//  Copyright (C) 1997, 1998 Olivetti & Oracle Research Laboratory
//
//  This file is part of the VNC system.
//
//  The VNC system is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
//  USA.
//
// If the source code for the VNC system is not available from the place 
// whence you received this file, check http://www.orl.co.uk/vnc or contact
// the authors on vnc@orl.co.uk for information on obtaining it.

#include "stdhdrs.h"
#include "vncviewer.h"
#include "VNCviewerApp.h"
#include "Exception.h"

// Register the Bell sound event

const char* BELL_APPL_KEY_NAME  = "AppEvents\\Schemes\\Apps\\VNCviewer";
const char* BELL_LABEL = "VNCviewerBell";

VNCviewerApp *pApp;

VNCviewerApp::VNCviewerApp(HINSTANCE hInstance, PSTR szCmdLine) {
	pApp = this;
	m_instance = hInstance;

	m_pdaemon = NULL;
	m_pflasher = NULL;

	// Read the command line
	m_options.SetFromCommandLine(szCmdLine);
	
	// Logging info
	log.SetLevel(m_options.m_logLevel);
	if (m_options.m_logToConsole) {
		log.SetMode(Log::ToConsole | Log::ToDebug);
	}
	if (m_options.m_logToFile) {
		log.SetFile(m_options.m_logFilename);
	}
	
	// Initialise winsock
	WORD wVersionRequested = MAKEWORD(2, 0);
	WSADATA wsaData;
	if (WSAStartup(wVersionRequested, &wsaData) != 0) {
		MessageBox(NULL, "Error initialising sockets library", "VNC info", MB_OK | MB_ICONSTOP);
		exit(1);
	}
	log.Print(3, _T("Started and Winsock (v %d) initialised\n"), wsaData.wVersion);
	
	// Allow bell sound to be changed from control panel
	RegisterSounds();
	
	// Load a requested keyboard layout
	if (m_options.m_kbdSpecified) {
		HKL hkl = LoadKeyboardLayout(  m_options.m_kbdname, 
			KLF_ACTIVATE | KLF_REPLACELANG | KLF_REORDER  );
		if (hkl == NULL) {
			MessageBox(NULL, "Error loading specified keyboard layout", 
				"VNC info", MB_OK | MB_ICONSTOP);
			exit(1);
		}
	}
	
	
	
	// Start listening daemons if requested
	
	if (m_options.m_listening) {
		log.Print(3, _T("In listening mode - staring daemons\n"));
		
		try {
			m_pflasher = new Flasher(FLASH_PORT_OFFSET);
			m_pdaemon = new Daemon(INCOMING_PORT_OFFSET);
		} catch (WarningException &e) {
			char msg[1024];
			sprintf(msg, "Error creating listening daemon:\n\r(%s)\n\r%s",
				e.m_info, "Perhaps another VNCviewer is already running?");
			MessageBox(NULL, msg, "VNCviewer error", MB_OK | MB_ICONSTOP);
			exit(1);
		}
		
	} 
	// Start connection if specified on command line
	// or if not listening
	
	if (m_options.m_connectionSpecified) {
		NewConnection(m_options.m_host, m_options.m_port);
	} else if (!m_options.m_listening) {
		NewConnection();
	}
	
}

// These should maintain a list of connections.

void VNCviewerApp::NewConnection() {
	ClientConnection *pcc = NULL;
	try {
		pcc = new ClientConnection(this);
		pcc->Run();
	} catch (Exception &e) {
		if (pcc != NULL) {
			pcc->Kill();
			e.Report();	
		}
	}
}

void VNCviewerApp::NewConnection(char *host, int port) {
	ClientConnection *pcc = NULL;
	try {
		pcc = new ClientConnection(this, host,port);
		pcc->Run();
	} catch (Exception &e) {
		if (pcc != NULL) {
			pcc->Kill();
			e.Report();	
		}
	}
}

void VNCviewerApp::NewConnection(SOCKET sock) {
	ClientConnection *pcc = NULL;
	try {
		pcc = new ClientConnection(this, sock);
		pcc->Run();
	} catch (Exception &e) {
		if (pcc != NULL) {
			pcc->Kill();
			e.Report();	
		}
	}
}


// --------------------------------------------------------------------------

void VNCviewerApp::RegisterConnection(ClientConnection *pConn) {
	m_clilist.push_front(pConn);
	log.Print(4,"Registered connection with app\n");
}

void VNCviewerApp::DeregisterConnection(ClientConnection *pConn) {

	m_clilist.remove(pConn);
	log.Print(4,"Deregistered connection from app\n");
	if (m_clilist.empty() && !pApp->m_options.m_listening) PostQuitMessage(0);
}

// ----------------------------------------------

void VNCviewerApp::RegisterSounds() {
	HKEY hBellKey;
	char keybuf[256];
	
	sprintf(keybuf, "AppEvents\\EventLabels\\%s", BELL_LABEL);
	// First create a label for it
	if ( RegCreateKey(HKEY_CURRENT_USER, keybuf, &hBellKey)  == ERROR_SUCCESS ) {
		RegSetValue(hBellKey, NULL, REG_SZ, "Bell", 0);
		RegCloseKey(hBellKey);
		
		// Then put the detail in the app-specific area
		
		if ( RegCreateKey(HKEY_CURRENT_USER, BELL_APPL_KEY_NAME, &hBellKey)  == ERROR_SUCCESS ) {
			
			sprintf(keybuf, "%s\\%s", BELL_APPL_KEY_NAME, BELL_LABEL);
			RegCreateKey(HKEY_CURRENT_USER, keybuf, &hBellKey);
			RegSetValue(hBellKey, NULL, REG_SZ, "Bell", 0);
			RegCloseKey(hBellKey);
			
			sprintf(keybuf, "%s\\%s\\.current", BELL_APPL_KEY_NAME, BELL_LABEL);
			if (RegOpenKey(HKEY_CURRENT_USER, keybuf, &hBellKey) != ERROR_SUCCESS) {
				RegCreateKey(HKEY_CURRENT_USER, keybuf, &hBellKey);
				RegSetValue(hBellKey, NULL, REG_SZ, "ding.wav", 0);
			}
			RegCloseKey(hBellKey);
			
			sprintf(keybuf, "%s\\%s\\.default", BELL_APPL_KEY_NAME, BELL_LABEL);
			if (RegOpenKey(HKEY_CURRENT_USER, keybuf, &hBellKey) != ERROR_SUCCESS) {
				RegCreateKey(HKEY_CURRENT_USER, keybuf, &hBellKey);
				RegSetValue(hBellKey, NULL, REG_SZ, "ding.wav", 0);
			}
			RegCloseKey(hBellKey);
		}
		
	} 
	
}

VNCviewerApp::~VNCviewerApp() {
		
	// We don't need to clean up pcc if the thread has been joined.
	if (m_pdaemon != NULL) delete m_pdaemon;
	if (m_pflasher != NULL) delete m_pflasher;
	
	
	// Clean up winsock
	WSACleanup();
	
	log.Print(2, _T("VNC viewer closing down\n"));

}
