#!/usr/bin/env python
# Data Tables - includes update logic, date routines, and gantt calculations

# Copyright 2004, 2005, 2006, 2007, 2008 by Brian C. Christensen

#    This file is part of GanttPV.
#
#    GanttPV is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    GanttPV is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with GanttPV; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# 040407 - first version w/ FillTable and Update to create some sample data
# 040408 - added date conversion tables
# 040409 - added GanttCalculation
# 040410 - Update will now update
# 040412 - added CheckChange and SetUndo; added OpenReports
# 040413 - added SetEmptyData
# 040414 - renamed file to Data; added load and store routines; corrections to Update; reserved ReportID's 1 & 2 for 'Project/Report List' and 'Resource List'
# 040415 - moved menu adjust logic here from Main.py; made various changes to correct or improve logic
# 040416 - changed gantt calculation to process all projects; wrote Undo and Redo routines
# 040417 - added optional parameter to Update to specify whether change should be added to UndoStack; in CheckChange now sets ChangedReport flag if zzStatus appears; moved flags and variables to front of file this; added MakeReady to do all the necessary computations after a new database is loaded; added AdjustReportRows to add report rows when records are added
# 040419 - reworked AdjustReportRows logic
# 040420 - added routines to open and close reports; revised MakeReady
# 040422 - fix so rows added by AdjustReportRows will always have a link to their parent; added row type colors  for Project/Report frame
# 040423 - added ReorderReportRows; moved AutoGantt to Option; added ConfirmScripts to Option
# 040424 - added ScriptDirectory to Option; added LoadOption and SaveOption; added GetRowList; fixed reset error in date tables;
# 040426 - fixed Recalculate to check ChangedReport; added check for None to GetRowList and ReorderReportRows; added table Assignment; in GanttCalculation ignore deleted Dependency records
# 040427 - in Resource table changed 'LongName' to 'Name'; added Width to ReportColumn; revised ReportColumn fields
# 040503 - added ReportType and ColumnType tables; moved UpdateDataPointers here from Main and GanttReport; added "Prerequisite" synonym for "Task" in Database
# 040504 - added default Width to ColumnType; in gantt calculation, will ignore tasks w/o projects; added GetColumnList and ReorderReportColumns; added GetToday
# 040505 - fixed "Computed"/"Calculated" inconsistency; changed some column type labels
# 040506 - add info line at beginning of saved files; added BaseBar to chart options
# 040508 - start w/ empty data instead of sample data
# 040510 - added header line to option file
# 040512 - added 'T' (source table) to ColumnType; several bug fixes; fixed handling of holidays in date tables
# 040520 - fixed bug where Assignment pointer was not set for Loaded files
# 040605 - fixed bug in ReorderReportColumns to properly handle zzStatus == None
# 040715 - Pierre_Rouleau@impathnetworks.com: removed all tabs, now use 4-space indentation level to comply with Official Python Guideline.
# 040815 - FileName not set to None on New (because not on globals list)
# 040906 - add "project name / report name" to report titles
# 040928 - Alexander - when making DateConv, ignore incorrectly formatted dates; in other places, ignore dates not in DateConv
# 041001 - added FindID and FindIDs
# 041009 - tightened edits on FirstDate and LastDate; added ValidDate() routine
# 041012 - moved AddTable, AddRow, and AddReportType here from "Add Earned Value Tracking.py"
# 041031 - don't automatically add report rows for "deleted" records
# 041203 - GetColumnHeader and GetCellValue logic moved here from GanttReport
# 041231 - added xref to find first day of next month; changed GetColumnHead to work for months and quarters; added routine to add months to a date index
# 050101 - added hours units
# 050104 - added backwards pass and float calculations
# 050328 - derive parent dates from children
# 050329 - support multi-value list type columns (for predecessors, successors, children, and resource names)
# 050402 - handle task w/ self for parent
# 050407 - Alexander - close deleted reports and prevent them from opening
# 050409 - Alexander - added GetModuleNames
# 050503 - Alexander - added App, for program quiting; and ActiveReport, for script-running and window-switching
# 050423 - moved GetPeriodInfo logic to calculate period start and hours to Data from GanttReport.py; added GetColumnDate; save "SubtaskCount" in Task table
# 050504 - Alexander - moved script-running logic here; added logic to prevent no-value entries in the database; added logic to update the Window menu.
# 050513 - Alexander - revised some dictionary fetches to ignore invalid keys (instead of raising an exception); tightened _Do logic
# 050519 - Brian - use TaskID instead of ParentTaskID to designate Task parent.
# 050521 - Alexander - fixed work week bug in SetupDateConv
# 050527 - Alexander - added 'Generation' column to designate levels in the task-parenting heirarchy; updated in GanttCalculation
# 050531 - Brian - in AdjustReportRows test to make sure parent exists before testing parent's select column value
# 050716 - Brian - in GetCellValue use more general routine to find period
# 050806 - Alexander - rewrote date conversion!
# 050814 - Alexander - added SearchByColumn
# 050826 - Alexander - rewrote row/column ordering!
# 050903 - Alexander - fixed bug in AddRow (was aborting when table was empty)
# 051121 - Brian - add FileSignature number
# 060131 - Alex - save main window size/location
# 060211 - Alex - changed AddRow and AddReportType so that, by default, they activate the records they modify
# 060314 - Alex - added ParseLinks, ParsePath, and ParseBranching
# 060316 - Alex - rewrote time period functions; changed GetRowList and GetColumnList to skip deleted rows and columns
# 060316 - Brian - moved startup data to separate file
# 060317 - Brian - moved to ReportAids all of the routines that know about wx, GanttPV, GanttReport, or Menu
# 060323 - Brian - change update to keep track of deltas from the server's data
# 060325 - Alex - moved here the "meat" of UpdateRowPointers and UpdateColumnPointers
# 060403 - Brian - add override of FirstDate in time scales
# 060404 - Brian - allow Resource and other measurement types (in addition to project ones)
#                  moved SetValue logic here from GanttReport as SetCellValue
#                  added suggested columns to new reports
# 060405 - Brian - fix an error in ResourceMeasurement table maping
# 060427 - Brian - include Task StartHour in critical path calculation
# 060506 - Alex - allow dependencies between parent tasks
# 060520 - Alex - fix holiday bugs; added AddColumn, AddReport, and GetSuggestedColumns
# 060603 - Brian - identify current platform
# 060709 - Brian - add ability to create aliases in Update
# 060723 - Brian - translate column type labels
# 060725 - Alex - added AddAlias and UpdateAliases
# 060802 - Brian - added Task's StartHour to check change
# 060826 - Alex - allow AddReportType to change column labels after all
# 070328 - Brian - Change SetCellValue so that it will not change protected cells
# 070731 - Alex - hide assignment or dependency records that point to a deleted task or resource (somewhat ad hoc)
# 080517 - Brian - added object api
# 080602 - Brian - add subtypes to object api
# 080604 - Brian - simplify return of objects via object api (don't require 'get')
# 080609 - Brian - simplify setting of object pointer values via object api
# 090112 - Brian - added update, set undo, open/close report events

import datetime, calendar
import cPickle
# import wx
import os, sys
# import Menu, GanttReport
import random
import StartupData
import Event
import re

debug = 1
if debug: print "load Data.py"

# On making changes in GanttPV
# 1- Use Update for all changes
# 2- CheckChange will decide the impact of the changes
# 3- Use SetUndo to tell GanttPV to fix anything in response to the changes

if sys.platform.startswith("darwin"):
    platform = "mac"
elif sys.platform.startswith("win32"):
    platform = "win"
elif sys.platform.startswith("linux"):
    platform = "linux"
else:
    platform = "other"

# this is the Data only version
# it is overriden by ReportAids.py
def GetModuleNames():
    """ Return the GanttPV modules as a namespace dictionary.

    This dictionary should be passed to any scripts run with execfile.
    """
#    import Data, GanttPV, GanttReport, ID, Menu, UI, wx
    import Data, ID  # GanttPV, GanttReport, Menu, and UI are not available to server scripts
    return locals()

App = None  # the application itself (deprecated)
ActiveReport = 1  # ReportID of most recently active frame

# Clients should treat these tables as read only data
Database = {}           # will contain pointers to all of these tables
ChangedData = False     # True if database needs to be saved
FileName = None         # Filename of current database
Path = ""               # Directory containing executed program, look for option file here

# save impact of change until it can be addressed
ChangedCalendar = False  # will affect calendar hours tables (gantt will have to be redone, too)
ChangedSchedule = False  # gantt calculation must be redone
ChangedReport = False    # report layout needs to be redone
ChangedRow = False       # a record was created, the rows on a report may be affected

UndoStack = []
RedoStack = []
OpenReports = {}  # keys are report id's of open reports. Main is always "1"; Report Options is always "2"

# month names (for time scales)

MonthNames = ['January', 'February', 'March', 'April', 'May', 'June',
    'July', 'August', 'September', 'October', 'November', 'December']

ShortMonthNames = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
    'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']

# these options are user preferences
Option = StartupData.GanttPVOptions()


# -------------- Setup Empty and Sample databases

# reserved table column names: 'Table', 'ID', 'zzStatus', anything ending with 'ID", anything starting with 'xx'

def SetEmptyData():
    global Database, FileName
    Database = StartupData.GanttPVData()
    FileName = None

    # CloseReports()
    # Database = Database
    # MakeReady()

# ---------

def AddTable(name):
    """ Add the table if it doesn't already exist """
    # (Note: this isn't reversed by Undo.)
    if name and name not in Database:
        Database[name] = {}
        Database['NextID'][name] = 1

def AddAlias(alias, tableName):
    """ Create an alias to an existing table """
    # (Note: this isn't reversed by Undo.)
    change = {'Table': 'TableAlias', 'ID': 1, alias: tableName}
    Update(change, 0)

def AddRow(change):
    """ Add or update row """
    changeTable = Database.get(change.get("Table")) or {}
    changeName = change.get("Name")
    for k, v in changeTable.iteritems():
        if v.get('Name') == changeName:
            change['ID'] = k
            break
    if 'zzStatus' not in change:
        change['zzStatus'] = None  # default to active status
    Update(change)

def AddReportType(reportType, columnTypes):
    # should add code to ensure all values are valid

    # convert "Also" name to record id
    also = reportType.get("Also")
    if also:
        alsoid = 0
        for k, v in Database['ReportType'].items():
            if v.get('Name') == also:
                alsoid = k
                break
        if alsoid:
            reportType['Also'] = alsoid
        else:
            if debug: print "couldn't convert Also value", also
            del reportType['Also']

    # Make sure tables exist that are referenced by ReportType
    tableNames =  map( reportType.get, ["TableA", "TableB"] )
    for name in tableNames:
        AddTable(name)

    # Either add or update ReportType
    reportTypeName = reportType["Name"]
    rtid = 0
    for k, v in Database['ReportType'].items():
        if v.get('Name') == reportTypeName:
            rtid = k
            break
    change = reportType
    change["Table"] = 'ReportType'
    if rtid:  # already exists
        change['ID'] = rtid
        if 'zzStatus' not in change:
            change['zzStatus'] = None  # default to active status
        Update(change)
        oldRT = True
    else:  # new
        undo = Update(change)
        rtid = undo['ID']
        oldRT = False

    # expects a list of ColumnTypes
    for change in columnTypes:
        typeName = change["Name"]

        # check whether column type already exists
        ctid = 0
        if oldRT:
            for k, v in Database['ColumnType'].items():
                if v.get('ReportTypeID') == rtid and v.get('Name') == typeName:
                    ctid = k
                    break
        if ctid:
            change['ID'] = ctid  # if column type already exists, change this to an update instead of an add
            # if change.has_key('Label'): del change['Label']  # don't change label on existing column type, is test necessary?
                # was intended to preserve translations; no longer needed
            if 'zzStatus' not in change:
                change['zzStatus'] = None  # default to active status
        change["Table"] = "ColumnType"
        change["ReportTypeID"] = rtid

        # check for required fields ??
        Update(change)

def AddColumn(reportid, column):
    """
    column['ColumnTypeID'] must be (ReportType name, ColumnType name)
    """
    change = column.copy()
    change['Table'] = 'ReportColumn'
    change['ReportID'] = reportid

    # convert ReportType and ColumnType names to id
    rtname, ctname = column.get('ColumnTypeID')
    rtid = SearchByColumn(ReportType, {'Name': rtname}, True)
    if not rtid: return
    ctid = SearchByColumn(ColumnType, {'Name': ctname, 'ReportTypeID': rtid},
        True)
    if not ctid: return
    change['ColumnTypeID'] = ctid

    if 'Width' not in column:
        ct = ColumnType.get(ctid, {})
        column['Width'] = ct.get('Width')

    id = Update(change)['ID']
    return id

def AddReport(projectid, report, columns=None):
    """
    report['ReportTypeID'] must be a ReportType id or a ReportType name
    columns[k]['ColumnTypeID'] must be (ReportType name, ColumnType name)
    """
    change = report.copy()
    change['Table'] = 'Report'
    change['ProjectID'] = projectid

    rtid = report.get('ReportTypeID')
    if not isinstance(rtid, int):
        # convert ReportType name to id
        rtid = SearchByColumn(ReportType, {'Name': rtid}, True)
        if not rtid: return
        change['ReportTypeID'] = rtid

    if 'Name' not in change:
        rt = ReportType.get(rtid, {})
        change['Name'] = rt.get('Label') or rt.get('Name') or '--'

    if projectid != 1:
        change['SelectColumn'] = 'ProjectID'
        change['SelectValue'] = projectid

    reportid = Update(change)['ID']
    if reportid and columns:
        clist = [AddColumn(reportid, c) for c in columns]
        ReorderReportColumns(reportid, clist)
    return reportid

def GetSuggestedColumns(reporttypeid):
    """
    Return a list of default columns for use with AddReport

    This is experimental. It may be replaced with a different mechanism.
    We don't promise it will work in the next release. Very tentative.
    """
    rt = ReportType.get(reporttypeid, {})
    suggestion = rt.get("SuggestedColumns", "")
    # if not suggestion:
    #     also = rt.get("also")
    #     also_rt = ReportType.get(reporttypeid, {})
    #     also_name = also_rt.get("Name")
    #     if also_name:
    #         suggestion = "%s;" % also_name
    defaults = [rt.get("Name"), 'Name', None, None]
    columns = []
    for column_string in suggestion.split(";"):
        elements = column_string.split(",")
        elements = [(v or defaults[k]) for k, v in enumerate(elements)]
        elements += defaults[len(elements):]

        rtname, ctname, width, periods = elements[:len(defaults)]
            # number of values extracted must equal len(defaults)
        if not (rtname and ctname):
            continue

        column = {'Table': 'ReportColumn', 'ColumnTypeID': (rtname, ctname)}
        if width:
            column['Width'] = int(width)
        if periods:
            column['Periods'] = int(periods)
        columns.append(column)
    return columns

# ---------
def GetTableNames():
    return [name for name in NextID
            if not name.startswith('_') and name != 'ID']

def FindID(table, field1, value1, field2=None, value2=None):  # deprecated v0.7
    search = {field1: value1, field2: value2}
    return SearchByColumn(Database.get(table), search, True)

def FindIDs(table, field1, value1, field2=None, value2=None): # deprecated v0.6
    search = {field1: value1, field2: value2}
    return SearchByColumn(Database.get(table), search).keys()

def SearchByColumn(table, search, unique=False):
    """ Search the table for records with specific column values

    table -- a dictionary of records
    search -- a dictionary of the form {field1: value1, ...}
    unique -- if true, return only the ID of the first matching record;
      otherwise, return a subset of the table: a dictionary that contains
      the matching records by ID.

    record matches if record.get(key) == val for every item in the search.
    """
    if unique:
        result = 0
    else:
        result = {}
    if not table: return result
    for id, record in table.iteritems():
        for key, val in search.iteritems():
            if record.get(key) != val: break
        else:
            if unique: return id
            result[id] = record
    return result

def CheckChange(change):
    """ Check for important changes

    change -- the undo info for the changes (only changed columns included)
    """
    global ChangedCalendar, ChangedSchedule, ChangedReport, ChangedRow
    if debug: print 'CheckChange:', change
    if not change.has_key('Table'):
        if debug: print "change does not specify table"
        return

    if ChangedRow: pass
    elif 'zzStatus' in change:  # something has been added or deleted
        ChangedRow = True
    else:
        for k in change:
            if k[-2:] == 'ID' and len(k) > 2:  # foreign key was changed
                ChangedRow = True; break

    if ChangedCalendar: pass
    elif change['Table'] == 'OtherData':
        for k in ('WeekHours',):
            if change.has_key(k): ChangedCalendar = True; break
    elif change['Table'] == 'Holiday':
        for k in ('Date', 'Hours', 'zzStatus'):
            if change.has_key(k): ChangedCalendar = True; break

    elif ChangedSchedule: pass
    elif change['Table'] == 'Task':
        for k in ('StartDate', 'DurationHours', 'zzStatus', 'ProjectID', 'TaskID', 'StartHour'):
            if change.has_key(k): ChangedSchedule = True; break
    elif change['Table'] == 'Dependency':
        for k in ('PrerequisiteID', 'TaskID', 'zzStatus'):
            if change.has_key(k): ChangedSchedule = True; break
    elif change['Table'] == 'Project':
        for k in ('zzStatus', 'StartDate'):
            if change.has_key(k) :  ChangedSchedule = True; break

    if ChangedRow or ChangedReport: pass
    elif change['Table'] == 'ReportRow':
        for k in ('NextRow', 'Hidden'):
            if change.has_key(k): ChangedReport = True; break
    elif change['Table'] == 'ReportColumn':
        for k in ('Type', 'NextColumn', 'Time', 'Periods', 'FirstDate'):
            if change.has_key(k): ChangedReport = True; break
    elif change['Table'] == 'Report':
        for k in ('Name', 'FirstColumn', 'FirstRow', 'ShowHidden', 'zzStatus'):
            if change.has_key(k): ChangedReport = True; break

# this routine is overriden by ReportAids.py
def RefreshReports():
    pass

# ----- undo and redo

def Recalculate(autogantt=True):
    global ChangedCalendar, ChangedSchedule, ChangedReport

    UpdateCalendar = ChangedCalendar and autogantt
    UpdateGantt = UpdateCalendar or (ChangedSchedule and autogantt)
    UpdateReports = ChangedReport or ChangedRow or UpdateCalendar

    # these routines shouldn't add anything to the undo stack
    if UpdateCalendar:
        SetupDateConv(); ChangedCalendar = False
    if UpdateGantt:
        GanttCalculation(); ChangedSchedule = False
    if UpdateReports:
        for v in OpenReports.values():
            if v: v.UpdatePointers()
        ChangedReport = False

    RefreshReports()

SetUndoEvent = Event.Event(msg='Set Undo')

def SetUndo(message):
    """ This is the last step in submitting a group of changes to the database.
    Adjust calculated values and update the displays.
    """
    global ChangedReport, ChangedRow, ChangedData, UndoStack, RedoStack
    if debug: print "Start SetUndo:", message

    if ChangedRow:
        AdjustReportRows(); ChangedReport = True; ChangedRow = False

    if UndoStack and not isinstance(UndoStack[-1], str):  # prevent empty batch
        UndoStack.append(message)
        RedoStack = []  # clear out the redo stack
        ChangedData = True  # file needs to be saved

    autogantt = Option.get('AutoGantt')
    Recalculate(autogantt)
    SetUndoEvent(message)
    
    if debug: print "End SetUndo"

def _Do(fromstack, tostack):
    global ChangedData, ChangedRow
    if fromstack and isinstance(fromstack[-1], str):
        savemessage = fromstack.pop()
        while fromstack and isinstance(fromstack[-1], dict):
            change = fromstack.pop()
            redo = Update(change, 0)  # '0' means don't put change into Undo Stack
            tostack.append(redo)
        tostack.append(savemessage)
        ChangedData = True  # file needs to be saved
        Recalculate()
        ChangedRow = False  # already handled by last SetUndo
        SetUndoEvent(savemessage)

def DoUndo():
    _Do(UndoStack, RedoStack)

def DoRedo():
    _Do(RedoStack, UndoStack)

# --------------------------

UpdateEvent = Event.Event(msg='Update')

# update routine
def Update(change, push=1):
    global ChangedRow
    if debug: print 'Update:', change
    tname = change.get('Table')
    if not tname:
        if debug: print 'change does not specify table'
        raise KeyError

    table = Database.get(tname)
    if table == None:
        if debug: print 'change specifies invalid table:', tname
        raise KeyError

    undo = {'Table': tname}
    if change.has_key('ID'):
        id = undo['ID'] = change['ID']
        if id not in table:
            if debug: print 'change specifies invalid record:', id
            raise KeyError

        record = table[id]
        for c, newval in change.iteritems():  # process each field
            # fieldnames must have at least one character
            # don't accept changes to '_' fields which are used to track server delta
            if not c or c == 'Table' or c == 'ID' or c[0] == '_': continue
            oldval = record.get(c)
            if newval != oldval:
                serverid = NextID.get('_' + tname)
                if serverid and id < serverid and not record.has_key('_' + c):
                    record['_' + c] = oldval
                undo[c] = oldval
                if newval or newval == 0:
                    record[c] = newval
                elif c in record:
                    del record[c]
        if len(undo) > 2:
            CheckChange(undo)
    else:
        record = {}
        for c, newval in change.iteritems():  # process each field
            if c == 'Table': continue
            if newval or newval == 0:
                record[c] = newval
                undo[c] = None
        undo['zzStatus'] = 'deleted'
        id = NextID[tname]
        if debug: print "Added new record:", id
        NextID[tname] = id + 1
        record['ID'] = undo['ID'] = id
        CheckChange(undo)
        table[id] = record

    if change['Table'] == 'TableAlias':  # add aliases if needed
        tableNames = GetTableNames()
        for k, v in change.iteritems():
            if k in ('Table', 'ID', 'zzStatus'):  # not an alias
                continue
            if (k in tableNames) or (v not in tableNames):  # invalid alias
                continue
            Database[k] = Database[v]
    if push and len(undo) > 2: UndoStack.append(undo)
    UpdateEvent(undo, push=push)
    return undo

# GanttPV Object API

class Object:
    def __init__(self, db, object_type, object_id):
        ''' object_type = table name; object_id = row id '''
        self.__dict__['db'] = db
        self.__dict__['Table'] = object_type
        self.__dict__['ID'] = object_id
##        self.Table = object_type  # avoiding use of __setattr__
##        self.ID = object_id

    def __str__(self):
        if self.Table in ('ReportRow', 'GraphicObject'):
            return ('Object Type %s with ID %d and target %s' %
                (self.Table, self.ID, self.TableName))
        return 'Object Type %s with ID %d' % (self.Table, self.ID)
        
    __repr__ = __str__  # is this right??
        
    def __getattr__(self, name):
# it is impossible to return values for 'Get', 'GetList', or 'Valid'
# the program only comes here if it can't find the real attribute
#     this applies to db, Table, and ID

        # edit on every access or on creation?
        if not self.Table in self.db.Database:
            return None
        rec = self.db.Database[self.Table].get(self.ID)
        if not rec:
            return None

        # if the name didn't refer to a value, maybe an object was intended
        result = rec.get(name)
        if result == None and not name.endswith('ID'):
            return self.Get(name)  # return object
        else:
            return result  # return value
##        # temporary version
##        if name.endswith('IDX'):  # return the object
##            return self.Get(name[:-3])
##        return rec.get(name)  # any column name

    def __nonzero__(self):  # if object == false if not valid
        if not self.Table in self.db.Database:
            return False
        rec = self.db.Database[self.Table].get(self.ID)
        if not rec:
            return False
        return True

    def __eq__(self, other):  # so the 'in' operator will work
        try:
            return (self.ID, self.Table) == (other.ID, other.Table)
        except AttributeError:
            return False

    def __hash__(self):  # IMPORTANT: hash values may be changed by share w/ server script
        return hash((self.Table, self.ID))

    def __setattr__(self, name, value):
        # edits
        if name in ('Table', 'ID', 'db', 'Valid', 'Get', 'GetList', 'GetGraphicList'):
            return  # silently ignore attempts to change these?
        if not self.Valid():
            return
        if isinstance(value, Object):  # make it easier to do foreign keys
            if name == 'Target':
                change = {'Table': self.Table, 'ID': self.ID,
                          'TableName': value.Table, 'TableID': value.ID }
            elif self.db._ConvertAlias(name) == value.Table:
                change = {'Table': self.Table, 'ID': self.ID,
                          name + 'ID': value.ID }
            else:  # silently ignore object vs. name mismatches
                return
        else:    
            change = {'Table': self.Table, 'ID': self.ID, name: value }
        Update(change)  # can only update the active database -- IMPORTANT
        return

    # ---- these are experimental ----
    def _SetInShell(self, tag, value):
        '''Don't update the database; save value in shell object'''
        self.__dict__[tag] = value

    def _SetToSubclass(self, subclass):
        '''Change the class of the object to a subtype of the class'''
        # self.__dict__['__class__'] = subclass  # how to make this work?
        self.__class__ = subclass  # or this??
        print self.__class__

    # ----- end -------
    
    def Valid(self):
        try:
            rec = self.db.Database[self.Table].get(self.ID)
            return (rec and rec.get('zzStatus') != 'deleted')
        except:
            return None

    def Delete(self):
        self.zzStatus = 'deleted'

    def Get(self, name=None, getDeleted=False):
        '''
returns an object that this one points to
returns object pointed to by name + ID
does not create a new object
        '''
        if not name: return None
        if not self.Valid(): return None

##        if self.Table in ('ReportRow', 'GraphicObject') and name == 'Target':
        if name == 'Target':
            if self.TableName and self.TableID:  # don't create new
                return self.db.GetObject(self.TableName, self.TableID, getDeleted=getDeleted)
            else:
                return None

        rec = self.db.Database[self.Table].get(self.ID)
        target_id = rec.get(name + 'ID')  # FK value
        if target_id:  # don't create new
            return self.db.GetObject(name, target_id, getDeleted=getDeleted)  # object
        else:
            return None

    def GetList(self, table_name, attribute=None, getDeleted=False):
        '''
returns a list of "table_name type" objects that point to this one
list_of_assignment_objects = ThisTask.Find('Assignment', 'Prerequisite')
Returns a list of all assignment objects that point to the original task object
        '''
        if attribute:
            fk = attribute
        else:
            fk = self.Table
        table_rows = self.db.Database.get(table_name) or {}
        search = { fk + 'ID': self.ID }
        result = SearchByColumn(table_rows, search)  # only works w/ current db -- IMPORTANT
        # if debug: print result
        result = [self.db.GetObject(table_name, x, getDeleted=getDeleted) for x in result.iterkeys()]
        return filter(None, result)

    def GetGraphicList(self, subtype_name, attribute=None, getDeleted=False):
        '''
returns a list of "table_name type" objects that point to this one
list_of_assignment_objects = ThisTask.Find('Assignment', 'Prerequisite')
Returns a list of all assignment objects that point to the original task object
        '''
        if attribute:
            fk = attribute
        else:
##            fk = 'GraphicObject'
            fk = self.Table
        table_rows = self.db.Database.get('GraphicObject') or {}
        search = { fk + 'ID': self.ID, 'Subtype': subtype_name }
        result = SearchByColumn(table_rows, search)  # only works w/ current db -- IMPORTANT
        # if debug: print result
        result = [self.db.GetObject('GraphicObject', x, getDeleted=getDeleted) for x in result.iterkeys()]
        return filter(None, result)

class DB:
    def __init__(self, database):
        ''' database '''
        self.Database = database
        self._ObjectXref = {}  # key = (table, id)

    def _ConvertAlias(self, object_type):
        # convert known aliases
        da = self.Database['TableAlias'][1]
        if object_type in da:
            object_type = da[object_type]
        return object_type
    
    _subtype = {}
    @classmethod
    def RegisterSubtype(cls, subtype_name, constructor):
        cls._subtype[subtype_name] = constructor

    @classmethod
    def UnRegisterSubtype(cls, subtype_name):
        if subtype_name in cls._subtype:
            del cls._subtype[subtype_name]

    _type = {}
    @classmethod
    def RegisterType(cls, type_name, constructor):
        cls._type[type_name] = constructor

    @classmethod
    def UnRegisterType(cls, type_name):
        if type_name in cls._type:
            del cls._type[type_name]

    def GetObject(self, object_type, object_id=None, subtype=None, getDeleted=False):
        if object_type in ('Table', 'ID'):
            return None
        object_type = self._ConvertAlias(object_type)
        # add new objects to database
        if not object_id:
            new = {'Table': object_type, 'Subtype': subtype}
            object_id = Update(new)['ID']  # only works w/ current db -- IMPORTANT
        else: # existing object
            new = None
        # create object, save reference for possible future use
        key = (object_type, object_id)
        if key in self._ObjectXref:
            ob = self._ObjectXref[key]
        else:
            if not new:  # lookup subtype in record
                subtype = self.Database[object_type][object_id].get('Subtype')
            if subtype in DB._subtype:
                ob = DB._subtype[subtype](self, object_type, object_id)
            elif object_type in DB._type:
                ob = DB._type[object_type](self, object_type, object_id)
            else:
                ob = Object(self, object_type, object_id)
            self._ObjectXref[key] = ob
        # I create the object even if I don't return it
        # that may be a problem? at minimum it is inefficient
        if getDeleted or ob.Valid():
            return ob
        else:
            return None

    def GetList(self, table_name, getDeleted=False):
        '''
returns a list of "table_name type"
        '''
        table = self.Database.get(table_name) or {}
        result = [self.GetObject(table_name, x, getDeleted=getDeleted) for x in table.iterkeys()]
        return filter(None, result)

# end of GanttPV object API

########## @@@@@@@@@@ Start Alex Date Conversion @@@@@@@@@@ ##########

# calendar setup

def SetupDateConv():
    UpdateWorkWeek()
    ReadHolidays()
    UpdateHolidayHours()

def UpdateWorkWeek():
    global WorkWeek, WeekSize, CumWeek
    global HoursPerWeek, DaysPerWeek, HoursPerDay, AllowDaysUnit
    global ChangedCalendar
    if debug: print 'start UpdateWorkWeek'
    Other = Database['OtherData'][1]

    WorkWeek = list(Other.get('WeekHours') or (8, 8, 8, 8, 8, 0, 0))
    WeekSize = len(WorkWeek)

    CumWeek = []
    HoursPerWeek = DaysPerWeek = 0
    for day in WorkWeek:
        CumWeek.append(HoursPerWeek)
        if day:
            HoursPerWeek += day
            DaysPerWeek += 1

    HoursPerDay = Other.get('HoursPerDay')
    if HoursPerDay:
        AllowDaysUnit = True
    else:
        HoursPerDay = HoursPerWeek.__truediv__(DaysPerWeek)
        AllowDaysUnit = True
        # AllowDaysUnit = (DaysPerWeek == WorkWeek.count(HoursPerDay))

    if debug: print 'end UpdateWorkWeek'

HolidayMap = {}
HolidayDate = []
HolidayHour = []
HolidayAdjust = []

def ReadHolidays():
    """ Read the holiday dates and lengths from the database """
    global HolidayMap, HolidayDate
    if debug: print 'start ReadHolidays'
    HolidayMap = {}
    for r in Database['Holiday'].itervalues():
        if r.get('zzStatus') == 'deleted': continue
        datestr = r.get('Date')
        try:
            date = StringToDate(datestr)
        except ValueError:
            continue
        hours = r.get('Hours')
        if hours:
            HolidayMap[date] = hours
        else:
            HolidayMap[date] = 0
    HolidayDate = HolidayMap.keys()
    HolidayDate.sort()
    if debug: print 'end ReadHolidays'

def UpdateHolidayHours():
    """ Prepare the holiday conversions """
    global HolidayHour, HolidayAdjust
    if debug: print 'start UpdateHolidayHours'
    HolidayHour = []
    HolidayAdjust = [0]
    adjust = 0
    for date in HolidayDate:
        w, dow = divmod(date, WeekSize)
        newLength = HolidayMap[date]
        oldLength = WorkWeek[dow]
        hour = w * HoursPerWeek + CumWeek[dow] + adjust + min(newLength, oldLength)
        adjust += newLength - oldLength
        HolidayHour.append(hour)
        HolidayAdjust.append(adjust)
    if debug: print 'end UpdateHolidayHours'


# date / hour conversion

def DateToHours(date):
    """ Convert a date from days to hours """
    key = CutByValue(HolidayDate, date - 1)
    week, dow = divmod(date, WeekSize)
    hour = week * HoursPerWeek + CumWeek[dow] + HolidayAdjust[key]
    if key > 0:
        hour = max(hour, HolidayHour[key-1])
    return hour

def HoursToDate(hours):
    """ Convert a date from hours to days """
    key = CutByValue(HolidayHour, hours)
    h = hours - HolidayAdjust[key]
    if (key > 0) and (h < HolidayHour[key-1]):
        date = HolidayDate[key-1]
        hours -= HolidayHour[key-1]
    else:
        week, hours = divmod(h, HoursPerWeek)
        dow = CutByValue(CumWeek, hours) - 1
        date = week * WeekSize + dow
        hours -= CumWeek[dow]
    return date, hours

def HoursToDateString(hours):
    date, hours = HoursToDate(hours)
    return DateToString(date), hours

def CutByValue(list, value):
    """ Return the index such that list[index-1] <= value < list[index]

    list -- an ascending sequence
    value -- the search value

    """
    start = 0
    end = len(list)
    while start < end:
        center = (start + end) / 2
        if list[center] <= value:
            start = center + 1
        else:
            end = center
    return start


base_date_object = datetime.date(2001, 1, 1)
BaseDate = base_date_object.toordinal()
BaseYear = base_date_object.year

Years_per_Calendar_Cycle = 400  # Gregorian calendar
Days_per_Calendar_Cycle = 146097

DateFormat = "%04d-%02d-%02d"
NegDateFormat = "-" + DateFormat


# the current date

def TodayDate():
    y, m, d = _get_today()
    return _ymd_to_date(y, m, d)

def TodayString():
    y, m, d = _get_today()
    return _ymd_to_str(y, m, d)

GetToday = TodayString

def _get_today():
    dateobj = datetime.date.today()
    return _date_tuple(dateobj)

def _date_tuple(dateobj):
    return dateobj.year, dateobj.month, dateobj.day


# parsing entered dates

def CheckDateString(s):
    try:
        y, m, d = _user_str_to_ymd(s)
        _ymd_to_date(y, m, d)
    except ValueError:
        return ""
    return _ymd_to_str(y, m, d)

def _user_str_to_ymd(s):
    if not s:
        raise ValueError

    if s == '=':
        return _get_today()
    if s[0] == '*':
        today = TodayDate()
        dow = int(s[1:]) - 1
        date = today + (dow - today) % WeekSize
        return _date_to_ymd(date)

    parts = s.split('-')
    if s[0] == '-':
        parts[:2] = ['-' + parts[1]]

    ymd = [int(p) for p in parts]

    if s[0] in ('+', '-'):
        while len(ymd) < 3:
            ymd.append(1)
    elif len(ymd) < 3:
        ymd[:0] = _get_today()[:-len(ymd)]
    else:
        magnitude = 10 ** len(parts[0])
        currentyear = _get_today()[0]

        if currentyear < 0:
            diff = (-currentyear % magnitude) - ymd[0]
        else:
            diff = ymd[0] - (currentyear % magnitude)
        year = currentyear + diff

        if diff < 0:
            if (-diff > magnitude / 2) and not (0 < -year < magnitude):
                year += magnitude
        elif (diff > magnitude / 2) and not (0 < year < magnitude):
            year -= magnitude

        ymd[0] = year

    return ymd


# string to date conversion

def StringToDate(s):
    y, m, d = _str_to_ymd(s)
    date = _ymd_to_date(y, m, d)
    return date

def _str_to_ymd(s):
    if not s:
        raise ValueError
    parts = s.split('-')
    if s[0] == '-':
        parts[:2] = ['-' + parts[1]]
    ymd = [int(p) for p in parts]
    return ymd

def _ymd_to_date(year, month, day):
    cycles, year = divmod(year - BaseYear, Years_per_Calendar_Cycle)
    dateobj = datetime.date(year + BaseYear, month, day)
    date = dateobj.toordinal() - BaseDate + cycles * Days_per_Calendar_Cycle
    return date

def _coerce_ymd_to_date(year, month, day):
    # wrap month around year (e.g. 2005-15 -> 2006-03)
    # if day is too high, use the last day of that month

    year += (month - 1) // 12
    month = (month - 1) % 12 + 1

    equiv_year = (year - BaseYear) % Years_per_Calendar_Cycle + BaseYear
    month_size = calendar.monthrange(equiv_year, month)[1]
    day = min(day, month_size)

    return _ymd_to_date(year, month, day)


# date to string conversion

def DateToString(date):
    y, m, d = _date_to_ymd(date)
    s = _ymd_to_str(y, m, d)
    return s

def ValidDate(s):
    try:
        StringToDate(s)
        return s
    except ValueError:
        return ""

def _date_to_ymd(date):
    cycles, date = divmod(date, Days_per_Calendar_Cycle)
    dateobj = datetime.date.fromordinal(int(date) + BaseDate)
    y, m, d = _date_tuple(dateobj)
    y += cycles * Years_per_Calendar_Cycle
    return y, m, d

def _ymd_to_str(year, month, day):
    if year < 0:
        return NegDateFormat % (-year, month, day)
    else:
        return DateFormat % (year, month, day)


# month intervals

def AddMonths(date, months):
    y, m, d = _date_to_ymd(date)
    return _coerce_ymd_to_date(y, m + months, d)


# transition objects (for backwards compatibility)

class _str_to_date:
    def __init__(self):
        pass
    def __contains__(self, s):
        return ValidDate(s)
    def __getitem__(self, s):
        return StringToDate(s)
    def has_key(self, key):
        return key in self

class _date_to_str:
    def __init__(self):
        pass
    def __getitem__(self, d):
        return DateToString(d)

class _date_info:
    def __init__(self):
        pass
    def __getitem__(self, d):
        dow = d % WeekSize
        dayhours = HolidayMap.get(d, WorkWeek[dow])
        cumhours = DateToHours(d)
        return dayhours, cumhours, dow

class _next_month:
    def __init__(self):
        pass
    def __contains__(self, s):
        return ValidDate(s)
    def __getitem__(self, s):
        date = StringToDate(s)
        return AddMonths(date, 1)
    def has_key(self, key):
        return key in self

# DateConv = {}   # usage: index = DateConv['2004-01-01']
# DateIndex = []  # usage: date = DateIndex[1]
# DateInfo = []   # dayhours, cumhours, dow = DateInfo[1]
# DateNextMonth = {} # usage: index = DateNextMonth['2004-01']  # index is first day of next month

DateConv = _str_to_date()
DateIndex = _date_to_str()
DateInfo = _date_info()
DateNextMonth = _next_month()


# time period functions (e.g., day, week, or month scale)

def AddPeriod(date, type, size):
    if type == "Day":
        date += size
    elif type == "Week":
        date += size * WeekSize
    elif type == "Month":
        y, m, d = _date_to_ymd(date)
        m += size
        date = _coerce_ymd_to_date(y, m, d)
    elif type == "Quarter":
        y, m, d = _date_to_ymd(date)
        m += size * 3
        date = _coerce_ymd_to_date(y, m, d)
    elif type == "Year":
        y, m, d = _date_to_ymd(date)
        y += size
        date = _coerce_ymd_to_date(y, m, d)
    else:
        date += size
    return date

def PeriodStart(date, type, size):
    if type == "Week":
        date -= date % WeekSize
    elif type == "Month":
        y, m, d = _date_to_ymd(date)
        date -= d - 1
    elif type == "Quarter":
        y, m, d = _date_to_ymd(date)
        m -= (m - 1) % 3
        date = _ymd_to_date(y, m, 1)
    elif type == "Year":
        y, m, d = _date_to_ymd(date)
        y -= y % size
        date = _ymd_to_date(y, 1, 1)
    return date

def GetPeriodStart(period, date, offset):
    return GetPeriodInfo(period, date, offset, True)

def GetPeriodInfo(period, date, offset, index=False):  # needed by server
    """
    If index is true, return index of starting date; otherwise, return tuple:
        length in hours, index of starting hour, starting day of week
    """
    period = period.split('/')[0]
    if '*' in period:
        type, size = period.split('*', 1)
        size = int(size)
    else:
        type, size = period, 1

    date = PeriodStart(date, type, size)
    date = AddPeriod(date, type, size * offset)
    if index:
        return date

    date2 = AddPeriod(date, type, size)
    dow = date % WeekSize
    cumh = DateToHours(date)
    dh = DateToHours(date2) - cumh
    return dh, cumh, dow

########## @@@@@@@@@@ End Alex Date Conversion @@@@@@@@@@ ##########

# -----------------

def GanttCalculation(): # Gantt chart calculations - all dates are in hours
    # change = { 'Table': 'Task' }  # will be used to apply updates  --->> Don't Use 'Update' here <<--
    # Set the project start date (use specified date or today)  later may be adjusted by the tasks' start dates
    Today = GetToday()
    if debug: print "today", Today

    ps = {}  # project start dates indexed by project id
    pre = {}  # task prerequisites indexed by task id number
    suc = {}  # task successors
    precnt = {}  # count of all unprocessed prerequisites
    succnt = {}  # count of all unprocessed successors
    tpid = {}  # project id by task
    parents = {}  # child count by parent id
    ancestry = {}  # ancestors by task id

    for k, v in Task.iteritems():  # init dependency counts, xrefs, and start dates
        # if debug: print "task", k, "data", v
        if v.get('zzStatus') == 'deleted': continue
        pid = v.get('ProjectID')
        # if debug: "task", k, "project", pid
        if pid not in Project or Project[pid].get('zzStatus') == 'deleted':
            continue # silently ignore task w/ invalid or deleted project

        pre[k] = {}
        suc[k] = {}
        tpid[k] = pid
        tsd = v.get('StartDate')
        if tsd and (tsd in DateConv):
            if (pid not in ps) or (tsd < ps[pid]):
                # project start date defaults to earliest task
                ps[pid] = tsd

        p = v.get('TaskID')  # parent task id
        if p != k and p in Task and Task[p].get('zzStatus') != 'deleted':
            if p in parents:
                parents[p] += 1  # count the number of children
            else:
                parents[p] = 1  # add to list of parents

        anc = {}  # task ancestors
        while p in Task:
            if p == k or p in anc or Task[p].get('zzStatus') == 'deleted':
                break
            anc[p] = None
            p = Task[p].get('TaskID')
        ancestry[k] = anc

    ProjectStartHour = {}; ProjectEndHour = {}
    for k, v in Project.iteritems():
        if v.get('zzStatus') == 'deleted': continue
        sd = v.get('StartDate')
        # if debug: print "project", k, ", startdate", sd
        if not (sd and sd in DateConv):
            # default to earliest task if possible; otherwise, default to today
            sd = ps.get(k, Today)

        # convert project start dates to hours format
        si = DateConv[sd]  # get starting date index
        sh = DateInfo[si][1]  # get cum hours for start date
        ProjectStartHour[k] = sh
        if debug: "project, start hour", k, sh
        ProjectEndHour[k] = 0  # prepare to save project end hour

    # find dependencies
    for k, v in Dependency.iteritems():
        # if debug: print "dependency record", v
        if v.get('zzStatus') == 'deleted': continue
        p = v['PrerequisiteID']
        s = v['TaskID']
        if (p in suc) and (s in pre):  # ignore dependencies for missing tasks
            if (p in ancestry[s]) or (s in ancestry[p]):
                # ignore dependencies between parents and their children
                continue
            pre[s][p] = None
            suc[p][s] = None

    # copy dependencies from parent tasks to their children
    for child in pre:
        for parent in ancestry[child]:
            for p in pre[parent]:
                pre[child][p] = None
                suc[p][child] = None
    for child in suc:
        for parent in ancestry[child]:
            for s in suc[parent]:
                pre[s][child] = None
                suc[child][s] = None

    # clear parents; will calculate dates from children
    if debug: print "parent counts", parents
    for p in parents:
        for k in suc.pop(p):  # remove parents from pass calculations
            del pre[k][p]
        for k in pre.pop(p):
            del suc[k][p]

        task = Task[p]
        # if debug: print "clearing parent", p, task
        # update database -- doesn't use Update, but may in the future
        task['hES'] = None
        task['hEF'] = None
        task['CalculatedStartDate'], task['CalculatedStartHour'] = None, None
        task['CalculatedEndDate'], task['CalculatedEndHour'] = None, None

        task['hLS'] = None
        task['hLF'] = None
        task['FreeFloatHours'] = None
        task['TotalFloatHours'] = None
        # if debug: print "cleared parent", p, task

    # count prerequisites and successors
    for k, v in pre.iteritems():
        precnt[k] = len(v)
    for k, v in suc.iteritems():
        succnt[k] = len(v)

    # forward pass
    moretodo = True
    while moretodo:
        moretodo = False  # stop if we don't process at least one task per loop
        for k, v in precnt.iteritems():  # k is the task id
            if v == 0:  # all prereqs have been processed; set to -1 when done
                moretodo = True  # make another pass through the tasks

                # get task information
                # note: end date is not currently used to compute a start date
                tsd = Task[k].get('StartDate')
                tsdh = Task[k].get('StartHour') or 0
                ted = Task[k].get('EndDate')
                td  = Task[k].get('DurationHours')

                # calculate early start for task
                if tsd and DateConv.has_key(tsd):
                    tsi = DateConv[tsd]  # date index
                    tsh = DateInfo[tsi][1]  # date hour
                    tsh += tsdh  # adjust the starting hour of the day
                    es = tsh
                else:
                    es = ProjectStartHour[tpid[k]]

                # consider task dependencies
                for t in pre[k]:
                    ef = Task[t]["hEF"]
                    if ef > es: es = ef

                # calculate early finish
                if tsd and ted and not td and DateConv.has_key(ted):  # use difference in dates to compute duration
                    tei = DateConv[ted]  # date index
                    teh = DateInfo[tei+1][1]  # first hour of next day
                    ef = es + (tsh - teh)
                elif td:
                    ef = es + td
                else:
                    ef = es + int(HoursPerDay)

                if ef > ProjectEndHour[tpid[k]]: ProjectEndHour[tpid[k]] = ef  # keep track of project end date

                # update database -- doesn't use Update, but may in the future
                Task[k]['hES'] = es
                Task[k]['hEF'] = ef
                Task[k]['CalculatedStartDate'], Task[k]['CalculatedStartHour'] = HoursToDateString(es)
                Task[k]['CalculatedEndDate'], Task[k]['CalculatedEndHour'] = HoursToDateString(ef)

                # change['ID'] = k
                # change['hES'] = es
                # change['hEF'] = ef
                # change['CalculatedStartDate'], change['CalculatedStartHour'] = HoursToDateString(es)
                # change['CalculatedEndDate'], change['CalculatedEndHour'] = HoursToDateString(ef)
                # Update(change, 0)

                # tell successor that I'm ready
                for t in suc[k]: precnt[t] -= 1
                precnt[k] = -1

    vs = precnt.values()
    if vs.count(-1) != len(vs):
        if debug: print "forward pass did ", vs.count(-1), " of ", len(vs), ". Probably a dependency loop."
        return  # didn't finish forward pass, skip backward pass

    # backward pass
    moretodo = True
    while moretodo:
        moretodo = False  # stop if we don't process at least one task per loop
        for k, v in succnt.iteritems():  # k is the task id
            if v == 0:  # all successors have been processed; set to -1 when done
                moretodo = True  # make another pass through the tasks

                # calculate late finish and free float for task
                lf = ProjectEndHour[tpid[k]]
                ses = lf  # successor early start
                for t in suc[k]:
                    ls = Task[t]["hLS"]
                    if ls < lf: lf = ls
                    es = Task[t]["hES"]
                    if es < ses: ses = es
                ff = ses - Task[k]["hEF"]

                # calculate late start
                td  = Task[k]["hEF"] - Task[k]["hES"]
                ls = lf - td

                # update database -- doesn't use Update, but may in the future
                Task[k]['hLS'] = ls
                Task[k]['hLF'] = lf
                Task[k]['FreeFloatHours'] = ff
                Task[k]['TotalFloatHours'] = ls - Task[k]["hES"]

                # change['ID'] = k
                # change['hLS'] = ls
                # change['hLF'] = lf
                # ---------- remember to add float lines
                # Update(change, 0)

                # tell predecessor that I'm ready
                for t in pre[k]: succnt[t] -= 1
                succnt[k] = -1

    for k, v in Task.iteritems():  # derive parent dates from children
        # reminder: make sure this handles deleted & purged records properly
        if k in parents:
            Task[k]['SubtaskCount'] = parents[k]  # save count of child tasks
            continue  # skip parent tasks
        if Task[k].has_key('SubtaskCount'): del Task[k]['SubtaskCount']

        hes, hef, hls, hlf = [ v.get(x) for x in ['hES', 'hEF', 'hLS', 'hLF']]
        for p in ancestry.get(k, []):
            # if debug: print "adjusting parent ", p, Task[p]
            phes, phef, phls, phlf = [ Task[p].get(x) for x in ['hES', 'hEF', 'hLS', 'hLF']]
            if phes == None or hes < phes:
                Task[p]['hES'] = hes
                Task[p]['CalculatedStartDate'], Task[p]['CalculatedStartHour'] = HoursToDateString(hes)
            if phef == None or hef > phef:
                Task[p]['hEF'] = hef
                Task[p]['CalculatedEndDate'], Task[p]['CalculatedEndHour'] = HoursToDateString(hef)
            if phls == None or hls < phls: Task[p]['hLS'] = hls
            if phlf == None or hlf > phlf: Task[p]['hLF'] = hlf
            # if debug: print "adjusted parent ", p, Task[p]

# end of GanttCalculation
# -----------------


# helper functions for indirect-access columns

def ParseLinks(record, links):
    # return a dictionary of the form {id1: record1, ...}
    selection = {0: record}
    for link in links:
        size = link.count('-')
        if size == 0:
            getcol, tname, searchcol = link + 'ID', link, 'ID'
        elif size == 1:
            getcol = 'ID'
            tname, searchcol = link.split('-')
        else:
            getcol, tname, searchcol = link.split('-', 2)

        table = Database.get(tname) or {}
        selection, old = {}, selection
        for record in old.itervalues():
            target = record.get(getcol)
            search = SearchByColumn(table, {searchcol: target})
            for k, v in search.iteritems():
                if v.get('zzStatus') != 'deleted':
                    selection[k] = v
    return selection

def ParsePath(record, path):
    # return a list of values
    links = path.strip().split('/')
    column = links.pop()
    selection = ParseLinks(record, links)
    records = selection.values()
    result = [r.get(column) for r in records]
    return result

def ParseBranching(record, path):
    # return a triple-nested list
    # from outer to inner, the layers represent: records, branches, and values
    if ':' in path:
        trunk, path = path.split(':', 1)
        links = trunk.strip().split('/')
        records = ParseLinks(record, links).values()
    else:
        records = [record]
    paths = path.split(',')
    result = [[ParsePath(rec, p) for p in paths] for rec in records]
    return result


# helper functions for grid reports

def GetColumnDate(colid, of, fast=False, xdate=None):  # required by server
        """
    colid == ReportColumn ID; of == the offset; returns the index of first day of period
        """
        ctid = ReportColumn[colid].get('ColumnTypeID')  # get column type record that corresponds to this column
        if ctid not in ColumnType: return  # shouldn't happen
        ct = ColumnType[ctid]
        period = ct.get('PeriodSize') or ct.get('Name')
        if fast:
            type = period.split('/')[0].split('*')[0]
            if type in ('Day', 'Week'):
                of *= 7
            elif type in ('Month', 'Quarter'):
                of *= 12
            else:
                of *= 10

        firstdate = xdate or ReportColumn[colid].get('FirstDate')
        if firstdate not in DateConv:
            firstdate = GetToday()
        date = DateConv[firstdate]
        date = GetPeriodStart(period, date, of)
        return date

def GetColumnHeader(colid, of, xdate=None):
        """
    colid == ReportColumn ID; of == the offset; returns the first and second header line
        """
        if of == -1:
            label = ReportColumn[colid].get('Label')
            if not label:
                ctid = ReportColumn[colid].get('ColumnTypeID')  # get column type record that corresponds to this column
                if not ctid or not ColumnType.has_key(ctid): return ""  # shouldn't happen
                ct = ColumnType[ctid]
                label = ct.get('Label') or ct.get('Name')
                if label:
                    label = _(label)
        else:
            name1 = name2 = ''
            date = GetColumnDate(colid, of, xdate=xdate)
            if date == None: return
            year, month, day = _date_to_ymd(date)

            ctid = ReportColumn[colid].get('ColumnTypeID')  # get column type record that corresponds to this column
            if ctid not in ColumnType: return  # shouldn't happen
            ct = ColumnType[ctid]
            ctperiod, ctfield = ct['Name'].split('/', 1)
            period = ct.get('PeriodSize') or ctperiod
            if '*' in period:
                type, size = period.split('*', 1)
                size = int(size)
            else:
                type, size = period, 1
            if ctfield != 'Gantt':
                of -= 1  # make room for column name

            if type == 'Day':
                if of == 0 or day <= size:
                    name1 = MonthNames[month-1] + ' ' + str(year)
                name2 = str(day)
            elif type == 'Week':
                if of == 0 or day <= size * WeekSize:
                    name1 = ShortMonthNames[month-1] + ' ' + str(year)[-2:]
                name2 = str(day)
            elif type == 'Month':
                if of == 0 or month <= size:
                    name1 = str(year)[-2:]
                name2 = str(month)
            elif type == 'Quarter':
                if of == 0 or month <= size * 3:
                    name1 = str(year)[-2:]
                name2 = 'Q' + str((month + 2) // 3)
            elif type == 'Year':
                if of == 0 or (year % 100) < size:
                    name1 = str(year)[:-2]
                name2 = str(year)[-2:]

            if of < 0:
                name1 = ctfield[:5]  # ?? try column width mod 8 ??
            label = name1 + '\n' + name2
        return label

def GetCellValue(rowid, colid, of, xdate=None, source=False):
        if colid not in ReportColumn: return ""
        rc = ReportColumn[colid]
        ctid = rc.get('ColumnTypeID')
        if ctid not in ColumnType: return ""
        ct = ColumnType[ctid]
        if of == -1:
            rr = ReportRow[rowid]
            rtable = rr.get('TableName')
            tid = rr['TableID']  # was 'TaskID'

            t = ct.get('T') or 'X'
            reportid = rr.get('ReportID')
            if reportid not in Report: return ""
            rtid = Report[reportid].get('ReportTypeID')
            ctable = ReportType[rtid].get('Table' + t)

            dt = ct.get('DataType')
            at = ct.get('AccessType')
            if rtable != ctable:
                value = ''
            elif at in ('d', 'i', 'path'):
                record = Database[rtable][tid]
                path = ct.get('Path') or ct.get('Name')
                vals = ParsePath(record, path)
                if len(vals) == 1:
                    value = vals[0]
                else:
                    vals = [unicode(x) for x in vals if x != None]
                    value = ", ".join(vals)
            elif at == 'role':
                record = Database[rtable][tid]
                path = ct.get('Path') or ct.get('Name')
                vals = []
                for record in ParseBranching(record, path):
                    subvals = []
                    for branch in record:
                        branch = [str(x) for x in branch if x != None]
                        v = ", ".join(branch)
                        subvals.append(v)
                    if subvals:
                        v = subvals.pop(0) or "--"
                        subvals = [x for x in subvals if x]
                        if subvals:
                            v += " (" + "; ".join(subvals) + ")"
                        vals.append(v)
                value = ", ".join(vals)
            elif at == 'role2':
                # like role, but instead of displaying the secondary values,
                # the primary values are sorted on them
                record = Database[rtable][tid]
                path = ct.get('Path') or ct.get('Name')
                vals = []
                for record in ParseBranching(record, path):
                    subvals = []
                    for branch in record:
                        branch = [str(x) for x in branch if x != None]
                        v = ", ".join(branch)
                        subvals.append(v)
                    if subvals:
                        v = subvals.pop(0) or "--"
                        vals.append((subvals, v))
                vals.sort()
                vals = [v[1] for v in vals]
                value = ", ".join(vals)
            elif at == 'list':  # deprecated v0.7
                try:
                    listcol, listtable, listselect, listtarget, listtable2, listcol2 = ct.get('Path').split('/')  # path to values
                except ValueError:
                    if debug: print "List column needs valid path, has: ", ct.get('Path')
                    value = ""
                else:
                    listvalue = Database[rtable][tid].get(listcol)
                    table = Database.get(listtable) or {}
                    records = SearchByColumn(table, {listselect: listvalue})
                    vals = [(r.get(listtarget) or "-") for r in records.itervalues() if r.get('zzStatus') != 'deleted']
                    if listtable2 and listcol2:
                        table = Database.get(listtable2) or {}
                        records = [table.get(v) for v in vals]
                        vals = [r.get(listcol2) or "-" for r in records if r and r.get('zzStatus') != 'deleted']
                    value = ", ".join( [ unicode(x) for x in vals ] )

            if dt == 'u' and isinstance(value, int) and value > 0:
                w, h = divmod(value, HoursPerWeek)
                if h > int(HoursPerWeek): w += 1; h = 0

                if AllowDaysUnit:
                    d, h = divmod(h, HoursPerDay)
                    if h > int(HoursPerDay): d += 1; h = 0
                else:
                    d = 0

                if h % 1: h += 1

                value = []
                if w: value.append(str(int(w)) + 'w')
                if d: value.append(str(int(d)) + 'd')
                if h: value.append(str(int(h)) + 'h')
                return ' '.join(value)
            if value == None: value = ''
            if source:
                return value, {} # {'Table': timename, 'ID': timeid, 'Column': fieldname}
            else:
                return value
        else:
            # ---- Here are some examples that this should handle ----
            # -- Report Type => Column Type --
            # TaskDay => Day/Gantt, Day/Hours
            # ResourceDay => Day/Hours
            # ProjectDay or ProjectWeek => Day/Measurement, Week/Measurement
            # TaskWeek => Week/PercentComplete, Week/Effort
            # ResourceWeek => Week/Effort

            ctperiod, ctfield = ct['Name'].split("/", 1)
            type = ct.get('DataType', 't')  # only needed if source=True

            if ctfield == "Gantt":  # don't display a value
                value = ''
            else:  # table name, field name, time period, and record id
                rr = ReportRow[rowid]
                tablename = rr.get('TableName')
                tid = rr['TableID']
                if ctfield == 'Measurement':  # find field name
                    mid = Database[tablename][tid].get('MeasurementID')  # point at measurement record
                    if mid:
                        fieldname = Database['Measurement'][mid].get('Name')  # measurement name == field name
                        type = Database['Measurement'][mid].get('DataType')  # override column type w/ measurement type
                        # type only needed for if source=True
                    else:
                        fieldname = None
                    if tablename.endswith("Measurement"):
                        newtablename = tablename[:-11]  # trim of "measurement" -- currently yields Project or Resource
                    else:
                        newtablename = 'Project'  # prior default -- this shouldn't happen
                    tid = Database[tablename][tid].get(newtablename + 'ID')  # point at measurement record
                    tablename = newtablename
                else:
                    fieldname = ctfield

                # find the period date
                timename = tablename + ctperiod
                date = DateIndex[GetColumnDate(colid, of, xdate=xdate)]  # should not use date index here
                timeid = FindID(timename, tablename + "ID", tid, 'Period', date)
                # if debug: print "timeid", timeid
                if timeid:
                    value = Database[timename][timeid].get(fieldname)
                    # if debug: print "timename, timeid, fieldname, value: ", timename, timeid, fieldname, value
                    # if debug: print "record: ", Database[timename][timeid]
                else:
                    value = None
                    # if debug: print "didn't find timeid", timeid, value

            if value == None: value = ''
            if source:
                searchdict = {tablename + "ID": tid, 'Period': date}
                return value, { 'Table': timename, 'ID': timeid, 'SearchKeys': searchdict, 'Column': fieldname, 'Type': type }
            else:
                return value

def SetCellValue(rowid, colid, of, value, xdate=None, raw=True):  # need a better name than raw
    """ Update of database using same report row and column ids
        returns true if successful
        """
    rr = ReportRow[rowid]
    rtable = rr.get('TableName')
    tid = rr['TableID']

    rc = ReportColumn[colid]
    ctid = rc.get('ColumnTypeID')
    ct = ColumnType[ctid]

    # is all protection set by the column type?
    if not ct.get('Edit'):  # don't apply changes to protected cells
        return 0

    type = ct.get('DataType', 't')

    if of != -1:
        ctperiod, ctfield = ct.get('Name').split("/")
        if ctfield == "Gantt":  # shouldn't happen
            return 0
        else:  # table name, field name, time period, and record id
            if debug: print 'ctfield', ctfield
            if ctfield == 'Measurement':  # find field name
                mid = Database[rtable][tid].get('MeasurementID')  # point at measurement record
                if mid: # measurement id
                    fieldname = Database['Measurement'][mid].get('Name')  # measurement name == field name
                    type = Database['Measurement'][mid].get('DataType')  # override column type w/ measurement type
                else:
                    fieldname = None
                if rtable.endswith("Measurement"):
                    newrtable = rtable[:-11]  # trim of "measurement" -- currently yields Project or Resource
                else:
                    newrtable = 'Project'  # prior default -- this shouldn't happen
                tid = Database[rtable][tid].get(newrtable + 'ID')  # point at measurement record
                rtable = newrtable
            else:
                fieldname = ctfield
        if debug: print 'offset, rtable, fieldname', of, rtable, fieldname
        if fieldname == None: return 0

        # find the period date
        timename = rtable + ctperiod
        date = GetColumnDate(colid, of, xdate=xdate)
        if date == None: return 0
        date = DateIndex[date]  # should use date not date index
        timeid = FindID(timename, rtable + "ID", tid, 'Period', date)

    # add processing of other date formats (per user preferences)
    # also add processing of dd, mm-dd, and yy-mm-dd for standard format
    if raw:
        if value == "": v = None
        elif type == 'i': 
            try:
                v = int(value)
            except ValueError:  # should I display an error message?
                return 0
        elif type == 'f': 
            try:
                v = float(value)
            except ValueError:  # should I display an error message?
                return 0
        elif type == 'd':
            v = CheckDateString(value)
            if not v: return 0
        elif type == 'u': 
            result = re.match(r"^\s*(\d+w)?\s*(\d+d)?\s*(\d+h)?\s*$|^\s*(\d+)\s*$", value, re.I)
            if result:
                groups = result.groups() # [weeks, days, hours, hours]
                v = 0
                if groups[0]: v += int(groups[0][:-1]) * HoursPerWeek
                if groups[1]: v += int(groups[1][:-1]) * HoursPerDay
                if groups[2]: v += int(groups[2][:-1])
                if groups[3]: v += int(groups[3])
                if v % 1: v += 1
                v = int(v)
                if v <= 0: return 0
            else: 
                 return 0
        else: v = value

    change = {}

    at = ct.get('AccessType')
    if at == 'd':
        column = ct.get('Name')
        table = rtable
        id = tid
        if not id: return 0  # don't make update if ID is invalid
        change['ID'] = id
    elif at == 'i':
        table, column = ct.get('Name').split('/')  # indirect table & column
        # id = self.data[rtable][tid].get(table+'ID')
        id = Database[rtable][tid].get(table+'ID')
        if not id: return 0  # don't make update if ID is invalid
        change['ID'] = id
    elif at == 's':
        table = timename
        column = fieldname
        if timeid: # don't add record if already exists
            change['ID'] = timeid
        else:
            change[rtable + 'ID'] = tid
            change['Period'] = date
    else:
        return 0  # we don't recognize how to find the record to update, so ignore it

    change['Table'] = table
    change[column] = v
    Update(change)
    label = rc.get('Label') or ct.get('Label') or ct.get('Name')
    label = label.replace('\n', ' ')
    return label  # for undo message
    # SetUndo(column)

# -----------------
# this is overriden by ReportAids.py
def UpdateDataPointers(self, reportid):
    pass


########## @@@@@@@@@@ Start Alex Row/Column Order @@@@@@@@@@ ##########

def GetColumnList(reportid):
    """ Return list of column id's in the current order """
    ids = []
    done = {}
    report = Report.get(reportid) or {}
    k = report.get('FirstColumn')
    while k in ReportColumn and k not in done:
        column = ReportColumn.get(k, {})
        if column.get('zzStatus') != 'deleted':
            ids.append(k)
        done[k] = None
        k = column.get('NextColumn')
    return ids

def GetColumnPointers(reportid):
    columns = []
    types = []
    offsets = []

    for colid in GetColumnList(reportid):
        rc = ReportColumn.get(colid) or {}
        ctid = rc.get('ColumnTypeID')
        ct = ColumnType.get(ctid) or {}
        at = ct.get('AccessType')
        if at == 's':
            # column is a time scale
            for of in xrange(rc.get('Periods') or 1):
                columns.append( colid )
                types.append( ctid )
                offsets.append( of )
        else:
            columns.append( colid )
            types.append( ctid )
            offsets.append( -1 )

    return columns, types, offsets

def ReorderReportColumns(reportid, columnids):
    """ Use the list of column ids as the columns sequence of the report

    Ignores duplicates.  Calling routine must call 'SetUndo'.

    """
    if debug: print "Start ReorderReportColumns"

    # remove duplicates
    stack = []
    done = {}
    for id in columnids:
        if (id not in done) and (id in ReportColumn):
            stack.append(id)
            done[id] = None
    stack.reverse()

    # apply new order
    next = None
    for id in stack:
        record = ReportColumn[id]
        if record.get('NextColumn') != next:
            change = {'Table': 'ReportColumn', 'ID': id, 'NextColumn': next}
            Update(change)
        next = id
    if Report[reportid].get('FirstColumn') != next:
        change = {'Table': 'Report', 'ID': reportid, 'FirstColumn': next}
        Update(change)

    if debug: print "End ReorderReportColumns"

def GetRowList(reportid):
    """ Returns a list of row ids in the current order """
    ids = []
    done = {}
    report = Report.get(reportid) or {}
    k = report.get('FirstRow')
    while k in ReportRow and k not in done:
        row = ReportRow.get(k, {})
        if row.get('zzStatus') != 'deleted':
            ids.append(k)
        done[k] = None
        k = row.get('NextRow')
    return ids

def IsLinkActive(table, id):
    """ Return false if the assignment or dependency uses a deleted record """
    if table == 'Assignment':
        ftables = ['Task', 'Resource']
    elif table == 'Dependency':
        ftables = ['Task', 'Prerequisite']
    else:
        return True
    try:
        for ftable in ftables:
            linkID = Database[table][id].get(ftable + 'ID')
            if Database[ftable][linkID].get('zzStatus') == 'deleted':
                return False
    except KeyError:
        return False
    return True

def GetRowLevels(reportid, showHidden=True):
    """ Return two lists: row ids and row levels (ie - ancestor counts) """
    rlist = GetRowList(reportid)
    rlevels = []

    stack = []
    for rid in rlist:
        rr = ReportRow.get(rid) or {}
        parent = rr.get('ParentRow')
        while stack:
            if stack[-1] == parent:
                break
            stack.pop()
        rlevels.append(len(stack))
        stack.append(rid)

    if showHidden:
        return rlist, rlevels

    # remove hidden and deleted rows from list
    rows = []
    rowlevels = []
    level = 0  # maximum level of next row

    for rowid, lev in zip(rlist, rlevels):
        if lev > level:
            # parent is not shown
            continue
        level = lev

        r = ReportRow.get(rowid) or {}
        hidden = r.get('Hidden')
        try:
            t = r['TableName']
            id = r['TableID']
            deleted = (Database[t][id].get('zzStatus') == 'deleted')
            deleted = deleted or not IsLinkActive(t, id)
        except KeyError:
            deleted = True

        if not (hidden or deleted):
            rows.append(rowid)
            rowlevels.append(lev)
            level += 1

    return rows, rowlevels

# this is overriden by ReportAids.py
def UpdateRowPointers(self):
    pass

def ReorderReportRows(reportid, rowids):
    """ Use the list of rowids as the row sequence of the report

    Ignore duplicates.  Keep children with parents.
    Calling routine must call 'SetUndo'.

    """
    if debug: print "Start ReorderReportRows"

    # remove duplicates
    queue = []
    done = {}
    for id in rowids:
        if (id not in done) and (id in ReportRow):
            queue.append(id)
            done[id] = None

    # examine hierarchy
    godfamily = {}
    family = {}  # {parent: [child, ...]}
        # godfamily is cross-table; family is same-table

    stack = []
    for id in queue:
        record = ReportRow[id]
        parent = record.get('ParentRow')
        if parent in ReportRow:
            if parent not in family:
                godfamily[parent] = []
                family[parent] = []
            prec = ReportRow[parent]
            if prec.get('TableName') == record.get('TableName'):
                family[parent].append(id)
            else:
                godfamily[parent].append(id)
        else:
            stack.append(id)

    # apply new order
    next = None
    while stack:
        id = stack[-1]
        if id in family:
            stack += godfamily.pop(id) + family.pop(id)
        else:
            stack.pop()
            if ReportRow[id].get('NextRow') != next:
                change = {'Table': 'ReportRow', 'ID': id, 'NextRow': next}
                Update(change)
            next = id
    if Report[reportid].get('FirstRow') != next:
        change = {'Table': 'Report', 'ID': reportid, 'FirstRow': next}
        Update(change)

    if debug: print "End ReorderReportRows"

def AdjustReportRows():
    """ Ensure that every report has the correct rows and hierarchy

    The primary steps:
    - build a list of all of the records that should appear in the report
    - scan the current rows
        - remove every row whose record isn't in the Should list
        - remember the ids for every row whose record is in the Should list
    - create a row for every record in the Should list that lacks a row
    - remove parenting loops
    - give each a row a reference to its parent row
    - call ReorderReportRows to link the rows

    """
    if debug: print "Start AdjustReportRows"
    newrow = {'Table': 'ReportRow'}
    oldrow = {'Table': 'ReportRow'}

    # process all non-deleted reports
    for rk, r in Report.iteritems():
        if r.get('zzStatus') == 'deleted' or r.get('AdjustRowOption'): continue
        newrow['ReportID'] = rk

        rtid = r.get('ReportTypeID')
        if rtid not in ReportType:
            if debug: print "invalid ReportType id: report", rk, ", type", rtid
            continue
        rt = ReportType[rtid]
        if rt.get('AdjustRowOption'): continue

        ta, tb = rt.get('TableA'), rt.get('TableB')
        tableA = Database.get(ta)
        if not tableA: continue

        if tb == ta:
            tableB == None
        else:
            tableB = Database.get(tb)

        # search for records that should be present
        should = {}  # {table name: {record id: row id}}

        map = {}
        selcol = r.get('SelectColumn')
        if selcol:
            selval = r.get('SelectValue')
            for id, record in tableA.iteritems():
                if record.get(selcol) == selval:
                    map[id] = None
        else:
            for id, record in tableA.iteritems():
                map[id] = None
        should[ta] = map

        if tableB:
            map = {}
            for id, record in tableB.iteritems():
                parent = record.get(ta + 'ID')
                if parent in should[ta]:
                    map[id] = None
            should[tb] = map

        # correlate records with existing rows
        rlist = []
        for rowid in GetRowList(rk):
            r = ReportRow[rowid]
            t = r.get('TableName')
            tid = r.get('TableID')
            if t in should and tid in should[t] and not should[t][tid]:
                should[t][tid] = rowid
                rlist.append(rowid)

        # create a row for every record that needs one;
        #   map parent records to child rows
        parents = {}  # {row id: (parent table, parent id)}
        godparents = {}
            # here, parents are same-table; godparents are cross-table

        for t, map in should.iteritems():
            newrow['TableName'] = t
            for tid, rowid in map.items():
                if not rowid:
                    newrow['TableID'] = tid
                    rowid = Update(newrow)['ID']
                    map[tid] = rowid
                    rlist.append(rowid)

                record = Database[t][tid]
                parent = record.get(t + 'ID')
                prec = Database[t].get(parent)

                if prec and prec.get('zzStatus') != 'deleted':
                    if t == ta or record.get(ta + 'ID') == prec.get(ta + 'ID'):
                        parents[rowid] = t, parent
                if t != ta:
                    godparents[rowid] = ta, record.get(ta + 'ID')

        # map parent rows to child rows
        parentrows = {}  # {child row: parent row}

        for rowid in rlist:
            lineage = {}
            while rowid in parents:
                t, tid = parents.pop(rowid)
                parent = should[t].get(tid)
                lineage[rowid] = parent
                if parent in lineage:
                    del lineage[parent]
                rowid = parent
            parentrows.update(lineage)

        # give each a row a reference to its parent row
        for rowid in rlist:
            parent = parentrows.get(rowid)
            if (not parent) and (rowid in godparents):
                t, tid = godparents[rowid]
                parent = should[t].get(tid)
            if ReportRow[rowid].get('ParentRow') != parent:
                oldrow['ID'] = rowid
                oldrow['ParentRow'] = parent
                Update(oldrow)

        # link the new rowlist
        ReorderReportRows(rk, rlist)

    if debug: print "End AdjustReportRows"

########## @@@@@@@@@@ End Alex Row/Column Order @@@@@@@@@@ ##########

# --------- Routines to load and save data ---------
OptionFile = None

bar_colors = {
    'ActualBar': (238, 238, 102),
    'ActualBarSelected': (238, 238, 102),
    'BaseBar': (0, 0, 187),
    'BaseBarSelected': (0, 0, 187),
    'CompletionBar': (0, 0, 0),
    'CompletionBarSelected': (0, 0, 0),
}

# overriden by ReportAids.py
def LoadOption(directory=None):
    """ Load the option file. """
    global OptionFile, Option
    if not directory:
        directory = Path

    OptionFile = os.path.join(directory, "Options.ganttpvo")
    if debug: print "load option file:", OptionFile
    try:
        f = open(OptionFile, "rb")
    except IOError:
        if debug: print "option file not found"
    else:
        header = f.readline()  # header will identify need for conversion of earlier versions or use of different file formats
        Option = cPickle.load(f)
        if not Option.get("CompletionBar"):
            Option.update(bar_colors)  # ad hoc fix to old, ugly bar colors
        f.close()
#        Menu.GetScriptNames()  # the ReportAids version keeps this line

def SaveOption():
    """ Save the Option file. """
    if debug: print "saving option file", OptionFile
    try:
        f = open(OptionFile, "wb")
    except IOError:
        if debug: print "SaveOption io error"
    else:
        f.write("GanttPV\t0.1\tO\n")
        cPickle.dump(Option, f)
        f.close()

# --------- Project files

OpenReportEvent = Event.Event(msg='Open Report')
CloseReportEvent = Event.Event(msg='Close Report')

# overriden by ReportAids.py
def OpenReport(id):
    OpenReportEvent(id)

# overriden by ReportAids.py
def CloseReport(id):
    CloseReportEvent(id)

# overriden by ReportAids.py
def CloseReports():
    pass

# overriden by ReportAids.py
def SetActiveReport(id):
    ActiveReport = id

def UpdateAliases():
    """ Refresh the list of table aliases """
    for alias, aTable in Database.iteritems():
        for tname in GetTableNames():
            if Database.get(tname) is aTable and (alias != tname):
                AddAlias(alias, tname)
                break

def PrepDatabase():
    """ After an database has been loaded or created, set all other values to match """
    global ChangedData, ChangedCalendar, ChangedSchedule, ChangedReport, UndoStack, RedoStack
    global Project, Task, Dependency, Report, ReportColumn, ReportRow
    global Resource, Assignment, Holiday, ReportType, ColumnType, OtherData, Other, NextID
    global DBObject
    
    Project =       Database['Project']
    Task =          Database['Task']
    Dependency =    Database['Dependency']
    Report =        Database['Report']
    ReportColumn =  Database['ReportColumn']
    ReportRow =     Database['ReportRow']
    Resource =      Database['Resource']
    Assignment =    Database['Assignment']
    Holiday =       Database['Holiday']
    ReportType =    Database['ReportType']
    ColumnType =    Database['ColumnType']
    NextID =        Database['NextID']

    Database['Other'] = OtherData = Database['OtherData']
        # OtherData and Database['OtherData'] are deprecated v0.6
    if 'Other' not in NextID:
        NextID['Other'] = 2
    Other = OtherData[1]
    if not Other.get('FileSignature'):
        Other['FileSignature'] = random.randint(1, 1000000000)

    if 'TableAlias' not in Database:
        AddTable('TableAlias')
        Update({'Table': 'TableAlias'})  # create the first record
    UpdateAliases()

    SetupDateConv()
    GanttCalculation()
    AdjustReportRows()
    DBObject = DB(Database)  # will get objects from this database

    UndoStack = []
    RedoStack = []
    ChangedData = False  # true if database needs to be saved
    ChangedCalendar = ChangedSchedule = ChangedReport = ChangedRow = False

# overriden by ReportAids.py
def MakeReady():
    pass

# overriden by ReportAids.py
def CreateReports():
    pass

def LoadContents(path=None):
    """ Load the contents of our document into memory. """
    global Database
    try:
        f = open(path or FileName, "rb")
        header = f.readline()  # add read line of text - will allow conversion of earlier versions or use of different file formats
        db = cPickle.load(f)
        f.close()
    except IOError:
        if debug: print "LoadContents io error"
    else:
        CloseReports()
        Database = db
        MakeReady()
        return True

def SaveContents(path=None):
    """ Save the contents of our document to disk. """
    global ChangedData
    try:
        f = open(path or FileName, "wb")
        f.write("GanttPV\t0.1\ta\n")
        cPickle.dump(Database, f)
        f.close()
    except IOError:
        if debug: print "SaveContents io error"
    else:
        ChangedData = False
        return True

def GetScriptDirectory():
    """ Return the directory to search for script files """
    return Option.get('ScriptDirectory') or os.path.join(Path, "Scripts")

def OpenFile(path):
    """ Open a file (any type) """
    ext = os.path.splitext(path)[1]
    if ext == '.ganttpv':
        OpenDatabase(path)
    elif ext == '.py':
        RunScript(path)
    else:
        if debug: print "unknown file extension:", ext

def OpenDatabase(path):
    """ Open a database file (.ganttpv) """
    global FileName
    if not LoadContents(path): return
    FileName = path
    RefreshReports()

def RunScript(path):
    """ Run a script file (.py) """
    if debug: print "begin script:", path

    name_dict = GetModuleNames()
    name_dict['self'] = OpenReports.get(ActiveReport)
    name_dict['thisfile'] = path
    scriptname = os.path.basename(path)
    scriptname = os.path.splitext(scriptname)[0]  # remove extension
    name_dict['scriptname'] = scriptname
    name_dict['debug'] = 1
    name_dict['_'] = _

    try:
        execfile(path, name_dict)
    except:
        error_info = sys.exc_info()
        sys.excepthook(*error_info)

    if debug: print "end of script:", path

# this is overriden by ReportAids.py (for now)
def AskIfUserWantsToSave(action):
    pass

# overriden by ReportAids.py
def Hint(message):
    if debug: print message

# ------- setup data for testing

# SetEmptyData()
# MakeReady()

if debug: print "end Data.py"
