#!/usr/bin/env python
# gantt report definition

# Copyright 2004, 2005, 2006, 2007 by Brian C. Christensen

#    This file is part of GanttPV.
#
#    GanttPV is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    GanttPV is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with GanttPV; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# 040408 - first version of this program
# 040409 - will display task values in report
# 040410 - SetValue will now set Task fields
# 040419 - changes for new ReportRow column names
# 040420 - changes to allow Main.py to open gantt reports; doClose will close this report; saves Report size and position
# 040424 - added doInsertTask, doDuplicateTask, doDeleteTask
# 040426 - added doMoveRow, added doPrerequisite
# 040427 - added doAssignResource, onHide, onShowHidden; fixed doDelete; save the result when the user changes columns widths; prevent changes to row height; changed some column names in ReportColumn
# 040503 - changes to use new ReportType and ColumnType tables; added OnInsertColumn; added row highlighting for two table reports
# 040504 - finished OnInsertColumn; added OnDeleteColumn, and OnMoveColumn
# 040505 - support UI changes (many references to 'Task' replaced with 'Row', format changes, and minor adjustments); revised OnInsertRow, OnDuplicateRow, and OnDeleteRow to work with all tables; added OnScroll and OnScrollToTask; use colors from Data.Option for chart
# 040508 - prevent deletion of project 1 or reports 1 or 2
# 040510 - set minimum row size for non-gantt column in "_updateColAttrs"; set Hidden and Deleted row color; copied Scripts menu processing from Main.py; changed doClose to use Data.doClose
# 040512 - added OnEditorShown; added support for indirect column types; several bug fixes
# 040518 - added doHome
# 040528 - in doDuplicate only dup rows of primary table's type; in onPrerequisite only include Tasks in the list of potential prerequisites
# 040531 - in onDraw make sure all rectangles use new syntax for version 2.5
# 040715 - Pierre_Rouleau@impathnetworks.com: removed all tabs, now use 4-space indentation level to comply with Official Python Guideline.
# 040906 - changed OnInsertColumn to ignore Labels w/ value of None; display "project name / report name" in report title.
# 040914 - handle indirect display if no ID is found; assign project id when inserting rows into reports that can be each.
# 040915 - allow entry of floating point numbers (type "f")
# 040918 - add week time scale; default FirstDate in column type to today or this week
# 040928 - Alexander - ignores dates not present in Data.DateConv; prevents entry of incorrectly formatted dates
# 041001 - display measurements in weekly time scale
# 041009 - Alexander & Brian - default dates to current year and month; Brian - change SetValue to work on measurement time scale data
# 041009 - change scroll to work w/ any time scale column
# 041010 - change "column insert" to set # periods for all timescale columns; changes to allow edit of non-measurement   time scale columns
# 041126 - draw bars for week timescale
# 041203 - moved get column header logic to Data
# 041204 - default time scale first date to today
# 050101 - previously added month  & quarter timescale
# 050101 - added entry of duration units datatype
# 050105 - fix part of bug: avoid report reset unless # rows or # columns changes (report un-scrolls when reset this was a problem when entering start dates and durations); problem still exists when adding a column or a row, but at least it will be less annoying.
# 050106 - fixed bug where deleted records threw off location of inserted rows
# 050202 - remove line feed from insert column menu text
# 050423 - move logic to calculate period start and hours to Data.GetPeriodInfo
# 050424 - fix ScrollToTask for M & Q
# 050426 - Alexander - fixed bug where moving columns threw off displayed widths
# 050503 - make wxPython 2.6.0.0 use same fonts as prior default
# 050504 - in OnPrerequisite, chain tasks if more than one non-parent task is selected
# 050504 - Alexander - implemented Window menu; moved some menu event-handling logic to Menu.py
# 050505 - Alexander - updated time units feature to use the work week
# 050513 - Alexander - replaced UpdateColumnWidths with new UpdateColumns; this fixes the bug where moving columns threw off cell renderers and read-only status
# 050517 - Alexander - fixed bug where row size was not properly adjusted for the presence of a gantt column
# 050520 - Alexander - renamed UpdateColumns into UpdateAttrs, and added a call to _updateRowAttrs; this fixes the bug where moving rows threw off row colors.
# 050527 - Alexander - added IndentedRenderer
# 050630 - Brian - use PlanBarColor to override default bar color
# 050826 - Alexander - rewrote row/column movement!
# 050903 - Brian - prevent negative window positions (fix problem caused when Windows iconizes reports)
# 051207 - Brian - changed GetSelectedRows to work around error in wx that treats shift-click rows as cells not rows
# 060131 - Alexander - store gantt-chart backgrounds in the column attributes (allows print script to override those colors)
# 060211 - Alexander - when inserting before a subtask, give the same parent to the new row
# 060314 - Alex - allow adding multiple reports at a time
# 060316 - Alex - updated time period access and scrolling
# 060404 - Brian - move SetValue logic into Data so GanttPV Server can use it
# 060422 - Alex - override drawing of column headers to produce more attractive time-scales
# 060506 - Alex - extended "Assign Resources" button to work in reverse (ie, assign tasks to a selected resource)
# 060606 - Alex - changed GetSelectedCols to work around wx shift-click bug; changed inserts to insert after the selection and to default selection to the grid cursor
# 060704 - Brian - connect new menu items for 0.7
# 060723 - Brian - connect Assign Task; translate column type labels in insert column
# 060729 - Brian - add expansion of resource work groups
# 060828 - Blake - change to make timescale header's visible in Linux
# 061020 - Alex - fixed bug in OnPrerequisite that made duplicate records
# 061125 - Brian - fix broken row select logic in wx.grid
# 061127 - Brian - make OnScrollToTask work for assignments (for resource assignment report)
# 061206 - Brian - move current cell to inserted row and scroll display to show it
# 070126 - Brian - add shift left and right to menu (manage parenting)
# 070323 - Brian - prevent windows from shifting down when reopened
# 070326 - Brian - fix bug reported by Blake, finish cell edit before inserting, deleting, or moving rows or columns
# 080221 - Alex - allow shifting of multiple rows
# 080305 - Alexander - fix bug that prevents scroll bars from appearing
# 080513 - Alexander - add first names to resource assignment dialog, changed the default name for new rows
# 080516 - Alexander - added search box to assignment dialog
# 080516 - Brian - added actual and baseline plan bars
# 080517 - Alex - sorted the insert column list
# 080605 - Alex - grid gets its default sizes from Data.Option

import wx, wx.grid
import datetime
from wx.lib.dialogs import MultipleChoiceDialog as wxMultipleChoiceDialog
import Data, UI, ID, Menu, ReportAids
# import images
import re
import os

debug = Data.debug
is24 = 1

if debug: print "load GanttReport.py"


# ------------ drawing helper functions ---------

def ClipText(dc, text, size):
    """ Returns a portion of text no larger than size. """
    if '\n' in text:
        result = []
        y = 0
        for line in text.splitlines():
            y += dc.GetTextExtent(line)[1]
            if y > size.height:
                break
            line = ClipText(dc, line, size)
            result.append(line)
        return '\n'.join(result)

    width, height = dc.GetTextExtent(text)
    if height > size.height:
        return ""
    if width > size.width:
        max_width = size.width - dc.GetTextExtent(" ...")[0]
        extents = dc.GetPartialTextExtents(text)
        for index, x in enumerate(extents):
            if x > max_width:
                return text[:index] + " ..."

    return text

def DrawClippedText(dc, text, rect):
    size = wx.Size(rect.width - 2, rect.height - 2)
    text = ClipText(dc, text, size)
    dc.DrawLabel(text, rect, wx.ALIGN_CENTER, -1)


# ------------ Table behind grid ---------

class GanttChartTable(wx.grid.PyGridTableBase):
    """ A custom wxGrid Table using user supplied data """
    def __init__(self, reportid):
        """ data is taken from SampleData """
        # The base class must be initialized *first*
        wx.grid.PyGridTableBase.__init__(self)

        # create local pointers to SampleData?
        self.UpdateDataPointers(reportid)
        self.UpdateColumnPointers()
        self.UpdateRowPointers()

        # self.colnames = _getColNames()

        # store the row length and col length to test if the table has changed size
        self._rows = self.GetNumberRows()
        self._cols = self.GetNumberCols()

    def GetNumberCols(self):
        return len(self.columns)

    def GetNumberRows(self):
        return len(self.rows)

    def GetColLabelValue(self, col):
        return Data.GetColumnHeader(self.columns[col], self.coloffset[col])

    def GetValue(self, row, col):
        return Data.GetCellValue(self.rows[row], self.columns[col], self.coloffset[col])

    def GetRawValue(self, row, col):  # same as GetValue  ( I don't know the difference. The example I'm following made them the same. )

        # Alexander - In the wxPython demo, GetValue converted to a string,
        #   while GetRawValue did not. But GetRawValue is not a standard
        #   function of the base class; we don't need to add it.

        value = GetValue(self, row, col)
        return value

    def SetValue(self, row, col, value):
        rowid = self.rows[row]
        of = self.coloffset[col]
        colid = self.columns[col]
        column = Data.SetCellValue(rowid, colid, of, value)
        if column:  # name of column updated (for undo message)
            Data.SetUndo(column)

    def ResetView(self, grid):  # deprecated
        grid.Reset()

    def UpdateValues(self, grid):  # deprecated
        # This sends an event to the grid table to update all of the values
        msg = wx.grid.GridTableMessage(self, wx.grid.GRIDTABLE_REQUEST_VIEW_GET_VALUES)
        grid.ProcessTableMessage(msg)

    # -- These are not part of the sample interface - my routines to make display easier

    def UpdateDataPointers(self, reportid):
        Data.UpdateDataPointers(self, reportid)

        # self.data = Data.Database
        # self.report = self.data["Report"][reportid]
        # self.reportcolumn = self.data["ReportColumn"]
        # self.reportrow = self.data["ReportRow"]

    def UpdateColumnPointers(self):
        reportid = self.report.get('ID')
        pointers = Data.GetColumnPointers(reportid)
        self.columns, self.ctypes, self.coloffset = pointers

    def UpdateRowPointers(self):
        Data.UpdateRowPointers(self)

    def _updateColAttrs(self, grid):
        """
        wxGrid -> update the column attributes to add the
        appropriate renderer given the column type.

        Otherwise default to the default renderer.
        """
        rsize = Data.Option.get("GridRowSize", 24)
        for col in xrange(len(self.columns)):
            attr = wx.grid.GridCellAttr()
            cid = self.columns[col]
            rc = Data.ReportColumn.get(cid) or {}
            ctid = self.ctypes[col]
            ct = self.columntype.get(ctid) or {}
            ctname = ct.get('Name')
            of = self.coloffset[col]
            if of > -1:
                # time scale column, so color-code work-days and holidays
                cid = self.columns[col]
                rc = Data.ReportColumn.get(cid) or {}
                fdate = rc.get('FirstDate')
                if fdate and Data.ValidDate(fdate):
                    ix = Data.StringToDate(fdate)
                else:
                    ix = Data.TodayDate()
                period = ct.get('PeriodSize') or ctname
                if Data.GetPeriodInfo(period, ix, of)[0] > 0:
                    bgcolor = Data.Option.get('WorkDay', wx.BLUE)
                else:
                    bgcolor = Data.Option.get('NotWorkDay', wx.BLUE)
                attr.SetBackgroundColour(bgcolor)
            if of > -1 and ctname and ctname[-6:] == "/Gantt":
                # gantt column, so use gantt renderer
                renderer = GanttCellRenderer(self)
                grid.SetColSize(col, renderer.colSize)
                if rsize < renderer.rowSize:
                    rsize = renderer.rowSize
                attr.SetReadOnly(True)
                attr.SetRenderer(renderer)
            else:
                # not a gantt column
                if ctname == 'Name':
                    sub_renderer = grid.GetDefaultRenderer()
                    renderer = IndentedRenderer(self, sub_renderer)
                    attr.SetRenderer(renderer)
                csize = (rc.get('Width') or ct.get('Width') or
                    grid.GetDefaultColSize())
                grid.SetColSize(col, csize)
                readonly = not ct.get('Edit')
                attr.SetReadOnly(readonly)
            grid.SetColAttr(col, attr)
        if rsize != grid.GetDefaultRowSize():
            grid.SetDefaultRowSize(rsize, True)

    def _updateRowAttrs(self, grid):
        """ Highlight parent rows in split-table reports """
        rtid = self.report['ReportTypeID']
        rt = Data.Database['ReportType'].get(rtid)
        ta, tb = rt.get('TableA'), rt.get('TableB')

        for row, rowid in enumerate(self.rows):
            rr = self.reportrow.get(rowid) or {}
            t = rr.get('TableName')
            tid = rr.get('TableID')
            try:
                deleted = (Data.Database[t][tid]['zzStatus'] == 'deleted')
            except KeyError:
                deleted = False
            hidden = rr.get('Hidden')

            attr = wx.grid.GridCellAttr()
            if Data.platform == "mac":
                attr.SetFont(wx.Font(11, wx.SWISS, wx.NORMAL, wx.NORMAL))

            if deleted:
                bgcolor = Data.Option.get('DeletedColor')
            elif hidden:
                bgcolor = Data.Option.get('HiddenColor')
            elif t == ta and tb:
                # parent table in two level report
                bgcolor = Data.Option.get('ParentColor')
            else:
                bgcolor = Data.Option.get('ChildColor')
                # attr.SetTextColour(wxBLACK)
                # attr.SetFont(wxFont(10, wxSWISS, wxNORMAL, wxBOLD))
                # attr.SetReadOnly(True)
                # self.SetRowAttr(row, attr)
            attr.SetBackgroundColour(bgcolor or "pale green")
            grid.SetRowAttr(row, attr)

# ---------------------- Draw Cells of Grid -----------------------------

class IndentedRenderer(wx.grid.PyGridCellRenderer):
    def __init__(self, table, renderer):
        wx.grid.PyGridCellRenderer.__init__(self)
        self.table = table
        self.renderer = renderer
        self.min_width = 20
        self.increment = 10

    def Draw(self, grid, attr, dc, rect, row, col, isSelected):
        x, y, width, height = rect
        max_indent = width - self.min_width
        if max_indent > 0:
             level = self.table.rowlevels[row]

             # subtract level of ancestor in primary table
             #   (because columns are not shared between tables)
             rt = self.table.reporttype
             ta = rt.get('TableA')
             rr = Data.ReportRow[self.table.rows[row]]
             t = rr.get('TableName')
             if t != ta:
                 parent = row
                 while t != ta and parent > 0:
                     parent -= 1
                     rr = Data.ReportRow[self.table.rows[parent]]
                     t = rr.get('TableName')
                 if parent >= 0:
                     level -= self.table.rowlevels[parent] + 1

             indent = level * self.increment
             if indent > 0:
                 if indent > max_indent:
                     indent = max_indent

                 # fill in the background
                 if isSelected:
                     color = wx.SystemSettings_GetColour(wx.SYS_COLOUR_HIGHLIGHT)
                 else:
                     color = attr.GetBackgroundColour()
                 dc.SetBrush(wx.Brush(color))
                 dc.SetPen(wx.Pen(color))
                 dc.DrawRectangle(x, y, indent, height)

                 # indent the drawing area
                 x += indent
                 width -= indent
                 rect = (x, y, width, height)

        self.renderer.Draw(grid, attr, dc, rect, row, col, isSelected)

class GanttCellRenderer(wx.grid.PyGridCellRenderer):
    def __init__(self, table):
        """
        Image Renderer Test.  This just places an image in a cell
        based on the row index.  There are N choices and the
        choice is made by  choice[row%N]
        """
        wx.grid.PyGridCellRenderer.__init__(self)
        self.table = table
        # self._choices = [images.getSmilesBitmap,
        #                  images.getMondrianBitmap,
        #                  images.get_10s_Bitmap,
        #                  images.get_01c_Bitmap]

        self.colSize = Data.Option.get("GanttColumnSize", 24)
        self.rowSize = Data.Option.get("GanttRowSize", 28)

    def Draw(self, grid, attr, dc, rect, row, col, isSelected):
        # choice = self.table.GetRawValue(row, col)
        # bmp = self._choices[ choice % len(self._choices)]()
        # image = wxMemoryDC()
        # image.SelectObject(bmp)
        o = Data.Option  # colors

        # Get data to draw gantt chart
        # if debug: print 'col & colid', col, self.table.columns[col]
        rc = self.table.reportcolumn[self.table.columns[col]]
        # if debug: print 'rc', rc

        # draw bar chart or not

        # there are two tests. the name test is new, the FirstDate test is used in verion 0.1
        ct = self.table.columntype[self.table.ctypes[col]]  # if the column is in a report it will always have a type
        ctname = ct.get('Name')
        # if debug: print "draw ctname", ctname
        if ctname and ctname[-6:] != "/Gantt": return

        fdate = rc.get('FirstDate') or Data.GetToday()
        if not fdate or not Data.DateConv.has_key(fdate): return  # this routine shouldn't be called for not gantt columns, but it is anyway - just ignore them
                                # the program seems to be refreshing three times when one is needed (040505)

        # if debug: print "-- didn't return --"

        ix = Data.DateConv[fdate]
        of = self.table.coloffset[col]

        period = ct.get('PeriodSize') or ctname
        dh, cumh, dow = Data.GetPeriodInfo(period, ix, of)  # if period not recognized, defaults to day
        # if debug: print 'ix, dh, cumh', ix, dh, cumh

        # clear the background
        dc.SetBackgroundMode(wx.SOLID)
        if isSelected and dh > 0:
            backcolor = o.get('WorkDaySelected', wx.BLUE)
        elif isSelected:
            backcolor = o.get('NotWorkDaySelected', wx.BLUE)
        else:
            backcolor = attr.GetBackgroundColour()

        dc.SetBrush(wx.Brush(backcolor, wx.SOLID))
        dc.SetPen(wx.Pen(backcolor, 1, wx.SOLID))
        if is24:
            dc.DrawRectangle(rect.x, rect.y, rect.width, rect.height)
        else:
            dc.DrawRectangle((rect.x, rect.y), (rect.width, rect.height))
        # draw gantt bar
        if dh > 0:  # only display bars on days that have working hours

            # get info needed to draw bar
            rr = self.table.reportrow[self.table.rows[row]]
            tname = rr['TableName']
            if tname != 'Task': return  # only draw for task records
            tid = rr['TableID']
            task = self.table.data['Task'][tid]

            # pick color
            if isSelected:
                plancolor = o.get('PlanBarSelected', wx.GREEN)
                actualcolor = o.get('ActualBarSelected', wx.Colour(255, 255, 0))
                basecolor = o.get('BaseBarSelected', wx.BLUE)
                completioncolor = o.get('CompletionBarSelected', wx.BLACK)
            else:
                plancolor = o.get('PlanBar', wx.GREEN)
                actualcolor = o.get('ActualBar', wx.Colour(255, 255, 0))
                basecolor = o.get('BaseBar', wx.BLUE)
                completioncolor = o.get('CompletionBar', wx.BLACK)

            override = rr.get('PlanBarColor')
            if override and re.match('^[0-9A-Fa-f]{6}$', override):
                plancolor = [ int(x,16) for x in (override[0:2], override[2:4], override[4:6]) ]

            # calculate bar location
            # (dh should be integer, but just to make sure I don't divide by a fraction below)
            es = task.get('hES', 0)  # if not found don't display gantt chart
            ef = task.get('hEF', 0)

            def drawbar(es, ef, barcolor, yof, yh):
                if es < (dh + cumh) and ef >= cumh:
                    if es <= cumh: xof = 0
                    else: xof = int( rect.width * (es - cumh)/dh)
                    if ef >= (cumh + dh): wof = 0
                    else: wof = int( rect.width * (cumh + dh - ef)/dh)
                    dc.SetBrush(wx.Brush(barcolor, wx.SOLID))
                    dc.SetPen(wx.Pen(barcolor, 1, wx.SOLID))
                    if is24:
                        dc.DrawRectangle(rect.x+xof, rect.y+yof, rect.width-wof-xof, yh)
                    else:
                        dc.DrawRectangle((rect.x+xof, rect.y+yof), (rect.width-wof-xof, yh))

            if task.get('SubtaskCount'):
                drawbar(es, ef, plancolor, 6, (rect.height-12)//2)  # half-height bar
            else:
                reportid = rr.get('ReportID')
                bars = rc.get('Bars') or 'p'  # could be 'pab'
                
                if len(bars) == 3:  # 'pab'  p
                    bar_height = (rect.height-12) // 3
                    y_offset = gap = 3
                elif len(bars) == 2:
                    bar_height = (rect.height-12) // 2
                    y_offset = gap = 4
                else:  # == 1
                    bar_height = (rect.height-12)
                    y_offset = gap = 6

                if 'p' in bars or 'c' in bars or 'C' in bars:
                    drawbar(
                        es, ef, plancolor,
                        y_offset, bar_height)
                    if 'c' in bars or 'C' in bars:  # completion percent
                        if task.get("ActualEndDate"):
                            pc = 100
                            position = ef
                        elif task.get("PercentComplete"):  # is this cell end past current position in task
                            pc = task.get("PercentComplete")
                            position = es + (pc * (ef-es) / 100) # current position of task
                        else:
                            pc = 0
                        if pc > 0:
                            if 'c' in bars:  # half height completion bar
                                drawbar(
                                    es, position, completioncolor,
                                    y_offset + bar_height//3, bar_height//3)
                            elif 'C' in bars:  # full height completion bar
                                drawbar(
                                    es, position, completioncolor,
                                    y_offset, bar_height)
                    y_offset += bar_height + gap

                asd = task.get('ActualStartDate')
                if 'a' in bars:
                    if asd:
                        ash = task.get('ActualStartHour') or 0
                        aed = task.get('ActualEndDate') or Data.TodayString()
                        aeh = task.get('ActualEndHour') or 0
                        drawbar(
                            Data.DateInfo[Data.DateConv[asd]][1] + ash,
                            Data.DateInfo[Data.DateConv[aed]][1] + aeh, actualcolor,
                            y_offset, bar_height)
                    y_offset += bar_height + gap

                bsd = task.get('BaseStartDate')
                if 'b' in bars and bsd:
                    bsh = task.get('BaseStartHour') or 0
                    bed = task.get('BaseEndDate') or bsd
                    beh = task.get('BaseEndHour') or 0
                    drawbar(
                        Data.DateInfo[Data.DateConv[bsd]][1] + bsh,
                        Data.DateInfo[Data.DateConv[bed]][1] + beh, basecolor,
                        y_offset, bar_height)

        # firstdate = table.reportcolumn[table.columns[col]]['FirstDate']
        # dateindex = Data.DateConv[firstdate] + of

        # date = Data.DateIndex[ index + of ]
        # rr = table.reportrow[table.rows[row]]
        # tid = rr['ID']
        # task = table.data['Task'][tid]
        # startdate = task.get('StartDate')

        # startdate = table.rows[row].get('CalculatedStartDate', None)
        # enddate = table.rows[row].get('CalculatedEndDate', None)
        # c, i = columns[col]
        # firstperiod =
        # thisperiod = GetPeriod(
        # If startdate != None and enddate != None:
        #         if startdate <= next period and end date > this period then skip
        #   if start date = this period and hours > 0 then no adjustment on left
        #   if end date > this period then no adjustment on right

# ------------------ Grid -------------------------

class GanttChartGrid(wx.grid.Grid):
    def __init__(self, parent, reportid):
        wx.grid.Grid.__init__(self, parent, -1) # initialize base class first

        self.table = GanttChartTable(reportid)
        self.SetTable(self.table)

        self.DisableDragRowSize()
        self.DisableDragGridSize()
        self.SetRowLabelSize(Data.Option.get("GridRowLabelSize", 40))
        self.SetColLabelSize(Data.Option.get("GridColumnLabelSize", 40))
        self.SetDefaultRowSize(Data.Option.get("GridRowSize", 24))
        self.SetDefaultColSize(Data.Option.get("GridColumnSize", 40))

        if Data.platform == "mac":
            self.SetLabelFont(wx.Font(11, wx.SWISS, wx.NORMAL, wx.BOLD))
        elif Data.platform == "linux":
            self.SetLabelFont(wx.Font(8, wx.SWISS, wx.NORMAL, wx.BOLD))
        elif Data.platform == "win":
            self.SetLabelFont(wx.Font(8, wx.SWISS, wx.NORMAL, wx.BOLD))

        wx.grid.EVT_GRID_RANGE_SELECT(self, self.OnSelect)
        wx.grid.EVT_GRID_SELECT_CELL(self, self.OnSelect)
        wx.grid.EVT_GRID_LABEL_LEFT_CLICK(self, self.OnLabelLeftClick)

        # override drawing of column headers
        header = self.GetGridColLabelWindow()
        wx.EVT_PAINT(header, self.OnColumnHeaderPaint)

        self.Reset()

    # ---- fix broken row select logic in wx.grid ----
    def OnLabelLeftClick(self, evt):
        # self.startrow == None if no selection is available to extend
        row = evt.GetRow()
        if debug: print "row", row
        if row != -1:
            selection = self.GetSelectedRows()
            if not selection: self.startrow = None  # no prior selection to extend
            if debug: print "selection", selection
            if evt.ControlDown() or evt.MetaDown():  # add row to selection (meta is mac's cmd key)
                if row in selection:
                    if debug: print "control click - remove from selection", row
                    self.BeginBatch()  # avoid confusing blink of prior selection
                    self.ClearSelection()
                    selection.remove(row)  # remove row from selection
                    for row in selection:  # define new selection without row
                        self.SelectRow(row, True)
                    self.EndBatch()
                else:
                    if debug: print "control click - add to selection", row
                    self.SelectRow(row, True)  # add to selection
                    self.startrow = row  # extend selections from here
            elif evt.ShiftDown():  # extend selection
                if debug: print "shift click - extend selection", row
                if self.startrow == None:  # same as plain click
                    self.SelectRow(row, False)
                    self.startrow = row  # extend selections from here
                elif row != self.startrow:
                    if row > self.startrow:
                        for r in range(self.startrow, row + 1):
                            if debug: print "add row to selection", r
                            self.SelectRow(r, True)
                    else:
                        for r in range(self.startrow, row - 1, -1):
                            if debug: print "add row to selection", r
                            self.SelectRow(r, True)
            else:  # plain click - start a new selection
                # if debug: print "left click - maybe new selection", row, selection
                # if not row in selection:  # allow clicks on selected rows w/o changing selection?
                if debug: print "left click - new selection", row
                self.SelectRow(row, False)
                self.startrow = row  # extend selections from here
                evt.Skip()  # this makes it so the drag select will still work
        else:  # clicked on column
            self.startrow = None  # for extending row selections
            evt.Skip()  # continue processing the event

    # -------- column header display
    def OnColumnHeaderPaint(self, evt):
        header = self.GetGridColLabelWindow()
        dc = wx.PaintDC(header)
        font = self.GetLabelFont()
        dc.SetFont(font)
        dc.SetBrush(wx.TRANSPARENT_BRUSH)
        dc.BeginDrawing()

        darkBorder = wx.Pen((68, 68, 68))

        columnCount = self.GetNumberCols()
        height = self.GetColLabelSize()
        middle = height / 2
        x = -self.GetViewStart()[0] * self.GetScrollPixelsPerUnit()[0]
        for col in range(columnCount):
            width = self.GetColSize(col)
            right = x + width
            label = self.GetColLabelValue(col)
            of = self.table.coloffset[col]
            if of < 0:
                dc.SetPen(darkBorder)
                dc.DrawRectangle(x - 1, 0, width + 1, height)

                dc.SetPen(wx.WHITE_PEN)  # draw highlights
                dc.DrawLine(x, 1, right - 1, 1)
                dc.DrawLine(x, 1, x, height - 1)

                rect = wx.Rect(x, 0, width, height)
                DrawClippedText(dc, label, rect)
            else:
                # time-scale headers have two parts
                lines = label.splitlines()
                name1, name2 = lines[0], lines[-1]

                # draw upper header
                if name1:
                    for c1 in xrange(col + 1, columnCount):
                        if self.table.coloffset[c1] < 1:
                            break
                        label = self.GetColLabelValue(c1)
                        lines = label.splitlines()
                        if (not lines) or lines[0]:
                            break
                    else:
                        c1 = columnCount

                    widths = [self.GetColSize(c) for c in xrange(col, c1)]
                    sumWidth = sum(widths)

                    dc.SetPen(darkBorder)
                    dc.DrawRectangle(x - 1, 0, sumWidth + 1, middle)

                    dc.SetPen(wx.WHITE_PEN)  # draw highlights
                    dc.DrawLine(x, 1, x + sumWidth - 1, 1)
                    dc.DrawLine(x, 1, x, middle - 1)

                    rect1 = wx.Rect(x, 0, sumWidth, middle)
                    DrawClippedText(dc, name1, rect1)

                # draw lower header
                dc.SetPen(darkBorder)
                dc.DrawRectangle(x - 1, middle, width + 1, middle)

                dc.SetPen(wx.WHITE_PEN)  # draw highlights
                dc.DrawLine(x, middle, right - 1, middle)
                dc.DrawLine(x, middle, x, height - 1)

                rect2 = wx.Rect(x, middle, width, middle)
                DrawClippedText(dc, name2, rect2)
            x = right

        dc.EndDrawing()

    def GetSelectedRows(self):
        """
        wx.grid.Grid treats shift-selected rows as merely selected cells.
        Add them back as selected rows.
        """
        selrow = wx.grid.Grid.GetSelectedRows(self)
        # if debug: print "old selrow", selrow
        seltl = self.GetSelectionBlockTopLeft()
        # if debug: print "seltl", seltl
        if seltl:  # found selected cells
            selbr = self.GetSelectionBlockBottomRight()
            # if debug: print "selbr", selbr
            cols = self.table.GetNumberCols()
            # if debug: print "cols", cols
            for (ra, ca), (rz, cz) in zip(seltl, selbr):
                if (ca == 0) and (cz == cols - 1):  # whole row selected
                    selrow += range(ra, rz+1)
            selrow = dict.fromkeys(selrow).keys()  # remove any duplicates
        # if debug: print "new selrow", selrow
        return selrow

    def GetSelectedCols(self):
        """
        wx.grid.Grid treats shift-selected columns as merely selected cells.
        Add them back as selected columns.
        """
        selcol = wx.grid.Grid.GetSelectedCols(self)
        seltl = self.GetSelectionBlockTopLeft()
        if seltl:  # found selected cells
            selbr = self.GetSelectionBlockBottomRight()
            rows = self.table.GetNumberRows()
            for (ra, ca), (rz, cz) in zip(seltl, selbr):
                if (ra == 0) and (rz == rows - 1):  # whole column selected
                    selcol += range(ca, cz+1)
            selcol = dict.fromkeys(selcol).keys()  # remove any duplicates
        return selcol

    def OnSelect(self, event):
        # if debug: print "OnSelect Cell"
        reportid = self.table.report['ID']
        f = Data.OpenReports.get(reportid)
        if f:
            Menu.AdjustMenus(f)
        event.Skip()

    def UpdateAttrs(self):
         self.table._updateRowAttrs(self)
         self.table._updateColAttrs(self)

# UpdateColumnWidths is no good -- if the column pointers have changed, then the
# column attributes must be updated, to preserve read-only status and cell renderers.
#    -- Alex

#     def UpdateColumnWidths(self):
#         rc = Data.ReportColumn
#         for i, c in enumerate(self.table.columns):
#             of = self.table.coloffset[i]
#             ct = Data.ColumnType[self.table.ctypes[i]]
#             ctname = ct.get('Name')
#             if of > -1 and ctname and ctname[-6:] == "/Gantt":
#                 self.SetColSize(i, 24)
#             else:
#                 cw = rc[c].get('Width') or 40
#                 self.SetColSize(i, cw)

    def Reset(self):
        """ Reset the view based on the data in the table.

        Call this when rows are added or destroyed
        """
        self.BeginBatch()

        for current, new, delmsg, addmsg in [
            (self.table._rows, self.table.GetNumberRows(), wx.grid.GRIDTABLE_NOTIFY_ROWS_DELETED, wx.grid.GRIDTABLE_NOTIFY_ROWS_APPENDED),
            (self.table._cols, self.table.GetNumberCols(), wx.grid.GRIDTABLE_NOTIFY_COLS_DELETED, wx.grid.GRIDTABLE_NOTIFY_COLS_APPENDED),
        ]:
            if new < current:
                msg = wx.grid.GridTableMessage(self.table,delmsg,new,current-new)
                self.ProcessTableMessage(msg)
            elif new > current:
                msg = wx.grid.GridTableMessage(self.table,addmsg,new-current)
                self.ProcessTableMessage(msg)
                msg = wx.grid.GridTableMessage(self.table, wx.grid.GRIDTABLE_REQUEST_VIEW_GET_VALUES)
                self.ProcessTableMessage(msg)

        self.table._rows = self.table.GetNumberRows()
        self.table._cols = self.table.GetNumberCols()

        # update the column rendering plugins
        self.UpdateAttrs()

        # update the scrollbars and the displayed part of the grid
        self.AdjustScrollbars()

        self.EndBatch()


#------------------ MultiSelect Frame -----------------------------------

class SearchSelection(UI.MultipleSelection):
    # should this be merged into the parent class in UI?

    def __init__(self, *args, **kwds):
        # begin wxGlade: ReportFrame.__init__
        UI.MultipleSelection.__init__(self, *args, **kwds)
        self.choices = []
        self.selections = {}

        wx.EVT_TEXT(self, self.Search.GetId(), self.onSearch)
        wx.EVT_CHAR(self.Search, self.onChar)
        wx.EVT_CHAR(self.SelectionListBox, self.onChar)

        ReportAids.RegisterSize(self, "SearchSelection")

    def GetValue(self):
        self.Search.SetValue("")
        self.RefreshSearch()
        return self.SelectionListBox.GetSelections()

    def onSearch(self, event):
        self.RefreshSearch()

    def RefreshSearch(self):
        # return a transformed list where every item is a unique tuple
        # identically-named choices are always shown at the same time,
        # in the same order, so we can tell them apart by their order
        def deDupper(oldlist):
            newlist = []
            count = dict.fromkeys(oldlist, 0)
            for x in oldlist:
                newlist.append((x, count[x]))
                count[x] += 1
            return newlist

        def reDupper(newlist):
            return [x[0] for x in newlist]

        # get the visible selection set
        choices = deDupper(self.SelectionListBox.GetStrings())
        sel_numbers = self.SelectionListBox.GetSelections()
        selections = dict.fromkeys([choices[n] for n in sel_numbers])

        # update the full selection set (includes invisible selections)
        for choice in choices:
            if choice in selections:
                self.selections[choice] = None
            elif choice in self.selections:
                del self.selections[choice]

        # save the original list of choices
        if not self.choices:
            self.choices = choices

        # limit the visible choices to those that match the search
        search = self.Search.GetValue()
        if search == "=":
            choices = [x for x in self.choices if x in self.selections]
        else:
            choices = [x for x in self.choices if re.search(search, x[0], re.I)]
                # x[0] is equivalent to reDupping
        self.SelectionListBox.Set(reDupper(choices))
        for i, choice in enumerate(choices):
            if choice in self.selections:
                self.SelectionListBox.Select(i)

    def onChar(self, event):
        if event.GetKeyCode() == wx.WXK_ESCAPE:
            self.Search.SetValue("")
            self.Search.SetFocus()
        event.Skip()

class MultiSelection(SearchSelection):
    # better name might be "LinkTableSelection"
    # or we could transfer the functionality of this class somewhere else

    def __init__(self, *args, **kwds):
        # begin wxGlade: ReportFrame.__init__
        SearchSelection.__init__(self, *args, **kwds)
        wx.EVT_BUTTON(self, wx.ID_OK, self.onOK)

    # ID == ID of this record
    # TargetIDs == IDs of candidate target records
    # Status == status of link records

    # LinkTable == table of link records (e.g., 'Assignment', 'Dependency')
    # Column1 == column that points to this record
    # Column2 == column that points to the target records
    # Message == undo message

    def onOK(self, event):
        if debug: print "Start onOK"
        if debug:
            print "link table:", self.LinkTable
            print "link status:", self.Status  # not used anymore, takes status at OK instead of at open
            print self.Column1 + ":", self.ID
            print self.Column2 + ":", self.TargetIDs
            print "message:", self.Message

        self.Search.SetValue("")
        self.RefreshSearch()

        linkdb = Data.Database[self.LinkTable]
        targetdb = Data.Database[self.Column2[:-2]]
        hasResourceGrouping = 'ResourceGrouping' in Data.Database
        expandGroups = self.LinkTable == 'Assignment' and self.Column2 == 'ResourceID' and hasResourceGrouping
        if debug:
            print 'hasResourceGrouping', hasResourceGrouping
            print 'expandGroups', expandGroups

        target_xref = {}  # from target id to link table id
        for k, v in linkdb.iteritems():
            if v.get(self.Column1) != self.ID: continue # include both active and inactive links
            target_xref[v.get(self.Column2)] = k
        if debug: print 'target_xref', target_xref

        vals = self.SelectionListBox.GetSelections()  # ??all targets in popup need to be accounted for?? -- ?no?
        if debug: print "selected values", vals

        targetids = {}  # active link targets
        for v in vals:  # offsets of user's selection
            target = self.TargetIDs[v]  # translate selection to target id
            if debug: print 'is group', targetdb[target].get('GroupType') == 'Work'
            if expandGroups and targetdb[target].get('GroupType') == 'Work':  # resource is a work group
                targets = Data.ParsePath(targetdb[target], 'ResourceGrouping-ResourceGroupID/Resource/ID') # get the list of resource ids for that group
                for x in targets:
                    targetids[x] = None  # remove duplicates
            else:
                targetids[target] = None
        if debug: print 'targetids', targetids

        for targetid in targetids.keys():  # turn on all selected targets
            linkid = target_xref.get(targetid)
            if not linkid:  # no record, must add
                change = {'Table': self.LinkTable, self.Column1: self.ID, self.Column2: targetid}
                Data.Update(change)
            else:
                del target_xref[targetid]  # prepare to clear all that are not set
                if linkdb[linkid].get('zzStatus') == 'deleted':  # record exists, but is deleted -> undelete
                    change = {'Table': self.LinkTable, 'ID': linkid, 'zzStatus': None}
                    Data.Update(change)

        for linkid in target_xref.values():  # turn off any prior records that were not selected
            if linkdb[linkid].get('zzStatus') != 'deleted':  # active -> must delete
                change = {'Table': self.LinkTable, 'ID': linkid, 'zzStatus': 'deleted'}
                Data.Update(change)

        Data.SetUndo(self.Message)
        event.Skip()


#------------------ Gantt Report Frame -----------------------------------

class GanttReportFrame(UI.ReportFrame):
    def __init__(self, reportid, *args, **kwds):
        if debug: "Start GanttReport init"
        if debug: print 'reportid', reportid
        UI.ReportFrame.__init__(self, *args, **kwds)

        # these three commands were moved out of UI.ReportFrame's init
        self.report_window = GanttChartGrid(self.Report_Panel, reportid)
        self.Report = self.report_window
        self.ReportID = reportid

        # self.Report, aka report_window, is a grid, not a report or a frame
        # report_window is referenced in one place in UI.ReportFrame.do_layout
        # Report is referenced in several scripts
        #     -- Alexander

        # Data.OpenReports[reportid] = self

        self.set_properties()  # these are in the parent class
        self.do_layout()

        Menu.doAddScripts(self)
        Menu.FillWindowMenu(self)
        Menu.AdjustMenus(self)
        self.SetReportTitle()

        # file menu events
        wx.EVT_MENU(self, wx.ID_NEW,    Menu.doNew)
        wx.EVT_MENU(self, wx.ID_OPEN,   Menu.doOpen)
        wx.EVT_MENU(self, wx.ID_CLOSE,  self.doClose)
        wx.EVT_MENU(self, wx.ID_CLOSE_ALL, Menu.doCloseReports)
        wx.EVT_MENU(self, wx.ID_SAVE,   Menu.doSave)
        wx.EVT_MENU(self, wx.ID_SAVEAS, Menu.doSaveAs)
        wx.EVT_MENU(self, ID.M_AUTO_SAVE, Menu.doAutoSaveOnQuit)
        wx.EVT_MENU(self, wx.ID_REVERT, Menu.doRevert)
        wx.EVT_MENU(self, ID.M_REVERT_OPEN, Menu.doRevertOpen)
        wx.EVT_MENU(self, wx.ID_EXIT, Menu.doExit)

        # edit menu events
        wx.EVT_MENU(self, wx.ID_UNDO,          Menu.doUndo)
        wx.EVT_MENU(self, wx.ID_REDO,          Menu.doRedo)
        wx.EVT_MENU(self, wx.ID_COPY, self.OnCopy)
        wx.EVT_MENU(self, wx.ID_PASTE, self.OnPaste)

        wx.EVT_MENU(self, ID.M_INSERT_ROW,     self.OnInsertRow)
        wx.EVT_MENU(self, ID.M_INSERT_ROW_ABOVE, self.OnInsertRow)
        wx.EVT_MENU(self, ID.M_INSERT_CHILD,    self.OnInsertChild)
        wx.EVT_MENU(self, ID.M_INSERT_RELATED,  self.OnInsertRow)
        wx.EVT_MENU(self, ID.M_DELETE_ROW,      self.OnDeleteRow)
        wx.EVT_MENU(self, ID.M_SHIFT_LEFT,     self.OnShiftRow)
        wx.EVT_MENU(self, ID.M_SHIFT_RIGHT,   self.OnShiftRow)
        wx.EVT_MENU(self, ID.M_MOVE_ROW_UP,     self.OnMoveRow)
        wx.EVT_MENU(self, ID.M_MOVE_ROW_DOWN,   self.OnMoveRow)

        wx.EVT_MENU(self, ID.M_EDIT_PROJECT_NAME,   self.OnEditProjectName)
        wx.EVT_MENU(self, ID.M_EDIT_REPORT_NAME,   self.OnEditReportName)

        # action menu events
        wx.EVT_MENU(self, ID.M_ASSIGN_PREREQUISITE, self.OnPrerequisite)
        wx.EVT_MENU(self, ID.M_LINK_TASKS,          self.OnPrerequisite)
        wx.EVT_MENU(self, ID.M_ASSIGN_RESOURCE,     self.OnAssignResource)
        wx.EVT_MENU(self, ID.M_RESOURCE_TO_GROUP,   self.OnResourceGroup)
        wx.EVT_MENU(self, ID.M_DEFINE_RESOURCE_GROUP, self.OnResourceGroup)
        wx.EVT_MENU(self, ID.M_ASSIGN_TASK,         self.OnAssignResource)

        # view menu events
        wx.EVT_MENU(self, ID.M_INSERT_COLUMN, self.OnInsertColumn)
        wx.EVT_MENU(self, ID.M_DELETE_COLUMN, self.OnDeleteColumn)
        wx.EVT_MENU(self, ID.M_MOVE_COLUMN_LEFT, self.OnMoveColumn)
        wx.EVT_MENU(self, ID.M_MOVE_COLUMN_RIGHT, self.OnMoveColumn)
        wx.EVT_MENU(self, ID.M_SCROLL_LEFT_FAST, self.OnScroll)
        wx.EVT_MENU(self, ID.M_SCROLL_LEFT, self.OnScroll)
        wx.EVT_MENU(self, ID.M_SCROLL_RIGHT, self.OnScroll)
        wx.EVT_MENU(self, ID.M_SCROLL_RIGHT_FAST, self.OnScroll)
        wx.EVT_MENU(self, ID.M_SCROLL_TO_TASK, self.OnScrollToTask)

        # script menu events
        wx.EVT_MENU(self, ID.FIND_SCRIPTS, Menu.doFindScripts)
        wx.EVT_MENU(self, ID.REFRESH_SCRIPTS, Menu.doRefreshScripts)
        wx.EVT_MENU(self, ID.REPEAT_SCRIPT, Menu.doRepeatScript)
        wx.EVT_MENU_RANGE(self, ID.FIRST_SCRIPT, ID.LAST_SCRIPT, Menu.doScript)

        # window menu events
        wx.EVT_MENU_RANGE(self, ID.FIRST_WINDOW, ID.LAST_WINDOW, self.doBringWindow)

        # help menu events
        wx.EVT_MENU(self, wx.ID_ABOUT, Menu.doShowAbout)
        wx.EVT_MENU(self, ID.QUICK_START, self.doQuick)
        wx.EVT_MENU(self, ID.ORM_QUICK_START, self.doORMQuick)
        wx.EVT_MENU(self, ID.SHORT_CUTS, self.doShort)
        wx.EVT_MENU(self, ID.HOME_PAGE, Menu.doHome)
        wx.EVT_MENU(self, ID.HELP_PAGE, Menu.doHelp)
        wx.EVT_MENU(self, ID.HELP_BOOK, Menu.doBook)
        wx.EVT_MENU(self, ID.FORUM, Menu.doForum)

        # frame events
        wx.EVT_ACTIVATE(self, self.OnActivate)
        wx.EVT_CLOSE(self, self.doClose)
        wx.EVT_SIZE(self, self.OnSize)
        wx.EVT_MOVE(self, self.OnMove)

        # grid events
        wx.grid.EVT_GRID_COL_SIZE(self, self.OnColSize)
        wx.grid.EVT_GRID_ROW_SIZE(self, self.OnRowSize)
        wx.grid.EVT_GRID_EDITOR_SHOWN(self, self.OnEditorShown)
        wx.grid.EVT_GRID_EDITOR_HIDDEN(self, self.OnEditorHidden)

        # tool bar events
        wx.EVT_TOOL(self, ID.INSERT_ROW, self.OnInsertRow)
        wx.EVT_TOOL(self, ID.DUPLICATE_ROW, self.OnDuplicateRow)
        wx.EVT_TOOL(self, ID.DELETE_ROW, self.OnDeleteRow)
        wx.EVT_TOOL(self, ID.MOVE_UP, self.OnMoveRow)
        wx.EVT_TOOL(self, ID.MOVE_DOWN, self. OnMoveRow)
        wx.EVT_TOOL(self, ID.PREREQUISITE, self.OnPrerequisite)
        wx.EVT_TOOL(self, ID.ASSIGN_RESOURCE, self.OnAssignResource)

        wx.EVT_TOOL(self, ID.HIDE_ROW, self.OnHide)
        wx.EVT_TOOL(self, ID.SHOW_HIDDEN, self.OnShowHidden)

        wx.EVT_TOOL(self, ID.INSERT_COLUMN, self.OnInsertColumn)
        wx.EVT_TOOL(self, ID.DELETE_COLUMN, self.OnDeleteColumn)
        wx.EVT_TOOL(self, ID.MOVE_LEFT, self.OnMoveColumn)
        wx.EVT_TOOL(self, ID.MOVE_RIGHT, self.OnMoveColumn)

        # wx.EVT_TOOL(self, ID.COLUMN_OPTIONS, self.OnColumnOptions)

        wx.EVT_TOOL(self, ID.SCROLL_LEFT_FAR, self.OnScroll)
        wx.EVT_TOOL(self, ID.SCROLL_LEFT, self.OnScroll)
        wx.EVT_TOOL(self, ID.SCROLL_RIGHT, self.OnScroll)
        wx.EVT_TOOL(self, ID.SCROLL_RIGHT_FAR, self.OnScroll)
        wx.EVT_TOOL(self, ID.SCROLL_TO_TASK, self.OnScrollToTask)

        self.Report.ForceRefresh()
        if debug: "End GanttReport init"

    # used my any commands that move grid rows or columns
    def FinishEdit(self):
        '''If a cell is being editted, save it before moving rows'''
        self.Report.SaveEditControlValue()
        self.Report.DisableCellEditControl()
        
    # ------ Tool Bar Commands ---------

    def OnInsertRow(self, event):
        self.FinishEdit()
        insertAfter = (event.GetId() in (ID.INSERT_ROW, ID.M_INSERT_ROW))
        self.CreateRow(insertAfter, False)

    def OnInsertChild(self, event):
        self.FinishEdit()
        self.CreateRow(True, True)

    def CreateRow(self, insertAfter, makeChild):
        if debug: print "Start InsertRow"

        r = Data.Report[self.ReportID]
        rt = Data.ReportType[r['ReportTypeID']]
        ta = rt.get('TableA')
        if not ta or ta in ('Report', 'ReportColumn', 'ReportRow', 'ReportType', 'ColumnType'): return  # need special handling

        sel = self.Report.GetSelectedRows()
        if sel:
            # consider only the best-ranking rows
            rowlevels = self.Report.table.rowlevels
            level = min([rowlevels[x] for x in sel])
            sel = [x for x in sel if rowlevels[x] == level]

            if insertAfter:
                row = max(sel)
            else:
                row = min(sel)
        elif self.Report.table.rows:
            row = self.Report.GetGridCursorRow()
        else:
            row = -1

        if row == -1:
            parent = None  # no rows are visible
        else:
            rowid = self.Report.table.rows[row]
            rr = Data.ReportRow[rowid]
            while row > 0 and rr['TableName'] != ta:  # find a primary row
                row -= 1
                rowid = self.Report.table.rows[row]
                rr = Data.ReportRow[rowid]
            if makeChild:
                parent = rowid  # child of selection
            else:
                parent = rr.get('ParentRow')  # same parent as selection

        change = {'Table': ta, 'Name': '--'}
        if ta != 'Project':
            change['ProjectID'] = r.get('ProjectID')
        if parent:
            change[ta + 'ID'] = Data.ReportRow.get(parent, {}).get('TableID')
        undo = Data.Update(change)
        newid = undo['ID']

        change = {'Table': ta, 'ID': newid, 'Name': _(ta) + " " + str(newid)}
        undo = Data.Update(change)

        change = {'Table': 'ReportRow', 'ReportID': self.ReportID, 'TableName': ta, 'TableID': newid, 'ParentRow': parent}
        undo = Data.Update(change)  # created here to control where inserted
        newrowid = undo['ID']

        rlist = Data.GetRowList(self.ReportID)
        if row == -1:
            pos = len(rlist)  # no rows are visible
        else:
            pos = rlist.index(rowid)
            if insertAfter: pos += 1
        rlist.insert(pos, newrowid)
        Data.ReorderReportRows(self.ReportID, rlist)

        Data.SetUndo(_('Insert %s') % _(ta))
        if debug: print "End InsertRow"

        # move current cell to inserted row
        try:
            r = self.Report.table.rows.index(newrowid)
        except:
            pass
        else:
            c = self.Report.GetGridCursorCol()
            self.Report.SetGridCursor(r, c)
            self.Report.MakeCellVisible(r, c)

    def OnDuplicateRow(self, event):
        self.FinishEdit()
        sel = self.Report.GetSelectedRows()  # current selection
        if len(sel) == 0:
            if debug: print "can't duplicate, empty selection"
            return
        rtid = Data.Report[self.ReportID].get('ReportTypeID')
        tablea = Data.ReportType[rtid].get('TableA')
        new = []
        for s in sel:
            rid = self.Report.table.rows[s]
            ta = Data.ReportRow[rid]['TableName']
            if ta != tablea: continue  # only duplicate rows of primary table type
            rcopy = Data.ReportRow[rid].copy()  # report row
            tid = rcopy['TableID']
            tcopy = Data.Database[ta][tid].copy()  # table row

            tcopy['Table'] = ta; del tcopy['ID']
            undo = Data.Update(tcopy)
            rcopy['Table'] = 'ReportRow'; del rcopy['ID']
            rcopy['TableID'] = undo['ID']
            undo = Data.Update(rcopy)
            new.append(undo['ID'])

        rlist = Data.GetRowList(self.ReportID)  # list of row id's in display order
        where = max(sel) + 1
        # print "rlist", rlist
        # print "new", new
        rlist[where:where] = new
        # print "new rlist", rlist
        Data.ReorderReportRows(self.ReportID, rlist)

        Data.SetUndo(_('Duplicate Row'))

    def OnDeleteRow(self, event):  # this is a shallow delete
        if debug: print "Start OnDeleteRow"
        self.FinishEdit()
        sel = self.Report.GetSelectedRows()  # current selection
        if len(sel) < 1:
            if debug: print "can't delete, no rows selected"
            return  # only move if rows selected
        change = { 'Table': None, 'ID': None, 'zzStatus': 'deleted' }
        cnt = 0
        for s in sel:
            rid = self.Report.table.rows[s]
            ta = Data.ReportRow[rid].get('TableName')
            id = Data.ReportRow[rid].get('TableID')
            if not id: continue  # silently skip invalid table id's
            if ta == 'Project' and id == 1: continue  # certain projects and reports can't be deleted
            elif ta == 'Report' and (id == 1 or id == 2): continue
            if Data.Database[ta][id].get('zzStatus') == 'deleted':
                change['zzStatus'] = None
            else:
                change['zzStatus'] = 'deleted'
            change['Table'] = ta
            change['ID'] = id
            undo = Data.Update(change)
            cnt += 1
        if cnt > 0: Data.SetUndo(_('Delete/Reactivate Row'))
        if debug: print "End OnDeleteRow"


    def OnShiftRow(self, event):
        """ Shift selected row one level up or down in hierarchy """
        # if left -> set parent to parent's parent
        # if right -> find prior record with same parent; make that record the parent

        self.FinishEdit()
        sel = self.Report.GetSelectedRows()
        # if sel and len(sel) > 1:  # not more than one selected row
        #     return
        
        if not sel:
            if not self.Report.table.rows:
                return
            sel = [self.Report.GetGridCursorRow()]  # if no rows are selected, use cursor row    
        sel.sort()

        rows = self.Report.table.rows
        rowlevels = self.Report.table.rowlevels
        firstrowid = rows[sel[0]]

        if event.GetId() in (ID.M_SHIFT_LEFT, ):
            # ignore lower-ranking child rows
            level = min([rowlevels[x] for x in sel])
            if level < 1: level = 1
            sel = [x for x in sel if rowlevels[x] == level]

        for row in sel:
            rid = rows[row]
            ta = Data.ReportRow[rid].get('TableName')
            id = Data.ReportRow[rid].get('TableID')
            if (ta not in Data.Database or id not in Data.Database[ta] or
                Data.Database[ta][id].get('zzStatus') == 'deleted'):  # silently skip invalid table id's
                return

            table = Data.Database[ta]
            parent = table[id].get(ta + 'ID')

            if event.GetId() in (ID.M_SHIFT_LEFT, ):
                if parent not in table or table[id].get('zzStatus') == 'deleted': continue
                newparent = table[parent].get(ta + 'ID')
                change = { 'Table': ta, 'ID': id, ta + 'ID': newparent }
                Data.Update(change)

            else: # ID.Shift_Right
                step = 1
                for s in range(row-1, -1, -1):  # check each prior row in the report
                    rid2 = rows[s]
                    ta2 = Data.ReportRow[rid2].get('TableName')
                    if ta != ta2: continue  # ignore rows that aren't in the same table
                    id2 = Data.ReportRow[rid2].get('TableID')
                    if (id2 not in Data.Database[ta2] or
                        Data.Database[ta2][id2].get('zzStatus') == 'deleted'):  # silently skip invalid table id's
                        continue
                    parent2 = table[id2].get(ta2 + 'ID')
                    if parent != parent2: continue  # the first prior row with the same parent will be this records new parent

                    change = { 'Table': ta, 'ID': id, ta + 'ID': id2 }
                    Data.Update(change)
                    break

        if event.GetId() in (ID.M_SHIFT_LEFT, ):
            Data.SetUndo(_('Shift Left'))
        else:
            Data.SetUndo(_('Shift Right'))

        # move current cell so it is on the same record
        try:
            r = self.Report.table.rows.index(firstrowid)
        except:
            pass
        else:
            c = self.Report.GetGridCursorCol()
            old_r = self.Report.GetGridCursorRow()
            if r != old_r:
                self.Report.SetGridCursor(r, c)
                self.Report.MakeCellVisible(r, c)

    def OnMoveRow(self, event):
        """ Move selected rows up or down by one position """
        self.FinishEdit()
        sel = self.Report.GetSelectedRows()
        if not sel:
#            return
            sel = [self.Report.GetGridCursorRow()]  # if no rows are selected, use cursor row
            if not self.Report.table.rows: return  # no cursor row (i.e. no rows in report)

        if event.GetId() in (ID.MOVE_UP, ID.M_MOVE_ROW_UP):
            step = -1
        else: # ID.MOVE_DOWN
            step = 1

        rows = self.Report.table.rows
        rowlevels = self.Report.table.rowlevels

        # move only the best-ranking rows of the selection
        level = min([rowlevels[x] for x in sel])
        move = [x for x in sel if rowlevels[x] == level]
        move.sort()

        row_set = dict.fromkeys(move)  # set of row positions
        move_row_ids = [rows[x] for x in move]  # list of report row ids

        # determine whether the selection is contiguous
        #   (ignoring children and invisible rows)
        start = move[0]
        end = move[-1] + 1

        for x in xrange(start, end):
            if x in row_set:
                continue
            lev = rowlevels[x]
            if lev <= level:
                contiguous = False
                break
        else:
            contiguous = True

            # find the next visible, same-parent row
            next_row_id = None

            if step < 0:
                x = start - 1
            else:
                x = end

            while 0 <= x < len(rows):
                lev = rowlevels[x]
                if lev == level:
                    next_row_id = rows[x]
                    break
                elif lev < level:
                    break
                x += step

        # check for invisible rows
        if not self.Report.table.report.get('ShowHidden'):
            # report may include rows that are not displayed

            # get report rows from database; convert row positions
            rows, rowlevels = Data.GetRowLevels(self.ReportID)

            positions = {}
            for x, rid in enumerate(rows):
                positions[rid] = x

            row_set = {}
            for rowid in move_row_ids:
                if rowid in positions:
                    pos = positions[rowid]
                    row_set[pos] = None
            if not row_set:
                return

            move = row_set.keys()
            move.sort()

            start = move[0]
            end = move[-1] + 1
        else:
            # don't modify the row list directly
            rows = rows[:]

        if not contiguous:
            # consolidate rows
            if step < 0:
                end = start + len(move)
            else:
                start = end - len(move)

            move.reverse()
            for x in move:
                del rows[x]
            rows[start:start] = move_row_ids

        elif next_row_id:
            # move next row to the other side of selection
            if step < 0:
                dest = end - 1
            else:
                dest = start
            rows.remove(next_row_id)
            rows.insert(dest, next_row_id)

        else:
            # nothing changed
            return

        # apply the new row order
        Data.ReorderReportRows(self.ReportID, rows)
        Data.SetUndo(_('Move Row'))

    def OnEditProjectName(self, event):
        projectid = Data.Report[self.ReportID].get('ProjectID')
        if projectid:
            Menu.doEditName('Project', projectid)

    def OnEditReportName(self, event):
        Menu.doEditName('Report', self.ReportID)

    def OnPrerequisite(self, event):
        # list tasks in the same order they appear now  -- use self.Report.table.rows
        # highlight the ones that are currently prerequisites
        sel = self.Report.GetSelectedRows()  # current selection
        if len(sel) < 1:
            if debug: print "must select at least one row"
            return
        elif len(sel) > 1:
            sel.sort()  # chain tasks in the order they appear on report
            rows = self.Report.table.rows
            alltids = [ Data.ReportRow[rows[x]].get('TableID') for x in sel if Data.ReportRow[rows[x]].get('TableName') == 'Task' ]
            tids = [ x for x in alltids if not Data.Task[x].get('SubtaskCount') ]
            if len(tids) > 1:
                for i in range(len(tids) - 1):  # try to match and link each pair
                    # look for an existing dependency record
                    did = Data.FindID('Dependency', 'PrerequisiteID', tids[i], 'TaskID', tids[i+1])
                    if did:
                        if Data.Database['Dependency'][did].get('zzStatus') == 'deleted':
                            change = { 'Table': 'Dependency', 'ID': did, 'zzStatus': None }
                            Data.Update(change)
                    else:
                        change = { 'Table': 'Dependency', 'PrerequisiteID': tids[i], 'TaskID': tids[i+1] }
                        Data.Update(change)
                Data.SetUndo(_("Set Dependencies"))
            return
        rows = self.Report.table.rows
        sel = sel[0]
        rowid = rows[sel]  # get selection's task id
        ta = Data.ReportRow[rowid].get('TableName')
        if ta != 'Task': return  # only on task rows
        sid = Data.ReportRow[rowid].get('TableID')
        assert sid, "tried to assign prereq's to an invalid task id"

        alltid = [ Data.ReportRow[x].get('TableID') for x in rows if Data.ReportRow[x].get('TableName') == 'Task']
        tid = [ x for x in alltid if Data.Task[x].get('zzStatus') != 'deleted' and x != sid]  # active displayed tasks
        tname = [ Data.Task[x].get('Name', "") or "--" for x in tid ]

        status = [0] * len(tid)  # array of 0's
        for k, v in Data.Dependency.iteritems():
            if sid != v.get('TaskID'): continue
            p = v.get('PrerequisiteID')
            try:
                i = tid.index(p)
            except: pass
            else:
                if v.get('zzStatus') != 'deleted':
                    status[i] = k
                else:
                    status[i] = -k

        dialog = MultiSelection(self, -1, "Assign Prerequisite", size=(240,320))
        dialog.Instructions.SetLabel("Select prerequisite tasks:")
        # dialog.SelectionListBox.Clear()
        dialog.SelectionListBox.Set(tname)
        for i, v in enumerate(status):
            if v > 0:
                dialog.SelectionListBox.SetSelection(i)

        dialog.ID = sid
        dialog.TargetIDs = tid
        dialog.Status = status

        dialog.LinkTable = 'Dependency'
        dialog.Column1 = 'TaskID'
        dialog.Column2 = 'PrerequisiteID'
        dialog.Message = 'Set Dependencies'

        dialog.Centre()  # centers dialog on screen
        dialog.ShowModal()

    def OnResourceGroup(self, event):
        # list names in the same order they appear now  -- use self.Report.table.rows
        # highlight the ones that are currently linked
        kind = 'Resource'

        # edit selection
        sel = self.Report.GetSelectedRows()  # current selection
        if len(sel) != 1:
            if debug: print "must select one row"
            return
        rows = self.Report.table.rows
        sel = sel[0]
        rowid = rows[sel]  # get selection's task id
        ta = Data.ReportRow[rowid].get('TableName')
        if ta != kind: return  # wrong kind of row
        sid = Data.ReportRow[rowid].get('TableID')
        assert sid, "tried to assign to an invalid id"

        # create list of names for menu
        allids = [ Data.ReportRow[x].get('TableID') for x in rows if Data.ReportRow[x].get('TableName') == kind]
        type = bool(Data.Resource[sid].get('GroupType'))
        # remove deleted, self, and same type (group vs. not group)
        ids = [ x for x in allids if Data.Resource[x].get('zzStatus') != 'deleted' and (type != bool(Data.Resource[x].get('GroupType'))) and x != sid]
        names = [ Data.Resource[x].get('Name', "") or "--" for x in ids ]

        status = [0] * len(ids)  # array of 0's
        if type:
            matchfield = 'ResourceGroupID'
            getfield = 'ResourceID'
            prompt = _('Select resources:')
        else:
            matchfield = 'ResourceID'
            getfield = 'ResourceGroupID'
            prompt = _('Select resource groups:')
        for k, v in Data.Database['ResourceGrouping'].iteritems():
            if sid != v.get(matchfield): continue
            p = v.get(getfield)
            try:
                i = ids.index(p)
            except: pass
            else:
                if v.get('zzStatus') != 'deleted':
                    status[i] = k
                else:
                    status[i] = -k

        dialog = MultiSelection(self, -1, "Resource Grouping", size=(240,320))
        dialog.Instructions.SetLabel(prompt)
        # dialog.SelectionListBox.Clear()
        dialog.SelectionListBox.Set(names)
        for i, v in enumerate(status):
            if v > 0:
                dialog.SelectionListBox.SetSelection(i)

        dialog.ID = sid
        dialog.TargetIDs = ids
        dialog.Status = status

        dialog.LinkTable = 'ResourceGrouping'
        dialog.Column1 = matchfield
        dialog.Column2 = getfield
        dialog.Message = prompt

        dialog.Centre()  # centers dialog on screen
        dialog.ShowModal()

    def OnAssignResource(self, event):
        # list resources in alphabetical order
        # highlight the ones that are currently assigned
        sel = self.Report.GetSelectedRows()  # user's selection
        if len(sel) != 1:
            if debug: print "only assignments for one task"
            return  # only move if rows selected
        rows = self.Report.table.rows
        sel = sel[0]  # make it not a list
        rowid = rows[sel]  # get selected task's id
        sid = Data.ReportRow[rowid].get('TableID')

        assert sid and sid > 0, "tried to assign to an invalid id"

        ta = Data.ReportRow[rowid].get('TableName')
        if ta == 'Task':
            res = []
            for k, v in Data.Resource.iteritems():
                if v.get('zzStatus') == 'deleted': continue
                name = v.get('Name') or '--'
                firstname = v.get('FirstName')
                if firstname: name += ", " + firstname
                res.append((k, name))
            res.sort()
            ids, names = zip(*res)
            column1 = 'TaskID'
            column2 = 'ResourceID'
            dialogPrompt = _("Select assigned resources:")
        elif ta == 'Resource':
            if Data.Database[ta][sid].get('GroupType') == 'Work': return  # can't assigne tasks to resource groups
            tasks = []
            for k, v in Data.Task.iteritems():
                if v.get('zzStatus') == 'deleted': continue
                name = v.get('Name') or '--'
                tasks.append((k, name))
            tasks.sort()
            ids, names = zip(*tasks)
            column1 = 'ResourceID'
            column2 = 'TaskID'
            dialogPrompt = _("Select assigned tasks:")
        else:
            return

        assigns = {}
        for k, v in Data.Assignment.iteritems():
            if sid != v.get(column1): continue
            id2 = v.get(column2)
            if v.get('zzStatus') == 'deleted':
                assigns[id2] = -k
            else:
                assigns[id2] = k
        status = [assigns.get(id, 0) for id in ids]

        dialog = MultiSelection(self, -1, "Assign Resource", size=(240,320))
        dialog.Instructions.SetLabel(dialogPrompt)
        # dialog.SelectionListBox.Clear()
        dialog.SelectionListBox.Set(names)
        for i, v in enumerate(status):
            if v > 0:
                dialog.SelectionListBox.SetSelection(i)

        dialog.ID = sid
        dialog.TargetIDs = ids
        dialog.Status = status

        dialog.LinkTable = 'Assignment'
        dialog.Column1 = column1
        dialog.Column2 = column2
        dialog.Message = 'Set Assignments'

        dialog.Centre()  # centers dialog on screen
        dialog.ShowModal()

    def OnHide(self, event):
        self.FinishEdit()
        sel = self.Report.GetSelectedRows()  # user's selection
        Menu.onHide(self.Report.table, event, sel)

    def OnShowHidden(self, event):
        self.FinishEdit()
        Menu.onShowHidden(self, event)

    def OnInsertColumn(self, event):
        if debug: print "Start OnInsertColumn"
        insertAfter = False

        self.FinishEdit()
        r = Data.Report[self.ReportID]
        rtid = r.get('ReportTypeID')  # these determine what kind of columns can be inserted
        also = Data.ReportType[rtid].get('Also')

        menuid = []  # list of types to be displayed for selection
        rlist = Data.GetRowList(2)  # report 2 defines the sequence of the report type selection list
        for k in rlist:
            rr = Data.ReportRow[k]
            hidden = rr.get('Hidden', False)
            table = rr.get('TableName')  # should always be 'ReportType' or 'ColumnType'
            id = rr.get('TableID')
            if (not hidden) and table == 'ColumnType' and id:
                xrtid = Data.ColumnType[id].get('ReportTypeID')
                active = Data.ColumnType[id].get('zzStatus') != 'deleted'
                if active and xrtid and ( (rtid == xrtid) or (also and (also == xrtid)) ):
                    menuid.append( id )
        def getname(id):
            name = Data.ColumnType[x].get('Label')
            if name:
                name = _(name)  # translate label
            else:
                name = Data.ColumnType[x].get('Name')
            return name
        menutext = [ getname(x) for x in menuid ]
        for i, v in enumerate(menutext):  # remove line feeds before menu display
            if '\n' in v:
                menutext[i] = v.replace('\n', ' ')
        menuT = [ Data.ColumnType[x].get('T') for x in menuid ]

        # sort menu
        menus = zip(menuT, menutext, menuid)
        menus.sort()
        menuT, menutext, menuid = zip(*menus)
        if debug: print menuid, menutext

        dlg = SearchSelection(self, -1, "New Columns", size=(240, 320))
        dlg.Instructions.SetLabel("Select columns to add:")
        dlg.SelectionListBox.Set(menutext)
        # dlg = wxMultipleChoiceDialog(self, "Select columns to add:", "New Columns", menutext, style=wx.DEFAULT_FRAME_STYLE, size=(240, 320))
        dlg.Centre()
        if (dlg.ShowModal() != wx.ID_OK): return
        newlist = dlg.GetValue()
        addlist = []

        change = { 'Table': 'ReportColumn', 'ReportID': self.ReportID }  # new record because no ID specified
        for n in newlist:
            change['ColumnTypeID'] = menuid[n]
            ct = Data.ColumnType[menuid[n]]
            if ct['Name'] == 'Day/Gantt':  # leaving this for compatibility with version 0.1
                change['Periods'] = 21
                change['FirstDate'] = Data.GetToday()
                undo = Data.Update(change)
                del change['Periods']; del change['FirstDate']
            elif ct.get('AccessType') == 's':
                change['Periods'] = 14
                # change['FirstDate'] = Data.GetToday()
                undo = Data.Update(change)
                del change['Periods']
                # del change['FirstDate']
            else:
                change['Width'] = ct.get('Width')
                undo = Data.Update(change)
                del change['Width']
            addlist.append(undo['ID'])

        clist = Data.GetColumnList(self.ReportID)
        sel = self.Report.GetSelectedCols()
        # if debug: print 'selection', sel
        if sel:
            if insertAfter:
                pos = max(sel)
            else:
                pos = min(sel)
        else:
            # pos = self.Report.GetGridCursorCol()
            pos = len(clist)
        if insertAfter: pos += 1
        clist[pos:pos] = addlist
        Data.ReorderReportColumns(self.ReportID, clist)

        Data.SetUndo(_('Insert Column'))
        if debug: print "End OnInsertColumn"

    def OnDeleteColumn(self, event):
        if debug: print "Start OnDeleteColumn"
        self.FinishEdit()
        sel = self.Report.GetSelectedCols()
        if not sel: return

        clist = Data.GetColumnList(self.ReportID)
        cids = self.Report.table.columns

        sel_cids = [cids[x] for x in sel]
        sel_map = dict.fromkeys(sel_cids)
        clist = [x for x in clist if x not in sel_map]

        # apply the new column order
        Data.ReorderReportColumns(self.ReportID, clist)
        Data.SetUndo('Delete Column')

        # clear the selection
        self.Report.ClearSelection()

        if debug: print "End OnDeleteColumn"

    def OnMoveColumn(self, event):
        """ Move selected columns left or right by one position """
        self.FinishEdit()
        sel = self.Report.GetSelectedCols()
        if not sel:
            return

        if event.GetId() in (ID.MOVE_LEFT, ID.M_MOVE_COLUMN_LEFT):
            step = -1
        else: # ID.MOVE_RIGHT
            step = 1

        cids = self.Report.table.columns

        # get report columns from database
        #   (grid uses multiple grid columns for certain report columns)
        clist = Data.GetColumnList(self.ReportID)
        cids = self.Report.table.columns

        map = {}
        for sel_x in sel:
            try:
                x = clist.index(cids[sel_x])
            except:
                continue
            map[x] = None
        if not map:
            return

        move = map.keys()
        move.sort()

        # perform the movement
        start = move[0]
        end = move[-1] + 1

        if end - start > len(move):
            # consolidate columns
            if step < 0:
                end = start + len(move)
            else:
                start = end - len(move)

            move_cids = [clist[x] for x in move]
            move.reverse()
            for x in move:
                del clist[x]
            clist[start:start] = move_cids

        else:
            # move past the next column
            if step < 0:
                x = start - 1
                dest = end - 1
            else:
                x = end
                dest = start

            if 0 <= x < len(clist):
                cid = clist.pop(x)
                clist.insert(dest, cid)
                start += step
                end += step
            else:
                return

        # apply the new column order
        Data.ReorderReportColumns(self.ReportID, clist)
        Data.SetUndo(_('Move Column'))

    def OnScroll(self, event):
        """ scroll the selected  """
        if debug: print "Start OnScroll"
        self.FinishEdit()
        # figure out which gantt chart to scroll -- either the ones w/ selected columns or the first one
        scrollcols = []
        sel = self.Report.GetSelectedCols()  # current selection
        if len(sel) > 0:
            for s in sel:
                cid = self.Report.table.columns[s]
                ctid = Data.ReportColumn[cid]['ColumnTypeID']
                if Data.ColumnType[ctid].get('AccessType') == 's':
                    if scrollcols.count(cid) == 0:
                        scrollcols.append(cid)
        if len(scrollcols) == 0:
            clist = Data.GetColumnList(self.ReportID)  # complete list of column id's in display order
            for cid in clist:
                ctid = Data.ReportColumn[cid]['ColumnTypeID']
                if Data.ColumnType[ctid].get('AccessType') == 's':
                    scrollcols.append(cid)
                    break
        offset = 0; fast = False
        id = event.GetId()  # move left or right?
        if id in (ID.SCROLL_LEFT_FAR, ID.M_SCROLL_LEFT_FAST): offset = 1; fast = True
        elif id in (ID.SCROLL_LEFT, ID.M_SCROLL_LEFT): offset = 1
        elif id in (ID.SCROLL_RIGHT, ID.M_SCROLL_RIGHT): offset = -1
        elif id in (ID.SCROLL_RIGHT_FAR, ID.M_SCROLL_RIGHT_FAST): offset = -1; fast = True
        somethingChanged = False
        for s in scrollcols:
            date = Data.GetColumnDate(s, offset, fast)
            if date == None: continue
            newdate = Data.DateIndex[date]
            change = { 'Table': 'ReportColumn', 'ID': s, 'FirstDate': newdate}
            Data.Update(change)
            somethingChanged = True
        if somethingChanged: Data.SetUndo(_('Scroll Timescale Columns'))
        if debug: print "End Scroll"

    def OnScrollToTask(self, event):
        if debug: print "Start ScrollToTask"
        self.FinishEdit()
        sel = self.Report.GetSelectedRows()  # current selection
        if len(sel) < 1:
            if debug: print "can't scroll, no tasks selected"
            return  # only move if columns selected
        s = self.Report.table.rows[min(sel)]  # just use first row
        rs = Data.ReportRow[s]
        if not rs.get('TableName') in ('Task', 'Assignment'):
            if debug: print "tablename", rs.get('TableName')
            return  # can only scroll to tasks or assignments
        if rs.get('TableName') == 'Task':
            tid = rs.get('TableID')
        else:  # assignment
            aid = rs.get('TableID')
            tid = Data.Assignment[aid].get('TaskID')
        newdate = Data.ValidDate(Data.Task[tid].get('StartDate')) or Data.Task[tid].get('CalculatedStartDate')
            # maybe this should always go with the CalculatedStartDate
        if debug: print "new date", newdate
        if not newdate: return  # no date to scroll to

        somethingChanged = False
        clist = Data.GetColumnList(self.ReportID)  # complete list of row id's in display order
        for c in clist:
            ctid = Data.ReportColumn[c]['ColumnTypeID']
            ct = Data.ColumnType[ctid]
            if ct.get('AccessType') != 's':  # not a time scale
                continue
            # period = ct.get('PeriodSize') or ct.get('Name')
            change = {'Table': 'ReportColumn', 'ID': c, 'FirstDate': newdate}
            Data.Update(change)
            somethingChanged = True
        if somethingChanged: Data.SetUndo(_('Scroll to Task'))
        if debug: print "End ScrollToTask"

    # ---- menu Commands -----

    def doClose(self, event):
        Data.CloseReport(self.ReportID)

    def OnCopy(self, event):
        rid = self.ReportID  # current report
        if rid == 1:
            hint("This command doesn't work on the main report.")
            return  # do nothing

        # find selection
        selcol = self.Report.GetSelectedCols()  # current selection

        selrow = self.Report.GetSelectedRows()  # current selection
        selcell = self.Report.GetSelectedCells()  # current selection
        selr = self.Report.GetGridCursorRow()  # current selection
        selc = self.Report.GetGridCursorCol()  # current selection
        seltl = self.Report.GetSelectionBlockTopLeft()  # current selection
        selbr = self.Report.GetSelectionBlockBottomRight()  # current selection

        if debug:
            print 'selcol', selcol
            print 'selrow', selrow
            print 'selcell', selcell
            print 'cursor', selr, selc
            print 'seltl', seltl
            print 'selbr', selbr

        cx = self.Report.table.columns
        ox = self.Report.table.coloffset
        rx = self.Report.table.rows
        if not (cx and rx):
            hint("There are no cells to copy yet.")
            return

        if selcol: # columns selected
            r = range(len(rx))
            c = selcol
        elif selrow:
            r = selrow
            c = range(len(cx))
        elif seltl and selbr:
            r = range(seltl[0][0], selbr[0][0] + 1)
            c = range(seltl[0][1], selbr[0][1] + 1)
        else:
            r = [ selr ]
            c = [ selc ]

        def stringval(x):
            if isinstance(x, int) or isinstance(x, float):
                return str(x)
            elif isinstance(x, str):
                if x.count('\t'): x = x.replace('\t', r'\t')
                if x.count('\n'): x = x.replace('\n', r'\n')
                if x.count('\r'): x = x.replace('\r', r'\r')
                return x
            else:
                return x

        values = [ [ stringval(Data.GetCellValue(rx[ri], cx[ci], ox[ci])) for ci in c ] for ri in r ]

        if debug: print values

        s = os.linesep.join( [ '\t'.join(row) for row in values ] )

        if len(values) > 1 or len(values[0]) > 1:
            s += os.linesep

        if debug: print s

        # s = 'for the clipboard\tmore\tmore' + os.linesep + '1\t2\t3'
        # s = 'for the clipboard\tmore\tmore' + '\r' + '1\t2\t3' + '\r'
        cbData = wx.PyTextDataObject()
        cbData.SetText(s)

        if wx.TheClipboard.Open():
            wx.TheClipboard.SetData(cbData)
            wx.TheClipboard.Flush()
            wx.TheClipboard.Close()

    # -- taken from "Paste Cells From Clipboard.py" --
    # the following two helper functions edit the grid without calling SetUndo
    # (we want to be able to undo the entire pasting at once)

    def _CreateRow(self):
        """ Add a row to the end of the report. """
        r = Data.Report[self.ReportID]
        rt = Data.ReportType[r['ReportTypeID']]
        ta = rt.get('TableA')
        if not ta or ta in ('Report', 'ReportColumn', 'ReportRow', 'ReportType', 'ColumnType'): return  # need special handling
    
        change = {'Table': ta, 'Name': '--'}
        if ta != 'Project':
            change['ProjectID'] = r.get('ProjectID')
        undo = Data.Update(change)
        newid = undo['ID']

        change = {'Table': ta, 'ID': newid, 'Name': _(ta) + " " + str(newid)}
        undo = Data.Update(change)

        change = {'Table': 'ReportRow', 'ReportID': self.ReportID, 'TableName': ta, 'TableID': newid}
        undo = Data.Update(change)  # created here to control where inserted
        newrowid = undo['ID']
    
        rlist = Data.GetRowList(self.ReportID)
        rlist.append(newrowid)
        Data.ReorderReportRows(self.ReportID, rlist)
        Data.Recalculate()
    
    def _SetCellValue(self, row, col, value):
        table = self.Report.table
        rowid = table.rows[row]
        of = table.coloffset[col]
        colid = table.columns[col]
        column = Data.SetCellValue(rowid, colid, of, value)

    def OnPaste(self, event):
        if not isinstance(self.Report, wx.grid.Grid):
            hint("Run this command from a grid report.")
            return
        if not self.Report.GetNumberCols():
            hint("Insert columns first.")
            return

        # open clipboard
        data = wx.PyTextDataObject()
        if not wx.TheClipboard.Open():
            hint("Couldn't open clipboard.")
            return
        wx.TheClipboard.GetData(data)
        wx.TheClipboard.Close()
        cells = data.GetText()

        # decide where to paste
        if self.Report.GetNumberRows():
            sel = self.Report.GetSelectedRows()
            if sel:
                row = min(sel)
            else:
                row = self.Report.GetGridCursorRow()

            sel = self.Report.GetSelectedCols()
            if sel:
                column = min(sel)
            else:
                column = self.Report.GetGridCursorCol()
        else:
            row = column = 0

        # paste the cells
        for record in cells.splitlines():
            if row >= self.Report.GetNumberRows():  # need more rows
                 self._CreateRow()
                 if row >= self.Report.GetNumberRows():  # couldn't add a row
                     break
            c = column
            for value in record.split('\t'):
                 if c >= self.Report.GetNumberCols():
                     break
                 # self.Report.SetCellValue(row, c, value)
                 self._SetCellValue(row, c, value)
                 c += 1
            row += 1

        Data.SetUndo("Paste Cells")

    def doBringWindow(self, event):
        Menu.doBringWindow(self, event)

    def doQuick(self, event):
        Menu.doQuick(self, event)

    def doORMQuick(self, event):
        Menu.doORMQuick(self, event)

    def doShort(self, event):
        Menu.doShort(self, event)

    # ----------------- frame events

    def OnSize(self, event):
        size = event.GetSize()
        r = Data.Database['Report'][self.ReportID]
        r['FrameSizeW'] = size.width
        r['FrameSizeH'] = size.height

        event.Skip()  # to call default handler; needed?

    def OnMove(self, event):
        pos = event.GetPosition()
        if pos.x > 0 and pos.y > 0:
            r = Data.Database['Report'][self.ReportID]
            if Data.platform == "win":  # else windows don't open in the right place
                r['FramePositionX'] = pos.x - 4
                r['FramePositionY'] = pos.y - 50
            else:
                r['FramePositionX'] = pos.x
                r['FramePositionY'] = pos.y

        event.Skip()  # needed?

    def OnActivate(self, event):
        if event.GetActive():
            Data.SetActiveReport(self.ReportID)
        else:
            self.Report.SaveEditControlValue()
            self.Report.DisableCellEditControl()
        event.Skip()

    # ---------------- Column/Row resizing
    def OnRowSize(self, evt):
        pass

    def OnColSize(self, evt):
        if debug: print "OnColSize", (evt.GetRowOrCol(), evt.GetPosition())
        if debug: print "get col width", self.Report.GetColSize(evt.GetRowOrCol())
        newsize = self.Report.GetColSize(evt.GetRowOrCol())
        col = evt.GetRowOrCol()
        if self.Report.table.coloffset[col] == -1:  # not a gantt column
            colid = self.Report.table.columns[col]
            change = { 'Table': 'ReportColumn', 'ID': colid, 'Width': newsize }
            Data.Update(change, 0)  #  --------------------- don't allow Undo (until I can figure out how)
            # Data.SetUndo('Change Column Width')

        evt.Skip()

    # -------------- Make sure only the right tables are edited per column
    def OnEditorShown(self, evt):
        rtid = Data.Report[self.ReportID].get('ReportTypeID')

        rid = self.Report.table.rows[evt.GetRow()]
        rtable = Data.ReportRow[rid].get('TableName')

        cid = self.Report.table.columns[evt.GetCol()]
        ctid = Data.ReportColumn[cid].get('ColumnTypeID')
        which = Data.ColumnType[ctid].get('T') or 'Z'  # can be 'A', 'B', or 'X'

        ctable = Data.ReportType[rtid].get('Table' + which)  # 'X' should always yield 'None'

        if which != 'X' and ((not ctable) or rtable != ctable):
            evt.Veto()
            return
        self.Edit.Enable(wx.ID_COPY, False)
        self.Edit.Enable(wx.ID_PASTE, False)

    def OnEditorHidden(self, evt):
        self.Edit.Enable(wx.ID_COPY, True)
        self.Edit.Enable(wx.ID_PASTE, True)

    # -----------
    def SetReportTitle(self):
        rname = Data.Report[self.ReportID].get('Name') or "-"
        pid = Data.Report[self.ReportID].get('ProjectID')
        if Data.Project.has_key(pid):
            pname = Data.Project[pid].get('Name') or "-"
        else:
            pname = "-"
        title = pname + " / " + rname
        if self.GetTitle() != title:
            self.SetTitle(title)
            Menu.UpdateWindowMenuItem(self.ReportID)

    def UpdatePointers(self, all=0):  # 1 = new database; 0 = changed report rows or columns
        if debug: print "Start GanttReport UpdatePointers"

        # don't refresh a report if the underlying report record is invalid
        if all or not Data.Report[self.ReportID].get('ReportTypeID'):
            Data.CloseReport(self.ReportID)
            return

        sr = self.Report.table
        rlen, clen = len(sr.rows), len(sr.columns)

        select_rows = {}
        for x in self.Report.GetSelectedRows():
             rowid = sr.rows[x]
             select_rows[rowid] = None
        select_columns = {}
        for x in self.Report.GetSelectedCols():
             colid = sr.columns[x]
             select_columns[colid] = None

        r = self.Report.GetGridCursorRow()
        try:
            rid = sr.rows[r]
        except IndexError:
            rid = 0

        sr.UpdateColumnPointers()
        sr.UpdateRowPointers()

        self.Report.BeginBatch()

        if rlen != len(sr.rows) or clen != len(sr.columns):
            self.Report.Reset()  # tell grid that the number of rows or columns has changed
        else:
            self.Report.UpdateAttrs()

        self.Report.ClearSelection()
        for x, rowid in enumerate(sr.rows):
             if rowid in select_rows:
                 self.Report.SelectRow(x, True)
        for x, colid in enumerate(sr.columns):
             if colid in select_columns:
                 self.Report.SelectCol(x, True)

        self.Report.EndBatch()

        try:
            r = self.Report.table.rows.index(rid)
        except:
            pass
        else:
            c = self.Report.GetGridCursorCol()
            old_r = self.Report.GetGridCursorRow()
            if r != old_r:
                self.Report.SetGridCursor(r, c)
                self.Report.MakeCellVisible(r, c)

        if debug: print "End GanttReport UpdatePointers"

#---------------------------------------------------------------------------

if __name__ == '__main__':
    app = wx.PySimpleApp()
    frame = GanttReportFrame(3, None, -1, "")  # reportid = 3
    frame.Show(True)
    app.MainLoop()

if debug: print "end GanttReport.py"
