#!/usr/bin/env python
# Implement shared menu commands

# Copyright 2004, 2006, 2008 by Brian C. Christensen

#    This file is part of GanttPV.
#
#    GanttPV is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    GanttPV is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with GanttPV; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# 040417 - first version of this program; moved AdjustMenus here from Data
# 040419 - Setup Undo and Redo to use the commands from Data
# 040420 - moved doClose back to individual reports; drafted toolbar adjustment for Main
# 040424 - added scripts menu processing
# 040427 - onHide and onShowHidden moved from Main
# 040503 - added Undo message text to Undo and Redo menus
# 040510 - fixed doOpen & doExit; changed AdjustMenus to adjust GanttReport toolbar
# 040517 - added Bitmap
# 040518 - changed doScript so that it will look in the default location if ScriptDirectory is not defined; added doHome
# 040519 - show Assign Resources and Assign Prerequisite only if single task selected; revise logic to enable Main's buttons
# 040520 - Added final slash to Help and Forum URL's
# 040605 - changed URL directory names to lower case
# 040715 - Pierre_Rouleau@impathnetworks.com: removed all tabs, now use 4-space indentation level to comply with Official Python Guideline.
# 041013 - in Save As set title of main window to file name; in New set title of main window to "Main".
# 050311 - add OGL import
# 050314 - add xmlrpclib import
# 050409 - Alex - implemented multi-level Scripts menu: AddMenuTree, SearchDir, and LinearTraversal functions; ScriptMenu and ScriptPath arrays.
# 050503 - Alex - implemented Window menu for easy window-switching
# 050504 - Alex - moved some menu event-handling here from GanttPV.py and GanttReport.py; centered dialogs on screen rather than current report.
# 050504 - enable assign dependency when selection > 1
# 050515 - Alex - fixed a bug in the Window menu that prevented the application build from starting
# 050527 - Alex - sort Script menu-items (required for Windows platform)
# 050601 - Alex - fixed a bug in the order of dependency chaining
# 060211 - Alex - added "Revert" to File menu
# 060401 - Alex - added RefreshScriptMenu and ChangeScriptFolder
# 060520 - Alex - renamed "Find Scripts" to "Change Script Folder" and always prompt for the folder
# 060704 - Brian - adjust new menu items for 0.7
# 060709 - Brian - implement quick start menus
# 060711 - Brian - added import of Zeroconf for script use (moved here from GanttPV.py)
# 060817 - Brian - added import of xml.dom.minidom to support MS Project XML import script
# 061127 - Brian - enable scroll to task for only task and assignment rows
# 061207 - Brian - use tabs in quick start window
# 061229 - Brian - added edit project/report name to edit menu
# 070221 - Brian - enable hierachy menu commands if column type exists
# 070725 - Brian - interface changed: add code to convert GetMenuItems reply to a list
# 070926 - Brian - Data.FileName not set to None on New (fixed again? see 040815 in Data)
# 080519 - Brian - separate logic for DiagramFrames
# 080720 - Alex - integrated the script shorcuts into Data.Option

import wx, os, webbrowser
import wx.html
import Data, ID, UI, QuickStart
# imports to make modules available to scripts
import xmlrpclib  # to call xml server
import wx.lib.ogl as ogl  # for network diagrams
import Zeroconf
import xml.dom.minidom # for MS Project XML import
import wx.py  # for pyshell, etc.

# import Main, GanttReport
# import inspect

debug = Data.debug
if debug: print "load Menu.py"

# --------- File Menu ------------

def doNew(event):
    """ Respond to the "New" menu command. """
    if debug: print "Start doNew"
    if Data.AskIfUserWantsToSave("creating new project file"):
        if debug: print "Setup empty database"
        Data.CloseReports()  # close all open reports except #1
        Data.SetEmptyData()
        Data.MakeReady()
    if debug: print "End doNew"

def doOpen(event):
    """ Respond to the "Open" menu command. """
    if not Data.AskIfUserWantsToSave("opening a project file"): return

    curDir = os.getcwd()
    fileName = wx.FileSelector("Open File",
        default_extension=".ganttpv",
        wildcard="*.ganttpv",
        flags = wx.OPEN | wx.FILE_MUST_EXIST)
    if fileName == "": return
    fileName = os.path.join(os.getcwd(), fileName)
    os.chdir(curDir)

    Data.OpenFile(fileName)

# I'm going to leave the close logic in each frame for the present
# def doClose(reportid):
#     """ Respond to the "Close" menu command. """
#    if reportid == 1:
#        doExit()
#    else:
#        Data.CloseReport(reportid == 1)

def doCloseReports(event):
    Data.CloseReports()

def doSave(event):
    """ Respond to the "Save" menu command. """
    if Data.FileName:
        Data.SaveContents()
    else:
        doSaveAs(event)

def doSaveAs(event):
    """ Respond to the "Save As" menu command. """
    curDir = os.getcwd()
    fileName = wx.FileSelector("Save File As", "Saving",
        default_filename='Untitled.ganttpv',
        default_extension="ganttpv",
        wildcard="*.ganttpv",
        flags = wx.SAVE | wx.OVERWRITE_PROMPT)
    if not fileName: return # User cancelled.
    fileName = os.path.join(os.getcwd(), fileName)
    os.chdir(curDir)

    if Data.SaveContents(fileName):
        Data.FileName = fileName
        Data.RefreshReports()

def doAutoSaveOnQuit(event):
    Data.Option['AutoSaveOnQuit'] = not Data.Option.get('AutoSaveOnQuit')
    Data.SaveOption()
    for k, v in Data.OpenReports.items():
        AdjustMenus(v)

def doExit(event):
    """ Respond to the "Quit" menu command. """
    if Data.ChangedData and Data.Option.get('AutoSaveOnQuit') and Data.FileName:
        Data.SaveContents()
    else:
        if not Data.AskIfUserWantsToSave("exiting GanttPV"): return

    if debug: print "Continue doExit"
    Data.SaveOption()
    Data.CloseReports()
    Data.CloseReport(1)
    App = wx.GetApp()
    if App:
        App.ExitMainLoop()

def doRevert(event):
    """ Respond to the "Revert" menu command. """
    if not Data.FileName: return

    if Data.ChangedData:
        response = wx.MessageBox(
            "Save changes to new file before reverting to prior version?",
            "Confirm", wx.YES_NO | wx.CANCEL)
        if response == wx.YES:
            path = wx.FileSelector("Save File As", "Saving",
                default_filename='Untitled.ganttpv',
                default_extension="ganttpv",
                wildcard="*.ganttpv",
                flags = wx.SAVE | wx.OVERWRITE_PROMPT)
            if not path: return
            Data.SaveContents(path)
        elif response == wx.CANCEL:
            return

    Data.LoadContents()

def doRevertOpen(event):
    """ Respond to the "Revert To Opened" menu command. """
    while Data.UndoStack:
        Data.DoUndo()

# ---------- Edit Menu -------

def doUndo(event):
    """ Respond to the "Undo" menu command. """
    Data.DoUndo()

def doRedo(event):
    """ Respond to the "Undo" menu command. """
    Data.DoRedo()

def doEditName(table, id):
    """ Edit Project or Report name """
    dlg = wx.TextEntryDialog(None, _('New %s Name') % _(table), 'Edit %s Name' % _(table), 'Python')
    oldvalue = Data.Database[table][id].get('Name')
    if oldvalue == None: oldvalue = ""
    dlg.SetValue(oldvalue)
    if dlg.ShowModal() == wx.ID_OK:
        change = { 'Table': table, 'ID': id, 'Name': dlg.GetValue() }
        Data.Update(change)
        Data.SetUndo(_('Edit %s Name') % table)
    dlg.Destroy()    

# ------------ Script Menu -----------------

ScriptMenu = []  # nested list of item and submenu names
ScriptPath = []  # relative pathnames, arranged by script id

def doAddScripts(frame): 
    """ Initialize scripts menu. """
    if len(ScriptMenu) > 1:
        mb = frame.GetMenuBar()
        menuItem = mb.FindItemById(ID.FIND_SCRIPTS)
        menu = menuItem.GetMenu()
        menu.AppendSeparator()
        AddMenuTree(menu, ScriptMenu, ID.FIRST_SCRIPT)

default_shortcuts = {
    "Print Report": "CTRL-P",
    "Select ID": "CTRL-2",
    "Add Checklist Item to Task": "CTRL-3",
    "Sort By Priority": "CTRL-4",
    "Sort By Selected Column": "CTRL-5",
    "Save Backup Copy of Database": "CTRL-B",
    }

def AddMenuTree(menu, list, nextID):
    """ Add a nested list of items and submenus to menu.

    Increment nextID after each menu append.
    Return the final value of nextID.
    List format is [(ignored), [item], [submenu, [item], ...] ...]
    """
    shortcuts = Data.Option.get('Shortcuts', default_shortcuts)
    for i in list[1:]:
        label = i[0].replace(':', '/')
        if len(i) > 1:
            submenu = wx.Menu()
            nextID = AddMenuTree(submenu, i, nextID)
            menu.AppendMenu(nextID, label, submenu)
        else:
            if label in shortcuts:
                label += "\t" + shortcuts[label]
            menu.Append(nextID, label)
        nextID += 1
    return nextID

def SearchDir(path, maxDepth=0, showExtension=None, hidePrefix=None):
    """ Return a list of the files contained in path.

    Allow at most maxDepth sublevels.
    If showExtension is given, list only the files with that extension.
    Ignore files and directories whose names begin with hidePrefix.
    List format is [path, [file], [subpath, [file], ...] ...]
    """
    found = []
    if not os.path.exists(path):
        return found

    title = [os.path.basename(path)]
    if not os.path.isdir(path):
        return title

    contents = os.listdir(path)
    for i in contents:
        # conflict in merge -- guessed these three lines were right -- bcc
        name = os.path.basename(i)
        if name[:len('zzMeasure')] == 'zzMeasure': continue # ignore the zzMeasure folder
        if hidePrefix and name[:len(hidePrefix)] == hidePrefix:
        # conflict in merging changes -- guessed that these two lines were wrong -- bcc
        # if i[:len('zzMeasure')] == 'zzMeasure': continue  # ignore the zzMeasure folder
        # if hidePrefix and i[:len(hidePrefix)] == hidePrefix:
            continue
        fullpath = os.path.join(path, i)
        if maxDepth and os.path.isdir(fullpath):
            subfind = SearchDir(fullpath, maxDepth-1, showExtension, hidePrefix)
            if len(subfind) > 1:
                found.append(subfind)
        else:
            root, ext = os.path.splitext(i)
            if not showExtension or ext == showExtension:
                found.append([root])

    found.sort()
    return title + found

def LinearTraversal(list, verbose=False, path=None):
    """ Return a depth-first traversal of a nested list.

    If verbose, prepend an accumulated path to each list item.
    Nested format is [(ignored), [item1], [item2, [item3], ...] ...]
    Traversed format is [item1, item3, item2, ...]
    """
    traversal = []
    for i in list[1:]:
        if verbose and path:
            name = os.path.join(path, i[0])
        else:
            name = i[0]
        if len(i) > 1:
            sublist = LinearTraversal(i, verbose, name)
            traversal.extend(sublist)
        traversal.append(name)
    return traversal

def GetScriptNames():
    global ScriptMenu, ScriptPath
    if debug: print 'start GetScriptNames'
    sd = Data.GetScriptDirectory()
    ScriptMenu = SearchDir(sd, 2, '.py', '_')
    ScriptPath = LinearTraversal(ScriptMenu, True)
    if debug: print 'end GetScriptNames'

def doFindScripts(event):
    if debug: print "Start doFindScripts"
    ChangeScriptFolder()
    RefreshScriptMenu()
    if debug: print "End doFindScripts"

def doRefreshScripts(event):
    GetScriptNames()
    RefreshScriptMenu()

def RefreshScriptMenu():
    global prior_script_id
    prior_script_id = None
    # remove old scripts
    for v in Data.OpenReports.values():
        mb = v.GetMenuBar()
        item = mb.FindItemById(ID.FIND_SCRIPTS)
        menu = item.GetMenu()
        menuItems = list(menu.GetMenuItems())
        menuItems.reverse()
        for item in menuItems:
            if ID.FIRST_SCRIPT <= item.GetId() <= ID.LAST_SCRIPT:
                menu.RemoveItem(item)
            else:
                if item.IsSeparator():
                    menu.RemoveItem(item)
                break

    # add new scripts
    for v in Data.OpenReports.values():
        doAddScripts(v)

def ChangeScriptFolder(path=None):
    if not path:
        # prompt user for path
        curDir = os.getcwd()  # remember current directory
        dlg = wx.DirDialog(None, "Choose script folder",
            style = wx.DD_DEFAULT_STYLE|wx.DD_NEW_DIR_BUTTON)
        if dlg.ShowModal() == wx.ID_OK:
            path = dlg.GetPath()
        os.chdir(curDir)

    if path:
        if Data.Option['ScriptDirectory'] != path:
            Data.Option['ScriptDirectory'] = path
            Data.SaveOption()
        GetScriptNames() # even if directory is same, contents may have changed

prior_script_id = None  # save id of last script executed so that it can be rerun

def doRepeatScript(event):
    if debug: print "Start doRepeatScript"
    if prior_script_id is not None:
        id = prior_script_id
        sd = Data.GetScriptDirectory()
        script = os.path.join(sd, ScriptPath[id] + '.py')
        Data.RunScript(script)
    if debug: print "End RepeatScript"

def doScript(event):
    global prior_script_id
    if debug: print "Start doScript"
    id = event.GetId() - ID.FIRST_SCRIPT
    prior_script_id = id  # save for repeat command
    # display the script name that would be repeated
    for v in Data.OpenReports.values():
        mb = v.GetMenuBar()
        item = mb.FindItemById(ID.REPEAT_SCRIPT)
        item.SetHelp("Repeat '" + ScriptPath[id] + "'")
    sd = Data.GetScriptDirectory()
    script = os.path.join(sd, ScriptPath[id] + '.py')
    Data.RunScript(script)
    if debug: print "End doScript"

# ------------ Window Menu -----------------

WindowOrder = [(1, None)]  # [(report id, title), ...]

def SearchWindowOrder(reportid):
    index = 1
    while index < len(WindowOrder):
        if reportid == WindowOrder[index][0]:
            return index
        index += 1
    return 0

def ResetWindowMenus(): 
    """ Clear and rebuild every Window menu. """
    global ReportTitles, WindowOrder, ReportIds
    for frame in Data.OpenReports.itervalues():
        ClearWindowMenu(frame)
    WindowOrder = [(1, None)]
    for k in Data.OpenReports.iterkeys():
        UpdateWindowMenuItem(k)

def ClearWindowMenu(frame):
    """ Clear the frame's Window menu. """
    menu = frame.WindowMenu
    menuItems = list(menu.GetMenuItems())
    menuItems.reverse()
    for item in menuItems:
        if ID.FIRST_WINDOW < item.GetId() <= ID.LAST_WINDOW:
            menu.RemoveItem(item)
        else:
            if item.IsSeparator():
                menu.RemoveItem(item)
            break

def UpdateWindowMenuItem(reportid):
    """ Update one item in every Window menu. """
    if reportid == 1: return
    report = Data.OpenReports.get(reportid)
    if report:
        title = report.GetTitle()
    else:
        title = None

    index = SearchWindowOrder(reportid)
    if index:
        if title == WindowOrder[index][1]:
            return
        WindowOrder[index] = (reportid, title)
    else:
        if title == None:
            return
        index = len(WindowOrder)
        if index > ID.LAST_WINDOW - ID.FIRST_WINDOW:
            return
        WindowOrder.append((reportid, title))

    for v in Data.OpenReports.values():
        RefreshWindowMenuItem(v, index)

def FillWindowMenu(frame):
    """ Fill the frame's Window menu. """
    index = 1
    while index < len(WindowOrder):
        RefreshWindowMenuItem(frame, index)
        index += 1

def RefreshWindowMenuItem(frame, index):
    """ Refresh one item in the frame's Window menu. """
    if 0 < index < len(WindowOrder):
        reportid, title = WindowOrder[index]
    else:
        return

    id = index + ID.FIRST_WINDOW
    menu = frame.WindowMenu
    if menu.FindItemById(id):
        menu.Remove(id)
    elif title == None:
        # Nothing was changed.
        return

    menuItems = list(menu.GetMenuItems())
    if not title:
        # Nothing to add, but don't leave a hanging separator.
        if menuItems:
            lastitem = menuItems[-1]
            if lastitem.IsSeparator():
                menu.RemoveItem(lastitem)
        return

    pos = 0    
    while pos < len(menuItems):
        # Skip the fixed items and the main window.
        if menuItems[pos].GetId() > ID.FIRST_WINDOW:
            break
        pos += 1
    if not (pos > 0 and menuItems[pos-1].IsSeparator()):
        # Ensure a separator before the window names.
        menu.InsertSeparator(pos)
        menuItems = list(menu.GetMenuItems())
        pos += 1
    while pos < len(menuItems):
        # Insert in lexicographical order.
        if menuItems[pos].GetLabel() >= title:
            break
        pos += 1
    while pos < len(menuItems):
        if menuItems[pos].GetLabel() > title:
            break
        # Keep same-title items in the order they were opened.
        r = menuItems[pos].GetId() - ID.FIRST_WINDOW
        if SearchWindowOrder(r) > index:
            break
        pos += 1

    menu.InsertCheckItem(pos, id, title, "Bring this window to front")
    if reportid == frame.ReportID:
        menu.Check(id, True)

def doBringWindow(frame, event):
    id = event.GetId()

    mb = frame.GetMenuBar()
    check = not event.IsChecked()
    mb.Check(id, check)

    index = id - ID.FIRST_WINDOW
    if index < len(WindowOrder):
        reportid = WindowOrder[index][0]
        report = Data.OpenReports.get(reportid)
        if report:
            report.Raise()
            report.Show()

# ------------ Help Menu -----------------

def doHome(event):
    p = Data.platform[0]
    webbrowser.open('http://www.PureViolet.net/ganttpv/?v=0.11b%s' % p, new=2)

def doHelp(event):
    p = Data.platform[0]
    webbrowser.open('http://www.PureViolet.net/ganttpv/help/?v=0.11b%s' % p, new=2)

def doBook(event):
    p = Data.platform[0]
    webbrowser.open('http://www.SimpleProjectManagement.com/ganttpv/?v=0.11b%s' % p, new=2)

def doForum(event):
    p = Data.platform[0]
    webbrowser.open('http://www.SimpleProjectManagement.com/forum/?v=0.11b%s' % p, new=2)

class HelpNB(wx.Notebook):
    def __init__(self, parent, pages):  # pages is a list of [name, text] 
        wx.Notebook.__init__(self, parent, -1, # size=(21,21),
                             #style=
                             #wx.NB_TOP # | wx.NB_MULTILINE
                             #wx.NB_BOTTOM
                             #wx.NB_LEFT
                             #wx.NB_RIGHT
                             )

        for page in pages:
            name, text = page
            html = wx.html.HtmlWindow(self)
            if "gtk2" in wx.PlatformInfo:  # this was in the sample I copied
                html.SetStandardFonts()
            html.SetPage(text)
            # win = self.makeColorPanel(wx.BLUE)
            self.AddPage(html, name)
 
##        # Show how to put an image on one of the notebook tabs,
##        # first make the image list:
##        il = wx.ImageList(16, 16)
##        idx1 = il.Add(images.getSmilesBitmap())
##        self.AssignImageList(il)
##
##        # now put an image on the first tab we just created:
##        self.SetPageImage(0, idx1)


##        self.Bind(wx.EVT_NOTEBOOK_PAGE_CHANGED, self.OnPageChanged)
##        self.Bind(wx.EVT_NOTEBOOK_PAGE_CHANGING, self.OnPageChanging)


##    def makeColorPanel(self, color):
##        p = wx.Panel(self, -1)
##        win = ColorPanel.ColoredPanel(p, color)
##        p.win = win
##
##        def OnCPSize(evt, win=win):
##            win.SetSize(evt.GetSize())
##
##        p.Bind(wx.EVT_SIZE, OnCPSize)
##        return p

##    def OnPageChanged(self, event):
##        old = event.GetOldSelection()
##        new = event.GetSelection()
##        sel = self.GetSelection()
##        self.log.write('OnPageChanged,  old:%d, new:%d, sel:%d\n' % (old, new, sel))
##        event.Skip()
##
##    def OnPageChanging(self, event):
##        old = event.GetOldSelection()
##        new = event.GetSelection()
##        sel = self.GetSelection()
##        self.log.write('OnPageChanging, old:%d, new:%d, sel:%d\n' % (old, new, sel))
##        event.Skip()


# add code to remember how these were resized (in the option file?)
class NBFrame(wx.Frame):
    def __init__(self, parent, title, list):
        if Data.platform == "mac":
            wx.Frame.__init__(self, parent, -1, title, size=(600, 600))
        else:
            wx.Frame.__init__(self, parent, -1, title, size=(500, 600))
        html = HelpNB(self, list)

class HtmlFrame(wx.Frame):
    def __init__(self, parent, title, text):
        wx.Frame.__init__(self, parent, -1, title, size=(500, 600))
        html = wx.html.HtmlWindow(self)
        if "gtk2" in wx.PlatformInfo:
            html.SetStandardFonts()
        html.SetPage(text)

def doQuick(self, event):
    list = []

    name = "Overview"
    text = QuickStart.overview
    list.append([name, text])

    name = "Reports"
    text = QuickStart.report_window
    list.append([name, text])

    name = "Tasks"
    text = QuickStart.define_tasks
    list.append([name, text])

    name = "Resources"
    text = QuickStart.define_resources
    list.append([name, text])

    name = "Work Time"
    text = QuickStart.work_time
    list.append([name, text])

    name = "Scripts"
    text = QuickStart.scripts
    list.append([name, text])

    name = "Finding Help"
    text = QuickStart.finding_help
    list.append([name, text])

#    name = "Menus"
#    text = QuickStart.finding_help
#    list.append([name, text])

#    name = "ToolBar"
#    text = QuickStart.finding_help
#    list.append([name, text])

    hwin = NBFrame(self, "Quick Start", list)
    hwin.Show()

def doORMQuick(self, event):
    list = []

    name = "Overview"
    text = QuickStart.orm_overview
    list.append([name, text])

    name = "Method"
    text = QuickStart.orm_steps
    list.append([name, text])

    name = "Facts"
    text = QuickStart.orm_facts
    list.append([name, text])

    name = "Scripts"
    text = QuickStart.orm_scripts
    list.append([name, text])

    name = "Shortcuts"
    text = QuickStart.orm_diagram
    list.append([name, text])

    hwin = NBFrame(self, "ORM Quick Start", list)
    hwin.Show()

##def doQuick(self, event):
##    name = "Quick Start"
##    text = QuickStart.quick_start
###    hwin = wx.html.HtmlWindow(self, name=name)
##    hwin = HtmlFrame(self, name, text)
##    hwin.Show()

##def doShort(self, event):
##    name = "Short Cuts"
##    text = QuickStart.short_cuts
##    hwin = HtmlFrame(self, name, text)
##    hwin.Show()

def doShort(self, event):
    list = []

    name = "Built In"
    text = QuickStart.shortcuts_a
    list.append([name, text])

    name = "Tricks"
    text = QuickStart.shortcuts_b
    list.append([name, text])

    name = "Scripted"
    text = QuickStart.shortcuts_c
    list.append([name, text])

    hwin = NBFrame(self, "Shortcuts", list)
    hwin.Show()

def doShowAbout(event):
    """ Respond to the "About GanttPV" menu command. """
    dialog = UI.AboutDialog(None, -1, "About GanttPV")
    dialog.Centre()
    dialog.ShowModal()
    dialog.Destroy()

# -- adjust menus

def AdjustMenus(self):
    """ Adjust menus and toolbar to reflect the current state. """
    # if debug: print "Adjusting Menus"
    # if debug: print 'self', self
    canSave   = True # bool(Data.FileName)
    canRevert = bool(Data.FileName)
    canUndo   = bool(Data.UndoStack and isinstance(Data.UndoStack[-1], str))
    canRedo   = bool(Data.RedoStack and isinstance(Data.RedoStack[-1], str))

    # selection = len(self.selection) > 0
    # onlyOne   = len(self.selection) == 1
    # isText    = onlyOne and (self.selection[0].getType() == obj_TEXT)
    # front     = onlyOne and (self.selection[0] == self.contents[0])
    # back      = onlyOne and (self.selection[0] == self.contents[-1])

    # Enable/disable our menu items.

    self.FileMenu.Enable(wx.ID_SAVE, canSave)
    self.FileMenu.Check(ID.M_AUTO_SAVE, bool(Data.Option.get('AutoSaveOnQuit')))
    self.FileMenu.Enable(wx.ID_REVERT, canRevert)
    self.FileMenu.Enable(ID.M_REVERT_OPEN, canUndo)

    self.Edit.Enable(wx.ID_UNDO, canUndo)
    self.Edit.Enable(wx.ID_REDO, canRedo)
    mbar = self.GetMenuBar()
    undoItem = mbar.FindItemById(wx.ID_UNDO)
    redoItem = mbar.FindItemById(wx.ID_REDO)
    # if debug and canUndo: print 'Data.UndoStack[-1]', Data.UndoStack[-1]
    if canUndo: undoItem.SetText('Undo ' + Data.UndoStack[-1] + '\tCtrl-Z')
    else: undoItem.SetText('Undo\tCtrl-Z')
    if canRedo: redoItem.SetText('Redo ' + Data.RedoStack[-1] + '\tCtrl-Shift-Z')
    else: redoItem.SetText('Redo\tCtrl-Shift-Z')

    # self.editMenu.Enable(menu_DUPLICATE, selection)
    # self.editMenu.Enable(menu_EDIT_TEXT, isText)
    # self.editMenu.Enable(menu_DELETE, selection)

    # self.toolsMenu.Check(menu_SELECT, self.x == self.y)

    # Enable/disable our toolbar icons.
    if isinstance(self, UI.MainFrame):
        item = self.Report.currentItem
        if debug: "item", item
        isSel = (item != None) and (0 <= item < len(self.Report.rows))
        if debug: "isSel", isSel

        ## needed only if menus change depending on whether project or report is selected
        selReport = False; selProject = False
        if isSel:
            rowid = self.Report.rows[item]
            rr = Data.ReportRow[rowid]
            rowtable = rr['TableName']
            selReport = (rowtable == 'Report')
            selProject = (rowtable == 'Project')
        
        self.FileMenu.Enable(ID.M_OPEN_REPORT, selReport)
        self.FileMenu.Enable(wx.ID_CLOSE, False)  # this closes the application, user should quit instead

        self.Edit.Enable(ID.M_INSERT_PROJECT, True)
        self.Edit.Enable(ID.M_INSERT_REPORT, isSel)
        self.Edit.Enable(ID.M_DELETE_ROW, isSel)
        self.Edit.Enable(ID.M_MOVE_ROW_UP, isSel)
        self.Edit.Enable(ID.M_MOVE_ROW_DOWN, isSel)
        self.Edit.Enable(ID.M_EDIT_PROJECT_NAME, isSel)
        self.Edit.Enable(ID.M_EDIT_REPORT_NAME, selReport)

        self.main_toolbar.EnableTool(ID.NEW_PROJECT, True)
        self.main_toolbar.EnableTool(ID.NEW_REPORT, isSel)
        self.main_toolbar.EnableTool(ID.EDIT, isSel)
        self.main_toolbar.EnableTool(ID.DUPLICATE, isSel)
        self.main_toolbar.EnableTool(ID.DELETE, isSel)
        self.main_toolbar.EnableTool(ID.SHOW_HIDDEN_REPORT, True)

        self.main_toolbar.EnableTool(ID.MOVE_UP, isSel)
        self.main_toolbar.EnableTool(ID.MOVE_DOWN, isSel)

        self.main_toolbar.ToggleTool(ID.SHOW_HIDDEN_REPORT, bool(Data.Report[1].get('ShowHidden')))

    elif isinstance(self, UI.ReportFrame):
        r = Data.Report[self.ReportID]
        rtid = r['ReportTypeID']
        rt = Data.ReportType[rtid]
        ta = rt.get('TableA')
        tb = rt.get('TableB')
        toolbar = self.report_toolbar
        isTask = (ta == "Task")
        isResource = (ta == "Resource")
        isAssignment = (ta == "Assignment")
        csel = self.Report.GetSelectedCols()
        rsel = self.Report.GetSelectedRows()
        isOneRow = bool(len(rsel) == 1)
        isSelCol = bool(csel)
        isSelRow = bool(rsel)
        multiRow = isSelRow and not isOneRow
        isEditorShown = self.Report.IsCellEditControlEnabled()
        noRows = not self.Report.table.rows  # no cursor if no rows

        # self.report_toolbar.EnableTool(ID.INSERT_ROW, True)
        self.report_toolbar.EnableTool(ID.DUPLICATE_ROW, isSelRow and isTask)
        self.report_toolbar.EnableTool(ID.DELETE_ROW, isSelRow)
        self.report_toolbar.EnableTool(ID.MOVE_UP, isSelRow)
        self.report_toolbar.EnableTool(ID.MOVE_DOWN, isSelRow)

        # check if parenting is allowed
        if noRows: # multiRow or noRows:  # or no rows exist
            canChild = False
        else:
            if isOneRow:
                rindex = rsel[0]
            else:
                rindex = self.Report.GetGridCursorRow()
            row = Data.ReportRow[self.Report.table.rows[rindex]]
            tsel = row.get('TableName')
            canChild = False
            if tsel == ta:
                dc = Data.ReportColumn
                dct = Data.ColumnType
##                for col in Data.GetColumnList(self.ReportID):
##                    ctid = dc[col].get('ColumnTypeID')
##                    if ctid:
##                        cTypeName = dct[ctid].get('Name')
##                        if cTypeName == ta + 'ID':  # parent id column type is <tablename> + 'ID'
##                            canChild = True
##                            break
                # is a parent column type defined for this report's primary table?
                also = rt.get('Also')
                if also and also in Data.ReportType:  # use columns types define for primary table
                    search_rtid = also
                else:
                    search_rtid = rtid
                canChild = Data.SearchByColumn(Data.ColumnType, { 'ReportTypeID': search_rtid, 'Name': ta + 'ID' }, unique=True)

        # self.Edit.Enable(ID.M_INSERT_CHILD, canChild)
        self.Edit.Enable(ID.M_SHIFT_LEFT, canChild)
        self.Edit.Enable(ID.M_SHIFT_RIGHT, canChild)

        if isSelRow:
            row = Data.ReportRow[self.Report.table.rows[min(rsel)]]
            tsel = row.get('TableName')
            
            self.Edit.SetLabel(ID.M_INSERT_ROW, _("Insert New %s\tCtrl-R") % _(ta)) # should be "tsel"
            self.Edit.SetLabel(ID.M_INSERT_ROW_ABOVE, _("Insert New %s Above Selection\tCtrl-Shift-R") % _(ta)) # should be "tsel"

            self.Edit.SetHelpString(ID.M_INSERT_ROW, _("Insert new %s below the selected row") % _(ta)) # should be "tsel"
            self.Edit.SetHelpString(ID.M_INSERT_ROW_ABOVE, _("Insert a new %s above the selected row") % _(ta)) # should be "tsel"

            self.report_toolbar.SetToolShortHelp(ID.INSERT_ROW, _("Insert new %s") % _(ta))
            self.report_toolbar.SetToolShortHelp(ID.DUPLICATE_ROW, _("Duplicate %s") % _(ta))

            self.report_toolbar.SetToolLongHelp(ID.INSERT_ROW, _("Insert new %s below selection") % _(ta))
            self.report_toolbar.SetToolLongHelp(ID.DUPLICATE_ROW, _("Duplicate selected %s") % _(ta))

            canRelated = False # bool(tb and tb not in ('Assignment', 'ResourceGrouping', 'Dependency', 'MeasurementDependency', 'Report', 'ReportType', 'ColumnType', 'ReportColumn', 'ReportRow', 'ProjectWeek', 'ProjectDay', 'ProjectResourceWeek', 'ProjectResourceDay', 'TaskWeek'))
            self.Edit.Enable(ID.M_INSERT_RELATED, canRelated)

            # action menu
            self.Action.Enable(ID.M_ASSIGN_PREREQUISITE, tsel == 'Task' and isOneRow)
            self.Action.Enable(ID.M_LINK_TASKS, tsel == 'Task' and not isOneRow)  # should test to make sure all are tasks
            self.Action.Enable(ID.M_ASSIGN_RESOURCE, tsel == 'Task' and isOneRow)
            oneResource = tsel == 'Resource' and isOneRow
            canResourceGroup = 'ResourceGrouping' in Data.Database and 'ResourceGroup' in Data.Database
            isResourceGroup = bool(tsel == 'Resource' and Data.Database[tsel][row.get('TableID')].get('GroupType'))
            self.Action.Enable(ID.M_RESOURCE_TO_GROUP, oneResource and canResourceGroup and not isResourceGroup)
            self.Action.Enable(ID.M_DEFINE_RESOURCE_GROUP, oneResource and canResourceGroup and isResourceGroup)
            self.Action.Enable(ID.M_ASSIGN_TASK, oneResource)
            self.View.Enable(ID.M_SCROLL_TO_TASK, tsel in ('Task', 'Assignment'))
            self.report_toolbar.EnableTool(ID.SCROLL_TO_TASK, tsel in ('Task', 'Assignment'))

        else:
            self.Edit.SetLabel(ID.M_INSERT_ROW, _("Insert New %s\tCtrl-R") % _(ta))
            self.Edit.SetLabel(ID.M_INSERT_ROW_ABOVE, _("Insert New %s Above Selection\tCtrl-Shift-R") % _(ta))

            self.Edit.SetHelpString(ID.M_INSERT_ROW, _("Insert new %s below the selected row") % _(ta))
            self.Edit.SetHelpString(ID.M_INSERT_ROW_ABOVE, _("Insert a new %s above the selected row") % _(ta)) # should be "tsel"

            self.report_toolbar.SetToolShortHelp(ID.INSERT_ROW, _("Insert new %s") % _(ta))
            self.report_toolbar.SetToolShortHelp(ID.DUPLICATE_ROW, _("Duplicate %s") % '')

            self.report_toolbar.SetToolLongHelp(ID.INSERT_ROW, _("Insert new %s below selection") % _(ta))
            self.report_toolbar.SetToolLongHelp(ID.DUPLICATE_ROW, _("Duplicate selected %s") % '')

            # self.Edit.Enable(ID.M_INSERT_CHILD, False)
            self.Edit.Enable(ID.M_INSERT_RELATED, False)

            # action menu
            self.Action.Enable(ID.M_ASSIGN_PREREQUISITE, False)
            self.Action.Enable(ID.M_LINK_TASKS, False)
            self.Action.Enable(ID.M_ASSIGN_RESOURCE, False)
            self.Action.Enable(ID.M_RESOURCE_TO_GROUP, False)
            self.Action.Enable(ID.M_DEFINE_RESOURCE_GROUP, False)
            self.Action.Enable(ID.M_ASSIGN_TASK, False)
            self.View.Enable(ID.M_SCROLL_TO_TASK, False)
            self.report_toolbar.EnableTool(ID.SCROLL_TO_TASK, False)

        # self.Edit.SetLabel(ID.M_INSERT_CHILD, _("Insert Child %s") % _(ta))
        self.Edit.SetLabel(ID.M_INSERT_RELATED, _("Insert Related %s\tCtrl-Alt-R") % _(tb or " "))
        # self.Edit.SetHelpString(ID.M_INSERT_CHILD, _("Insert a child %s") % _(ta))
        self.Edit.SetHelpString(ID.M_INSERT_RELATED, _("Insert a new %s") % _(tb or " "))

        # self.Edit.Enable(ID.M_INSERT_ROW_ABOVE, isSelRow)
        self.Edit.Enable(ID.M_DELETE_ROW, isSelRow)
##        self.Edit.Enable(ID.M_MOVE_ROW_UP, isSelRow)
##        self.Edit.Enable(ID.M_MOVE_ROW_DOWN, isSelRow)
        self.Edit.Enable(ID.M_MOVE_ROW_UP, not noRows)  # what if there are no rows??
        self.Edit.Enable(ID.M_MOVE_ROW_DOWN, not noRows)

        self.report_toolbar.EnableTool(ID.PREREQUISITE, isTask and isSelRow)
        self.report_toolbar.EnableTool(ID.ASSIGN_RESOURCE, (isTask or isResource) and len(rsel) == 1)

        self.report_toolbar.EnableTool(ID.HIDE_ROW, isSelRow)
        self.report_toolbar.ToggleTool(ID.SHOW_HIDDEN, bool(r.get('ShowHidden')))

        self.View.Enable(ID.M_DELETE_COLUMN, isSelCol)
        self.View.Enable(ID.M_MOVE_COLUMN_LEFT, isSelCol)
        self.View.Enable(ID.M_MOVE_COLUMN_RIGHT, isSelCol)

        # self.report_toolbar.EnableTool(ID.INSERT_COLUMN, True)
        self.report_toolbar.EnableTool(ID.DELETE_COLUMN, isSelCol)
        self.report_toolbar.EnableTool(ID.MOVE_LEFT, isSelCol)
        self.report_toolbar.EnableTool(ID.MOVE_RIGHT, isSelCol)
        self.report_toolbar.EnableTool(ID.COLUMN_OPTIONS, False)
        # self.report_toolbar.EnableTool(ID.SCROLL_LEFT_FAR, True)
        # self.report_toolbar.EnableTool(ID.SCROLL_LEFT, True)
        # self.report_toolbar.EnableTool(ID.SCROLL_RIGHT, True)
        # self.report_toolbar.EnableTool(ID.SCROLL_RIGHT_FAR, True)

    elif isinstance(self, UI.DiagramFrame):
        r = Data.Report[self.ReportID]
        rtid = r['ReportTypeID']
        rt = Data.ReportType[rtid]
##        if rt.get('PanelType') == 'ORMDiagram': return  # IMPORTANT - fix this
        ta = rt.get('TableA')
        tb = rt.get('TableB')
##        toolbar = self.report_toolbar
##        isTask = (ta == "Task")
##        isResource = (ta == "Resource")
##        isAssignment = (ta == "Assignment")
##        csel = self.Report.GetSelectedCols()
##        rsel = self.Report.GetSelectedRows()
##        isOneRow = bool(len(rsel) == 1)
##        isSelCol = bool(csel)
##        isSelRow = bool(rsel)
##        multiRow = isSelRow and not isOneRow
##        isEditorShown = self.Report.IsCellEditControlEnabled()
##        noRows = not self.Report.table.rows  # no cursor if no rows

        # self.report_toolbar.EnableTool(ID.INSERT_ROW, True)
##        self.report_toolbar.EnableTool(ID.DUPLICATE_ROW, isSelRow and isTask)
##        self.report_toolbar.EnableTool(ID.DELETE_ROW, isSelRow)
##        self.report_toolbar.EnableTool(ID.MOVE_UP, isSelRow)
##        self.report_toolbar.EnableTool(ID.MOVE_DOWN, isSelRow)

        # check if parenting is allowed
##        if noRows: # multiRow or noRows:  # or no rows exist
##            canChild = False
##        else:
##            if isOneRow:
##                rindex = rsel[0]
##            else:
##                rindex = self.Report.GetGridCursorRow()
##            row = Data.ReportRow[self.Report.table.rows[rindex]]
##            tsel = row.get('TableName')
##            canChild = False
##            if tsel == ta:
##                dc = Data.ReportColumn
##                dct = Data.ColumnType
####                for col in Data.GetColumnList(self.ReportID):
####                    ctid = dc[col].get('ColumnTypeID')
####                    if ctid:
####                        cTypeName = dct[ctid].get('Name')
####                        if cTypeName == ta + 'ID':  # parent id column type is <tablename> + 'ID'
####                            canChild = True
####                            break
##                # is a parent column type defined for this report's primary table?
##                also = rt.get('Also')
##                if also and also in Data.ReportType:  # use columns types define for primary table
##                    search_rtid = also
##                else:
##                    search_rtid = rtid
##                canChild = Data.SearchByColumn(Data.ColumnType, { 'ReportTypeID': search_rtid, 'Name': ta + 'ID' }, unique=True)
##
##        # self.Edit.Enable(ID.M_INSERT_CHILD, canChild)
##        self.Edit.Enable(ID.M_SHIFT_LEFT, canChild)
##        self.Edit.Enable(ID.M_SHIFT_RIGHT, canChild)

##        if isSelRow:
##            row = Data.ReportRow[self.Report.table.rows[min(rsel)]]
##            tsel = row.get('TableName')
##            
##            self.Edit.SetLabel(ID.M_INSERT_ROW, _("Insert New %s\tCtrl-R") % _(ta)) # should be "tsel"
##            self.Edit.SetLabel(ID.M_INSERT_ROW_ABOVE, _("Insert New %s Above Selection\tCtrl-Shift-R") % _(ta)) # should be "tsel"
##
##            self.Edit.SetHelpString(ID.M_INSERT_ROW, _("Insert new %s below the selected row") % _(ta)) # should be "tsel"
##            self.Edit.SetHelpString(ID.M_INSERT_ROW_ABOVE, _("Insert a new %s above the selected row") % _(ta)) # should be "tsel"
##
##            self.report_toolbar.SetToolShortHelp(ID.INSERT_ROW, _("Insert new %s") % _(ta))
##            self.report_toolbar.SetToolShortHelp(ID.DUPLICATE_ROW, _("Duplicate %s") % _(ta))
##
##            self.report_toolbar.SetToolLongHelp(ID.INSERT_ROW, _("Insert new %s below selection") % _(ta))
##            self.report_toolbar.SetToolLongHelp(ID.DUPLICATE_ROW, _("Duplicate selected %s") % _(ta))
##
##            canRelated = False # bool(tb and tb not in ('Assignment', 'ResourceGrouping', 'Dependency', 'MeasurementDependency', 'Report', 'ReportType', 'ColumnType', 'ReportColumn', 'ReportRow', 'ProjectWeek', 'ProjectDay', 'ProjectResourceWeek', 'ProjectResourceDay', 'TaskWeek'))
##            self.Edit.Enable(ID.M_INSERT_RELATED, canRelated)
##
##            # action menu
##            self.Action.Enable(ID.M_ASSIGN_PREREQUISITE, tsel == 'Task' and isOneRow)
##            self.Action.Enable(ID.M_LINK_TASKS, tsel == 'Task' and not isOneRow)  # should test to make sure all are tasks
##            self.Action.Enable(ID.M_ASSIGN_RESOURCE, tsel == 'Task' and isOneRow)
##            oneResource = tsel == 'Resource' and isOneRow
##            canResourceGroup = 'ResourceGrouping' in Data.Database and 'ResourceGroup' in Data.Database
##            isResourceGroup = bool(tsel == 'Resource' and Data.Database[tsel][row.get('TableID')].get('GroupType'))
##            self.Action.Enable(ID.M_RESOURCE_TO_GROUP, oneResource and canResourceGroup and not isResourceGroup)
##            self.Action.Enable(ID.M_DEFINE_RESOURCE_GROUP, oneResource and canResourceGroup and isResourceGroup)
##            self.Action.Enable(ID.M_ASSIGN_TASK, oneResource)
##            self.View.Enable(ID.M_SCROLL_TO_TASK, tsel in ('Task', 'Assignment'))
##            self.report_toolbar.EnableTool(ID.SCROLL_TO_TASK, tsel in ('Task', 'Assignment'))
##
##        else:
##            self.Edit.SetLabel(ID.M_INSERT_ROW, _("Insert New %s\tCtrl-R") % _(ta))
##            self.Edit.SetLabel(ID.M_INSERT_ROW_ABOVE, _("Insert New %s Above Selection\tCtrl-Shift-R") % _(ta))
##
##            self.Edit.SetHelpString(ID.M_INSERT_ROW, _("Insert new %s below the selected row") % _(ta))
##            self.Edit.SetHelpString(ID.M_INSERT_ROW_ABOVE, _("Insert a new %s above the selected row") % _(ta)) # should be "tsel"
##
##            self.report_toolbar.SetToolShortHelp(ID.INSERT_ROW, _("Insert new %s") % _(ta))
##            self.report_toolbar.SetToolShortHelp(ID.DUPLICATE_ROW, _("Duplicate %s") % '')
##
##            self.report_toolbar.SetToolLongHelp(ID.INSERT_ROW, _("Insert new %s below selection") % _(ta))
##            self.report_toolbar.SetToolLongHelp(ID.DUPLICATE_ROW, _("Duplicate selected %s") % '')
##
##            # self.Edit.Enable(ID.M_INSERT_CHILD, False)
##            self.Edit.Enable(ID.M_INSERT_RELATED, False)
##
##            # action menu
##            self.Action.Enable(ID.M_ASSIGN_PREREQUISITE, False)
##            self.Action.Enable(ID.M_LINK_TASKS, False)
##            self.Action.Enable(ID.M_ASSIGN_RESOURCE, False)
##            self.Action.Enable(ID.M_RESOURCE_TO_GROUP, False)
##            self.Action.Enable(ID.M_DEFINE_RESOURCE_GROUP, False)
##            self.Action.Enable(ID.M_ASSIGN_TASK, False)
##            self.View.Enable(ID.M_SCROLL_TO_TASK, False)
##            self.report_toolbar.EnableTool(ID.SCROLL_TO_TASK, False)
##
##        # self.Edit.SetLabel(ID.M_INSERT_CHILD, _("Insert Child %s") % _(ta))
##        self.Edit.SetLabel(ID.M_INSERT_RELATED, _("Insert Related %s\tCtrl-Alt-R") % _(tb or " "))
##        # self.Edit.SetHelpString(ID.M_INSERT_CHILD, _("Insert a child %s") % _(ta))
##        self.Edit.SetHelpString(ID.M_INSERT_RELATED, _("Insert a new %s") % _(tb or " "))
##
##        # self.Edit.Enable(ID.M_INSERT_ROW_ABOVE, isSelRow)
##        self.Edit.Enable(ID.M_DELETE_ROW, isSelRow)
####        self.Edit.Enable(ID.M_MOVE_ROW_UP, isSelRow)
####        self.Edit.Enable(ID.M_MOVE_ROW_DOWN, isSelRow)
##        self.Edit.Enable(ID.M_MOVE_ROW_UP, not noRows)  # what if there are no rows??
##        self.Edit.Enable(ID.M_MOVE_ROW_DOWN, not noRows)
##
##        self.report_toolbar.EnableTool(ID.PREREQUISITE, isTask and isSelRow)
##        self.report_toolbar.EnableTool(ID.ASSIGN_RESOURCE, (isTask or isResource) and len(rsel) == 1)
##
##        self.report_toolbar.EnableTool(ID.HIDE_ROW, isSelRow)
##        self.report_toolbar.ToggleTool(ID.SHOW_HIDDEN, bool(r.get('ShowHidden')))
##
##        self.View.Enable(ID.M_DELETE_COLUMN, isSelCol)
##        self.View.Enable(ID.M_MOVE_COLUMN_LEFT, isSelCol)
##        self.View.Enable(ID.M_MOVE_COLUMN_RIGHT, isSelCol)
##
##        # self.report_toolbar.EnableTool(ID.INSERT_COLUMN, True)
##        self.report_toolbar.EnableTool(ID.DELETE_COLUMN, isSelCol)
##        self.report_toolbar.EnableTool(ID.MOVE_LEFT, isSelCol)
##        self.report_toolbar.EnableTool(ID.MOVE_RIGHT, isSelCol)
##        self.report_toolbar.EnableTool(ID.COLUMN_OPTIONS, False)
##        # self.report_toolbar.EnableTool(ID.SCROLL_LEFT_FAR, True)
##        # self.report_toolbar.EnableTool(ID.SCROLL_LEFT, True)
##        # self.report_toolbar.EnableTool(ID.SCROLL_RIGHT, True)
##        # self.report_toolbar.EnableTool(ID.SCROLL_RIGHT_FAR, True)


    # if debug: print 'Finished adjusting menus'

# ----------- common tool buttons -------

def onHide(self, event, sel):  # self is the object where 'rows' is stored
    """ Toggle Hide flag on report row """
    if debug: print "Start OnHide"
    for x in sel:   
        rrid = self.rows[x]  # item -> report row id
        rr = self.reportrow[rrid]  # report row id -> report row record
        oldval = rr.get('Hidden', False)
        if debug: print "old hide value", oldval
        change = { 'Table': 'ReportRow', 'ID': rrid, 'Hidden': not oldval }
        Data.Update(change)
        if debug: print "change", change
    Data.SetUndo('Hide Report Row')
    if debug: print "End OnHide"

def onShowHidden(self, event):  # self is the frame
    """ Toggle ShowHidden flag on report """
    if debug: print "Start OnShowHidden"
    rep = Data.Report[self.ReportID]
    change = {'Table':'Report', 'ID': self.ReportID, 'ShowHidden': not (rep.get('ShowHidden', False))}
    # if debug: print "change", change
    Data.Update(change)
    Data.SetUndo('Show Hidden Rows')
    if debug: print "End OnShowHidden"

def Bitmap(file, flags):
    """ Replacement for code that reads bitmaps in UI.py """
    parts = file.split('/') # separate out the file name
    path = os.path.join(Data.Path, "icons", parts[-1])
    # if debug: print 'path', path
    bm = wx.Bitmap(path, flags)
    # if not isinstance(bm, wx.Bitmap): bm = wx.NullBitmap
    return bm
    # return wx.Bitmap(os.path.join(Data.Path, "icons", "Report.bmp"), wx.BITMAP_TYPE_BMP)

def IconPath(path):
    filename = os.path.basename(path)
    path = os.path.join(Data.Path, "icons", filename)
    return path

if debug: print "end Menu.py"
