# Copyright 2007, 2008 by Brian C. Christensen

#    This file is part of GanttPV.
#
#    GanttPV is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    GanttPV is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with GanttPV; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# 070806 - First experiments with combining FlatNoteBook and OGL
# 070807 - Added pseudo dc window
# 070825 - Added separate sections for adding orm objext to the model and to the display
# 071128 - Ready do commit changes to standardize names to 'object type' and 'fact type'
#           Revise window layout within frame. Implement a draft of auto layout.
#           Implement multiple selection.
# 080129 - Some clean up before the demo
# 080517 - Moved script into GanttPV as ORM.py
# 080602 - Revised orm object model
# 080604 - Added Role entries to Fact Type popup menu
# 080605 - Will add and connect Role and Subtype constraints
# 080605 - Alex - workaround for bug in GetPixel on MacOSX; moved rolebox connector logic into the fact shape; replaced ctrl-click with cmd-click on Mac
# 080606 - Brian - display constraint symbols, connect correctly to adjacent pairs of roles
# 080606 - Alex - deontic constraints; three-way role constraints; shift-click on constraint to add role sequence; new icons
# 080617 - Alex - elegant algorithm for following a role connector; added pencil cursor
# 080716 - Alex - extended "delete role sequence" to all sequence numbers
# 080723 - Alex - graphics for ring constraints, simplify undo typing, delete role names with their connectors
# 080731 - Alex - separated role and ring constraints, simplified binding of menu events
# 090102 - Brian - separate canvas logic into separate file

from __future__ import division

import wx
# import images
import sys
import math  # to calculate angles of connectors
import random
import string
import os

import wx.lib.ogl as ogl
# ogl.OGLInitialize()  # after app object, but before OGL is used

import Data, ID, Menu

debug = 1

SelectedColor = 220, 220, 220
SelectedTextColor = 210, 230, 255


# workaround for bugs in wx.DC.GetPixel and wx.PsuedoDC.FindObjects

if Data.platform == "mac":
    def GetPixel(self, x, y):
        x, y = self.LogicalToDeviceX(x), self.LogicalToDeviceY(y)
        bitmap = self.GetAsBitmap(wx.Rect(x, y, 1, 1))
        image = wx.ImageFromBitmap(bitmap)
        rgba = image.GetRed(0, 0), image.GetGreen(0, 0), image.GetBlue(0, 0), image.GetAlpha(0, 0)
        return wx.Colour(*rgba)

    wx.DC.GetPixel = GetPixel

if Data.platform in ["mac", "linux"]:
    def FindObjects(self, x, y, hitradius=1, bgcolor=wx.WHITE):
        bitmap = wx.EmptyBitmap(1, 1)
        memdc = wx.MemoryDC(bitmap)
        memdc.SetDeviceOrigin(-x, -y)
        result = []
        for id in self.FindObjectsByBBox(x, y):
            memdc.SetBackground(wx.Brush(bgcolor))
            memdc.Clear()
            self.DrawIdToDC(id, memdc)
            if memdc.GetPixel(x, y) != bgcolor:
                result.append(id)
        return result
 
    wx.PseudoDC.FindObjects = FindObjects


# implement "any" function for lesser versions of python
try:
    any
except NameError:
    def any(iterable):
        for element in iterable:
            if element:
                return True
        return False


# geometric functions for positioning role names

def NearestPointOnLine(pointA, pointB, pointC):
    """ Find the nearest point on the line from A to B """
    x0, y0 = pointA
    x1, y1 = pointB
    x2, y2 = pointC
    if x0 == x1:
        return x0, y2
    dx, dy = x1 - x0, y1 - y0
    x = ( ((dx * dx) * x2 + (dx * dy) * (y2 - y0) + (dy * dy) * x0) /
          (dx * dx + dy * dy) )
    y = (dy / dx) * (x - x0) + y0
    return x, y

def dist(pointA, pointB):
    x0, y0 = pointA
    x1, y1 = pointB
    dx, dy = x1 - x0, y1 - y0
    return math.sqrt(dx * dx + dy * dy)

def SetLineFollowerMetrics(line, follower):
    """ Impossible to explain without drawing a diagram. """ 
    pointA, pointB = line.GetEnds()
    pointC = follower.GetPos()
    junction = NearestPointOnLine(pointA, pointB, pointC)
    parallel = dist(pointA, junction) / dist(pointA, pointB)
    orthagonal = dist(junction, pointC)
    # if follower is clockwise from the line, make orthagonal negative
    if ((pointB[1] > pointA[1] and pointC[0] < junction[0]) or
        (pointB[1] < pointA[1] and pointC[0] > junction[0]) or
        (pointB[0] > pointA[0] and pointC[1] > junction[1]) or
        (pointB[0] < pointA[0] and pointC[1] < junction[1])):
        orthagonal = -orthagonal
    follower._SetInShell('parallel', parallel)
    follower._SetInShell('orthagonal', orthagonal)

# list of variables that use SetInShell
# =================
# TempX, TempY, parallel, orthagonal
# followers, dcid, canvas
# IsSelected, Cursor, CharMode, InDelete
# Relation, Speed, Force, ForceR, ForceNR

#---------------------------------------------------------------------------
# Add object to model - independent of how it is drawn
#---------------------------------------------------------------------------

def AddNodeToDiagram(report_object, shape_type, x, y):
    ''' Add report row, add target object'''
    if debug: print report_object
    projectid = report_object.ProjectID
    today = Data.TodayString()

    object_type = shape_type[:-5]  # default name trims 'shape'
    model = {'ProjectID': projectid, 'DateAdded': today}
    if shape_type == 'ORMObjectTypeShape':
        model['Type'] = 'Entity'
    elif shape_type == 'ORMFactTypeShape':  # should point to a reading?
        model['Nary'] = 0  # number of roles
    elif shape_type == 'ORMNoteShape':
        pass
    elif shape_type == 'ORMConstraintShape':
        model['Operator'] = 'Unique'  # default
    elif shape_type == 'ORMSubtypeConstraintShape':
        model['Operator'] = 'Exclusive'  # default
    model['Table'] = object_type 
    object_id = Data.Update(model)['ID']

    graphic = {'Table': 'GraphicObject', 'ProjectID': projectid, 'ReportID': report_object.ID,
               'Subtype': shape_type,
               'TableName': object_type, 'TableID': object_id,
               'PosX': x, 'PosY': y,
               'Width': 5, 'Height': 5,  # temporary values
               'DateAdded': today}
    if shape_type == 'ORMFactTypeShape':
        graphic['Orientation'] = 0  # 0..3
    newrowid = Data.Update(graphic)['ID']

##    rlist = Data.GetRowList(report_object.ID)  # insert row at beginning of row list
##    rlist.append(newrowid)
##    Data.ReorderReportRows(report_object.ID, rlist)

    return report_object.db.GetObject('GraphicObject', newrowid)

def AddConnectorToDiagram(report_object, shape_type, nodea, nodeb):
    ''' Add report row, add target object, connect to end objects'''
    if debug: print 'adding %s to %s' % (shape_type, report_object)
    projectid = report_object.ProjectID
    today = Data.TodayString()

    # I think that subtype is the only example of linking the same types to each other

    object_type = shape_type[:-5]  # default name trims 'shape'
    model = {'ProjectID': projectid, 'DateAdded': today}
    model[nodea.TableName + 'ID'] = nodea.TableID
    if shape_type == 'ORMRoleConnectorShape':
        object_type = 'ORMRole'
        model['ORMObjectTypeID'] = nodeb.TableID
    elif shape_type == 'ORMSubtypeConnectorShape':  # what should this be called?
        model['ORMObjectTypeID'] = nodeb.TableID
        model['ORMSubtypeID'] = nodea.TableID
    elif shape_type == 'ORMNoteConnectorShape':  # may connect to anything
        model['TableName'] = nodeb.TableName
        model['TableID'] = nodeb.TableID
    else:
        model[nodeb.TableName + 'ID'] = nodeb.TableID
    model['Table'] = object_type
    object_id = Data.Update(model)['ID']
    if shape_type == 'ORMRoleConnectorShape':
        fact = fact_object = nodea.Target
        AddRole(fact, fact.db.GetObject('ORMRole', object_id))

    graphic = {'Table': 'GraphicObject', 'ProjectID': projectid, 'ReportID': report_object.ID,
               'Subtype': shape_type,
               'TableName': object_type, 'TableID': object_id,
               'NodeAID': nodea.ID, 'NodeBID': nodeb.ID,  # objects pointed to
               'PosX': -1, 'PosY': -1,
               'DateAdded': today}
    newrowid = Data.Update(graphic)['ID']

    return report_object.db.GetObject('GraphicObject', newrowid) 

def AddFollowerToDiagram(report_object, shape_type, nodea):
    ''' Add report row, add target object, connect to end objects'''
    if debug: print 'adding %s to %s' % (shape_type, report_object)
    projectid = report_object.ProjectID
    today = Data.TodayString()

    # I think that subtype is the only example of linking the same types to each other

    object_type = shape_type[:-5]  # default name trims 'shape'
    model = {'ProjectID': projectid, 'DateAdded': today}
    model[nodea.TableName + 'ID'] = nodea.TableID
    model['Table'] = object_type 
    if shape_type == 'ORMFactReadingShape':
        model2 = {'Table': 'ORMRoleSequence', 'Seq': 1,
                 'ProjectID': projectid, 'DateAdded': today}
        role_seq_id = Data.Update(model2)['ID']
        model['ORMRoleSequenceID'] = role_seq_id

        object_id = Data.Update(model)['ID']
        fact_object = nodea.Target
        fact_object.ORMFactReadingID = object_id  # pointer back
    elif shape_type == 'ORMRoleNameShape':  # is this really needed
        object_type = 'ORMRole'
        object_id = nodea.TableID   # same target as nodea
    elif shape_type == 'ORMSampleTableShape':  # is this really needed?
        object_type = nodea.TableName  # same target as nodea
        object_id = nodea.TableID
    elif shape_type == 'ORMObjectTypeShape':
        model['Type'] = 'Objectified'
        object_id = Data.Update(model)['ID']

    graphic = {'Table': 'GraphicObject', 'ProjectID': projectid, 'ReportID': report_object.ID,
               'Subtype': shape_type,
               'TableName': object_type, 'TableID': object_id,
               'NodeAID': nodea.ID, # 'NodeBID': nodeb.ID,  # objects pointed to
               'PosX': nodea.PosX + 10, 'PosY': nodea.PosY + 20,
               'DateAdded': today}
    if shape_type == 'ORMRoleNameShape':
        graphic['ORMRoleID'] = object_id
    newrowid = Data.Update(graphic)['ID']

    return report_object.db.GetObject('GraphicObject', newrowid) 

# -----------
class ORMShape(Data.Object):
    def __init__(self, *parms):
        Data.Object.__init__(self, *parms)
        self._SetInShell('followers', [])

##    def ClearFollowers(self):  # objects don't have this method soon enough
##        self._SetInShell('followers', [])
    def Follow(self, remove=None):  # identify which objects you follow
        pass  # overriden in subtypes
    def AddFollower(self, follower_shape):
        if not follower_shape in self.followers:
            self.followers.append(follower_shape)
    def RemoveFollower(self, follower_shape):
        if follower_shape in self.followers:
            self.followers.remove(follower_shape)
    def GetFollowers(self, filter=''):  # use this instead of accessing followers directly
        return [ x for x in self.followers if x.Valid() and (not filter or x.TableName == filter)]    
    def SetTempPos(self, x, y):
        self._SetInShell('TempX', x)
        self._SetInShell('TempY', y)
    def GetPos(self):
        return self.TempX or self.PosX or 100, self.TempY or self.PosY or 100  # return something reasonable
    def CommitPos(self):  # need error check - what if they don't exist
        # set official position
        self.PosX = self.TempX
        self.PosY = self.TempY
        
    def SetClipBox(self, box_x, box_y, box_w, box_h):  # here or in ORMBox?
        x, y = self.GetPos()
##        self.ClipX = box_x - x
##        self.ClipY = box_y - y
##        self.ClipW = box_w
##        self.ClipH = box_h
        change = {
            'Table': self.Table,
            'ID': self.ID,
            'ClipX': box_x - x,
            'ClipY': box_y - y,
            'ClipW': box_w,
            'ClipH': box_h,
            }
        Data.Update(change, 0)  # don't put this on the undo stack
    def GetClipBox(self):
        x, y = self.GetPos()
        return self.ClipX + x, self.ClipY + y, self.ClipW, self.ClipH
    def SetWithoutUndo(self, name, value):
        change = {
            'Table': self.Table,
            'ID': self.ID,
            name: value,
            }
        Data.Update(change, 0)  # don't put this on the undo stack

    def Char(self, char):
        pass
        # if debug: print "char", ord(char), "'", char, "'"

    def MoveCursor(self, keycode):
        if keycode in (wx.WXK_LEFT, wx.WXK_RIGHT):
            cursor = self.Cursor
            if cursor != None:
                if keycode == wx.WXK_LEFT:
                    cursor -= 1
                else:
                    cursor += 1
                self._SetInShell('Cursor', cursor)

    def InsertCursor(self, text):
        cursor = self.Cursor
        if cursor == None:
            cursor = len(text)
            self._SetInShell('Cursor', cursor)
        cursor = max(0, min(len(text), cursor))
        self._SetInShell('Cursor', cursor)
        return text[:cursor] + '|' + text[cursor:]

    def ApplyChar(self, char, text):
        if char == '\r': char = '\n'
        cursor = self.Cursor
        if text:
            if cursor == None: cursor = len(text)
            if char == '\x08':  # this is for windows, cross platform?
                if cursor > 0:
                    text = text[:cursor-1] + text[cursor:]  # delete 1 char
                    self._SetInShell('Cursor', cursor-1)
            else:
                text = text[:cursor] + char + text[cursor:]
                self._SetInShell('Cursor', cursor+1)
        else:
            if char in '\r\x08':  # this is for windows, cross platform?
                pass
            else:
                text = char
                self._SetInShell('Cursor', 1)
        return text

    def SetSelected(self):
        self._SetInShell('IsSelected', True)
        self._SetInShell('CharMode', '')
##        self._SetInShell('Cursor', None)
    def ClearSelected(self):
        self._SetInShell('IsSelected', False)

    def GetSelected(self):
        return self.IsSelected
    def NoneSelected(self):
        return not self.canvas.selection
    def Moveable(self):
        ''' if any nodes are selected, then only selected nodes are moveable'''
        return self.NoneSelected() or self.GetSelected()

    def SetFocus(self):  # object will receive chars from keyboard
        pass
    def ClearFocus(self):  # object will no longer receive chars from keyboard
        pass

    def OnBottom(self):  # overridden by ORMObjectTypeShape
        pass

    def IsUIDeletable(self):  # overriden in subclasses
        return False
    def Delete(self):  # overriden in subclasses
        self.Follow(remove=True)  # may be repeated, but shouldn't cause harm if so
        for shape in self.GetFollowers():
            shape.Delete()
        if self.dcid and self.canvas:  # if diagram is open
            try:
                self.canvas.DeleteDCID(self.dcid)
            except:
                pass
        # self._SetInShell('dcid', None)
        # self._SetInShell('canvas', None)
        Data.Object.Delete(self)

    # these are for the autolayout algorithm
    def SetRelation(self, intensity):  # get the intensity of the relationship -- CHANGE NAME
        self._SetInShell('Relation', intensity)
    def SetSpeed(self, vec):
        self._SetInShell('Speed', vec)
    def SetForce(self, vec):
        self._SetInShell('Force', vec)
    def SetForceR(self, vec):
        self._SetInShell('ForceR', vec)
    def SetForceNR(self, vec):
        self._SetInShell('ForceNR', vec)
    def InitPosition(self, vec):  # I should be able to create the vector here
        # is this necessary??
##        if 'TempX' not in self.__dict__:
##            self._SetInShell('TempX', self.PosX)
##        if 'TempY' not in self.__dict__:
##            self._SetInShell('TempY', self.PosY)
        # set value
        vec.x = self.TempX
        vec.y = self.TempY
        self._SetInShell('Position', vec)
    def SetPosition(self, vec):
        self._SetInShell('Position', vec)
##        self._SetInShell('TempX', vec.X())   # move object to new position?
##        self._SetInShell('TempY', vec.Y())
##    def CommitPosition(self):
##        vec = self.Position
##        self._SetInShell('TempX', vec.X())   # move object to new position?
##        self._SetInShell('TempY', vec.Y())
##        self.CommitPos()
    def AddForce(self, vec):
##        self.Force += vec  # same vector object -- NO, not implemented that way
        self.SetForce(self.Force + vec)  # same vector object
    def AddForceR(self, vec):
        self.SetForceR(self.ForceR + vec)  # same vector object
    def AddForceNR(self, vec):
        self.SetForceNR(self.ForceNR + vec)  # same vector object

class ORMBox(ORMShape):
    def AdjustEnd(self, other_end):  # use separate x & y or a tuple?
        '''x & y are for the other end of the line'''
        centerx, centery = self.GetPos()
        if self.canvas == None: return centerx, centery  # make sure diagram will load
        
        r = self.canvas.pdc.GetIdBounds(self.dcid)
        x, y = wx.lib.ogl.FindEndForBox(r.GetWidth(), r.GetHeight(), centerx, centery, other_end[0], other_end[1])
        return int(x), int(y)

class ORMCircle(ORMShape):
    def AdjustEnd(self, other_end):  # use separate x & y or a tuple?
        '''x & y are for the other end of the line'''
        r = self.canvas.pdc.GetIdBounds(self.dcid)
        # width == height
        centerx, centery = self.GetPos()
        x, y = wx.lib.ogl.FindEndForCircle(r.GetWidth()/2, centerx, centery, other_end[0], other_end[1])
        return int(x), int(y)

class ORMConnector(ORMShape):
    def AdjustEnd(self, other_end):  # use separate x & y or a tuple?
        '''x & y are for the other end of the line'''
#        r = self.canvas.pdc.GetIdBounds(self.dcid)
#        x, y = wx.lib.ogl.FindEndForBox(r.GetWidth(), r.GetHeight(), self.PosX, self.PosY, other_end[0], other_end[1])
#        return int(x), int(y)
        return self.GetPos()

    def GetEnds(self):
        x, y = self.GetPos()
        nodea, nodeb = self.NodeA, self.NodeB
        endb = nodeb.GetPos()
        enda = nodea.AdjustEnd(endb)
        endb = nodeb.AdjustEnd(enda)
        return enda, endb

    def AddFollower(self, follower_shape):
        if not follower_shape in self.followers:
            self.followers.append(follower_shape)
            SetLineFollowerMetrics(self, follower_shape)

    def Follow(self, remove=None):
        nodea = self.NodeA  # these objects might not already be in the diagram
        nodeb = self.NodeB
        if not remove:
            nodea.AddFollower(self)  # must be the right subtype to have this method
            nodeb.AddFollower(self)
        else:
            nodea.RemoveFollower(self)  # must be the right subtype to have this method
            nodeb.RemoveFollower(self)

class ORMFollowText(ORMBox):
    def Follow(self, remove=None):
        nodea = self.NodeA  # these objects might not already be in the diagram
        if not remove:
            nodea.AddFollower(self)  # must be the right subtype to have this method
        else:
            nodea.RemoveFollower(self)  # must be the right subtype to have this method

class ORMObjectTypeShape(ORMBox):
    def Follow(self, remove=None):
        nodea = self.NodeA
        if nodea:  # if objectified fact type
            if not remove:
                nodea.AddFollower(self)
            else:
                nodea.RemoveFollower(self)

    def OnBottom(self):
        nodea = self.NodeA
        if nodea:  # if objectified fact type
            nodea.OnTop()
        
    def CommitPos(self):  # if self moved while fact type was not selected
        ORMBox.CommitPos(self)
        if self.NodeA:
            self.NodeA.CommitPos()

    def Draw(self, dc):
        orm_object = self.Get('Target')
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)

        x, y = self.GetPos()

        left_margin = 6
        right_margin = 5
        top_margin = 4
        bottom_margin = 6

        text = orm_object.Name or 'Object Type %s' % self.TableID
        word = orm_object.RefMode or ''
        if self.IsSelected and self.dcid == self.canvas.keyboard_target_dcid:
            if self.CharMode == '':
                if orm_object.Name:
                    text = self.InsertCursor(text)  # add cursor to text for display
            else:
                word = self.InsertCursor(word)

        if orm_object.Independent:
            text += " !"
        if orm_object.Derived:
            text += " " + orm_object.Derived
        if self.NodeA:  # objectified fact type
            text = '"' + text + '"'
        lines = text.splitlines()
        if word:
            word = '(' + word + ')'
            lines.append(word)

        if len(lines) == 1:  # to make it look better
            # top_margin += 0
            bottom_margin -= 2

        sizes = [ self.canvas.GetFullTextExtent(line)[0:2] for line in lines ]  # pull out only w and h
        w = left_margin * 2
        h = 0
        for line_w,line_h in sizes:
            w = max(w, line_w)
            h += line_h

        if orm_object.Type in ('Entity','Objectified'):
            pen = self.canvas.CachedPen('Black', 1, wx.SOLID)
        else:
            pen = self.canvas.CachedPen('Black', 1, wx.DOT)
        dc.SetPen(pen)
        if self.GetSelected():
            dc.SetBrush(self.canvas.CachedBrush(SelectedColor))
        elif self.canvas.keyboard_nav_dcid == self.dcid:
            dc.SetBrush(self.canvas.CachedBrush((255, 102, 204))) # (150,220,220)))
        else:
            dc.SetBrush(self.canvas.CachedBrush((254, 254, 254)))
            # dc.SetBrush(self.canvas.CachedBrush('White'))

        if self.NodeA:  # objectified fact type
            fact_object = self.NodeA.Target
            box_w = role_box_width * fact_object.Nary + left_margin + right_margin + 4
            box_h = role_box_width + top_margin + bottom_margin

            box_x = x - box_w/2
            box_y = y - box_h/2 + 3
    #            dc.DrawRoundedRectangle(box_x,box_y,box_w,box_h,8)
            dc.DrawRoundedRectangle(box_x,box_y,box_w,box_h,5)
            start_text = top_margin + h  # start text this far above the box
            yy = box_y - start_text
        else:
            box_w = w + left_margin + right_margin
            box_h = h + top_margin + bottom_margin

            box_x = x - box_w/2
            box_y = y - box_h/2
    #            dc.DrawRoundedRectangle(box_x,box_y,box_w,box_h,8)
            dc.DrawRoundedRectangle(box_x,box_y,box_w,box_h,5)
            start_text = 0  # only for objectified fact types
            yy = box_y + top_margin

        dc.SetFont(self.canvas.GetFont())
        if orm_object.Name:
            dc.SetTextForeground('Black')
        else:
            dc.SetTextForeground((128, 128, 128))  # grey out default name
            if self.IsSelected and self.CharMode == '' and self.dcid == self.canvas.keyboard_target_dcid:
                dc.SetPen(wx.Pen((254, 254, 254), 2))
                dc.SetBrush(wx.Brush(SelectedTextColor))
                rect = wx.Rect(x - sizes[0][0]/2, yy, sizes[0][0], sizes[0][1])
                dc.DrawRectangleRect(rect)

        dc.SetTextBackground('White')
        for i in range(len(lines)):
            dc.DrawText(lines[i], x - sizes[i][0]/2, yy)  # center text in box
            # dc.DrawText(lines[1], x - sizes[1][0]/2, box_y+box_h/2+1)
            yy += sizes[i][1]
            dc.SetTextForeground('Black')


        width_adj = max(w - box_w, 0)  # if the text is wider that the box
        r = wx.Rect(box_x - width_adj //2,box_y - start_text,box_w + width_adj,box_h + start_text)
#        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)  # bounds inclose all of the drawing??

    def Char(self, char):
        if debug: print "char", ord(char), "'", char, "'"
        orm_object = self.Get('Target')

        if char == '(':
            self._SetInShell('CharMode', 'refmode')
            return
        elif char == ')':
            self._SetInShell('CharMode', '')
            return
        elif char == '!':
            if orm_object.Independent == '!':
                orm_object.Independent = ''
            else:
                orm_object.Independent = '!'
            return
        elif char in ('+', '*'):
            if orm_object.Derived == char:
                orm_object.Derived = ''
            else:
                orm_object.Derived = char
            return
        elif char in ('$'):
            if orm_object.Type == 'Entity':
                orm_object.Type = 'Value'
            else:
                orm_object.Type = 'Entity'
            return

        if self.CharMode == '':
            orm_object.Name = self.ApplyChar(char, orm_object.Name)
##            if orm_object.Name:
##                if char == '\x08':  # this is for windows, cross platform?
##                    orm_object.Name = orm_object.Name[:-1]
##                elif char == '\r':  # this is for windows, cross platform?
##                    orm_object.Name += '\n'
##                else:
##                    orm_object.Name += char
##            else:
##                if char in '\r\x08':  # this is for windows, cross platform?
##                    pass
##                else:
##                    orm_object.Name = char
        elif self.CharMode == 'refmode':
            orm_object.RefMode = self.ApplyChar(char, orm_object.RefMode)
##            if orm_object.RefMode:
##                if char == '\x08':  # this is for windows, cross platform?
##                    orm_object.RefMode = orm_object.RefMode[:-1]
##                elif char == '\r':  # this is for windows, cross platform?
##                    orm_object.RefMode += '\n'
##                else:
##                    orm_object.RefMode += char
##            else:
##                if char in '\r\x08':  # this is for windows, cross platform?
##                    pass
##                else:
##                    orm_object.RefMode = char

    def IsUIDeletable(self):  # allow user to delete this graphic object
        return True

    def Delete(self):
        if self.InDelete: return
        self._SetInShell('InDelete', True)
        self.Follow(remove=True)  # before they are deleted
        for shape in self.GetGraphicList('ORMRoleConnectorShape', 'NodeB'):
            shape.Delete()
        for shape in self.GetGraphicList('ORMSubtypeConnectorShape', 'NodeA'):
            shape.Delete()
        for shape in self.GetGraphicList('ORMSubtypeConnectorShape', 'NodeB'):
            shape.Delete()
        if self.Target:
            other_graphics = [ x for x in self.Project.GetGraphicList('ORMObjectTypeShape')
                           if x.Target == self.Target and not x == self ]
            if not other_graphics:
                self.Target.Delete()
        ORMShape.Delete(self)
        self._SetInShell('InDelete', None)

role_box_height = 8 # 12
role_box_width = 12 # 12

class ORMFactTypeShape(ORMBox):
    def AdjustEnd(self, other_end, seq=None):
        fact_object = self.Target
        reading_object = self.Target.ORMFactReading
        # I'm not sure whether the fact shape should point to the fact shape or the fact
        # reading. I've added these two definitions to each method to make it easier
        # to convert to pointing at the reading.

        nary = fact_object.Nary
        if not seq:
            seq = (nary + 1) / 2.0

        ax, ay = self.GetPos()
        bx, by = other_end
        if bx == None: bx = 100  # fix error that prevents load of diagram
        if by == None: by = 100
        
        midway = ay + role_box_height / 2
        right = ax + role_box_width * seq

        if seq == 1 and bx < ax and abs(bx - ax) > abs(by - midway):
            ay = midway
        elif seq == nary and right < bx and abs(bx - right) > abs(by - midway):
            ax = right
            ay = midway
        else:
            ax += role_box_width * seq - role_box_width / 2
            if by > midway:
                ay += role_box_width
        return ax, ay

    def GetCenter(self):
        fact_object = self.Target
        reading_object = self.Target.ORMFactType

        x, y = self.GetPos()
        y += role_box_height // 2
        x += role_box_width * fact_object.Nary // 2
        return x, y

    def Draw(self, dc):
        dc.ClearId(self.dcid)
        fact_object = self.Target
        if not fact_object: return
        reading_object = self.Target.ORMFactReading
        if not reading_object: return  # object has probably been undone
        dc.SetId(self.dcid)

        x, y = self.GetPos()

        prior_r = self.canvas.pdc.GetIdBounds(self.dcid)
        if debug: print "prior rectangle", prior_r
        nary = fact_object.Nary or 1
        alignment = self.Orientation or 0
        # eventually allow any of 4 orientations (90 degree rotations)
        if alignment == 0:
            w = (role_box_width) * (nary)
            h = role_box_height
        box_w = w + 6
        box_h = h + 6

        pen = self.canvas.CachedPen('Black', 1, wx.SOLID)
        dc.SetPen(pen)
        if self.GetSelected():
            dc.SetBrush(self.canvas.CachedBrush(SelectedColor))
        elif self.canvas.keyboard_nav_dcid == self.dcid:
            dc.SetBrush(self.canvas.CachedBrush((255, 102, 204))) # (150,220,220)))
        else:
            dc.SetBrush(self.canvas.CachedBrush((254, 254, 254)))
            # dc.SetBrush(self.canvas.CachedBrush('White'))
        for i in range(nary):
            dc.DrawRectangle(x+i*role_box_width,y + 3,role_box_width+1,h)

        role_unique = False
        role_unique_alethic = False
        # roles = fact_object.GetList('ORMRole')
        roles = reading_object.ORMRoleSequence.GetList('ORMRolePosition')
        roles.sort(cmp=lambda x,y: cmp(x.Seq, y.Seq))
        for i, r in enumerate(roles):
##            print 'role', r
##            print 'rolelist', self.canvas.rolelist
            if r.ORMRole in self.canvas.rolelist:      # selection numbers to put in role boxes
                seq = self.canvas.rolelist.index(r.ORMRole)
                dc.SetFont(self.canvas.GetFont())
                dc.SetTextForeground('Black')
                dc.SetTextBackground('White')
                offsetx = i*role_box_width + 2
                dc.DrawText(str(seq+1), x + offsetx, y)

            if roles[i].ORMRole.Unique in ('a', 'p', 'd'):    # role unique bars
                role_unique = True
                if roles[i].ORMRole.Unique in ('a', 'p'):
                    role_unique_alethic = True
                    color = 'Violet'
                else:  # 'd'
                    color = 'Blue'
                pen = self.canvas.CachedPen(color, 1, wx.SOLID)
                dc.SetPen(pen)
                x0 = x+i*role_box_width + 2
                y0 = y
                x1 = x0 + role_box_width - 3
                y1 = y
                dc.DrawLine(x0, y0, x1, y1)

        role_spans = 0
        y0 = y
        if role_unique: y0 -= 2
        y1 = y0
        for i in range(len(roles)):
            if roles[i].ORMRole.UniqueOther in ('ax', 'px', 'dx'):
                if roles[i].ORMRole.UniqueOther in ('ax', 'px'):
                    role_spans += 1
                    color = 'Violet'
                else:  # 'dx'
                    color = 'Blue'
                pen = self.canvas.CachedPen(color, 1, wx.SOLID)
                penx = self.canvas.CachedPen(color, 1, wx.DOT)
                dc.SetPen(pen)
                if i == 0:
                    x0 = x + role_box_width
                    x1 = x0 + role_box_width * (len(roles)-1)
                    dc.DrawLine(x0, y0, x1, y1)
                elif i == len(roles)-1:
                    x0 = x
                    x1 = x0 + role_box_width * (len(roles)-1)
                    dc.DrawLine(x0, y0, x1, y1)
                else:
                    x0 = x
                    x1 = x0 + role_box_width * i
                    dc.DrawLine(x0, y0, x1, y1)

                    x0 = x1 + 2  # start with a gap   
                    x1 = x0 + role_box_width - 4  # end with gap
                    dc.SetPen(penx)
                    dc.DrawLine(x0, y0, x1, y1)
                    
                    x0 = x1 + 2  # after gap
                    x1 = x0 + role_box_width * (len(roles)-i-1)
                    dc.SetPen(pen)
                    dc.DrawLine(x0, y0, x1, y1)
                y0 = y0 - 2
                y1 = y0

        if not role_unique_alethic and not role_spans and fact_object.Unique:
            pen = self.canvas.CachedPen('Violet', 1, wx.SOLID)
            dc.SetPen(pen)
            x0 = x+ 1
##            y0 = y
            x1 = x0 + len(roles) * role_box_width - 2
            y1 = y0
            dc.DrawLine(x0, y0, x1, y1)

        r = wx.Rect(x,y,box_w,box_h + 3)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)

        r = self.canvas.pdc.GetIdBounds(self.dcid)
        if debug: print "new rectangle", r

    def Char(self, char):
        # if no reading object, create one
        fact_object = self.Target
        reading_object = self.Target.ORMFactReading

        if char in ('+', '*'):
            if fact_object.Derived == char:
                fact_object.Derived = ''
            else:
                fact_object.Derived = char
            return

        if reading_object.Reading:
            if char == '\x08':  # this is for windows, cross platform?
                reading_object.Reading = reading_object.Reading[:-1]
            elif char == '\r':  # this is for windows, cross platform?
                reading_object.Reading += '\n'
            else:
                reading_object.Reading += char
        else:
            if char in '\r\x08':  # this is for windows, cross platform?
                pass
            else:
                reading_object.Reading = char

    def GetRole(self, rx, ry):
        '''In which role was the fact type clicked?'''
        fact_object = self.Target
        reading_object = self.Target.ORMFactReading

        roles = reading_object.ORMRoleSequence.GetList('ORMRolePosition')
        roles.sort(cmp=lambda x,y: cmp(x.Seq, y.Seq))
        if debug: print 'GetRole:', roles
        fx, fy = self.GetPos()
        if debug: print 'fx, rx', fx, rx
        rolenum = (rx - fx) // role_box_width
        if 0 <= rolenum < len(roles):
            if debug: print 'rolenum:', rolenum
            return roles[rolenum].ORMRole
        return None

    def SetSelected(self):
        fact_object = self.Target
        reading_object = self.Target.ORMFactReading

        x, y = self.canvas.clickx, self.canvas.clicky
        role = self.GetRole(x, y)
        if role and (role not in self.canvas.rolelist):
            for r in reading_object.ORMRoleSequence.GetList('ORMRolePosition'):
                self.canvas.RemoveRole(r.ORMRole)
            self.canvas.AddRole(role)
        self._SetInShell('IsSelected', True)
        self._SetInShell('CharMode', '')

    def ToggleRole(self):
        if debug: print 'toggle role'
        fact_object = self.Target
        reading_object = self.Target.ORMFactReading

        x, y = self.canvas.clickx, self.canvas.clicky
        role = self.GetRole(x, y)
        if role in self.canvas.rolelist:  # already on, turn it off
            self.canvas.RemoveRole(role)
            return any([ r.ORMFactTypeID == fact_object.ID for r in self.canvas.rolelist ])
        else:  # not on, turn it on
            self.canvas.AddRole(role)
            return True

    def OnTop(self):  # must run after this object has a dcid
##        fact_object = self.Target
##        reading_object = self.Target.ORMFactReading

        followers = self.GetFollowers()
        object_type_shape = None
        for f in followers:
            if f.Subtype == 'ORMObjectTypeShape':
                object_type_shape = f
                break
        if object_type_shape:
##            print 'found objectified'
            dc = self.canvas.pdc
            print self.dcid, object_type_shape.dcid
            if True: # self.dcid < object_type_shape.dcid:
                print 'reseting fact dcid'
                self.canvas.DeleteDCID(self.dcid)
##                dc.RemoveId(self.dcid)  # clear the old id
                self.canvas.SetGraphic(dc, self, newid=True)  # create new one
            for f in followers:
##                print 'looking at', f
##                if debug:
##                    print "object dcid, other dcid", f.dcid, object_type_shape.dcid
                if f.Subtype != 'ORMObjectTypeShape': # f.dcid < object_type_shape.dcid:  # if behind object type shape
                    self.canvas.DeleteDCID(f.dcid)
##                    dc.RemoveId(f.dcid)  # clear the old id
                    self.canvas.SetGraphic(dc, f, newid=True)  # create new one

    def IsUIDeletable(self):  # allow user to delete this graphic object
        return True

    def Delete(self):
        fact_object = self.Target
        reading_object = self.Target.ORMFactReading

        if self.InDelete: return
        self._SetInShell('InDelete', True)
        self.Follow(remove=True)  # before they are deleted
        for shape in self.GetFollowers():
            shape.Delete()
##        for shape in self.GetGraphicList('ORMRoleConnectorShape', 'NodeA'):
##            shape.Delete()
##        for shape in self.GetGraphicList('ORMFactReadingShape', 'NodeA'):
##            shape.Delete()
##        for shape in self.GetGraphicList('ORMConstraintConnectorShape', 'NodeA'):
##            shape.Delete()
        if self.NodeA:
            self.NodeA.Delete()  # Objectified fact type
        if self.Target:
            for obj in self.Target.GetList('ORMFactReading'):
                obj.Delete()
            self.Target.Delete()
        ORMShape.Delete(self)
        self._SetInShell('InDelete', None)

    def DeleteRole(self, role):
        '''use this to delete a role from a fact type'''
        fact_object = self.Target
##        reading_object = self.Target.ORMFactReading

        DeleteRole(fact_object, role)

    def AddRole(self, role):
        '''use this to add a role from a fact type'''
        fact_object = self.Target
##        reading_object = self.Target.ORMFactReading

        AddRole(fact_object, role)

def DeleteRole(fact_object, delete_role):  # in a separate function "just in case"
    readings = fact_object.GetList('ORMFactReading')
    for reading in readings:
        roles = reading.ORMRoleSequence.GetList('ORMRolePosition')
        roles.sort(cmp=lambda x,y: cmp(x.Seq, y.Seq))
        i = 0
        for r in roles:
            if r.ORMRole == delete_role:
                r.Delete()
            else:  # renumber remaining role positions
                i += 1
                r.Seq = i

    delete_role.Delete()
    roles = fact_object.GetList('ORMRole')
    roles.sort(cmp=lambda x,y: cmp(x.Seq, y.Seq))
    for i, r in enumerate(roles):
        r.Seq = i + 1
    fact_object.Nary = len(roles)
    AdjustSampleRows(fact_object)  # remove cells

def AddRole(fact_object, add_role):  # in a separate function "just in case"
    readings = fact_object.GetList('ORMFactReading')
    today = Data.TodayString()
    for reading in readings:
        roles = reading.ORMRoleSequence.GetList('ORMRolePosition')
        newpos = fact_object.db.GetObject('ORMRolePosition')  # new
        newpos.ORMRoleSequenceID = reading.ORMRoleSequence.ID
        newpos.Seq = len(roles) + 1
        newpos.ORMRoleID = add_role.ID
        newpos.DateAdded = today

##    if not add_role.ORMFactType == fact_object:  # if it hasn't already been added
    add_role.ORMFactTypeID = fact_object.ID
    roles = fact_object.GetList('ORMRole')
    fact_object.Nary = len(roles)
    add_role.Seq = len(roles)
    AdjustSampleRows(fact_object)  # add cells

class ORMFactReadingShape(ORMFollowText):
    def Draw(self, dc):
        orm_object = self.Get('Target')
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)

        x, y = self.GetPos()

        text = orm_object.Reading or ''
        lines = text.splitlines()
        sizes = [ self.canvas.GetFullTextExtent(line)[0:2] for line in lines ]  # pull out only w and h
        w = 5
        h = 0
        for line_w,line_h in sizes:
            w = max(w, line_w)
            h += line_h
        box_w = w + 6
        box_h = h + 6
        box_x = x - box_w/2
        box_y = y - box_h/2

        if text == '':
            pen = self.canvas.CachedPen('Blue', 1, wx.DOT)
        else:
            pen = self.canvas.CachedPen('White', 1, wx.SOLID)
        dc.SetPen(pen)
        if self.GetSelected():
            dc.SetBrush(self.canvas.CachedBrush(SelectedColor))
        else:
            dc.SetBrush(self.canvas.CachedBrush((254, 254, 254)))
#        dc.SetBrush(self.canvas.CachedBrush('White'))
        dc.DrawRectangle(box_x,box_y,box_w,box_h)

        dc.SetFont(self.canvas.GetFont())
        dc.SetTextForeground('Black')
        dc.SetTextBackground('White')
        w = 3; h = 2
        for i in range(len(lines)):
            line = lines[i]
            line_w, line_h = sizes[i]
            dc.DrawText(line, box_x+w, box_y+h)
            h += line_h

        r = wx.Rect(box_x,box_y,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)

    def Char(self, char):
        if debug: print "char", ord(char), "'", char, "'"
        orm_object = self.Get('Target')
        if orm_object.Reading:
            if char == '\x08':  # this is for windows, cross platform?
                orm_object.Reading = orm_object.Reading[:-1]
            elif char == '\r':  # this is for windows, cross platform?
                orm_object.Reading += '\n'
            else:
                orm_object.Reading += char
        else:
            if char in '\r\x08':  # this is for windows, cross platform?
                pass
            else:
                orm_object.Reading = char

class ORMSampleTableShape(ORMFollowText):  # follows Object or Fact Type
    def Draw(self, dc):
        orm_object = self.Get('Target')  # ORMFactType or ORMObjectType
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)

        x, y = self.GetPos()

        rows = orm_object.GetList('ORMSampleRow')
        rows.sort(cmp=lambda x,y: cmp(x.Seq, y.Seq))
        table = [ row.GetList('ORMSampleCell') for row in rows ]
        
        if orm_object.Table == 'ORMFactType':  # sort on reading role seq
##            role_to_seq = {}
##            try:
##                rolep = orm_object.ORMFactReading.ORMRoleSequence.GetList('ORMRolePosition')
##            except:
##                rolep = []
##            for rp in rolep:
##                role_to_seq[rp.ORMRole] = rp.Seq

            for row in table:
##                if not row: continue
                for cell in row:
##                    cell._SetInShell('seq', role_to_seq.get(cell.ORMRolePosition or cell.ORMRole))  # cell.ORMRole is just temporary
                    cell._SetInShell('seq', cell.ORMRolePosition.Seq)
        else:  # sort on object type name (arbitrary, but consistent)
               # (should use the sequence from the preferred identifier constraint
               # if more than one)
            for row in table:
##                if not row: continue
                for cell in row:
                    cell._SetInShell('seq', cell.ORMCellValue.ORMObjectType.Name)

        for row in table:
            row.sort(cmp=lambda x,y: cmp(x.seq, y.seq))

        width = {}  # max width in column
        height = {} # max height in row
        table_text = {}
        for i, row in enumerate(table):
##            if not row: continue
            for j, cell in enumerate(row):
                # if cell is selected cell, use cached value and add cursor
                # if ???:
                #     pass
                text = GetCellText(cell)
                if self.canvas.focus_cell is cell:
                    text = self.InsertCursor(self.EditText or '')  # add cursor to text for display
##                if cell.ORMSampleValue:
##                    text = cell.ORMSampleValue.Value
##                elif cell.ORMSampleCompositeValue:
##                    text = GetCompositValue(cell.ORMSampleCompositeValue)
##                else:
##                    text = ''
                cell_w, cell_h = self.canvas.GetFullTextExtent(text)[0:2]  # pull out only w and h
                width[j] = max(cell_w, width.setdefault(j, 5))
                height[i] = max(cell_h, height.setdefault(i, 5))
                table_text[(i,j)] = text

        offset = 3
        for k in width.iterkeys():
            width[k] += offset*2
        for k in height.iterkeys():
            height[k] += offset*2
        
        box_w = sum(width.values())
        box_h = sum(height.values())
        box_x = x - box_w/2
        box_y = y - box_h/2

        pen = self.canvas.CachedPen('Green', 1, wx.SOLID)
        dc.SetPen(pen)
        if self.GetSelected():
            dc.SetBrush(self.canvas.CachedBrush(SelectedColor))
        else:
            dc.SetBrush(self.canvas.CachedBrush((254, 254, 254)))
#        dc.SetBrush(self.canvas.CachedBrush('White'))
        dc.DrawRectangle(box_x,box_y,box_w,box_h)
        sumw = sumh = 0
        clip_columns = [ sumw ]
        clip_rows = [ sumh ]
        for i in range(len(width)-1): # vertical
            w = width[i]
            sumw += w
            dc.DrawLine(box_x+sumw, box_y, box_x+sumw, box_y+box_h)
            clip_columns.append(sumw)
        clip_columns.append(box_w)
        
        for i in range(len(height)-1):
            h = height[i]
            sumh += h
            dc.DrawLine(box_x, box_y+sumh, box_x+box_w, box_y+sumh)
            clip_rows.append(sumh)
        clip_rows.append(box_h)

        dc.SetFont(self.canvas.GetFont())
        dc.SetTextForeground('Black')
        dc.SetTextBackground('White')

        h = 0
        for i, row in enumerate(table):
            w = 0
            for j, cell in enumerate(row):
                line = table_text[(i,j)]
                if line:
                    dc.DrawText(line, box_x+w+offset, box_y+h+offset)
                w += width[j]
            h += height[i]

        r = wx.Rect(box_x,box_y,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)

        self.SetClipBox(box_x, box_y, box_w, box_h)
        self.SetWithoutUndo('ClipRows', ','.join([ str(x) for x in clip_rows ]))
        self.SetWithoutUndo('ClipColumns', ','.join([ str(x) for x in clip_columns ]))

# if ctrl-r is entered, add a new role object
# if move to new cell, the process cell contents
# or if object selection status is changed

    def Char(self, char):
        if debug: print "char", ord(char), "'", char, "'"
        if char == '\t':
            prior_cell = self.canvas.focus_cell
            self.ClearFocus()
            # set focus to next cell
            cells = prior_cell.ORMSampleRow.GetList('ORMSampleCell')
            cells.sort(cmp=lambda x,y: cmp(x.ORMRolePosition.Seq, y.ORMRolePosition.Seq))
            pos = cells.index(prior_cell)
            if pos < len(cells) - 1:  # use the next cell in this row
                self.SetFocusCell(cells[pos+1])
            else:
                prior_row = prior_cell.ORMSampleRow
                mo = prior_row.ORMObjectType or prior_row.ORMFactType
                rows = mo.GetList('ORMSampleRow')
                rows.sort(cmp=lambda x,y: cmp(x.Seq, y.Seq))
                pos = rows.index(prior_row)
                if pos < len(rows) - 1:  # use the first cell in the next row
                    new_row = rows[pos + 1]
                else:  # use first cell in first row
                    new_row = rows[0]
                cells = new_row.GetList('ORMSampleCell')
                cells.sort(cmp=lambda x,y: cmp(x.ORMRolePosition.Seq, y.ORMRolePosition.Seq))
                self.SetFocusCell(cells[0])
        else:
            self._SetInShell('EditText', self.ApplyChar(char, self.EditText))

##    def OpenCell(self, row, col):
##        self._SetInShell('text', xx)

    def GetCell(self, cx, cy):
        '''In which cell was the sample table clicked?'''
        model_object = self.Target
        if not self.ClipRows or not model_object:
            print "no cliprows"
            return None

        box_x, box_y, box_w, box_h = self.GetClipBox()
        if (cx < box_x or cx > box_x + box_w
            or cy < box_y or cy > box_y + box_h):
            print "didn't click in sample table"
            return None

        w = cx - box_x
        h = cy - box_y
        row_number = col_number = 0
        clip_rows = [ int(x) for x in self.ClipRows.split(',') ]
        clip_cols = [ int(x) for x in self.ClipColumns.split(',') ]
        for i, row_h in enumerate(clip_rows[1:]):
            if h < row_h:
                row_number = i
                break
        for i, col_w in enumerate(clip_cols[1:]):
            if w < col_w:
                col_number = i
                break

        rows = model_object.GetList('ORMSampleRow')
        rows.sort(cmp=lambda x,y: cmp(x.Seq, y.Seq))
        if len(rows) == 0:
            print "error, no sample rows"
            return None
        elif row_number < len(rows):
            row = rows[row_number]
        else:
            row = rows[-1]
        cols = row.GetList('ORMSampleCell')
        cols.sort(cmp=lambda x,y: cmp(x.ORMRolePosition.Seq, y.ORMRolePosition.Seq))
        if len(cols) == 0:
            print "error, no sample cells"
            return None
        elif col_number < len(cols):
            col = cols[col_number]
        else:
            col = cols[-1]
        return col

    def SetSelected(self):
        model_object = self.Target

        x, y = self.canvas.clickx, self.canvas.clicky
        cell = self.GetCell(x, y)
        self.SetFocusCell(cell)
        self._SetInShell('IsSelected', True)

    def SetFocusCell(self, cell):
        self.canvas.focus_cell = cell
        self._SetInShell('EditText', GetCellText(cell))
        self._SetInShell('Cursor', None)

    def ClearFocus(self):
        model_object = self.Target

        if self.canvas.focus_cell:  # finish up cell edit
            print "----- cell value might have changed -----"
            cell = self.canvas.focus_cell
            # should cell be value or composite value?
            if not cell.ORMSampleValue:
                cell.ORMSampleValue = AddSampleValue(cell)
            if cell.ORMSampleValue:
                oldsv = cell.ORMSampleValue
                if oldsv.Value != self.EditText:
                    print "----- cell value changed -----"
                    # is this sv used by anyone else?
                    references = oldsv.GetList('ORMSampleCell')
                    # is there another sv with desired value?
                    oldsv_list = oldsv.ORMObjectType.GetList('ORMSampleValue')
                    print 'oldsv_list', oldsv_list
                    svs = [ x for x in oldsv_list                                
                            if x.Value == self.EditText ]
                    print 'svs', svs
                    if svs:  # use existing sv
                        print "----- desired value already exists -----"
                        cell.ORMSampleValue = svs[0]
                        if len(references) == 1:  # this was only user of sv
                            print "----- delete unneeded sv -----"
                            oldsv.Delete()  # old sv not needed by anyone
                    else:  # desired sv doesn't exist
                        print "----- desired value doesn't already exist -----"
                        if len(references) == 1:  # change this sv
                            print "----- recycle old value object -----"
                            oldsv.Value = self.EditText
                        else:  # create new sv
                            print "----- creaste new sv object -----"
                            newsv = self.db.GetObject('ORMSampleValue')
                            newsv.ProjectID = oldsv.ProjectID
                            newsv.ORMObjectTypeID = oldsv.ORMObjectTypeID
                            newsv.Seq = len(oldsv_list) + 1
                            newsv.Value = self.EditText
                            newsv.DateAdded = Data.TodayString()
                            cell.ORMSampleValue = newsv
                    Data.SetUndo('Cell value changed')
            elif cell.ORMSampleValue:  # multiple identifier
                pass
            else:
                pass

            self.canvas.focus_cell = None

def GetCellText(cell):
    if not cell:
        return ''
    elif cell.ORMSampleValue:
        return cell.ORMSampleValue.Value or ''
    elif cell.ORMSampleCompositeValue:
        cells = [ x for x in cell.ORMSampleCompositeValue.GetList('ORMSampleCell')
                  if x.ORMRolePosition ]
        cells.sort(cmp=lambda x,y:
                   cmp(x.ORMRolePosition.Seq, y.ORMRolePosition.Seq))
        return '|'.join([ GetCellText(x) for x in cells ])
    else:
        return ''

def AddSampleValue(cell):
    new = cell.db.GetObject('ORMSampleValue')
    new.ProjectID = cell.ProjectID
    ot = cell.ORMRolePosition.ORMRole.ORMObjectType
    new.ORMObjectTypeID = ot.ID
    new.Value = ''
    new.Seq = len(ot.GetList('ORMSampleValue')) + 1
    new.DateAdded = Data.TodayString()
    return new

def AddSampleCell(row, rp):
    new = row.db.GetObject('ORMSampleCell')
    new.ProjectID = row.ProjectID
    new.ORMSampleRowID = row.ID
    new.ORMRolePosition = rp
    new.DateAdded = Data.TodayString()
    return new

def AddSampleRow(mo, seq=1):
    new = mo.db.GetObject('ORMSampleRow')
    if mo.Table == 'ORMObjectType':
        new.ORMObjectTypeID = mo.ID
    elif mo.Table == 'ORMFactType':
        new.ORMFactTypeID = mo.ID
    else:
        pass # shouldn't happen
    new.ProjectID = mo.ProjectID
    new.Seq = seq
    new.DateAdded = Data.TodayString()
    AdjustSampleRows(mo)
    return new

def AdjustSampleRows(mo):
    rows = mo.GetList('ORMSampleRow')
    rows.sort(cmp=lambda x,y: cmp(x.Seq, y.Seq))
    if mo.Table == 'ORMObjectType':
        pass  # ignore for now
    elif mo.Table == 'ORMFactType':
        role_pos = mo.ORMFactReading.ORMRoleSequence.GetList('ORMRolePosition')
        role_pos.sort(cmp=lambda x,y: cmp(x.Seq, y.Seq))
        for i, row in enumerate(rows):
            if row.Seq != i + 1: row.Seq = i + 1
            # delete cells that don't match a role position
            cells = row.GetList('ORMSampleCell')
            for cell in cells:
                if cell.ORMRolePosition not in role_pos:
                    cells.remove(cell)
                    cell.Delete()
            for rp in role_pos:
                # make sure that each role pos matches a cell
                found = False
                for cell in cells:
                    if rp is cell.ORMRolePosition:
                        found = True
                        break
                if not found:
                    new_cell = AddSampleCell(row, rp)
    else:
        pass # shouldn't happen

class ORMRoleNameShape(ORMFollowText):
    def Draw(self, dc):
        orm_object = self.Get('ORMRole')  # make sure this is there?
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)

        x, y = self.GetPos()

        text = (orm_object.Name or '')
        if text: text = '[' + text + ']'
        lines = text.splitlines()
        sizes = [ self.canvas.GetFullTextExtent(line)[0:2] for line in lines ]  # pull out only w and h
        w = 5
        h = 0
        for line_w,line_h in sizes:
            w = max(w, line_w)
            h += line_h
        box_w = w + 6
        box_h = h + 6
        box_x = x - box_w/2
        box_y = y - box_h/2

        if text == '':
            pen = self.canvas.CachedPen('Blue', 1, wx.DOT)
        else:
            pen = self.canvas.CachedPen('White', 1, wx.SOLID)
        dc.SetPen(pen)
        if self.GetSelected():
            dc.SetBrush(self.canvas.CachedBrush(SelectedColor))
        else:
            dc.SetBrush(self.canvas.CachedBrush((254, 254, 254)))
        dc.DrawRectangle(box_x,box_y,box_w,box_h)

        dc.SetFont(self.canvas.GetFont())
        dc.SetTextForeground('Black')
        dc.SetTextBackground('White')
        w = 3; h = 2
        for i in range(len(lines)):
            line = lines[i]
            line_w, line_h = sizes[i]
            dc.DrawText(line, box_x+w, box_y+h)
            h += line_h

        r = wx.Rect(box_x,box_y,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)

    def Char(self, char):
        if debug: print "char", ord(char), "'", char, "'"
        # orm_object = self.Get('Target')
        orm_object = self.Get('ORMRole')  # make sure this is there?
        if orm_object.Name:
            if char == '\x08':  # this is for windows, cross platform?
                orm_object.Name = orm_object.Name[:-1]
            elif char == '\r':  # this is for windows, cross platform?
                orm_object.Name += '\n'
            else:
                orm_object.Name += char
        else:
            if char in '\r\x08':  # this is for windows, cross platform?
                pass
            else:
                orm_object.Name = char

    def IsUIDeletable(self):  # allow user to delete this graphic object
        return True

    def Delete(self):
        self._SetInShell('InDelete', True)
        ORMShape.Delete(self)
        self._SetInShell('InDelete', None)

class ORMNoteShape(ORMBox):
    def Draw(self, dc):
        orm_object = self.Get('Target')
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)

        x, y = self.GetPos()
##        x = self.PosX  # treat this as the center of the shape
##        y = self.PosY

        text = orm_object.Text or 'empty note'
        if self.IsSelected and self.dcid == self.canvas.keyboard_target_dcid:
            text = self.InsertCursor(text)  # add cursor to text for display

        lines = text.splitlines()
        sizes = [ self.canvas.GetFullTextExtent(line)[0:2] for line in lines ]  # pull out only w and h
        w = 5
        h = 0
        for line_w,line_h in sizes:
            w = max(w, line_w)
            h += line_h
        box_w = w + 6
        box_h = h + 6
        box_x = x - box_w/2
        box_y = y - box_h/2

        pen = self.canvas.CachedPen('Blue', 1, wx.DOT)
        dc.SetPen(pen)
        if self.GetSelected():
            dc.SetBrush(self.canvas.CachedBrush(SelectedColor))
        else:
            dc.SetBrush(self.canvas.CachedBrush((254, 254, 254)))
            # dc.SetBrush(self.canvas.CachedBrush('White'))
        dc.DrawRectangle(box_x,box_y,box_w,box_h)

        dc.SetFont(self.canvas.GetFont())
        dc.SetTextForeground('Black')
        dc.SetTextBackground('White')
        w = 3; h = 2
        for i in range(len(lines)):
            line = lines[i]
            line_w, line_h = sizes[i]
            dc.DrawText(line, box_x+w, box_y+h)
            h += line_h

        r = wx.Rect(box_x,box_y,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)

    def Char(self, char):
        if debug: print "char", ord(char), "'", char, "'"
        orm_object = self.Get('Target')
        orm_object.Text = self.ApplyChar(char, orm_object.Text)
##        if orm_object.Text:
##            if char == '\x08':  # this is for windows, cross platform?
##                orm_object.Text = orm_object.Text[:-1]
##            elif char == '\r':  # this is for windows, cross platform?
##                orm_object.Text += '\n'
##            else:
##                orm_object.Text += char
##        else:
##            if char in '\r\x08':  # this is for windows, cross platform?
##                pass
##            else:
##                orm_object.Text = char

    def IsUIDeletable(self):  # allow user to delete this graphic object
        return True

    def Delete(self):
        if self.InDelete: return
        self._SetInShell('InDelete', True)
        self.Follow(remove=True)  # before they are deleted
##        for shape in self.GetGraphicList('ORMNoteConnectorShape', 'NodeA'):
##            shape.Delete()
        if self.Target:
            self.Target.Delete()
        ORMShape.Delete(self)
        self._SetInShell('InDelete', None)

class ORMConstraintShape(ORMCircle):
    ''' external constraint?'''
    def Draw(self, dc):
        orm_object = self.Get('Target')
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)

        x, y = self.GetPos()
##        x = self.PosX  # treat this as the center of the shape
##        y = self.PosY

        if orm_object.Deontic:
            color = 'Blue'
        else:  # Alethic
            color = 'Violet'

        pen = self.canvas.CachedPen(color, 1, wx.SOLID)
        dc.SetPen(pen)
        if self.GetSelected():
            bgbrush = self.canvas.CachedBrush(SelectedColor)
        else:
            bgbrush = self.canvas.CachedBrush((254, 254, 254))
        colorbrush = self.canvas.CachedBrush(color)
   
        if orm_object.Operator in RingConstraints:
            operators = orm_object.Operator.split(" and ")
            symmetric = "Symmetric" in operators
            asymmetric = "Asymmetric" in operators
            antisymmetric = "Antisymmetric" in operators
            irreflexive = "Irreflexive" in operators
            intransitive = "Intransitive" in operators
            acyclic = "Acyclic" in operators
    
            radius = 7 # 10
            ellipse_width = 18 # 25
            dot_radius = 2 # 3
            mini_radius = 4 # 6
            mini_dot_radius = 1.5 # 2
            box_h = (radius + dot_radius) * 2
        
            def DrawIrreflexive(radius, dot_radius):
                dc.SetBrush(bgbrush)
                dc.DrawCircle(x, y, radius)
                dc.SetBrush(colorbrush)
                dc.DrawCircle(x - radius, y, dot_radius)
                dc.DrawLine(x, y + radius - dot_radius, x, y + radius + dot_radius)
    
            def DrawTrianglePoints(radius, dot_radius, lines=False):
                base = y + radius / 2
                pt1 = (x, y - radius)
                pt2 = (x - radius * math.sqrt(3) / 2, base)
                pt3 = (x + radius * math.sqrt(3) / 2, base)
                dc.SetBrush(bgbrush)
                if lines:
                    dc.DrawPolygon((pt1, pt2, pt3))
                    dc.DrawLine(x, base - dot_radius, x, base + dot_radius)
                dc.SetBrush(colorbrush)
                for pt in (pt1, pt2, pt3):
                    dc.DrawCirclePoint(pt, dot_radius)

            def DrawIntransitive(radius, dot_radius):
                DrawTrianglePoints(radius, dot_radius, lines=True)

            if symmetric or asymmetric or antisymmetric:
                box_w = ellipse_width + dot_radius * 2
                left_end = x - ellipse_width / 2
                right_end = x + ellipse_width / 2
                dc.SetBrush(bgbrush)
                dc.DrawEllipse(left_end, y - radius, ellipse_width, radius * 2)
                dc.SetBrush(colorbrush)
                dc.DrawCircle(left_end, y, dot_radius)
                if antisymmetric:
                    dc.SetBrush(bgbrush)
                dc.DrawCircle(right_end, y, dot_radius)
                if irreflexive:
                    DrawIrreflexive(mini_radius, mini_dot_radius)
                if intransitive:
                    DrawIntransitive(mini_radius, mini_dot_radius)
                if not symmetric:
                    dc.DrawLine(x, y + radius - dot_radius, x, y + radius + dot_radius)
            else:
                box_w = box_h
                if acyclic:
                    dc.SetBrush(bgbrush)
                    dc.DrawCircle(x, y, radius)
                    DrawTrianglePoints(radius, dot_radius)
                    dc.DrawLine(x, y + radius - dot_radius, x, y + radius + dot_radius)
                if irreflexive:
                    DrawIrreflexive(radius, dot_radius)
                if intransitive:
                    DrawIntransitive(radius, dot_radius)

            box_x = x - box_w/2
            box_y = y - box_h/2

        else:  # non-ring constraint
 
            box_w = 14  # 20
            box_h = box_w
            box_x = x - box_w/2
            box_y = y - box_h/2
    
            dc.SetBrush(bgbrush)
            dc.DrawCircle(x,y,box_w/2)
    
            if orm_object.Operator in ('InclusiveOr', 'ExclusiveOr'):  # inclusive
                dc.SetBrush(colorbrush)
                dc.DrawCircle(x,y,box_w/4)
                dc.SetBrush(bgbrush)
    
            if orm_object.Operator in ('Exclusion', 'ExclusiveOr'):  # exclusive
                xx, yy = self.AdjustEnd((box_x, box_y))
                adjx = x - xx - 2
                adjy = y - yy - 2
                dc.DrawLine(x-adjx, y-adjy, x+adjx+1, y+adjy+1)
                dc.DrawLine(x-adjx, y+adjy, x+adjx+1, y-adjy-1)
    
            elif orm_object.Operator == 'Unique':  # external unique
                dc.DrawLine(x-box_w/2, y, x+box_w/2, y)
    
            elif orm_object.Operator == 'Preferred':  # external prefered unique
                dc.DrawLine(x-box_w/2, y-1, x+box_w/2, y-1)
                dc.DrawLine(x-box_w/2, y+1, x+box_w/2, y+1)
    
            elif orm_object.Operator == 'Equality':  # equal
                dc.DrawLine(x-box_w/2+4, y-2, x+box_w/2-4, y-2)
                dc.DrawLine(x-box_w/2+4, y+1, x+box_w/2-4, y+1)
    
            elif orm_object.Operator == 'Subset':  # subset
                left = x-box_w/2+6
                right = x+box_w/2-4
                gap = 2
                dc.DrawLine(left, y-gap*2, right, y-gap*2)
    
                # arc doesn't work because it draws the diameter line
                # spline doesn't work because it isn't implemented by pseudo dc
                dc.DrawLine(left, y-gap*2, left, y-gap)
    #            dc.DrawArc(left, y-gap*2, left, y, left, y-gap)
    #            dc.DrawSpline([wx.Point(left, y-gap*2), wx.Point(left-3, y-gap), wx.Point(left, y)])
    
                dc.DrawLine(left, y, right, y)
                dc.DrawLine(x-box_w/2+4, y+2, right, y+2)
    
    ##        dc.SetFont(self.canvas.GetFont())
    ##        dc.SetTextForeground('Black')
    ##        dc.SetTextBackground('White')
    ##
    ##        w = 3; h = 2
    ##        for i in range(len(lines)):
    ##            line = lines[i]
    ##            line_w, line_h = sizes[i]
    ##            dc.DrawText(line, box_x+w, box_y+h)
    ##            h += line_h
    
        r = wx.Rect(box_x,box_y,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)  # must set before self.AdjustEnd can be used

    def IsUIDeletable(self):  # allow user to delete this graphic object
        return True

    def Delete(self):
        if self.InDelete: return
        self._SetInShell('InDelete', True)
        self.Follow(remove=True)  # before they are deleted
##        for shape in self.GetGraphicList('ORMConstraintConnectorShape', 'NodeB'):
##            shape.Delete()
        if self.Target:
            for obj in self.Target.GetList('ORMRoleSequence'):
                for pos in obj.GetList('ORMRolePosition'):
                    pos.Delete()
                obj.Delete()
            self.Target.Delete()
        ORMShape.Delete(self)
        self._SetInShell('InDelete', None)

class ORMSubtypeConstraintShape(ORMCircle):
    ''' '''
    def Draw(self, dc):
        orm_object = self.Get('Target')
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)

        x, y = self.GetPos()

        box_w = 14  # 20
        box_h = box_w
        box_x = x - box_w/2
        box_y = y - box_h/2

        if orm_object.Deontic:
            color = 'Blue'
        else:  # Alethic
            color = 'Violet'

        pen = self.canvas.CachedPen(color, 1, wx.SOLID)
        dc.SetPen(pen)
        if self.GetSelected():
            dc.SetBrush(self.canvas.CachedBrush(SelectedColor))
        else:
            dc.SetBrush(self.canvas.CachedBrush((254, 254, 254)))
            # dc.SetBrush(self.canvas.CachedBrush('White'))
        dc.DrawCircle(x,y,box_w/2)

        r = wx.Rect(box_x,box_y,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)  # must set before self.AdjustEnd can be used

        if orm_object.Operator in ('Total', 'Partition'):  # inclusive
            dc.SetBrush(self.canvas.CachedBrush(color))
            dc.DrawCircle(x,y,box_w/4)

        if orm_object.Operator in ('Exclusive', 'Partition'):  # exclusive
            xx, yy = self.AdjustEnd((box_x, box_y))
            adjx = x - xx - 2
            adjy = y - yy - 2
            dc.DrawLine(x-adjx, y-adjy, x+adjx+1, y+adjy+1)
            dc.DrawLine(x-adjx, y+adjy, x+adjx+1, y-adjy-1)

##        dc.SetFont(self.canvas.GetFont())
##        dc.SetTextForeground('Black')
##        dc.SetTextBackground('White')
##
##        w = 3; h = 2
##        for i in range(len(lines)):
##            line = lines[i]
##            line_w, line_h = sizes[i]
##            dc.DrawText(line, box_x+w, box_y+h)
##            h += line_h

    def IsUIDeletable(self):  # allow user to delete this graphic object
        return True

    def Delete(self):
        if self.InDelete: return
        self._SetInShell('InDelete', True)
        self.Follow(remove=True)  # before they are deleted
##        for shape in self.GetGraphicList('ORMSubtypeConstraintConnectorShape', 'NodeA'):
##            shape.Delete()
        if self.Target:
            for obj in self.Target.GetList('ORMSubtypeConstraintConnector', 'NodeA'):
                obj.Delete()
            self.Target.Delete()
        ORMShape.Delete(self)
        self._SetInShell('InDelete', None)

class ORMRoleConnectorShape(ORMConnector):
    def GetEnds(self):
        orm_object = self.Get('Target')
        x, y = self.GetPos()
        nodea, nodeb = self.NodeA, self.NodeB

        # connect base on reading sequence, not fact sequence
        fact_object = nodea.Target
        reading_object = nodea.Target.ORMFactReading
        seq = [ x.Seq for x in reading_object.ORMRoleSequence.GetList('ORMRolePosition')
                if x.ORMRole.ID == orm_object.ID ][0]
        endb = nodeb.GetPos()
        enda = nodea.AdjustEnd(endb, seq)  # nodeA should be a fact shape
        endb = nodeb.AdjustEnd(enda)
        return enda, endb

    def Draw(self, dc):
        orm_object = self.Get('Target')
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)
        
        x, y = self.GetPos()
        nodea = self.NodeA  # these objects might not already be in the diagram
        nodeb = self.NodeB

        # connect base on reading sequence, not fact sequence
        reading_object = nodea.Target.ORMFactReading
        seq = [ x.Seq for x in reading_object.ORMRoleSequence.GetList('ORMRolePosition')
                if x.ORMRole.ID == orm_object.ID ][0]
        endb = nodeb.GetPos()
        enda = nodea.AdjustEnd(endb, seq)  # nodeA should be a fact shape
        endb = nodeb.AdjustEnd(enda)

        minx = min(enda[0], endb[0])
        maxx = max(enda[0], endb[0])
        miny = min(enda[1], endb[1])
        maxy = max(enda[1], endb[1])
        
        box_w = maxx - minx + 6
        box_h = maxy - miny + 6

        if self.IsSelected:
            pen = self.canvas.CachedPen(SelectedColor, 3, wx.SOLID)
            dc.SetPen(pen)
            dc.DrawLine(enda[0], enda[1], endb[0], endb[1])

        pen = self.canvas.CachedPen('Black', 1, wx.SOLID)
        dc.SetPen(pen)
        dc.DrawLine(enda[0], enda[1], endb[0], endb[1])
        self.SetTempPos((enda[0]+endb[0]) / 2, (enda[1] + endb[1]) / 2)
        # need to add logic to draw the role name
        # need to save the calculated PosX and PosY

        # place mandatory participation
        if orm_object.Mandatory: # mandatory
            dot_size = 3
            if 1: # one end
                dot = wx.lib.ogl.GetPointOnLine(enda[0], enda[1], endb[0], endb[1], dot_size)
            else:
                dot = wx.lib.ogl.GetPointOnLine(endb[0], endb[1], enda[0], enda[1], dot_size)
            if orm_object.Mandatory == 'a':  #
                color = 'Violet'
            else:  # d
                color = 'Blue'
            pen = self.canvas.CachedPen(color, 1, wx.SOLID)
            dc.SetBrush(self.canvas.CachedBrush(color))
            dc.SetPen(pen)
            dc.DrawCircle(int(dot[0]),int(dot[1]),dot_size)

        r = wx.Rect(minx,miny,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)

    def Char(self, char):
        if debug: print "char", ord(char), "'", char, "'"
        orm_object = self.Get('Target')
        if orm_object.Name:
            if char == '\x08':  # this is for windows, cross platform?
                orm_object.Name = orm_object.Name[:-1]
            elif char == '\r':  # this is for windows, cross platform?
                orm_object.Name += '\n'
            else:
                orm_object.Name += char
        else:
            if char in '\r\x08':  # this is for windows, cross platform?
                pass
            else:
                orm_object.Name = char

    def Delete(self):
        if self.InDelete: return  # prevent loops
        self._SetInShell('InDelete', True)
        self.Follow(remove=True)  # before they are deleted
        if self.NodeA:
            self.NodeA.DeleteRole(self.Target)
        for f in self.GetFollowers():
            if f.Subtype == 'ORMRoleNameShape':
                f.Delete()
        ORMShape.Delete(self)
        self._SetInShell('InDelete', None)

class ORMSubtypeConnectorShape(ORMConnector):
    def Draw(self, dc):
        orm_object = self.Get('Target')
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)

        x, y = self.GetPos()  # treat this as the center of the shape
        nodea = self.NodeA  # these objects might not already be in the diagram
        nodeb = self.NodeB

        enda = nodea.GetPos()
        endb = nodeb.GetPos()
        enda, endb = nodea.AdjustEnd(endb), nodeb.AdjustEnd(enda)
        
        minx = min(enda[0], endb[0])
        maxx = max(enda[0], endb[0])
        miny = min(enda[1], endb[1])
        maxy = max(enda[1], endb[1])
        
        box_w = maxx - minx + 6
        box_h = maxy - miny + 6

        if self.IsSelected:
            pen = self.canvas.CachedPen(SelectedColor, 3, wx.SOLID)
            dc.SetPen(pen)
            dc.DrawLine(enda[0], enda[1], endb[0], endb[1])

        pen = self.canvas.CachedPen('Violet', 1, wx.SOLID)
        dc.SetPen(pen)
        dc.DrawLine(enda[0], enda[1], endb[0], endb[1])
        self.SetTempPos((enda[0]+endb[0]) / 2, (enda[1] + endb[1]) / 2)
        # need to add logic to draw the role name
        # need to save the calculated PosX and PosY

        # subtype arrow
        arrow_length = 3
        arrow_width = 3
        arrow = list(wx.lib.ogl.GetArrowPoints(enda[0], enda[1], endb[0], endb[1], arrow_length, arrow_width))
        if debug: print repr(arrow)
        draw_arrow = []
        while arrow:
            draw_arrow.append((int(arrow.pop(0)+.5), int(arrow.pop(0)+.5)),)
        if debug: print repr(draw_arrow)
        pen = self.canvas.CachedPen('Violet', 1, wx.SOLID)
        dc.SetPen(pen)
        dc.SetBrush(self.canvas.CachedBrush('Violet'))
        dc.DrawPolygon(draw_arrow)  # this works!

        r = wx.Rect(minx,miny,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)

    def IsUIDeletable(self):  # allow user to delete this graphic object
        return True
## or handle via menu command

    def Delete(self):
        if self.InDelete: return  # prevent loops
        self._SetInShell('InDelete', True)
        self.Follow(remove=True)  # before they are deleted
        if self.Target:
            self.Target.Delete()
        ORMShape.Delete(self)
        self._SetInShell('InDelete', None)

class ORMNoteConnectorShape(ORMConnector):
    def Draw(self, dc):
        orm_object = self.Get('Target')
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)

        x, y = self.GetPos()
        nodea, nodeb = self.NodeA, self.NodeB

        enda = nodea.GetPos()
        endb = nodeb.GetPos()
        enda, endb = nodea.AdjustEnd(endb), nodeb.AdjustEnd(enda)

        minx = min(enda[0], endb[0])
        maxx = max(enda[0], endb[0])
        miny = min(enda[1], endb[1])
        maxy = max(enda[1], endb[1])
        
        box_w = maxx - minx + 6
        box_h = maxy - miny + 6

        if self.IsSelected:
            pen = self.canvas.CachedPen(SelectedColor, 3, wx.SOLID)
            dc.SetPen(pen)
            dc.DrawLine(enda[0], enda[1], endb[0], endb[1])

        pen = self.canvas.CachedPen('Blue', 1, wx.DOT)
        dc.SetPen(pen)
        dc.DrawLine(enda[0], enda[1], endb[0], endb[1])
        self.SetTempPos((enda[0]+endb[0]) / 2, (enda[1] + endb[1]) / 2)

        r = wx.Rect(minx,miny,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)

    def IsUIDeletable(self):  # allow user to delete this graphic object
        return True

    def Delete(self):
        if self.InDelete: return  # prevent loops
        self._SetInShell('InDelete', True)
        self.Follow(remove=True)  # before they are deleted
        if self.Target:
            self.Target.Delete()
        ORMShape.Delete(self)
        self._SetInShell('InDelete', None)

class ORMConstraintConnectorShape(ORMConnector):
    def GetEnds(self):
        x, y = self.GetPos()
        nodea, nodeb = self.NodeA, self.NodeB
        fact_object = nodea.Target
        reading_object = nodea.Target.ORMFactReading

        constraintlist = self.Get('Target')
        positions = constraintlist.GetList('ORMRolePosition')
        fact_roles = [p.ORMRole for p in positions
                      if p.ORMRole and p.ORMRole.ORMFactTypeID == fact_object.ID]

        seq_xref = dict([ (x.ORMRoleID, x.Seq) for x in reading_object.ORMRoleSequence.GetList('ORMRolePosition') ])
        seq_numbers = [seq_xref[role.ID] for role in fact_roles]
        if not seq_numbers: return

        min_seq = min(seq_numbers)
        max_seq = max(seq_numbers)
        seq = (min_seq + max_seq) / 2.0

        endb = nodeb.GetPos()
        enda = nodea.AdjustEnd(endb, seq)  # nodeA should be a fact shape
        endb = nodeb.AdjustEnd(enda)
        return enda, endb

    '''Connects to Fact Type at one or more Role boxes'''
    def Draw(self, dc):
        orm_object = self.Get('Target')
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)
        
        x, y = self.GetPos()
        nodea = self.NodeA  # these objects might not already be in the diagram
        nodeb = self.NodeB
        fact_object = nodea.Target
        reading_object = nodea.Target.ORMFactReading

        constraintlist = self.Get('Target')
        positions = constraintlist.GetList('ORMRolePosition')
        # remove constraint roles that aren't for this fact 
        fact_roles = [p.ORMRole for p in positions
                      if p.ORMRole and p.ORMRole.ORMFactTypeID == fact_object.ID]

        seq_xref = dict([ (x.ORMRoleID, x.Seq) for x in reading_object.ORMRoleSequence.GetList('ORMRolePosition') ])
        seq_numbers = [seq_xref[role.ID] for role in fact_roles]
        if not seq_numbers: return

        if orm_object.ORMConstraint.Deontic:
            color = 'Blue'
        else:  # Alethic
            color = 'Violet'

        min_seq = min(seq_numbers)
        max_seq = max(seq_numbers)
        seq = (min_seq + max_seq) / 2.0

        endb = nodeb.GetPos()
        enda = nodea.AdjustEnd(endb, seq)  # nodeA should be a fact shape
        endb = nodeb.AdjustEnd(enda)

        minx = min(enda[0], endb[0])
        maxx = max(enda[0], endb[0])
        miny = min(enda[1], endb[1])
        maxy = max(enda[1], endb[1])

        if max_seq - min_seq > 1:
            # make room for the horizontal bar
            factX, factY = enda
            if factY < nodea.GetCenter()[1]:
                barY = factY - 4
            else:
                barY = factY + 4
            enda = factX, barY

            leftX = nodea.AdjustEnd(enda, min_seq)[0]
            rightX = nodea.AdjustEnd(enda, max_seq)[0]
            pen = self.canvas.CachedPen(color, 1, wx.SOLID)
            dc.SetPen(pen)
            dc.DrawLine(leftX, barY, rightX, barY)

            for s in seq_numbers:
                roleX = nodea.AdjustEnd(enda, s)[0]
                dc.DrawLine(roleX, factY, roleX, barY)

            minx = min(minx, leftX)
            maxx = max(maxx, rightX)
            miny = min(miny, barY, factY)
            maxy = max(maxy, barY, factY)
        
        box_w = maxx - minx + 6
        box_h = maxy - miny + 6

        if self.IsSelected:
            pen = self.canvas.CachedPen(SelectedColor, 3, wx.SOLID)
            dc.SetPen(pen)
            dc.DrawLine(enda[0], enda[1], endb[0], endb[1])

        pen = self.canvas.CachedPen(color, 1, wx.DOT)
        dc.SetPen(pen)
        dc.DrawLine(enda[0], enda[1], endb[0], endb[1])
        self.SetTempPos((enda[0]+endb[0]) / 2, (enda[1] + endb[1]) / 2)

        if orm_object.ORMConstraint.Operator == 'Subset' and orm_object.Seq == 2:
            # subtype arrow
            arrow_length = 3
            arrow_width = 3
            arrow = list(wx.lib.ogl.GetArrowPoints(endb[0], endb[1], enda[0], enda[1], arrow_length, arrow_width))
            draw_arrow = []
            while arrow:
                draw_arrow.append((int(arrow.pop(0)+.5), int(arrow.pop(0)+.5)),)
            brush = self.canvas.CachedBrush(color)
            dc.SetBrush(brush)
            dc.DrawPolygon(draw_arrow)  # this works!

        r = wx.Rect(minx,miny,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)

class ORMSubtypeConstraintConnectorShape(ORMConnector):
    '''Connects to Subtype arrow'''
    def Draw(self, dc):
        orm_object = self.Get('Target')
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)

        x, y = self.GetPos()
        nodea, nodeb = self.NodeA, self.NodeB

        enda = nodea.GetPos()
        endb = nodeb.GetPos()
        enda, endb = nodea.AdjustEnd(endb), nodeb.AdjustEnd(enda)

        minx = min(enda[0], endb[0])
        maxx = max(enda[0], endb[0])
        miny = min(enda[1], endb[1])
        maxy = max(enda[1], endb[1])
        
        box_w = maxx - minx + 6
        box_h = maxy - miny + 6

        if orm_object.ORMSubtypeConstraint.Deontic:
            color = 'Blue'
        else:  # Alethic
            color = 'Violet'
        pen = self.canvas.CachedPen(color, 1, wx.DOT)
        dc.SetPen(pen)
        dc.DrawLine(enda[0], enda[1], endb[0], endb[1])
        self.SetTempPos((enda[0]+endb[0]) / 2, (enda[1] + endb[1]) / 2)

        r = wx.Rect(minx,miny,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)

##    def IsUIDeletable(self):  # allow user to delete this graphic object
##        return True
## or handle via menu command

# Relational Table Shapes

class RelationalTableShape(ORMBox):
    def GetTextLines(self):
        orm_object = self.Get('Target')  # this should be a relational table
        if not orm_object: return  # object has probably been undone
        if orm_object.Table == 'RelationalTable':
            cols = [ x for x in orm_object.GetList('RelationalColumn') if x.InUse ]
        else:  # 'StarTable'
            cols = orm_object.GetList('StarColumn')
        cols.sort(cmp=lambda x,y: cmp(x.Seq, y.Seq))
        text = [ x.Name for x in cols ]
        lines = [orm_object.Name] + text
        return lines

    def AdjustEnd(self, other_end, seq=None):
        if not seq or not self.ClipRows:
            return ORMBox.AdjustEnd(self, other_end)

        clip_rows = [ int(x) for x in self.ClipRows.split(',') ]

        if not seq < len(clip_rows) - 1:  # for average calculation below
            return ORMBox.AdjustEnd(self, other_end)

        spacing = 0
        x, y = self.GetPos()
        bx, by = other_end
        lines = self.GetTextLines()
##        sizes = [ self.canvas.GetFullTextExtent(line)[0:2] for line in lines ]
##        sizes[0] = sizes[0][0], sizes[0][1] + 3

##        box_w = 5
##        box_h = 0
##        for line_w, line_h in sizes:
##            box_w = max(box_w, line_w)
##            box_h += line_h + spacing
##        box_w += 6
##        box_h += 6
        left, top, box_w, box_h = self.GetClipBox()
        right = left + box_w
        
##        left = x - box_w / 2
##        right = x + box_w / 2
##        top = y - box_h / 2

        if bx <= x:  # can add a constant here to avoid crossing over
            ax = left
        else:
            ax = right

##        ay = top + 2  # from h = 2 ... box_y+h
##        line_h = 0  # just in case sizes list is empty
##        for line_w, line_h in sizes[:seq + 1]:  # include the header line
##            ay += line_h + spacing
##        ay -= line_h / 2  # center on the line of text
        ay = top + (clip_rows[seq] + clip_rows[seq + 1]) / 2
            # average of the two rows to center the line

        return ax, ay

    def Draw(self, dc):
        orm_object = self.Get('Target')  # this should be a relational table
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)

##        if orm_object.Table == 'RelationalTable':
##            cols = orm_object.GetList('RelationalColumn')
##        else:  # 'StarTable'
##            cols = orm_object.GetList('StarColumn')
##        cols.sort(cmp=lambda x,y: cmp(x.Seq, y.Seq))
##        text = [ x.Name for x in cols ]
##
####        text = orm_object.Text or 'empty note'
####        if self.IsSelected and self.dcid == self.canvas.keyboard_target_dcid:
####            text = self.InsertCursor(text)  # add cursor to text for display
##
##        lines = [orm_object.Name, ''] + text  # .splitlines()

        spacing = 0
        x, y = self.GetPos()
        lines = self.GetTextLines()
        sizes = [ self.canvas.GetFullTextExtent(line)[0:2] for line in lines ]  # pull out only w and h
        sizes[0] = sizes[0][0], sizes[0][1] + 3

        w = 5
        h = 0
        for line_w,line_h in sizes:
            w = max(w, line_w)
            h += line_h + spacing
        box_w = w + 6
        box_h = h + 6
        box_x = x - box_w/2
        box_y = y - box_h/2

        pen = self.canvas.CachedPen('Blue', 1, wx.SOLID)
        dc.SetPen(pen)
        if self.GetSelected():
            dc.SetBrush(self.canvas.CachedBrush(SelectedColor))
        else:
            dc.SetBrush(self.canvas.CachedBrush((254, 254, 254)))
            # dc.SetBrush(self.canvas.CachedBrush('White'))
        dc.DrawRectangle(box_x,box_y,box_w,box_h)

        dc.SetFont(self.canvas.GetFont())
        dc.SetTextForeground('Black')
        dc.SetTextBackground('White')
        w = 3; h = 2
        clip_rows = [ h ]
        for i in range(len(lines)):
            line = lines[i]
            line_w, line_h = sizes[i]
            dc.DrawText(line, box_x+w, box_y+h)
            h += line_h + spacing
            clip_rows.append(h)

        tablenamebox = box_y + sizes[0][1] + 0  # line offset from top of box
        dc.DrawLine(box_x, tablenamebox, box_x + box_w, tablenamebox) # box table name

        r = wx.Rect(box_x,box_y,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)

        # save this information, but don't put on undo stack
        # because before this function didn't change the database
        # and in use might not always be followed by a SetUndo
        self.SetClipBox(box_x, box_y, box_w, box_h)
##        self.ClipRows = ','.join([ str(x) for x in clip_rows ])
        self.SetWithoutUndo('ClipRows', ','.join([ str(x) for x in clip_rows ]))

##    def Char(self, char):
##        if debug: print "char", ord(char), "'", char, "'"
##        orm_object = self.Get('Target')
##        orm_object.Text = self.ApplyChar(char, orm_object.Text)

##        if orm_object.Text:
##            if char == '\x08':  # this is for windows, cross platform?
##                orm_object.Text = orm_object.Text[:-1]
##            elif char == '\r':  # this is for windows, cross platform?
##                orm_object.Text += '\n'
##            else:
##                orm_object.Text += char
##        else:
##            if char in '\r\x08':  # this is for windows, cross platform?
##                pass
##            else:
##                orm_object.Text = char

    def IsUIDeletable(self):  # allow user to delete this graphic object
        return False

    def Delete(self):
        if self.InDelete: return
        self._SetInShell('InDelete', True)
        self.Follow(remove=True)  # before they are deleted
##        for shape in self.GetGraphicList('ORMNoteConnectorShape', 'NodeA'):
##            shape.Delete()
        if self.Target:
            self.Target.Delete()
        ORMShape.Delete(self)
        self._SetInShell('InDelete', None)

class RelationalConnectorShape(ORMConnector):
    def Draw(self, dc):
        orm_object = self.Get('Target')
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)

        x, y = self.GetPos()
        nodea, nodeb = self.NodeA, self.NodeB

        if orm_object.RelationalReference:
            seqa = orm_object.RelationalReference.Seq
        else:
            seqa = None
        seqb = orm_object.Seq
        
        enda = nodea.GetPos()
        endb = nodeb.GetPos()
        enda, endb = nodea.AdjustEnd(endb, seq=seqa), nodeb.AdjustEnd(enda, seq=seqb)

        platform = 20  # arbitrary width

        if enda[0] < nodea.GetPos()[0]:
            jointa = enda[0] - platform, enda[1]
        else:
            jointa = enda[0] + platform, enda[1]

        if endb[0] < nodeb.GetPos()[0]:
            jointb = endb[0] - platform, endb[1]
        else:
            jointb = endb[0] + platform, endb[1]

        minx = min(enda[0], endb[0], jointa[0], jointb[0])
        maxx = max(enda[0], endb[0], jointa[0], jointb[0])
        miny = min(enda[1], endb[1])
        maxy = max(enda[1], endb[1])
        
        box_w = maxx - minx + 6
        box_h = maxy - miny + 6

        if self.IsSelected:
            pen = self.canvas.CachedPen(SelectedColor, 3, wx.SOLID)
            dc.SetPen(pen)
            dc.DrawLine(enda[0], enda[1], endb[0], endb[1])

        pen = self.canvas.CachedPen('Blue', 1, wx.SOLID)
        dc.SetPen(pen)
        dc.DrawLine(enda[0], enda[1], jointa[0], jointa[1])
        dc.DrawLine(jointa[0], jointa[1], jointb[0], jointb[1])
        dc.DrawLine(jointb[0], jointb[1], endb[0], endb[1])
        self.SetTempPos((jointa[0] + jointb[0]) / 2, (jointa[1] + jointb[1]) / 2)

        r = wx.Rect(minx,miny,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)

    def IsUIDeletable(self):  # allow user to delete this graphic object
        return False

    def Delete(self):
        if self.InDelete: return  # prevent loops
        self._SetInShell('InDelete', True)
        self.Follow(remove=True)  # before they are deleted
        if self.Target:
            self.Target.Delete()
        ORMShape.Delete(self)
        self._SetInShell('InDelete', None)

# Register shapes
_subtype = {
    # generic shapes
#    'Shape': ORMShape,  # abstract shape (no instances)
##    'Box': ORMBox,
##    'Circle': ORMCircle,
##    'Connector': ORMConnector,
##    'FollowText': ORMFollowText,
    # ORM shapes
    'ORMObjectTypeShape': ORMObjectTypeShape,
    'ORMFactTypeShape': ORMFactTypeShape,
    'ORMFactReadingShape': ORMFactReadingShape,
    # 'ORMObjectificationNameShape': ORMObjectificationNameShape,  # ??always above & centered??
    'ORMSampleTableShape': ORMSampleTableShape,
    'ORMRoleNameShape': ORMRoleNameShape,
    'ORMNoteShape': ORMNoteShape,
    # 'ORMTextShape': ORMTextShape,
    # 'ORMSampleDataShape': ORMSampleDataShape,
    'ORMConstraintShape': ORMConstraintShape,
    'ORMSubtypeConstraintShape': ORMSubtypeConstraintShape,
    # 'ORMValueConstraintShape': ORMValueConstraintShape,
    # 'ORMFrequencyConstraintShape': ORMFrequencyConstraintShape,
    'ORMRoleConnectorShape': ORMRoleConnectorShape,
    'ORMSubtypeConnectorShape': ORMSubtypeConnectorShape,
    'ORMNoteConnectorShape': ORMNoteConnectorShape,
    'ORMConstraintConnectorShape': ORMConstraintConnectorShape,
    'ORMSubtypeConstraintConnectorShape': ORMSubtypeConstraintConnectorShape,

    'RelationalTableShape': RelationalTableShape,
    'RelationalConnectorShape': RelationalConnectorShape,
    'RelationalTableConnectorShape': RelationalConnectorShape,  # temporary
}
for k, v in _subtype.iteritems():
    Data.DB.RegisterSubtype(k, v)

RingConstraints = ['Irreflexive', 'Symmetric', 'Asymmetric', 'Antisymmetric', 'Intransitive', 'Acyclic', 'Acyclic and Intransitive', 'Asymmetric and Intransitive', 'Symmetric and Intransitive', 'Symmetric and Irreflexive']

