#!/usr/bin/env python
# orm report definition

# Copyright 2004, 2005, 2006, 2007, 2008 by Brian C. Christensen

#    This file is part of GanttPV.
#
#    GanttPV is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    GanttPV is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with GanttPV; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# 080517 - first version of this program based on GanttReport

import wx, wx.grid
import datetime
from wx.lib.dialogs import MultipleChoiceDialog as wxMultipleChoiceDialog
import Data, UI, ID, Menu, ReportAids
import ORMCanvas
# import images
import re
import os
import sys

debug = Data.debug
is24 = 1

if debug: print "load ORMReport.py"

#------------------ ORM Report Frame -----------------------------------
def OpenReport(parent, reportID):  # parent is parent frame
#    nb = parent.book
    nb = parent
    log = sys.stdout
#    frame = parent.book
    
##    pid = 2  # need to create this
##    rtid = Data.SearchByColumn(Data.Database['ReportType'], {'Name': 'ORM Diagram'}, unique=True)
##    columns = Data.GetSuggestedColumns(rtid)
##    report = {'Table': 'Report', 'ReportTypeID': rtid, 'AdjustRowOption': 'ORM Diagram'}
##    reportID = Data.AddReport(pid, report, columns)
##    #### db = Data._dbDatabase(Data.Database)  # will get objects from this database
    report = Data.DBObject.GetObject('Report', reportID)

    pnl = wx.Panel(nb, wx.ID_ANY,size=(200,30))
    pnl2 = ORMCanvas.ControlPanel(pnl,wx.ID_ANY)
    pnl2.demo = parent  # so control panel can save data at application level
    win = ORMCanvas.MyCanvas(pnl, wx.ID_ANY, log, report=report)
    pnl2.win = win  # so toolbar buttons can reach canvas
    pnl.win = win  # so ormreport can call canvas
    win.demo = parent  # so canvas can save data at application level
    win.demo.draw_mode = False  # -- IMPORTANT -- where should this really be?
    sz = wx.BoxSizer(wx.VERTICAL)
    sz.Add(pnl2,0,wx.EXPAND|wx.ALL,5)
    sz.Add(win,1,wx.EXPAND)
    pnl.SetSizerAndFit(sz)

    wx.EVT_KEY_DOWN(win, pnl2.OnKeyDown)
    wx.EVT_KEY_UP(win, pnl2.OnKeyUp)
    return pnl

class ORMReportFrame(UI.DiagramFrame):
    def __init__(self, reportid, *args, **kwds):
        if debug: "Start ORMReport init"
        if debug: print 'reportid', reportid
        UI.DiagramFrame.__init__(self, *args, **kwds)

        # these three commands were moved out of UI.ReportFrame's init
##        self.report_window = GanttChartGrid(self.Report_Panel, reportid)
##        self.Report = self.report_window
        self.report_window = OpenReport(self.Report_Panel, reportid)
        self.Report = self.report_window
        self.ReportID = reportid

        # self.Report, aka report_window, is a grid, not a report or a frame
        # report_window is referenced in one place in UI.ReportFrame.do_layout
        # Report is referenced in several scripts
        #     -- Alexander

        # Data.OpenReports[reportid] = self

        self.set_properties()  # these are in the parent class
        self.do_layout()

        Menu.doAddScripts(self)
        Menu.FillWindowMenu(self)
#        Menu.AdjustMenus(self)    # -- IMPORTANT - need to fix this
        self.SetReportTitle()

        # file menu events
        wx.EVT_MENU(self, wx.ID_NEW,    Menu.doNew)
        wx.EVT_MENU(self, wx.ID_OPEN,   Menu.doOpen)
        wx.EVT_MENU(self, wx.ID_CLOSE,  self.doClose)
        wx.EVT_MENU(self, wx.ID_CLOSE_ALL, Menu.doCloseReports)
        wx.EVT_MENU(self, wx.ID_SAVE,   Menu.doSave)
        wx.EVT_MENU(self, wx.ID_SAVEAS, Menu.doSaveAs)
        wx.EVT_MENU(self, ID.M_AUTO_SAVE, Menu.doAutoSaveOnQuit)
        wx.EVT_MENU(self, wx.ID_REVERT, Menu.doRevert)
        wx.EVT_MENU(self, ID.M_REVERT_OPEN, Menu.doRevertOpen)
        wx.EVT_MENU(self, wx.ID_EXIT, Menu.doExit)

        # edit menu events
        wx.EVT_MENU(self, wx.ID_UNDO, self.doUndo)
        wx.EVT_MENU(self, wx.ID_REDO, self.doRedo)
        wx.EVT_MENU(self, wx.ID_COPY, self.OnCopy)
        wx.EVT_MENU(self, wx.ID_PASTE, self.OnPaste)

##        wx.EVT_MENU(self, ID.M_INSERT_ROW,     self.OnInsertRow)
##        wx.EVT_MENU(self, ID.M_INSERT_ROW_ABOVE, self.OnInsertRow)
##        wx.EVT_MENU(self, ID.M_INSERT_CHILD,    self.OnInsertChild)
##        wx.EVT_MENU(self, ID.M_INSERT_RELATED,  self.OnInsertRow)
##        wx.EVT_MENU(self, ID.M_DELETE_ROW,      self.OnDeleteRow)
##        wx.EVT_MENU(self, ID.M_SHIFT_LEFT,     self.OnShiftRow)
##        wx.EVT_MENU(self, ID.M_SHIFT_RIGHT,   self.OnShiftRow)
##        wx.EVT_MENU(self, ID.M_MOVE_ROW_UP,     self.OnMoveRow)
##        wx.EVT_MENU(self, ID.M_MOVE_ROW_DOWN,   self.OnMoveRow)
##
##        wx.EVT_MENU(self, ID.M_EDIT_PROJECT_NAME,   self.OnEditProjectName)
##        wx.EVT_MENU(self, ID.M_EDIT_REPORT_NAME,   self.OnEditReportName)

        # action menu events
##        wx.EVT_MENU(self, ID.M_ASSIGN_PREREQUISITE, self.OnPrerequisite)
##        wx.EVT_MENU(self, ID.M_LINK_TASKS,          self.OnPrerequisite)
##        wx.EVT_MENU(self, ID.M_ASSIGN_RESOURCE,     self.OnAssignResource)
##        wx.EVT_MENU(self, ID.M_RESOURCE_TO_GROUP,   self.OnResourceGroup)
##        wx.EVT_MENU(self, ID.M_DEFINE_RESOURCE_GROUP, self.OnResourceGroup)
##        wx.EVT_MENU(self, ID.M_ASSIGN_TASK,         self.OnAssignResource)

        # view menu events
##        wx.EVT_MENU(self, ID.M_INSERT_COLUMN, self.OnInsertColumn)
##        wx.EVT_MENU(self, ID.M_DELETE_COLUMN, self.OnDeleteColumn)
##        wx.EVT_MENU(self, ID.M_MOVE_COLUMN_LEFT, self.OnMoveColumn)
##        wx.EVT_MENU(self, ID.M_MOVE_COLUMN_RIGHT, self.OnMoveColumn)
##        wx.EVT_MENU(self, ID.M_SCROLL_LEFT_FAST, self.OnScroll)
##        wx.EVT_MENU(self, ID.M_SCROLL_LEFT, self.OnScroll)
##        wx.EVT_MENU(self, ID.M_SCROLL_RIGHT, self.OnScroll)
##        wx.EVT_MENU(self, ID.M_SCROLL_RIGHT_FAST, self.OnScroll)
##        wx.EVT_MENU(self, ID.M_SCROLL_TO_TASK, self.OnScrollToTask)

        # script menu events
        wx.EVT_MENU(self, ID.FIND_SCRIPTS, Menu.doFindScripts)
        wx.EVT_MENU(self, ID.REFRESH_SCRIPTS, Menu.doRefreshScripts)
        wx.EVT_MENU(self, ID.REPEAT_SCRIPT, Menu.doRepeatScript)
        wx.EVT_MENU_RANGE(self, ID.FIRST_SCRIPT, ID.LAST_SCRIPT, Menu.doScript)

        # window menu events
        wx.EVT_MENU_RANGE(self, ID.FIRST_WINDOW, ID.LAST_WINDOW, self.doBringWindow)

        # help menu events
        wx.EVT_MENU(self, wx.ID_ABOUT, Menu.doShowAbout)
        wx.EVT_MENU(self, ID.QUICK_START, self.doQuick)
        wx.EVT_MENU(self, ID.ORM_QUICK_START, self.doORMQuick)
        wx.EVT_MENU(self, ID.SHORT_CUTS, self.doShort)
        wx.EVT_MENU(self, ID.HOME_PAGE, Menu.doHome)
        wx.EVT_MENU(self, ID.HELP_PAGE, Menu.doHelp)
        wx.EVT_MENU(self, ID.HELP_BOOK, Menu.doBook)
        wx.EVT_MENU(self, ID.FORUM, Menu.doForum)

        # frame events
        wx.EVT_ACTIVATE(self, self.OnActivate)
        wx.EVT_CLOSE(self, self.doClose)
        wx.EVT_SIZE(self, self.OnSize)
        wx.EVT_MOVE(self, self.OnMove)

        # grid events
##        wx.grid.EVT_GRID_COL_SIZE(self, self.OnColSize)
##        wx.grid.EVT_GRID_ROW_SIZE(self, self.OnRowSize)
##        wx.grid.EVT_GRID_EDITOR_SHOWN(self, self.OnEditorShown)
##        wx.grid.EVT_GRID_EDITOR_HIDDEN(self, self.OnEditorHidden)

        # tool bar events
##        wx.EVT_TOOL(self, ID.INSERT_ROW, self.OnInsertRow)
##        wx.EVT_TOOL(self, ID.DUPLICATE_ROW, self.OnDuplicateRow)
##        wx.EVT_TOOL(self, ID.DELETE_ROW, self.OnDeleteRow)
##        wx.EVT_TOOL(self, ID.MOVE_UP, self.OnMoveRow)
##        wx.EVT_TOOL(self, ID.MOVE_DOWN, self. OnMoveRow)
##        wx.EVT_TOOL(self, ID.PREREQUISITE, self.OnPrerequisite)
##        wx.EVT_TOOL(self, ID.ASSIGN_RESOURCE, self.OnAssignResource)
##
##        wx.EVT_TOOL(self, ID.HIDE_ROW, self.OnHide)
##        wx.EVT_TOOL(self, ID.SHOW_HIDDEN, self.OnShowHidden)
##
##        wx.EVT_TOOL(self, ID.INSERT_COLUMN, self.OnInsertColumn)
##        wx.EVT_TOOL(self, ID.DELETE_COLUMN, self.OnDeleteColumn)
##        wx.EVT_TOOL(self, ID.MOVE_LEFT, self.OnMoveColumn)
##        wx.EVT_TOOL(self, ID.MOVE_RIGHT, self.OnMoveColumn)
##
##        # wx.EVT_TOOL(self, ID.COLUMN_OPTIONS, self.OnColumnOptions)
##
##        wx.EVT_TOOL(self, ID.SCROLL_LEFT_FAR, self.OnScroll)
##        wx.EVT_TOOL(self, ID.SCROLL_LEFT, self.OnScroll)
##        wx.EVT_TOOL(self, ID.SCROLL_RIGHT, self.OnScroll)
##        wx.EVT_TOOL(self, ID.SCROLL_RIGHT_FAR, self.OnScroll)
##        wx.EVT_TOOL(self, ID.SCROLL_TO_TASK, self.OnScrollToTask)

#        self.Report.ForceRefresh()  # -- IMPORTANT - fix this, too
        if debug: "End ORMReport init"

    # used my any commands that move grid rows or columns
    def FinishEdit(self):
        '''If a cell is being editted, save it before moving rows'''
        self.Report.SaveEditControlValue()
        self.Report.DisableCellEditControl()

    # temporary until I can figure out a better way to redraw at the right times
    def doUndo(self, event):
        Menu.doUndo(event)
        self.Report.win.RedrawAll()
    def doRedo(self, event):
        Menu.doRedo(event)
        self.Report.win.RedrawAll()

    # ------ Tool Bar Commands ---------

    def OnInsertRow(self, event):
        self.FinishEdit()
        insertAfter = (event.GetId() in (ID.INSERT_ROW, ID.M_INSERT_ROW))
        self.CreateRow(insertAfter, False)

    def OnInsertChild(self, event):
        self.FinishEdit()
        self.CreateRow(True, True)

    def CreateRow(self, insertAfter, makeChild):
        if debug: print "Start InsertRow"

        r = Data.Report[self.ReportID]
        rt = Data.ReportType[r['ReportTypeID']]
        ta = rt.get('TableA')
        if not ta or ta in ('Report', 'ReportColumn', 'ReportRow', 'ReportType', 'ColumnType'): return  # need special handling

        sel = self.Report.GetSelectedRows()
        if sel:
            # consider only the best-ranking rows
            rowlevels = self.Report.table.rowlevels
            level = min([rowlevels[x] for x in sel])
            sel = [x for x in sel if rowlevels[x] == level]

            if insertAfter:
                row = max(sel)
            else:
                row = min(sel)
        elif self.Report.table.rows:
            row = self.Report.GetGridCursorRow()
        else:
            row = -1

        if row == -1:
            parent = None  # no rows are visible
        else:
            rowid = self.Report.table.rows[row]
            rr = Data.ReportRow[rowid]
            while row > 0 and rr['TableName'] != ta:  # find a primary row
                row -= 1
                rowid = self.Report.table.rows[row]
                rr = Data.ReportRow[rowid]
            if makeChild:
                parent = rowid  # child of selection
            else:
                parent = rr.get('ParentRow')  # same parent as selection

        change = {'Table': ta, 'Name': '--'}
        if ta != 'Project':
            change['ProjectID'] = r.get('ProjectID')
        if parent:
            change[ta + 'ID'] = Data.ReportRow.get(parent, {}).get('TableID')
        undo = Data.Update(change)
        newid = undo['ID']

        change = {'Table': ta, 'ID': newid, 'Name': _(ta) + " " + str(newid)}
        undo = Data.Update(change)

        change = {'Table': 'ReportRow', 'ReportID': self.ReportID, 'TableName': ta, 'TableID': newid, 'ParentRow': parent}
        undo = Data.Update(change)  # created here to control where inserted
        newrowid = undo['ID']

        rlist = Data.GetRowList(self.ReportID)
        if row == -1:
            pos = len(rlist)  # no rows are visible
        else:
            pos = rlist.index(rowid)
            if insertAfter: pos += 1
        rlist.insert(pos, newrowid)
        Data.ReorderReportRows(self.ReportID, rlist)

        Data.SetUndo(_('Insert %s') % _(ta))
        if debug: print "End InsertRow"

        # move current cell to inserted row
        try:
            r = self.Report.table.rows.index(newrowid)
        except:
            pass
        else:
            c = self.Report.GetGridCursorCol()
            self.Report.SetGridCursor(r, c)
            self.Report.MakeCellVisible(r, c)

    def OnDuplicateRow(self, event):
        self.FinishEdit()
        sel = self.Report.GetSelectedRows()  # current selection
        if len(sel) == 0:
            if debug: print "can't duplicate, empty selection"
            return
        rtid = Data.Report[self.ReportID].get('ReportTypeID')
        tablea = Data.ReportType[rtid].get('TableA')
        new = []
        for s in sel:
            rid = self.Report.table.rows[s]
            ta = Data.ReportRow[rid]['TableName']
            if ta != tablea: continue  # only duplicate rows of primary table type
            rcopy = Data.ReportRow[rid].copy()  # report row
            tid = rcopy['TableID']
            tcopy = Data.Database[ta][tid].copy()  # table row

            tcopy['Table'] = ta; del tcopy['ID']
            undo = Data.Update(tcopy)
            rcopy['Table'] = 'ReportRow'; del rcopy['ID']
            rcopy['TableID'] = undo['ID']
            undo = Data.Update(rcopy)
            new.append(undo['ID'])

        rlist = Data.GetRowList(self.ReportID)  # list of row id's in display order
        where = max(sel) + 1
        # print "rlist", rlist
        # print "new", new
        rlist[where:where] = new
        # print "new rlist", rlist
        Data.ReorderReportRows(self.ReportID, rlist)

        Data.SetUndo(_('Duplicate Row'))

    def OnDeleteRow(self, event):  # this is a shallow delete
        if debug: print "Start OnDeleteRow"
        self.FinishEdit()
        sel = self.Report.GetSelectedRows()  # current selection
        if len(sel) < 1:
            if debug: print "can't delete, no rows selected"
            return  # only move if rows selected
        change = { 'Table': None, 'ID': None, 'zzStatus': 'deleted' }
        cnt = 0
        for s in sel:
            rid = self.Report.table.rows[s]
            ta = Data.ReportRow[rid].get('TableName')
            id = Data.ReportRow[rid].get('TableID')
            if not id: continue  # silently skip invalid table id's
            if ta == 'Project' and id == 1: continue  # certain projects and reports can't be deleted
            elif ta == 'Report' and (id == 1 or id == 2): continue
            if Data.Database[ta][id].get('zzStatus') == 'deleted':
                change['zzStatus'] = None
            else:
                change['zzStatus'] = 'deleted'
            change['Table'] = ta
            change['ID'] = id
            undo = Data.Update(change)
            cnt += 1
        if cnt > 0: Data.SetUndo(_('Delete/Reactivate Row'))
        if debug: print "End OnDeleteRow"


    def OnShiftRow(self, event):
        """ Shift selected row one level up or down in hierarchy """
        # if left -> set parent to parent's parent
        # if right -> find prior record with same parent; make that record the parent

        self.FinishEdit()
        sel = self.Report.GetSelectedRows()
        # if sel and len(sel) > 1:  # not more than one selected row
        #     return
        
        if not sel:
            if not self.Report.table.rows:
                return
            sel = [self.Report.GetGridCursorRow()]  # if no rows are selected, use cursor row    
        sel.sort()

        rows = self.Report.table.rows
        rowlevels = self.Report.table.rowlevels
        firstrowid = rows[sel[0]]

        if event.GetId() in (ID.M_SHIFT_LEFT, ):
            # ignore lower-ranking child rows
            level = min([rowlevels[x] for x in sel])
            if level < 1: level = 1
            sel = [x for x in sel if rowlevels[x] == level]

        for row in sel:
            rid = rows[row]
            ta = Data.ReportRow[rid].get('TableName')
            id = Data.ReportRow[rid].get('TableID')
            if (ta not in Data.Database or id not in Data.Database[ta] or
                Data.Database[ta][id].get('zzStatus') == 'deleted'):  # silently skip invalid table id's
                return

            table = Data.Database[ta]
            parent = table[id].get(ta + 'ID')

            if event.GetId() in (ID.M_SHIFT_LEFT, ):
                if parent not in table or table[id].get('zzStatus') == 'deleted': continue
                newparent = table[parent].get(ta + 'ID')
                change = { 'Table': ta, 'ID': id, ta + 'ID': newparent }
                Data.Update(change)

            else: # ID.Shift_Right
                step = 1
                for s in range(row-1, -1, -1):  # check each prior row in the report
                    rid2 = rows[s]
                    ta2 = Data.ReportRow[rid2].get('TableName')
                    if ta != ta2: continue  # ignore rows that aren't in the same table
                    id2 = Data.ReportRow[rid2].get('TableID')
                    if (id2 not in Data.Database[ta2] or
                        Data.Database[ta2][id2].get('zzStatus') == 'deleted'):  # silently skip invalid table id's
                        continue
                    parent2 = table[id2].get(ta2 + 'ID')
                    if parent != parent2: continue  # the first prior row with the same parent will be this records new parent

                    change = { 'Table': ta, 'ID': id, ta + 'ID': id2 }
                    Data.Update(change)
                    break

        if event.GetId() in (ID.M_SHIFT_LEFT, ):
            Data.SetUndo(_('Shift Left'))
        else:
            Data.SetUndo(_('Shift Right'))

        # move current cell so it is on the same record
        try:
            r = self.Report.table.rows.index(firstrowid)
        except:
            pass
        else:
            c = self.Report.GetGridCursorCol()
            old_r = self.Report.GetGridCursorRow()
            if r != old_r:
                self.Report.SetGridCursor(r, c)
                self.Report.MakeCellVisible(r, c)

    def OnMoveRow(self, event):
        """ Move selected rows up or down by one position """
        self.FinishEdit()
        sel = self.Report.GetSelectedRows()
        if not sel:
#            return
            sel = [self.Report.GetGridCursorRow()]  # if no rows are selected, use cursor row
            if not self.Report.table.rows: return  # no cursor row (i.e. no rows in report)

        if event.GetId() in (ID.MOVE_UP, ID.M_MOVE_ROW_UP):
            step = -1
        else: # ID.MOVE_DOWN
            step = 1

        rows = self.Report.table.rows
        rowlevels = self.Report.table.rowlevels

        # move only the best-ranking rows of the selection
        level = min([rowlevels[x] for x in sel])
        move = [x for x in sel if rowlevels[x] == level]
        move.sort()

        row_set = dict.fromkeys(move)  # set of row positions
        move_row_ids = [rows[x] for x in move]  # list of report row ids

        # determine whether the selection is contiguous
        #   (ignoring children and invisible rows)
        start = move[0]
        end = move[-1] + 1

        for x in xrange(start, end):
            if x in row_set:
                continue
            lev = rowlevels[x]
            if lev <= level:
                contiguous = False
                break
        else:
            contiguous = True

            # find the next visible, same-parent row
            next_row_id = None

            if step < 0:
                x = start - 1
            else:
                x = end

            while 0 <= x < len(rows):
                lev = rowlevels[x]
                if lev == level:
                    next_row_id = rows[x]
                    break
                elif lev < level:
                    break
                x += step

        # check for invisible rows
        if not self.Report.table.report.get('ShowHidden'):
            # report may include rows that are not displayed

            # get report rows from database; convert row positions
            rows, rowlevels = Data.GetRowLevels(self.ReportID)

            positions = {}
            for x, rid in enumerate(rows):
                positions[rid] = x

            row_set = {}
            for rowid in move_row_ids:
                if rowid in positions:
                    pos = positions[rowid]
                    row_set[pos] = None
            if not row_set:
                return

            move = row_set.keys()
            move.sort()

            start = move[0]
            end = move[-1] + 1
        else:
            # don't modify the row list directly
            rows = rows[:]

        if not contiguous:
            # consolidate rows
            if step < 0:
                end = start + len(move)
            else:
                start = end - len(move)

            move.reverse()
            for x in move:
                del rows[x]
            rows[start:start] = move_row_ids

        elif next_row_id:
            # move next row to the other side of selection
            if step < 0:
                dest = end - 1
            else:
                dest = start
            rows.remove(next_row_id)
            rows.insert(dest, next_row_id)

        else:
            # nothing changed
            return

        # apply the new row order
        Data.ReorderReportRows(self.ReportID, rows)
        Data.SetUndo(_('Move Row'))

    def OnEditProjectName(self, event):
        projectid = Data.Report[self.ReportID].get('ProjectID')
        if projectid:
            Menu.doEditName('Project', projectid)

    def OnEditReportName(self, event):
        Menu.doEditName('Report', self.ReportID)

    def OnPrerequisite(self, event):
        # list tasks in the same order they appear now  -- use self.Report.table.rows
        # highlight the ones that are currently prerequisites
        sel = self.Report.GetSelectedRows()  # current selection
        if len(sel) < 1:
            if debug: print "must select at least one row"
            return
        elif len(sel) > 1:
            sel.sort()  # chain tasks in the order they appear on report
            rows = self.Report.table.rows
            alltids = [ Data.ReportRow[rows[x]].get('TableID') for x in sel if Data.ReportRow[rows[x]].get('TableName') == 'Task' ]
            tids = [ x for x in alltids if not Data.Task[x].get('SubtaskCount') ]
            if len(tids) > 1:
                for i in range(len(tids) - 1):  # try to match and link each pair
                    # look for an existing dependency record
                    did = Data.FindID('Dependency', 'PrerequisiteID', tids[i], 'TaskID', tids[i+1])
                    if did:
                        if Data.Database['Dependency'][did].get('zzStatus') == 'deleted':
                            change = { 'Table': 'Dependency', 'ID': did, 'zzStatus': None }
                            Data.Update(change)
                    else:
                        change = { 'Table': 'Dependency', 'PrerequisiteID': tids[i], 'TaskID': tids[i+1] }
                        Data.Update(change)
                Data.SetUndo(_("Set Dependencies"))
            return
        rows = self.Report.table.rows
        sel = sel[0]
        rowid = rows[sel]  # get selection's task id
        ta = Data.ReportRow[rowid].get('TableName')
        if ta != 'Task': return  # only on task rows
        sid = Data.ReportRow[rowid].get('TableID')
        assert sid, "tried to assign prereq's to an invalid task id"

        alltid = [ Data.ReportRow[x].get('TableID') for x in rows if Data.ReportRow[x].get('TableName') == 'Task']
        tid = [ x for x in alltid if Data.Task[x].get('zzStatus') != 'deleted' and x != sid]  # active displayed tasks
        tname = [ Data.Task[x].get('Name', "") or "--" for x in tid ]

        status = [0] * len(tid)  # array of 0's
        for k, v in Data.Dependency.iteritems():
            if sid != v.get('TaskID'): continue
            p = v.get('PrerequisiteID')
            try:
                i = tid.index(p)
            except: pass
            else:
                if v.get('zzStatus') != 'deleted':
                    status[i] = k
                else:
                    status[i] = -k

        dialog = MultiSelection(self, -1, "Assign Prerequisite", size=(240,320))
        dialog.Instructions.SetLabel("Select prerequisite tasks:")
        # dialog.SelectionListBox.Clear()
        dialog.SelectionListBox.Set(tname)
        for i, v in enumerate(status):
            if v > 0:
                dialog.SelectionListBox.SetSelection(i)

        dialog.ID = sid
        dialog.TargetIDs = tid
        dialog.Status = status

        dialog.LinkTable = 'Dependency'
        dialog.Column1 = 'TaskID'
        dialog.Column2 = 'PrerequisiteID'
        dialog.Message = 'Set Dependencies'

        dialog.Centre()  # centers dialog on screen
        dialog.ShowModal()

    def OnResourceGroup(self, event):
        # list names in the same order they appear now  -- use self.Report.table.rows
        # highlight the ones that are currently linked
        kind = 'Resource'

        # edit selection
        sel = self.Report.GetSelectedRows()  # current selection
        if len(sel) != 1:
            if debug: print "must select one row"
            return
        rows = self.Report.table.rows
        sel = sel[0]
        rowid = rows[sel]  # get selection's task id
        ta = Data.ReportRow[rowid].get('TableName')
        if ta != kind: return  # wrong kind of row
        sid = Data.ReportRow[rowid].get('TableID')
        assert sid, "tried to assign to an invalid id"

        # create list of names for menu
        allids = [ Data.ReportRow[x].get('TableID') for x in rows if Data.ReportRow[x].get('TableName') == kind]
        type = bool(Data.Resource[sid].get('GroupType'))
        # remove deleted, self, and same type (group vs. not group)
        ids = [ x for x in allids if Data.Resource[x].get('zzStatus') != 'deleted' and (type != bool(Data.Resource[x].get('GroupType'))) and x != sid]
        names = [ Data.Resource[x].get('Name', "") or "--" for x in ids ]

        status = [0] * len(ids)  # array of 0's
        if type:
            matchfield = 'ResourceGroupID'
            getfield = 'ResourceID'
            prompt = _('Select resources:')
        else:
            matchfield = 'ResourceID'
            getfield = 'ResourceGroupID'
            prompt = _('Select resource groups:')
        for k, v in Data.Database['ResourceGrouping'].iteritems():
            if sid != v.get(matchfield): continue
            p = v.get(getfield)
            try:
                i = ids.index(p)
            except: pass
            else:
                if v.get('zzStatus') != 'deleted':
                    status[i] = k
                else:
                    status[i] = -k

        dialog = MultiSelection(self, -1, "Resource Grouping", size=(240,320))
        dialog.Instructions.SetLabel(prompt)
        # dialog.SelectionListBox.Clear()
        dialog.SelectionListBox.Set(names)
        for i, v in enumerate(status):
            if v > 0:
                dialog.SelectionListBox.SetSelection(i)

        dialog.ID = sid
        dialog.TargetIDs = ids
        dialog.Status = status

        dialog.LinkTable = 'ResourceGrouping'
        dialog.Column1 = matchfield
        dialog.Column2 = getfield
        dialog.Message = prompt

        dialog.Centre()  # centers dialog on screen
        dialog.ShowModal()

    def OnAssignResource(self, event):
        # list resources in alphabetical order
        # highlight the ones that are currently assigned
        sel = self.Report.GetSelectedRows()  # user's selection
        if len(sel) != 1:
            if debug: print "only assignments for one task"
            return  # only move if rows selected
        rows = self.Report.table.rows
        sel = sel[0]  # make it not a list
        rowid = rows[sel]  # get selected task's id
        sid = Data.ReportRow[rowid].get('TableID')

        assert sid and sid > 0, "tried to assign to an invalid id"

        ta = Data.ReportRow[rowid].get('TableName')
        if ta == 'Task':
            res = []
            for k, v in Data.Resource.iteritems():
                if v.get('zzStatus') == 'deleted': continue
                name = v.get('Name') or '--'
                firstname = v.get('FirstName')
                if firstname: name += ", " + firstname
                res.append((k, name))
            res.sort()
            ids, names = zip(*res)
            column1 = 'TaskID'
            column2 = 'ResourceID'
            dialogPrompt = _("Select assigned resources:")
        elif ta == 'Resource':
            if Data.Database[ta][sid].get('GroupType') == 'Work': return  # can't assigne tasks to resource groups
            tasks = []
            for k, v in Data.Task.iteritems():
                if v.get('zzStatus') == 'deleted': continue
                name = v.get('Name') or '--'
                tasks.append((k, name))
            tasks.sort()
            ids, names = zip(*tasks)
            column1 = 'ResourceID'
            column2 = 'TaskID'
            dialogPrompt = _("Select assigned tasks:")
        else:
            return

        assigns = {}
        for k, v in Data.Assignment.iteritems():
            if sid != v.get(column1): continue
            id2 = v.get(column2)
            if v.get('zzStatus') == 'deleted':
                assigns[id2] = -k
            else:
                assigns[id2] = k
        status = [assigns.get(id, 0) for id in ids]

        dialog = MultiSelection(self, -1, "Assign Resource", size=(240,320))
        dialog.Instructions.SetLabel(dialogPrompt)
        # dialog.SelectionListBox.Clear()
        dialog.SelectionListBox.Set(names)
        for i, v in enumerate(status):
            if v > 0:
                dialog.SelectionListBox.SetSelection(i)

        dialog.ID = sid
        dialog.TargetIDs = ids
        dialog.Status = status

        dialog.LinkTable = 'Assignment'
        dialog.Column1 = column1
        dialog.Column2 = column2
        dialog.Message = 'Set Assignments'

        dialog.Centre()  # centers dialog on screen
        dialog.ShowModal()

    def OnHide(self, event):
        self.FinishEdit()
        sel = self.Report.GetSelectedRows()  # user's selection
        Menu.onHide(self.Report.table, event, sel)

    def OnShowHidden(self, event):
        self.FinishEdit()
        Menu.onShowHidden(self, event)

    def OnInsertColumn(self, event):
        if debug: print "Start OnInsertColumn"
        insertAfter = False

        self.FinishEdit()
        r = Data.Report[self.ReportID]
        rtid = r.get('ReportTypeID')  # these determine what kind of columns can be inserted
        also = Data.ReportType[rtid].get('Also')

        menuid = []  # list of types to be displayed for selection
        rlist = Data.GetRowList(2)  # report 2 defines the sequence of the report type selection list
        for k in rlist:
            rr = Data.ReportRow[k]
            hidden = rr.get('Hidden', False)
            table = rr.get('TableName')  # should always be 'ReportType' or 'ColumnType'
            id = rr.get('TableID')
            if (not hidden) and table == 'ColumnType' and id:
                xrtid = Data.ColumnType[id].get('ReportTypeID')
                active = Data.ColumnType[id].get('zzStatus') != 'deleted'
                if active and xrtid and ( (rtid == xrtid) or (also and (also == xrtid)) ):
                    menuid.append( id )
        def getname(id):
            name = Data.ColumnType[x].get('Label')
            if name:
                name = _(name)  # translate label
            else:
                name = Data.ColumnType[x].get('Name')
            return name
        menutext = [ getname(x) for x in menuid ]
        for i, v in enumerate(menutext):  # remove line feeds before menu display
            if '\n' in v:
                menutext[i] = v.replace('\n', ' ')
        menuT = [ Data.ColumnType[x].get('T') for x in menuid ]

        # sort menu
        menus = zip(menuT, menutext, menuid)
        menus.sort()
        menuT, menutext, menuid = zip(*menus)
        if debug: print menuid, menutext

        dlg = SearchSelection(self, -1, "New Columns", size=(240, 320))
        dlg.Instructions.SetLabel("Select columns to add:")
        dlg.SelectionListBox.Set(menutext)
        # dlg = wxMultipleChoiceDialog(self, "Select columns to add:", "New Columns", menutext, style=wx.DEFAULT_FRAME_STYLE, size=(240, 320))
        dlg.Centre()
        if (dlg.ShowModal() != wx.ID_OK): return
        newlist = dlg.GetValue()
        addlist = []

        change = { 'Table': 'ReportColumn', 'ReportID': self.ReportID }  # new record because no ID specified
        for n in newlist:
            change['ColumnTypeID'] = menuid[n]
            ct = Data.ColumnType[menuid[n]]
            if ct['Name'] == 'Day/Gantt':  # leaving this for compatibility with version 0.1
                change['Periods'] = 21
                change['FirstDate'] = Data.GetToday()
                undo = Data.Update(change)
                del change['Periods']; del change['FirstDate']
            elif ct.get('AccessType') == 's':
                change['Periods'] = 14
                # change['FirstDate'] = Data.GetToday()
                undo = Data.Update(change)
                del change['Periods']
                # del change['FirstDate']
            else:
                change['Width'] = ct.get('Width')
                undo = Data.Update(change)
                del change['Width']
            addlist.append(undo['ID'])

        clist = Data.GetColumnList(self.ReportID)
        sel = self.Report.GetSelectedCols()
        # if debug: print 'selection', sel
        if sel:
            if insertAfter:
                pos = max(sel)
            else:
                pos = min(sel)
        else:
            # pos = self.Report.GetGridCursorCol()
            pos = len(clist)
        if insertAfter: pos += 1
        clist[pos:pos] = addlist
        Data.ReorderReportColumns(self.ReportID, clist)

        Data.SetUndo(_('Insert Column'))
        if debug: print "End OnInsertColumn"

    def OnDeleteColumn(self, event):
        if debug: print "Start OnDeleteColumn"
        self.FinishEdit()
        sel = self.Report.GetSelectedCols()
        if not sel: return

        clist = Data.GetColumnList(self.ReportID)
        cids = self.Report.table.columns

        sel_cids = [cids[x] for x in sel]
        sel_map = dict.fromkeys(sel_cids)
        clist = [x for x in clist if x not in sel_map]

        # apply the new column order
        Data.ReorderReportColumns(self.ReportID, clist)
        Data.SetUndo('Delete Column')

        # clear the selection
        self.Report.ClearSelection()

        if debug: print "End OnDeleteColumn"

    def OnMoveColumn(self, event):
        """ Move selected columns left or right by one position """
        self.FinishEdit()
        sel = self.Report.GetSelectedCols()
        if not sel:
            return

        if event.GetId() in (ID.MOVE_LEFT, ID.M_MOVE_COLUMN_LEFT):
            step = -1
        else: # ID.MOVE_RIGHT
            step = 1

        cids = self.Report.table.columns

        # get report columns from database
        #   (grid uses multiple grid columns for certain report columns)
        clist = Data.GetColumnList(self.ReportID)
        cids = self.Report.table.columns

        map = {}
        for sel_x in sel:
            try:
                x = clist.index(cids[sel_x])
            except:
                continue
            map[x] = None
        if not map:
            return

        move = map.keys()
        move.sort()

        # perform the movement
        start = move[0]
        end = move[-1] + 1

        if end - start > len(move):
            # consolidate columns
            if step < 0:
                end = start + len(move)
            else:
                start = end - len(move)

            move_cids = [clist[x] for x in move]
            move.reverse()
            for x in move:
                del clist[x]
            clist[start:start] = move_cids

        else:
            # move past the next column
            if step < 0:
                x = start - 1
                dest = end - 1
            else:
                x = end
                dest = start

            if 0 <= x < len(clist):
                cid = clist.pop(x)
                clist.insert(dest, cid)
                start += step
                end += step
            else:
                return

        # apply the new column order
        Data.ReorderReportColumns(self.ReportID, clist)
        Data.SetUndo(_('Move Column'))

    def OnScroll(self, event):
        """ scroll the selected  """
        if debug: print "Start OnScroll"
        self.FinishEdit()
        # figure out which gantt chart to scroll -- either the ones w/ selected columns or the first one
        scrollcols = []
        sel = self.Report.GetSelectedCols()  # current selection
        if len(sel) > 0:
            for s in sel:
                cid = self.Report.table.columns[s]
                ctid = Data.ReportColumn[cid]['ColumnTypeID']
                if Data.ColumnType[ctid].get('AccessType') == 's':
                    if scrollcols.count(cid) == 0:
                        scrollcols.append(cid)
        if len(scrollcols) == 0:
            clist = Data.GetColumnList(self.ReportID)  # complete list of column id's in display order
            for cid in clist:
                ctid = Data.ReportColumn[cid]['ColumnTypeID']
                if Data.ColumnType[ctid].get('AccessType') == 's':
                    scrollcols.append(cid)
                    break
        offset = 0; fast = False
        id = event.GetId()  # move left or right?
        if id in (ID.SCROLL_LEFT_FAR, ID.M_SCROLL_LEFT_FAST): offset = 1; fast = True
        elif id in (ID.SCROLL_LEFT, ID.M_SCROLL_LEFT): offset = 1
        elif id in (ID.SCROLL_RIGHT, ID.M_SCROLL_RIGHT): offset = -1
        elif id in (ID.SCROLL_RIGHT_FAR, ID.M_SCROLL_RIGHT_FAST): offset = -1; fast = True
        somethingChanged = False
        for s in scrollcols:
            date = Data.GetColumnDate(s, offset, fast)
            if date == None: continue
            newdate = Data.DateIndex[date]
            change = { 'Table': 'ReportColumn', 'ID': s, 'FirstDate': newdate}
            Data.Update(change)
            somethingChanged = True
        if somethingChanged: Data.SetUndo(_('Scroll Timescale Columns'))
        if debug: print "End Scroll"

    def OnScrollToTask(self, event):
        if debug: print "Start ScrollToTask"
        self.FinishEdit()
        sel = self.Report.GetSelectedRows()  # current selection
        if len(sel) < 1:
            if debug: print "can't scroll, no tasks selected"
            return  # only move if columns selected
        s = self.Report.table.rows[min(sel)]  # just use first row
        rs = Data.ReportRow[s]
        if not rs.get('TableName') in ('Task', 'Assignment'):
            if debug: print "tablename", rs.get('TableName')
            return  # can only scroll to tasks or assignments
        if rs.get('TableName') == 'Task':
            tid = rs.get('TableID')
        else:  # assignment
            aid = rs.get('TableID')
            tid = Data.Assignment[aid].get('TaskID')
        newdate = Data.ValidDate(Data.Task[tid].get('StartDate')) or Data.Task[tid].get('CalculatedStartDate')
            # maybe this should always go with the CalculatedStartDate
        if debug: print "new date", newdate
        if not newdate: return  # no date to scroll to

        somethingChanged = False
        clist = Data.GetColumnList(self.ReportID)  # complete list of row id's in display order
        for c in clist:
            ctid = Data.ReportColumn[c]['ColumnTypeID']
            ct = Data.ColumnType[ctid]
            if ct.get('AccessType') != 's':  # not a time scale
                continue
            # period = ct.get('PeriodSize') or ct.get('Name')
            change = {'Table': 'ReportColumn', 'ID': c, 'FirstDate': newdate}
            Data.Update(change)
            somethingChanged = True
        if somethingChanged: Data.SetUndo(_('Scroll to Task'))
        if debug: print "End ScrollToTask"

    # ---- menu Commands -----

    def doClose(self, event):
        Data.CloseReport(self.ReportID)

    def OnCopy(self, event):
        rid = self.ReportID  # current report
        if rid == 1:
            hint("This command doesn't work on the main report.")
            return  # do nothing

        # find selection
        selcol = self.Report.GetSelectedCols()  # current selection

        selrow = self.Report.GetSelectedRows()  # current selection
        selcell = self.Report.GetSelectedCells()  # current selection
        selr = self.Report.GetGridCursorRow()  # current selection
        selc = self.Report.GetGridCursorCol()  # current selection
        seltl = self.Report.GetSelectionBlockTopLeft()  # current selection
        selbr = self.Report.GetSelectionBlockBottomRight()  # current selection

        if debug:
            print 'selcol', selcol
            print 'selrow', selrow
            print 'selcell', selcell
            print 'cursor', selr, selc
            print 'seltl', seltl
            print 'selbr', selbr

        cx = self.Report.table.columns
        ox = self.Report.table.coloffset
        rx = self.Report.table.rows
        if not (cx and rx):
            hint("There are no cells to copy yet.")
            return

        if selcol: # columns selected
            r = range(len(rx))
            c = selcol
        elif selrow:
            r = selrow
            c = range(len(cx))
        elif seltl and selbr:
            r = range(seltl[0][0], selbr[0][0] + 1)
            c = range(seltl[0][1], selbr[0][1] + 1)
        else:
            r = [ selr ]
            c = [ selc ]

        def stringval(x):
            if isinstance(x, int) or isinstance(x, float):
                return str(x)
            elif isinstance(x, str):
                if x.count('\t'): x = x.replace('\t', r'\t')
                if x.count('\n'): x = x.replace('\n', r'\n')
                if x.count('\r'): x = x.replace('\r', r'\r')
                return x
            else:
                return x

        values = [ [ stringval(Data.GetCellValue(rx[ri], cx[ci], ox[ci])) for ci in c ] for ri in r ]

        if debug: print values

        s = os.linesep.join( [ '\t'.join(row) for row in values ] )

        if len(values) > 1 or len(values[0]) > 1:
            s += os.linesep

        if debug: print s

        # s = 'for the clipboard\tmore\tmore' + os.linesep + '1\t2\t3'
        # s = 'for the clipboard\tmore\tmore' + '\r' + '1\t2\t3' + '\r'
        cbData = wx.PyTextDataObject()
        cbData.SetText(s)

        if wx.TheClipboard.Open():
            wx.TheClipboard.SetData(cbData)
            wx.TheClipboard.Flush()
            wx.TheClipboard.Close()

    # -- taken from "Paste Cells From Clipboard.py" --
    # the following two helper functions edit the grid without calling SetUndo
    # (we want to be able to undo the entire pasting at once)

    def _CreateRow(self):
        """ Add a row to the end of the report. """
        r = Data.Report[self.ReportID]
        rt = Data.ReportType[r['ReportTypeID']]
        ta = rt.get('TableA')
        if not ta or ta in ('Report', 'ReportColumn', 'ReportRow', 'ReportType', 'ColumnType'): return  # need special handling
    
        change = {'Table': ta, 'Name': '--'}
        if ta != 'Project':
            change['ProjectID'] = r.get('ProjectID')
        undo = Data.Update(change)
        newid = undo['ID']

        change = {'Table': ta, 'ID': newid, 'Name': _(ta) + " " + str(newid)}
        undo = Data.Update(change)

        change = {'Table': 'ReportRow', 'ReportID': self.ReportID, 'TableName': ta, 'TableID': newid}
        undo = Data.Update(change)  # created here to control where inserted
        newrowid = undo['ID']
    
        rlist = Data.GetRowList(self.ReportID)
        rlist.append(newrowid)
        Data.ReorderReportRows(self.ReportID, rlist)
        Data.Recalculate()
    
    def _SetCellValue(self, row, col, value):
        table = self.Report.table
        rowid = table.rows[row]
        of = table.coloffset[col]
        colid = table.columns[col]
        column = Data.SetCellValue(rowid, colid, of, value)

    def OnPaste(self, event):
        if not isinstance(self.Report, wx.grid.Grid):
            hint("Run this command from a grid report.")
            return
        if not self.Report.GetNumberCols():
            hint("Insert columns first.")
            return

        # open clipboard
        data = wx.PyTextDataObject()
        if not wx.TheClipboard.Open():
            hint("Couldn't open clipboard.")
            return
        wx.TheClipboard.GetData(data)
        wx.TheClipboard.Close()
        cells = data.GetText()

        # decide where to paste
        if self.Report.GetNumberRows():
            sel = self.Report.GetSelectedRows()
            if sel:
                row = min(sel)
            else:
                row = self.Report.GetGridCursorRow()

            sel = self.Report.GetSelectedCols()
            if sel:
                column = min(sel)
            else:
                column = self.Report.GetGridCursorCol()
        else:
            row = column = 0

        # paste the cells
        for record in cells.splitlines():
            if row >= self.Report.GetNumberRows():  # need more rows
                 self._CreateRow()
                 if row >= self.Report.GetNumberRows():  # couldn't add a row
                     break
            c = column
            for value in record.split('\t'):
                 if c >= self.Report.GetNumberCols():
                     break
                 # self.Report.SetCellValue(row, c, value)
                 self._SetCellValue(row, c, value)
                 c += 1
            row += 1

        Data.SetUndo("Paste Cells")

    def doBringWindow(self, event):
        Menu.doBringWindow(self, event)

    def doQuick(self, event):
        Menu.doQuick(self, event)

    def doORMQuick(self, event):
        Menu.doORMQuick(self, event)

    def doShort(self, event):
        Menu.doShort(self, event)

    # ----------------- frame events

    def OnSize(self, event):
        size = event.GetSize()
        r = Data.Database['Report'][self.ReportID]
        r['FrameSizeW'] = size.width
        r['FrameSizeH'] = size.height

        event.Skip()  # to call default handler; needed?

    def OnMove(self, event):
        pos = event.GetPosition()
        if pos.x > 0 and pos.y > 0:
            r = Data.Database['Report'][self.ReportID]
            if Data.platform == "win":  # else windows don't open in the right place
                r['FramePositionX'] = pos.x - 4
                r['FramePositionY'] = pos.y - 50
            else:
                r['FramePositionX'] = pos.x
                r['FramePositionY'] = pos.y

        event.Skip()  # needed?

    def OnActivate(self, event):
        if event.GetActive():
            Data.SetActiveReport(self.ReportID)
        else:
            pass  # what is really needed here?
##            self.Report.SaveEditControlValue()
##            self.Report.DisableCellEditControl()
        event.Skip()

    # ---------------- Column/Row resizing
    def OnRowSize(self, evt):
        pass

    def OnColSize(self, evt):
        if debug: print "OnColSize", (evt.GetRowOrCol(), evt.GetPosition())
        if debug: print "get col width", self.Report.GetColSize(evt.GetRowOrCol())
        newsize = self.Report.GetColSize(evt.GetRowOrCol())
        col = evt.GetRowOrCol()
        if self.Report.table.coloffset[col] == -1:  # not a gantt column
            colid = self.Report.table.columns[col]
            change = { 'Table': 'ReportColumn', 'ID': colid, 'Width': newsize }
            Data.Update(change, 0)  #  --------------------- don't allow Undo (until I can figure out how)
            # Data.SetUndo('Change Column Width')

        evt.Skip()

    # -------------- Make sure only the right tables are edited per column
    def OnEditorShown(self, evt):
        rtid = Data.Report[self.ReportID].get('ReportTypeID')

        rid = self.Report.table.rows[evt.GetRow()]
        rtable = Data.ReportRow[rid].get('TableName')

        cid = self.Report.table.columns[evt.GetCol()]
        ctid = Data.ReportColumn[cid].get('ColumnTypeID')
        which = Data.ColumnType[ctid].get('T') or 'Z'  # can be 'A', 'B', or 'X'

        ctable = Data.ReportType[rtid].get('Table' + which)  # 'X' should always yield 'None'

        if which != 'X' and ((not ctable) or rtable != ctable):
            evt.Veto()
            return
        self.Edit.Enable(wx.ID_COPY, False)
        self.Edit.Enable(wx.ID_PASTE, False)

    def OnEditorHidden(self, evt):
        self.Edit.Enable(wx.ID_COPY, True)
        self.Edit.Enable(wx.ID_PASTE, True)

    # -----------
    def SetReportTitle(self):
        rname = Data.Report[self.ReportID].get('Name') or "-"
        pid = Data.Report[self.ReportID].get('ProjectID')
        if Data.Project.has_key(pid):
            pname = Data.Project[pid].get('Name') or "-"
        else:
            pname = "-"
        title = pname + " / " + rname
        if self.GetTitle() != title:
            self.SetTitle(title)
            Menu.UpdateWindowMenuItem(self.ReportID)

    def UpdatePointers(self, all=0):  # 1 = new database; 0 = changed report rows or columns
        if debug: print "Start ORMReport UpdatePointers"

        # don't refresh a report if the underlying report record is invalid
        if all or not Data.Report[self.ReportID].get('ReportTypeID'):
            Data.CloseReport(self.ReportID)
            return
### I don't know if this should be replaced with something else
##        sr = self.Report.table
##        rlen, clen = len(sr.rows), len(sr.columns)
##
##        select_rows = {}
##        for x in self.Report.GetSelectedRows():
##             rowid = sr.rows[x]
##             select_rows[rowid] = None
##        select_columns = {}
##        for x in self.Report.GetSelectedCols():
##             colid = sr.columns[x]
##             select_columns[colid] = None
##
##        r = self.Report.GetGridCursorRow()
##        try:
##            rid = sr.rows[r]
##        except IndexError:
##            rid = 0
##
##        sr.UpdateColumnPointers()
##        sr.UpdateRowPointers()
##
##        self.Report.BeginBatch()
##
##        if rlen != len(sr.rows) or clen != len(sr.columns):
##            self.Report.Reset()  # tell grid that the number of rows or columns has changed
##        else:
##            self.Report.UpdateAttrs()
##
##        self.Report.ClearSelection()
##        for x, rowid in enumerate(sr.rows):
##             if rowid in select_rows:
##                 self.Report.SelectRow(x, True)
##        for x, colid in enumerate(sr.columns):
##             if colid in select_columns:
##                 self.Report.SelectCol(x, True)
##
##        self.Report.EndBatch()
##
##        try:
##            r = self.Report.table.rows.index(rid)
##        except:
##            pass
##        else:
##            c = self.Report.GetGridCursorCol()
##            old_r = self.Report.GetGridCursorRow()
##            if r != old_r:
##                self.Report.SetGridCursor(r, c)
##                self.Report.MakeCellVisible(r, c)

        if debug: print "End ORMReport UpdatePointers"

#---------------------------------------------------------------------------

if __name__ == '__main__':
    app = wx.PySimpleApp()
    frame = GanttReportFrame(3, None, -1, "")  # reportid = 3
    frame.Show(True)
    app.MainLoop()

if debug: print "end GanttReport.py"
