# Copyright 2009 by Brian C. Christensen

#    This file is part of GanttPV.
#
#    GanttPV is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    GanttPV is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with GanttPV; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# 090102 - created this file
# 090124 - settled on StateObjects.py for the file name

import ORM

# State Shapes

class SDStateShape(ORM.ORMBox):
    def GetTextLines(self):
        orm_object = self.Get('Target')  # this should be a relational table
        if not orm_object: return  # object has probably been undone
        if orm_object.Table == 'RelationalTable':
            cols = [ x for x in orm_object.GetList('RelationalColumn') if x.InUse ]
        else:  # 'StarTable'
            cols = orm_object.GetList('StarColumn')
        cols.sort(cmp=lambda x,y: cmp(x.Seq, y.Seq))
        text = [ x.Name for x in cols ]
        lines = [orm_object.Name] + text
        return lines

    def AdjustEnd(self, other_end, seq=None):
        if not seq or not self.ClipRows:
            return ORMBox.AdjustEnd(self, other_end)

        clip_rows = [ int(x) for x in self.ClipRows.split(',') ]

        if not seq < len(clip_rows) - 1:  # for average calculation below
            return ORMBox.AdjustEnd(self, other_end)

        spacing = 0
        x, y = self.GetPos()
        bx, by = other_end
        lines = self.GetTextLines()
##        sizes = [ self.canvas.GetFullTextExtent(line)[0:2] for line in lines ]
##        sizes[0] = sizes[0][0], sizes[0][1] + 3

##        box_w = 5
##        box_h = 0
##        for line_w, line_h in sizes:
##            box_w = max(box_w, line_w)
##            box_h += line_h + spacing
##        box_w += 6
##        box_h += 6
        left, top, box_w, box_h = self.GetClipBox()
        right = left + box_w
        
##        left = x - box_w / 2
##        right = x + box_w / 2
##        top = y - box_h / 2

        if bx <= x:  # can add a constant here to avoid crossing over
            ax = left
        else:
            ax = right

##        ay = top + 2  # from h = 2 ... box_y+h
##        line_h = 0  # just in case sizes list is empty
##        for line_w, line_h in sizes[:seq + 1]:  # include the header line
##            ay += line_h + spacing
##        ay -= line_h / 2  # center on the line of text
        ay = top + (clip_rows[seq] + clip_rows[seq + 1]) / 2
            # average of the two rows to center the line

        return ax, ay

    def Draw(self, dc):
        orm_object = self.Get('Target')  # this should be a relational table
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)

##        if orm_object.Table == 'RelationalTable':
##            cols = orm_object.GetList('RelationalColumn')
##        else:  # 'StarTable'
##            cols = orm_object.GetList('StarColumn')
##        cols.sort(cmp=lambda x,y: cmp(x.Seq, y.Seq))
##        text = [ x.Name for x in cols ]
##
####        text = orm_object.Text or 'empty note'
####        if self.IsSelected and self.dcid == self.canvas.keyboard_target_dcid:
####            text = self.InsertCursor(text)  # add cursor to text for display
##
##        lines = [orm_object.Name, ''] + text  # .splitlines()

        spacing = 0
        x, y = self.GetPos()
        lines = self.GetTextLines()
        sizes = [ self.canvas.GetFullTextExtent(line)[0:2] for line in lines ]  # pull out only w and h
        sizes[0] = sizes[0][0], sizes[0][1] + 3

        w = 5
        h = 0
        for line_w,line_h in sizes:
            w = max(w, line_w)
            h += line_h + spacing
        box_w = w + 6
        box_h = h + 6
        box_x = x - box_w/2
        box_y = y - box_h/2

        pen = self.canvas.CachedPen('Blue', 1, wx.SOLID)
        dc.SetPen(pen)
        if self.GetSelected():
            dc.SetBrush(self.canvas.CachedBrush(SelectedColor))
        else:
            dc.SetBrush(self.canvas.CachedBrush((254, 254, 254)))
            # dc.SetBrush(self.canvas.CachedBrush('White'))
        dc.DrawRectangle(box_x,box_y,box_w,box_h)

        dc.SetFont(self.canvas.GetFont())
        dc.SetTextForeground('Black')
        dc.SetTextBackground('White')
        w = 3; h = 2
        clip_rows = [ h ]
        for i in range(len(lines)):
            line = lines[i]
            line_w, line_h = sizes[i]
            dc.DrawText(line, box_x+w, box_y+h)
            h += line_h + spacing
            clip_rows.append(h)

        tablenamebox = box_y + sizes[0][1] + 0  # line offset from top of box
        dc.DrawLine(box_x, tablenamebox, box_x + box_w, tablenamebox) # box table name

        r = wx.Rect(box_x,box_y,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)

        # save this information, but don't put on undo stack
        # because before this function didn't change the database
        # and in use might not always be followed by a SetUndo
        self.SetClipBox(box_x, box_y, box_w, box_h)
##        self.ClipRows = ','.join([ str(x) for x in clip_rows ])
        self.SetWithoutUndo('ClipRows', ','.join([ str(x) for x in clip_rows ]))

##    def Char(self, char):
##        if debug: print "char", ord(char), "'", char, "'"
##        orm_object = self.Get('Target')
##        orm_object.Text = self.ApplyChar(char, orm_object.Text)

##        if orm_object.Text:
##            if char == '\x08':  # this is for windows, cross platform?
##                orm_object.Text = orm_object.Text[:-1]
##            elif char == '\r':  # this is for windows, cross platform?
##                orm_object.Text += '\n'
##            else:
##                orm_object.Text += char
##        else:
##            if char in '\r\x08':  # this is for windows, cross platform?
##                pass
##            else:
##                orm_object.Text = char

    def IsUIDeletable(self):  # allow user to delete this graphic object
        return False

    def Delete(self):
        if self.InDelete: return
        self._SetInShell('InDelete', True)
        self.Follow(remove=True)  # before they are deleted
##        for shape in self.GetGraphicList('ORMNoteConnectorShape', 'NodeA'):
##            shape.Delete()
        if self.Target:
            self.Target.Delete()
        ORMShape.Delete(self)
        self._SetInShell('InDelete', None)

class SDStateTransitionShape(ORM.ORMConnector):
    def Draw(self, dc):
        orm_object = self.Get('Target')
        dc.ClearId(self.dcid)
        if not orm_object: return  # object has probably been undone
        dc.SetId(self.dcid)

        x, y = self.GetPos()
        nodea, nodeb = self.NodeA, self.NodeB

        if orm_object.RelationalReference:
            seqa = orm_object.RelationalReference.Seq
        else:
            seqa = None
        seqb = orm_object.Seq
        
        enda = nodea.GetPos()
        endb = nodeb.GetPos()
        enda, endb = nodea.AdjustEnd(endb, seq=seqa), nodeb.AdjustEnd(enda, seq=seqb)

        platform = 20  # arbitrary width

        if enda[0] < nodea.GetPos()[0]:
            jointa = enda[0] - platform, enda[1]
        else:
            jointa = enda[0] + platform, enda[1]

        if endb[0] < nodeb.GetPos()[0]:
            jointb = endb[0] - platform, endb[1]
        else:
            jointb = endb[0] + platform, endb[1]

        minx = min(enda[0], endb[0], jointa[0], jointb[0])
        maxx = max(enda[0], endb[0], jointa[0], jointb[0])
        miny = min(enda[1], endb[1])
        maxy = max(enda[1], endb[1])
        
        box_w = maxx - minx + 6
        box_h = maxy - miny + 6

        if self.IsSelected:
            pen = self.canvas.CachedPen(SelectedColor, 3, wx.SOLID)
            dc.SetPen(pen)
            dc.DrawLine(enda[0], enda[1], endb[0], endb[1])

        pen = self.canvas.CachedPen('Blue', 1, wx.SOLID)
        dc.SetPen(pen)
        dc.DrawLine(enda[0], enda[1], jointa[0], jointa[1])
        dc.DrawLine(jointa[0], jointa[1], jointb[0], jointb[1])
        dc.DrawLine(jointb[0], jointb[1], endb[0], endb[1])
        self.SetTempPos((jointa[0] + jointb[0]) / 2, (jointa[1] + jointb[1]) / 2)

        r = wx.Rect(minx,miny,box_w,box_h)
        r.Inflate(pen.GetWidth(),pen.GetWidth())
        dc.SetIdBounds(self.dcid,r)

    def IsUIDeletable(self):  # allow user to delete this graphic object
        return False

    def Delete(self):
        if self.InDelete: return  # prevent loops
        self._SetInShell('InDelete', True)
        self.Follow(remove=True)  # before they are deleted
        if self.Target:
            self.Target.Delete()
        ORMShape.Delete(self)
        self._SetInShell('InDelete', None)

_subtype = {
    # State shapes
    'SDStateShape': SDStateShape,
    'SDStateTransitionShape': SDStateTransitionShape,
##    'ORMFactReadingShape': ORMFactReadingShape,
}
for k, v in _subtype.iteritems():
    Data.DB.RegisterSubtype(k, v)
