/*
 *-----------------------------------------------------------------
 * Name:	eee-inlines.h
 * Copyright (C) 2001, Agile Storage, Inc.
 * Description:	eee inline functions
 *-----------------------------------------------------------------
 */
#if !defined(__EEE_INLINES__)
#define __EEE_INLINES__

extern address_t get_pc(void);

#define EEE_REMOTE_POOL EEE_FP_POOL

#ifdef NFX_SMP

/*++

Routine Description:

    Initialize the global allocation pool.
    
Arguments:

    global_pool - global allocation pool
    num_avail - the total number of objects
    num_slab - the number of elements in a slab
    id - the pool Id
    buf_base - the beginning of buffer memory
    buf_end - the end of buffer memory

Return Values:

    None.

--*/

void
eee_global_pool_init(struct eee_global_stack_pool *global_pool,
                     uint32 num_avail,
                     uint32 num_slab,
                     uint32 id,
                     void *buf_base,
                     void *buf_end);

/*++

Routine Description:

    Initialize local allocation pool.
    
Arguments:

    pool - local allocation pool
    global_pool - global allocation pool
    num_slab - the number of elements in a slab

Return Values:

    None.

--*/

void
eee_local_pool_init(struct eee_local_stack_pool *pool,
                    struct eee_global_stack_pool *global_pool,
                    uint32 num_slab);


/*++

Routine Description:

    Deallocate the buffer to the local pool.
    
Arguments:

    pool - the allocation pool
    obj - the pointer to the beginning of the buffer 

Return Values:

    None.

--*/

#ifdef EEE_VALIDATE
void
eee_dealloc_buf_ra(struct eee_local_stack_pool *pool, void *obj, address_t ra);
#else
#define eee_dealloc_buf_ra(p, o, r) eee_dealloc_buf(p, o)
#endif
void
eee_dealloc_buf(struct eee_local_stack_pool *pool,
                void *obj);


/*++

Routine Description:

    Allocate a buffer from the local pool.
    
Arguments:

    pool - the pool to use for the allocation

Return Values:

    Returns the allocated buffer or NULL if not enough memory.

--*/

#ifdef EEE_VALIDATE
void *
eee_alloc_buf_ra(struct eee_local_stack_pool *pool, address_t ra);
#else
#define eee_alloc_buf_ra(p, r) eee_alloc_buf(p)
#endif
void *
eee_alloc_buf(struct eee_local_stack_pool *pool);

#endif

#ifndef NFX_SMP
#if defined(__EEE_DESC_COMPILE)
/*-----------------------------------------------------------------
 * Name:	eee_allocateDesc(uint32 mempool, uint32 cos)
 *
 * Description:	Allocate an EEE descriptor.
 *				For optimal performance, the caller should know where
 *				the packet will be destined.  This knowledge will
 *				allow the allocation routine to get the descriptor
 *				from the best pool, according to the module's
 *				architecture.
 *
 * Created by:	Rick Lund
 *
 * Date Created:	4/9/01
 *
 *-----------------------------------------------------------------
 */
eee_descPtr_t
eee_allocateDesc(uint32 mempool, uint32 cos unused__)
#else
static inline eee_descPtr_t
EEE_ALLOCATE_DESC(
	uint32	mempool unused__,	/* mem_pool_id */
	uint32	cos	 unused__		/* NFX_COS_HIGH, NFX_COS_LOW */
)
#endif	
{
	address_t		ptr;
	eee_descPtr_t	desc;

#if defined(EEE_VALIDATE)
#if defined(__NO_INLINE__) || defined(__EEE_DESC_COMPILE)
	address_t ra = get_ra();
#else
	address_t ra = get_pc();
#endif
#endif

	NFX_SPIN_LOCK(&eee.descpool_spin);

	if (mempool == EEE_SHARED_POOL || mempool == EEE_MY_POOL_ID) {
	try_shared:
		{
			eee_stack_pool_cntl_t *stack =
				&eee.desc_pool[EEE_MY_SHARED_POOL].stack;
			if (stack->num_avail) {
				--stack->num_avail;
				ptr = *stack->top++;
#if defined(EEE_VALIDATE)
				EEE_VALIDATE_ALLOC_DESC(ptr, EEE_MY_SHARED_POOL, ra);
				if (stack->num_avail <
					eee_stats->low_watermark_desc[EEE_MY_SHARED_POOL])
					eee_stats->low_watermark_desc[EEE_MY_SHARED_POOL] =
						stack->num_avail;
#endif			
				eee_attrAssignDesc((void *)ptr, EEE_ATTR_ALLOCATE);
			}
			else {
				eee_stats->no_desc_avail[EEE_MY_SHARED_POOL]++;
                NFX_SPIN_UNLOCK(&eee.descpool_spin);
				return NULL;
			}
		}
	}
#if defined(NFP_FP) || defined(NFP_TXRX)		
	else if (mempool == EEE_REMOTE_POOL) {
		eee_queue_pool_cntl_t *queue =
			&eee.desc_pool[EEE_REMOTE_POOL - SHARED_POOL_OFFSET].queue;
		ptr = *queue->head;
		if ((void*)ptr != NULL) {
			(void*)*queue->head = NULL;
			if (((address_t)queue->head & CACHELINE_MASK) == 0)
				*(queue->tail_ptr) = queue->head;
			queue->head++;
			if (queue->head == queue->end)
				queue->head = queue->start;
			eee_attrAssignDesc((void *)ptr, EEE_ATTR_RQ_DEQUEUE);
		}
		else {
			eee_stats->no_desc_avail[EEE_REMOTE_POOL - SHARED_POOL_OFFSET]++;
			goto try_shared;
		}
	}
#endif
	else if (mempool == EEE_LOCAL_POOL) {
		eee_stack_pool_cntl_t *stack = &eee.desc_pool[0].stack;
		if (stack->num_avail) {
			stack->num_avail--;
			ptr = *stack->top++;
#if defined(EEE_VALIDATE)
			EEE_VALIDATE_ALLOC_DESC(ptr, EEE_LOCAL_POOL, ra);
#endif	
			eee_attrAssignDesc((void *)ptr, EEE_ATTR_ALLOCATE);
		}
		else {
			eee_stats->no_desc_avail[0]++;
			goto try_shared;
		}
	}
	else {
		*(volatile uint64 *)0 = 0;
		return NULL;
	}
	
	NFX_SPIN_UNLOCK(&eee.descpool_spin);
	
	desc = (eee_descPtr_t)ptr;

	return desc;
}	/* eee_allocateDesc() */


#if defined(__EEE_DESC_COMPILE)
/*-----------------------------------------------------------------
 * Name:	eee_allocateBuffer(uint32 buf_size, uint32 cos)
 *
 * Description:	Allocates a system buffer from the free pools
 *				Like the descriptor allocation routine, for optimal
 *				performance, the calling function should know where
 *				the buffer is to be ultimately used.
 *
 * Created by:	Rick Lund
 *
 * Date Created:	4/9/01
 *
 *-----------------------------------------------------------------
 */
void *				/* returns the buffer pointer, or NULL if none
					 * available from the given cos
					 */
eee_allocateBuffer(
	uint32	mempool,		/* mem_pool_id */
	uint16	buf_size,		/* EEE_BUF_SMALL
							 * EEE_BUF_LARGE
							 * EEE_BUF_ULTRA
							 */
	uint32	cos unused__ /* NFX_COS_HIGH, NFX_COS_LOW */
)
#else
static inline void *
EEE_ALLOCATE_BUFFER(uint32 mempool,
					uint16 buf_size,
                    uint32 cos unused__)
#endif
{
	address_t		ptr;

#if defined(EEE_VALIDATE)
#if defined(__NO_INLINE__) || defined(__EEE_DESC_COMPILE) 
	address_t ra = get_ra();
#else
	address_t ra = get_pc();
#endif
#endif

	NFX_SPIN_LOCK(&eee.bufpool_spin);

	if (buf_size >= EEE_NUM_BUF_SIZES) {
		*(volatile uint64 *)0 = 0;
		return NULL;
	}

	if (mempool == EEE_SHARED_POOL || mempool == EEE_MY_POOL_ID) {
	try_shared:
		{
			eee_stack_pool_cntl_t *stack =
				&eee.buf_pool[EEE_MY_SHARED_POOL][buf_size].stack;
			if (stack->num_avail) {
				--stack->num_avail;
				ptr = *stack->top++;
#if defined(EEE_VALIDATE)
				EEE_VALIDATE_ALLOC_BUF(ptr, EEE_MY_SHARED_POOL, buf_size, ra);
				if (stack->num_avail <
					eee_stats->low_watermark_buf[EEE_MY_SHARED_POOL][buf_size])
					eee_stats->low_watermark_buf[EEE_MY_SHARED_POOL][buf_size] =
						stack->num_avail;
#endif			
				eee_attrAssignBuf((void *)ptr, EEE_ATTR_ALLOCATE);
			}
			else {
				NFX_SPIN_UNLOCK(&eee.bufpool_spin);
				ATOMIC_INC_INT(
					&eee_stats->no_buf_avail[EEE_MY_SHARED_POOL][buf_size], 1);
				return NULL;
			}
		}
	}
#if defined(NFP_FP) || defined(NFP_TXRX)		
	else if (mempool == EEE_REMOTE_POOL) {
		eee_queue_pool_cntl_t *queue =
			&eee.buf_pool[EEE_REMOTE_POOL - SHARED_POOL_OFFSET][buf_size].queue;
		ptr = *queue->head;
		if ((void*)ptr != NULL) {
			(void*)*queue->head = NULL;
			if (((address_t)queue->head & CACHELINE_MASK) == 0)
				*(queue->tail_ptr) = queue->head;
			queue->head++;
			if (queue->head == queue->end)
				queue->head = queue->start;
			eee_attrAssignBuf((void *)ptr, EEE_ATTR_RQ_DEQUEUE);
		}
		else {
			ATOMIC_INC_INT(&eee_stats->no_buf_avail
					   [EEE_REMOTE_POOL - SHARED_POOL_OFFSET][buf_size], 1);
			goto try_shared;
		}
	}
#endif
	else if (mempool == EEE_LOCAL_POOL) {
		eee_stack_pool_cntl_t *stack = &eee.buf_pool[0][buf_size].stack;
		if (stack->num_avail) {
			stack->num_avail--;
			ptr = *stack->top++;
#if defined(EEE_VALIDATE)
			EEE_VALIDATE_ALLOC_BUF(ptr, EEE_LOCAL_POOL, buf_size, ra);
#endif	
			eee_attrAssignBuf((void *)ptr, EEE_ATTR_ALLOCATE);
		}
		else {
			eee_stats->no_buf_avail[0][buf_size]++;
			goto try_shared;
		}
	}
	else {
		*(volatile uint64 *)0 = 0;
		return NULL;
	}

	NFX_SPIN_UNLOCK(&eee.bufpool_spin);

#if defined(DEBUG_SCSI_WRITE_REQ_CORRUPTION) && notdef
    if ((void *)ptr != NULL) {
        /* Use pattern 0xcc for entries on the free queue.
         * Just init 2x the size of the scsi_writeReqMsg_t
         */
        memset((void *)ptr, 0xcc, 64 * 2);
        CACHE_FLUSH_STRUCT((address_t)ptr, 64 * 2);
    }
#endif
    return (void *)ptr;
}	/* eee_allocateBuffer() */

/*++

Routine Description:

    Deallocate descriptor into shared pool.
    
Arguments:

    edesc - EEE descriptor

Return Values:

    None.

--*/

static inline void
eee_deallocateSharedDesc(eee_descPtr_t edesc)
{
#if defined(EEE_VALIDATE)
#if defined(__NO_INLINE__) || defined(__EEE_DESC_COMPILE) 
	address_t ra = get_ra();
#else
	address_t ra = get_pc();
#endif
#endif	
	eee_stack_pool_cntl_t *pool = &eee.desc_pool[EEE_MY_SHARED_POOL].stack;

	edesc->hdr.next = 0;
	edesc->hdr.offset = 0;
    edesc->hdr.control = 0;

	NFX_SPIN_LOCK(&eee.descpool_spin);
#ifdef EEE_ATTR
	if (!(eee_attrOkToFreeDesc(edesc))) {
		eee_attrDisplay((address_t)edesc);
		panic("eee_attr failed in deallocateDescRA, edesc = %p, ra = %p\n", edesc, ra);
	}
	eee_attrSetDescValidFreeAttr(edesc, 0, 0, 0, 0);
#endif

#ifdef EEE_VALIDATE
	if (edesc < eee.desc_base[EEE_MY_SHARED_POOL]
		|| edesc >= eee.desc_end[EEE_MY_SHARED_POOL]) {
		panic("descriptor is not shared descriptor\n");
	}
	EEE_VALIDATE_FREE_DESC(edesc, EEE_MY_SHARED_POOL, ra);
#endif

	*(--pool->top) = (address_t)edesc;
	++pool->num_avail;

	NFX_SPIN_UNLOCK(&eee.descpool_spin);

#ifdef EEE_ATTR
	eee_attrAssignDesc(edesc, EEE_ATTR_FREE);
#endif
}

/*++

Routine Description:

    Deallocate buffer into shared pool of the specified size
    
Arguments:

    buf - the buffer to deallocate
    size - the buffer size

Return Values:

    None.

--*/

static inline void
eee_deallocateSharedBuffer(void *buf, uint16 size)
{
#if defined(EEE_VALIDATE)
#if defined(__NO_INLINE__) || defined(__EEE_DESC_COMPILE) 
	address_t ra = get_ra();
#else
	address_t ra = get_pc();
#endif
#endif	
	eee_stack_pool_cntl_t *pool =
		&eee.buf_pool[EEE_MY_SHARED_POOL][size].stack;

	NFX_SPIN_LOCK(&eee.bufpool_spin);

#ifdef EEE_ATTR
	if (!(eee_attrOkToFreeBuf(buf))) {
		eee_attrDisplay((address_t)buf);
		panic("eee_attr failed in deallocateBufferRA, buf = %p, ra = %p\n", buf, ra);
	}
	eee_attrSetBufValidFreeAttr(buf, 0, 0, 0, 0);
#endif

#ifdef EEE_VALIDATE
	EEE_VALIDATE_FREE_BUF(buf, EEE_MY_SHARED_POOL, size, ra);
#endif

    CACHE_INVALIDATE_STRUCT((address_t)buf, size);
	*(--pool->top) = (address_t)buf;
	++pool->num_avail;

#ifdef EEE_ATTR
	eee_attrAssignBuf(buf, EEE_ATTR_FREE);
#endif
	NFX_SPIN_UNLOCK(&eee.bufpool_spin);
}
#else


/*++

Routine Description:

    Deallocate descriptor into shared pool.
    
Arguments:

    edesc - EEE descriptor

Return Values:

    None.

--*/

static inline void
eee_deallocateSharedDesc(eee_desc_t *edesc)
{
    edesc->hdr.offset = 0;
    edesc->hdr.next = 0;
    edesc->hdr.control = 0;
    eee_dealloc_buf_ra(&local_desc_pool, edesc, get_ra());
}


/*++

Routine Description:

    Deallocate buffer into shared pool of the specified size
    
Arguments:

    buf - the buffer to deallocate
    size - the buffer size

Return Values:

    None.

--*/

static inline void
eee_deallocateSharedBuffer(void *buf, uint16 size)
{
    eee_dealloc_buf_ra(&local_buf_pool[size], buf, get_ra());
}

#endif


static inline int
eee_isLocalAddr(address_t addr)
{
	return (addr >> 36) == (uint16)(LOCAL_MEM >> 36);
}

static inline int
eee_isRemoteAddr(address_t addr)
{
	return (addr >> 36) == (uint16)(REMOTE_MEM >> 36);
}

static inline int
eee_isRemoteCachedAddr(address_t addr)
{
	return (addr >> 36) == (uint16)(REMOTE_CACHED_MEM >> 36);
}

/*-----------------------------------------------------------------
 * Name :	eee_deallocateRemoteDesc()
 *
 * Description: Deallocate descriptor into forwarding queue
 *
 * Created by:	Maxim Kozlovsky
 *
 * Date Created:	09/27/02
 *
 *-----------------------------------------------------------------
 */
static inline void
eee_deallocateRemoteDesc(eee_descPtr_t edesc)
{
#if defined(EEE_ATTR)
#if defined(__NO_INLINE__) || defined(__EEE_DESC_COMPILE) 
	address_t ra = get_ra();
#else
	address_t ra = get_pc();
#endif
#endif	
#ifdef EEE_ATTR
	if (!(eee_attrOkToFreeDesc(edesc))) {
		eee_attrDisplay((address_t)edesc);
		panic("eee_attr failed in deallocateDescRA, edesc = %p, ra = %p\n",
			  edesc, ra);
	}
	
	eee_attrSetDescValidFreeAttr(edesc, 0, 0, 0, 0);
#endif
	ATOMIC_INC_INT(&eee_stats->fwd_free_desc[EEE_REMOTE_QUEUE_INDEX], 1);
#ifdef EEE_ATTR
	eee_attrAssignDesc(edesc, EEE_ATTR_FWD_TO_FREE);
#endif
	FWD_PACKET((address_t)edesc | EEE_FREE_DESC, EEE_REMOTE_QUEUE_INDEX, TRUE);
}


static inline void
eee_deallocateDescMaybeCached(eee_descPtr_t edesc)
{
#if !defined(FS_STANDALONE)
	/* Done with the descriptor */
	if (eee_isRemoteCachedAddr((address_t)edesc)) {
		__asm__("cache %0,0(%1)" : : "i"(Hit_Invalidate_D),"r"(edesc));
		__asm__("cache %0,32(%1)" : : "i"(Hit_Invalidate_D),"r"(edesc));
		__asm__("cache %0,64(%1)" : : "i"(Hit_Invalidate_D),"r"(edesc));
		__asm__("cache %0,96(%1)" : : "i"(Hit_Invalidate_D),"r"(edesc));
		eee_deallocateRemoteDesc((eee_descPtr_t)
			(((address_t)edesc - REMOTE_CACHED_MEM) + REMOTE_MEM));
	}
	else {
		eee_deallocateDesc(edesc);
	}
#endif
}

/*-----------------------------------------------------------------
 * Name:    eee_getNbrOfBufs()
 *
 * Description: 
 *
 * Created by:  Jan Vanhoutte
 *
 * Date Created:    5/3/02
 *
 *-----------------------------------------------------------------
 */
static inline uint32
eee_getTotalNbrOfBufs(int buf_size)
{
	return eee.sm_num_buf[buf_size];
}

/*-----------------------------------------------------------------
 * Name:    eee_getNbrOfBufs()
 *
 * Description: 
 *
 * Created by:  Jan Vanhoutte
 *
 * Date Created:    5/3/02
 * int		buf_size => EEE_BUF_SMALL, EEE_BUF_LARGE, EEE_BUF_ULTRA
 *-----------------------------------------------------------------
 */
static inline uint32
eee_getAvailableNbrOfBufs(int buf_size)
{
	return eee.buf_pool[EEE_MY_SHARED_POOL][buf_size].stack.num_avail;
}

/*++

Routine Description:
    Get the number of available descriptors in the system.
Arguments:
    None.
Return Value:
    Returns the number of descriptors available.
--*/

static inline long
eee_getAvailableNbrOfDescs()
{
    return eee.desc_pool[EEE_MY_SHARED_POOL].stack.num_avail;
}

/*++
Routine Description:
    Convert the shared memory pool id.
Arguments:
    pool_id - the shared memory pool id
Return Value:
    Returns the index of the shared memory pool.
--*/

static inline int
eee_pool_index(uint32 pool_id)
{
    switch (pool_id) {
    case EEE_SHARED_POOL:
    case EEE_MY_POOL_ID:
        return EEE_MY_SHARED_POOL;
    default:
        // ASSERT(0);
#if defined(EEE_REMOTE_POOL)
    case EEE_REMOTE_POOL:
        return EEE_REMOTE_POOL;
#else
        return EEE_LOCAL_POOL;
#endif        
    }
}


/*-----------------------------------------------------------------
 * Name:	EEE_BUF_BASE()
 *
 * Description:	From an address within a buffer, figure out the real
 *				base address of the buffer.
 *
 * Created by:	Rick Lund
 *
 * Date Created:	5/1/01
 *
 *-----------------------------------------------------------------
 */
#define EEE_BUF_BASE(_bufptr, _pool_id, _size_id)                   \
({                                                                  \
	address_t _r;                                                   \
    address_t _bufaddr = (address_t)_bufptr;                        \
                                                                    \
    /* Verify that the buffer was really allocated from this pool.  \
     */                                                             \
    // ASSERT((eee.buf_base[eee_pool_index(_pool_id)][_size_id] <=     \
            (void *)_bufptr) &&                                     \
           ((void *)_bufptr <                                       \
            eee.buf_end[eee_pool_index(_pool_id)][_size_id]));      \
                                                                    \
	if ((_size_id) == EEE_BUF_SMALL) {                              \
		_r = _bufaddr & eee.buf_mask[EEE_BUF_SMALL];                \
    }                                                               \
	else if ((_size_id) == EEE_BUF_LARGE) {                         \
        _r = (_bufaddr - (((uint32)_bufaddr) % EEE_BUFSIZE_LARGE)); \
	}                                                               \
	else if ((_size_id) == EEE_BUF_ULTRA) {                         \
        _r = (_bufaddr - (((uint32)_bufaddr) % EEE_BUFSIZE_ULTRA)); \
	}                                                               \
	else if ((_size_id) == EEE_BUF_MGMT) {                          \
        _r = (_bufaddr - (((uint32)_bufaddr) % EEE_BUFSIZE_MGMT));  \
	}                                                               \
	else {                                                          \
		panic("Invalid size_id, 0x%x\n", _size_id);                 \
		_r = 0;                                                     \
	}                                                               \
	_r;                                                             \
})

#endif
