/*
 *-----------------------------------------------------------------
 * Name:    eee-msg.c
 * Copyright (C) 2009, OnStor, Inc.
 * Description: Messaging engine functions
 *-----------------------------------------------------------------
 */
#include "nfx-incl.h"
#include "../sm-eee/eee-api.h"
#include "../sm-chassis/cm-api.h"
#include "../sm-debug/debug.h"

extern uint16 eee_packetId;

//#define NET_HDR_LEN (sizeof(agile_hdr_t) + sizeof(enet_hdr_t))
#define NET_HDR_LEN 64 //-- Leave room for lmux header too.


/*-----------------------------------------------------------------
 * Name :   eee_sendFragmentedMessage()
 *
 * Description: Send message to another slot. Allocate message descriptors fragmented according
 *              to the backplane mtu for the packet data type.
 *              If firstDesc is NULL, then eDesc chain is allocated and returned in
 *              *firstDesc. No attempt is made to forward the allocated eDesc chain 
 *-----------------------------------------------------------------
 */
int32
eee_sendFragmentedMessage(void *src,
    uint32 num_bytes,
    uint32 dest_port,
    uint32 source_port,
    eee_desc_t **firstDesc)
{
    int hdrlen = sizeof(agile_hdr_t);
    int mtu = 1500;
    eee_descPtr_t startDesc = NULL;
    eee_descPtr_t *pprev = &startDesc;
    eee_descPtr_t edesc;
    uchar8 *buf;
    eee_descHdr_t *hdr;
    eee_descBuf_t *bd;
    uchar8 *srcbuf = src;
    uint32 offset = 0;
    uint32 nextlen;
    uint32 moreFragments;
    uint16 pktid;
    uint32 buftype;
    uint32 mp;
    
    if(!src && !firstDesc) return NFX_ERR;

    pktid = eee_nextPacketId();

    switch (EEE_GET_DTYPE(dest_port)) {

    case EEE_MGMT:
        mtu = 1500;
#if defined(NFP_FP) || defined(NFP_TXRX)            
        /* Use large buffers instead of management buffers because
         * they are more plentiful.
         */
        buftype = EEE_BUF_LARGE;
#if defined(NFP_FP)
        mp = GET_MEMPOOL_INDEX_FROM_CPU_ID(0);
#else
        mp = EEE_SHARED_POOL;
#endif      
#else
        mp = EEE_SHARED_POOL;
        if (num_bytes <= eee.buf_size[EEE_BUF_SMALL] - NET_HDR_LEN) {
            buftype = EEE_BUF_SMALL;
        }
        else {
            buftype = EEE_BUF_LARGE;
        }
#endif
        break;

    case EEE_CONTROL:
        mtu = 1500;
        mp = EEE_SHARED_POOL;
        if (num_bytes <= eee.buf_size[EEE_BUF_SMALL] - NET_HDR_LEN) {
            buftype = EEE_BUF_SMALL;
        }
        else {
            buftype = EEE_BUF_LARGE;
        }
        break;

    case EEE_USER_DATA:
    case EEE_SAC:
        mtu = 8704 - NET_HDR_LEN;
        mp = EEE_SHARED_POOL;
        if (num_bytes <= eee.buf_size[EEE_BUF_SMALL] - NET_HDR_LEN) {
            buftype = EEE_BUF_SMALL;
        }
        else if (num_bytes <= eee.buf_size[EEE_BUF_LARGE] - NET_HDR_LEN) {
            buftype = EEE_BUF_LARGE;
        }
        else {
            buftype = EEE_BUF_ULTRA;
        }
        break;

    default:
        printk("eee_sendFragmentedMessage: invalid dtype %d\n", EEE_GET_DTYPE(dest_port));
        return NFX_ERR;
    }

    while (offset < num_bytes) {

        edesc = eee_allocateDesc(mp, NFX_COS_HIGH);
        buf = eee_allocateBuffer(mp, buftype, NFX_COS_HIGH);

        if (edesc != NULL && buf != NULL) {
            hdr = &edesc->hdr;
            bd = &edesc->bd[1];

            nextlen = num_bytes - offset;
            if (nextlen + hdrlen > mtu) {
                /* Fragments will be at least 64 bytes long */
                if (nextlen + hdrlen < mtu + 63)
                    nextlen = nextlen / 2;
                else
                    nextlen = mtu - hdrlen;
                nextlen &= ~63;
                moreFragments = AGILEHDR_MF;
            }
            else {
                moreFragments = 0;
            }

            hdr->src_port = source_port;
            hdr->dest_port = dest_port;
            hdr->control = nextlen | (1 << EDESC_BUFCNT_SHIFT);
            hdr->offset = offset | moreFragments
                | (pktid << AGILEHDR_ID_SHIFT);

            bd->control = EDESC_BD_CMD_IN_USE;
            bd->len = nextlen;

            /* Make room for the network header in the buffer */
            bd->buf = buf + NET_HDR_LEN;

            if(srcbuf)
                memcpy(bd->buf, &srcbuf[offset], nextlen);
            
            *pprev = edesc;
            pprev = &edesc->hdr.next;
            
            offset += nextlen;
            
            hdr->control |= EDESC_COMMAND_DONT_CONVERT;
        }
        else {
            if (edesc) eee_deallocateDesc(edesc);
            if (buf) eee_deallocateBuffer(buf);
            eee_freePacket(startDesc);
            printk("eee_sendFragmentedMessage: no buffers\n");
            return EEE_ERR_NO_BUF;
        }
    }

    *pprev = NULL;

    if(firstDesc) {
        *firstDesc = startDesc;
        return 0;
    }
    return eee_forwardPacket(startDesc);
}

/*
Routine Description:
    Allocate the packet chain for sending message and copy the data from the
    buffer into the message. If first_edesc is not null, the allocated
    packet is returned here. If first_edesc is null, the message is send.

Arguments:
    src - message buffer
    num_bytes - message length
    dest_port - destination port
    source_port - source port
    first_edesc - if not null, the allocated descriptor is returned here

Return Value:
    NFX_OK
    EEE_ERR_NO_BUF
    EEE_ERR_NO_DESC
    EEE_ERR_UNKNOWN_PORT
    EEE_ERR_PACKET_TOO_LARGE
    EEE_ERR_BAD_PARAM
    
    EEE_FORWARDED_PACKET
    EEE_DROP_PACKET
    EEE_TX_QUEUE_FULL
 */
int32
eee_alloc_or_send_message(void *src,
    uint32 num_bytes,
    uint32 dest_port,
    uint32 source_port,
    eee_desc_t **first_edesc)
{
    eee_descPtr_t edesc;
    uint32 mp;
    uint32 bytes_remain;
    eee_descPtr_t startDesc = NULL;
    eee_descPtr_t *pprev = &startDesc;
    uchar8 *srcbuf = src;
    int error = 0;

    // printk("eee_sendMessage(%p, 0x%x, 0x%x, 0x%x)\n", src, num_bytes, dest_port, source_port);

    if (num_bytes == 0) {
        /* The caller is trying to send a 0 byte message.
         */
        ASSERT(FALSE);
        return EEE_ERR_BAD_PARAM;
    }

    if (EEE_GET_SLOT_NUM(dest_port) != nfxMySlotId) {
        return eee_sendFragmentedMessage(src, num_bytes, dest_port, source_port, first_edesc);
    }

    /*
     * Allocate descriptor
     */
    mp = GET_MEMPOOL_INDEX_FROM_CPU_ID(EEE_GET_DEST_CPU(dest_port));

    /*
     * can't send chained packets to different cpu on same card
     */
    ASSERT(eee_is_local_slot_cpu(EEE_GET_SLOT_NUM(dest_port), 
                                 EEE_GET_DEST_CPU(dest_port)) ||
           (num_bytes <= EEE_BUFSIZE_LARGE * (EEE_MAX_BUFS-1)));

    /*
     * Allocate buffers
     */
    bytes_remain = num_bytes;

    while (bytes_remain > 0 && !error) {
        edesc = eee_allocateDesc(mp, NFX_COS_HIGH);
        if (edesc != NULL) {
            *pprev = edesc;
            pprev = &edesc->hdr.next;
            edesc->hdr.next = NULL;
            edesc->hdr.control = 0;
#if defined(NETEEE_FRAGMENT)
            edesc->hdr.offset = 0;
#endif          
            while (bytes_remain > 0 && !error
                   && EEE_GET_NUM_BUFS(edesc) < EEE_MAX_BUFS - 1) {
                eee_descBuf_t *bd = &edesc->bd[EEE_GET_NUM_BUFS(edesc) + 1];
                bd->buf = NULL;
                if (bd->buf == NULL
                    && bytes_remain < eee.buf_size[EEE_BUF_SMALL]
                    - eee.buf_offset) {
                    bd->buf = eee_allocateBuffer(mp, EEE_BUF_SMALL,
                                                      NFX_COS_HIGH);
                    if (bd->buf != NULL) {
                        bd->control = EDESC_BD_SMALL | EDESC_BD_CMD_IN_USE;
                        bd->len = bytes_remain;
                    }
                }
                if (bd->buf == NULL
                    && bytes_remain < eee.buf_size[EEE_BUF_LARGE]
                    - eee.buf_offset) {
                    bd->buf = eee_allocateBuffer(mp, EEE_BUF_LARGE,
                                                          NFX_COS_HIGH);
                    if (bd->buf != NULL) {
                        bd->control = EDESC_BD_LARGE | EDESC_BD_CMD_IN_USE;
                        bd->len = bytes_remain;
                    }
                }
                if (bd->buf == NULL) {
                    bd->buf = eee_allocateBuffer(mp, EEE_BUF_LARGE,
                                                      NFX_COS_HIGH);
                    if (bd->buf != NULL) {
                        bd->control = EDESC_BD_LARGE | EDESC_BD_CMD_IN_USE;
                        bd->len = EEE_BUFSIZE_LARGE;
                    }
                }
                if (bd->buf != NULL) {
                    memcpy(bd->buf, srcbuf, bd->len);
                    bytes_remain -= bd->len;
                    srcbuf += bd->len;
                    EEE_SET_NUM_BUFS(edesc, EEE_GET_NUM_BUFS(edesc) + 1);
                }
                else {
                    error = EEE_ERR_NO_BUF;
                }
            }
        }
        else {
            error = EEE_ERR_NO_DESC;
        }
    }

    if (error || bytes_remain != 0) {
        eee_freePacket(startDesc);
    } else {
        EEE_SET_PKT_LEN(startDesc, num_bytes);
        startDesc->hdr.src_port = source_port;
        startDesc->hdr.dest_port = dest_port;

        if(first_edesc != NULL) {
            *first_edesc = startDesc;
            error = 0;
        } else {
            error = eee_forwardPacket(startDesc);
            if (error == EEE_TX_QUEUE_FULL) {
                eee_freePacket(startDesc);
            }
        }
    }
    return error;
}

/*-----------------------------------------------------------------
 * Name:    eee_sendMessage()
 *
 * Description: The sendMessage routine will COPY local data into
 *              system buffers and attach them to an EEE descriptor,
 *              which can be passed through the system.  The local
 *              buffer is still owned by the caller upon return.
 * Return Values:
 * NFX_OK
 * EEE_ERR_NO_BUF
 * EEE_ERR_NO_DESC
 * EEE_ERR_UNKNOWN_PORT
 * EEE_ERR_PACKET_TOO_LARGE
 * EEE_ERR_BAD_PARAM
 * EEE_FORWARDED_PACKET
 * EEE_DROP_PACKET
 * EEE_TX_QUEUE_FULL
 *-----------------------------------------------------------------
 */
int32
eee_sendMessage(void *src,  /* start address of bytes to send */
    uint32  num_bytes,      /* number of bytes to send */
    uint32  dest_port,      /* dst port address,  used as the dest_port in the EEE descriptor hdr */
    uint32  source_port)
{
    return eee_alloc_or_send_message(src, num_bytes, dest_port, source_port, NULL);
}  


