/* vi:ts=4
 *-----------------------------------------------------------------
 * Name:	eee-poll.h
 * Description:	Polling routines
 *-----------------------------------------------------------------
 */
#ifndef _EEE_POLL_H
#define _EEE_POLL_H

#define EEE_HIGH_PRIORITY		1
#define EEE_LOW_PRIORITY		2

#define POLLING_CAN_CONTINUE(x)	((x)--)

typedef struct _eee_poll_cfg_t {
	uchar8	* pollCfgName;
	void	*func;
} eee_poll_cfg_t;

#define POLLCFG_RSP_SECTION_LABEL	0xFFFFFFFFFFFFFFFELL
#define POLLCFG_RSP_END			0xFFFFFFFFFFFFFFFFLL
#define POLLCFG_RSP_STR_LEN		16
#define POLLCFG_RSP_MAX			128

typedef struct _eee_poll_cfg_rsp_t {
	uchar8	str[POLLCFG_RSP_STR_LEN];
	uint64	cnt;
} eee_poll_cfg_rsp_t;

typedef struct _eee_poll_cfg_req_t {
	uint32	cmd;
} eee_poll_cfg_req_t;

struct local_queue {
	spinlock_t lq_spin;
	struct edesc *lq_head;
	struct edesc *lq_tail;
	uint64	num_called;
	uint64	poll_idle;
	uint64	poll_valid;
	uint64	rcv_cnt;
        uint64  enqueued;
        uint64  dequeued;
	uint32	hwm;
	uint32	max_val;
};

#if defined(NFP_TXRX)
extern struct local_queue local_queueA; /* local queue from Txrx ACPU */
#endif

/*-----------------------------------------------------------------
 * Name:	eee_registerPollingFunc(priority, polling_func, cb)
 *
 * Description:	Polling functions will be called periodically.
 *				Drivers should use polling functions to process receive
 *				rings, transmit cleanup, etc.
 *				The polling function will be called with the user-supplied
 *				control block, and a timer reference value.  The timer
 *				reference will be utilized via some EEE layer macros.
 *
 * Created by:	Rick Lund
 *
 * Date Created:	4/9/01
 *
 *-----------------------------------------------------------------
 */
int32						/* returns handle for polling function or
							 * EEE_ERR_POLLING_TABLE_FULL
							 */
eee_registerPollingFunc(
	uint32	priority,			/* EEE_HIGH_PRIORITY, EEE_LOW_PRIORITY */
	void	(*polling_func)(		/* function to be called as part of
									 * EEE layer polling loop
									 */
				void	* cb,		/* user defined control block for
									 * polling function
									 */
				uint32	timer_ref	/* some indication of how much time
									 * this routine can use the CPU, or
									 * how many packets to process in a
									 * loop - supplied by the EEE layer
									 */
			),
	void	* cb,
	eee_poll_cfg_t	*poll_cfg_struct
);

/*-----------------------------------------------------------------
 * Name:	eee_unregisterPollingFunc(priority, handle)
 *
 * Description:	Basically provided for completeness.  A polling function
 *				can be removed from polling loop.
 *
 * Created by:	Rick Lund
 *
 * Date Created:	4/9/01
 *
 *-----------------------------------------------------------------
 */
int32						/* returns NFX_OK,
							 * EEE_ERR_POLLING_FUNC_INVALID
							 */
eee_unregisterPollingFunc(
	int32	handle				/* from eee_registerPollingFunc()
								 * return code
								 */
);

/*++

Routine Description:

    Stop all cores except the current one.
    
Arguments:

    org_dont_poll - the original state of dont_poll variables for the cores.
    stopped - if the corresponding element in the array is 1, the core
              was stopped successfully

Return Value:

    NFX_OK
    NFX_ERR

--*/

int
eee_freeze_all(boolean *org_dont_poll, 
               boolean *stopped);

/*++

Routine Description:

    Restart the stopped cores.

Arguments:

    org_dont_poll - the array of dont_poll values for the cores. If 0 restart
                    the core.

Return Value:

    None.

--*/

void
eee_thaw_all(boolean *org_dont_poll);

/*++

Routine Description:

    Disable the external links and stop the cores.

Arguments:

    None.

Return Value:

    None.

--*/

void
eee_halt_prepare(void);

/*++

Routine Description:

    Enter the halt loop.

Arguments:

    None.

Return Value:

    None.

--*/

void
eee_halt(void);


/*-----------------------------------------------------------------
 * Name:	eee_poll()
 *
 * Description: Main polling loop
 *
 * Created by:	Rick Lund
 *
 * Date Created:	4/10/01
 *
 *-----------------------------------------------------------------
 */
#if !defined(_KERNEL)
void
eee_poll(uint32 num_loops);
#endif

/*
 * Poll config functions
 */
void eee_pollCfgSyntax(uchar8 *cmd_name);
void eee_pollCfgView(uint32 just_hwm);
void eee_pollCfgClear(void);
void eee_pollCfgSet( uchar8 *poll_name, uint32 max_val, uchar8 *cmd_name);
void eee_pollCfgSSC(uint32 just_hwm, eee_poll_cfg_rsp_t *rsp, uint32 max_rsp);

#endif	/* _EEE_POLL_H */

