/* vi:ts=4
 *-----------------------------------------------------------------
 * Copyright (C) 2003, ClariStor Inc.
 * RCS:  $Id: Exp $
 * Name         : pkt-queue-api.h
 * Description  : Defines API for managing queues of packets.
 * Created by   : Maxim Kozlovsky
 * Date Created : 01/31/2003
 *-----------------------------------------------------------------
 */
#ifndef _PKT_QUEUE_API_H
#define _PKT_QUEUE_API_H

#ifdef LATER
#include "../sm-cifs/ntstatus.h"
#include "../sm-pkt/pkt-queue-struct.h"
#include "../sm-req-queue/req-queue-api.h"
#include "../sm-cifs/cifs-conn-struct.h"
#include "../sm-tpl/tpl-api.h"
#include "../sm-tpl/tpl.h"
#include "../sm-malloc-slab/malloc-api.h"

#else

typedef struct any_request {
   unsigned int req_type;
} any_request_t, *any_request_ptr_t;

typedef struct any_acpu_conn {
  unsigned int v, maxBuflenFirst, maxBuflen;
} any_acpu_conn_t, *any_acpu_conn_ptr_t;

typedef struct acpu_conn {
   unsigned int maxBuflenFirst, maxBuflen;
   unsigned int mtuSizeFirst, mtuSize;
} acpu_conn_t, *acpu_conn_ptr_t;
#endif /* LATER */

#if defined(BASIC_STATS)
#define UPDATE_STAT_INC(stat) ++(stat)
#define UPDATE_MAXSTAT_INC(stat, maxstat) {if (++(stat) > maxstat) maxstat = stat;}
#define UPDATE_STAT_DEC(stat) --(stat)
#define UPDATE_STAT_MINUS(stat, dec) ((stat) -= dec)
#define UPDATE_STAT_PLUS(stat, plus) ((stat) += plus)
#define UPDATE_MAXSTAT_PLUS(stat, plus, max) {if (((stat) += (plus) > maxstat) maxstat = stat;}
#define UPDATE_MAXSTAT(stat, max) {if ((stat) > (maxstat)) maxstat = stat;}
#else
#define UPDATE_STAT_INC(stat)
#define UPDATE_MAXSTAT_INC(stat, maxstat)
#define UPDATE_STAT_DEC(stat)
#define UPDATE_STAT_MINUS(stat, dec)
#define UPDATE_STAT_PLUS(stat, plus)
#define UPDATE_MAXSTAT_PLUS(stat, plus, max)
#define UPDATE_MAXSTAT(stat, maxstat) 
#endif

/* Number of packet with large buffers that can be allocated on ACPU.
 */
#define PKT_ACPU_LIMIT 10000

/* Packet allocator free list.
 */
extern struct pkt_free_list pkt_acpu_free_list;
extern struct pkt_free_list pkt_ncpu_free_list;
extern struct sk_buff *pkt_acpu_memory;

/* The allocation statistics.
 */
extern int32 pkt_alloc_stats[];

/*++

Routine Description:

    Validate the packet.
    
Arguments:

    pkt - packet descriptor

Return Values:

    Returns true if the packet is valid.

--*/

boolean
pkt_validate(struct sk_buff *pkt);


/*++

Routine Description:
    Validate the packet and it's next pointers.
    
Arguments:
    pkt_head - packet descriptor

Return Values:
    Returns true if the packet is valid.
--*/
boolean
pkt_validate_chain(struct sk_buff *pkt_head);

/*
Routine Description:

    This routine initializes the packet queue to contain the packet
    specified by the argument packet. The pq_current field
    is initialized to point to the start of the queue, pq_offset
    and pq_start_offset are initialized to 0.

Arguments:

    pq - queue to initialize
    packet - pointer to the first packet in the packet chain
    len - the length of the queue

Return Values:

    None.
*/
void pq_init_from_packet(pkt_queue_t *pq, struct sk_buff *packet, uint32 len);

/*
Routine Description:

    This routine makes an empty queue.

Arguments:

    pq - queue to initialize

Return Values:

    None.    
*/
void pq_make_empty(pkt_queue_t *pq);


/*
Routine Description:
    Check if the queue is empty.

Arguments:
    pq - packet queue

Return Values:
    Returns TRUE if the queue is empty.
*/
static inline int
pq_is_empty(pkt_queue_t *pq)
{
    return pq->pq_head == NULL;
}

/*
Routine Description:
    This routine adds a packet to the tail of the queue. The queue length
    is adjusted according to the value of hdrSize field of packet.

Arguments:
    pq - packet queue
    packet - packet to add

Return Values:
    None.
*/    
void pq_add_packet(pkt_queue_t *pq, struct sk_buff *packet);

/*++

Routine Description:

    This routine appends a packet queue specified by the argument
    add_queue to the tail of the packet queue specified by the
    argument pq.

Arguments:

    pq - packet queue
    add_queue - packet queue to add

Return Values:

    None.

--*/
 
void pq_append_queue(pkt_queue_t *pq, pkt_queue_t *add_queue);

/*--
  
Routine Description:

    This routine appends a packet chain to the tail of the packet queue
    specified by the argument pq.

Arguments:

    pq - packet queue
    head - start of the packet chain
    tail - tail of the packet chain
    size - the size of the packet chain

Return Values:

    None.
    
++*/

void 
pq_append_chain_to_queue(pkt_queue_t *pq, struct sk_buff *head,
    struct sk_buff *tail, uint32 size);

/*
Routine Description:

    This routine advances the pq_current, pq_offset and
    pq_start_offset fields by the number of bytes specified by the
    argument bytes towards the end of the queue.

    If there is no sufficient data in the queue, the pq_current
    will be NULL in the end. The caller should either check pq_current
    pointer after calling this function, or check if
    pq_len is greater than pq_start_offset + bytes before calling this
    function.

Arguments:

    pq - packet queue
    bytes - the number of bytes to advance

Return Values:

    None.
*/
void pq_adjust(pkt_queue_t *pq, uint32 bytes);


/*
Routine Description:

    Set the queue current pointers to the given offset from the
    beginning of the packet queue.

Arguments:

    pq - packet queue
    offset_from_start - offset from the beginning of the packet
                        queue

Return Values:

    None.
*/
void pq_seek_to(pkt_queue_t *pq, uint32 offset_from_start);

/* Allocation types
 */
#define M_WAIT 0
#define M_NOWAIT 1

/*++
  
Routine Description:

    This routines makes sure that the packet data starting at the current
    pointer is occupying contiguous locations.

    If M_NOWAIT is specified in allocation flags, the packet allocation is
    required and there are no memory available, STATUS_INSUFF_SERVER_RESOURCES
    will be returned. Otherwise, the request will block for memory allocation.

    This routine may have to perform memory allocation to have large
    enough contiguous region, so the caller should prepare to block
    before calling this routine.

Arguments:

    req - current request
    pq - packet queue
    pullup_size - the required contiguous size
    free_list - free list to use for the packet allocation if
                required
    flags - allocation flags 

Return Values:

    STATUS_SUCCESS - pullup successful.
    STATUS_INSUFF_SERVER_RESOURCES - not enough memory and
                                      M_NOWAIT was specified in the
                                      allocation flags
    STATUS_PENDING - need to wait for the memory allocation.
    
--*/

#ifdef LATER
NTSTATUS
pq_pullup(req_hdr_t *req, pkt_queue_t *pq,
          uint32 size, struct pkt_free_list *free_list, uint32 flags);

#else
NTSTATUS
pq_pullup(void *req, pkt_queue_t *pq, uint32 size, struct pkt_free_list *free_list, uint32 flags);
#endif /* LATER */

/*
Routine Description:

    This routine checks if the packet data at the current pointer
    is contiguous.

Arguments:

    pq - packet queue
    size - requested size of contiguous region

Return Values:

    Returns true if the data is contiguous, otherwise returns false.
*/
int
pq_is_contig(pkt_queue_t *pq, uint32 size);


/*++

Routine Description:

    This routine returns whether the given pointer resides within the given
    packet.

Arguments:

    pkt - packet descriptor
    ptr - The pointer to check.

Return Value:

    TRUE - ptr resides within pkt.
    FALSE - Otherwise.

--*/

static inline boolean
pkt_contains_ptr(struct sk_buff *pkt, char *ptr)
{
#ifdef LATER
    return ((ptr >= pkt->pHdr) && (ptr <= (pkt->pHdr + pkt->hdrSize)));
#else
    return 1;
#endif
}


/*
Routine Description:

    Get the length of the buffer attached to the packet descriptor.

Arguments:

    pkt - packet descriptor

Return Values:

    Returns the length of the buffer.
*/
static inline uint32
pkt_buflen(struct sk_buff *pkt)
{
#ifdef LATER
    return EEE_BUFSIZE_LARGE - sizeof(struct sk_buff);
#endif
    return ((uint32)pkt->len);
}

#define COPY_TYPE_FROM_PKT    1
#define COPY_TYPE_TO_PKT      2

/*
Routine Description: 

    Copies a flat buffer into a packet descriptor queue
    Or copies a packet descriptor queue into a flat buffer
    Assumes that there is sufficient data/space in the pkt_desc

Arguments:
 
    copy_type     - type of copy, COPY_TYPE_FROM_PKT or COPY_TYPE_TO_PKT
    pkt           - packet descriptor
    pkt_offset    - start offset in the pkt desc.
    buf           - flat buffer
    num_bytes     - number of bytes to copy
    pktEnd        - ending packet desc. (optional arg)
    pkt_endOffset - ending packet offset (optional arg)
 
Return Value:

    None
*/
void
pq_copyBufPkt(int copy_type, struct sk_buff *pkt, uint16 pkt_offset, 
    char *buf, uint32 num_bytes, struct sk_buff **pktEnd, uint16 *pkt_endOffset);

/*++
Routine Description:
    This routine copies data from one packet chain into another.

Arguments:
    pkt_src - The source data packet.
    pkt_src_offset - The starting offset within the source data packet.
    num_bytes - The number of bytes to copy.
    pkt_dest - The destination data packet.
    pkt_dest_offset - The starting offset within the destination data packet.

Return Value:
    None.
--*/
void
pq_copyPkt(struct sk_buff *pkt_src,
           uint32 pkt_src_offset,
           uint32 num_bytes, 
           struct sk_buff *pkt_dest,
           uint32 pkt_dest_offset);

/*
Routine Description:

    Wrapper around pq_copyBufPkt
    Copies data from a pkt_desc chain into a flat buffer

    See pq_copyBufPkt for more details
*/    
static inline void
pq_copydata(struct sk_buff *pkt, uint16 pkt_offset, char *buf, uint32 num_bytes,
            struct sk_buff **pktEnd, uint16 *pkt_endOffset)
{
    pq_copyBufPkt(COPY_TYPE_FROM_PKT, pkt, pkt_offset, buf, num_bytes,
                  pktEnd, pkt_endOffset);
}

/*
Routine Description:

    Wrapper around pq_copyBufPkt
    Copies data from a flat buffer into a pkt_desc chain

    See pq_copyBufPkt for more details
*/    
static inline void
pq_setdata(struct sk_buff *pkt, uint16 pkt_offset, void *buf, uint32 num_bytes,
            struct sk_buff **pktEnd, uint16 *pkt_endOffset)
{
    pq_copyBufPkt(COPY_TYPE_TO_PKT, pkt, pkt_offset, buf, num_bytes,
                  pktEnd, pkt_endOffset);
}

/* 
Routine Description: 
    Copies a eDesc chain into a packet descriptor queue
    Or copies a packet descriptor queue into a eDesc chain
    Assumes that there is sufficient space in the eDesc chain
    and pkt desc chain
  
Arguments:
 
    copy_type       - type of copy, COPY_TYPE_FROM_PKT or COPY_TYPE_TO_PKT
    pkt             - packet descriptor
    pkt_offset      - start offset in the pkt desc.
    eDesc           - eDesc chain
    eDesc_offset    - start offset in eDesc chain first eDesc
    num_bytes       - number of bytes to copy
    pktEnd          - ending packet desc. (optional arg)
    pkt_endOffset   - ending packet offset (optional arg)
    eDescEnd        - ending eDesc (optional arg)
    eDesc_endOffset - ending eDesc offset (optional arg)
 
Return Value:

    None
*/
void
pq_copyEdescPkt(int copy_type, struct sk_buff *pkt, uint16 pkt_offset,
                eee_desc_t *eDesc, uint16 eDesc_offset, uint32 num_bytes,
                struct sk_buff **pktEnd, uint16 *pkt_endOffset); 

/*
Routine Description:

    Wrapper around pq_copyEdescPkt
    Copies data from an eDesc chain to a pkt_desc chain

    See pq_copyEdescPkt for more details
*/    
static inline void
pq_copyEdescToPkt(struct sk_buff *pkt, uint16 pkt_offset,
    eee_desc_t *eDesc, uint16 eDesc_offset, uint32 num_bytes,
    struct sk_buff **pktEnd, uint16 *pkt_endOffset) 
{
    pq_copyEdescPkt(COPY_TYPE_TO_PKT, pkt, pkt_offset, eDesc, 
                    eDesc_offset, num_bytes, pktEnd, pkt_endOffset);
}
                    
/*
Routine Description:

    Wrapper around pq_copyEdescPkt
    Copies data from a pkt_desc chain to an eDesc chain

    See pq_copyEdescPkt for more details
*/    
static inline void
pq_copyPktToEdesc(struct sk_buff *pkt, uint16 pkt_offset,
                  eee_desc_t *eDesc, uint16 eDesc_offset, uint32 num_bytes,
                  struct sk_buff **pktEnd, uint16 *pkt_endOffset) 
{
    pq_copyEdescPkt(COPY_TYPE_FROM_PKT, pkt, pkt_offset, eDesc, 
                    eDesc_offset, num_bytes, pktEnd, pkt_endOffset);
}

#define PQ_SOURCE_UNDERFLOW    1
#define PQ_DESTN_OVERFLOW      2

/*
Routine Description:

    Parses a packet looking for an NULL Terminated ASCII string
    Returns the string in the destination

Arguments:
    pkt           - source packet
    pkt_offset    - starting offset in the packet
    destn         - destination string
    destn_size    - maximum size of destination (including NULL terminator)
    pktEnd        - packet containg next data byte (optional arg)
    pkt_endOffset - start of next data byte (optonal arg)

Return Values:
    NFX_OK  - if string was successfully parsed from packet
    PQ_SOURCE_UNDERFLOW - if the source got exhausted and NULL terminator was
                          not found
    PQ_DESTN_OVERFLOW   - if the destination was too small
*/    
int32
pq_parseAsciiString(struct sk_buff *pkt, uint16 pkt_offset,
                    char *destn, uint16 *destn_size,
                    struct sk_buff **pktEnd, uint16 *pkt_endOffset);

/*++
Routine Description:
    Setup a request packet to use for sending a reply.

Arguments:
    pkt - packet

Return Values:
    None.
--*/
static inline void
pkt_reuse_request_packet_for_reply(struct sk_buff *pkt)
{
#ifdef LATER
    ASSERT((pkt->flags & PKT_DESC_INUSE));
    uint32 alloc_flags = (pkt->flags & PKT_ALLOCATED_ON_ACPU);

    if (!(pkt->flags & PKT_ALLOCATED_ON_ACPU)) {
#ifdef REQ_VALIDATE_PACKET_COUNTS
        if (creq.v != NULL) {
            creq.v->rh_packets_allocated++;
        }
#endif
    }
    pkt->pHdr = ((void *)pkt + PKT_BUFF_INDENT);
    pkt->pktSize = 0;
    pkt->flags = (PKT_DESC_INUSE | PKT_FAST_PATH | PKT_DATA | alloc_flags);
    pkt->p_time_stamp = 0;
#endif
}


/*++

Routine Description:
    Validate the queue fields.

Arguments:
    pq - packet queue

Return Value:
    None.
--*/
void
pq_validate(pkt_queue_t *pq);

#if !defined(DEBUG)
#define pq_validate(pq)
#endif


/*++

Routine Description:

    This routine attempts to align the data starting from the current pointer
    to the request alignment.  If possible the data will be right shifted
    within the packet, but if that isn't possible a new packet will be
    allocated.

    We will not try to align more than the maximum bytes specified by the
    caller.

    If the amount of data remaining in the current packet is not a multiple
    of the given alignment size, we will pullup some characters from the
    next packet in the chain so that there are no partial units.

    This primary purpose of this call is to pullup misaligned Unicode
    strings.  In that case there should be at most a single byte movement
    but we do want to avoid having a code point span a packet boundary.

    WARNING - This move is a destructive process.  If the caller were to
              pq_seek backwards they would not be able to easily locate the
              buffer.  For example, the CIFS Unicode string would not be
              found in the expected place after the fixed portion of the
              request structure.  As long as the caller knew that they had
              made this call then they could use that knowlege to pq_adjust
              them packet to the location we will be moving things.

    This is a blocking call as a new packet might need to be allocated.

Arguments:

    reqHdr - The request.
    pq - packet queue
    align_size - The alignment size in bytes.
    max_size - The maximum number of bytes to pull up.  We may pull up
               less, but it must be a multiple of the alignment size.

Return Value:

    STATUS_SUCCESS - Success.
    STATUS_PENDING - The request was queued.  Return back to the dispatcher.
--*/
#ifdef LATER
NTSTATUS
pq_alignAtOffset(req_hdr_t *req, pkt_queue_t *pq, uint32 align_size, uint32 max_size);
#else
int
pq_alignAtOffset(void *req, pkt_queue_t *pq, uint32 align_size, uint32 max_size);
#endif

/*++

Routine Description:

    Set the packet header size.

Arguments:

    pkt - packet
    hdrSize - header size

Return Value:

    None.

--*/

static inline void
pkt_setHdrSize(struct sk_buff *pkt, uint32 hdrSize)
{
#ifdef LATER
    ASSERT((hdrSize <= pkt_buflen(pkt)) || (pkt->flags & PKT_DCACHE_REF));
    pkt->hdrSize = hdrSize;
#else
    pkt->len = hdrSize;
#endif
}


/*++

Routine Description:
    Split the current packet at the current offset moving the rest of the data
    into the provided packet.

Arguments:
    pq - packet queue
    new_packet - new packet

Return Value:
    None.
--*/
void
pq_split_current_packet_right(pkt_queue_t *pq, struct sk_buff *new_packet);

/*++

Routine Description:
    Split the packet in the queue at the given offset moving the rest of
    the data into the provided packet.

Arguments:
    pq - packet queue
    split_packet - packet to split
    offset - offset in the split_packet
    new_packet - new packet

Return Value:
    None.

--*/

void
pq_split_right(pkt_queue_t *pq,
               struct sk_buff *split_packet,
               uint32 offset,
               struct sk_buff *new_packet);

/*++

Routine Description:
    Split the current packet at the current offset. The data before the
	current position is moved to the new packet.

Arguments:
    pq - packet queue
    new_packet - new packet

Return Value:
    None.
--*/
void
pq_split_current_packet_left(pkt_queue_t *pq, struct sk_buff *new_packet);

/*++

Routine Description:
    Save the current queue data so it can be compared after a queue
    modification operation.

Arguments:
    pq - packet queue

Return Value:
    None.
--*/
static inline void
pq_save_data(pkt_queue_t *pq )
{
#if defined(JONG_HACK)
    pq->pq_save_data = eee_ramAlloc(pq->pq_len);
    if (pq->pq_save_data != NULL) {
        pq_copydata(pq->pq_head, 0, pq->pq_save_data, pq->pq_len, NULL, NULL);
    }
#endif    
}

/*++

Routine Description:
    Compare current queue data against saved data. The saved data is
    discarded after successful comparison.

Arguments:
    pq - packet queue

Return Value:
    None.
--*/
static inline void
pq_compare_saved_data(pkt_queue_t *pq )
{
#if defined(JONG_HACK)
    if (pq->pq_save_data != NULL) {
        void *new_data = eee_ramAlloc(pq->pq_len);
        if (new_data != NULL) {
            pq_copydata(pq->pq_head, 0, new_data, pq->pq_len, NULL, NULL);
            ASSERT(memcmp(pq->pq_save_data, new_data, pq->pq_len) == 0);
            eee_ramDealloc(new_data);
        }
        eee_ramDealloc(pq->pq_save_data);
        pq->pq_save_data = NULL;
    }
#endif    
}

/*++
Routine Description:
    Get the pointer to the current data.

Arguments:
    pq - packet queue

Return Value:
    Returns the pointer to the queue data in the current packet at the current offset.
--*/

static inline void *
pq_current_ptr(pkt_queue_t *pq)
{
#ifdef LATER
    return pq->pq_current->pHdr + pq->pq_offset;
#else
    return pq->pq_current->next;
#endif
}

/*++

Routine Description:
    Allocate a packet descriptor from free packet list.

Arguments:
    free_list - the free list to allocate from
    alloc_type - the allocation type

Return Value:
    Returns pointer to the packet descriptor allocated or NULL if not enough
    memory.

--*/

struct sk_buff *
pkt_alloc_from_list(struct pkt_free_list *free_list, int32 alloc_type);

/*++

Routine Description:
    Allocate a packet with an associated buffer from ACPU free packet list.

Arguments:
    alloc_type - allocation type

Return Value:
    Returns pointer to the packet descriptor allocated or NULL if not enough memory.

--*/
static inline struct sk_buff *
pkt_acpu_alloc(int alloc_type)
{
#ifdef LATER
    return pkt_alloc_from_list(&pkt_acpu_free_list, alloc_type);
#else
    return NULL;
#endif
}

/*++

Routine Description:
    Free the single packet to the ACPU free packet list.

Arguments:
    pkt - packet descriptor

Return Value:
    None.
--*/
void
pkt_acpu_free(struct sk_buff *pkt);


/*++

Routine Description:
    Free a chain of packets linked with pNext field.

Arguments:
    pkt - start of the packet chain

Return Value:
    None.

--*/

void
pkt_acpu_free_chain(struct sk_buff *pkt);


/*++

Routine Description:
    Free a packet. 

Arguments:
    pkt - packet to free

Return Value:
    None.
--*/
void
pkt_free(struct sk_buff *pkt);


/*++
Routine Description:
    Wait until the memory in the packet free list becomes available.

    The request state should be either CIFS_REQ_ALLOC_PACKETS_CONTINUE if the
    request is allocating into a packet queue or REQ_WAIT_ONE_PACKET if the
    request wants to allocate just one packet.

Arguments:
    req - request

Return Value:
    Returns STATUS_PENDING.
--*/
NTSTATUS
pkt_acpu_wait(any_request_ptr_t req);

/*++

Routine Description:

    Get the size of the network headers in the packet.    
    
Arguments:

    protocol - the protocol of the connection the allocated packet is
               going to be sent on.
    is_first - if TRUE, this is the first packet in the reply chain

Return Values:

    Returns the number of bytes used by the network headers.

--*/

static inline uint32
pkt_network_header_size(uint32 protocol, boolean is_first, uint32 vlan)
{
#ifdef LATER
    ASSERT((protocol == TPL_PROT_TCP) || (protocol == TPL_PROT_UDP));
    uint32 hdrsize;

    hdrsize = ((protocol == TPL_PROT_TCP) ?
            (is_first ?
             PKT_TCP_FIRST_HEADER_SIZE :
             PKT_TCP_HEADER_SIZE) :
            (is_first ?
             PKT_UDP_FIRST_HEADER_SIZE :
             PKT_UDP_HEADER_SIZE));

    hdrsize += (vlan) ? PKT_VLAN_EXTRA_ETH_HEADER_SIZE : 0;
    return (hdrsize);
#else
    return 20;
#endif
}

/*++

Routine Description:
    Get the length usable for the data according to the connection MTU.
    
Arguments:
    conn - the connection
    is_first - if TRUE, this is the first packet in the reply chain

Return Values:

    Returns the number of bytes that can be used. 

--*/

static inline uint32
pkt_usable_data_len(acpu_conn_t *conn, boolean is_first)
{
    return is_first ? conn->mtuSizeFirst : conn->mtuSize;
}

/*++

Routine Description:
    Find out how much bytes in the packet buffer can be used depending on the 
    allocation type.  

Arguments:
    conn - the connection
    is_first - if TRUE, this is the first packet in the reply chain

Return Value:
    Returns the number of bytes that can be used. 
--*/
static inline uint32
pkt_usable_len(acpu_conn_t *conn, boolean is_first)
{
    return is_first ? conn->maxBuflenFirst : conn->maxBuflen; 
}   


/*++

Routine Description:

    Send a packet allocated from ACPU allocator.

Arguments:

    conn - tpl connection
    pkt - packet descriptor
    req - request

Return Value:

    None.

--*/

static inline void
pkt_acpu_send(any_acpu_conn_ptr_t pConnId, struct sk_buff *pkt, any_request_ptr_t req)
{
#ifdef LATER
#ifdef DEBUG
    struct sk_buff *ipkt = pkt;
    while (ipkt != NULL) {
        ASSERT(ipkt->hdrSize <= pkt_usable_data_len(pConnId.v, FALSE));
        ASSERT(pkt_validate(ipkt));

        ipkt = ipkt->pNext;

#ifdef REQ_VALIDATE_PACKET_COUNTS
        if (creq.v != NULL) {
            creq.v->rh_packets_sent++;
        }
#endif /* REQ_VALIDATE_PACKET_COUNTS */
    }
#endif /* DEBUG */
    pkt->pTplConn = pConnId.v;
    ASSERT(pkt->p_req == NULL);

    if ((req.v != NULL) && (req.v->rh_kpi.ks_graph_level >= KPI_LEVEL_3)) {
        pkt->p_req = req.v;
        ASSERT(!(req.v->rh_flags & REQ_PKT_REFERENCE));
        req.v->rh_flags |= REQ_PKT_REFERENCE;
        kpi_enter(&req.v->rh_kpi, KPI_PKT_NCPU_QUEUE, KPI_LEVEL_3);
    }
    tpl_xmtPkt(pkt);
#endif /* LATER */
}


/*++

Routine Description:
    Send a packet allocated from ACPU allocator via the slow path.

Arguments:
    raddr - remote address in network byte order
    rport - remote port in network byte order
    sock - socket
    pkt - packet descriptor

Return Value:
    None.
--*/
static inline void
pkt_acpu_send_slow(uint32 raddr, uint16 rport, void *sock, struct sk_buff *pkt)
{
#ifdef LATER
    ASSERT(sock != NULL);

#ifdef DEBUG
    struct sk_buff *ipkt = pkt;
    while (ipkt != NULL) {
        ASSERT(pkt_validate(ipkt));

        ipkt = ipkt->pNext;

#ifdef REQ_VALIDATE_PACKET_COUNTS
        if (creq.v != NULL) {
            creq.v->rh_packets_sent++;
        }
#endif /* REQ_VALIDATE_PACKET_COUNTS */
    }
#endif /* DEBUG */
    pkt->flags &= ~PKT_FAST_PATH;
    pkt->p_time_stamp = 0;
    tpl_xmtPktSlow(raddr,
                   rport,
                   pkt,
                   sock);
#endif /* LATER */
}


/*++

Routine Description:

    Allocate a packet for a request. The pointer to allocated packet is stored
    in the request structure and can be accessed by calling req_last_result().

    This is a blocking call. The caller needs to setup the request
    state before calling this function.

Arguments:

    req - request

Return Value:

    STATUS_SUCCESS - the packet was allocated
    STATUS_PENDING - the request was blocked for memory allocation

--*/

NTSTATUS
req_alloc_packet(any_request_ptr_t req);


/*++

Routine Description:

    Check if the packet queue contains not more than one packet.

Arguments:

    pq - packet queue

Return Value:

    Returns TRUE if the queue is empty or contains only one packet.

--*/

static inline boolean
pq_is_first_packet(pkt_queue_t *pq)
{
#ifdef LATER
    ASSERT((pq->pq_head == NULL) == (pq->pq_tail == NULL));
#endif
    return pq->pq_head == pq->pq_tail;
}


/*++

Routine Description:

    Find out how many bytes can be used in the tail of the response queue.  

Arguments:

    pq - packet queue
    conn - the connection 

Return Value:

    Returns the number of bytes that can be used in the last packet.

--*/

static inline uint32
pq_usable_len(pkt_queue_t *pq, acpu_conn_t *conn)
{
    return pkt_usable_len(conn, pq_is_first_packet(pq));
}


/*++

Routine Description:
    Allocate packets for the request output queue.

Arguments:
    req - request
    pq - packet queue to do allocations for
    need_bytes - desired queue length
    wait_state - state to wait for memory allocation

Return Value:
    STATUS_SUCCESS - allocated successfully
    STATUS_PENDING - request was queued, return to dispatcher.
--*/
NTSTATUS
pq_alloc_packets(any_request_ptr_t req, pkt_queue_t *pq, uint32 need_bytes, req_state_t wait_state);
                 

/*++
Routine Description:
    Discard unused data at the end of the queue, updating the current
    pointers if necessary.

Arguments:
    pq - packet queue
    num_bytes - number of bytes to discard

Return Value:
    None.
--*/
void
pq_discard(pkt_queue_t *pq, uint32 num_bytes);

/*++
Routine Description:
    Intentionally misalign and split a packet at a random offset.

Arguments:
    pq - packet queue

Return Value:
    None.
--*/
void
pq_split_at_random_offset(pkt_queue_t *pq);


/*++
  
Routine Description:
    Free a NCPU packet chain except the first packet.
Arguments:
    pkt - start of the chain
Return Value:
    None.
--*/
static inline void
pkt_free_chain_except_first(struct sk_buff *pkt)
{
#ifdef LATER
    ASSERT(! (pkt->flags & PKT_ALLOCATED_ON_ACPU));
    
    if (pkt->pNext != NULL) {
        pkt_freePkt(pkt->pNext);
        pkt->pNext = NULL;
    }
#else
    return;
#endif /* LATER */
}


/*++
Routine Description:
    Copy out the cod entry into the response packet queue at the
    current queue position.

    The queue position is adjusted to point after the last written byte.

Arguments:
    pq - packet queue to fill
    cod_offset - offset in cod
    cod - cod
    length - data length

Return Value:
    None.
--*/
void
req_copy_cod_entry(pkt_queue_t *pq, uint32 cod_offset, nfx_cod_t cod, uint32 length);

/*++

Routine Description:
    Add pad at the end of the response queue. The caller should
    make sure that there is enough space in the last packet of
    the queue.

    When the padding is less than 4 bytes, there is always enough
    space in the queue and there should be no need to check.

    The current queue position is advanced forward by pad.

Arguments:
    pq - packet queue
    pad - pad to add.

Return Value:
    None.
--*/
void
pq_pad(pkt_queue_t *pq, uint32 pad);


/*++

Routine Description:

    Get the current queue location.

Arguments:

    pq - packet queue

Return Value:

    Returns the current queue location.

--*/

static inline struct pq_location 
pq_location(pkt_queue_t *pq)
{
    return ((struct pq_location) { pq->pq_current, pq_current_ptr(pq) });
}


/*++

Routine Description:

    Get the NULL queue location. This location is not equal to any valid
    queue location.

Arguments:

    None.

Return Value:

    Returns the NULL queue location.

--*/

static inline struct pq_location
pq_null_location(void)
{
    return (struct pq_location){ NULL, NULL };
}


/*++

Routine Description:

    Check if the queue location is NULL.
    
Arguments:

    location - queue location
    
Return Value:

    Returns TRUE if the location is NULL.
    
--*/

static inline boolean
pq_is_null_location(struct pq_location *location)
{
    return location->pql_data == NULL;
}


/*++

Routine Description:

    Check if two queue locations are equal.    
    
Arguments:

    loc1 - queue location
    loc2 - queue location

Return Values:

    Returns TRUE if the locations are equal.

--*/
static inline boolean
pq_location_equal(struct pq_location *loc1, struct pq_location *loc2)
{
    return loc1->pql_data == loc2->pql_data;
}


/*++

Routine Description:

    Delete some data in the middle of the queue.  It is assumed that the range 
    to be deleted spans at most two packets.  If the range is ending before 
    the end of the packet, the remaining data in the packet is moved towards 
    the beginning of the packet.  If any data was moved, The addresses of the 
    affected range are returned in affectedRangeStart and affectedRangeEnd.  
    
Arguments:

    pq - packet queue 
    location - the location of the beginning of the range to be deleted
    prev_pkt - the previous packet (prev_pkt->pNext == pkt) or NULL if the 
               pkt is the head of the queue
    size - the size of the range to be deleted in bytes
    affectedRangeStart - the start of the moved range is returned here
    affectedRangeEnd - the end of the moved range is returned here
    moveBy - offset by which the range was moved
    
Return Value:

    None.
    
--*/

void
pq_delete_range(pkt_queue_t *pq,    
                struct pq_location *location,
                struct sk_buff *prev_pkt,
                uint32 size,
                void **affectedRangeStart,
                void **affectedRangeEnd,
                int *moveBy);


/*++

Routine Description:

    Free a packet to a free list.

Arguments:

    free_list - packet free list
    pkt - packet descriptor

Return Value:

    None.

--*/

static inline void
pkt_free_to_list(struct pkt_free_list *free_list, struct sk_buff *pkt)
{
#ifdef LATER
#ifdef PKT_ALLOC_USE_TAILQ
    /* @@@ need fast method to determine if packet is on free list.
     */

    *(free_list->pf_tail) = pkt;
    free_list->pf_tail = &pkt->pNext;
    pkt->pNext = NULL;

#else
    // pkt->pNext = free_list->pf_head;
    // free_list->pf_head = pkt;
#endif /* PKT_ALLOC_USE_TAILQ */
    // UPDATE_STAT_DEC(free_list->pf_inuse);
#endif /* LATER */
}

/*++

Routine Description:

    Initialize packet free list.

Arguments:

    free_list - free list to initialize
    num_element - number of elements in the list

Return Value:

    None.

--*/

void
pkt_init_free_list(struct pkt_free_list *free_list, int num_elements);

/*++

Routine Description:

    Validate the packet queue after the allocation done by pq_alloc_packets().
    The allocated packets should match the request's connection MTU and the
    first packet should have the space reserved for the RPC header according
    to the protocol.

Arguments:

    req - request
    pq - packet queue

Return Value:

    None.

--*/

void
pq_validate_after_alloc(any_request_ptr_t req, pkt_queue_t *pq);
#ifndef DEBUG
#define pq_validate_after_alloc(req, pq)
#endif


/*++

Routine Description:

    Make a reference to the dcache entries in the packets of the packet
    queue.
    
Arguments:
    
    req - request.
    ctx - the request context

Return Values:

    STATUS_SUCCESS - copied the data successfully
    STATUS_PENDING - the request was queued, return to the dispatcher.

--*/

#ifdef LATER

NTSTATUS    
req_copy_dcaches_proc(any_request_ptr_t req, struct req_copy_dcaches_ctx *ctx);

/*++

Routine Description:

    Free the dcache entries stored in req_copy_dcaches() context.    
    
Arguments:

    ctx - req_copy_dcaches context

Return Values:

    None.

--*/

void
req_free_dcache_entries(struct req_copy_dcaches_ctx *ctx);

/*++

Routine Description:

    Make a reference to the dcache entries in the packets of the packet
    queue.
    
    The referenced dcaches will have their wb_busyCnt incremented. This will
    prevent them from being overwritten until the transfer of the data is complete.
    
    The wb_busyCnt will be decremented when the packet referencing the dcache is
    freed back to the free list.
    
    The caller should allocate the context structure and setup nr_read_offset, 
    nr_read_count, nr_read_num_dcache_entries, nr_read_dcache_entries before 
    calling this function.  
    
    The copy can be blocked for the queue memory allocation.
    
    See the sibyte documentation for the ugly manipulations necessary to 
    conform to the hardware restrictions.
    
Arguments:
    
    req - request
    ctx - request context
    pq - packet queue to use

Return Values:

    STATUS_SUCCESS - copied the data successfully
    STATUS_PENDING - the request was queued, return to the dispatcher.

--*/

static inline NTSTATUS
req_copy_dcaches(any_request_ptr_t req,
                 struct req_copy_dcaches_ctx *ctx,
                 pkt_queue_t *pq)
{
    ASSERT(ctx->nr_read_num_dcache_entries > 0);
    ctx->nr_read_dcache_index = 0;
    ctx->nr_pq = pq;
    ctx->nr_mtu_unused = 0;
    
    /* Save the end of the queue for undo.
     */
    ctx->nr_save_tail = pq->pq_tail;
    ctx->nr_save_hdrSize = pq->pq_tail->hdrSize;

    return req_copy_dcaches_proc(req, ctx);
}

#endif /* LATER */

/*++

Routine Description:

    Align the packet header so the region of the specified length starting at
    the packet header ends at a cache line boundary.

Arguments:

    pkt - the packet
    len - the region length 

Return Values:

    None.

--*/

static inline void
pkt_align(struct sk_buff *pkt, int len)
{
#ifdef LATER
    pkt->pHdr = alignptr(pkt->pHdr + len, CACHELINE_SIZE) - len;
#endif
}

#endif /* _PKT_QUEUE_API_H */
