/* 
 *-----------------------------------------------------------------
 * Name:        tpl-api.h
 * Copyright (C) 2009, OnStor, Inc.
 * Description: Transport Packet Layer(TPL) inteface definitions
 *-----------------------------------------------------------------
 */
#ifndef _TPL_API_H
#define _TPL_API_H

#include "bqueue.h"

#include "tpl-ipc.h"
#include "pkt-queue-struct.h"

#ifdef LATER
#include "../sm-stats/kpi.h"
#endif

struct _tpl_connCb;

/*
 * Structure containing arguments to create a socket
 */
typedef struct bsd_sarg_t {
	unsigned int laddr;
	unsigned int faddr;
	unsigned short lport;
	unsigned short fport;
	void *handle;
	unsigned int bufType;
	void *cb;
	void *db;
	unsigned int flags;
} bsd_sarg_t;

#define TPL_PROT_UDP   17
#define TPL_PROT_TCP    6

#define	TPL_FREE_PKT	PKT_FREE_PKT
#define	TPL_HOLD_PKT	PKT_HOLD_PKT

#define	TPL_ERR_NO_SHMEM_RSRC	(TPL_ERROR_BASE)
#define	TPL_ERR_NO_BIND_RSRC	(TPL_ERR_NO_SHMEM_RSRC-1)
#define	TPL_ERR_NO_CAM_RSRC	(TPL_ERR_NO_BIND_RSRC-1)
#define	TPL_ERR_NO_PKT_RSRC	(TPL_ERR_NO_CAM_RSRC-1)
#define	TPL_ERR_NO_RSRC		(TPL_ERR_NO_PKT_RSRC-1)
#define	TPL_ERR_DUP_CONN	(TPL_ERR_NO_RSRC-1)
#define	TPL_ERR_BAD_HDL		(TPL_ERR_DUP_CONN-1)
#define	TPL_ERR_BAD_PROT	(TPL_ERR_BAD_HDL-1)
#define	TPL_ERR_BAD_FUNC	(TPL_ERR_BAD_PROT-1)
#define	TPL_ERR_BAD_BIND	(TPL_ERR_BAD_FUNC-1)
#define	TPL_ERR_BAD_CONN	(TPL_ERR_BAD_BIND-1)
#define	TPL_ERR_BAD_EVT		(TPL_ERR_BAD_CONN-1)
#define	TPL_ERR_BAD_PKT		(TPL_ERR_BAD_EVT-1)
#define	TPL_ERR_DUP_BIND	(TPL_ERR_BAD_PKT-1)
#define	TPL_ERR_NOT_BIND	(TPL_ERR_DUP_BIND-1)
#define	TPL_ERR_UL_ADD_FAIL	(TPL_ERR_NOT_BIND-1)
#define	TPL_ERR_UL_DEL_FAIL	(TPL_ERR_UL_ADD_FAIL-1)
#define	TPL_ERR_RING_FULL	(TPL_ERR_UL_DEL_FAIL-1)
#define	TPL_ERR_XMT_FAIL	(TPL_ERR_RING_FULL-1)
#define TPL_ERR_NO_LADDR        (TPL_ERR_XMT_FAIL-1)
#define TPL_ERR_NO_RADDR        (TPL_ERR_NO_LADDR-1)
#define TPL_ERR_NO_LPORT        (TPL_ERR_NO_RADDR-1)
#define TPL_ERR_NO_RPORT        (TPL_ERR_NO_LPORT-1)
#define TPL_ERR_RADDR_IN_SRVR   (TPL_ERR_NO_RPORT-1)
#define TPL_ERR_RPORT_IN_SRVR   (TPL_ERR_RADDR_IN_SRVR-1)
#define TPL_ERR_BAD_MODE        (TPL_ERR_RPORT_IN_SRVR-1)
#define TPL_ERR_BAD_LINK        (TPL_ERR_BAD_MODE-1)
#define TPL_ERR_VSTACK_ERR      (TPL_ERR_BAD_LINK -1)
#define TPL_ERR_MAX        	(TPL_ERR_VSTACK_ERR)

#define TPL_ANY_PORT		0
#define TPL_BIND_SERVER		0x0000
#define TPL_BIND_CLIENT		0x0001	/* Bind client option   */

#define TPL_BIND_USE_RPC	0x0010	/* App. uses RPC header */
#define TPL_BIND_USE_NETBIOS    0x0020  /* App. uses NetBIOS header */
#define TPL_BIND_CLOSE		0x0040	/* bind pts is closing */
#define TPL_BIND_ONLY		0x0080	/* Do only bind...no listen or connect */

#define TPL_BIND_FP_CONN	0x0100	/* This is an FP socket */
#define TPL_BIND_USE_KEEPALIVE  0x0200	/* Use TCP Keepalive option on this socket */
#define TPL_BIND_REUSEPORT	0x0400	/* Reuse the port */
#define TPL_BIND_ROUTETOIF      0x0800	/* Bind socket to IP addr */
/* Do NOT use route lookup mechanism to figure out the source IP address */

#define TPL_FLAG_CLIENT		TPL_BIND_CLIENT
#define TPL_FLAG_USE_RPC	TPL_BIND_USE_RPC
#define TPL_FLAG_USE_NETBIOS	TPL_BIND_USE_NETBIOS
#define TPL_FLAG_BIND_ONLY	TPL_BIND_ONLY
#define TPL_FLAG_FP_CONN	TPL_BIND_FP_CONN
#define TPL_FLAG_USE_KEEPALIVE  TPL_BIND_USE_KEEPALIVE  

/*
 * Routine Description:
 *  The function type called by TPL when a packet chain is received on ACPU.
 *  The application assumes ownership of packet chain and is responsible for freeing it.
 *
 * Arguments:
 *  app_handle - the application handle setup in tpl_addConnInd() routine
 *  pkt - the packet chain received
 *  data - pointer to the beginning of the packet data
 *  tval - receive timestamp
 *
 * Return Values:
 *  None.
 */
typedef     
void (*tpl_rcvPktInd_func) (void *app_handle, struct sk_buff *pkt, void *data, unsigned long tval);

/* The values for acpuState.
 */
typedef enum tpl_AcpuConnState {
    TPL_IPC_CONN_INIT=0,

    /* Connection is active.
     */ 
    TPL_IPC_CONN_ADDED=1,
    
    /* Close has been sent from the ACPU for this connection.
     */
    TPL_IPC_CONN_CLOSE_SENT=2,
    
    /* Received a delete connection request from the ACPU for this
     * connection and the delConnInd() callback for this connection has been
     * executed.
     */
    TPL_IPC_CONN_DELED=3 /* TPL_IPC_CONN_DELETED */
} tpl_AcpuConnState;

/*------------------------------------
 * TPL Bind Control Block.
 *------------------------------------
 */
typedef struct _tpl_bindCb {
	unsigned int	laddr;		/* Local IP address */
	unsigned int	raddr;		/* Remote IP address */
	unsigned short	lport;		/* Local  TCP/UDP port */
	unsigned short	rport;		/* Remote TCP/UDP port */

	unsigned short	tplProt;	/* Protocol UDP/TCP flag */
	unsigned short	bindFlags;	/* Binding flags */

        unsigned int     vsId;          /* Virtual Server ID  */

	void	*usrInfo;		/* Usr cookie */
	void	*pSock;			/* Address of Linux socket */
	void	*pConnList;		/* ptr to ConnCb List */

	void	*acpuMsg;		/* msg from acpu, used by unbind */
	void	*fpMsg;			/* msg from fp, used by accept and connect */

	int	numConnCb;		/* num Connection CB's associated with bindCb */
	int	pad;			/* padding to 8 bytes alignment on 64-bit machine */

        tpl_AcpuConnState acpuState;  /* bind control block state as ACPU sees it */

	/* Add Connection Upcall
	 */
	int	(*ul_addConnInd) (struct _tpl_bindCb *pBind, /* Bind Cb ptr */
		void *pConnId,   /* Conn ID */	
		void **ppAppHdl, /* App Hdl ptr */
		uint32 raddr, 	 /* Remote IP */
		ushort16 rport); /* Remote port */

	/* Delete Connection Upcall
	 */
	int	(*ul_delConnInd) (void *pAppHdl, void *pConnId);

	/* Bind Request acknowleding upcall
	 */
	int	(*ul_bindReqAck) (void *pAppHdl, int rc, void *pBind);

	/* Unbind request ack up call
	 */
	int	(*ul_unbindReqAck) (int rc,
			unsigned int laddr,
			unsigned short lport,
			unsigned short tplProt,
			void *usrInfo,
			void *pBind);

	/* Receive Packet Upcall
	 */
	void	(*ul_rcvPktInd) (void *pAppHdl,
			struct sk_buff *skb,
			void *pHdr,
			uint64 tval);
} tpl_bindCb_t;

/*  Connection flags.
 */
#define TPL_CONN_INUSE		0x80000000 /* Connection structure is in use */
#define TPL_CONN_USE_RPC	0x00000001 /* TCP connection should reassemble RPC packets on rcv */
#define TPL_CONN_RPC_REASM	0x00000002 /* RPC reassembly is currently in progress */
#define TPL_CONN_DEL_PENDING	0x00000004
#define TPL_CONN_NOMAC		0x00000008

#define TPL_CONN_VIF_LA		0x00000010 /* conn has link that is part of vif-la */
#define TPL_CONN_USE_NETBIOS	0x00000020 /* connection should reassemble netbios packets on rcv */
#define TPL_CONN_BAD_APP	0x00000040 /* application open failed */	
#define TPL_CONN_CLOSED	        0x00000080 /* conn is closed on ncpu side */

#define TPL_CONN_LAST_RPC	0x00000100 /* last rpc fragment has been rcved */
#define TPL_CONN_GOOD_APP       0x00000200 /* app. open OKed */
#define TPL_CONN_WAIT_PACKETS   0x00000400 /* connection waiting for pkt allocation */

#define TPL_CONN_SENT_DEL_CONN  0x00000800 /* NCPU flag => ACPU has explicitly closed connection */
#define TPL_CONN_FP_CONN        0x00001000  /* This is an FP TPL connection */

#define TPL_CANTRECVMORE  (TPL_CONN_DEL_PENDING|TPL_CONN_BAD_APP|TPL_CONN_CLOSED|TPL_CONN_SENT_DEL_CONN)

/*------------------------------------
 * TPL Connection Control Block.
 *------------------------------------
 */
#define TPL_FP_READ_MSG     0   /* fpMsg inx, read and close will never be happening at the same time */
#define TPL_FP_WRITE_MSG    1

typedef struct _tpl_connCb {
    int flags;		/* See above TPL_CONN_XXX connection specific definitions */

    LIST_ENTRY(_tpl_connCb) hash_entry;    /* entry in lookup hash table */

    unsigned int     prot;	/* UDP/TCP, in socket */
    unsigned int     laddr;	/* local  IP address, in socket */
    unsigned int     raddr;	/* remote IP address, in socket */
    unsigned short   lport;	/* local  UDP/TCP port, in socket */
    unsigned short   rport;	/* remote UDP/TCP port, in socket */
    unsigned int     vsid;      /* parent virtual server id */

    void *pSock;		/* Linux socket */

    int maxBuflenFirst; /* max application usable buffer length for first packet in RPC response */
    int maxBuflen;  /* max application usable buffer length */

    tpl_bindCb_t	*pBindCb;	/* binding Ctrl Blk ptr */

    void *pAppHdl;	/* application handle */    

    tpl_AcpuConnState acpuState; /* connection state as acpu sees it */
    tpl_rcvPktInd_func rcv_func; /* The receive function copied from pBindCb */

    pkt_queue_t     rpc_queue;  /* RPC reassembly queue and queue for holding TPL-FP packets */

    int	rpcRasmLen; /* total RPC Reassembly Len */
    int	last_aTic;  /*num. of tics since last activity */

    struct _tpl_connCb *pNext;	/* next ConnCb of the same "bind" */
    SIMPLEQ_ENTRY(_tpl_connCb) waitNext; /* Next connection on the wait list */

    struct sk_buff *acpuPktQ;  /* pkt rcvd before conn. up evnt rcved */

    eee_desc_t *fpMsg[2];  /* pending request from FP */
    struct sk_buff *arpTx;     /* su, 05/13/02, arp queue because no mac */

} tpl_connCb_t;

/*-----------------------------------------------------------------
 * Name:        tpl_init
 * Description: Perform Transport Layer initialization.
 *-----------------------------------------------------------------
 */
void tpl_init(void);

/*-----------------------------------------------------------------
 * Name:        tpl_bindReq
 *
 * Description: Binds upper layer to Transport layer. Also binds
 *              TPL to Linux layer
 *
 *              The following table shows the fields required  
 *              in the tpl_bindCb for Server or Client bind request.
 *
 *                          SERVER          CLIENT
 *              ------------------------------------
 *              laddr       Required        Required
 *              lport       Required        Required
 *              raddr       No              Required
 *              rport       No              Required
 *              addConnInd  Required        Required
 *              delConnInd  Required        Required
 *              rcvPktInd   Required        Required
 *              xmtCmpInd   Required        Required
 * 
 *              NOTE: laddr, lport, raddr, rport are expected to
 *                    be in NETWORK ORDER (Big Endian)
 *-----------------------------------------------------------------
 */
int
tpl_bindReq(tpl_bindCb_t *bindReq, void *handle, unsigned int bindOpt);

/*-----------------------------------------------------------------
 * Name:        tpl_unbindReq
 *
 * Description: Unbinds upper layer from Transport layer.
 *              Deletes all connections associated with bindCB.
 *              Also removes TPL from BSD layer for slow-path.
 *
 * Returns:     NFX_OK or TPL error and
 *              Address of CB returned at tpl_bindReq
 *-----------------------------------------------------------------
 */
int
tpl_unbindReq(tpl_bindCb_t *pBind);

/*-----------------------------------------------------------------
 * Name:        tpl_delConnReq
 * Description: Called by Upper layers (e.g. NFS/Mount/CIFS) to 
 *		delete a connection.
 * Returns:     NFX_OK or TPL error
 *-----------------------------------------------------------------
 */
int
tpl_delConnReq(void * pConn);

/*-----------------------------------------------------------------
 * Name:        tpl_delConnAck
 * Description: Called by Upper layers (e.g. NFS/Mount/CIFS) to 
 *		ACK a delete connection request from NCPU.
 * Returns:     NFX_OK or TPL error
 *-----------------------------------------------------------------
 */
int
tpl_delConnAck(void * pConn);

/*-----------------------------------------------------------------
 * Name:  	tpl_freePktByApp
 * Description: Consult application before call tpl_freePkt
 * Returns:     NFX_OK or TPL error
 *-----------------------------------------------------------------
 */
int
tpl_freePktByApp(void *pConnId, struct sk_buff *skb);

/*-----------------------------------------------------------------
 * Name:        tpl_xmtPkt
 * 
 * Description: Transmit function called by upper layer to send packet.
 * Returns:     NFX_OK or TPL error
 *-----------------------------------------------------------------
 */
int
tpl_xmtPkt(void *pConnId, struct sk_buff *skb);

/*-----------------------------------------------------------------
 * Name:  		tpl_xmtCmp
 * Description: Transmit complete function called by lower layer (LUC)
 *		after sending a packet.
 *-----------------------------------------------------------------
 */
void
tpl_xmtCmp(struct sk_buff *skb);

/*-----------------------------------------------------------------
 * Name:        tpl_rcvPkt
 * Description: Receive function called by lower layer when packet received.
 *              pHdr should point beyound UDP/TCP header to RPC header.
 *-----------------------------------------------------------------
 */
void
tpl_rcvPkt(void *pConnId, struct sk_buff *skb);

/*-----------------------------------------------------------------
 * Name:        TPL_ALLOC_PACKET
 * Description: Macro to allocate a packet from the Transport Layer
 *-----------------------------------------------------------------
 */
#define TPL_ALLOC_PACKET(_conn)	tpl_allocPkt(_conn)

/*-----------------------------------------------------------------
 * Name:        TPL_FREE_PACKET
 * Description: Macro to free packet allocated from transport Layer
 *-----------------------------------------------------------------
 */
#define TPL_FREE_PACKET(_pdesc) tpl_freePkt(_pdesc)

/*-----------------------------------------------------------------
 * Name:        tpl_process_receive_queue
 * Description:
 *
 *  This function processes the data in the connection input queue.
 *  The incoming data is split into packet chains according to the RPC boundary
 *  and the resulting packet chains are sent to the application on ACPU.
 *
 *  This function is called by tcp_input() when the connection receives new
 *  data or after the packets required to finish the queue split operation
 *  become available.
 *
 *  If it is required to allocate new packets and there are currently no
 *  packets available, the connection will be put on the allocator wait
 *  list. The allocator will call this function again after the packets will
 *  become available.
 *
 * Arguments:
 *  pConn - TPL connection.
 *
 * Return Value:
 *  None.
 *-----------------------------------------------------------------
 */
void
tpl_process_receive_queue(struct _tpl_connCb *pConn);

/*-----------------------------------------------------------------
 * Name:        tpl_loopback_packet
 * Description: Procedure called to "loop-back" a packet to ourselves
 */
void
tpl_loopback_packet(struct sk_buff *skb);

#endif /* _TPL_API_H */
