/*
 *--------------------------------------------------------------------------
 *
 * Name: tpl-fp-api.h
 *
 * Copyright (C) 2004, ONStor, Inc.
 *
 * Description: Transport Layer Connection API for FP <=> TxRx communication
 *
 * Created by:	Nitin Bahadur
 *
 * Date Created: 07/20/2004
 *
 *--------------------------------------------------------------------------
 */
#ifndef __TPL_FP_API_H
#define __TPL_FP_API_H

#if !defined(SSC)
#include "tpl-fp.h"
#else
typedef void tpl_fp_conn_t;
#endif

#ifndef MIN
#define MIN(a,b) (((a) < (b)) ? (a) : (b))
#endif

#define TPL_FP_PROT_TCP    1
#define TPL_FP_PROT_UDP    2

#define TPL_INADDR_ANY     0

#define TPL_FP_DEFAULT_MAX_DATA	(512*1024)

/* Max data/buffer size = 8K
 * So 360 buffers * 1456 bytes = 512 KB max data queued
 */
#define TPL_FP_DEFAULT_NUM_BUFS	512

/* TPL_FP_DEFAULT_NUM_BUFS is also used as the max number of buffers.
 * An array of buffer pointers and lengths is stored in an ultra buffer so
 * make sure the array will fit.
 */
NFX_ASSERT_STATIC((sizeof(void*) * 2 * TPL_FP_DEFAULT_NUM_BUFS) <= EEE_BUFSIZE_ULTRA);


/* Pass this value to tpl_read and tpl_write to wait forever.
 */
#define TPL_FP_NO_TIMEOUT 0xffffffff

/* Message types
 */
enum tpl_fp_mid {
    TPL_FP_OPEN = TPL_FP_MSG_BASE,
    TPL_FP_CLOSE,
    TPL_FP_LISTEN,
    TPL_FP_ACCEPT,
    TPL_FP_CONNECT,
    TPL_FP_READ,
    TPL_FP_WRITE,
    TPL_FP_UNBIND,
};


/* Read/Write Data types
 */
enum tpl_fp_data_type
{
    TPL_FP_DATA_BUFFER,
    TPL_FP_DATA_EDESC
};


/* Message header
 */
typedef struct 
{
    /* Message id
     */
    enum tpl_fp_mid mid;

    /* Response status
     */
    NTSTATUS status;

#ifndef SSC
    /* Connection structure
     * This should be physical address and not virtual address
     */
    tpl_fp_conn_t *conn;
#else
    /* client private returned in the response 
     */
    uint64 app_handle;
#endif
} tpl_fp_msg_hdr_t;


/* TPL Generic Response 
 * Sent from TxRx to FP when an operation completes
 */
typedef struct tpl_fp_generic_rsp
{
    tpl_fp_msg_hdr_t    hdr;
} tpl_fp_generic_rsp_t;
 
 
/* TPL Open Request
 * Sent from FP to TxRx when application calls tpl_open()
 */
typedef struct tpl_fp_open_req
{
    tpl_fp_msg_hdr_t    hdr;        /* TPL_FP_OPEN */
    uint16              protocol;   /* TCP or UDP */
    uint16              pad;        /* padding */
    vs_id_t             vs_id;      /* virtual server id */
    uint32              local_addr; /* local bind address */
    uint32              local_port; /* local bind port */
} tpl_fp_open_req_t;

/* TPL Open Response
 * Sent from TxRx to FP when a new socket has been opened
 */
typedef struct tpl_fp_open_rsp
{
    tpl_fp_msg_hdr_t    hdr;		    /* TPL_FP_OPEN */
    void                *tpl_bind_cb;   /* tpl_bindCb_t* */
    uint32              local_addr;     /* local bind address */
    uint32              local_port;     /* local bind port */
} tpl_fp_open_rsp_t;


/* TPL Close Request
 * Sent from FP to TxRx when application calls tpl_close()
 */
typedef struct tpl_fp_close_req
{   
    tpl_fp_msg_hdr_t        hdr;        /* TPL_FP_CLOSE */
} tpl_fp_close_req_t;

/* TPL Close Response
 * Sent from TxRx to FP when socket close completes
 */
typedef tpl_fp_generic_rsp_t tpl_fp_close_rsp_t;


/* TPL Listen Request
 * Sent from FP to TxRx when application calls tpl_listen()
 */
typedef struct tpl_fp_listen_req
{
    tpl_fp_msg_hdr_t    hdr;        /* TPL_FP_LISTEN */
    vs_id_t             vs_id;      /* virtual server id */
    uint32              local_addr; /* local listen address */
    uint32              local_port; /* local listen port */
    uint32              pad;        /* padding */
} tpl_fp_listen_req_t;

/* TPL Listen Response
 * Sent from TxRx to FP once listen has been started
 */
typedef struct tpl_fp_listen_rsp
{
    tpl_fp_msg_hdr_t    hdr;            /* TPL_FP_CONNECT */
    uint32              laddr;          /* listen addr */
    uint16              lport;          /* listen port */
    uint16              pad;            /* padding */
    void                *tpl_bind_cb;   /* pointer to tpl_bindCb_t on TxRx */
} tpl_fp_listen_rsp_t;


/* TPL Connect Request
 * Sent from FP to TxRx when application calls tpl_connect()
 */
typedef struct tpl_fp_connect_req
{
    tpl_fp_msg_hdr_t    hdr;		    /* TPL_FP_CONNECT */
    vs_id_t             vs_id;          /* virtual server id */
    uint32              addr;           /* remote address */
    uint32              port;           /* remote port */
    uint32              pad;
} tpl_fp_connect_req_t;

/* TPL Connect Response
 * Sent from TxRx to FP when connect completes (with success or failure)
 */
typedef struct tpl_fp_connect_rsp
{
    tpl_fp_msg_hdr_t    hdr;            /* TPL_FP_CONNECT */
    uint32              laddr;          /* local address of connection */
    uint32              raddr;          /* remote address of connection */
    uint16              lport;          /* local port of connection */
    uint16              rport;          /* remote port of connection */
    uint32              mtu_size;       /* connection mtu size */
    void                *tpl_conn_cb;   /* pointer to tpl_connCb_t on TxRx */
} tpl_fp_connect_rsp_t;


/* TPL Accept Request
 * Sent from FP to TxRx when application calls tpl_accept()
 */
typedef struct tpl_fp_accept_req
{
    tpl_fp_msg_hdr_t    hdr;			/* TPL_FP_ACCEPT */
    void                *accept_conn;	/* tpl_fp_conn_t*  This SHOULD be the physical address and not
                                           FP virtual address */
} tpl_fp_accept_req_t;

/* TPL Accept Response
 * Sent from TxRx to FP when the accept completes
 */
typedef struct tpl_fp_accept_rsp
{
    tpl_fp_msg_hdr_t    hdr;            /* TPL_FP_ACCEPT */
    uint32              laddr;          /* local address of connection */
    uint32              raddr;          /* remote address of connection */
    uint16              lport;          /* local port of connection */
    uint16              rport;          /* remote port of connection */
    uint32              mtu_size;       /* connection mtu size */
    void                *tpl_conn_cb;   /* pointer to tpl_connCb_t on TxRx */
} tpl_fp_accept_rsp_t;


/* TPL Read Request
 * Sent from FP to TxRx when tpl_read() is called.
 * It notifies the TxRx that input space is available on FP
 */
typedef struct tpl_fp_read_req
{
    tpl_fp_msg_hdr_t    hdr;        /* TPL_FP_READ */
} tpl_fp_read_req_t;

/* TPL Read Response
 * Sent from TxRx to FP when data is received from client
 * and copied into the FP input queue
 */
typedef tpl_fp_generic_rsp_t tpl_fp_read_rsp_t;


/* TPL Write Request
 * Sent from FP to TxRx when data is copied into the output queue
 * TxRx will move the data into the socket output queue and
 * start the transmission
 */
typedef struct tpl_fp_write_req
{
    tpl_fp_msg_hdr_t    hdr;        /* TPL_FP_WRITE */
    uint32              write_size; /* minimum write space needed */
    uint32              pad;        /* padding */
} tpl_fp_write_req_t;


/* TPL Write Response
 * Sent from TxRx to FP when data has been moved from output queue into
 * socket queue. It notifies the FP that space is now available and the
 * application can continue writing more data
 */
typedef tpl_fp_generic_rsp_t tpl_fp_write_rsp_t;


/* TPL Unbind IP Address
 * Sent from SSC to FP when an IP address is deleted.
 * FP will generate errors on all listening sockets on that IP.
 */
typedef struct tpl_fp_unbind_req
{
    tpl_fp_msg_hdr_t    hdr;        /* TPL_FP_UNBIND */
    vs_id_t             vs_id;      /* virtual server id */
    uint32              addr;       /* ip address to unbind */
} tpl_fp_unbind_req_t;

/* TPL Unbind Response
 * Sent from FP to SSC to indicate successful receipt of the unbind request.
 */
typedef tpl_fp_generic_rsp_t tpl_fp_unbind_rsp_t;

#if !defined(SSC)

/* Callbacks for various TPL API Calls
 */
typedef void tpl_callback(tpl_fp_conn_t *conn, void *arg, NTSTATUS status);
typedef void tpl_listen_callback(tpl_fp_conn_t **conn, void *arg, NTSTATUS status);
typedef void tpl_accept_callback(tpl_fp_conn_t *conn, tpl_fp_conn_t *conn2, void *arg, NTSTATUS status);

/* TPL API Routines
 */

/*-----------------------------------------------------------------
 * Name:        tpl_listen
 *
 * Created by:  Nitin Bahadur
 *
 * Date Created:07/20/04
 *
 * Description: Open a listening socket on FP. 
 *              If the local_addr is not INADDR_ANY, it must match one of
 *              the virtual server IP addresses.
 *              If the local_addr is INADDR_ANY, the incoming connections will
 *              be accepted on all IP addresses the virtual server uses, 
 *              otherwise the connections will be accepted only on the address
 *              specified.
 *
 * Arguments:
 *  vs_id      - the virtual server this connection belongs to
 *  local_addr - the local address in the network byte order.
 *               if the address is INADDR_ANY, TPL will choose any address
 *  local_port - the local port in the network byte order. If the port
 *               is 0, TPL will chose the port. If the port is not 0,
 *               it can not be in the range reserved for the SSC.
 *  conn       - address where to store the tpl connection pointer if listen
 *               completes successfully
 *  callback   - function to be called on listen completion
 *  arg        - argument to be returned on callback
 *
 * Return Values:
 *  Callback will be invoked when the TPL sends back the result message.
 *  
 *  STATUS_SUCCESS - opened and bound the connection successfully
 *  STATUS_INSUFF_SERVER_RESOURCES - not enough memory
 *  STATUS_INVALID_ADDRESS - the virtual server/local_addr combination is not
 *                            valid.
 *  STATUS_INVALID_PARAMETER - the protocol is not valid
 *  STATUS_ADDRESS_ALREADY_EXISTS - the connection could not be bound to
 *   specified address because it is already in use
 *-----------------------------------------------------------------
 */
void
tpl_listen(vs_id_t vs_id, 
           uint32 local_addr, 
           uint16 local_port,
           tpl_fp_conn_t **conn, 
           tpl_listen_callback *callback, 
           void *arg);

           
/*-----------------------------------------------------------------
 * Name:        tpl_listen_multiple
 *
 * Created by:  Nitin Bahadur
 *
 * Date Created:07/20/04
 *
 * Description: Open a listening sockets on FP. 
 *              If the local_addr is not INADDR_ANY, it must match one of
 *              the virtual server IP addresses.
 *              If the local_addr is INADDR_ANY, the incoming connections will
 *              be accepted on all IP addresses the virtual server uses, 
 *              otherwise the connections will be accepted only on the address
 *              specified.
 *
 * Arguments  :
 *  vs_id      - the virtual server this connection belongs to
 *  num_addrs  - number of addresses to listen on
 *  laddrs     - the local addresses in the network byte order.
 *               if the address is INADDR_ANY, TPL will choose any address
 *  lports     - the local ports in the network byte order. If the port
 *               is 0, TPL will chose the port. If the port is not 0,
 *               it can not be in the range reserved for the SSC.
 *  conn       - list of connections. *conn[i] == NULL if listen failed
 *  callback   - function to be called on listen completion
 *  arg        - argument to be returned on callback
 *
 * Return Values:
 *  See tpl_listen
 *-----------------------------------------------------------------
 */
void 
tpl_listen_multiple(vs_id_t vs_id, 
                    uint32 num_addrs, 
                    uint32 *laddrs, 
                    uint16 *lports,
                    tpl_fp_conn_t **conn, 
                    tpl_listen_callback *callback, 
                    void *arg);


/*-----------------------------------------------------------------
 * Name:        tpl_open
 *
 * Created by : Nitin Bahadur
 *
 * Date Created:07/20/04
 *
 * Description: Open the socket for establishing outgoing TPL connection 
 *              on FP. The socket has to be connected to the remote 
 *              server before the data can be sent.
 *
 * Arguments  :
 *  protocol   - the protocol to use, one of TPL_PROT_TCP and TPL_PROT_UDP.
 *               Only TPL_PROT_TCP will be supported initially.
 *  vs_id      - the virtual server this connection belongs to
 *  local_addr - the local address in the network byte order.  If the address
 *               is INADDR_ANY, the TPL will chose the address.
 *  local_port - the local port in the network byte order. If the port is 0, 
 *               the TPL will chose the local por
 *  input_buf_limit    - the maximum number of buffers that can be used for
 *                       the input queue
 *  input_bytes_limit  - the maximum number of data bytes that can be placed
 *                       in the input queue
 *  output_buf_limit   - the maximum number of buffers that can be used for
 *                       the output queue
 *  output_bytes_limit - the maximum number of data bytes that can be placed
 *                       in the output queue
 *  callback           - function to be called on open completion
 *  arg                - argument to be returned on callback
 *
 * Return Values:
 *   The callback will be invoked when the reply from the TPL is received.
 *
 *   STATUS_SUCCESS - opened and bound the connection successfully
 *   STATUS_INSUFF_SERVER_RESOURCES - not enough memory
 *   STATUS_INVALID_ADDRESS - the virtual server/local_addr combination is not
 *                            valid.
 *   STATUS_ADDRESS_ALREADY_EXISTS - the connection could not be bound to
 *    specified address because it is already in use
 *  
 *-----------------------------------------------------------------
 */
void 
tpl_open(uint16 protocol, 
         vs_id_t vs_id, 
         uint32 local_addr, 
         uint16 local_port,
         uint32 input_buf_limit, 
         uint32 input_bytes_limit,
         uint32 output_buf_limit, 
         uint32 output_bytes_limit,
         tpl_callback *callback,
         void *arg);


/*-----------------------------------------------------------------
 * Name:         tpl_close
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Close the open connection. The callback function will be 
 *               invoked when the reply is received from the TxRx. After the
 *               callback function returns, the connection structure will be
 *               freed. The application should not have any references on
 *               the connection structure when calling tpl_close().
 *
 * Arguments:
 *  conn      - the TPL connection
 *  callback  - function to be called on close completion
 *  arg       - argument to be returned on callback
 *
 * Return Values:
 *   The callback will be invoked when the reply from the TPL is received.
 *
 *   STATUS_SUCCESS - closed connection successfully
 *   STATUS_INVALID_HANDLE - the connection is not valid
 *-----------------------------------------------------------------
 */
void 
tpl_close(tpl_fp_conn_t *conn, 
          tpl_callback *callback, 
          void *arg);


/*-----------------------------------------------------------------
 * Name:         tpl_accept
 *
 * Created by:   Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Accept the new connection. The callback will be invoked when
 *               the new client connection has been established.    
 *               The accept_conn argument to the callback function is the new
 *               connection to use for communication with the client.
 *               Parallel accepts/multiple accepts on same connection are not
 *               supported
 *
 * Arguments:
 *  conn               - the TPL connection that was argument to tpl_listen()
 *  input_buf_limit    - the maximum number of buffers that can be used for
 *                       the input queue
 *  input_bytes_limit  - the maximum number of data bytes that can be placed
 *                       in the input queue
 *  output_buf_limit   - the maximum number of buffers that can be used for
 *                       the output queue
 *  output_bytes_limit - the maximum number of data bytes that can be placed
 *                       in the output queue
 *  callback           - function to be called on accept completion
 *  arg                - argument to be returned on callback
 *
 * Return Values:
 *  The callback will be invoked when the reply from the TPL is received.
 *
 *  STATUS_SUCCESS - the connection accepted successfully
 *  STATUS_CONNECTION_RESET - the client has reset the connection
 *  STATUS_INVALID_HANDLE - the connection is not valid
 *  STATUS_INVALID_PARAMETER - the connection state is invalid for an accept
 *  STATUS_LOCAL_DISCONNECT - the listening socket has been closed
 *-----------------------------------------------------------------
 */
void 
tpl_accept(tpl_fp_conn_t *conn, 
           uint32 input_buf_limit, 
           uint32 input_bytes_limit,
           uint32 output_buf_limit, 
           uint32 output_bytes_limit,
           tpl_accept_callback *callback, 
           void *arg);


/*-----------------------------------------------------------------
 * Name:         tpl_connect
 *
 * Created by:  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Connect to the remote server.
 *
 * Arguments:
 *  conn     - the TPL connection
 *  raddr    - the remote IP address in network byte order
 *  rport    - the remote port in network byte order
 *  callback - function to be called on accept completion
 *  arg      - argument to be returned on callback
 *
 * Return Values:
 *  The callback will be invoked when the reply from the TPL is received.
 *
 *  STATUS_SUCCESS - connected successfully
 *  STATUS_INSUFF_SERVER_RESOURCES - not enough memory
 *  STATUS_CONN_REFUSED - the remote server is not listening
 *  STATUS_NETWORK_NOT_REACHABLE - no route to host
 *  STATUS_TIMEOUT - the connection timed out in the TCP layer
 *  STATUS_INVALID_HANDLE - the connection was already used in call to 
 *                          tcp_listen()
 *  STATUS_INVALID_PARAMETER - the connection state is invalid for a connect
 *-----------------------------------------------------------------
 */
void 
tpl_connect(tpl_fp_conn_t *conn, 
            uint32 raddr, 
            uint16 rport, 
            tpl_callback *callback, 
            void *arg);


/*-----------------------------------------------------------------
 * Name:         tpl_request_read
 *
 * Created by:   Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Send request to TPL read the data from the connection.  
 *               When the data is available from the client, it is placed 
 *               into the connection input buffer. It can be read from the
 *               connection using tpl_read() function.
 *               If data is already available, then we don't do anything
 *               The application can issue both tpl_request_read() and 
 *               tpl_request_write() simultaneously.
 *               If no callback function is specified, then no callback
 *               will be done once there is data available from client.
 *
 * Arguments:
 *  conn       - the TPL connection
 *  callback   - function to be called on read completion
 *  arg        - argument to be returned on callback
 *  
 * Return Values:
 *  The callback will be invoked when the reply from the TPL is received.
 *
 *  STATUS_SUCCESS - the requested data is available
 *  STATUS_ONLY_IF_CONNECTED - the connection is in invalid state. It can be
 *   already closed on FP side, or has not been connected yet.
 *  STATUS_INVALID_HANDLE - the connection is not valid
 *  STATUS_PENDING if the request is pending
 *-----------------------------------------------------------------
 */
NTSTATUS
tpl_request_read(tpl_fp_conn_t *conn, 
                 tpl_callback *callback, 
                 void *arg);


/*-----------------------------------------------------------------
 * Name:         tpl_read
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Read the data from the connection into the provided buffer.
 *               The number of bytes read can be less then number of bytes 
 *               requested. Data copy from TxRx buffers to FP buffers will
 *               make use of the data-mover only if the caller is a thread
 *               (and not a state machine).
 *
 * Arguments:
 *  conn        - the TPL connection
 *  nbytes      - the number of bytes to read
 *  type        - TPL_FP_DATA_BUFFER or TPL_FP_DATA_EDESC
 *  data        - the user buffer (char* or edesc**)
 *  nbytes_read - the number of bytes read is returned here.
 *
 * Return Values:
 *  STATUS_SUCCESS - read some data successfully
 *  STATUS_ALREADY_DISCONNECTED - all available data was read and the 
 *   connection has been closed by the peer
 *  STATUS_ONLY_IF_CONNECTED - the connection is in invalid state. It can be
 *   already closed on FP side, or has not been connected yet.
 *  STATUS_INVALID_HANDLE - the connection is not valid
 *-----------------------------------------------------------------
 */
NTSTATUS 
tpl_read(tpl_fp_conn_t *conn, 
         uint32 nbytes, 
         enum tpl_fp_data_type type,
         void *data,
         uint32 *nbytes_read);

/*-----------------------------------------------------------------
 * Name:         tpl_request_write
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Send request to TPL to notify the application when the space
 *               in the output buffer is available.    
 *               The application can issue both tpl_request_read() and 
 *               tpl_request_write() simultaneously.
 *               If no callback function is specified, then no callback
 *               will be done once there is space available in the output buffer.
 *
 * Arguments:
 *  conn       - the TPL connection
 *  callback   - function to be called on write completion
 *  arg        - argument to be returned on callback
 *  write_size - optional write size...callback will be called only when this
 *               much space is available in the queue
 *
 * Return Values:
 *  The callback will be invoked when the reply from the TPL is received.
 *
 *  STATUS_SUCCESS - the requested space is available in the socket buffer
 *  STATUS_CONNECTION_RESET- the connection was reset by peer
 *  STATUS_ONLY_IF_CONNECTED - the connection is in invalid state
 *  STATUS_INVALID_HANDLE - the connection is not valid
 *  STATUS_PENDING if the request is pending
 *-----------------------------------------------------------------
 */
NTSTATUS
tpl_request_write(tpl_fp_conn_t *conn, 
                  tpl_callback *callback, 
                  void *arg,
                  uint32 write_size);


/*-----------------------------------------------------------------
 * Name:         tpl_write
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Send the data to the client
 *
 * Arguments:
 *  conn            - the TPL connection
 *  nbytes          - the number of bytes to write
 *  type            - TPL_FP_DATA_BUFFER or TPL_FP_DATA_EDESC
 *  data            - the user buffer (char* or edesc*)
 *  nbytes_written  - the number of bytes written is returned here.
 *
 * 
 * Return Values:
 *  STATUS_SUCCESS - the data was written successfully
 *  STATUS_CONNECTION_RESET- the connection was reset by peer
 *  STATUS_ONLY_IF_CONNECTED - the connection is in invalid state
 *  STATUS_INVALID_HANDLE - the connection is not valid
 *-----------------------------------------------------------------
 */
NTSTATUS 
tpl_write(tpl_fp_conn_t *conn, 
          uint32 nbytes, 
          enum tpl_fp_data_type type,
          void *data, 
          uint32 *nbytes_written);


/*-----------------------------------------------------------------
 * Name:         tpl_conn_get_peeraddr
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Get the connection peer address.
 *
 * Arguments:
 *  conn  - the TPL connection
 *  raddr - the remote address in the network byte order is returned here
 *  rport - the remore port in the network byte order is returned here
 *
 * Return Values:
 *  STATUS_SUCCESS - read the values successfully
 *  STATUS_ONLY_IF_CONNECTED - the connection is not in connected state
 *  STATUS_ALREADY_DISCONNECTED - the connection has been closed.
 *-----------------------------------------------------------------
 */
NTSTATUS 
tpl_conn_get_peeraddr(tpl_fp_conn_t *conn, 
                      uint32 *raddr, 
                      uint16 *rport);

/*-----------------------------------------------------------------
 * Name:         tpl_conn_get_localaddr
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Get the connection local address.
 *
 * Arguments:
 *  conn  - the TPL connection
 *  laddr - the local address in the network byte order is returned here
 *  lport - the local port in the network byte order is returned here
 *
 * Return Values:
 *  STATUS_SUCCESS - read the values successfully.
 *  STATUS_ALREADY_DISCONNECTED - the connection has been closed.
 *-----------------------------------------------------------------
 */
NTSTATUS 
tpl_conn_get_localaddr(tpl_fp_conn_t *conn, 
                       uint32 *laddr, 
                       uint16 *lport);

/*-----------------------------------------------------------------
 * Name:         tpl_connect_multiple
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Try to connect to all the servers in the list, returning as
 *               soon as one connection succeeds. The connections will be 
 *               tried one by one in the order they are supplied in the array.
 *
 * Arguments:
 *  
 *  conn      - the TPL connection
 *  num_addrs - the number of remote addresses
 *  raddrs    - the array of IP addresses in the network byte order
 *  rports    - the array of ports in the network byte order
 *  timeout   - the timeout in seconds before giving up on the connection
 *              attempt.
 *  callback  - function to be called on connect completion
 *  arg       - argument to be returned on callback
 *
 * Return Values:
 *  The specified callback routine will be invoked when one of the connection 
 *  attempts succeeds or all connection attempts will fail.
 *   
 *  STATUS_SUCCESS - connected successfully
 *  STATUS_INSUFF_SERVER_RESOURCES - not enough memory
 *  STATUS_TIMEOUT - connection timed out
 *-----------------------------------------------------------------
 */
void 
tpl_connect_multiple(tpl_fp_conn_t *conn, 
                     uint32 num_addrs, 
                     uint32 *raddrs, 
                     uint16 *rports, 
                     uint32 timeout,
                     tpl_callback *callback, 
                     void *arg);


/*-----------------------------------------------------------------
 * Name:         tpl_read_n
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Read the specified number of bytes from the connection.
 *               Invoke the callback when all the requested bytes have been 
 *               read or the connection has been closed by the peer.
 *               Data copy from TxRx buffers to FP buffers will
 *               make use of the data-mover only if the caller is a thread
 *               (and not a state machine).
 *               If no callback function is specified, then the read will
 *               proceed in the context of the calling thread and return back
 *               to the calling thread only when read is done processing.
 *
 * Arguments:
 *  conn      - TPL connection
 *  nbytes    - the number of bytes to read
 *  type      - TPL_FP_DATA_BUFFER or TPL_FP_DATA_EDESC
 *  data      - the user buffer (char* or edesc**)
 *  timeout   - timeout in milliseconds. 
 *              The timeout of 0 means return
 *              immediately if the data is not available.
 *              Timeout of TPL_FP_NO_TIMEOUT means infinite wait
 *  callback  - function to be called on read completion
 *  arg       - argument to be returned on callback
 *
 * Return Values:
 *  The callback function is invoked when the data has been received or
 *  the peer closed the connection.
 * 
 *  STATUS_SUCCESS - read all the requested bytes successfully
 *  STATUS_INSUFF_SERVER_RESOURCES - not enough memory
 *  STATUS_TIMEOUT - the read has timed out
 *  STATUS_ALREADY_DISCONNECTED - the connection has been closed.
 *  STATUS_UNSUCCESSFUL - read failed due to some other reason
 *-----------------------------------------------------------------
 */
NTSTATUS
tpl_read_n(tpl_fp_conn_t *conn, 
           uint32 nbytes, 
           enum tpl_fp_data_type type,
           void *data,
           uint32 timeout,
           tpl_callback *callback, 
           void *arg);


/*-----------------------------------------------------------------
 * Name:         tpl_write_n
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Write the specified number of bytes to the connection.
 *               Invoke the callback after all the requested bytes have been
 *               placed in the socket output buffer. This does not necessarily
 *               mean that the data will eventaully reach the peer.
 *               If no callback function is specified, then the write will
 *               proceed in the context of the calling thread and return back
 *               to the calling thread only when write is done processing.
 *
 * Arguments:
 *  conn      - TPL connection
 *  nbytes    - the number of bytes to write
 *  type      - TPL_FP_DATA_BUFFER or TPL_FP_DATA_EDESC
 *  data      - the user buffer (char* or edesc*)
 *  timeout   - timeout in milliseconds. The timeout of 0 means return
 *              immediately if the data is not available.
 *              Timeout of TPL_FP_NO_TIMEOUT means infinite wait
 *  callback  - function to be called on write completion
 *  arg       - argument to be returned on callback
 *
 * Return Values:
 *  The callback function is invoked when the data has been sent of the peer
 *  closed the connection.
 *   
 *  STATUS_SUCCESS - wrote all the requested bytes successfully
 *  STATUS_INSUFF_SERVER_RESOURCES - not enough memory
 *  STATUS_TIMEOUT - the read has timed out
 *  STATUS_ALREADY_DISCONNECTED - the connection has been closed.
 *  STATUS_UNSUCCESSFUL - write failed due to some other reason
 *-----------------------------------------------------------------
 */
NTSTATUS
tpl_write_n(tpl_fp_conn_t *conn, 
            uint32 nbytes, 
            enum tpl_fp_data_type type,
            void *data,
            uint32 timeout,
            tpl_callback *callback, 
            void *arg);


/*-----------------------------------------------------------------------------
 * Name:         tpl_inet_ntoa
 *
 * Created by:   Nitin Bahadur      
 *
 * Date Created: 08/04/04
 *
 * Description:  Converts an address in network byte order to string format
 *
 * Arguments:
 *	addr	- address in network byte order
 *
 * Returns:
 *	address in the form A.B.C.D
 *-----------------------------------------------------------------------------
 */
char* 
tpl_inet_ntoa(uint32 addr);


/*-----------------------------------------------------------------------------
 * Name:         tpl_inet_addr
 *
 * Created by:   Nitin Bahadur      
 *
 * Date Created: 08/04/04
 *
 * Description:  Converts an ip address in string format to network byte order
 *               uint32 format
 *
 * Arguments:
 *	string      - address in string format
 *
 * Returns:
 *	address in the form of a uint32 in network byte order
 *-----------------------------------------------------------------------------
 */
uint32
tpl_inet_addr(char *string);


/*-----------------------------------------------------------------
 * Name         : tpl_get_num_write_bufs
 *
 * Created by   : Nitin Bahadur
 *
 * Date Created : 08/20/05
 *
 * Description  : Returns number of buffers in a queue available for writing
 *
 * Arguments:
 *  queue       - TPL queue which is to be queried
 *
 * Return Values:
 *  number of buffers available for writing
 *-----------------------------------------------------------------
 */
static inline uint32
tpl_get_num_write_bufs(tpl_fp_queue_t *queue)
{
	uint32 max_bufs_to_write = 0;

	/* if queue is empty, then all buffers are available for writing
	 */
	if (queue->write_idx == queue->read_idx)
		max_bufs_to_write = queue->max_buffers;

	/* if write index is ahead of read index in circular buffer
	 */
	else if (queue->write_idx > queue->read_idx)
		max_bufs_to_write = queue->max_buffers - (queue->write_idx - queue->read_idx);

	/* if read index is ahead of write index in circular buffer
	 */
	else if (queue->write_idx < queue->read_idx)
		max_bufs_to_write = queue->read_idx - queue->write_idx;

#ifdef USE_SAME_BUFFER
#error Depricated/unused code
    /* if we have only filled up 1 buf partially, then we have all
       bufs still available
     */
	else if (queue->write_idx == queue->read_idx &&
             queue->num_bytes_written - queue->num_bytes_read < queue->mtu_size)
	    max_bufs_to_write = queue->max_buffers;

	/* if we have filled up current buf only partially, then we can use
	   this buf also
	 */
	if (queue->cur_write_offset != 0 && queue->write_idx != queue->read_idx) {
		NFX_ASSERT(queue->cur_write_offset < queue->mtu_size);
		max_bufs_to_write++;
	}
#endif

	NFX_ASSERT(max_bufs_to_write > 0);
	
	return max_bufs_to_write - 1;
}

/*-----------------------------------------------------------------
 * Name         : tpl_get_num_write_bufs
 *
 * Created by   : Nitin Bahadur
 *
 * Date Created : 08/20/05
 *
 * Description  : Returns number of buffers in a queue available for writing
 *                This takes into account buffers which are currently being
 *                written to by the data-mover.
 *
 * Arguments:
 *  queue       - TPL queue which is to be queried
 *
 * Return Values:
 *  number of buffers available for writing
 *-----------------------------------------------------------------
 */
static inline uint32
tpl_get_local_write_bufs(tpl_fp_queue_t *queue)
{
	uint32 max_bufs_to_write = 0;

	/* if we have read all bytes, then all buffers are available for writing
	 */
	if (queue->local_write_idx == queue->read_idx)
		max_bufs_to_write = queue->max_buffers;

	/* if write index is ahead of read index in circular buffer
	 */
	else if (queue->local_write_idx > queue->read_idx)
		max_bufs_to_write = queue->max_buffers - (queue->local_write_idx - queue->read_idx);

	/* if read index is ahead of write index in circular buffer
	 */
	else if (queue->local_write_idx < queue->read_idx)
		max_bufs_to_write = queue->read_idx - queue->local_write_idx;

	NFX_ASSERT(max_bufs_to_write > 0);

    return max_bufs_to_write - 1;
}


/*-----------------------------------------------------------------
 * Name:         tpl_snd_datalen
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/21/04
 *
 * Description:  Returns the amount of write data present in the output queue
 *
 * Arguments:
 *  conn      - TPL connection
 *
 * Return Value:
 *  Data length in bytes
 *-----------------------------------------------------------------
 */
#if defined (NFP_FP)
static inline uint32
tpl_snd_datalen(tpl_fp_conn_t *conn)
{
    return (uint32)(conn->out_queue.local_bytes_written - conn->out_queue.num_bytes_read);
}

/*-----------------------------------------------------------------
 * Name:         tpl_rcv_datalen
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Returns the amount of read data present in the input queue
 *
 * Arguments:
 *  conn      - TPL connection
 *
 * Return Value:
 *  Data length in bytes
 *-----------------------------------------------------------------
 */
static inline uint32
tpl_rcv_datalen(tpl_fp_conn_t *conn)
{
    return (uint32)(conn->in_queue.num_bytes_written - conn->in_queue.local_bytes_read);
}


/*-----------------------------------------------------------------
 * Name:         tpl_sndq_space
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 08/18/04
 *
 * Description:  Returns the amount of available space present in the output queue
 *               W A R N I N G - if the caller logic does not fill in each queue 
 *               slot completely, then this space count may be misleading.  
 *               As revealed by Defect 16653, if the caller writes records smaller 
 *               than the queue slot/buffer size (mtu_size), then more queue space will 
 *               actually be required by the caller than intially appears.
 *               The user should use the alternative function: tpl_can_queue_hold_edesc
 *               which contains more checking pertaining to this condition
 *
 * Arguments:
 *  conn      - TPL connection
 *
 * Return Value:
 *  Data length in bytes
 *-----------------------------------------------------------------
 */
static inline uint32
tpl_sndq_space(tpl_fp_conn_t *conn)
{
    tpl_fp_queue_t  queue;

    memcpy(&queue, &conn->out_queue, sizeof(queue));
    return (min(queue.max_bytes - (queue.local_bytes_written - queue.num_bytes_read),
                tpl_get_local_write_bufs(&queue) * queue.mtu_size));
}


/*-----------------------------------------------------------------
 * Name:         tpl_rcvq_space
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 08/18/04
 *
 * Description:  Returns the amount of available space present in the input queue
 *
 * Arguments:
 *  conn      - TPL connection
 *
 * Return Value:
 *  Data length in bytes
 *-----------------------------------------------------------------
 */
static inline uint32
tpl_rcvq_space(tpl_fp_conn_t *conn)
{
    return (uint32)(conn->in_queue.max_bytes - (conn->in_queue.num_bytes_written - conn->in_queue.num_bytes_read));
}

#endif /* NFP_FP */


/*-----------------------------------------------------------------
 * Name:         tpl_get_buflen
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 08/09/04
 *
 * Description:  Returns the amount of valid data in a buffer
 *
 * Arguments:
 *  buffers  - array of buffers
 *  index    - buffer index
 *  max_bufs - maximum buffers in the queue
 *
 * Return Value:
 *  Data length in bytes
 *-----------------------------------------------------------------
 */
static inline uint32
tpl_get_buflen(void **buffers, uint32 index, uint32 max_bufs)
{
    return (index < max_bufs) ? (uint32)((uint64)buffers[max_bufs + index] & 0xFFFFFFFF) : 0;
}


/*-----------------------------------------------------------------
 * Name:         tpl_set_buflen
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 08/09/04
 *
 * Description:  Sets the amount of valid data in a buffer
 *
 * Arguments:
 *  buffers  - array of buffers
 *  index    - buffer index
 *  max_bufs - maximum buffers in the queue
 *  len      - amount of valid data in the buffer
 *
 *-----------------------------------------------------------------
 */
static inline void
tpl_set_buflen(void **buffers, uint32 index, uint32 max_bufs, uint32 len)
{
    if (index < max_bufs)
        buffers[max_bufs + index] = (void *)((uint64)len);
}
#endif


/*-----------------------------------------------------------------
 * Name:         tpl_queue_can_hold_edesc
 *
 * Created by :  Mike Lee
 *
 * Date Created: 12/03/2006
 *
 * Description:  Determine if the data in the buffers of a specified edesc chain can fit into the available slots 
 *               of the tpl ring buffer queue
 *               This function provides stronger checking than using tpl_sndq_space()
 *               Example: 
 *                 Assume caller is trying to write to the tpl queue an edesc chain 
 *                 containing 64 buffers, each of which contained 1K worth of data
 *                 Let's say that the tpl queue has exactly 64K bytes of empty space
 *                 and 63 buffer slots, where the mtu size is 1456 bytes.
 *                 As a result, tpl_sndq_space() would return 64K, which would suggestes to  
 *                 the caller that the amount of space in the queue is large enough.
 *                 However, if the caller subsequently issues tpl_write(), the underlying
 *                 function tpl_write_edesc_data() will exercise a loop that writes
 *                 a 1024 byte record per iteration, it will run out of queue slot
 *                 (since 64 are needed and only 63 are available).  This condition is 
                   documented in defect 16653, and fixed by introducing this new function.
 *               Specifically, in handling an edesc chain, tpl_write (or more accurately, the underlying
 *               tpl_write_edesc_data() function) is not doing packed writing.  So, if the edesc buffer 
 *               is smaller than the size of the tpl queue slot, then N buffers in the edesc chain 
 *               should technically fit in <N slots in the tpl queue, but without packed writing, they 
 *               will actually take up >= N slots instead.
 *
 * Arguments:
 *  conn_p      - TPL connection
 *  edesc_p     - pointer to the edesc chain whose data size is to be checked against the tpl queue
 *
 * Return Value:
 *  TRUE - if the tpl queue can accomodate the data in the specified edesc chain
 *  FALSE - otherwise
 *-----------------------------------------------------------------
 */
#ifdef LATER

inline unsigned int
tpl_get_local_write_bufs(tpl_fp_queue_t *)
{
  return 1024;
}
#define EEE_GET_NUM_BUFS(eee_desc_t *it_p) itp_p->foobar;

static inline boolean
tpl_can_queue_hold_edesc(tpl_fp_conn_t* conn_p, eee_desc_t* edesc_p)
{
       tpl_fp_queue_t  queue;
       memcpy(&queue, &conn_p->out_queue, sizeof(queue));

	uint32 qsize = min(queue.max_bytes - (queue.local_bytes_written - queue.num_bytes_read),
			    tpl_get_local_write_bufs(&queue) * queue.mtu_size);

	//loop to see how many buffers are in the edesc_chain
	uint32 total_bufs = 0; 
	eee_desc_t* it_p = edesc_p; 
	while (it_p != NULL) {
		total_bufs += EEE_GET_NUM_BUFS(it_p);
		it_p = it_p->hdr.next;
	}

	//the data can fit in the queue if 
	// (1) the total of size of the queue exceeds the packet size
	// (2) there are enought empty slots in the queue for every buffer in the edesc chain
	boolean rc = ((qsize >= EEE_GET_PKT_LEN(edesc_p)) && (tpl_get_local_write_bufs(&queue) >= total_bufs));
	return rc;
}
#else
boolean tpl_can_queue_hold_edesc(tpl_fp_conn_t* conn_p, eee_desc_t* edesc_p);
#endif /* LATER */

#endif /* __TPL_FP_API_H */
