/*
 *--------------------------------------------------------------------------
 * Name: tpl-msg.c
 * Description: Transport Packet Layer(TPL) Message Handlers
 *              between TxRx <=> FP
 *--------------------------------------------------------------------------
 */
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>

#include <linux/gfp.h>
#include <linux/slab.h>

#include "nfx-types.h"
#include "nfx-defs.h"
#include "nfx-error.h"

#include "eee-desc.h"
#include "tpl-api.h"
#include "tpl.h"
#include "tpl-ipc.h"

#include "pkt-api.h"
#include "pkt-queue-struct.h"
#include "pkt-queue-api.h"

#ifdef LATER
#include "../sm-cifs/win-compat.h"
#endif /* LATER */

#include "tpl-fp-api.h"

extern tpl_bindCb_t *tpl_findBindCbByUsrInfo(void *connectId);
extern int tpl_connectReq(tpl_bindCb_t *bindRsp, unsigned int addr, unsigned short port);
extern int linux_socket_close(struct sock *sk);

#define E_LOG(a,b,c,d,e,f,g)

#define STATUS_INVALID_HANDLE         0x1
#define STATUS_ADDRESS_ALREADY_EXISTS 0x2
#define STATUS_INVALID_ADDRESS        0x3
#define STATUS_ALREADY_DISCONNECTED   0x4
#define STATUS_ONLY_IF_CONNECTED      0x5
#define STATUS_CONNECTION_REFUSED     0x6
#define STATUS_TIMEOUT                0x7
#define STATUS_NETWORK_UNREACHABLE    0x8
#define STATUS_UNSUCCESSFUL           0x9

/* on txrx and cougar only */
#define HW_HT_REMOTE_BASE_ADDRESS 0x5000000000UL
#define HW_HT_LOCAL_BASE_ADDRESS  0x4000000000UL

#define nfxMySlotId 1
#define nfxMyCPUId  0
#define EEE_USER_DATA 0x2
#define NFX_NFP_FP_ANY  0x4

extern tpl_cfg_t *pTplCfg;

extern int eee_sendMessage(void *, unsigned int len, unsigned short src_port, unsigned short dst_port);
extern void eee_freePacket(eee_desc_t *desc_p);
extern int eee_getApplicationId(char *name);

extern int tpl_listenReq(tpl_bindCb_t *request, unsigned int address, unsigned short port);

/*
 * Function Prototypes
 */
void tpl_procFpOpenReq(tpl_fp_open_req_t *req, eee_desc_t *edesc);
void tpl_procFpCloseReq(tpl_fp_close_req_t *req, eee_desc_t *edesc);

void tpl_procFpListenReq(tpl_fp_listen_req_t *req, eee_desc_t *edesc);
void tpl_procFpAcceptReq(tpl_fp_accept_req_t *req, eee_desc_t *edesc);

void tpl_procFpConnectReq(tpl_fp_connect_req_t *req, eee_desc_t *edesc);

void tpl_procFpReadReq(tpl_fp_read_req_t *req, eee_desc_t *edesc);
void tpl_procFpWriteReq(tpl_fp_write_req_t *req, eee_desc_t *edesc);

void tpl_fpTxPkt(tpl_connCb_t *pConn, tpl_fp_conn_t *conn, tpl_fp_queue_t *queue);

struct sk_buff *tpl_fpAllocPktDesc(tpl_connCb_t *pConn, char *buf, uint32 size);

/* Global Function Prototypes */
int32 tpl_addFpConnInd(tpl_bindCb_t *pBind, void *pConnId, void **ppAppHdl, uint32 raddr, ushort16 rport);
int32 tpl_delFpConnInd(void *pAppHdl, void *pConnId);
void tpl_rcvFpPktInd(void *pAppHdl, struct sk_buff *pDesc, void *pHdr, uint64 tval);
void tpl_fpContinueWrite(void *sock, void *pConnId, int unused);

/*-----------------------------------------------------------------
 * Name:         tpl_receiveMsg
 * Description:  Processes messages received from FP
 * Arguments:
 *  edesc - incoming message
 *-----------------------------------------------------------------
 */
void 
tpl_receiveMsg(eee_desc_t *edesc)
{
	eee_descBuf_t       *bd;
	tpl_fp_msg_hdr_t    *hdr;
    
	bd = edesc_firstBuf(edesc);
	hdr = (tpl_fp_msg_hdr_t *)bd->buf;
    
	switch(hdr->mid) {
		case TPL_FP_OPEN:
			tpl_procFpOpenReq((tpl_fp_open_req_t *)hdr, edesc);
			break;

		case TPL_FP_CLOSE:
			tpl_procFpCloseReq((tpl_fp_close_req_t *)hdr, edesc);
			break;

		case TPL_FP_LISTEN:
			tpl_procFpListenReq((tpl_fp_listen_req_t *)hdr, edesc);
			break;

		case TPL_FP_ACCEPT:
			tpl_procFpAcceptReq((tpl_fp_accept_req_t *)hdr, edesc);
			return;	/* return here because we don't want to free the pkt */

		case TPL_FP_CONNECT:
			tpl_procFpConnectReq((tpl_fp_connect_req_t *)hdr, edesc);
			return;

		case TPL_FP_READ:
			tpl_procFpReadReq((tpl_fp_read_req_t *)hdr, edesc);
			return;

		case TPL_FP_WRITE:
			tpl_procFpWriteReq((tpl_fp_write_req_t *)hdr, edesc);
			return;

		default:
			NFX_ASSERT(0);
            E_LOG(class_1, LOG_ERR, pTplCfg->tplAppId, 0, 0, 0,
					("%s: Invalid msg of type %d recd.\n", __FUNCTION__, hdr->mid));
			break;
	}
    eee_freePacket(edesc);
}

/*-----------------------------------------------------------------
 * Name:         tpl_procFpOpenReq
 * Description:  Processes socket open request from FP
 * Arguments:
 *  req - open request
 *-----------------------------------------------------------------
 */
void
tpl_procFpOpenReq(tpl_fp_open_req_t *req, eee_desc_t *edesc)
{
        tpl_bindCb_t bindReq, *bindRsp;
	tpl_fp_open_rsp_t rsp;
	int32 rc;

	memset(&bindReq, 0, sizeof(bindReq));
	
	/* create bind request */
	bindReq.laddr = req->local_addr;
	bindReq.lport = req->local_port;
	bindReq.tplProt = TPL_PROT_TCP;
	bindReq.vsId = req->vs_id;
	bindReq.bindFlags = TPL_FLAG_FP_CONN | TPL_FLAG_BIND_ONLY;
	bindReq.ul_addConnInd = tpl_addFpConnInd;
	bindReq.ul_delConnInd = tpl_delFpConnInd;
	bindReq.ul_rcvPktInd = tpl_rcvFpPktInd;

	/* create the socket and do the bind */
        bindRsp = NULL;

	rc = tpl_bindReq(&bindReq, &bindRsp, (TPL_BIND_CLIENT | TPL_BIND_ONLY));
	
	/* construct open response */
	memset(&rsp, 0, sizeof(rsp));

	rsp.hdr.mid = req->hdr.mid;
	rsp.hdr.conn = req->hdr.conn;

	switch (rc) {
		case NFX_OK:
			rsp.hdr.status = STATUS_SUCCESS;
			rsp.tpl_bind_cb = bindRsp;
			rsp.local_addr = bindRsp->laddr;
			rsp.local_port = bindRsp->lport;

			/* save the FP side tpl connection structure */
			bindRsp->usrInfo = req->hdr.conn;
			break;
			
		case TPL_ERR_DUP_BIND:
			rsp.hdr.status = STATUS_ADDRESS_ALREADY_EXISTS;
			break;

		case TPL_ERR_NO_BIND_RSRC:
		case TPL_ERR_VSTACK_ERR:
		default:
			rsp.hdr.status = STATUS_INVALID_ADDRESS;
			break;
	}
	
	/* send back response */
	rc = eee_sendMessage(&rsp, sizeof(rsp), edesc->hdr.src_port, edesc->hdr.dest_port);
	if (rc != NFX_OK) {
		TPL_LOGERR( TPL_ERR_XMT_FAIL, &rsp); 
	}
	return;
}

/*-----------------------------------------------------------------
 * Name:         tpl_procFpCloseReq
 * Description:  Processes socket close request from FP
 * Arguments:
 *  req - close request
 *-----------------------------------------------------------------
 */
void
tpl_procFpCloseReq(tpl_fp_close_req_t *req, eee_desc_t *edesc)
{
	tpl_fp_close_rsp_t rsp;
	int32 rc = 0;
	tpl_bindCb_t *pBind = NULL;
	tpl_connCb_t *pConn = NULL;
	boolean	serverSocket = FALSE;
	boolean asyncClose = FALSE;
	
	rsp.hdr.mid = req->hdr.mid;
	rsp.hdr.conn = req->hdr.conn;

	E_LOG(class_1, debug_s, NULL, 0, 0, 0,
          ("%s: close on conn %p\n", __FUNCTION__, req->hdr.conn));

	/*
	 * If the socket is connected, then first delete the
	 * connection and then unbind the socket (fall through)
	 */
	pConn = tpl_findConnByAppHdl(req->hdr.conn);

	if (pConn) {
		tpl_flushRpcRasm(pConn);

		/* unset any existing upcall for snd queue acks */
		if (pConn->pSock)
	            tpl_setSockUpcall(pConn, NULL);
    
		pBind = pConn->pBindCb;

		/* delete any pending requests from FP (read/write) on
		   this this connection
		 */
		if (pConn->fpMsg[TPL_FP_READ_MSG]) {
			eee_freePacket((eee_desc_t *)pConn->fpMsg[TPL_FP_READ_MSG]);
			pConn->fpMsg[TPL_FP_READ_MSG] = NULL;
		}
		if (pConn->fpMsg[TPL_FP_WRITE_MSG]) {
			eee_freePacket((eee_desc_t *)pConn->fpMsg[TPL_FP_WRITE_MSG]);
			pConn->fpMsg[TPL_FP_WRITE_MSG] = NULL;
		}

		/* if peer has already closed the connection, then
		   simply get rid of it
		 */
		if (pConn->flags & TPL_CONN_CLOSED) {
			      if (pConn->pSock) {
				pConn->pSock = NULL;
			      }

			      E_LOG(class_1, debug_s, NULL, 0, 0, 0,
                          ("%s: close socket conn %p\n", 
                           __FUNCTION__, req->hdr.conn));

			      (void)linux_socket_close(pConn->pSock);

			      E_LOG(class_1, debug_s, NULL, 0, 0, 0,
                          ("%s:%u  tpl_delConnCore on conn %p\n", 
                           __FUNCTION__, __LINE__, req->hdr.conn));

			      rc = tpl_delConn(pConn);

			      E_LOG(class_1, debug_s, NULL, 0, 0, 0,
                          ("%s:%u tpl_delConnCore on conn %p, return %d\n", 
                           __FUNCTION__, __LINE__, req->hdr.conn, rc));
		} else {
			/* close the connection asynchronously...wait until peer closes
			   the connection or the close times out
			 */
                E_LOG(class_1, debug_s, NULL, 0, 0, 0,
                      ("%s:%u  tpl_delConnCore on conn %p\n", 
                       __FUNCTION__, __LINE__, req->hdr.conn));
				rc = tpl_delConnReq(pConn);
                E_LOG(class_1, debug_s, NULL, 0, 0, 0,
                      ("%s:%u tpl_delConnCore on conn %p, return %d\n", 
                       __FUNCTION__, __LINE__, req->hdr.conn, rc));

		          if (rc == NFX_OK) {
					asyncClose = TRUE;

					/* unset the user handles so that we can't find them via any
					   more FP messages.
					 */
					pConn->pAppHdl = NULL;
					if (pBind->bindFlags & TPL_BIND_CLIENT)
						pBind->usrInfo = NULL;
			  }
			}
	}

	/* Unbind a open socket and close it */
	if (rc == 0) {
		if (pBind == NULL) {
			pBind = tpl_findBindCbByUsrInfo(req->hdr.conn);
			if (pBind != NULL) {
				/* this is a server socket we are closing
				   disassociate the socket from any user handle
				 */
				pBind->usrInfo = NULL;
				serverSocket = TRUE;
			}
		}
			
		if (pBind != NULL) {
				
            E_LOG(class_1, debug_s, NULL, 0, 0, 0,
                  ("%s:%u conn %p, asyncClose=%u, bindFlags=0x%x, serverSocket=%u\n", 
                   __FUNCTION__, __LINE__, req->hdr.conn, asyncClose,
                   pBind->bindFlags, serverSocket));
			/* if this is a client socket not undergoing asynchronous close..i.e. it has been closed,
			   then unbind the socket
			   if this is a listen(server) socket and it has been closed and has no open pConns, 
			   then unbind the socket
			 */
			if (!asyncClose && ((pBind->bindFlags & (TPL_BIND_CLIENT|TPL_BIND_CLOSE)) || serverSocket))
            {
			    /* delete any pending requests from FP (accept/read/write) on
    			   this this connection
	    		 */
		    	if (pBind->fpMsg) {
			        eee_freePacket((eee_desc_t *)pBind->fpMsg);
    				pBind->fpMsg = NULL;
	    		}

			rc = tpl_unbindReq(pBind);

                E_LOG(class_1, debug_s, NULL, 0, 0, 0,
                      ("%s:%u tpl_unbindReqCore on conn %p, return %d\n", 
                       __FUNCTION__, __LINE__, req->hdr.conn, rc));
                           if (rc == NFX_OK && pBind->numConnCb == 0)
			     tpl_freeBindCb(pBind);
                      }
		}
		else
			rc = TPL_ERR_BAD_HDL;
	}

	switch (rc) {
		case NFX_OK:
			rsp.hdr.status = STATUS_SUCCESS;
			break;
		default:
			rsp.hdr.status = STATUS_INVALID_HANDLE;
			break;
	}

	/* send back response */
	E_LOG(class_1, debug_s, NULL, 0, 0, 0,
          ("%s:%u send response for conn %p\n", 
           __FUNCTION__, __LINE__, req->hdr.conn));

	rc = eee_sendMessage(&rsp, sizeof(rsp), edesc->hdr.src_port, edesc->hdr.dest_port);
	if (rc != NFX_OK) {
		TPL_LOGERR( TPL_ERR_XMT_FAIL, &rsp); 
	}
}

/*-----------------------------------------------------------------
 * Name:         tpl_procFpListenReq
 * Description:  Processes socket listen request from FP
 * Arguments:
 *  req - listen request
 *-----------------------------------------------------------------
 */
void
tpl_procFpListenReq(tpl_fp_listen_req_t *req, eee_desc_t *edesc)
{
        tpl_bindCb_t bindReq, *bindRsp;
	tpl_fp_listen_rsp_t rsp;
	int32 rc;

	/* check if a socket has already been created for this tpl req */
	bindRsp = tpl_findBindCbByUsrInfo(req->hdr.conn);

	if (bindRsp == NULL) { /* create bind request, high frequency case */

		memset(&bindReq, 0, sizeof(bindReq));
		bindReq.laddr = req->local_addr;
		bindReq.lport = req->local_port;
		bindReq.tplProt = TPL_PROT_TCP;
		bindReq.vsId = req->vs_id;
		bindReq.bindFlags = TPL_FLAG_FP_CONN;
		bindReq.ul_addConnInd = tpl_addFpConnInd;
		bindReq.ul_delConnInd = tpl_delFpConnInd;
		bindReq.ul_rcvPktInd = tpl_rcvFpPktInd;

		/* create a listening socket and do the bind and listen */
		rc = tpl_bindReq(&bindReq, &bindRsp, TPL_BIND_SERVER);

	} else { /* already exists, so just do the listen */
		rc = tpl_listenReq(bindRsp, req->local_addr, req->local_port);
	}

	/* construct open response */
	rsp.hdr.mid = req->hdr.mid;
	rsp.hdr.conn = req->hdr.conn;

	switch (rc) {
		case NFX_OK:
			rsp.hdr.status = STATUS_SUCCESS;
			rsp.laddr = bindRsp->laddr;
			rsp.lport = bindRsp->lport;
			rsp.pad = 0;
			rsp.tpl_bind_cb = bindRsp;

			/* save the FP side tpl connection structure */
			bindRsp->usrInfo = req->hdr.conn;
			bindRsp->bindFlags &= ~TPL_FLAG_BIND_ONLY;
			break;
			
		case TPL_ERR_DUP_BIND:
		case TPL_ERR_NO_BIND_RSRC:
			rsp.hdr.status = STATUS_ADDRESS_ALREADY_EXISTS;
			break;
		default:
			rsp.hdr.status = STATUS_INVALID_ADDRESS;
			break;
	}
	
	/* send back response */
	rc = eee_sendMessage(&rsp, sizeof(rsp), edesc->hdr.src_port, edesc->hdr.dest_port);
	if (rc != NFX_OK) {
		TPL_LOGERR( TPL_ERR_XMT_FAIL, &rsp); 
	}
	return;
}

/*-----------------------------------------------------------------
 * Name:         tpl_procFpAcceptReq
 * Description:  Processes socket accept request from FP
 * Arguments:
 *  req - accept request
 *-----------------------------------------------------------------
 */
void
tpl_procFpAcceptReq(tpl_fp_accept_req_t *req, eee_desc_t *edesc)
{
	tpl_bindCb_t *pBind;
	tpl_fp_conn_t *conn;
	tpl_connCb_t *pConn;
	tpl_fp_accept_rsp_t rsp;
	int i, rc;
	
	pBind = tpl_findBindCbByUsrInfo(req->hdr.conn);
	if (pBind == NULL) {
		rsp.hdr.status = STATUS_INVALID_HANDLE;
	} else {
		pConn = (tpl_connCb_t *)pBind->pConnList;
startloop:
		/* look through the socket searching for any connected clients */
		for (i = 0; pConn && i < pBind->numConnCb; i++, pConn = pConn->pNext) {
			/* if an unassigned connection is found */
			if (pConn->pAppHdl == NULL) {
				/* check if there is a delete pending on this connection.
				   If so, just get rid of it
				 */
			    if (pConn->flags & TPL_CONN_DEL_PENDING) {
					tpl_delConnInd(pConn);
					pConn = (tpl_connCb_t *)pBind->pConnList;
					goto startloop;
				}
				break;
			}
		}

		/* if no connection was found, then save request and return */
		if (pConn == NULL) {
			pBind->fpMsg = edesc;
			return;
		}

		/* save the connection */
		conn = req->accept_conn;
		pConn->pAppHdl = req->accept_conn;
		
		/* Convert to TxRx address */
		conn = (void *)(PHYS_TO_XKPHYS(K_CALG_UNCACHED,
				(address_t)req->accept_conn) | HW_HT_REMOTE_BASE_ADDRESS);	/* HT access */
		conn->tpl_txrx_cb = pConn;
		
		/* we have found a connection! */
		/* construct accept response */
		rsp.hdr.status = STATUS_SUCCESS;
		rsp.laddr = pConn->laddr;
		rsp.raddr = pConn->raddr;
		rsp.lport = pConn->lport;
		rsp.rport = pConn->rport;
		rsp.mtu_size = 0; // pConn->mtuSize;
		rsp.tpl_conn_cb = pConn;

		/* set tpl upcall to know where is more space in the socket buffer */
		tpl_setSockUpcall(pConn, tpl_fpContinueWrite);
	}

	rsp.hdr.mid = req->hdr.mid;
	rsp.hdr.conn = req->hdr.conn;

	/* send back response */
	rc = eee_sendMessage(&rsp, sizeof(rsp), edesc->hdr.src_port, edesc->hdr.dest_port);

	if (rc != NFX_OK) {
		TPL_LOGERR( TPL_ERR_XMT_FAIL, &rsp); 
	}
	/* Need to free here since it won't be freed in the calling function */
	eee_freePacket(edesc);
	return;
}

/*-----------------------------------------------------------------
 * Name:         tpl_procFpConnectReq
 * Description:  Processes socket connect request from FP
 * Arguments:
 *  req - connect request
 *-----------------------------------------------------------------
 */
void
tpl_procFpConnectReq(tpl_fp_connect_req_t *req, eee_desc_t *edesc)
{
        tpl_bindCb_t bindReq, *bindRsp;
	tpl_fp_connect_rsp_t rsp;
	int32 rc;

	/* check if a socket has already been created for this tpl req */
	bindRsp = tpl_findBindCbByUsrInfo(req->hdr.conn);

	if (bindRsp == NULL) {
		/* create bind request */
		memset(&bindReq, 0, sizeof(bindReq));
		bindReq.laddr = 0;
		bindReq.lport = 0;
		bindReq.raddr = req->addr;
		bindReq.rport = req->port;
		bindReq.tplProt = TPL_PROT_TCP;
		bindReq.vsId = req->vs_id;
		bindReq.bindFlags = TPL_FLAG_FP_CONN;
		bindReq.ul_addConnInd = tpl_addFpConnInd;
		bindReq.ul_delConnInd = tpl_delFpConnInd;
		bindReq.ul_rcvPktInd = tpl_rcvFpPktInd;

		/* create a listening socket and do the bind and connect */
		rc = tpl_bindReq(&bindReq, &bindRsp, TPL_BIND_CLIENT);
	} else {
		/* if we had bound this socket, then it means we must
		   have issued an open for it with just bind set 
		 */
		NFX_ASSERT(bindRsp->bindFlags & TPL_FLAG_BIND_ONLY);
		
		/* just do the connect */
		rc = tpl_connectReq(bindRsp, req->addr, req->port);
	}

	switch (rc) {
		case NFX_OK:
			/* connect is going through now....we will get
			   a callback once it completes
			   save the FP side tpl connection structure 
			 */
			bindRsp->usrInfo = req->hdr.conn;
			bindRsp->bindFlags &= ~TPL_FLAG_BIND_ONLY;
			bindRsp->fpMsg = edesc;
			return;
			
		case TPL_ERR_DUP_BIND:
			rsp.hdr.status = STATUS_ADDRESS_ALREADY_EXISTS;
			break;

		case TPL_ERR_NO_BIND_RSRC:
		default:
			rsp.hdr.status = STATUS_INVALID_ADDRESS;
			break;
	}

	/* construct connect response */
	rsp.hdr.mid = req->hdr.mid;
	rsp.hdr.conn = req->hdr.conn;

	/* send back response */
	rc =  eee_sendMessage(&rsp, sizeof(rsp), edesc->hdr.src_port, edesc->hdr.dest_port);
	if (rc != NFX_OK) {
		TPL_LOGERR( TPL_ERR_XMT_FAIL, &rsp); 
	}
}

/*-----------------------------------------------------------------
 * Name:         tpl_procFpReadReq
 * Description:  Processes socket read request from FP
 * Arguments:
 *  req - read request
 *-----------------------------------------------------------------
 */
void
tpl_procFpReadReq(tpl_fp_read_req_t *req, eee_desc_t *edesc)
{
	tpl_connCb_t		*pConn;
	tpl_fp_conn_t		*conn;
	tpl_fp_read_rsp_t	rsp;
	uint64				num_bytes_read;
	uint64				num_bytes_written;
	int32 rc;
	
	/* find connection structure */
	pConn = tpl_findConnByAppHdl(req->hdr.conn);
	if (pConn == NULL) {
		rsp.hdr.status = STATUS_INVALID_HANDLE;
	} else {
		/* if there is still unread data in the TPL FP input queue
		 * then send the read response immediately
		 * else 
		 *    if connection has been closed, then return error rsp
		 *    else wait for data to arrive on the TCP connection
		 */
		conn = pConn->pAppHdl;

		/* Convert to TxRx address */
		conn = (tpl_fp_conn_t *)(PHYS_TO_XKPHYS(K_CALG_UNCACHED,
				(address_t)conn) | HW_HT_REMOTE_BASE_ADDRESS);	/* HT access */

		/* start flushing data from the socket buffers to the input queue */
		tpl_rcvFpPktInd(pConn->pAppHdl, NULL, NULL, 0);
		
		num_bytes_read = conn->in_queue.num_bytes_read;
		num_bytes_written = conn->in_queue.num_bytes_written;
		
		if (num_bytes_written - num_bytes_read > 0) {
			rsp.hdr.status = STATUS_SUCCESS;
		} else if (pConn->flags & TPL_CONN_CLOSED) {
			rsp.hdr.status = STATUS_ALREADY_DISCONNECTED;
		} else {
			/* If there is a prev. read request pending, then just
			 * delete that. It might be that that req. timed out on the FP
			 */
			if (pConn->fpMsg[TPL_FP_READ_MSG] != NULL) {
				eee_freePacket(pConn->fpMsg[TPL_FP_READ_MSG]);
			}
			pConn->fpMsg[TPL_FP_READ_MSG] = edesc;
			return;
		}
	}
	rsp.hdr.mid = req->hdr.mid;
	rsp.hdr.conn = req->hdr.conn;
	
	/* send back response */
	rc = eee_sendMessage(&rsp, sizeof(rsp), edesc->hdr.src_port, edesc->hdr.dest_port);
	if (rc != NFX_OK) {
		TPL_LOGERR( TPL_ERR_XMT_FAIL, &rsp); 
	}
	eee_freePacket(edesc);
}

/*-----------------------------------------------------------------
 * Name:         tpl_procFpWriteReq
 * Description:  Processes socket write request from FP
 * Arguments:
 *  req - write request
 *-----------------------------------------------------------------
 */
void
tpl_procFpWriteReq(tpl_fp_write_req_t *req, eee_desc_t *edesc)
{
	tpl_connCb_t		*pConn;
	tpl_fp_conn_t		*conn;
	tpl_fp_write_rsp_t	rsp;
	
	/* find connection structure */
	pConn = tpl_findConnByAppHdl(req->hdr.conn);
	if (pConn == NULL) {
		rsp.hdr.status = STATUS_INVALID_HANDLE;
	} else if(pConn->flags & TPL_CONN_CLOSED) {
		rsp.hdr.status = STATUS_ONLY_IF_CONNECTED;
	} else {
		tpl_fp_queue_t  q;
		
		/* if there is space in the TPL FP output queue
		 * then send the write response immediately
		 * else wait for data to go out on the TCP connection
		 */
		conn = pConn->pAppHdl;

		/* Convert to TxRx address */
		conn = (tpl_fp_conn_t *)(PHYS_TO_XKPHYS(K_CALG_UNCACHED,
				(address_t)conn) | HW_HT_REMOTE_BASE_ADDRESS);	/* HT access */
		
		/* start flushing the output queue */
		tpl_fpTxPkt(pConn, conn, &conn->out_queue);

		memcpy(&q, &conn->out_queue, sizeof(q));

		/* if there is space available for FP to write, then return a favorable response */
		if ((q.max_bytes - 
		     (q.num_bytes_written - q.num_bytes_read) >= req->write_size) && 
		        tpl_get_num_write_bufs(&q) > 0) {

			rsp.hdr.status = STATUS_SUCCESS;
		} else {
			/* If there is a prev. write request pending, then just
			 * delete that. It might be that that req. timed out on the FP
			 */
			if (pConn->fpMsg[TPL_FP_WRITE_MSG] != NULL) {
			       eee_freePacket((eee_desc_t *)pConn->fpMsg[TPL_FP_WRITE_MSG]);
			}
			pConn->fpMsg[TPL_FP_WRITE_MSG] = edesc;
			return;
		}
	}
	rsp.hdr.mid = req->hdr.mid;
	rsp.hdr.conn = req->hdr.conn;
	
	/* send back response */
	if (eee_sendMessage(&rsp, sizeof(rsp), edesc->hdr.src_port, 
						edesc->hdr.dest_port) != NFX_OK) {
		TPL_LOGERR( TPL_ERR_XMT_FAIL, &rsp); 
	}
	eee_freePacket(edesc);
}

/*-----------------------------------------------------------------
 * Name:         tpl_addFpConnInd
 * Description:  This function is called when someone connects to a listening socket. 
 *	If an accept() is pending, we return a response
 *	Else if a connect() is pending, we return a response
 *	Else we do nothing. Later on when an accept req. comes in
 *	 we shall usr this connection for the accept response
 *
 * Arguments:
 *  pBind    - bind control block pointer
 *  pConnId  - connection pointer
 *  ppAppHdl - application pointer is saved in here
 *  raddr    - remote address of the connection
 *  rport    - remote port of the connection
 *
 * Return Values:
 *	NFX_OK if we were able to send accept() response to FP
 *	NFX_ERR otherwise
 *-----------------------------------------------------------------
 */
int32
tpl_addFpConnInd(tpl_bindCb_t *pBind, void *pConnId, void **ppAppHdl, uint32 raddr, ushort16 rport)
{
	tpl_connCb_t *pConn = (tpl_connCb_t *)pConnId;
	int rc = NFX_OK;
	eee_desc_t *edesc;
	eee_descBuf_t *bd;
	tpl_fp_msg_hdr_t *hdr;
	tpl_fp_conn_t *conn;
	int32 rc2;

	pConn->flags |= TPL_CONN_FP_CONN;

	/* send the accept response if an accept request was issued */
	if (pBind->fpMsg) {
		edesc = (eee_desc_t *)pBind->fpMsg;
		bd = edesc_firstBuf(edesc);
		hdr = (tpl_fp_msg_hdr_t *)bd->buf;
		
		switch (hdr->mid) {
			case TPL_FP_ACCEPT:
			{
				tpl_fp_accept_req_t	*accept_req = (tpl_fp_accept_req_t *)hdr;
				tpl_fp_accept_rsp_t	accept_rsp;
				
				/*
				 * Save fp tpl connection structure and
				 * convert to TxRx address
				 */
				*ppAppHdl = accept_req->accept_conn;

				/* Convert to TxRx address for* HT access */
				conn = (tpl_fp_conn_t *)(PHYS_TO_XKPHYS(K_CALG_UNCACHED,
					(address_t)accept_req->accept_conn) | HW_HT_REMOTE_BASE_ADDRESS);

				conn->tpl_txrx_cb = pConn;

				/* create accept response */
				accept_rsp.hdr.mid = accept_req->hdr.mid;
				accept_rsp.hdr.conn = accept_req->hdr.conn;
				accept_rsp.hdr.status = STATUS_SUCCESS;
				accept_rsp.laddr = pConn->laddr;
				accept_rsp.raddr = pConn->raddr;
				accept_rsp.lport = pConn->lport;
				accept_rsp.rport = pConn->rport;
				accept_rsp.mtu_size = 0; // pConn->mtuSize;
				accept_rsp.tpl_conn_cb = pConnId;

				/* set tpl upcall to know where is more space in the socket buffer */
				tpl_setSockUpcall(pConn, tpl_fpContinueWrite);

				/* send back response */
				rc2 = eee_sendMessage(&accept_rsp, sizeof(accept_rsp), edesc->hdr.src_port,
						edesc->hdr.dest_port);

				if (rc2 != NFX_OK) {
					TPL_LOGERR( TPL_ERR_XMT_FAIL, &accept_rsp); 
					rc = NFX_ERR;
				}
				eee_freePacket(edesc);
				pBind->fpMsg = NULL;
				break;
			}	

			case TPL_FP_CONNECT:
			{
				tpl_fp_connect_req_t	*connect_req = (tpl_fp_connect_req_t *)hdr;
				tpl_fp_connect_rsp_t	connect_rsp;

				/* save the fp tpl connection structure */
				/* convert to TxRx address */
				*ppAppHdl = connect_req->hdr.conn;

				/* Convert to TxRx address */
				conn = (tpl_fp_conn_t *)(PHYS_TO_XKPHYS(K_CALG_UNCACHED, 
					     (address_t)connect_req->hdr.conn) | 
	 		 		     HW_HT_REMOTE_BASE_ADDRESS);	/* HT access */
				conn->tpl_txrx_cb = pConn;

				connect_rsp.hdr.mid = connect_req->hdr.mid;
				connect_rsp.hdr.conn = connect_req->hdr.conn;
				connect_rsp.hdr.status = STATUS_SUCCESS;
				connect_rsp.laddr = pConn->laddr;
				connect_rsp.raddr = pConn->raddr;
				connect_rsp.lport = pConn->lport;
				connect_rsp.rport = pConn->rport;
				connect_rsp.mtu_size = 0; // pConn->mtuSize;
				connect_rsp.tpl_conn_cb = pConn;

				/* set tpl upcall to know where is more space in the socket buffer */
				tpl_setSockUpcall(pConn, tpl_fpContinueWrite);

				/* our bind socket should be the same as the connected socket */
				NFX_ASSERT(pConn->pSock == pConn->pBindCb->pSock);
				pConn->pBindCb->pSock = NULL;
				
				/* send back response */
				rc2  = eee_sendMessage(&connect_rsp, sizeof(connect_rsp), edesc->hdr.src_port,
									edesc->hdr.dest_port);
				if (rc2 != NFX_OK) {
					TPL_LOGERR( TPL_ERR_XMT_FAIL, &connect_rsp); 
					rc = NFX_ERR;
				}
				eee_freePacket(edesc);
				pBind->fpMsg = NULL;
				break;
			}
			default:
				break;
		}
	} else {
		*ppAppHdl = NULL;
	}
	return rc;
}

/*-----------------------------------------------------------------
 * Name:         tpl_delFpConnInd
 * Description:  This function is called when a connection is closed
 * by the peer. If the socket was closed because it is
 * currently bind-only socket and connection could not be 
 * established, we do nothing.
 * Else we unbind the socket.
 * For a connected socket as the next read/write
 * request will detect this error and act accordingly
 *
 * Arguments:
 *  pAppHdl  - application handle to the connection
 *  pConnId  - connection pointer
 *
 * Return Values:
 *	NFX_OK
 *  Result of unbinding the socket
 *-----------------------------------------------------------------
 */
int32
tpl_delFpConnInd(void *pAppHdl, void *pConnId)
{
	tpl_connCb_t *pConn = pConnId;
	tpl_bindCb_t *pBind = pConn->pBindCb;
	tpl_fp_connect_req_t *req;
	tpl_fp_connect_rsp_t rsp;
	tpl_fp_close_rsp_t	closeMsg;
	uint32 error, srcPort, destPort;
	eee_desc_t *edesc;
	eee_descBuf_t *bd;
	int32 rc = NFX_OK;
	int32 rc2;

	/* if there was a connect pending on this socket,
	   then reply to that connect
	 */
	error = tpl_getSockError(pConnId);

	if (pBind->fpMsg) {

		edesc = pBind->fpMsg;
		bd = edesc_firstBuf(edesc);
		req = (tpl_fp_connect_req_t *)bd->buf;

		switch (req->hdr.mid) {
			case TPL_FP_CONNECT:
				memset(&rsp, 0, sizeof(rsp));
				rsp.hdr.mid = req->hdr.mid;
				rsp.hdr.conn = req->hdr.conn;

				switch(error) {
					case ECONNREFUSED:
						rsp.hdr.status = STATUS_CONNECTION_REFUSED;
						break;
					case ETIMEDOUT:
						rsp.hdr.status = STATUS_TIMEOUT;
						break;
					case ENETUNREACH:
						rsp.hdr.status = STATUS_NETWORK_UNREACHABLE;
						break;
					case ECONNABORTED:
					case ECONNRESET:
					default:
						rsp.hdr.status = STATUS_UNSUCCESSFUL;
						break;
				}
			
				/* send back response */
				rc2 = eee_sendMessage(&rsp, sizeof(rsp), edesc->hdr.src_port, 
						edesc->hdr.dest_port);
				if (rc2 != NFX_OK) {
					TPL_LOGERR( TPL_ERR_XMT_FAIL, &rsp); 
					rc = NFX_ERR;
				}
				eee_freePacket(edesc);
				pBind->fpMsg = NULL;
				return rc;

			default:
				break;
		}
	}
	
	/* if the socket has been closed by both FP and remote side, then get rid of it
	 */
	if (pConn->pAppHdl == NULL) {
		rc = tpl_delConn(pConn);

		/* if this is a client socket not belonging to a listening connection then
		   get rid of the bind point also
		   or if this is a server socket and the bind point has been closed
		   and this was the last connection belonging to that pBind, then
		   get rid of the bind point
		 */
		if (pBind->numConnCb == 0 && (pBind->bindFlags & (TPL_FLAG_CLIENT | TPL_BIND_CLOSE))) {
			tpl_freeBindCb(pBind);
		}
	} else {
		/* else, inform the FP that the socket has been closed.
		 * FP can then take action accordingly.
		 * drain the socket queue
		 */
		tpl_rcvFpPktInd(pConn->pAppHdl, NULL, NULL, 0);

		/* delete any pending requests from FP (read/write) on
		   this this connection
		 */
		if (pConn->fpMsg[TPL_FP_READ_MSG]) {
			eee_freePacket((eee_desc_t *)pConn->fpMsg[TPL_FP_READ_MSG]);
			pConn->fpMsg[TPL_FP_READ_MSG] = NULL;
		}
		if (pConn->fpMsg[TPL_FP_WRITE_MSG]) {
			eee_freePacket((eee_desc_t *)pConn->fpMsg[TPL_FP_WRITE_MSG]);
			pConn->fpMsg[TPL_FP_WRITE_MSG] = NULL;
		}
		closeMsg.hdr.mid = TPL_FP_CLOSE;
		closeMsg.hdr.status = STATUS_SUCCESS;
		closeMsg.hdr.conn = pConn->pAppHdl;

		srcPort = PACK_UNICAST_PORT_ADDRESS(nfxMySlotId,
           		nfxMyCPUId,
			eee_getApplicationId("tpl"),
                        EEE_USER_DATA,
		        NFX_COS_HIGH,
        		PORT_BEST_FIT);

		destPort = PACK_UNICAST_PORT_ADDRESS(nfxMySlotId,
          		NFX_NFP_FP_ANY,
			eee_getApplicationId("tpl-fp"),
                        EEE_USER_DATA,
		        NFX_COS_HIGH,
        		PORT_BEST_FIT);

		rc2 = eee_sendMessage(&closeMsg, sizeof(closeMsg), destPort, srcPort);
		if (rc2 != NFX_OK) {
			TPL_LOGERR( TPL_ERR_XMT_FAIL, &rsp); 
		}
	}
	return rc;
}

/*-----------------------------------------------------------------
 * Name:        tpl_fpConnFailedInd
 * Description: This function is called when a connect fails.
 *		We do the socket cleanup here.
 * Arguments:
 *  pBind    - bind control block pointer
 *
 * Return Values:
 *  Result of tpl_delFpConnInd()
 *-----------------------------------------------------------------
 */
int32
tpl_fpConnFailedInd(tpl_bindCb_t *pBind)
{
	tpl_connCb_t	pConn;

	memset(&pConn, 0, sizeof(pConn));
	pConn.pBindCb = pBind;
	return tpl_delFpConnInd(NULL, (void *)&pConn);
}

/*-----------------------------------------------------------------
 * Name:         tpl_rcvFpPktInd
 * Description:  Procedure is called when we receive a pkt from the peer
 *
 * Arguments:
 *  pAppHdl  - application handle to the connection
 *  pDesc    - packet descriptor
 *  pHdr     - packet data start
 *  tval - timestamp
 *-----------------------------------------------------------------
 */
void
tpl_rcvFpPktInd(void *pAppHdl, struct sk_buff *pd, void *pHdr, uint64 tval)
{
	tpl_fp_conn_t *conn = pAppHdl;
	tpl_fp_queue_t queue;
	tpl_connCb_t *pConn;
	pkt_queue_t *pktq;
	struct sk_buff *pDesc;
	eee_desc_t *edesc;
	eee_descBuf_t *bd;
	tpl_fp_msg_hdr_t *req;
	tpl_fp_read_rsp_t read_rsp;
	void **buffers;
	uint32 maxBufsToWrite;
	int64 maxBytes;
	int i;
	int32 rc2;
	
	if (pAppHdl == NULL) {
	   printk("%s: got bogus pAppHdl[%p]\n", __FUNCTION__, pAppHdl);
	   return;
	}

	/* Convert to TxRx address */
	conn = (tpl_fp_conn_t *)(PHYS_TO_XKPHYS(K_CALG_UNCACHED, (address_t)conn) | 
 		 		    HW_HT_REMOTE_BASE_ADDRESS);	/* HT access */

	memcpy(&queue, &conn->in_queue, sizeof(queue));

	buffers = (void **)(PHYS_TO_XKPHYS(K_CALG_UNCACHED, (address_t)queue.buffers) |
                                            HW_HT_REMOTE_BASE_ADDRESS);	/* HT access */
	pConn = conn->tpl_txrx_cb;
	pktq = &pConn->rpc_queue;

	/* go through the pkt descriptor list, extract the buffers and attach them
	 * to the tpl-fp input queue
	 */
	maxBufsToWrite = tpl_get_num_write_bufs(&queue);
	maxBytes = queue.max_bytes - (queue.num_bytes_written - queue.num_bytes_read);
	
	for (i = 0; i < maxBufsToWrite && maxBytes > 0 && !pq_is_empty(pktq); i++) {
		/* remove packet from head of queue
		 */
		pDesc = pktq->pq_head;
		pktq->pq_head = pktq->pq_current = pDesc->next;
		pktq->pq_len -= pDesc->len;
		maxBytes -= pDesc->len;
		if (pktq->pq_head == NULL)
			pq_make_empty(pktq);
		pDesc->next = NULL;
		pq_validate(pktq);

		/* attach packet data to TPL-FP queue
		 */
		if (pDesc->len > 0) {
			buffers[queue.write_idx] = pDesc->data;
			tpl_set_buflen(buffers, queue.write_idx, queue.max_buffers, pDesc->len);
			queue.write_idx++;
			if (queue.write_idx == queue.max_buffers)
				queue.write_idx = 0;
			queue.num_bytes_written += pDesc->len;
		} else {
			kfree_skb(pDesc);
		}
	}
	conn->in_queue.write_idx = queue.write_idx;

	/* Flush the write_idx before updating num_bytes_written to ensure that the
	 * FP sees the update to write_idx before the num_bytes_written update.
	 */
	// mips_wbflush();

	conn->in_queue.num_bytes_written = queue.num_bytes_written;

#ifdef LATER
    /* Send a window update message to the peer
     */
    if (pConn->pSock) {
        /* Switch to the correct vstack context
         */
        if (vstack_switchContext(pConn->vsid) == VSTACK_SUCCESS) {
            tcp_output(&((struct socket *)pConn->pSock)->so_tcpcb);
        } else {
            ASSERT(FALSE);
        }
    }
#endif
	/* if FP was waiting for data, then send it a read response
	 */
	if (pConn->fpMsg[TPL_FP_READ_MSG]) {
		edesc = (eee_desc_t *)pConn->fpMsg[TPL_FP_READ_MSG];
		bd = edesc_firstBuf(edesc);
		req = (tpl_fp_msg_hdr_t *)bd->buf;
		
		if (req->mid == TPL_FP_READ) {
			read_rsp.hdr.mid = req->mid;
			read_rsp.hdr.conn = req->conn;
			read_rsp.hdr.status = STATUS_SUCCESS;

			pConn->fpMsg[TPL_FP_READ_MSG] = NULL;

			/* send back response */
			rc2 = eee_sendMessage(&read_rsp, sizeof(read_rsp),
				edesc->hdr.src_port, edesc->hdr.dest_port);
			if (rc2 != NFX_OK) {
				TPL_LOGERR( TPL_ERR_XMT_FAIL, &read_rsp); 
			}
			eee_freePacket(edesc);	
		}
	}
	return;
}

/*-----------------------------------------------------------------
 * Name:        tpl_fpAllocPktDesc
 * Description: Uses part of data buffer as packet queue header for 
 *		the data to be sent.
 * Arguments:
 *  pConn - connection on which data is to be sent
 *  buf	  - data to be sent
 *  size  - data length
 *-----------------------------------------------------------------
 */
struct sk_buff *
tpl_fpAllocPktDesc(tpl_connCb_t *pConn, char *buf, uint32 size)
{
      struct sk_buff *pDesc;

      // pDesc = eee_getBufBase(buf);
      // pktStats.cntAllocPkt++;

      // pDesc->pTplConn = pConn;
      // pDesc->len = size;
      // pDesc->data = buf;
      // pDesc->next = NULL;

      // pDesc->len = pDesc->pktSize = size;
      // pDesc->flags = (PKT_FAST_PATH | PKT_DESC_INUSE | PKT_DATA);
      // pDesc->pTplNext = NULL;
      // pDesc->p_time_stamp = 0;

      pDesc = alloc_skb(size, (__GFP_HIGHMEM | __GFP_NOFAIL | __GFP_DMA));
      pDesc->tstamp = 0;

      return pDesc;
}

/*-----------------------------------------------------------------
 * Name:         tpl_fpTxPkt
 * Description:  Takes data from FP output queue, converts it into
 *               a packet queue and queues it on the socket buffers
 *
 * Arguments:
 *  pConn - TPL TxRx connection structure
 *  conn  - TPL FP connection structure
 *  queue - TPL FP queue
 *-----------------------------------------------------------------
 */
void
tpl_fpTxPkt(tpl_connCb_t *pConn, tpl_fp_conn_t *conn, tpl_fp_queue_t *queue)
{
	struct sk_buff	*pDesc;
	int64		len;
	char		*buf;
	uint32		bufLen;
	void        **buffers;
	tpl_fp_queue_t q;
	
	memcpy(&q, queue, sizeof(q));
	buffers = (void **)(PHYS_TO_XKPHYS(K_CALG_UNCACHED,
			(address_t)q.buffers) | HW_HT_REMOTE_BASE_ADDRESS);	/* HT access */

	/* while there is data to send and there is space in the socket
	 * buffers, keep queuing data into socket buffers
	 */
	while((len = ((int64)q.num_bytes_written - (int64)q.num_bytes_read)) > 0) {
		/* check if there is space in socket buffer to send 1 more buffer */
		buf = buffers[q.read_idx];
		
		/* check if there is enough space to queue the packet into the socket
		 * stream 
		 */
		bufLen = tpl_get_buflen(buffers, q.read_idx, q.max_buffers);
		NFX_ASSERT(buf != NULL && bufLen > 0);
#ifdef LATER
		if (tpl_getSndSockSpace(pConn) < bufLen) {
			break;
		}
#endif /* LATER */
		/* allocate packet desc. queue for sending the pkt */
		pDesc = tpl_fpAllocPktDesc(pConn, buf, bufLen);
		NFX_ASSERT(pDesc);

		/* transmit the packet queue */
		NFX_ASSERT(pDesc->len != 0);
		// NFX_ASSERT(pDesc->flags != 0);
		NFX_ASSERT(pDesc->data != 0);

		if (tpl_xmtPkt(pConn, pDesc) == NFX_OK) {

			buffers[q.read_idx] = NULL;
			tpl_set_buflen(buffers, q.read_idx, q.max_buffers, 0);
			q.cur_read_offset = 0;
			q.read_idx++;

		    if (q.read_idx == q.max_buffers)
			    q.read_idx = 0;
			q.num_bytes_read += bufLen;
			pTplCfg->tplStats.cntFpTxRcv++;
		} else {
			// VERIFY(0);
			TPL_LOGERR( TPL_ERR_XMT_FAIL, pDesc); 
			break;
		}
	}
	queue->read_idx = q.read_idx;
	queue->cur_read_offset = q.cur_read_offset;
	queue->num_bytes_read = q.num_bytes_read;
	return;
}

/*-----------------------------------------------------------------
 * Name:         tpl_fpContinueWrite
 * Description:  Callback routine called when there is more space in
 *               the socket buffer to write
 *               This routine pulls out more data from the TPL-FP ring
 *               buffers and puts them into the socket buffer for transmission
 * Arguments:
 *  sock    - tcp socket associated with the TPL connection
 *  pConnId - pointer to tpl connection
 *  unused  - unused variable
 *-----------------------------------------------------------------
 */
void
tpl_fpContinueWrite(void *sock, void *pConnId, int unused)
{
    tpl_connCb_t *pConn = pConnId;
    tpl_fp_conn_t *conn = pConn->pAppHdl;
    eee_desc_t *edesc;
    eee_descBuf_t *bd;
    tpl_fp_write_req_t *write_req;
    tpl_fp_write_rsp_t write_rsp;
    tpl_fp_queue_t queue;
    int32 rc;

    conn = (tpl_fp_conn_t *)(PHYS_TO_XKPHYS(K_CALG_UNCACHED,
		(address_t)conn) | HW_HT_REMOTE_BASE_ADDRESS); /* HT access */

    /* continue to read the FP queue and write into socket buffer */
    tpl_fpTxPkt(pConn, conn, &conn->out_queue);

    /* if FP is waiting to write more data into the queue, then inform FP */
    if (pConn->fpMsg[TPL_FP_WRITE_MSG]) {

	edesc = (eee_desc_t *)pConn->fpMsg[TPL_FP_WRITE_MSG];
	bd = edesc_firstBuf(edesc);

	write_req = (tpl_fp_write_req_t *)bd->buf;
	memcpy(&queue, &conn->out_queue, sizeof(queue));

	if (write_req->hdr.mid == TPL_FP_WRITE) {

		/* if there is space available for FP to write, then return a favorable response */
		if ((queue.max_bytes - (queue.num_bytes_written - queue.num_bytes_read)
		     < write_req->write_size) || tpl_get_num_write_bufs(&queue)  == 0) {
			return;
		}
		write_rsp.hdr.mid = write_req->hdr.mid;
		write_rsp.hdr.conn = write_req->hdr.conn;
		write_rsp.hdr.status = STATUS_SUCCESS;

		pConn->fpMsg[TPL_FP_WRITE_MSG] = NULL;

		/* send back response */
		rc = eee_sendMessage(&write_rsp, sizeof(write_rsp), edesc->hdr.src_port, edesc->hdr.dest_port);
		if (rc != NFX_OK) {
			TPL_LOGERR( TPL_ERR_XMT_FAIL, &write_rsp); 
		}
		eee_freePacket(edesc);
	}
     }
     return;
}
