/*
 *-----------------------------------------------------------------
 * Name: tpl.c 
 * Copyright (C) 2009, OnStor, Inc.
 * Description: Transport Packet Layer (TPL) External Operations
 *-----------------------------------------------------------------
 */
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>

#include <linux/gfp.h>
#include <linux/slab.h>

#include "nfx-defs.h"
#include "nfx-types.h"

#include "eee-desc.h"
#include "tpl-api.h"
#include "tpl.h"

/*
 * External Procedures called by Linux netfilter code
 */
#ifdef LATER
tpl_bindCb_t *tpl_findBindCb(unsigned short prot, unsigned short lport , unsigned int laddr, unsigned int vsId);
tpl_bindCb_t *tpl_findBindCbByUsrInfo(void *usrInfo);
tpl_bindCb_t *tpl_allocBindCb(void);
void tpl_freeBindCb(tpl_bindCb_t *pBind);

int tpl_bsdConnInd(void *sock, void *pcb, int evt, void **pHdl);
int tpl_bsdRcvInd(void *sock, void *pDesc, unsigned int saddr, unsigned short sport, unsigned int len, void *hdl);

/*
 * NCPU called functions
 */
int tpl_listenReq(tpl_bindCb_t *pBind, unsigned int laddr, unsigned short lport);
int tpl_connectReq(tpl_bindCb_t *pBind, unsigned int raddr, unsigned short rport);
int tpl_abortConn(tpl_connCb_t *pConn);
void tpl_abortAllConnForIP(unsigned int ipaddr);

/*
 * ACPU called functions
 */
int tpl_bindReq(tpl_bindCb_t *bindReq, void *handle, unsigned int bindOpt);
int tpl_unbindReq(tpl_bindCb_t *pBind);
int tpl_addConnInd(unsigned short prot,	unsigned int laddr, unsigned int raddr,	unsigned short lport
	unsigned short rport, void *pInp, void *pTcp, void *sock, void **ppConn);
int tpl_delConnReq(void *connPtr);
int tpl_delConnAck(void *connPtr);
int tpl_delConnInd(void *pConn);
tpl_connCb_t *tpl_findConnByAppHdl(void *appHdl);
unsigned int tpl_getSockError(tpl_connCb_t *pConn);
void tpl_setSockUpcall(tpl_connCb_t *pConn, void (*upcall)(void *sock, void *arg, int unused));

#endif /* LATER */


/* Global variables and constants
 */
unsigned int tpl_abort_log_flag = 0;
 
#define TPL_CONN_PRINTK(a)	printk a

#define E_LOG(a,b,c,d,e,f,g)

#define	TPL_REQ		1	/* Upper Layer Request */
#define	TPL_IND		0	/* Lower Layer Indication */

int bsdrl_appid = 0xf;

tpl_cfg_t *pTplCfg;

/* External procedures
 */
extern void * kmalloc(size_t size, gfp_t flags);
extern atomic_t txrx_requests_pending;

extern struct sk_buff *skb_alloc(void);
extern void pkt_acpu_free_from_ncpu(struct sk_buff *pkt);
extern int tpl_ipcSendEvntMsg(int, tpl_bindCb_t *, int, unsigned int);
extern void tpl_ipcq_write(tpl_ipcq_t *, tpl_evntMsg_t *, int);
extern void pq_make_empty(pkt_queue_t *);

#ifdef LATER
extern struct pkt_free_list pkt_acpu_free_list;
extern struct pkt_free_list pkt_ncpu_free_list;
#endif /* LATER */

extern void *linux_socket(int protocol, int bindType, bsd_sarg_t *sockArg);
extern int linux_listen(struct socket *, int, int, int);
extern int linux_connect(struct socket *, unsigned int, unsigned int);
extern int linux_socket_close(struct socket *);
extern int linux_socket_abort(struct socket *);

/*-----------------------------------------------------------------
 * Name: 		tpl_findBindCb
 * Description: Returns Bind CB that matches protocol & port number.
 *-----------------------------------------------------------------
 */
tpl_bindCb_t * 
tpl_findBindCb(unsigned short prot, unsigned short lport, unsigned int laddr, unsigned int vsId)
{
	TPL_CONN_PRINTK(("%s: prot %d lport %d, laddr 0x%x, vsId %d\n",
		__FUNCTION__, prot, lport, laddr, vsId));

#ifdef LATER
	tpl_bindCb_t	*pBind;
	int i;

	TPL_CONN_PRINTK(("%s: prot %d lport %d, laddr 0x%x, vsId %d\n",
		__FUNCTION__, prot, lport, laddr, rport, vsId));

	/*-------------------------------
	 * Find Bind CB that matches
	 * IP, port & protocol.
	 *-------------------------------
	 */
	pBind = pTplCfg->bindTblBase;
	for (i=0; i < pTplCfg->maxBindCb; i++) {
		if ((pBind->lport != TPL_ANY_PORT) && 
		    (pBind->lport   == lport) && 
		    (pBind->tplProt == prot) && 
		    (pBind->laddr == laddr) &&
		    (pBind->vsId  == vsId))
			return(pBind);

		pBind++;
	}
#endif /* LATER */
	return (NULL);
}

/*-----------------------------------------------------------------
 * Name: 		tpl_findBindCbByUsrInfo
 * Description: Returns Bind CB that matches usrInfo
 *-----------------------------------------------------------------
 */
tpl_bindCb_t *
tpl_findBindCbByUsrInfo(void *usrInfo)
{
	TPL_CONN_PRINTK(("%s\n", __FUNCTION__));
#ifdef LATER
	tpl_bindCb_t	*pBind;
	int			i;

	TPL_CONN_PRINTK(("%s\n", __FUNCTION__));

	/*-----------------------------------
	 * Find Bind CB that matches usrInfo
	 *-----------------------------------
	 */
	pBind = pTplCfg->bindTblBase;
	for (i=0; i < pTplCfg->maxBindCb; i++) {
		if (pBind->usrInfo == usrInfo)
			return(pBind);
		pBind++;
	}
#endif /* LATER */
	return (NULL);
}

/*-----------------------------------------------------------------
 * Name: tpl_allocBindCb
 * Description: Allocates a Bind CB. Returns NULL if none available.
 *-----------------------------------------------------------------
 */
tpl_bindCb_t *
tpl_allocBindCb(void)
{
	TPL_CONN_PRINTK(("%s\n", __FUNCTION__));
#ifdef LATER
	tpl_bindCb_t	*pBind;
	int i;

	TPL_CONN_PRINTK(("%s\n", __FUNCTION__));

	pBind = pTplCfg->bindTblBase;

	for (i=0; i < pTplCfg->maxBindCb; i++) {

		if (!(pBind->lport) &&
		    !(pBind->tplProt) && 
		    !(pBind->ul_addConnInd) &&
		    !(pBind->ul_delConnInd) && 
		    !(pBind->ul_rcvPktInd))
			return(pBind);

		pBind++;
	}
#endif /* LATER */
	return (NULL);
}

/*-----------------------------------------------------------------
 * Name: tpl_freeBindCb
 * Description: Frees a Bind CB. 
 *-----------------------------------------------------------------
 */
void
tpl_freeBindCb(tpl_bindCb_t *pBind)
{
	TPL_CONN_PRINTK(("%s\n", __FUNCTION__));
	memset((void *)pBind, 0, sizeof(tpl_bindCb_t));
	return;
}

/*-----------------------------------------------------------------
 * Name: tpl_validateBindReq
 * Description: Validate user inputs to bind request 
 *-----------------------------------------------------------------
 */
int
tpl_validateBindReq(tpl_bindCb_t *pBind, int bindMode)
{
	TPL_CONN_PRINTK(("%s, bindMode 0x%x\n", __FUNCTION__, bindMode));

	switch (bindMode) {
        /*------------------------------------
         * If its a client bind only request
         * then remote address and remote port
         * are not required
         *------------------------------------
         */
        case    TPL_BIND_CLIENT | TPL_BIND_ONLY:
                break;

		/*-------------------------------
		 * For client bind, raddr 
		 * & rport are REQUIRED.
		 *-------------------------------
		 */
	case	TPL_BIND_CLIENT:
		if (pBind->raddr == 0) 
			return (TPL_ERR_NO_RADDR);

		if (pBind->rport == 0)
			return (TPL_ERR_NO_RPORT);
		break;

		/*-------------------------------
		 * For server bind, raddr * & rport 
		 * MUST NOT be set.
		 *-------------------------------
		 */
		case	TPL_BIND_SERVER:
			if (pBind->raddr != 0) 
				return (TPL_ERR_RADDR_IN_SRVR);
			if (pBind->rport != 0)
				return (TPL_ERR_RPORT_IN_SRVR);
		break;

        case    TPL_BIND_REUSEPORT:
        case    TPL_BIND_ROUTETOIF:
                break; 

		default:
			return (TPL_ERR_BAD_MODE );
	}

	/*-------------------------------
	 * Accepts TCP & UDP only.
	 *-------------------------------
	 */
	if ((pBind->tplProt != TPL_PROT_UDP) && 
		(pBind->tplProt != TPL_PROT_TCP)) {
		return (TPL_ERR_BAD_PROT );
	}
	/*-------------------------------
	 * Checks for upcalls are set.
	 *-------------------------------
	 */
	if ( !(pBind->ul_addConnInd) || !(pBind->ul_delConnInd) || !(pBind->ul_rcvPktInd)) {
		return (TPL_ERR_BAD_FUNC );
	}
	return 0;
}

/*-----------------------------------------------------------------
 * Name:        tpl_bindReqCore
 *
 * Description: Binds upper layer to Transport layer.
 *              Also binds TPL to Linux layer.
 *
 *              The following table shows the fields required 
 *              in the tpl_bindCb for Server or Client bind request.
 *
 *                          SERVER          CLIENT
 *              ------------------------------------
 *              laddr       Required        Required
 *              lport       Required        Optional
 *              raddr       No              Required
 *              rport       No              Required
 *              addConnInd  Required        Required
 *              delConnInd  Required        Required
 *              rcvPktInd   Required        Required
 *              xmtCmpInd   Required        Required
 *
 *              NOTE: laddr, lport, raddr, rport are expected to
 *                    be in NETWORK ORDER (Big Endian)
 *
 * Arguments:
 * tpl_bindCb_t *pReq ptr to Bind Req Cb
 * tpl_bindCb_t **pRsp ptr to Bind Response Cb ptr
 * int bindOpt       see tpl-api.h for options
 *-----------------------------------------------------------------
 */
int
tpl_bindReqCore(tpl_bindCb_t *pReq, tpl_bindCb_t **pRsp, unsigned int bindOpt)
{
	tpl_bindCb_t	*pBind;
	void *pSock;
	int protocol, bindType, rc;
	unsigned short flags;
	bsd_sarg_t sockArg;

	TPL_CONN_PRINTK(("%s, bindOpt 0x%x\n", __FUNCTION__, bindOpt));

	pBind = pReq;
	flags = pBind->bindFlags;

	/*-------------------------------
	 * Check user input.
	 *-------------------------------
	 */
	rc = tpl_validateBindReq( pReq, (bindOpt & (TPL_BIND_CLIENT | TPL_BIND_ONLY)));
	if (rc != 0) {
		TPL_LOGERR( rc, pBind );
                *pRsp = pReq;
		return (rc);
	}
	/*-------------------------------
	 * Checks for duplication.
	 *-------------------------------
	 */
	if (tpl_findBindCb( pBind->tplProt, 
                        pBind->lport, 
                        pBind->laddr,
                        pBind->vsId ) != NULL) {
		TPL_LOGERR( TPL_ERR_DUP_BIND, pBind );
                *pRsp = pReq;
		return (TPL_ERR_DUP_BIND);
	}

	/*-------------------------------
	 * Alloc tpl_bindCb.
	 *-------------------------------
	 */
	pBind = tpl_allocBindCb();
	if (pBind == NULL) {
		TPL_LOGERR( TPL_ERR_NO_BIND_RSRC, pBind );
                *pRsp = pReq;
		return (TPL_ERR_NO_BIND_RSRC);
	}

	/*-------------------------------
	 * Copy request into new bindCb.
	 * Return real bindCb in pRsp.
	 *-------------------------------
	 */
	memcpy((void *)pBind, (void *)pReq, sizeof(tpl_bindCb_t));
	pBind->pConnList = NULL;
	pBind->pSock  = NULL;
	pBind->acpuMsg = NULL;
	pBind->numConnCb = 0;
	pBind->acpuState = TPL_IPC_CONN_INIT;
	
	/*-------------------------------
	 * Setup BSD socket arg for binding.
	 *-------------------------------
	 */
	memset((void *)&sockArg, 0, sizeof(bsd_sarg_t));
	sockArg.laddr   = pBind->laddr;		/* Local  IP */
	sockArg.faddr   = pBind->raddr;		/* Remote IP */
	sockArg.lport   = pBind->lport;		/* Local  Port */
	sockArg.fport   = pBind->rport;		/* Remote Port */
	sockArg.handle  = pBind;		/* Handle */
	sockArg.bufType = 0; /* BSD_BUF_DESCR, Expects pkt desc */
	sockArg.cb = tpl_bsdConnInd;	/* Callback for connection event */
	sockArg.db = tpl_bsdRcvInd;	/* Callback for rcv data event */
	sockArg.flags   = 0;
	// ((bindOpt & TPL_BIND_REUSEPORT) ? BSD_SO_REUSEPORT : 0)
	// | ((bindOpt & TPL_BIND_ROUTETOIF) ? BSD_SO_ROUTETOIF: 0);

	if (bindOpt & TPL_BIND_CLIENT)
		pBind->bindFlags |= TPL_FLAG_CLIENT;

	if (bindOpt & TPL_BIND_USE_RPC)
		pBind->bindFlags |= TPL_FLAG_USE_RPC;

	if (bindOpt & TPL_BIND_USE_NETBIOS)
	        pBind->bindFlags |= TPL_FLAG_USE_NETBIOS;

	if (bindOpt & TPL_BIND_ONLY)
		pBind->bindFlags |= TPL_FLAG_BIND_ONLY;

	if (bindOpt & TPL_BIND_USE_KEEPALIVE) {
	  /* This bind endpoint and all the subsequent children of this endpoint
	   * shall inherit this value
	   */
	        pBind->bindFlags |= TPL_FLAG_USE_KEEPALIVE;
	}
	/*-------------------------------
	 * Setup bsd bind protocol & type.
	 *-------------------------------
	 */
	// bindType = (bindOpt & TPL_BIND_CLIENT) ? BSD_SO_CLIENT : BSD_SO_SERVER;
	// bindType |= (bindOpt & TPL_BIND_ONLY) ? BSD_SO_BIND_ONLY : 0;
	// protocol = (pBind->tplProt == TPL_PROT_UDP) ? BSD_IPPROTO_UDP : BSD_IPPROTO_TCP;
	bindType = 0; /* XXX */
	protocol = pBind->tplProt; /* XXX */

	/*-------------------------------
	 * Bind with FreeBSD Code.
	 *-------------------------------
	 */
	pSock = linux_socket(protocol, bindType, &sockArg);

	if (!pSock) {
		TPL_LOGERR( TPL_ERR_NO_BIND_RSRC, pBind );
		*pRsp = pReq;
		tpl_freeBindCb(pBind);
		return (TPL_ERR_NO_BIND_RSRC);
	}

	pBind->pSock = pSock;
	if (pBind->lport == 0)
	  pBind->lport = sockArg.lport;

	*pRsp = pBind;

	pTplCfg->tplStats.cntBindReq++;
	pTplCfg->numBindCb++;

	return 0;
}

/*-----------------------------------------------------------------
 * Name:        tpl_bindReq
 * Description: bind request on ACPU
 *-----------------------------------------------------------------
 */
int
tpl_bindReq(tpl_bindCb_t *bindReq, void *handle, unsigned int bindOpt)
{
	tpl_bindCb_t myBindReq;
	
	memcpy((void *)&myBindReq, (void *)bindReq, sizeof(tpl_bindCb_t));
	
	/* use flags field for options, and pSock field for handle */
	myBindReq.pSock = handle;
	myBindReq.bindFlags = bindOpt;
	
	/* send to NCPU */
	return tpl_ipcSendEvntMsg(TPL_EVNT_BIND_REQ, &myBindReq, 0, bindReq->vsId);
}

/*-----------------------------------------------------------------
 * Name:        tpl_unbindReqCore
 * Description: Unbinds upper layer from Transport layer. 
 *              Deletes all connections associated with bindCB and
 *              removes TPL from Linux layer.
 *-----------------------------------------------------------------
 */
int
tpl_unbindReqCore(tpl_bindCb_t *pBind)
{
    tpl_connCb_t	*pConn, *npConn;
    tpl_evntMsg_t	*evntMsg;
    tpl_ipcq_t		*evntQNtoAPtr;
	
    /*-------------------------------
     * Check if bindCb is valid.
     *-------------------------------
     */
    if (tpl_validateBindCb( pBind ) != 0) {
        return ( TPL_ERR_NOT_BIND );
    }

    /*-------------------------------
     * Removes associated connections.
     * note that list can be modify
     *-------------------------------
     */
    if (!(pBind->bindFlags & TPL_BIND_CLIENT)) {
        pConn = pBind->pConnList;
        while (pConn) {
            npConn = pConn->pNext;
            tpl_abortConn(pConn);
            pConn = npConn;
        }
    }
	
	/*-------------------------------
	 * Close assoc. BSD sock.
	 *-------------------------------
	 */
	if (pBind->pSock) {
		linux_socket_close(pBind->pSock);
		pBind->pSock = NULL;
	}
	pBind->bindFlags |= TPL_BIND_CLOSE;

	/* pending pConn, wait */
	if (pBind->pConnList)
		return 0;

	/* if there is request from acpu, send reply
	 */
	if (pBind->acpuMsg) {
		evntMsg = (tpl_evntMsg_t*)pBind->acpuMsg;
		evntQNtoAPtr = &(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_NTOA]);
		evntMsg->evnt_msg = TPL_EVNT_UNBIND_REQ_ACK;
		evntMsg->evnt_rc = 0;

		tpl_ipcq_write(evntQNtoAPtr, evntMsg, TPL_EVNT_UNBIND_REQ_ACK);
	}
	return 0;
}

/*-----------------------------------------------------------------
 * Name:        tpl_unbindReq
 * Description: unbind request on ACPU
 *-----------------------------------------------------------------
 */
int
tpl_unbindReq(tpl_bindCb_t *pBind)
{
    int rc;

    if (pBind->acpuState == TPL_IPC_CONN_ADDED) {
        pBind->acpuState = TPL_IPC_CONN_CLOSE_SENT;
	rc = tpl_ipcSendEvntMsg(TPL_EVNT_UNBIND_REQ, pBind, 0, pBind->vsId);
    } else {
      rc = -1; /* XXX */
    }
    return rc;
}

/*-----------------------------------------------------------------
 * Name:        tpl_listenReq
 * Description: listen request on NCPU
 *-----------------------------------------------------------------
 */
int
tpl_listenReq(tpl_bindCb_t *pBind, unsigned int laddr, unsigned short lport)
{
    int rc;
    pBind->bindFlags &= ~TPL_FLAG_CLIENT;

    rc = linux_listen(pBind->pSock, laddr, lport, 0);
    return rc;
}

/*-----------------------------------------------------------------
 * Name:        tpl_connectReq
 * Description: connect request on NCPU
 *-----------------------------------------------------------------
 */
int
tpl_connectReq(tpl_bindCb_t *pBind, unsigned int raddr, unsigned short rport)
{
    return linux_connect(pBind->pSock, raddr, rport);
}

/*-----------------------------------------------------------------
 * Name: tpl_addConnInd
 * Description: Called by Linux tcp stack to add a new connection.
 * Returns: 0, or err
 *-----------------------------------------------------------------
 */
int
tpl_addConnInd(unsigned short prot, 	/* protocol UDP/TCP */
	unsigned int laddr, 	/* local  IP address */
	unsigned int raddr,	/* remote IP address */
	unsigned short lport,	/* local  Port # */
	unsigned short rport,	/* remote Port # */
	void	 *pInp,	/* ptr to in_pcb (bsd code)*/
	void	 *pTcp,	/* ptr to tcpcb (bsd code)*/
	void 	 *sock,
	void	 **ppConn) /* ptr to pConn, ret to caller */
{
	tpl_bindCb_t	*pBind;
	tpl_connCb_t	*pConn;
	int rc;
	
	TPL_CONN_PRINTK(("tpl_addConnInd prot %d, la 0x%x, ra 0x%x, lp %d, rp %d\n",
		prot, ntohl(laddr), ntohl(raddr), ntohs(lport), ntohs(rport)));

#ifdef LATER
	struct inpcb    *pInpcb;
	int camIdx;

	pInpcb = pInp;

	/*-------------------------------
	 * use pBind passed during tpl_bindReq() 
	 *-------------------------------
	 */
	pBind = *ppConn; 

	/*-------------------------------
	 * Check if protocol & port have been bound to TPL.
	 *-------------------------------
	 */
	if (tpl_validateBindCb(pBind) != 0) {
		return (TPL_ERR_NOT_BIND);
	}

	/*-------------------------------
	 * Allocate tpl_connCb of same idx.
	 *-------------------------------
	 */
	camIdx = 0; /* XXX */
	pConn = (tpl_connCb_t *)(&pTplCfg->connTblBase[camIdx]);

	if (pConn->flags & TPL_CONN_INUSE) {
		/*-------------------------------
		 * CAM & TPL Connection Out of Sync!
		 *-------------------------------
		 */
		pTplCfg->tplStats.cntBadConnErr++;
 		return (TPL_ERR_BAD_CONN);
	}
#else
	pBind = *ppConn; 
	pConn = (tpl_connCb_t *)kmalloc(sizeof(tpl_connCb_t), (__GFP_HIGHMEM | __GFP_NOFAIL));
#endif /* LATER */

	/*-------------------------------
	 * Setup tpl conn fields.
	 *-------------------------------
	 */
	pConn->flags = TPL_CONN_INUSE;
	pConn->prot = prot;
	pConn->laddr = laddr;
	pConn->raddr = raddr;
	pConn->lport = lport;
	pConn->rport = rport;
	pConn->pBindCb = pBind;
	pConn->rcv_func = pBind->ul_rcvPktInd;
	pConn->vsid = 0; // vstack_getCurVsid()
	pConn->pSock = sock;

	pq_make_empty(&pConn->rpc_queue);
	pConn->rpcRasmLen = 0;

	// pConn->arpTx = 0;
	
	if (pBind->bindFlags & TPL_FLAG_USE_RPC)
		pConn->flags |= TPL_CONN_USE_RPC;

	if (pBind->bindFlags & TPL_FLAG_USE_NETBIOS)
	        pConn->flags |= TPL_CONN_USE_NETBIOS;

	/*-------------------------------
	 * Upcall app. for new connection.
	 * FP Connections need to use callback
	 *------------------------------------
	 */
	if (pBind->bindFlags & TPL_FLAG_FP_CONN) {
		rc =  (pBind->ul_addConnInd)(pBind, pConn, &pConn->pAppHdl ,raddr, rport);
        } else {
	        rc = tpl_ipcSendEvntMsg(TPL_EVNT_ADD_CONN, pBind, 1, pBind->vsId);
        }
	if (rc) {
		pConn->flags = 0;
		pTplCfg->tplStats.cntUlAddFail++;
		return TPL_ERR_UL_ADD_FAIL;
	}
#ifdef LATER
	/*-------------------------------
	 * Insert new conn at head of list.
	 *-------------------------------
	 */
	pConn->pNext = pBind->pConnList;
	pBind->pConnList = pConn;
#endif /* LATER */

	pBind->numConnCb++;
	pTplCfg->tplStats.cntAddConnInd++;
	pTplCfg->numConnCb++;

	*ppConn = pConn;
	return 0;
}

/*-----------------------------------------------------------------
 * Name:	tpl_delConnCore
 * Description: Called to delete a connection.
 * Returns: 0, or error
 *-----------------------------------------------------------------
 */
int
tpl_delConnCore(tpl_connCb_t *pConn)
{
#ifdef LATER
	tpl_bindCb_t *pBind = pConn->pBindCb;
	tpl_connCb_t *prevPtr, *currPtr;

	if (pConn->arpTx) {
	  pTplCfg->tplStats.cntXmtMacQueue--;
	  pkt_acpu_free_from_ncpu(pConn->arpTx);
	}
	
	struct sk_buff *tp, *ntp;
	/*
	 * Remove any acpuPktQ, if there are queued pkts ??
	 */
        for (tp = pConn->acpuPktQ; tp;) {

		ntp = tp->next;
		tp->next = NULL;

		atomic_dec(&txrx_requests_pending);

		// pkt_freePkt(tp);
		tp = ntp;
	}
	/*-------------------------------
	 * Increment del req/ind counter.
	 *-------------------------------
	 */
	pTplCfg->tplStats.cntDelConnInd++;
	pTplCfg->numConnCb--;
	pBind->numConnCb--;
	
	/*-------------------------------
	 * Delete conn from head of list.
	 *-------------------------------
	 */
	if (pBind->pConnList == pConn) {
		pBind->pConnList = pConn->pNext;
	} else {
		/*-------------------------------
		 * Go thru list to find matching 
		 * conn to be removed.
		 *-------------------------------
		 */
		prevPtr = pBind->pConnList;
		currPtr = prevPtr->pNext;
		while(currPtr) {
			if (currPtr == pConn) {
				prevPtr->pNext = currPtr->pNext;
				break;
			}
			prevPtr = currPtr;
			currPtr = currPtr->pNext;
		}
	}

	/*-------------------------------
	 * Clear tpl_connCb.
	 *-------------------------------
	 */
	memset((void *)pConn, 0, sizeof(tpl_connCb_t));
	
	/* Su, 10/02/02 
	 * If actively opened conn, clear BindCB
	 */
	if ((pBind->bindFlags & TPL_FLAG_CLIENT)) {
		if (pBind->pConnList || pBind->numConnCb) {
			/* For actively open Conn, each 
			 * bindCB at most has one pConn associate
			 * to it, and this pConn had been deleted
			 * now.
			 */
			TPL_LOGERR(TPL_ERR_BAD_BIND, pBind);
		}

		/* socket has already been closed */
		pBind->pSock = NULL;

		if (pBind->bindFlags & TPL_FLAG_FP_CONN) {
			tpl_freeBindCb(pBind);

			tpl_unbindReqCore(pBind);
	  		(pBind->ul_delConnInd)( pConn->pAppHdl , pConn);
        }
        else if (pBind->bindFlags & TPL_BIND_CLOSE) {
			tpl_unbindReqCore(pBind);
             } 
	}
	else if ((pBind->bindFlags & TPL_BIND_CLOSE) &&
			 (!pBind->pConnList)) {
		tpl_unbindReqCore(pBind);
	}
#endif
    return 0;
}	

/*-----------------------------------------------------------------
 * Name:	tpl_delConn
 * Description: Called by Linux tcp stack to delete a connection.
 * Returns: 0, or error
 *-----------------------------------------------------------------
 */
int
tpl_delConn(tpl_connCb_t *pConn , int dir)
{
	tpl_bindCb_t *pBind;
	int rc;

	TPL_CONN_PRINTK(("tpl_delConn %s pConn 0x%p\n", (dir)? "Req":"Ind", pConn));

	/*-------------------------------
	 * Make sure connCb is still vaild.
	 *-------------------------------
	 */
	if (tpl_validateConnCb(pConn) != 0)
		return (TPL_ERR_BAD_CONN);

	/*-------------------------------
	 * Make sure binding is still valid.
	 * Upcall app. for delete conn.
	 *-------------------------------
	 */
	pBind = pConn->pBindCb;
	if (tpl_validateBindCb(pBind) != 0)
		return (TPL_ERR_BAD_CONN);

	/* connection is closed, no more incoming pkts */
	pConn->flags |= TPL_CONN_CLOSED;

#ifdef LATER
	/* Remove the connection from the wait list if it is currently waiting for
	 * the packets.
	 */
	if (pConn->flags & TPL_CONN_WAIT_PACKETS) {
	  SIMPLEQ_REMOVE((&(pkt_ncpu_free_list.pf_conn_waiters)),
                       pConn, _tpl_connCb, waitNext);
	  pConn->flags &= ~TPL_CONN_WAIT_PACKETS;
	}
#endif /* LATER */

	/* remove any rpc queue */
	if (!(pConn->flags & TPL_CONN_FP_CONN))
		tpl_flushRpcRasm(pConn);
	
	/* send msg to appliction if application exist */
	if (pConn->flags & TPL_CONN_BAD_APP) {
	  rc = tpl_delConnCore(pConn);
	} else {

      /* Check if we have received a ACPU close before, if so
       * dont send the DEL_CONN event
       */
      if(!(pConn->flags & TPL_CONN_SENT_DEL_CONN)) {
          pConn->flags |= TPL_CONN_SENT_DEL_CONN;
	  
          /* if FP connection, then indicate to FP of connection deletion */
          if (pConn->flags & TPL_CONN_FP_CONN)
              rc = (pBind->ul_delConnInd)( pConn->pAppHdl , pConn);
          else 
              rc = tpl_ipcSendEvntMsg(TPL_EVNT_DEL_CONN, pBind, 1, pBind->vsId);
      } else {
          rc = 0;
      }
    }
    return rc;
}

/*-----------------------------------------------------------------
 * Name:		tpl_delConnReq
 * Description: Called by NFS/Mount/CIFS layer to delete a connection.
 * Returns: 0, or err
 *-----------------------------------------------------------------
 */
int
tpl_delConnReq(void *connPtr )
{
    tpl_connCb_t *pConn = connPtr;

    TPL_CONN_PRINTK(("%s: pConn=0x%p, lport %d, rport %d, pSock 0x%p\n",
		__FUNCTION__, 
		(void *)pConn, 
		ntohs(pConn->lport),
		ntohs(pConn->rport), 
		(void *)pConn->pSock));

	/*
	 * su, 07/02/02
	 * make sure pConn still in use
	 */
	if (!(pConn->flags & TPL_CONN_INUSE))
		return 0;
	
	/*-------------------------------
	 * Return, if del is pending.
	 *-------------------------------
	 */
	if ((pConn->flags & TPL_CONN_DEL_PENDING) || (pConn->flags & TPL_CONN_CLOSED)) {

	    TPL_CONN_PRINTK(("%s: pConn 0x%p, DELETE already pending!\n",
				        __FUNCTION__, (void *)pConn ));
        return 0;
	}

	/*-------------------------------
	 * Set pending flag.
	 * Call BSD to close sock.
	 * 
	 * Will get an delConnInd from BSD
	 * lower socket is closed. 
	 * Perform actual closing then.
	 *-------------------------------
	 */
	pConn->flags |= TPL_CONN_DEL_PENDING;
	if (pConn->pSock)
		linux_socket_close( pConn->pSock);

	return 0;
}

/*-----------------------------------------------------------------
 * Name:		tpl_delConnReqCore
 * Description: Called by NFS/Mount/CIFS layer to delete a connection.
 * Returns: 0, or err
 *-----------------------------------------------------------------
 */
int
tpl_delConnReqCore(void *connPtr)
{
    int rc;

    ((tpl_connCb_t *)connPtr)->acpuState = TPL_IPC_CONN_CLOSE_SENT;

    rc = tpl_ipcSendEvntMsg(TPL_EVNT_CLOSE_REQ, connPtr, 0, ((tpl_connCb_t *)connPtr)->pBindCb->vsId);
    return rc;
}

/*-----------------------------------------------------------------
 * Name:		tpl_delConnAck
 * Description: Called by NFS/Mount/CIFS layer to delete a connection.
 * Returns: 0, or err
 *-----------------------------------------------------------------
 */
int
tpl_delConnAck(void *connPtr)
{
    int rc;
    rc = tpl_ipcSendEvntMsg(TPL_EVNT_DEL_CONN_ACK, connPtr, 0, ((tpl_connCb_t *)connPtr)->pBindCb->vsId);
    return rc;
}

/*-----------------------------------------------------------------
 * Name:		tpl_delConnId
 * Description: Called by Linux tcp stack to delete a connection.
 * Returns: 0, or err
 *-----------------------------------------------------------------
 */
int
tpl_delConnInd(void *pConn )
{
	int rc;
	rc = tpl_delConn(pConn, TPL_IND);
	return rc;
}

#define	SO_CONN_UP_IND         1
#define	SO_CONN_CONNFAILED_IND 2
#define	SO_CONN_DOWN_IND       3
#define	SO_CONN_RTCHG_IND      4
#define	SO_CONN_CLOSE_REQC     5 

/*-----------------------------------------------------------------
 * Name:        tpl_bsdConnInd
 * Description: Called by Linux tcp stack to indicate a connection event.
 *              Callback for connection event
 * Returns: 0, or err
 *-----------------------------------------------------------------
 */
int
tpl_bsdConnInd(void *sock, void *pcb, int evt, void **pHdl)
{
	int rc = 0;
#ifdef LATER
	struct inpcb *pInpcb;
	tpl_connCb_t *pConn = (tpl_connCb_t*)*pHdl;
	struct sk_buff *skb;

	pInpcb = pcb;
	switch (evt) {
	/*-------------------------------
	 * Add connection event.
	 *-------------------------------
	 */
	case	SO_CONN_UP_IND:
		/*-------------------------------
		 * inpcb has all the info we need
		 * to add a TPL connection.
		 * Preserved NETWORK ORDER on
		 * addresses & ports.
		 *-------------------------------
		 */
		rc = tpl_addConnInd(pInpcb->inp_ip_p,
					(pInpcb->inp_laddr.s_addr),
					(pInpcb->inp_faddr.s_addr),
					(pInpcb->inp_lport),
					(pInpcb->inp_fport),
					pInpcb,
					pInpcb->inp_ppcb,
					sock,
					pHdl);
	
		if (rc != 0) 
			TPL_LOGERR( rc, *pHdl);
		break;

	/*------------------------------------
	 * Connection Refused/Timed-out event.
	 *------------------------------------
	 */
	case	SO_CONN_CONNFAILED_IND: 
		if (pConn->flags & TPL_CONN_FP_CONN) {
			rc = tpl_fpConnFailedInd((tpl_bindCb_t *)pConn->pBindCb);
  			if (rc != 0)
    			TPL_LOGERR( rc, *pHdl);
			break;
		}
		/* else... fall through */
		/*-------------------------------
		 * Delete connection event.
		 *-------------------------------
		 */
	case	SO_CONN_DOWN_IND:
	    /* su, 07/02/02
	     * set  TPL_CONN_DEL_PENDING
	     * so socket won't be closed twice
	     */
		if (pConn->flags & TPL_CONN_FP_CONN) {
			if (!(pConn->flags & TPL_CONN_DEL_PENDING)) {
	    		pConn->flags |= TPL_CONN_DEL_PENDING;
				rc = tpl_delConnInd(pConn);
			}
		} else {
		    	pConn->flags |= TPL_CONN_DEL_PENDING;
	    		linux_socket_close(sock); 
		}
		break;

	case	SO_CONN_RTCHG_IND:
		/* su, 05/13/02. unset NOMAC flags, and send any queue pkt */
		pConn->flags &= ~TPL_CONN_NOMAC;

		if ((skb = pConn->arpTx)) {
			pConn->arpTx = NULL;
			pTplCfg->tplStats.cntXmtMacQueue--;
			tpl_xmtPkt(pConn, skb);
		}
		break;
	
    	case	SO_CONN_CLOSE_REQC:
		/* filter out binding socket */
		if (((void*)*pHdl < (void*)pTplCfg->bindTblBase ) ||
			((void*)*pHdl > (void*)pTplCfg->bindTblEnd )) {

			rc = tpl_delConnInd( *pHdl );
			if (rc != 0)
				TPL_LOGERR( rc, *pHdl);
		}
		break; 		  

	default:
		TPL_LOGERR( TPL_ERR_BAD_EVT, *pHdl );
		break;
	}
#endif /* LATER */
    return rc;
}

/*
 * Callback for packet received indication event
 */
void
tpl_bsdRcvInd(void *sock, void *pDesc, unsigned int saddr, unsigned short sport, unsigned int len, void *hdl)
{
  return;
}

/*----------------------------------------------------------------
 * Name:  tpl_abortConn
 *
 * Description: Terminate network connection immediately. 
 *              This function is called when 
 *              1) route for an connection no longer exist(for example, IP removed)
 *              2) application issued an unbind.
 *----------------------------------------------------------------
 */
int
tpl_abortConn(tpl_connCb_t *pConn)
{
	int			rc;
	tpl_bindCb_t	*pBind;

	TPL_CONN_PRINTK(("%s: pConn 0x%p\n", __FUNCTION__, pConn));
	
	/* sanity chk */
	rc = tpl_validateConnCb(pConn);
	if (rc != 0) {
		TPL_LOGERR(rc, pConn);
		return (rc);
	}

	pBind = pConn->pBindCb;
	rc = tpl_validateBindCb(pBind);
	if (rc != 0) {
		TPL_LOGERR(rc, pConn);
		return (rc);
	}

	/* if connection is already closed,
	 * do nothing, the proper events will be
	 * generated.
	 */
	if (pConn->flags & TPL_CONN_CLOSED || pConn->flags & TPL_CONN_DEL_PENDING)
		return 0;

	/* abort network layer connection */
	(void)linux_socket_abort(pConn->pSock);

	pConn->pSock = NULL;
	
	if (tpl_abort_log_flag == TRUE) {

	  E_LOG (class_1, info_s, bsdrl_appid, 0, 0, 0,
               ("%s: Aborting connection {0x%x,0x%x,0x%x,0x%x}: RA = %p\n",
                __FUNCTION__,
                pConn->laddr,
                pConn->lport,
                pConn->raddr,
                pConn->rport,
                (void *) get_ra()));
	}

	/* close pCOnn */
	tpl_delConnInd(pConn);
	
	return 0;
}

/*----------------------------------------------------------------
 * Name:  tpl_abortAllConnForIP
 *
 * Description: Terminate network connection for this IP address immediately. 
 *              This function is called when IP interface is going down.
 *----------------------------------------------------------------
 */
void
tpl_abortAllConnForIP(unsigned int ipaddr)
{
#ifdef LATER
    struct inpcb *inp, *ninp;

    inp = curVStack->udb.lh_first;
    while (inp) {
        ninp = inp->inp_list.le_next;

        if ((inp->inp_laddr.s_addr == ipaddr) && 
            (inp->inp_faddr.s_addr != 0) &&
            (inp->inp_fport != 0))
            tpl_abortConn(inp->inp_socket->so_handle);
        inp = ninp;
    }

    inp = curVStack->tcb.lh_first;
    while (inp) {
        ninp = inp->inp_list.le_next;

        if ((inp->inp_laddr.s_addr == ipaddr) && 
            (inp->inp_faddr.s_addr != 0) &&
            (inp->inp_fport != 0))
            tpl_abortConn(inp->inp_socket->so_handle);
        inp = ninp;
    }
#endif
    return;
}

/*-----------------------------------------------------------------
 * Name: 	tpl_findConnByAppHdl
 * Description: Returns Conn CB that matches appHdl
 *-----------------------------------------------------------------
 */
tpl_connCb_t *
tpl_findConnByAppHdl(void *appHdl)
{
	tpl_connCb_t	*pConn;
	int				i;

	for (i=0; i < pTplCfg->maxConnCb; i++) {
		pConn = (tpl_connCb_t *)(&pTplCfg->connTblBase[i]);
		if (pConn->pAppHdl == appHdl)
			return (pConn);
	}
	return (NULL);
}

/*-----------------------------------------------------------------
 * Name: 	tpl_getSockError
 * Description: Returns any error associated with a TPL connection bsd socket
 *-----------------------------------------------------------------
 */
unsigned int
tpl_getSockError(tpl_connCb_t *pConn)
{
#ifdef LATER
	struct socket *sock = (pConn->pSock) ? pConn->pSock : 
			(pConn->pBindCb) ? pConn->pBindCb->pSock: NULL;
	return (sock) ? sock->so_error : 0;
#else
	return 0;
#endif
}

/*-----------------------------------------------------------------
 * Name: 	tpl_getSockUpcall
 * Description: Sets the upcall for the bsd socket to get notified
 *	when there is more space in the socket buffer
 *-----------------------------------------------------------------
 */
void
tpl_setSockUpcall(tpl_connCb_t *pConn,
	void (*upcall)(void *sock, void *arg, int unused))
{
#ifdef LATER
	upcall_calbck_t	sock_upcall = (upcall_calbck_t)upcall;

	linux_SetSockUpcall(pConn->pSock, sock_upcall, (void *)pConn);
#endif /* LATER */
	return;
}

/*-----------------------------------------------------------------
 * Name:        tpl_allocPkt
 * Description: Allocates packet from free list and associate with connection.
 *              This runs on NCPU
 *-----------------------------------------------------------------
 */
struct sk_buff * 		
tpl_allocPkt(struct socket *so, uint32 size)
{
        struct sk_buff *skb = alloc_skb(size, (__GFP_HIGHMEM | __GFP_NOFAIL | __GFP_DMA));

	if (skb) {
	  pTplCfg->tplStats.cntAllocPkt++;
	  pTplCfg->tplStats.cntOutConnAlloc++;
	} else {
	  pTplCfg->tplStats.cntNoPktRsrc++;
	}
	return skb;
}

/*-----------------------------------------------------------------
 * Name:	tpl_freePkt
 * Description: Free pkt allocated using tpl_allocPkt() back to free pkt list.
 *-----------------------------------------------------------------
 */
void
tpl_freePkt(struct sk_buff *skb)
{
#ifdef LATER
    // ASSERT(CM_IS_TXRX_NCPU);
    // ASSERT(skb->pTplNext == NULL);

    if (! (skb->flags & (PKT_USR_LUC_TX | PKT_USR_TCP_TX_Q))) {

        /* Free the packet unless it is still in the driver transmit queue.
         */
        pkt_freePkt(skb);
	pTplCfg->tplStats.cntFreePkt++;
    }
#endif /* LATER */
}

/*-----------------------------------------------------------------
 * Name: 	tpl_init
 * Description: Initialize Transport Layer data structures.
 *-----------------------------------------------------------------
 */
void 
tpl_init(void)
{
	tpl_cfg_t *pCfg;
	int allocSize = sizeof(tpl_cfg_t);

	pCfg = (tpl_cfg_t *)kmalloc(allocSize, (__GFP_HIGHMEM | __GFP_NOFAIL));
	if (pCfg == NULL) {
		TPL_LOGERR( TPL_ERR_NO_SHMEM_RSRC, pCfg );
		return;
	}
	memset((void *)pCfg, 0, allocSize);

	/*-------------------------------
	 * Allocate connection CB table out of shared memory
	 *-------------------------------
	 */
	allocSize = sizeof(tpl_any_conn_t) * (TPL_MAX_CONN_CB + 1);
	pCfg->maxConnCb = TPL_MAX_CONN_CB;

	/* XXX Get the memory cache line aligned */
	// pCfg->connTblBase = (tpl_any_conn_t *)((void *)kmalloc(allocSize) & (~CACHELINE_MASK));
	pCfg->connTblBase = (tpl_any_conn_t *)kmalloc(allocSize, (__GFP_HIGHMEM | __GFP_NOFAIL));

	if (pCfg->connTblBase == NULL) {
		TPL_LOGERR( TPL_ERR_NO_SHMEM_RSRC, pCfg );
		return;
	}
	pCfg->connTblEnd  = &pCfg->connTblBase[TPL_MAX_CONN_CB];

	/*--------------------------
	 * Clear Connection CB table
	 *--------------------------
	 */
	memset((void *)pCfg->connTblBase, 0, (allocSize - sizeof(tpl_any_conn_t)));

	/*-------------------------------
	 * Allocates bind CB table out of shared mem.
	 *-------------------------------
	 */
	allocSize = sizeof(tpl_bindCb_t)*TPL_MAX_BIND_CB;
	pCfg->maxBindCb = TPL_MAX_BIND_CB;

	pCfg->bindTblBase = (tpl_bindCb_t *)kmalloc(allocSize, (__GFP_HIGHMEM | __GFP_NOFAIL));

	if (pCfg->bindTblBase == NULL) {
		TPL_LOGERR( TPL_ERR_NO_SHMEM_RSRC, pCfg );
		return;
	}
	pCfg->bindTblEnd = pCfg->bindTblBase + TPL_MAX_BIND_CB;

	memset((void *)(pCfg->bindTblBase), 0, allocSize);

	pTplCfg = pCfg;
	return;
}

