/*
 *-----------------------------------------------------------------
 * Name:        tpl.h
 * RCS:  $Id: tpl.h,v 1.35 2002/10/29 01:21:34 swang Exp $
 * Copyright (C) 2009, OnStor, Inc.
 * Description: Transport Layer internal data definitions.
 *-----------------------------------------------------------------
 */
#ifndef _TPL_H
#define _TPL_H

#include "pkt-queue-struct.h"

#ifdef LATER
#include "rpc/rpc.h"
#include "../sm-nfs/nfs-conn-struct.h"
#include "../sm-cifs/cifs-conn-struct.h"
#endif

struct socket;
 
/* Maximum number of client requests that can be executed on TxRx.
 * 
 * In the worst case of 64K write request a single request requires
 * ~44 buffers. 4K TCP connections with 64K receive window
 * require another 44 buffers each.
 *
 * The number of buffers on TxRx is ~650000 for 2GB configuration
 * the number below is chosen to not exceed this limit in the worst
 * case of maximum pending requests and all connection buffers used.
 */
//@@@; this needs to be reworked, or be made platform dependent.
#define TXRX_MAX_PENDING_REQUESTS 5000

/* The number of requests currently executing on TxRx.
 */
extern atomic_t txrx_requests_pending;

/* The number of requests dropped because the pending requests have
 * exceeded the limit
 */
extern unsigned int txrx_requests_dropped;

#define TPL_SIM				1	/* turn on TPL Simulation */
#define TPL_DEBUG			1

#define TPL_TIMER_INTERVAL	10000	/* timer function invoked every 10 sec */

#define TPL_LOGERR( code, info )	tpl_logErr( __FILE__, __LINE__, code, info)
#define TPL_PANIC( )				{ *(char *)0 = 1; }

/*------------------------------------
 * Some TPL Max. Needs to be increased
 * in  'real' system.
 *------------------------------------
 */
#define TPL_MAX_CONN_CB		 NFX_MAX_CONN
#define TPL_MAX_BIND_CB		 1024*24	/*  1024*24 Binding CB max. */

/* The maximum length of RPC response. This is set to 66K to allow 64K write requests.
 * We have the limit to catch completely bogus client RPCs.
 */
#define TPL_MAX_RPC_LEN		(66*1024)

/* Indexes in times array.
 */
typedef enum tpl_time_idx {
    /* The time to process a received packet.
     */
    TPL_TIME_RX,
    TPL_TIME_MAX
} tpl_time_idx;

/*------------------------------------
 * TPL Statitics counters.
 *------------------------------------
 */
typedef struct {
	/* separate acpu stats from ncpu stats */
        // defined(NFX_SMP)
	unsigned int	cntAcpuAllocPkt;
	unsigned int	cntAcpuOutConnAlloc;
	unsigned int	cntAcpuFreePkt;
	unsigned int    cntIpcRxRcv;
	unsigned int    cntIpcRxDrop;
	unsigned int    cntIpcTxFwd;
	unsigned int    cntIpcTxDrop;
	unsigned int    cntIpcFreeFwd;
	unsigned int    cntIpcFreeDrop;
	unsigned int    cntIpcCompRcv;
	unsigned int    cntIpcCompDrop;
	unsigned int	AtoNMsgCnt;		/* cnt of AtoN evnt msgs currently allocated */
        // endif NFX_SMP
	unsigned int	cntBadBindErr;	/* count invalid Bind CB error */
	unsigned int	cntBadConnErr;	/* count invalid Conn CB error */
	unsigned int	cntBadPktErr;	/* count invalid Pkt Desc error */
	unsigned int	cntBadLinkErr;	/* count invalid Link Id error */
	unsigned int	cntNoCamRsrc;	/* count no more CAM entry */
	unsigned int	cntNoPktRsrc;	/* count no more free PKT entry */
	unsigned int	cntUlAddFail;	/* count Upper layer add fail */
	unsigned int	cntUlDelFail;	/* count Upper layer delete fail */
	unsigned int	cntBindReq;		/* count bind req. rcvd */
	unsigned int	cntUnbindReq;	/* count unbind req. rcvd */
	unsigned int	cntAddConnInd;	/* count add Conn Ind rcvd */
	unsigned int	cntDelConnInd;	/* count del Conn Ind rcvd */
	unsigned int	cntDelConnReq;	/* count del Conn Req rcvd */
	unsigned int	cntAllocPkt;	/* count tpl pkt alloc */
	unsigned int	cntFreePkt;		/* count tpl pkt free */
	unsigned int	cntOutConnAlloc;/* count out of conn. alloc */
	unsigned int	cntOutHWater;
	unsigned int	cntRcvPkt;		/* count tpl pkt rcv */
	unsigned int	cntIpCsumErr;	/* count IP checksum error rcvd*/
	unsigned int	cntUdpCsumErr;	/* count UDP checksum error rcvd*/
	unsigned int	cntTcpCsumErr;	/* count TCP checksum error rcvd*/
	unsigned int	cntXmtPkt;		/* count tpl pkt xmt */
	unsigned int	cntXmtCmp;		/* count tpl xmt complete*/
	unsigned int	cntXmtUpCall;	/* count tpl xmt complete Up Calls*/
	unsigned int	cntXmtDownCall;	/* count tpl pkt send from application */
	unsigned int	cntXmtFail;		/* count tpl xmt failure*/
	unsigned int	cntInTcpXmt;	/* count desc with PKT_USR_TCP_TRANSIT set*/
	unsigned int	cntInTcpTxQ;	/* count desc with PKT_USR_TCP_TX_Q set */
	unsigned int	cntRpcLenErr;	/* count length error in RPC Marker */
	unsigned int	cntRpcRasmErr;	/* count length error in RPC Marker */
	unsigned int	cntXmtNoMacErr;	/* application pkt dropped because no mac */
	unsigned int	cntXmtMacQueue;	/* num. of pkt waiting in mac queue */
	unsigned int	cntXmtMacUncall;/* mac is available, yet there was no upcall */
	unsigned int	cntXmtDup;	    /* count dup. pkt send by application */
	unsigned int	cntXmtNoTplUsr; /* count pkt with PKT_USR_TPL_TX not set */
	unsigned int	cntXmtTcpDup;   /* tcp attempt to send pkts still in luc */
	unsigned int	cntXmtTcpErr;   /* fast path tcp send error */
	unsigned int	cntXmtTcpQueue;	/* num. of pkt in tcp send queue */
	unsigned int	cntConnLinkChge;/* conn. update due to link change */
	unsigned int	cntPktLinkChge;	/* pkt update due to link change */
        // defined(NFX_SMP)
	unsigned int    cntIpcRxFwd;
	unsigned int    cntIpcTxRcv;
	unsigned int    cntIpcFreeRcv;
	unsigned int    cntIpcCompFwd;
	unsigned int	NtoAMsgCnt;		/* cnt of NtoA evnt msgs currently allocated */
	unsigned int	msgAllocFail;	/* message allocation failed */
	unsigned int	cntFpRxFwd;		/* number of packes forwarded to FP */
	unsigned int	cntFpTxRcv;		/* number of packets recd. from FP for transmission */
        // endif NFX_SMP
	unsigned int	cntPktDropFail;		/* failures in pkt drop due to bad pkt */

    /* Packet processing times.
     */
#ifdef LATER
    kpi_time_count tpl_times_counts[TPL_TIME_MAX];
#endif
} tpl_stats_t;

/* su, 05/03/02, pending queue when no mac avail */
typedef struct tpl_pend {
	struct tpl_pend	*next;
	struct tpl_pend *prev;
	tpl_connCb_t	*pConn;
	int             expire;     /* counter to expire pConn */
} tpl_pend_t;	

/* Union of all know connection types.
 */
typedef union __attribute__((transparent_union)) {
#ifdef LATER
    struct acpu_conn v;
    struct cifs_conn cifs;
    struct nfs_conn  nfs;
#else
    void *acpu_conn_v;
    void *cifs_conn_cifs;
    void *nfs_conn_nfs;
#endif /* LATER */
} tpl_any_conn_t;

/*------------------------------------
 * TPL Configuration Control Block
 *------------------------------------
 */
#define TPL_IPC_QUEUE_CNT           3
typedef struct {
        tpl_ipcq_t tplIpcQ[TPL_IPC_QUEUE_CNT];	/* TPL_IPC_Q_CNT same but defined in tpl-ipc.h */

        tpl_bindCb_t    *bindTblBase;	/* bind CB table base */
        tpl_bindCb_t    *bindTblEnd;	/* bind CB table end */

        tpl_any_conn_t  *connTblBase;	/* connection CB table base */
        tpl_any_conn_t  *connTblEnd;	/* connection CB table end */
        unsigned int          numConnCb;		/* num of conn. cb in use*/
        unsigned int          numBindCb;		/* num of bind. cb in use*/
        unsigned int          maxConnCb;		/* max. num of conn. cb */
        unsigned int          maxBindCb;		/* max. num of bind. cb */
	int		      tplTimer;	    /* timer handle  */
	int		      tplAppId;		/* tpl application id */
	unsigned int	      tplAddr;		/* tpl application address for EEE communication */
	unsigned int          pad;
	tpl_any_conn_t	*next_rcycConn; /* next conn when recycle func. run again */
	tpl_stats_t	tplStats;		/* TPL I/F stats */
} tpl_cfg_t;

int tpl_bsdConnInd(void *sock, void *pcb, int evt, void **pHdl);
void tpl_bsdRcvInd(void *sock, void *pDesc, unsigned int saddr, unsigned short sport, unsigned int len, void *hdl);

void tpl_logErr(char *fileName, int lineNum, int errCode, void *info);
void tpl_showCfg(void);
void tpl_showStats(void);
void tpl_showBind(unsigned short lport);
void tpl_showConn(unsigned short lport);
void tpl_unbindPort(unsigned short lport);
void tpl_clearStats(void);

tpl_bindCb_t * tpl_findBindCb( unsigned short prot, unsigned short lport, unsigned int laddr, unsigned int vsId);
tpl_bindCb_t * tpl_findBindCbByUsrInfo( void *usrInfo);

int tpl_validateConnCb( void *pConn );
int tpl_validateBindCb( tpl_bindCb_t *pBind );

tpl_connCb_t * tpl_findConnByAppHdl( void *appHdl);
void tpl_abortAllConnForIP(unsigned int ipaddr);
int tpl_abortConn(tpl_connCb_t *pConn);
void  tpl_recycleConn(void);
int tpl_delConnInd( void *pConn );
unsigned int tpl_getSockError( tpl_connCb_t *pConn);
void tpl_setSockUpcall(tpl_connCb_t *pConn, void (*upcall)(void *sock, void *arg, int unused));

#ifdef LATER
/* if defined(NFX_SMP) */
int tpl_delConnReqCore(void *connPtr);
int tpl_bindReqCore(tpl_bindCb_t *pReq, tpl_bindCb_t **pRsp, int bindOpt);
int tpl_unbindReqCore(tpl_bindCb_t *pBind);
int tpl_listenReqCore(tpl_bindCb_t *pBind, unsigned int laddr, uint16 lport);
int tpl_connectReqCore(tpl_bindCb_t *pBind, unsigned int raddr, uint16 rport);
int tpl_delConnCore(tpl_connCb_t *pconn);
int tpl_timerFunc(unsigned int handle, void *parm1, void *parm2, void *parm3, void *parm4);

void tpl_receiveMsg(eee_desc_t *edesc);
boolean tpl_fpSpaceAvailable(void *pAppHdl, struct sk_buff *pDesc);
int tpl_fpConnFailedInd(tpl_bindCb_t *pBind);

/* endif => NFX_SMP */
#endif /* LATER */

void tpl_flushRpcRasm(tpl_connCb_t *pConn);
void tpl_xmt(tpl_connCb_t *conn, struct socket *so, struct sk_buff *pDesc);

/*
 *
 * Routine Description:
 *  Send the packet to the application on ACPU to process.  If the application 
 *  has not yet acknowledged the connection establishment, queue the packet 
 *  to be send later.
 * Arguments:
 *   pConn - the TPL connection
 *   pDesc - the packet descriptor
 * Return Values:
 *   None.
 */
void
tpl_givePktToApp(tpl_connCb_t *pConn, struct sk_buff *pDesc);

/*
 * Routine Description:
 *  Free the bind control block.
 * Arguments:
 *  pBind - the TPL bind CB structure to be freed
 * Return Values:
 *  None.
 */
void
tpl_freeBindCb(tpl_bindCb_t *pBind);

/*
 * Description:
 * Finish the request processing on the ACPU. Decrement the number of pending
 * requests and notify the NCPU if the number of requests has dropped below
 * the maximum and there are blocked connections on the NCPU side.
 */
void
txrx_request_complete(void);

#endif
