/*
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2001, 2002, 2004 Ralf Baechle
 * Copyright (C) 2007 Onstor Inc.
 *   Andrew Sharp (andy.sharp@onstor.com)
 */


#include <linux/init.h>
#include <linux/console.h>
#include <linux/kdev_t.h>
#include <linux/major.h>
#include <linux/termios.h>
#include <linux/sched.h>
#include <linux/tty.h>
#include <linux/delay.h>

#include <linux/serial.h>
#include <linux/serial_core.h>
#include <asm/serial.h>
#include <asm/io.h>
#include <asm/mach-bobcat/bobcat.h>

extern unsigned long uart_base;

#ifdef CONFIG_BOBCAT_8250_DEBUG_CONSOLE
struct bc_debug_uart {
	union {
		struct {
			uint8_t		bu_thr;		/* xmit holding reg */
			uint8_t		bu_ier;		/* int enable reg */
			uint8_t		bu_fcr;		/* fifo ctrl reg */
			uint8_t		bu_lcr;		/* line ctrl reg */
			uint8_t		bu_mcr;		/* modem ctrl reg */
		} bu_xmit_regs;
		struct {
			uint8_t		bu_rhr;		/* recv holding reg */
			uint8_t		nada1;		/* not used */
			uint8_t		bu_isr;		/* int status reg */
			uint8_t		nada2;		/* not used */
			uint8_t		nada3;		/* not used */
			uint8_t		bu_lsr;		/* line status reg */
			uint8_t		bu_msr;		/* modem status reg */
		} bu_recv_regs;
		struct {
			uint8_t		bu_div_lsb;		/* LSB of divisor latch */
			uint8_t		bu_div_msb;		/* MSB of divisor latch */
			uint8_t		bu_alt_fr;		/* alternate function register */
		} secondary;
	} bu_regs;
	uint8_t		bu_scr;		/* scratchpad reg */
};

#define BU_ISR(a)	((struct bc_debug_uart *)(a))->bu_regs.bu_recv_regs.bu_isr
#define BU_LSR(a)	((struct bc_debug_uart *)(a))->bu_regs.bu_recv_regs.bu_lsr
#define BU_LCR(a)	((struct bc_debug_uart *)(a))->bu_regs.bu_xmit_regs.bu_lcr
#define BU_MCR(a)	((struct bc_debug_uart *)(a))->bu_regs.bu_xmit_regs.bu_mcr
#define BU_IER(a)	((struct bc_debug_uart *)(a))->bu_regs.bu_xmit_regs.bu_ier
#define BU_FCR(a)	((struct bc_debug_uart *)(a))->bu_regs.secondary.bu_alt_fr
#define BU_LSB(a)	((struct bc_debug_uart *)(a))->bu_regs.secondary.bu_div_lsb
#define BU_MSB(a)	((struct bc_debug_uart *)(a))->bu_regs.secondary.bu_div_msb
#define BU_THR(a)	((struct bc_debug_uart *)(a))->bu_regs.bu_xmit_regs.bu_thr
#define BU_RHR(a)	BU_THR(a)

#define         UART16550_READ(y)    (*((volatile uint8_t *)(&y)))
#define         UART16550_WRITE(y, z)  ((*((volatile uint8_t *)(&y))) = z)

 static int //__init
bc_8250_setup(struct console *con, char *cmd_str)
{
	int foo;

	/*
	 * Set the uart_base address in the private data area for the console.
	 * Set up the fifo and interrupts (we don't have interrupts for the
	 * 16552) and buad rate if necessary.
	 */
	con->data = (void *)uart_base;

	if (con->data) {
		/*
		 * clear any pending interrupts and disable interrupts
		 */
		UART16550_WRITE(BU_IER(con->data), 0); /* disable interrupts */

		foo = UART16550_READ(BU_ISR(con->data));
		foo = UART16550_READ(BU_LSR(con->data));

		foo = UART16550_READ(BU_ISR(con->data)); /* clear ISR */
		UART16550_WRITE(BU_FCR(con->data), 7); /* reset and enable fifos */
		udelay(100);

		/*
		 * set baud rate of 57600
		 * first, set bit in LCR to use special regs
		 * second, divisor is 2500000, formula is (2500000/(16 * buad))
		 * set in LSB and MSB, then undo bit in LCR
		 */
		UART16550_WRITE(BU_LCR(con->data), 0x80);
		UART16550_WRITE(BU_LSB(con->data), 0x1b);
		UART16550_WRITE(BU_MSB(con->data), 0);
		UART16550_WRITE(BU_LCR(con->data), 0); /* back to regular regs */

		UART16550_WRITE(BU_LCR(con->data), 3); /* 8 bit, no parity */
		//UART16550_WRITE(BU_MCR(con->data), 3); /* DTR | RTS */

		UART16550_WRITE(BU_IER(con->data), 0); /* disable interrupts */

	}

	return con->data ? 0 : -ENODEV;
}

 void
bc_8250_puts(struct console *con, const char *s, unsigned n)
{
	char *c = (char *)s;

	while (n--) {
		while ((UART16550_READ(BU_LSR(con->data)) & 0x20) == 0) ;
		if (*c == '\n') {
			UART16550_WRITE(BU_THR(con->data), '\r');
		}
		/*
		 * device has a fifo, so we don't need to wait before
		 * writing the second char
		 */
		UART16550_WRITE(BU_THR(con->data), *c++);
	}
}

 static int
bc_8250_gets(struct console *con, char *s, unsigned int n)
{
	int i;

	for (i = 0; i < n; i++) {
		while ((UART16550_READ(BU_LSR(con->data)) & 0x1) == 0) ;
		s[i] = UART16550_READ(BU_RHR(con->data));
	}

	return i;
}

static struct console pcon = {
	.name = "ttyS",
	.write = bc_8250_puts,
	.read = bc_8250_gets,
	.setup = bc_8250_setup,
	.flags = CON_PRINTBUFFER,
	.index = -1,
};

 void __init
bc_8250_console_setup(void)
{
	register_console(&pcon);

	printk("Enabling early printk on 8250 console\n");
}

 void __init
disable_early_printk(void)
{
	printk("Disabling early printk on 8250 port\n");
	unregister_console(&pcon);
}

#endif /* CONFIG_BOBCAT_8250_DEBUG_CONSOLE */

#ifdef CONFIG_BOBCAT_MPSC_EARLY_PRINTK
#define	MPSC_CHR_1			0x00c
#define	MPSC_CHR_2			0x010
#define MPSC_CHR_4			0x0018
#define MPSC_CHR_5			0x001c
#define	MPSC_CHR_2_TTCS		0x200
#define MPSC_CHR_5_V		0x8000
#define MPSC_CHR_4_Z		0x20000000
#define MPSC_CHR_10			0x30
#define MPSC_CHR_10_RCRn_mask (0xff << 16)

static volatile char *muart_base = (volatile char *)0xbc008000;

 static void
mpsc_early_putc(char c)
{
	unsigned long r;

	writel(c, muart_base + MPSC_CHR_1);
	r = readl(muart_base + MPSC_CHR_2);
	writel(r | MPSC_CHR_2_TTCS, muart_base + MPSC_CHR_2);
	while (readl(muart_base + MPSC_CHR_2) & MPSC_CHR_2_TTCS) ;
}

 static void
mpsc_early_puts(struct console *con, const char *s, unsigned n)
{
	char *c = (char *)s;

	while (n--) {
		mpsc_early_putc(*c);
		if (*c == '\n') {
			mpsc_early_putc('\r');
		}
		c++;
	}
}

 static char
getDebugChar(void)
{
	unsigned long r;
	char c;

	while (((r = readl(muart_base + MPSC_CHR_10)) &
				MPSC_CHR_10_RCRn_mask) == 0);

	/*
	 * clear the received character in MPSC_CHR_10.
	 */
	writel((r | MPSC_CHR_10_RCRn_mask), muart_base + MPSC_CHR_10);

	c = (r & MPSC_CHR_10_RCRn_mask) >> 16;
	return c;
}

 static int
mpsc_early_gets(struct console *con, char *s, unsigned int n)
{
	int i;

	for (i = 0; i < n; i++) {
		s[i] = getDebugChar();
	}

	return i;
}


 static int
mpsc_early_setup(struct console *con, char *options)
{
	writel(MPSC_CHR_4_Z, muart_base + MPSC_CHR_4);
	writel(MPSC_CHR_5_V, muart_base + MPSC_CHR_5);

	return 0;
}

static struct console mpsc_early_console = {
	.name	= "ttyMM",
	.write	= mpsc_early_puts,
	.read	= mpsc_early_gets,
	.setup	= mpsc_early_setup,
	.flags	= CON_PRINTBUFFER | CON_BOOT,
	.index	= -1
};

 void __init
mpsc_early_printk_init(void)
{
	return register_console(&mpsc_early_console);

	printk("Registered MPSC early console\n");
}

 void __init
disable_early_printk(void)
{
	printk("Disabling early printk on MPSC port\n");

	writel(0, muart_base + MPSC_CHR_4);
	writel(0, muart_base + MPSC_CHR_5);

	unregister_console(&mpsc_early_console);
}
#endif /* CONFIG_BOBCAT_MPSC_EARLY_PRINTK */
