/*
 * Dallas Semiconductor 1511 Real Time Clock interface for Linux.
 *
 * Copyright (C) 2005 Onstor, Inc. 2005
 */
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/miscdevice.h>
#include <linux/time.h>
#include <linux/bcd.h>

#include <linux/rtc.h>
#include <asm/time.h>
#include <asm/addrspace.h>
#include <linux/proc_fs.h>

#include <asm/debug.h>
#include <asm/mach-bobcat/bobcat.h>

static unsigned long ds1511_base;

static int wd_sec = 0;
static int wd_msec = 0;
static int wd_active_flag = 0;
static int wd_reset_flag = 0;
static int wd_kickonce_flag = 0;
static int wd_timer_initialized = 0;
static struct timer_list ds1511_wdtimer;

#include "ds1511.h"

#define DS1511_VERSION	"1.1"

static DECLARE_WAIT_QUEUE_HEAD(ds1511_wait);

static ssize_t ds1511_read(struct file *file, char *buf, size_t count,
		loff_t *ppos);

static int ds1511_ioctl(struct inode *inode, struct file *file,
			unsigned int cmd, unsigned long arg);

static unsigned int ds1511_poll(struct file *file, poll_table *wait);

static void ds1511_get_alm_time (struct rtc_time *alm_tm);
static void ds1511_get_rtctime(struct rtc_time *rtc_tm);
static void ds1511_kick_wdtimer(unsigned long);
static int ds1511_set_rtctime(struct rtc_time *rtc_tm);

static DEFINE_SPINLOCK(ds1511_lock);

static int ds1511_read_proc(char *page, char **start, off_t off,
                            int count, int *eof, void *data);

/*
 *	Bits in rtc_status. (7 bits of room for future expansion)
 */

#define RTC_IS_OPEN		0x01	/* means /dev/rtc is in use	*/
#define RTC_TIMER_ON		0x02	/* missed irq timer active	*/
#define RTC_WD_ENABLED		0x04	/* user-level watchdog enabled */

static unsigned char ds1511_status;

static unsigned char days_in_mo[] = {
	0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
};

 static inline void
rtc_write(uint8_t val, uint32_t reg)
{
	*((uint8_t *)(ds1511_base + reg * 8)) = val;
}

 static inline void
rtc_write_alarm(uint8_t val, ds1511reg_t reg)
{
	rtc_write((val | 0x80), reg);
}

 static inline uint8_t
rtc_read(ds1511reg_t reg)
{
	return (*((uint8_t *)(ds1511_base + reg * 8)));
}

 static inline void
rtc_disable_update(void)
{
	rtc_write((rtc_read(RTC_CMD) & ~RTC_DATE_SET), RTC_CMD);
}

 static inline void
rtc_enable_update(void)
{
	rtc_write((rtc_read(RTC_CMD) | RTC_DATE_SET), RTC_CMD);
}

/*
 *	Now all the various file operations that we export.
 */

 static ssize_t
ds1511_read(struct file *file, char *buf, size_t count, loff_t *ppos)
{
	return -EIO;
}

 static int
ds1511_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		unsigned long arg)
{
	struct rtc_time wtime;

	switch (cmd) {
	case RTC_AIE_OFF:	/* Mask alarm int. enab. bit	*/
	{
		unsigned int flags;
		unsigned char val;

		if (!capable(CAP_SYS_TIME))
			return -EACCES;

		spin_lock_irqsave(&ds1511_lock, flags);
		val = rtc_read(RTC_CMD);
		val &=  ~RTC_TIE;
		rtc_write(val, RTC_CMD);
		spin_unlock_irqrestore(&ds1511_lock, flags);

		return 0;
	}
	case RTC_AIE_ON:	/* Allow alarm interrupts.	*/
	{
		unsigned int flags;
		unsigned char val;

		if (!capable(CAP_SYS_TIME))
			return -EACCES;

		spin_lock_irqsave(&ds1511_lock, flags);
		val = rtc_read(RTC_CMD);
		val |=  RTC_TIE;
		rtc_write(val, RTC_CMD);
		spin_unlock_irqrestore(&ds1511_lock, flags);

		return 0;
	}
	case RTC_WIE_OFF:	/* Mask watchdog int. enab. bit	*/
	{
		unsigned int flags;

		if (!capable(CAP_SYS_TIME))
			return -EACCES;

		spin_lock_irqsave(&ds1511_lock, flags);
		ds1511_status &= ~RTC_WD_ENABLED;
		spin_unlock_irqrestore(&ds1511_lock, flags);

		return 0;
	}
	case RTC_WIE_ON:	/* Allow watchdog interrupts.	*/
	{
		unsigned int flags;

		if (!capable(CAP_SYS_TIME))
			return -EACCES;

		spin_lock_irqsave(&ds1511_lock, flags);
		ds1511_status |= RTC_WD_ENABLED;
		spin_unlock_irqrestore(&ds1511_lock, flags);

		return 0;
	}
	case RTC_ALM_READ:	/* Read the present alarm time */
	{
		/*
		 * This returns a struct rtc_time. Reading >= 0xc0
		 * means "don't care" or "match all". Only the tm_hour,
		 * tm_min, and tm_sec values are filled in.
		 */

		memset(&wtime, 0, sizeof(wtime));
		ds1511_get_alm_time(&wtime);
		break;
	}
	case RTC_ALM_SET:	/* Store a time into the alarm */
	{
		/*
		 * This expects a struct rtc_time. Writing 0xff means
		 * "don't care" or "match all". Only the tm_hour,
		 * tm_min and tm_sec are used.
		 */
		unsigned char hrs, min, sec;
		struct rtc_time alm_tm;

		if (!capable(CAP_SYS_TIME))
			return -EACCES;

		if (copy_from_user(&alm_tm, (struct rtc_time*)arg,
				   sizeof(struct rtc_time)))
			return -EFAULT;

		hrs = alm_tm.tm_hour;
		min = alm_tm.tm_min;
		sec = alm_tm.tm_sec;

		if (hrs >= 24)
			hrs = 0xff;

		if (min >= 60)
			min = 0xff;

		sec = BIN2BCD(sec);
		min = BIN2BCD(min);
		hrs = BIN2BCD(hrs);

		spin_lock(&ds1511_lock);
		rtc_write_alarm(sec, RTC_ALARM_SEC);
		rtc_write_alarm(min, RTC_ALARM_MIN);
		rtc_write_alarm(hrs, RTC_ALARM_HOUR);
		rtc_write_alarm(0, RTC_ALARM_DATE);
		spin_unlock(&ds1511_lock);

		return 0;
	}
	case RTC_RD_TIME:	/* Read the time/date from RTC	*/
	{
		memset(&wtime, 0, sizeof(wtime));
		ds1511_get_rtctime(&wtime);
		break;
	}
	case RTC_SET_TIME:	/* Set the RTC */
	{
		if (!capable(CAP_SYS_TIME))
			return -EACCES;

		if (copy_from_user(&wtime, (struct rtc_time*)arg,
				   sizeof(struct rtc_time)))
			return -EFAULT;

		return ds1511_set_rtctime(&wtime);
	}
	default:
		return -EINVAL;
	}
	return copy_to_user((void *)arg, &wtime, sizeof wtime) ? -EFAULT : 0;
}

/*
 *	We enforce only one user at a time here with the open/close.
 *	Also clear the previous interrupt data on an open, and clean
 *	up things on a close.
 */

 static int
ds1511_open(struct inode *inode, struct file *file)
{
	spin_lock_irq(&ds1511_lock);

	if (ds1511_status & RTC_IS_OPEN)
		goto out_busy;

	ds1511_status |= RTC_IS_OPEN;

	spin_unlock_irq(&ds1511_lock);
	return 0;

out_busy:
	spin_lock_irq(&ds1511_lock);
	return -EBUSY;
}

 static int
ds1511_release(struct inode *inode, struct file *file)
{
	ds1511_status &= ~RTC_IS_OPEN;

	return 0;
}

 static unsigned int
ds1511_poll(struct file *file, poll_table *wait)
{
	poll_wait(file, &ds1511_wait, wait);

	return 0;
}

/*
 *	The various file operations we support.
 */

static struct file_operations ds1511_fops = {
	.llseek		= no_llseek,
	.read		= ds1511_read,
	.poll		= ds1511_poll,
	.ioctl		= ds1511_ioctl,
	.open		= ds1511_open,
	.release	= ds1511_release,
};

static struct miscdevice ds1511_dev=
{
	.minor	= RTC_MINOR,
	.name	= "rtc",
	.fops	= &ds1511_fops,
};

//static rtc_device *ds1511_rtc;

 static int __init
ds1511_init(void)
{
	int err;

	printk(KERN_INFO "DS1511 Real Time Clock Driver v%s\n", DS1511_VERSION);

	if (!ds1511_base) {
		ds1511_base = (unsigned long)ioremap(BC_RTC_BASE_PADDR,
			BOBCAT_RTC_SIZE);
		if (!ds1511_base) {
			goto out;
		}
	}

	//rtc = rtc_device_register("ds1511", &ds1511_dev, ds1511_ops, THIS_MODULE);
	err = misc_register(&ds1511_dev);
	if (err)
		goto out;

	if (!create_proc_read_entry("driver/rtc", 0, 0, ds1511_read_proc, NULL)) {
		err = -ENOMEM;
		goto out_deregister;
	}

	return 0;

out_deregister:
	misc_deregister(&ds1511_dev);

out:
	return err;
}

 static void __exit
ds1511_exit(void)
{
	remove_proc_entry("driver/rtc", NULL);
	misc_deregister(&ds1511_dev);
}

static char *days[] = {
	"***", "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};

/*
 *	Info exported via "/proc/driver/rtc".
 */
 static int
ds1511_proc_output(char *buf)
{
	char *p, *s;
	struct rtc_time tm;
	unsigned char month, cmd, cmd1, amode;

	p = buf;

	ds1511_get_rtctime(&tm);

	p += sprintf(p,
	             "rtc_time\t: %02d:%02d:%02d\n"
	             "rtc_date\t: %04d-%02d-%02d\n",
		     tm.tm_hour, tm.tm_min, tm.tm_sec,
		     tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday);

	/*
	 * We implicitly assume 24hr mode here. Alarm values >= 0xc0 will
	 * match any value for that particular field. Values that are
	 * greater than a valid time, but less than 0xc0 shouldn't appear.
	 */
	ds1511_get_alm_time(&tm);
	p += sprintf(p, "alarm\t\t: %s ", days[tm.tm_wday]);
	if (tm.tm_hour <= 24)
		p += sprintf(p, "%02d:", tm.tm_hour);
	else
		p += sprintf(p, "**:");

	if (tm.tm_min <= 59)
		p += sprintf(p, "%02d\n", tm.tm_min);
	else
		p += sprintf(p, "**\n");

	month = rtc_read(RTC_MON);

	amode = ((rtc_read(RTC_ALARM_MIN) & 0x80) >> 5) |
	        ((rtc_read(RTC_ALARM_HOUR) & 0x80) >> 6) |
	        ((rtc_read(RTC_ALARM_DATE) & 0x80) >> 7);
	if (amode == 7)      s = "each minute";
	else if (amode == 3) s = "minutes match";
	else if (amode == 1) s = "hours and minutes match";
	else if (amode == 0) s = "days, hours and minutes match";
	else                 s = "invalid";
	p += sprintf(p, "alarm_mode\t: %s\n", s);

	cmd = rtc_read(RTC_CMD);
	cmd1 = rtc_read(RTC_CMD1);
	p += sprintf(p,
	             "user-level wdog_alarm\t: %s\n"
	             "alarm_mask\t: %s\n"
	             "wdog_alarm_mask\t: %s\n",
		     (ds1511_status & RTC_WD_ENABLED)
		          ? "enabled" : "disabled",
		     (cmd & (RTC_RESET_ENABLE | RTC_COLDRESET_ENABLE))
		          ? "yes" : "no",
		     (cmd & RTC_WDE) ? "disabled" : "enabled");

	return  p - buf;
}

 static int
ds1511_read_proc(char *page, char **start, off_t off, int count, int *eof,
		void *data)
{
	int len = ds1511_proc_output (page);

	if (len <= off+count)
		*eof = 1;
	*start = page + off;
	len -= off;
	if (len>count)
		len = count;
	if (len<0)
		len = 0;

	return len;
}

/*
 * Initialize the clock's control registers.
 */
 unsigned long
ds1511_rtc_init(void)
{
	if (!ds1511_base) {
		ds1511_base = (unsigned long)ioremap(BC_RTC_BASE_PADDR,
			BOBCAT_RTC_SIZE);
		if (!ds1511_base) {
			return 0;
		}
	}
	rtc_write(0, RTC_CMD);
	rtc_write(0, RTC_CMD1);
	rtc_enable_update();

	return ds1511_base;
}

 static void
ds1511_get_rtctime(struct rtc_time *rtc_tm)
{
	unsigned int flags;
	int century;

	/*
	 * Only the values that we read from the RTC are set. We leave
	 * tm_wday, tm_yday and tm_isdst untouched. Even though the
	 * RTC has RTC_DAY_OF_WEEK, we ignore it, as it is only updated
	 * by the RTC when initially set to a non-zero value.
	 */
	spin_lock_irqsave(&ds1511_lock, flags);
	rtc_disable_update();

	rtc_tm->tm_sec = rtc_read(RTC_SEC) & 0x7f;
	rtc_tm->tm_min = rtc_read(RTC_MIN) & 0x7f;
	rtc_tm->tm_hour = rtc_read(RTC_HOUR) & 0x3f;
	rtc_tm->tm_mday = rtc_read(RTC_DOM) & 0x3f;
	rtc_tm->tm_wday = rtc_read(RTC_DOW) & 0x7;
	rtc_tm->tm_mon = rtc_read(RTC_MON) & 0x1f;
	rtc_tm->tm_year = rtc_read(RTC_YEAR) & 0x7f;
	century = rtc_read(RTC_CENTURY);

	rtc_enable_update();
	spin_unlock_irqrestore(&ds1511_lock, flags);

	rtc_tm->tm_sec = BCD2BIN(rtc_tm->tm_sec);
	rtc_tm->tm_min = BCD2BIN(rtc_tm->tm_min);
	rtc_tm->tm_hour = BCD2BIN(rtc_tm->tm_hour);
	rtc_tm->tm_mday = BCD2BIN(rtc_tm->tm_mday);
	rtc_tm->tm_wday = BCD2BIN(rtc_tm->tm_wday);
	rtc_tm->tm_mon = BCD2BIN(rtc_tm->tm_mon);
	rtc_tm->tm_year = BCD2BIN(rtc_tm->tm_year);
	century = BCD2BIN(century) * 100;

	/*
	 * Account for differences between how the RTC uses the values
	 * and how they are defined in a struct rtc_time;
	 */
	century += rtc_tm->tm_year;
	rtc_tm->tm_year = century - 1900;

	rtc_tm->tm_mon--;
}

/*
 * set the rtc chip's idea of the time
 * stupidly, some callers call with year unmolested; and some call with
 * year = year - 1900.
 */
 static int
ds1511_set_rtctime(struct rtc_time *rtc_tm)
{
	u8 mon, day, dow, hrs, min, sec, yrs, cen, leap_yr;
	unsigned int flags;

	if (rtc_tm->tm_year < 1900) {
		rtc_tm->tm_year += 1900;
	}
	if (rtc_tm->tm_year < 1970) {
		return -EINVAL;
	}
	leap_yr = ((!(rtc_tm->tm_year % 4) && (rtc_tm->tm_year % 100)) ||
			!(rtc_tm->tm_year % 400));
	yrs = rtc_tm->tm_year % 100;
	cen = rtc_tm->tm_year / 100;
	mon = rtc_tm->tm_mon + 1;   /* tm_mon starts at zero */
	day = rtc_tm->tm_mday;
	dow = rtc_tm->tm_wday & 0x7; /* automatic BCD */
	hrs = rtc_tm->tm_hour;
	min = rtc_tm->tm_min;
	sec = rtc_tm->tm_sec;

	if ((mon > 12) || (day == 0))
		return -EINVAL;

	if (day > (days_in_mo[mon] + ((mon == 2) && leap_yr)))
		return -EINVAL;

	if ((hrs >= 24) || (min >= 60) || (sec >= 60))
		return -EINVAL;

	/*
	 * each register is a different number of valid bits
	 */
	sec = BIN2BCD(sec) & 0x7f;
	min = BIN2BCD(min) & 0x7f;
	hrs = BIN2BCD(hrs) & 0x3f;
	day = BIN2BCD(day) & 0x3f;
	mon = BIN2BCD(mon) & 0x1f;
	yrs = BIN2BCD(yrs) & 0xff;
	cen = BIN2BCD(cen) & 0xff;

	spin_lock_irqsave(&ds1511_lock, flags);
	rtc_disable_update();
	rtc_write(cen, RTC_CENTURY);
	rtc_write(yrs, RTC_YEAR);
	rtc_write((rtc_read(RTC_MON) & 0xe0) | mon, RTC_MON);
	rtc_write(day, RTC_DOM);
	rtc_write(hrs, RTC_HOUR);
	rtc_write(min, RTC_MIN);
	rtc_write(sec, RTC_SEC);
	rtc_write(dow, RTC_DOW);
	rtc_enable_update();
	spin_unlock_irqrestore(&ds1511_lock, flags);

	return 0;
}

 int
ds1511_set_time(unsigned long sec)
{
	struct rtc_time rtc_tm;

	/*
	 * convert to a standard time format -- note months
	 * count from zero.
	 */
	to_tm(sec, &rtc_tm);
	ds1511_set_rtctime(&rtc_tm);
	return 0;
}

 unsigned long
ds1511_get_time(void)
{
	struct rtc_time rtc_tm;

	ds1511_get_rtctime(&rtc_tm);
	return mktime(rtc_tm.tm_year, rtc_tm.tm_mon, rtc_tm.tm_mday,
	    rtc_tm.tm_hour, rtc_tm.tm_min, rtc_tm.tm_sec);
}

 static void
ds1511_get_alm_time(struct rtc_time *alm_tm)
{
	unsigned int flags;

	/*
	 * Only the values that we read from the RTC are set. That
	 * means only tm_sec, tm_hour, tm_min.
	 */
	spin_lock_irqsave(&ds1511_lock, flags);
	alm_tm->tm_sec = rtc_read(RTC_ALARM_SEC) & 0x7f;
	alm_tm->tm_min = rtc_read(RTC_ALARM_MIN) & 0x7f;
	alm_tm->tm_hour = rtc_read(RTC_ALARM_HOUR) & 0x3f;
	spin_unlock_irqrestore(&ds1511_lock, flags);

	alm_tm->tm_sec = BCD2BIN(alm_tm->tm_sec);
	alm_tm->tm_min = BCD2BIN(alm_tm->tm_min);
	alm_tm->tm_hour = BCD2BIN(alm_tm->tm_hour);
}

/*
 * set the watchdog timer.
 */
 static void
ds1511_set_wdtimer(int sec, int msec)
{
	uint8_t bcd_sec;
	uint8_t bcd_msec;

	bcd_sec = BIN2BCD(sec);
	bcd_msec = BIN2BCD(msec);
	rtc_write(bcd_sec, RTC_WD_SEC);
	rtc_write(bcd_msec, RTC_WD_MSEC);
}

/*
 * clear the watchdog event.
 */
 static void
ds1511_clear_wdtimer(void)
{
	rtc_write((rtc_read(RTC_CMD) & ~RTC_WDE), RTC_CMD);
	rtc_write((rtc_read(RTC_CMD1) & ~(DS1511_CONTROL_A_IRQF | RTC_WDF)),
		RTC_CMD1);
}

/*
 * enable the watchdog timer interrupt.
 */
 static void
ds1511_enable_wdtimer(int resetflag)
{
	int flag;

	if (resetflag)
		flag = RTC_COLDRESET_ENABLE;
	else
		flag = RTC_RESET_ENABLE;

	rtc_write(rtc_read(RTC_CMD) | flag, RTC_CMD);
}

/*
 * keep kicking the watchdog timer to prevent the watchdog from
 * expiring.
 */
 static void
ds1511_kick_wdtimer(unsigned long arg)
{
	db_assert(wd_sec > 0 || wd_msec > 0);
	ds1511_clear_wdtimer();
	ds1511_set_wdtimer(wd_sec, wd_msec);
	ds1511_enable_wdtimer(wd_reset_flag);
	if (wd_kickonce_flag) {
		/*
		 * wd_kickonce_flag lets watchdog timer expire.
		 */
		return;
	}
	if (wd_sec)
		mod_timer(&ds1511_wdtimer, jiffies + (wd_sec / 2) * HZ);
	else if (wd_msec)
		mod_timer(&ds1511_wdtimer, jiffies + (wd_msec / 2) * (HZ/100));
}

/*
 * set the watchdog timer and [re]start it
 *
 * call with seconds = -1 to use the configured number of seconds
 */
 void
ds1511_start_wdtimer(int seconds, int onceflag, int resetflag)
{
	if (seconds >= 0) {
		if (seconds == 0) {
			seconds = 1;
		} else if (seconds > RTC_MAX_WD_SEC) {
			seconds = RTC_MAX_WD_SEC;
		}
		wd_sec = seconds;
	}
	if (!wd_active_flag) {
		wd_active_flag = 1;
		wd_kickonce_flag = onceflag;
		wd_reset_flag = resetflag;
		ds1511_kick_wdtimer(0);
	}
}

/*
 * stop the watchdog kicker and disable the watchdog interrupt.
 */
 void
ds1511_stop_wdtimer(void)
{
	if (wd_active_flag) {
		ds1511_clear_wdtimer();
		del_timer_sync(&ds1511_wdtimer);
		wd_active_flag = 0;
	}
}

/*
 * initialize the watchdog timer and turn the watchdog timer on.
 */
 void
ds1511_init_wdtimer(int sec, int msec, int resetflag)
{
	if (wd_timer_initialized)
		del_timer_sync(&ds1511_wdtimer);

	wd_sec = sec;
	wd_msec = msec;
	ds1511_clear_wdtimer();
	ds1511_set_wdtimer(wd_sec, wd_msec);
	init_timer(&ds1511_wdtimer);
	ds1511_wdtimer.expires = jiffies + (msec/2) * HZ;
	ds1511_wdtimer.data = 0;
	ds1511_wdtimer.function = &ds1511_kick_wdtimer;
	add_timer(&ds1511_wdtimer);
	ds1511_enable_wdtimer(resetflag);
	wd_timer_initialized = 1;
}

module_init(ds1511_init);
module_exit(ds1511_exit);
