/*
 * Copyright (C) 2002 Momentum Computer, Inc.
 * Author: Matthew Dharm, mdharm@momenco.com
 * Copyright (C) 2007 Onstor Inc.
 * Author: Andrew Sharp (andy.sharp@onstor.com)
 *
 * Based on work by:
 *   Copyright (C) 2000 RidgeRun, Inc.
 *   Author: RidgeRun, Inc.
 *   glonnon@ridgerun.com, skranz@ridgerun.com, stevej@ridgerun.com
 *
 *   Copyright 2001 MontaVista Software Inc.
 *   Author: Jun Sun, jsun@mvista.com or jsun@junsun.net
 *
 *   Copyright (C) 2000, 01, 06 Ralf Baechle (ralf@linux-mips.org)
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/signal.h>
#include <linux/types.h>
#include <asm/irq_cpu.h>
#include <asm/mipsregs.h>
#include <asm/time.h>
#include <asm/marvell.h>
#include <linux/mv643xx.h>

static unsigned int irq_base;

static inline int ls1bit32(unsigned int x)
{
	int b = 31, s;

	s = 16; if (x << 16 == 0) s = 0; b -= s; x <<= s;
	s =  8; if (x <<  8 == 0) s = 0; b -= s; x <<= s;
	s =  4; if (x <<  4 == 0) s = 0; b -= s; x <<= s;
	s =  2; if (x <<  2 == 0) s = 0; b -= s; x <<= s;
	s =  1; if (x <<  1 == 0) s = 0; b -= s;

        return b;
}

/*
 * Interrupt handler for interrupts coming from the Marvell chip.
 * It could be built in ethernet ports etc...
 */
void ll_bc_mv64440_irq(void)
{
	unsigned int irq_src_low, irq_src_high;
	unsigned int irq_mask_low, irq_mask_high;

	/* read the interrupt status registers */
	irq_mask_low = MV_READ(MV64340_CPU_INTERRUPT0_MASK_LOW);
	irq_mask_high = MV_READ(MV64340_CPU_INTERRUPT0_MASK_HIGH);

	irq_src_low = MV_READ(MV64340_MAIN_INTERRUPT_CAUSE_LOW);
	irq_src_high = MV_READ(MV64340_MAIN_INTERRUPT_CAUSE_HIGH);

	/* mask for just the interrupts we want */
	irq_src_low &= irq_mask_low;
	irq_src_high &= irq_mask_high;

	if (irq_src_low)
		do_IRQ(ls1bit32(irq_src_low) + irq_base);
	else
		do_IRQ(ls1bit32(irq_src_high) + irq_base + 32);
}

asmlinkage void plat_irq_dispatch(void)
{
	unsigned int pending = read_c0_cause() & read_c0_status();

	if (pending & STATUSF_IP0)
		do_IRQ(0);
	else if (pending & STATUSF_IP1)
		do_IRQ(1);
	else if (pending & STATUSF_IP2)
		do_IRQ(2);
	else if (pending & STATUSF_IP3)
		ll_bc_mv64440_irq();
	else if (pending & STATUSF_IP4)
		do_IRQ(4);
	else if (pending & STATUSF_IP5)
		do_IRQ(5);
	else if (pending & STATUSF_IP6)
		do_IRQ(6);
	else if (pending & STATUSF_IP7)
		ll_timer_interrupt(7);
	else {
		/*
		 * Now look at the extended interrupts
		 */
		pending = (read_c0_cause() & (read_c0_intcontrol() << 8)) >> 16;
		/*
		 * dp83816
		 */
		if (pending & (STATUSF_IP8 | STATUSF_IP9)) {
			do_IRQ(pending & STATUSF_IP8 ? 8 : 9);
		/*
		 * pcmcia
		 */
		} else if (pending & STATUSF_IP10) {
			do_IRQ(10);
		} else if (pending & STATUSF_IP11) {
			do_IRQ(11);
		} else if (pending & STATUSF_IP12) {
			do_IRQ(12);
		} else {
			spurious_interrupt();
		}
	}
}

/*
 * mask off an interrupt -- 1 is enable, 0 is disable
 */
 static inline void
mask_bc_mv64440_irq(unsigned int irq)
{
	uint32_t value;

	if (irq < (irq_base + 32)) {
		value = MV_READ(MV64340_CPU_INTERRUPT0_MASK_LOW);
		value &= ~(1 << (irq - irq_base));
		MV_WRITE(MV64340_CPU_INTERRUPT0_MASK_LOW, value);
	} else {
		value = MV_READ(MV64340_CPU_INTERRUPT0_MASK_HIGH);
		value &= ~(1 << (irq - irq_base - 32));
		MV_WRITE(MV64340_CPU_INTERRUPT0_MASK_HIGH, value);
	}
}

/* unmask an interrupt -- 1 is enable, 0 is disable */
 static inline void
unmask_bc_mv64440_irq(unsigned int irq)
{
	uint32_t value;

	if (irq < (irq_base + 32)) {
		value = MV_READ(MV64340_CPU_INTERRUPT0_MASK_LOW);
		value |= 1 << (irq - irq_base);
		MV_WRITE(MV64340_CPU_INTERRUPT0_MASK_LOW, value);
	} else {
		value = MV_READ(MV64340_CPU_INTERRUPT0_MASK_HIGH);
		value |= 1 << (irq - irq_base - 32);
		MV_WRITE(MV64340_CPU_INTERRUPT0_MASK_HIGH, value);
	}
}

static struct irqaction cascade_mv64440 = {
	no_action, IRQF_DISABLED, CPU_MASK_NONE, "MV64440-Cascade", NULL, NULL
};

struct irq_chip bc_mv64440_irq_type = {
	.name = "BC-MV-64440",
	.ack = mask_bc_mv64440_irq,
	.mask = mask_bc_mv64440_irq,
	.mask_ack = mask_bc_mv64440_irq,
	.unmask = unmask_bc_mv64440_irq,
};

void __init arch_init_irq(void)
{
	int i;
	/*
	 * Clear all of the interrupts while we change the table around a bit.
	 * int-handler is not on bootstrap
	 */
	clear_c0_status(ST0_IM);

	mips_cpu_irq_init();
	rm7k_cpu_irq_init();
	rm9k_cpu_irq_init();

	/* set up the cascading interrupts */
	setup_irq(3, &cascade_mv64440);

	for (i = 16; i < 16 + 64; i++) {
		set_irq_chip_and_handler(i, &bc_mv64440_irq_type, handle_level_irq);
	}

	irq_base = 16;

	set_c0_status(ST0_IM);
}
