/*
 * BRIEF MODULE DESCRIPTION
 * Onstor Bobcat board dependent setup routines
 *
 * Copyright (C) 1996, 1997, 2001, 04, 06  Ralf Baechle (ralf@linux-mips.org)
 * Copyright (C) 2000 RidgeRun, Inc.
 * Copyright (C) 2001 Red Hat, Inc.
 * Copyright (C) 2002 Momentum Computer
 * Copyright (C) 2007 Onstor Inc.
 *
 * Author: Matthew Dharm, Momentum Computer
 *   mdharm@momenco.com
 *
 * Louis Hamilton, Red Hat, Inc.
 *   hamilton@redhat.com  [MIPS64 modifications]
 *
 * Author: RidgeRun, Inc.
 *   glonnon@ridgerun.com, skranz@ridgerun.com, stevej@ridgerun.com
 *
 * Copyright 2001 MontaVista Software Inc.
 * Author: jsun@mvista.com or jsun@junsun.net
 *
 * Andrew Sharp, Onstor Inc.
 *   andy.sharp@onstor.com
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <linux/bcd.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/mm.h>
#include <linux/bootmem.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/swap.h>
#include <linux/ioport.h>
#include <linux/pm.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/timex.h>
#include <linux/vmalloc.h>
#include <linux/mv643xx.h>

#include <asm/time.h>
#include <asm/bootinfo.h>
#include <asm/page.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/processor.h>
#include <asm/reboot.h>
#include <asm/tlbflush.h>
#include <asm/pmon.h>
#include <asm/mach-bobcat/bobcat.h>

#include "bobcat_bm_fpga.h"
#include "ds1511.h"

extern volatile uint8_t __iomem *mv64440_sram_base;
unsigned long cpu_clock;
unsigned long bus_clock;

/* These functions are used for rebooting or halting the machine*/
extern void onstor_bobcat_restart(char *command);
extern void onstor_bobcat_halt(void);
extern void onstor_bobcat_power_off(void);
extern int pci_probe_only;
#ifdef CONFIG_BOBCAT_8250_DEBUG_CONSOLE
extern void bc_8250_console_setup(void);
#endif
#ifdef CONFIG_BOBCAT_MPSC_EARLY_PRINTK
extern void mpsc_early_printk_init(void);
#endif

static char reset_reason;

 static inline unsigned long
ENTRYLO_CACHED(unsigned long paddr)
{
	return ((paddr & PAGE_MASK) |
		(_PAGE_PRESENT | __READABLE | __WRITEABLE | _PAGE_GLOBAL |
			_CACHE_WB)) >> 6;
}

 static inline unsigned long
ENTRYLO(unsigned long paddr)
{
	return ((paddr & PAGE_MASK) |
		(_PAGE_PRESENT | __READABLE | __WRITEABLE | _PAGE_GLOBAL |
			_CACHE_UNCACHED)) >> 6;
}

void __init bus_error_init(void) { /* nothing */ }

/*
 * Load a few TLB entries for the MV64340 and perhiperals. The MV64340 is going
 * to be hit on every IRQ anyway - there's absolutely no point in letting it be
 * a random TLB entry, as it'll just cause needless churning of the TLB. And we
 * use the other half for the serial port, which is just a PITA otherwise :)
 *
 *	Device			Physical	Virtual
 *	MV64440 Internal Regs	0xbc000000	0xbc000000
 *	//Ocelot-C[S] PLD (CS0)	0xfc000000	0xfc000000
 *	NVRAM (CS1)		0xfc800000	0xfc800000
 *	UARTs (CS2)		0xfd000000	0xfd000000
 *	Internal SRAM	0xbc040000	0xbc040000
 *	External SRAM	0x40000000	FIXADDR_TOP - 32MB
 */

unsigned long marvell_base = 0UL;
unsigned long marvell_mem;
unsigned long bobcat_fpga_base;
unsigned long uart_base;
volatile void __iomem *ocd_base;
volatile void __iomem *mcpu_shared_mem;

EXPORT_SYMBOL(marvell_base);
EXPORT_SYMBOL(ocd_base);

 static int __init
per_cpu_mappings(void)
{
	/*
	 * supposedly, marvell_base might have been passed in by the prom
	 */
	if (marvell_base) {
		marvell_base = (unsigned long)ioremap(CPHYSADDR(marvell_base), 0x10000);
	} else {
		marvell_base = (unsigned long)ioremap(BC_MARVELL_BASE_PADDR, 0x10000);
	}
#ifdef CONFIG_BOBCAT_8250_DEBUG_CONSOLE
	uart_base    = BC_UART0_BASE_VADDR;
#endif
	mvmemregion.pa_start = SHARED_MEM_START;
	mvmemregion.pa_end = mvmemregion.pa_start + SHARED_DEVMEM_SIZE -
		PAGE_SIZE - 1;
	mvmemregion.va_start = (unsigned long)ioremap(SHARED_MEM_START,
		SHARED_DEVMEM_SIZE - PAGE_SIZE /*0x10000000*/);
	dprintk("mvmemregion.va_start = %p", (void *)mvmemregion.va_start);
dprintk("marvel_base=%p", marvell_base);

	return 0;
}
arch_initcall(per_cpu_mappings);

 void __init
plat_timer_setup(struct irqaction *irq)
{
	int e;

	e = setup_irq(7, irq);
}

static uint32_t mv_io_base, mv_io_size;

static struct resource mv_pci_io1_resource = {
	.name	= "MV64440 PCI1 IO MEM",
	.flags	= IORESOURCE_IO,
};

static struct resource mv_pci_mem1_resource = {
	.name	= "MV64440 PCI1 MEM",
	.flags	= IORESOURCE_MEM,
};

static struct mv_pci_controller mv_bus1_controller = {
	.pcic = {
		.pci_ops	= &mv_pci_ops,
		.io_resource	= &mv_pci_io1_resource,
		.mem_resource	= &mv_pci_mem1_resource,
	},
	.config_addr	= MV64340_PCI_1_CONFIG_ADDR,
	.config_vreg	= MV64340_PCI_1_CONFIG_DATA_VIRTUAL_REG,
};

 static void __init
bobcat_pci1_init(void)
{
	uint32_t mem_base, mem_size;
	uint32_t mem0_base, mem0_size;
	uint32_t mem1_base, mem1_size;
	uint32_t mem2_base, mem2_size;
	uint32_t io_base, io_size;

	/* remap compact flash to PCI address 0x800000. */
	MV_WRITE(MV64340_PCI_1_MEMORY2_LOW_ADDR_REMAP, 0x80);

	io_base = MV_READ(MV64340_PCI_1_IO_BASE_ADDR) << 16;
	io_size = (MV_READ(MV64340_PCI_1_IO_SIZE) + 1) << 16;
	dprintk("io_base = %#x, io_size = %#x", io_base, io_size);
	mem0_base = MV_READ(MV64340_PCI_1_MEMORY0_BASE_ADDR) << 16;
	mem0_size = (MV_READ(MV64340_PCI_1_MEMORY0_SIZE) + 1) << 16;
	dprintk("mem0_base = %#x, mem0_size = %#x", mem0_base, mem0_size);
	mem1_base = MV_READ(MV64340_PCI_1_MEMORY1_BASE_ADDR) << 16;
	mem1_size = (MV_READ(MV64340_PCI_1_MEMORY1_SIZE) + 1) << 16;
	dprintk("mem1_base = %#x, mem1_size = %#x", mem1_base, mem1_size);
	mem2_base = MV_READ(MV64340_PCI_1_MEMORY2_BASE_ADDR) << 16;
	mem2_size = (MV_READ(MV64340_PCI_1_MEMORY2_SIZE) + 1) << 16;
	dprintk("mem2_base = %#x, mem2_size = %#x", mem2_base, mem2_size);
	mem_base = mem2_base;
	mem_size = mem1_size + mem2_size;
	dprintk("mem_base = %#x, mem_size = %#x", mem_base, mem_size);

	dprintk("mv_io_base = %#x, mv_io_size = %#x", mv_io_base, mv_io_size);
	/*
	 * Set the io and memory resource windows for the bus
	 */
	mv_pci_io1_resource.start	= io_base;
	mv_pci_io1_resource.end		= io_base + io_size - 1;
	mv_pci_mem1_resource.start		= mem_base;
	mv_pci_mem1_resource.end		= mem_base + mem_size - 1;
	mv_bus1_controller.pcic.io_map_base = io_base;
	mv_bus1_controller.pcic.io_map_base = -1UL;
	mv_bus1_controller.pcic.io_offset = io_base;
	mv_bus1_controller.pcic.mem_offset = mem_base;

	/*
	 * set up the non-pci resources that are on the pci bus ~:^)
	 */
	ioport_resource.start	= 0;
	ioport_resource.end		= io_base + io_size - 1;

	register_pci_controller(&mv_bus1_controller.pcic);

	mv_io_base = io_base;
	mv_io_size = io_size;
	dprintk("mv_io_base = %#x, mv_io_size = %#x", mv_io_base, mv_io_size);
}

 static int __init
bobcat_pci_init(void)
{
	unsigned long io_v_base;
	uint32_t enable;

	enable = ~MV_READ(MV64340_BASE_ADDR_ENABLE);

	/*
	 * Initialize PCI1 bus on the 64440
	 */

	if (enable & (0x01 << 14) || enable & (0x01 << 15)) {
		bobcat_pci1_init();
	}

	if (mv_io_size) {
		io_v_base = (unsigned long) ioremap(mv_io_base, mv_io_size);
		dprintk("pci io_v_base = %#lx\n", io_v_base);
		if (!io_v_base) {
			panic("Could not ioremap I/O port range");
		}

		set_io_port_base(io_v_base);
	}

	dprintk("mips_io_port_base = %#lx\n", mips_io_port_base);

	pci_probe_only = 1;

	dprintk("pci_probe_only=%d", pci_probe_only);

	return 0;
}

arch_initcall(bobcat_pci_init);

 static void __init
bobcat_map_ocd(void)
{
	ocd_base = ioremap(BOBCAT_OCD_ADDR, OCD_SIZE);
	if (!ocd_base) {
		printk("Mapping OCD failed - game over.  Your score is 0.");
	}
}

 static void __init
bobcat_rtc_setup(void)
{
	unsigned int rtc;

	/* initialize the ds1511 controll registers */
	if (!(rtc = ds1511_rtc_init())) {
		printk("%s: failed to map RTC\n", __FUNCTION__);
	} else {
		printk("Real Time Clock base address = %p\n", (void *)rtc);

		rtc_mips_get_time = ds1511_get_time;
		rtc_mips_set_time = ds1511_set_time;

		write_seqlock(&xtime_lock);
		xtime.tv_sec = ds1511_get_time();
		xtime.tv_nsec = 0;

		set_normalized_timespec(&wall_to_monotonic, -xtime.tv_sec,
			-xtime.tv_nsec);
		write_sequnlock(&xtime_lock);
	}
	/*
	 * intialize the watchdog timer.
	 */
	ds1511_init_wdtimer(RTC_MAX_WD_SEC, 0, 1);
}

/*
 * take care of various time intializations
 * - initialize the hpt frequency for this beast
 * - start the rtc clock
 */
 void __init
bobcat_time_init(void)
{
	mips_hpt_frequency = cpu_clock / 2;

	bobcat_map_ocd();
	bobcat_rtc_setup();
}

/*
 * depressing boobcat memory factoid #1
 *
 * We are told by the hardware people that the Marvell southbridge,
 * and all devices that hang off its PCI busses, cannot see the
 * RM9K local memory.  The SysAd bus connecting the Marvell to the
 * PMC-RM9K is a one way bus -- RM9K to Marvell.  Devices on the
 * Marvell PCI busses cannot therefore DMA to the RM9K local memory.
 *
 * The upshot of that is we have to hack the driver of every DMA device
 * on the Marvell PCI busses that the RM9K uses to use Marvell local
 * memory for DMA, and then memcpy that data back across the SysAd bus
 * to the RM9K local memory.
 *
 * This includes the natsemi DP83816 driver, and the MPSC driver for
 * the built in Marvell serial duart.
 */
 void __init
plat_mem_setup(void)
{
	unsigned int tmpword = 0;

	local_flush_tlb_all();

	board_time_init = bobcat_time_init;

	_machine_restart = onstor_bobcat_restart;
	_machine_halt = onstor_bobcat_halt;
	pm_power_off = onstor_bobcat_power_off;

	bobcat_fpga_base = BOBCAT_CS0_VADDR;
#ifdef CONFIG_BOBCAT_8250_DEBUG_CONSOLE
	uart_base    = BC_UART0_BASE_VADDR;
	bob_vectors->tty1_printf("testing 8250 UART line 2\n");
	bc_8250_console_setup();
#endif
#ifdef CONFIG_BOBCAT_MPSC_EARLY_PRINTK
	mpsc_early_printk_init();
#endif
	/* What is the marvell sram used for?  anything? */
	mv64440_sram_base = (uint8_t *)BC_MARVELL_SRAM_VADDR;

	/*
	 * initrd_start = (unsigned long)bobcat_initrd_start;
	 * initrd_end = (unsigned long)bobcat_initrd_start +
	 *		(ulong)bobcat_initrd_size;
	 * initrd_below_start_ok = 1;
	 */

	tmpword = BOBCAT_FPGA_READ(BOARD_STATUS);
	printk("OnStor Bobcat: Board Assembly Rev. %c%d\n",
		'A'+(tmpword&0x4000), (tmpword >> 16) & 0xf);
	printk("Booting from %sprom\n", tmpword & 2 ? "" : "recovery ");
	printk("Bus clock speed: 1%sMHz\n", tmpword & 0x8000 ? "33" : "00");

	reset_reason = tmpword;

	/*
	 * add rm9k local memory
	 *
	 * we want to save the prom, at least on debug builds, so that
	 * a ^E will drop you to the PROM monitor
	 */
	add_memory_region(0x0, BOBCAT_PROM_SZ, BOOT_MEM_ROM_DATA);
	add_memory_region(BOBCAT_PROM_SZ, 0x10000000 - BOBCAT_PROM_SZ,
		BOOT_MEM_RAM);
	/*
	 * 128 MiB of FC memory for us, yum.
	 */
	add_memory_region(0x20000000, 0x8000000, BOOT_MEM_RESERVED);
	//add_wired_entry(ENTRYLO(0x20000000), ENTRYLO(0x24000000),
		//ENTRYLO_CACHED(0x10000000), PM_64M);
	/*
	 * the other 128MiB of FC memory
	 */
	add_memory_region(0x28000000, 0x8000000, BOOT_MEM_RESERVED);
	//add_wired_entry(ENTRYLO(0x28000000), ENTRYLO(0x2c000000),
		//ENTRYLO_CACHED(0x10000000), PM_64M);

	/*
	 * 1024MiB of MV-64440 DDR
	 *
	 * add the  marvell memory.  we will use first hunk of it
	 * for DMA for the natsemi and mspc drivers.  suckage alert.
	 */
	add_memory_region(SHARED_MEM_START, SHARED_MEM_SIZE, BOOT_MEM_RESERVED);
}
