/*
 * Create platform device for DS1511 RTC used in Cougar motherboards
 *
 * Copyright (C) 2007 Onstor, Inc.
 * Copyright (C) 2007 Andrew Sharp <andy.sharp@onstor.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor
 *   Boston, MA  02110-1301  USA
 */
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/platform_device.h>

#include <asm/time.h>
#include <asm/sibyte/sb1250.h>
#include <asm/sibyte/sb1250_regs.h>
#include <asm/mach-cougar/cougar.h>

extern irqreturn_t sbwdog_interrupt(int, void *);

static struct resource cougar_rtc_res[] __initdata = {
	{
		.start	=	COUGAR_RTC_PADDR,
		.end	=	COUGAR_RTC_PADDR + COUGAR_RTC_SIZE - 1,
		.flags	=	IORESOURCE_MEM,
	},
};

static struct device *pd;

#ifdef CONFIG_RTC_DRV_DS1511
extern int ds1511_rtc_read_time(struct device *, struct rtc_time *);
extern int ds1511_rtc_set_time(struct device *, struct rtc_time *);

 static int
cougar_rtc_set_time(unsigned long t)
{
	struct rtc_time tm;

	rtc_time_to_tm(t, &tm);
	return ds1511_rtc_set_time(pd, &tm);
}

 static unsigned long
cougar_rtc_get_time(void)
{
	struct rtc_time tm;

	ds1511_rtc_read_time(pd, &tm);
	return mktime(tm.tm_year > 1900 ? tm.tm_year : tm.tm_year + 1900,
		tm.tm_mon + 1, tm.tm_hour, tm.tm_mday, tm.tm_min, tm.tm_sec);
}

 static int __init
cougar_rtc_setup(void)
{
	struct platform_device *pdev;
	int err;

	pdev = platform_device_alloc("ds1511", -1);
	if (!pdev) {
		return -ENOMEM;
	}

	err = platform_device_add_resources(pdev, cougar_rtc_res,
			ARRAY_SIZE(cougar_rtc_res));
	if (err) {
		goto err_free_device;
	}

	err = platform_device_add(pdev);
	if (err) {
		goto err_free_device;
	}

	pd = &pdev->dev;

	rtc_mips_get_time = cougar_rtc_get_time;
	rtc_mips_set_time = cougar_rtc_set_time;

	return 0;

err_free_device:
	platform_device_put(pdev);

	return err;
}
device_initcall(cougar_rtc_setup);
#endif /* CONFIG_RTC_DRV_DS1511 */

/*
 * le coug is booted first by prom which uses the first sibyte
 * watchdog as, well, a watchdog.  we keep it going in case the kernel
 * gets stuck.  we shouldn't need to set up the registers or anything
 * because the prom should already have done that.
 */
 void
cougar_wd_setup(void)
{
	int ret;

	ret = request_irq(0, sbwdog_interrupt, IRQF_DISABLED | IRQF_SHARED,
		"Cougar Kernel Watchdog", IOADDR(A_SCD_WDOG_CFG_0));
	if (ret) {
		printk(KERN_CRIT "Watchdog IRQ zero(0) failed to be requested - %d\n",
			ret);
	}
}
