/*
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Set up the mapping of PCI addr 1a00,0000 to point to
 * the shared memory segment starting at SHARED_MEM_START
 *
 * Copyright (C) 2008, Onstor, Inc.
 * Author Andrew Sharp (andy.sharp@onstor.com)
 */
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/pci.h>

#include <asm/pci.h>
#include <asm/io.h>
#include <asm/mipsregs.h>
#include <asm/sibyte/bcm1480_regs.h>

#include <tuxrx.h>

#ifdef NOT_NEEDED
/*
 * Do platform specific device initialization at pci_enable_device() time
 */
int
pcibios_plat_dev_init(struct pci_dev *dev)
{
	printk(KERN_INFO "%s: vendor:device  subsystem_device  driver_name\n", __func__);
	printk(KERN_INFO "pcibios_plat_dev_init:   %04x:%04x        %04x              %s\n",
	       dev->vendor, dev->device, dev->subsystem_device, dev->driver->name);
	return 0;
}
#endif /* NOT_NEEDED */

/*
 * Read/write values in config space.
 */
#define READCFG(off) (*(u32 *)(cfg_space + (off & ~3)))
#define WRITECFG(off, data) *(u32 *)(cfg_space + ((off) & ~3)) = (data)

/*
 * BCM1480 PCI header definitions
 */
#define R_BCM1480_PHB_EXTCONFIGDIS      0x98
#define R_BCM1480_PHB_MAP(n)            (0x44 + ((n) * 4))

/*
 * program the config map registers
 * to point to shared memory chunk set up
 * in plat_smp_setup(), of all places
 */
 static void
tuxrx_pci_program_shared_mem_config(struct pci_dev *dev)
{
	void *cfg_space;
	u64 addr, addr_shift;
	int i;
	u32 contents, data, offset;

	printk(KERN_INFO "%s: vendor:device  bus_no\n", __func__);
	printk(KERN_INFO "tuxrx_pci_program_shared_mem_config:   %04x:%04x      %04x\n",
	       dev->vendor, dev->device, dev->bus->number);

	cfg_space = ioremap(A_BCM1480_PHYS_PCI_CFG_MATCH_BITS, 16*1024*1024);
	addr = SHARED_MEM_START;

	printk(KERN_INFO "%s: cfg_space 0x%p, addr 0x%p\n",
	       __func__, cfg_space, (void *)addr);

	/*
	 * Enable access to map registers
	 */
	WRITECFG(R_BCM1480_PHB_EXTCONFIGDIS, 1);
	/*
	 * Enable access to 16 1MB chunks
	 */
	for (i = 0; i < 16; i++) {
	        offset = R_BCM1480_PHB_MAP(i);
		addr_shift = (addr >> 8);
		data = (u32 )((addr >> 8) | 0x1);

#ifdef TUXRX_PCI_DEBUG
	printk(KERN_INFO "\tWriting PCI Config Space: offset 0x%x, addr_shift: 0x%p, data 0x%x\n",
		       offset, (void *)addr_shift, data);
#endif /* TUXRX_PCI_DEBUG */

		WRITECFG(offset, data);
		addr += (1024 * 1024);
	}

	/*
	 * Read back what was written into config space
	 */
	for (i = 0; i < 16; i++) {
	        offset = R_BCM1480_PHB_MAP(i);
	        contents = READCFG(offset);

#ifdef TUXRX_PCI_DEBUG
		printk(KERN_INFO "\tReading PCI Config Space: offset 0x%x, contents 0x%x\n",
		       offset, contents);
#endif /* TUXRX_PCI_DEBUG */
	}

	/*
	 * Disable access to map registers
	 */
	WRITECFG(R_BCM1480_PHB_EXTCONFIGDIS, 0);
	return;
}

DECLARE_PCI_FIXUP_EARLY(PCI_VENDOR_ID_SIBYTE, PCI_DEVICE_ID_BCM1480_PCI,
    tuxrx_pci_program_shared_mem_config);

