/*
 * Watchdog driver for SiByte SB1 SoCs
 *
 * Copyright (C) 2007,2008 OnStor, Inc. * Andrew Sharp <andy.sharp@onstor.com>
 *
 * This driver is intended to make the second [and third] of two [four]
 * hardware watchdogs on the Sibyte 12XX and 11XX [1x55 and 1x80] SoCs
 * available to the user.  There are two [four] such devices available
 * on the SoC, but it seems that there isn't an enumeration class for
 * watchdogs in Linux like there is for RTCs.
 *
 * It is a simple timer, and there is an interrupt that is raised the
 * first time the timer expires.  The second time it expires, the chip
 * is reset and there is no way to redirect that NMI.  Which could
 * be problematic in some cases where this chip is sitting on the HT
 * bus and has just taken responsibility for providing a cache block.
 * Since the reset can't be redirected to the external reset pin, it is
 * possible that other HT connected processors might hang and not reset.
 * For Linux, a soft reset would probably be even worse than a hard reset.
 * There you have it.
 *
 * The timer takes 23 bits of a 64 bit register (?) as a count value,
 * and decrements the count every microsecond, for a max value of
 * 0x7fffff usec or about 8.3ish seconds.
 *
 * Support for the 1x55 and 1x80 SoCs is now added, which includes
 * the 1480.  These have four watchdog timers, but due to an errata,
 * the 4th doesn't work, so only 1 extra over the 1250 and derivatives
 * is available.
 *
 * This watchdog borrows some user semantics from the softdog driver,
 * in that if you close the fd, it leaves the watchdog running, unless
 * you previously wrote a 'V' to the fd, in which case it disables
 * the watchdog when you close the fd like some other drivers.
 *
 * Based on various other watchdog drivers, which are probably all
 * loosely based on something Alan Cox wrote years ago.
 *
 *	(c) Copyright 1996 Alan Cox <alan@redhat.com>, All Rights Reserved.
 *				http://www.redhat.com
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	version 1 or 2 as published by the Free Software Foundation.
 *
 */
#include <linux/module.h>
#include <linux/io.h>
#include <linux/uaccess.h>
#include <linux/fs.h>
#include <linux/reboot.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/interrupt.h>

#include <asm/sibyte/sb1250.h>
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
# include <asm/sibyte/bcm1480_regs.h>
# include <asm/sibyte/bcm1480_int.h>
# include <asm/sibyte/bcm1480_scd.h>
#elif defined(CONFIG_SIBYTE_SB1250) || defined(CONFIG_SIBYTE_BCM112X)
# include <asm/sibyte/sb1250_regs.h>
# include <asm/sibyte/sb1250_int.h>
# include <asm/sibyte/sb1250_scd.h>
#else
# error invalid SiByte configuration
#endif

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
# define SBWDOG_VBASE(n) \
	(IO_BASE + (unsigned long)A_BCM1480_SCD_WDOG_BASE(n) + 0x10)
# define SBWDOG_INT_1	K_BCM1480_INT_WATCHDOG_TIMER_1
#else
# define SBWDOG_VBASE(n) \
	(IO_BASE + (unsigned long)A_SCD_WDOG_BASE(n) + 0x10)
# define SBWDOG_INT_1	K_INT_WATCHDOG_TIMER_1
#endif

/*
 * set the initial count value of a timer
 *
 * wdog is the iomem address of the cfg register
 */
 void
sbwdog_set(char __iomem *wdog, unsigned long t)
{
	__raw_writeb(0, wdog - 0x10);
	__raw_writeq(t & 0x7fffffUL, wdog);
}

/*
 * cause the timer to [re]load it's initial count and start counting
 * all over again
 *
 * wdog is the iomem address of the cfg register
 */
 void
sbwdog_pet(char __iomem *wdog)
{
	__raw_writeb(__raw_readb(wdog) | 1, wdog);
}

static unsigned long sbwdog_gate0; /* keeps it to one thread only */
static char __iomem *kern_dog =  (char __iomem *)SBWDOG_VBASE(0);
static char __iomem *user_dog0 = (char __iomem *)SBWDOG_VBASE(1);
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
static unsigned long sbwdog_gate1; /* keeps it to one thread only */
static char __iomem *user_dog1 = (char __iomem *)SBWDOG_VBASE(2);
/* static char __iomem *user_dog2 = (char __iomem *)SBWDOG_VBASE(3); */
#endif
static unsigned long timeout = 0x7fffffUL;	/* useconds: 8.3ish secs. */
static int expect_close;

static struct watchdog_info ident = {
	.options	= WDIOF_CARDRESET | WDIOF_SETTIMEOUT | WDIOF_KEEPALIVEPING,
	.identity	= "SiByte User Watchdog",
};

/*
 * Allow only a single thread to walk the dog
 */
 static int
sbwdog_open(struct inode *inode, struct file *file)
{
	unsigned long *gate = &sbwdog_gate0;
	void *dog = user_dog0;

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
	const char *wname = file->f_dentry->d_name.name;

	/*
	 * this can't be the right way to do this
	 */
	if (!strcmp(wname, "watchdog1")) {
		dog = user_dog1;
		gate = &sbwdog_gate1;
	}
#endif

	nonseekable_open(inode, file);

	if (test_and_set_bit(0, gate)) {
		return -EBUSY;
	}

	/*
	 * Activate the timer
	 */
	sbwdog_set(dog, timeout);
	__raw_writeb(1, dog);

	return 0;
}

/*
 * stop the user watchdogs
 */
 static void
sbwdog_stop_user_dogs(void)
{
	__raw_writeb(0, user_dog0);
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
	__raw_writeb(0, user_dog1);
	/* __raw_writeb(0, user_dog2); */
#endif
}

/*
 * Put the dog back in the kennel.
 */
 static int
sbwdog_release(struct inode *inode, struct file *file)
{
	const char *wname = file->f_dentry->d_name.name;
	unsigned long *gate = &sbwdog_gate0;
	void *dog = user_dog0;

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
	/*
	 * this can't be the right way to do this
	 */
	if (!strcmp(wname, "watchdog1")) {
		dog = user_dog1;
		gate = &sbwdog_gate1;
	}
#endif

	if (expect_close == 42) {
		__raw_writeb(0, dog);
	} else {
        /*
         * ONSTOR change - chassisd does not keep the fd open and we don't want
         *                 this message everytime the fd is closed.
         */
#if !defined(CONFIG_ONSTOR_BOBCAT) && !defined(CONFIG_ONSTOR_COUGAR)
		printk(KERN_CRIT "%s: Unexpected close, not stopping watchdog!\n",
			ident.identity);
#endif
		sbwdog_pet(user_dog0);
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
		sbwdog_pet(user_dog1);
		/* sbwdog_pet(user_dog2); */
#endif
	}
	clear_bit(0, gate);
	expect_close = 0;

	return 0;
}

/*
 * 42 - the answer
 */
 static ssize_t
sbwdog_write(struct file *file, const char __user *data, size_t len,
	loff_t *ppos)
{
	const char *wname = file->f_dentry->d_name.name;
	void *dog = user_dog0;
	int i;

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
	/*
	 * this can't be the right way to do this
	 */
	if (!strcmp(wname, "watchdog1")) {
		dog = user_dog1;
	}
#endif

	if (len) {
		/*
		 * restart the timer
		 */
		expect_close = 0;

		for (i = 0; i != len; i++) {
			char c;

			if (get_user(c, data + i)) {
				return -EFAULT;
			}
			if (c == 'V') {
				expect_close = 42;
			}
		}
		sbwdog_pet(dog);
	}

	return len;
}

 static int
sbwdog_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
	unsigned long arg)
{
	int ret = -ENOTTY;
	unsigned long time;
	void __user *argp = (void __user *)arg;
	int __user *p = argp;
	void *dog = user_dog0;

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
	const char *wname = file->f_dentry->d_name.name;

	/*
	 * this can't be the right way to do this
	 */
	if (!strcmp(wname, "watchdog1")) {
		dog = user_dog1;
	}
#endif

	switch (cmd) {
	case WDIOC_GETSUPPORT:
		ret = copy_to_user(argp, &ident, sizeof(ident)) ? -EFAULT : 0;
		break;

	case WDIOC_GETSTATUS:
		/*
		 * return the bits from the config register
		 */
		ret = put_user(__raw_readb(dog), p);
		break;

	case WDIOC_SETTIMEOUT:
		ret = get_user(time, p);
		if (ret) {
			break;
		}

		time *= 1000000;
		if (time > 0x7fffffUL) {
			ret = -EINVAL;
			break;
		}
		timeout = time;
		sbwdog_set(dog, timeout);
		sbwdog_pet(dog);

	case WDIOC_GETTIMEOUT:
		/*
		 * get the remaining count from the ... count register
		 * which is 1*8 before the config register
		 */
		ret = put_user(__raw_readq(dog - 8) / 1000000, p);
		break;

	case WDIOC_KEEPALIVE:
		sbwdog_pet(dog);
		ret = 0;
		break;
	}
	return ret;
}

/*
 *	Notifier for system down
 */
 static int
sbwdog_notify_sys(struct notifier_block *this, unsigned long code, void *erf)
{
	if (code == SYS_DOWN || code == SYS_HALT) {
		/*
		 * sit and sit
		 */
		__raw_writeb(0, kern_dog);
		sbwdog_stop_user_dogs();
	}

	return NOTIFY_DONE;
}

static const struct file_operations sbwdog_fops =
{
	.owner		= THIS_MODULE,
	.llseek		= no_llseek,
	.write		= sbwdog_write,
	.ioctl		= sbwdog_ioctl,
	.open		= sbwdog_open,
	.release	= sbwdog_release,
};

static struct miscdevice sbwdog_miscdev_0 =
{
	.minor		= WATCHDOG_MINOR,
	.name		= "watchdog0",
	.fops		= &sbwdog_fops,
};
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
static struct miscdevice sbwdog_miscdev_1 =
{
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= "watchdog1",
	.fops		= &sbwdog_fops,
};
/*
 * 4th WD on 1480 doesn't work
static struct miscdevice sbwdog_miscdev_2 =
{
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= "watchdog2",
	.fops		= &sbwdog_fops,
};
*/
#endif

static struct notifier_block sbwdog_notifier = {
	.notifier_call	= sbwdog_notify_sys,
};

/*
 * interrupt handler
 *
 * doesn't do a whole lot for user, but oh so cleverly written so kernel
 * code can use it to re-up the watchdog, thereby saving the kernel from
 * having to create and maintain a timer, just to tickle another timer,
 * which is just so wrong.
 */
 irqreturn_t
sbwdog_interrupt(int irq, void *addr)
{
	unsigned long wd_init;
	char *wd_cfg_reg = (char *)addr;
	u8 cfg;

	cfg = __raw_readb(wd_cfg_reg);
	wd_init = __raw_readq(wd_cfg_reg - 8) & 0x7fffff;

	/*
	 * if it's not the kernel watchdog timer, it's for those users
	 */
	if (wd_cfg_reg != kern_dog) {
		printk(KERN_CRIT
			"%s in danger of initiating system reset in %ld.%01ld seconds\n",
			ident.identity, wd_init / 1000000, (wd_init / 100000) % 10);
	} else {
		cfg |= 1;
	}

	__raw_writeb(cfg, wd_cfg_reg);

	return IRQ_HANDLED;
}

 static int __init
sbwdog_init(void)
{
	int ret;
	int reg_ret;

	/*
	 * register a reboot notifier
	 */
	ret = register_reboot_notifier(&sbwdog_notifier);
	if (ret) {
		printk (KERN_ERR "%s: cannot register reboot notifier: %d)\n",
			ident.identity, ret);
		return ret;
	}

	/*
	 * register the device(s)
	 */
	ret = misc_register(&sbwdog_miscdev_0);
	if (ret) {
		reg_ret = 1;
		printk(KERN_ERR "%s: unable to regster device %s: %d\n", __func__,
			sbwdog_miscdev_0.name, ret);
	}

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
	ret = misc_register(&sbwdog_miscdev_1);
	if (ret) {
		reg_ret |= 2;
		printk(KERN_ERR "%s: unable to regster device %s: %d\n", __func__,
			sbwdog_miscdev_1.name, ret);
	}
#endif

	/*
	 * get the resources
	 */

	ret = request_irq(SBWDOG_INT_1, sbwdog_interrupt, IRQF_DISABLED | IRQF_SHARED,
		ident.identity, (void *)user_dog0);
	if (ret) {
		printk(KERN_ERR "%s: failed to request irq 1 - %d\n", ident.identity,
			ret);
		if (!(reg_ret & 1)) {
			misc_deregister(&sbwdog_miscdev_0);
		}
		return ret;
	}
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
	ret = request_irq(K_BCM1480_INT_WATCHDOG_TIMER_2, sbwdog_interrupt,
		IRQF_DISABLED | IRQF_SHARED, ident.identity, (void *)user_dog1);
	if (ret) {
		printk(KERN_ERR "%s: failed to request irq 2 - %d\n", ident.identity,
			ret);
		if (!(reg_ret & 2)) {
			misc_deregister(&sbwdog_miscdev_1);
		}
		return ret;
	}
#endif

	/*
	 * make sure no errant wd timers have been set that don't
	 * have handlers
	 */
	sbwdog_stop_user_dogs();

	if (ret == 0) {
		printk(KERN_INFO "%s: timeout is %ld.%ld secs\n", ident.identity,
			timeout / 1000000, (timeout / 100000) % 10);
	}

	return ret;
}

 static void __exit
sbwdog_exit(void)
{
	misc_deregister(&sbwdog_miscdev_0);
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
	misc_deregister(&sbwdog_miscdev_1);
	/* misc_deregister(&sbwdog_miscdev_2); */
#endif
}

module_init(sbwdog_init);
module_exit(sbwdog_exit);

MODULE_AUTHOR("Andrew Sharp <andy.sharp@onstor.com>");
MODULE_DESCRIPTION("SiByte Watchdog");
MODULE_VERSION("1.2");

module_param(timeout, ulong, 0);
MODULE_PARM_DESC(timeout,
	"Watchdog timeout in microseconds (max/default 8388607 or 8.3ish secs)");

MODULE_LICENSE("GPL");
MODULE_ALIAS_MISCDEV(WATCHDOG_MINOR);

/*
 * example code that can be put in a platform code area to utilize the
 * first watchdog timer for the kernel's own purpose.

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
# define SBWDOG_INT_0	K_BCM1480_INT_WATCHDOG_TIMER_0
#else
# define SBWDOG_INT_0	K_INT_WATCHDOG_TIMER_0
#endif

 void
platform_wd_setup(void)
{
	int ret;

	ret = request_irq(SBWDOG_INT_0, sbwdog_interrupt,
		IRQF_DISABLED | IRQF_SHARED, "Kernel Watchdog",
		IOADDR(A_SCD_WDOG_CFG_0));
	if (ret) {
		printk(KERN_CRIT "Watchdog IRQ zero(0) failed to be requested - %d\n",
			ret);
	}
}


 */
