/*
 * Name            : mgmt-bus.c
 * Description     : SSC or TxRx Linux shared memory management bus driver
 * Original Authors: Tim Gardner(EEE/OpenBSD), Max Kozlovsky(Linux), Fisher(Tuxrx)
 * Created On      : Thu Jun 24 15:31:17 2004
 * Last Modifed By : William Fisher, Major rework for TxRx Linux
 *                    for common SSC and TxRx Linux Device Driver
 * Last Modifed    : Monday January 12, 2009 17:33:53
 *
 * Copyright (c) 2004-2009 OnStor, Inc.
 * All rights reserved.
 *
 * Purpose:
 *
 * This Linux driver is implements a pseudo PPP network device driver which
 * utilizes the low level management bus to support sending/receiving management bus
 * packets to/from the SSC, TxRx and FP, via the PCI bus.
 * This driver executes on BOTH the SSC or TxRx Linux kernels and communicates
 * via using the shared memory area mapped onto the PCI bus address space.
 * The driver is compiled separately for either the SSC(uniprocessor, 1125) or
 * the TXRX(SMP, 1480) hardware.
 *
 * NOTE: This drivers supports ONLY the Cougar platform and has NO
 * support for Bobcat nor any prior platforms.
 *
 * The management bus protocol is described by the Bobcat Management Bus
 * Protocol Design Specification. Goodluck in finding that specification,
 * the code is the spec I believe.
 * This driver resides in the Linux kernel and supports BOTH the neteee and IP
 * protocols stacks. The eee protocol module 'neteee.c', can be found under the
 * net/neteee kernel source directory.
 *
 * The driver implements a shared memory point-to-point network driver.
 * For each logical point-to-point link (SSC to TXRX, TXRX to SSC, etc),
 * two pairs of message descriptor rings exist. In the case of the SSC,
 * one pair is used to send messages FROM the SSC and the other is used
 * to send messages TO the SSC.  Each pair consists of a ring of 'available'
 * message descriptors and another of 'used' message descriptors.
 *
 * When transmitting a packet, the source CPU copies the packet data into
 * the buffer associated with the next available descriptor on the 'available' ring.
 * The corresponding descriptor in the 'used' ring is updated to inform the
 * destination CPU a packet is available to be received.
 *
 * Available descriptor rings are always located in memory local to the sender.
 * This is done by having the kernel map the PCI address space onto a chunk
 * of memory local to the processor. Access from other core's are done via
 * the PCI bus.
 * In the same way, the 'used' descriptor rings are also located in memory local
 * to the receiving CPU.
 * This enables all packet transfers to be done using only PCI writes.
 * Hence PCI reads are eliminated and we avoid all potential CPU stalls
 * waiting for a PCI Read completion.
 *
 * The flags field in each descriptor indicates it's state, either 'available'
 * or 'used', with 'used' indicating the descriptor contains a packet requiring
 * receive processing.
 *
 * Initialization:
 *
 * Initialization is performed by the 'mgmtbus_init' procedure which is called
 * either during kernel booting or when the driver module is loaded.
 * All descriptor rings residing in shared memory are initialized.
 * We ASSUME the SSC boots first and completes it's initialization before
 * any of the other CPUs are booted.
 *
 * A configuration table is located at a well known address which contains the
 * address and size of every descriptor ring.  Upon initialization, each CPU
 * initializes it's portion of the table for the rings residing in memory local
 * to that CPU. The Shared Memory area(s), statically identified for each
 * core from the PCI Address space are mapped into local memory for that CPU.
 * Other core's access this memory via the PCI bus uncached. This mapping
 * is done by the Linux kernel prior to driver initialization.
 *
 * Packet Transmission:
 *
 * Packet transmit is done by the mgmtbus_hard_start_xmit() procedure which is called
 * by the neteee or IP protocol modules. The packet to be transmitted is described
 * by an 'skb' descriptor and contains a 'agile_hdr_aligned_t' PPP header.
 * The 'agile_hdr_t' header which contains an extra 2 byte pad on the front for
 * alignment purposes.The transmit procedure must translate the header containing
 * the source and destination addresses, before sending.
 *
 * Transfer is accomplished by copying the 'skb' packet data into the buffer whose
 * address resides in the next descriptor on the available message descriptor ring.
 * This descriptor and associated buffer were allocated from the shared memory area.
 *
 * An error is returned if no 'available' message descriptors can be found
 * to transmit the packet.
 *
 * Ideally DMA should be used instead of a processor copy, unfortunately the Galileo
 * IDMA controller, only on the Bobcat platform could not access local memory
 * (only Marvell memory) and all outbound skb's reside in local memory. That
 * is not an issue under Cougar however DMA is not used.
 *
 * Packet Reception:
 *
 * Packet receive processing is performed under the 'mgmtbus_intr' interrupt
 * procedure which is called by either the Linux NAPI poll function if this
 * is a NAPI compliant driver, or directly from the kernel interrupt handling code.
 * The Linux kernel TuxRx running on the TxRx Core, generates an interrupt to
 * the SSC and vice versa when messages are transmitted.
 *
 * The interrupt handler checks all 'used' descriptor rings for messages awaiting processing.
 * Each message is processed by allocating a 'skb' to contain the message data
 * residing in shared memory. If an skb cannot be allocated, message processing is deferred
 * until the next interrupt.
 *
 * All incomming packets begin with an agile_hdr_aligned_t header which is translated
 * into an agile_hdr_t header before processing. If the packet is destined for the
 * neteee protocol, the header is left intact. When an IP packet is received the
 * 'agile_hdr_aligned_t' is stripped, so the packet begins with an IP header.
 * The protocol type is indicated by a 16-Bit protocol type field which will be
 * added to the 'agile_hdr_aligned_t', using 16-bits from the last 32-bit pad word.
 */
#include <linux/types.h>
#include <linux/param.h>
#include <linux/spinlock_types.h>
#include <linux/socket.h>
#include <linux/ioctl.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/if.h>
#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <linux/if_ether.h>

#include <linux/ip.h>
#include <linux/in.h>

#include <linux/etherdevice.h>
#include <net/neteee/edesc.h>
#include "mgmt-bus.h"
#include "rcon.h"

/*
 * CONFIG_ONSTOR_TUXRX
 * CONFIG_ONSTOR_COUGAR ONLY
 */
#include <asm/io.h>
#include <asm/mach-cougar/cougar.h>

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
#include <asm/sibyte/bcm1480_regs.h>
#else /* defined(CONFIG_SIBYTE_SB1250) || defined(CONFIG_SIBYTE_BCM112X) */
#include <asm/sibyte/sb1250_regs.h>
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

#include <asm/sibyte/sb1250_int.h>
#include "../../arch/mips/onstor/cougar/cougar_bm_fpga.h"

/*
 *****************************************************************
 * The following constants were contained in the old EEE specific
 * TXRX header file, located at:
 * ../nfx-tree/Includes/cg/NFP_TXRX/mgmt-bus-regs.h file
 * Maybe move these into kernel include file mgmt-bus.h ?
 *
 * The following PCI offsets correspond to the mailboxes
 * SSC  Mailbox starts 0x7000.0000, REQUIRES top 8-bits phys addr 0xF8
 * TXRX Mailbox starts 0x8000.0000, REQUIRES top 8-bits phys addr 0x10
 * FP   Mailbox starts 0x9000.0000, REQUIRES top 8-bits phys addr 0x10
 *
 * NOTE: The Linux kernel uses mailbox_0 in a totally different manner
 * than the EEE. Hence we must co-exist with the Linux kernel's mailbox
 * useage for IPI, MSP support, etc.
 *****************************************************************
 */
/* The register type. */
#define MGMTBUS_SSC_INTERRUPT_REG_SIZE uint64_t

/*
 * This TXRX PCI write generates a SSC mailbox interrupt.
 * The TXRX physical PCI address MUST have 0x10 in top 8-bits,
 * Location of mailbox register to generate interrupt on SSC from TXRX.
 * The 1125 "mailbox set" register accessed through PCI BAR2.
 */
#define MGMTBUS_SSC_INTR_MAILBOX 0x1070000008UL
#define MGMTBUS_SSC_INTERRUPT_REG PHYS_TO_XKSEG_UNCACHED(MGMTBUS_SSC_INTR_MAILBOX)

/* The value to write to the register. Generates MBOX_INT_3 on the 1125 */
#define MGMTBUS_SSC_INTERRUPT_VAL 0xFFFFUL

/*
 * This SSC PCI write generates a TXRX mailbox interrupt.
 * The SSC physical PCI address MUST have 0xF8 in top 8-bits,
 * Location of mailbox register to generate interrupt on TXRX from SSC.
 * The 1480 "mailbox set" register accessed through PCI BAR2.
 */
#define MGMTBUS_TXRX_INTR_MAILBOX 0xF880000008UL
#define MGMTBUS_TXRX_INTERRUPT_REG PHYS_TO_XKSEG_UNCACHED(MGMTBUS_TXRX_INTR_MAILBOX)

/*
 * Value to write into register. Generates MBOX_INT_3 to the 1480
 */
#define MGMTBUS_TXRX_INTERRUPT_VAL 0xFFFFUL

/*
 * Location of register used to generate console interrupt.
 * The console uses the same mailbox register but generates
 * a different interrupt.
 */
#define SM_CONSOLE_SSC_INTERRUPT_REG MGMTBUS_SSC_INTERRUPT_REG

/*
 * Compile in some development debugging.
 */
/* define MGMTBUS_DEBUG 1 */

#ifdef MGMTBUS_DEBUG
#define MGMTBUS_PRINTK(x) printk x
#else
#define MGMTBUS_PRINTK(x)
#endif /* MGMTBUS_DEBUG */

#define MGMTBUS_PRINTK_X(x)

/*
 * Device Driver software context.
 */
typedef struct {
    unsigned int flags;
    spinlock_t tx_lock;

    mgmtbus_ring_config_t *ring_config;

    void *txrx_pci_virt;
    void *fp_pci_virt;
    void *ssc_pci_virt;

    unsigned long txrx_pci_base;
    unsigned long fp_pci_base;
    unsigned long ssc_pci_base;

    unsigned long txrx_pci_next_virt_avail;  /* Next virtual address to allocate */
    unsigned long fp_pci_next_virt_avail;
    unsigned long ssc_pci_next_virt_avail;

    struct net_device_stats mgmtbus_stats;
} mgmtbus_softc_t;

/*
 * Device Driver Global Variables
 */
mgmtbus_softc_t Mgmtbus_softc;
mgmtbus_ring_config_t *magicmanagementbusringconfig;

/*
 * Holding spot for the model number until the mgmtbus driver init routine runs.
 */
static int model_num = 0x104;  /* seed it with a funny number so we can see if it's been changed */

/*
 * Forward referenced/external procedures
 */
struct net_device_stats * mgmtbus_get_stats(struct net_device *dev);

int mgmtbus_hard_start_xmit(struct sk_buff *skb, struct net_device *dev);

#ifdef CONFIG_NET_POLL_CONTROLLER
void mgmtbus_poll_controller(struct net_device *dev);
#endif /* CONFIG_NET_POLL_CONTROLLER */

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
void mgmtbus_send_interrupt_to_ssc(void);
#else
void mgmtbus_send_interrupt_to_txrx(void);
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

/*
 * Management bus net_device structure
 */
struct net_device mgmtbus_dev = {
    .name = "mgmtbus",

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
    .features = NETIF_F_FRAGLIST | NETIF_F_HIGHDMA,
    .tx_queue_len = MGMTBUS_TXRX_2_SSC_RING_SIZE,
#else
    .features = NETIF_F_FRAGLIST | NETIF_F_HIGHDMA | NETIF_F_LLTX,
    .tx_queue_len = MGMTBUS_SSC_2_TXRX_RING_SIZE,
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

    .get_stats = mgmtbus_get_stats,
    .flags = IFF_POINTOPOINT | IFF_NOARP,
    .mtu = 1536,
    .type = ARPHRD_ETHER,
    .hard_start_xmit = mgmtbus_hard_start_xmit,
    .hard_header = eth_header,
    .hard_header_len = ETH_HLEN,
    .addr_len = ETH_ALEN,
    .rebuild_header = eth_rebuild_header,

#ifdef CONFIG_NET_POLL_CONTROLLER
    .poll_controller = mgmtbus_poll_controller,
#endif /* CONFIG_NET_POLL_CONTROLLER */
};

/*
 * L1 and L2 cache flush/sync macro.
 * Bobcat design had the Marvell memory mapped uncached
 * and does not define the macro MGMTBUS_CACHEFLUSH.
 * Since Cougar is cache coherent, this isn't required.
 */
#ifdef MGMTBUS_CACHEFLUSH
#define MGMTBUS_SYNC_CACHE(addr, size, op)  xxxxx
#else
#define MGMTBUS_SYNC_CACHE(addr, size, op)
#endif

/*
 * Macro for aligning addresses and sizes to a the size of a cache line.
 */
#define MGMTBUS_ALIGNMENT   32
#define MGMTBUS_ALIGN(addr) ALIGN(addr, MGMTBUS_ALIGNMENT)

/*
 * Physical start of PCI space, different on SSC versus TxRx
 */
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
/*
 * On TxRx PCI Physical Address'es start at following address on Linux
 */
#define PCI_PHYS_START 0x1000000000UL
#else
#define PCI_PHYS_START 0xF800000000UL
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

/*
 * On TxRx Linux this PCI Share Memory(16 MB) area is mapped to
 * the following Physical Address of local memory.
 */
#define TXRX_PCI_PHY_START 0x1F0000000UL
#define TXRX_PCI_PHY_SIZE     0x800000UL

/*
 * Offsets of the CPUs memory in the PCI space.
 */
#define SSC_PCI_OFFSET      0x60000000UL
#define TXRX_PCI_OFFSET     0x1A000000UL
#define FP_PCI_OFFSET       0x1B000000UL

struct net_device_stats *
mgmtbus_get_stats(struct net_device *dev)
{
     return (&Mgmtbus_softc.mgmtbus_stats);
}

/*
 * Address conversion procedures for converting between physical and virtual
 * addresses.  All addresses in the management bus ring configuration table
 * are physical.
 *
 * On cougar the addresses passed around are 32-Bit offsets in PCI address space.
 * For the TLB memory mappings for the PCI areas, check out:
 * For TxRx Linux(tuxrx) on Cougar: ./include/asm-mips/mach-tuxrx/cougar.h
 * For TxRx Linux(tuxrx) on Cougar for PCI BIOS map irq: ./arch/mips/pci/fixup-tuxrx.c
 * Under EEE, the PCI mapping setup for TxRx is in:  ./nfx-tree/code/sm-pci/pci-init.c
 */
void *
mgmtbus_phys_to_virt(unsigned long phys)
{
    void *virt_addr = 0;

    if (MGMTBUS_SSC_BASE_ADDR <= phys &&
                phys < MGMTBUS_SSC_BASE_ADDR + SHARED_MEM_MGMT_SIZE) {

            virt_addr = (void *)(Mgmtbus_softc.ssc_pci_virt + ((phys) - MGMTBUS_SSC_BASE_ADDR));
	    MGMTBUS_PRINTK_X(("%s: SSC BASE_ADDR: phys 0x%p, virtual 0x%p\n",
		__FUNCTION__, (void *)phys, virt_addr));

    } else if (Mgmtbus_softc.ssc_pci_base <= phys &&
               phys < Mgmtbus_softc.ssc_pci_base + SHARED_MEM_MGMT_SIZE) {

            virt_addr = (void *)(Mgmtbus_softc.ssc_pci_virt +
				 (phys - Mgmtbus_softc.ssc_pci_base));
	    MGMTBUS_PRINTK_X(("%s: SSC ssc_pci_base: phys 0x%p, virtual 0x%p\n",
		__FUNCTION__, (void *)phys, virt_addr));

    } else if (Mgmtbus_softc.txrx_pci_base <= phys &&
               phys < Mgmtbus_softc.txrx_pci_base + SHARED_MEM_MGMT_SIZE) {
            virt_addr = (void *)(Mgmtbus_softc.txrx_pci_virt +
				 (phys - Mgmtbus_softc.txrx_pci_base));
	    MGMTBUS_PRINTK_X(("%s: TXRX phys 0x%p, virtual 0x%p\n",
		__FUNCTION__, (void *)phys, virt_addr));

    } else if (Mgmtbus_softc.fp_pci_base <= phys &&
               phys < Mgmtbus_softc.fp_pci_base + SHARED_MEM_MGMT_SIZE) {
             virt_addr = (void *)(Mgmtbus_softc.fp_pci_virt +
				  (phys - Mgmtbus_softc.fp_pci_base));
	    MGMTBUS_PRINTK_X(("%s: FP phys 0x%p, virtual 0x%p\n",
		__FUNCTION__, (void *)phys, virt_addr));
    } else {
		struct pt_regs *regs = get_irq_regs();

		printk("%s: invalid phys addr %#lx\n", __FUNCTION__, phys);
		die("Oops", regs);
    }
    return virt_addr;
}

void
mgmtbus_dumpRingConfig(mgmtbus_ring_config_t *t)
{
    printk("%s: DUMP START\n", __FUNCTION__);
    /*
     * Dump out all entries associated with SSC
     */
    printk("%s: table[SSC][SSC] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_SSC].avail_ring_address,
	   t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_SSC].used_ring_address,
	   t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_SSC].ring_size);

    printk("%s: table[SSC][TXRX] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].avail_ring_address,
	   t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].used_ring_address,
	   t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].ring_size);

    printk("%s: table[SSC][FP] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FP].avail_ring_address,
	   t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FP].used_ring_address,
	   t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FP].ring_size);

    printk("%s: table[SSC][FC] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FC].avail_ring_address,
	   t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FC].used_ring_address,
	   t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FC].ring_size);
    /*
     * Dump out all entries associated with TXRX
     */
    printk("%s: table[TXRX][SSC] avail 0x%x, used 0x%x, size 0x%x\n",
	__FUNCTION__,
	   t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].avail_ring_address,
	   t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].used_ring_address,
	   t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].ring_size);

    printk("%s: table[TXRX][TXRX] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_TXRX].avail_ring_address,
	   t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_TXRX].used_ring_address,
	   t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_TXRX].ring_size);

    printk("%s: table[TXRX][FP] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_FP].avail_ring_address,
	   t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_FP].used_ring_address,
	   t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_FP].ring_size);

    printk("%s: table[TXRX][FC] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_FC].avail_ring_address,
	   t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_FC].used_ring_address,
	   t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_FC].ring_size);
    /*
     * Dump out all entries associated with TXRX
     */
    printk("%s: table[FP][SSC] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_SSC].avail_ring_address,
	   t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_SSC].used_ring_address,
	   t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_SSC].ring_size);

    printk("%s: table[FP][TXRX] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_TXRX].avail_ring_address,
	   t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_TXRX].used_ring_address,
	   t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_TXRX].ring_size);

    printk("%s: table[FP][FP] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_FP].avail_ring_address,
	   t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_FP].used_ring_address,
	   t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_FP].ring_size);

    printk("%s: table[FP][FC] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_FC].avail_ring_address,
	   t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_FC].used_ring_address,
	   t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_FC].ring_size);

    printk("%s: table[FC][SSC] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_SSC].avail_ring_address,
	   t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_SSC].used_ring_address,
	   t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_SSC].ring_size);

    printk("%s: table[FC][TXRX] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_TXRX].avail_ring_address,
	   t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_TXRX].used_ring_address,
	   t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_TXRX].ring_size);

    printk("%s: table[FC][FP] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_FP].avail_ring_address,
	   t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_FP].used_ring_address,
	   t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_FP].ring_size);

    printk("%s: table[FC][FC] avail 0x%x, used 0x%x, size 0x%x\n",
	   __FUNCTION__,
	   t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_FC].avail_ring_address,
	   t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_FC].used_ring_address,
	   t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_FC].ring_size);

    printk("%s: DUMP END\n", __FUNCTION__);
    return;
}

/*
 * Routine Description:
 * Initialize management bus ring configuration table.  Addresses of rings
 * residing in SSC or TXRX memory are initialized to a valid address.
 * All other ring addresses are initialized to 0.  Each remote CPU is
 * responsible for initializing the addresses of its rings when they boot.
 *
 * Arguments:
 *  none
 *
 * Return Value:
 *  None.
 *
 * Saves the next available address for ssc OR txrx buffer allocations
 * in the management bus's state block.
 */
void
initConfigTable(void)
{
    mgmtbus_ring_config_t *t = Mgmtbus_softc.ring_config;
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
    unsigned long txrx_addr = Mgmtbus_softc.txrx_pci_base;
#else
    unsigned long ssc_addr = Mgmtbus_softc.ssc_pci_base;
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

    /*
     * Set the model number into the structure
     */
    t->model_num = (u16)(model_num & 0xffff);

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
    /*
     *******************************************
     * TxRx, aka 1480, running the driver.
     * Initialize our portion of ring configuration area
     * NOTE: The address in the RingConfig structure is accessed via
     * the PCI bus on the TxRx and via local memory addresses on SSC.
     *******************************************
     */
#ifdef MGMTBUS_DEBUG
    /*
     * On 1480, aka TxRx, dump SSC Initialized ringConfig Table
     */
    printk("%s: SSC Sent: ring_config 0x%p\n",
           __FUNCTION__, (void *)t);
    mgmtbus_dumpRingConfig(t);
#endif /* MGMTBUS_DEBUG */

    MGMTBUS_PRINTK(("%s: initRingTable START: txrx_addr 0x%p\n",
		    __FUNCTION__,
		    (void *)txrx_addr));
    /*
     * Set AVAILABLE addresses for TxRx Entries as src index
     */
    t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].ring_size = MGMTBUS_TXRX_2_SSC_RING_SIZE;
    t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].avail_ring_address = txrx_addr;

    /* t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].used_ring_address, SSC Sets */
    txrx_addr += MGMTBUS_RING_SIZE(MGMTBUS_TXRX_2_SSC_RING_SIZE);

    /*
     * Messages destined for the FP and FC are proxied through the TXRX.
     * This can be changed LATER when the FP is changed to directly handle
     * it's own messages. Until then, TxRx initializes it's view.
     *
     * The source cpu (SSC) sets 'used_ring_address' and the destination cpu (dst)
     * sets 'avail_ring_address' when it initializes the global config table.
     *
     * table[src][dst].avail_ring_address set by 'src' cpu, TXRX here
     * table[src][dst].used_ring_address set by 'dst' cpu, SSC here.
     *
     * NOTE: Since the TxRx is proxying messages from SSC destined to
     * either ACPU or FP, we set the 'avail_ring_address' fields for FP and FC here.
     */
    t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_FP].ring_size = MGMTBUS_TXRX_2_FP_RING_SIZE;
    t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_FP].avail_ring_address = txrx_addr;
    /* t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_FP].used_ring_address FP sets */
    txrx_addr += MGMTBUS_RING_SIZE(MGMTBUS_TXRX_2_FP_RING_SIZE);

    t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_FC].ring_size = MGMTBUS_TXRX_2_FC_RING_SIZE;
    t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_FC].avail_ring_address = txrx_addr;
    /* t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_FC].used_ring_address FC sets */
    txrx_addr += MGMTBUS_RING_SIZE(MGMTBUS_TXRX_2_FC_RING_SIZE);

    /*
     * Set USED addresses for TxRx Entries as dst index
     */
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].ring_size = MGMTBUS_SSC_2_TXRX_RING_SIZE;
    /* t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].avail_ring_address, SSC Sets */
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].used_ring_address = txrx_addr;
    txrx_addr += MGMTBUS_RING_SIZE(MGMTBUS_SSC_2_TXRX_RING_SIZE);

    t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_TXRX].ring_size = MGMTBUS_FP_2_TXRX_RING_SIZE;
    /* t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_TXRX].avail_ring_address = txrx_addr; => TxRx PROXY for FP */
    t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_TXRX].used_ring_address = txrx_addr;
    txrx_addr += MGMTBUS_RING_SIZE(MGMTBUS_FP_2_TXRX_RING_SIZE);

    t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_TXRX].ring_size = MGMTBUS_FC_2_TXRX_RING_SIZE;
    /* t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_TXRX].avail_ring_address = txrx_addr; => TxRx PROXY for FC */
    t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_TXRX].used_ring_address = txrx_addr;
    txrx_addr += MGMTBUS_RING_SIZE(MGMTBUS_FC_2_TXRX_RING_SIZE);

    MGMTBUS_PRINTK(("%s: initRingTable END: txrx_addr 0x%p\n",
		    __FUNCTION__,
		    (void *)txrx_addr));

    Mgmtbus_softc.txrx_pci_next_virt_avail = txrx_addr;
#else
    /*
     *******************************************
     * SSC Linux driver running on 1125, runs first
     * and ZERO'es Ring configuration area before
     * initializing the ring address'es.
     * ******************************************
     */
    memset(t, 0, sizeof(mgmtbus_ring_config_t));

    ssc_addr += MGMTBUS_ALIGN(sizeof(mgmtbus_ring_config_t));
    ssc_addr += sizeof(struct rcon_queue) * RCON_MAX_CORES;

    /* Save next available ssc buffer address for later */
    Mgmtbus_softc.ssc_pci_next_virt_avail = ssc_addr;

    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].ring_size = MGMTBUS_SSC_2_TXRX_RING_SIZE;
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].avail_ring_address = ssc_addr;
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].used_ring_address = 0; /* Set by TxRx */
    ssc_addr += MGMTBUS_RING_SIZE(MGMTBUS_SSC_2_TXRX_RING_SIZE);

    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FP].ring_size = MGMTBUS_SSC_2_FP_RING_SIZE;
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FP].avail_ring_address = ssc_addr;
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FP].used_ring_address = 0; /* Set by TxRx for FP */
    ssc_addr  += MGMTBUS_RING_SIZE(MGMTBUS_SSC_2_FP_RING_SIZE);

    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FC].ring_size = MGMTBUS_SSC_2_FC_RING_SIZE;
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FC].avail_ring_address = ssc_addr;
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FC].used_ring_address = 0; /* Set by TxRx for FC */
    ssc_addr  += MGMTBUS_RING_SIZE(MGMTBUS_SSC_2_FC_RING_SIZE);

    t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].ring_size = MGMTBUS_TXRX_2_SSC_RING_SIZE;
    t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].avail_ring_address = 0; /* Set by TxRx */
    t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].used_ring_address = ssc_addr;
    ssc_addr += MGMTBUS_RING_SIZE(MGMTBUS_TXRX_2_SSC_RING_SIZE);

    t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_SSC].ring_size = MGMTBUS_FP_2_SSC_RING_SIZE;
    t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_SSC].avail_ring_address = 0; /* Set by TxRx for FP */
    t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_SSC].used_ring_address = ssc_addr;
    ssc_addr += MGMTBUS_RING_SIZE(MGMTBUS_FP_2_SSC_RING_SIZE);

    t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_SSC].ring_size = MGMTBUS_FC_2_SSC_RING_SIZE;
    t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_SSC].avail_ring_address = 0; /* Set by TxRx for FC */
    t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_SSC].used_ring_address = ssc_addr;
    ssc_addr += MGMTBUS_RING_SIZE(MGMTBUS_FC_2_SSC_RING_SIZE);

    /* Save next available ssc buffer address for later */
    Mgmtbus_softc.ssc_pci_next_virt_avail = ssc_addr;

    MGMTBUS_SYNC_CACHE(t, sizeof(mgmtbus_ring_config_t), 1);

#ifdef MGMTBUS_DEBUG
    printk("%s: SSC Initialized: ring_config 0x%p\n",
           __FUNCTION__, (void *)t);

    mgmtbus_dumpRingConfig(t);
#endif /* MGMTBUS_DEBUG */

#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */
    return;
}

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
/*
 * Routine Description:
 *  Initialize all descriptor rings on TXRX
 *  residing in the PCI address space that was
 *  mapped to local memory.
 *
 * Arguments:
 *  None.
 *
 * Return Value:
 *  None.
 */
static void
initDescriptorRingsOnTxRx(void)
{
    mgmtbus_ring_config_t *t = Mgmtbus_softc.ring_config;
    unsigned long txrx_addr;
    u32 i, ring_size, rx_avail_ring_phy, rx_used_ring_phy, tx_avail_ring_phy;
    mgmtbus_msg_desc_ring_t *rx_avail_ring_virt, *rx_used_ring_virt, *tx_avail_ring_virt;

    /*
     * TXRX: Initialize AVAILABLE Descriptor Rings in TXRX memory.
     * These rings are used for sending data from TXRX to other CPUs.
     * Mark descriptors as used. The avail_ring_address is set by local
     * processor and the 'used_ring_address' is set by dst processor.
     * The destination processor initializes the ring table[src][dst]
     * entries when it starts.
     * 'src' processor initializes ring_table[src][dst].avail_ring_address 
     * 'dst' processor initializes ring_table[src][dst].used_ring_address
     *
     * Set txrx_addr to the starting address of the TXRX message buffers.
     * The message buffer memory is located immediately following the TXRX
     * descriptor ring memory.
     *
     * NOTE: Under eee, the txrx code did NOT set the entry for
     * table[MGMTBUS_ADDR_MAX][MGMTBUS_ADDR_TXRX].used_ring_address
     *
     * Hence logic was reworked to store 'next_avail_addr' in context block.
     */
    txrx_addr = Mgmtbus_softc.txrx_pci_next_virt_avail;

    MGMTBUS_PRINTK(("%s: ENTRY; ring_config 0x%p, txrx 0x%p\n",
		    __FUNCTION__,
		    (void *)t, (void *)txrx_addr));

    // Following should have been set in initConfig when TXRX runs, prior to calling us
    // tx_avail_ring = t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].avail_ring_address;

    // Following should have been set in initConfig when SSC runs, prior to calling us
    // tx_used_ring =  t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].used_ring_address;

    MGMTBUS_PRINTK(("%s: t->table[TXRX][SSC].avail(tx_avail_ring) Phys: 0x%x, Used(tx_used_ring) Phys 0x%x\n",
		__FUNCTION__,
		t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].avail_ring_address,
		t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].used_ring_address));

    rx_avail_ring_phy = t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].avail_ring_address; 
    rx_avail_ring_virt = mgmtbus_phys_to_virt(rx_avail_ring_phy);

    MGMTBUS_PRINTK(("%s: t->table[SSC][TXRX].avail_ring_addr(rx_avail) <Phys: 0x%x, Virt: 0x%p>\n",
		__FUNCTION__,
		rx_avail_ring_phy, rx_avail_ring_virt));

    /*
     * t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].used_ring_address
     * is set by TXRX in initConfigTable
     */
    rx_used_ring_phy = t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].used_ring_address;
    rx_used_ring_virt = mgmtbus_phys_to_virt(rx_used_ring_phy);

    MGMTBUS_PRINTK(("%s: t->table[SSC][TXRX].used_ring_addr(rx_used) <Phys: 0x%x, Virt: 0x%p>\n",
		__FUNCTION__,
		rx_used_ring_phy,
		rx_used_ring_virt));

    rx_used_ring_virt->next_index = 0;
    /*
     * Initialize USED Descriptor Rings in TxRx memory.
     * These rings are used for receiving messages sent by remote CPUs.
     * Initialize descriptors to point to TxRx message buffers.
     *
     * t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].ring_size
     * is set in initConfigTable()
     */
    MGMTBUS_PRINTK(("%s: INIT USED DESC; t->table[TXRX][SSC].ring_size 0x%x\n",
		__FUNCTION__,
		t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].ring_size));

    rx_used_ring_virt->ring_size = ring_size = t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].ring_size;

    for (i = 0; i < ring_size; i++) {
	rx_used_ring_virt->ring[i].flags = 0;
	rx_used_ring_virt->ring[i].index = i;
	rx_used_ring_virt->ring[i].size = MGMTBUS_MTU;
	rx_used_ring_virt->ring[i].address = txrx_addr;

	rx_avail_ring_virt->ring[i].flags = 0;
	rx_avail_ring_virt->ring[i].index = i;
	rx_avail_ring_virt->ring[i].size = MGMTBUS_MTU;
	rx_avail_ring_virt->ring[i].address = txrx_addr;
        txrx_addr += MGMTBUS_MTU;
    }

#ifdef MGMTBUS_DEBUG
    for (i = 0; i < 4; i++) {
    MGMTBUS_PRINTK(("%s: rx_used_ring_virt->ring[%d] 0x%p, index %d, size %d, address Phys: 0x%x\n",
		__FUNCTION__,
		i,
		&(rx_used_ring_virt->ring[i]),
		rx_used_ring_virt->ring[i].index,
		rx_used_ring_virt->ring[i].size,
		rx_used_ring_virt->ring[i].address));

    MGMTBUS_PRINTK(("%s: rx_avail_ring_virt->ring[%d] 0x%p, index %d, size %d, address Phys: 0x%x\n",
		__FUNCTION__,
		i,
		&(rx_avail_ring_virt->ring[i]),
		rx_avail_ring_virt->ring[i].index,
		rx_avail_ring_virt->ring[i].size,
		rx_avail_ring_virt->ring[i].address));
    }
#endif /* MGMTBUS_DEBUG */

    /*
     * Initialize AVAIL Descriptor Rings residing in TxRx memory.
     * These rings are used to transmit messages to remote CPUs.
     * Initialize buffer address'es in the descriptors to point
     * to SSC message buffers, which have already been setup.
     *
     * t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].used_ring_address
     * was set by SSC during it's initialization.
     *
     * We initialize the avail ring descriptors associated with txrx transmit
     * support. The descriptors are found in
     * t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].avail_ring_address
     * We use the 'used' buffer offsets supplied by SSC.
     */
    rx_used_ring_phy = t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].used_ring_address;
    rx_used_ring_virt = mgmtbus_phys_to_virt(rx_used_ring_phy);

    tx_avail_ring_phy = t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].avail_ring_address;
    tx_avail_ring_virt = mgmtbus_phys_to_virt(tx_avail_ring_phy);

    tx_avail_ring_virt->ring_size = ring_size = t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].ring_size;

    MGMTBUS_PRINTK(("%s: TXRX INIT AVAIL DESC, t->table[TXRX][SSC].avail virt 0x%p, phy 0x%x, ring_size 0x%x\n",
		__FUNCTION__,
		tx_avail_ring_virt, tx_avail_ring_phy, ring_size));

    MGMTBUS_PRINTK(("%s: TXRX INIT AVAIL DESC, SSC Used; t->table[TXRX][SSC].used virt 0x%p, phy 0x%x\n",
		__FUNCTION__,
		rx_used_ring_virt,
		rx_used_ring_phy));

    for (i = 0; i < ring_size; i++) {
	tx_avail_ring_virt->ring[i].flags = 0;
	tx_avail_ring_virt->ring[i].index = i;
	tx_avail_ring_virt->ring[i].size = MGMTBUS_MTU;
	tx_avail_ring_virt->ring[i].address = rx_used_ring_virt->ring[i].address;
    }

#ifdef MGMTBUS_DEBUG
    for (i = 0; i < 4; i++) {
    MGMTBUS_PRINTK(("%s: SSC Set; rx_used_ring_virt->ring[%d] 0x%p, index %d, size %d, address Phys: 0x%x\n",
		__FUNCTION__,
		i,
		&(rx_used_ring_virt->ring[i]),
		rx_used_ring_virt->ring[i].index,
		rx_used_ring_virt->ring[i].size,
		rx_used_ring_virt->ring[i].address));

    MGMTBUS_PRINTK(("%s: TXRX Set; tx_avail_ring_virt->ring[%d] 0x%p, index %d, size %d, address Phys: 0x%x\n",
		__FUNCTION__,
		i,
		&(tx_avail_ring_virt->ring[i]),
		tx_avail_ring_virt->ring[i].index,
		tx_avail_ring_virt->ring[i].size,
		tx_avail_ring_virt->ring[i].address));
    }
#endif /* MGMTBUS_DEBUG */

    MGMTBUS_SYNC_CACHE(ring,
     sizeof(mgmtbus_msg_desc_ring_t)+(ring->ring_size*sizeof(mgmtbus_msg_desc_t)),1);

    /* Save next available txrx buffer address */
    Mgmtbus_softc.txrx_pci_next_virt_avail = txrx_addr;

    MGMTBUS_PRINTK(("%s: RingConfig 0x%p, END Ring Init: phy txrx_addr 0x%p\n",
		    __FUNCTION__, (void *)t, (void *)txrx_addr));

    mgmtbus_dumpRingConfig(t);
    return;
}

#else

/*
 * Routine Description:
 *  Initialize all descriptor rings on SSC
 *  residing in the PCI address space that was
 *  mapped to local memory.
 *
 * Arguments:
 *  None.
 *
 * Return Value:
 *  None.
 */
void
initDescriptorRingsOnSSC(void)
{
    mgmtbus_ring_config_t *t = Mgmtbus_softc.ring_config;
    uint32_t dest, src, i;
    mgmtbus_msg_desc_ring_t *ring;
    unsigned int mgmtbus_addr_index;
    u32 ring_phy;
    unsigned long ssc_addr;

    /*
     * Set ssc_addr to the starting address of the SSC message buffers.
     * The message buffer memory is located immediately following the SSC
     * descriptor ring memory.
     */
    mgmtbus_addr_index = MGMTBUS_ADDR_SSC;
    ssc_addr = Mgmtbus_softc.ssc_pci_next_virt_avail;

    MGMTBUS_PRINTK(("%s: ring_config 0x%p, ssc 0x%p\n",
		    __FUNCTION__,
		    (void *)t, (void *)ssc_addr));
    /*
     * SSC: Initialize AVAILABLE Descriptor Rings in SSC memory.
     * These rings are used for sending data from the SSC to other CPUs.
     * Mark descriptors as used. They will be initialized
     * by each destination processor when it starts.
     * NOTE: Loop index starts at 1, since SSC index is zero.
     */
    for (dest = 1; dest <= MGMTBUS_ADDR_MAX; dest++) {

        ring_phy = t->table[MGMTBUS_ADDR_SSC][dest].avail_ring_address;

        ring = mgmtbus_phys_to_virt(ring_phy);
        ring->next_index = 0;
        ring->ring_size = t->table[MGMTBUS_ADDR_SSC][dest].ring_size;

	MGMTBUS_PRINTK(("%s: Initialize AVAIL DESC RING ring 0x%p, ring->ring[0] 0x%p, ring_size %d\n",
		__FUNCTION__,
		ring,
		&(ring->ring[0]),
	        ring->ring_size));

        for (i = 0; i < ring->ring_size; i++) {
	        /*
		 * Keep SSC from transmitting until TxRx init and alloc's buffers
		 * The flags field is cleared by TxRx
		 */
		ring->ring[i].flags = MGMTBUS_DESC_FLAGS_USED;
		ring->ring[i].index = i;
		ring->ring[i].size = 0;
		ring->ring[i].address = 0; /* Set by TxRx */
        }

	if (dest == MGMTBUS_ADDR_TXRX) { /* dump first few ring entries */

	  for (i = 0; i < 4; i++) {
 MGMTBUS_PRINTK(("%s: AVAIL Entry; table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX]; ring->ring[%d] 0x%p, flags 0x%x, index %d, size %d, address 0x%x\n",
		__FUNCTION__,
		i,
		&(ring->ring[i]),
		ring->ring[i].flags,
		ring->ring[i].index,
		ring->ring[i].size,
		ring->ring[i].address));
	  }
	}

        MGMTBUS_SYNC_CACHE(ring,
         sizeof(mgmtbus_msg_desc_ring_t)+(ring->ring_size*sizeof(mgmtbus_msg_desc_t)),1);
    }
    /*
     * Initialize USED Descriptor Rings in use local SSC memory.
     * These rings are used for receiving messages sent by remote CPUs.
     * Initialize descriptors to point to SSC message buffers.
     */
    for (src = 1; src <= MGMTBUS_ADDR_MAX; src++) {

        ring_phy = t->table[src][MGMTBUS_ADDR_SSC].used_ring_address;
        ring = mgmtbus_phys_to_virt(ring_phy);
        ring->next_index = 0;
        ring->ring_size = t->table[src][MGMTBUS_ADDR_SSC].ring_size;

	MGMTBUS_PRINTK(("%s: Initialize USED ring 0x%p, ring->ring[0] 0x%p, ring_size %d\n",
		__FUNCTION__,
		ring,
		&(ring->ring[0]),
	        ring->ring_size));

        for (i = 0; i < ring->ring_size; i++) {
            ring->ring[i].flags = 0;
            ring->ring[i].index = i;
            ring->ring[i].size = MGMTBUS_MTU;
            ring->ring[i].address = ssc_addr;
            ssc_addr += MGMTBUS_MTU;
        }

	if (src == MGMTBUS_ADDR_TXRX) { /* dump first few ring entries */

	  for (i = 0; i < 4; i++) {
 MGMTBUS_PRINTK(("%s: USED Entry; table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC]; ring->ring[%d] 0x%p, flags 0x%x, index %d, size %d, address 0x%x\n",
		__FUNCTION__,
		i,
		&(ring->ring[i]),
		ring->ring[i].flags,
		ring->ring[i].index,
		ring->ring[i].size,
		ring->ring[i].address));
	  }
	}

        MGMTBUS_SYNC_CACHE(ring,
         sizeof(mgmtbus_msg_desc_ring_t)+(ring->ring_size*sizeof(mgmtbus_msg_desc_t)),1);
    }

    /* Save next available ssc buffer address */
    Mgmtbus_softc.ssc_pci_next_virt_avail = ssc_addr;

    MGMTBUS_PRINTK(("%s: RingConfig 0x%p, END Ring Init: phy ssc_addr 0x%p\n",
		    __FUNCTION__, (void *)t, (void *)ssc_addr));
    mgmtbus_dumpRingConfig(t);

    return;
}
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

/*
 * Routine Description:
 *  Convert an eee address to a management bus address.
 *  A management bus address is the index of the ring
 *   within the ring configuration table.
 *
 * Arguments:
 *  port - eee destination port
 *
 * Return Value:
 *  Returns the index in the configuration table.
 */
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
/*
 * TXRX Transmist destination neteee address parser
 */
inline
int mgmtbus_convert_eee_addr(uint32_t port) {return MGMTBUS_ADDR_SSC;}
#else
/*
 * SSC Transmist destination neteee address parser
 */
int
mgmtbus_convert_eee_addr(uint32_t port)
{
    int addr = -1;
    uint32_t dest_slot, dest_cpu;

    dest_cpu = EEE_GET_DEST_CPU(port);
    dest_slot = EEE_GET_SLOT_NUM(port);

    switch (dest_slot) {

    case NFX_SLOTTYPE_SSC:
        if (dest_cpu == NFX_SSC_UNIX) {
            addr = MGMTBUS_ADDR_SSC;
        }
        break;

    case NFX_SLOTTYPE_NFP:
        if ((NFX_NFP_TXRX1 <= dest_cpu) && (dest_cpu <= NFX_NFP_FP_ANY)) {
		/*
		 * XXX
		 * @@@ TxRx currently provides a proxy service for FP messages so
		 * all FP messages need to be sent to the TxRx.
		 * This can be changed when the embedded OS is changed.
		 * Sometime in the future.
		 */
            addr = MGMTBUS_ADDR_TXRX;
        }
        break;

    case NFX_SLOTTYPE_FC:
        addr = MGMTBUS_ADDR_TXRX;
        break;

    default:
        printk("%s: BOGUS Address: 0x%x, dest_cpu 0x%x, dest_slot 0x%x\n",
	       __FUNCTION__, port, dest_cpu, dest_slot);
        break;
    }
    return addr;
}
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)

/*
 * Executing on 1480 TXRX generate mailbox 0 interrupt on 1125 SSC
 */
void
mgmtbus_send_interrupt_to_ssc(void)
{
#ifdef MGMTBUS_DEBUG
    uint64_t *mailbox = (uint64_t *)PHYS_TO_XKSEG_UNCACHED(MGMTBUS_SSC_INTR_MAILBOX);

    printk("%s: Writing MGMTBUS_SSC_INTERRUPT_MAILBOX: 0x%p\n",
		__FUNCTION__, mailbox);
#endif /* MGMTBUS_DEBUG */

     *((volatile uint64_t *)MGMTBUS_SSC_INTERRUPT_REG) = MGMTBUS_SSC_INTERRUPT_VAL;
     return;
}
#else

/*
 * Executing on 1125 SSC generate mailbox 0 interrupt on 1480 TXRX
 */
void
mgmtbus_send_interrupt_to_txrx(void)
{
#ifdef MGMTBUS_DEBUG
     uint64_t *mailbox = (uint64_t *)PHYS_TO_XKSEG_UNCACHED(MGMTBUS_TXRX_INTR_MAILBOX);

     printk("%s: Writing MGMTBUS_TXRX_INTERRUPT_MAILBOX: 0x%p\n",
		__FUNCTION__, mailbox);
#endif /* MGMTBUS_DEBUG */

     *((volatile uint64_t *)MGMTBUS_TXRX_INTERRUPT_REG) = MGMTBUS_TXRX_INTERRUPT_VAL;
     return;
}
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

/*
 * Routine Description:
 *  Management bus device driver transmit function
 *  Packet in skb arrrives with following headers:
 *
 *  1) Ethernet Header => 14 bytes
 *  2) Agile Header(18 Bytes defined, 16 on bus)
 *     2 bytes of header pad are NOT sent!
 *  3) Data Bytes
 *
 * Arguments:
 *  skb - Address of skb to send
 *  dev - Address of network device structure
 *
 * Return Value:
 *  Returns 0
 *
 * Messages that can NOT be transmitted due to ring
 * full conditions, no skb's, etc are dropped and
 * the interfaces statistics updated to reflect either
 * errors and drops.
 */
int
mgmtbus_hard_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
    mgmtbus_ring_config_t *t;
    mgmtbus_msg_desc_ring_t *used_ring, *avail_ring;
    volatile mgmtbus_msg_desc_t *used_msg_desc, *avail_msg_desc;
    agile_hdr_t *hdr, a_hdr;
    int table_dst_index, table_src_index;
    int agile_hdr_len, copy_len, len, send_size;
    void *virt;
    unsigned long irq_save_flags;
    unsigned short skb_protocol;
    struct iphdr *ip_hdr;
#ifdef MGMTBUS_DEBUG
    int i, dump_size;
    unsigned char data_byte;
    unsigned char *src_addr, *dst_addr;
#endif /* MGMTBUS_DEBUG */
    
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
    table_src_index = MGMTBUS_ADDR_TXRX;
#else
    table_src_index = MGMTBUS_ADDR_SSC;
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

    /* Get ring configuration record */
    t = Mgmtbus_softc.ring_config;
 
    /* Get pointers to mac and agile header from skb */
    skb_protocol = ntohs(skb->protocol);

    /*
     * Get pointer to ppp like stock header on packet
     */
    hdr = (agile_hdr_t *)skb_mac_header(skb);

    MGMTBUS_PRINTK(("%s: TX Pkt START; skb 0x%p, skb->len %d, skb->protocol 0x%x, dev: %s\n",
	__FUNCTION__, skb, skb->len, skb_protocol, skb->dev->name));

    MGMTBUS_PRINTK(("%s: TX Pkt; skb->data 0x%p, skb->mac_hdr 0x%p, skb->net_hdr 0x%p, hdr 0x%p\n",
	__FUNCTION__, skb->data, skb->mac_header, skb->network_header, hdr));

    /*
     * Claim lock to prevent transmit ring corruption
     */
    spin_lock_irqsave(&Mgmtbus_softc.tx_lock, irq_save_flags);

    switch (skb_protocol) {

      case ETH_P_IP:   /* Transmit IPv4 Packet to other end */

	MGMTBUS_PRINTK(("%s: TX IPv4 Pkt; skb_protocol 0x%x\n",
		__FUNCTION__, skb_protocol));

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
	/* On TxRx, send to SSC */
	table_dst_index = MGMTBUS_ADDR_SSC;
#else
	/* On SSC, send to TXRX */
	table_dst_index = MGMTBUS_ADDR_TXRX;
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */
	break;

      case ETH_P_IPV6:

	/* drop it until IPv6 support added, if ever for forwarding */
	MGMTBUS_PRINTK(("%s: TX IPv6 Pkt; DROPPING skb_protocol 0x%x\n",
		__FUNCTION__,
		skb_protocol));
	Mgmtbus_softc.mgmtbus_stats.rx_dropped++;;
	goto out;

       case ETH_P_EEE:

	MGMTBUS_PRINTK(("%s: TX NETEEE Pkt; skb_protocol 0x%x\n",
		__FUNCTION__, skb_protocol));

	hdr = (agile_hdr_t *)skb_network_header(skb);

#ifdef MGMTBUS_DEBUG
	printk("%s: TX NETEEE Pkt; Agile hdr; dst 0x%x, src 0x%x, offset 0x%x, h_proto 0x%x\n",
	__FUNCTION__,
	hdr->dest_port,
	hdr->src_port,
	hdr->offset,
	ntohs(hdr->h_proto));

	src_addr = (unsigned char *)hdr;
	printk("%s: TX NETEEE Pkt; AGILE Hdr\n", __FUNCTION__);
	for (i = 0; i < sizeof(agile_hdr_t); i++) {
	  data_byte = *((unsigned char *)(src_addr+i));
	  printk("%02x ", data_byte);
	}
	printk("\n");
#endif /* MGMTBUS_DEBUG */

	MGMTBUS_PRINTK(("%s: TX NETEEE Pkt; <slot,cpu,app> FROM %d.%d.%d TO %d:%d:%d, h_proto 0x%x\n",
		__FUNCTION__,
		EEE_GET_SLOT_NUM(hdr->src_port),
		EEE_GET_DEST_CPU(hdr->src_port),
		EEE_GET_APP_ID(hdr->src_port),
		EEE_GET_SLOT_NUM(hdr->dest_port),
		EEE_GET_DEST_CPU(hdr->dest_port),
		EEE_GET_APP_ID(hdr->dest_port),
		ntohs(hdr->h_proto)));

	 /*
	  * NetEEE pkt so examine Agile header
	  * Convert header dst address tuple
	  * to a dst mgmtbus index.
	  */
	 table_dst_index = mgmtbus_convert_eee_addr(hdr->dest_port);

	 if (table_dst_index == -1)  { /* failure */

    MGMTBUS_PRINTK(("%s: TX NETEEE Pkt; Dropping pkt No dest_port(addr) support 0x%x, <slot,cpu,app> FROM %d.%d.%d TO %d:%d:%d\n",
		__FUNCTION__,
		hdr->dest_port,
		EEE_GET_SLOT_NUM(hdr->src_port),
		EEE_GET_DEST_CPU(hdr->src_port),
		EEE_GET_APP_ID(hdr->src_port),
		EEE_GET_SLOT_NUM(hdr->dest_port),
		EEE_GET_DEST_CPU(hdr->dest_port),
		EEE_GET_APP_ID(hdr->dest_port)));

             Mgmtbus_softc.mgmtbus_stats.tx_dropped++;
	     goto out;
	 }
	break;

       default: /* Ignore protocol type */
         MGMTBUS_PRINTK(("%s: TX Pkt END; UNKNOWN skb_protocol 0x%x, Dropping Pkt\n",
		__FUNCTION__, skb_protocol));
	 Mgmtbus_softc.mgmtbus_stats.rx_dropped++;;
	 goto out;
    }

    MGMTBUS_SYNC_CACHE(t, sizeof(mgmtbus_ring_config_t), 0);
    /*
     * Verify destination CPU 'addr' has completed initialization.
     */
    if (t->table[table_src_index][table_dst_index].used_ring_address == 0) {

      MGMTBUS_PRINTK(("%s: TX Pkt END; <slot,cpu,app> TO %d:%d:%d, NULL table[%d][%d].used_ring_address\n",
		      __FUNCTION__,
		      EEE_GET_SLOT_NUM(hdr->dest_port),
		      EEE_GET_DEST_CPU(hdr->dest_port),
		      EEE_GET_APP_ID(hdr->dest_port),
		      table_src_index, table_dst_index));

        Mgmtbus_softc.mgmtbus_stats.tx_dropped++;
        goto out;
    }

    len = skb->len;
    used_ring = (mgmtbus_msg_desc_ring_t *)
      mgmtbus_phys_to_virt(t->table[table_src_index][table_dst_index].used_ring_address);

    avail_ring = (mgmtbus_msg_desc_ring_t *)
      mgmtbus_phys_to_virt(t->table[table_src_index][table_dst_index].avail_ring_address);

    avail_msg_desc = &avail_ring->ring[avail_ring->next_index];
    MGMTBUS_SYNC_CACHE(avail_msg_desc, sizeof(*avail_msg_desc), 0);

    MGMTBUS_PRINTK(("%s: TX Pkt; table[%d][%d] avail_ring 0x%p, used_ring 0x%p, avail_ring->next_index %d\n",
        __FUNCTION__,
        table_src_index, table_dst_index,
        avail_ring, used_ring, avail_ring->next_index));

    MGMTBUS_PRINTK(("%s: TX Pkt; avail_msg_desc 0x%p, addr 0x%x, flags 0x%x, size %d\n",
        __FUNCTION__,
	avail_msg_desc, avail_msg_desc->address, avail_msg_desc->flags, avail_msg_desc->size));

    MGMTBUS_PRINTK(("%s: TX Pkt; skb 0x%p, skb->len %d\n",
        __FUNCTION__,
        skb, skb->len));
    /*
     * Verify that ring has free slots to use
     */
    if (avail_msg_desc->flags & MGMTBUS_DESC_FLAGS_USED) { /* BAIL out as used entry */

#ifdef MGMTBUS_DEBUG
        printk("%s: TX Pkt; Available Ring FULL, skb 0x%p, skb->len %d, used_ring 0x%p, avail_ring 0x%p\n",
        __FUNCTION__,
        skb, skb->len,
	used_ring, avail_ring);

        printk("%s: TX Pkt END; avail_msg_desc, 0x%p, flags 0x%x, addr 0x%x, next_index %d\n",
        __FUNCTION__,
	avail_msg_desc, avail_msg_desc->address, avail_msg_desc->flags, avail_ring->next_index);
#else
        printk("%s: TX Pkt; Available Ring FULL, Pkt Dropped\n", __FUNCTION__);
#endif /* MGMTBUS_DEBUG */

        Mgmtbus_softc.mgmtbus_stats.tx_dropped++;
        goto out;
    }

    /*
     * Verify descriptor references a message buffer.
     */
    if (avail_msg_desc->address == 0) {

        Mgmtbus_softc.mgmtbus_stats.tx_errors++;
        Mgmtbus_softc.mgmtbus_stats.tx_dropped++;

	printk(KERN_ERR "%s: TX Pkt END; NULL ADDRESS avail_msg_desc 0x%p, avail next_index %d\n",
	       __FUNCTION__,
	       avail_msg_desc, avail_ring->next_index);

	printk(KERN_ERR "%s: TX Pkt END; NULL ADDRESS addr 0x%x, flags 0x%x, index %d, size %d\n",
	       __FUNCTION__,
	       avail_msg_desc->address, avail_msg_desc->flags,
	       avail_msg_desc->index, avail_msg_desc->size);
        goto out;
    }

    /*
     * Get virtual address of shared memory message buffer
     * from available descriptor.
     *
     * Packet arrives with the following headers and data:
     *
     * Ethernet header (14 bytes)
     * Agile_header_t excludes first 2 byte pad (16 bytes)
     * Data
     */
    virt = mgmtbus_phys_to_virt(avail_msg_desc->address);

    switch (skb_protocol) {

      case ETH_P_IP:
	 /*
	  * Transmit IPv4 Packet to other end
	  * Arrival Packet format
	  *
	  * Ethernet Header (14 Bytes)
	  * IP Header (20 Bytes)
	  * Data
	  *
	  * We strip off the Ethernet header
	  * and create a shortened Agile header
	  * minus the first 2 pad bytes.
	  * Departure Packet format
	  *
	  * Agile Header (16 bytes)
	  * IP Header (20 Bytes)
	  * Data
	  */
	ip_hdr = (struct iphdr *)skb_network_header(skb);

	MGMTBUS_PRINTK(("%s: TX IPv4 Pkt; IP hdr 0x%p\n", __FUNCTION__, ip_hdr));

#ifdef MGMTBUS_DEBUG
	 src_addr = (unsigned char *)ip_hdr;
	 for (i = 0; i < sizeof(struct iphdr); i++) {
	   data_byte = *((unsigned char *)(src_addr+i));
	   printk("%02x ", data_byte);
	 }
	 printk("\n");

#endif /* MGMTBUS_DEBUG */

	 memset(&a_hdr, 0, sizeof(agile_hdr_t));
	 a_hdr.h_proto = htons(ETH_P_IP);

	 agile_hdr_len = sizeof(agile_hdr_t) - sizeof(hdr->pad);

	 /*
	  * Remove Ethernet header(14) from total skb length
	  */
	 copy_len = skb->len - ETH_HLEN;

	 /*
	  * COPY Agile header and data into AVAILABLE Descriptor Ring.
	  * The offset of the buffer can be found there for this cpu.
	  * Copy in the Agile Header, shortened into descriptor buffer
	  */
	 memcpy(virt, &a_hdr.dest_port, agile_hdr_len);

	 /* Copy in IP header and data into descriptor ring buffer */
	 memcpy((void *)(virt+agile_hdr_len), ip_hdr, copy_len);
	 send_size = agile_hdr_len + copy_len;

#ifdef MGMTBUS_DEBUG
	 src_addr = (unsigned char *)(skb->data);
	 printk("%s: TX Pkt; SRC(skb): skb->data: 0x%p, skb->len %d\n",
		__FUNCTION__, src_addr, skb->len);
	 for (i = 0; i < skb->len; i++) {
	   data_byte = *((unsigned char *)(src_addr+i));
	   printk("%02x ", data_byte);
	 }
	 printk("\n");

	 src_addr = (unsigned char *)(&a_hdr.dest_port);
	 printk("%s: TX Pkt; SRC(Agile hdr): Virt 0x%p, agile_hdr_len %d\n",
		__FUNCTION__, src_addr, agile_hdr_len);
	 for (i = 0; i < agile_hdr_len; i++) {
	   data_byte = *((unsigned char *)(src_addr+i));
	   printk("%02x ", data_byte);
	 }
	 printk("\n");

	 src_addr = (unsigned char *)(ip_hdr);
	 printk("%s: TX Pkt; SRC(IP hdr+data): Virt 0x%p, copy_len %d\n",
		__FUNCTION__, src_addr, copy_len);

	 for (i = 0; i < copy_len; i++) {
	   data_byte = *((unsigned char *)(src_addr+i));
	   printk("%02x ", data_byte);
	 }
	 printk("\n");

	 printk("%s: TX Pkt; DST(avail desc): Virt 0x%p, phys 0x%x, send_size %d\n",
		__FUNCTION__, virt, avail_msg_desc->address, send_size);

	 dst_addr = (unsigned char *)virt;
	 dump_size = agile_hdr_len + copy_len;
	 for (i = 0; i < dump_size; i++) {
	   data_byte = *((unsigned char *)(dst_addr+i));
	   printk("%02x ", data_byte);
	 }
	 printk("\n");
#endif /* MGMTBUS_DEBUG */
	 break;

       case ETH_P_EEE:
	 /*
	  * Arriving packet to transmit, aka NETEEE pkt
	  * Packet to other end, arrives with format:
	  *
	  * Ethernet Header (14 bytes)
	  * Agile Header (16 bytes), stock is 18 bytes
	  *   the 2 pad bytes are removed before xmit
	  * Data
	  *
	  * Departing packet sent over mgmt bus leaves
	  * with following format:
	  *
	  * Agile Header (16 bytes) 2 pad bytes removed
	  * Data
	  *
	  * NOTE: hdr here points to Agile header
	  * The guess is that chopping off two bytes
	  * of agile header pad is for alignment purposes
	  * into the descriptors.
	  */
	 send_size = skb->len - sizeof(hdr->pad) - ETH_HLEN;

	 /*
	  * COPY skb headers and data into AVAILABLE Descriptor Ring
	  * area located in the local memory for this cpu
	  */
	 memcpy(virt, &hdr->dest_port, send_size);

#ifdef MGMTBUS_DEBUG
	 src_addr = (unsigned char *)(&hdr->dest_port);

	 printk("%s: TX Pkt; SRC(skb buffer): Virt 0x%p\n",
		__FUNCTION__, src_addr);

	 dump_size = ETH_HLEN + sizeof(agile_hdr_t);
	 printk("%s: TX Pkt; SRC(skb data)\n", __FUNCTION__);
	 for (i = 0; i < dump_size; i++) {
	   data_byte = *((unsigned char *)(src_addr+i));
	   printk("%02x ", data_byte);
	 }
	 printk("\n");

	 printk("%s: TX Pkt; DST(avail desc): Virt 0x%p, phys 0x%x, send_size %d\n",
		__FUNCTION__, virt, avail_msg_desc->address, send_size);

	 dst_addr = (unsigned char *)virt;
	 printk("%s: TX Pkt; DST(avail_msg_desc data)\n", __FUNCTION__);
	 for (i = 0; i < dump_size; i++) {
	   data_byte = *((unsigned char *)(dst_addr+i));
	   printk("%02x ", data_byte);
	 }
	 printk("\n");
#endif /* MGMTBUS_DEBUG */
	 break;

      default:
	MGMTBUS_PRINTK(("%s: TX Pkt; Drop Pkt; UNKNOWN skb_protocol 0x%x\n",
			__FUNCTION__, skb_protocol));
         Mgmtbus_softc.mgmtbus_stats.tx_dropped++;
	 goto out;
    }

    used_msg_desc = &used_ring->ring[avail_ring->next_index];

    /*
     * Update local end's 'available' descriptor, and
     * indicate descriptor contains packet data.
     */
    avail_msg_desc->flags = MGMTBUS_DESC_FLAGS_USED;
    MGMTBUS_SYNC_CACHE(avail_msg_desc, sizeof(*avail_msg_desc), 1);

    /*
     * Update remote end's 'used' descriptor, hence
     * informing them of the arrival of the new message.
     */
    used_msg_desc->size = send_size;
    used_msg_desc->flags = MGMTBUS_DESC_FLAGS_USED;
    MGMTBUS_SYNC_CACHE(used_msg_desc, sizeof(*used_msg_desc), 1);

    MGMTBUS_PRINTK(("%s: TX Pkt; used_msg_desc 0x%p, addr 0x%x, flags 0x%x, size %d\n",
        __FUNCTION__,
	used_msg_desc, used_msg_desc->address, used_msg_desc->flags, used_msg_desc->size));

    /*
     * Advance the next available slot index
     */
    avail_ring->next_index++;

    /*
     * Check for warp-around case and handle appropriately
     */
    if (avail_ring->next_index == avail_ring->ring_size) {
          avail_ring->next_index = 0;
    }

    /*
     * Update the transmit packet and byte counts
     */
    Mgmtbus_softc.mgmtbus_stats.tx_packets++;
    Mgmtbus_softc.mgmtbus_stats.tx_bytes += send_size;

    /*
     * Drop spin lock before sending
     * interrupt signal to other end
     */
    spin_unlock_irqrestore(&Mgmtbus_softc.tx_lock, irq_save_flags);
    kfree_skb(skb);

    /*
     * Send interrupt to other end to notify new
     * message arrival. The message content is
     * accessed via shared PCI space mappings of
     * the buffers between the processors.
     */
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
    /* Running on 1480 TXRX generate interrupt on 1125 SSC */
    mgmtbus_send_interrupt_to_ssc();
#else
    /* Running on 1125 SSC generate interrupt on 1480 TXRX */
    mgmtbus_send_interrupt_to_txrx();
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

    MGMTBUS_PRINTK(("%s: TX Pkt END; avail_ring->ring_size %d, avail_ring->next_index %d\n",
	__FUNCTION__,
	avail_ring->ring_size, avail_ring->next_index));

    return 0;
out:
    spin_unlock_irqrestore(&Mgmtbus_softc.tx_lock, irq_save_flags);
    kfree_skb(skb);
    return 0;
}

/*
 * Routine Description:
 *  Common interrupt handler for pci message register interrupts.
 *  The handler runs on either 1125(SSC) or 1480(TxRx, SMP) Linux kernels.
 *
 * Arguments:
 *  cause - the combined interrupt cause bit mask. If the bit is set,
 *          the corresponding cpu input ring requires processing.
 *
 * Return Value:
 *  None.
 */
void
mgmtbus_intr(uint32_t cause)
{
    mgmtbus_ring_config_t *t;
    struct sk_buff *skb;
    agile_hdr_t *hdr, *agile_hdr_rx;
    struct ethhdr *eth;
    u32 used_ring_phy, avail_ring_phy;
    mgmtbus_msg_desc_ring_t *used_ring, *avail_ring;
    mgmtbus_msg_desc_t *used_desc, *avail_desc;
    unsigned int table_src_index, table_dst_index;
    void *used_desc_virt_addr;
    char *skb_dst_addr;
    unsigned short hdr_h_proto;
    struct iphdr *ip_hdr_rx;
    unsigned int hdr_size, copy_size;
#ifdef MGMTBUS_DEBUG
    int i, dump_size;
    unsigned char data_byte;
    unsigned char *dst_addr, *src_addr;
#endif /* MGMTBUS_DEBUG */

    t = Mgmtbus_softc.ring_config;
    MGMTBUS_SYNC_CACHE(t, sizeof(mgmtbus_ring_config_t), 0);

    MGMTBUS_PRINTK(("%s: RX PKT START; cause 0x%x\n", __FUNCTION__, cause));

    /*
     * Originally for all hardware we checked all the cases
     * for (src = MGMTBUS_ADDR_SSC; src <= MGMTBUS_ADDR_MAX; src++)
     * On Cougar the FC and FP cases are NOT used. Hence:
     *
     * On TxRx, only check the SSC for src index table[SSC][TXRX]
     * On TXRX, cause is 0x1 on entry
     *
     * On SSC, only check the TXRX for src index table[TXRX][SSC]
     * On SSC, cause is 0x2 on entry
     */
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)

    /* Running on TxRx, Only SSC needs to be checked as source of packets */
    table_src_index = MGMTBUS_ADDR_SSC;
    table_dst_index = MGMTBUS_ADDR_TXRX;
#else

    /* Running on SSC, Only TxRx needs to be checked as source of packets */
    table_src_index = MGMTBUS_ADDR_TXRX;
    table_dst_index = MGMTBUS_ADDR_SSC;
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

        /*
         * Verify remote CPU has completed initialization.
         */
        if (t->table[table_src_index][table_dst_index].avail_ring_address == 0) { /* skip if not done */

	  MGMTBUS_PRINTK(("%s: RX Pkt END; table[%d][%d].avail_ring_address NULL, cause 0x%x\n",
                     __FUNCTION__, table_src_index, table_dst_index, cause));
            return;
        }

        used_ring_phy = t->table[table_src_index][table_dst_index].used_ring_address;
        used_ring = mgmtbus_phys_to_virt(used_ring_phy);

        avail_ring_phy =t->table[table_src_index][table_dst_index].avail_ring_address;
        avail_ring = mgmtbus_phys_to_virt(avail_ring_phy);

	MGMTBUS_PRINTK(("%s: RX Pkt; table[%d][%d] used <0x%p, 0x%x>, avail <0x%p, 0x%x>\n",
			__FUNCTION__, table_src_index, table_dst_index,
			used_ring, used_ring_phy,
			avail_ring, avail_ring_phy));
	MGMTBUS_PRINTK(("%s: RX Pkt; table[%d][%d] used_ring->next_index %d\n",
                     __FUNCTION__, table_src_index, table_dst_index,
		     used_ring->next_index));
        /*
         * Interrogate the used ring for new packets.
         */
        while (1) {

            used_desc = &used_ring->ring[used_ring->next_index];
            MGMTBUS_SYNC_CACHE(used_desc, sizeof(*used_desc), 0);

            if ((used_desc->flags & MGMTBUS_DESC_FLAGS_USED) == 0) { /* quit on first one NOT used */
                break;
            }
            avail_desc = &avail_ring->ring[used_ring->next_index];
            MGMTBUS_SYNC_CACHE(mgmtbus_phys_to_virt(used_desc->address),
                               used_desc->size, 0);

	    MGMTBUS_PRINTK(("%s: RX Pkt; table[%d][%d] used_desc 0x%p, used_ring->next_index %d\n",
                     __FUNCTION__, table_src_index, table_dst_index,
			used_desc, used_ring->next_index));
	    MGMTBUS_PRINTK(("%s: RX Pkt; table[%d][%d] avail_desc 0x%p\n",
                     __FUNCTION__, table_src_index, table_dst_index,
		     avail_desc));

	    used_desc_virt_addr = mgmtbus_phys_to_virt(used_desc->address);

            /* KLUDGE since first 2 pad bytes are not transferred */
            hdr = (agile_hdr_t *)(used_desc_virt_addr - 2);
            hdr_h_proto = ntohs(hdr->h_proto);

	    MGMTBUS_PRINTK(("%s: RX Pkt; used_desc 0x%p, addr <virt 0x%p, phy 0x%x>\n",
                     __FUNCTION__,
		     used_desc,
		     used_desc_virt_addr,
		     used_desc->address));
	    MGMTBUS_PRINTK(("%s: RX Pkt; used_desc <flags 0x%x, size %d>, h_proto 0x%x\n",
                     __FUNCTION__,
		     used_desc->flags,
		     used_desc->size,
		     hdr_h_proto));

            /*
             * Check if we have a Legal Agile header
	     * so valid data from EEE Pkt follows
             */
            switch (hdr_h_proto) {

	    case 0:
	    case ETH_P_EEE:
            /*
             * Allocate 2 bytes more to restore mangled agile_hdr_t.
             */
            skb = dev_alloc_skb(used_desc->size + 2 + ETH_HLEN);

            if (skb == NULL) {
	        Mgmtbus_softc.mgmtbus_stats.rx_dropped++;;
                break;
            }

            eth = (struct ethhdr *)skb_put(skb, ETH_HLEN);
            memcpy(eth->h_dest, mgmtbus_dev.dev_addr, ETH_ALEN);
            memset(eth->h_source, 0, ETH_ALEN);
            eth->h_proto = htons(ETH_P_EEE);

            /*
             * Account and zero the 2 byte pad
	     * in the Agile header.
	     */
            hdr = (agile_hdr_t *)skb_put(skb, 2);
            memset(hdr, 0, 2);

            /*
             * Compute buffer address'es and prepare to get the Agile header
             * Copy message from shared memory descriptor area
	     * into skb's local buffer
             */
            skb_dst_addr = skb_put(skb, used_desc->size);
	    used_desc_virt_addr = mgmtbus_phys_to_virt(used_desc->address);

            memcpy(skb_dst_addr,
                   used_desc_virt_addr,
                   used_desc->size);

	    MGMTBUS_PRINTK(("%s: RX Pkt; ETH_P_EEE Agile h_proto 0x%x\n",
                     __FUNCTION__, hdr_h_proto));

            MGMTBUS_PRINTK(("%s: RX Pkt; h_proto 0x%x, <slot:cpu:app> FROM %d:%d:%d TO %d:%d:%d\n",
                     __FUNCTION__,
                     hdr_h_proto,
                     EEE_GET_SLOT_NUM(hdr->src_port),
                     EEE_GET_DEST_CPU(hdr->src_port),
                     EEE_GET_APP_ID(hdr->src_port),
                     EEE_GET_SLOT_NUM(hdr->dest_port),
                     EEE_GET_DEST_CPU(hdr->dest_port),
                     EEE_GET_APP_ID(hdr->dest_port)));

#ifdef MGMTBUS_DEBUG
	       printk("%s: RX Pkt; SRC(used_desc) 0x%p, used__desc buff <virt 0x%p, phy 0x%x>\n",
                     __FUNCTION__,
		     used_desc,
		     used_desc_virt_addr,
		     used_desc->address);
	       printk("%s: RX Pkt; SRC(used_desc) flags 0x%x, size %d\n",
                     __FUNCTION__,
		     used_desc->flags,
		     used_desc->size);

	       dump_size = ETH_HLEN + sizeof(agile_hdr_t);
	       src_addr = (unsigned char *)used_desc_virt_addr;

	       printk("%s: RX Pkt; SRC(used_desc data)\n", __FUNCTION__);
	       for (i = 0; i < dump_size; i++) {
		 data_byte = *((unsigned char *)(src_addr+i));
		 printk("%02x ", data_byte);
	       }
	       printk("\n");

	       printk("%s: RX Pkt; DST(skb) 0x%p, skb->len %d, skb buffer 0x%p\n",
                     __FUNCTION__,
		     skb,
		     skb->len,
		     skb_dst_addr);

	       dst_addr = (unsigned char *)skb_dst_addr;
	       printk("%s: RX Pkt; DST(skb buffer)\n", __FUNCTION__);
	       for (i = 0; i < dump_size; i++) {
		 data_byte = *((unsigned char *)(dst_addr+i));
		 printk("%02x ", data_byte);
	       }
	       printk("\n");
#endif /* MGMTBUS_DEBUG */

	       skb->protocol = eth_type_trans(skb, &mgmtbus_dev);

	       /* Update network interface statistics */
	       Mgmtbus_softc.mgmtbus_stats.rx_packets++;
	       Mgmtbus_softc.mgmtbus_stats.rx_bytes += used_desc->size;

	       MGMTBUS_PRINTK(("%s: RX Pkt; After eth_type_trans(), skb->len %d, skb->protocol 0x%x\n",
			__FUNCTION__, skb->len, ntohs(skb->protocol)));

	       /* Send packet to kernel protocol stack */
	       netif_rx(skb);
	       break;

	    case ETH_P_IP:

	    MGMTBUS_PRINTK(("%s: RX IPv4 Pkt; Agile h_proto 0x%x\n",
                     __FUNCTION__, hdr_h_proto));
	      /*
	       * Allocate buffer for complete packet
	       * Note this could be smaller since we
	       * discard the shortened Agile header.
	       * ie (sizeof(agile_hdr_t) - sizeof(hdr->pad))
	       * Add some for alignment for the IP header.
	       */
	      hdr_size = ETH_HLEN + sizeof(struct iphdr) + 2;

	      skb = dev_alloc_skb(used_desc->size + hdr_size);

	      if (skb == NULL) {
			Mgmtbus_softc.mgmtbus_stats.rx_dropped++;;
			break;
	      }

	      /*
	       * Get skb position for Ethernet header & create it
	       * Update skb to reflect Ethernet header length
	       */
	      eth = (struct ethhdr *)skb_put(skb, ETH_HLEN);
	      memcpy(eth->h_dest, mgmtbus_dev.dev_addr, ETH_ALEN);
	      memset(eth->h_source, 0, ETH_ALEN);
	      eth->h_proto = htons(ETH_P_IP);

	      MGMTBUS_PRINTK(("%s: RX IPv4 Pkt; After ETH HDR, skb->len %d\n",
			__FUNCTION__, skb->len));
	       /*
		* Compute copy len excluding short Agile header
		* This should cover the IP header and transmitted data.
		* Seems requiring skipping first 2 byte pad in agile hdr!
		*/
	      hdr_size = sizeof(agile_hdr_t) - sizeof(hdr->pad);
	      copy_size = used_desc->size - hdr_size;

	      /*
	       * Compute dst skb buffer address to hold
	       * everything except shortened Agile header.
	       * This update's skb len to reflect the new total
	       * skb_dst_addr should now point to dst IP hdr
	       */
	      skb_dst_addr = skb_put(skb, copy_size);

	      MGMTBUS_PRINTK(("%s: RX IPv4 Pkt; After copy_size, skb->len %d, copy_size %d\n",
			__FUNCTION__, skb->len, copy_size));
	      /*
	       * Compute src address of IP header + data
	       * After this point, used_desc_virt_addr should
	       * point to the src IP hdr
	       */
	      used_desc_virt_addr = mgmtbus_phys_to_virt(used_desc->address);

	      agile_hdr_rx = (agile_hdr_t *)used_desc_virt_addr;
	      ip_hdr_rx = (struct iphdr *)((void *)(used_desc_virt_addr + hdr_size));

	       /*
		* Copy message from shared memory descriptor area into
		* skb local buffer, skipping Agile header without
		* the two pad header pad, which was NOT sent over bus
		*/
		memcpy(skb_dst_addr,
		       used_desc_virt_addr + hdr_size,
		       copy_size);

#ifdef MGMTBUS_DEBUG
		printk("%s: RX IPv4 Pkt; SRC(used_desc) data\n", __FUNCTION__);
		dst_addr = (unsigned char *)used_desc_virt_addr;

		for (i = 0; i < used_desc->size; i++) {
			data_byte = *((unsigned char *)(dst_addr+i));
			printk("%02x ", data_byte);
		}
		printk("\n");

		printk("%s: RX IPv4 Pkt; DST(skb) data\n", __FUNCTION__);
		dst_addr = (unsigned char *)eth;
		for (i = 0; i < (copy_size + ETH_HLEN); i++) {
			data_byte = *((unsigned char *)(dst_addr+i));
			printk("%02x ", data_byte);
		}
		printk("\n");

		printk("%s: RX IPv4 Pkt; Agile hdr 0x%p\n", __FUNCTION__, agile_hdr_rx);
		dst_addr = (unsigned char *)agile_hdr_rx;

		for (i = 0; i < hdr_size; i++) {
			data_byte = *((unsigned char *)(dst_addr+i));
			printk("%02x ", data_byte);
		}
		printk("\n");

		printk("%s: RX IPv4 Pkt; IP hdr 0x%p\n", __FUNCTION__, ip_hdr_rx);

		dst_addr = (unsigned char *)ip_hdr_rx;
		for (i = 0; i < sizeof(struct iphdr); i++) {
			data_byte = *((unsigned char *)(dst_addr+i));
			printk("%02x ", data_byte);
		}
		printk("\n");
#endif /* MGMTBUS_DEBUG */

	       MGMTBUS_PRINTK(("%s: RX IPv4 Pkt; After memcpy(), skb->len %d, copy_size %d, skb->protocol 0x%x\n",
			__FUNCTION__, skb->len, copy_size, ntohs(skb->protocol)));

		skb->protocol = eth_type_trans(skb, &mgmtbus_dev);

		/* Tell upper layer where IP header starts */
		skb_set_network_header(skb, ETH_HLEN);

		MGMTBUS_PRINTK(("%s: RX IPv4 Pkt; Before netif_rx(), skb->len %d, skb->protocol 0x%x\n",
			__FUNCTION__, skb->len, ntohs(skb->protocol)));
		MGMTBUS_PRINTK(("%s: RX IPv4 Pkt; skb head 0x%p, data 0x%p, tail 0x%x, end 0x%x\n",
				__FUNCTION__, skb->head, skb->data, skb->tail, skb->end));
		MGMTBUS_PRINTK(("%s: RX IPv4 Pkt; skb mac 0x%x, network 0x%x, transport 0x%x\n",
				__FUNCTION__, skb->mac_header, skb->network_header, skb->transport_header));

		/* Update network interface statistics */
		Mgmtbus_softc.mgmtbus_stats.rx_packets++;
		Mgmtbus_softc.mgmtbus_stats.rx_bytes += used_desc->size;

		/*
		 * Send IP Pkt to kernel protocol stack
		 */
		netif_rx(skb);
		break;

            default:
	       MGMTBUS_PRINTK((
		"%s: RX Pkt; UNKNOWN Agile h_proto 0x%x, <slot,cpu,app> FROM %d:%d:%d TO %d:%d:%d\n",
                __FUNCTION__,
		hdr_h_proto,
                EEE_GET_SLOT_NUM(hdr->src_port),
                EEE_GET_DEST_CPU(hdr->src_port),
                EEE_GET_APP_ID(hdr->src_port),
                EEE_GET_SLOT_NUM(hdr->dest_port),
                EEE_GET_DEST_CPU(hdr->dest_port),
                EEE_GET_APP_ID(hdr->dest_port)));

	        Mgmtbus_softc.mgmtbus_stats.rx_dropped++;;
	        break;
	} /* end of switch */

        /*
         * Clear descriptor flags to indicate the
	 * descriptor is now available
         */
         used_desc->flags &= ~MGMTBUS_DESC_FLAGS_USED;
         avail_desc->flags &= ~MGMTBUS_DESC_FLAGS_USED;

         MGMTBUS_SYNC_CACHE(used_desc, sizeof(*used_desc), 1);
         MGMTBUS_SYNC_CACHE(avail_desc, sizeof(*avail_desc), 1);

         used_ring->next_index++;
         if (used_ring->next_index == used_ring->ring_size) {
               used_ring->next_index = 0;
         }
         used_desc = &used_ring->ring[used_ring->next_index];
    } /* end of while for-ever loop processing loop */

    MGMTBUS_PRINTK(("%s: RX PKT END\n", __FUNCTION__));
    return;
}

#ifdef CONFIG_NET_POLL_CONTROLLER
void
mgmtbus_poll_controller(struct net_device *dev)
{
	uint32_t cause;

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
	/* On TxRx, cause should indicate intr came from SSC which is 0x2 */
	cause = (1 << MGMTBUS_ADDR_SSC);
#else
	/* On SSC, cause should indicate intr came from TXRX which is 0x1 */
	cause = (1 << MGMTBUS_ADDR_TXRX);
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

	mgmtbus_intr(cause);
	return;
}

void __exit
mgmtbus_exit(void)
{
	unregister_netdev(&mgmtbus_dev);
	return;
}
#else
void __exit
mgmtbus_exit(void)
{
	mgmtbus_irq_free(&mgmtbus_dev);
	unregister_netdev(&mgmtbus_dev);
	return;
}
#endif /* CONFIG_NET_POLL_CONTROLLER */

void
mgmtBus_waitforRingConfig(mgmtbus_ring_config_t *t)
{
    volatile mgmtbus_ring_config_t *rcfg = t;

    printk("%s: Waiting for SSC to finish mgmtbus init; ring_config 0x%p\n",
	__FUNCTION__,
        &(rcfg->magic));
    /*
     * Wait for magic word to change in SSC ring configuration area
     */
    while (*((volatile uint32_t *)(&rcfg->magic)) != MGMTBUS_RING_CONFIG_MAGIC);

    printk("%s: Continuing mgmtbus initialization....\n",
        __FUNCTION__);
    return;
}

/*
 * Routine Description:
 *  Module initialization function
 *
 * Arguments:
 *  None.
 *
 * Return Value:
 *  None.
 */
int
__init mgmtbus_init(void)
{
    void *ssc_base_phy_addr;
    phys_t ssc_base_offset;
    int ret;

    if (MGMTBUS_MAX_SHARED != SHARED_MEM_MGMT_SIZE) {
        return -ENOMEM;
    }
    memset(&Mgmtbus_softc.mgmtbus_stats, 0, sizeof(struct net_device_stats));
    Mgmtbus_softc.flags = 0;

    /*
     * Allocate the SSC shared memory from MGMTBUS_SSC_BASE_ADDR.
     * The SSC shared memory area contains the shared Ring Configuration
     * structure first, followed by the RCON structure, the ring descriptors
     * and ring buffers.
     * This physical address is different under SSC versus TXRX cores.
     */
    ssc_base_offset = MGMTBUS_SSC_BASE_ADDR;
    Mgmtbus_softc.ssc_pci_base = SSC_PCI_OFFSET;
    Mgmtbus_softc.txrx_pci_base = TXRX_PCI_OFFSET;
    Mgmtbus_softc.fp_pci_base = FP_PCI_OFFSET;

    spin_lock_init(&Mgmtbus_softc.tx_lock);

   /*
    * Set the Common shared configuration ring address
    * NOTE: This is a physical address in local SSC memory.
    * ALL of the offset's are relative to the SSC's memory map
    * for the SSC Linux device driver.
    *
    * For the TxRx Linux device driver, these address'es ARE
    * different. CONFIG_ONSTOR_TUXRX is defined for Linux on TxRx.
    *
    * Other core's access the shared memory area via the PCI bus.
    * Those address'es are mapped and access by different physical
    * locations on each core. Top 8-bits of physical addresses are
    * SSC 0xF8, TxRx 0x10 and FP 0x10 for the PCI address'es.
    *
    * The SSC Linux device driver uses 'SHARED_MEM_START' for ssc_base_offset
    * is 0x8F000000UL and DOES NOT set top 8-Bits of 40-Bit Physical address
    * for uncached mapping operation.
    *
    * On TxRx, the SSC shared memory area is PCI accessed with a
    * different physical address, which is required for the map operation.
    */
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
    ssc_base_phy_addr = (void *)(ssc_base_offset | PCI_PHYS_START);

    MGMTBUS_PRINTK(("%s: ringConfAddr ssc_pci_base 0x%p, ssc_base_offset 0x%p, phy 0x%p\n",
	__FUNCTION__,
	(void *)Mgmtbus_softc.ssc_pci_base,
	(void *)(ssc_base_offset),
	ssc_base_phy_addr));

    /* NOTE: Compiler complains due to the macro definition or'ing top bits to void * ptr */
    Mgmtbus_softc.ssc_pci_virt = (void *)PHYS_TO_XKSEG_UNCACHED((ssc_base_offset | PCI_PHYS_START));

    MGMTBUS_PRINTK(("%s: ssc <virt,pci_base> 0x%p, 0x%p\n",
	__FUNCTION__,
	(void *)Mgmtbus_softc.ssc_pci_virt,
	(void *)Mgmtbus_softc.ssc_pci_base));

    Mgmtbus_softc.ring_config = Mgmtbus_softc.ssc_pci_virt;
    magicmanagementbusringconfig = Mgmtbus_softc.ring_config;

    /*
     * On TxRx, Phyical address for start of txrx shared memory
     * area is different from SSC.
     * The symbol 'PCI_PHYS_START' is defined for each case.
     * The txrx_pci_base is an offset into this area.
     *
     * HOWEVER: On the TxRx the memory area mapped to this
     * area is NOT at 'PCI_PHYS_START' but 0x1F000 0000
     * Get and uncached virtual address for this memory.
     *
     * (void *)PHYS_TO_XKSEG_UNCACHED((Mgmtbus_softc.txrx_pci_base | PCI_PHYS_START));
     * Mgmtbus_softc.txrx_pci_virt = (void *)__va((void *)(TXRX_PCI_PHY_START));
     */
    Mgmtbus_softc.txrx_pci_virt = (void *)ioremap(TXRX_PCI_PHY_START, TXRX_PCI_PHY_SIZE);

    MGMTBUS_PRINTK(("%s: txrx <virt,pci_base> 0x%p, 0x%p\n",
	__FUNCTION__,
	(void *)Mgmtbus_softc.txrx_pci_virt,
	(void *)Mgmtbus_softc.txrx_pci_base));

    Mgmtbus_softc.fp_pci_virt =
        (void *)PHYS_TO_XKSEG_UNCACHED((Mgmtbus_softc.fp_pci_base | PCI_PHYS_START));

    MGMTBUS_PRINTK(("%s: fp <virt,pci_base> 0x%p, 0x%p\n",
	__FUNCTION__,
	(void *)Mgmtbus_softc.fp_pci_virt,
	(void *)Mgmtbus_softc.fp_pci_base));

    /*
     * On TxRx we must Wait for SSC to write the magic location in ringConfig
     * area before TxRx can initialize it's fields in the ring_config area.
     */
    mgmtBus_waitforRingConfig(Mgmtbus_softc.ring_config);

#else
    ssc_base_phy_addr = (void *)(ssc_base_offset);
    Mgmtbus_softc.ssc_pci_virt = (void *)PHYS_TO_XKSEG_UNCACHED(ssc_base_offset);

    MGMTBUS_PRINTK(("%s: ringConfAddr ssc_base_offset 0x%p, ssc_base_phy_addr 0x%p\n",
	__FUNCTION__,
	(void *)(ssc_base_offset),
	ssc_base_phy_addr));

    MGMTBUS_PRINTK(("%s: ssc <virt,pci_base> 0x%p, 0x%p\n",
	__FUNCTION__,
	(void *)Mgmtbus_softc.ssc_pci_virt,
	(void *)Mgmtbus_softc.ssc_pci_base));

    Mgmtbus_softc.ring_config = Mgmtbus_softc.ssc_pci_virt;
    magicmanagementbusringconfig = Mgmtbus_softc.ring_config;

    /*
     * On SSC, Top 8-Bits of physical address need to be 0xF8 for PCI access
     */
    Mgmtbus_softc.txrx_pci_virt =
        (void *)PHYS_TO_XKSEG_UNCACHED(Mgmtbus_softc.txrx_pci_base | PCI_PHYS_START);

    MGMTBUS_PRINTK(("%s: txrx <virt,pci_base> 0x%p, 0x%p\n",
	__FUNCTION__,
	(void *)Mgmtbus_softc.txrx_pci_virt,
	(void *)Mgmtbus_softc.txrx_pci_base));

    Mgmtbus_softc.fp_pci_virt =
        (void *)PHYS_TO_XKSEG_UNCACHED(Mgmtbus_softc.fp_pci_base | PCI_PHYS_START);

    MGMTBUS_PRINTK(("%s: fp <virt,pci_base> 0x%p, 0x%p\n",
	__FUNCTION__,
	(void *)Mgmtbus_softc.fp_pci_virt,
	(void *)Mgmtbus_softc.fp_pci_base));
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

    initConfigTable();
    // initDescriptorRings();

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
    initDescriptorRingsOnTxRx();
#else
    initDescriptorRingsOnSSC();
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

    mgmtbus_init_hostid();
    mgmtbus_dev.dev_addr[0] = 0x4;
    mgmtbus_dev.dev_addr[1] = 0x0;
    mgmtbus_dev.dev_addr[2] = 0x0;
    mgmtbus_dev.dev_addr[3] = 0x0;
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
    mgmtbus_dev.dev_addr[4] = 0x2; /* txrx */
#else
    mgmtbus_dev.dev_addr[4] = 0x1;
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */
    mgmtbus_dev.dev_addr[5] = 0x0;

    /*
     * Indicate the SSC initialization is complete.
     * by writing the magic word into ring configuration area.
     * The TxRx waits for the SSC to complete initialization
     * as indicated by this magic word. It's not used in the
     * other direction, so it's here only for code symmetry.
     */
    Mgmtbus_softc.ring_config->magic = MGMTBUS_RING_CONFIG_MAGIC;

    MGMTBUS_SYNC_CACHE(Mgmtbus_softc.ring_config, sizeof(mgmtbus_ring_config_t), 1);

    ret = register_netdev(&mgmtbus_dev);
    if (ret) {
	printk(KERN_ERR "mgmtbus: register_netdev() FAILED, net device: %s, ret %d\n",
		mgmtbus_dev.name, ret);
	return ret;
    }

#ifdef CONFIG_NET_POLL_CONTROLLER
    /*
     * Setup to poll, so no explict
     * interrupt number assigned
     */
    printk(KERN_INFO "mgmtbus: NET_POLL_CONTROLLER dev: %s, RingConfig 0x%p\n", 
	       mgmtbus_dev.name, Mgmtbus_softc.ring_config);
#else
    /*
     * Call platform specific code, aka cougar here, to setup the
     * interrupt handlers for the mailboxes
     */
    ret = mgmtbus_irq_setup(&mgmtbus_dev);
    if (ret) {
	printk(KERN_ERR "mgmtbus: mgmtbus_irq_setup() FAILED, net device: %s, ret %d\n",
		mgmtbus_dev.name, ret);

        unregister_netdev(&mgmtbus_dev);
    } else {
	printk(KERN_INFO "mgmtbus: registered net device: %s, RingConfig 0x%p\n", 
	       mgmtbus_dev.name, Mgmtbus_softc.ring_config);
    }
#endif /* CONFIG_NET_POLL_CONTROLLER */

    return ret;
}

/*
 * Routine Description:
 *  Function called by kernel command line parser code to
 *  parse the onstor_model parameter that the prom passes as
 *  a kernel argument.
 *
 * Arguments:
 *  str - the string containing the value of the argument
 *
 * Return Value: 0
 */
static int __init
mgmtbus_set_model_number(char *str)
{
    /*
     * Convert the model string into a number for easier transport to the TXRX
     * It is assumed that model numbers are of the form TLA-SYS-NUMBER
     */
    char *str1;

    /*
     * Pick some default that turns everything on in the embedded layer.
     */
    model_num = 0;

    if (str == NULL) {
      return 0;
    }
    str1 = strrchr(str, '-');
    if (str1 == NULL) {
        return 0;
    }
    str1++;

    if (sscanf(str1, "%d", &model_num) != 1) {
        model_num = 0;
    }
    return 0;
}

__setup("onstor_model=", mgmtbus_set_model_number);

module_init(mgmtbus_init);
module_exit(mgmtbus_exit);
MODULE_DESCRIPTION("Management Bus Driver");
MODULE_LICENSE("GPL");

