
/*-----------------------------------------------------------------
 *
 * Name: rcon.c
 *
 * Copyright (C) 2007, Onstor, Inc.
 *
 * Description: device driver for access to remote consoles.
 *
 *-----------------------------------------------------------------
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/proc_fs.h>
#include <linux/fcntl.h>
#include <asm/system.h>
#include <asm/uaccess.h>
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <linux/poll.h>
#include "mgmt-bus.h"
#include "rcon.h"

#ifdef CONFIG_ONSTOR_BOBCAT
# include <linux/mv643xx.h>
# include <asm/marvell.h>
# include <asm/mach-bobcat/bobcat.h>
#elif defined(CONFIG_ONSTOR_COUGAR)
# include <asm/mach-cougar/cougar.h>
# include <asm/sibyte/sb1250_int.h>
# include <asm/sibyte/sb1250_regs.h>
#else
# error "Error in configuration: no platform specified."
#endif

extern void *mcpu_shared_memory;

#define EEE_DEBUG
#undef EEE_DEBUG

#ifdef EEE_DEBUG
#define eee_dbg(x) printk x
#else
#define eee_dbg(x)
#endif

MODULE_LICENSE("GPL");

/* Location of fp and txrx queues in the SSC memory.
 */
#ifdef CONFIG_ONSTOR_COUGAR
# define RCON_FP_QUEUES PHYS_TO_XKSEG_UNCACHED(0xf81b000380)
# define RCON_TXRX_QUEUES PHYS_TO_XKSEG_UNCACHED(0xf81a000000)
#elif defined(CONFIG_ONSTOR_BOBCAT)
# define RCON_FP_QUEUES 0xbb000380
# define RCON_TXRX_QUEUES 0xba000000
#endif

/*
 * Our device major number
 */
int rcon_major;

/*
 * Number of cores per txrx/fp
 */
#define RCON_CORES_PER_CPU 4

/*
 * The remote console device structure.
 */
struct rcon_dev {
    /*
     * If true, somebody has device opened for reading.  Only one process can
     * have the device opened for writing.
     */
    bool open_read;

    /*
     * If true, somebody has device opened for reading.  Only one process can
     * have the device opened for reading.
     */
    bool open_write;

    /*
     * The offset in print_buf where the next character from the input queue
     * will be copied.
     */
    int print_buf_pos;

    /*
     * The number of characters copied into the print buffer from the input
     * queue.
     */
    uint64_t print_buf_last_read;

    /*
     * The buffer collecting the data from the input queue
     */
    char print_buf[SM_RCON_QUEUE_MAX];

    /*
     * This cpu queue.
     */
    struct rcon_queue *my_rcon_queue;

    /*
     * The remote cpu queue.
     */
    struct rcon_queue *peer_rcon_queue;

    /*
     * Queue to wait for the data to become available.
     */
    wait_queue_head_t read_wait;

    /*
     * Queue to wait for the space in the queue to become available.
     */
    wait_queue_head_t write_wait;
} rcon_devs[RCON_MAX_CORES];


int
rcon_open(struct inode *inode, struct file *filp)
{
    int minor = MINOR(inode->i_rdev);

    if (minor >= RCON_MAX_CORES) {
        return -ENODEV;
    }

    /*
     * Prevent more than one process to have the device opened for reading or
     * writing.
     */
    if ((((filp->f_flags & O_ACCMODE) == O_RDONLY) ||
         ((filp->f_flags & O_ACCMODE) == O_RDWR)) &&
        rcon_devs[minor].open_read) {
        return -EBUSY;
    }

    if ((((filp->f_flags & O_ACCMODE) == O_WRONLY) ||
         ((filp->f_flags & O_ACCMODE) == O_RDWR)) &&
        rcon_devs[minor].open_write) {
        return -EBUSY;
    }

    /*
     * Increment open count.
     */
    if (((filp->f_flags & O_ACCMODE) == O_RDONLY) ||
        ((filp->f_flags & O_ACCMODE) == O_RDWR)) {
        rcon_devs[minor].open_read++;
    }

    if (((filp->f_flags & O_ACCMODE) == O_WRONLY) ||
        ((filp->f_flags & O_ACCMODE) == O_RDWR)) {
        rcon_devs[minor].open_write++;
    }

    filp->private_data = &rcon_devs[minor];

    return 0;
}


int
rcon_release(struct inode *inode, struct file *filp)
{
    int minor = MINOR(inode->i_rdev);

    if (minor >= RCON_MAX_CORES) {
        return -ENODEV;
    }

    /*
     * Decrement the corresponding open count.
     */
    if (((filp->f_flags & O_ACCMODE) == O_RDONLY) ||
        ((filp->f_flags & O_ACCMODE) == O_RDWR)) {
        rcon_devs[minor].open_read--;
    }

    if (((filp->f_flags & O_ACCMODE) == O_WRONLY) ||
        ((filp->f_flags & O_ACCMODE) == O_RDWR)) {
        rcon_devs[minor].open_write--;
    }

    return 0;
}


ssize_t
rcon_read(struct file *filp, char *buf, size_t count, loff_t *f_pos)
{
    struct rcon_dev *dev = filp->private_data;
    struct rcon_queue *my_q = dev->my_rcon_queue;
    struct rcon_queue *peer_q = dev->peer_rcon_queue;
    uint64_t peer_wrote;
    uint64_t i_read;
    int rc;
    uint64_t remain;
    uint64_t copied;
    char *copy_to;

    /*
     * Return immediately if not blocking and no data available.
     */
    if ((filp->f_flags & O_NONBLOCK) &&
        (my_q->bytes_peer_wrote <= my_q->bytes_i_read)) {
        return -EAGAIN;
    }

    /*
     * Wait for the data to become available.
     */
    if (wait_event_interruptible(
            dev->read_wait,
            my_q->bytes_peer_wrote > my_q->bytes_i_read)) {
        return -ERESTARTSYS;
    }

    peer_wrote = my_q->bytes_peer_wrote;
    i_read = my_q->bytes_i_read;

    eee_dbg(("%s: w=%lld r=%lld\n", __FUNCTION__, peer_wrote, i_read));

    remain = min(peer_wrote - i_read, (uint64_t)count);
    copied = remain;
    copy_to = buf;
    while (remain > 0) {
        uint64_t queue_offset = (i_read % SM_RCON_QUEUE_MAX);
        uint64_t copy_this_time = min(SM_RCON_QUEUE_MAX - queue_offset,
                                      remain);
        rc = copy_to_user(copy_to, my_q->data + queue_offset,
                          copy_this_time);
        if (rc < 0) {
            return rc;
        }

        i_read += copy_this_time;
        remain -= copy_this_time;
        copy_to += copy_this_time;
    }

    my_q->bytes_i_read = i_read;
    peer_q->bytes_peer_read = i_read;

    return copied;
}

ssize_t
rcon_write(struct file *filp, const char __user *buf, size_t count,
           loff_t *f_pos)
{
    struct rcon_dev *dev = filp->private_data;
    struct rcon_queue *my_q = dev->my_rcon_queue;
    struct rcon_queue *peer_q = dev->peer_rcon_queue;
    uint64_t i_wrote = my_q->bytes_i_wrote;
    int rc;
    uint64_t remain;
    const char __user *copy_from = buf;

    uint64_t copied;

    eee_dbg(("%s\n", __FUNCTION__));

    if (count == 0) {
        return 0;
    }

    remain = min((uint64_t)count,
                 my_q->bytes_peer_read + SM_RCON_QUEUE_MAX - i_wrote);

    /*
     * Return if no space in queue and blocking, otherwise wait for the space
     * to become available.
     */
    if (remain == 0) {
        if (filp->f_flags & O_NONBLOCK) {
            return -EAGAIN;
        } else {
            if (wait_event_interruptible(
                    dev->write_wait,
                    my_q->bytes_peer_read + SM_RCON_QUEUE_MAX > i_wrote)) {
                return -ERESTARTSYS;
            }
            remain = min((uint64_t)count,
                         my_q->bytes_peer_read + SM_RCON_QUEUE_MAX - i_wrote);
        }
    }

    copied = remain;

    while (remain > 0) {
        uint64_t queue_offset = i_wrote % SM_RCON_QUEUE_MAX;
        uint64_t copy_this_time = min(remain, SM_RCON_QUEUE_MAX - queue_offset);

        eee_dbg(("%s: copy_from %p %p %lld\n",
                 __FUNCTION__,
                 peer_q->data + queue_offset, copy_from, copy_this_time));

        rc = copy_from_user(peer_q->data + queue_offset, copy_from,
                            copy_this_time);
        if (rc < 0) {
            return rc;
        }

        i_wrote += copy_this_time;
        remain -= copy_this_time;
        copy_from += copy_this_time;
    }

    peer_q->bytes_peer_wrote = i_wrote;
    my_q->bytes_i_wrote = i_wrote;

    return copied;
}


/*++

Routine Description:

    Process the interrupt for the specific queue.

Arguments:

    i - device minor number

Return Value:

    None.

--*/

void
rcon_process(int i)
{
    struct rcon_dev *dev = &rcon_devs[i];
    struct rcon_queue *my_q = dev->my_rcon_queue;
    struct rcon_queue *peer_q = dev->peer_rcon_queue;
    uint64_t peer_wrote = my_q->bytes_peer_wrote;
    uint64_t i_read = dev->print_buf_last_read;
    uint64_t remain;

    /*
     * Wakeup anybody waiting for the data.
     */
    if (peer_wrote > my_q->bytes_i_read) {
        wake_up(&dev->read_wait);
    }

    /*
     * Wakeup anybody waiting for the space in the queue.
     */
    if (my_q->bytes_i_wrote < (my_q->bytes_peer_read + SM_RCON_QUEUE_MAX)) {
        wake_up(&dev->write_wait);
    }

    /*
     * Copy the data into the local buffer. If encountered line end, output
     * the contents of the buffer and clear the buffer. If the buffer is
     * full, output it as well.
     *
     * The output is skipped if some user process is reading the device.
     */
    while (peer_wrote > i_read) {
        remain = peer_wrote - i_read;
        while (remain > 0) {
            int c = my_q->data[i_read % SM_RCON_QUEUE_MAX];
            bool output = c == '\n';
            if (!output) {
                /*
                 * Strip carriage returns.
                 */
                if (c != '\r') {
                    dev->print_buf[dev->print_buf_pos] = c;
                    dev->print_buf_pos++;
                }
                output = (dev->print_buf_pos == (SM_RCON_QUEUE_MAX - 1));
            }
            if (output) {
                dev->print_buf[dev->print_buf_pos] = 0;
                if (!dev->open_read) {
                    printk(KERN_INFO "%s%d: %s\n",
                           i < RCON_CORES_PER_CPU ? "tx" : "fp",
                           i < RCON_CORES_PER_CPU ? i : i - RCON_CORES_PER_CPU,
                           dev->print_buf);
                }
                dev->print_buf_pos = 0;
            }

            i_read++;
            remain--;
        }
        dev->print_buf_last_read = i_read;

        /*
         * If nobody is reading, remove the data from the queue.
         */
        if (!dev->open_read) {
            peer_q->bytes_peer_read = i_read;
            my_q->bytes_i_read = i_read;
        }
    }
}

unsigned int
rcon_poll(struct file *filp, poll_table *wait)
{
    struct rcon_dev *dev = filp->private_data;
    struct rcon_queue *my_q = dev->my_rcon_queue;
    unsigned int mask = 0;

    poll_wait(filp, &dev->read_wait, wait);
    poll_wait(filp, &dev->write_wait, wait);

    eee_dbg(("%s: dev %d peer_wrote = %lld i_read = %lld\n",
             __FUNCTION__, dev - rcon_devs, my_q->bytes_peer_wrote,
             my_q->bytes_i_read));

    if (my_q->bytes_peer_wrote > my_q->bytes_i_read) {
        mask |= POLLIN | POLLRDNORM;
    }

    if ((my_q->bytes_i_wrote - my_q->bytes_peer_read) < SM_RCON_QUEUE_MAX) {
        mask |= POLLOUT | POLLWRNORM;
    }

    return mask;
}


struct file_operations rcon_fops = {
    .read = rcon_read,
    .write = rcon_write,
    .open = rcon_open,
    .poll = rcon_poll,
    .release = rcon_release
};

static struct class *rcon_dev_class;


static void
rcon_uninit(void)
{
    printk(KERN_INFO "rcon device has left the building\n");

    rcon_irq_free(rcon_devs);
    unregister_chrdev(rcon_major, "rcon");
}

static int
rcon_init(void)
{
    int rc;
    int minor;
    unsigned long start = (unsigned long)magicmanagementbusringconfig;
    struct rcon_queue *queue_start;
    struct rcon_queue *fp_queue = (struct rcon_queue *)RCON_FP_QUEUES;
    struct rcon_queue *txrx_queue = (struct rcon_queue *)RCON_TXRX_QUEUES;

    start += sizeof(mgmtbus_ring_config_t);
    start = ALIGN(start, 32);

    printk("rcon device will be in the hizzy @ %p\n", (void *)start);

    queue_start = (struct rcon_queue *)start;

    rc = register_chrdev(0, "rcon", &rcon_fops);
    if (rc < 0) {
        printk("%s: failed to register: %d\n", __FUNCTION__, rc);
        return rc;
    }

    rcon_major = rc;

    rc = rcon_irq_setup(rcon_devs);
    if (rc < 0) {
        goto out;
    }

    rcon_dev_class = class_create(THIS_MODULE, "rcon");
    if (IS_ERR(rcon_dev_class)) {
        rc = -1;
        goto out;
    }

    for (minor = 0; minor < RCON_MAX_CORES; ++minor) {
        char dev_name[256];

        init_waitqueue_head(&rcon_devs[minor].read_wait);
        init_waitqueue_head(&rcon_devs[minor].write_wait);

        rcon_devs[minor].my_rcon_queue = &queue_start[minor];
        if (minor >= RCON_CORES_PER_CPU) {
            rcon_devs[minor].peer_rcon_queue =
                &fp_queue[minor - RCON_CORES_PER_CPU];
        } else {
            rcon_devs[minor].peer_rcon_queue =
                &txrx_queue[minor];
        }

        rcon_devs[minor].my_rcon_queue->bytes_i_read = 0;
        rcon_devs[minor].my_rcon_queue->bytes_i_wrote = 0;

        sprintf(dev_name, "rcon%d", minor);

        class_device_create(rcon_dev_class, NULL, MKDEV(rcon_major, minor),
                            NULL, dev_name);
    }

    return 0;

  out:
    rcon_uninit();
    return rc;
}


module_init(rcon_init);
module_exit(rcon_uninit);

