/*
 * tuxrx_mgmt_bus.c
 *
 * Description:
 *  Cougar platform but Linux Sibyte 1480 TxRx specific routines.
 *
 * Created by:  William Fisher, November 2008
 * Copyright (c) 2008 OnStor, Inc.
 * All rights reserved.
 */
#include <linux/types.h>
#include <linux/irqreturn.h>
#include <linux/netdevice.h>
#include <net/neteee/edesc.h>
#include "mgmt-bus.h"
#include "rcon.h"
#include <asm/mach-cougar/cougar.h>

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
#include <asm/sibyte/bcm1480_regs.h>
#include <asm/sibyte/bcm1480_int.h>
#else /* defined(CONFIG_SIBYTE_SB1250) || defined(CONFIG_SIBYTE_BCM112X) */
#include <asm/sibyte/sb1250_regs.h>
#include <asm/sibyte/sb1250_int.h>
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

#include <asm/sibyte/sb1250.h>

extern uint64_t sbmac_orig_hwaddr[];

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
unsigned int mgmtbus_rcon_mailbox_value = K_BCM1480_INT_MBOX_0_2;
unsigned int mgmtbus_mailbox_intr_value = K_BCM1480_INT_MBOX_0_3;
#else
unsigned int mgmtbus_rcon_mailbox_value = K_INT_MBOX_2;
unsigned int mgmtbus_mailbox_intr_value = K_INT_MBOX_3;
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

/*
 * Routine Description:
 *  Initialize host id using 4th+5th+6th bytes of the MAC address,
 *   counting from 1.
 *
 * Arguments:
 *  None
 *
 * Return Value:
 *  Void
 */
void
mgmtbus_init_hostid()
{
    unsigned char *c = (unsigned char *)&sbmac_orig_hwaddr[0];

    magicmanagementbusringconfig->hostid = (c[3] << 16) | (c[4] << 8) | c[5];
    return;
}

/*
 * Routine Description:
 *  Interrupt handler for mbox interrupt.
 *
 * Arguments:
 *  irq - interrupt number
 *  dev_id - pointer to device structure
 *
 * Return Value:
 * Returns IEQ_HANDLED.
 */
static irqreturn_t
mgmtbus_mailbox_intr(int irq, void *dev_id)
{
    /* Acknowledge the interrupt by clearing the mailbox.
     */
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
     /* Sibyte 1480 mailbox register on TxRx
      */
    /* XXX */
    *(uint64_t *)PHYS_TO_XKSEG_UNCACHED(
        A_IMR_REGISTER(0, R_IMR_MAILBOX_CLR_CPU)) = 0xFFFFUL;

   mgmtbus_intr(1 << MGMTBUS_ADDR_SSC);
#else
     /* Sibyte 1125 mailbox register on SSC
      */
    *(uint64_t *)PHYS_TO_XKSEG_UNCACHED(
        A_IMR_REGISTER(0, R_IMR_MAILBOX_CLR_CPU)) = 0xFFFFUL;

   mgmtbus_intr(1 << MGMTBUS_ADDR_TXRX);
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */
   return IRQ_HANDLED;
}

/*
 * Routine Description:
 *  Release the mailbox assigned to mgmtbus driver interrupt handler.
 *
 * Arguments:
 *  dev - device structure
 *
 * Return Value:
 *  Void
 */
void
mgmtbus_irq_free(struct net_device *dev)
{
    free_irq(mgmtbus_mailbox_intr_value, dev);
    printk(KERN_INFO "MGMTBUS: releasing irq %d\n", mgmtbus_mailbox_intr_value);
    return;
}

/*
 * Routine Description:
 *  Setup the interrupts for the mgmtbus driver.
 *
 * Arguments:
 *  dev - device structure
 *
 * Return Value:
 *  0 - success, -error on failure.
 */
int
mgmtbus_irq_setup(struct net_device *dev)
{
    int ret;

    /* Don't share if possible, aka no IRQF_SHARED */
    ret = request_irq(mgmtbus_mailbox_intr_value, mgmtbus_mailbox_intr,
                      0, dev->name, dev);
    if (ret) {
        printk(KERN_ERR "%s: mgmtbus: request_irq() FAILED, irq %d, err %d\n",
	       __FUNCTION__, mgmtbus_mailbox_intr_value, ret);
    } else {
        printk(KERN_INFO "mgmtbus: using irq %d\n", mgmtbus_mailbox_intr_value);
    }
    return ret;
}

#if defined(CONFIG_RCON)
/*
 * Routine Description:
 *  Remote Console(RCON) Interrupt handler.
 *
 * Arguments:
 *  irq - interrupt number
 *  dev - device structure
 *
 * Return Value:
 *  IRQ_HANDLED
 */
irqreturn_t
rcon_intr(int irq, void *dev_id)
{
    int i;

    /*
     * Ack the interrupt by clearing the mailbox.
     */
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
     /* Sibyte 1480 mailbox register on TxRx
      */
     /* XXX */
    *(uint64_t *)PHYS_TO_XKSEG_UNCACHED(
        A_IMR_REGISTER(0, R_IMR_MAILBOX_CLR_CPU)) = 0xFFFF0000UL;
#else
     /* Sibyte 1125 mailbox register on SSC
      */
    *(uint64_t *)PHYS_TO_XKSEG_UNCACHED(
        A_IMR_REGISTER(0, R_IMR_MAILBOX_CLR_CPU)) = 0xFFFF0000UL;

#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

    for (i = 0; i < RCON_MAX_CORES; ++i) {
        rcon_process(i);
    }
    return IRQ_HANDLED;
}

/*
 * Routine Description:
 *  Setup/register Remote Console(RCON) Interrupt handler.
 *
 * Arguments:
 *  dev - device structure
 *
 * Return Value:
 *  0 - success, -error on failure.
 */
int
rcon_irq_setup(struct rcon_dev *dev)
{
    int rc = 0;

     /* Sibyte 1480 mailbox register on TxRx OR Sibyte 1125 on SSC
      */
    rc = request_irq(mgmtbus_rcon_mailbox_value, rcon_intr, IRQF_SHARED, "rcon", dev);
    if (rc < 0) {
        printk("%s: failed to register irq %d: %d\n", __FUNCTION__, 
               mgmtbus_rcon_mailbox_value, rc);
    }
    return rc;
}

/*
 * Routine Description:
 *  Release/free Remote Console(RCON) Interrupt handler assignment.
 *
 * Arguments:
 *  dev - device structure
 *
 * Return Value:
 *  Void
 */
void
rcon_irq_free(struct rcon_dev *dev)
{
     /* Sibyte 1480 mailbox register on TxRx or Sibyte 1125 on SSC
      */
    free_irq(mgmtbus_rcon_mailbox_value, dev);
    return;
}
#endif /* CONFIG_RCON */

