/* bobcat_mgmt_bus.c
 *
 * Description     : Bobcat platform specific routines.
 *
 * Copyright (c) 2007 OnStor, Inc.
 * All rights reserved.
 */


#include <linux/types.h>
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <linux/netdevice.h>
#include <net/neteee/edesc.h>
#include "mgmt-bus.h"
#include "rcon.h"
#include <linux/mv643xx.h>
#include <asm/marvell.h>
#include <asm/mach-bobcat/bobcat.h>
#include "../../arch/mips/onstor/bobcat/bobcat_bm_fpga.h"

/*++

Routine Description:

    Interrupt handler for pci inbound register. The register is written
    by TXRX when a message is added into the SSC input ring.

Arguments:

    irq - interrupt number
    dev_id - pointer to device structure

Return Value:

    Returns IRQ_HANDLED.

--*/

 irqreturn_t
mgmtbus_pci_intr(int irq, void *dev_id)
{
    uint64_t cause = MV_READ(MV64340_I2O_INBOUND_INTERRUPT_CAUSE_REG_CPU1_SIDE);
    uint32_t mask;

    /*
     * Ack the interrupt.
     */
    MV_WRITE(MV64340_I2O_INBOUND_INTERRUPT_CAUSE_REG_CPU1_SIDE,
             (cause & 0x00010001));

    mask = 0;
    if ((cause & 0x00000001) != 0) {
        mask |= (1 << MGMTBUS_ADDR_TXRX);
    }

    if ((cause & 0x00010000) != 0) {
        mask |= (1 << MGMTBUS_ADDR_FP);
    }

    if (mask) {
        mgmtbus_intr(mask);
    }

    return IRQ_HANDLED;
}

/*++

Routine Description:

    Interrupt handler for bmfpga message interrupt.

Arguments:

    irq - interrupt number
    dev_id - pointer to device structure

Return Value:
)
    Returns IRQ_HANDLED.

--*/

static irqreturn_t
mgmtbus_bmfpga_intr(int irq, void *dev_id)
{
    /*
     * Read the message in register to clear the interrupt.
     */
    uint32_t msg = ONSTOR_FPGA_READ(SSC_MBX_IN);

    if (msg) {
        mgmtbus_intr(1 << MGMTBUS_ADDR_FC);
    }

    return IRQ_HANDLED;
}


/*++

Routine Description:

    Release the interrupt resources for the mgmtbus driver.

Arguments:

    dev - device structure

Return Value:

    None.

--*/

void
mgmtbus_irq_free(struct net_device *dev)
{
    /*
     * Mask the interrupts.
     */
    MV_WRITE(MV64340_CPU_INTERRUPT0_MASK_LOW,
             MV_READ(MV64340_CPU_INTERRUPT0_MASK_LOW) & ~(1 << 22));
    MV_WRITE(MV64340_CPU_INTERRUPT0_MASK_LOW,
             MV_READ(MV64340_CPU_INTERRUPT0_MASK_LOW) & ~(1 << 20));
    MV_WRITE(MV64340_I2O_INBOUND_INTERRUPT_MASK_REG_CPU1_SIDE,
             MV_READ(MV64340_I2O_INBOUND_INTERRUPT_MASK_REG_CPU1_SIDE) |
			 0x00010001);

    free_irq(ONSTOR_BMPGA_MSG_IN_IRQ, NULL);

    free_irq(16 + 22, dev);

    free_irq(16 + 20, dev);


    printk(KERN_INFO "MGMTBUS: released irqs %d, %d, %d\n", 36, 38,
           ONSTOR_BMPGA_MSG_IN_IRQ);
}


/*++

Routine Description:

    Setup the interrupts for the mgmtbus driver.

Arguments:

    dev - device structure

Return Value:

    0 - success, -error on failure.

--*/

int
mgmtbus_irq_setup(struct net_device *dev)
{
    int ret;

    ret = request_irq(16 + 20, mgmtbus_pci_intr, IRQF_SHARED, 
                      dev->name, dev);
    if (ret) {
        printk(KERN_ERR "%s: can't register irq 36, %d\n", 
               __FUNCTION__, ret);
        goto out;
    }

    ret = request_irq(16 + 22, mgmtbus_pci_intr, IRQF_SHARED,
                      dev->name, dev);

    if (ret) {
        printk(KERN_ERR "%s: can't register irq 38, %d\n",
               __FUNCTION__, ret);
        free_irq(36, NULL);
        goto out;
    }

    ret = request_irq(ONSTOR_BMPGA_MSG_IN_IRQ, mgmtbus_bmfpga_intr,
                      IRQF_SAMPLE_RANDOM, dev->name, NULL);
    if (ret) {
        printk(KERN_ERR "%s: can't register irq %d, %d\n", __FUNCTION__,
               ONSTOR_BMPGA_MSG_IN_IRQ, ret);
        free_irq(36, NULL);
        free_irq(38, NULL);
        goto out;
    }

    /*
     * Unmask the interrupts.
     */
    MV_WRITE(MV64340_CPU_INTERRUPT0_MASK_LOW,
             MV_READ(MV64340_CPU_INTERRUPT0_MASK_LOW) | (1 << 20));
    MV_WRITE(MV64340_CPU_INTERRUPT0_MASK_LOW,
             MV_READ(MV64340_CPU_INTERRUPT0_MASK_LOW) | (1 << 22));
    MV_WRITE(MV64340_I2O_INBOUND_INTERRUPT_MASK_REG_CPU1_SIDE,
             ~0x00010001);

    printk(KERN_INFO
           "MGMTBUS: using irqs %d, %d, %d\n", 36, 38,
           ONSTOR_BMPGA_MSG_IN_IRQ);
  out:
    return ret;
}


irqreturn_t
rcon_intr(int irq, void *dev_id)
{
    uint64_t cause = MV_READ(MV64340_I2O_INBOUND_INTERRUPT_CAUSE_REG_CPU1_SIDE);
    int i;

    if (cause & 0x10000) {
        /*
         * Ack the interrupt.
         */
        MV_WRITE(MV64340_I2O_INBOUND_INTERRUPT_CAUSE_REG_CPU1_SIDE, 0x10000);

        for (i = 0; i < RCON_MAX_CORES; ++i) {
            rcon_process(i);
        }
    }

    return IRQ_HANDLED;
}

int
rcon_irq_setup(struct rcon_dev *dev)
{
    int rc;

    rc = request_irq(16 + 20, rcon_intr, IRQF_SHARED, "rcon", dev);
    if (rc < 0) {
        printk("%s: failed to register irq 20: %d\n", __FUNCTION__, rc);
        goto out;
    }

    rc = request_irq(16 + 22, rcon_intr, IRQF_SHARED, "rcon", dev);
    if (rc < 0) {
        printk("%s: failed to register irq 22: %d\n", __FUNCTION__, rc);
        goto out;
    }

    MV_WRITE(MV64340_CPU_INTERRUPT0_MASK_LOW,
             MV_READ(MV64340_CPU_INTERRUPT0_MASK_LOW) | (1 << 20));
    MV_WRITE(MV64340_CPU_INTERRUPT0_MASK_LOW,
             MV_READ(MV64340_CPU_INTERRUPT0_MASK_LOW) | (1 << 22));
    MV_WRITE(MV64340_I2O_INBOUND_INTERRUPT_MASK_REG_CPU1_SIDE,
             MV_READ(MV64340_I2O_INBOUND_INTERRUPT_MASK_REG_CPU1_SIDE) &
             ~0x00010000);

  out:
    return rc;
}


void
rcon_irq_free(struct rcon_dev *dev)
{
    MV_WRITE(MV64340_I2O_INBOUND_INTERRUPT_MASK_REG_CPU1_SIDE,
             MV_READ(MV64340_I2O_INBOUND_INTERRUPT_MASK_REG_CPU1_SIDE) |
             0x00010000);

    free_irq(16 + 20, dev);
    free_irq(16 + 22, dev);
}

extern uint8_t eee_ssc_ext_mac[2][6];

/*
 * The host id is 4th+5th bytes of the MAC address (counting from 1).
 */
void
mgmtbus_init_hostid()
{
    magicmanagementbusringconfig->hostid = ((eee_ssc_ext_mac[0][3] << 8) |
                                            eee_ssc_ext_mac[0][4]);
}
