/* bobcat_mgmt_bus.c
 *
 * Description     : Cougar platform specific routines.
 *
 * Copyright (c) 2007 OnStor, Inc.
 * All rights reserved.
 */

#include <linux/types.h>
#include <linux/irqreturn.h>
#include <linux/netdevice.h>
#include <net/neteee/edesc.h>
#include "mgmt-bus.h"
#include "rcon.h"
#include <asm/mach-cougar/cougar.h>
#include <asm/sibyte/sb1250_int.h>
#include <asm/sibyte/sb1250_regs.h>
#include <asm/sibyte/sb1250.h>

/*++

Routine Description:

    Interrupt handler for mbox interrupt.

Arguments:

    irq - interrupt number
    dev_id - pointer to device structure

Return Value:

    Returns IEQ_HANDLED.

--*/

static irqreturn_t
mgmtbus_mailbox_intr(int irq, void *dev_id)
{
    /* Acknowledge the interrupt by clearing the mailbox.
     */
    *(uint64_t *)PHYS_TO_XKSEG_UNCACHED(
        A_IMR_REGISTER(0, R_IMR_MAILBOX_CLR_CPU)) = 0xFFFFUL;

   mgmtbus_intr(1 << MGMTBUS_ADDR_TXRX);

   return IRQ_HANDLED;
}

/*++

Routine Description:

    Release the interrupt for the mgmtbus driver.

Arguments:

    dev - device structure

Return Value:

    N/A

--*/

void
mgmtbus_irq_free(struct net_device *dev)
{
    free_irq(K_INT_MBOX_3, dev);
    printk(KERN_INFO "MGMTBUS: releasing irq %d\n", K_INT_MBOX_3);
}

/*++

Routine Description:

    Setup the interrupts for the mgmtbus driver.

Arguments:

    dev - device structure

Return Value:

    0 - success, -error on failure.

--*/

int
mgmtbus_irq_setup(struct net_device *dev)
{
    int ret;

    ret = request_irq(K_INT_MBOX_3, mgmtbus_mailbox_intr,
                      IRQF_SHARED, dev->name, dev);
    if (ret) {
        printk(KERN_ERR "%s: can't register irq %d: %d\n", __FUNCTION__,
               K_INT_MBOX_3, ret);
        goto out;
    }

    printk(KERN_INFO "MGMTBUS: using irq %d\n", K_INT_MBOX_3);

  out:
    return ret;
}

irqreturn_t
rcon_intr(int irq, void *dev_id)
{
    int i;

    /*
     * Ack the interrupt by clearing the mailbox.
     */
    *(uint64_t *)PHYS_TO_XKSEG_UNCACHED(
        A_IMR_REGISTER(0, R_IMR_MAILBOX_CLR_CPU)) = 0xFFFF0000UL;

    for (i = 0; i < RCON_MAX_CORES; ++i) {
        rcon_process(i);
    }

    return IRQ_HANDLED;
}

int
rcon_irq_setup(struct rcon_dev *dev)
{
    int rc;

    rc = request_irq(K_INT_MBOX_2, rcon_intr, IRQF_SHARED, "rcon", dev);
    if (rc < 0) {
        printk("%s: failed to register irq %d: %d\n", __FUNCTION__, 
               K_INT_MBOX_2, rc);
    }

    return rc;
}

void
rcon_irq_free(struct rcon_dev *dev)
{
    free_irq(K_INT_MBOX_2, dev);
}


extern uint64_t sbmac_orig_hwaddr[];

/*
 * The host id is 4th+5th+6th bytes of the MAC address (counting from 1).
 */
void
mgmtbus_init_hostid()
{
    unsigned char *c = (unsigned char *)&sbmac_orig_hwaddr[0];
    magicmanagementbusringconfig->hostid =
        (c[3] << 16) | (c[4] << 8) | c[5];
}
