/*                               -*- Mode: C -*-
 * mgmt-bus.c
 *
 * Description     : SSC shared memory management bus driver.
 *
 * Author          : Tim Gardner
 * Created On      : Thu Jun 24 15:31:17 2004
 *
 * Copyright (c) 2007 OnStor, Inc.
 * All rights reserved.
 */

/*
 * This driver is implemented as a pseudo network driver.
 *
 * To configure:
 *   ifconfig mgmtbus0 eee 2.0.0 up
 *
 * To enable debug output:
 *   ifconfig mgmtbus0 debug
 * To disable debug output:
 *   ifconfig mgmtbus0 -debug
 */

/*
 * Theory of operation:
 *
 * Purpose:
 *
 * This driver is the low level SSC management bus driver that supports
 * sending/receiving management bus packets to/from the TxRx, FP, and FC.  The
 * management bus protocol implemented by this driver is document in the
 * Bobcat Management Bus Protocol Design Specification.  This driver exists
 * below the eee protocol module in the kernel.  The eee protocol module
 * (neteee) code resides in net/neteee.
 *
 * This driver is basically a shared memory point-to-point network driver.
 * For each point-to-point link (SSC to FC, etc) there exists two pairs of
 * message descriptor rings.  One pair is used for sending messages from the
 * SSC; the other is used for sending messages to the SSC.  Each pair consists
 * of a ring of available message descriptors and a ring of used message
 * descriptors.  When sending a packet, the source CPU copies the packet data
 * to the buffer referenced by the next descriptor in the available ring.  The
 * associated descriptor in the used ring is then updated to inform the
 * destination CPU that a packet is available to be recieved.  Available
 * descriptor rings are always located in memory local to the sender.  Used
 * descriptor rings are always located in memory local to the receiver.  This
 * enables all packet transfers to be done using only PCI writes.  No PCI
 * reads are needed eliminating the CPU stalls waiting on completion of a PCI
 * read.  A flags field in each descriptor indicates the state of the
 * descriptor - available or used.  Used meaning that the descriptor
 * references a packet in need of receive processing.
 *
 * Initialization:
 *
 * Initialization is performed by the mgmtbusattach function which is called
 * during kernel boot.  All descriptor rings residing in shared memory are
 * initialized.  It is assummed that the SSC will always boot and complete
 * initialization before any of the other CPUs boot.  A configuration table
 * located at a well known address provides the address and size of every
 * descriptor ring.  Upon initialization, each CPU initializes this table for
 * the rings residing in memory local to the CPU.
 *
 * Packet Transmission:
 *
 * Packet output is performed by the mgmtbus_output function which is called
 * by neteee.  Each packet to be transmitted is in the form of an mbuf.  At
 * the beginning of every packet is a header defined by agile_hdr_aligned_t.
 * The packet received from neteee actually has an agile_hdr_t header which
 * begins with 2 pad bytes.  The output code translates the header before
 * sending.  This header contains the source and destination addresses for the
 * transfer.  Transfer is accomplished by copying the packet data to the
 * buffer referenced by the next descriptor in the available message
 * descriptor ring.  An ENOBUFS error is returned to neteee if there is no
 * available message descriptor.  Ideally DMA should be used instead of a
 * copy.  Unfortunately the Galileo IDMA controller cannot access local memory
 * (only Marvell memory) and outbound mbufs are all located in local memory.
 * This last is only applicable to Bobcat platform.
 *
 * Packet Reception:
 *
 * Packet receive processing is performed by the mgmtbus_intr function.  This
 * interrupt handler is currently called periodically when a timer expires.
 * Eventually this needs to be changed so that it is called as a result of an
 * interrupt generated by the TxRx/FP/FC.  The interrupt handler checks all
 * used descriptor rings for messages waiting to be processed.  Each message
 * is processed by allocating an extended mbuf, initializing the mbuf to
 * directly reference the message data in shared memory, and forwarding the
 * mbuf to neteee.  A message block is taken from the free pool to replace the
 * used message descriptor.  When neteee frees the mbuf, the mgmtbus_mbuf_free
 * function is called which returns the mbuf to the free pool.  If the free
 * pool is empty when processing a message, then a regular mbuf is allocated
 * and the data is copied from the message block to the mbuf.  If an mbuf
 * cannot be allocated, then message processing is deferred until the next
 * interrupt.  Incomming packets begin with an agile_hdr_aligned_t header.
 * This is translated into an agile_hdr_t header before passing the mbuf to
 * neteee.
 */

#include <linux/types.h>
#include <linux/param.h>
#include <linux/socket.h>
#include <linux/ioctl.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/if.h>
#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <linux/etherdevice.h>
#include <net/neteee/edesc.h>
#ifdef CONFIG_ONSTOR_BOBCAT
# include <linux/mv643xx.h>
# include <asm/marvell.h>
#endif

#include "mgmt-bus.h"
#include "rcon.h"

/*
 * there has to be a better way...
 */
#ifdef CONFIG_ONSTOR_BOBCAT
# include <asm/mach-bobcat/bobcat.h>
# include "../../arch/mips/onstor/bobcat/bobcat_bm_fpga.h"
#elif defined(CONFIG_ONSTOR_COUGAR)
# include <asm/mach-cougar/cougar.h>
# include <asm/sibyte/sb1250_regs.h>
# include <asm/sibyte/sb1250_int.h>
# include "../../arch/mips/onstor/cougar/cougar_bm_fpga.h"
#else
# error "Error in configuration: no platform specified."
#endif

#define EEE_DEBUG
#undef EEE_DEBUG

#ifdef EEE_DEBUG
#define eee_dbg(x) printk x
#else
#define eee_dbg(x)
#endif

/*
 * @@@
 * L1 and L2 cache flush/sync macro.
 * Current design has the Marvell memory mapped uncached, and does not
 * define the macro MGMTBUS_CACHEFLUSH.
 */
#ifdef MGMTBUS_CACHEFLUSH
#define MGMTBUS_SYNC_CACHE(addr, size, op)  xxxxx
#else
#define MGMTBUS_SYNC_CACHE(addr, size, op)
#endif

/*
 * Macro for aligning addresses and sizes to a the size of a cache line.
 */
#define MGMTBUS_ALIGNMENT   32
#define MGMTBUS_ALIGN(addr) ALIGN(addr, MGMTBUS_ALIGNMENT)

/* Address conversion macros for converting between physical and virtual
 * addresses.  All addresses in the management bus ring configuration table
 * are physical.
 */

/*
 * Driver software context.
 */
typedef struct
{
    mgmtbus_ring_config_t *ring_config;
    void *txrx_pci_virt;
    void *fp_pci_virt;
    void *ssc_pci_virt;
    unsigned long txrx_pci_base;
    unsigned long fp_pci_base;
    unsigned long ssc_pci_base;
} mgmtbus_softc_t;

mgmtbus_softc_t Mgmtbus_softc;
mgmtbus_ring_config_t *magicmanagementbusringconfig;

/*
 * Holding spot for the model number until the mgmtbus driver init routine
 * runs.
 */
static int model_num = 0x104;  /* seed it with a funny number
								  so we can see if it's been changed */

#if defined(CONFIG_ONSTOR_BOBCAT)
/*
 * Two memory regions to be concerned with: Marvell and PCI.
 *
 * Marvell physical memory is located at 0x40000000 and is mapped into virtual
 * memory starting at 0xf0000000.
 *
 * PCI physical addresses are above 0x9a000000 and are mapped into virtual
 * memory starting at 0x80000000 (cached) and 0xa0000000 (uncached).
 */
/*
 * actually, what's going on is that we are getting a
 * cached virtual address 0x9axxxxxx plugged into a spot
 * (t->table[src][MGMTBUS_ADDR_SSC].avail_ring_address - line 688)
 * which should have a physical address in it 0x1axxxxxx.  And the
 * virtual address shouldn't be the cached one anyway, it should be
 * the uncached one 0xbaXXXXXX
 *
 * so, to fix this bug (it's probably an issue in the txrx/fp runtime code
 * i'm guessing, since it isn't in this driver), we just add the missing
 * 0x20000000 to the virtual address to make it an uncached address
 * instead of a cached one.  but we're probably still going to have problems.
 */
//# define MGMTBUS_PHYS2VIRT(phys) phys_to_virt((unsigned long)phys)

# define MGMTBUS_PHYS2VIRT(phys) ((phys < 0x50000000) ? \
									phys_to_virt(phys) : \
									(void *)(phys | 0x20000000))

#elif defined(CONFIG_ONSTOR_COUGAR)

/*
 * Physical start of PCI space.
 */
#define PCI_PHYS_START 0xF800000000UL

/*
 * Offsets of the CPUs memory in the PCI space.
 */
#define SSC_PCI_OFFSET  0x60000000UL
#define TXRX_PCI_OFFSET 0x1a000000UL
#define FP_PCI_OFFSET   0x1b000000UL

/*
 * On cougar the addresses passed around are 32bit offsets in the PCI
 * address space.
 */

static void *
MGMTBUS_PHYS2VIRT(unsigned long phys)
{
    if (MGMTBUS_SSC_BASE_ADDR <= phys &&
        phys < MGMTBUS_SSC_BASE_ADDR + SHARED_MEM_MGMT_SIZE) {
        return Mgmtbus_softc.ssc_pci_virt + ((phys) - MGMTBUS_SSC_BASE_ADDR);
    } else if (Mgmtbus_softc.ssc_pci_base <= phys &&
               phys < Mgmtbus_softc.ssc_pci_base + SHARED_MEM_MGMT_SIZE) {
        return Mgmtbus_softc.ssc_pci_virt +
            (phys - Mgmtbus_softc.ssc_pci_base);
    } else if (Mgmtbus_softc.txrx_pci_base <= phys &&
               phys < Mgmtbus_softc.txrx_pci_base + SHARED_MEM_MGMT_SIZE) {
        return Mgmtbus_softc.txrx_pci_virt +
            (phys - Mgmtbus_softc.txrx_pci_base);
    } else if (Mgmtbus_softc.fp_pci_base <= phys &&
               phys < Mgmtbus_softc.fp_pci_base + SHARED_MEM_MGMT_SIZE) {
        return Mgmtbus_softc.fp_pci_virt +
            (phys - Mgmtbus_softc.fp_pci_base);
    } else {
		struct pt_regs *regs = get_irq_regs();
		printk("%s: invalid phys addr %#lx\n", __FUNCTION__, phys);
		die("Oops", regs);
    }
}


#if 0
/*
 * Not used currently, leave the code just in case.
 */
static unsigned long
MGMTBUS_VIRT2PHYS(void *virt)
{
    if (Mgmtbus_softc.ssc_pci_virt <= virt &&
        virt < Mgmtbus_softc.ssc_pci_virt + SHARED_MEM_MGMT_SIZE) {
        return Mgmtbus_softc.ssc_pci_base + virt - Mgmtbus_softc.ssc_pci_virt;
    } else if (Mgmtbus_softc.fp_pci_virt <= virt &&
        virt < Mgmtbus_softc.fp_pci_virt + SHARED_MEM_MGMT_SIZE) {
        return Mgmtbus_softc.fp_pci_base + virt - Mgmtbus_softc.fp_pci_virt;
    } if (Mgmtbus_softc.txrx_pci_virt <= virt &&
        virt < Mgmtbus_softc.txrx_pci_virt + SHARED_MEM_MGMT_SIZE) {
        return Mgmtbus_softc.txrx_pci_base + virt - Mgmtbus_softc.txrx_pci_virt;
    } else {
        panic("%s: invalid virt addr %p\n", __FUNCTION__, virt);
    }
}
#endif

#endif

extern phys_t marvel_mem_alloc(int, int *);

/*++

Routine Description:

    Initialize management bus ring configuration table.  Addresses of rings
    residing in SSC memory are initialized to a valid address.  All other ring
    addresses are initialized to 0.  Each remote CPU is responsible for
    initializing the addresses of its rings when the CPU boots.

Arguments:

    None.

Return Value:

    None.

--*/

static void
initConfigTable(void)
{
    mgmtbus_ring_config_t*  t = Mgmtbus_softc.ring_config;
    unsigned long ssc_addr  = Mgmtbus_softc.ssc_pci_base;

    ssc_addr += MGMTBUS_ALIGN(sizeof(mgmtbus_ring_config_t));

    memset(t, 0, sizeof(mgmtbus_ring_config_t));

	/*
	 * set the model number into the structure
	 */
	t->model_num = (u16)(model_num & 0xffff);

    ssc_addr += sizeof(struct rcon_queue) * RCON_MAX_CORES;

#ifdef MGMTBUS_TEST
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_SSC].ring_size = MGMTBUS_SSC_2_SSC_RING_SIZE;
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_SSC].avail_ring_address =
        ssc_addr;
    ssc_addr  += MGMTBUS_RING_SIZE(MGMTBUS_SSC_2_SSC_RING_SIZE);
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_SSC].used_ring_address = ssc_addr;
    ssc_addr += MGMTBUS_RING_SIZE(MGMTBUS_SSC_2_SSC_RING_SIZE);
#endif

    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].ring_size = MGMTBUS_SSC_2_TXRX_RING_SIZE;
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].avail_ring_address = ssc_addr;
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_TXRX].used_ring_address  = 0;
    ssc_addr += MGMTBUS_RING_SIZE(MGMTBUS_SSC_2_TXRX_RING_SIZE);

    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FP].ring_size = MGMTBUS_SSC_2_FP_RING_SIZE;
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FP].avail_ring_address   = ssc_addr;
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FP].used_ring_address    = 0;
    ssc_addr  += MGMTBUS_RING_SIZE(MGMTBUS_SSC_2_FP_RING_SIZE);

    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FC].ring_size = MGMTBUS_SSC_2_FC_RING_SIZE;
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FC].avail_ring_address = ssc_addr;
    t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_FC].used_ring_address = 0;
    ssc_addr  += MGMTBUS_RING_SIZE(MGMTBUS_SSC_2_FC_RING_SIZE);

    t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].ring_size = MGMTBUS_TXRX_2_SSC_RING_SIZE;
    t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].avail_ring_address = 0;
    t->table[MGMTBUS_ADDR_TXRX][MGMTBUS_ADDR_SSC].used_ring_address  = ssc_addr;
    ssc_addr += MGMTBUS_RING_SIZE(MGMTBUS_TXRX_2_SSC_RING_SIZE);

    t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_SSC].ring_size = MGMTBUS_FP_2_SSC_RING_SIZE;
    t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_SSC].avail_ring_address = 0;
    t->table[MGMTBUS_ADDR_FP][MGMTBUS_ADDR_SSC].used_ring_address = ssc_addr;
    ssc_addr += MGMTBUS_RING_SIZE(MGMTBUS_FP_2_SSC_RING_SIZE);

    t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_SSC].ring_size = MGMTBUS_FC_2_SSC_RING_SIZE;
    t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_SSC].avail_ring_address = 0;
    t->table[MGMTBUS_ADDR_FC][MGMTBUS_ADDR_SSC].used_ring_address = ssc_addr;
    ssc_addr += MGMTBUS_RING_SIZE(MGMTBUS_FC_2_SSC_RING_SIZE);

    MGMTBUS_SYNC_CACHE(t, sizeof(mgmtbus_ring_config_t), 1);
}


/*++

Routine Description:

    Initialize all descriptor rings residing in SSC memory.

Arguments:

    None.

Return Value:

    None.

--*/

static void
initDescriptorRings(void)
{
    mgmtbus_ring_config_t*      t = Mgmtbus_softc.ring_config;
    unsigned long ssc_addr;
    uint32_t                        dest;
    uint32_t                        src;
    uint32_t                        i;
    mgmtbus_msg_desc_ring_t*    ring;

    /* Set ssc_addr to the starting address of the SSC message buffers.  The
     * message buffer memory is located immediately following the SSC
     * descriptor ring memory.
     */
    ssc_addr = t->table[MGMTBUS_ADDR_MAX][MGMTBUS_ADDR_SSC].used_ring_address;
    ssc_addr += MGMTBUS_RING_SIZE(t->table[MGMTBUS_ADDR_MAX][MGMTBUS_ADDR_SSC].ring_size);

#ifdef MGMTBUS_TEST
    {
        /*
         * Initialize rings used for loopback.
         */
        mgmtbus_msg_desc_ring_t*    avail_ring;
        mgmtbus_msg_desc_ring_t*    used_ring;

        avail_ring = MGMTBUS_PHYS2VIRT(t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_SSC].avail_ring_address);
        used_ring = MGMTBUS_PHYS2VIRT(t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_SSC].used_ring_address);
        avail_ring->next_index = 0;
        avail_ring->ring_size = t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_SSC].ring_size;
        used_ring->next_index = 0;
        used_ring->ring_size = t->table[MGMTBUS_ADDR_SSC][MGMTBUS_ADDR_SSC].ring_size;

        for (i = 0; i < avail_ring->ring_size; i++) {
            avail_ring->ring[i].flags    = 0;
            avail_ring->ring[i].index    = i;
            avail_ring->ring[i].size     = MGMTBUS_MTU;
            avail_ring->ring[i].address  = ssc_addr;
            used_ring->ring[i].flags     = 0;
            used_ring->ring[i].index     = i;
            used_ring->ring[i].size      = MGMTBUS_MTU;
            used_ring->ring[i].address   = ssc_addr;

            ssc_addr += MGMTBUS_MTU;
        }

        MGMTBUS_SYNC_CACHE(avail_ring,
                           sizeof(mgmtbus_msg_desc_ring_t) +
                           (avail_ring->ring_size * sizeof(mgmtbus_msg_desc_t)),
                           1);
        MGMTBUS_SYNC_CACHE(used_ring,
                           sizeof(mgmtbus_msg_desc_ring_t) +
                           (used_ring->ring_size * sizeof(mgmtbus_msg_desc_t)),
                           1);
    }
#endif

    /*
     * Initialize all available descriptor rings in SSC memory.
     * These rings are used for sending data from the SSC to other CPUs.
     * Mark descriptors as used. They will be initialized
     * by each destination processor when it starts.
     */
    for (dest = 1; dest <= MGMTBUS_ADDR_MAX; dest++) {
        ring = MGMTBUS_PHYS2VIRT(
            t->table[MGMTBUS_ADDR_SSC][dest].avail_ring_address);
        ring->next_index = 0;
        ring->ring_size  = t->table[MGMTBUS_ADDR_SSC][dest].ring_size;

        for (i = 0; i < ring->ring_size; i++)
        {
            ring->ring[i].flags    = MGMTBUS_DESC_FLAGS_USED;
            ring->ring[i].index    = i;
            ring->ring[i].size     = 0;
            ring->ring[i].address  = 0;
        }

        MGMTBUS_SYNC_CACHE(ring,
                           sizeof(mgmtbus_msg_desc_ring_t) +
                           (ring->ring_size * sizeof(mgmtbus_msg_desc_t)),
                           1);
    }

    /*
     * Initialize all used descriptor rings in SSC memory.
     * These rings are used for receiving messages sent by remote CPUs.
     * Initialize descriptors to point to SSC message buffers.
     */
    for (src = 1; src <= MGMTBUS_ADDR_MAX; src++) {
        ring = MGMTBUS_PHYS2VIRT(
            t->table[src][MGMTBUS_ADDR_SSC].used_ring_address);
        ring->next_index = 0;
        ring->ring_size  = t->table[src][MGMTBUS_ADDR_SSC].ring_size;

        for (i = 0; i < ring->ring_size; i++) {
            ring->ring[i].flags    = 0;
            ring->ring[i].index    = i;
            ring->ring[i].size     = MGMTBUS_MTU;
            ring->ring[i].address  = ssc_addr;
            ssc_addr += MGMTBUS_MTU;
        }

        MGMTBUS_SYNC_CACHE(ring,
                           sizeof(mgmtbus_msg_desc_ring_t) +
                           (ring->ring_size * sizeof(mgmtbus_msg_desc_t)),
                           1);
    }

    eee_dbg(("%s: start = %p end %p\n", __FUNCTION__, t,
             (void *)phys_to_virt(ssc_addr)));
}

/*++

Routine Description:

    Convert a eee address to a management bus address.  A management bus
    address is just the index of the rings within the ring configuration
    table.

Arguments:

    port - eee destination port

Return Value:

    Returns the index in the configuration table.

--*/

static int
mgmtbus_convert_eee_addr(uint32_t port)
{
    int     addr = -1;
    uint32_t dest_cpu = EEE_GET_DEST_CPU(port);

    switch (EEE_GET_SLOT_NUM(port)) {
    case NFX_SLOTTYPE_SSC:
        if (dest_cpu == NFX_SSC_UNIX) {
            addr = MGMTBUS_ADDR_SSC;
        }
#ifdef MGMTBUS_TEST
        else {
            /*
             * For unit testing, all packets sent to an invalid CPU are looped
             * back.
             */
            addr = MGMTBUS_ADDR_SSC;
        }
#endif
        break;

    case NFX_SLOTTYPE_NFP:
        if ((NFX_NFP_TXRX1 <= dest_cpu) && (dest_cpu <= NFX_NFP_FP_ANY)) {
			/*
			 * @@@ TxRx currently provides a proxy service for FP messages so
			 * all FP messages need to be sent to the TxRx.  Change this once
			 * embedded is changed.
			 */
            addr = MGMTBUS_ADDR_TXRX;
        }
        break;

    case NFX_SLOTTYPE_FC:
#ifdef CONFIG_ONSTOR_BOBCAT
        addr = MGMTBUS_ADDR_FC;
#else
        addr = MGMTBUS_ADDR_TXRX;
#endif
        break;

    default:
        break;
    }

    return addr;
}


/*++

Routine Description:

    Driver transmit function.

Arguments:

    skb - the skb to send
    dev - pointer to the device structure

Return Value:

    Returns 0, the messages are dropped if can not transmit.

--*/

static int
mgmtbus_hard_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
    int                             len;
    agile_hdr_t*                    hdr = (agile_hdr_t *)skb_network_header(skb);
    int                             addr;
    mgmtbus_ring_config_t*          t = Mgmtbus_softc.ring_config;
    mgmtbus_msg_desc_ring_t*        used_ring;
    mgmtbus_msg_desc_ring_t*        avail_ring;
    volatile mgmtbus_msg_desc_t*    used_msg_desc;
    volatile mgmtbus_msg_desc_t*    avail_msg_desc;
    void*                           virt;
    int                             send_size;
    
    eee_dbg(("%s: %d/%d/%d len %d\n", __FUNCTION__,
             EEE_GET_SLOT_NUM(hdr->dest_port),
             EEE_GET_DEST_CPU(hdr->dest_port),
             EEE_GET_APP_ID(hdr->dest_port),
             skb->len));

    /*
     * Convert the destination address to a management bus address.
     */
    addr = mgmtbus_convert_eee_addr(hdr->dest_port);

    if (addr == -1) {
        goto out;
    }

#ifndef MGMTBUS_TEST
    if (addr == MGMTBUS_ADDR_SSC) {
        goto out;
    }
#endif

    MGMTBUS_SYNC_CACHE(t, sizeof(mgmtbus_ring_config_t), 0);

    /*
     * Verify destination CPU has completed initialization.
     */
    if (t->table[MGMTBUS_ADDR_SSC][addr].used_ring_address == 0) {
        goto out;
    }

    len = skb->len;


    used_ring = (mgmtbus_msg_desc_ring_t*)MGMTBUS_PHYS2VIRT(t->table[MGMTBUS_ADDR_SSC][addr].used_ring_address);

    avail_ring = (mgmtbus_msg_desc_ring_t*)MGMTBUS_PHYS2VIRT(t->table[MGMTBUS_ADDR_SSC][addr].avail_ring_address);
    avail_msg_desc = &avail_ring->ring[avail_ring->next_index];
    MGMTBUS_SYNC_CACHE(avail_msg_desc, sizeof(*avail_msg_desc), 0);

    /*
     * Verify ring is not full.
     */
    if ((avail_msg_desc->flags & MGMTBUS_DESC_FLAGS_USED) != 0) {
        printk(KERN_WARNING "%s: ring is full\n", __FUNCTION__);
        goto out;
    }

    /*
     * Verify descriptor references a message buffer.
     */
    if (avail_msg_desc->address == 0) {
        panic("null message buffer address\n");
    }

    virt = MGMTBUS_PHYS2VIRT(avail_msg_desc->address);

    send_size = skb->len - sizeof(hdr->pad) - ETH_HLEN;
    memcpy(virt, &hdr->dest_port, send_size);
    
    used_msg_desc = &used_ring->ring[avail_ring->next_index];

    avail_msg_desc->flags = MGMTBUS_DESC_FLAGS_USED;
    MGMTBUS_SYNC_CACHE(avail_msg_desc, sizeof(*avail_msg_desc), 1);

    used_msg_desc->size   = send_size;
    used_msg_desc->flags  = MGMTBUS_DESC_FLAGS_USED;
    MGMTBUS_SYNC_CACHE(used_msg_desc, sizeof(*used_msg_desc), 1);

    if (++avail_ring->next_index == avail_ring->ring_size) {
        avail_ring->next_index = 0;
    }

out:
    kfree_skb(skb);

    return 0;
}


static struct net_device_stats mgmtbus_stats;

static struct net_device_stats *
mgmtbus_get_stats(struct net_device *dev)
{
    return &mgmtbus_stats;
}


struct net_device mgmtbus_dev = {
    .name           = "mgmtbus",
    .mtu            = 1536,
    .hard_start_xmit    = mgmtbus_hard_start_xmit,
    .hard_header        = eth_header,
    .hard_header_len    = ETH_HLEN,
    .addr_len       = ETH_ALEN,
    .get_stats = mgmtbus_get_stats,
    .tx_queue_len       = 1000,
    .type = ARPHRD_ETHER,
    .rebuild_header = eth_rebuild_header,
    .flags = IFF_POINTOPOINT | IFF_NOARP,
    .features       = NETIF_F_FRAGLIST
#ifdef LOOPBACK_TSO
                  | NETIF_F_TSO
#endif
                  | NETIF_F_HIGHDMA
                  | NETIF_F_LLTX,
};


/*++

Routine Description:

    Common interrupt handler for pci message register and bmfpga interrupts.

Arguments:

    cause - the combined interrupt cause, if the bit is set the
            corresponding cpu input ring needs to be processed.

Return Value:

    None.

--*/

void
mgmtbus_intr(uint32_t cause)
{
    int                         src;
    mgmtbus_ring_config_t*      t = Mgmtbus_softc.ring_config;
    struct sk_buff *skb;
    agile_hdr_t *hdr;
    struct ethhdr *eth;

    MGMTBUS_SYNC_CACHE(t, sizeof(mgmtbus_ring_config_t), 0);

    /*
     * Check each ring for packets to be processed.
     */
    for (src = 0; src <= MGMTBUS_ADDR_MAX; src++) {
        mgmtbus_msg_desc_ring_t*    used_ring;
        mgmtbus_msg_desc_ring_t*    avail_ring;
        mgmtbus_msg_desc_t*         used_desc;
        mgmtbus_msg_desc_t*         avail_desc;

        if ((cause & (1 << src)) == 0) {
            continue;
        }

        /*
         * Verify remote CPU has completed initialization.
         */
        if (t->table[src][MGMTBUS_ADDR_SSC].avail_ring_address == 0) {
            continue;
        }

        used_ring  = MGMTBUS_PHYS2VIRT(t->table[src][MGMTBUS_ADDR_SSC].used_ring_address);

        avail_ring = MGMTBUS_PHYS2VIRT(t->table[src][MGMTBUS_ADDR_SSC].avail_ring_address);

        while (1) {
            used_desc = &used_ring->ring[used_ring->next_index];
            MGMTBUS_SYNC_CACHE(used_desc, sizeof(*used_desc), 0);

            if (!(used_desc->flags & MGMTBUS_DESC_FLAGS_USED)) {
                break;
            }

            avail_desc = &avail_ring->ring[used_ring->next_index];
            MGMTBUS_SYNC_CACHE(MGMTBUS_PHYS2VIRT(used_desc->address),
                               used_desc->size, 0);

            /*
             * Allocate 2 bytes more to restore mangled agile_hdr_t.
             */
            skb = dev_alloc_skb(used_desc->size + 2 + ETH_HLEN);

            if (skb == NULL) {
                break;
            }

            eth = (struct ethhdr *)skb_put(skb, ETH_HLEN);
            memset(eth->h_source, 0, ETH_ALEN);
            memcpy(eth->h_dest, mgmtbus_dev.dev_addr, ETH_ALEN);
            eth->h_proto = htons(ETH_P_EEE);

            /*
             * Add missing pad to the header.
             */
            hdr = (agile_hdr_t *)skb_put(skb, 2);
            memset(hdr, 0, 2);

            memcpy(skb_put(skb, used_desc->size),
                   MGMTBUS_PHYS2VIRT(used_desc->address),
                   used_desc->size);

            eee_dbg(("%s: rx from %d/%d/%d to %d/%d/%d\n",
                     __FUNCTION__,
                     EEE_GET_SLOT_NUM(hdr->src_port),
                     EEE_GET_DEST_CPU(hdr->src_port),
                     EEE_GET_APP_ID(hdr->src_port),
                     EEE_GET_SLOT_NUM(hdr->dest_port),
                     EEE_GET_DEST_CPU(hdr->dest_port),
                     EEE_GET_APP_ID(hdr->dest_port)));

            skb->protocol = eth_type_trans(skb, &mgmtbus_dev);

            netif_rx(skb);

            /*
             * Mark the descriptors as available.
             */
            used_desc->flags &= ~MGMTBUS_DESC_FLAGS_USED;
            avail_desc->flags &= ~MGMTBUS_DESC_FLAGS_USED;

            MGMTBUS_SYNC_CACHE(used_desc, sizeof(*used_desc), 1);
            MGMTBUS_SYNC_CACHE(avail_desc, sizeof(*avail_desc), 1);

            if (++used_ring->next_index == used_ring->ring_size) {
                used_ring->next_index = 0;
            }
            used_desc = &used_ring->ring[used_ring->next_index];
        }
    }
}

void __exit mgmtbus_exit(void)
{
	mgmtbus_irq_free(&mgmtbus_dev);
	unregister_netdev(&mgmtbus_dev);
}


/*++

Routine Description:

    Module initialization function

Arguments:

    None.

Return Value:

    None.

--*/

int __init mgmtbus_init(void)
{
    int ret = 0;
    phys_t base_addr;

    /*
     * Allocate the marvel memory at MGMTBUS_SSC_BASE_ADDR.
     */
    base_addr = MGMTBUS_SSC_BASE_ADDR;
    if (MGMTBUS_MAX_SHARED != SHARED_MEM_MGMT_SIZE) {
        return -ENOMEM;
    }

#ifdef CONFIG_ONSTOR_BOBCAT
    Mgmtbus_softc.ring_config = ioremap(base_addr, SHARED_MEM_MGMT_SIZE);
	magicmanagementbusringconfig = Mgmtbus_softc.ring_config;
    Mgmtbus_softc.ssc_pci_base = base_addr;
#elif defined(CONFIG_ONSTOR_COUGAR)
    Mgmtbus_softc.ssc_pci_base = SSC_PCI_OFFSET;
    Mgmtbus_softc.ssc_pci_virt = (void *)PHYS_TO_XKSEG_UNCACHED(base_addr);
    Mgmtbus_softc.ring_config = Mgmtbus_softc.ssc_pci_virt;
	magicmanagementbusringconfig = Mgmtbus_softc.ring_config;
    Mgmtbus_softc.txrx_pci_base = TXRX_PCI_OFFSET;
    Mgmtbus_softc.fp_pci_base = FP_PCI_OFFSET;
    Mgmtbus_softc.txrx_pci_virt =
        (void *)PHYS_TO_XKSEG_UNCACHED(Mgmtbus_softc.txrx_pci_base |
                                       PCI_PHYS_START);
    Mgmtbus_softc.fp_pci_virt =
        (void *)PHYS_TO_XKSEG_UNCACHED(Mgmtbus_softc.fp_pci_base |
                                       PCI_PHYS_START);
#endif

    initConfigTable();
    initDescriptorRings();

    mgmtbus_init_hostid();

    /*
     * Indicate that SSC initialization is complete.
     */
    Mgmtbus_softc.ring_config->magic = MGMTBUS_RING_CONFIG_MAGIC;
    MGMTBUS_SYNC_CACHE(Mgmtbus_softc.ring_config,
                       sizeof(mgmtbus_ring_config_t), 1);

    mgmtbus_dev.dev_addr[0] = 0x4;
    mgmtbus_dev.dev_addr[1] = 0x0;
    mgmtbus_dev.dev_addr[2] = 0x0;
    mgmtbus_dev.dev_addr[3] = 0x0;
    mgmtbus_dev.dev_addr[4] = 0x1;
    mgmtbus_dev.dev_addr[5] = 0x0;
    ret = register_netdev(&mgmtbus_dev);
	if (ret) {
		printk(KERN_ERR "Failed to register %s net device: %d\n",
			mgmtbus_dev.name, ret);
		goto out;
	}

    ret = mgmtbus_irq_setup(&mgmtbus_dev);
    if (ret) {
        unregister_netdev(&mgmtbus_dev);
        goto out;
    }

    printk(KERN_INFO
           "MGMTBUS: registered net device %s @ %p\n", 
           mgmtbus_dev.name, Mgmtbus_softc.ring_config);

out:
    return ret;
}

/*++

Routine Description:

    Function called by kernel command line parser code to
	parse the onstor_model parameter that the prom passes as
	a kernel argument.

Arguments:

    str - the string containing the value of the argument

Return Value:

    0

--*/

static int __init
mgmtbus_set_model_number(char *str)
{
    /*
     * Convert the model string into a number for easier transport to the TXRX
     * It is assumed that model numbers are of the form TLA-SYS-NUMBER
     */
    char *str1;

    /*
     * Pick some default that turns everything on in the embedded layer.
     */
	model_num = 0;

	if (str == NULL) {
		return 0;
	}
    str1 = strrchr(str, '-');
    if (str1 == NULL) {
        return 0;
    }
    str1++;

    if (sscanf(str1, "%d", &model_num) != 1) {
        model_num = 0;
    }
    return 0;
}

__setup("onstor_model=", mgmtbus_set_model_number);


module_init(mgmtbus_init);
module_exit(mgmtbus_exit);
MODULE_DESCRIPTION("Management Bus Driver");
MODULE_LICENSE("GPL");

