/*                                -*- Mode: C -*-
 * mgmt-bus.h
 *
 * Description     : Management bus API.
 *
 * Author          : Tim Gardner
 * Created On      : Fri Jun 25 16:19:26 2004
 * Last Modified By: Tim Gardner
 * Last Modified On: Thu Sep  2 11:59:52 2004
 * Update Count    : 49
 *
 * Copyright (c) 2003 OnStor, Inc.
 * All rights reserved.
 */

#ifndef _mgmt_bus_h_
#define _mgmt_bus_h_

/*
 * Maximum packet size including neteee header.
 * Must be a multiple of 32 (cache line size).
 */
#define MGMTBUS_MTU     1536

/*
 * This bit indicates that the descriptor references a message block that
 * contains data that is ready to be processed by the receiver.
 */
#define MGMTBUS_DESC_FLAGS_USED 0x1

typedef struct mgmtbus_msg_desc
{
    /*
     * control flags
     */
    u32 flags;

    /*
     * index of the descriptor within the ring
     */
    u32 index;

    /*
     * size of the message block referenced by address
     */
    u32 size;

    /*
     * SSC address of the message block.  This address needs to be translated
     * before use by FP, TXRX, FC
     */
    u32 address;

    /*
     * cache line alignment pad
     */
    u32 pad[4];
} mgmtbus_msg_desc_t;


typedef struct
{
    /*
     * index of next descriptor in the ring to be used
     */
    u32                next_index;

    /*
     * number of descriptors in the ring
     */
    u32                ring_size;

    /*
     * cache line alignment pad
     */
    u32                pad[6];
    mgmtbus_msg_desc_t  ring[0];
} mgmtbus_msg_desc_ring_t;


#define MGMTBUS_RING_SIZE(numDesc) (sizeof(mgmtbus_msg_desc_ring_t) + (numDesc * sizeof(mgmtbus_msg_desc_t)))

typedef struct mgmtbus_ring_config_element
{
    /*
     * number of descriptors in the ring
     */
    u32                ring_size;

    /*
     * physical memory address
     */
    u32 avail_ring_address;

    /*
     * physical memory address
     */
    u32 used_ring_address;
} mgmtbus_ring_config_element_t;

#define MGMTBUS_ADDR_SSC    0
#define MGMTBUS_ADDR_TXRX   1
#define MGMTBUS_ADDR_FP     2
#define MGMTBUS_ADDR_FC     3
#define MGMTBUS_ADDR_MAX    3

/*
 * mgmtbus_ring_config_t.magic is set to this to indicate that the ring config
 * table has been initialized and that SSC initialization is complete.
 */
#define MGMTBUS_RING_CONFIG_MAGIC   0xbabecafe

/* Ring configuration table.
 * This data structure is located at MGMTBUS_RING_CONFIG_ADDR in Marvell
 * memory.
 *
 * Note that this is also used to communicate the number of cores we should
 * enable on the txrx and fp.  While this doesn't have a great deal to do with
 * the mgmt bus, there was little reason to duplicate the work to get a few
 * more bytes sent to the eee layer from the SSC.
 */
typedef struct
{
    /* Magic number to verify valid data.
     */
    u32 magic;

    /* The list of ring buffers configurations associated with each MAC.
     */
    mgmtbus_ring_config_element_t table[MGMTBUS_ADDR_MAX+1][MGMTBUS_ADDR_MAX+1];

    /*
     * The hostid passed to the embedded CPUs to initialize their MAC
     * addresses.
     */
    u32 hostid;

    /* The model number.
     */
    u16 model_num;

    /* Padding.
     */
    u16 pad2;

    /*
     * Pad to cache line.
     */
    u32 pad[5];
} mgmtbus_ring_config_t;


#define MGMTBUS_SSC_BASE_ADDR       SHARED_MEM_MGMT

/*
 * Location of the management bus ring configuration table in Marvell memory.
 */
#define MGMTBUS_RING_CONFIG_ADDR    MGMTBUS_SSC_BASE_ADDR

/*
 * Maximum size (in bytes) of marvell memory used for management bus.  This
 * memory starts at MGMTBUS_SSC_BASE_ADDR
 */
#define MGMTBUS_MAX_SHARED         SHARED_MEM_MGMT_SIZE

/*
 * Descriptor ring sizes (number of descriptors)
 */
#ifdef MGMTBUS_TEST
/*
 * loopback for unit testing
 */
#define MGMTBUS_SSC_2_SSC_RING_SIZE     128
#else
#define MGMTBUS_SSC_2_SSC_RING_SIZE     0
#endif

#define MGMTBUS_NUM_BUFFERS     1024
#define MGMTBUS_SSC_2_TXRX_RING_SIZE    MGMTBUS_NUM_BUFFERS
#define MGMTBUS_SSC_2_FP_RING_SIZE  MGMTBUS_NUM_BUFFERS
#define MGMTBUS_SSC_2_FC_RING_SIZE  MGMTBUS_NUM_BUFFERS
#define MGMTBUS_TXRX_2_SSC_RING_SIZE    MGMTBUS_NUM_BUFFERS
#define MGMTBUS_FP_2_SSC_RING_SIZE  MGMTBUS_NUM_BUFFERS
#define MGMTBUS_FC_2_SSC_RING_SIZE  MGMTBUS_NUM_BUFFERS
#define MGMTBUS_MIN_FREE_POOL_SIZE  MGMTBUS_NUM_BUFFERS

struct net_device;
struct rcon_dev;

void mgmtbus_intr(u32);
int mgmtbus_irq_setup(struct net_device *dev);
void mgmtbus_irq_free(struct net_device *dev);
void mgmtbus_init_hostid(void);
int rcon_irq_setup(struct rcon_dev *dev);
void rcon_process(int minor);
void rcon_irq_free(struct rcon_dev *dev);

extern mgmtbus_ring_config_t *magicmanagementbusringconfig;

#endif /* _mgmt_bus_h_ */
