/*
 *-----------------------------------------------------------------
 *
 * Name:	prom-part-api.h
 *
 * RCS:  $Id: prom-part-api.h,v 1.5 2002/11/07 17:55:17 sandrineb Exp $
 *
 * Copyright (C) 2002, Agile Storage, Inc.
 *
 * Description: header file for Flash Prom Partitioning
 *
 * Created by:	John Hahn
 *
 * Date Created:	07/25/2002
 *
 *-----------------------------------------------------------------
 */

#ifndef	_PROM_PART_API_H
#define	_PROM_PART_API_H

/******************************************************************************/

#define     PROM_PART_UNKNOWN       0
#define     PROM_PART_R7K           1
#define     PROM_PART_SIBYTE        2
#define     PROM_PART_MGMT          3
#define     PROM_PART_R7K_DIAG      4
#define     PROM_PART_SIBYTE_DIAG   5
#define     PROM_PART_SYS           6
#define     PROM_PART_LUC_HEX       7
#define     PROM_PART_LUC_BIN       8
#define     PROM_PART_LMUX_HEX      9
#define     PROM_PART_LMUX_BIN      10
#define     PROM_PART_R7K_RPROM     11
#define     PROM_PART_SIBYTE_RPROM  12
#define     PROM_PART_R9K           13
#define     PROM_PART_R9K_DIAG      14
#define     PROM_PART_R9K_RPROM     15

#ifdef LANGUAGE_C
/******************************************************************************/

/*
 * Prom Partition Descriptor for Prom memory layout
 */

typedef struct prom_part_desc {
    /* static desc */
    char *          part_name;
    unsigned int    part_offs;          // offset of this partition in prom
    unsigned int    part_size;          // size of this partition in prom
    unsigned int    part_type;          // see PROM_PART_*
} prom_part_desc_t, * prom_part_desc_p;

#endif /* LANGUAGE_C */
/******************************************************************************/

#define PROM_BOOT_OFFS      0x0
#define PROM_BOOT_SIZE      0xc0000
#define PROM_DIAG_OFFS      PROM_BOOT_SIZE
#define PROM_DIAG_SIZE      0xc0000
#define PROM_RPRM_OFFS      0x0
#define PROM_RPRM_SIZE      0x80000

#define PROM_RPRM_CODE_SIZE 0x7A000

#define PROM_PART_SYS_INFO_STR  "SYS Partition"

#ifdef LANGUAGE_C

#define PART_DESC_NULL        {NULL,       0x0, 0x0, 0}

/*
 * Partition Layout for the 4MB sibyte prom
 */
#define PART_DESC_SIBYTE_BOOT   {"sibyte_boot", PROM_BOOT_OFFS, PROM_BOOT_SIZE, PROM_PART_SIBYTE}
#define PART_DESC_SIBYTE_DIAG   {"sibyte_diag", PROM_DIAG_OFFS, PROM_DIAG_SIZE, PROM_PART_SIBYTE_DIAG}
#define PART_DESC_SIBYTE_LMUX   {"lmux_hex",    0x180000,0x080000,PROM_PART_LMUX_HEX}
#define PART_DESC_SIBYTE_LUC    {"luc_hex",     0x200000,0x180000,PROM_PART_LUC_HEX}
                                /* unused       0x380000 0x070000 */
#define PART_DESC_SIBYTE_SYS    {"sys",         0x3f0000,0x010000,PROM_PART_SYS}

#define PART_DESC_SIBYTE_LAYOUT \
    {\
        PART_DESC_SIBYTE_BOOT,\
        PART_DESC_SIBYTE_DIAG,\
        PART_DESC_SIBYTE_LMUX,\
        PART_DESC_SIBYTE_LUC,\
        PART_DESC_SIBYTE_SYS,\
        PART_DESC_NULL,\
    }

#define PART_DESC_SIBYTE_FP_LAYOUT \
    {\
        PART_DESC_SIBYTE_BOOT,\
        PART_DESC_SIBYTE_DIAG,\
        PART_DESC_SIBYTE_SYS,\
        PART_DESC_NULL,\
    }

/*
 * Partition Layout for the 2MB R7000 prom
 */
#define PART_DESC_R7K_BOOT      {"r7k_boot", PROM_BOOT_OFFS, PROM_BOOT_SIZE, PROM_PART_R7K}
#define PART_DESC_R7K_DIAG      {"r7k_diag", PROM_DIAG_OFFS, PROM_DIAG_SIZE, PROM_PART_R7K_DIAG}
                                /* unused    0x180000  0x070000 */
#define PART_DESC_R7K_SYS       {"sys",      0x1f0000, 0x010000, PROM_PART_SYS}
#define PART_DESC_R7K_LAYOUT \
    {\
        PART_DESC_R7K_BOOT,\
        PART_DESC_R7K_DIAG,\
        PART_DESC_R7K_SYS,\
        PART_DESC_NULL,\
    }

/*
 * Partition Layout for the 2MB R9000 prom
 */
#define PART_DESC_R9K_BOOT      {"r9k_boot", PROM_BOOT_OFFS, PROM_BOOT_SIZE, PROM_PART_R9K}
#define PART_DESC_R9K_DIAG      {"r9k_diag", PROM_DIAG_OFFS, PROM_DIAG_SIZE, PROM_PART_R9K_DIAG}
                                /* unused    0x180000  0x070000 */
#define PART_DESC_R9K_SYS       {"sys",      0x1f0000, 0x010000, PROM_PART_SYS}
#define PART_DESC_R9K_LAYOUT \
    {\
        PART_DESC_R9K_BOOT,\
        PART_DESC_R9K_DIAG,\
        PART_DESC_R9K_SYS,\
        PART_DESC_NULL,\
    }

/*
 * Partition Layout for the 0.5MB SSC-Mgmt (toshiba) prom, use RPROM defs
 */
#define PART_DESC_MGMT_BOOT     {"ssc_mgmt_boot", PROM_RPRM_OFFS, PROM_RPRM_SIZE, PROM_PART_MGMT}
#define PART_DESC_MGMT_SYS      {"sys",           PROM_RPRM_SIZE, 0x010000, PROM_PART_SYS}
#define PART_DESC_MGMT_LAYOUT \
    {\
        PART_DESC_MGMT_BOOT,\
        PART_DESC_MGMT_SYS,\
        PART_DESC_NULL,\
    }

/*
 * Partition Layout for the 0.5MB Sibyte RProm
 */
#define PART_DESC_SIBYTE_RPROM  {"sibyte_rprom",  PROM_RPRM_OFFS, PROM_RPRM_SIZE, PROM_PART_SIBYTE_RPROM}
#define PART_DESC_SIBYTE_RPROM_LAYOUT \
    {\
        PART_DESC_SIBYTE_RPROM,\
        PART_DESC_NULL,\
    }

/*
 * Partition Layout for the 0.5MB R7000 RProm
 */
#define PART_DESC_R7K_RPROM     {"r7k_rprom",     PROM_RPRM_OFFS, PROM_RPRM_SIZE, PROM_PART_R7K_RPROM}
#define PART_DESC_R7K_RPROM_LAYOUT \
    {\
        PART_DESC_R7K_RPROM,\
        PART_DESC_NULL,\
    }

/*
 * Partition Layout for the 0.5MB R9000 RProm
 */
#define PART_DESC_R9K_RPROM     {"r9k_rprom",     PROM_RPRM_OFFS, PROM_RPRM_SIZE, PROM_PART_R9K_RPROM}
#define PART_DESC_R9K_RPROM_LAYOUT \
    {\
        PART_DESC_R9K_RPROM,\
        PART_DESC_NULL,\
    }

/*
 * Partition descriptor and control block dump macros
 */

#define PART_DESC_PRINT(_dp) \
    {\
        printk("  Partition Descriptor\n"); \
        printk("\tpart_name           : %s\n", (_dp)->part_name);\
        printk("\tpart_offs           : 0x%x\n", (_dp)->part_offs);\
        printk("\tpart_size           : 0x%x\n", (_dp)->part_size);\
        printk("\tpart_type           : 0x%x\n", (_dp)->part_type);\
    }

#define PART_CB_PRINT(_cb, _bp) \
    {\
        printk("  Partition Control Block @%p, Partition Base %p\n",(_cb),(_bp));\
        /* check for uninitialized info_str in flash ie. all 1's */ \
        if (!((_cb)->info_str[0]&0x80)) printk("    %s\n",(_cb)->info_str);\
        printk("\tsignature           : 0x%x\n", (_cb)->signature);\
        printk("\tflags               : 0x%x\n", (_cb)->flags);\
        printk("\tprom_type           : 0x%x\n", (_cb)->prom_type);\
        printk("\tcompressed_start    : 0x%x\n", (_cb)->compressed_start);\
        printk("\tcompressed_size     : 0x%x\n", (_cb)->compressed_size);\
        printk("\tdecompressed_size   : 0x%x\n", (_cb)->decompressed_size);\
        printk("\tpartition_crc32     : 0x%x\n", (_cb)->partition_crc32);\
        printk("\theader_crc32        : 0x%x\n", (_cb)->header_crc32);\
    }

/******************************************************************************/

#define PROM_PART_SIGNATURE             ('P'<<24 | 'r'<<16 | 'c'<<8 | 'B')

#define PROM_PART_COMPRESSION_FLAG      0x0001  /* compressed part present */
#define PROM_PART_CRC32_FLAG            0x0002  /* crc32 over data present */
#define PROM_PART_BOOT_USE_DISABLE      0x0004  /* boot auto use disabled  */
#define PROM_PART_DISABLED              0x8000  /* disable always          */
#define PROM_PART_INFO_STR_LEN          (80)

typedef struct prom_part_cb {
	unsigned char       info_str[PROM_PART_INFO_STR_LEN];
    unsigned int	    reserved[5];        // 80
    unsigned int	    signature;          //100
    unsigned short      flags;              //104
    unsigned short      prom_type;          //106 see PROM_PART_*
    unsigned int        compressed_start;   //108
    unsigned int        compressed_size;    //112
    unsigned int	    decompressed_size;  //116
    unsigned int	    partition_crc32;    //120
	unsigned int	    header_crc32;       //124
} prom_part_cb_t, * prom_part_cb_p;

/*
 * the info str len should match the
 * ImageVerString[80] in the generated file Tools/Version/$PLATFORM/image-ver.c
 * and #define CM_VER_STRING_LEN 80 in sm-chassis/cm-api.h
 */
#define PROM_PART_CB_SIZE		sizeof(prom_part_cb_t)
#define PROM_PART_CB_CRC32_SIZE (unsigned int)(&((prom_part_cb_p)(0))->header_crc32)

/******************************************************************************/

/*
 * compression record for old prom format
 */
typedef struct crt {
	unsigned int	signature;
	unsigned int	compressed_start;
	unsigned int	compressed_size;
	unsigned int	prom_type;		/* 1 = r7k, 2 = sibyte */
	unsigned int	res[4];
} comp_rec_t;


#define COMPRESS_RECORD_SIZE        sizeof(comp_rec_t)

#endif /* LANGUAGE_C */

#define OLD_PROM_COMP_REC_OFFSET    (0x6ffe0)
#define COMPRESS_SIGNATURE          0xA1E50AE1

#ifdef LANGUAGE_C

/******************************************************************************/

typedef int (*prom_part_proc_user_f)(void *uhdl, char * buf, int dlen);

prom_part_desc_p prom_part_get_desc_from_type(unsigned short  prom_type);
prom_part_desc_p prom_part_get_desc_from_name(char * part_name);
prom_part_cb_p   prom_part_get_cb(char * buf_p, int len);
int              prom_part_verify(prom_part_cb_p cb_p, char * base_p, int verbose);
int              prom_part_upgrade(char * base_p, int len);
int              prom_part_update_flag(char * part_name,
                    unsigned short flags, int set);
int              prom_part_get_cb_flags(char * part_name, unsigned short * flags);
void             prom_part_print_info(void * buf_p, int len);
void             prom_part_init(void);

struct prom_part_desc *prom_get_layout(int unit);

#ifdef PROM
int              prom_part_proc_data(unsigned short prom_type,
                     prom_part_proc_user_f ucb_fct,
                     void * user_handle);

extern int from_get_prom_info(char *buf, int len, prom_part_cb_p *,
							  prom_part_desc_p *);


#endif  /* PROM */

#endif  /* LANGUAGE_C */

#endif	/* _PROM_PART_API_H */

