
/*-----------------------------------------------------------------
 *
 * Name: prom.c
 *
 * Copyright (C) 2007, Onstor, Inc.
 *
 * Description: device driver for prom upgrades.
 *
 *-----------------------------------------------------------------
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/proc_fs.h>
#include <linux/fcntl.h>
#include <asm/system.h>
#include <asm/uaccess.h>
#ifdef CONFIG_ONSTOR_BOBCAT
# include <linux/mv643xx.h>
# include <asm/marvell.h>
# include <asm/mach-bobcat/bobcat.h>
#endif
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <linux/poll.h>
#include "from-api.h"
#include <linux/crc32.h>
#include <linux/delay.h>


#define EEE_DEBUG
#undef EEE_DEBUG

#ifdef EEE_DEBUG
#define debug(x) printk x
#else
#define debug(x)
#endif

MODULE_LICENSE("GPL");

#define PROM_NUM_DEVS 5

#define FROM_CBX_PRIMARY   0
#define FROM_CBX_ALTERNATE 1


#include "from-api.h"

#include "prom-part-api.h"

#undef DEBUG_PRINT
#ifdef DEBUG_PRINT
#define DPRINT(_x)  printf _x
#else
#define DPRINT(_x)
#endif

uint8_t	* from_base;
uint8_t	* from_ptr;
uint32_t	  from_len;
#ifdef CONFIG_ONSTOR_BOBCAT
prom_part_desc_t prom_part_rprom_layout[]   = PART_DESC_R9K_RPROM_LAYOUT;
prom_part_desc_t prom_part_rt_layout[]      = PART_DESC_R9K_LAYOUT;
#elif defined(CONFIG_ONSTOR_COUGAR)
prom_part_desc_t prom_part_rprom_layout[]   = PART_DESC_SIBYTE_RPROM_LAYOUT;
prom_part_desc_t prom_part_rt_layout[]      = PART_DESC_SIBYTE_LAYOUT;
#else
# error Platform configuration error - which PROM is it?
#endif

prom_part_desc_p prom_part_layout;

struct prom_part_desc *prom_layouts[2];

void
prom_part_init(void)
{
    if (primary_is_rprom) {
		prom_layouts[0] = prom_part_rprom_layout;
		prom_layouts[1] = prom_part_rt_layout;
        prom_part_layout = prom_part_rprom_layout;
    }
	else {
		prom_layouts[0] = prom_part_rt_layout;
		prom_layouts[1] = prom_part_rprom_layout;
		prom_part_layout = prom_part_rt_layout;
	}
}

/*
 * print old prom compress record
 */
void
from_old_prom_print(
        uint8_t *    buf_p)
{
    comp_rec_t * cr_p = (comp_rec_t *) &buf_p[OLD_PROM_COMP_REC_OFFSET];

    printk("  !!! OLD PROM FORMAT compress rec @%p, Base %p\n",cr_p, buf_p);
    /* check for uninitialized info_str in flash ie. all 1's */
    printk("\tsignature           : 0x%x\n", cr_p->signature);
    printk("\tcompressed_start    : 0x%x\n", cr_p->compressed_start);
    printk("\tcompressed_size     : 0x%x\n", cr_p->compressed_size);
    printk("\tprom_type           : 0x%x\n", cr_p->prom_type);
}

/*
 * get the prom type and valid length for the partition(s)
 */
int
from_old_prom_type(
        uint8_t *    buf_p,
        int32_t       prom_len)
{
    /* check for the classic compress record
     * bootroms, for 512KB Am29LV040B and prom_rec at 0x6ffec
     */
    if ((prom_len == AMD_29LV040_SIZE) || (prom_len == ST_M29W040_SIZE)) {
        comp_rec_t * cr_p = (comp_rec_t *) &buf_p[OLD_PROM_COMP_REC_OFFSET];

        if (cr_p->signature == COMPRESS_SIGNATURE) {
            if (cr_p->prom_type == PROM_PART_R7K ||
                cr_p->prom_type == PROM_PART_R9K ||
                cr_p->prom_type == PROM_PART_SIBYTE ||
                cr_p->prom_type == PROM_PART_MGMT) {

                return cr_p->prom_type;
            }
        }
    }
    return PROM_PART_UNKNOWN;
}


/*-----------------------------------------------------------------
 * Name :	from_check_prom_type()
 *
 * Description:
 *
 * Created by:	John Hahn
 *
 * Date Created:	08/09/02
 *
 *-----------------------------------------------------------------
 */
int
from_check_old_prom_type(uint8_t *buf, int len)
{
	return from_old_prom_type(buf, len) == PROM_PART_R7K;
}

static int
prom_part_check_full_size(int len)
{
    return (len == AMD_29F040_SIZE ||
            len == AMD_29LV040_SIZE ||
            len == AMD_29LV017_SIZE ||
            len == AMD_29LV033_SIZE ||
	    len == ST_M29W040_SIZE);
}

/*
 * check if the partition prom_type is RPROM
 * or if the len is rprom full size
 */
static int
prom_part_check_rprom(prom_part_cb_p cb_p, int len)
{
    switch (cb_p->prom_type) {
        case PROM_PART_R7K_RPROM:
            return 1;
            break;
        default:
            if ((len == AMD_29LV040_SIZE) || (len == ST_M29W040_SIZE))
                return 1;
            break;
    }
    return 0;
}

/*
 * get the prom type and valid length for the partition(s)
 */
int
from_get_prom_info(
        char *              buf_p,
        int                 data_len,
        prom_part_cb_p *    cb_pp,
        prom_part_desc_p *  desc_pp)
{
    char *              base_p;
    prom_part_cb_p      cb_p;
    prom_part_desc_p    desc_p;
    int                 full_size;

    desc_p = NULL;
    base_p = buf_p;
    cb_p = prom_part_get_cb(buf_p, data_len);

    full_size = prom_part_check_full_size(data_len);

    if (!full_size) {
        if (prom_part_verify(cb_p, base_p, 1 /*XXX verbose*/) != 0)
            return -1;
    }
    else {
        /* full prom in buf, go thru valid partitions in prom buf */
        if (!prom_part_check_rprom(cb_p, data_len)) {
#if defined(SIBYTE)
            if (isFP())
                desc_p = prom_part_rt_fp_layout;
            else
#endif
                desc_p = prom_part_rt_layout;
        }
        else
            desc_p = prom_part_rprom_layout;

        while (desc_p->part_name != NULL) {
            if (desc_p->part_type == PROM_PART_SYS)
                goto get_prom_info_exit;

            base_p = buf_p + desc_p->part_offs;
            cb_p = prom_part_get_cb(base_p, desc_p->part_size);
            if (prom_part_verify(cb_p, base_p, 1 /*XXX verbose*/) != 0)
                return -1;

            if (prom_part_check_rprom(cb_p, 0))
                goto get_prom_info_exit;
            desc_p++;
        }
    }

    desc_p = prom_part_get_desc_from_type(cb_p->prom_type);
    if (desc_p == NULL)
        return -1;

    switch (cb_p->prom_type) {
        case PROM_PART_R7K:
        case PROM_PART_R7K_DIAG:
        case PROM_PART_R7K_RPROM:
            if ((data_len + PROM_PART_CB_SIZE) > desc_p->part_size)
                return -1;
            break;

        default:
            return -1;
            break;
    }
get_prom_info_exit:
    *cb_pp   = cb_p;
    *desc_pp = desc_p;
    return 0;
}

/*-----------------------------------------------------------------
 * Name:            prom_part_upgrade
 *
 * Description:     install the prom binary image in buf_p
 *
 * Created by:	    John Hahn
 *
 * Date Created:    7/25/2002
 *
 *-----------------------------------------------------------------
 */
int
prom_part_upgrade(
        char *      buf_p,
        int         data_len)
{
    int                 rv;
    int                 from_sav;
    prom_part_desc_p    desc_p;
    prom_part_cb_p      cb_p;

    DPRINT(("prom_part_upgrade: buf_p %p data_len 0x%x\n", buf_p, data_len));

    if (data_len < PROM_PART_CB_SIZE)
        return -1;

    /*
     * check for the old type of bootrom.bin and take care of it
     */
    if (from_check_old_prom_type(buf_p, data_len)) {

        from_sav = from_select_recov();
        rv = from_programbytes(0, (void *)buf_p, data_len);
        from_set_current(from_sav);
        return rv;
    }

    rv = from_get_prom_info(buf_p, data_len, &cb_p, &desc_p);
    if (rv != 0)
        return rv;

#ifdef DEBUG_PRINT
    DPRINT(("partition info: (rprom=%d)\n",prom_part_check_rprom(cb_p, data_len)));
    PART_DESC_PRINT(desc_p);
    PART_CB_PRINT(cb_p, buf_p);
#endif

    if (prom_part_check_rprom(cb_p, data_len)) {
        // for RPRM the last sector may be shared w/ crashdump and fake seepenv
        if ((cb_p->compressed_start + cb_p->compressed_size) > PROM_RPRM_CODE_SIZE) {
            printk("Upgrade Error RPROM code size 0x%x + 0x%x > 0x%x\n",
                cb_p->compressed_start,
                cb_p->compressed_size,
                PROM_RPRM_CODE_SIZE);
            return -1;
        }


        from_sav = from_select_recov();
    }
    else
        from_sav = from_select_std();

    if (from_sav == FROM_SEL_INVALID)
        return -1;

    /*
     * full prom upgrade
     */
    if (prom_part_check_full_size(data_len)){
        rv = from_programbytes(0, (void *)buf_p, data_len);
        goto prom_part_upgrade_exit;
    }

    rv = from_programbytes(
            desc_p->part_offs,
            (void *)buf_p,
            data_len - PROM_PART_CB_SIZE);

    if (rv != 0)
        goto prom_part_upgrade_exit;

    rv = from_programbytes(
            desc_p->part_offs + desc_p->part_size - PROM_PART_CB_SIZE,
            (void *)cb_p,
            PROM_PART_CB_SIZE);

    /*
     * find the prom_part_cb at the end of the buffer
     * verify the crc32 over header and data
     * if full prom check the data_len, cb's  and program it
     * if single prom image, check the cb program the data then the cb
     * at the end of the partition for the image
     *
     */
prom_part_upgrade_exit:
    from_set_current(from_sav);
    return rv;
}

/*
 * return prom partition descriptor of partition name
 * NOTE: the desc may not be the one for the from_current
 * and you need know wether to switch froms !!!
 */
prom_part_desc_p
prom_part_get_desc_from_name(
        char * name)
{
    prom_part_desc_p desc_p = prom_part_layout;

    while (desc_p->part_name != NULL) {
        if (strcmp(name, desc_p->part_name) == 0)
            return desc_p;
        desc_p++;
    }

    if (prom_part_layout == prom_part_rprom_layout) {
        desc_p = prom_part_rt_layout;
    }
    else
        desc_p = prom_part_rprom_layout;

    while (desc_p->part_name != NULL) {
        if (strcmp(name, desc_p->part_name) == 0)
            return desc_p;
        desc_p++;
    }

    return NULL;
}

/*
 * return prom partition descriptor of partition prom_type
 */
prom_part_desc_p
prom_part_get_desc_from_type(
        unsigned short  prom_type)
{
    prom_part_desc_p desc_p = prom_part_layout;

    if (prom_type == PROM_PART_R7K_RPROM ||
        prom_type == PROM_PART_R9K_RPROM ||
        prom_type == PROM_PART_SIBYTE_RPROM)
        desc_p = prom_part_rprom_layout;
    else {
        desc_p = prom_part_rt_layout;
    }

    while (desc_p->part_name != NULL) {
        if (prom_type == desc_p->part_type)
            return desc_p;
        desc_p++;
    }
    return NULL;
}


struct prom_part_desc *
prom_get_layout(int unit)
{
	return prom_layouts[unit];
}


/*
 * return partition control block at the end of the given buffer of len
 */
prom_part_cb_p
prom_part_get_cb(
        char *  buf_p,
        int     len)
{
    return (prom_part_cb_p) (buf_p + len - PROM_PART_CB_SIZE);
}

/*
 * following are verified on the partition:
 * in the control block: signature, header crc32, partition data crc32
 */
int
prom_part_verify(
        prom_part_cb_p  cb_p,
        char *          base_p,
        int             verbose)
{
    uint32_t crc32;

    if (cb_p->signature != PROM_PART_SIGNATURE) {
        if (verbose)
            printk("Error partition cb@%p bad signature 0x%x, expect 0x%x\n",
                    cb_p, cb_p->signature, PROM_PART_SIGNATURE);
        return -1;
    }

    crc32 = crc32(0, cb_p, PROM_PART_CB_CRC32_SIZE);
    if (crc32 != cb_p->header_crc32) {
        if (verbose)
            printk("Error partition cb@%p header corrupt crc32 0x%x hdr 0x%x\n",
                    cb_p, crc32, cb_p->header_crc32);
        return -1;
    }
    switch (cb_p->prom_type) {
        case PROM_PART_R7K:
        case PROM_PART_R7K_DIAG:
        case PROM_PART_R7K_RPROM:
            // partition is allowed for this platform, proceed ...
            break;

        case PROM_PART_SYS:
            return 0;
            break;

        default:
            // if we got here then the partition doesnt belong
            if (verbose)
                printk("Error prom partition type %d doesnt belong here\n", cb_p->prom_type);
            return -1;
    }
    if (cb_p->flags | PROM_PART_CRC32_FLAG) {
        if (cb_p->compressed_start + cb_p->compressed_size) {
            crc32 = crc32(0, base_p,
                             cb_p->compressed_start + cb_p->compressed_size);
            if (crc32 == cb_p->partition_crc32) {
                return 0;
            }
            else {
                if (verbose)
                    printk("Error, partition data corrupt crc32 0x%x data 0x%x\n", crc32, cb_p->partition_crc32);
            }
        }
        else {
            if (verbose)
                printk("Error, no partition data\n");
        }
    }
    return -1;
}

static void
prom_part_print(
    prom_part_desc_p    desc_p,
    prom_part_cb_p      cb_p,
    char *              base_p)
{

    if (desc_p != NULL)
        PART_DESC_PRINT(desc_p);

    PART_CB_PRINT(cb_p, base_p);
}


int
prom_part_update_flag(
    char *          part_name,
    unsigned short  flags,
    int             set)
{
    int rv;
    int from_sav;
    char *              base_p;
    prom_part_desc_p    desc_p;
    prom_part_cb_p      cb_p;
    prom_part_cb_t      control_block;

    desc_p = prom_part_get_desc_from_name(part_name);
    if (desc_p == NULL ||
        (desc_p->part_type != PROM_PART_LUC_HEX &&
         desc_p->part_type != PROM_PART_LMUX_HEX))
    {
        printk("prom_part_update_flag: invalid partition %s\n", part_name);
        return -1;
    }

    from_sav = from_select_std();
    if (from_sav == FROM_SEL_INVALID)
        return -1;

    base_p = FROM_ADDR(char *, desc_p->part_offs);
    cb_p = prom_part_get_cb(base_p, desc_p->part_size);

    control_block = *cb_p;

    if (set) {
        control_block.flags |= flags;
    }
    else {  /* set the bits */
        control_block.flags &= ~flags;
    }
    if (control_block.flags == cb_p->flags) {
        /* nothing to change */
        rv = 0;
        goto update_flag_exit;
    }

    printk("prom_part_update_flag: old=%04x new=%04x\n",
            cb_p->flags, control_block.flags);

    /* recalculate the header crc32 */
    control_block.header_crc32 = crc32(
            0, &control_block, PROM_PART_CB_CRC32_SIZE);
    rv = from_programbytes(
            desc_p->part_offs + desc_p->part_size - PROM_PART_CB_SIZE,
            (void *)&control_block,
            PROM_PART_CB_SIZE);

update_flag_exit:
    from_set_current(from_sav);
    return rv;
}

/*
 * return prom partition control block flags
 */
int
prom_part_get_cb_flags(
    char *          part_name,
    unsigned short  * flags)
{
    int from_sav;
    char *              base_p;
    prom_part_desc_p    desc_p;
    prom_part_cb_p      cb_p;

    desc_p = prom_part_get_desc_from_name(part_name);
    if (desc_p == NULL ||
        (desc_p->part_type != PROM_PART_LUC_HEX &&
         desc_p->part_type != PROM_PART_LMUX_HEX))
    {
        printk("prom_part_update_flag: invalid partition %s\n", part_name);
        return -1;
    }

    from_sav = from_select_std();
    if (from_sav == FROM_SEL_INVALID)
        return -1;

    base_p = FROM_ADDR(char *, desc_p->part_offs);
    cb_p = prom_part_get_cb(base_p, desc_p->part_size);
    *flags = cb_p->flags;

    from_set_current(from_sav);
    return 0;
}

void
prom_part_print_info(
        void *  addr,
        int     len)
{
    char *              base_p;
    char *              part_p;
    prom_part_desc_p    desc_p;
    prom_part_cb_p      cb_p;
    int                 full_size;

    full_size = prom_part_check_full_size(len);

    if (!full_size && addr) {
        base_p = (char *) addr;

        if (from_check_old_prom_type(base_p, len)) {
            from_old_prom_print(base_p);
        }
        else {
            cb_p   = prom_part_get_cb(base_p, len);
            desc_p = prom_part_get_desc_from_type(cb_p->prom_type);
            prom_part_verify(cb_p, base_p, 1 /*verbose*/);
            prom_part_print(desc_p, cb_p, base_p);
        }
    }
    else {
        if (addr) {
            base_p = (char *)addr;
        }
        else {
            printk("\nPrimary flash memory\n");
            base_p = FROM_ADDR(char *, 0);
        }

        if (from_check_old_prom_type(base_p, AMD_29LV040_SIZE)) {
            from_old_prom_print(base_p);
        }
	else if (from_check_old_prom_type(base_p, ST_M29W040_SIZE)) {
            from_old_prom_print(base_p);
        }
        else {

            if (addr && len) {
                if (!prom_part_check_rprom((void *)base_p, len)) {
#if defined(SIBYTE)
                    if (isFP())
                        desc_p = prom_part_rt_fp_layout;
                    else
#endif
                        desc_p = prom_part_rt_layout;
                }
                else
                    desc_p = prom_part_rprom_layout;
            }
            else
                desc_p = prom_part_layout;

            while (desc_p->part_name != NULL) {
                part_p = base_p + desc_p->part_offs;

                cb_p = prom_part_get_cb(part_p, desc_p->part_size);
                prom_part_verify(cb_p, part_p, 1 /*verbose*/);
                prom_part_print(desc_p, cb_p, part_p);
                printk("\n");
                desc_p++;
            }
        }

        // user  requested the default print out all flash partition info
        // check if we should print out the alternate flash
        if (!addr && (from_probed_proms > 1)) {
            int from_sav = from_swap_base();

            printk("\nAlternate flash memory\n");
            if (from_check_old_prom_type(FROM_ADDR(char *,0), FROM_SIZE)) {
                from_old_prom_print(FROM_ADDR(char *,0));
            }
            else {
                // set to the other layout
                if (prom_part_layout == prom_part_rprom_layout) {
#if defined(SIBYTE)
                    if (isFP())
                        desc_p = prom_part_rt_fp_layout;
                    else
#endif
                        desc_p = prom_part_rt_layout;
                }
                else
                    desc_p = prom_part_rprom_layout;

                while (desc_p->part_name != NULL) {
                    part_p = FROM_ADDR(char *, desc_p->part_offs);
                    cb_p = prom_part_get_cb(part_p, desc_p->part_size);
                    prom_part_verify(cb_p, part_p, 1 /*verbose*/);
                    prom_part_print(desc_p, cb_p, part_p);
                    printk("\n");
                    desc_p++;
                }
            }
            from_set_current(from_sav);
        }
    }
}

int from_current;
int primary_is_rprom;
int from_probed_proms;

#define HW_MAX_PROM_NUM     2

/*
 * flash rom device driver control block
 */
typedef struct {
    char *              name;               // description of chip
    TYPE_LEN            mfg_id;             // chip manufacturer Id
    TYPE_LEN            dev_id;             // chip device Id
    volatile TYPE_LEN * base_p;             // sw accessible adress to offset 0
    unsigned int        sector_size;        // sector size in bytes
    unsigned int        num_sectors;        // number of sectors
    unsigned int        size;               // chip size in bytes
} FROM_CB_T, * FROM_CB_P;

FROM_CB_T from_cb[HW_MAX_PROM_NUM];


#if 0
#define SAFEROUTINE(name,protoargs,callargs)					\
int32_t name protoargs											\
{																\
	int32_t	(*fn) protoargs;									\
	int32_t	ret;												\
	int32_t	codesize = (char8 *)name - (char8 *)_##name;		\
																\
	fn = eee_ramAlloc (codesize);								\
	if (!fn)													\
		return FROM_ERR_NOMEM;									\
	memcpy (fn, _##name, codesize);								\
	mips_clean_cache (fn, codesize);							\
																\
	ret = (*fn) callargs;										\
	eee_ramDealloc(fn);											\
	return ret;													\
}
#else
#define SAFEROUTINE(name,protoargs,callargs)					\
extern int32_t name protoargs;									\
int32_t name protoargs											\
{																\
	int32_t	ret;												\
	ret = _##name callargs;										\
	return ret;													\
}
#endif

#define PAUSE	udelay(1)

static void
fromunlock (volatile uint8_t *sec_base)
{
	*(TYPE_LEN *)(sec_base + CMD_UNLOCK_ADDR1) = CMD_DATA1;
	PAUSE;
	*(TYPE_LEN *)(sec_base + CMD_UNLOCK_ADDR2) = CMD_DATA2;
	PAUSE;
}

static void
fromcmd (volatile uint8_t *sec_base, TYPE_LEN cmd)
{
	*(TYPE_LEN *)(sec_base + CMD_UNLOCK_ADDR1) = CMD_DATA1;
	PAUSE;
	*(TYPE_LEN *)(sec_base + CMD_UNLOCK_ADDR2) = CMD_DATA2;
	PAUSE;
	*(TYPE_LEN *)(sec_base + CMD_ADDR3) = cmd;
	PAUSE;
}

static int32_t
frompoll (volatile TYPE_LEN *fp, TYPE_LEN expect)
{
	TYPE_LEN	poll;

	while (1) {
		poll = *fp;
		if (((poll ^ expect) & DQPOLL) == 0)
			break;
		if ((poll & DQTIMEEXCEEDED) == DQTIMEEXCEEDED) {
			poll = *fp;
			if (((poll ^ expect) & DQPOLL) == 0)
				break;
			*fp = CMD_DATA_RESET;
			(void) *fp;
			return FROM_ERR_FATAL;
		}
	}
	return 0;
}

/*-----------------------------------------------------------------
 * Name:	from_readId()
 *
 * Description: read manufacture and device id from chip
 *
 * Created by:	Rick Lund
 *
 * Date Created:	10/8/01
 *
 *-----------------------------------------------------------------
 */
static int32_t
from_readId(TYPE_LEN * man, TYPE_LEN * dev)
{
	volatile TYPE_LEN *sec = FROM_ADDR(volatile TYPE_LEN *, 0);
	int32_t			prot;

	fromcmd ((uint8_t *)sec, CMD_DATA_AUTOSELECT);

	/* get manufacture and device id */
	if (man)
		*man = sec[0];
	if (dev)
		*dev = sec[1];

	/* get sector protect status */
	prot = *(volatile TYPE_LEN *)((TYPE_LEN *)sec + 2) & 1;

	sec[0] = CMD_DATA_RESET;

	(void) sec[0];

	return prot;
}

/*-----------------------------------------------------------------
 * Name:	from_programbytes()
 *
 * Description:
 *
 * Created by:	Rick Lund
 *
 * Date Created:	10/8/01
 *
 *-----------------------------------------------------------------
 */
/*
 *	We build a vector of these, each representing the data to be
 *	programmed into one sector.
 */
struct flashvec {
	uint32_t		sec;			/* sector num */
	uint32_t		offs;			/* sector offset */
	void		* mem;			/* src pointer */
	uint32_t		num_bytes;
	uint32_t		erase;			/* unused */
	int32_t		scratch;		/* unused */
};

uint8_t	secdata[FROM_SECTORSIZE];

static int32_t
from_reallyprogrambytes (struct flashvec *v, int32_t nv)
{
	volatile TYPE_LEN * fp;
	volatile TYPE_LEN * sp, * mp;
	int32_t			status = 0;
	uint32_t			j, o;
	int32_t			stat;

	while (nv-- != 0) {
		sp = FROM_ADDR(volatile TYPE_LEN *, (v->sec * FROM_SECTORSIZE));

		/*
		 * do we need to do a read and merge?
		 */
		if ((v->offs != 0) || (v->num_bytes != FROM_SECTORSIZE)) {
			volatile uint8_t	* dst, * src;
			uint32_t			i;

			/*
			 * read in current data
			 */
			dst = secdata;
			src = (uint8_t *)sp;
			for (i=0; i<FROM_SECTORSIZE; i++)
				*dst++ = *src++;
			/*
			 * overwrite with new data
			 */
			dst = &(secdata[v->offs]);
			src = v->mem;
			for (i=0; i<v->num_bytes; i++)
				*dst++ = *src++;
			v->mem = secdata;
			v->num_bytes = FROM_SECTORSIZE;
			v->offs = 0;
		}

		/*
		 * erase sector
		 */
        DPRINT(("program/erase sector at %p ...", sp));
		fromcmd ((uint8_t *)sp, CMD_DATA_ERASE);
		fromunlock ((uint8_t *)sp);
		*sp = CMD_DATA_ERASESECT;
		stat = frompoll (sp, ~0);
		if (stat != 0)
			status = stat;

		/*
		 * program sector
		 */
		mp = (TYPE_LEN *)v->mem;
#ifndef SSC_MGMT
		for (j = v->num_bytes, o = v->offs; j != 0; j--, o++, mp++) {
#else
		for (j = v->num_bytes, o = v->offs; j != 0; j-=2, o++, mp++) {
#endif
			/* otherwise program flash with sequential byte stream */
			fp = (volatile TYPE_LEN *)(sp + o);

			fromcmd ((uint8_t *)sp, CMD_DATA_PROGRAM);
			*fp = *mp;
			stat = frompoll (fp, *mp);
			if (stat != 0)
				status = stat;
		}

		/* advance to next sector */
		v++;
	}

	return status;
}


/* Preallocate array for from_programbytes so we can do crashdump even if out
 * of memory. 64 is maximum number of sectors in any supported flash.
 */
static struct flashvec vecs[64];

int32_t
from_programbytes(int32_t offset, TYPE_LEN * src_buf, uint32_t len)
{
	struct flashvec	*   v;
	uint32_t			    o, nb;
	int32_t			    nv = 0;
	int32_t			    ret;
	TYPE_LEN *          sp = src_buf;


    v = vecs;

	/* check limits */
	if (offset+len > FROM_SIZE) {
		ret = FROM_ERR_FAIL;
        goto exit;
	}

	if (!src_buf || len == 0) {
		ret = 0;
        goto exit;
    }

	/* construct the flash programming vector */
	for (o = offset, nb = len; nb != 0;) {
		uint32_t			sec_offset, sec_num;
		uint32_t			n;

		/* find starting sector */
		sec_offset = FROM_SECTOR_OFFSET(o);
		sec_num = FROM_SECTOR_NUM(o);

		/* how many bytes do we want to program in this sector */
		n = FROM_SECTORSIZE - sec_offset;
		if (n > nb)
			n = nb;

		/* add to programming vector */
		v->sec = sec_num;
		v->offs = sec_offset;
		v->num_bytes = n;
		v->mem = sp;
		v->scratch = 0;
		v->erase = 0;

        DPRINT(("program vector, sec = %d, offs = %d, mem = %p, len = %d\n",
                    v->sec, v->offs, v->mem, v->num_bytes));
		/* advance to next sector */
		v++;
		nv++;
		sp = (TYPE_LEN *)((uint8_t *)sp + n);
		o += n;
		nb -= n;
	}

#if 0
	/* wipe this area of the flash out of the data cache */
	mips_clean_dcache (PA_TO_KVA0(FROM_BASE + offset), len);
#endif

	/* now really do the programming operation */
	ret = from_reallyprogrambytes (vecs, nv);

exit:

	return ret;
}

/*-----------------------------------------------------------------
 * Name:	from_readbytes()
 *
 * Description:	Read bytes from Flash ROM to user supplied buffer.
 *
 * Created by:	Rick Lund
 *
 * Date Created:	10/8/01
 *
 *-----------------------------------------------------------------
 */
int32_t
from_readbytes(int32_t offset, uint8_t * dest_buf, uint32_t len)
{
	volatile uint8_t	* sp;
	int32_t			i;

	/* check limits */
	if (offset+len > FROM_SIZE)
		return FROM_ERR_FAIL;

	sp = FROM_ADDR(volatile uint8_t *, offset);

	/* copy bytes to dest_buf array */
	for (i=0; i<len; i++)
		dest_buf[i] = sp[i];

	return 0;
}


/*-----------------------------------------------------------------
 * Name :	from_device_present()
 *
 * Description: Return true if the flash device is present
 *
 * Created by:	Maxim Kozlovsky
 *
 * Date Created:	11/05/02
 *
 *-----------------------------------------------------------------
 */
int
from_device_present(int id)
{
	return from_cb[id].dev_id != 0xff;
}


/*
 * dump the flash device driver info for diag purposes
 */
void
from_print_info(void)
{
    int i;
    FROM_CB_P fcb_p;

    printk("from_current=%d\n", from_current);
    for (i=0, fcb_p=from_cb; i < HW_MAX_PROM_NUM; i++, fcb_p++) {
        printk("Device[%d]: %s\n", i, fcb_p->name);
        printk("\tIds: Manufacturer=0x%02x Device=0x%02x\n",
                fcb_p->mfg_id, fcb_p->dev_id);
        printk("\tbase_addr:\t%p\n", fcb_p->base_p);
        printk("\tsectors: num=%d * size=%d (0x%x)\n",
                fcb_p->num_sectors, fcb_p->sector_size, fcb_p->sector_size);
        printk("\tsize=%d (0x%x)\n", fcb_p->size, fcb_p->size);
    }
}

/*
 * number of flash chips on the system
 */
int
from_cfg_proms(void)
{
    return FROM_RECOV_PROM_CFG;
}

/*
 * initialize the flash driver
 */
void
from_init(void)
{
    int i;
    FROM_CB_P fcb_p;

    from_probed_proms = 0;
    for (i=0, fcb_p=from_cb; i < from_cfg_proms(); i++, fcb_p++) {
        if (i == FROM_CBX_PRIMARY) {
            fcb_p->base_p = (volatile TYPE_LEN *)0xbfc00000;
#ifdef CONFIG_ONSTOR_COUGAR
            fcb_p->base_p = (volatile TYPE_LEN *)0xffffffffbfc00000;
#endif
        } else {
            fcb_p->base_p = (volatile TYPE_LEN *)0xbf800000;
#ifdef CONFIG_ONSTOR_COUGAR
            fcb_p->base_p = (volatile TYPE_LEN *)0xffffffffbf800000;
#endif
        }

        from_current = i;       // need to set for from_readId call
        from_readId(&fcb_p->mfg_id, &fcb_p->dev_id);

        switch (fcb_p->dev_id) {
            case DEV_29F040:

                printk("%s: DEV_29F040 ", __FUNCTION__);

                fcb_p->num_sectors = AMD_29F040_MAX_SECTORS;
                fcb_p->name = AMD_29F040_STR;
                from_probed_proms++;
                break;

            case DEV_29LV040:

                printk("%s: DEV_29LV040 ", __FUNCTION__);

                fcb_p->num_sectors = AMD_29LV040_MAX_SECTORS;
                fcb_p->name = AMD_29LV040_STR;
                from_probed_proms++;
                break;

            case DEV_29LV017:
                printk("%s: DEV_29LV017 ", __FUNCTION__);
                fcb_p->num_sectors = AMD_29LV017_MAX_SECTORS;
                fcb_p->name = AMD_29LV017_STR;
                from_probed_proms++;
                break;

            case DEV_29LV033:
                printk("%s: DEV_29LV033 ", __FUNCTION__);
                fcb_p->num_sectors = AMD_29LV033_MAX_SECTORS;
                fcb_p->name = AMD_29LV033_STR;
                from_probed_proms++;
                break;

            case DEV_M29W040:
                printk("%s: DEV_M29W040 ", __FUNCTION__);
                fcb_p->num_sectors = ST_M29W040_MAX_SECTORS;
                fcb_p->name = ST_M29W040_STR;
                from_probed_proms++;
                break;

            default:
                printk("%s: not recognized 0x%02x\n", __FUNCTION__,
                    fcb_p->dev_id);

				fcb_p->dev_id = 0xff;
                break;
        }
        fcb_p->sector_size = FROM_SECTORSIZE;
        fcb_p->size = fcb_p->sector_size * fcb_p->num_sectors;

        printk("sector size = %d size = %d\n", fcb_p->sector_size,
               fcb_p->num_sectors * fcb_p->sector_size);
    }
    from_set_primary();
    if ((from_cb[FROM_CBX_PRIMARY].dev_id == DEV_29LV040)||
        (from_cb[FROM_CBX_PRIMARY].dev_id == DEV_M29W040))
        primary_is_rprom = 1;
    else {
        primary_is_rprom = 0;
    }
    prom_part_init();
}


unsigned long
from_get_base(void)
{
    return (unsigned long)from_cb[from_current].base_p;
}

uint32_t
from_get_size(void)
{
    return from_cb[from_current].size;
}

void
from_set_primary(void)
{
    from_current = FROM_CBX_PRIMARY;
}

void
from_set_alternate(void)
{
    if (from_probed_proms > 1)
        from_current = FROM_CBX_ALTERNATE;
}

void
from_set_current(int select)
{
    if (from_probed_proms > 1)
        from_current = select;
}

int
from_swap_base(void)
{
    int rv = from_current;

    if (from_probed_proms > 1) {
        if (from_current == FROM_CBX_PRIMARY)
            from_set_alternate();
        else
            from_set_primary();
    }
    return rv;
}

int
from_select_recov(void)
{
    int rv = from_current;

    if (from_probed_proms > 1) {
        if (primary_is_rprom) {
            from_set_primary();
        }
        else {
            from_set_alternate();
        }
    }
    return rv;
}

int
from_select_std(void)
{
    int rv = from_current;

    if (from_probed_proms > 1) {
        if (primary_is_rprom) {
            from_set_alternate();
        }
        else {
            from_set_primary();
        }
    }
    else {
        rv = FROM_SEL_INVALID;
    }
    return rv;
}


static int
unit_prom(int unit)
{
	if (0 <= unit && unit <= 1) {
		if (unit == 1 && primary_is_rprom)
			return -1;
		return FROM_CBX_PRIMARY;
	}
	else if (2 <= unit && unit <= 3) {
		if (unit == 3 && !primary_is_rprom)
			return -1;
		return FROM_CBX_ALTERNATE;
	}
	else if (unit == 4) {
		if (from_device_present(FROM_CBX_ALTERNATE)) {
			return primary_is_rprom ? FROM_CBX_ALTERNATE : FROM_CBX_PRIMARY;
		}
		else
			return -1;
	}
	else {
		return -1;
	}
}


static int
unit_part_no(int unit)
{
	if (0 <= unit && unit <= 1)
		return unit;
	else if (2 <= unit && unit <= 3)
		return unit - 2;
	else
		return 0;
}


static struct prom_part_desc *
unit_part_desc(int unit)
{
	return &prom_get_layout(unit_prom(unit))[unit_part_no(unit)];
}


static int
unit_offset(int unit)
{
	if (unit == 4) {
		if (primary_is_rprom) {
			return prom_get_layout(FROM_CBX_ALTERNATE)[0].part_offs;
		}
		else {
			return prom_get_layout(FROM_CBX_PRIMARY)[0].part_offs;
		}
	}
	else
		return unit_part_desc(unit)->part_offs;
}


static int
unit_size(int unit)
{
	if (unit == 4) {
		return from_get_size();
	}
	else
		return unit_part_desc(unit)->part_size;
}


/*
 * Provides exclusive access to prom device.
 */
bool prom_opened;

/*
 * Our device major number
 */
int prom_major;


int
prom_open(struct inode *inode, struct file *filp)
{
	int prom = unit_prom(MINOR(inode->i_rdev));


    /*
     * Only exclusive open is allowed.
     */

    if (prom_opened) {
        return -EBUSY;
    }

    if (prom == -1) {
        return -ENXIO;
    }

    prom_opened = true;

    from_set_current(prom);

	return 0;
}


int
prom_release(struct inode *inode, struct file *filp)
{
    prom_opened = false;

    return 0;
}


ssize_t
prom_read(struct file *filp, char *buf, size_t count, loff_t *f_pos)
{
    int rc;

    int unit = MINOR(filp->f_dentry->d_inode->i_rdev);
    int prom = unit_prom(unit);

    /*
     * Allow only sector aligned reads of sector size for
     * simplicity.
     */

    if (count != FROM_SECTORSIZE) {
        return -EINVAL;
    }

    if (*f_pos % FROM_SECTORSIZE) {
        return -EINVAL;
    }

    /*
     * Check if reading past the end of prom.
     */
    if (*f_pos > unit_size(unit)) {
        return -EINVAL;
    }

    /*
     * If at the last byte of prom, return end of file.
     */
    if (*f_pos == unit_size(unit)) {
        return 0;
    }

    if (prom != -1) {
        uint8_t *mem = kmalloc(FROM_SECTORSIZE, 0);
        loff_t offset = unit_offset(unit) + *f_pos;

        debug(("%s: read at 0x%llx(0x%llx)\n",
               __FUNCTION__, offset, *f_pos));

        rc = from_readbytes(offset, mem, FROM_SECTORSIZE);

        if (rc != 0) {
            printk("%s: from_readbytes() at 0x%llx(0x%llx) failed\n",
                   __FUNCTION__, offset, *f_pos);
            kfree(mem);
            return -EIO;
        }

        rc = copy_to_user(buf, mem, count);
        if (rc < 0) {
            kfree(mem);
            return rc;
        }

        *f_pos = *f_pos + count;

        kfree(mem);
        return count;
    }

    return -ENODEV;
}

ssize_t
prom_write(struct file *filp, const char __user *buf, size_t count,
           loff_t *f_pos)
{
    int rc;

    int unit = MINOR(filp->f_dentry->d_inode->i_rdev);
    int prom = unit_prom(unit);

    /*
     * Allow only sector aligned writes of sector size for
     * simplicity.
     */

    if (count != FROM_SECTORSIZE) {
        return -EINVAL;
    }

    if (*f_pos % FROM_SECTORSIZE) {
        return -EINVAL;
    }

    /*
     * Check if writing past the end of the prom.
     */
    if (*f_pos >= unit_size(unit)) {
        return -EINVAL;
    }

    if (prom != -1) {
        uint8_t *mem = kmalloc(FROM_SECTORSIZE, 0);
        loff_t offset = unit_offset(unit) + *f_pos;

        debug(("%s: write at 0x%llx(0x%llx)\n",
               __FUNCTION__, offset, *f_pos));

        rc = copy_from_user(mem, buf, count);
        if (rc < 0) {
            kfree(mem);
            return rc;
        }

        rc = from_programbytes(offset, mem, FROM_SECTORSIZE);

        if (rc != 0) {
            printk("%s: from_programbytes() at 0x%llx(0x%llx) failed\n",
                   __FUNCTION__, offset, *f_pos);
            kfree(mem);
            return -EIO;
        }

        *f_pos = *f_pos + count;

        kfree(mem);

        return count;
    }

    return -ENODEV;
}


struct file_operations prom_fops = {
    .read = prom_read,
    .write = prom_write,
    .open = prom_open,
    .release = prom_release
};

static struct class *prom_dev_class;


static void
prom_uninit(void)
{
    printk("%s\n", __FUNCTION__);

    class_device_destroy(prom_dev_class, MKDEV(prom_major, 4));

    class_destroy(prom_dev_class);
    prom_dev_class = NULL;

    unregister_chrdev(prom_major, "prom");
}

static int
prom_init(void)
{
    int rc;

    printk("%s\n", __FUNCTION__);

    from_init();

    if (!from_device_present(FROM_CBX_PRIMARY)) {
        printk("%s: primary prom device not found\n", __FUNCTION__);
        return -ENODEV;
    }

    if (!from_device_present(FROM_CBX_ALTERNATE)) {
        /*
         * Not really an error, continue after warning.
         */
        printk("%s: alternate prom device not found\n", __FUNCTION__);
    }

    rc = register_chrdev(0, "prom", &prom_fops);
    if (rc < 0) {
        printk("%s: failed to register: %d\n", __FUNCTION__, rc);
        return rc;
    }

    prom_major = rc;

    prom_dev_class = class_create(THIS_MODULE, "prom");
    if (IS_ERR(prom_dev_class))
        return -1;

    /*
     * BSD version used to have a bunch of nodes to access different prom
     * partitions, but we really need only one for the whole prom. Call it
     * "promc" with minor number 4 for BSD compatibility.
     */
    class_device_create(prom_dev_class, NULL, MKDEV(prom_major, 4),
                        NULL, "promc");

    return 0;
}


module_init(prom_init);
module_exit(prom_uninit);
