/*
 * Name: neteee.c
 *
 * Description: Linux kernel implementation of OnStor NetEEE protocol suite
 * Copyright (c) 2005-2009 OnStor, Inc.
 * All rights reserved.
 */
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/fcntl.h>
#include <linux/socket.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/if_packet.h>
#include <linux/wireless.h>
#include <linux/kmod.h>
#include <net/ip.h>
#include <net/protocol.h>
#include <linux/skbuff.h>
#include <net/sock.h>
#include <linux/errno.h>
#include <linux/timer.h>
#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/ioctls.h>
#include <asm/page.h>
#include <asm/io.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/poll.h>
#include <linux/module.h>
#include <linux/init.h>
#include <net/neteee/eee.h>
#include <net/neteee/edesc.h>
#include <linux/if_arp.h>
#include <linux/etherdevice.h>

/* #define NETEEE_DEBUG 1 */

#ifdef NETEEE_DEBUG
#define NETEEE_PRINTK(x) printk x
#else
#define NETEEE_PRINTK(x)
#endif

#ifdef CONFIG_MGMT_BUS
extern struct net_device mgmtbus_dev;
struct net_device *net_dev = &mgmtbus_dev;
#else
struct net_device *net_dev = &loopback_dev;
#endif

uint8_t global_eee_cpu = 0; /* SSC(0), TxRx => ACPU(0), NCPU(1) */
uint8_t loopback_cpu = 0;

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)

/* TXRX slot is 1, for bind's local address AND getsockname() */
uint8_t global_eee_slot = 1;
uint8_t loopback_slot = 1;

#else

/* SSC slot is 0, for bind's local address AND getsockname() */
uint8_t global_eee_slot = 0;
uint8_t loopback_slot = 0;

/* All external IP packets go here.
 */
extern struct net_device relay_dev;

#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

/* Relayer socket.
 */
struct socket *relay_sock;

/* The last port that was allocated for ADDR_ANY.
 */
uint16_t eee_last_allocated_num = EEE_RESERVED_APP;

/* hash table of bound eee sockets.
 */
struct hlist_head eee_hashtable[EDESC2_APP_NUM];

/* Lock for the table and eee_last_allocated_num.
 */
DEFINE_RWLOCK(eee_hash_lock);

/* Lock for the reassembly queues.
 */
DEFINE_RWLOCK(eee_reass_lock);

/* Reassembly queues.
 */
LIST_HEAD(eee_reass_queues);

/* Current number of reassemblies in progress.
 */
int eee_num_reass;

/* Maximum number of reassemblies in progress. Each core can generate a
 * fragmented packet, we have at most 9 of cores, add a few entries
 * just in case.
 */
#define EEE_REASS_MAX 16

/* Reassembly queue entry.
 */
struct eee_reass_qentry {
    struct list_head list;    /* Linked list entry */
    struct sk_buff *sk_head;  /* The head of the reassembly list */
    struct sk_buff *sk_tail;    /* The tail of the reassembly list */
    int len;                    /* The length of reassembled data */
    int truesize;    /* True size of data */
    int data_len;    /* The length of the fragments except the first */
};

/* EEE socket structure.
 */
struct eee_sock {
    struct sock sk;        /* Common socket field */
    uint16_t num;          /* Our EEE application ID */
    struct eee_addr peer;  /* Peer address */
};

/* Datagram ID.
 */
uint16_t eee_packet_id;

static __inline struct eee_sock *
eee_sk(const struct sock *sk)
{
	return (struct eee_sock *)sk;
}

/* Some functions that are currently unused, make sure that we
 * return an error if they somehow get called.
 */
static int
eee_connect(struct sock *sk, struct sockaddr *uaddr, int addr_len)
{
    return -EPROTONOSUPPORT;
}

static int
eee_disconnect(struct sock *sk, int flags)
{
    return -EPROTONOSUPPORT;
}

static int
eee_ioctl(struct sock *sk, int cmd, unsigned long arg)
{
    return -EPROTONOSUPPORT;
}

static int
eee_destroy_sock(struct sock *sk)
{
    return 0;
}

static int
eee_setsockopt(struct sock *sk, int level, int optname,
               char __user *optval, int optlen)
{
    return -EPROTONOSUPPORT;
}

static int
eee_getsockopt(struct sock *sk, int level, int optname,
               char __user *optval, int __user *optlen)
{
    return -EPROTONOSUPPORT;
}

static int
eee_sendmsg(struct kiocb *iocb, struct sock *sk, struct msghdr *msg,
            size_t len)
{
    return -EPROTONOSUPPORT;
}

int
eee_sendpage(struct sock *sk, struct page *page, int offset,
             size_t size, int flags)
{
    return -EPROTONOSUPPORT;
}

void
eee_hash(struct sock *sk)
{
    return;
}

/*
 * Routine Description:
 *  Close eee socket.
 *
 * Arguments:
 *  sk - socket
 *  timeout - ignored
 *
 * Return Value:
 *  None.
 */
static void
eee_close(struct sock *sk, long timeout)
{
    sk_common_release(sk);
    return;
}

/*
 * Routine Description:
 *  Receive the message on EEE socket.
 *
 * Arguments:
 *  sk - socket
 *  msg - message header
 *  len - the requested length
 *  noblock - if true, don't block if no data available.
 *  addr_len - if the sender address is returned, the length is stored here
 *
 * Return Value:
 *  Returns the length of the received data or -errno.
 */
static int
eee_recvmsg(struct kiocb *iocb, struct sock *sk, struct msghdr *msg,
            size_t len, int noblock, int flags, int *addr_len)
{
    int copied, err;
    struct sk_buff *skb;
    struct sockaddr_eee *seee;
    agile_hdr_t *hdr;

    /* Get the pending data if any.
     */
    skb = skb_recv_datagram(sk, flags, noblock, &err);
    if (!skb) {
        goto out;
    }

    /* Setup the address length if required.
     */
    if (addr_len) {
        *addr_len = sizeof(struct sockaddr_eee);
    }

    copied = skb->len - sizeof(agile_hdr_t);

    /* Adjust the copy size for the provided buffer size and set the message
     * flags if truncated.
     */
    if (copied > len) {
        copied = len;
        msg->msg_flags |= MSG_TRUNC;
    }

    /* Copy the data into the user space.
     */
    err = skb_copy_datagram_iovec(skb, sizeof(agile_hdr_t),
                                  msg->msg_iov, copied);
    if (err) {
        goto out_free;
    }

    /* Fill in the message name if any.
     */
    if (msg->msg_name) {
        seee = msg->msg_name;
        hdr = (agile_hdr_t *)skb->data;

        seee->seee_family = AF_EEE;
        seee->seee_addr.eee_app = EEE_GET_APP_ID(hdr->src_port);
        seee->seee_addr.eee_cpu = EEE_GET_DEST_CPU(hdr->src_port);
        seee->seee_addr.eee_slot = EEE_GET_SLOT_NUM(hdr->src_port);
        seee->seee_addr.eee_bplane = 0;
    }

    /* In the success case the return value is the length of the data
     * received.
     */
    err = copied;

    /* Report the real message length if MSG_TRUNC is specified.
     */
    if (flags & MSG_TRUNC) {
        err = skb->len - sizeof(agile_hdr_t);
    }
out_free:
    skb_free_datagram(sk, skb);
out:
    return err;
}

/*
 * Routine Description:
 *  Remove the socket from the hash table, called on the socket destruction.
 *
 * Arguments:
 *  sk - socket
 *
 * Return Value:
 *  None.
 */
static __inline void
eee_unhash(struct sock *sk)
{
    write_lock_bh(&eee_hash_lock);
    if (sk_del_node_init(sk)) {
        eee_sk(sk)->num = 0;
        sock_prot_dec_use(sk->sk_prot);
    }
    write_unlock_bh(&eee_hash_lock);
    return;
}

/*
 * Routine Description:
 *  Should not be called, 1 means error.
 *
 * Arguments:
 *  sk - socket
 *  snum - port number
 *
 * Return Value:
 *  Returns 1.
 */
static __inline int
eee_get_port(struct sock *sk, unsigned short snum)
{
	return 1;
}

/* EEE protocol switch.
 */
struct proto eee_prot = {
	.name		   = "EEE",
	.owner		   = THIS_MODULE,
	.close		   = eee_close,
	.connect	   = eee_connect, //ip4_datagram_connect,
	.disconnect	   = eee_disconnect, // udp_disconnect,
	.ioctl		   = eee_ioctl, //udp_ioctl,
	.destroy	   = eee_destroy_sock,
	.setsockopt	   = eee_setsockopt, //udp_setsockopt,
	.getsockopt	   = eee_getsockopt, //udp_getsockopt,
	.sendmsg	   = eee_sendmsg, //udp_sendmsg,
	.recvmsg	   = eee_recvmsg, // udp_recvmsg,
	.sendpage          = eee_sendpage, // udp_sendpage,
	.hash		   = eee_hash,
	.unhash		   = eee_unhash,
	.get_port	   = eee_get_port, //udp_v4_get_port,
	.obj_size	   = sizeof(struct eee_sock),
};

/*
 * Routine Description:
 *  Release EEE socket, called on socket close.
 *
 * Arguments:
 *  sock - socket
 *
 * Return Value:
 *  0 - no error.
 */
int
eee_release(struct socket *sock)
{
	struct sock *sk = sock->sk;
	long timeout = 0;

	if (sk) {
		sock->sk = NULL;
		sk->sk_prot->close(sk, timeout);
	}
	return 0;
}

/*
 * Routine Description:
 *  Bind eee socket to the specified address. If the application id is 0,
 *  the port is dynamically allocated in the range above EEE_RESERVED_APP.
 *
 * Arguments:
 *  sock - socket
 *  uaddr - address to bind
 *    addr_len - the address length
 *
 * Return Value:
 *  EINVAL - the address length is not valid
 *  EINVAL - the slot/cpu are not SSC slot/cpu
 *  EINVAL - the application id is not valid
 *  EADDRNOTAVAIL - all addresses are in use
 *  EADDRINUSE - the requested address is in use
 */
int
eee_bind(struct socket *sock, struct sockaddr *uaddr, int addr_len)
{
    struct sockaddr_eee *addr_eee = (struct sockaddr_eee *)uaddr;
    struct sock *sk = sock->sk;
    int err = 0;
    struct hlist_head *head;
    uint16_t snum;

    /* Check the address length.
     */
    if (addr_len < sizeof(struct sockaddr_eee)) {
        err = -EINVAL;
        goto out;
    }

    /*
     * Check for invalid address, like trying to bind to something
     * on TxRx other than TxRx NCPU address <slot,cpu>, 1/0.
     */
    if ((addr_eee->seee_addr.eee_cpu != global_eee_cpu) ||
        (addr_eee->seee_addr.eee_slot != global_eee_slot) ||
        (addr_eee->seee_addr.eee_app >= EDESC2_APP_NUM)) {
        err = -EINVAL;
	goto out;
    }

    lock_sock(sk);
    snum = addr_eee->seee_addr.eee_app;

    /* Check for the socket already bound.
     */
    if (eee_sk(sk)->num) {
        err = -EINVAL;
        goto out_release_sock;
    }
    write_lock_bh(&eee_hash_lock);

    if (snum == 0) {
        /* Pick us an application id above the reserved range.
         */
        for (snum = eee_last_allocated_num + 1;
              snum != eee_last_allocated_num;
              ++snum ) {
            if ((snum >= EDESC2_APP_NUM) ||
                (snum < EEE_RESERVED_APP)) {
                      snum = EEE_RESERVED_APP;
            }
	    head = &eee_hashtable[snum];
	    if (hlist_empty(head)) {
	      eee_last_allocated_num = snum;
	      goto got_it;
	    }
        }
        err = -EADDRNOTAVAIL;
        goto out_release_hash;

    } else {
        /* Use the specified application id.
         */
        head = &eee_hashtable[snum];
        err = -EADDRINUSE;

        if (!hlist_empty(head)) {
            goto out_release_hash;
        }
    }

got_it:
    err = 0;
    /* Save the allocated application id in the socket structure.
     */
    eee_sk(sk)->num = snum;
    sk->sk_hash = snum;

    /* Insert the socket in the hash.
     */
    if (sk_unhashed(sk)) {
        head = &eee_hashtable[snum];
        sk_add_node(sk, head);
        sock_prot_inc_use(sk->sk_prot);
    }

out_release_hash:
    write_unlock_bh(&eee_hash_lock);

out_release_sock:
    release_sock(sk);

out:
    return err;
}

/*
 * Routine Description:
 *  Connect EEE socket to the peer.
 *
 * Arguments:
 *  sock - socket
 *  uaddr - the peer address
 *  addr_len - the address length
 *  flags - ignored
 *
 * Return Value:
 *  EINVAL - address invalid
 *  EAFNOSUPPORT - address family invalid
 *  Any error from eee_bind() if the socket was not bound previously.
 */
int
eee_datagram_connect(struct socket *sock, struct sockaddr * uaddr,
                  int addr_len, int flags)
{
    struct sock *sk = sock->sk;
    struct sockaddr_eee *seee = (struct sockaddr_eee *)uaddr;
    struct sockaddr_eee bind_eee_addr;
    int err = 0;

    /* Verify the address length.
     */
    if (addr_len < sizeof(*seee)) {
        err = -EINVAL;
        goto out;
    }

    /* Verify the address family.
     */
    if (seee->seee_family != AF_EEE) {
        err = -EAFNOSUPPORT;
        goto out;
    }

    /* Check if the address is valid.
     */
    if ((seee->seee_addr.eee_app == 0) ||
        (seee->seee_addr.eee_app >= EDESC2_APP_NUM)) {
        err = -EINVAL;
        goto out;
    }

    /* Bind the socket if it is not bound yet.
     */
    if (!eee_sk(sk)->num) {

        memset(&bind_eee_addr, 0, sizeof(struct sockaddr_eee));
        bind_eee_addr.seee_family = AF_EEE;
        bind_eee_addr.seee_slot = global_eee_slot;

        err = eee_bind(sock, (struct sockaddr *)&bind_eee_addr,
                       sizeof(struct sockaddr_eee));
        if (err) {
            goto out;
        }
    }

    /* Save the peer address in the socket structure.
     */
    eee_sk(sk)->peer = seee->seee_addr;
out:
    return err;
}

/*
 * Routine Description:
 *  Get our socket name of the name of the peer.
 *
 * Arguments:
 *  sock - socket
 *  uaddr - the name is stored here if successful
 *  uaddr_len - the name length is stored here
 *  peer - if true, get the address of the peer, otherwise of this socket.
 *
 * Return Value:
 *  ENOTCONN - tried to get peer address of not connected socket
 */
int
eee_getpeername(struct socket *sock, struct sockaddr *uaddr,
            int *uaddr_len, int peer)
{
    struct sock *sk = sock->sk;
    struct sockaddr_eee *seee = (struct sockaddr_eee *)uaddr;

    if (peer) {
        /* Check for not connected socket.
         */
        if (eee_sk(sk)->peer.eee_app == 0) {
            return -ENOTCONN;
        }
        seee->seee_family = AF_EEE;
        seee->seee_addr = eee_sk(sk)->peer;
    } else {
        seee->seee_family = AF_EEE;
        seee->seee_addr.eee_slot = global_eee_slot;
        seee->seee_addr.eee_cpu = 0;
        seee->seee_addr.eee_bplane = 2;
        seee->seee_addr.eee_app = eee_sk(sk)->num;
    }

    *uaddr_len = sizeof(struct sockaddr_eee);
    return 0;
}

int
eee_datagram_ioctl(struct socket *sock, unsigned int cmd, unsigned long arg)
{
    return -EPROTONOSUPPORT;
}

/*
 * Routine Description:
 *  Increment the Global EEE datagram ID.
 *
 * Arguments:
 *  None.
 *
 * Return Value:
 *  None.
 */
static __inline void
eee_inc_id(void)
{
    ++eee_packet_id;
    if (eee_packet_id > AGILEHDR_ID_MAX) {
        eee_packet_id = 0;
    }
    return;
}

/*
 * Routine Description:
 *  Fill in the EEE packet header.
 *
 * Arguments:
 *  sk - socket
 *  hdr - the packet header
 *  fragment_flag - the fragment flag to set
 *  fragment_offset - the offset of the fragment
 *
 * Return Value:
 *  None.
 */
static __inline void
eee_fill_header(struct sock *sk,
    agile_hdr_t *hdr,
    struct eee_addr *dst_addr,
    unsigned fragment_flag,
    unsigned fragment_offset)
{
    memset(hdr, 0, sizeof(*hdr));

    hdr->dest_port = ((dst_addr->eee_slot << EDESC2_SLOT_SHIFT) |
                      (dst_addr->eee_cpu << EDESC2_CPU_SHIFT) |
                      dst_addr->eee_app |
                      (EEE_MGMT << EDESC2_DATA_TYPE_SHIFT));

    /* For SSC the slot and cpu are ZERO, on TxRx the slot is not zero */
    hdr->src_port = ((global_eee_slot << EDESC2_SLOT_SHIFT) |
                     (global_eee_cpu << EDESC2_CPU_SHIFT) |
		     (eee_sk(sk)->num |
                     (EEE_MGMT << EDESC2_DATA_TYPE_SHIFT)));

    hdr->offset = (fragment_flag | fragment_offset |
                   (eee_packet_id << AGILEHDR_ID_SHIFT));
    hdr->h_proto = htons(ETH_P_EEE);
    return;
}

/*
 * Routine Description:
 *  Send the message to the peer.
 *
 * Arguments:
 *  iocb - ignored
 *  sock - socket
 *  msg - the message to send
 *  size - the length of the message
 *
 * Return Value:
 *  Returns the length of the message sent on success, otherwise returns
 *   -errno.
 *  EDESTADDRREQ - tried to send on unconnected socket without providing
 *                 the destination address
 *  EINVAL - the address is invalid
 *  EAFNOSUPPORT - the address family is invalid
 *  ENOBUF - memory allocation failed
 */
int
eee_datagram_sendmsg(struct kiocb *iocb, struct socket *sock, struct msghdr *msg, size_t size)
{
    struct sock *sk = sock->sk;
    struct eee_addr dst_addr;
    struct sockaddr_eee *seee = msg->msg_name;
    struct net_device *dev;
    struct sk_buff *skb;
    int err, res;
    agile_hdr_t *hdr;
    int fragment_flag;
    int fragment_offset;
    int size_this_fragment;
    int size_remain;

    err = 0;
    memset(&dst_addr, 0, sizeof(struct eee_addr));

    if (msg->msg_name == NULL) {
        if (eee_sk(sk)->peer.eee_app == 0) {
            return -EDESTADDRREQ;
        }

        dst_addr = eee_sk(sk)->peer;
    } else {
        if (msg->msg_namelen < sizeof(*seee)) {
            return -EINVAL;
        }

        if (seee->seee_family != AF_EEE) {
            return -EAFNOSUPPORT;
        }

        if (seee->seee_addr.eee_app == 0) {
            return -EINVAL;
        }

        dst_addr = seee->seee_addr;
    }

    /* Route the packet.
     */
    if ((dst_addr.eee_cpu == loopback_cpu) &&
        (dst_addr.eee_slot == loopback_slot)) {
        dev = &loopback_dev;
    } else {
        dev = net_dev;
    }

    NETEEE_PRINTK(("%s: START; <slot,cpu,app> FROM: %d:%d:%d TO: %d:%d:%d, size %u, dev: %s\n",
	__FUNCTION__,
	loopback_slot, loopback_cpu, eee_sk(sk)->num,
        dst_addr.eee_slot, dst_addr.eee_cpu, dst_addr.eee_app,
	(unsigned int)size, dev->name));

    dev_hold(dev);

    eee_inc_id();

    /* Copy the user data into skbs, fragmenting if necessary and
     * send them to the device.
     */
    fragment_offset = 0;
    size_remain = size;
    do {
        fragment_flag = ((dev->mtu < (size_remain + sizeof(agile_hdr_t) +
                         LL_RESERVED_SPACE(dev))) ? AGILEHDR_MF : 0);

        size_this_fragment = (fragment_flag ?
                         (dev->mtu - (sizeof(agile_hdr_t) + LL_RESERVED_SPACE(dev)))
                         : size_remain);

        /* Allocate the socket buffer.
         */
        skb = sock_alloc_send_skb(sk,
                                  size_this_fragment + sizeof(agile_hdr_t) +
                                  LL_RESERVED_SPACE(dev),
                                  msg->msg_flags & MSG_DONTWAIT, &err);
        if (skb == NULL) {
	    /* err should be set to -ENOBUF here via failing allocation call */
            goto out_unlock;
        }

        skb_reserve(skb, LL_RESERVED_SPACE(dev));

        /* Attach agile header
         */
        hdr = (agile_hdr_t *)skb_put(skb, sizeof(agile_hdr_t));
        skb_reset_network_header(skb);
        eee_fill_header(sk, hdr, &dst_addr, fragment_flag, fragment_offset);

	NETEEE_PRINTK(("%s: <slot,cpu,app> FROM: %d:%d:%d TO: %d:%d:%d, dev: %s\n",
		       __FUNCTION__,
		       loopback_slot, loopback_cpu, eee_sk(sk)->num,
		       dst_addr.eee_slot, dst_addr.eee_cpu, dst_addr.eee_app,
		       dev->name));

        /* Attach device header.
         */
        if (dev->hard_header) {
            res = dev->hard_header(skb, dev, ETH_P_EEE, NULL, NULL,
                                   size_this_fragment);
            if (res < 0) {
	        err = -EINVAL;
                goto out_free;
            }
        }

        /* Move in the user data.
         */
        err = memcpy_fromiovec(skb_put(skb, size_this_fragment),
                               msg->msg_iov,
                               size_this_fragment);
        if (err) {
            goto out_free;
        }

        skb->dev = dev;
        skb->priority = sk->sk_priority;
        skb->protocol = htons(ETH_P_EEE);

        NETEEE_PRINTK(("%s: dev_queue_xmit() skb 0x%p, skb->len %u, msg->msg_iov 0x%p\n",
                       __FUNCTION__, skb, skb->len, msg->msg_iov));
        NETEEE_PRINTK(("%s: dev_queue_xmit() <dev_name: %s, state 0x%x, features 0x%x>\n",
                       __FUNCTION__, skb->dev->name, skb->dev->state, skb->dev->features));

        err = dev_queue_xmit(skb);

#ifdef NETEEE_DEBUG
        if (err < 0) {
                printk("%s: dev_queue_xmit() err %d\n", __FUNCTION__, -err);
        }
#endif /* NETEEE_DEBUG */

        if (err > 0 && (err = net_xmit_errno(err)) != 0) {
            goto out_unlock;
        }

        size_remain -= size_this_fragment;
        fragment_offset += size_this_fragment;
   } while (size_remain != 0);

    dev_put(dev);
    NETEEE_PRINTK(("%s: return size %u\n", __FUNCTION__, (unsigned int)size));
    return size;

out_free:
    kfree_skb(skb);

out_unlock:
    if (dev != NULL) {
        dev_put(dev);
    }

    NETEEE_PRINTK(("%s: return err %d\n", __FUNCTION__, -err));
    return err;
}

/* EEE socket operations switch.
 */
const struct proto_ops eee_datagram_ops = {
	.family		   = PF_EEE,
	.owner		   = THIS_MODULE,
	.release	   = eee_release, //inet_release
	.bind		   = eee_bind, //inet_bind,
	.connect	   = eee_datagram_connect, //inet_dgram_connect,
	.socketpair	   = sock_no_socketpair,
	.accept		   = sock_no_accept,
	.getname	   = eee_getpeername, //inet_getname,
	.poll		   = datagram_poll,
	.ioctl		   = eee_datagram_ioctl, //inet_ioctl,
	.listen		   = sock_no_listen,
	.shutdown	   = sock_no_shutdown,
	.setsockopt	   = sock_common_setsockopt,
	.getsockopt	   = sock_common_getsockopt,
	.sendmsg	   = eee_datagram_sendmsg, //inet_sendmsg
	.recvmsg	   = sock_common_recvmsg,
	.mmap		   = sock_no_mmap,
	.sendpage	   = sock_no_sendpage,
};

/*
 * Routine Description:
    EEE socket destructor.

 * Arguments:
    sk - socket

 * Return Value:
    None.
 */
void
eee_sock_destruct(struct sock *sk)
{
	__skb_queue_purge(&sk->sk_receive_queue);
	__skb_queue_purge(&sk->sk_error_queue);

	if (!sock_flag(sk, SOCK_DEAD)) {
		printk(KERN_INFO "Attempt to release a live inet socket %p\n", sk);
	}
	return;
}

/*
 * Routine Description:
 *  Create the AF_EEE socket.
 *
 * Arguments:
 *  sock - the socket
 *  protocol - the protocol, should be 0 .
 *
 * Return Value:
 *  EPROTNOSUPPORT - socket type or protocol is not valid
 *  ENOBUF - memory allocation failed
 */
static int
eee_create(struct socket *sock, int protocol)
{
    struct sock *sk;
    int err = 0;

    /* Verify socket type.
     */
    if (sock->type != SOCK_DGRAM) {
        err = -EPROTONOSUPPORT;
        goto out;
    }

    /* Verify protocol.
     */
    if (protocol != 0) {
        err = -EPROTONOSUPPORT;
        goto out;
    }

    /* Allocate and initialize the socket. EEE sockets start unconnected
     * with 0 source and destination address.
     */
    sock->state = SS_UNCONNECTED;
    sock->ops = &eee_datagram_ops;

    sk = sk_alloc(PF_INET, GFP_KERNEL, &eee_prot, 1);

    if (sk == NULL) {
        err = -ENOBUFS;
        goto out;
    }
    sock_init_data(sock, sk);

    eee_sk(sk)->num = 0;
    memset(&eee_sk(sk)->peer, 0, sizeof(eee_sk(sk)->peer));

    sk->sk_destruct = eee_sock_destruct;
    sk->sk_family = PF_EEE;
    sk->sk_protocol = 0;
    sk->sk_prot = &eee_prot;

out:
    return err;
}

/* EEE address family switch
 */
static struct net_proto_family eee_family_ops = {
	.family = PF_EEE,
	.create = eee_create,
	.owner	= THIS_MODULE,
};

/*
 * Routine Description:
 *  Free the fragment queue, decrement the number of active reassemblies.
 *
 * Arguments:
 *  reass_queue - the queue to free
 *
 * Return Value:
 *  None.
 */
static void
eee_fragment_queue_free(struct eee_reass_qentry *reass_queue)
{
    --eee_num_reass;
    kfree(reass_queue);
    return;
}

/*
 * Routine Description:
 *  Free the fragments in the reassembly queue.
 *
 * Arguments:
 *  reass_queue  - the queue
 *
 * Return Value:
 *  None.
 */
static void
eee_fragment_free(struct eee_reass_qentry *reass_queue)
{
    struct sk_buff *sk = reass_queue->sk_head;

    while (sk != NULL) {
        struct sk_buff *next = sk->next;
        sk->next = NULL;
        kfree_skb(sk);
        sk = next;
    }
    return;
}

/*
 * Routine Description:
 *  Check if the fragment matches the queue contents.
 *
 * Arguments:
 *  reass_queue - the queue
 *  skb - the fragment
 *
 * Return Value:
 *  None.
 */
static bool
eee_fragment_match(struct eee_reass_qentry *reass_queue,
                   struct sk_buff *skb)
{
    agile_hdr_t *qhdr = (agile_hdr_t *)skb_network_header(reass_queue->sk_head);
    agile_hdr_t *hdr = (agile_hdr_t *)skb_network_header(skb);

    return (((qhdr->offset & AGILEHDR_ID_MASK) ==
             (hdr->offset & AGILEHDR_ID_MASK)) &&
            (hdr->src_port == qhdr->src_port) &&
            (hdr->dest_port == qhdr->dest_port));
}

/*
 * Routine Description:
 *  Delete the reassembly queue and return the reassembled list.
 *
 * Arguments:
 *  reass_queue - the queue
 *
 * Return Value:
 *  Returns the reassembled skb.
 */
static struct sk_buff *
eee_fragment_get(struct eee_reass_qentry *reass_queue)
{
    struct sk_buff *skb = reass_queue->sk_head;

    list_del(&reass_queue->list);

    skb_shinfo(skb)->frag_list = skb->next;
    skb->data_len = reass_queue->data_len;
    skb->len = reass_queue->len + sizeof(agile_hdr_t);
    skb->truesize = reass_queue->truesize;
    skb->next = NULL;

    eee_fragment_queue_free(reass_queue);
    return skb;
}

/*
 * Routine Description:
 *  Add the fragment to the queue.
 *
 * Arguments:
 *  reass_queue - the reassembly queue
 *  skb - the fragment
 *
 * Return Value:
 *  Returns NULL if the fragment was queued for reassembly, otherwise
 *  returns the reassembled datagram.
 */
static struct sk_buff *
eee_fragment_add(struct eee_reass_qentry *reass_queue,
                 struct sk_buff *skb)
{
    agile_hdr_t *hdr = (agile_hdr_t *)skb_network_header(skb);
    int frag_offset = hdr->offset & ~(AGILEHDR_ID_MASK | AGILEHDR_MF);

    if (reass_queue->len != frag_offset) {
        /* Out of order fragment, kill the reassembly.
         */
        list_del(&reass_queue->list);
        eee_fragment_free(reass_queue);
        eee_fragment_queue_free(reass_queue);
        kfree_skb(skb);
        return NULL;
    } else {
        /* Drop the header and add the fragment to the tail of the list.
         */
        skb_pull(skb, sizeof(agile_hdr_t));

        reass_queue->sk_tail->next = skb;
        reass_queue->sk_tail = skb;
        reass_queue->len += skb->len;
        reass_queue->data_len += skb->len;
        reass_queue->truesize += skb->truesize;

        if (!(hdr->offset & AGILEHDR_MF)) {
            /* Reassembly complete.
             */
            return eee_fragment_get(reass_queue);
        } else {
            /* Fragment queued.
             */
            return NULL;
        }
    }
}

/*
 * Routine Description:
 *  Allocate a new reassembly queue, recycling an old one if necessary.
 *
 * Arguments:
 *  skb - the head fragment
 *
 * Return Value:
 *  None.
 */
static void
eee_fragment_new(struct sk_buff *skb)
{
    struct eee_reass_qentry *reass_queue;
    agile_hdr_t *hdr = (agile_hdr_t *)skb_network_header(skb);

    if (hdr->offset & ~(AGILEHDR_ID_MASK | AGILEHDR_MF)) {
        /* Out of order fragment.
         */
        kfree_skb(skb);
        return;
    }

    if (eee_num_reass < EEE_REASS_MAX) {
        reass_queue = kmalloc(sizeof(*reass_queue), GFP_ATOMIC);
        if (reass_queue == NULL) {
            /* Drop if out of memory.
             */
            kfree_skb(skb);
            return;
        }
        ++eee_num_reass;
    } else {
        /* Recycle the oldest entry, which is at the head of the list.
         * New entries are added at tail.
         */
        reass_queue = list_entry(eee_reass_queues.next,
                                 struct eee_reass_qentry, list);

        list_del(&reass_queue->list);
    }

    list_add_tail(&reass_queue->list, &eee_reass_queues);

    reass_queue->sk_head = skb;
    reass_queue->sk_tail = skb;
    reass_queue->len = skb->len - sizeof(agile_hdr_t);
    reass_queue->data_len = 0;
    reass_queue->truesize = skb->truesize;
    return;
}

/* Receive function for EEE protocol.
 */
int
eee_rcv(struct sk_buff *skb,
	struct net_device *dev,
        struct packet_type *pt,
	struct net_device *orig_dev)
{
    agile_hdr_t *hdr;
    int app_id, rc;
    struct sock *sk = NULL;
    struct eee_reass_qentry *reass_queue;

    /* Check for enough length for the header.
     */
    if (!pskb_may_pull(skb, sizeof(agile_hdr_t))) {
        goto drop;
    }

    hdr = (agile_hdr_t *)skb_network_header(skb);
    app_id = EEE_GET_APP_ID(hdr->dest_port);

#ifdef NETEEE_DEBUG
    {
        int i;
        printk("%s: <slot,cpu,app> FROM <%d:%d:%d> TO <%d:%d:%d>, skb->len %d, hdr->offset 0x%x\n",
               __FUNCTION__,
               EEE_GET_SLOT_NUM(hdr->src_port),
               EEE_GET_DEST_CPU(hdr->src_port),
               EEE_GET_APP_ID(hdr->src_port),
               EEE_GET_SLOT_NUM(hdr->dest_port),
               EEE_GET_DEST_CPU(hdr->dest_port),
               app_id,
               skb->len,
               hdr->offset);

        printk("%s: skb 0x%p, data 0x%p, tail 0x%x\n",
	       __FUNCTION__,
               (void *)skb,
               (void *)skb->data,
	       skb->tail);

        printk("%s: skb 0x%p, network_hdr 0x%p, transport_hdr 0x%p\n",
	       __FUNCTION__,
               (void *)skb,
	       (void *)skb_network_header(skb),
	       (void *)skb_transport_header(skb));

	printk("%s: skb->data\n", __FUNCTION__);
        for (i = 0; i < 14+20; ++i) {
            printk("%02x ", skb->data[i]);
        }
        printk("\n");
    }
#endif /* NETEEE_DEBUG */

    /* Check for correct destination, if running on TxRx or SSC
     */
    if ((EEE_GET_SLOT_NUM(hdr->dest_port) != global_eee_slot) ||
        (EEE_GET_DEST_CPU(hdr->dest_port) != 0) ||
        (app_id == 0) || (app_id >= EDESC2_APP_NUM)) {

        NETEEE_PRINTK(("%s: <slot,cpu,app> TO <%d:%d:%d>, Dropping BOGUS Pkt\n",
               __FUNCTION__,
               EEE_GET_SLOT_NUM(hdr->dest_port),
               EEE_GET_DEST_CPU(hdr->dest_port),
	       app_id));
        goto drop;
    }

    if (hdr->offset & ~AGILEHDR_ID_MASK) { /* Perform reassembly */

        write_lock(&eee_reass_lock);
        list_for_each_entry(reass_queue, &eee_reass_queues, list) {

            if (eee_fragment_match(reass_queue, skb)) {
                skb = eee_fragment_add(reass_queue, skb);
                break;
            }
        }

        if (&reass_queue->list == &eee_reass_queues) {
            eee_fragment_new(skb);
            skb = NULL;
        }
        write_unlock(&eee_reass_lock);

        if (skb == NULL) { /* Queued fragment for reassembly */
            return 0;
        }
    }

    /* We have a complete reassembled datagram, or no reassembly was
     * required.
     */
#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
        /*
         * No Relay checking required as we're the target
         */
#else
    if (app_id == EEE_RELAY_APP_ID) {
        /*
         * This is IP relayed packet.  Strip the agile header and relay header
         * and resubmit the packet to stack.
         */
        NETEEE_PRINTK(("%s: Received relay packet: skb->len %d\n", __FUNCTION__, skb->len));

        skb_pull(skb, sizeof(agile_hdr_t) + sizeof(struct rl_hdr));

        skb->dev = &relay_dev;
        skb->protocol = eth_type_trans(skb, &relay_dev);

        rc = netif_receive_skb(skb);
    } else
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */
    {
        /*
         * Regular packet received
	 * Lookup socket by application ID and queue if found
         */
        read_lock(&eee_hash_lock);
        if (hlist_empty(&eee_hashtable[app_id])) {
	    NETEEE_PRINTK(("%s: NO Socket Found; Dropping Pkt, app_id %d\n",
		__FUNCTION__, app_id));
            goto drop_unlock;
        }

        sk = hlist_entry(eee_hashtable[app_id].first, typeof(*sk), sk_node);
        sock_hold(sk);
        read_unlock(&eee_hash_lock);

        rc = sock_queue_rcv_skb(sk, skb);
        if (rc < 0) {
	    NETEEE_PRINTK(("%s: sock_queue_rcv_skb() Enqueued FAILED: skb->len %d\n",
		__FUNCTION__, skb->len));
            goto drop;
        } else {
	    NETEEE_PRINTK(("%s: sock_queue_rcv_skb() Enqueued packet: skb->len %d\n",
		__FUNCTION__, skb->len));
        }
        sock_put(sk);
    }
    return 0;

drop_unlock:
    read_unlock(&eee_hash_lock);

drop:
    if (sk != NULL) {
        sock_put(sk);
    }
    kfree_skb(skb);
    return NET_RX_DROP;
}

/* EEE packet type switch.
 */
static struct packet_type eee_packet_type = {
	.type = __constant_htons(ETH_P_EEE),
	.func = eee_rcv,
};

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
/*
 * Not required for txrx linux as we're the relay target
 * The network device is a only a place holder on txrx
 * to avoid changing other OnStor SSC IPv4 networking code
 */
struct net_device relay_dev = {
    .name = "bp0",
    .flags = IFF_NOARP | IFF_BROADCAST,
    .mtu = 1500,
    .dev_addr = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05},
    .features = NETIF_F_FRAGLIST | NETIF_F_HIGHDMA | NETIF_F_LLTX,
    .addr_len = 0,
    .tx_queue_len = 0,
    .addr_len = ETH_ALEN,
    .type = ARPHRD_VOID,
};

#else

/*
 * Routine Description:
 *  On SSC, this is the transmit function for relay device.
 *  Attach ethernet header, relay header, agile header and forward the packet
 *  through the mgmtbus device.
 *
 *  Arguments:
 *   skb - skb to send
 *   dev - relay device
 *
 * Return Value:
 *  Returns 0.
 */
int
relay_hard_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
    struct sock *sk;
    agile_hdr_t *hdr;
    struct rl_hdr *rh;
    int err, res;

    err = 0;
    if (relay_sock) {
	sk = relay_sock->sk;
    } else {
	return 0;
    }

    if (!skb->dst) return 0;

    eth_header(skb, skb->dst->dev, ETH_P_IP, NULL, NULL, skb->len);
    rh = (struct rl_hdr *)skb_push(skb, sizeof(struct rl_hdr));

    skb_reset_network_header(skb);

    memset(rh, 0, sizeof(*rh));
    rh->rh_cpu = 0;
    rh->rh_slot = 1;
    rh->rh_ctrl = 1;

    hdr = (agile_hdr_t *)skb_push(skb, sizeof(agile_hdr_t));
    eee_inc_id();
    skb_reset_network_header(skb);
    eee_fill_header(sk, hdr, &eee_sk(sk)->peer, 0, 0);

    if (dev->hard_header) {

        res = net_dev->hard_header(skb, net_dev, ETH_P_EEE, NULL, NULL, skb->len);

        if (res < 0) {
	    err = -EINVAL;
            kfree_skb(skb);
            return 0;
        }
    }
    skb->protocol = htons(ETH_P_EEE);
    skb->dev = net_dev;
    err = dev_queue_xmit(skb);

    if (err > 0 && (err = net_xmit_errno(err)) != 0) {
        kfree_skb(skb);
    }
    return 0;
}

int
relay_hard_header(struct sk_buff *skb,
                  struct net_device *dev,
                  unsigned short type,
                  void *daddr,
                  void *saddr,
                  unsigned len)
{
    return 0;
}

static struct net_device_stats relay_stats;

static struct net_device_stats *
relay_get_stats(struct net_device *dev)
{
    return &relay_stats;
}

struct net_device relay_dev = {
    .name = "bp0",
    .flags = IFF_NOARP | IFF_BROADCAST,
    .mtu = 1500,
    .dev_addr = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05},
    .features = NETIF_F_FRAGLIST | NETIF_F_HIGHDMA | NETIF_F_LLTX,
    .hard_start_xmit = relay_hard_start_xmit,
    .hard_header = relay_hard_header,
    .hard_header_len  = (sizeof(agile_hdr_t) + sizeof(struct ethhdr) +
                         sizeof(struct rl_hdr)) + sizeof(struct ethhdr),
    .get_stats = relay_get_stats,
    .addr_len = 0,
    .tx_queue_len = 0,
    .addr_len = ETH_ALEN,
    .type = ARPHRD_VOID,
};

static int
neteee_init_relay(void)
{
    struct sockaddr_eee addr;
    int rc;

    rc = register_netdev(&relay_dev);
    if (rc) {
	dev_remove_pack(&eee_packet_type);
	proto_unregister(&eee_prot);
        goto out;
    }

    rc = sock_create_kern(AF_EEE, SOCK_DGRAM, 0, &relay_sock);

    if (rc != 0) {
        printk(KERN_ERR "%s: failed to create relay socket: %d\n",
          __FUNCTION__, rc);

	unregister_netdev(&relay_dev);
	dev_remove_pack(&eee_packet_type);
	proto_unregister(&eee_prot);
	goto out;
    }

    memset(&addr, 0, sizeof(addr));
    addr.seee_family = AF_EEE;
    addr.seee_addr.eee_cpu = 0;
    addr.seee_addr.eee_app = EEE_RELAY_APP_ID;

    /* SSC slot is 0, for bind's local address, for TxRx it is 1 */
    addr.seee_addr.eee_slot = global_eee_slot;

    rc = relay_sock->ops->bind(relay_sock, (struct sockaddr *)&addr, sizeof(addr));
    if (rc != 0) {
        printk(KERN_ERR "%s: failed to bind relay socket: %d\n", __FUNCTION__, rc);
	unregister_netdev(&relay_dev);
	dev_remove_pack(&eee_packet_type);
	proto_unregister(&eee_prot);
	goto out;
    }

    addr.seee_family = AF_EEE;
    addr.seee_addr.eee_slot = 1; /* pass to txrx cpu(0) */
    addr.seee_addr.eee_cpu = 0;
    addr.seee_addr.eee_app = EEE_RELAY_APP_ID;

    rc = relay_sock->ops->connect(relay_sock,
                                  (struct sockaddr *)&addr, sizeof(addr), 0);
    if (rc != 0) {
        printk(KERN_ERR "%s: failed to connect relay socket: %d\n",
               __FUNCTION__, rc);
    }
out:
    return rc;
}
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

static void
__exit neteee_exit(void)
{
	;
}

int
__init neteee_init(void)
{
    int rc;

    rc = proto_register(&eee_prot, 1);
    if (rc) {
        goto out;
    }

    (void)sock_register(&eee_family_ops);
    dev_add_pack(&eee_packet_type);

#if defined(CONFIG_SIBYTE_BCM1x55) || defined(CONFIG_SIBYTE_BCM1x80)
    /* No relay initialization required */
#else
    rc = neteee_init_relay();
#endif /* CONFIG_SIBYTE_BCM1x55 || CONFIG_SIBYTE_BCM1x80 */

out:
    return rc;
}

module_init(neteee_init);
module_exit(neteee_exit);
MODULE_LICENSE("GPL");
MODULE_ALIAS_NETPROTO(PF_EEE);

