/*
 *-----------------------------------------------------------------
 *
 * Name:	eee-debug.c
 *
 * RCS:  $Id: eee-debug.c,v 1.30 2002/11/07 01:20:03 rick Exp $
 *
 * Copyright (C) 2001, Agile Storage, Inc.
 *
 * Description:
 *
 * Created by:	Rick Lund
 *
 * Date Created:	5/16/01
 *
 *-----------------------------------------------------------------
 */

#include "nfx-incl.h"
#include "../sm-eee/eee-api.h"
#include "../sm-chassis/cm-api.h"
#include "../sm-timer/timer-api.h"

#include "../sm-eee/eee-debug.h"
#ifdef DMALLOC
#include "dmalloc.h"
#endif

#include "../sm-hw/pte-api.h"
#include "../sm-hw/mem-api.h"



uchar8 * bufsize_string[] = {"Small", "Large", "Ultra", "Mgmt"};

#ifdef NFX_MOD_NFP
uchar8	* mempool_string[] = {"LOCAL", "TXRX", "FP"};
uchar8	* ipcpool_string[] = {"TX", "RX", "FP1", "FP2"};
uchar8	* fwdpool_string[] = {"TXRX", "FP"};
#endif

#if defined(FCNIM)
uchar8	* mempool_string[] = {"LOCAL", "FP1", "FP2"};
uchar8	* ipcpool_string[] = {"FP1", "FP2"};
#endif

#ifdef NFX_MOD_SSC
uchar8	* mempool_string[] = {"LOCAL", "SSC", "SSC_MGMT"};
uchar8	* ipcpool_string[] = {"SSC", "SSC_MGMT"};
#endif

#ifdef NFX_MOD_LINUX_TEST
uchar8	* mempool_string[] = {"LOCAL", "LINUX_1", "LINUX_2", "LINUX_3"};
uchar8	* ipcpool_string[] = {"LINUX_1", "LINUX_2", "LINUX_3"};
#endif

#ifdef PROM
uchar8	* mempool_string[] = {"LOCAL", "PROM"};
uchar8	* ipcpool_string[] = {"??", "???"};
#endif

#ifdef EEE_POLLING_BUCKET
extern int32 eeePollingBuckets[EEE_NBR_POLLING_BUCKETS];
#endif 

/*-----------------------------------------------------------------
 * Name:	eee_displayEEEconfig()
 *
 * Description:	Display some of the EEE config structure
 *
 * Created by:	Rick Lund
 *
 * Date Created:	5/17/01
 *
 *-----------------------------------------------------------------
 */
void
eee_displayEEEconfig()
{
	uint32	mem, cpu, slot;
	uint32	i, j, size;
	uint32	bufsize;

	GET_MY_MEMPOOL_INDEX(mem);

	printf("EEE config details:\n");
	printf("   Number Shared Descriptors = %d\n", eee.sm_num_desc);
	printf("   Number Shared Descriptors Available = %d\n",
										eee.desc_pool[mem].stack.num_avail);
	printf("   Number Shared Small Buffers = %d\n", eee.sm_num_buf[0]);
	printf("   Number Shared Large Buffers = %d\n", eee.sm_num_buf[1]);
	printf("   Number Shared Ultra Buffers = %d\n", eee.sm_num_buf[2]);
#ifdef EEE_BUF_MGMT
    if(eee.sm_num_buf[3]) {
        /* Don't bother displaying anything if not configured to use these buffers.
         */
        printf("   Number Shared Mgmt Buffers = %d\n", eee.sm_num_buf[3]);
    }
    
#endif
	printf("   Number Shared Small Buffers Available = %d\n",
										eee.buf_pool[mem][0].stack.num_avail);
	printf("   Number Shared Large Buffers Available = %d\n",
										eee.buf_pool[mem][1].stack.num_avail);
	printf("   Number Shared Ultra Buffers Available = %d\n",
										eee.buf_pool[mem][2].stack.num_avail);
#ifdef EEE_BUF_MGMT
    if(eee.sm_num_buf[3]) {
        printf("   Number Shared Mgmt Buffers Available = %d\n",
               eee.buf_pool[mem][3].stack.num_avail);
    }
    
#endif
	printf("   Low Watermark Desc = %d\n",
			eee_stats->low_watermark_desc[eee.my_pool_id - SHARED_POOL_OFFSET]);
	printf("   Low Watermark Buf Small = %d\n",
		eee_stats->low_watermark_buf[eee.my_pool_id - SHARED_POOL_OFFSET][0]);
	printf("   Low Watermark Buf Large = %d\n",
		eee_stats->low_watermark_buf[eee.my_pool_id - SHARED_POOL_OFFSET][1]);
	printf("   Low Watermark Buf Ultra = %d\n",
		eee_stats->low_watermark_buf[eee.my_pool_id - SHARED_POOL_OFFSET][2]);
#ifdef EEE_BUF_MGMT
    if(eee.sm_num_buf[3]) {
        printf("   Low Watermark Buf Mgmt = %d\n",
               eee_stats->low_watermark_buf[eee.my_pool_id - SHARED_POOL_OFFSET][3]);
    }
    
#endif
	printf("   Small Buffer Base = %p, phys = %p\n", 
           eee.buf_base[mem][0], (char*)virt_to_phys((address_t)eee.buf_base[mem][0]));
    printf("   Small Buffer End  = %p, phys = %p, size = 0x%x\n", 
           eee.buf_end[mem][0], (char*)virt_to_phys((address_t)eee.buf_end[mem][0]),
           (int)(eee.buf_end[mem][0] - eee.buf_base[mem][0]));
	printf("   Large Buffer Base = %p, phys = %p\n", 
           eee.buf_base[mem][1], (char*)virt_to_phys((address_t)eee.buf_base[mem][1]));
	printf("   Large Buffer End  = %p, phys = %p, size = 0x%x\n", 
           eee.buf_end[mem][1], (char*)virt_to_phys((address_t)eee.buf_end[mem][1]),
           (int)(eee.buf_end[mem][1] - eee.buf_base[mem][1]));
	printf("   Ultra Buffer Base = %p, phys = %pn", 
           eee.buf_base[mem][2], (char*)virt_to_phys((address_t)eee.buf_base[mem][2]));
	printf("   Ultra Buffer End  = %p, phys = %p, size = 0x%x\n", 
           eee.buf_end[mem][2], (char*)virt_to_phys((address_t)eee.buf_end[mem][2]),
           (int)(eee.buf_end[mem][2] - eee.buf_base[mem][2]));
#ifdef EEE_BUF_MGMT
    if(eee.sm_num_buf[3]) {
        printf("   Mgmt Buffer Base = %p, phys = %p\n", 
               eee.buf_base[mem][3], (char*)virt_to_phys((address_t)eee.buf_base[mem][3]));
        printf("   Mgmt Buffer End  = %p, phys = %p, size = 0x%x\n", 
               eee.buf_end[mem][3], (char*)virt_to_phys((address_t)eee.buf_end[mem][3]),
               (int)(eee.buf_end[mem][3] - eee.buf_base[mem][3]));
    }
#endif

	printf("   Descriptor Resource Queues:\n");
	for (i=EEE_NUM_LM_POOLS; i<EEE_NUM_MEM_POOLS; i++) {
		if (i == mem)
			continue;
		size = eee.desc_pool[i].queue.end - eee.desc_pool[i].queue.start;
		printf("      %s --> %s, number of queue entries = 0x%x\n",
										mempool_string[i], mempool_string[mem],
										size);
		printf("      %s --> %s, number allocated = 0x%x\n",
										mempool_string[mem], mempool_string[i],
										eee_stats->rq_desc_allocated[i]);
	}

	for (bufsize = 0; bufsize < EEE_NUM_BUF_SIZES; bufsize++) {
		printf("   %s Buffer Resource Queues:\n", bufsize_string[bufsize]);
		for (i=EEE_NUM_LM_POOLS; i<EEE_NUM_MEM_POOLS; i++) {
			if (i == mem)
				continue;
			size = eee.buf_pool[i][bufsize].queue.end -
										eee.buf_pool[i][bufsize].queue.start;
			printf("      %s --> %s, number of queue entries = 0x%x\n",
										mempool_string[i], mempool_string[mem],
										size);
			printf("      %s --> %s, number allocated = 0x%x\n",
									mempool_string[mem], mempool_string[i],
									eee_stats->rq_buf_allocated[i][bufsize]);
		}
	}

	printf("   Forwarding:\n");
	printf("      slot     cpu       pkts \n");
	printf("      ----     ---    --------\n");
	for (slot = 0; slot < NFX_NUM_SLOTS_PER_CHASSIS; slot++) {
		for (cpu = 0; cpu < NFX_MAX_CPUS_PER_SLOT; cpu++) {
			if (eee_stats->fwd_pkts[slot][cpu]) {
				printf("       %2d     %2d         %8d\n",
								slot, cpu, eee_stats->fwd_pkts[slot][cpu]);
			}
		}
	}

	printf("   Application Interface:\n");
	for (i=0; i<EEE_MAX_APP_CLIENTS; i+=4) {
		printf("   ");
		for (j=0; j<4; j++) {
			printf("   %2d: %8d", i+j, eee_stats->app_rcv_pkts[i+j]);
		}
		printf("\n");
	}
	printf("      Invalid Application Ids : 0x%x\n",
										eee_stats->invalid_app_id);

#if defined(NFP_TXRX)
	printf("      Copied management packet bytes : %lld\n",
		   eee_stats->txrx_mgmt_copied);
#endif		   

}	/* eee_displayEEEconfig() */

/*-----------------------------------------------------------------
 * Name:
 *
 * Description:
 *
 * Created by:	Rick Lund
 *
 * Date Created:	3/2/02
 *
 *-----------------------------------------------------------------
 */
void
eee_clearFwdRcvStats()
{
	uint32	i, slot, cpu;

	for (i=0; i<EEE_MAX_APP_CLIENTS; i++)
		eee_stats->app_rcv_pkts[i] = 0;

	for (slot = 0; slot < NFX_NUM_SLOTS_PER_CHASSIS; slot++)
		for (cpu = 0; cpu < NFX_MAX_CPUS_PER_SLOT; cpu++)
			eee_stats->fwd_pkts[slot][cpu] = 0;

}

/*-----------------------------------------------------------------
 * Name: eee_showPerf
 *
 * Description:
 *
 * Created by:	Jan Vanhoutte
 *
 * Date Created:
 *
 *-----------------------------------------------------------------
 */

void
eee_showPerf(int32 clear)
{
    eeeGMT_t now;
#if defined(EEE_POLLING_BUCKET) || defined(EEE_POLLING_PERF)
    int32 i;
#endif
#ifdef EEE_POLLING_BUCKET
    int32 j;
#endif
#ifdef EEE_POLLING_PERF
    int64 curr, min, max;
#endif
            
    eee_rtcGMT (&now);

#ifdef EEE_POLLING_BUCKET
    printf("Function    ");
    for (i=0; i < (EEE_NBR_POLLING_BUCKETS - 1); i++) 
    {
        printf("> %6dus  ", eeePollingBuckets[i]);
    }
    printf("\n            ");
    for (i=1; i < (EEE_NBR_POLLING_BUCKETS - 1); i++) 
    {
        printf("< %6dus  ", eeePollingBuckets[i]);
    }
    printf("\n");

    for (i=1; i<EEE_MAX_POLL_HIGH_ROUTINES; i++) 
    {
 		if (eee.pfunc.high[i].func != NULL) 
        {
            printf("H %p  ", eee.pfunc.high[i].func);
       		for (j=1; j<EEE_NBR_POLLING_BUCKETS; j++) 
            {
                printf("%10d  ", eee.pfunc.high[i].bucket[j]);
            }
            printf("\n");
        }
    }
    for (i=1; i<EEE_MAX_POLL_LOW_ROUTINES; i++) 
    {
 		if (eee.pfunc.low[i].func != NULL) 
        {
            printf("L %p  ", eee.pfunc.low[i].func);
       		for (j=1; j<EEE_NBR_POLLING_BUCKETS; j++) 
            {
                printf("%10d  ", eee.pfunc.low[i].bucket[j]);
            }
            printf("\n");
        }
    }
    if ((eee.pfunc.clearTime.sec == 0) &&
        (eee.pfunc.clearTime.usec == 0))
    {
        eee.pfunc.clearTime.sec = bootGMTTime.sec ;
        eee.pfunc.clearTime.usec = bootGMTTime.usec ;
    }
#endif

    printf ("Time            %s\n", 
            eee_humandate(now.sec));
    
    printf ("Boot time       %s (%d day, %d hour, %d min, %d sec\n", 
            eee_humandate(bootGMTTime.sec),
            ((now.sec - bootGMTTime.sec) / SEC_PER_DAY),
            (((now.sec - bootGMTTime.sec) % SEC_PER_DAY) / SEC_PER_HOUR),
            (((now.sec - bootGMTTime.sec) % SEC_PER_HOUR) / SEC_PER_MIN),
            ((now.sec - bootGMTTime.sec) % SEC_PER_MIN));

#ifdef EEE_POLLING_BUCKET
    printf ("Clear counters  %s (%d day, %d hour, %d min, %d sec)\n", 
            eee_humandate(eee.pfunc.clearTime.sec),
            ((now.sec - eee.pfunc.clearTime.sec) / SEC_PER_DAY),
            (((now.sec - eee.pfunc.clearTime.sec) % SEC_PER_DAY) / SEC_PER_HOUR),
            (((now.sec - eee.pfunc.clearTime.sec) % SEC_PER_HOUR) / SEC_PER_MIN),
            ((now.sec - eee.pfunc.clearTime.sec) % SEC_PER_MIN));

    if (clear)
    {
        for (i=1; i<EEE_MAX_POLL_HIGH_ROUTINES; i++) 
        {
       		for (j=0; j<EEE_NBR_POLLING_BUCKETS; j++) 
            {
                eee.pfunc.high[i].bucket[j] = 0;
            }
        }
        for (i=1; i<EEE_MAX_POLL_LOW_ROUTINES; i++) 
        {
       		for (j=0; j<EEE_NBR_POLLING_BUCKETS; j++) 
            {
                eee.pfunc.low[i].bucket[j] = 0;
            }
        }
        eee_rtcGMT (&eee.pfunc.clearTime);
    }
#else
    // printf("Polling buckets not available\n");
#endif

#ifdef EEE_POLLING_PERF
    printf("Load indication (min %lld max %lld) : ", 
           eeePollingCounterMin, eeePollingCounterMax);
    max = MAX(eeePollingCounterMax, 1);
    min = MIN(eeePollingCounterMin, 0);

    for (i=0; i < EEE_POLLING_HIST; i++)
    {
        if (max <= min)
            printf ("MAX  ");
        else
        {
            curr = eeePollingCountersHist[(eeePollingCounterIndex + i + 1) % EEE_POLLING_HIST];
            curr = 100 - (curr - min) * 100 / (max - min);
            
            printf("%d%%  ", (int32)curr);
        }
    }
    printf("\n");

    if (clear)
    {
        eeePollingCounterMin = 999999999;
        eeePollingCounterMax = 0;
    }
    
#endif

}

/*-----------------------------------------------------------------
 * Name:	eee_displayErrorStats()
 *
 * Description:	Displays error counters
 *
 * Created by:	Rick Lund
 *
 * Date Created:	5/20/01
 *
 *-----------------------------------------------------------------
 */
void
eee_displayErrorStats()
{
	uint32	i, j;

	for (i=EEE_NUM_LM_POOLS; i<EEE_NUM_MEM_POOLS; i++) {
			printf("No desc available (%s) = %d\n", mempool_string[i],
											eee_stats->no_desc_avail[i]);
	}

	for (i=EEE_NUM_LM_POOLS; i<EEE_NUM_MEM_POOLS; i++) {
		for (j=0; j<EEE_NUM_BUF_SIZES; j++)
			printf("No %s buffer available (%s) = %d\n", bufsize_string[j],
														mempool_string[i],
												eee_stats->no_buf_avail[i][j]);
	}

	for (i=0; i<EEE_NUM_FWD_QUEUES; i++)
		printf("No FWD queue entry available (%s) = %d\n", ipcpool_string[i],
											eee_stats->no_fwd_avail[i]);

	printf("Invalid descriptor address = %d\n", eee_stats->invalid_desc_addr);
	printf("Invalid buffer address = %d\n", eee_stats->invalid_buf_addr);
    printf("Convert to management buffer failed = %d\n",
           eee_stats->convert_fail);

}	/* eee_displayErrorStats() */

#ifdef NFX_MOD_NFP
/*-----------------------------------------------------------------
 * Name:	eee_displayDescQueues(far)
 *
 * Description:	Display control structures for descriptor queues
 *
 * Created by:	Rick Lund
 *
 * Date Created:	5/16/01
 *
 *-----------------------------------------------------------------
 */
void
eee_displayDescQueues(
	uint32	far					/* 0, display local desc queue addresses,
								 * 1, display remote resource queue addresses
								 */
)
{
	uint32	my_mempool, i;

	GET_MY_MEMPOOL_INDEX(my_mempool);

	if (far) {
		printf("\nResource Queue Descriptor addresses:\n");
		for (i=EEE_NUM_LM_POOLS; i<EEE_NUM_MEM_POOLS; i++) {
			if (i == my_mempool)
				continue;
			printf("   %s --> %s, start = %p, end = %p\n",
								mempool_string[my_mempool], mempool_string[i],
								eee.far_desc_pool[i].start,
								eee.far_desc_pool[i].end);
		}
	}
	else {
#ifndef NFX_SMP
		printf("\nFree Descriptor Stack\n");
		printf("   %s, top = %p, num_avail = 0x%x\n",
									mempool_string[my_mempool],
									eee.desc_pool[my_mempool].stack.top,
									eee.desc_pool[my_mempool].stack.num_avail);
#endif
	}
}	/* eee_displayDescQueues() */

/*-----------------------------------------------------------------
 * Name:	eee_displayBufQueues(far)
 *
 * Description:	Display control structures for buffer queues
 *
 * Created by:	Rick Lund
 *
 * Date Created:	5/16/01
 *
 *-----------------------------------------------------------------
 */
void
eee_displayBufQueues(
	uint32	far					/* 0, display local buf queue addresses,
								 * 1, display remote resource queue addresses
								 */
)
{
	uint32	i, j, my_mempool;

	GET_MY_MEMPOOL_INDEX(my_mempool);

	if (far) {
		printf("\nResource Queue Buffer addresses:\n");
		for (i=EEE_NUM_LM_POOLS; i<EEE_NUM_MEM_POOLS; i++) {
			if (i == my_mempool)
				continue;
			for (j=0; j<EEE_NUM_BUF_SIZES; j++) {
				printf("   %s --> %s %s, start = %p, end = %p\n",
								mempool_string[my_mempool], mempool_string[i],
								bufsize_string[j],
								eee.far_buf_pool[i][j].start,
								eee.far_buf_pool[i][j].end);
			}
		}
	}
	else {
#ifndef NFX_SMP
		printf("\nFree Buffer Stack\n");
		for (j=0; j<EEE_NUM_BUF_SIZES; j++) {
			printf("   %s %s, top = %p, num_avail = 0x%x\n",
								mempool_string[my_mempool],
								bufsize_string[j],
								eee.buf_pool[my_mempool][j].stack.top,
								eee.buf_pool[my_mempool][j].stack.num_avail);
		}
#endif
	}
}	/* eee_displayBufQueues() */

/*-----------------------------------------------------------------
 * Name:	eee_displayIPCQueues(far)
 *
 * Description:	Display control structures for traditional IPC queues
 *
 * Created by:	Rick Lund
 *
 * Date Created:	5/16/01
 *
 *-----------------------------------------------------------------
 */
void
eee_displayIPCQueues(
	uint32	far					/* 0, display local IPC queue addresses,
								 * 1, display remote resource queue addresses
								 */
)
{
	uint32	i, ipc;

	GET_MY_IPC_INDEX(ipc);

	if (far) {
		printf("\nIPC Queues in remote pools\n");
		for (i=0; i<EEE_NUM_IPC_QUEUES; i++) {
			if (i == ipc)
				continue;
			printf("   %s --> %s, start = %p, end = %p\n",
								ipcpool_string[ipc], ipcpool_string[i],
								eee.ipc_send_start[i],
								eee.ipc_send_end[i]);
		}
	}
	else {
		printf("\nIPC Queues\n");
		for (i=0; i<EEE_NUM_IPC_QUEUES; i++) {
			if (i == ipc)
				continue;
			printf("   %s --> %s, start = %p, end = %p\n",
								ipcpool_string[i], ipcpool_string[ipc],
								eee.ipc_rcv_start[i],
								eee.ipc_rcv_end[i]);
		}
	}
}	/* eee_displayIPCQueues() */

/*-----------------------------------------------------------------
 * Name:	eee_displayFWDQueues(far)
 *
 * Description:	Display control structures for traditional FWD queues
 *
 * Created by:	Rick Lund
 *
 * Date Created:	5/16/01
 *
 *-----------------------------------------------------------------
 */
void
eee_displayFWDQueues(
	uint32	far					/* 0, display local FWD queue addresses,
								 * 1, display remote resource queue addresses
								 */
)
{
	uint32	i, fwd;

	GET_MY_FWD_INDEX(fwd);

	if (far) {
		printf("\nFWD Queues in remote pools\n");
		for (i=0; i<EEE_NUM_FWD_QUEUES; i++) {
			if (i == fwd)
				continue;
			printf("   %s --> %s, start = %p, end = %p\n",
								fwdpool_string[fwd], fwdpool_string[i],
								eee.fwd_queue[i].fwd_start,
								eee.fwd_queue[i].fwd_end);
			printf("                head = %p, tail = %p\n",
								eee.fwd_queue[i].fwd_head,
								*(eee.fwd_queue[i].fwd_tail_ptr));
			printf("                queued 0x%x entries\n", eee.fwd_queue[i].fwd_cnt);
		}
	}
	else {
		printf("\nRCV Queues\n");
		for (i=0; i<EEE_NUM_FWD_QUEUES; i++) {
			if (i == fwd)
				continue;
			printf("   %s --> %s, start = %p, end = %p\n",
								fwdpool_string[i], fwdpool_string[fwd],
								eee.rcv_queue[i].rcv_start,
								eee.rcv_queue[i].rcv_end);
			printf("                head = %p, tail = %p\n",
								eee.rcv_queue[i].rcv_head,
								*(eee.rcv_queue[i].rcv_tail_ptr));
			printf("                dequeued 0x%llx entries\n",
				   eee.rcv_queue[i].rcv_cnt);
		}
	}
}	/* eee_displayFWDQueues() */
#endif

/*-----------------------------------------------------------------
 * Name:        take_an_edump
 *
 * Description: Display eee_descPtr_t and associated parts
 *
 * Created by:  eric choi
 *
 * Date Created:        7/20/01
 *
 *-----------------------------------------------------------------
 */

void
take_an_edump(eee_descPtr_t edesc)
{
	uint32 i,x;
	uchar8 *displayPtr;

    printf("\nedesc at %p\n", edesc);
	printf("\nHDR:  control          %x",edesc->hdr.control);
	printf("\n      offset           %x",edesc->hdr.offset);
	printf("\n      dest_port        %x",edesc->hdr.dest_port);
	printf("\n      src_port         %x",edesc->hdr.src_port);
	printf("\n      attr             %p", (void *)edesc->hdr.attr);
	printf("\n      next             %p",edesc->hdr.next);

	for (i = 0; i < EEE_MAX_BUFS; i++) {
		if (edesc->bd[i].control & EDESC_BD_CMD_IN_USE) {
			printf("\nBD[%d]: control:       %x",i, edesc->bd[i].control);
			printf("\n        len:           %x",edesc->bd[i].len);
			printf("\n        buf:           %p\n",edesc->bd[i].buf);
			for (x = 0, displayPtr = edesc->bd[i].buf; x < edesc->bd[i].len;
															displayPtr++) {
				printf("%x ",*displayPtr);
				if (!(++x & 0x1F))
					printf("\n");
			}
			printf("\n");
		}
	}
}

/*-----------------------------------------------------------------
 * Name:
 *
 * Description:
 *
 * Created by:	Rick Lund
 *
 * Date Created:	1/30/02
 *
 *-----------------------------------------------------------------
 */
void
eee_debug_ui()
{
	uchar8	cmd[80];
	uint32	sel;
#ifdef EEE_ATTR
    address_t   addr;
#endif

	while (1) {
		printf("EEE debug menu:\n");
		printf("1. Display FWD queues\n");
		printf("2. Display RCV queues\n");
		printf("3. Display Resource queues\n");
		printf("4. Display Buffer stats\n");
#ifdef EEE_ATTR
		printf("5. Display Attribute Config\n");
		printf("6. Display Attribute\n");
		printf("7. Display All Descriptor Attributes\n");
		printf("8. Display All Buffer Attributes\n");

#endif
		printf("99. Quit\n");
		while (1) {
			printf("Enter selection: ");
			memset(cmd, 0, 80);
			get_cmd(cmd);
			if (strlen(cmd))
				break;
		}
		sscanf(cmd, "%d", &sel);
		printf("\n");

		switch (sel) {
			case 1:
				printf("FWD queues:\n");
#ifdef NFX_MOD_NFP
				eee_displayFWDQueues(1);
#endif
				break;
			case 2:
				printf("RCV queues:\n");
#ifdef NFX_MOD_NFP
				eee_displayFWDQueues(0);
#endif
				break;
			case 3:
#ifdef NFX_MOD_NFP
				printf("RQ queues:\n");
				eee_displayDescQueues(1);
				eee_displayBufQueues(1);
#endif
				break;
			case 4:
				printf("Buffer stats:\n");
				break;
#ifdef EEE_ATTR
			case 5:
				eee_attrDisplayConfig();
				break;
			case 6:
				printf("Enter buf/desc address: ");
				memset(cmd, 0, 80);
				get_cmd(cmd);
				addr = atolx(cmd);
				eee_attrDisplay(addr);
				break;
			case 7:
				eee_attrShowDescAll();
				break;
			case 8:
				eee_attrShowBufAll();
				break;
#endif
			case 99:
				return;
			default:
				printf("Invalid selection, %d\n", sel);
				break;
		}
	}
}

#ifdef EEE_VALIDATE

/*++

Routine Description:

    Display count of allocated buffers or descriptors and address 
    of allocating functions.

Arguments:

    pool - Must be 1.  Only pool 1 allocations are instrumented.
    size - 0-2 for small, large or ultra. Use size = EEE_NUM_BUF_SIZES 
              to display descriptors.
    alloc - Display allocated if true.
Return Value:


--*/

void
eee_displayAllocatedPoolBuffers(uint32 pool, uint32 size, boolean alloc)
{
#define ADDRESSES_TO_DISPLAY 100

    address_t addresses[ADDRESSES_TO_DISPLAY];
    int addressCounters[ADDRESSES_TO_DISPLAY];
    address_t savedAddress;
    eee_validate_t *validatePtr;
    int totalAllocated = 0;
    int bufCount;
    int i;
    int j;
    char *allocFree;
    /* Only '1' is used by the code that is saving the addresses.
     */
    if (pool != 1) {
        printf("%s: pool != 1\n", __FUNCTION__);
        return;
    }

    if (size > EEE_NUM_BUF_SIZES) {
        printf("%s: invalid buf size = %d\n", __FUNCTION__, size);
        return;
    }

    for (i = 0; i < ADDRESSES_TO_DISPLAY; i++) {
        addresses[i] = 0;
        addressCounters[i] = 0;
    }

    /* If size == EEE_NUM_BUF_SIZES, display descriptors.
     */
    if (size == EEE_NUM_BUF_SIZES) {
        validatePtr = eee_desc_validate[pool];
        bufCount = eee.sm_num_desc;
    } else {
        validatePtr = eee_buf_validate[pool][size];
        bufCount = eee.sm_num_buf[size];
    }

    if (alloc == TRUE) {
        allocFree = "allocated";
    } else {
        allocFree = "freed";
    }

    for (i = 0; i < bufCount; i++) {
        if (alloc == TRUE) {

            savedAddress = validatePtr[i].alloc;
        } else {
            savedAddress = validatePtr[i].free;
        }

        /* Is this buffer is currently allocated?
         */
        if (savedAddress) {

            /* Have we seen this address before?
             */
            for (j = 0; j < ADDRESSES_TO_DISPLAY; j++) {
                if ((addresses[j] == 0) || (addresses[j] == savedAddress)) {
                    /* Save the address and count it.
                     */
                    addressCounters[j]++;
                    addresses[j] = savedAddress;
                    break;
                }
            }
            totalAllocated++;
        }
    }

    if (size == EEE_NUM_BUF_SIZES) {
        printf("desc %s: %d \n", allocFree, totalAllocated);
    } else {
        printf("%s buffers %s: %d \n", bufsize_string[size], allocFree,
               totalAllocated);
    }

    if (totalAllocated) {
        for (i = 0; i < ADDRESSES_TO_DISPLAY; i++) {
            if (addressCounters[i]) {
                printf("  %s from 0x%lx: %d\n", allocFree, addresses[i],
                       addressCounters[i]);
            } else {
                break;
            }

        }
    }
}

/*++

Routine Description:

    Display count of allocated buffers or descriptors and address 
    of allocating functions.

Arguments:

    size - 0-2 for small, large or ultra. Use size = EEE_NUM_BUF_SIZES 
              to display descriptors.
    alloc - Display allocated if true.
Return Value:


--*/

void eee_displayAllocatedBuffers(uint32 size, boolean alloc)
{
    eee_displayAllocatedPoolBuffers(1, size, alloc);
}

#endif
