/*
 *-----------------------------------------------------------------
 * Name:    eee-desc.c
 * Description:
 * Copyright (C) 2009, OnStor, Inc.
 *-----------------------------------------------------------------
 */
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>

#include <linux/gfp.h>
#include <linux/slab.h>

#include "../tpl/nfx-defs.h"
#include "../tpl/nfx-types.h"
#include "../tpl/nfx-error.h"

#include "../tpl/bqueue.h"
#include "eee-drv.h"
#include "../tpl/eee-desc.h"

#include "eee-error.h"
#include "eee-validate.h"
#include "eee-attr.h"
#include "eee-poll.h"
#include "eee-app.h"
#include "eee-ipc.h"
#include "eee.h"

extern uint32 nfxMyCPUId;

/*-----------------------------------------------------------------
 * Name:	eee_buf_base()
 *
 * Description:	From an address within a buffer,
 *              return real base address of the buffer.
 *-----------------------------------------------------------------
 */
void *
eee_buf_base(void *_bufptr, unsigned int _pool_id, unsigned int _size_id)
{
    void *_r, *_bufaddr;

    _bufaddr = _bufptr;
    /* Verify that the buffer was really allocated from this pool
     */
    // ASSERT((eee.buf_base[eee_pool_index(_pool_id)][_size_id] <=
    // (void *)_bufptr) && ((void *)_bufptr < eee.buf_end[eee_pool_index(_pool_id)][_size_id]));

#define BUFADDR_M(a, m) (((u64)(a) % m) & 0x0ffffffffULL)
	if ((_size_id) == EEE_BUF_SMALL) {
		// _r = (_bufaddr & ((void *)eee.buf_mask[EEE_BUF_SMALL]));
		_r = _bufaddr - BUFADDR_M(_bufaddr, EEE_BUFSIZE_SMALL);
	} else if ((_size_id) == EEE_BUF_LARGE) {
		_r = _bufaddr - BUFADDR_M(_bufaddr, EEE_BUFSIZE_LARGE);
	} else if ((_size_id) == EEE_BUF_ULTRA) {
		_r = _bufaddr - BUFADDR_M(_bufaddr, EEE_BUFSIZE_ULTRA);
	} else if ((_size_id) == EEE_BUF_MGMT) {
		_r = _bufaddr - BUFADDR_M(_bufaddr, EEE_BUFSIZE_MGMT);
	} else {
		EEE_PRINTK(("Invalid size_id, 0x%x\n", _size_id));
		_r = NULL;
	}
	return _r;
}

#ifdef EEE_VALIDATE
/*
 * Routine Description:
    Get alloc/free validation structure for the descriptor.

 * Arguments:
    edesc - descriptor
    pool - memory pool

 * Return Value:
    Returns the validation structure.
 */
eee_validate_t *
eee_validate_desc_cb(eee_desc_t *edesc, int pool)
{
    uint32 vpool = (pool == 0) ? 0 : 1;
    int index = (((address_t)edesc - (address_t)eee.desc_base[pool]) / sizeof(eee_desc_t)); 
    return &eee_desc_validate[vpool][index];
}

/*
 * Routine Description:
    Get alloc/free validation structure for the eee buffer.

 * Arguments:
    buf - buffer
    pool - memory pool
    sz - buffer size

 * Return Value:
    Returns the validation structure.
 */
eee_validate_t *
eee_validate_buf_cb(void *buf, int pool, int sz)
{
    uint32 vpool = (pool == 0) ? 0 : 1;
    int index = (((address_t)buf - (address_t)eee.buf_base[pool][sz]) / eee.buf_size[sz]); 
    return &eee_buf_validate[vpool][sz][index];
}

/*
 * Routine Description:
    Validate if allocating object is OK.
 * Arguments:
    v - validation structure
    ra - caller's address
 * Return Value:
    None
 */
void
eee_validate_alloc(eee_validate_t *v, address_t ra)
{
    if (v->alloc) {
        printk("Alloc validate error, ra = %p prev_alloc = %p\n", ra, v->alloc);	
    }
    v->alloc = ra;					
    v->free = 0;
    v->refcnt = 0;
}

/*
 * Routine Description:
    Validate if freeing object is OK.
 * Arguments:
    v - validation structure
    ra - caller's address
 * Return Value:
    None
 */
void
eee_validate_free(eee_validate_t *v, address_t ra)
{
    if (v->free) {
      printk("Free validate error, ra=%p prev_free=%p\n", ra, v->free); 
    }
    if (v->refcnt != 0) {
        printk("Freeing referenced object ra=%p alloc=%p\n", ra, v->alloc);
    }
    v->alloc = 0;					
    v->free = ra;
}

/*
 * Routine Description:
    Validate if allocating eee descriptor is OK.
 * Arguments:
    desc - descriptor
    pool - memory pool
    ra - caller's address
 * Return Value:
    None
 */
void
eee_validate_alloc_desc(eee_desc_t *edesc, int pool, address_t ra)
{
    eee_validate_alloc(eee_validate_desc_cb(edesc, pool), ra);
}

/*
 * Routine Description:
    Validate if freeing eee descriptor is OK.
 * Arguments:
    desc - descriptor
    pool - memory pool
    ra - caller's address
 * Return Value:
    None
 */
void
eee_validate_free_desc(eee_desc_t *edesc, int pool, address_t ra)
{
    eee_validate_free(eee_validate_desc_cb(edesc, pool), ra);
}

/*
 * Routine Description:
    Validate if allocating a buffer is OK.
 * Arguments:
    buf - buffer
    pool - memory pool
    sz - buffer size
    ra - caller's address
 * Return Value:
    None
 */
void
eee_validate_alloc_buf(void *buf, int pool, int sz, address_t ra)
{
    eee_validate_alloc(eee_validate_buf_cb(buf, pool, sz), ra);
}

/*
 * Routine Description:
    Validate if freeing a buffer is OK.
 * Arguments:
    buf - buffer
    pool - memory pool
    sz - buffer size
    ra - caller's address
 * Return Value:
    None
 */
void
eee_validate_free_buf(void *buf, int pool, int sz, address_t ra)
{
    eee_validate_free(eee_validate_buf_cb(buf, pool, sz), ra);
}

/*
 * Routine Description:
    Increment reference count on the descriptor. Used only when 
    compiled with EEE_VALIDATE.
 * Arguments:
    edesc - EEE descriptor    
 * Return Value:
    Returns the descriptor.
 */
eee_desc_t *
eee_desc_hold(eee_desc_t *edesc)
{
    if (!eee_isRemoteAddr((uintptr_t)edesc)) {
        eee_validate_t *v = eee_validate_desc_cb(edesc, EEE_MY_SHARED_POOL);
        // VERIFY(v->refcnt < (uint32)-1);
        atomic_inc(&v->refcnt);
    }
    return edesc;
}

/*
 * Routine Description:
    Release a reference on the descriptor. Used only when compiled with EEE_VALIDATE.
 * Arguments:
    edesc - EEE descriptor
 * Return Value:
    None
 */
void
eee_desc_rele(eee_desc_t *edesc)
{
    if (!eee_isRemoteAddr((uintptr_t)edesc)) {
        eee_validate_t *v = eee_validate_desc_cb(edesc, EEE_MY_SHARED_POOL);
        // VERIFY(v->refcnt > 0);
        atomic_dec(&v->refcnt);
    }
}

/*
 * Routine Description:
    Get the buffer size index.
 * Arguments:
    buf - buffer
 * Return Value:
    Returns the buffer size index.
 */
int
eee_local_buf_size_idx(void *buf)
{
    int ix;

    if (buf < eee.buf_end[EEE_MY_SHARED_POOL][EEE_BUF_SMALL]) {
        ix = EEE_BUF_SMALL;
    } else if (buf < eee.buf_end[EEE_MY_SHARED_POOL][EEE_BUF_LARGE]) {
        ix = EEE_BUF_LARGE;
    } else if (buf < eee.buf_end[EEE_MY_SHARED_POOL][EEE_BUF_ULTRA]) {
        ix = EEE_BUF_ULTRA;
    } else {
        // VERIFY(0);
        ix = 0;
    }
    return ix;
}

/*
 * Routine Description:
    Increment reference count on the buffer. Used only when 
    compiled with EEE_VALIDATE.
 * Arguments:
    buf - buffer
 * Return Value:
    Returns the buffer.
 */
void *
eee_buf_hold(void *buf)
{
    eee_validate_t *v;

    if (!eee_isRemoteAddr((uintptr_t)buf)) {
        v = eee_validate_buf_cb(buf, EEE_MY_SHARED_POOL, eee_local_buf_size_idx(buf));
        // VERIFY(v->refcnt < (uint32)-1);
        atomic_inc(&v->refcnt);
    }
    return buf;
}

/*
 * Routine Description:
 *  Release a reference on the buffer. Used only when compiled with EEE_VALIDATE.
 * Arguments:
 *  buf - buffer
 * Return Value:
 *  None
 */
void
eee_buf_rele(void *buf)
{
    if (!eee_isRemoteAddr((uintptr_t)buf)) {
        eee_validate_t *v = eee_validate_buf_cb(buf, EEE_MY_SHARED_POOL, eee_local_buf_size_idx(buf));
        // VERIFY(v->refcnt > 0);
        atomic_dec(&v->refcnt);
    }
}

/*
 * Routine Description:
    Execute the code if compiled with validate on.
 * Arguments:
    code - the code to execute
 * Return Values:
    None
 */
#define VALIDATE(code) code
#else

#define eee_validate_alloc_desc(e, p, ra)
#define eee_validate_free_desc(e, p, ra)
#define eee_validate_alloc_buf(e, p, sz, ra)
#define eee_validate_free_buf(e, p, sz, ra)
#define VALIDATE(code)
#endif /* VALIDATE */

/*
 * Routine Description:
    Get the address of function which allocated the buffer.
 * Arguments:
    bufPtr - pointer within the buffer.
 * Return Values:
    Address of the function that allocated the buffer.
 */
address_t
eee_getBufferAllocAddr(void *bufPtr)
{
#if defined(EEE_VALIDATE)
    uint32  i, j;
    void *buf_base;

    for (i=0; i<EEE_NUM_MEM_POOLS; i++) {
        for (j=0; j<EEE_NUM_BUF_SIZES; j++) {
            if ((bufPtr >= eee.buf_base[i][j]) && (bufPtr < eee.buf_end[i][j])) {

                buf_base = eee_buf_base(bufPtr, i, j);
                if ((i == 0) || (i == eee.my_pool_id - SHARED_POOL_OFFSET)) {
                    return eee_validate_buf_cb(bufPtr, i, j)->alloc;
                } else {
                    /* buffer from other CPU */
                    return 0;
                }
            }
        }
    }
#endif
    return 0;
}

#define REMOTE_SHARED_POOL (EEE_REMOTE_POOL - SHARED_POOL_OFFSET)

#if defined(NFX_SMP)

#ifdef EEE_VALIDATE

/* Desciptors pool id.
 */
#define EEE_DESC_POOL_ID EEE_NUM_BUF_SIZES

/*
 * Routine Description:
    Check if the pool is used to allocate the EEE descriptors. This
    is used in validation code to determine which particular validation
    routine to call.
    
 * Arguments:
    pool - allocation pool

Return Values:
    Returns TRUE if the pool is used to allocate descriptors.
 */
boolean
eee_is_desc_pool(struct eee_global_stack_pool *pool)
{
    return pool->pool_id == EEE_DESC_POOL_ID;
}

/*
 * Routine Description:
    Validate if it is OK to free the buffer into the pool.
 * Arguments:
    pool - local allocation pool
    buf - the buffer to free
    ra - the caller's address
 * Return Values:
    None
 */
void
eee_buf_validate_before_free(struct eee_global_stack_pool *pool, void *buf, address_t ra)
{
    address_t buf_addr = (address_t)buf;

    if (pool->pool_id > EEE_DESC_POOL_ID) {
        return;
    }

    // ASSERT(pool->buf_base <= buf_addr);
    // ASSERT(pool->buf_end > buf_addr);
    if (eee_is_desc_pool(pool)) {
#ifdef EEE_ATTR
        // ASSERT(eee_attrOkToFreeDesc(edesc));
        eee_attrSetDescValidFreeAttr(edesc, 0, 0, 0, 0);
#endif
        eee_validate_free_desc(buf, EEE_MY_SHARED_POOL, ra);
    } else {
#ifdef EEE_ATTR
        // ASSERT(eee_attrOkToFreeBuf(buf));
        eee_attrSetBufValidFreeAttr(buf, 0, 0, 0, 0);
#endif
        eee_validate_free_buf(buf, EEE_MY_SHARED_POOL, pool->pool_id, ra);
    }
}

/*
 * Routine Description:
    Record that the buffer is freed.
 * Arguments:
    pool - local allocation pool
    buf - the buffer to free
    ra - the caller's address
 * Return Values:
    None
 */
void
eee_buf_record_free(struct eee_global_stack_pool *pool, void *buf, address_t ra)
{
    if (pool->pool_id > EEE_DESC_POOL_ID) {
        return;
    }
    if (eee_is_desc_pool(pool)) {
        eee_attrAssignDesc(edesc, EEE_ATTR_FREE);
   } else {
        eee_attrAssignBuf((void *)buf_base, EEE_ATTR_FREE);
    }
}

/*
 * Routine Description:
    Record that the buffer was allocated.
 * Arguments:
    pool - local allocation pool
    buf - the allocated buffer
    ra - the caller's address
 * Return Values:
    None
 */
void
eee_buf_record_alloc(struct eee_global_stack_pool *pool, void *buf, address_t ra)
{
    if (pool->pool_id > EEE_DESC_POOL_ID) {
        return;
    }

    if (eee_is_desc_pool(pool)) {
        eee_validate_alloc_desc(buf, EEE_MY_SHARED_POOL, ra);
        eee_attrAssignDesc(buf, EEE_ATTR_ALLOCATE);
    } else {
        eee_validate_alloc_buf(buf, EEE_MY_SHARED_POOL, pool->pool_id, ra);
        eee_attrAssignBuf(buf, EEE_ATTR_ALLOCATE);
    }
}
#endif /* EEE_VALIDATE */

/*
 * Routine Description:
    Move completely empty allocation slab to the list of empty slabs.
 * Arguments:
    global_pool - global allocation pool
    top - the slab stack top
    num_slab - number of elements in the slab
 * Return Values:
    None
 */
void
eee_pool_add_empty_slab(struct eee_global_stack_pool *global_pool, address_t *top, uint32 num_slab)
{
    struct eee_global_pool_buf_hdr *empty_buf = (void *)(top - num_slab);
    empty_buf->top = top;
    SLIST_INSERT_HEAD(&global_pool->empty_list, empty_buf, next);
}

/*
 * Routine Description:
    Initialize the global allocation pool. The pool is originally empty
    and has enough control memory allocated for num_avail number of objects.
    Two extra control stacks are allocated so the list of empty control stacks
    is never exhausted.

 * Arguments:
    global_pool - global allocation pool
    num_avail - the total number of objects
    num_slab - the number of elements in a slab
    id - the pool Id
    buf_base - the beginning of buffer memory
    buf_end - the end of buffer memory

Return Values:
    None
 */
void
eee_global_pool_init(struct eee_global_stack_pool *global_pool,
    uint32 num_avail, uint32 num_slab, uint32 id, void *buf_base, void *buf_end)
{
    int i;
    uint32 total_num_elts = (((num_avail + num_slab - 1) / num_slab) * num_slab) + (SMP_MAX_CPUS * num_slab);
    address_t *stack_mem = eee_allocateBlock(total_num_elts * sizeof(address_t));

    if (stack_mem == NULL) {
        printk("Can not allocate memory for global_pool %d\n", id);
    }

    NFX_SPIN_INIT(&global_pool->spin, SPINLOCK_LEVEL_UNKNOWN);

    SLIST_INIT(&global_pool->empty_list);
    SLIST_INIT(&global_pool->free_list);

#ifdef EEE_VALIDATE
    global_pool->pool_id = id;
    global_pool->buf_base = (uintptr_t)buf_base;
    global_pool->buf_end = (uintptr_t)buf_end;
#endif

    for (i = 0; i < total_num_elts; 
          i += num_slab, stack_mem += num_slab) {
        eee_pool_add_empty_slab(global_pool, stack_mem + num_slab, num_slab);
    }

    global_pool->num_avail = 0;
}

struct eee_local_stack_pool local_desc_pool;
struct eee_local_stack_pool local_buf_pool[EEE_NUM_BUF_SIZES];

/*
 * Routine Description:
    Initialize local allocation pool.
    
 * Arguments:
    pool - local allocation pool
    global_pool - global allocation pool
    num_slab - the number of elements in a slab

Return Values:
    None
 */
void
eee_local_pool_init(struct eee_local_stack_pool *pool,
                    struct eee_global_stack_pool *global_pool,
                    uint32 num_slab)
{
    if (pool->free_list.tqh_first == NULL) {
        TAILQ_INIT(&pool->free_list);
        NFX_SPIN_INIT(&pool->spin, SPINLOCK_LEVEL_UNKNOWN);
        pool->global_pool = global_pool;
    }
}

/*
 * Routine Description:
    Deallocate the buffer to the local pool.
 * Arguments:
    pool - the allocation pool
    obj - the pointer to the beginning of the buffer 
 * Return Values:
    None
 */
#ifdef EEE_VALIDATE

void
eee_dealloc_buf(struct eee_local_stack_pool *pool, void *obj)
{
    eee_dealloc_buf_ra(pool, obj, get_ra());
}

void
eee_dealloc_buf_ra(struct eee_local_stack_pool *pool, void *obj, address_t ra)
#else
void
eee_dealloc_buf(struct eee_local_stack_pool *pool, void *obj)
#endif
{
    NFX_SPIN_LOCK(&pool->spin);

    VALIDATE(eee_buf_validate_before_free(pool->global_pool, obj, ra));

    struct eee_local_stack_entry *ctl = obj;
    TAILQ_INSERT_TAIL(&pool->free_list, ctl, entry);

    VALIDATE(eee_buf_record_free(pool->global_pool, obj, ra));

    ++pool->global_pool->num_avail;
        
    NFX_SPIN_UNLOCK(&pool->spin);
}   

/*
 * Routine Description:
    Allocate a buffer from the local pool.
    
 * Arguments:
    pool - the pool to use for the allocation

Return Values:
    Returns the allocated buffer or NULL if not enough memory.
 */
#ifdef EEE_VALIDATE

void *
eee_alloc_buf(struct eee_local_stack_pool *pool)
{
    return eee_alloc_buf_ra(pool, get_ra());
}

void *
eee_alloc_buf_ra(struct eee_local_stack_pool *pool, address_t ra)
#else

void *
eee_alloc_buf(struct eee_local_stack_pool *pool)
#endif /* EEE_VALIDATE */
{
    NFX_SPIN_LOCK(&pool->spin);

    struct eee_local_stack_entry *ctl = TAILQ_FIRST(&pool->free_list);
    if (ctl != NULL) {
        TAILQ_REMOVE(&pool->free_list, ctl, entry);
        ctl->entry.tqe_next = NULL;
        ctl->entry.tqe_prev = NULL;
        VALIDATE(eee_buf_record_alloc(pool->global_pool, ctl, ra));

        --pool->global_pool->num_avail;
    }
    NFX_SPIN_UNLOCK(&pool->spin);
    return ctl;
}

#if defined(EEE_VALIDATE)

eee_desc_t *
eee_allocateDesc(uint32 mempool, uint32 cos unused__)
{
    return eee_allocateDescRA(mempool, cos, get_ra());
}

eee_desc_t *
eee_allocateDescRA(uint32 mempool, uint32 cos unused__, address_t ra)
#else
eee_desc_t *
eee_allocateDesc(uint32 mempool, uint32 cos unused__)
#endif
{
    eee_queue_pool_cntl_t *queue;
    address_t ptr;

    if (mempool != EEE_REMOTE_POOL) {
        try_shared:
            return eee_alloc_buf_ra(&local_desc_pool, ra);
    } else {
        NFX_SPIN_LOCK(&eee.descpool_spin);
        queue =	&eee.desc_pool[EEE_REMOTE_POOL - SHARED_POOL_OFFSET].queue;
	ptr = *queue->head;

	if ((void*)ptr != NULL) {
			(void*)*queue->head = NULL;
			if (((address_t)queue->head & CACHELINE_MASK) == 0)
				*(queue->tail_ptr) = queue->head;
			queue->head++;
			if (queue->head == queue->end)
				queue->head = queue->start;
			eee_attrAssignDesc((void *)ptr, EEE_ATTR_RQ_DEQUEUE);
            NFX_SPIN_UNLOCK(&eee.descpool_spin);
            return (void *)ptr;
        } else {
            NFX_SPIN_UNLOCK(&eee.descpool_spin);
	    eee_stats->no_desc_avail[EEE_REMOTE_POOL - SHARED_POOL_OFFSET]++;
	    goto try_shared;
	}
    }
}

#if defined(EEE_VALIDATE)
void
eee_deallocateBufferRA(void *bufPtr, address_t ra)
#else
void
eee_deallocateBuffer(void *bufPtr)
#endif
{
    struct eee_local_stack_pool *pool;
    address_t   buf_base;
    int size unused__;

    if (!eee_isRemoteAddr((address_t)bufPtr)) {

        if (bufPtr < eee.buf_end[EEE_MY_SHARED_POOL][EEE_BUF_SMALL]) {
            buf_base = (address_t)bufPtr & ~(EEE_BUFSIZE_SMALL - 1);
            pool = &local_buf_pool[EEE_BUF_SMALL];
            size = EEE_BUF_SMALL;
        }
        else if (bufPtr < eee.buf_end[EEE_MY_SHARED_POOL][EEE_BUF_LARGE]) {
            buf_base = (address_t)bufPtr -
                (uint32)(address_t)bufPtr % EEE_BUFSIZE_LARGE;
            pool = &local_buf_pool[EEE_BUF_LARGE];
            size = EEE_BUF_LARGE;
        }
        else if (bufPtr < eee.buf_end[EEE_MY_SHARED_POOL][EEE_BUF_ULTRA]) {
            buf_base = (address_t)bufPtr -
                (uint32)(address_t)bufPtr % EEE_BUFSIZE_ULTRA;
            pool = &local_buf_pool[EEE_BUF_ULTRA];
            size = EEE_BUF_ULTRA;
        }
        else {
            *(volatile uint64 *)0 = 0;
            return;
        }
        eee_dealloc_buf_ra(pool, (void *)buf_base, ra);
    } else {

#ifdef EEE_ATTR
        if (!(eee_attrOkToFreeBuf(bufPtr))) {
            eee_attrDisplay(bufPtr);
            printk("eee_attr failed in deallocateBufferRA, buf = %p, ra = %p\n", bufPtr, ra);
        }
        eee_attrSetBufValidFreeAttr(bufPtr, 0, 0, 0, 0);
#endif  

        if (bufPtr < eee.buf_end[REMOTE_SHARED_POOL][EEE_BUF_SMALL]) {
            buf_base = (address_t)bufPtr & ~(EEE_BUFSIZE_SMALL - 1);
            size = EEE_BUF_SMALL;
        }
        else if (bufPtr < eee.buf_end[REMOTE_SHARED_POOL][EEE_BUF_LARGE]) {
            buf_base = (address_t)bufPtr -
                (uint32)(address_t)bufPtr % EEE_BUFSIZE_LARGE;
            size = EEE_BUF_LARGE;
        }
        else if (bufPtr < eee.buf_end[REMOTE_SHARED_POOL][EEE_BUF_ULTRA]) {
            buf_base = (address_t)bufPtr -
                (uint32)(address_t)bufPtr % EEE_BUFSIZE_ULTRA;
            size = EEE_BUF_ULTRA;
        }
        else {
            *(volatile uint64 *)0 = 0;
            return;
        }

#if defined(EEE_VALIDATE)
        if (buf_base < (address_t)eee.buf_base[REMOTE_SHARED_POOL][size]) {
            printk("invalid buf %p (%p %p) sz %d\n",
                  buf_base,
                  eee.buf_base[REMOTE_SHARED_POOL][size],
                  eee.buf_end[REMOTE_SHARED_POOL][size],
                  size);
        }
#endif
        atomic_inc(&eee_stats->fwd_free_buf[EEE_REMOTE_QUEUE_INDEX]);

        eee_attrAssignBuf((void *)buf_base, EEE_ATTR_FWD_TO_FREE);

        if (FWD_PACKET(buf_base | EEE_FREE_BUF
                       | (size << EEE_FWD_BUF_SIZE_SHIFT),
                       EEE_REMOTE_QUEUE_INDEX,
                       TRUE) != EEE_FORWARDED_PACKET) {
#if defined(EEE_VALIDATE)           
            printk("No forwarding entry available to free buf, "
                   "buf = %lx, ra = %lx\n", buf_base, ra);
#endif          
        }
    }
}

#ifdef EEE_VALIDATE
void *
eee_allocateBuffer(uint32 mempool, uint16 buf_size, uint32 cos unused__)
{
    void *ptr = eee_allocateBufferRA(mempool, buf_size, cos, get_ra());
    return ptr;
}

void *
eee_allocateBufferRA(uint32 mempool, uint16 buf_size, uint32 cos unused__, address_t ra)
#else
void *
eee_allocateBuffer(uint32 mempool, uint16 buf_size, uint32 cos unused__)
#endif
{
  void *ptr = NULL;
  eee_queue_pool_cntl_t *queue;

  if (mempool != EEE_REMOTE_POOL) {
                ptr = eee_alloc_buf_ra(&local_buf_pool[buf_size], ra);
  } else {
                NFX_SPIN_LOCK(&eee.bufpool_spin);
		queue =	&eee.buf_pool[EEE_REMOTE_POOL - SHARED_POOL_OFFSET][buf_size].queue;
		ptr = *queue->head;

		if (ptr) {
			(void*)*queue->head = NULL;
			if (((address_t)queue->head & CACHELINE_MASK) == 0)
				*(queue->tail_ptr) = queue->head;
			queue->head++;
			if (queue->head == queue->end)
				queue->head = queue->start;
			eee_attrAssignBuf((void *)ptr, EEE_ATTR_RQ_DEQUEUE);
			NFX_SPIN_UNLOCK(&eee.bufpool_spin);

		} else { /* try agin before failing */
			++eee_stats->no_buf_avail[EEE_REMOTE_POOL - SHARED_POOL_OFFSET][buf_size];
			NFX_SPIN_UNLOCK(&eee.bufpool_spin);
			ptr = eee_alloc_buf_ra(&local_buf_pool[buf_size], ra);
		}
  }
  return ptr;
}

#ifdef EEE_VALIDATE
void
eee_deallocateDescRA(eee_desc_t *edesc, address_t ra)
#else
void
eee_deallocateDesc(eee_desc_t *edesc)
#endif
{
    (address_t)edesc &= ~EDESC_ADDR_MASK;

    if (eee_isLocalAddr((address_t)edesc)) {
        edesc->hdr.next = 0;
        edesc->hdr.offset = 0;
        edesc->hdr.control = 0;
        eee_dealloc_buf_ra(&local_desc_pool, edesc, ra);
    } else if (eee_isRemoteCachedAddr((address_t)edesc)) {
        __asm__("cache %0,0(%1)" : : "i"(Hit_Invalidate_D),"r"(edesc));
        __asm__("cache %0,32(%1)" : : "i"(Hit_Invalidate_D),"r"(edesc));
        __asm__("cache %0,64(%1)" : : "i"(Hit_Invalidate_D),"r"(edesc));
        __asm__("cache %0,96(%1)" : : "i"(Hit_Invalidate_D),"r"(edesc));
        edesc = (eee_descPtr_t)
            (((address_t)edesc - REMOTE_CACHED_MEM) + REMOTE_MEM);
        goto free_uncached;
    } else {
        int rc unused__;

    free_uncached:
        
#ifdef EEE_ATTR
    if (!(eee_attrOkToFreeDesc(edesc))) {
        eee_attrDisplay(edesc);
        printk("eee_attr failed in deallocateDescRA, edesc = %p, ra = %p\n", edesc, ra);
    }
    eee_attrSetDescValidFreeAttr(edesc, 0, 0, 0, 0);
#endif

        atomic_inc(&eee_stats->fwd_free_desc[EEE_REMOTE_QUEUE_INDEX]);
#ifdef EEE_ATTR
        eee_attrAssignDesc(edesc, EEE_ATTR_FWD_TO_FREE);
#endif
        rc = FWD_PACKET((address_t)edesc | EEE_FREE_DESC,
                        EEE_REMOTE_QUEUE_INDEX, TRUE);
#if defined(EEE_VALIDATE)
        if (edesc < eee.desc_base[EEE_REMOTE_POOL - SHARED_POOL_OFFSET]
            || edesc >= eee.desc_end[EEE_REMOTE_POOL - SHARED_POOL_OFFSET]) {
            printk("desciptor %p is not remote descriptor (%p %p)\n",
                  edesc, eee.desc_base[EEE_REMOTE_POOL - SHARED_POOL_OFFSET],
                  eee.desc_end[EEE_REMOTE_POOL - SHARED_POOL_OFFSET]);
        }
        if (rc != EEE_FORWARDED_PACKET) {
            printk("No forwarding entry available to free desc\n");
        }
#endif
    }
}

#else

#if defined(EEE_VALIDATE)

void
cache_invalidate_struct(address_t buf_base, unsigned int buf_size)
{
}

/*-----------------------------------------------------------------
 * Name:    eee_deallocateBuffer
 *
 * Description: Return buffer to free pool
 *              If this is one of my buffers, then add it back onto
 *              either the local pool's free stack, or the shared pool's
 *              free stack, depending on the address.
 *              If not one of mine, then deliver to rightful owner.
 *-----------------------------------------------------------------
 */

void
eee_deallocateBufferRA(void *bufPtr, address_t   ra)
#else
void
eee_deallocateBuffer(void *bufPtr)
#endif
{
#if defined(FCNIM) && (EEE_NUM_SM_POOLS == 0)
    test_eee_deallocateBuffer(bufPtr);
#else

    address_t   buf_base;
    int size unused__;

#if defined (NFP_FP) || defined (FEATURE_DIAG_PROM)
    if (mm_isMallocBuffer(bufPtr)) {
        eee_ramDealloc(bufPtr);
        return;
    }
#endif

#ifdef EEE_ATTR
    if (!(eee_attrOkToFreeBuf(bufPtr))) {
        eee_attrDisplay(bufPtr);
        printk("eee_attr failed in deallocateBufferRA, buf = %p, ra = %p\n", bufPtr, ra);
    }
    eee_attrSetBufValidFreeAttr(bufPtr, 0, 0, 0, 0);
#endif

    // ASSERT(!eee_isRemoteCachedAddr((address_t)bufPtr));

#if defined(NFP_FP) || defined(NFP_TXRX)
    if (!eee_isRemoteAddr((address_t)bufPtr))
#endif
    {
        eee_stack_pool_cntl_t *pool;

        if (bufPtr < eee.buf_end[EEE_MY_SHARED_POOL][EEE_BUF_SMALL]) {
            buf_base = (address_t)bufPtr & ~(EEE_BUFSIZE_SMALL - 1);
            pool = &eee.buf_pool[EEE_MY_SHARED_POOL][EEE_BUF_SMALL].stack;
            size = EEE_BUF_SMALL;
            cache_invalidate_struct(buf_base, EEE_BUFSIZE_SMALL);
        }
        else if (bufPtr < eee.buf_end[EEE_MY_SHARED_POOL][EEE_BUF_LARGE]) {
            buf_base = (address_t)bufPtr -
                (uint32)(address_t)bufPtr % EEE_BUFSIZE_LARGE;
            pool = &eee.buf_pool[EEE_MY_SHARED_POOL][EEE_BUF_LARGE].stack;
            size = EEE_BUF_LARGE;
            cache_invalidate_struct(buf_base, EEE_BUFSIZE_LARGE);
        }
        else if (bufPtr < eee.buf_end[EEE_MY_SHARED_POOL][EEE_BUF_ULTRA]) {
            buf_base = (address_t)bufPtr -
                (uint32)(address_t)bufPtr % EEE_BUFSIZE_ULTRA;
            pool = &eee.buf_pool[EEE_MY_SHARED_POOL][EEE_BUF_ULTRA].stack;
            size = EEE_BUF_ULTRA;
            cache_invalidate_struct(buf_base, EEE_BUFSIZE_ULTRA);
        }
        else {
            *(volatile uint64 *)0 = 0;
            return;
        }

#if defined(EEE_VALIDATE)
        if (buf_base < (address_t)eee.buf_base[EEE_MY_SHARED_POOL][size]) {
            printk("invalid buf %p (%p %p) sz %d\n",
                  buf_base,
                  eee.buf_base[EEE_MY_SHARED_POOL][size],
                  eee.buf_end[EEE_MY_SHARED_POOL][size],
                  size);
        }
        eee_validate_free_buf(buf_base, EEE_MY_SHARED_POOL, size, ra);
#endif
        NFX_SPIN_LOCK(&eee.bufpool_spin);
        *(--pool->top) = buf_base;
        ++pool->num_avail;

        eee_attrAssignBuf((void *)buf_base, EEE_ATTR_FREE);
        NFX_SPIN_UNLOCK(&eee.bufpool_spin);
    }
#if defined(NFP_FP) || defined(NFP_TXRX)
    else {
        if (bufPtr < eee.buf_end[REMOTE_SHARED_POOL][EEE_BUF_SMALL]) {
            buf_base = (address_t)bufPtr & ~(EEE_BUFSIZE_SMALL - 1);
            size = EEE_BUF_SMALL;
            cache_invalidate_struct(buf_base, EEE_BUFSIZE_SMALL);
        }
        else if (bufPtr < eee.buf_end[REMOTE_SHARED_POOL][EEE_BUF_LARGE]) {
            buf_base = (address_t)bufPtr -
                (uint32)(address_t)bufPtr % EEE_BUFSIZE_LARGE;
            size = EEE_BUF_LARGE;
            cache_invalidate_struct(buf_base, EEE_BUFSIZE_LARGE);
        }
        else if (bufPtr < eee.buf_end[REMOTE_SHARED_POOL][EEE_BUF_ULTRA]) {
            buf_base = (address_t)bufPtr -
                (uint32)(address_t)bufPtr % EEE_BUFSIZE_ULTRA;
            size = EEE_BUF_ULTRA;
            cache_invalidate_struct(buf_base, EEE_BUFSIZE_ULTRA);
        }
        else {
            *(volatile uint64 *)0 = 0;
            return;
        }

#if defined(EEE_VALIDATE)
        if (buf_base < (address_t)eee.buf_base[REMOTE_SHARED_POOL][size]) {
            printk("invalid buf %p (%p %p) sz %d\n",
                  buf_base,
                  eee.buf_base[REMOTE_SHARED_POOL][size],
                  eee.buf_end[REMOTE_SHARED_POOL][size],
                  size);
        }
#endif
        atomic_inc(&eee_stats->fwd_free_buf[EEE_REMOTE_QUEUE_INDEX]);
        cache_invalidate_struct(buf_base, eee.buf_size[j]);

        eee_attrAssignBuf((void *)buf_base, EEE_ATTR_FWD_TO_FREE);

        if (FWD_PACKET(buf_base | EEE_FREE_BUF
                       | (size << EEE_FWD_BUF_SIZE_SHIFT),
                       EEE_REMOTE_QUEUE_INDEX, TRUE) != EEE_FORWARDED_PACKET) {
#if defined(EEE_VALIDATE)           
            printk("No forwarding entry available to free buf, "
                   "buf = %lx, ra = %lx\n", buf_base, ra);
#endif          
        }
    }
#endif  
#endif
}

/*-----------------------------------------------------------------
 * Name:    eee_deallocateDesc
 *
 * Description: Releases an EEE descriptor back into the free pool.
 *              If this is one of my descriptors, then add it back onto
 *              either the local pool's free stack, or the shared pool's
 *              free stack, depending on the address.
 *              If not one of mine, then deliver to rightful owner.
 *-----------------------------------------------------------------
 */
#if defined(EEE_VALIDATE)
void
eee_deallocateDescRA(eee_descPtr_t edesc_param, address_t ra)
#else
void
eee_deallocateDesc(eee_descPtr_t edesc_param)
#endif  
{
    eee_descPtr_t edesc;
    // (address_t)edesc &= ~EDESC_ADDR_MASK;
    // XXX
    edesc = edesc_param;

#ifdef EEE_ATTR
    if (!(eee_attrOkToFreeDesc(edesc))) {
        eee_attrDisplay(edesc);
        printk("eee_attr failed in deallocateDescRA, edesc = %p, ra = %p\n", edesc, ra);
    }
    eee_attrSetDescValidFreeAttr(edesc, 0, 0, 0, 0);
#endif

    if (eee_isLocalAddr((address_t)edesc)) {
        eee_stack_pool_cntl_t *pool = &eee.desc_pool[EEE_MY_SHARED_POOL].stack;
#ifdef EEE_VALIDATE
        if (edesc < eee.desc_base[EEE_MY_SHARED_POOL]
            || edesc >= eee.desc_end[EEE_MY_SHARED_POOL]) {
            printk("descriptor %p is not shared descriptor (%p %p)\n",
                  edesc, eee.desc_base[EEE_MY_SHARED_POOL],
                  eee.desc_end[EEE_MY_SHARED_POOL]);
        }
        // EEE_VALIDATE_FREE_DESC(edesc, EEE_MY_SHARED_POOL, ra);
#endif
        edesc->hdr.next = 0;
        edesc->hdr.offset = 0;
        edesc->hdr.control = 0;
        
        NFX_SPIN_LOCK(&eee.descpool_spin);

        *(--pool->top) = (address_t)edesc;
        ++pool->num_avail;

#ifdef EEE_ATTR
        eee_attrAssignDesc(edesc, EEE_ATTR_FREE);
#endif
        NFX_SPIN_UNLOCK(&eee.descpool_spin);
    }
#if defined(NFP_FP) || defined(NFP_TXRX)
    else if (eee_isRemoteCachedAddr((address_t)edesc)) {
        __asm__("cache %0,0(%1)" : : "i"(Hit_Invalidate_D),"r"(edesc));
        __asm__("cache %0,32(%1)" : : "i"(Hit_Invalidate_D),"r"(edesc));
        __asm__("cache %0,64(%1)" : : "i"(Hit_Invalidate_D),"r"(edesc));
        __asm__("cache %0,96(%1)" : : "i"(Hit_Invalidate_D),"r"(edesc));
        edesc = (eee_descPtr_t)
            (((address_t)edesc - REMOTE_CACHED_MEM) + REMOTE_MEM);
        goto free_uncached;
    }
    else 
    {
        int rc unused__;
    free_uncached:
        atomic_inc(&eee_stats->fwd_free_desc[EEE_REMOTE_QUEUE_INDEX]);
#ifdef EEE_ATTR
        eee_attrAssignDesc(edesc, EEE_ATTR_FWD_TO_FREE);
#endif
        rc = FWD_PACKET((address_t)edesc | EEE_FREE_DESC,
                        EEE_REMOTE_QUEUE_INDEX, TRUE);
#if defined(EEE_VALIDATE)
        if (edesc < eee.desc_base[EEE_REMOTE_POOL - SHARED_POOL_OFFSET]
            || edesc >= eee.desc_end[EEE_REMOTE_POOL - SHARED_POOL_OFFSET]) {
            printk("desciptor %p is not remote descriptor (%p %p)\n",
                  edesc, eee.desc_base[EEE_REMOTE_POOL - SHARED_POOL_OFFSET],
                  eee.desc_end[EEE_REMOTE_POOL - SHARED_POOL_OFFSET]);
        }
        if (rc != EEE_FORWARDED_PACKET) {
            printk("No forwarding entry available to free desc\n");
        }
#endif
    }
#endif  
}
#endif

#if defined(EEE_VALIDATE)
/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_deallocateDesc(eee_descPtr_t edesc)
{
    address_t ra = (address_t)get_ra();
    eee_deallocateDescRA(edesc, ra);
}
#endif

/*-----------------------------------------------------------------
 * Name:    eee_freePacket
 * Description: Free an EEE descriptor and the associated buffers
 *-----------------------------------------------------------------
 */
void
eee_freePacket(eee_descPtr_t edesc)
{
    address_t ra unused__ = get_ra();
    eee_descBuf_t   * bd;
    uint32          num_bufs, i;
    uint32          free_rout_ind;
    uint32          cont;

    while (edesc != NULL) {
        eee_descPtr_t next_desc = eee_nextDesc(edesc);
        cont = edesc->hdr.control;
        if ((cont & EDESC_PKT_NO_DEALLOC) != 0)
            return;
        
        eee_unlinkNextDesc(edesc);
        free_rout_ind = (cont & EDESC_FREE_ROUTINE_INDEX) >>
            EDESC_FREE_ROUTINE_SHIFT;
        if (free_rout_ind) {
            // VERIFY(eee.free_routine[free_rout_ind-1] != 0);
            (*eee.free_routine[free_rout_ind-1])(edesc);
        }
        
        bd = &(edesc->bd[0]);
        num_bufs = EEE_GET_NUM_BUFS(edesc);

        if (!(edesc->hdr.control & EDESC_COMMAND_USE_FIRST_BUFFER))
            bd++;

        for (i=0; i<num_bufs; i++, bd++) {
            if ((bd->control & EDESC_BD_CMD_IN_USE) &&
                !(bd->control & EDESC_BD_CMD_NO_DEALLOC)) {
#if defined(NFP_TXRX) || defined(NFP_FP)
                if (eee_isRemoteCachedAddr((address_t)bd->buf)) {
                    /* In the case we did conversion to
                     * cached address invalidate the cache lines
                     * the application potentially accessed and
                     * convert back to uncached before deallocating
                     * the buffer.
                     */
                    sibyte_cacheInvalidate(
                        ((address_t)bd->buf & 0xFFFFFFFFFFFFFFE0LL),
                        bd->len+((address_t)bd->buf & 0x1F));
                    eee_deallocateBufferRA(
                        (void *)cached_to_uncached((address_t)bd->buf), ra);
                }
                else {
                    eee_deallocateBufferRA(bd->buf, ra);
                }
#else
                eee_deallocateBufferRA(bd->buf, ra);
#endif
            }
        }
        eee_deallocateDescRA(edesc, ra);
        edesc = next_desc;
    }
}

/*-----------------------------------------------------------------
 * Name :   eee_nextBuf()
 *
 * Description: Move the pointers to the next buffer. The new descriptor
 * is stored in the location specified by the argument edesc.
 *
 * Returns: new buffer descriptor or NULL if this is last buffer descriptor.
 *-----------------------------------------------------------------
 */
eee_descBuf_t *
eee_nextBuf(eee_descPtr_t *edesc, eee_descBuf_t *bd)
{
    // ASSERT(edesc != NULL);
    // ASSERT(&(*edesc)->bd[edesc_firstBufferIdx(*edesc)] <= bd
    // && bd < &(*edesc)->bd[EEE_GET_NUM_BUFS(*edesc) + edesc_firstBufferIdx(*edesc)]);

    ++bd;
    if (bd == &(*edesc)->bd[EEE_GET_NUM_BUFS(*edesc) + edesc_firstBufferIdx(*edesc)]) {
        *edesc = eee_nextDesc(*edesc);
        bd = (*edesc) != NULL ? &(*edesc)->bd[edesc_firstBufferIdx(*edesc)] : NULL;
    }
    return bd;
}

/*-----------------------------------------------------------------
 * Name :   eee_copyPacketDataFrom()
 * Description: Copy num_bytes of descriptor data from the given
 *              location to a local buffer. Location is specified by buffer index
 *              and pointer into the buffer memory.
 * Returns:     Number of bytes copied
 *-----------------------------------------------------------------
 */
int
eee_copyPacketDataFrom(
    void *dest,
    eee_descPtr_t edesc,
    int num_bytes,
    int bufidx,
    char *bufmem
    )
{
    uchar8 *dp = dest;
    int copied;
    int buflen;
    eee_descBuf_t *bd;
    
    /* Copy first (maybe partial) buffer */
    bd = &edesc->bd[bufidx];
    buflen = MIN(num_bytes, eee_remainBufLen(bd, bufmem));
    memcpy(dest, bufmem, buflen);
    dp += buflen;
    copied = buflen;
    bd = eee_nextBuf(&edesc, bd);
    while (bd != NULL && copied < num_bytes) {
        buflen = bd->len;
        if (copied + buflen >= num_bytes) {
            memcpy(dp, bd->buf, num_bytes - copied);
            copied = num_bytes;
        }
        else {
            memcpy(dp, bd->buf, buflen);
            copied += buflen;
            dp += buflen;
            bd = eee_nextBuf(&edesc, bd);
        }
    }

    return copied;
}

/*
 *  Routine Description:
    This routine verifies the arguments to the copy function and update the
    number of bytes to copy according to the size of the packet. 

 *  Arguments:
    dest - The destination buffer.
    edesc - The descriptor to copy.
    offset - The offset with the descriptor where the copy will start.
    numBytes - This starts as the number of bytes to copy from the descriptor
                to dest.  If there is more data in the packet to copy than
                this value we will either return an error, or adjust this
                size.  truncOK controls this behavior.
    truncOK - TRUE if it's OK to increase the size of numBytes to include
              all of the data in the descriptor at 'offset'.  FALSE will
              result in an error if there is more data to copy than
              numBytes.

 *  Return Value:
    NFX_OK if arguments are correct, otherwise EEE_ERR_PACKET_COPY_FAILED.
 */
int
eee_verifyCopyArgs(void *dest,
                   eee_descPtr_t edesc,
                   uint32 offset,
                   uint32 *numBytes,
                   boolean truncOK)
{
    int pktlen = EEE_GET_PKT_LEN(edesc);
    int buf_size;
    if (offset >= pktlen) {
        return EEE_ERR_PACKET_COPY_FAILED;
    }
    if ((offset + *numBytes) > pktlen) {
        if (truncOK) {
            *numBytes = pktlen - offset;
        } else {
            return EEE_ERR_PACKET_COPY_FAILED;
        }
    }

    /* If the destination buffer is a malloc'd buffer, then verify size
     */
    if (mm_isMallocBuffer(dest)) {
        if (!(mm_isSizeOkay(dest, *numBytes, &buf_size))) {
            printk("%s, numBytes = %d, buf = %p, buf_size = %d\n",
                  __FUNCTION__, *numBytes, dest, buf_size);
            return EEE_ERR_PACKET_COPY_FAILED;
        }
    }
    return NFX_OK;
}

/*-----------------------------------------------------------------
 * Name :   eee_copyPacketData()
 *
 * Description: Copies bytes from a system buffer starting from the
 *              given offset into a contiguous memory area.
 * Returns:     Number of bytes copied or EEE_ERR_PACKET_COPY_FAILED
 *-----------------------------------------------------------------
 */
int32
eee_copyPacketData(void *dest,                 /* destination address of local buffer */
    eee_descPtr_t edesc,        /* source descriptor */
    uint32 offset,              /* offset in the descriptor */
    uint32 num_bytes,           /* bytes to copy */
    uint32 truncate_ok          /* if non-zero, allow truncation */
    )
{
   int bufidx;
    uchar8 *copyfrom;
    int32 rv;

    rv = eee_verifyCopyArgs(dest, edesc, offset, &num_bytes, truncate_ok);
    if (rv == NFX_OK) {
        eee_skipBufsFromStart(&edesc, offset, &bufidx, &copyfrom);
        rv = eee_copyPacketDataFrom(dest, edesc, num_bytes, bufidx, copyfrom);
    }
    return rv;
}

/*-------------------------------------------------------------------------------
 * Name: eee_copyBufferToPacket()
 *
 * Description:
 *     This function copies a flat buffer into a (chained) edesc after skipping
 *     a given number of bytes.  It has the option of copying only part of the
 *     flat buffer if the edesc is not large enough to hold the entire buffer.
 *
 * Arguments:
 *     buf - This is the flat buffer to be copied into the edesc.
 *     bufSize - This is the size of buf.
 *     edesc - This is the edesc to be copied into.
 *     numBytesToSkip - This is the number of bytes to skip within the edesc to
 *                      start copying buf into.
 *     allow_partial_copy - If this is FALSE then this function will only copy
 *                          buf into edesc if it is possible to copy all of buf
 *                          into edesc.  If this is TRUE then as much of buf
 *                          will be copied into edesc as there is space.
 *
 * Return Values:
 *     Number of bytes copied.
 *------------------------------------------------------------------------------
 */
uint32
eee_copyBufferToPacket(const void *buf,
                       int bufSize,
                       eee_desc_t *edesc,
                       uint32 numBytesToSkip,
                       boolean allow_partial_copy)
{
    int total_bytes_to_copy = bufSize;
    int remain = bufSize;
    const char *copy_from = buf;
    int bufIdx;
    uchar8 *memptr;
    eee_descBuf_t *current_bd;

    if (((EEE_GET_PKT_LEN(edesc) + numBytesToSkip) < bufSize) &&
        !allow_partial_copy) {
        return 0;
    }

    eee_skipBufsFromStart(&edesc, numBytesToSkip, &bufIdx, &memptr);

    current_bd = &edesc->bd[bufIdx];

    while ((remain > 0) && (current_bd != NULL)) {
        int current_bd_size = eee_remainBufLen(current_bd, memptr);
        if (current_bd_size >= remain) {
            memcpy(memptr, copy_from, remain);
            return total_bytes_to_copy;
        } else {
            memcpy(memptr, copy_from, current_bd_size);
            remain -= current_bd_size;
            copy_from += current_bd_size;
        }
        current_bd = eee_nextBuf(&edesc, current_bd);
        if (current_bd != NULL)
            memptr = current_bd->buf;
    }
    return total_bytes_to_copy - remain;
}

/*------------------------------------------------------------------------------
 * Name:  eee_rcvContigData()
 * Description:
 *     eee_rcvContigData() is a function that given a eDesc will return a
 *     pointer to a contiguous buffer of the data that it contains.  If the
 *     data within the edesc fits with in the edesc's first buf then the
 *     pointer returned will be within the edesc, else the pointer will point
 *     to an allocated buffer containing the data from the packet in it.  To
 *     skip a given amount of bytes from the start of the data in the edesc,
 *     set minExpectedDataLenAfterSkip to a non-zero value.
 *
 * Arguments:
 *     eDesc - Get a pointer to contiguous data held in this packet.
 *     numBytesToSkip - Point the returned pointer to data within the eDesc
 *                      starting after this many bytes.
 *     flatBufferAllocated - This is set to TRUE if the returned pointer points
 *                           to an allocated buffer rather within the edesc bufs.
 * Return Values:
 *     - NULL - If no memory was available to allocate a buffer.
 *     - Pointer to contiguous data contained in the inputted edesc.
 *     - boolean pointed to by flatBufferAllocated is set to TRUE if buffer was allocated.
 */
void *
eee_rcvContigData(eee_desc_t *eDesc, uint32 numBytesToSkip, boolean *flatBufferAllocated)
{
    int pktlen = EEE_GET_PKT_LEN(eDesc);
    void *data;
    int bufIdx;
    int dataSize = pktlen - numBytesToSkip;

    *flatBufferAllocated = FALSE;
    // ASSERT(numBytesToSkip < pktlen);

    if (EEE_GET_NUM_BUFS(eDesc) > 1)    {
        data = eee_ramAlloc(dataSize);

        if (data != NULL) {
            int32 size unused__ = eee_copyPacketData(data,
                                                     eDesc,
                                                     numBytesToSkip,
                                                     dataSize,
                                                     FALSE);
            // ASSERT(size == dataSize);
            *flatBufferAllocated = TRUE;
        }
        return data;
    } else {
        eee_skipBufsFromStart(&eDesc,
                              numBytesToSkip,
                              &bufIdx,
                              (uchar8 **)&data);
        return data;
    }
}

/*
 * Routine Description:
    This routine copies bytes from a system buffer into a flat memory area.

 * Arguments:
    dest - The destination buffer.
    edesc - The descriptor to copy.
    numBytes - This starts as the number of bytes to copy from the descriptor
               to dest.  If there is more data in the packet to copy than
               this value we will either return an error, or adjust this
               size.  truncOK controls this behavior.
    truncOK - TRUE if it's OK to increase the size of numBytes to include
              all of the data in the descriptor'.  FALSE will
              result in an error if there is more data to copy than
              numBytes.

 * Return Value:
    NFX_OK if arguments are correct, otherwise EEE_ERR_PACKET_COPY_FAILED.
 */
int32
eee_copyPacketToLocalBuffer(void *dest, eee_descPtr_t edesc, uint32 numBytes, boolean truncOK)
{
    uint32 tmpBytes = numBytes;
    int rc = eee_verifyCopyArgs(dest, edesc, 0, &tmpBytes, truncOK);
    if (rc == NFX_OK) {
        eee_descPtr_t *tmpDesc = &edesc;
        uint8 *memptr;
        int bufIdx;
        eee_skipBufsFromStart(tmpDesc, 0, &bufIdx, &memptr);
        rc = eee_copyPacketDataFrom(dest, *tmpDesc, tmpBytes, bufIdx, memptr);
    }
    return rc;
}

/*-----------------------------------------------------------------
 * Name:        eee_createPacketFromLocalBuffer
 *
 * Description: Creates a system buffer and copies bytes from a flat
 *              memory buffer into it
 *-----------------------------------------------------------------
 */
eee_descPtr_t
eee_createPacketFromLocalBuffer(uchar8  * buf,      //Local buffer
    uint32  len,        //Local buffer length
    uint32  memPool,    //Memory pool to use for the system buffer
    uint32  bufType     //System buffer type to use (EEE_BUF_SMALL/LARGE/ULTRA)
)
{
    eee_descPtr_t edesc, first_edesc;
    eee_descBuf_t   * bd;
    uint32 save_len = len;
    uint32 off = 0, num_bufs;
    uint32 bufSize = eee.buf_size[bufType];

    // ASSERT(bufType == EEE_BUF_SMALL || bufType == EEE_BUF_LARGE || bufType == EEE_BUF_ULTRA);

    first_edesc = edesc = eee_allocateDesc(memPool, NFX_COS_HIGH);
    if (!edesc)
        return edesc;

    while (edesc) {
        num_bufs = 0;
        bd = &(edesc->bd[1]);
        while (len) {
            bd->buf = eee_allocateBuffer(memPool, bufType, NFX_COS_HIGH);
            if (!bd->buf) {
                eee_freePacket(first_edesc);
                return NULL;
            }
            bd->control = EDESC_BD_CMD_IN_USE | bufType;
            if (len > bufSize) {
                bd->len = bufSize;
                len -= bufSize;
            }
            else {
                bd->len = len;
                len = 0;
            }
            memcpy(bd->buf, buf+off, bd->len);
            off += bd->len;

            num_bufs++;
            if (num_bufs == EEE_MAX_BUFS - 1)
                break;
            else
                bd++;
        }
        edesc->hdr.control = 0;
        EEE_SET_NUM_BUFS(edesc, num_bufs);

        if (len) {
            edesc->hdr.next = eee_allocateDesc(memPool, NFX_COS_HIGH);
            if (!edesc->hdr.next)   {
                eee_freePacket(first_edesc);
                return NULL;
            }
        }
        else
            edesc->hdr.next = 0;
        edesc = edesc->hdr.next;
    }
    EEE_SET_PKT_LEN(first_edesc, save_len);
    return first_edesc;
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 * Argument:
 * bufPtr => address within buffer to be released
 *-----------------------------------------------------------------
 */
#if defined(EEE_VALIDATE)
void
eee_deallocateBuffer(void    *bufPtr)
{
    address_t   ra = (address_t)get_ra();
    eee_deallocateBufferRA(bufPtr, ra);
}
#endif

/*-----------------------------------------------------------------
 * Name:    eee_isUltraBuf()
 * Description: From an address within a buffer, figure out if its
 *              an ULTRA buf and return its base address.
 *-----------------------------------------------------------------
 */
boolean
eee_isUltraBuf(void *bufptr, void **buf_base)
{
    int32   pool_id;

    for (pool_id = 0; pool_id < EEE_NUM_MEM_POOLS; pool_id++) {
        if (bufptr >= eee.buf_base[pool_id][EEE_BUF_ULTRA] &&
            bufptr < eee.buf_end[pool_id][EEE_BUF_ULTRA]) {
            *buf_base = (void *)eee_buf_base(bufptr, pool_id, EEE_BUF_ULTRA);
            return TRUE;
        }
    }
    return FALSE;
}

uint32
eee_getAvailableNbrOfDesc()
{
    uint32  mem;

    GET_MY_MEMPOOL_INDEX(mem);
    return (eee.desc_pool[mem].stack.num_avail);
}

/*
 *    uint32  buf_size       EEE_BUF_SMALL, EEE_BUF_LARGE, EEE_BUF_ULTRA
 */
uint32
eee_getLWMBufs(uint32  buf_size)
{
    return (eee_stats->low_watermark_buf[eee.my_pool_id - SHARED_POOL_OFFSET][buf_size]);
}

#ifdef NFX_MOD_NFP
uint64
eee_getTotalSizeOfLargeBufsOnTxrx()
{
    uint64  num;

    num = (address_t)eee.buf_end[EEE_TXRX_POOL-SHARED_POOL_OFFSET][EEE_BUF_LARGE] -
        ((address_t)eee.buf_base[EEE_TXRX_POOL-SHARED_POOL_OFFSET][EEE_BUF_LARGE]);

    return num;
}
#endif

/*-----------------------------------------------------------------
 * Name:    eee_initDeallocateSharedBuffers()
 *
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_initDeallocateSharedBuffers(uint32  pass)
{
    uchar8  * bufptr, * trash;
    uint32  pool_id, i;
    uint32  wd_kick_ctr = 0;
    int testdebugflag=0;
    uint64 addr;

    EEE_PRINTK(("[eee_initDeallocateSharedBuffers], pass = %d\n", pass));

    pool_id = eee.my_pool_id - SHARED_POOL_OFFSET;

    for (i=0; i<EEE_NUM_BUF_SIZES; i++) {
        if (eee.sm_num_buf[i]) {
         if (pass == 0) {
            trash = (uchar8 *)eee_allocateAlignedBlock(eee.buf_size[i]);
	    EEE_PRINTK(("initDeallocatesharedBuffers, trash = %p\n", trash));

            bufptr = (uchar8 *)eee_allocateBlock((eee.sm_num_buf[i] + 1) * eee.buf_size[i]);
            bufptr += eee.buf_size[i] - (uint32)(address_t)bufptr % eee.buf_size[i];

            eee.buf_base[pool_id][i] = bufptr;
            eee.buf_end[pool_id][i] = bufptr + (eee.sm_num_buf[i] * eee.buf_size[i]); 
	    EEE_PRINTK(("EEE.BUF_BASE[%d][%d] = %p\n", pool_id, i, eee.buf_base[pool_id][i]));
	    EEE_PRINTK(("EEE.BUF_END[%d][%d] = %p\n", pool_id, i, eee.buf_end[pool_id][i]));
        } else if (pass == 1) {

#ifdef NFX_SMP
    eee_local_pool_init(&local_buf_pool[i], 
                        &eee.buf_pool[pool_id][i].stack,
                        EEE_BUF_POOL_SLAB_SIZE);
#endif
                testdebugflag=0;
                
		for (bufptr = eee.buf_base[pool_id][i];
                     (bufptr + eee.buf_size[i]) <= (uchar8 *)eee.buf_end[pool_id][i];
                     bufptr += eee.buf_size[i]) {

		  if ((wd_kick_ctr++ & 0xFFF) == 0)
		    watchdog_kick();
                    if (testdebugflag) {
                        testdebugflag = 0;
                        EEE_PRINTK(("next  %p %p \n", bufptr, bufptr + eee.buf_size[i]));
                    }

                    /* The Fibre Channel controller cannot handle DMA transfers
                     * that cross a 4GB boundary (upper 32 bits of address are
                     * fixed and not part of DMAs counter).
                     * If a buffer is going to cross a 4GB boundary, round up
                     * the start address to the 4GB boundary.
                     */
                    addr = (uint64)bufptr;

                    if (((addr + eee.buf_size[i]) ^ addr) & 0xffffffff00000000) {
                        /* start and end addresses for this buffer differ in the
                         * upper 32 bits, round start address up.  But don't use
                         * this buffer at all if it is the last one and rounding
                         * the start address up runs past the available space. 
                         */
                        EEE_PRINTK(("Crossed 4GB %p %llx pool %d size %d \n",
				   bufptr, addr + eee.buf_size[i], pool_id, i));

                        bufptr = (uchar8 *) ((uint64)(bufptr + eee.buf_size[i]) & 0xffffffff00000000);
                        testdebugflag = 1;

                        if((bufptr  + eee.buf_size[i]) > (uchar8 *)eee.buf_end[pool_id][i]) {
                            /* No more room, all done */
                            break;
                        }
                        
                    }

		    if (spans_dram_blocks((address_t)bufptr, eee.buf_size[i])) {
		      EEE_PRINTK(("unusable buffer, %p, phys_base = %lx, phys_end = %lx\n",
				 bufptr,
				 virt_to_phys((address_t)bufptr),
				 virt_to_phys((address_t)bufptr+eee.buf_size[i])));
		      continue;
		    }
		    eee_deallocateBuffer(bufptr);
            }
          }
        }
    }
    EEE_PRINTK(("[eee_initDeallocateSharedBuffers], leaving\n"));
}

/*-----------------------------------------------------------------
 * Name:    eee_initDeallocateSharedDesc()
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_initDeallocateSharedDesc(uint32  pass)
{
    eee_descPtr_t   edesc;
    uint32          pool_id;

    EEE_PRINTK(("[eee_initDeallocateSharedDesc] pass = %d\n", pass));

    pool_id = eee.my_pool_id - SHARED_POOL_OFFSET;

    if (pass == 0) {
    eee.desc_base[pool_id] = (eee_descPtr_t)eee_allocateBlock(
                                    eee.sm_num_desc * sizeof(eee_desc_t) + 1);
    eee.desc_base[pool_id] = (eee_descPtr_t)
        (((address_t)eee.desc_base[pool_id] + sizeof(eee_desc_t) - 1)
         & ~(sizeof(eee_desc_t) - 1));
    eee.desc_end[pool_id] = eee.desc_base[pool_id] + eee.sm_num_desc;
    EEE_PRINTK(("EEE.DESC_BASE[%d] = %p\n", pool_id, eee.desc_base[pool_id]));

    EEE_PRINTK(("EEE.DESC_END[%d] = %p\n", pool_id, eee.desc_end[pool_id]));

    } else if (pass == 1) {

    watchdog_kick();
#ifdef NFX_SMP
    eee_local_pool_init(&local_desc_pool,
                        &eee.desc_pool[pool_id].stack, 
                        EEE_DESC_POOL_SLAB_SIZE);
#endif

      for (edesc = eee.desc_base[pool_id]; edesc < eee.desc_end[pool_id]; edesc++) {
        eee_deallocateDesc(edesc);
      }
    }
    EEE_PRINTK(("[eee_initDeallocateSharedDesc], leaving\n"));
}

/*-----------------------------------------------------------------
 * Name:    eee_initTailPointers()
 * Description: Write initial values into tail pointers in remote memory pools
 *-----------------------------------------------------------------
 */
void
eee_initTailPointers()
{
    uint32  i, j, my_index;

    GET_MY_MEMPOOL_INDEX(my_index);

    EEE_PRINTK(("[eee_initTailPointers]\n"));
    for (i=EEE_NUM_LM_POOLS; i<EEE_NUM_MEM_POOLS; i++) {
        if (i == my_index)
            continue;
        if (eee.shmem_avail[i-EEE_NUM_LM_POOLS]) {
            EEE_PRINTK(("writing %p to %p\n", eee.desc_pool[i].queue.end-4,
                        eee.desc_pool[i].queue.tail_ptr));
            *(eee.desc_pool[i].queue.tail_ptr) = eee.desc_pool[i].queue.end-4;

            for (j=0; j<EEE_NUM_BUF_SIZES; j++) {
                if (eee.buf_pool[i][j].queue.head == eee.buf_pool[i][j].queue.end) {
                EEE_PRINTK(("writing 0 to %p\n", eee.buf_pool[i][j].queue.tail_ptr));
                *(eee.buf_pool[i][j].queue.tail_ptr) = 0;
		} else {
                EEE_PRINTK(("writing %p to %p\n",
                            eee.buf_pool[i][j].queue.end-4,
                            eee.buf_pool[i][j].queue.tail_ptr));
                *(eee.buf_pool[i][j].queue.tail_ptr) = eee.buf_pool[i][j].queue.end-4;
		}
	    } /* end of for */
        }
        else
            printk("shmem pool %d unavailable\n", i-EEE_NUM_LM_POOLS);
    }
    for (i=0; i<EEE_NUM_FWD_QUEUES; i++) {
        if (i == eee.my_fwd_id)
            continue;
#if 0
        if (eee.shmem_avail[eee_index[2][i] - EEE_NUM_LM_POOLS]) {
#else
        if (eee.shmem_avail[eee_index[2][i]]) {
#endif
            EEE_PRINTK(("I'm now writing %p to %p\n",
                       eee.rcv_queue[i].rcv_end-1,
                       eee.rcv_queue[i].rcv_tail_ptr));
            *(eee.rcv_queue[i].rcv_tail_ptr) = eee.rcv_queue[i].rcv_end-1;
        }
    }
    EEE_PRINTK(("[eee_initTailPointers:done]\n"));

}   /* eee_initTailPointers() */

/*-----------------------------------------------------------------
 * Name:    eee_descDisplay()
 * Description: 
 *-----------------------------------------------------------------
 */
void
eee_descDisplay(eee_descPtr_t   edesc)
{
    printk("edesc at 0x%p\n", edesc);
    printk("      control   = 0x%08x\n", edesc->hdr.control);
    printk("      dest_port = 0x%08x\n", edesc->hdr.dest_port);
    printk("      src_port  = 0x%08x\n", edesc->hdr.src_port);
    printk("      attr      = %p\n", (void *)edesc->hdr.attr);
    printk("      next      = %p\n", edesc->hdr.next);
    printk("      offset    = %x\n", edesc->hdr.offset);
    printk("      bd[0]     = 0x%04x 0x%04x %p\n", edesc->bd[0].control, edesc->bd[0].len, edesc->bd[0].buf);
    printk("      bd[1]     = 0x%04x 0x%04x %p\n", edesc->bd[1].control, edesc->bd[1].len, edesc->bd[1].buf);
    printk("      bd[2]     = 0x%04x 0x%04x %p\n", edesc->bd[2].control, edesc->bd[2].len, edesc->bd[2].buf);
    printk("      bd[3]     = 0x%04x 0x%04x %p\n", edesc->bd[3].control, edesc->bd[3].len, edesc->bd[3].buf);
}

/*-----------------------------------------------------------------
 * Name:
 *
 * Description:
 * Get number of bytes in this buffer type
 *-----------------------------------------------------------------
 */
int32
eee_getBufSize(int32 buf_type)
{
    return (eee.buf_size[buf_type]);
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 * EEE_BUF_SMALL, EEE_BUF_LARGE, EEE_BUF_ULTRA, -1 on error
 *-----------------------------------------------------------------
 */
int32
eee_getBufType(void *_bufptr)
{
    int32   _pool,_size;

    for (_pool=0; _pool<EEE_NUM_MEM_POOLS; _pool++) {
        for (_size=0; _size<EEE_NUM_BUF_SIZES; _size++) {
            if ((_bufptr >= eee.buf_base[_pool][_size]) && (_bufptr < eee.buf_end[_pool][_size]))
                return _size;
        }
    }
    return -1;
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 * Return buffer base address or -1
 *-----------------------------------------------------------------
 */
void *
eee_getBufBase(void *buf)
{
    address_t   base;
    uint32      index;
    int32   _pool,_size;

    for (_pool=0; _pool<EEE_NUM_MEM_POOLS; _pool++) {
        for (_size=0; _size<EEE_NUM_BUF_SIZES; _size++) {
            if ((buf >= eee.buf_base[_pool][_size]) && (buf < eee.buf_end[_pool][_size])) {
                if (_size == EEE_BUF_SMALL)
                    return (void *)((address_t)(buf) & eee.buf_mask[EEE_BUF_SMALL]);
                else if (_size == EEE_BUF_LARGE) {
                    base = (address_t)eee.buf_base[(_pool)][EEE_BUF_LARGE];
                    index = (uint32)((address_t)(buf) - base) / (EEE_BUFSIZE_LARGE);
                    return (void *)(base + (index * EEE_BUFSIZE_LARGE));
                } else {
                    base = (address_t)eee.buf_base[(_pool)][EEE_BUF_ULTRA];
                    index = (uint32)((address_t)(buf) - base) / (EEE_BUFSIZE_ULTRA);
                    return (void *)(base + (index * EEE_BUFSIZE_ULTRA));
                }
            }
        }
    }

    return (void *)-1;
}

int32
spans_dram_blocks(address_t   start,uint32      len)
{
#ifdef SIBYTE
    if ((virt_to_phys(start+len - 1) - virt_to_phys(start)) == (len - 1))
        return 0;
    else
        return 1;
#else
    return 0;
#endif
}

address_t get_pc()
{
    return get_ra();
}

uint32
eee_getPktNumBufs(eee_descPtr_t   ed)
{
    uint32  nb = 0;
    while (ed) {
        nb += EEE_GET_NUM_BUFS(ed);
        ed = ed->hdr.next;
    }
    return nb;
}

/*
 * Routine Description:
    Validate buffer pointer and length.

 * Arguments:
    ptr - a pointer 
    ptr_size - the size of the memory to check starting from pointer
    ra - caller's address

 * Return Value:
    Returns TRUE if the pointer and length are valid, otherwise printks.
 */
boolean
eee_buf_valid(const void *ptr, uint32 ptr_size, address_t ra)
{
#if !defined(NFP_TXRX) && !defined(NFP_FP)
    /* @@@ add this for other cpus
     */
    return TRUE;
#else
    address_t buf_base = 0;
    uint64 buf_size = (uint64)-1;
    boolean rc;

    if (mm_isMallocBuffer(ptr)) {
        uint32 buf_size;
        rc = mm_isSizeOkay(ptr, ptr_size, &buf_size);
        if (!rc) {
            printk("eee_buf_valid failed, malloc buffer %p sz %d ra=%lx\n",
                  ptr,
                  ptr_size,
                  ra);
        }
        return rc;
    }

    if (eee_isRemoteCachedAddr((address_t)ptr)) {
        ptr = (void *)cached_to_uncached((address_t)ptr);
    }

    if ((eee.buf_base[EEE_MY_SHARED_POOL][EEE_BUF_SMALL] <= ptr) &&
        (ptr < eee.buf_end[EEE_MY_SHARED_POOL][EEE_BUF_SMALL])) {
        buf_base = (address_t)ptr & ~(EEE_BUFSIZE_SMALL - 1);
        buf_size = EEE_BUFSIZE_SMALL;
    }
    else if ((eee.buf_base[EEE_MY_SHARED_POOL][EEE_BUF_LARGE] <= ptr) &&
        (ptr < eee.buf_end[EEE_MY_SHARED_POOL][EEE_BUF_LARGE])) {
        buf_base = (address_t)ptr -
            (uint32)(address_t)ptr % EEE_BUFSIZE_LARGE;
        buf_size = EEE_BUFSIZE_LARGE;
    }
    else if ((eee.buf_base[EEE_MY_SHARED_POOL][EEE_BUF_ULTRA] <= ptr) &&
        (ptr < eee.buf_end[EEE_MY_SHARED_POOL][EEE_BUF_ULTRA])) {
        buf_base = (address_t)ptr -
            (uint32)(address_t)ptr % EEE_BUFSIZE_ULTRA;
        buf_size = EEE_BUFSIZE_ULTRA;
    }
#if defined(NFP_FP) || defined(NFP_TXRX)
    else 
        if ((eee.buf_base[REMOTE_SHARED_POOL][EEE_BUF_SMALL] <= ptr) &&
            (ptr < eee.buf_end[REMOTE_SHARED_POOL][EEE_BUF_SMALL])) {
                buf_base = (address_t)ptr & ~(EEE_BUFSIZE_SMALL - 1);
                buf_size = EEE_BUFSIZE_SMALL;
            }
        else if ((eee.buf_base[REMOTE_SHARED_POOL][EEE_BUF_LARGE] <= ptr) &&
            (ptr < eee.buf_end[REMOTE_SHARED_POOL][EEE_BUF_LARGE])) {
                buf_base = (address_t)ptr -
                    (uint32)(address_t)ptr % EEE_BUFSIZE_LARGE;
                buf_size = EEE_BUFSIZE_LARGE;
            }
        else if ((eee.buf_base[REMOTE_SHARED_POOL][EEE_BUF_ULTRA] <= ptr) &&
            (ptr < eee.buf_end[REMOTE_SHARED_POOL][EEE_BUF_ULTRA])) {
                buf_base = (address_t)ptr -
                    (uint32)(address_t)ptr % EEE_BUFSIZE_ULTRA;
                buf_size = EEE_BUFSIZE_ULTRA;
            }
#endif

    rc = ((buf_base <= (address_t)ptr) &&
          ((address_t)ptr + ptr_size <= buf_base + buf_size));
    if (!rc) {
        printk("eee_buf_valid failed, eee buffer %p base=%p size=%ld ra=%lx\n",
              ptr,
              buf_base,
              buf_size,
              ra);
    }
    return rc;
#endif    
}

#ifdef NFX_SMP
/*-----------------------------------------------------------------
 * Name: eee_allocateNBuffers
 *
 * Description:	Allocates system buffers from the free pools
 *		Like the descriptor allocation routine, for optimal
 *		performance, the calling function should know where
 *		the buffer is to be ultimately used.
 * uint32	mempool => mem_pool_id
 * uint16	buf_size => EEE_BUF_SMALL, EEE_BUF_LARGE, EEE_BUF_ULTRA
 * uint32	cos => NFX_COS_HIGH, NFX_COS_LOW
 *-----------------------------------------------------------------
 */
uint32
eee_allocateNBuffers(uint32 mempool, uint16 buf_size, uint32 cos, void ** buf, uint32 req_cnt)
{
#ifdef EEE_VALIDATE
    address_t ra = get_ra();
#endif
    // ASSERT(mempool == EEE_SHARED_POOL);
    struct eee_local_stack_pool *pool = &local_buf_pool[buf_size];
    int i;
    for (i = 0; i < req_cnt; ++i, ++buf) {
        void *alloc_buf = eee_alloc_buf_ra(pool, ra);
        if (alloc_buf != NULL) {
            *buf = alloc_buf;
        } else {
            break;
        }
    }
    return i;
}


/*-----------------------------------------------------------------
 * Name:	eee_allocateNDesc
 * Description:	Allocate multiple EEE descriptors.
 *		For optimal performance, the caller should know where
 *		the packet will be destined.  This knowledge will
 *		allow the allocation routine to get the descriptor
 *		from the best pool, according to the module's architecture.
 *-----------------------------------------------------------------
 */
uint32
eee_allocateNDesc(uint32 mempool, uint32 cos unused__, eee_descPtr_t * desc, uint32 req_cnt)
{
#ifdef EEE_VALIDATE
    address_t ra = get_ra();
#endif
    // ASSERT(mempool == EEE_SHARED_POOL);
    struct eee_local_stack_pool *pool = &local_desc_pool;
    int i;
    for (i = 0; i < req_cnt; ++i, ++desc) {
        void *alloc_buf = eee_alloc_buf_ra(pool, ra);
        if (alloc_buf != NULL) {
            *desc = alloc_buf;
        } else {
            break;
        }
    }
    return i;
}
#endif /* FP */

/*
 *------------------------------------------------------------------------------
 * Function Name:   eee_splitDesc
 * Description:	    Splits an edesc into 2 parts. The original edesc will
 *		    contain the specified length and the remaining data
 *		    will be returned in the new edesc
 *		    Code partly borrowed from fs_restoreMoveEdescBufs()
 * Returns:	    New edesc containing remaining data, 0 on error
 *------------------------------------------------------------------------------
 */
eee_desc_t*
eee_splitDesc(eee_desc_t *edesc, uint32	len)
{
	eee_desc_t*	newEdesc;
	eee_desc_t*	origDesc = edesc;
	uchar8*		data;
	uint32		pktLen = EEE_GET_PKT_LEN(edesc);
	int			destBufIdx = 0;
	int srcBufIdx, useFirst, i;
	uint32 bufLen;

	// NFX_ASSERT(len <= pktLen);
	if (len == pktLen)
		return 0;
			
    eee_skipBufsFromStart(&edesc, len, &srcBufIdx, &data);

    useFirst = edesc_firstBufferIdx(edesc);

	/* If the data ends exactly at a buffer boundary,
	 * then no copying is required.
	 *
	 */
	if (srcBufIdx == useFirst && data == (uchar8 *)edesc->bd[srcBufIdx].buf)
	{
		newEdesc = edesc;
		edesc = origDesc;
    	eee_skipBufsFromStart(&edesc, len - 1, &srcBufIdx, &data);
		edesc->hdr.next = NULL;
	} else {
		/* allocate a new edesc
		 */
		newEdesc = eee_allocateDesc(EEE_LOCAL_POOL, NFX_COS_HIGH);
		if (!newEdesc) {
			printk("eee_splitDesc: failed to allocate edesc\n");
			return 0;
		}
		memset(newEdesc, 0, sizeof(eee_desc_t));
		newEdesc->hdr.src_port = origDesc->hdr.src_port;
		newEdesc->hdr.dest_port = origDesc->hdr.dest_port;
		
		/* If the data ends exactly at a buffer boundary,
		 * no need to copy any part of that buffer
		 */
		if (data != edesc->bd[srcBufIdx].buf)
		{
			
			/* copy the partial buffer data into a new edesc
			 * and set the bd lengths correctly
			 */
			newEdesc->bd[0].buf = eee_allocateBuffer(EEE_LOCAL_POOL, EEE_BUF_ULTRA, NFX_COS_HIGH);
			if (newEdesc->bd[0].buf == NULL) {
				printk("eee_splitDesc: failed to allocate ultra buf\n");
				eee_deallocateDesc(newEdesc);
				return 0;
			}
			bufLen = (address_t)data - (address_t)edesc->bd[srcBufIdx].buf;
			newEdesc->bd[0].len = edesc->bd[srcBufIdx].len - bufLen;
			edesc->bd[srcBufIdx].len = bufLen;

			memcpy(newEdesc->bd[0].buf, data, newEdesc->bd[0].len);
			newEdesc->hdr.control |= EDESC_COMMAND_USE_FIRST_BUFFER;

			srcBufIdx++;
		}
		destBufIdx++;

		/* copy any remaining bd's in the src edesc
		 */
    	for (i = srcBufIdx; i < useFirst + EEE_GET_NUM_BUFS(edesc); i++) {
        	    newEdesc->bd[destBufIdx++] = edesc->bd[i];
    	}
		EEE_SET_NUM_BUFS(edesc, srcBufIdx - useFirst);
		EEE_SET_NUM_BUFS(newEdesc, i - srcBufIdx + 
				 ((newEdesc->hdr.control & EDESC_COMMAND_USE_FIRST_BUFFER) ? 1:0));
		/*
	 	 * If this is an edesc chain, then chain the remaining desc to
	 	 * the destination
		 */
		newEdesc->hdr.next = edesc->hdr.next;
	 	edesc->hdr.next = NULL;
	}

	/* set the length field correctly in the 2 descriptors
	 */
	EEE_SET_PKT_LEN(newEdesc, pktLen - len);
	EEE_SET_PKT_LEN(origDesc, len);
	return newEdesc;
}

/*
 * Routine Description:
    Calculate the descriptor length in buffers.
    
 * Arguments:
    edesc - EEE descriptor

Return Values:
    Returns the length of the data in buffers.
 */
int 
edesc_calc_len(eee_desc_t *edesc)
{
    int len = 0;
    int i;
    struct edesc_buf *bd = edesc_firstBuf(edesc);
    for (i = 0; i < EEE_GET_NUM_BUFS(edesc); ++i, ++bd) {
        len += bd->len;
        // ASSERT(bd->control & EDESC_BD_CMD_IN_USE);
    }
    return len;
}

/*
 * Routine Description:
    Sanity check on the iterator state. 
        
 * Arguments:
    it - iterator

Return Values:
    Returns TRUE otherwise printk's.
 */
boolean
edesc_iter_verify(struct edesc_iter *it)
{
   int i, num_bufs;
   struct edesc_buf *bd;

    if (edesc_iter_valid(it)) {
        bd = edesc_firstBuf(it->edesc);
        num_bufs = EEE_GET_NUM_BUFS(it->edesc);

        // ASSERT(it->last_bd == (bd + num_bufs));
        for (i = 0; i < num_bufs; ++i, ++bd) {
            if (bd == it->bd) {
                // ASSERT(bd->len >= it->buf_remain);
                // ASSERT((bd->buf + (bd->len - it->buf_remain)) == it->buf);
                return TRUE;
            }
        }
        // ASSERT(i != num_bufs);
        return FALSE;
    } else {
        return TRUE;
    }
}

/*
 * Routine Description:
    Calculate the iterator position from the head of the list.
    
 * Arguments:
    edesc - EEE descriptor
    it - iterator

Return Values:
    Returns the offset of the current position from the head of the list in 
    bytes.  
 */
int
edesc_iter_pos(eee_desc_t *edesc, struct edesc_iter *it)
{
    int pos = 0;

    if (!edesc_iter_valid(it)) {
        while (edesc != NULL) {
            pos += edesc_calc_len(edesc);
            edesc = edesc->hdr.next;
        }
    } else {
        while (edesc != NULL) {
            struct edesc_buf *bd = edesc_firstBuf(edesc);
            int i;
            for (i = 0; i < EEE_GET_NUM_BUFS(edesc); ++i, ++bd) {
                if (bd == it->bd) {
                    // ASSERT(it->edesc == edesc);
                    return pos + ((uint8 *)it->buf - bd->buf);
                }
                pos += bd->len;
            }
            edesc = edesc->hdr.next;
        }
    }
    return pos;
}

/*
 * Routine Description:
    Initialize the EEE descriptor list iterator.
    
 * Arguments:
    it - iterator
    edesc - the head of the list

Return Values:
    Returns TRUE if there is valid data in the list.
 */
boolean
edesc_iter_init(struct edesc_iter *it, eee_desc_t *edesc)
{
    it->edesc = edesc;
    if (edesc != NULL) {
        it->bd = edesc_firstBuf(edesc);
        it->last_bd = it->bd + EEE_GET_NUM_BUFS(edesc);
        it->buf = it->bd->buf;
        it->buf_remain = it->bd->len;
    }

    // ASSERT(edesc_iter_verify(it));
    // ASSERT(edesc_iter_pos(edesc, it) == 0);

    return edesc_iter_valid(it);
}

/*
 * Routine Description:
    Advance the current position forward by num_bytes.
    
 * Arguments:
    it - iterator
    num_bytes - number of bytes to advance by.

Return Values:
    Returns TRUE if there is more valid data in the descriptor list,
    returns FALSE if advancing by the given number of bytes moves the position
    past the end of the data. 
 */
boolean
edesc_iter_advance(struct edesc_iter *it, int num_bytes)
{
    int start_pos unused__ = edesc_iter_pos(it->edesc, it);
    eee_desc_t *start_desc unused__ = it->edesc;
    int num_bytes_remain;
    int len_this_buf;

    if (num_bytes == 0) {
        return edesc_iter_valid(it);
    }

    for (num_bytes_remain = num_bytes; 
        (num_bytes_remain > 0) && edesc_iter_valid(it);
        num_bytes_remain -= len_this_buf) {
        if (it->buf_remain <= num_bytes_remain) {
            len_this_buf = it->buf_remain;
            it->bd++;
            if (it->bd == it->last_bd) {
                edesc_iter_init(it, it->edesc->hdr.next);
            } else {
                it->buf_remain = it->bd->len;
                it->buf = it->bd->buf;
            }
        } else {
            len_this_buf = num_bytes_remain;
            it->buf_remain -= len_this_buf;
            it->buf += len_this_buf;
        }
    }

    // ASSERT(edesc_iter_verify(it));
    // ASSERT(edesc_iter_valid(it) || (edesc_iter_pos(start_desc, it) == (start_pos + num_bytes)));

    return edesc_iter_valid(it);
}

/*
 * Routine Description:
    Make a EEE descriptor containing enough buffers to hold num_bytes of data.
    Only single descriptor will be allocated.

 * Arguments:
    desc_pool - pool to use for the descriptor allocation
    buf_pool - pool to use for the buffer allocation
    buf_size - pool size to use for the buffer allocation
    buf_len - the length of the buffers in this pool
    num_bytes - number of bytes to allocate

Return Values:
    Returns pointer to allocated descriptor or NULL if not enough memory.
 */
eee_desc_t *
eee_make_desc(int desc_pool,
              int buf_pool,
              int buf_size,
              int buf_len,
              int num_bytes)
{
    eee_desc_t *edesc = eee_allocateDesc(desc_pool, NFX_COS_HIGH);

    // VERIFY(num_bytes <= (buf_len * EEE_MAX_BUFS));

    if (edesc != NULL) {
        struct edesc_buf *bd;
        int num_bytes_remain;

        edesc->hdr.control = EDESC_COMMAND_USE_FIRST_BUFFER;
        EEE_SET_PKT_LEN(edesc, num_bytes);

        for (bd = &edesc->bd[0],
             num_bytes_remain = num_bytes; 
            num_bytes_remain > 0; 
            ++bd, num_bytes_remain -= buf_len) {
            bd->buf = eee_allocateBuffer(buf_pool, buf_size, NFX_COS_HIGH);
            if (bd->buf != NULL) {
                bd->len = MIN(num_bytes_remain, buf_len);
                bd->control = EDESC_BD_CMD_IN_USE;
            } else {
                eee_freePacket(edesc);
                return NULL;
            }
        }

        EEE_SET_NUM_BUFS(edesc, bd - &edesc->bd[0]);

        // ASSERT(EEE_GET_PKT_LEN(edesc) == edesc_calc_len(edesc));
        // ASSERT(EEE_GET_PKT_LEN(edesc) == num_bytes);
    }

    return edesc;
}

/*
 * Routine Description:
    Compare the contents of two EEE descriptors.
 * Arguments:
    edesc_1 - EEE descriptor 1
    edesc_2 - EEE descriptor 2
    len - number of bytes to compare starting from beginning
Return Values:
    Similar to memcmp():

    < 0 - edesc_1 < edesc_2
    > 0 - edesc_1 > edesc_2
    == 0 - edesc_1 == edesc_2
 */
int
edesc_cmp(eee_desc_t *edesc_1, eee_desc_t *edesc_2, int len)
{
    int cmp_this_time, len_remain;
    struct edesc_iter iter_1, iter_2;
    int res = 0;

    for (len_remain = len,
         edesc_iter_init(&iter_1, edesc_1),
         edesc_iter_init(&iter_2, edesc_2);
        ((len_remain > 0) && 
         (res == 0) && 
         edesc_iter_valid(&iter_1) &&
         edesc_iter_valid(&iter_2));
        edesc_iter_advance(&iter_1, cmp_this_time),
        edesc_iter_advance(&iter_2, cmp_this_time),
        len_remain -= cmp_this_time) {
        cmp_this_time = MIN(iter_1.buf_remain, iter_2.buf_remain);
        res = memcmp(iter_1.buf, iter_2.buf, cmp_this_time);
    }
    return res;
}

/*
 * Routine Description:
    Find out the buffer size.

 * Arguments:
    buf - pointer into the buffer

 * Return Value:
    EEE_BUFSIZE_SMALL
    EEE_BUFSIZE_LARGE
    EEE_BUFSIZE_ULTRA
    -1 - not a buffer
 */
int
eee_buf_size(void *buf)
{
    int i,j;

    for (i = 0; i < array_len(eee.buf_base); ++i) {
        for (j = 0; j < array_len(eee.buf_base[i]); ++j) {
            if ((eee.buf_base[i][j] <= buf) && 
                (buf < eee.buf_end[i][j])) {
                return j;
            }
        }
    }
    return -1;
}

/*
 * Routine Description:
    Calculate the space occupied by the descriptor buffers.
 * Arguments:
    edesc - EEE descriptor
 * Return Value:
 *  Returns the space occupied by the descriptor buffers in bytes or -1 on error.
 */
int
edesc_calc_space(eee_desc_t *edesc)
{
    eee_desc_t *it;
    struct edesc_buf *bd, *last_bd;
    int total = 0;

    for (it = edesc; it != NULL; it = it->hdr.next) {
        for (bd = edesc_firstBuf(it),
              last_bd = bd + EEE_GET_NUM_BUFS(it);
              bd < last_bd; ++bd) {
            int space;

            switch (eee_buf_size(bd->buf)) {

            case EEE_BUF_ULTRA:
                space = EEE_BUFSIZE_ULTRA;
                break;

            case EEE_BUF_LARGE:
                space = EEE_BUFSIZE_LARGE;
                break;

            case EEE_BUF_SMALL:
                space = EEE_BUFSIZE_SMALL;
                break;
            default:
                return -1;
            }
            total += space;
        }
    }
    return total;
}

