/*
 *-----------------------------------------------------------------
 *
 * Name:    eee-desc.h
 *
 * RCS:  $Id: eee-desc.h,v 1.54 2002/12/03 23:17:09 maximk Exp $
 *
 * Copyright (C) 2001, Agile Storage Inc.
 *
 * Description: EEE descriptor typedef
 *
 * Created by:  Rick Lund
 *
 * Date Created:    4/9/01
 *
 *-----------------------------------------------------------------
 */

#ifndef _EEE_DESC_H
#define _EEE_DESC_H

#include "nfx-incl.h"

/*------------------------------------------------------------------*/
/*
 * Control word bit definitions
 */
/*------------------------------------------------------------------*/
#define EDESC_PKT_NO_DEALLOC            0x80000000
#define EDESC_COMMAND_MASK              0x70000000
#define EDESC_COMMAND_USE_FIRST_BUFFER  0x10000000
#define EDESC_COMMAND_CHAIN_LAST_BUFFER 0x20000000
#define EDESC_COMMAND_DONT_CONVERT      0x40000000
#define EDESC_COMMAND_REUSE_HEADER      0x08000000
#define EDESC_FREE_ROUTINE_INDEX        0x07000000
#define EDESC_FREE_ROUTINE_SHIFT        24
#define EDESC_BUFCNT_MASK               0x00F00000
#define EDESC_BUFCNT_SHIFT              20
#define EDESC_TOTAL_LEN                 0x000FFFFF
#define EDESC_MAX_PACKET_LEN            EDESC_TOTAL_LEN

/*------------------------------------------------------------------*/
/*
 * dest_port, src_port bit definitions
 */
/*------------------------------------------------------------------*/
#define EDESC2_DATA_TYPE_MASK   0xE0000000
#define EDESC2_DATA_TYPE_SHIFT  29
#define EDESC2_DATA_TYPE_NUM    8
#define EDESC2_PORT_MASK        0x1F000000
#define EDESC2_PORT_SHIFT       24
#define EDESC2_PORT_NUM         32
#define EDESC2_MCAST_IND        0x00800000
#define EDESC2_COS_MASK         0x00600000
#define EDESC2_COS_SHIFT        21
#define EDESC2_COS_NUM          4
#define EDESC2_CPU_MASK         0x001F0000
#define EDESC2_CPU_SHIFT        16
#define EDESC2_CPU_NUM          32
#define EDESC2_SLOT_MASK        0x0000F000
#define EDESC2_SLOT_SHIFT       12
#define EDESC2_SLOT_NUM         16
#define EDESC2_APP_MASK         0x00000FFF
#define EDESC2_APP_SHIFT        0
#define EDESC2_APP_NUM          4096

#define PACK_UNICAST_PORT_ADDRESS(_slot, _cpu, _app, _dtype, _cos, _port)   \
({                                                                          \
    uint32  _r;                                                             \
    _r =(((_dtype)  << EDESC2_DATA_TYPE_SHIFT)  |                           \
        ((_port)    << EDESC2_PORT_SHIFT)       |                           \
        ((_cos)     << EDESC2_COS_SHIFT)        |                           \
        ((_cpu)     << EDESC2_CPU_SHIFT)        |                           \
        ((_slot)    << EDESC2_SLOT_SHIFT)       |                           \
        ((_app)     << EDESC2_APP_SHIFT));                                  \
    _r;                                                                     \
})

/*------------------------------------------------------------------*/
/*
 * EEE Descriptor Format
 *
 * bits for the fields are defined above.
 */
/*------------------------------------------------------------------*/
#ifdef __long64
typedef struct edesc_hdr {
    uint32      control;
    uint32      offset;
    uint32      dest_port;
    uint32      src_port;
    void        * attr;
    struct edesc* next;
} eee_descHdr_t, * eee_descHdrPtr_t;
#else
typedef struct edesc_hdr {
    uint32      control;
    uint32      dest_port;
    uint32      src_port;
    uint32      attr;
    struct edesc *next;
    uint32      offset;
} eee_descHdr_t, * eee_descHdrPtr_t;
#endif

#ifdef LARGE_MEMORY
#define EEE_NUM_BUF_SIZES       4
#else
#define EEE_NUM_BUF_SIZES       3
#endif
#define EEE_BUF_SMALL           0
#define EEE_BUF_LARGE           1
#define EEE_BUF_ULTRA           2
#ifdef LARGE_MEMORY
#define EEE_BUF_MGMT            3
#endif

#define EEE_BUFSIZE_SMALL       512
#ifdef EEE_NEW_BUFSIZE
#define EEE_BUFSIZE_LARGE       1664
#define EEE_BUFSIZE_MGMT        1664
#else
//#define EEE_BUFSIZE_LARGE       2048
#define EEE_BUFSIZE_LARGE       1792
#define EEE_BUFSIZE_MGMT        EEE_BUFSIZE_LARGE
#endif
#define EEE_BUFSIZE_ULTRA       8704

typedef struct edesc_buf {
    ushort16    control;
#define EDESC_BD_CMD_MASK       0xFFF0
#define EDESC_BD_CMD_IN_USE     0x8000
#define EDESC_BD_CMD_NO_DEALLOC 0x4000
#define EDESC_BD_CMD_START_IN_USE 0x2000 /* The area between the buffer start
                                            and the current buffer pointer
                                            is used by another descriptor */
#define EDESC_BD_BUFSIZE_MASK   0x000F
#define EDESC_BD_SMALL          (EEE_BUF_SMALL)
#define EDESC_BD_LARGE          (EEE_BUF_LARGE)
#define EDESC_BD_ULTRA          (EEE_BUF_ULTRA)
    ushort16    len;
#ifdef __long64
    uint32      res_;
#endif
    uchar8      * buf;
} eee_descBuf_t, * eee_descBufPtr_t;

#ifdef __long64
#define EEE_MAX_BUFS    6
#else
#define EEE_MAX_BUFS    13
#endif

typedef struct edesc {
    eee_descHdr_t   hdr;
    eee_descBuf_t   bd[EEE_MAX_BUFS];
} eee_desc_t, * eee_descPtr_t;

#define EDESC_ADDR_MASK         0x3F

NFX_ASSERT_STATIC(sizeof(eee_desc_t) == 128);

#define EEE_MSG_WINDOW_UPDATE ELAYER_MSG_BASE

/*--------------------------------------------------------------------------------
 * Name:            EEE_WU_CHAR_SPACE/EEE_WU_MBUF_SPACE
 *
 * Description:     Convert a message length to the amount of char/mbuf space the message will use.
 *
 * Created by:      Tim Gardner
 *
 * Date Created:    15-Jul-2003
 *
 *--------------------------------------------------------------------------------
 */
#define EEE_WU_CHAR_SPACE(x)    (((x) >= 28) ? ((x)+20) : (28+20))
#define EEE_WU_MBUF_SPACE(x)    (((x)/1024 + 1) * 2048 + ((x)/1024 + 3) * 128)

/*-----------------------------------------------------------------
 * Name:    EEE_GET_PKT_LEN(edesc)
 *
 * Description:
 *
 * Created by:  Rick Lund
 *
 * Date Created:
 *
 *-----------------------------------------------------------------
 */
#define EEE_GET_PKT_LEN(_edesc) ((_edesc)->hdr.control & EDESC_TOTAL_LEN)

/*-----------------------------------------------------------------
 * Name:    EEE_SET_PKT_LEN(edesc, len)
 *
 * Description:
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/02/2002
 *
 *-----------------------------------------------------------------
 */
#define EEE_SET_PKT_LEN(_edesc, len)        \
((_edesc)->hdr.control = (((_edesc)->hdr.control & ~EDESC_TOTAL_LEN) | (len)))
    


/*-----------------------------------------------------------------
 * Name:    EEE_GET_NUM_BUFS(edesc)
 *
 * Description:
 *
 * Created by:  Rick Lund
 *
 * Date Created:    11/10/01
 *
 *-----------------------------------------------------------------
 */
#define EEE_GET_NUM_BUFS(_edesc)                    \
    (((_edesc)->hdr.control & EDESC_BUFCNT_MASK) >> EDESC_BUFCNT_SHIFT)


/*-----------------------------------------------------------------
 * Name:    EEE_SET_NUM_BUFS(edesc, nbufs)
 *
 * Description:
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/02/2002
 *
 *-----------------------------------------------------------------
 */
#define EEE_SET_NUM_BUFS(_edesc, nbufs)     \
((_edesc)->hdr.control = ((_edesc)->hdr.control & ~EDESC_BUFCNT_MASK) \
    | ((nbufs) << EDESC_BUFCNT_SHIFT))

    
/*-----------------------------------------------------------------
 * Name:    EEE_GET_SLOT_NUM(dest_prot)
 *
 * Description: returns slot id from the dest_port word
 *
 * Created by:  Rick Lund
 *
 * Date Created:    5/19/01
 *
 *-----------------------------------------------------------------
 */
#define EEE_GET_SLOT_NUM(_dp) \
    (((_dp) & EDESC2_SLOT_MASK) >> EDESC2_SLOT_SHIFT)

/*-----------------------------------------------------------------
 * Name:    EEE_GET_DEST_CPU(dest_port)
 *
 * Description: returns major_id from the dest_port word
 *
 * Created by:  Rick Lund
 *
 * Date Created:    4/30/01
 *
 *-----------------------------------------------------------------
 */
#define EEE_GET_DEST_CPU(_dp) \
    (((_dp) & EDESC2_CPU_MASK) >> EDESC2_CPU_SHIFT)

/*-----------------------------------------------------------------
 * Name:    EEE_GET_APP_ID(dest_port)
 *
 * Description: returns the minor_id from the dest_port word
 *
 * Created by:  Rick Lund
 *
 * Date Created:    4/30/01
 *
 *-----------------------------------------------------------------
 */
#define EEE_GET_APP_ID(_dp) \
    (((_dp) & EDESC2_APP_MASK) >> EDESC2_APP_SHIFT)

/*-----------------------------------------------------------------
 * Name:    EEE_GET_DTYPE(dest_port)
 *
 * Description:
 *
 * Created by:  Rick Lund
 *
 * Date Created:    5/20/01
 *
 *-----------------------------------------------------------------
 */
#define EEE_GET_DTYPE(_dp) \
    (((_dp) & EDESC2_DATA_TYPE_MASK) >> EDESC2_DATA_TYPE_SHIFT)


/*-----------------------------------------------------------------
 * Name:
 *
 * Description:
 *
 * Created by:  Rick Lund
 *
 * Date Created:    11/10/01
 *
 *-----------------------------------------------------------------
 */
#define EEE_GET_PORT_ID(_dp)                                    \
    (((_dp) & EDESC2_PORT_MASK) >> EDESC2_PORT_SHIFT)

#define EEE_SET_PORT_ID(_dp, _port)                             \
({                                                              \
    uint32  _r;                                                 \
    _r = (_dp) & ~EDESC2_PORT_MASK;                             \
    _r |= (_port << EDESC2_PORT_SHIFT);                         \
    _dp = _r;                                                   \
})


/*------------------------------------------------------------------*/

#define EEE_DEST_PORT_TO_MAC_ADDR(_da, _dest_port)      \
{                                                       \
    _da[0] = 0x00;                                      \
    _da[1] = 0x11;                                      \
    _da[2] = 0x22;                                      \
    _da[3] = 0x33;                                      \
    _da[4] = 0x44;                                      \
    _da[5] = 0x55;                                      \
}

#define EEE_DEST_PORT_TO_VLAN(_vlan, _dest_port)        \
{                                                       \
    _vlan = 0x1234;                                     \
}

/*------------------------------------------------------------------*/
/*
 * Memory Pool definitions for allocation routines
 */
#define EEE_LOCAL_POOL          0
#define EEE_SHARED_POOL         1

#if defined(NFX_MOD_NFP)
    #define SHARED_POOL_OFFSET      1
    #define EEE_TXRX_POOL           2
    #define EEE_FP_POOL             3
#endif  /* NFX_MOD_NFP */

#if defined(FCNIM)
    #define SHARED_POOL_OFFSET      4
    #define EEE_FC_POOL             5
#endif  /* FCNIM && CHEETAH */

#if defined(NFX_MOD_SSC)
    #define SHARED_POOL_OFFSET      5
    #define EEE_SSC_MGMT_POOL       6
    #define EEE_SSC_7000_POOL       7
#endif  /* NFX_MOD_SSC */

#if defined(PROM)
    #define SHARED_POOL_OFFSET      9
    #define EEE_PROM_POOL           10
#endif  /* PROM */


/*-----------------------------------------------------------------
 * Name:    GET_DESC_POOL_PTRS()
 *
 * Description: sets up primary and secondary pointers to
 *              eee_pool_cntl_t structures and error counters.
 *
 * Created by:  Rick Lund
 *
 * Date Created:    5/1/01
 *
 *-----------------------------------------------------------------
 */
#ifdef NFX_MOD_NFP
#define GET_DESC_POOL_PTRS(_mp, _pri, _sec, _epri, _esec, _is_q)        \
{                                                                       \
    if (((_mp) == EEE_FP_POOL) || ((_mp) == EEE_TXRX_POOL)) {           \
        (_pri) = &(eee.desc_pool[(_mp)-SHARED_POOL_OFFSET]);            \
        (_epri) = &(eee_stats->no_desc_avail[(_mp) -                    \
                                                SHARED_POOL_OFFSET]);   \
    }                                                                   \
    else if ((_mp) == EEE_LOCAL_POOL) {                                 \
        (_pri) = &(eee.desc_pool[0]);                                   \
        (_epri) = &(eee_stats->no_desc_avail[0]);                       \
    }                                                                   \
    else if ((_mp) == EEE_SHARED_POOL) {                                \
        (_pri) = &(eee.desc_pool[eee.my_pool_id-SHARED_POOL_OFFSET]);   \
        (_epri) = &(eee_stats->no_desc_avail[eee.my_pool_id -           \
                                                SHARED_POOL_OFFSET]);   \
    }                                                                   \
    else {                                                              \
        panic("Invalid memory pool, 0x%x\n", (_mp));                    \
    }                                                                   \
                                                                        \
    if (((_mp) == eee.my_pool_id) || ((_mp) == EEE_SHARED_POOL)) {      \
        (_sec) = NULL;                                                  \
    }                                                                   \
    else {                                                              \
        (_sec) = &(eee.desc_pool[eee.my_pool_id-SHARED_POOL_OFFSET]);   \
        (_esec) = &(eee_stats->no_desc_avail[eee.my_pool_id -           \
                                                SHARED_POOL_OFFSET]);   \
    }                                                                   \
                                                                        \
    if (((_mp) == EEE_LOCAL_POOL) || ((_mp) == EEE_SHARED_POOL) ||      \
                                            ((_mp) == eee.my_pool_id))  \
        (_is_q) = 0;                                                    \
    else                                                                \
        (_is_q) = 1;                                                    \
}
#elif (defined(FCNIM) && defined(CHEETAH)) || defined(PROM) || (defined(FCNIM) && defined(BOBCAT))
#define GET_DESC_POOL_PTRS(_mp, _pri, _sec, _epri, _esec, _is_q)        \
{                                                                       \
    if ((_mp) == EEE_LOCAL_POOL) {                                      \
        (_pri) = &(eee.desc_pool[0]);                                   \
        (_epri) = &(eee_stats->no_desc_avail[0]);                       \
    }                                                                   \
    else if (((_mp) == eee.my_pool_id) || ((_mp) == EEE_SHARED_POOL)) {     \
        (_pri) = &(eee.desc_pool[eee.my_pool_id-SHARED_POOL_OFFSET]);   \
        (_epri) = &(eee_stats->no_desc_avail[eee.my_pool_id -           \
                                                SHARED_POOL_OFFSET]);   \
    }                                                                   \
    else {                                                              \
        panic("Invalid memory pool 0x%x\n", (_mp));                     \
    }                                                                   \
    (_sec) = NULL;                                                      \
    (_is_q) = 0;                                                        \
}
#elif defined(NFX_MOD_SSC)
#define GET_DESC_POOL_PTRS(_mp, _pri, _sec, _epri, _esec, _is_q)        \
{                                                                       \
    if (((_mp) == EEE_SSC_7000_POOL) || ((_mp) == EEE_SSC_MGMT_POOL)) { \
        (_pri) = &(eee.desc_pool[(_mp)-SHARED_POOL_OFFSET]);            \
        (_epri) = &(eee_stats->no_desc_avail[(_mp) -                    \
                                                SHARED_POOL_OFFSET]);   \
    }                                                                   \
    else if ((_mp) == EEE_LOCAL_POOL) {                                 \
        (_pri) = &(eee.desc_pool[0]);                                   \
        (_epri) = &(eee_stats->no_desc_avail[0]);                       \
    }                                                                   \
    else if ((_mp) == EEE_SHARED_POOL) {                                \
        (_pri) = &(eee.desc_pool[eee.my_pool_id-SHARED_POOL_OFFSET]);   \
        (_epri) = &(eee_stats->no_desc_avail[eee.my_pool_id -           \
                                                SHARED_POOL_OFFSET]);   \
    }                                                                   \
    else {                                                              \
        panic("Invalid memory pool 0x%x\n", (_mp));                     \
    }                                                                   \
                                                                        \
    if (((_mp) == eee.my_pool_id) || ((_mp) == EEE_SHARED_POOL)) {      \
        (_sec) = NULL;                                                  \
    }                                                                   \
    else {                                                              \
        (_sec) = &(eee.desc_pool[eee.my_pool_id-SHARED_POOL_OFFSET]);   \
        (_esec) = &(eee_stats->no_desc_avail[eee.my_pool_id -           \
                                                SHARED_POOL_OFFSET]);   \
    }                                                                   \
                                                                        \
    if (((_mp) == EEE_LOCAL_POOL) || ((_mp) == EEE_SHARED_POOL) ||      \
                                            ((_mp) == eee.my_pool_id))  \
        (_is_q) = 0;                                                    \
    else                                                                \
        (_is_q) = 1;                                                    \
}
#endif

/*-----------------------------------------------------------------
 * Name:    GET_BUF_POOL_PTRS
 *
 * Description: sets up primary and secondary pointers to
 *              eee_pool_cntl_t structures and error counters.
 *
 * Created by:  Rick Lund
 *
 * Date Created:    5/1/01
 *
 *-----------------------------------------------------------------
 */
#ifdef NFX_MOD_NFP
#define GET_BUF_POOL_PTRS(_mp, _size, _pri, _sec, _epri, _esec, _is_q)  \
{                                                                       \
    if ((_size) >= EEE_NUM_BUF_SIZES)                                   \
        panic("Invalid buffer size, 0x%x\n", (_size));                  \
                                                                        \
    if (((_mp) == EEE_FP_POOL) || ((_mp) == EEE_TXRX_POOL)) {           \
        (_pri) = &(eee.buf_pool[(_mp)-SHARED_POOL_OFFSET][(_size)]);    \
        (_epri) = &(eee_stats->no_buf_avail[(_mp) -                     \
                                        SHARED_POOL_OFFSET][(_size)]);  \
    }                                                                   \
    else if ((_mp) == EEE_LOCAL_POOL) {                                 \
        (_pri) = &(eee.buf_pool[0][(_size)]);                           \
        (_epri) = &(eee_stats->no_buf_avail[0][(_size)]);               \
    }                                                                   \
    else if ((_mp) == EEE_SHARED_POOL) {                                \
        (_pri) = &(eee.buf_pool[eee.my_pool_id -                        \
                                        SHARED_POOL_OFFSET][(_size)]);  \
        (_epri) = &(eee_stats->no_buf_avail[eee.my_pool_id -            \
                                        SHARED_POOL_OFFSET][(_size)]);  \
    }                                                                   \
    else {                                                              \
        panic("Invalid memory pool, 0x%x\n", (_mp));                    \
    }                                                                   \
                                                                        \
    if (((_mp) == eee.my_pool_id) || ((_mp) == EEE_SHARED_POOL)) {      \
        (_sec) = NULL;                                                  \
    }                                                                   \
    else {                                                              \
        (_sec) = &(eee.buf_pool[eee.my_pool_id -                        \
                                        SHARED_POOL_OFFSET][(_size)]);  \
        (_esec) = &(eee_stats->no_buf_avail[eee.my_pool_id -            \
                                        SHARED_POOL_OFFSET][(_size)]);  \
    }                                                                   \
                                                                        \
    if (((_mp) == EEE_LOCAL_POOL) || ((_mp) == EEE_SHARED_POOL) ||      \
                                            ((_mp) == eee.my_pool_id))  \
        (_is_q) = 0;                                                    \
    else                                                                \
        (_is_q) = 1;                                                    \
}
#elif (defined(FCNIM) && defined(CHEETAH)) || defined(PROM) || (defined(FCNIM) && defined(BOBCAT))
#define GET_BUF_POOL_PTRS(_mp, _size, _pri, _sec, _epri, _esec, _is_q)  \
{                                                                       \
    if ((_size) >= EEE_NUM_BUF_SIZES)                                   \
        panic("Invalid buffer size, 0x%x\n", (_size));                  \
                                                                        \
    if ((_mp) == EEE_LOCAL_POOL) {                                      \
        (_pri) = &(eee.buf_pool[0][(_size)]);                           \
        (_epri) = &(eee_stats->no_buf_avail[0][(_size)]);               \
    }                                                                   \
    else if (((_mp) == eee.my_pool_id) || ((_mp) == EEE_SHARED_POOL)) {     \
        (_pri) = &(eee.buf_pool[eee.my_pool_id -                        \
                                        SHARED_POOL_OFFSET][(_size)]);  \
        (_epri) = &(eee_stats->no_buf_avail[eee.my_pool_id -            \
                                        SHARED_POOL_OFFSET][(_size)]);  \
    }                                                                   \
    else {                                                              \
        panic("Invalid memory pool 0x%x\n", (_mp));                     \
    }                                                                   \
    (_sec) = NULL;                                                      \
    (_is_q) = 0;                                                        \
}
#elif defined(NFX_MOD_SSC)
#define GET_BUF_POOL_PTRS(_mp, _size, _pri, _sec, _epri, _esec, _is_q)  \
{                                                                       \
    if ((_size) >= EEE_NUM_BUF_SIZES)                                   \
        panic("Invalid buffer size, 0x%x\n", (_size));                  \
                                                                        \
    if (((_mp) == EEE_SSC_7000_POOL) || ((_mp) == EEE_SSC_MGMT_POOL)) { \
        (_pri) = &(eee.buf_pool[(_mp)-SHARED_POOL_OFFSET][(_size)]);    \
        (_epri) = &(eee_stats->no_buf_avail[(_mp) -                     \
                                        SHARED_POOL_OFFSET][(_size)]);  \
    }                                                                   \
    else if ((_mp) == EEE_LOCAL_POOL) {                                 \
        (_pri) = &(eee.buf_pool[0][(_size)]);                           \
        (_epri) = &(eee_stats->no_buf_avail[0][(_size)]);               \
    }                                                                   \
    else if ((_mp) == EEE_SHARED_POOL) {                                \
        (_pri) = &(eee.buf_pool[eee.my_pool_id -                        \
                                        SHARED_POOL_OFFSET][(_size)]);  \
        (_epri) = &(eee_stats->no_buf_avail[eee.my_pool_id -            \
                                        SHARED_POOL_OFFSET][(_size)]);  \
    }                                                                   \
    else {                                                              \
        panic("Invalid memory pool 0x%x\n", (_mp));                     \
    }                                                                   \
                                                                        \
    if (((_mp) == eee.my_pool_id) || ((_mp) == EEE_SHARED_POOL)) {      \
        (_sec) = NULL;                                                  \
    }                                                                   \
    else {                                                              \
        (_sec) = &(eee.buf_pool[eee.my_pool_id -                        \
                                        SHARED_POOL_OFFSET][(_size)]);  \
        (_esec) = &(eee_stats->no_buf_avail[eee.my_pool_id -            \
                                        SHARED_POOL_OFFSET][(_size)]);  \
    }                                                                   \
                                                                        \
    if (((_mp) == EEE_LOCAL_POOL) || ((_mp) == EEE_SHARED_POOL) ||      \
                                            ((_mp) == eee.my_pool_id))  \
        (_is_q) = 0;                                                    \
    else                                                                \
        (_is_q) = 1;                                                    \
}
#endif

/*-----------------------------------------------------------------
 * Name:    CHECK_POOLS
 *
 * Description: Gets an entry from the buf pools or desc pools
 *
 * Created by:  Rick Lund
 *
 * Date Created:    5/9/01
 *
 *-----------------------------------------------------------------
 */
#define CHECK_POOLS(_ptr, _pri, _sec, _epri, _esec, _is_q)              \
{                                                                       \
    /*                                                                  \
     * Check primary pool                                               \
     */                                                                 \
    if (_is_q) {                                                        \
        _ptr = *_pri->queue.head;                                       \
        if (_ptr != NULL) {                                             \
            *_pri->queue.head = NULL;                                   \
            *(_pri->queue.tail_ptr) = _pri->queue.head;                 \
            _pri->queue.head++;                                         \
            if (_pri->queue.head == _pri->queue.end)                    \
                _pri->queue.head = _pri->queue.start;                   \
            goto got_it;                                                \
        }                                                               \
        else                                                            \
            (*_epri)++;                                                 \
    }                                                                   \
    else {                                                              \
        if (_pri->stack.num_avail) {                                    \
            _ptr = *(_pri->stack.top++);                                \
            _pri->stack.num_avail--;                                    \
            goto got_it;                                                \
        }                                                               \
        else                                                            \
            (*_epri)++;                                                 \
    }                                                                   \
                                                                        \
    /*                                                                  \
     * Check secondary (backup/spillover/etc) pool                      \
     */                                                                 \
    if (_sec) {                                                         \
        if (_sec->stack.num_avail) {                                    \
            _ptr = *_sec->stack.top++;                                  \
            _sec->stack.num_avail--;                                    \
            _is_q = 0;                                                  \
            goto got_it;                                                \
        }                                                               \
        else                                                            \
            (*_esec)++;                                                 \
    }                                                                   \
                                                                        \
    /*                                                                  \
     * otherwise, return NULL                                           \
     */                                                                 \
    _ptr = NULL;                                                        \
}

/*-----------------------------------------------------------------
 * Name:    eee_allocateDesc(uint32 mempool, uint32 cos)
 *
 * Description: Allocate an EEE descriptor.
 *
 * Created by:  Rick Lund
 *
 * Date Created:    4/9/01
 *
 *-----------------------------------------------------------------
 */
eee_descPtr_t       /* Returns a pointer to a desciptor, or NULL if no
                     * descriptors are available for the given cos
                     */
eee_allocateDesc(
    uint32  mempool,    /* mem_pool_id */
    uint32  cos         /* NFX_COS_HIGH, NFX_COS_LOW */
);

#if !defined(EEE_VALIDATE)
#define eee_allocateDescRA(m, c, ra) eee_allocateDesc(m, c)
#else
eee_desc_t *eee_allocateDescRA(uint32 mempool, uint32 cos, address_t ra);
#endif


/*-----------------------------------------------------------------
 * Name:    eee_deallocateDesc(eee_descPtr_t edesc)
 *
 * Description: Releases an EEE descriptor back into the free pool.
 *
 * Created by:  Rick Lund
 *
 * Date Created:    4/9/01
 *
 *-----------------------------------------------------------------
 */
void
eee_deallocateDesc(
    eee_descPtr_t   edesc       /* pointer to descriptor to free */
);

#if !defined(EEE_VALIDATE)
#define eee_deallocateDescRA(desc, ra) eee_deallocateDesc(desc)
#else
void eee_deallocateDescRA(eee_desc_t *desc, address_t ra);
#endif


void
eee_deallocateBufferRA(
    void    *bufPtr,        /* address within buffer to be released */
    address_t   ra
);

#if !defined(EEE_VALIDATE)
#define eee_deallocateBufferRA(buf, ra) eee_deallocateBuffer(buf)
#endif

/*-----------------------------------------------------------------
 * Name:    eee_freePacket(eee_descPtr_t edesc)
 *
 * Description: Free an EEE descriptor and the associated buffers
 *
 * Created by:  Rick Lund
 *
 * Date Created:    4/9/01
 *
 *-----------------------------------------------------------------
 */
void
eee_freePacket(
    eee_descPtr_t   edesc       /* pointer to packet to free */
);

/*-------------------------------------------------------------------------------
 * Name:  ee_copyBufferToPacket()
 *
 * Description:
 *
 *     This function copies a flat buffer into a (chained) edesc after skipping
 *     a given number of bytes.  It has the option of copying only part of the
 *     flat buffer if the edesc is not large enough to hold the entire buffer.
 *
 * Arguments:
 *
 *     buf - This is the flat buffer to be copied into the edesc.
 *     bufSize - This is the size of buf.
 *     edesc - This is the edesc to be copied into.
 *     numBytesToSkip - This is the number of bytes to skip within the edesc to
 *                      start copying buf into.
 *     allow_partial_copy - If this is FALSE then this function will only copy
 *                          buf into edesc if it is possible to copy all of buf
 *                          into edesc.  If this is TRUE then as much of buf
 *                          will be copied into edesc as there is space.
 *
 * returns:
 *
 *     Number of bytes copied.
 *
 * Created by:     Ian Brown
 *
 * Date Created:   10/06/03
 *------------------------------------------------------------------------------
 */
uint32
eee_copyBufferToPacket(const void *buf,
                       int bufSize,
                       eee_desc_t *edesc,
                       uint32 numBytesToSkip,
                       boolean allow_partial_copy);

/*------------------------------------------------------------------------------
 * Name:  eee_rcvContigData()
 *
 * Description:
 *
 *     eee_rcvContigData() is a function that given a eDesc will return a
 *     pointer to a contiguous buffer of the data that it contains.  If the
 *     data within the edesc fits with in the edesc's first buf then the
 *     pointer returned will be within the edesc, else the pointer will point
 *     to an allocated buffer containing the data from the packet in it.  To
 *     skip a given amount of bytes from the start of the data in the edesc,
 *     set minExpectedDataLenAfterSkip to a non-zero value.
 *
 * Arguments:
 *
 *     eDesc - Get a pointer to contiguous data held in this packet.
 *     numBytesToSkip - Point the returned pointer to data within the eDesc
 *                      starting after this many bytes.
 *     flatBufferAllocated - This is set to TRUE if the returned pointer points
 *                           to an allocated buffer rather within the edesc bufs.
 *     minExpectedDataLenAfterSkip - If the edesc does not hold a minimum of
 *                                   this many bytes after numBytesToSkip bytes
 *                                   are skipped then NULL is returned.
 *
 * Returns:
 *
 *     - NULL - If no memory was available to allocate a buffer.
 *     - Pointer to contiguous data contained in the inputted edesc.
 *     - The boolean pointed to by flatBufferAllocated is set to TRUE if a
 *       buffer was allocated.
 *
 * Created by:  Ian Brown
 *
 * Date Created: 10/14/03
 *------------------------------------------------------------------------------
 */
void *
eee_rcvContigData(eee_desc_t *eDesc,
                  uint32 numBytesToSkip,
                  boolean *flatBufferAllocated);

/*++

Routine Description:

    This routine copies bytes from a system buffer into a flat memory area.

Arguments:

    dest - The destination buffer.
    edesc - The descriptor to copy.
    numBytes - This starts as the number of bytes to copy from the descriptor
               to dest.  If there is more data in the packet to copy than
               this value we will either return an error, or adjust this
               size.  truncOK controls this behavior.
    truncOK - TRUE if it's OK to increase the size of numBytes to include
              all of the data in the descriptor.  FALSE will
              result in an error if there is more data to copy than
              numBytes.

Return Value:

    NFX_OK if arguments are correct, otherwise EEE_ERR_PACKET_COPY_FAILED.

--*/

int32
eee_copyPacketToLocalBuffer(void *dest,
                            eee_descPtr_t edesc,
                            uint32 numBytes,
                            boolean truncOK);


/*-----------------------------------------------------------------
 * Name:        eee_createPacketFromLocalBuffer
 *
 * Description: Creates a system buffer and copies bytes from a flat
 *              memory buffer into it
 *
 * Created by:  Rick Lund
 *              Laurent Brard 
 *
 * Date Created:    04/23/03
 *
 *-----------------------------------------------------------------
 */
eee_descPtr_t
eee_createPacketFromLocalBuffer(
    uchar8  * buf,      //Local buffer
    uint32  len,        //Local buffer length
    uint32  memPool,    //Memory pool to use for the system buffer
    uint32  bufType     //System buffer type to use (EEE_BUF_SMALL/LARGE/ULTRA)
);

/*-----------------------------------------------------------------
 * Name:    eee_allocateBuffer(uint32 mempool, uint32 buf_size, uint32 cos)
 *
 * Description: Allocates a system buffer from the free pools
 *
 * Created by:  Rick Lund
 *
 * Date Created:    4/9/01
 *
 *-----------------------------------------------------------------
 */
void *              /* returns the buffer pointer, or NULL if none
                     * available from the given cos
                     */
eee_allocateBuffer(
    uint32  mempool,
    uint16  buf_size,       /* EDESC_BUFHDR_256,
                             * EDESC_BUFHDR_512,
                             * EDESC_BUFHDR_2048,
                             * EDESC_BUFHDR_8664
                             */
    uint32  cos             /* NFX_COS_HIGH, NFX_COS_LOW */
);

#if !defined(EEE_VALIDATE)
#define eee_allocateBufferRA(m, s, c, ra) eee_allocateBuffer(m, s, c)
#else
void *eee_allocateBufferRA(uint32 mempool, 
                           uint16 size, 
                           uint32 cos, 
                           address_t ra);
#endif



/*-----------------------------------------------------------------
 * Name:    eee_deallocateBuffer(void * bufPtr)
 *
 * Description: Return buffer to free pool
 *
 * Created by:  Rick Lund
 *
 * Date Created:    4/9/01
 *
 *-----------------------------------------------------------------
 */
void
eee_deallocateBuffer(
    void    *bufPtr         /* address within buffer to be released */
);

/*-----------------------------------------------------------------
 * Name:    eee_isUltraBuf()
 *
 * Description: From an address within a buffer, figure out if its
 *              an ULTRA buf and return its base address.
 *
 * Created by:  Bob Fozard
 *
 * Date Created:    2/20/02
 *
 *-----------------------------------------------------------------
 */
boolean
eee_isUltraBuf(
    void    *bufptr,
    void    **buf_base
);

#if defined(NETEEE_FRAGMENT)

/*-----------------------------------------------------------------
 * Name:    eee_fragmentOffset
 *
 * Description: Returns the offset of a fragment in the packet
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    2/20/02
 *
 *-----------------------------------------------------------------
 */
static inline uint32
eee_fragmentOffset(const agile_hdr_t *hdr)
{
    return hdr->offset & ~(AGILEHDR_MF | AGILEHDR_ID_MASK);
}


/*-----------------------------------------------------------------
 * Name:    eee_moreFragments
 *
 * Description: Returns true if there are more fragments following this
 * fragment.
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    2/20/02
 *
 *-----------------------------------------------------------------
 */
static inline uint32
eee_moreFragments(const agile_hdr_t *hdr)
{
    return hdr->offset & AGILEHDR_MF;
}

/*-----------------------------------------------------------------
 * Name:    edesc_fragmentOffset
 *
 * Description: Returns the offset of a fragment in the packet
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    2/20/02
 *
 *-----------------------------------------------------------------
 */
static inline uint32
edesc_fragmentOffset(const eee_descPtr_t edesc)
{
    return edesc->hdr.offset & ~(AGILEHDR_MF | AGILEHDR_ID_MASK);
}


/*-----------------------------------------------------------------
 * Name:    edesc_moreFragments
 *
 * Description: Returns true if there are more fragments following this
 * fragment.
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    2/20/02
 *
 *-----------------------------------------------------------------
 */
static inline uint32
edesc_moreFragments(const eee_descPtr_t edesc)
{
    return edesc->hdr.offset & AGILEHDR_MF;
}


/*-----------------------------------------------------------------
 * Name :   eee_seqId()
 *
 * Description: Return the packet sequence ID
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/11/02
 *
 *-----------------------------------------------------------------
 */
static inline uint16
eee_seqId(const agile_hdr_t *hdr)
{
    return (hdr->offset & AGILEHDR_ID_MASK) >> AGILEHDR_ID_SHIFT;
}


int eee_nextPacketId(void);


/*-----------------------------------------------------------------
 * Name :   edesc_seqId()
 *
 * Description: returns the packet sequence id
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/11/02
 *
 *-----------------------------------------------------------------
 */
static inline uint16
edesc_seqId(eee_descPtr_t desc)
{
    return (desc->hdr.offset & AGILEHDR_ID_MASK) >> AGILEHDR_ID_SHIFT;
}


#else

static inline int
edesc_fragmentOffset(eee_descPtr_t desc)
{
    return 0;
}


static inline int
edesc_moreFragments(eee_descPtr_t desc)
{
    return 0;
}


static inline int
eee_fragmentOffset(agile_hdr_t *desc)
{
    return 0;
}


static inline int
eee_moreFragments(agile_hdr_t *desc)
{
    return 0;
}

#endif


/*-----------------------------------------------------------------
 * Name :   eee_chainDesc()
 *
 * Description: add a new descriptor to the chain
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/19/02
 *
 *-----------------------------------------------------------------
 */
static inline void
eee_chainDesc(eee_descPtr_t desc, eee_descPtr_t chainDesc)
{
    desc->hdr.next = chainDesc;
}


/*-----------------------------------------------------------------
 * Name :   eee_nextDesc()
 *
 * Description: Returns the next descriptor in the chain or NULL if this
 * is the last descriptor
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/19/02
 *
 *-----------------------------------------------------------------
 */
static inline eee_descPtr_t
eee_nextDesc(eee_descPtr_t desc)
{
    return desc->hdr.next;
}


/*-----------------------------------------------------------------
 * Name :   eee_unlinkNextDesc()
 *
 * Description: unlink the rest of the chain from the descriptor
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/19/02
 *
 *-----------------------------------------------------------------
 */
static inline void
eee_unlinkNextDesc(eee_descPtr_t desc)
{
    desc->hdr.next = NULL;
}


/*-----------------------------------------------------------------
 * Name :   edesc_firstBufferIdx()
 *
 * Description: Return the index of the first buffer in the descriptor.
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/08/02
 *
 *-----------------------------------------------------------------
 */
static inline int
edesc_firstBufferIdx(const eee_descPtr_t edesc)
{
    return (edesc->hdr.control & EDESC_COMMAND_USE_FIRST_BUFFER) ? 0 : 1;
}


/*-----------------------------------------------------------------
 * Name :   edesc_firstBuf()
 *
 * Description: Return the first buffer descriptor in eee descriptor.
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/26/02
 *
 *-----------------------------------------------------------------
 */
static inline eee_descBuf_t *
edesc_firstBuf(const eee_descPtr_t desc)
{
    return &desc->bd[edesc_firstBufferIdx(desc)];
}


/*-----------------------------------------------------------------
 * Name :   eee_addHeader()
 *
 * Description: Add a header buffer containing agile header and
 * ethernet header to the descriptor for sending the packet through
 * the backplane.
 *
 * Returns: NFX_OK if succeeds
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/10/02
 *
 *-----------------------------------------------------------------
 */
int
eee_addHeader(eee_desc_t *pktDesc, int remotePort, uchar8 *myMacAddr);


/*-----------------------------------------------------------------
 * Name :   eee_removeHeader()
 *
 * Description: strip the mac header and agile header, fill the descriptor header.
 * Assumes the descriptor does not use the first buffer.
 *
 * Returns: the size of the removed header
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/23/02
 *
 *-----------------------------------------------------------------
 */
int
eee_removeHeader(eee_descPtr_t edesc);


/*-----------------------------------------------------------------
 * Name :   eee_skipBufs()
 *
 * Description: Skip len bytes from the current position in the
 * descriptor.  The position is specified by bufptr and memptr, which
 * are the current buffer index and the current pointer into the
 * buffer memory.  On return the locations specified by the arguments
 * descptr, bufptr and memptr will be updated to contain new
 * descriptor position.
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/23/02
 *
 *----------------------------------------------------------------- */
void
eee_skipBufs(eee_descPtr_t *descptr, int len, int *bufptr, uchar8 **memptr);


/*-----------------------------------------------------------------
 * Name :   eee_nextBuf()
 *
 * Description: Move the pointers to the next buffer. The new descriptor
 * is stored in the location specified by the argument edesc.
 *
 * Returns: new buffer descriptor or NULL is this was the last buffer
 * descriptor.
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/24/02
 *
 *-----------------------------------------------------------------
 */
eee_descBuf_t *
eee_nextBuf(eee_descPtr_t *edesc, eee_descBuf_t *bd);


/*-----------------------------------------------------------------
 * Name :   eee_skipBufsFromStart()
 *
 * Description: Skip len bytes from the beginning of descriptor. The
 * index of the buffer containing len + 1 byte will be stored into the
 * location specified by the argument buf, the pointer into the
 * buffer will be stored into the location specified by the argument
 * memptr.
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/04/02
 *
 *-----------------------------------------------------------------
 */
void
eee_skipBufsFromStart(eee_descPtr_t *descptr, int len, int *bufptr,
                      uchar8 **memptr);

/*-----------------------------------------------------------------
 * Name :   eee_remainBufLen()
 *
 * Description: return the number of bytes remaining available in the buffer
 * starting from memptr.
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/04/02
 *
 *-----------------------------------------------------------------
 */
int
eee_remainBufLen(eee_descBuf_t *bufdesc, uchar8 *memptr);

/*-----------------------------------------------------------------
 * Name :   eee_copyPacketData()
 *
 * Description: Copies bytes from a system buffer starting from the
 * given offset into a contiguous memory area.
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/23/02
 *
 *-----------------------------------------------------------------
 */
int32                           /* returns number of bytes copied,
                                 or EEE_ERR_PACKET_COPY_FAILED */
eee_copyPacketData(
    void *dest,                 /* destination address of local buffer */
    eee_descPtr_t edesc,        /* source descriptor */
    uint32 offset,              /* offset in the descriptor */
    uint32 num_bytes,           /* bytes to copy */
    uint32 truncate_ok          /* if non-zero, allow truncation */
    );

/*-----------------------------------------------------------------
 * Name :   eee_copyPacketDataFrom()
 *
 * Description: Copy num_bytes of descriptor data from the given
 * location to a local buffer. Location is specified by buffer index
 * and pointer into the buffer memory.
 *
 * Returns: number of bytes copied.
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    04/24/02
 *
 *----------------------------------------------------------------- */
int
eee_copyPacketDataFrom(
    void *dest,
    eee_descPtr_t edesc,
    int num_bytes,
    int bufidx,
    char *bufmem
    );

uint32
eee_getLWMBufs(
    uint32  buf_size        /* EEE_BUF_SMALL
                             * EEE_BUF_LARGE
                             * EEE_BUF_ULTRA
                             */
);

#ifdef NFX_MOD_NFP
uint64 
eee_getTotalSizeOfLargeBufsOnTxrx();
#endif

/*-----------------------------------------------------------------
 * Name :   eee_getBufferAllocAddr()
 *
 * Description: Return the address of function which allocated the
 * buffer.
 *
 * Created by:  Maxim Kozlovsky
 *
 * Date Created:    09/06/02
 *
 *-----------------------------------------------------------------
 */
address_t
eee_getBufferAllocAddr(
    void    *bufPtr     /* address within buffer */
);

/*-----------------------------------------------------------------
 * Name:
 *
 * Description:
 *
 * Created by:  Rick Lund
 *
 * Date Created:
 *
 *-----------------------------------------------------------------
 */
uint32
eee_getAvailableNbrOfDesc(void);

/*-----------------------------------------------------------------
 * Name:    eee_initDeallocateSharedBuffers()
 *
 * Description:
 *
 * Created by:  Rick Lund
 *
 * Date Created:    5/11/01
 *
 *-----------------------------------------------------------------
 */
void
eee_initDeallocateSharedBuffers(
    uint32  pass
);

/*-----------------------------------------------------------------
 * Name:    eee_initDeallocateSharedDesc()
 *
 * Description:
 *
 * Created by:  Rick Lund
 *
 * Date Created:
 *
 *-----------------------------------------------------------------
 */
void
eee_initDeallocateSharedDesc(
    uint32  pass
);

/*-----------------------------------------------------------------
 * Name:    eee_initTailPointers()
 *
 * Description: Write initial values into tail pointers in remote memory
 *              pools
 *
 * Created by:  Rick Lund
 *
 * Date Created:    5/16/01
 *
 *-----------------------------------------------------------------
 */
void
eee_initTailPointers(void);

/*-----------------------------------------------------------------
 * Name:    eee_descDisplay()
 *
 * Description:
 *
 * Created by:  Rick Lund
 *
 * Date Created:    12/11/01
 *
 *-----------------------------------------------------------------
 */
void
eee_descDisplay(
    eee_descPtr_t   edesc
);

/*-----------------------------------------------------------------
 * Name:
 *
 * Description:
 *
 * Created by:  Rick Lund
 *
 * Date Created:    3/2/02
 *
 *-----------------------------------------------------------------
 */
int32
eee_getBufSize(
    int32   buf_type        /* number of bytes in this buffer type */
);

/*-----------------------------------------------------------------
 * Name:
 *
 * Description:
 *
 * Created by:  Rick Lund
 *
 * Date Created:    3/2/02
 *
 *-----------------------------------------------------------------
 */
int32                   /* EEE_BUF_SMALL,
                         *  EEE_BUF_LARGE,
                         *  EEE_BUF_ULTRA,
                         *  -1 on error
                         */
eee_getBufType(
    void    * _bufptr
);

/*-----------------------------------------------------------------
 * Name:
 *
 * Description:
 *
 * Created by:  Rick Lund
 *
 * Date Created:    3/2/02
 *
 *-----------------------------------------------------------------
 */
void *                  /* buffer base address or -1 */
eee_getBufBase(
    void    * buf
);

/*-----------------------------------------------------------------
 * Name:
 *
 * Description:
 *
 * Created by:  Rick Lund
 *
 * Date Created:
 *
 *-----------------------------------------------------------------
 */
int32
spans_dram_blocks(
    address_t   start,
    uint32      len
);

/*-----------------------------------------------------------------
 * Name:
 *
 * Description:
 *
 * Created by:  Rick Lund
 *
 * Date Created:
 *
 *-----------------------------------------------------------------
 */
uint32
eee_getPktNumBufs(
    eee_descPtr_t   ed
);


int eee_forwardLocalPktUni(uint32 dest_port, eee_desc_t *edesc);

/*++

Routine Description:

    Validate buffer pointer and length.

Arguments:

    ptr - a pointer 
    ptr_size - the size of the memory to check starting from pointer
    ra - caller's address

Return Value:

    Returns TRUE if the pointer and length are valid.

--*/

boolean
eee_buf_valid(const void *ptr, uint32 ptr_size, address_t ra);


/*-----------------------------------------------------------------
 * Name:	eee_allocateNBuffers(uint32 buf_size, uint32 cos,
 *										void ** buf, uint32 req_cnt)
 *
 * Description:	Allocates system buffers from the free pools
 *				Like the descriptor allocation routine, for optimal
 *				performance, the calling function should know where
 *				the buffer is to be ultimately used.
 *
 * Created by:	Rick Lund
 *
 * Date Created:	2/21/03
 *
 *-----------------------------------------------------------------
 */
uint32
eee_allocateNBuffers(
	uint32	mempool,		/* mem_pool_id */
	uint16	buf_size,		/* EEE_BUF_SMALL
							 * EEE_BUF_LARGE
							 * EEE_BUF_ULTRA
							 */
	uint32	cos unused__,			/* NFX_COS_HIGH, NFX_COS_LOW */
	void	** buf,
	uint32	req_cnt
);


/*-----------------------------------------------------------------
 * Name:	eee_allocateNDesc(uint32 mempool, uint32 cos,
 *								eee_descPtr_t * desc, uint32 req_cnt)
 *
 * Description:	Allocate multiple EEE descriptors.
 *				For optimal performance, the caller should know where
 *				the packet will be destined.  This knowledge will
 *				allow the allocation routine to get the descriptor
 *				from the best pool, according to the module's
 *				architecture.
 *
 * Created by:	Rick Lund
 *
 * Date Created:	2/21/03
 *
 *-----------------------------------------------------------------
 */
uint32
eee_allocateNDesc(uint32 mempool,
                  uint32 cos unused__,
                  eee_descPtr_t * desc,
                  uint32 req_cnt);

/*
 *------------------------------------------------------------------------------
 * Function Name:   eee_splitDesc
 *
 * Description:		Splits an edesc into 2 parts. The original edesc will
 *					contain the specified length and the remaining data
 *					will be returned in the new edesc
 *					Code partly borrowed from fs_restoreMoveEdescBufs()
 *
 * Created by:      Nitin Bahadur
 *
 * Data Created:    11/04/04
 *
 * Returns:			new edesc containing remaining data
 *------------------------------------------------------------------------------
 */
eee_desc_t*
eee_splitDesc(
	eee_desc_t*		edesc,
	uint32			len);

/*-----------------------------------------------------------------
 * Name:		eee_tryFreeReceiveQueue()
 *
 * Description:		try to free buf or edesc entries in rcv_queue
 *
 * Date Created:	10/31/07
 *-----------------------------------------------------------------
 */
void
eee_tryFreeReceiveQueue(void   *cb,
                        uint32 tref);

/* Structure for iterating over lists of EEE descriptors.
 */
struct edesc_iter {
    /* The current descriptor.
     */
    eee_desc_t *edesc;

    /* The current buffer descriptor.
     */
    struct edesc_buf *bd;

    /* Next after last valid buffer descriptor.
     */
    struct edesc_buf *last_bd;

    /* Pointer to the current position in the current buffer.
     */
    void *buf;

    /* The number of bytes remaining in the current buffer from the current 
     * position.  
     */
    int buf_remain;
};


/*++

Routine Description:

    Calculate the descriptor length in buffers.
    
Arguments:

    edesc - EEE descriptor

Return Values:

    Returns the length of the data in buffers.

--*/

int 
edesc_calc_len(eee_desc_t *edesc);

/*++

Routine Description:

    Calculate the space occupied by the descriptor buffers.

Arguments:

    edesc - EEE descriptor

Return Value:

    Returns the space occupied by the descriptor buffers in bytes or 
    -1 on error.

--*/

int
edesc_calc_space(eee_desc_t *edesc);

/*++

Routine Description:

    Check if there is valid data after the current position.
    
Arguments:

    it - iterator

Return Values:

    Returns TRUE if there is valid data after the current position,
    otherwise returns FALSE.

--*/

static inline boolean
edesc_iter_valid(struct edesc_iter *it)
{
    return it->edesc != NULL;
}


/*++

Routine Description:

    Sanity check on the iterator state. 
        
Arguments:

    it - iterator

Return Values:

    Returns TRUE otherwise panics.

--*/

boolean
edesc_iter_verify(struct edesc_iter *it);


/*++

Routine Description:

    Calculate the iterator position from the head of the list.
    
Arguments:

    edesc - EEE descriptor
    it - iterator

Return Values:

    Returns the offset of the current position from the head of the list in 
    bytes.  

--*/

int
edesc_iter_pos(eee_desc_t *edesc,
               struct edesc_iter *it);


/*++

Routine Description:

    Initialize the EEE descriptor list iterator.
    
Arguments:

    it - iterator
    edesc - the head of the list

Return Values:

    Returns TRUE if there is valid data in the list.

--*/

boolean
edesc_iter_init(struct edesc_iter *it,
                eee_desc_t *edesc);


/*++

Routine Description:

    Advance the current position forward by num_bytes.
    
Arguments:

    it - iterator
    num_bytes - number of bytes to advance by.

Return Values:
    
    Returns TRUE if there is more valid data in the descriptor list,
    returns FALSE if advancing by the given number of bytes moves the position
    past the end of the data. 

--*/
boolean
edesc_iter_advance(struct edesc_iter *it, int num_bytes);


/*++

Routine Description:

    Make a EEE descriptor containing enough buffers to hold num_bytes of data.
    
    Only single descriptor will be allocated.

Arguments:

    desc_pool - pool to use for the descriptor allocation
    buf_pool - pool to use for the buffer allocation
    buf_size - pool size to use for the buffer allocation
    buf_len - the length of the buffers in this pool
    num_bytes - number of bytes to allocate

Return Values:

    Returns pointer to allocated descriptor or NULL if not enough memory.

--*/

eee_desc_t *
eee_make_desc(int desc_pool,
              int buf_pool,
              int buf_size,
              int buf_len,
              int num_bytes);


/*++

Routine Description:

    Compare the contents of two EEE descriptors.
    
Arguments:

    edesc_1 - EEE descriptor 1
    edesc_2 - EEE descriptor 2
    len - number of bytes to compare starting from beginning

Return Values:

    Similar to memcmp():

    < 0 - edesc_1 < edesc_2
    > 0 - edesc_1 > edesc_2
    == 0 - edesc_1 == edesc_2

--*/

int
edesc_cmp(eee_desc_t *edesc_1,
          eee_desc_t *edesc_2,
          int len);

/*++

Routine Description:

    Find out the buffer size.

Arguments:

    buf - pointer into the buffer

Return Value:

    EEE_BUFSIZE_SMALL
    EEE_BUFSIZE_LARGE
    EEE_BUFSIZE_ULTRA
    -1 - not a buffer

--*/

int
eee_buf_size(void *buf);

/*++

Routine Description:

    Increment reference count on the descriptor. Used only when 
    compiled with EEE_VALIDATE.

Arguments:

    edesc - EEE descriptor    

Return Value:

    Returns the descriptor.

--*/

eee_desc_t *
eee_desc_hold(eee_desc_t *edesc);

/*++

Routine Description:

    Release a reference on the descriptor. Used only when compiled
    with EEE_VALIDATE.

Arguments:

    edesc - EEE descriptor

Return Value:

    None.

--*/

void
eee_desc_rele(eee_desc_t *edesc);

/*++

Routine Description:

    Increment reference count on the buffer. Used only when 
    compiled with EEE_VALIDATE.

Arguments:

    buf - buffer

Return Value:

    Returns the buffer.

--*/

void *
eee_buf_hold(void *buf);

/*++

Routine Description:

    Release a reference on the buffer. Used only when compiled
    with EEE_VALIDATE.

Arguments:

    buf - buffer

Return Value:

    None.

--*/

void
eee_buf_rele(void *buf);


#ifndef EEE_VALIDATE
#define eee_desc_hold(e) (e)
#define eee_desc_rele(e)
#define eee_buf_hold(e) (e)
#define eee_buf_rele(e)
#endif

#endif  /* _EEE_DESC_H */
