/*
 *-----------------------------------------------------------------
 * Name:	eee-fwd.c
 * Description: Routines that handle the forward/receive queues
 * Copyright (C) 2009, OnStor, Inc.
 *-----------------------------------------------------------------
 */
#ifdef LATER
#include "nfx-incl.h"
#include "bqueue.h"
#include "../sm-eee/eee-api.h"
#include "../sm-chassis/cm-api.h"
#if defined(SSC) || defined(SSC_MGMT) || defined(FCNIM)
#include "../sm-malloc/malloc-api.h"
#else
#include "../sm-malloc-slab/malloc-api.h"
#endif
#include "../sm-hw/mem-api.h"
#include "../sm-timer/timer-api.h"
#if defined(NFP_TXRX)
#include "../sm-sbm-rev2/sbm-api.h"
#endif
#include "../sm-backplane/backplane-api.h"
#include "eee-inlines.h"
#ifdef SSC
#include "../sm-anpssc/anpssc-api.h"
#endif

#include "../sm-spinlock/spinlock-api.h"

#ifdef LINUX_TEST
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#ifdef LINUX_POLL_NONBLOCKING_FD
#include <sys/socket.h>
#include <sys/un.h>
#include <fcntl.h>
#ifdef DMALLOC
#include "dmalloc.h"
#endif
#else

#endif /* LATER */

/* rx polling indicator socket fd for this cpu */
int eee_linuxFwdPollRxFd;
/* tx polling indicator socket fd's for destinations */
int eee_linuxFwdPollTxFd[NFX_SLOT_MAX][NFX_MAX_CPUS_PER_SLOT];
/* state of tx polling indicator socket fd's for destinations */
int eee_linuxFwdPollTxFdState[NFX_SLOT_MAX][NFX_MAX_CPUS_PER_SLOT];
#endif
#endif

#if defined(NFX_SMP)
static void eee_processPacket(struct edesc *);
#endif

uint16 eee_packetId;	/* The packet sequence identifier */
uint64 ht_bytes_rx;

/*
Routine Description:
    Check if the slot is local. On Cougar, everything that is not SSC is on
    the same slot as FP/TXRX.

Arguments:
    slot - slot

Return Value:
    Returns true if the slot is local.
 */
inline int
eee_is_local_slot(int slot)
{
#ifdef COUGAR
    return (slot != NFX_SLOTTYPE_SSC);
#else
    return slot == nfxMySlotId;
#endif
}

/*

Routine Description:

    Check if the cpu is on this node. It is assumed the slot matches the current
    slot.
    
Arguments:

    cpu - cpu number, one of NFX_NFP_* or NFX_FCNIM_*

Return Value:

    Returns true if the cpu is on this node.

 */

int
eee_is_local_cpu(int cpu)
{
#if defined(NFP_FP)
    return (((NFX_NFP_FP1 <= cpu) && (cpu <= NFX_NFP_FP_LAST)) || 
            (cpu == NFX_NFP_FP_ANY));
#elif defined(NFP_TXRX)
    return (((NFX_NFP_TXRX1 <= cpu) && (cpu <= NFX_NFP_TXRX_LAST)) || 
            (cpu == NFX_NFP_TXRX_ANY));
#else
    return cpu == nfxMyCPUId;
#endif
}

/*

Routine Description:

    Check if the slot/cpu is on this node.

Arguments:

    slot - slot number, one of NFP_SLOTTYPE_*
    cpu - cpu number, one of NFX_NFP_* or NFX_FCNIM_*

Return Value:

    Returns true if the slot/cpu is on this node.

 */

boolean
eee_is_local_slot_cpu(int slot, int cpu)
{
#if defined(COUGAR) && defined(NFP_FP)
    /* On cougar FC is on the same node as FP.
     */
    return (((slot == NFX_SLOTTYPE_NFP) && eee_is_local_cpu(cpu)) ||
            (slot == NFX_SLOTTYPE_FC));
#else
    return ((slot == nfxMySlotId) && eee_is_local_cpu(cpu));
#endif
}

#if defined(NFX_SMP)

struct local_queue local_queue;

void
eee_receivePacket(struct edesc *desc)
{
	NFX_SPIN_LOCK(&local_queue.lq_spin);
	if (local_queue.lq_tail) {
		local_queue.lq_tail->hdr.next = desc;
	}
	else {
		local_queue.lq_head = desc;
	}
	local_queue.lq_tail = desc;
    local_queue.enqueued++;
	NFX_SPIN_UNLOCK(&local_queue.lq_spin);
}

void
eee_receiveChain(struct edesc *desc)
{
      NFX_SPIN_LOCK(&local_queue.lq_spin);
      if (local_queue.lq_tail) {
          /* Prevent descriptor from chaining to itself. This is for the
           * case when there is only one desc in the queue and rmc attempts
           * to resend the same desc. This happens when the TXRX N-CPU is
           * too busy to service the queue before rmc retries on TXRX A-CPU.
           */
          if (local_queue.lq_tail != desc) { 
              local_queue.lq_tail->hdr.next = desc;
          } else {
              NFX_SPIN_UNLOCK(&local_queue.lq_spin);
              return;
          }
      } else {
              local_queue.lq_head = desc;
      }
      while (desc->hdr.next != NULL) {
              desc = desc->hdr.next;
      }
      local_queue.lq_tail = desc;
      local_queue.enqueued++;
      NFX_SPIN_UNLOCK(&local_queue.lq_spin);
}

struct edesc *
eee_dequeue_packet(struct local_queue *queue)
{
	struct edesc *desc;
	NFX_SPIN_LOCK(&queue->lq_spin);
	desc = queue->lq_head;
	if (desc) {
		if (!desc->hdr.next) {
			queue->lq_tail = NULL;
		}
		queue->lq_head = desc->hdr.next;
		desc->hdr.next = NULL;
		queue->dequeued++;
	}
	NFX_SPIN_UNLOCK(&queue->lq_spin);
	return desc;
}

void
eee_poll_local_queue(void *cb, uint32 tref)
{
	struct local_queue *queue = cb;
	struct edesc *desc;
	uint32	cnt = 0;

#if defined(NFP_TXRX)
	/* txrx acpu does not poll local_queue */
	if (!CM_IS_TXRX_NCPU)
		return;
#endif /* NFP_TXRX */

	queue->num_called++;

	while ((cnt < queue->max_val) &&
						(desc = eee_dequeue_packet(queue))) {
		eee_processPacket(desc);
		cnt++;
	}
	if (cnt == 0)
		queue->poll_idle++;
	else {
		queue->poll_valid++;
		queue->rcv_cnt += cnt;
		if (cnt > queue->hwm)
			queue->hwm = cnt;
	}
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_fwdLocalQueuePollCfg(uint32	max_val)
{
	local_queue.max_val = max_val;
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_fwdLocalQueuePollView(uint32 just_hwm)
{
	NFX_SPIN_LOCK(&local_queue.lq_spin);

	if (just_hwm) {
		printk("LOCALQ: max_val     = %d\n", local_queue.max_val);
		printk("        HWM         = %d\n", local_queue.hwm);
	}
	else {
		printk("LOCALQ: called      = %lld\n", local_queue.num_called);
		printk("        max_val     = %d\n", local_queue.max_val);
		printk("        poll_idle   = %lld\n", local_queue.poll_idle);
		printk("        poll_valid  = %lld\n", local_queue.poll_valid);
		printk("        rcvd        = %lld\n", local_queue.rcv_cnt);
		printk("        enqueued    = %lld\n", local_queue.enqueued);
		printk("        dequeued    = %lld\n", local_queue.dequeued);
		printk("        HWM         = %d\n", local_queue.hwm);
		printk("        AVG         = %lld\n", local_queue.poll_valid ?
						local_queue.rcv_cnt / local_queue.poll_valid : 0);
	}

	NFX_SPIN_UNLOCK(&local_queue.lq_spin);
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_fwdLocalQueuePollSendToSSC(uint32 just_hwm, eee_poll_cfg_rsp_t *rsp, uint32 max_rsps, uint32 *num_added)
{
	uint32	num = 0;

	NFX_SPIN_LOCK(&local_queue.lq_spin);

	if (just_hwm) {
		if (num < max_rsps) {
			strncpy(rsp->str, "LOCALQ", POLLCFG_RSP_STR_LEN);
			rsp->cnt = POLLCFG_RSP_SECTION_LABEL;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "max_val", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queue.max_val;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "HWM", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queue.hwm;
			rsp++; num++;
		}
	}
	else {
		if (num < max_rsps) {
			strncpy(rsp->str, "LOCALQ", POLLCFG_RSP_STR_LEN);
			rsp->cnt = POLLCFG_RSP_SECTION_LABEL;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "called", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queue.num_called;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "max_val", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queue.max_val;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "poll_idle", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queue.poll_idle;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "poll_valid", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queue.poll_valid;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "rcvd", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queue.rcv_cnt;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "HWM", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queue.hwm;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "AVG", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queue.poll_valid ?
						local_queue.rcv_cnt / local_queue.poll_valid : 0;
			rsp++; num++;
		}
	}

	*num_added = num;

	NFX_SPIN_UNLOCK(&local_queue.lq_spin);
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_fwdLocalQueuePollClear()
{
	NFX_SPIN_LOCK(&local_queue.lq_spin);

	local_queue.num_called = 0;
	local_queue.poll_idle = 0;
	local_queue.poll_valid = 0;
	local_queue.hwm = 0;
	local_queue.rcv_cnt = 0;
	local_queue.enqueued = 0;
	local_queue.dequeued = 0;

	NFX_SPIN_UNLOCK(&local_queue.lq_spin);
}

eee_poll_cfg_t eee_fwdLocalQueuePollCfgStruct[] = {
	{"view",	eee_fwdLocalQueuePollView},
	{"clear",	eee_fwdLocalQueuePollClear},
	{"ssc",		eee_fwdLocalQueuePollSendToSSC},
	{"localq",	eee_fwdLocalQueuePollCfg},
	{NULL, 0}
};

#if defined(NFP_TXRX)
struct local_queue local_queueA; /* local queue from Txrx ACPU */

void
eee_receivePacketA(struct edesc *desc)
{
	//printk("give message to acpu, dp %x, sp %x\n",
	//	   desc->hdr.dest_port, desc->hdr.src_port);
	
	NFX_SPIN_LOCK(&local_queueA.lq_spin);
	if (local_queueA.lq_tail) {
		local_queueA.lq_tail->hdr.next = desc;
	} else {
		local_queueA.lq_head = desc;
	}
	local_queueA.lq_tail = desc;
	local_queueA.enqueued++;
	NFX_SPIN_UNLOCK(&local_queueA.lq_spin);
}

int
eee_poll_local_queueA(void *cb, uint32 tref)
{
	struct local_queue *queue = cb;
	struct edesc *desc;
	uint32	cnt = 0;

	queue->num_called++;

    if (queue->lq_head != NULL) {
        while ((cnt < tref) &&
               (desc = eee_dequeue_packet(queue))) {
            eee_processPacket(desc);
            cnt++;
        }
    }

        if (cnt == 0) {
		queue->poll_idle++;
	} else {
		queue->poll_valid++;
		queue->rcv_cnt += cnt;
		if (cnt > queue->hwm)
			queue->hwm = cnt;
	}

    return cnt;
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_fwdLocalQueueAPollClear()
{
	NFX_SPIN_LOCK(&local_queueA.lq_spin);

	local_queueA.num_called = 0;
	local_queueA.poll_idle = 0;
	local_queueA.poll_valid = 0;
	local_queueA.hwm = 0;
	local_queueA.rcv_cnt = 0;
	local_queueA.enqueued = 0;
	local_queueA.dequeued = 0;

	NFX_SPIN_UNLOCK(&local_queueA.lq_spin);
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_fwdLocalQueueAPollView(uint32 just_hwm)
{
	NFX_SPIN_LOCK(&local_queueA.lq_spin);

	if (just_hwm) {
		printk("LOCALQA: max_val    = %d\n", local_queueA.max_val);
		printk("         HWM        = %d\n", local_queueA.hwm);
	} else {
		printk("LOCALQA: called     = %lld\n", local_queueA.num_called);
		printk("         max_val    = %d\n", local_queueA.max_val);
		printk("         poll_idle  = %lld\n", local_queueA.poll_idle);
		printk("         poll_valid = %lld\n", local_queueA.poll_valid);
		printk("         rcvd       = %lld\n", local_queueA.rcv_cnt);
		printk("         enqueued   = %lld\n", local_queueA.enqueued);
		printk("         dequeued   = %lld\n", local_queueA.dequeued);
		printk("         HWM        = %d\n", local_queueA.hwm);
		printk("         AVG        = %lld\n", local_queueA.poll_valid ?
						local_queueA.rcv_cnt / local_queueA.poll_valid : 0);
	}

	NFX_SPIN_UNLOCK(&local_queueA.lq_spin);
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_fwdLocalQueueAPollSendToSSC(uint32	just_hwm,
	eee_poll_cfg_rsp_t	* rsp,
	uint32  max_rsps,
	uint32	* num_added)
{
	uint32	num = 0;

	NFX_SPIN_LOCK(&local_queueA.lq_spin);

	if (just_hwm) {
		if (num < max_rsps) {
			strncpy(rsp->str, "LOCALQA", POLLCFG_RSP_STR_LEN);
			rsp->cnt = POLLCFG_RSP_SECTION_LABEL;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "max_val", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queueA.max_val;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "HWM", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queueA.hwm;
			rsp++; num++;
		}
	}
	else {
		if (num < max_rsps) {
			strncpy(rsp->str, "LOCALQA", POLLCFG_RSP_STR_LEN);
			rsp->cnt = POLLCFG_RSP_SECTION_LABEL;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "called", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queueA.num_called;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "max_val", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queueA.max_val;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "poll_idle", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queueA.poll_idle;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "poll_valid", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queueA.poll_valid;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "rcvd", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queueA.rcv_cnt;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "HWM", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queueA.hwm;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "AVG", POLLCFG_RSP_STR_LEN);
			rsp->cnt = local_queueA.poll_valid ?
						local_queueA.rcv_cnt / local_queueA.poll_valid : 0;
			rsp++; num++;
		}
	}

	*num_added = num;

	NFX_SPIN_UNLOCK(&local_queueA.lq_spin);
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_fwdLocalQueueAPollCfg(uint32	max_val)
{
	local_queueA.max_val = max_val;
}

eee_poll_cfg_t eee_fwdLocalQueueAPollCfgStruct[] = {
	{"view",	eee_fwdLocalQueueAPollView},
	{"ssc",		eee_fwdLocalQueueAPollSendToSSC},
	{"clear",	eee_fwdLocalQueueAPollClear},
	{"localqa",	eee_fwdLocalQueueAPollCfg},
	{NULL, 0}
};

/*-----------------------------------------------------------------
 * Name :	eee_txrxFilterToAcpu
 *
 * Description: create the filter that determines whether to fwd pkt
 *		to txrx Acpu from ncpu
 *-----------------------------------------------------------------
 */
#define TXRX_MANAGEMENT_FILTER	0 /* manage plane filter  */
#define TXRX_FWDQ_FILTER		1 /* fwd queue filter */
#define TXRX_NET_FILTER			2 /* network traffic filter */

#define TXRX_FORWARD_ALL 1
#define TXRX_FORWARD_MATCH_CPU_ONLY 2

char	txrx_filters[3][256];		
void
eee_txrxMapAcpu()
{
	int i;
	
	bzero(txrx_filters, sizeof(txrx_filters));
	
	/* initilize management plane filter */
	for (i=0; i < 256; i++) {
		if (i == eee_getApplicationId("timekeeper")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] =
                TXRX_FORWARD_MATCH_CPU_ONLY;
			continue;
		}

		if (i == eee_getApplicationId("rmc")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}

		if (i == eee_getApplicationId("cifsauth")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}
		
		if (i == eee_getApplicationId("cifs_srvr")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}

		if (i == eee_getApplicationId("dm")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}

		if (i == eee_getApplicationId("mnt_srvr")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}

		if (i == eee_getApplicationId("nfs_cmd")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}

		if (i == eee_getApplicationId("nfs_srvr")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}

		if (i == eee_getApplicationId("cifs")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}

		if (i == eee_getApplicationId("nfs")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}

		if (i == eee_getApplicationId("nshare")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}
        
		if (i == eee_getApplicationId("cshare")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}

		if (i == eee_getApplicationId("audit")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}

		if (i == eee_getApplicationId("dcache")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}

		if (i == eee_getApplicationId("symlinkmap")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}

		if (i == eee_getApplicationId("author_authen")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}

		if (i == eee_getApplicationId("author_fs")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}

		if (i == eee_getApplicationId("file-server")) {
			txrx_filters[TXRX_MANAGEMENT_FILTER][i] = TXRX_FORWARD_ALL;
			txrx_filters[TXRX_FWDQ_FILTER][i] = TXRX_FORWARD_ALL;
			continue;
		}
	}
	
	return;
}

/*-----------------------------------------------------------------
 * Name :	eee_txrxIsPktToAcpu
 * Description: SMP txrx, it determines whether a recived pkt by 
 *		NCPU txrx should be forwarded to ACPU txrx.
 *		edescp is given to acpu if
 *		1. src_cpu is fp and dst_cpu is txrx 
 *		2. if src port is dp83820 and selected appId
 *-----------------------------------------------------------------
 */
int32	
eee_txrxIsPktToAcpu(eee_descPtr_t edesc)
{
	uint32	dp,sp, appId;
	
	dp = edesc->hdr.dest_port;
	sp = edesc->hdr.src_port;


	appId = EEE_GET_APP_ID(dp);
	
	/* msg from fp to txrx */
	if ((EEE_GET_SLOT_NUM(sp) == nfxMySlotId) && 
		( EEE_GET_SLOT_NUM(dp) == nfxMySlotId)) {

		if ((txrx_filters[TXRX_FWDQ_FILTER][appId] == TXRX_FORWARD_ALL)
            || ((txrx_filters[TXRX_FWDQ_FILTER][appId] ==
                 TXRX_FORWARD_MATCH_CPU_ONLY) &&
                (EEE_GET_DEST_CPU(dp) == NFX_NFP_TXRX2))) {
			return 1;
        }
	}
	else if (EEE_GET_SLOT_NUM(sp) != nfxMySlotId) {

		/* management msg */
		if ((txrx_filters[TXRX_MANAGEMENT_FILTER][appId] == TXRX_FORWARD_ALL)
            || ((txrx_filters[TXRX_MANAGEMENT_FILTER][appId] ==
                 TXRX_FORWARD_MATCH_CPU_ONLY) &&
                (EEE_GET_DEST_CPU(dp) == NFX_NFP_TXRX2))) {
			return 1;
        }
	}
	
	return 0;
}
#endif /* NFP_TXRX */
#endif /* NFX_SMP */

/*-----------------------------------------------------------------
 * Name :	eee_remainBufLen()
 * Description: return the number of bytes remaining available in the buffer
 *              starting from memptr.
 *-----------------------------------------------------------------
 */
int
eee_remainBufLen(eee_descBuf_t *bufdesc, uchar8 *memptr)
{
	/* Check that the pointer is inside the buffer */
	NFX_ASSERT(bufdesc->buf <= memptr && memptr < bufdesc->buf + bufdesc->len);
	return bufdesc->len - (memptr - bufdesc->buf);
}

#if defined(NETEEE_FRAGMENT)

typedef struct fragment_queue {
	eee_descPtr_t desc;			/* The packet descriptor */
	eee_descPtr_t last_desc;	/* The last descriptor in chain */
	int timeout;				/* Decrement on timeout pass, delete
                                   the queue entry when goes to zero */
	LIST_ENTRY(fragment_queue) q_list;
} fragment_queue_t;

LIST_HEAD(, fragment_queue) fragment_queue_head;
#if defined(NFX_SMP)
nfx_global_spinlock_t fragment_spin;
#endif

/*-------------------------------------------------------------------
 * Name :	matchFragmentQueue()
 * Description: return true if the packet fragment matches the fragment queue
 *-------------------------------------------------------------------
 */
int
matchFragmentQueue(fragment_queue_t *q, eee_descPtr_t edesc)
{
	return (q->desc->hdr.dest_port == edesc->hdr.dest_port
			&& q->desc->hdr.src_port == edesc->hdr.src_port
			&& edesc_seqId(q->desc) == edesc_seqId(edesc));
}


/*-----------------------------------------------------------------
 * Name :	findFragmentQueue()
 * Description: find a fragment queue containing the packet for the fragment
 *              specified by the argument edesc.
 *-----------------------------------------------------------------
 */
fragment_queue_t *
findFragmentQueue(eee_descPtr_t edesc)
{
	fragment_queue_t *q;

	for (q = LIST_FIRST(&fragment_queue_head); q != NULL;
		 q = LIST_NEXT(q, q_list)) {
		if (matchFragmentQueue(q, edesc)) {
			return q;
		}
	}
	return NULL;
}

/*-----------------------------------------------------------------
 * Name :	addFragmentQueue()
 * Description: add new fragment queue to the list of queues
 *-----------------------------------------------------------------
 */
void
addFragmentQueue(fragment_queue_t *q)
{
	LIST_INSERT_HEAD(&fragment_queue_head, q, q_list);
}

/*-----------------------------------------------------------------
 * Name :	deleteFragmentQueue()
 * Description: delete the queue from the list of fragment queues and
 *              free it.
 *-----------------------------------------------------------------
 */
void
deleteFragmentQueue(fragment_queue_t *q)
{
	LIST_REMOVE(q, q_list);
	eee_ramDealloc(q);
}

/*-----------------------------------------------------------------
 * Name :	dropFragmentQueue()
 * Description: delete the fragments in the queue and remove the queue
 *              from the list
 *-----------------------------------------------------------------
 */
void
dropFragmentQueue(fragment_queue_t *q)
{
	eee_freePacket(q->desc);
	deleteFragmentQueue(q);
}

/*-----------------------------------------------------------------
 * Name :	fragmentTimeout()
 * Description: decrement the timeout and delete timed out fragment queues
 *-----------------------------------------------------------------
 */
int32 
fragmentTimeout(uint32 tmr_handle, void	*usr_p1, void *usr_p2, void *usr_p3, void *usr_p4)
{
	fragment_queue_t *nq, *q;
	for (q = LIST_FIRST(&fragment_queue_head); q != NULL; q = nq) {
		nq = LIST_NEXT(q, q_list);
		if (--q->timeout == 0) {
			printk("fragmentTimeout: drop queue\n");
			dropFragmentQueue(q);
		}
	}
	return NFX_OK;
}

/*-----------------------------------------------------------------
 * Name :	initFragmentQueue()
 * Description: initialize fragment queue with the start fragment
 *-----------------------------------------------------------------
 */
void
initFragmentQueue(fragment_queue_t *q, eee_descPtr_t edesc)
{
	q->desc = edesc;
	q->last_desc = edesc;
	q->timeout = 2;
}

/*-----------------------------------------------------------------
 * Name :	allocFragmentQueue()
 * Description: allocate and initialize new fragment queue
 *-----------------------------------------------------------------
 */
fragment_queue_t *
allocFragmentQueue(eee_descPtr_t edesc)
{
	fragment_queue_t *q = (fragment_queue_t *)
		eee_ramAlloc(sizeof(fragment_queue_t));
	if (q) {
		initFragmentQueue(q, edesc);
	}

	return q;
}

/*-----------------------------------------------------------------
 * Name :	fragmentOffsetMatch()
 * Description: Returns true if the fragment offset matches the current
 *              fragment queue contents.
 *-----------------------------------------------------------------
 */
int
fragmentOffsetMatch(fragment_queue_t *q, eee_descPtr_t desc)
{
	return EEE_GET_PKT_LEN(q->desc) == edesc_fragmentOffset(desc);
}

/*-----------------------------------------------------------------
 * Name :	addFragmentToQueue()
 * Description: add fragment to the fragment queue
 *-----------------------------------------------------------------
 */
int
addFragmentToQueue(eee_descPtr_t desc, fragment_queue_t *q)
{
	int naddBufs = EEE_GET_NUM_BUFS(desc);
	int startBuf = EEE_GET_NUM_BUFS(q->last_desc);
	int endBuf; 
	int newBufIdx, bufIdx;

	NFX_ASSERT(fragmentOffsetMatch(q, desc));
	if (!(q->last_desc->hdr.control & EDESC_COMMAND_USE_FIRST_BUFFER))
		++startBuf;
	endBuf = startBuf + naddBufs;
	newBufIdx = (desc->hdr.control
				 & EDESC_COMMAND_USE_FIRST_BUFFER) ? 0 : 1;
	for (bufIdx = startBuf; bufIdx < endBuf; bufIdx++, newBufIdx++) {
		if (bufIdx == EEE_MAX_BUFS) {
			eee_descPtr_t newDesc = eee_allocateDesc(EEE_SHARED_POOL, NFX_COS_HIGH);
			if (newDesc == NULL) {
				return FALSE;
			} else {
				EEE_PRINTK(("addFragmentToQueue: chain new desc\n"));
				newDesc->hdr.control = 0;
				newDesc->bd[0].control = 0;
				eee_chainDesc(q->last_desc, newDesc);
				q->last_desc = newDesc;
				bufIdx = 1;
				endBuf = endBuf - EEE_MAX_BUFS + 1;
			}
		}
		EEE_SET_NUM_BUFS(q->last_desc, EEE_GET_NUM_BUFS(q->last_desc) + 1);
		q->last_desc->bd[bufIdx] = desc->bd[newBufIdx];
	}

	EEE_SET_PKT_LEN(q->desc, EEE_GET_PKT_LEN(q->desc)
					+ EEE_GET_PKT_LEN(desc));
	
	/* The new descriptor does not own any buffers now, so no need to
	   call eee_freePacket */
	eee_deallocateDesc(desc);

	return TRUE;
}

/*-----------------------------------------------------------------
 * Name :	eee_reassemble()
 *
 * Description: Attempt to reassemble the packet. Return NULL if the
 * fragment was queued for later reassembly, or the complete packet if
 * the reassembly was complete.
 *-----------------------------------------------------------------
 */
eee_descPtr_t
eee_reassemble(eee_descPtr_t edesc)
{
	uint32 offset = edesc_fragmentOffset(edesc);
	fragment_queue_t *queue;
	
	NFX_ASSERT(offset != 0 || edesc_moreFragments(edesc));

	EEE_PRINTK(("eee_reassemble: off=%d more=%s\n",
			offset, edesc_moreFragments(edesc) ? "yes" : "no"));

	NFX_SPIN_LOCK(&fragment_spin);
	if (offset == 0) {
		if ((queue = findFragmentQueue(edesc)) == NULL
			&& (queue = allocFragmentQueue(edesc)) != NULL) {
			addFragmentQueue(queue);
			NFX_SPIN_UNLOCK(&fragment_spin);
			return NULL;
		}
		else {
			if (queue != NULL) printk("duplicate queue\n");
			if (queue == NULL) printk("can't allocate queue\n");
		}
	} else {
		if ((queue = findFragmentQueue(edesc)) != NULL) {
			if (fragmentOffsetMatch(queue, edesc)) {
				int more_fragments = edesc_moreFragments(edesc);
				if (addFragmentToQueue(edesc, queue)) {
					if (!more_fragments) {
						/* Reassembly complete, delete the queue */
						eee_descPtr_t edesc = queue->desc;
						deleteFragmentQueue(queue);
						NFX_SPIN_UNLOCK(&fragment_spin);
						return edesc;
					}
					else {
						/* more fragments follow */
						NFX_SPIN_UNLOCK(&fragment_spin);
						return NULL;
					}
				}
				else {
					printk("eee_reassemble: can't add packet to queue, "
						   "dropping\n");
				}
			}
			else {
				printk("eee_reassemble: non-matching off=%d queue-len=%d, "
					   "dropping\n", edesc_fragmentOffset(edesc),
					   EEE_GET_PKT_LEN(queue->desc));
				deleteFragmentQueue(queue);
			}
		}
	}

	NFX_SPIN_UNLOCK(&fragment_spin);
	eee_freePacket(edesc);
	return NULL;
}

/*-----------------------------------------------------------------
 * Name :	eee_aliasData()
 *
 * Description: Make descriptor specified by the argument to reference
 * the same data of len bytes as descriptor specified by the argument
 * from starting from offset in descriptor "from". The "to" descriptor
 * will become the owner of the buffers completely contained in the
 * new descriptor.
 *-----------------------------------------------------------------
 */
void
eee_aliasData(eee_descPtr_t from, int offset, eee_descPtr_t to, int len)
{
	int aliasLen = 0;
	int fromBuf;
	uchar8 *fromMem;
	int toBuf = (to->hdr.control & EDESC_COMMAND_USE_FIRST_BUFFER) ? 0 : 1;
	int nbufs = 0;

	NFX_ASSERT(EEE_GET_PKT_LEN(from) - offset >= len);

	EEE_PRINTK(("eee_aliasData: off=%d pktlen=%d nbufs=%d len=%d\n", offset,
			   EEE_GET_PKT_LEN(from), EEE_GET_NUM_BUFS(from), len));

	eee_skipBufsFromStart(&from, offset, &fromBuf, &fromMem);
	EEE_PRINTK(("eee_aliasData: nbufs=%d frombuf=%d fromMem=%p\n",
			   EEE_GET_NUM_BUFS(from), fromBuf, fromMem));

	while (aliasLen < len) {
		int fromBufLen = eee_remainBufLen(&from->bd[fromBuf], fromMem);
		int thisAliasLen = MIN(fromBufLen, len - aliasLen);
		to->bd[toBuf].len = thisAliasLen;
		to->bd[toBuf].buf = fromMem;
		to->bd[toBuf].control = EDESC_BD_CMD_IN_USE |
			(from->bd[fromBuf].control & EDESC_BD_BUFSIZE_MASK);

		to->bd[toBuf].control |= (from->bd[fromBuf].control &
							      EDESC_BD_CMD_NO_DEALLOC);

		if (fromBufLen > len - aliasLen) {
			to->bd[toBuf].control |= EDESC_BD_CMD_NO_DEALLOC;
		}
		else {
			from->bd[fromBuf].control |= EDESC_BD_CMD_NO_DEALLOC;
		}
		if (fromMem != from->bd[fromBuf].buf) {
			/* If the buffer points does not point to the beginning of the
			   original buffer, the drivers can not use the area before
			   the buffer pointer to prepend headers */
			to->bd[toBuf].control |= EDESC_BD_CMD_START_IN_USE;
		}
 		toBuf++;
 		fromBuf++;
		if (edesc_firstBufferIdx(from) +
			EEE_GET_NUM_BUFS(from) == fromBuf) {
			from = eee_nextDesc(from);
			if (from != NULL)
				fromBuf = edesc_firstBufferIdx(from);
		}
	 	nbufs++;
		if (from != NULL)
			fromMem = from->bd[fromBuf].buf;
 		aliasLen += thisAliasLen;
	} 

	EEE_SET_NUM_BUFS(to, nbufs);
	EEE_SET_PKT_LEN(to, len);
}

int
eee_nextPacketId()
{
	int id;
	NFX_SPIN_LOCK(&fragment_spin);
	id = eee_packetId;
	++eee_packetId;
	if (eee_packetId > AGILEHDR_ID_MAX)
		eee_packetId = 0;
	NFX_SPIN_UNLOCK(&fragment_spin);
	return id;
}

			
/*-----------------------------------------------------------------
 * Name :	eee_convertPacket()
 * Description: fragment the packet to be output with the given mtu.
 *              On txrx, copy the data from the original descriptor if sending
 *              through management plane. Otherwise alias the original data into
 *              new descriptors to avoid copying.
 *-----------------------------------------------------------------
 */
eee_descPtr_t
eee_convertPacket(eee_descPtr_t edesc, int mtu)
{
	int hdrlen = sizeof(agile_hdr_t);
	int nextlen;
	int pktlen = EEE_GET_PKT_LEN(edesc);
	eee_descPtr_t newDesc;
	int rc = NFX_OK;
	int offset;
	int moreFragments;
	eee_descPtr_t firstDesc = NULL;
	eee_descPtr_t lastDesc = NULL;
	int alias_data = 1;
	int packetId;

	EEE_PRINTK(("eee_convertPacket: pktlen=%d mtu=%d dst=%x src=%x\n",
			   pktlen, mtu, edesc->hdr.dest_port, edesc->hdr.src_port));

	offset = 0;

	packetId = eee_nextPacketId();

	while (rc == NFX_OK && offset < pktlen) {
		nextlen = pktlen - offset;
		if (nextlen + hdrlen > mtu) {
			/* Fragments will be at least 64 bytes long */
			if (nextlen + hdrlen < mtu + 63)
				nextlen = nextlen / 2;
			else
				nextlen = mtu - hdrlen;
			nextlen &= ~63;
			moreFragments = AGILEHDR_MF;
		}
		else {
			moreFragments = 0;
		}
		
		if ((newDesc = eee_allocateDesc(EEE_SHARED_POOL, NFX_COS_HIGH)) != NULL) {
			eee_descHdr_t *hdr = &newDesc->hdr;

			hdr->src_port = edesc->hdr.src_port;
			hdr->dest_port = edesc->hdr.dest_port;
			hdr->offset = offset | moreFragments |
				(packetId << AGILEHDR_ID_SHIFT);
			
			if (alias_data) {
				hdr->control = 0;
				eee_aliasData(edesc, offset, newDesc, nextlen);
			}
#if defined(NFP_TXRX)
			else {
				uint8 *newBuf = eee_allocateBuffer(eee.my_pool_id, EEE_BUF_MGMT, NFX_COS_HIGH);
				if (newBuf) {
					eee_descBuf_t *bd = &newDesc->bd[1];
					hdr->control = nextlen | (1 << EDESC_BUFCNT_SHIFT);
					bd->len = nextlen;
					bd->buf = newBuf;
					bd->control = EDESC_BD_CMD_IN_USE | EDESC_BD_LARGE;
					eee_copyPacketData(newBuf, edesc, offset, nextlen, 0);
					eee_stats->txrx_mgmt_copied += nextlen;
				}
				else {
                    eee_deallocateDesc(newDesc);
					rc = NFX_ERR;
				}
			}
#endif		

            if (rc != NFX_ERR) {
                
                /* Add the new descriptor to the output packet
                 */
                
                if (lastDesc == NULL) {
                    lastDesc = newDesc;
                    firstDesc = newDesc;
                }
                else {
                    eee_chainDesc(lastDesc, newDesc);
                    lastDesc = newDesc;
                }
            }
		}
		else {
			rc = NFX_ERR;
		}
		offset += nextlen;
	}

	if (rc == NFX_OK) {
		if ((alias_data) && 
		    ((edesc->hdr.control & EDESC_PKT_NO_DEALLOC) == 0)) {
			while (edesc) {
				eee_descPtr_t next = eee_nextDesc(edesc);
				eee_unlinkNextDesc(edesc);
				eee_deallocateDesc(edesc);
				edesc = next;
			}
		}
		else {
			eee_freePacket(edesc);
		}

		return firstDesc;
	}
	else {
		if (firstDesc != NULL) {
			eee_freePacket(firstDesc);
		}
		return NULL;
	}
}

#else

eee_descPtr_t
eee_reassemble(eee_descPtr_t desc)
{
	return desc;
}

int32 
fragmentTimeout(uint32	tmr_handle,
	void	* usr_p1,
	void	* usr_p2,
	void	* usr_p3,
	void	* usr_p4)
{
	return NFX_OK;
}

#endif


/*-----------------------------------------------------------------
 * Name :	eee_skipBufs()
 *
 * Description: Skip len bytes from the current position in the
 * descriptor.  The position is specified by bufptr and memptr, which
 * are the current buffer index and the current pointer into the
 * descriptor.  On return the locations specified by the arguments
 * descptr, bufptr and memptr will be updated to contain new
 * descriptor position.
 *----------------------------------------------------------------- */
void
eee_skipBufs(eee_descPtr_t *descptr, int len, int *bufptr, uchar8 **memptr)
{
	int skipLen = 0;
	eee_descPtr_t desc = *descptr;
	int buf = *bufptr;
	uchar8 *mem = *memptr;
	int bufRemainLen;

	while (skipLen + (bufRemainLen = eee_remainBufLen(&desc->bd[buf], mem)) <= len) {
		skipLen += bufRemainLen;
		++buf;
		if (buf == edesc_firstBufferIdx(desc) + EEE_GET_NUM_BUFS(desc)) {
			desc = eee_nextDesc(desc);
			/* Check if tried to skip too much */
			NFX_ASSERT(desc != NULL);
			buf = edesc_firstBufferIdx(desc);
		}
		mem = desc->bd[buf].buf;
	}
	*memptr = desc->bd[buf].buf + (len - skipLen);
	*bufptr = buf;
	*descptr = desc;
}


/*-----------------------------------------------------------------
 * Name :	eee_skipBufsFromStart()
 * Description: Skip len bytes from the beginning of descriptor. The
 * number of buffer containing len + 1 byte will be stored into the
 * location specified by the argument buf, the pointer into the
 * buffer will be stored into the location specified by the argument memptr.
 *-----------------------------------------------------------------
 */
void
eee_skipBufsFromStart(eee_descPtr_t *descptr, int len, int *bufptr, uchar8 **memptr)
{
	*bufptr = edesc_firstBufferIdx(*descptr);
	*memptr = (*descptr)->bd[*bufptr].buf;
	eee_skipBufs(descptr, len, bufptr, memptr);
}

/*-----------------------------------------------------------------
 * Name:	eee_processReceiveQueue()
 * Description:	polling function that reads entries from the receive queue
 *		and acts on the entries.
 *-----------------------------------------------------------------
 */
void
eee_processReceiveQueue(void *cb, uint32 tref)
{
	struct eee_rcv_queue *queue = cb;
	address_t entry;

#if defined (NFP_TXRX) && defined(NFP_SMP)
        /* txrx acpu does not poll receive_queue */
        if (!CM_IS_TXRX_NCPU)
            return;
#endif /* NFP_SMP */
 	
	queue->num_called++;

    if (!(((address_t)(*queue->rcv_head)) & EEE_FWD_OWN)) {
		queue->poll_idle++;
        return;
	}

	if (!NFX_SPIN_TRYLOCK(&queue->rcv_spin)) {
		return;
	}

	if ((entry = (address_t)(*queue->rcv_head)) & EEE_FWD_OWN) {
		int cnt = 0;
		int max = queue->rcv_max;
		eee_descPtr_t *rcv_end = queue->rcv_end;
		queue->poll_valid++;
		do {
			++cnt;
			switch (entry & EEE_FWD_WHAT_MASK) {
			case EEE_FWD_ACTION_FWD:
			{
				eee_descPtr_t edesc =
					(eee_descPtr_t)(entry & EEE_FWD_ADDR_MASK);
				queue->rcv_pkts_forwarded++;
#if defined(NFP_TXRX) || defined(NFP_FP)
				if (eee_isRemoteAddr((address_t)edesc)) {
					eee_desc_t *new_edesc = eee_alloc_buf(&local_desc_pool);

					ht_memcpy_ass(new_edesc,
								  (void *)uncached_to_cached((address_t)edesc),
							  sizeof(eee_desc_t) / CACHELINE_SIZE);

					/* JTOF FIX - make sure the edesc copy doesn't 
		             * have the NO_DEALLOC set - but orinal stays
					 * set and doesn't get cleared cause RMC owns it 
					 */
					if (new_edesc->hdr.control & EDESC_PKT_NO_DEALLOC) 
						new_edesc->hdr.control &= ~EDESC_PKT_NO_DEALLOC;

					if ((edesc->hdr.control & EDESC_PKT_NO_DEALLOC) == 0)
						eee_deallocateRemoteDesc(edesc);

					edesc = new_edesc;
				}
#endif

				ht_bytes_rx += sizeof(edesc) + EEE_GET_PKT_LEN(edesc);
#ifdef NFX_SMP
				eee_receiveChain(edesc);
#else
				eee_receivePacket(edesc);
#endif
				break;
			}
			case EEE_FREE_BUF:
				eee_deallocateSharedBuffer(
					(void *) (entry & EEE_FWD_ADDR_MASK),
					(entry & EEE_FWD_BUF_SIZE_MASK) >> EEE_FWD_BUF_SIZE_SHIFT);
				queue->rcv_free_buf++;
				break;
			case EEE_FREE_DESC:
				eee_deallocateSharedDesc((void *)
										 (entry & EEE_FWD_ADDR_MASK));
				queue->rcv_free_desc++;
				break;
			default:
				printk("Invalid rcv queue entry, %lx\n", entry);
				break;
			}

			*(queue->rcv_head) = 0;
			*(queue->rcv_tail_ptr) = queue->rcv_head;
			QUEUE_PTR_INC(	queue->rcv_head,
							rcv_end,
							queue->rcv_start);
		} while (((entry = (address_t)(*queue->rcv_head)) & EEE_FWD_OWN)
			&& cnt != max);
		queue->rcv_cnt += cnt;
		if (cnt > queue->rcv_hwm)
			queue->rcv_hwm = cnt;
	}

	NFX_SPIN_UNLOCK(&queue->rcv_spin);
}

/*-----------------------------------------------------------------
 * Name:		eee_tryFreeReceiveQueue()
 * Description:		try to free buf or edesc entries in rcv queue
 *-----------------------------------------------------------------
 */
void
eee_tryFreeReceiveQueue(void *cb, uint32 tref)
{
        struct eee_rcv_queue *queue = cb;
        address_t entry;

#if defined (NFP_TXRX) && defined(NFP_SMP)
        /* txrx acpu does not poll receive_queue */
        if (!CM_IS_TXRX_NCPU)
            return;
#endif /* NFP_SMP */

        NFX_SPIN_LOCK(&queue->rcv_spin);

        if (!(((address_t)(*queue->rcv_head)) & EEE_FWD_OWN)) {
            NFX_SPIN_UNLOCK(&queue->rcv_spin);
            return;
        }

        if ((entry = (address_t)(*queue->rcv_head)) & EEE_FWD_OWN) {
                int cnt = 0;
                eee_descPtr_t *rcv_end = queue->rcv_end;
                do {
                        ++cnt;
                        switch (entry & EEE_FWD_WHAT_MASK) {
                        case EEE_FWD_ACTION_FWD:
                                break;
                        case EEE_FREE_BUF:
                                eee_deallocateSharedBuffer(
                                        (void *) (entry & EEE_FWD_ADDR_MASK),
                                        (entry & EEE_FWD_BUF_SIZE_MASK) >> EEE_FWD_BUF_SIZE_SHIFT);
                                queue->rcv_free_buf++;
                                break;
                        case EEE_FREE_DESC:
                                eee_deallocateSharedDesc((void *) (entry & EEE_FWD_ADDR_MASK));
				queue->rcv_free_desc++;
                                break;
                        default:
                                printk("Invalid rcv queue entry, %lx\n", entry);
                                break;
                        }

                        *(queue->rcv_head) = 0;
                        *(queue->rcv_tail_ptr) = queue->rcv_head;
                        QUEUE_PTR_INC(  queue->rcv_head,
                                                        rcv_end,
                                                        queue->rcv_start);
                } while (((entry = (address_t)(*queue->rcv_head)) & EEE_FWD_OWN));
                queue->rcv_cnt += cnt;
        }

        NFX_SPIN_UNLOCK(&queue->rcv_spin);
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_fwdReceiveQueuePollCfg(uint32	max_val)
{
	eee_my_receive_queue()->rcv_max = max_val;
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_fwdReceiveQueuePollView(uint32 just_hwm)
{
	NFX_SPIN_LOCK(&(eee_my_receive_queue()->rcv_spin));

	if (just_hwm) {
		printk("RCVQ: max_val       = %d\n", eee_my_receive_queue()->rcv_max);

		printk("      HWM           = %d\n", eee_my_receive_queue()->rcv_hwm);
	} else {
		printk("RCVQ: called        = %lld\n",eee_my_receive_queue()->num_called);
		printk("      max_val       = %d\n",eee_my_receive_queue()->rcv_max);
		printk("      poll_idle     = %lld\n",eee_my_receive_queue()->poll_idle);
		printk("      poll_valid    = %lld\n",eee_my_receive_queue()->poll_valid);
		printk("      rcvd          = %lld\n",eee_my_receive_queue()->rcv_cnt);
		printk("      HWM           = %d\n",eee_my_receive_queue()->rcv_hwm);
		printk("      AVG           = %lld\n",eee_my_receive_queue()->poll_valid ?
			eee_my_receive_queue()->rcv_cnt / eee_my_receive_queue()->poll_valid : 0);
	}
	NFX_SPIN_UNLOCK(&(eee_my_receive_queue()->rcv_spin));
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_fwdReceiveQueuePollSendToSSC(uint32	just_hwm, eee_poll_cfg_rsp_t *rsp, uint32 max_rsps, uint32 *num_added)
{
	uint32	num = 0;

	NFX_SPIN_LOCK(&(eee_my_receive_queue()->rcv_spin));

	if (just_hwm) {
		if (num < max_rsps) {
			strncpy(rsp->str, "RCVQ", POLLCFG_RSP_STR_LEN);
			rsp->cnt = POLLCFG_RSP_SECTION_LABEL;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "max_val", POLLCFG_RSP_STR_LEN);
			rsp->cnt = eee_my_receive_queue()->rcv_max;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "HWM", POLLCFG_RSP_STR_LEN);
			rsp->cnt = eee_my_receive_queue()->rcv_hwm;
			rsp++; num++;
		}
	} else {
		if (num < max_rsps) {
			strncpy(rsp->str, "RCVQ", POLLCFG_RSP_STR_LEN);
			rsp->cnt = POLLCFG_RSP_SECTION_LABEL;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "called", POLLCFG_RSP_STR_LEN);
			rsp->cnt = eee_my_receive_queue()->num_called;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "max_val", POLLCFG_RSP_STR_LEN);
			rsp->cnt = eee_my_receive_queue()->rcv_max;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "poll_idle", POLLCFG_RSP_STR_LEN);
			rsp->cnt = eee_my_receive_queue()->poll_idle;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "poll_valid", POLLCFG_RSP_STR_LEN);
			rsp->cnt = eee_my_receive_queue()->poll_valid;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "rcvd", POLLCFG_RSP_STR_LEN);
			rsp->cnt = eee_my_receive_queue()->rcv_cnt;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "HWM", POLLCFG_RSP_STR_LEN);
			rsp->cnt = eee_my_receive_queue()->rcv_hwm;
			rsp++; num++;
		}
		if (num < max_rsps) {
			strncpy(rsp->str, "AVG", POLLCFG_RSP_STR_LEN);
			rsp->cnt = eee_my_receive_queue()->poll_valid ?
							eee_my_receive_queue()->rcv_cnt /
							eee_my_receive_queue()->poll_valid : 0;
			rsp++; num++;
		}
	}

	*num_added = num;

	NFX_SPIN_UNLOCK(&(eee_my_receive_queue()->rcv_spin));
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_fwdReceiveQueuePollClear()
{
	NFX_SPIN_LOCK(&(eee_my_receive_queue()->rcv_spin));

	eee_my_receive_queue()->num_called = 0;
	eee_my_receive_queue()->poll_idle = 0;
	eee_my_receive_queue()->poll_valid = 0;
	eee_my_receive_queue()->rcv_cnt = 0;
	eee_my_receive_queue()->rcv_hwm = 0;

	NFX_SPIN_UNLOCK(&(eee_my_receive_queue()->rcv_spin));
}

eee_poll_cfg_t eee_fwdReceiveQueuePollCfgStruct[] = {
	{"view",	eee_fwdReceiveQueuePollView},
	{"ssc",		eee_fwdReceiveQueuePollSendToSSC},
	{"clear",	eee_fwdReceiveQueuePollClear},
	{"rcvq",	eee_fwdReceiveQueuePollCfg},
	{NULL, 0}
};

/*-----------------------------------------------------------------
 * Name:	eee_deliverPacketToApp()
 * Description: This function should deliver the packet to a
 *		registered application receive function on the current CPU.
 * Returns:
 * NfX_OK, EEE_ERR_INVALID_APP_ID
 *-----------------------------------------------------------------
 */
int32
eee_deliverPacketToApp(eee_descPtr_t edesc, int app_id)
{
#ifdef SSC
	
	uchar8	app[80];
	uint32  srcPort;

	/*
	printk("eee_deliverPackToApp(%p, 0x%x)\n", edesc, app_id);
	*/

	srcPort = edesc->hdr.src_port;
/*
	We go through the ANPSSC stuff if it's destined for the Unix CPU and
	its not addressed to a e-layer application.
 */
	if ((EEE_GET_DEST_CPU(edesc->hdr.dest_port) == NFX_SSC_UNIX) &&
	    ((app_id >= EEE_MAX_APP_CLIENTS) || !eee.rcv.uni[app_id]))
	{
		uchar8	* a;
		uchar8	* bufptr;
		uint32	pkt_len;
		extern int32	ssc_anp_fd;
		a = eee_getAppNameFromId(app_id);
		if (a) {
			memcpy(app, a, 80);
		}
		else {
			sprintf (app, "Agile%d", app_id);
		}

/*
printk("SSC : app_id (%d), sending to app string \"%s\"\n", app_id, app);
*/

		pkt_len = EEE_GET_PKT_LEN(edesc);
		bufptr = (uchar8 *)malloc(pkt_len+sizeof(ans_ssc_msghdr_t));

		((ans_ssc_msghdr_t *)bufptr)->appid =EEE_GET_APP_ID(edesc->hdr.src_port);
		((ans_ssc_msghdr_t *)bufptr)->slot = EEE_GET_SLOT_NUM(edesc->hdr.src_port);
		((ans_ssc_msghdr_t *)bufptr)->cpu = EEE_GET_DEST_CPU(edesc->hdr.src_port);
		((ans_ssc_msghdr_t *)bufptr)->plane = EEE_GET_DTYPE(edesc->hdr.src_port);


		eee_copyPacketToLocalBuffer(bufptr+sizeof(ans_ssc_msghdr_t), edesc, pkt_len, 0);

		if (sendAgileMsg(ssc_anp_fd,
			bufptr,
			pkt_len + sizeof(ans_ssc_msghdr_t),
			app,
			0,
			0,
			0,
			srcPort,
			0,
			0) < 0)
			printk("error sending ANP SSC sendAgileMsg\n");
		else 
			printk("ANP SSC message sent successfully\n");
		eee_freePacket(edesc);
		return NFX_OK;
	}

#endif

	if (app_id >= EEE_MAX_APP_CLIENTS) {
		ATOMIC_INC_INT(&eee_stats->invalid_app_id, 1);
//		panic("Invalid application id, %d\n", app_id);
		return EEE_ERR_INVALID_APP_ID;
	}

	if (eee.rcv.uni[app_id]) {
		ATOMIC_INC_INT(&eee_stats->app_rcv_pkts[app_id], 1);
#ifdef EEE_ATTR
		eee_attrAssignPkt(edesc, app_id);
#endif
        (*eee.rcv.uni[app_id])(edesc);
    }
	else {
		ATOMIC_INC_INT(&eee_stats->invalid_app_id, 1);

/*
		printk("ERROR: deliverPacketToApp\n");
*/
		return EEE_ERR_INVALID_APP_ID;
	}

	return NFX_OK;

}	/* eee_deliverPacketToApp() */

/*-----------------------------------------------------------------
 * Name:	eee_receivePacket()
 * Description:
 *-----------------------------------------------------------------
 */
#if !defined(NFX_SMP)
void
eee_receivePacket(eee_descPtr_t	edesc)
#else
void
eee_processPacket(eee_descPtr_t   edesc)
#endif
{
    address_t ra unused__ = get_ra();
    uint32	dp;
	uint32	app_id;

	dp = edesc->hdr.dest_port;

#if defined(NFX_SMP) && defined(NFP_TXRX)
	/* acpu, deliver to application directly */
	if (nfxMyCPUId == NFX_NFP_TXRX2) // TXRX2 is the acpu
    {
        /* If we received a buffer in the remote uncached memory,
         * convert the address to cached before delivering the
         * packet to the application.
         */
        eee_descBuf_t *bd = edesc_firstBuf(edesc);
        if (eee_isRemoteAddr((address_t)bd->buf)) {
            bd->buf = (void *)uncached_to_cached((address_t)bd->buf);
            /* Prefetch the beginning of the buffer for the
             * application since most probably the first thing
             * it will do is access the first bytes to get
             * the message type.
             */
            PREFETCH(PREF_LOAD, bd->buf, 0);
            goto deliver_pkt;
        }
        else if (bd->buf != NULL) {
            PREFETCH(PREF_LOAD, bd->buf, 0);
            goto deliver_pkt;
        }
        else {
            /* Drop packets without a buffer 
             */
            ASSERT(FALSE);
            eee_freePacket(edesc);
        }
    }
#endif /* NFX_SMP, NFP_TXRX */

#ifdef NFP_TXRX
        /*
         * If received packet for different slot or for FP cpus, forward
         * it.
         */
	if ((EEE_GET_DTYPE(dp) == EEE_MGMT) &&
            !eee_is_local_slot_cpu(EEE_GET_SLOT_NUM(dp), EEE_GET_DEST_CPU(dp))) {

            EEE_SET_PORT_ID(dp, PORT_BEST_FIT);
            edesc->hdr.dest_port = dp;
            eee_forwardPacket(edesc);
            return;
	}
#endif

#if defined(NFX_SMP) && defined(NFP_TXRX)
		/* is this message to be processed on ncpu or give to
		 * acpu. 
		 * edescp is given to acpu if
		 * 1. src_cpu is fp and dst_cpu is txrx 
		 * 2. if src port is dp83820 and selected appId
		 */
		if (eee_txrxIsPktToAcpu(edesc)) {
			eee_receivePacketA(edesc);
			return;
		}

deliver_pkt:
#endif /* NFX_SMP, NFP_TXRX */

	if (edesc_fragmentOffset(edesc) != 0 || edesc_moreFragments(edesc)) {
		edesc = eee_reassemble(edesc);
	}
	if (edesc != NULL) {
		/* Reassembly successful, or no reassembly was necessary */
		app_id = EEE_GET_APP_ID(dp);
#if defined(DEBUG_SCSI_WRITE_REQ_CORRUPTION)
        int destAppId = EEE_GET_APP_ID(edesc->hdr.dest_port);
        int scsiAppid = eee_getApplicationId("scsi");

        if (destAppId == scsiAppid) {
            scsi_check_write_req_corruption(edesc_firstBuf(edesc)->buf,
                                            edesc_firstBuf(edesc)->len,
                                            ra, FALSE);
        }
#endif
		if (eee_deliverPacketToApp(edesc, app_id) != NFX_OK) {
			eee_freePacket(edesc);
		}
	}

}	/* eee_receivePacket() */


/*-----------------------------------------------------------------
 * Name :	eee_findPort()
 * Description: find the driver port to send a packet to the destination
 *              specified by the argument dp. 
 *-----------------------------------------------------------------
 */
int
eee_findPort(uint32 port, uint32 dp, eee_port_t **drv_port)
{
	int rc = NFX_OK;
	if (port == PORT_BEST_FIT) {
		*drv_port = &eee.drv.dtype_default[EEE_GET_DTYPE(dp)];
	} else {
		if (dp & EDESC2_MCAST_IND) {
			panic("Multicast transmit\n");
			*drv_port = &eee.drv.multi[port];
		}
		else {
			*drv_port = &eee.drv.uni[port];
		}
	}
	
	if ((*drv_port)->func == NULL) {
		printk("no transmit function for port %d, dp = 0x%x\n", port, dp);
		rc = EEE_DROP_PACKET;
	}
	return rc;
}

/*-----------------------------------------------------------------
 * Name :	eee_doTransmit()
 * Description: transmit the packet and update the statistics.
 *-----------------------------------------------------------------
 */
int
eee_doTransmit(eee_port_t *drv_port, uint32 dp, eee_descPtr_t edesc)
{
	int rc;
	ATOMIC_INC_INT(&eee_stats->fwd_pkts[EEE_GET_SLOT_NUM(dp)] [EEE_GET_DEST_CPU(dp)], 1);

	rc = (*(drv_port->func))(dp, edesc);

	switch (rc) {

	case EEE_FORWARDED_PACKET:
		ATOMIC_INC_INT(&drv_port->forwarded, 1);
		break;

	case EEE_DROP_PACKET:
		ATOMIC_INC_INT(&drv_port->dropped, 1);
		eee_freePacket(edesc);
		break;

	case EEE_TX_QUEUE_FULL:
#ifndef SSC_MGMT
		panic("EEE_TX_QUEUE_FULL, dp = 0x%x\n", dp);
#endif
		break;
	default:
		panic("Bad return code, 0x%x\n", rc);
		break;
	}
	return rc;
}

/*-----------------------------------------------------------------
 * Name:	eee_forwardPacket()
 *
 * Description: This function is used to send a packet through the system.
 *
 *		An application that needs to send a packet should use this function.
 *		A driver that receives a packet should use this function
 *		to deliver the packet to appropriate destination.
 *
 * 		This function will handle packets destined for applications
 *		on the local CPU, other CPUs on the local slot, or
 *		sending to registered transmit drivers for delivery across
 *		the backplane switch fabrics.
 *
 *
 *		Calling function should look something like this:
 *
 *		rc = forwardPacket(edesc);
 *		if (rc == EEE_DROP_PACKET) {
 *			// The packet was dropped by either the driver or
 *			// the EEE layer.  The descriptor has been freed.
 *			// The caller should make no further references to
 *			// the packet.
 *		}
 *		else if (rc == EEE_TX_QUEUE_FULL) {
 *			// The packet could not be queued due a tx queue full
 *			// condition.  The caller can take whatever action
 *			// it wants with the packet, since it is still the
 *			// owner.
 *		}
 *		else {
 *			// The packet was queued for transmit successfully.
 *			// The caller should not free the packet (or touch
 *			// the packet in any way), as it is now owned by
 *			// someone else.
 *		}
 *
 * Returns:
 * EEE_FORWARDED_PACKET, EEE_DROP_PACKET, EEE_TX_QUEUE_FULL
 *-----------------------------------------------------------------
 */
int32
eee_forwardPacket(eee_descPtr_t	edesc)
{
    uint64 ra unused__ = get_ra();
	uint32	dp;
	uint32  slot_id, cpu_id, app_id, dtype;
	uint32	port;
	int32	rc;
	eee_port_t *drv_port;

	dp = edesc->hdr.dest_port;

#if defined(NFP_FP) && defined(DEBUG_SCSI_WRITE_REQ_CORRUPTION)
    scsi_check_write_req_corruption(edesc_firstBuf(edesc)->buf,
                                    edesc_firstBuf(edesc)->len,
                                    ra, FALSE);
#endif
	if (EEE_GET_APP_ID(dp) == 0)
		panic("eee_forwardPacket, appid = 0, edesc = %p\n", edesc);

	if (EEE_GET_PKT_LEN(edesc) == 0)
		panic("eee_forwardPacket, pktlen = 0, edesc = %p\n", edesc);

	slot_id = EEE_GET_SLOT_NUM(dp);
	cpu_id = EEE_GET_DEST_CPU(dp);
	dtype = EEE_GET_DTYPE(dp);
	port = EEE_GET_PORT_ID(dp);

	/*
	 * Is this destined for another application on same CPU?
	 */
	if (((port == PORT_BEST_FIT) || (port == PORT_LOCAL_APP)) && 
        eee_is_local_slot_cpu(slot_id, cpu_id)) {
        app_id = EEE_GET_APP_ID(dp);

#if defined(NFX_SMP) && defined(NFP_TXRX)
		/* is this message to be processed on ncpu or give to
		 * acpu. 
		 * edescp is given to acpu if
		 * 1. src_cpu is fp and dst_cpu is txrx 
		 * 2. if src port is dp83820 and selected appId
		 */
		if ((nfxMyCPUId == NFX_NFP_TXRX1) && eee_txrxIsPktToAcpu(edesc)) {
			eee_receivePacketA(edesc);
			return NFX_OK;
		}
#endif
        if (eee_deliverPacketToApp(edesc, app_id) != NFX_OK) {
            eee_freePacket(edesc);
            return EEE_DROP_PACKET;
        } else {
            return EEE_FORWARDED_PACKET;
		}
	}

#ifdef NFP_FP
	if (dtype == EEE_MGMT) {
		//printk("MGMT packet, change port to LOCAL_QUEUE\n");
		port = NFP_PORT_LOCAL_QUEUE;
	}
#endif

	/*
	 * Can this packet be queued locally?
	 */
#if defined(NFX_MOD_NFP) || defined(NFX_MOD_SSC)
	if (port == PORT_BEST_FIT) {
		if (eee_is_local_slot(slot_id))
#ifdef NFX_MOD_NFP
			port = NFP_PORT_LOCAL_QUEUE;
#else
			port = SSC_PORT_LOCAL_QUEUE;
#endif
	}
#endif

#if defined(NFP_TXRX) && defined(NFX_SMP)
	/* if managment msg on acpu, give to ncpu */
	if ((dtype == EEE_MGMT) && (nfxMyCPUId != NFX_NFP_TXRX1)) {
		eee_receiveChain(edesc);
		return EEE_FORWARDED_PACKET;
	}
#endif
	
	if ((rc = eee_findPort(port, dp, &drv_port)) == NFX_OK) {
		eee_descPtr_t packetDesc;
		if (!(edesc->hdr.control & EDESC_COMMAND_DONT_CONVERT)) {
			int size_too_big =
				EEE_GET_PKT_LEN(edesc) + sizeof(agile_hdr_t) > drv_port->mtu;
#if defined(NFP_TXRX)
			/* Need to satisfy driver requirements about buffer location */
			int need_convert = size_too_big
				|| (dtype == EEE_MGMT && port != NFP_PORT_LOCAL_QUEUE);
#else
			int need_convert = size_too_big;
#endif
			
			if (need_convert) {
				packetDesc = eee_convertPacket(edesc, drv_port->mtu);
			}
			else {
				packetDesc = edesc;
			}
		}
		else {
			packetDesc = edesc;
		}

#if defined(NFP_FP)
		if (port == NFP_PORT_LOCAL_QUEUE)
			rc = eee_doTransmit(drv_port, dp, packetDesc);
		else {
#endif			

            if (packetDesc != NULL) {
                int rc = EEE_FORWARDED_PACKET;
                while (packetDesc != NULL) {
                    eee_descPtr_t nextDesc = eee_nextDesc(packetDesc);
                    eee_unlinkNextDesc(packetDesc);

                    if (rc == EEE_FORWARDED_PACKET) {
                        rc = eee_doTransmit(drv_port, dp, packetDesc);
                    }
                    else {
                        printk("eee_doTransmit error, free rest of chain\n");
                        eee_freePacket(packetDesc);
                    }
                    packetDesc = nextDesc;
                }
            } else {
                rc = EEE_DROP_PACKET;
                eee_freePacket(edesc);
                ATOMIC_INC_INT(&eee.stats.convert_fail, 1);
            }
#if defined(NFP_FP)		
		}
#endif		
	} else {
		if (rc == EEE_DROP_PACKET) {
			eee_freePacket(edesc);
		}
	}

//	EEE_PRINTK(("eee_forwardPacket: returning %d\n", rc));
	return rc;
}	/* eee_forwardPacket() */

#if defined(NFX_MOD_NFP)
/*-----------------------------------------------------------------
 * Name:	eee_forwardLocalPktUni()
 * Description:	This function queues a packet to a local forwarding queue
 *		for delivery to a CPU on the local card that can be accessed
 *		via shared memory.
 * Returns:
 * EEE_FORWARDED_PACKET, EEE_TX_QUEUE_FULL
 *-----------------------------------------------------------------
 */
int32
eee_forwardLocalPktUni(uint32 dest_port, eee_desc_t *edesc)
{
	uint32	cpu_id;
	uint32	fwd_q_index;
	int32	rc;

	cpu_id = EEE_GET_DEST_CPU(dest_port);
#ifdef NFP_TXRX
	fwd_q_index = 1;
#elif defined(NFP_FP)
	fwd_q_index = 0;
#else
	fwd_q_index = GET_FWD_INDEX_FROM_CPU_ID(cpu_id);
#endif

#ifdef EEE_ATTR
	eee_attrAssignPkt(edesc, EEE_ATTR_FWD);
#endif

	rc = FWD_PACKET(((address_t)edesc & ~EDESC_ADDR_MASK)
					| EEE_FWD_ACTION_FWD, fwd_q_index, TRUE);
	return rc;

}	/* eee_forwardLocalPktUni() */

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
int32
eee_forwardLocalPktMulti(uint32 dest_port, eee_desc_t *edesc)
{
	panic("eee_forwardLocalPktMulti!\n");
	return EEE_DROP_PACKET;
}
#endif

/*-----------------------------------------------------------------
 * Name :	eee_initFragmentTimeout()
 * Description: initialize the fragments timeout facility
 *-----------------------------------------------------------------
 */
void
eee_initFragmentTimeout()
{
	int32 th;

	th = eee_timerCreate(2 * MSEC_PER_SEC, EEE_TIMER_REPEAT);
	if (th != TIMER_ERR_NO_TIMER_AVAILABLE) {
		int rc unused__ = eee_timerStart(th, fragmentTimeout,
                                         NULL, NULL, NULL, NULL);
		NFX_ASSERT(rc == NFX_OK);
	} else {
		panic("Can not initialize fragments timeout\n");
	}
}

/*-----------------------------------------------------------------
 * Name :	eee_addHeader()
 *
 * Description: Add a header buffer containing agile header and
 * ethernet header to the descriptor for sending the packet through
 * the backplane.
 *
 * Returns: NFX_OK if succeeeds, otherwise NFX_ERR
 *
 * Created by:	Maxim Kozlovsky
 *
 * Date Created:	04/10/02
 *
 *-----------------------------------------------------------------
 */
int
eee_addHeader(eee_desc_t *pktDesc, int remotePort, uchar8 *myMacAddr)
{
	int pktLen = EEE_GET_PKT_LEN(pktDesc);
	int bufCnt = EEE_GET_NUM_BUFS(pktDesc);
	int dtype = EEE_GET_DTYPE(pktDesc->hdr.dest_port);
	char *hdrBuf;
	enet_hdr_t *ethHdr;
	agile_hdr_t *eeeHdr;
	uint32 hdrLen = sizeof(enet_hdr_t) + sizeof(agile_hdr_t);
	uint32 offset;

	// Check for room in the first buffer.
	hdrBuf = pktDesc->bd[1].buf;	
#if defined(EEE_NEW_BUFSIZE)
    offset = (address_t)hdrBuf & 0x7F;
#else
	offset = (address_t)hdrBuf & 0xFF;
#endif	

	/* The descriptor passed should not use the first buffer */
	NFX_ASSERT(!(pktDesc->hdr.control & EDESC_COMMAND_USE_FIRST_BUFFER));

	if( offset >= hdrLen
		&& !(pktDesc->bd[1].control & EDESC_BD_CMD_START_IN_USE)) {

#if defined(EEE_VALIDATE)		
		NFX_ASSERT((address_t)eee_getBufBase(hdrBuf) <=
				   (address_t)hdrBuf - hdrLen);
#endif		
		hdrBuf -= hdrLen;

		
		ethHdr = (enet_hdr_t *)hdrBuf;
		eeeHdr = (agile_hdr_t *)(hdrBuf + sizeof(enet_hdr_t));
		bp_setDestMacAddr(pktDesc->hdr.dest_port, remotePort, ethHdr->DA);
		memcpy(ethHdr->SA, myMacAddr, 6);
		if (dtype == EEE_SAC)
			ethHdr->type = EEE_ETH_TYPE_SAC;
		else
			ethHdr->type = EEE_ETH_TYPE;

		eeeHdr->pad = 0;
		eeeHdr->dest_port = pktDesc->hdr.dest_port;
		eeeHdr->src_port = pktDesc->hdr.src_port;
#if defined(NETEEE_FRAGMENT)
		eeeHdr->offset = pktDesc->hdr.offset;
#endif
		pktDesc->bd[1].buf = hdrBuf;
		pktDesc->bd[1].len += hdrLen;
		pktLen += hdrLen;

        pktDesc->hdr.control = 0;
        EEE_SET_PKT_LEN(pktDesc, pktLen);
        EEE_SET_NUM_BUFS(pktDesc, bufCnt);
	} else {
#ifdef NFP_TXRX
		hdrBuf = eee_allocateBuffer(EEE_SHARED_POOL, EEE_BUF_MGMT, 0);
#else
		hdrBuf = eee_allocateBuffer(EEE_SHARED_POOL, EEE_BUF_SMALL, 0);
#endif
		if (hdrBuf == NULL) {
			printk("eee_addHeader: Error cannot allocate buffer\n");
			return NFX_ERR;
		}

		ethHdr = (enet_hdr_t *)hdrBuf;
		eeeHdr = (agile_hdr_t *)(hdrBuf + sizeof(enet_hdr_t));

		bp_setDestMacAddr(pktDesc->hdr.dest_port, remotePort, ethHdr->DA);
		memcpy(ethHdr->SA, myMacAddr, 6);
		if (dtype == EEE_SAC)
			ethHdr->type = EEE_ETH_TYPE_SAC;
		else
			ethHdr->type = EEE_ETH_TYPE;

		eeeHdr->pad = 0;
		eeeHdr->dest_port = pktDesc->hdr.dest_port;
		eeeHdr->src_port = pktDesc->hdr.src_port;
#if defined(NETEEE_FRAGMENT)
		eeeHdr->offset = pktDesc->hdr.offset;
#endif

		// Add header buffer to the edesc.
		pktDesc->bd[0].buf     = hdrBuf;
		pktDesc->bd[0].len     = hdrLen;
		pktDesc->bd[0].control = EDESC_BD_CMD_IN_USE | EDESC_BD_SMALL;
		pktLen += hdrLen;
		bufCnt++;

        pktDesc->hdr.control = 0;
        EEE_SET_PKT_LEN(pktDesc, pktLen);
        EEE_SET_NUM_BUFS(pktDesc, bufCnt);
        pktDesc->hdr.control |= EDESC_COMMAND_USE_FIRST_BUFFER;

		CACHE_FLUSH(hdrBuf, hdrBuf + hdrLen);
	}

	return NFX_OK;
}

/*-----------------------------------------------------------------
 * Name :	eee_removeHeader()
 * Description: strip the mac header and agile header, fill the descriptor header.
 * Returns: the size of the removed header
 *-----------------------------------------------------------------
 */
int
eee_removeHeader(eee_descPtr_t edesc)
{
	agile_hdr_t *eeeHdr = (agile_hdr_t *)(edesc->bd[1].buf
										  + sizeof(enet_hdr_t));
	int hdrlen = sizeof(enet_hdr_t) + sizeof(agile_hdr_t) + eeeHdr->pad;
	edesc->hdr.dest_port = eeeHdr->dest_port;
	edesc->hdr.src_port = eeeHdr->src_port;
#if defined(NETEEE_FRAGMENT)
	edesc->hdr.offset = eeeHdr->offset;
#endif		
	edesc->bd[1].buf += hdrlen;
	edesc->bd[1].len -= hdrlen;
	return hdrlen;
}

struct eee_rcv_queue *
eee_my_receive_queue(void)
{
	 int i;
	 uint32 my_fwd_q;
	 GET_MY_FWD_INDEX(my_fwd_q);
	 for (i = 0; i < EEE_NUM_FWD_QUEUES; i++) {
		 if (i == my_fwd_q)
			 continue;
		 return &eee.rcv_queue[i];
	 }
	 return NULL;
}

/*-----------------------------------------------------------------
 * Name:	eee_fwdInit()
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_fwdInit()
{
#ifdef LINUX_TEST
	uint32	cpu;
	int32	shmid, slot;
	uchar8	* shmptr;
#ifdef LINUX_POLL_NONBLOCKING_FD
    struct sockaddr_un sockAddr;
    char pathName[40];
#endif
	extern int32	eee_linuxForwardRemotePktUni();
	extern int32	eee_linuxForwardRemotePktMulti();
	extern void		eee_linuxForwardReceive();

#if 0
	extern int32	eee_linuxSockFwdPktUni();
	extern int32	eee_linuxSockFwdPktMulti();
	extern void		eee_linuxSockFwdRx();
#endif
#endif
	NFX_SPIN_INIT(&fragment_spin, SPINLOCK_LEVEL_UNKNOWN);

#if (EEE_NUM_FWD_QUEUES > 0)
	eee_my_receive_queue()->rcv_max = 64;
	eee_my_receive_queue()->rcv_hwm = 0;
	eee_registerPollingFunc(EEE_HIGH_PRIORITY,
		eee_processReceiveQueue,
		eee_my_receive_queue(),
		eee_fwdReceiveQueuePollCfgStruct);

#if defined(NFX_MOD_NFP)
	eee_registerTransmitFunc(NFP_PORT_LOCAL_QUEUE,
		eee_forwardLocalPktUni,
		eee_forwardLocalPktMulti,
		EDESC_TOTAL_LEN);
#elif defined(NFX_MOD_SSC)
#if !defined(AGILE_HARDWARE)
	eee_registerTransmitFunc(SSC_PORT_LOCAL_QUEUE,
		eee_forwardLocalPktUni,
		eee_forwardLocalPktMulti,
		EDESC_TOTAL_LEN);
#endif	
#endif

#endif

#if defined(NFX_SMP)
	NFX_SPIN_INIT(&local_queue.lq_spin, SPINLOCK_LEVEL_UNKNOWN);
	local_queue.max_val = 256;
	eee_registerPollingFunc(EEE_HIGH_PRIORITY,
		eee_poll_local_queue, &local_queue, eee_fwdLocalQueuePollCfgStruct);


#if defined(NFP_TXRX)
	NFX_SPIN_INIT(&local_queueA.lq_spin, SPINLOCK_LEVEL_UNKNOWN);
	local_queueA.max_val = 256;
	eee_txrxMapAcpu();
#endif /* NFP_TXRX */	

#endif	

#ifdef LINUX_TEST
/*
 * Management Plane queues + semaphores
 */
	/*
	 * Slot 0, SSC (cpu0 = 7000, MGMT has no MP port)
	 */
	if ((linux_mp_sem[0][0] = semget(SEM_MP_SSC, 0, 0)) == -1)
		printk("error getting MP_SSC semaphore\n");
	if ((shmid = shmget(SHM_SEG_MP_SSC, 0, SHM_R | SHM_W)) < 0) {
		printk("error in MP_SSC shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("MP_SSC shmat error\n");
	}
	printk("MP_SSC : base at 0x%x\n", shmptr);
	linux_mp[0][0] = (shm_tx_t *)shmptr;

	/*
	 * Slot 1, NFP (TXRX1, TXRX2, FP1, FP2)
	 */
	if ((linux_mp_sem[1][0] = semget(SEM_MP_TXRX1, 0, 0)) == -1)
		printk("error getting MP_TXRX1 semaphore\n");
	if ((shmid = shmget(SHM_SEG_MP_TXRX1, 0, SHM_R | SHM_W)) < 0) {
		printk("error in MP_TXRX1 shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("MP_TXRX1 shmat error\n");
	}
	printk("MP_TXRX1 : base at 0x%x\n", shmptr);
	linux_mp[1][0] = (shm_tx_t *)shmptr;

	if ((linux_mp_sem[1][1] = semget(SEM_MP_TXRX2, 0, 0)) == -1)
		printk("error getting MP_TXRX2 semaphore\n");
	if ((shmid = shmget(SHM_SEG_MP_TXRX2, 0, SHM_R | SHM_W)) < 0) {
		printk("error in MP_TXRX2 shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("MP_TXRX2 shmat error\n");
	}
	printk("MP_TXRX2 : base at 0x%x\n", shmptr);
	linux_mp[1][1] = (shm_tx_t *)shmptr;

	if ((linux_mp_sem[1][2] = semget(SEM_MP_FP1, 0, 0)) == -1)
		printk("error getting MP_FP1 semaphore\n");
	if ((shmid = shmget(SHM_SEG_MP_FP1, 0, SHM_R | SHM_W)) < 0) {
		printk("error in MP_FP1 shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("MP_FP1 shmat error\n");
	}
	printk("MP_FP1 : base at 0x%x\n", shmptr);
	linux_mp[1][2] = (shm_tx_t *)shmptr;

	if ((linux_mp_sem[1][3] = semget(SEM_MP_FP2, 0, 0)) == -1)
		printk("error getting MP_FP2 semaphore\n");
	if ((shmid = shmget(SHM_SEG_MP_FP2, 0, SHM_R | SHM_W)) < 0) {
		printk("error in MP_FP2 shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("MP_FP2 shmat error\n");
	}
	printk("MP_FP2 : base at 0x%x\n", shmptr);
	linux_mp[1][3] = (shm_tx_t *)shmptr;

	/*
	 * Slot 2, FCNIM (FC1, FC2, FC3, FC4)
	 */
	if ((linux_mp_sem[2][0] = semget(SEM_MP_FC1, 0, 0)) == -1)
		printk("error getting MP_FC1 semaphore\n");
	if ((shmid = shmget(SHM_SEG_MP_FC1, 0, SHM_R | SHM_W)) < 0) {
		printk("error in MP_FC1 shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("MP_FC1 shmat error\n");
	}
	printk("MP_FC1 : base at 0x%x\n", shmptr);
	linux_mp[2][0] = (shm_tx_t *)shmptr;

	if ((linux_mp_sem[2][1] = semget(SEM_MP_FC2, 0, 0)) == -1)
		printk("error getting MP_FC2 semaphore\n");
	if ((shmid = shmget(SHM_SEG_MP_FC2, 0, SHM_R | SHM_W)) < 0) {
		printk("error in MP_FC2 shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("MP_FC2 shmat error\n");
	}
	printk("MP_FC2 : base at 0x%x\n", shmptr);
	linux_mp[2][1] = (shm_tx_t *)shmptr;

	if ((linux_mp_sem[2][2] = semget(SEM_MP_FC3, 0, 0)) == -1)
		printk("error getting MP_FC3 semaphore\n");
	if ((shmid = shmget(SHM_SEG_MP_FC3, 0, SHM_R | SHM_W)) < 0) {
		printk("error in MP_FC3 shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("MP_FC3 shmat error\n");
	}
	printk("MP_FC3 : base at 0x%x\n", shmptr);
	linux_mp[2][2] = (shm_tx_t *)shmptr;

	if ((linux_mp_sem[2][3] = semget(SEM_MP_FC4, 0, 0)) == -1)
		printk("error getting MP_FC4 semaphore\n");
	if ((shmid = shmget(SHM_SEG_MP_FC4, 0, SHM_R | SHM_W)) < 0) {
		printk("error in MP_FC4 shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("MP_FC4 shmat error\n");
	}
	printk("MP_FC4 : base at 0x%x\n", shmptr);
	linux_mp[2][3] = (shm_tx_t *)shmptr;

/*
 * Data Plane queues + semaphores
 */
	/*
	 * Slot 1, NFP (TXRX, FP)
	 */
	if ((linux_dp_sem[1][0] = semget(SEM_DP_TXRX, 0, 0)) == -1)
		printk("error getting DP_TXRX semaphore\n");
	if ((shmid = shmget(SHM_SEG_DP_TXRX, 0, SHM_R | SHM_W)) < 0) {
		printk("error in DP_TXRX shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("DP_TXRX shmat error\n");
	}
	printk("DP_TXRX : base at 0x%x\n", shmptr);
	linux_dp[1][0] = (shm_tx_t *)shmptr;

	if ((linux_dp_sem[1][1] = semget(SEM_DP_FP, 0, 0)) == -1)
		printk("error getting DP_FP semaphore\n");
	if ((shmid = shmget(SHM_SEG_DP_FP, 0, SHM_R | SHM_W)) < 0) {
		printk("error in DP_FP shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("DP_FP shmat error\n");
	}
	printk("DP_FP : base at 0x%x\n", shmptr);
	linux_dp[1][1] = (shm_tx_t *)shmptr;

	/*
	 * Slot 2, FCNIM (FC1, FC2, FC3, FC4)
	 */
	if ((linux_dp_sem[2][0] = semget(SEM_DP_FC1, 0, 0)) == -1)
		printk("error getting DP_FC1 semaphore\n");
	if ((shmid = shmget(SHM_SEG_DP_FC1, 0, SHM_R | SHM_W)) < 0) {
		printk("error in DP_FC1 shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("DP_FC1 shmat error\n");
	}
	printk("DP_FC1 : base at 0x%x\n", shmptr);
	linux_dp[2][0] = (shm_tx_t *)shmptr;

	if ((linux_dp_sem[2][1] = semget(SEM_DP_FC2, 0, 0)) == -1)
		printk("error getting DP_FC2 semaphore\n");
	if ((shmid = shmget(SHM_SEG_DP_FC2, 0, SHM_R | SHM_W)) < 0) {
		printk("error in DP_FC2 shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("DP_FC2 shmat error\n");
	}
	printk("DP_FC2 : base at 0x%x\n", shmptr);
	linux_dp[2][1] = (shm_tx_t *)shmptr;

	if ((linux_dp_sem[2][2] = semget(SEM_DP_FC3, 0, 0)) == -1)
		printk("error getting DP_FC3 semaphore\n");
	if ((shmid = shmget(SHM_SEG_DP_FC3, 0, SHM_R | SHM_W)) < 0) {
		printk("error in DP_FC3 shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("DP_FC3 shmat error\n");
	}
	printk("DP_FC3 : base at 0x%x\n", shmptr);
	linux_dp[2][2] = (shm_tx_t *)shmptr;

	if ((linux_dp_sem[2][3] = semget(SEM_DP_FC4, 0, 0)) == -1)
		printk("error getting DP_FC4 semaphore\n");
	if ((shmid = shmget(SHM_SEG_DP_FC4, 0, SHM_R | SHM_W)) < 0) {
		printk("error in DP_FC4 shmget, shmid = 0x%x\n", shmid);
	}
	if ((shmptr = shmat(shmid, 0, 0)) == (void *) -1) {
		printk("DP_FC4 shmat error\n");
	}
	printk("DP_FC4 : base at 0x%x\n", shmptr);
	linux_dp[2][3] = (shm_tx_t *)shmptr;

	/*
	 * Local Forwarding queues on NFP and SSC
	 */
#if defined(NFX_MOD_NFP)
	if ((linux_fwd_sem[1][0] = semget(SEM_FWD_TO_TXRX, 0, 0)) == -1)
		printk("error getting FWD_TO_TXRX semaphore\n");
	if ((linux_fwd_sem[1][1] = semget(SEM_FWD_TO_FP, 0, 0)) == -1)
		printk("error getting FWD_TO_FP semaphore\n");
#endif

#ifdef LINUX_POLL_NONBLOCKING_FD
    /* Create, bind and register our poll indication socket */
    eee_linuxFwdPollRxFd = socket(AF_UNIX, SOCK_DGRAM, 0);
    if (eee_linuxFwdPollRxFd < 0)
    {
        perror("eee_fwd socket");
        panic("eee_fwd pollFd socket create failed");
    }
	if (fcntl(eee_linuxFwdPollRxFd, F_SETFL, O_NONBLOCK))
	{
		perror("eee_fwd fcntl");
		panic("eee_fwd pollFd socket fcntl failed");
	}
    sockAddr.sun_family = AF_UNIX;
    sprintf(pathName, "/tmp/EEE_SLOT_%d_CPU_%d", nfxMySlotId, nfxMyCPUId);
    strcpy(sockAddr.sun_path, pathName);
	/* unlink from previous run */
	unlink(pathName);
    if (bind(eee_linuxFwdPollRxFd, &sockAddr, sizeof(sockAddr)) != 0)
    {
        perror("eee_fwd bind");
        panic("eee_fwd pollFd bind of %s failed", pathName);
    }
    eee_addPollIndicationFd(eee_linuxFwdPollRxFd);
    /* Mark destination sockets as uninitialized, we'll initialize
     * each as needed.
     */
    for (slot = 0; slot < NFX_SLOT_MAX; slot++)
        for (cpu = 0; cpu < NFX_MAX_CPUS_PER_SLOT; cpu++) {
            eee_linuxFwdPollTxFd[slot][cpu] = -1;
            eee_linuxFwdPollTxFdState[slot][cpu] = -1;
        }
#endif

#if NFX_SOCKET_BACKPLANE
	eee_registerPollingFunc(EEE_HIGH_PRIORITY, eee_linuxSockFwdRx, NULL);
#endif // NFX_SOCKET_BACKPLANE

	eee_registerDefaultTransmitFunc(EEE_USER_DATA,
					eee_linuxForwardRemotePktUni,
					EDESC_TOTAL_LEN);
	eee_registerDefaultTransmitFunc(EEE_CONTROL,
					eee_linuxForwardRemotePktUni,
					EDESC_TOTAL_LEN);
	eee_registerDefaultTransmitFunc(EEE_MGMT,
					eee_linuxForwardRemotePktUni,
					EDESC_TOTAL_LEN);
	eee_registerDefaultTransmitFunc(EEE_JOURNAL,
					eee_linuxForwardRemotePktUni,
					EDESC_TOTAL_LEN);
	eee_registerDefaultTransmitFunc(EEE_METADATA,
					eee_linuxForwardRemotePktUni,
					EDESC_TOTAL_LEN);

#if defined(NFP_TXRX)

	/* 
	 * eee_registerTransmitFunc() will be done in luc-sim.c 
	 */
#elif defined(NFP_FP)

	eee_registerTransmitFunc(NFP_PORT_FP_ETH0,
					eee_linuxForwardRemotePktUni,
					eee_linuxForwardRemotePktMulti,
					EDESC_TOTAL_LEN);
	eee_registerTransmitFunc(NFP_PORT_FP_ETH1,
					eee_linuxForwardRemotePktUni,
					eee_linuxForwardRemotePktMulti,
					EDESC_TOTAL_LEN);
	eee_registerTransmitFunc(NFP_PORT_FP_ETH2,
					eee_linuxForwardRemotePktUni,
					eee_linuxForwardRemotePktMulti,
					EDESC_TOTAL_LEN);

#elif defined(FCNIM) && defined(CHEETAH)

	eee_registerTransmitFunc(FCNIM_PORT_FC,
					NULL,
					NULL,
					EDESC_TOTAL_LEN);
	eee_registerTransmitFunc(FCNIM_PORT_BP_DATA,
					eee_linuxForwardRemotePktUni,
					eee_linuxForwardRemotePktMulti,
					EDESC_TOTAL_LEN);
	eee_registerTransmitFunc(FCNIM_PORT_BP_MGMT,
					eee_linuxForwardRemotePktUni,
					eee_linuxForwardRemotePktMulti,
					EDESC_TOTAL_LEN);

#elif defined(NFX_MOD_SSC)

	eee_registerTransmitFunc(SSC_PORT_BP_MGMT,
					eee_linuxForwardRemotePktUni,
					eee_linuxForwardRemotePktMulti,
					EDESC_TOTAL_LEN);

#endif

#if defined(NFP_TXRX) || defined(NFP_FP) || (defined(FCNIM) && defined(CHEETAH))
	/* data plane */
	eee_registerPollingFunc(EEE_HIGH_PRIORITY, eee_linuxForwardReceive, (void *)0);
#endif
#if defined(NFP_TXRX) || defined(NFP_FP) || (defined(FCNIM) && defined(CHEETAH)) || defined(SSC)
	/* mgmt plane */
	eee_registerPollingFunc(EEE_HIGH_PRIORITY, eee_linuxForwardReceive, (void *)1);
#endif

#endif	/* LINUX_TEST */

}	/* eee_fwdInit() */

