/*
 *-----------------------------------------------------------------
 *
 * Name:	eee-fwd.h
 *
 * RCS:  $Id: eee-fwd.h,v 1.12 2002/11/13 01:14:33 maximk Exp $
 *
 * Copyright (C) 2001, Agile Storage, Inc.
 *
 * Description:	Forwarding definitions
 *
 *				All entries that are added to a forwarding queue must be
 *				64 byte aligned.  The bottom 6 bits are encoded with
 *				control information for the receiving side to know what to
 *				do with this entry.
 *
 * Created by:	Rick Lund
 *
 * Date Created:	5/9/01
 *
 *-----------------------------------------------------------------
 */

#ifndef _EEE_FWD_H
#define _EEE_FWD_H

/*
 * Format of entries in the forwarding queues
 *
 * So the way it works:
 *		if (ownership bit is set),
 *			check action field
 *			if (action == fwd)
 *				then the data is an EEE descriptor
 *			if (action == free)		// we can free a buffer, desc, or packet
 *				if (EEE_FWD_BUF is set)
 *					check BUF_SIZE_MASK and add to the proper buffer stack
 *				if (EEE_FWD_BUF is clear)
 *					if (EEE_FWD_DESC is set)
 *						then the data is an EEE descriptor, add it to the RQ
 *					if (EEE_FWD_DESC is clear)
 *						then the data is an EEE descriptor containing a packet
 *						free the buffers and the descriptor
 *
 */
#if BITS_PER_LONG == 64
#define EEE_FWD_ADDR_MASK		0xFFFFFFFFFFFFFFC0		/* 64 byte aligned */
#else
#define EEE_FWD_ADDR_MASK		0xFFFFFFC0		/* 64 byte aligned */
#endif

#define EEE_FWD_BUF_SIZE_SHIFT  4
#define EEE_FWD_BUF_SIZE_MASK	0x00000030
#define EEE_FWD_BUF_MGMT		(EEE_BUF_MGMT << EEE_FWD_BUF_SIZE_SHIFT)
#define EEE_FWD_BUF_ULTRA		(EEE_BUF_ULTRA << EEE_FWD_BUF_SIZE_SHIFT) 
#define EEE_FWD_BUF_LARGE		(EEE_BUF_LARGE << EEE_FWD_BUF_SIZE_SHIFT)
#define EEE_FWD_BUF_SMALL		(EEE_BUF_SMALL << EEE_FWD_BUF_SIZE_SHIFT)

#define EEE_FWD_DESC			0x00000008
#define EEE_FWD_PACKET			0x00000000

#define EEE_FWD_BUF				0x00000004

#define EEE_FWD_ACTION_MASK		0x00000002
#define EEE_FWD_ACTION_FREE		0x00000002
#define EEE_FWD_ACTION_FWD		0x00000000

#define EEE_FWD_WHAT_MASK 		(EEE_FWD_DESC|EEE_FWD_BUF|EEE_FWD_ACTION_MASK)
#define EEE_FWD_ACTION_FREE_BUF         (EEE_FWD_ACTION_FREE|EEE_FWD_BUF)
#define EEE_FWD_ACTION_FREE_DESC        (EEE_FWD_ACTION_FREE|EEE_FWD_DESC)

#define EEE_FWD_OWN			0x00000001

/*------------------------------------------------------------------
 *
 * Macros for forwarding
 *
 *------------------------------------------------------------------*/
#define EEE_FREE_DESC			EEE_FWD_ACTION_FREE | EEE_FWD_DESC
#define EEE_FREE_BUF			EEE_FWD_ACTION_FREE | EEE_FWD_BUF

/*-----------------------------------------------------------------
 * Name:	FWD_PACKET()
 *
 * Description:	Macro that will add entry to a shared memory fwd queue.
 *
 * Created by:	Rick Lund
 *
 * Date Created:	5/19/01
 *
 *-----------------------------------------------------------------
 */
#define FWD_PACKET(_data, _fwd_q_index, lock_queue)                  \
({                                                                   \
	int32	_rc;                                                     \
    struct eee_fwd_queue *queue = &eee.fwd_queue[_fwd_q_index];      \
    eee_descPtr_t *end = queue->fwd_end;                             \
    eee_descPtr_t *head;                                             \
    eee_descPtr_t * volatile *tail = queue->fwd_tail_ptr;            \
    eee_descPtr_t writeData = (eee_descPtr_t)(uintptr_t)(_data | EEE_FWD_OWN);  \
    uint32 fwd_cnt;                                                  \
                                                                     \
    if (lock_queue) {                                                \
        NFX_SPIN_LOCK(&queue->fwd_spin);                             \
    }                                                                \
    head = queue->fwd_head;                                          \
    fwd_cnt = queue->fwd_cnt;                                        \
    while (head == *tail) {                                          \
    ;                                                                \
    }                                                                \
   	*head = writeData;                                               \
	queue->fwd_cnt = fwd_cnt + 1;                                    \
	QUEUE_PTR_INC(head, end, queue->fwd_start);                      \
	_rc = EEE_FORWARDED_PACKET;                                      \
    queue->fwd_head = head;                                          \
                                                                     \
    if (lock_queue) {                                                \
        NFX_SPIN_UNLOCK(&queue->fwd_spin);                           \
    }                                                                \
	_rc;                                                             \
})                                                                   


/*-----------------------------------------------------------------
 * Name :	eee_initFragmentTimeout()
 *
 * Description: initialize the fragments timeout facility
 *
 * Created by:	Maxim Kozlovsky
 *
 * Date Created:	04/04/02
 *
 *-----------------------------------------------------------------
 */
void eee_initFragmentTimeout(void);

/*-----------------------------------------------------------------
 * Name:
 *
 * Description:
 *
 * Created by:	Rick Lund
 *
 * Date Created:
 *
 *-----------------------------------------------------------------
 */
void
eee_fwdInit(void);


#endif	/* _EEE_FWD_H */

