/*
 *-----------------------------------------------------------------
 * Name:	eee-ipc.c
 * Description:	IPC messaging functions
 * Copyright (C) 2009, OnStor, Inc.
 *-----------------------------------------------------------------
 */
#include "nfx-incl.h"
#include "../sm-eee/eee-api.h"
#include "../sm-chassis/cm-api.h"

ipc_client_t	ipc_client[EEE_IPC_NUM_CLIENTS];

int32	eee_ipc_app_rcv();

/*-----------------------------------------------------------------
 * Name:	eee_initIPCQueues()
 * Description:	allocate memory for IPC messaging queues
 *-----------------------------------------------------------------
 */
void
eee_initIPCQueues()
{
	eee_ipcEntry_t	* base;
	uint32 cpu, my_index, i, q_size;

	EEE_PRINTK(("eee_initIPCQueues()\n"));
	GET_MY_IPC_INDEX(my_index);
	EEE_PRINTK(("my_index = %d\n", my_index));

	for (i=0; i<EEE_NUM_IPC_QUEUES; i++) {
		if (i == my_index)
			continue;
		q_size = EEE_IPC_NUM_ENTRIES * (EEE_IPC_ENTRY_SIZE * sizeof(uint32));
		base = (eee_ipcEntry_t *)eee_allocateBlock(q_size);
		eee.ipc_rcv_curr[i] = (volatile eee_ipcEntry_t *)base;
		eee.ipc_rcv_start[i] = base;
		eee.ipc_rcv_end[i] = eee.ipc_rcv_start[i] + EEE_IPC_NUM_ENTRIES;
	}
}

/*-----------------------------------------------------------------
 * Name:	eee_registerIPCClient
 * Description:	To receive IPC messages, client must register an IPC receive function.
 *
 * Return Values: NFX_OK, EEE_ERR_IPC_BAD_CLIENT,
 *-----------------------------------------------------------------
 */
int32
eee_registerIPCClient(uint32 client_id,	/* from client_id table */
	int32	(*ipc_rcv_func)( /* function called when IPC for this client_id is received */
	uint32	ipc_command,	 /* from sender's sendIPC call */
	uint32	data_size,	 /* number of valid IPC bytes */
	uint32	*ipc_data,	 /* IPC data */
	uint32	*ipc_response,	 /* response from receiver */
	uint32	*ipc_response_data_size,  /* size of IPC response */
	uint32	**ipc_response_data))	  /* IPC response data */
{
	if (client_id >= EEE_IPC_NUM_CLIENTS)		/* invalid id */
		return EEE_ERR_IPC_BAD_CLIENT;

	if (ipc_client[client_id].func)				/* already registered */
		return EEE_ERR_IPC_BAD_CLIENT;

	ipc_client[client_id].func = ipc_rcv_func;
	return NFX_OK;
}

/*-----------------------------------------------------------------
 * Name:	eee_sendIPC()
 * Description:	To send an IPC message
 * Return Values:
 * NFX_OK
 * EEE_ERR_BAD_PARAM
 * EEE_ERR_IPC_ACK_TIMEOUT
 * EEE_ERR_IPC_BAD_CLIENT
 * EEE_ERR_IPC_NO_SPACE
 * EEE_ERR_IPC_NACK
 *-----------------------------------------------------------------
 */
int32
eee_sendIPC(uint32 dest_cpu,	/* IPC_NIM_RX, IPC_NIM_TX, IPC_FP1, IPC_FP2 */
	uint32	client_id,	/* from client_id table */
	uint32	control,
	uint32	command,	/* command to be sent to receive function */
	uint32	data_size,	/* num bytes of valid buffer data */
	uint32	*ipc_data,	/* local buffer containing IPC message */
	uint32	*ipc_response,	/* if desired, area for the response */
	uint32	*ipc_response_data_size, /* if desired, area for size of response message */
	uint32	**ipc_response_data)	/* if desired, buffer for response data */
{
	volatile eee_ipcEntry_t	* ipc_ptr;
	uint32 my_ipc_q, i, j, cont, size;
	int32 ipc_q, ret;

	ipc_q = GET_IPC_Q_INDEX(dest_cpu);
	if (ipc_q < 0)
		return EEE_ERR_BAD_PARAM;
	if (client_id >= EEE_IPC_NUM_CLIENTS)
		return EEE_ERR_IPC_BAD_CLIENT;

	ipc_ptr = eee.ipc_send_curr[ipc_q];
	GET_MY_IPC_INDEX(my_ipc_q);
	if (ipc_q == my_ipc_q) {
		printk("IPC destined for self\n");
		return NFX_OK;
	}
	ret = NFX_OK;

	for (i=0; i<IPC_WAIT_FOR_SPACE_COUNT; i++) {

		if ((ipc_ptr->control & IPC_CONT_OWN) == 0) {
			ipc_ptr->client_id = client_id;
			ipc_ptr->command = command;

			if (data_size)
				memcpy((uchar8 *)&(ipc_ptr->msg[0]), ipc_data, data_size);

			ipc_ptr->control = IPC_CONT_OWN | data_size;
			if (control & IPC_ACK_REQ) ipc_ptr->control |= IPC_CONT_ACK_REQ;

			if (control & IPC_ACK_REQ) {
				for (j=0; j<IPC_WAIT_FOR_ACK_COUNT; j++) {

					cont = ipc_ptr->control;
					if (cont & IPC_CONT_RESP_ACK) {

						size = ipc_ptr->control & IPC_CONT_SIZE_MASK;
						if (ipc_response)
							*ipc_response = ipc_ptr->command;
						if (ipc_response_data_size)
							*ipc_response_data_size = size;
						if (ipc_response_data)
							*ipc_response_data = (uint32 *)&(ipc_ptr->msg[0]);

						eee_stats->ipc_ok[client_id]++;
						ipc_ptr->control = 0;
						break;
					}
					else if (cont & IPC_CONT_RESP_NACK) {
						eee_stats->ipc_nack[client_id]++;
						ipc_ptr->control = 0;
						ret = EEE_ERR_IPC_NACK;
						break;
					}
				}
			}
			else
				eee_stats->ipc_ok[client_id]++;

			QUEUE_PTR_INC(	eee.ipc_send_curr[ipc_q],
							eee.ipc_send_end[ipc_q],
							eee.ipc_send_start[ipc_q]);

			if (j == IPC_WAIT_FOR_ACK_COUNT)
				return EEE_ERR_IPC_ACK_TIMEOUT;

			return (ret);
		}
	}
	eee_stats->ipc_no_space[ipc_q]++;
	return EEE_ERR_IPC_NO_SPACE;
}

/*-----------------------------------------------------------------
 * Name:	eee_checkIPCQueues(cb, tref)
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_checkIPCQueues(void	* cb, uint32	tref)
{
	volatile eee_ipcEntry_t	*ipc_ptr;
	uint32 my_ipc_q;
	uint32 ipc_q, cont, cmd, client, len;
	uint32 resp_cont, resp, resp_size, *resp_msg;
	int32  rc;

	GET_MY_IPC_INDEX(my_ipc_q);

	for (ipc_q=0; ipc_q<EEE_NUM_IPC_QUEUES; ipc_q++) {

		if (ipc_q == my_ipc_q)
			continue;

		ipc_ptr = eee.ipc_rcv_curr[ipc_q];

		while ((cont = ipc_ptr->control) & IPC_CONT_OWN) {

			cmd = ipc_ptr->command;
			client = ipc_ptr->client_id;	
			len = cont & IPC_CONT_SIZE_MASK;

			if (client > EEE_IPC_NUM_CLIENTS) {
				printk("IPC error, client_id 0x%x\n", client);
				if (cont & IPC_CONT_ACK_REQ)
					resp_cont = IPC_CONT_RESP_NACK;
			}

			if (!(ipc_client[client].func)) {
				printk("IPC error, no client function, 0x%x\n", client);
				if (cont & IPC_CONT_ACK_REQ)
					resp_cont = IPC_CONT_RESP_NACK;
			}

			ipc_client[client].msgs_rcvd++;
			rc = (*ipc_client[client].func)(cmd, len, ipc_ptr->msg,
										&resp, &resp_size, &resp_msg);

			if (cont & IPC_CONT_ACK_REQ) {
				if (rc == NFX_OK) {
					ipc_ptr->command = resp;
					if (resp_size <= EEE_IPC_DATA_SIZE_MAX*4) {
						memcpy((uchar8 *)&(ipc_ptr->msg[0]), resp_msg, resp_size);
						resp_cont = IPC_CONT_RESP_ACK | resp_size;
					}
					else
						resp_cont = IPC_CONT_RESP_NACK;
				}
				else {
					resp_cont = IPC_CONT_RESP_NACK;
				}
			}
			ipc_ptr->control = resp_cont;
			QUEUE_PTR_INC(ipc_ptr, eee.ipc_rcv_end[ipc_q], eee.ipc_rcv_start[ipc_q]);
		}
		eee.ipc_rcv_curr[ipc_q] = ipc_ptr;
	}
}

/*-----------------------------------------------------------------
 * Name:	eee_showIPCStats()
 * Description:	Display counts for messages to IPC clients
 *-----------------------------------------------------------------
 */
void
eee_showIPCStats()
{
	uint32	i;

	printk("\nIPC counters:\n");
	printk("   client    sent OK    sent NACK    rcv\n");
	printk("   ------    -------    ---------    ---\n");

	for (i=0; i<EEE_IPC_NUM_CLIENTS; i++) {
		if ((ipc_client[i].msgs_rcvd) || (eee_stats->ipc_ok) ||	(eee_stats->ipc_nack))
			printk("   %4d       %4d         %4d     %4d\n",
		 i, eee_stats->ipc_ok[i], eee_stats->ipc_nack[i], ipc_client[i].msgs_rcvd);
	}
	for (i=0; i<EEE_NUM_IPC_QUEUES; i++) {
/*
		if (eee_stats->ipc_msg_too_big[i])
*/
		printk("   IPC queue %d message size errors : %d\n", i, eee_stats->ipc_msg_too_big[i]);
	}
	for (i=0; i<EEE_NUM_IPC_QUEUES; i++) {
/*
		if (eee_stats->ipc_no_space[i])
*/
		printk("   IPC queue %d full : %d\n", i, eee_stats->ipc_no_space[i]);
	}
}

uint32	ipc_resp_msg[80];

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
int32 eee_ipc_app_rcv(uint32	cmd,
	uint32	size,
	uint32	* data,
	uint32	* resp_cmd,
	uint32	* resp_size,
	uint32	** resp_data)
{
	printk("eee_ipc_app_rcv(%d, %d)\n", cmd, size);

	switch (cmd) {
		case EEE_IPC_CMD_SYNC:
			
			ipc_resp_msg[0] = 0x12345678;

			if (data[0] == 0x30303030) {
				*resp_cmd = 0xa5a5a5a5;
				*resp_size = 4;
				*resp_data = &(ipc_resp_msg[0]);

				printk("returning OK\n");
				return NFX_OK;
			}
			else {
				printk("Error, data[0] != 0x30303030\n");
				return -1;
			}

		default:
			printk("Error, unknown command, 0x%x\n", cmd);
			return -1;
	}
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_ipcInit()
{
	uint32	i;

	for (i=0; i<EEE_IPC_NUM_CLIENTS; i++)
		ipc_client[i].func = 0;

	if (eee_registerPollingFunc(EEE_HIGH_PRIORITY, eee_checkIPCQueues, NULL) < 0)
		printk("Error registering polling function, eee_checkIPCQueues\n");

	eee_registerIPCClient(IPC_CLIENT_EEE, eee_ipc_app_rcv);
}

