/*
 *-----------------------------------------------------------------
 * Name:	eee-mem.c
 * Description:	Memory initialization and allocation routines for EEE layer.
 *              These are NOT the routines for generalmemory management of the heap (malloc(), dealloc())!
 * Copyright (C) 2009, Onstor, Inc.
 *-----------------------------------------------------------------
 */
#include "nfx-incl.h"
#include "../sm-eee/eee-api.h"
#ifdef DMALLOC
#include "dmalloc.h"
#endif

/*
 * Resource Queue sizes
 */
#ifdef NFX_MOD_NFP
#ifdef EEE_BUF_MGMT
rq_sizes_t	rq_size[EEE_NUM_MEM_POOLS] = {
	{0,		0,		0,		0,	0},
	{RQ_TXRX_SMALL_BUF,	RQ_TXRX_LARGE_BUF,	RQ_TXRX_ULTRA_BUF,	RQ_TXRX_MGMT_BUF, RQ_TXRX_DESC},
	{RQ_FP_SMALL_BUF,	RQ_FP_LARGE_BUF,	RQ_FP_ULTRA_BUF,	RQ_FP_MGMT_BUF, RQ_FP_DESC},
};
#else
rq_sizes_t	rq_size[EEE_NUM_MEM_POOLS] = {
	{0,		0,		0,		0},
	{RQ_TXRX_SMALL_BUF,	RQ_TXRX_LARGE_BUF,	RQ_TXRX_ULTRA_BUF,	RQ_TXRX_DESC},
	{RQ_FP_SMALL_BUF,	RQ_FP_LARGE_BUF,	RQ_FP_ULTRA_BUF,	RQ_FP_DESC},
};
#endif

uint32	eee_index[3][EEE_NUM_CPUS] = {
	{ 1, 1, 2, 2, 1, 2 },		/* mem pool index */
	{ 0, 1, 2, 3, 0, 2 },		/* ipc queue index */
	{ 0, 0, 1, 1, 0, 1 },		/* fwd queue index */
};

#if defined(NFP_TXRX)
uint32	fwd_size[EEE_NUM_FWD_QUEUES] = {
	FWD_TXRX_TXRX, FWD_TXRX_FP,
};
#elif defined(NFP_FP)
uint32	fwd_size[EEE_NUM_FWD_QUEUES] = {
	FWD_FP_TXRX, FWD_FP_FP,
};
#endif

#endif	/* NFX_MOD_NFP */

/*-----------------------------------------------------------------
 * Name:	eee_memInit()
 * Description:	Set up memory pool
 *-----------------------------------------------------------------
 */
void
eee_memInit(uint32	pass)
{
	if (pass == 1) {
		/*
		 * Memory pool should be aligned on ALIGN_SIZE boundary.  All updates
		 * to curr_addr will keep the alignment on ALIGN_SIZE boundaries.
		 *
		 * So, bytes_avail will always decrement by multiples of ALIGN_SIZE.
		 */
		eee.sm_pool.curr_addr = (uint32 *)eee_platGetSharedMemoryBase();
		if ((address_t)eee.sm_pool.curr_addr & (ALIGN_SIZE-1))
			printf("Invalid base address, %p\n", eee.sm_pool.curr_addr);
		eee.sm_pool.bytes_avail = eee.sm_size;
	} else {
#ifdef EEE_BUF_MGMT
		eee.mgmt_pool.curr_addr = (uint32 *)(eee.mgmt_space + eee.mgmt_size_buf);
		eee.mgmt_pool.bytes_avail = eee.mgmt_size_cont;
#endif
	}
}

/*-----------------------------------------------------------------
 * Name:	eee_allocateBlock
 *
 * Description:	allocates a block of memory aligned on cacheline boundary.
 *		This function should be used at init time (by drivers, etc)
 *		to preallocate a piece of memory.  The caller should assume
 *		that this memory is NOT being allocated from a
 *		shared-between-CPUs pool.
 *-----------------------------------------------------------------
 */
void *					/* returns memory base address, or NULL */
eee_allocateBlock(uint32 num_bytes)
{
	uint32	alloc_bytes;
	uint32	alloc_words;
	uint32	alloc_lines;
	uint32	* ptr;
#ifndef LARGE_MEMORY
	uint32	i;
#endif	

	if (num_bytes == 0)
		num_bytes = 1;

	alloc_lines = (num_bytes + ALIGN_SIZE - 1) >> ALIGN_SHIFT;
	alloc_words = alloc_lines * 8;
	alloc_bytes = alloc_words * 4;

	if (alloc_bytes > eee.sm_pool.bytes_avail) {
		ptr = NULL;
		goto alloc_mem_return;
	}

	ptr = (uint32 *)eee.sm_pool.curr_addr;
#ifdef LARGE_MEMORY
	watchdog_stop();
#endif
#ifndef LARGE_MEMORY
	for (i = 0; i < alloc_words; i++) {
		*(ptr + i) = 0;
	}
#endif
#ifdef LARGE_MEMORY
	watchdog_restart();
#endif

	eee.sm_pool.curr_addr += alloc_words;
	eee.sm_pool.bytes_avail -= alloc_bytes;

alloc_mem_return:
	EEE_PRINTF(("Allocating 0x%x bytes, ", num_bytes));
	EEE_PRINTF(("remaining = 0x%x, ", eee.sm_pool.bytes_avail));
	EEE_PRINTF(("Base address = %p\n", ptr));

	return ptr;

}

/*-----------------------------------------------------------------
 * Name:	eee_allocateAlignedBlock
 *
 * Description:	allocates a block of memory aligned according to
 *		the size requested (2 KB bytes, aligned on 2 KB byte boundary).
 *-----------------------------------------------------------------
 */
void *					/* returns memory base address, or NULL */
eee_allocateAlignedBlock(uint32	num_bytes)
{
	address_t	addr, aba, mask;
	uint32	nl, nw, nb;

	EEE_PRINTF(("Allocating aligned 0x%x bytes\n", num_bytes));
	/*
	 * Find highest order bit on which to align allocation
	 */
	for (mask = 0x1F; mask <= 0xFFFFFFF; mask = (mask << 1) | 1) {
		if (!(~mask & num_bytes))
			break;
	}
	mask >>= 1;

	EEE_PRINTF(("curr_addr = %p, mask = 0x%lx\n", eee.sm_pool.curr_addr, mask));

	addr = (address_t)eee.sm_pool.curr_addr;
	aba = (addr+mask) & ~mask;
	nl = (aba-(address_t)eee.sm_pool.curr_addr) >> ALIGN_SHIFT;
	nw = nl << 3;
	nb = nw << 2;
	eee.sm_pool.curr_addr += nw;
	eee.sm_pool.bytes_avail -= nb;

	return (eee_allocateBlock(num_bytes));

}	/* eee_allocateAlignedBlock() */

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
uint32
eee_remainBytes()
{
	return eee.sm_pool.bytes_avail;
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void *
eee_mgmtAlloc(uint32 num_bytes)
{
        uint32 alloc_bytes, alloc_words, alloc_lines, i, *ptr;

	alloc_lines = (num_bytes + ALIGN_SIZE - 1) >> ALIGN_SHIFT;
	alloc_words = alloc_lines * 8;
	alloc_bytes = alloc_words * 4;

	if (alloc_bytes > eee.mgmt_pool.bytes_avail)
		return NULL;

	ptr = (uint32 *)eee.mgmt_pool.curr_addr;
	for (i = 0; i < alloc_words; i++) {
		*(ptr + i) = 0;
	}

	eee.mgmt_pool.curr_addr += alloc_words;
	eee.mgmt_pool.bytes_avail -= alloc_bytes;

	EEE_PRINTF(("Allocating 0x%x bytes, ", num_bytes));
	EEE_PRINTF(("remaining = 0x%x, ", eee.mgmt_pool.bytes_avail));
	EEE_PRINTF(("Base address = %p\n", ptr));
	return ptr;
}
