/*
 *-----------------------------------------------------------------
 * Name:	eee-poll.c
 * Description:	EEE layer polling routines
 * Copyright (C) 2009, OnStor, Inc.
 *-----------------------------------------------------------------
 */
#include "nfx-incl.h"
#include "../sm-eee/eee-api.h"
#include "../sm-timer/timer-api.h"
#include <smp-api.h>

#ifdef NFP_TXRX
#include "../sm-except/except-api.h"
#include "../sm-utils/txrx-utils-api.h"
#endif

extern void rconPoll(void);
volatile int32 dont_poll CPU_PRIVATE;
volatile int32 seen_dont_poll CPU_PRIVATE;

/*-----------------------------------------------------------------
 * Name:	eee_registerPollingFunc(priority, polling_func, cb)
 *
 * Description:	Polling functions will be called periodically.
 *		Drivers should use polling functions to process receive
 *		rings, transmit cleanup, etc.
 *		The polling function will be called with the user-supplied
 *		control block, and a timer reference value.  The timer
 *		reference will be utilized via some EEE layer macros.
 *
 * Arguments:
 * priority => EEE_HIGH_PRIORITY, EEE_LOW_PRIORITY
 * void	(*polling_func)( => function to be called as part of EEE layer polling loop
 *   void	* cb, =>  user defined control block for polling function
 *   uint32	timer_ref => indicate how much time this routine can use the CPU, or
 *		   how many packets to process in a loop - supplied by the EEE layer
 * Return Values:
 *
 * Returns handle for polling function OR
 * EEE_ERR_NOT_INITED
 * EEE_ERR_POLLING_TABLE_FULL
 * EEE_ERR_BAD_PARAM
 */
int32
eee_registerPollingFunc(uint32	priority,
	void	(*polling_func)(void *cb, uint32 timer_ref),
	void	* cb,
	eee_poll_cfg_t	* poll_cfg_struct)
{
	uint32 i;
	int32 rc = 0;

	if (!eee.init_done)
		return EEE_ERR_NOT_INITED;

	switch (priority) {
	case EEE_HIGH_PRIORITY:
		for (i=0; i<EEE_MAX_POLL_HIGH_ROUTINES - 1; i++) {

			if (eee.pfunc.high[i].func == NULL) {
				eee.pfunc.high[i].func = polling_func;
				eee.pfunc.high[i].cb = cb;
				eee.pfunc.high[i].poll_cfg_struct = poll_cfg_struct;

				return (i | EEE_HIGH_PRIORITY_IND);
			}
		}
		rc = EEE_ERR_POLLING_TABLE_FULL;
		break;

	case EEE_LOW_PRIORITY:
		for (i=0; i<EEE_MAX_POLL_LOW_ROUTINES - 1; i++) {
			if (eee.pfunc.low[i].func == NULL) {
				eee.pfunc.low[i].func = polling_func;
				eee.pfunc.low[i].cb = cb;
				eee.pfunc.low[i].poll_cfg_struct = poll_cfg_struct;
				return (i | EEE_LOW_PRIORITY_IND);
			}
		}
		rc = EEE_ERR_POLLING_TABLE_FULL;
		break;
	default:
		printf("Invalid EEE priority, 0x%x\n", priority);
		rc = EEE_ERR_BAD_PARAM;
	}
	return rc;
}

#if 0
/*-----------------------------------------------------------------
 * Name:	eee_unregisterPollingFunc(handle)
 *
 * Description:	Basically provided for completeness.  A polling function
 *				can be removed from polling loop.
 *
 * Return Values:
 * NFX_OK, EEE_ERR_POLLING_FUNC_INVALID, EEE_ERR_BAD_PARAM
 *-----------------------------------------------------------------
 */
int32
eee_unregisterPollingFunc(int32	handle) /* from eee_registerPollingFunc() return code */
{
	int32	index;

	if (handle & EEE_HIGH_PRIORITY_IND) {
		index = (handle & ~EEE_HIGH_PRIORITY_IND);
		if ((index >= 0) && (index < EEE_MAX_POLL_HIGH_ROUTINES)) {
			if (eee.pfunc.high.func[index] != NULL) {
				eee.pfunc.high.func[index] = NULL;
				return NFX_OK;
			}
			else
				return EEE_ERR_POLLING_FUNC_INVALID;
		}
		else
			return EEE_ERR_BAD_PARAM;
	}
	else if ((handle >= 0) && (handle < EEE_MAX_POLL_LOW_ROUTINES)) {
		if (eee.pfunc.low.func[handle] != NULL) {
			eee.pfunc.low.func[handle] = NULL;
			return NFX_OK;
		}
		else
			return EEE_ERR_POLLING_FUNC_INVALID;
	}
	else
		return EEE_ERR_BAD_PARAM;
}	/* eee_unregisterPollingFunc() */
#endif

/*-----------------------------------------------------------------
 * Name:	eee_poll()
 * Description:	Main polling loop
 *-----------------------------------------------------------------
 */
#ifdef EEE_POLLING_BUCKET
int32 eeePollingBuckets[EEE_NBR_POLLING_BUCKETS] = { 500,1000,10000,30000,100000,100000000};
#endif

#ifdef EEE_POLLING_PERF
int64 eeePollingCounter = 0;
int64 eeePollingCounterMin = 999999999;
int64 eeePollingCounterMax = 0;
int64 eeePollingCountersHist[EEE_POLLING_HIST] = {0,0,0,0,0,0,0,0,0,0};
int32 eeePollingCounterIndex = 0;
int32 eeePollingPrintf = 0;
#endif

#if defined(NFP_TXRX)

#include "../sm-req-queue/req-queue-api.h"
#include "../sm-tpl/tpl.h"
#include "../sm-tpl/tpl-ipc.h"

extern void eee_processTimers(int, int);
extern void sb1250dm_movCompletePoll_acpu(void);
extern void cmGpioCardStatePoll(void * cb, uint32 tref);
extern int eee_poll_local_queueA(void *, int);   
extern int32 tpl_ipcAcpuProc(tpl_ipcq_t *);
extern int32 tpl_ipcAcpuEvntRxPktProc(void);
extern uint32 max_timers_per_loop;
extern int eee_process_FP_TxRx_queue(struct eee_rcv_queue *queue,
                                     int max_count);
extern int eee_check_TxRx_FP_queue_waiters(void *cb, int tref);


/* Number of requests to run before running the timers. 
 */
#define REQUEST_PER_TIMER_POLL 256

/*
Routine Description:
    Run TxRx ACPU polling routines that need to be invoked periodically.
Arguments:
    None.
Return Value:
    None.
 */
void
run_periodic_polls(void)
{
    eee_processTimers(max_timers_per_loop, 0); 
    cmGpioCardStatePoll(NULL, 0);
}

/*
Routine Description:
    Poll the periodic polling routines if the number of requests
    processed is greater than REQUEST_PER_TIMER_POLL.
Arguments:
    None.
Return Value:
    None.
 */
#define check_periodic_polls()                      \
if (total_processed >= REQUEST_PER_TIMER_POLL) {    \
    total_processed -= REQUEST_PER_TIMER_POLL;      \
    run_periodic_polls();                           \
}

/*
Routine Description:
    TxRx ACPU polling function.
Arguments:
    num_loops - number of polling loops to run before returning.
Return Value:
    None.
 */
void
eee_txrxACPU_poll(uint32 num_loops)
{
    int idle_passes = 0; 
    int total_processed = 0;
    int requests_processed, messages_processed, tpl_events_processed;
    boolean idle_pass = TRUE;

    while (num_loops--) {
#ifdef USE_WATCHDOG
        if ((num_loops & 0x3) == 0)
            wd_kick();
#endif

        boolean idle_pass = TRUE;
        do { /* Process all pending requests */
            do {
                requests_processed = (req_poll_queue(
                                          &req_queues[REQ_PRIORITY_WAKEUP],
                                          REQUEST_PER_TIMER_POLL));

                if (requests_processed != 0) {
                    total_processed += requests_processed;
                    idle_pass = FALSE;
                }

                check_periodic_polls();
            } while (requests_processed == REQUEST_PER_TIMER_POLL);

            do { /* Process filesystem responses */

                messages_processed = (eee_process_FP_TxRx_queue(
                        &eee.rcv_queue[EEE_FS_RPC_QUEUE_INDEX],
                        REQUEST_PER_TIMER_POLL));
                
                if (messages_processed != 0) {
                    total_processed += messages_processed;
                    idle_pass = FALSE;
                }

                check_periodic_polls();

            } while (messages_processed == REQUEST_PER_TIMER_POLL);

            do { /* Process filesystem and SSC responses */

                messages_processed = (eee_poll_local_queueA(
                                          &local_queueA,
                                          REQUEST_PER_TIMER_POLL));
                                  
                if (messages_processed != 0) {
                    total_processed += messages_processed;
                    idle_pass = FALSE;
                }

                check_periodic_polls();
                
            } while (messages_processed == REQUEST_PER_TIMER_POLL);
			
			/* Process the datamover completions */
			sb1250dm_movCompletePoll_acpu();

            /* Process TPL events excluding the received packets
	     */
            tpl_events_processed = tpl_ipcAcpuProc(&(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_PKTS_COMP]));

            if (tpl_events_processed != 0) {
                total_processed += tpl_events_processed;
                idle_pass = FALSE;
            }

            check_periodic_polls();

            /* If processed some pending requests or filesystem or SSC
             * responses or TPL events, start over to check if any requests
             * were woken up.
             */
            
        } while ((messages_processed != 0) || (requests_processed != 0) ||(tpl_events_processed != 0));

        /* Process new requests if there are no current running requests and
         * no pending filesystem or SSC responses.
         */
        int packets_processed = tpl_ipcAcpuEvntRxPktProc();

        total_processed += packets_processed;
        
        check_periodic_polls();

        /* If processed any incoming packets, check if any requests were woken
         * up, otherwise process low priority requests.
         */
        if (packets_processed == 0) {

            if (!io_throttle_enabled ||
                ((!SIMPLEQ_EMPTY(&req_queues[REQ_PRIORITY_NEW])) &&
                 (io_throttle_try_reserve_req() == STATUS_SUCCESS))) {
                requests_processed =
                    req_poll_queue(&req_queues[REQ_PRIORITY_NEW], 1);

            }
            if (idle_pass) {
                idle_passes++;
                if (idle_passes == 8) {
                    run_periodic_polls();
                    idle_passes = 0;
                    total_processed = 0;
                } else if (total_processed >= REQUEST_PER_TIMER_POLL) {
                    run_periodic_polls();
                    total_processed -= REQUEST_PER_TIMER_POLL;
                    idle_passes = 0;
                }
            }
        }
    }
}
#endif

void
eee_poll(uint32	num_loops)
{
	struct eee_poll_cb * poll_cb, * high = &(eee.pfunc.high[0]);

#ifdef EEE_POLLING_PERF
    eeeRTC_t duration;
	eeeRTC_t lastPollingTime;
	eeeRTC_t interval;
#endif            
#if defined(EEE_POLLING_BUCKET) || defined(EEE_POLLING_PERF)
	boolean first = TRUE;
#endif
#ifdef EEE_POLLING_BUCKET
	eeeRTC_t eeePollingBucketsRTC[EEE_NBR_POLLING_BUCKETS];
#endif

	if (dont_poll) {
        seen_dont_poll = TRUE;
#ifdef USE_WATCHDOG
		watchdog_kick();
#endif
		return;
	}

	PREFETCH(PREF_LOAD,high,0);

#ifdef NFX_SMP
	rconPoll();
#endif

#if defined(EEE_POLLING_BUCKET) || defined(EEE_POLLING_PERF)
	if (first) {
		first = FALSE;
#ifdef EEE_POLLING_BUCKET
		{
			int32 i;
			// Setup buckets in RTC form
			for (i=0; i<EEE_NBR_POLLING_BUCKETS; i++) {
				eee_usecToRTC((uint64)eeePollingBuckets[i], &(eeePollingBucketsRTC[i]));
			}
		}
#endif
#ifdef EEE_POLLING_PERF
        lastPollingTime = *(eee_rtcZero());
#endif
	}
#endif

#if defined(NFP_TXRX)
    if (!CM_IS_TXRX_NCPU) {
        eee_txrxACPU_poll(num_loops);
        return;
    }
#endif    

	while (num_loops--) {
#ifdef USE_WATCHDOG
        if ((num_loops & 0x3) == 0)
            wd_kick();
#endif
		for (poll_cb = high; poll_cb->func; ++poll_cb) {
#ifdef EEE_POLLING_BUCKET
			eeeRTC_t start, delta;
			uint32 bucket = 0, loop;

			eee_rtc(&start);
#endif            
			PREFETCH(PREF_LOAD, poll_cb, 32);
			(poll_cb->func)(poll_cb->cb, 2);

#ifdef EEE_POLLING_BUCKET
			eee_rtcDuration (&start, &delta);
			for (loop = 0; loop < EEE_NBR_POLLING_BUCKETS; loop++) {
				if (eee_rtcCompare(&delta, &(eeePollingBucketsRTC[loop])) <= 0)	{
					bucket = loop;
					break;
				}
			}
			if (loop >= EEE_NBR_POLLING_BUCKETS) {
				bucket = EEE_NBR_POLLING_BUCKETS - 1;
				if (eeePollingPrintf) {
					eeeGMT_t startTV, endTV;
					eeeRTC_t end;

					eee_rtcToTV(&start, &startTV);
					eee_rtcAdd(&start, &delta, &end);
					eee_rtcToTV(&end, &endTV);
					printf ("POLLING TIME %d %d %d %d %lld\n", 
							startTV.sec, startTV.usec, endTV.sec, endTV.usec,
							eee_rtcToUsec(&delta));
				}
			}
			poll_cb->bucket[bucket] += 1;
#endif
		}

		for (poll_cb = &eee.pfunc.low[0]; poll_cb->func; ++poll_cb) {
#ifdef EEE_POLLING_BUCKET
			eeeRTC_t start, delta;
			uint32 bucket = 0, loop;

			eee_rtc(&start);
#endif
			poll_cb->func(poll_cb->cb, 2);

#ifdef EEE_POLLING_BUCKET
			eee_rtcDuration (&start, &delta);
			for (loop = 0; loop < EEE_NBR_POLLING_BUCKETS; loop++) {
				if (eee_rtcCompare(&delta, &(eeePollingBucketsRTC[loop])) <= 0) {
					bucket = loop;
					break;
				}
			}
			if (loop >= EEE_NBR_POLLING_BUCKETS) {
				bucket = EEE_NBR_POLLING_BUCKETS - 1;
				if (eeePollingPrintf) {
					eeeGMT_t startTV, endTV;
					eeeRTC_t end;

					eee_rtcToTV(&start, &startTV);
					eee_rtcAdd(&start, &delta, &end);
					eee_rtcToTV(&end, &endTV);
					printf ("POLLING TIME %d %d %d %d %lld\n", 
							startTV.sec, startTV.usec, endTV.sec, endTV.usec,
							eee_rtcToUsec(&delta));
				}
			}
			poll_cb->bucket[bucket] += 1;
#endif
		}
	}

#ifdef EEE_POLLING_PERF
	/* This code records the number of poll loops handled per sec, for the
	   last EEE_POLLING_HIST seconds. */
	eeePollingCounter++;

	if (eee_rtcCompare(&lastPollingTime, eee_rtcZero()) == 0) {
		eee_rtc(&lastPollingTime);
		eee_msToRTC(EEE_POLLING_HIST_START_AFTER_N_SEC * MSEC_PER_SEC,&interval);
	} else {
		eee_rtcDuration(&lastPollingTime, &duration);

		// Todo: case when we skip 1 sec

		if (eee_rtcCompare(&duration, &interval) > 0) {
			eeePollingCountersHist[eeePollingCounterIndex] = eeePollingCounter;
			eeePollingCounterIndex++;
			if (eeePollingCounterIndex >= EEE_POLLING_HIST)
				eeePollingCounterIndex = 0;

			if (eeePollingCounter < eeePollingCounterMin)
				eeePollingCounterMin = eeePollingCounter;
			if (eeePollingCounter > eeePollingCounterMax)
				eeePollingCounterMax = eeePollingCounter;
		} else {
			eeePollingCounterMin = eeePollingCounter;
			eeePollingCounterMax = eeePollingCounter;
			eeePollingCounter = 0;
		}
		eeePollingCounter = 0;
		eee_rtc(&lastPollingTime);
	}
#endif
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
int32
eee_start_polling_ui(int32 argc, uchar8 *argv[])
{
	if (argc != 2)
		return -1;

	if (!strcmp(argv[1], "on"))
		dont_poll = 0;
	else if (!strcmp(argv[1], "off"))
		dont_poll = 1;
	else
		return -1;

	return 0;
}

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_pollCfgSyntax(uchar8 *cmd_name)
{
	struct eee_poll_cb	*poll_cb;
	eee_poll_cfg_t		*pollcfg;
	uint32	first = 1;

	printf("usage: %10s show\n", cmd_name);
	printf("       %10s clear\n", cmd_name);
	printf("       %10s set <", cmd_name);
	for (poll_cb = &(eee.pfunc.high[0]); poll_cb->func; ++poll_cb) {
		pollcfg = poll_cb->poll_cfg_struct;
		while (pollcfg && pollcfg->pollCfgName) {
			if (!(strequ(pollcfg->pollCfgName, "view") ||
							 strequ(pollcfg->pollCfgName, "ssc") ||
							 strequ(pollcfg->pollCfgName, "clear"))) {
				printf("%s%s", !first ? "|" : "", pollcfg->pollCfgName);
				first = 0;
			}
			pollcfg++;
		}
	}
	printf("> <max_val>\n");
}

typedef void (*Func)();

void
eee_pollCfgView(uint32	just_hwm)
{
	struct eee_poll_cb	*poll_cb;
	eee_poll_cfg_t		*pollcfg;

	for (poll_cb = &(eee.pfunc.high[0]); poll_cb->func; ++poll_cb) {
		pollcfg = poll_cb->poll_cfg_struct;

		while (pollcfg && pollcfg->pollCfgName) {
			if (strequ(pollcfg->pollCfgName, "view")) {
				((Func)(pollcfg->func))(just_hwm);
				break;
			}
			pollcfg++;
		}
	}
}

void
eee_pollCfgSSC(uint32 just_hwm, eee_poll_cfg_rsp_t *rsp, uint32 max_rsp)
{
	struct eee_poll_cb	*poll_cb;
	eee_poll_cfg_t		*pollcfg;
	uint32 num_rsp = 0, num_added;

	for (poll_cb = &(eee.pfunc.high[0]); poll_cb->func; ++poll_cb) {
		pollcfg = poll_cb->poll_cfg_struct;
		while ((num_rsp < max_rsp-1) && (pollcfg) && (pollcfg->pollCfgName)) {
			if (strequ(pollcfg->pollCfgName, "ssc")) {
				((Func)(pollcfg->func))(just_hwm, rsp, max_rsp-num_rsp-1, &num_added);
				rsp += num_added;
				num_rsp += num_added;
				break;
			}
			pollcfg++;
		}
		rsp->cnt = POLLCFG_RSP_END;
	}
}

void
eee_pollCfgClear(void)
{
	struct eee_poll_cb	*poll_cb;
	eee_poll_cfg_t		*pollcfg;

	for (poll_cb = &(eee.pfunc.high[0]); poll_cb->func; ++poll_cb) {
		pollcfg = poll_cb->poll_cfg_struct;
		while (pollcfg && pollcfg->pollCfgName) {
			if (strequ(pollcfg->pollCfgName, "clear")) {
				((Func)(pollcfg->func))();
				break;
			}
			pollcfg++;
		}
	}
}

void
eee_pollCfgSet(uchar8 *poll_name, uint32 max_val, uchar8 *cmd_name)
{
	struct eee_poll_cb	*poll_cb;
	eee_poll_cfg_t		*pollcfg;

	for (poll_cb = &(eee.pfunc.high[0]); poll_cb->func; ++poll_cb) {
		pollcfg = poll_cb->poll_cfg_struct;
		while (pollcfg && pollcfg->pollCfgName) {
			if (strequ(pollcfg->pollCfgName, poll_name)) {
				((Func)(pollcfg->func))(max_val);
				return;
			}
			pollcfg++;
		}
	}
	eee_pollCfgSyntax(cmd_name);
}

#ifdef NFX_MOD_NFP
/*
Routine Description:
    This macro calculates the address of a CPU private variable in the shared
    address space and dereferences.  This allows it to be used on either
    the lvalue or rvalue of an expression.
Arguments:
    VARNAME     - The CPU private variable
    CPUID       - The index of the CPU in the FP kernel
Return Value:
    lvalue/rvalue of the per-cpu variable.
 */
#define REMOTE_CPU_PRIVATE(VARNAME, CPUID) ( * (volatile typeof(&VARNAME)) ( \
                (char *)(smp_ClientPrivatePhysSpace[CPUID]) + \
                ( (char *)(&VARNAME) - _private_start ) ) )

#define PRINTF (wsdebug == TRUE) && printf

#define WS_TO         16
#define WS_ROLLCALL   ((0x1 << nfx_num_cpus) - 1)

extern char _private_start[];

/*
Routine Description:
    Stop all cores except the current one.
Arguments:
    org_dont_poll - the original state of dont_poll variables for the cores.
    stopped - if the corresponding element in the array is 1, the core was stopped successfully
Return Value:
    NFX_OK
    NFX_ERR
 */
int
eee_freeze_all(boolean *org_dont_poll, boolean *stopped)
{
    uint32 rollcall, i;
    eeeGMT_t start_time, now_time;
    
    /*
     * Save off the value of dont_poll for each CPU and set each to 1
     */
    for (i = 0; i < nfx_num_cpus; ++i) {
        if (i != nfxMyCoreId) {
            REMOTE_CPU_PRIVATE(seen_dont_poll, i) = FALSE;
            org_dont_poll[i] = REMOTE_CPU_PRIVATE(dont_poll, i);
            REMOTE_CPU_PRIVATE(dont_poll, i) = 1;
        } else {
            org_dont_poll[i] = 0;
        }
    }

    /*
     * Wait for CPUs to have seen the change in their dont_poll
     */
    rollcall = (0x1 << nfxMyCoreId);
    eee_rtcGMT(&start_time);

    while (rollcall != WS_ROLLCALL) {
        int32 spinning_time;

        for (i = 0; i < nfx_num_cpus; ++i) {
            if (i != nfxMyCoreId) {
                if (REMOTE_CPU_PRIVATE(seen_dont_poll, i) == TRUE) {
                    rollcall |= (0x1 << i);
                }
            }
        }

        eee_rtcGMT(&now_time);
        spinning_time = now_time.sec - start_time.sec;
        if(spinning_time >= WS_TO) {
            break;
        }
        watchdog_kick();
    }

    for (i = 0; i < nfx_num_cpus; ++i) {
        stopped[i] = rollcall & (1 << nfxMyCoreId);
    }

    if (rollcall == WS_ROLLCALL) {
        return NFX_OK;
    } else {
        eee_thaw_all(org_dont_poll);
        return NFX_ERR;
    }
}

/*

Routine Description:
    Restart the stopped cores.
Arguments:
    org_dont_poll - the array of dont_poll values for the cores. If 0 restart the core.
Return Value:
    None.
 */
void
eee_thaw_all(boolean *org_dont_poll)
{
    int i;
    for (i = 0; i < nfx_num_cpus; ++i) {
        if (i != nfxMyCoreId) {
            REMOTE_CPU_PRIVATE(dont_poll, i) = org_dont_poll[i];
        }
    }
}
#else

/*

Routine Description:
    Stop all cores except the current one.
Arguments:
    org_dont_poll - the original state of dont_poll variables for the cores.
    stopped - if the corresponding element in the array is 1, the core
              was stopped successfully
Return Value:
    NFX_OK
    NFX_ERR
 */
int
eee_freeze_all(boolean *org_dont_poll unused__, boolean *stopped unused__)
{
    return 0;
}

/*
Routine Description:
    Restart the stopped cores.
Arguments:
    org_dont_poll - the array of dont_poll values for the cores. If 0 restart the core.
Return Value:
    None.
 */
void
eee_thaw_all(boolean *org_dont_poll)
{
}

#endif

#ifdef NFP_FP
uint64 *
dump_esms_copy_stacks(boolean wsdebug, int *stacks_dumpedp, char *errstrp, int errstrsize);

/*

Routine Description:
    This routine freezes the FP, copies the stacks of all threads and then
    unfreezes the FP.
	
	The FP is "frozen" by turning off the polling loop on other CPUs.
	This ensures that only the thread this routine is running on is
	running.  A separate routine dump_esms_copy_stacks() is invoked to
	copy off the stacks.  After this the other CPUs are allowed to start
	on their polling loops.
	This routine has to be called with the "pause_threads_spinlock" held

Arguments:
    wsdebug        - Flag to print debug messages. Passed in from the fscmd request
    stacks_dumpedp - Used to return the number of stacks copied into the returned buffer
    errstrp        - A char array used to return an error message from the writestacks sub-system
    errstrsize     - Size of the array pointed to by errstrp

Return Value:
    Returns the buffer returned by dump_esms_copy_stacks() containing the
    packed stacks
 */
uint64 *
dump_esms_freeze_copy_thaw(boolean wsdebug, int *stacks_dumpedp, char *errstrp, int errstrsize)
{
    uint64 *retval;
    boolean org_dont_poll[SMP_MAX_CPUS];
    boolean stopped[SMP_MAX_CPUS];
    int rc;

    rc = eee_freeze_all(org_dont_poll, stopped); 

    /*
     * If some CPUs have not responded after 16 seconds return the list of
     * them to the SSC
     */
    if (rc == NFX_OK) {
        /*
         * Proceed with the writestacks only if all CPUs have responded
         */
        retval = dump_esms_copy_stacks(wsdebug, stacks_dumpedp, errstrp,
                                       errstrsize);
    } else {
        int rem_size = 0;
        int i = 0;

        /*
         * @@@ snprintf() doesn't seem to return the number of characters
         * of the output string as per snprintf(1) man page.  Used strlen()
         * eventually.
         */
        snprintf(errstrp, errstrsize, "Scheduler on the following "
                 "CPU(s) did not respond after %d seconds:",
                 WS_TO);
        for (i = 0; i < nfx_num_cpus; ++i) {
            errstrp[errstrsize - 1] = '\0';
            rem_size = errstrsize - strlen(errstrp);
            if (rem_size <= 1) {
                break;
            }
            if (!stopped[i]) {
                snprintf(errstrp + (errstrsize - rem_size), rem_size, " FP%d", i);
            }
        }
        PRINTF("rem_size is finally %d\n", rem_size);
        retval = NULL;
    }

    eee_thaw_all(org_dont_poll);
 
    PRINTF("Returning from dump_esms_freeze_copy_thaw()\n");
    return retval;
}
#endif

void bmc12500Eth_stop_all(void);
void ispfc_stop_all(void);

/*
Routine Description:
    Disable the external links and stop the cores.
Arguments:
    None.
Return Value:
    None.
 */
void
eee_halt_prepare(void)
{
    boolean org_dont_poll[SMP_MAX_CPUS];
    boolean stopped[SMP_MAX_CPUS];
    int rc;

    rc = eee_freeze_all(org_dont_poll, stopped);
    if (rc != NFX_OK) {
        panic("%s: failed to stop the cores\n", __FUNCTION__);
    }

#if defined(NFP_TXRX)
    bmc12500Eth_stop_all();
#elif defined(NFP_FP) && defined(COUGAR)
    ispfc_stop_all();
#endif
}

/*
Routine Description:
    Enter the halt loop.
Arguments:
    None.
Return Value:
    None.
 */
void
eee_halt(void)
{
    wd_disable();

    for (;;) {
    }
}
