/*
 *-----------------------------------------------------------------
 * Name:    eee-test.c
 * Description:    Test routines for the EEE layer.
 *
 *         There are 2 test options:
 *         1) simple - send a single packet to a node.
 *            The simple test rcv function should receive
 *             the packet.  Counters will be updated.
 *
 *         2) complex - send packets from each CPU on a module
 *             to every other CPU on that module.  This
 *             will occur until something triggers a stop condition.
 *             The 'complex' test runs in 2 modes:
 *                 1) allocate/deallocate a packet each time it gets sent.
 *                 2) allocate packets at init time and have EEE layer just
 *                    spin them around.
 * Copyright (C) 2009, OnStor, Inc.
 *-----------------------------------------------------------------
 */
#include "nfx-incl.h"
#include "../sm-eee/eee-api.h"
#include "../sm-eee/eee-test.h"
#include "../sm-malloc-slab/malloc-api.h"

#include "../sm-chassis/cm-api.h"
#include "../sm-timer/timer-api.h"

/* #define EEE_TEST_SPIN        0 */
#define EEE_TEST_ACK_SIMPLE    1
#define EEE_TEST_NUM_PKT_SPEC_USER_DATA  8         
#define EEE_TEST_NUM_PKT_SPEC_MGMT     8    

eee_testPktSpec_t    eee_test_spec[3][EEE_TEST_NUM_PKT_SPEC] = {
    {   {2,        8192},            /* EEE_USER_DATA */
        {2,        8192},
        {2,        8192},
        {2,        8192},
        {2,        8192},
        {2,        8192},
        {2,        8192},
        {2,        8192},
    },
    {   {0,        1024},            /* EEE_CONTROL */
        {0,        1024},
        {0,        1024},
        {1,        1024},
        {1,        1024},
        {0,        1024},
        {1,        1024},
        {1,        1024},
    },
    {   {0,        1400},            /* EEE_MGMT */
        {0,        1400},
        {0,        1400},
        {1,        1400},
        {1,        1400},
        {0,        1400},
        {1,        1400},
        {1,        1400},
    }
};

#define EEE_TEST_PKT_SIZE_MAX    3200

uchar8    eee_mgmt_tst_pkt[EEE_TEST_PKT_SIZE_MAX];

uint32    eee_test_spec_index[3];

uint32    next_exp_seq_num[NFX_NUM_SLOTS_PER_CHASSIS][NFX_MAX_CPUS_PER_SLOT][3];

uint32    simple_exp_next_seq_num;

/* System DVT Variables  */

eee_dvt_test_stats    dvtstats;
eee_dvt_txrx_stats    dvtstats_txrx;
eee_test_hdr_t    * test_hdr;
/* [cpu][plane]  Now max cpu on NFP/NFC is 4 per card */
uint32  sentFromSSC[4][3];
uint32    eee_test_numPkts_index[3]={EEE_TEST_NUM_PKT_SPEC_USER_DATA, 0, EEE_TEST_NUM_PKT_SPEC_MGMT};
#if defined(FCNIM)
int32 ispfc_fcadmin(int32 argc, char **argv);
#endif

uint32  incoming_appid;

void eee_reinit_validity_table();
void dvtRemake_pkt(uint32 slot, uint32 cpu, uint32 seq, eee_test_hdr_t *pktHdr);
void eee_systemDVTinit();
void eee_systemDVTSend(uint32 slot, uint32 cpu, uchar8 * bfr, uint32 size);

/* Functions to handle TXRX pkts through ETH ports */
void dvt_front_loopback(eee_test_hdr_t    *test_hdr);
void eee_dvttestFrontEnd(uint32 plane);
void dvt_rcvd_txrx_front_pkt(eee_descPtr_t edesc);

/* global variables for sysdvt */
volatile uint32 dvtStop;
extern sysdvt_qlogic_t  *qdvt;
extern uint32  pktSize;
extern uint32 test_timer;
uint32 timerStarted = 0;
extern uint32    dont_poll;

/*-----------------------------------------------------------------
 * Name:    eee_testCreateSimplePacket()
 *
 * Description:    Create an EEE packet of the given size
 * Return Values:
 *  NFX_OK,
 *  EEE_ERR_NO_DESC,
 *  EEE_ERR_NO_BUF,
 *  EEE_ERR_BAD_PARAM,
 *  EEE_ERR_PACKET_TOO_LARGE
 *-----------------------------------------------------------------
 */
int32
eee_testCreateSimplePacket(uint32    dest_slot,            /* slot number */
    uint32    dest_cpu,  /* cpu id on that slot */
    uint32    size,      /* number of bytes in test message to send */
    uint32    bufsize,   /* EEE_BUF_SMALL, EEE_BUF_LARGE, EEE_BUF_ULTRA */
    uint32    plane,     /* 0 = data, 1 = mgmt, 2 = control */
    uint32    seq_num)
{
    uint32            mp;
    eee_desc_t        * edesc;
    eee_descBuf_t     * bd;
    uchar8            * buf[EEE_MAX_BUFS];
    uint32            bd_size[EEE_MAX_BUFS];
    uint32            num_bufs;
    uint32            i, j;
    int32             rc;
    uint32            buf_bytes;
    uint32            remain;
    uint32            dtype;
    uint32            *bufhdr;

    /*
     * Verify parameters
     */
    if (bufsize >= EEE_NUM_BUF_SIZES)
        return EEE_ERR_BAD_PARAM;

    if (size > ((eee.buf_size[bufsize]-eee.buf_offset) * EEE_MAX_BUFS))
        return EEE_ERR_BAD_PARAM;

    if (size < 64) {
        printf("nbytes must be >= 64\n");
        return EEE_ERR_BAD_PARAM;
    }

    /*
     * Allocate descriptor from appropriate memory pool
     */
    if (dest_slot == nfxMySlotId) {
        if (dest_cpu >= EEE_NUM_CPUS)
            return EEE_ERR_BAD_PARAM;
        mp = GET_MEMPOOL_INDEX_FROM_CPU_ID(dest_cpu);
    }
    else
        mp = eee.my_pool_id;

    edesc = eee_allocateDesc(mp, NFX_COS_HIGH);
    if (!edesc)
        return EEE_ERR_NO_DESC;

    /*
     * Allocate buffers from appropriate memory pool
     */
    num_bufs = 0;
    buf_bytes = eee.buf_size[bufsize] - eee.buf_offset;
    remain = size;
    do {
        buf[num_bufs] = eee_allocateBuffer(mp, bufsize, NFX_COS_HIGH);
        if (!buf[num_bufs]) {
            for (j=0; j<num_bufs; j++)
                eee_deallocateBuffer(buf[j]);
            eee_deallocateDesc(edesc);
            return EEE_ERR_NO_BUF;
        }
        if (remain > buf_bytes) {
            bd_size[num_bufs] = buf_bytes;
            remain -= buf_bytes;
        }
        else {
            bd_size[num_bufs] = remain;
            remain = 0;
        }
        num_bufs++;
    } while (remain);

    /*
     * Fill in buffers with test data
     */
    for (i=0; i<num_bufs; i++) {
        for (j=0; j<bd_size[i]; j++)
            buf[i][j] = j & 0xff;
    }
    bufhdr = (uint32 *)&(buf[0][0]);
    *bufhdr++ = nfxMySlotId;
    *bufhdr++ = nfxMyCPUId;
    *bufhdr++ = size;

    /*
     * Create packet
      */
    bd = &(edesc->bd[0]);
    edesc->hdr.control = (num_bufs << EDESC_BUFCNT_SHIFT) | size;
    if (num_bufs == EEE_MAX_BUFS) {
        edesc->hdr.control |= EDESC_COMMAND_USE_FIRST_BUFFER;
    }
    else
        bd++;

    if (plane == 0)
        dtype = EEE_USER_DATA;
    else if (plane == 1)
        dtype = EEE_MGMT;
    else if (plane == 2)
        dtype = EEE_CONTROL;
    else if (plane == 3)
        dtype = EEE_SAC;
    else {
        dtype = 0;
        panic("invalid plane, %u\n", plane);
    }

    *bufhdr++ = dtype;
    *bufhdr++ = seq_num;

    edesc->hdr.dest_port = PACK_UNICAST_PORT_ADDRESS(dest_slot, dest_cpu,
                           eee.simple_test_app_id,
                           dtype, NFX_COS_HIGH, 0);
    edesc->hdr.src_port = PACK_UNICAST_PORT_ADDRESS(nfxMySlotId, nfxMyCPUId,
                          eee.simple_test_app_id,
                          dtype, NFX_COS_HIGH, 0);
    edesc->hdr.attr = 0;

    for (i=0; i<num_bufs; i++) {
        bd->control = EDESC_BD_CMD_IN_USE | bufsize;
        bd->len = bd_size[i];
        bd->buf = buf[i];
        bd++;
    }

    /*
     * Send packet
     */
    if ((rc = eee_forwardPacket(edesc)) != EEE_FORWARDED_PACKET)
        return rc;

    return NFX_OK;

}    /* eee_testCreateSimplePacket() */

/*-----------------------------------------------------------------
 * Name:    eee_testCreateDVTTestPkt()
 * Description:    Create a packet from a known sequence
 * Return Values:
 *  returns edesc addr or NULL
 *-----------------------------------------------------------------
 */
eee_descPtr_t
eee_testCreateDVTTestPkt(uint32 slot, uint32 cpu, eee_test_hdr_t *pktSpec, uint32 seq)
{
    eee_descPtr_t    edesc;
    uint32           size = 0, bs = 0;
    uint32           remain;
    eee_descBuf_t    * bd;
    uchar8           * buf[EEE_MAX_BUFS];
    uint32           bd_size[EEE_MAX_BUFS];
    uint32           num_bufs;
    uint32           buf_bytes;
    uint32           i;
    eee_test_hdr_t   * test_hdr;
    uint32           plane = pktSpec->plane;
    uint32           spin = pktSpec->spin;
// Debug -------------
     if(!((plane >= 0) && (plane < 3))) {
         printf("In eee_testCreateDVTTestPkt() plane = %0x not between 0-2\n",
                 plane);
     }
// Debug -------------

    /*
     * Get info from test_pkt_spec structure about packet to be created
     */
    if(plane == EEE_MGMT) {
        bs = 1;
        size = 1400;
    }
    if(plane == EEE_USER_DATA) {
        bs = 2;
        size = 8192;
    }

    /* If pktSize is passed from SSC */
    if(pktSpec->size) {
        size = pktSpec->size;    
        if(plane == EEE_MGMT)
            bs = 1;
        else if(plane == EEE_USER_DATA)
            bs = 2;    
    }
    /*
     * Allocate descriptor from my pool
     */
    edesc = eee_allocateDesc(eee.my_pool_id, NFX_COS_HIGH);
    if (!edesc) {
        eee.test_dvt_send_err[plane]++;
        return NULL;
    }

    /*
     * Allocate buffers from my pool
     */
    num_bufs = 0;
    buf_bytes = eee.buf_size[bs] - eee.buf_offset;
    remain = size;
    do {
        buf[num_bufs] = eee_allocateBuffer(eee.my_pool_id, bs, NFX_COS_HIGH);
        if (!buf[num_bufs]) {
            for (i=0; i<num_bufs; i++)
                eee_deallocateBuffer(buf[i]);
            eee_deallocateDesc(edesc);
            return NULL;
        }
        if (remain > buf_bytes) {
            bd_size[num_bufs] = buf_bytes;
            remain -= buf_bytes;
        }
        else {
            bd_size[num_bufs] = remain;
            remain = 0;
        }
        memset(buf[num_bufs], 0xab, buf_bytes);
        num_bufs++;
    } while (remain);

    /*
     * Create packet
     */
    bd = &(edesc->bd[0]);
    edesc->hdr.control = (num_bufs << EDESC_BUFCNT_SHIFT) | size;
    if (num_bufs == EEE_MAX_BUFS) {
        edesc->hdr.control |= EDESC_COMMAND_USE_FIRST_BUFFER;
    }
    else
        bd++;

    for (i=0; i<num_bufs; i++) {
        bd->control = EDESC_BD_CMD_IN_USE | bs;
        bd->len = bd_size[i];
        bd->buf = buf[i];
        bd++;
    }

    edesc->hdr.dest_port = PACK_UNICAST_PORT_ADDRESS(slot,
                                                    cpu,
                                                    eee.dvt_test_app_id,
                                                    plane,
                                                    NFX_COS_HIGH,
                                                    PORT_BEST_FIT);


    edesc->hdr.src_port = PACK_UNICAST_PORT_ADDRESS(nfxMySlotId,
                                                    nfxMyCPUId,
                                                    eee.dvt_test_app_id,
                                                    plane,
                                                    NFX_COS_HIGH,
                                                    PORT_BEST_FIT);
    edesc->hdr.attr = 0;
    test_hdr = (eee_test_hdr_t *)(buf[0]);
    test_hdr->cmd = EEE_MGMT_TEST_CMD_DATA;
    /* For missing pkts we need to create pkt with only the missed seq no.
     * So, when seq is not given(0xFFFF), base seq no from eee_test_spec_index
     * otherwise, make pkt with provided seq no.
     */ 
    if(seq == 0xFFFF)
        test_hdr->seq = eee_test_spec_index[plane];
    else
        test_hdr->seq = seq;

    test_hdr->size = size;
    test_hdr->orig_slot = nfxMySlotId;
    test_hdr->orig_cpu = nfxMyCPUId;
    test_hdr->plane = plane;
    test_hdr->spin = spin;
    if(seq == 0xFFFF)
        eee_test_spec_index[plane] = 
            (eee_test_spec_index[plane] + 1) % eee_test_numPkts_index[plane];
    
    return (edesc);

}    /* eee_testCreateDVTTestPkt() */

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
int32
eee_testSendDVTStartPkt(
    uint32    slot,
    uint32    cpu,
    uint32    plane,
    uint32    num_pkts,
    uint32    spin
)
{
    eee_test_hdr_t test_msg;
    uint32 dp, sp;

// Debug -------------
    if(!((plane >= 0) && (plane < 3))) {
        printf("In eee_testSendDVTStartPkt() plane = %0x not between 0-2\n",
                plane);
    }
// Debug -------------

    test_msg.cmd = EEE_MGMT_TEST_CMD_START;
    test_msg.size = num_pkts;
    test_msg.plane = plane;
    test_msg.orig_slot = nfxMySlotId;
    test_msg.orig_cpu = nfxMyCPUId;
    test_msg.spin = spin;

    dp = PACK_UNICAST_PORT_ADDRESS(slot,
                                    cpu,
                                    eee.dvt_test_app_id,
                                    plane,
                                    NFX_COS_HIGH,
                                    PORT_BEST_FIT);

    sp = PACK_UNICAST_PORT_ADDRESS(nfxMySlotId,
                                    nfxMyCPUId,
                                    eee.dvt_test_app_id,
                                    plane,
                                    NFX_COS_HIGH,
                                    PORT_BEST_FIT);

    return (eee_sendMessage(&test_msg, sizeof(eee_test_hdr_t), dp, sp));

}    /* eee_testSendDVTStartPkt() */

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
int32
eee_testSendDVTStopPkt(uint32 slot, uint32 cpu, uint32 plane)
{
    eee_test_hdr_t    test_msg;
    uint32            dp;
    uint32            sp;

// Debug -------------
    if(!((plane >= 0) && (plane < 3))) {
        printf("In eee_testSendDVTStopPkt() plane = %0x not between 0-2\n",
                plane);
    }
// Debug -------------

    test_msg.cmd = EEE_MGMT_TEST_CMD_STOP;
    test_msg.orig_slot = nfxMySlotId;
    test_msg.orig_cpu = nfxMyCPUId;
    test_msg.plane = plane;

    dp = PACK_UNICAST_PORT_ADDRESS(slot,
                                    cpu,
                                    eee.dvt_test_app_id,
                                    plane,
                                    NFX_COS_HIGH,
                                    PORT_BEST_FIT);

    sp = PACK_UNICAST_PORT_ADDRESS(nfxMySlotId,
                                    nfxMyCPUId,
                                    eee.dvt_test_app_id,
                                    plane,
                                    NFX_COS_HIGH,
                                    PORT_BEST_FIT);

    return (eee_sendMessage(&test_msg, sizeof(eee_test_hdr_t), dp, sp));

}    /* eee_testSendDVTStopPkt() */

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
int32
eee_testSendDVTClearPkt(uint32 slot, uint32 cpu, uint32 plane)
{
    eee_test_hdr_t    test_msg;
    uint32            dp;
    uint32            sp;

// Debug -------------
    if(!((plane >= 0) && (plane < 3))) {
        printf("In eee_testSendDVTClearPkt() plane = %0x not between 0-2\n",
                plane);
    }
// Debug -------------

    test_msg.cmd = EEE_MGMT_TEST_CMD_CLEAR;
    test_msg.orig_slot = nfxMySlotId;
    test_msg.orig_cpu = nfxMyCPUId;
    test_msg.plane = plane;

    dp = PACK_UNICAST_PORT_ADDRESS(slot,
                                    cpu,
                                    eee.dvt_test_app_id,
                                    plane,
                                    NFX_COS_HIGH,
                                    PORT_BEST_FIT);

    sp = PACK_UNICAST_PORT_ADDRESS(nfxMySlotId,
                                    nfxMyCPUId,
                                    eee.dvt_test_app_id,
                                    plane,
                                    NFX_COS_HIGH,
                                    PORT_BEST_FIT);

    return (eee_sendMessage(&test_msg, sizeof(eee_test_hdr_t), dp, sp));

}    /* eee_testSendDVTStopPkt() */

/*-----------------------------------------------------------------
 * Name:    eee_testDVTStart(mode)
 * Description:    Start EEE layer test
 *-----------------------------------------------------------------
 */
int32
eee_testDVTStart(uint32    num_pkts, uint32    plane, uint32    spin)
{
    uint32    slot, cpu, i;
    int32     rc;
    eee_descPtr_t    edesc;
    eee_test_hdr_t  pktHdr;

// Debug -------------
    if(!((plane >= 0) && (plane < 3))) {
        printf("In testDVTStart plane = %0x not between 0-2\n",plane); 
        printf(" num_pkts = %0x spin = %0x\n",num_pkts,spin); 
    }
// Debug -------------

    if (!eee.test_dvt_valid[nfxMySlotId][nfxMyCPUId][plane]) {
        printf("%s DVT test not valid on this CPU\n",
                        (plane == EEE_MGMT) ? "MGMT" :
                        ((plane == EEE_CONTROL) ? "CONTROL" : "DATA"));
        return NFX_ERR;
    }

    if (eee.test_dvt_running[plane]) {
        printf("EEE test already running eee_testDVTStart()"); 
        printf("  eee.test_dvt_running[]  plane=%d \n",plane);
        return NFX_OK;
    }

    /*
     * Broadcast start packet
     */
    for (slot=0; slot<NFX_NUM_SLOTS_PER_CHASSIS; slot++) {
        for (cpu=0; cpu<NFX_MAX_CPUS_PER_SLOT; cpu++) {
            if (eee.test_start_valid[slot][cpu][plane]) {
                if ((slot == nfxMySlotId) && (cpu == nfxMyCPUId))
                    continue;
                if (eee_testSendDVTStartPkt(slot, cpu, plane, 
                                            num_pkts, spin) != NFX_OK)
                    eee.test_dvt_send_err[plane]++;
            }
        }
    }

    /*
     * Send num_pkts to each slot/cpu
     */
    //NFX_NUM_SLOTS_PER_CHASSIS = 12

    for (slot=0; slot<NFX_NUM_SLOTS_PER_CHASSIS; slot++) {

      // NFX_MAX_CPUS_PER_SLOT = 16

        for (cpu=0; cpu<NFX_MAX_CPUS_PER_SLOT; cpu++) {
            if (eee.test_dvt_valid[slot][cpu][plane]) {
                if ((slot == nfxMySlotId) && (cpu == nfxMyCPUId))
                    continue;
                next_exp_seq_num[slot][cpu][plane] = 0;

                for (i=0; i<num_pkts; i++) {
                    pktHdr.plane = plane;    
                    pktHdr.spin = spin;    
                    edesc = eee_testCreateDVTTestPkt(slot, cpu, 
                                                     (eee_test_hdr_t *)&pktHdr,
                                                     0xFFFF);
                    if (edesc) {
                        if ((rc = eee_forwardPacket(edesc)) !=
                                                     EEE_FORWARDED_PACKET)
                            eee.test_dvt_send_err[plane]++;
                        else {
                            printf("slot %d cpu %d plane %d test pkt %d\n",
                                    slot,cpu, test_hdr->plane,
                                    eee_test_spec_index[test_hdr->plane]);
#ifdef USE_SMP                        
                            ATOMIC_INC_INT(&dvtstats.packets_sent[plane], 1);
                            ATOMIC_INC_INT(&dvtstats.bytes_sent[plane],
                                       (edesc->hdr.control & EDESC_TOTAL_LEN));
#else                        
                            dvtstats.packets_sent[plane]++;
                            dvtstats.bytes_sent[plane] += 
                                       (edesc->hdr.control & EDESC_TOTAL_LEN);
#endif                            
                        }    
                    }
                    else
                        eee.test_dvt_send_err[plane]++;
                }
            }
        }
    }

// --debug
    // eee.test_dvt_running[plane]  could be writting out of bounds
    VERIFY((plane >= 0) && (plane < 3));
// --debug
#ifdef USE_SMP                        
    ATOMIC_INC_INT(&eee.test_dvt_running[plane], 1);
#else                        
    eee.test_dvt_running[plane] = 1;
#endif
    return NFX_OK;

}    /* eee_testDVTStart() */


/*-----------------------------------------------------------------
 * Name:    eee_testDVTEnd()
 * Description:    Stop EEE layer test
 *-----------------------------------------------------------------
 */
void
eee_testDVTEnd(uint32    plane)
{
    uint32    slot, cpu;

// Debug -------------
    if(!((plane >= 0) && (plane < 3))) {
        printf("In testDVTEnd plane = %0x not between 0-2\n",plane); 
    }
// Debug -------------


    if (!eee.test_dvt_running[plane]) {
        printf("EEE test not running  eee_testDVTEnd()plane=%d \n",plane);
        return;
    }

    dvtStop = 1;
    /*
     * Broadcast start packet
     */
    // Debug FP and TXRX crash in sysDvt
    VERIFY((plane >= 0) && (plane < 3));
    VERIFY((nfxMyCPUId >= 0) && (nfxMyCPUId < 4));

    if(!sentFromSSC[nfxMyCPUId][plane]) {
        for (slot=0; slot<NFX_NUM_SLOTS_PER_CHASSIS; slot++) {
            for (cpu=0; cpu<NFX_MAX_CPUS_PER_SLOT; cpu++) {
                if (eee.test_start_valid[slot][cpu][plane]) {
                    if ((slot == nfxMySlotId) && (cpu == nfxMyCPUId))
                        continue;
#ifdef NFP_FP
                    /* To prevent a recursive loop */
                    if((slot == NFX_SLOTTYPE_NFP) && ((cpu == NFX_NFP_FP1) 
                                || (cpu == NFX_NFP_FP2)))
                        continue;
#endif                
                    if (eee_testSendDVTStopPkt(slot, cpu, plane) != NFX_OK)
                        eee.test_dvt_send_err[plane]++;
                }
            }
        }
    }
// --debug
    // eee.test_dvt_running[plane]  could be writting out of bounds
    VERIFY((plane >= 0) && (plane < 3));
// --debug
#ifdef USE_SMP                        
    ATOMIC_AND_INT(&eee.test_dvt_running[plane], 0);
#else                        
    eee.test_dvt_running[plane] = 0;
#endif    
    /* If test was initiated from the SSC, set the correct Init table */
    if(sentFromSSC[nfxMyCPUId][plane])
        eee_testInit();

    // Debug FP and TXRX crash in sysDvt
    VERIFY((plane >= 0) && (plane < 3));
    VERIFY((nfxMyCPUId >= 0) && (nfxMyCPUId < 4));

    sentFromSSC[nfxMyCPUId][plane] = 0;

}    /* eee_testDVTEnd() */

/*-----------------------------------------------------------------
 * Name:    eee_testDVTClear()
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_testDVTClear(uint32    plane)
{
    uint32    slot, cpu;

// Debug -------------
    if(!((plane >= 0) && (plane < 3))) {
        printf("In testDVTClear plane = %0x not between 0-2\n",plane); 
    }
// Debug -------------

    /*
     * Broadcast start packet
     */
    for (slot=0; slot<NFX_NUM_SLOTS_PER_CHASSIS; slot++) {
        for (cpu=0; cpu<NFX_MAX_CPUS_PER_SLOT; cpu++) {
            if (eee.test_start_valid[slot][cpu][plane]) {
 // wtg debug
                printf(" - Array test_start_valid "); 
                printf("slot=[%d] cpu=[%d] plane=[%d] \n",slot,cpu,plane);
 // -- debug 
                if ((slot == nfxMySlotId) && (cpu == nfxMyCPUId))
                    continue;
                if (eee_testSendDVTClearPkt(slot, cpu, plane) != NFX_OK)
                    eee.test_dvt_send_err[plane]++;
            }
        }
    }

// --debug
    // eee.test_dvt_running[plane]  could be writting out of bounds
    VERIFY((plane >= 0) && (plane < 3));
// --debug
#ifdef USE_SMP                        
    ATOMIC_AND_INT(&eee.test_dvt_running[plane], 0);
#else                        
    eee.test_dvt_running[plane] = 0;
#endif

}    /* eee_testDVTEnd() */

/*-----------------------------------------------------------------
 * Name:    eee_testDVTPrintStats()
 * Description:    Show send/receive stats for EEE test
 *-----------------------------------------------------------------
 */
void
eee_testDVTPrintStats(uint32    plane)
{
    uint32    i, j;
    uint32    tot_sent=0, tot_rcvd=0;

    printf("\nDVT BP %s Test : %s\n", (plane == EEE_MGMT) ? "MGMT" :
                        ((plane == EEE_CONTROL) ? "CONTROL" : "DATA"),
                        eee.test_dvt_running[plane] ? "RUNNING" : "STOPPED");
    printf("     Local Pkts:\n");
    printf("        slot  cpu       sent       rcv\n");
    printf("        ----  ---   --------  --------\n");
    for (i=0; i<NFX_NUM_SLOTS_PER_CHASSIS; i++)
        for (j=0; j<NFX_MAX_CPUS_PER_SLOT; j++)
            if (eee.test_dvt_valid[i][j][plane]) {
                printf("          %2d   %2d   %8d  %8d\n",
                                i, j,
                                eee.test_dvt_my_sent[i][j][plane],
                                eee.test_dvt_my_rcvd[i][j][plane]);
                tot_sent += eee.test_dvt_my_sent[i][j][plane];
                tot_rcvd += eee.test_dvt_my_rcvd[i][j][plane];
            }
    printf("\n");
    printf("     Remote Pkts:\n");
    printf("        slot  cpu        rcv      sent\n");
    printf("        ----  ---   --------  --------\n");
    for (i=0; i<NFX_NUM_SLOTS_PER_CHASSIS; i++)
        for (j=0; j<NFX_MAX_CPUS_PER_SLOT; j++)
            if (eee.test_dvt_valid[i][j][plane]) {
                printf("          %2d   %2d   %8d  %8d\n",
                                i, j,
                                eee.test_dvt_other_rcvd[i][j][plane],
                                eee.test_dvt_other_sent[i][j][plane]);
                tot_sent += eee.test_dvt_other_sent[i][j][plane];
                tot_rcvd += eee.test_dvt_other_rcvd[i][j][plane];
            }
    printf("\n");
    printf("     Send Errors: %d\n", eee.test_dvt_send_err[plane]);
    printf("     Rcv Errors : %d\n", eee.test_dvt_rcv_err[plane]);
    printf("\n");


    printf("Totals: sent:  %d\n", tot_sent);
    printf("        rcvd:  %d\n", tot_rcvd);

}    /* eee_testDVTPrintStats() */

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_testDVTClearStats(uint32    plane)
{
    uint32    i, j;
// Debug -------------
    if(!((plane >= 0) && (plane < 3))) {
        printf("In eee_testDVTClearStats() plane = %0x not between 0-2\n",
               plane);
    }
// Debug -------------

    for (i=0; i<NFX_NUM_SLOTS_PER_CHASSIS; i++) {
        for (j=0; j<NFX_MAX_CPUS_PER_SLOT; j++) {
            eee.test_dvt_my_sent[i][j][plane] = 0;
            eee.test_dvt_my_rcvd[i][j][plane] = 0;
            eee.test_dvt_other_sent[i][j][plane] = 0;
            eee.test_dvt_other_rcvd[i][j][plane] = 0;
        }
    }

    eee.test_dvt_send_err[plane] = 0;
    eee.test_dvt_rcv_err[plane] = 0;

}    /* eee_testDVTClearStats() */


/*-----------------------------------------------------------------
 * Name:    eee_testSimpleRcv()
 * Description:    Receive function for EEE test packets
 *-----------------------------------------------------------------
 */
void
eee_testSimpleRcv(eee_descPtr_t    edesc)
{
    uint32            pkt_len, bufcnt;
    uint32            i;
    eee_descBuf_t     * bd;
    uint32            * buf;
    uint32            src_slot, src_cpu, src_len, src_plane, src_seq_num;
    uchar8            * lbuf;
    int               error;
    int               ncopied;

    bufcnt = EEE_GET_NUM_BUFS(edesc);
    pkt_len = EEE_GET_PKT_LEN(edesc);

#if EEE_TEST_ACK_SIMPLE

//    printf("eee_testSimpleRcv, %p\n", edesc);
//    printf("pkt_len = %d bytes\n", pkt_len);
//    printf("num_bufs = %d\n", bufcnt);

    if (edesc->hdr.control & EDESC_COMMAND_USE_FIRST_BUFFER)
        i = 0;
    else
        i = 1;
    bd = &(edesc->bd[i]);

    buf = (uint32 *)bd->buf;

    src_slot = *buf++;
    src_cpu = *buf++;
    src_len = *buf++;
    src_plane = *buf++;
    src_seq_num = *buf++;

// Debug -------------
    if(!((src_plane >= 0) && (src_plane < 3))) {
        printf("In eee_testSimpleRcv() src_plane = %0x not between 0-2\n",
                src_plane);
    }
// Debug -------------

    if (src_len != pkt_len) {
        printf("Error in simple test pkt_len, src_len = %d, rcv_len = %d\n",
                                                            src_len, pkt_len);
        eee_freePacket(edesc);
        return;
    }
    if (src_plane != EEE_GET_DTYPE(edesc->hdr.dest_port)) {
        printf("Error in simple test dtype = %d, src_plane = %d\n",
                                EEE_GET_DTYPE(edesc->hdr.dest_port), src_plane);
        eee_freePacket(edesc);
        return;
    }

    lbuf = eee_ramAlloc(pkt_len);
    if (lbuf == NULL) {
        printf("eee_testSimpleRcv: cannot allocate local buffer\n");
        eee_freePacket(edesc);
        return;
    }
    if ((ncopied = eee_copyPacketToLocalBuffer(lbuf, edesc, pkt_len, 0)) != 
                                                                 pkt_len) {
        printf("eee_testSimpleRcv: copy failed ncopied=%d pktlen=%d\n",
               ncopied, pkt_len);
    }
    if (!eee_is_local_slot_cpu(src_slot, src_cpu)) {
        if ((error = eee_sendMessage(lbuf, pkt_len, edesc->hdr.src_port,
                                     edesc->hdr.dest_port)) != NFX_OK) {
            printf("eee_testSimpleRcv: send message failed err=%d\n", error);
        }
    }
    eee_ramDealloc(lbuf);
    eee_freePacket(edesc);
#else
    eee_freePacket(edesc);
#endif

}    /* eee_testSimpleRcv() */

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_testDVTRcv(eee_descPtr_t    edesc)
{
    /*
     * if (packet == START) {
     *        if (!test_running)
     *            send START pkt to all slots/cpus
     *            send N pkts to all slots/cpus
     *            test_running = 1;
     *        drop packet
     * }
     * if (packet == STOP) {
     *        if (test_running) {
     *            test_running = 0;
     *            send STOP pkt to all slots/cpus
     *        }
     *        drop packet
     * }
     * if (packet == DATA) {
     *        if (test_running) {
     *            spin packet around to source
     *        }
     *        else {
     *            if (orig_slot == myslot) && (orig_cpu == mycpu)
     *                drop packet
     *            else
     *                spin packet around to source
     *        }
     * }
     */
    eee_test_hdr_t    * test_hdr;
    eee_descBuf_t     * bd;
    uint32            cmd;
    int32             rc;
    uint32            src_slot, src_cpu, src_plane;
    uint32            i;
    // could be used uninit?? added (=0)compiler error when just defined
    uint32            plane=0;
#ifdef NFP_FP
    static uint32     firstAfterBootup=0;
#endif    
#ifdef SSC_MGMT
    uint32            sval=0;
#endif              
    bd = &(edesc->bd[0]);
    if (!(edesc->hdr.control & EDESC_COMMAND_USE_FIRST_BUFFER))
        bd++;

    src_slot = EEE_GET_SLOT_NUM(edesc->hdr.src_port);
    src_cpu = EEE_GET_DEST_CPU(edesc->hdr.src_port);
    src_plane = EEE_GET_DTYPE(edesc->hdr.src_port);
// Debug -------------
    if(!((src_plane >= 0) && (src_plane < 3))) {
        printf("In eee_testDVTRcv() src_plane = %0x not between 0-2\n",
                src_plane);
    }
// Debug -------------

    test_hdr = (eee_test_hdr_t *)(bd->buf);

    cmd = test_hdr->cmd;

// Debug..
    if(  (!((src_plane >= 0) && (src_plane < 3))) || 
       (src_plane != test_hdr->plane)  ) {
           printf(" DVTRcv  src_slot %x src_cpu %x src_plane %x \n",
                     src_slot,src_cpu,src_plane);
           printf("  test_hdr->cmd %x test_hdr->port %x test_hdr->plane %x\n",
                    test_hdr->cmd,test_hdr->port,test_hdr->plane);
    }
    switch(cmd) {
        case EEE_MGMT_TEST_CMD_START : {
            if((src_slot == 0) && (src_cpu == 0)) {
                
#ifdef SSC_MGMT
                /* Start the Fan test */
                /* Bring Fans Up */
                sval = *(volatile unsigned int *)(HW_BMFPGA_BASE + 0x10);
                sval &= ~0x44;
                *(volatile unsigned int *)(HW_BMFPGA_BASE + 0x10) = sval;
                return;
#endif

#ifdef NFP_FP
                /* Do not allow packets to be rcvd on second core of FP */
                if(nfxMyCPUId == NFX_NFP_FP2)
                    dont_poll = 1;    

                 /* if running test for the very first time after reboot,
                  * allow second core of FP to stop its polling loop.
                  */      
                if(!firstAfterBootup) {
                    firstAfterBootup ++;
                    eee_freePacket(edesc);
                    return;
                }    
#endif                
                // Debug FP and TXRX crash in sysDvt
                // Fails here ... VERIFY((test_hdr->plane >= 0) && 
                //                        (test_hdr->plane < 3));
                if(!((test_hdr->plane >= 0) && (test_hdr->plane < 3))) {
                    printf("In CMD_START  plane = %0x not between 0-2\n",
                            test_hdr->plane); 
                    printf(" nfxMyCPUId = %0x \n",nfxMyCPUId); 
                }
                VERIFY((nfxMyCPUId >= 0) && (nfxMyCPUId < 4));
                // Since test_hdr->plane comes in as a 4 (SYS_DVT_FRONT) 
                //  don't overwrite bounds
                // of the array  sentFromSSC[4][3] ...
                if(test_hdr->plane == SYS_DVT_FRONT) {
                    sentFromSSC[nfxMyCPUId][EEE_USER_DATA] = 1;
                    printf(" - sentFromSSC[%d][%d]\n",
                            nfxMyCPUId,EEE_USER_DATA);
                }
                else {
                    sentFromSSC[nfxMyCPUId][test_hdr->plane] = 1;
                    printf(" - sentFromSSC[%d][%d] else \n",
                             nfxMyCPUId,test_hdr->plane);
                }
                eee_reinit_validity_table();

                if(test_hdr->plane == EEE_CONTROL)
                        return;

                if(test_hdr->plane == SYS_DVT_FRONT)
                {
#ifdef NFP_TXRX                    
                    if (eee.test_dvt_running[EEE_USER_DATA]) {
                        printf("EEE test already running ");
                        printf("isCheetah[In CMD_START] ");
                        printf("eee.test_dvt_running[EEE_USER_DATA] ");
                        printf(" plane = %x\n",test_hdr->plane);
                        eee_freePacket(edesc);
                        return;
                    }    
                        dvt_front_loopback(test_hdr);    

#if defined (NFP_TXRX) || defined (NFP_FP) || defined(FCNIM)                    
                eee_testSysDVTStart(test_hdr);
#endif                
            }    
            else  /* Not initiated from SSC */
                eee_testDVTStart(test_hdr->size, test_hdr->plane, 
                                 test_hdr->spin);

            printf(" [in CMD START] src_slot = %x src_cpu = %x plane = %x \n",
                    src_slot,src_cpu,test_hdr->plane);

            eee_freePacket(edesc);
            break;
        }
        case EEE_MGMT_TEST_CMD_STOP : {
            if((src_slot == 0) && (src_cpu == 0)) {

#ifdef SSC_MGMT
              /* Bring Fans Down */
              sval = *(volatile unsigned int *)(HW_BMFPGA_BASE + 0x10);
              sval |= 0x24;
              *(volatile unsigned int *)(HW_BMFPGA_BASE + 0x10) = sval;
#endif
            /* if sent from ssc, stop all traffic : 
             * need to modify later to support only 1 plane
             */
                for(plane=0;plane<3;plane++) {    
#ifdef NFP_TXRX                    
                    if((isTXRX()) & (plane == EEE_USER_DATA))    
                        eee_dvttestFrontEnd(plane);
                    else
#endif                        
                        eee_testDVTEnd(plane);
                }    
            }    
            else    
                eee_testDVTEnd(test_hdr->plane);

            eee_freePacket(edesc);
            break;
        }
        case EEE_MGMT_TEST_CMD_CLEAR : {
            eee_testDVTClearStats(test_hdr->plane);
            eee_freePacket(edesc);
            break;
        }
        case EEE_MGMT_TEST_CMD_DATA : {

// Debug -------------
            if(!((test_hdr->plane >= 0) && (test_hdr->plane < 3))) {
                printf("In CMD_DATA  plane = %0x not between 0-2\n",
                        test_hdr->plane); 
                printf(" nfxMyCPUId = %0x \n",nfxMyCPUId); 
            }
            if(!((src_plane >= 0) && (src_plane < 3))) {
                printf("In CMD_DATA src_plane = %0x not between 0-2\n",
                        src_plane); 
                printf(" nfxMyCPUId = %0x \n",nfxMyCPUId); 
            }
// Debug -------------
            if (test_hdr->orig_slot == nfxMySlotId) {
#ifdef USE_SMP                        
                ATOMIC_INC_INT(&dvtstats.packets_received[src_plane], 1);
                ATOMIC_INC_INT(
                        &dvtstats.bytes_received[src_plane], test_hdr->size);
#else                        
                dvtstats.packets_received[src_plane]++;
                dvtstats.bytes_received[src_plane] += test_hdr->size;
#endif                
                if (!eee.test_dvt_running[test_hdr->plane])
                        eee_freePacket(edesc);
                else {
                    uint32    t;
                    uint32    seq;
                    seq = test_hdr->seq;

#ifdef NFP_TXRX                            
                    if((isTXRX()) && (test_hdr->plane == EEE_USER_DATA))
                        dvt_rcvd_txrx_front_pkt(edesc);
                    else 
#endif                        
                    {
                        /* If rcvd seq no > expected seq no, remake missing pkt,
                         * spin the rcvd pkt around.
                         */         
                        if (seq >
                             next_exp_seq_num[src_slot][src_cpu][src_plane]) {
                            printf("%s test error, expected seq %d, got %d, ",
                              (src_plane == EEE_MGMT) ? "MGMT" :
                              ((src_plane == EEE_CONTROL) ? "CONTROL" : "DATA"),
                              next_exp_seq_num[src_slot][src_cpu][src_plane],
                              seq );
                              printf(" slot %d, cpu %d Replacing pkt!\n",
                              src_slot, src_cpu);
#ifdef USE_SMP                        
                            ATOMIC_INC_INT(&eee.test_dvt_rcv_err[src_plane], 1);
#else                        
                            eee.test_dvt_rcv_err[src_plane]++;
#endif                        

                            /* DO NOT halt test if started from SSC */
                            // Debug FP and TXRX crash in sysDvt
                            VERIFY((plane >= 0) && (plane < 3));
                            VERIFY((nfxMyCPUId >= 0) && (nfxMyCPUId < 4));

                            VERIFY((src_plane >= 0) && (src_plane < 3));
                            VERIFY((test_hdr->plane >= 0) && (test_hdr->plane < 3));

                            VERIFY((src_slot >= 0) && (src_slot < 3));
                            VERIFY((src_cpu >= 0) && (src_cpu < 3));

                            if(!sentFromSSC[nfxMyCPUId][src_plane])
                                eee_testDVTEnd(src_plane);
                            else {
                                eee_test_hdr_t  pktHdr;
                                /*Since pkt is missing we need to create another one
                                 * to fill missing gap and proceed with test
                                 */     
                                pktHdr.plane = test_hdr->plane;    
                                pktHdr.spin =  test_hdr->spin;    
                                pktHdr.size =  test_hdr->size;    
                                dvtRemake_pkt(src_slot, src_cpu, 
                                     next_exp_seq_num[src_slot][src_cpu][src_plane],
                                     (eee_test_hdr_t *)&pktHdr);
                            }    
                            /* synchronise the counter to the current seq no */
                            next_exp_seq_num[src_slot][src_cpu][src_plane] = seq;
                        }
                        else if(seq <
                             next_exp_seq_num[src_slot][src_cpu][src_plane]) {
                            if(next_exp_seq_num[src_slot][src_cpu][src_plane] == 
                                  (eee_test_numPkts_index[src_plane] - 1)) {        
                                eee_test_hdr_t  pktHdr;
                                printf("%s test error, expected seq %d,", 
                                        "DATA", 
                                next_exp_seq_num[src_slot][src_cpu][src_plane]);
                                printf(" got %d, slot %d, cpu %d .\n",
                                    seq, src_slot, src_cpu);
#ifdef USE_SMP                        
                                ATOMIC_INC_INT(
                                         &eee.test_dvt_rcv_err[src_plane], 1);
#else                        
                                eee.test_dvt_rcv_err[src_plane]++;
#endif                        
                                pktHdr.plane = test_hdr->plane;    
                                pktHdr.spin =  test_hdr->spin;    
                                pktHdr.size =  test_hdr->size;    
                                dvtRemake_pkt(src_slot, src_cpu, 
                                     next_exp_seq_num[src_slot][src_cpu][src_plane],
                                     (eee_test_hdr_t *)&pktHdr);
                                next_exp_seq_num[src_slot][src_cpu][src_plane] =
                                                                      seq;
                            }
                            else {        
                            /* if we are receiving a delayed packet, 
                             * drop the pkt because we have already 
                             * replaced the missing packet
                             * from the stream.
                             */     
                                printf("%s test error, expected seq %d, got %d",
                                  (src_plane == EEE_MGMT) ? "MGMT" :
                              ((src_plane == EEE_CONTROL) ? "CONTROL" : "DATA"),
                                next_exp_seq_num[src_slot][src_cpu][src_plane],
                                   seq);
                                printf(" slot %d, cpu %d Dropping pkt!\n",
                                         src_slot, src_cpu);
                                eee_freePacket(edesc);
                                break;
                            }    
                        }
                        /* if seq no is as expected */    
                        next_exp_seq_num[src_slot][src_cpu][src_plane] = 
                          (next_exp_seq_num[src_slot][src_cpu][src_plane] + 1) %
                                     eee_test_numPkts_index[src_plane];
                        /* Swap the destn and src addresses */
                        t = edesc->hdr.dest_port;
                        edesc->hdr.dest_port = edesc->hdr.src_port;
                        edesc->hdr.src_port = t;
                        if ((rc = eee_forwardPacket(edesc)) != 
                                          EEE_FORWARDED_PACKET)
#ifdef USE_SMP                        
                            ATOMIC_INC_INT(&eee.test_dvt_send_err[src_plane],1);
#else                        
                            eee.test_dvt_send_err[src_plane]++;
#endif                            
                        else
                        {        
#ifdef USE_SMP                        
                            ATOMIC_INC_INT(&dvtstats.packets_sent[src_plane],1);
                            ATOMIC_INC_INT(&dvtstats.bytes_sent[src_plane],
                                         test_hdr->size);
#else                        
                            dvtstats.packets_sent[src_plane]++;
                            dvtstats.bytes_sent[src_plane] += test_hdr->size;
#endif                            
                        }    
                    }    
                } // test is running    
            } // got my OWN pkt
            else {
#ifdef USE_SMP                        
                ATOMIC_INC_INT(
                           &dvtstats.packets_received[test_hdr->plane],1);
                ATOMIC_INC_INT(
                           &dvtstats.bytes_received[test_hdr->plane],
                           test_hdr->size);
#else                        
                dvtstats.packets_received[test_hdr->plane]++;
                dvtstats.bytes_received[test_hdr->plane] += test_hdr->size;
#endif                
                if (!eee.test_dvt_running[test_hdr->plane])
                    eee_freePacket(edesc);
                else {
                    uint32    t;
                    t = edesc->hdr.dest_port;

#if defined(FCNIM)
                    uint32    seq;
                    seq = test_hdr->seq;
#endif                    
                    edesc->hdr.dest_port = edesc->hdr.src_port;
                    edesc->hdr.src_port = t;
                    if (test_hdr->spin) {
                        if ((rc = eee_forwardPacket(edesc)) != 
                                            EEE_FORWARDED_PACKET)
#ifdef USE_SMP                        
                            ATOMIC_INC_INT(
                                    &eee.test_dvt_send_err[src_plane], 1);
#else                        
                            eee.test_dvt_send_err[src_plane]++;
#endif                            
                        else {
#ifdef USE_SMP                        
                            ATOMIC_INC_INT(
                                    &dvtstats.packets_sent[test_hdr->plane], 1);
                            ATOMIC_INC_INT(
                                    &dvtstats.bytes_sent[test_hdr->plane],
                                    test_hdr->size);
#else                        
                            dvtstats.packets_sent[test_hdr->plane]++;
                            dvtstats.bytes_sent[test_hdr->plane] += 
                                                       test_hdr->size;
#endif                            
                        }    
                    delay_us(100);
                    }
                    else
                        eee_freePacket(edesc);
                }
            }
            break;
        }
        /*
         * found out that there are other apps that are calling...
         * There is a snmp MIB function that when called gets here to
         * return stats ( ie Fan, PS, and Pkt counts) No prints here.
         * Not just a test for manufacturing... like I thought :( 
         */
        case EEE_MGMT_STATS_REQUEST: {

            src_slot = EEE_GET_SLOT_NUM(edesc->hdr.src_port);
            src_cpu = EEE_GET_DEST_CPU(edesc->hdr.src_port);
            src_plane = EEE_GET_DTYPE(edesc->hdr.src_port);
            incoming_appid = EEE_GET_APP_ID(edesc->hdr.src_port);
// Debug -------------
            if(!((src_plane >= 0) && (src_plane < 3))) {
                SYSDVT_DPRINT(("In MGMT_STATS_REQUEST src_plane =")); 
                SYSDVT_DPRINT(("%0x  not between 0-2\n",src_plane)); 
                SYSDVT_DPRINT((" nfxMyCPUId = %0x \n",nfxMyCPUId)); 
            }
// Debug -------------

#ifdef SSC_MGMT            
            dvtstats.command = EEE_MGMT_STATS_REPORT;
            dvtstats.slot = nfxMySlotId;
            dvtstats.cpu = nfxMyCPUId;
            sval = *(volatile unsigned int *)(HW_BMFPGA_BASE + 0x10);
            dvtstats.fatal_errors = sval;
            eee_systemDVTSend(src_slot, src_cpu, (uchar8 *)&dvtstats, 
                               sizeof(eee_dvt_test_stats));
#endif            
            eee_reinit_validity_table();

            for(i=0;i<3;i++) {
                dvtstats.packet_send_errors[i] = eee.test_dvt_send_err[i];
                dvtstats.seq_errors[i] = eee.test_dvt_rcv_err[i];
            }
            /* For front plane pkts - since DATA is not valid 
             *  in the validity table 
             */ 
            if((isTXRX()) && (nfxMyCPUId == NFX_NFP_TXRX1)) { 
                /* update the MGMT traffic counters from the old struct 
                 * to the new struct 
                 */    
                dvtstats_txrx.packets_sent = dvtstats.packets_sent[EEE_MGMT];
                dvtstats_txrx.bytes_sent = dvtstats.bytes_sent[EEE_MGMT];
                dvtstats_txrx.packets_received = 
                                         dvtstats.packets_received[EEE_MGMT];
                dvtstats_txrx.bytes_received = 
                                         dvtstats.bytes_received[EEE_MGMT];
                dvtstats_txrx.packet_send_errors = 
                                         dvtstats.packet_send_errors[EEE_MGMT];
                dvtstats_txrx.seq_errors = dvtstats.seq_errors[EEE_MGMT];
                dvtstats_txrx.pktComp_errors = 
                                         dvtstats.pktComp_errors[EEE_MGMT];
            }    
            if((isTXRX()) && (nfxMyCPUId == NFX_NFP_TXRX1)) { 
                dvtstats_txrx.command = EEE_MGMT_STATS_REPORT;
                dvtstats_txrx.slot = nfxMySlotId;
                dvtstats_txrx.cpu = nfxMyCPUId;
                eee_systemDVTSend(src_slot, src_cpu, 
                                  (uchar8 *)&dvtstats_txrx, 
                                  sizeof(eee_dvt_txrx_stats));
            }
            else {        
                dvtstats.command = EEE_MGMT_STATS_REPORT;
                dvtstats.slot = nfxMySlotId;
                dvtstats.cpu = nfxMyCPUId;
                eee_systemDVTSend(src_slot, src_cpu, 
                                  (uchar8 *)&dvtstats, 
                                  sizeof(eee_dvt_test_stats));
            }        
            eee_freePacket(edesc);
            //eee_testInit();
            break;
        }
        case EEE_MGMT_STATS_CLEAR: {
            memset(&dvtstats,0,sizeof(eee_dvt_test_stats));
            memset(&dvtstats_txrx,0,sizeof(eee_dvt_txrx_stats));
            memset(qdvt,0,sizeof(sysdvt_qlogic_t));
            // Error (i=0;i<4;i++)  3 Not Valid Plane
            for(i=0;i<3;i++) {
                eee_testDVTClearStats(i);
            }    
            eee_freePacket(edesc);
            break;
        }
        default:
            printf("Invalid EEE test cmd, %d\n", cmd);
    }
}    /* eee_testDVTRcv() */

/*-----------------------------------------------------------------
 * Name:    eee_anpTestRcv()
 * Description:
 *-----------------------------------------------------------------
 */
void
eee_anpTestRcv(eee_descPtr_t edesc)
{
    uint32    dp, sp, len;
    uchar8    * buf;

    printf("eee_anpTestRcv: edesc = %p\n", edesc);

    sp = edesc->hdr.src_port;
    dp = edesc->hdr.dest_port;
    len = edesc->hdr.control & EDESC_TOTAL_LEN;

    printf("    source port = 0x%x\n", sp);
    printf("    dest port   = 0x%x\n", dp);
    printf("    msg len     = %d bytes\n", len);

    buf = eee_ramAlloc(16);
    memcpy(buf, edesc->bd[1].buf, 16);
    eee_freePacket(edesc);

    memcpy (buf, "Response", 8);
    eee_sendMessage(buf, len, sp, dp);
    eee_ramDealloc(buf);

}    /* eee_anpTestRcv() */

/*-----------------------------------------------------------------
 * Name:
 * Description:
 *-----------------------------------------------------------------
 */
int32
eee_sendMessageTest(uint32    slot, uint32    cpu, uint32    num_bytes)
{
    uint32    * buf;
    uint32    i;
    uint32    dp, sp;
    int32    rc;

    dp = PACK_UNICAST_PORT_ADDRESS(slot, cpu, eee.simple_test_app_id,
                                    EEE_MGMT, NFX_COS_HIGH, 0);
    sp = PACK_UNICAST_PORT_ADDRESS(nfxMySlotId, nfxMyCPUId,
                                    eee.simple_test_app_id,
                                    EEE_MGMT, NFX_COS_HIGH, 0);
    buf = eee_ramAlloc(num_bytes);
    if (!buf) {
        printf("Failed to allocate buffer\n");
        return -1;
    }
    for (i=0; i<num_bytes/4; i++)
        buf[i] = i;

    rc = eee_sendMessage((uchar8 *)buf, num_bytes, dp, sp);

    eee_ramDealloc(buf);

    return rc;
}

/*-----------------------------------------------------------------
 * Name:    eee_testInit()
 * Description:    Initialize receive functions for test application
 *-----------------------------------------------------------------
 */
void
eee_testInit()
{
    static uint32 testInitDone = 0;
    if(!testInitDone) {
    eee_unregisterReceiveFunc(eee.simple_test_app_id);
    eee_registerReceiveFunc(eee.simple_test_app_id, eee_testSimpleRcv,
                                                    eee_testSimpleRcv);

    eee_unregisterReceiveFunc(eee.dvt_test_app_id);
    eee_registerReceiveFunc(eee.dvt_test_app_id, eee_testDVTRcv,
                                                    eee_testDVTRcv);

    eee_unregisterReceiveFunc(eee.anp_test_app_id);
    eee_registerReceiveFunc(eee.anp_test_app_id, eee_anpTestRcv,
                                                    eee_anpTestRcv);
    testInitDone = 1;
    eee_systemDVTinit();
    }

    /*
     * Which CPUs should receive start/stop/clear messages
     */
    eee.test_start_valid[0][0][EEE_MGMT] = 0;
    eee.test_start_valid[0][1][EEE_MGMT] = 0;
    eee.test_start_valid[1][0][EEE_MGMT] = 1;
    eee.test_start_valid[1][1][EEE_MGMT] = 0;
    eee.test_start_valid[1][2][EEE_MGMT] = 0;
    eee.test_start_valid[1][3][EEE_MGMT] = 0;
    eee.test_start_valid[2][0][EEE_MGMT] = 0;
    // only 2 0
    eee.test_start_valid[2][1][EEE_MGMT] = 0;
    eee.test_start_valid[2][2][EEE_MGMT] = 0;
    eee.test_start_valid[2][3][EEE_MGMT] = 0;

    eee.test_start_valid[0][0][EEE_CONTROL] = 0;
    eee.test_start_valid[0][1][EEE_CONTROL] = 0;
    eee.test_start_valid[1][0][EEE_CONTROL] = 0;
    eee.test_start_valid[1][1][EEE_CONTROL] = 0;
    eee.test_start_valid[1][2][EEE_CONTROL] = 0;
    eee.test_start_valid[1][3][EEE_CONTROL] = 1;
    eee.test_start_valid[2][0][EEE_CONTROL] = 0;
    // only 2 0
    eee.test_start_valid[2][1][EEE_CONTROL] = 0;
    eee.test_start_valid[2][2][EEE_CONTROL] = 0;
    eee.test_start_valid[2][3][EEE_CONTROL] = 0;

    eee.test_start_valid[0][0][EEE_USER_DATA] = 0;
    eee.test_start_valid[0][1][EEE_USER_DATA] = 0;
    eee.test_start_valid[1][0][EEE_USER_DATA] = 0;
    eee.test_start_valid[1][1][EEE_USER_DATA] = 0;
    eee.test_start_valid[1][2][EEE_USER_DATA] = 0;
    eee.test_start_valid[1][3][EEE_USER_DATA] = 1;
    eee.test_start_valid[2][0][EEE_USER_DATA] = 0;
    // only 2 0
    eee.test_start_valid[2][1][EEE_USER_DATA] = 0;
    eee.test_start_valid[2][2][EEE_USER_DATA] = 0;
    eee.test_start_valid[2][3][EEE_USER_DATA] = 0;

    /*
     * Which CPUs should receive data packets
     */
    eee.test_dvt_valid[0][0][EEE_MGMT] = 0;
    eee.test_dvt_valid[0][1][EEE_MGMT] = 0;
    eee.test_dvt_valid[1][0][EEE_MGMT] = 1;
    eee.test_dvt_valid[1][1][EEE_MGMT] = 0;
    eee.test_dvt_valid[1][2][EEE_MGMT] = 0;
    eee.test_dvt_valid[1][3][EEE_MGMT] = 0;
    eee.test_dvt_valid[2][0][EEE_MGMT] = 0;
    // only 2 0
    eee.test_dvt_valid[2][1][EEE_MGMT] = 0;
    eee.test_dvt_valid[2][2][EEE_MGMT] = 0;
    eee.test_dvt_valid[2][3][EEE_MGMT] = 0;

    if (isFP()) {
        eee.test_dvt_valid[0][0][EEE_CONTROL] = 0;
        eee.test_dvt_valid[0][1][EEE_CONTROL] = 0;
        eee.test_dvt_valid[1][0][EEE_CONTROL] = 0;
        eee.test_dvt_valid[1][1][EEE_CONTROL] = 0;
        eee.test_dvt_valid[1][2][EEE_CONTROL] = 0;
        eee.test_dvt_valid[1][3][EEE_CONTROL] = 1;
        eee.test_dvt_valid[2][0][EEE_CONTROL] = 0;
        eee.test_dvt_valid[2][1][EEE_CONTROL] = 0;
        eee.test_dvt_valid[2][2][EEE_CONTROL] = 0;
	// only 2 0
        eee.test_dvt_valid[2][3][EEE_CONTROL] = 0;

        eee.test_dvt_valid[0][0][EEE_USER_DATA] = 0;
        eee.test_dvt_valid[0][1][EEE_USER_DATA] = 0;
        eee.test_dvt_valid[1][0][EEE_USER_DATA] = 0;
        eee.test_dvt_valid[1][1][EEE_USER_DATA] = 0;
        eee.test_dvt_valid[1][2][EEE_USER_DATA] = 0;
        eee.test_dvt_valid[1][3][EEE_USER_DATA] = 1;
        eee.test_dvt_valid[2][0][EEE_USER_DATA] = 0;
        eee.test_dvt_valid[2][1][EEE_USER_DATA] = 0;
        eee.test_dvt_valid[2][2][EEE_USER_DATA] = 0;
	// only 2 0
        eee.test_dvt_valid[2][3][EEE_USER_DATA] = 0;
    }
    else if (isFC()) {
        if (nfxMyCPUId == NFX_FCNIM_CPUID3) {
            eee.test_dvt_valid[0][0][EEE_CONTROL] = 0;
            eee.test_dvt_valid[0][1][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][0][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][1][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][2][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][3][EEE_CONTROL] = 1;
            eee.test_dvt_valid[2][0][EEE_CONTROL] = 0;
            eee.test_dvt_valid[2][1][EEE_CONTROL] = 0;
            eee.test_dvt_valid[2][2][EEE_CONTROL] = 0;
            eee.test_dvt_valid[2][3][EEE_CONTROL] = 0;

            eee.test_dvt_valid[0][0][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[0][1][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][0][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][1][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][2][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][3][EEE_USER_DATA] = 1;
            eee.test_dvt_valid[2][0][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[2][1][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[2][2][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[2][3][EEE_USER_DATA] = 0;
        }
        else if (nfxMyCPUId == NFX_FCNIM_CPUID2) {
            eee.test_dvt_valid[0][0][EEE_CONTROL] = 0;
            eee.test_dvt_valid[0][1][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][0][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][1][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][2][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][3][EEE_CONTROL] = 0;
            eee.test_dvt_valid[2][0][EEE_CONTROL] = 0;
	    // only 2 0
            eee.test_dvt_valid[2][1][EEE_CONTROL] = 0;
            eee.test_dvt_valid[2][2][EEE_CONTROL] = 0;
            eee.test_dvt_valid[2][3][EEE_CONTROL] = 0;

            eee.test_dvt_valid[0][0][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[0][1][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][0][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][1][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][2][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][3][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[2][0][EEE_USER_DATA] = 0;
	    // only 2 0
            eee.test_dvt_valid[2][1][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[2][2][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[2][3][EEE_USER_DATA] = 0;
        }
        else if (nfxMyCPUId == NFX_FCNIM_CPUID1) {
            eee.test_dvt_valid[0][0][EEE_CONTROL] = 0;
            eee.test_dvt_valid[0][1][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][0][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][1][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][2][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][3][EEE_CONTROL] = 0;
            eee.test_dvt_valid[2][0][EEE_CONTROL] = 0;
	    // only 2 0
            eee.test_dvt_valid[2][1][EEE_CONTROL] = 0;
            eee.test_dvt_valid[2][2][EEE_CONTROL] = 0;
            eee.test_dvt_valid[2][3][EEE_CONTROL] = 0;

            eee.test_dvt_valid[0][0][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[0][1][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][0][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][1][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][2][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][3][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[2][0][EEE_USER_DATA] = 0;
	    // only 2 0
            eee.test_dvt_valid[2][1][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[2][2][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[2][3][EEE_USER_DATA] = 0;
        }
         // NFX_FCNIM_CPUID0 = 0 (nfxMyCPUId=0 is fc slice 3)
        else if (nfxMyCPUId == NFX_FCNIM_CPUID0) { 
            eee.test_dvt_valid[0][0][EEE_CONTROL] = 0;
            eee.test_dvt_valid[0][1][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][0][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][1][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][2][EEE_CONTROL] = 0;
            eee.test_dvt_valid[1][3][EEE_CONTROL] = 1;
            eee.test_dvt_valid[2][0][EEE_CONTROL] = 0;
            eee.test_dvt_valid[2][1][EEE_CONTROL] = 0;
            eee.test_dvt_valid[2][2][EEE_CONTROL] = 0;
            eee.test_dvt_valid[2][3][EEE_CONTROL] = 0;

            eee.test_dvt_valid[0][0][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[0][1][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][0][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][1][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][2][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[1][3][EEE_USER_DATA] = 1;
            eee.test_dvt_valid[2][0][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[2][1][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[2][2][EEE_USER_DATA] = 0;
            eee.test_dvt_valid[2][3][EEE_USER_DATA] = 0;
        }
    }

}    /* eee_testSimple() */


/*-----------------------------------------------------------------
 * Name:    eee_testSysDVTStart(mode)
 *
 * Description:    Start system DVT test 
 *
 * Created by: lipika    
 *
 * Date Created:    6/01/02
 *
 *-----------------------------------------------------------------
 */
int32
eee_testSysDVTStart(eee_test_hdr_t  *test_hdr)
{
    uint32    slot, cpu, i;
    int32    rc;
    eee_descPtr_t    edesc;

// Debug -------------
                if(!((test_hdr->plane >= 0) && (test_hdr->plane < 3))) {
                    printf("In test(Sys)DVTStart plane = %0x not between 0-2\n",test_hdr->plane); 
                }
// Debug -------------

    if (!eee.test_dvt_valid[nfxMySlotId][nfxMyCPUId][test_hdr->plane]) {
        printf("%s DVT test not valid on this CPU\n",
                        (test_hdr->plane == EEE_MGMT) ? "MGMT" :
                        ((test_hdr->plane == EEE_CONTROL) ? "CONTROL" : "DATA"));
        return NFX_ERR;
    }

    if (eee.test_dvt_running[test_hdr->plane]) {
        printf("EEE test already running  in testSysDVTStart()return NFX_OK  plane = %x \n",test_hdr->plane);
        return NFX_OK;
    }

    /* We allow the second core to start test as the msg 
     * from SSC can go to either core
     */ 
#ifdef NFP_FP
    /* Do not start test from the second core of the FP */
    if(nfxMyCPUId == NFX_NFP_FP2)
        return NFX_ERR;
#endif    
    /*
     * Send num_pkts to each slot/cpu
     */
    //                NFX_NUM_SLOTS_PER_CHASSIS = 12
    for (slot=0; slot<NFX_NUM_SLOTS_PER_CHASSIS; slot++) {
        //              NFX_MAX_CPUS_PER_SLOT = 16
        for (cpu=0; cpu<NFX_MAX_CPUS_PER_SLOT; cpu++) { 

            /* Allowed to create test pkt, and have NOT created pkt on this 
             * cpu for this data type
             */     
// Debug 
#if 0
if( (slot < 3) && (cpu < 4))
 printf(" -debug- test_dvt_valid[%d][%d][%d] %d\n",slot,cpu,test_hdr->plane,
            eee.test_dvt_valid[slot][cpu][test_hdr->plane]);
#endif
            if (eee.test_dvt_valid[slot][cpu][test_hdr->plane] > 0) 
            {
                if (slot == nfxMySlotId)
                    continue;
                if((isTXRX()) || (isFP()))
                   {
// -- debug  
 printf(" -debug- test_dvt_valid[%d][%d][%d]=  %d\n",slot,cpu,test_hdr->plane,
            eee.test_dvt_valid[slot][cpu][test_hdr->plane]);
// -- debug
                        next_exp_seq_num[slot][cpu][test_hdr->plane] = 0;
                        /* Create packets to be spun around */
                        for (i=0; i<eee_test_numPkts_index[test_hdr->plane]; i++) {
                            edesc = eee_testCreateDVTTestPkt(slot, cpu, test_hdr, 0xFFFF);
                            if (edesc) {
                                if ((rc = eee_forwardPacket(edesc)) != EEE_FORWARDED_PACKET)
                                    eee.test_dvt_send_err[test_hdr->plane]++;
                                else {
#ifdef USE_SMP                        
                                    ATOMIC_INC_INT(&dvtstats.packets_sent[test_hdr->plane], 1);
                                    ATOMIC_INC_INT(&dvtstats.bytes_sent[test_hdr->plane],(edesc->hdr.control & EDESC_TOTAL_LEN));
#else                        
                                    dvtstats.packets_sent[test_hdr->plane]++;
                                    dvtstats.bytes_sent[test_hdr->plane] += (edesc->hdr.control & EDESC_TOTAL_LEN);
#endif                            
                                    // DEBUG: Start the timer after 1st pkt has been sent
                                    /* if((i==0) && (timerStarted == 0)) {
                                    rc = eee_timerStart(test_timer, testTimeout, NULL, NULL, NULL, NULL);
                                    if(rc != NFX_OK)
                                            printf(": failed to start test_timer. Error %08x\n",rc);
                                    else {
                                            timerStarted ++;    
                                            printf("Started timer!!\n");
                                    }        
                                    } */
                                }    
                            }
                            else
                                eee.test_dvt_send_err[test_hdr->plane]++;
                        }
                }        
            }
        }
    }
// --debug
 // eee.test_dvt_running[plane]  could be writting out of bounds
        VERIFY((test_hdr->plane >= 0) && (test_hdr->plane < 3));
// --debug

#ifdef USE_SMP                        
    ATOMIC_INC_INT(&eee.test_dvt_running[test_hdr->plane], 1);
#else                        
    eee.test_dvt_running[test_hdr->plane] = 1;
#endif    
    return NFX_OK;

}    /* eee_testSysDVTStart() */

#if 0    // @@@ obsolete??? called by code that is commented out
static int32 
testTimeout(
    uint32    tmr_handle,
    void    * usr_p1,
    void    * usr_p2,
    void    * usr_p3,
    void    * usr_p4
)
{
    uint32 plane=0;

    for(plane=0;plane<3;plane++) {    
#ifdef NFP_TXRX                    
        if((isTXRX()) && (plane == EEE_USER_DATA))    
            eee_dvttestFrontEnd(plane);
        else
#endif                        
            eee_testDVTEnd(plane);
    }    
    timerStarted = 0;
    return NFX_OK;
}
#endif
