/*
 *-----------------------------------------------------------------
 * Name:	eee.h
 * Copyright (C) 2001, Agile Storage, Inc.
 * Description:	Local EEE structure and config structure
 *-----------------------------------------------------------------
 */
#ifndef _EEE_H
#define _EEE_H

#define EEE_NEW_RATIO

// #define EEE_VALIDATE

#if defined(EEE_DEBUG)
#define EEE_PRINTF(x) printf x
#define EEE_PRINTK(x) printk x
#else
#define EEE_PRINTF(x)
#define EEE_PRINTK(x)
#endif

#define EEE_POLLING_BUCKET 1
#define EEE_POLLING_PERF   1
#define EEE_POLLING_HIST_START_AFTER_N_SEC     5
#define EEE_POLLING_HIST                       10

extern int64 eeePollingCounterMin;
extern int64 eeePollingCounterMax;
extern int64 eeePollingCountersHist[EEE_POLLING_HIST];
extern int32 eeePollingCounterIndex;

/*------------------------------------------------------------------
 *	Local Polling Interface structures
 *------------------------------------------------------------------
 */
#define EEE_MAX_POLL_HIGH_ROUTINES	32
#define EEE_MAX_POLL_LOW_ROUTINES	32
#define EEE_NBR_POLLING_BUCKETS         6

struct eee_poll_cb {
	void	(*func)(void);
	void	*cb;
	eee_poll_cfg_t	*poll_cfg_struct;
#ifdef EEE_POLLING_BUCKET
	uint32  bucket[EEE_NBR_POLLING_BUCKETS];
#endif
};

typedef struct {
        int32 sec;
        int32 usec;
} eeeGMT_t;

typedef struct polling {
	struct eee_poll_cb high[EEE_MAX_POLL_HIGH_ROUTINES];
	struct eee_poll_cb low[EEE_MAX_POLL_LOW_ROUTINES];
#ifdef EEE_POLLING_BUCKET
        eeeGMT_t clearTime;
#endif
} eee_pfunc_t;

#define EEE_HIGH_PRIORITY_IND	0x4000
#define EEE_LOW_PRIORITY_IND	0x0

/*------------------------------------------------------------------
 *	Local Receive Interface structures
 *------------------------------------------------------------------
 */
#define EEE_MAX_APP_CLIENTS		128

typedef struct rcv {
	void	(*uni[EEE_MAX_APP_CLIENTS])();
	void	(*multi[EEE_MAX_APP_CLIENTS])();
} eee_rcv_t;

/*------------------------------------------------------------------
 *	Forwarding Interface structures
 *------------------------------------------------------------------
 */
typedef struct eee_port {
	eee_transmitFunc_t func;
	uint32	dropped;
	uint32	forwarded;
	int mtu;
} eee_port_t;

typedef struct drv {
	eee_port_t uni[EDESC2_PORT_NUM];
	eee_port_t multi[EDESC2_PORT_NUM];
	eee_port_t dtype_default[EDESC2_DATA_TYPE_NUM];
	uint32	no_uni;
	uint32	no_multi;
} eee_drv_t;

/*------------------------------------------------------------------
 * Memory sizing definitions
 *------------------------------------------------------------------
 */
#define EEE_NUM_CPUS			6		/* 4 CPUs + TXRX_ANY + FP_ANY */

#define EEE_NUM_SM_POOLS		2
#define EEE_NUM_LM_POOLS		1
#define EEE_NUM_IPC_QUEUES		6		/* num CPUs */
#define EEE_NUM_FWD_QUEUES		2
#define EEE_NUM_FIXED_FWD_QUEUES 1

extern uint32	eee_index[3][EEE_NUM_CPUS];

		/*
		 * These are per 16 MB of Shared Memory
		 */
		/* no TXRX -> TXRX packets */
		#define RQ_TXRX_SMALL_BUF		0
		#define RQ_TXRX_LARGE_BUF		0
		#define RQ_TXRX_ULTRA_BUF		0

		#define RQ_TXRX_MGMT_BUF		0

		#define RQ_TXRX_DESC			0

		/* for TXRX -> FP packets */
		#define RQ_FP_SMALL_BUF			0x40
		#define RQ_FP_LARGE_BUF			0x40
		#define RQ_FP_ULTRA_BUF			0
		#define RQ_FP_MGMT_BUF			0
		#define RQ_FP_DESC			0x80	/* SMALL + LARGE + ULTRA */
		#define SMALL_RATIO			5
		#define LARGE_RATIO			500
		#define ULTRA_RATIO			1
		#define DESC_RATIO			(SMALL_RATIO + LARGE_RATIO + ULTRA_RATIO)
		#define FWD_RATIO			(DESC_RATIO)

		#define ALLOC_DIVISOR           ((SMALL_RATIO * EEE_BUFSIZE_SMALL) + \
                                        (LARGE_RATIO * EEE_BUFSIZE_LARGE) + \
                                        (ULTRA_RATIO * EEE_BUFSIZE_ULTRA) + \
                                        (DESC_RATIO * sizeof(eee_desc_t)) + \
                                        (FWD_RATIO * sizeof(address_t)) + \
                                        ((SMALL_RATIO +     \
                                            LARGE_RATIO +   \
                                            ULTRA_RATIO +   \
                                            DESC_RATIO) * sizeof(address_t)) + \
                                        ((SMALL_RATIO +     \
                                            LARGE_RATIO +   \
                                            ULTRA_RATIO +   \
                                            DESC_RATIO) * sizeof(eee_attr_t))   \
                                        )
		/*
		 * These are per 512 forwarding entries
		 */
		#define FWD_TXRX_TXRX			0
		#define FWD_TXRX_FP				512
#if 0
		#define FWD_TX_TX				0
		#define FWD_TX_RX				64
		#define FWD_TX_FP1				192
		#define FWD_TX_FP2				192
		#define FWD_RX_TX				64
		#define FWD_RX_RX				0
		#define FWD_RX_FP1				192
		#define FWD_RX_FP2				192
#endif

		/*
		 * These are per 512 forwarding entries
		 */
		#define FWD_FP_TXRX				512
		#define FWD_FP_FP				0
#if 0
		#define FWD_FP1_TX				64
		#define FWD_FP1_RX				256
		#define FWD_FP1_FP1				0
		#define FWD_FP1_FP2				64
		#define FWD_FP2_TX				64
		#define FWD_FP2_RX				256
		#define FWD_FP2_FP1				64
		#define FWD_FP2_FP2				0
#endif


/*
 * These are per 8 small buffers
 */
#define SMALL_TO_LARGE_RATIO	16		/* S = L */
#define SMALL_TO_ULTRA_RATIO	32		/* U = 0 */
#define DESC_TO_SMALL_RATIO	48		/* D = S + L + U */
#define FWD_TO_SMALL_RATIO	0		/* F = D */

extern uint32	eee_index[3][EEE_NUM_CPUS];

#define EEE_NUM_MEM_POOLS	(EEE_NUM_LM_POOLS+EEE_NUM_SM_POOLS)

/*------------------------------------------------------------------*/

#define GET_MY_MEMPOOL_INDEX(_index)		(_index) = eee_index[0][nfxMyCPUId]
#define GET_MY_IPC_INDEX(_index)			(_index) = eee_index[1][nfxMyCPUId]
#define GET_MY_FWD_INDEX(_index)			(_index) = eee_index[2][nfxMyCPUId]

#define GET_FWD_INDEX_FROM_CPU_ID(_c_id)			\
({													\
	uint32	_r;										\
													\
	_r = eee_index[2][_c_id];						\
	_r;												\
});

#define FWD_QUEUE_FROM_MEM_POOL(_index)				\
({													\
	uint32	_r = 0xFFFFFFFF;						\
	uint32	_i;										\
													\
	for (_i=0; _i<EEE_NUM_CPUS; _i++) {				\
		if (eee_index[0][_i] == (_index)) {			\
			_r = eee_index[2][_i];					\
			break;									\
		}											\
	}												\
	if (_i == EEE_NUM_CPUS)  						\
		printf("Error finding fwd queue index\n");	\
	_r;												\
});

#define GET_MEMPOOL_INDEX_FROM_CPU_ID(_c_id)		\
({													\
	uint32	_r;										\
													\
	_r = SHARED_POOL_OFFSET + eee_index[0][_c_id];	\
});

/* Control structures for resource allocation
 */

#ifdef EEE_ALLOC_USE_TAILQ

/* The structure to link the descriptors or buffers on the free list.
 */
struct eee_local_stack_entry {
    TAILQ_ENTRY(eee_local_stack_entry) entry;
};

/* The free list.
 */
struct eee_local_stack_pool {
    /* The head of the free list.
     */
    TAILQ_HEAD(, eee_local_stack_entry) free_list;

    /* Spinlock to protect the structure.  We do not use per-cpu pools since 
     * performance does not matter in the debugging build and single list 
     * preserves the global order of deallocations.  
     */
    spinlock_t spin;

    /* The global pool to use for the allocation if not enough local buffers. 
     */
    struct eee_global_stack_pool *global_pool;
};

#else

/* The structure for allocations of local buffers or descriptors. Each
 * CPU has such structure for each buffer and descriptor type.
 *
 * The local pools provide the CPU with buffering space for the memory 
 * allocation to avoid the synchronization on every allocation.  
 */
struct eee_local_stack_pool {
    /* Pointer to the next element to be allocated. If the stack is empty,
     * top points to the first address past the stack array. If the stack
     * is full, top points to the beginning of the stack array.
     *
     * Allocating elements from the array moves the top pointer to the 
     * end of the array, deallocating elements moves the pointer to
     * beginning of the array.
     *
     * The following is true:
     *
     * top + num_avail - max_avail = beginning of the stack memory
     * top + num_avail = first address past the stack memory
     */
    address_t *top;

    /* The current available number of free elements in the pool.
     * 
     * If there are no free elements available and the allocation is
     * requested, the empty slab stack is exchanged in the global
     * pool for the full slab.
     */
    uint32 num_avail;

    /* The maximum available number of free elements in the pool.
     *
     * If the number of free elements in the pool gets higher than
     * this number, the full slab is freed to the global pool in
     * exchange for an empty slab.
     */
    uint32 max_avail;

    /* The global pool to use for the allocation if not enough local buffers.  
     */
    struct eee_global_stack_pool *global_pool;
};

#endif

/* The header of the buffer deallocated to the global pool.
 */
struct eee_global_pool_buf_hdr {
    /* The next free slab.
     */
    SLIST_ENTRY(eee_global_pool_buf_hdr) next;

    /* The address of allocation control array.
     * 
     * When on the free_list, this is pointer to the beginning of the stack
     * array.
     *
     * When on the empty list, this is pointer past the end of the stack array.
     */
    address_t *top;
};


/* The global buffers or descriptors pool shared between the CPUs.
 */
struct eee_global_stack_pool {
    /* The spinlock protecting this structure.
     */
    spinlock_t spin;

    /* Number of elements currently available, multiple of max_avail in
     * local pools.
     */
    uint32 num_avail;
    
    /* The head of full slabs list.
     */
    SLIST_HEAD(, eee_global_pool_buf_hdr) free_list;

    /* The head of empty slabs list.
     */
    SLIST_HEAD(, eee_global_pool_buf_hdr) empty_list;

#ifdef EEE_VALIDATE
    /* The size ID of the buffers in the pool, EEE_NUM_BUFSIZES means 
     * descriptor pool.  
     */
    uint32 pool_id;
    
    /* The start of the memory allocated from this pool.
     */ 
    address_t buf_base;

    /* The end of the memory allocated from this pool.
     */
    address_t buf_end;
#endif
};

extern struct eee_local_stack_pool local_desc_pool;
extern struct eee_local_stack_pool local_buf_pool[];

/* The number of elements in the slab is chosen to be large enough to make the 
 * number of times we have to go to the global pool and synchronize small 
 * enough for performance gains and also small enough compared to the total 
 * number of buffers/descriptors to not get false negatives on memory 
 * allocations (the number of descriptors is ~160K on 2GB system).  
 */

/* Number of descriptors in slab. 
 */
#define EEE_DESC_POOL_SLAB_SIZE 1024

/* Number of buffers in slab
 *
 * @@@ make dependent on buffer type.
 */
#define EEE_BUF_POOL_SLAB_SIZE 200

/* Number of bytes to reserve for extra stack control pages.  Each cpu has to
 * have an non-full slab so the reserved space depends on the number of cpus.
 */
#define EEE_STACK_POOL_RESERVE \
    (((SMP_MAX_CPUS * EEE_DESC_POOL_SLAB_SIZE * sizeof(void *)) + \
      (SMP_MAX_CPUS * EEE_NUM_BUF_SIZES * EEE_BUF_POOL_SLAB_SIZE *  \
       sizeof(void *)) + 8192 - 1) & ~(8192 - 1))

typedef struct stack_pool {
	address_t	* top;
	uint32		num_avail;
} eee_stack_pool_cntl_t;

typedef struct queue_pool {
	address_t	* start;
	address_t	* end;
	address_t	* head;
	address_t	** tail_ptr;
} eee_queue_pool_cntl_t;

typedef union pool_cntl {
#ifdef NFX_SMP
	struct eee_global_stack_pool stack;
#else
        struct stack_pool stack;
#endif
	eee_queue_pool_cntl_t queue;
} eee_pool_cntl_t;

typedef struct mem_pool {
	uint32		* curr_addr;
	uint32		bytes_avail;
} eee_mem_pool_cntl_t;

/*------------------------------------------------------------------
 *	Statistics
 *------------------------------------------------------------------
 */
typedef struct eee_stats {

	uint32	no_buf_avail[EEE_NUM_MEM_POOLS][EEE_NUM_BUF_SIZES];
	uint32	no_desc_avail[EEE_NUM_MEM_POOLS];

	uint32	low_watermark_buf[EEE_NUM_MEM_POOLS][EEE_NUM_BUF_SIZES];
	uint32	low_watermark_desc[EEE_NUM_MEM_POOLS];

	uint32	ipc_no_space[EEE_NUM_IPC_QUEUES];
	uint32	ipc_msg_too_big[EEE_NUM_IPC_QUEUES];
	uint32	ipc_nack[EEE_IPC_NUM_CLIENTS];
	uint32	ipc_ok[EEE_IPC_NUM_CLIENTS];

	uint32	no_fwd_avail[EEE_NUM_FWD_QUEUES];

	uint32	packet_too_big;

	uint32	invalid_desc_addr;
	uint32	invalid_buf_addr;

	uint32	invalid_app_id;

	uint32	rq_desc_allocated[EEE_NUM_MEM_POOLS];
	uint32	rq_buf_allocated[EEE_NUM_MEM_POOLS][EEE_NUM_BUF_SIZES];

	uint32	fwd_free_pkt[EEE_NUM_FWD_QUEUES];
	uint32	fwd_free_buf[EEE_NUM_FWD_QUEUES];
	uint32	fwd_free_desc[EEE_NUM_FWD_QUEUES];
/*
	uint32	fwd_pkts_forwarded[EEE_NUM_FWD_QUEUES];
	uint32	fwd_pkts_remote[NFX_NUM_SLOTS_PER_CHASSIS];
*/

	uint32	fwd_pkts[NFX_NUM_SLOTS_PER_CHASSIS][NFX_MAX_CPUS_PER_SLOT];
	uint32	app_rcv_pkts[EEE_MAX_APP_CLIENTS];

#ifdef CONFIG_ONSTOR_TUXRX
	uint64  txrx_mgmt_copied;
#endif

    /* Failed to convert the packet into the management buffers.
     */
    uint32  convert_fail;
} eee_stats_t;

struct eee_rcv_queue {
	eee_descPtr_t	* rcv_end;
	eee_descPtr_t	* rcv_head;
	eee_descPtr_t	** rcv_tail_ptr;
	spinlock_t  rcv_spin;
	eee_descPtr_t	* rcv_start;
	uint64		rcv_cnt;
	uint32 		rcv_max;
	uint32		rcv_hwm;
	uint32		rcv_free_pkt;
	uint32		rcv_free_buf;
	uint32		rcv_free_desc;
	uint32		rcv_pkts_forwarded;
	uint64		num_called;
	uint64		poll_idle;
	uint64		poll_valid;
};

/*
 * Rcon definitions should be kept in sync with linux driver code.
 */

/* 
 * Number of cores connected to console.
 */
#define RCON_MAX_CORES 8

/*
 * Number of cores per txrx/fp
 */
#define RCON_CORES_PER_CPU 4

/* 
 * The remote console circular buffer size.
 */
#define SM_RCON_QUEUE_MAX 512

/* 
 * Queue structure.
 */
struct rcon_queue {
    /* 
     * The number of bytes written to the queue by this cpu.
     */
    uint64_t bytes_i_wrote;

    /* 
     * The number of bytes the remote cpu read from the queue. This
     * field is updated by the remote cpu only.
     */
    uint64_t bytes_peer_read;
    uint32_t pad[4];

    /*
     * The number of bytes the remote cpu wrote to the queue. This field
     * is updated by the remote cpu only.
     */
    uint64_t bytes_peer_wrote;

    /*
     * The number of bytes this cpu read from the queue.
     */
    uint64_t bytes_i_read;
    uint32_t pad_1[4];

    /*
     * The circular buffer.
     */
    char data[SM_RCON_QUEUE_MAX];
};

/* EEE forward queue structure.
 */
struct eee_fwd_queue {
    
    /* Pointer to the next read pointer
     */
    eee_descPtr_t **fwd_tail_ptr;

    /* Pointer to the first element in the circular buffer
     */
    eee_descPtr_t *fwd_start;

    /* Pointer to the last element in the circular buffer
     */
    eee_descPtr_t *fwd_end;
    
    /* The current write pointer
     */
    eee_descPtr_t *fwd_head;

#if defined(NFX_SMP)
    /* Spinlock protecting the structure
     */
    spinlock_t fwd_spin;
#endif
    
    /* Number of entries forwarded
     */
    uint32 fwd_cnt;

};
    
#define NFX_NUM_CONSOLES 4

/*------------------------------------------------------------------
 *	 Config structure
 *------------------------------------------------------------------
 */
typedef struct eee_config {

	address_t	shmem_base;
	address_t	wka;
	uint32	my_pool_id;
	uint32	my_fwd_id;
	uint32	buf_size[EEE_NUM_BUF_SIZES];
	address_t	buf_mask[EEE_NUM_BUF_SIZES];
	uint32	buf_offset;
	uint32	sm_size;

	uint32	sm_for_desc_and_buf;
	uint32	bytes_per_small_buf;

	uint32	sm_num_buf[EEE_NUM_BUF_SIZES];
	uint32	sm_num_desc;
	uint32	sm_num_fwd;		/* total number of fwd rcv q entries */

	address_t	mgmt_space;	/* base addr of region for pci accesses on sibyte */
	uint32		mgmt_size_buf;	/* for buffers */
	uint32		mgmt_size_cont;	/* for other structures (descriptors) */
	uint32		mgmt_stack_size;

	/*
	 * Memory pools for allocation of resources
	 */
	eee_mem_pool_cntl_t		sm_pool;		/* shared memory pool */
	eee_mem_pool_cntl_t		lm_pool;		/* local memory pool */
	eee_mem_pool_cntl_t		mgmt_pool;		/* shared memory pool */

	/*
	 * Is the memory pool available (did the CPU put it's RQ entries to the
	 * WKA during init?
	 */
	uint32	shmem_avail[EEE_NUM_SM_POOLS];

    struct rcon_dev {
        /*
         * This cpu queue.
         */
        struct rcon_queue *my_rcon_queue;

        /* The remote cpu queue.
         */
        struct rcon_queue *peer_rcon_queue;
    } rcon_devs[NFX_NUM_CONSOLES];

    /* 
     * Memory allocated for incoming rcon queues.
     */
    struct rcon_queue *rcon_queues;

	/*
	 * Buffer address ranges
	 */
	void	* buf_base[EEE_NUM_MEM_POOLS][EEE_NUM_BUF_SIZES];
	void	* buf_end[EEE_NUM_MEM_POOLS][EEE_NUM_BUF_SIZES];

	/*
	 * Descriptor address ranges
	 */
	eee_descPtr_t	desc_base[EEE_NUM_MEM_POOLS];
	eee_descPtr_t	desc_end[EEE_NUM_MEM_POOLS];

	/*------------------------------------------------------------------*
	 *
	 * Structures allocated in my memory pool
	 *
	 *------------------------------------------------------------------*/

	/*
	 * The spin lock that protects the descriptor pools.
	 */
         spinlock_t descpool_spin;

	/*
	 * Free Descriptor Pools
	 */
	eee_pool_cntl_t		desc_pool[EEE_NUM_MEM_POOLS];

	/*
	 * The spin lock that protects the buffer pools.
	 */
        spinlock_t bufpool_spin;

       /*
	 * Free Buffer Pools
	 */
	eee_pool_cntl_t		buf_pool[EEE_NUM_MEM_POOLS][EEE_NUM_BUF_SIZES];

	/*
	 * traditional IPC queues
	 */
	eee_ipcEntry_t	* ipc_rcv_start[EEE_NUM_IPC_QUEUES];
	eee_ipcEntry_t	* ipc_rcv_end[EEE_NUM_IPC_QUEUES];
	volatile eee_ipcEntry_t	* ipc_rcv_curr[EEE_NUM_IPC_QUEUES];

	/*
	 * Attribute structures (for descriptors and buffers)
	 */
	eee_attr_t		* desc_attr[EEE_NUM_MEM_POOLS];
	eee_attr_t		* buf_attr[EEE_NUM_MEM_POOLS][EEE_NUM_BUF_SIZES];

        /*
	 * Forwarding (IPC) queues
	 */
	struct eee_rcv_queue rcv_queue[EEE_NUM_FWD_QUEUES + EEE_NUM_FIXED_FWD_QUEUES];
        struct eee_fwd_queue fwd_queue[EEE_NUM_FWD_QUEUES + EEE_NUM_FIXED_FWD_QUEUES];

	/*------------------------------------------------------------------*
	 *
	 * Structures allocated from other CPU's memory pools.  At init time,
	 * this CPU will need to read these addresses from the structures at
	 * the Well Known Address.
	 *
	 *------------------------------------------------------------------*/

	/*
	 * The spin lock that protects the remote queues during resupply.
	 */
    spinlock_t rq_spin;

	eee_queue_pool_cntl_t	far_desc_pool[EEE_NUM_MEM_POOLS];
	eee_queue_pool_cntl_t	far_buf_pool[EEE_NUM_MEM_POOLS][EEE_NUM_BUF_SIZES];

    eee_ipcEntry_t	* ipc_send_start[EEE_NUM_IPC_QUEUES];
	eee_ipcEntry_t	* ipc_send_end[EEE_NUM_IPC_QUEUES];
	eee_ipcEntry_t	* ipc_send_curr[EEE_NUM_IPC_QUEUES];

	/*
	 * Application Ids
	 */
	uint32	app_id;

	/*
	 * Driver and application interfaces
	 */
	eee_rcv_t		rcv;
	eee_pfunc_t		pfunc;

	eee_drv_t		drv;

	/*
	 * registered freeing routines
	 */
	void			(*free_routine[EEE_MAX_FREE_ROUTINES])(eee_descPtr_t);

	/*
	 * Statistics and error counters
	 */
	eee_stats_t		stats;

	/*
	 * Test stuff
	 */
	uint32			test_dvt_running[3];	/* EEE_USER_DATA, CONTROL, MGMT */
	uint32			simple_test_app_id;
	uint32			dvt_test_app_id;
	uint32			anp_test_app_id;

	uint32			test_start_valid[NFX_NUM_SLOTS_PER_CHASSIS][NFX_MAX_CPUS_PER_SLOT][3];
	uint32			test_dvt_valid[NFX_NUM_SLOTS_PER_CHASSIS][NFX_MAX_CPUS_PER_SLOT][3];
	uint32			test_dvt_my_sent[NFX_NUM_SLOTS_PER_CHASSIS][NFX_MAX_CPUS_PER_SLOT][3];
	uint32			test_dvt_my_rcvd[NFX_NUM_SLOTS_PER_CHASSIS][NFX_MAX_CPUS_PER_SLOT][3];
	uint32			test_dvt_other_sent[NFX_NUM_SLOTS_PER_CHASSIS][NFX_MAX_CPUS_PER_SLOT][3];
	uint32			test_dvt_other_rcvd[NFX_NUM_SLOTS_PER_CHASSIS][NFX_MAX_CPUS_PER_SLOT][3];
	uint32			test_dvt_send_err[3];
	uint32			test_dvt_rcv_err[3];
} eee_config_t;

extern eee_config_t	 eee;
extern eee_stats_t	*eee_stats;

#define EEE_MY_POOL_ID EEE_TXRX_POOL
#define EEE_MY_SHARED_POOL (EEE_MY_POOL_ID - SHARED_POOL_OFFSET)

#define EEE_FS_RPC_QUEUE_INDEX 2
#define EEE_REMOTE_QUEUE_INDEX 1

extern struct eee_rcv_queue *eee_my_receive_queue(void);

#endif	/* _EEE_H */

