/*  $OpenBSD: queue.h,v 1.16 2000/09/07 19:47:59 art Exp $  */
/*  $NetBSD: queue.h,v 1.11 1996/05/16 05:17:14 mycroft Exp $   */

/*
 * Copyright (c) 1991, 1993
 *  The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *  This product includes software developed by the University of
 *  California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *  @(#)queue.h 8.5 (Berkeley) 8/20/94
 */

#ifndef _SYS_QUEUE_H_
#define _SYS_QUEUE_H_

/*
 * This file defines five types of data structures: singly-linked lists, 
 * lists, simple queues, tail queues, and circular queues.
 *
 *
 * A singly-linked list is headed by a single forward pointer. The elements
 * are singly linked for minimum space and pointer manipulation overhead at
 * the expense of O(n) removal for arbitrary elements. New elements can be
 * added to the list after an existing element or at the head of the list.
 * Elements being removed from the head of the list should use the explicit
 * macro for this purpose for optimum efficiency. A singly-linked list may
 * only be traversed in the forward direction.  Singly-linked lists are ideal
 * for applications with large datasets and few or no removals or for
 * implementing a LIFO queue.
 *
 * A list is headed by a single forward pointer (or an array of forward
 * pointers for a hash table header). The elements are doubly linked
 * so that an arbitrary element can be removed without a need to
 * traverse the list. New elements can be added to the list before
 * or after an existing element or at the head of the list. A list
 * may only be traversed in the forward direction.
 *
 * A simple queue is headed by a pair of pointers, one the head of the
 * list and the other to the tail of the list. The elements are singly
 * linked to save space, so elements can only be removed from the
 * head of the list. New elements can be added to the list before or after
 * an existing element, at the head of the list, or at the end of the
 * list. A simple queue may only be traversed in the forward direction.
 *
 * A tail queue is headed by a pair of pointers, one to the head of the
 * list and the other to the tail of the list. The elements are doubly
 * linked so that an arbitrary element can be removed without a need to
 * traverse the list. New elements can be added to the list before or
 * after an existing element, at the head of the list, or at the end of
 * the list. A tail queue may be traversed in either direction.
 *
 * A circle queue is headed by a pair of pointers, one to the head of the
 * list and the other to the tail of the list. The elements are doubly
 * linked so that an arbitrary element can be removed without a need to
 * traverse the list. New elements can be added to the list before or after
 * an existing element, at the head of the list, or at the end of the list.
 * A circle queue may be traversed in either direction, but has a more
 * complex end of list detection.
 *
 * For details on the use of these macros, see the queue(3) manual page.
 */

/*
 * Singly-linked List definitions.
 */
#define SLIST_HEAD(name, type)                      \
struct name {                               \
    struct type *slh_first; /* first element */         \
}
 
#define SLIST_HEAD_INITIALIZER(head)                    \
    { NULL }
 
#define SLIST_ENTRY(type)                       \
struct {                                \
    struct type *sle_next;  /* next element */          \
}
 
/*
 * Singly-linked List access methods.
 */
#define SLIST_FIRST(head)   ((head)->slh_first)
#define SLIST_END(head)     NULL
#define SLIST_EMPTY(head)   (SLIST_FIRST(head) == SLIST_END(head))
#define SLIST_NEXT(elm, field)  ((elm)->field.sle_next)

#define SLIST_FOREACH(var, head, field)                 \
    for((var) = SLIST_FIRST(head);                  \
        (var) != SLIST_END(head);                   \
        (var) = SLIST_NEXT(var, field))

/* This macro iterates through the list just like SLIST_FOREACH but is
 * intended for use when we are going to delete one or more entries from
 * the list.  At any point prevVarPtr points to the entry that points to
 * 'var'.  To remove 'var' call SLIST_REMOVE_WITH_DEL. 'dstate' is a
 * boolean that is used to detect when SLIST_REMOVE_WITH_DEL has been called.
 */
#define SLIST_FOREACH_DEL(var, prevVarPtr, head, field, dstate) \
    for((var) = SLIST_FIRST(head),                       \
        (prevVarPtr) = &(SLIST_FIRST(head)),             \
        (dstate) = FALSE;                                \
        (var) != SLIST_END(head);                        \
        (prevVarPtr) = (dstate) ? (prevVarPtr) : (&(SLIST_NEXT(var, field))), \
        (var) = *(prevVarPtr), \
        (dstate) = FALSE)


/* Remove the element elt from the single-linked list
 */
#define SLIST_REMOVE(head, elt, field, type) do {        \
    struct type **prev_ptr;                              \
    for (prev_ptr = &(SLIST_FIRST(head));                \
         *(prev_ptr) != NULL;                            \
         prev_ptr = &(SLIST_NEXT(*prev_ptr, field))) {  \
        if (*prev_ptr == (elt)) {                           \
            *prev_ptr = SLIST_NEXT(elt, field);          \
            break;                                       \
        }                                                \
    }                                                    \
} while (0)

/*
 * Singly-linked List functions.
 */
#define SLIST_INIT(head) {                      \
    SLIST_FIRST(head) = SLIST_END(head);                \
}

#define SLIST_INSERT_AFTER(slistelm, elm, field) do {           \
    (elm)->field.sle_next = (slistelm)->field.sle_next;     \
    (slistelm)->field.sle_next = (elm);             \
} while (0)

#define SLIST_INSERT_HEAD(head, elm, field) do {            \
    (elm)->field.sle_next = (head)->slh_first;          \
    (head)->slh_first = (elm);                  \
} while (0)

#define SLIST_INSERT_TAIL(head, elm, field, type) do {      \
    struct type **sil_prevPtr = &(SLIST_FIRST(head));       \
    for(;                                                   \
        *sil_prevPtr != SLIST_END(head);                    \
        sil_prevPtr = &(SLIST_NEXT(*sil_prevPtr, field)))   \
        ;                                                   \
    *sil_prevPtr = (elm);                                   \
    (elm)->field.sle_next = NULL;                           \
} while (0)

#define SLIST_REMOVE_HEAD(head, field) do {             \
    (head)->slh_first = (head)->slh_first->field.sle_next;      \
} while (0)

#define SLIST_REMOVE_WITH_DEL(prevElmPtr, elm, field, dstate) do { \
    *(prevElmPtr) = SLIST_NEXT(elm, field);                        \
    (dstate) = TRUE;                                               \
} while (0)

/*
 * List definitions.
 */
#define ONS_LIST_HEAD(name, type)                       \
struct name {                               \
    struct type *lh_first;  /* first element */         \
}
#ifndef LIST_HEAD
# define LIST_HEAD ONS_LIST_HEAD
#endif /* ONS_LIST_HEAD */

#define LIST_HEAD_INITIALIZER(head)                 \
    { NULL }

#define LIST_ENTRY(type)                        \
struct {                                \
    struct type *le_next;   /* next element */          \
    struct type **le_prev;  /* address of previous next element */  \
}

/*
 * List access methods
 */
#define LIST_FIRST(head)        ((head)->lh_first)
#define LIST_END(head)          NULL
#define LIST_EMPTY(head)        (LIST_FIRST(head) == LIST_END(head))
#define LIST_NEXT(elm, field)       ((elm)->field.le_next)

#define LIST_FOREACH(var, head, field)                  \
    for((var) = LIST_FIRST(head);                   \
        (var)!= LIST_END(head);                 \
        (var) = LIST_NEXT(var, field))

/*
 * List functions.
 */
#define LIST_INIT(head) do {                        \
    LIST_FIRST(head) = LIST_END(head);              \
} while (0)

#define LIST_INSERT_AFTER(listelm, elm, field) do {         \
    if (((elm)->field.le_next = (listelm)->field.le_next) != NULL)  \
        (listelm)->field.le_next->field.le_prev =       \
            &(elm)->field.le_next;              \
    (listelm)->field.le_next = (elm);               \
    (elm)->field.le_prev = &(listelm)->field.le_next;       \
} while (0)

#define LIST_INSERT_BEFORE(listelm, elm, field) do {            \
    (elm)->field.le_prev = (listelm)->field.le_prev;        \
    (elm)->field.le_next = (listelm);               \
    *(listelm)->field.le_prev = (elm);              \
    (listelm)->field.le_prev = &(elm)->field.le_next;       \
} while (0)

#define LIST_INSERT_HEAD(head, elm, field) do {             \
    if (((elm)->field.le_next = (head)->lh_first) != NULL)      \
        (head)->lh_first->field.le_prev = &(elm)->field.le_next;\
    (head)->lh_first = (elm);                   \
    (elm)->field.le_prev = &(head)->lh_first;           \
} while (0)

#define LIST_REMOVE(elm, field) do {                    \
    if ((elm)->field.le_next != NULL)               \
        (elm)->field.le_next->field.le_prev =           \
            (elm)->field.le_prev;               \
    *(elm)->field.le_prev = (elm)->field.le_next;           \
} while (0)


/*++

Routine Description:

    Initialize list element.

Arguments:

    elm - the list element
    field - the list entry field

Return Value:

    None.

--*/

#define LIST_INIT_ELM(elm, field) do { \
    (elm)->field.le_prev = NULL;                   \
    (elm)->field.le_next = NULL;                   \
    } while (0)


/*++

Routine Description:

    Check if the list element is cleared.

Arguments:

    elm - the list element
    field - the list entry field

Return Value:

    Returns true if the list element is cleared, this means that
    list element is not on any list.

--*/

#define LIST_ELM_ZEROED(elm, field) \
    (((elm)->field.le_prev == NULL) && ((elm)->field.le_next == NULL))


/*++

Routine Description:

    Remove the element from the list and clear the pointers.
    Same as LIST_REMOVE except provides consistent state for the
    removed elements.

Arguments:

    elm - the list element
    field - list entry field

Return Value:

    None.

--*/

#define LIST_REMOVEZ(elm, field) do {              \
    LIST_REMOVE(elm, field);                       \
    LIST_INIT_ELM(elm, field);                     \
} while (0)

#define LIST_REPLACE(elm, elm2, field) do {             \
    if (((elm2)->field.le_next = (elm)->field.le_next) != NULL) \
        (elm2)->field.le_next->field.le_prev =          \
            &(elm2)->field.le_next;             \
    (elm2)->field.le_prev = (elm)->field.le_prev;           \
    *(elm2)->field.le_prev = (elm2);                \
} while (0)

#define LIST_MOVE(srclist, destlist, field) do {                        \
    ASSERT(LIST_EMPTY(destlist));                                       \
    if (!LIST_EMPTY(srclist)) {                                         \
        LIST_FIRST(destlist) = LIST_FIRST(srclist);                     \
        LIST_FIRST(destlist)->field.le_prev = &(destlist)->lh_first;    \
        LIST_INIT(srclist);                                             \
    }                                                                   \
} while (0)

/*
 * Singly-linked Tail queue definitions.
 */
#define STAILQ_HEAD(name, type)                     \
struct name {                               \
    struct type *stqh_first;/* first element */         \
    struct type **stqh_last;/* addr of last next element */     \
}

#define STAILQ_HEAD_INITIALIZER(head)                   \
    { NULL, &(head).stqh_first }

#define STAILQ_ENTRY(type)                      \
struct {                                \
    struct type *stqe_next; /* next element */          \
}

/*
 * Singly-linked Tail queue functions.
 */
#define STAILQ_EMPTY(head) ((head)->stqh_first == NULL)

#define STAILQ_INIT(head) do {                      \
    (head)->stqh_first = NULL;                  \
    (head)->stqh_last = &(head)->stqh_first;            \
} while (0)

#define STAILQ_FIRST(head)  ((head)->stqh_first)
#define STAILQ_LAST(head)   (*(head)->stqh_last)
#define STAILQ_END(head)    NULL

#define STAILQ_INSERT_HEAD(head, elm, field) do {           \
    if (((elm)->field.stqe_next = (head)->stqh_first) == NULL)  \
        (head)->stqh_last = &(elm)->field.stqe_next;        \
    (head)->stqh_first = (elm);                 \
} while (0)

#define STAILQ_INSERT_TAIL(head, elm, field) do {           \
    (elm)->field.stqe_next = NULL;                  \
    *(head)->stqh_last = (elm);                 \
    (head)->stqh_last = &(elm)->field.stqe_next;            \
} while (0)

#define STAILQ_INSERT_AFTER(head, tqelm, elm, field) do {       \
    if (((elm)->field.stqe_next = (tqelm)->field.stqe_next) == NULL)\
        (head)->stqh_last = &(elm)->field.stqe_next;        \
    (tqelm)->field.stqe_next = (elm);               \
} while (0)

#define STAILQ_NEXT(elm, field) ((elm)->field.stqe_next)

#define STAILQ_REMOVE_HEAD(head, field) do {                \
    if (((head)->stqh_first =                   \
         (head)->stqh_first->field.stqe_next) == NULL)      \
        (head)->stqh_last = &(head)->stqh_first;        \
} while (0)

#define STAILQ_REMOVE_HEAD_UNTIL(head, elm, field) do {         \
    if (((head)->stqh_first = (elm)->field.stqe_next) == NULL)  \
        (head)->stqh_last = &(head)->stqh_first;        \
} while (0)


#define STAILQ_REMOVE(head, elm, type, field) do {          \
    if ((head)->stqh_first == (elm)) {              \
        STAILQ_REMOVE_HEAD(head, field);            \
    }                               \
    else {                              \
        struct type *curelm = (head)->stqh_first;       \
        while( curelm->field.stqe_next != (elm) )       \
            curelm = curelm->field.stqe_next;       \
        if((curelm->field.stqe_next =               \
            curelm->field.stqe_next->field.stqe_next) == NULL)  \
            (head)->stqh_last = &(curelm)->field.stqe_next; \
    }                               \
} while (0)

#define STAILQ_FOREACH(var, head, field)                  \
    for((var) = STAILQ_FIRST(head);                       \
        (var)!= STAILQ_END(head);                         \
        (var) = STAILQ_NEXT(var, field))


/*
 * Simple queue definitions.
 */
#define SIMPLEQ_HEAD(name, type)                    \
struct name {                               \
    struct type *sqh_first; /* first element */         \
    struct type **sqh_last; /* addr of last next element */     \
}

#define SIMPLEQ_HEAD_INITIALIZER(head)                  \
    { NULL, &(head).sqh_first }

#define SIMPLEQ_ENTRY(type)                     \
struct {                                \
    struct type *sqe_next;  /* next element */          \
}

/*
 * Simple queue access methods.
 */
#define SIMPLEQ_FIRST(head)     ((head)->sqh_first)
#if __GNUC__ >= 3
#define SIMPLEQ_LAST(head, field)                       \
    ((typeof((head)->sqh_first))                          \
      ((void *)(head)->sqh_last - \
       offsetof(typeof(*(head)->sqh_first), field)))
#endif
#define SIMPLEQ_END(head)       NULL
#define SIMPLEQ_EMPTY(head)     (SIMPLEQ_FIRST(head) == SIMPLEQ_END(head))
#define SIMPLEQ_NEXT(elm, field)    ((elm)->field.sqe_next)

#define SIMPLEQ_FOREACH(var, head, field)               \
    for((var) = SIMPLEQ_FIRST(head);                \
        (var) != SIMPLEQ_END(head);                 \
        (var) = SIMPLEQ_NEXT(var, field))

#define SIMPLEQ_FOREACH_DEL(elm, prevElmPtr, head, field, dstate)                 \
        for ((elm) = SIMPLEQ_FIRST(head),                                         \
                 (prevElmPtr) = &(SIMPLEQ_FIRST(head)),                           \
                 (dstate) = FALSE;                                                \
             (elm) != SIMPLEQ_END(head);                                          \
             (prevElmPtr) = (dstate) ? (prevElmPtr) : &(SIMPLEQ_NEXT(elm, field)),\
                 (elm) = *(prevElmPtr),                                           \
                 (dstate) = FALSE)

/*
 * Simple queue functions.
 */
#define SIMPLEQ_INIT(head) do {                     \
    (head)->sqh_first = NULL;                   \
    (head)->sqh_last = &(head)->sqh_first;              \
} while (0)

#define SIMPLEQ_INSERT_HEAD(head, elm, field) do {          \
    if (((elm)->field.sqe_next = (head)->sqh_first) == NULL)    \
        (head)->sqh_last = &(elm)->field.sqe_next;      \
    (head)->sqh_first = (elm);                  \
} while (0)

#define SIMPLEQ_INSERT_TAIL(head, elm, field) do {          \
    (elm)->field.sqe_next = NULL;                   \
    *(head)->sqh_last = (elm);                  \
    (head)->sqh_last = &(elm)->field.sqe_next;          \
} while (0)

#define SIMPLEQ_INSERT_AFTER(head, listelm, elm, field) do {        \
    if (((elm)->field.sqe_next = (listelm)->field.sqe_next) == NULL)\
        (head)->sqh_last = &(elm)->field.sqe_next;      \
    (listelm)->field.sqe_next = (elm);              \
} while (0)

#define SIMPLEQ_REMOVE_HEAD(head, elm, field) do {          \
    if (((head)->sqh_first = (elm)->field.sqe_next) == NULL)    \
        (head)->sqh_last = &(head)->sqh_first;          \
} while (0)

#define SIMPLEQ_REMOVE(head, elm, type, field) do {         \
    if ((head)->sqh_first == (elm)) {               \
        SIMPLEQ_REMOVE_HEAD(head, elm, field);          \
    }                               \
    else {                              \
        struct type *curelm = (head)->sqh_first;        \
        while( curelm->field.sqe_next != (elm) )        \
            curelm = curelm->field.sqe_next;        \
        if((curelm->field.sqe_next =                \
            curelm->field.sqe_next->field.sqe_next) == NULL)    \
            (head)->sqh_last = &(curelm)->field.sqe_next;   \
    }                               \
} while (0)

#define SIMPLEQ_REMOVE_WITH_DEL(elm, prevElmPtr, head, field, dstate) do {    \
        if (SIMPLEQ_NEXT(elm, field) == SIMPLEQ_END(head)) {                  \
                (head)->sqh_last = (prevElmPtr);                              \
        }                                                                     \
        *(prevElmPtr) = SIMPLEQ_NEXT(*(prevElmPtr), field);                   \
        (dstate) = TRUE;                                                      \
} while (0)

/*++

Routine Description:

    Move contents of the queue q2 to q1. q2 is reinitialized to empty.

Arguments:

    q1 - queue to move to
    q2 - queue to move from
    
Return Value:

    None.

--*/

#define SIMPLEQ_MOVE(q1, q2)  \
do {                          \
    if (SIMPLEQ_EMPTY(q2)) {  \
        SIMPLEQ_INIT(q1);     \
    } else {                  \
        *(q1) = *(q2);        \
        SIMPLEQ_INIT(q2);     \
    }                         \
} while (0)                   \


/*++

Routine Description:

    Append contents of queue q2 to q1. q2 is reinitialized to empty.

Arguments:

    q1 - queue to append to
    q2 - queue to append

Return Value:

    None.

--*/

#define SIMPLEQ_APPEND(q1, q2)                 \
do {                                           \
   if (!SIMPLEQ_EMPTY(q2)) {                   \
       if (SIMPLEQ_EMPTY(q1)) {                \
           SIMPLEQ_MOVE(q1, q2);               \
       } else {                                \
           *(q1)->sqh_last = (q2)->sqh_first;  \
           (q1)->sqh_last = (q2)->sqh_last;    \
           SIMPLEQ_INIT(q2);                   \
       }                                       \
   }                                           \
} while (0)                                    
    

/*
 * Tail queue definitions.
 */
#define TAILQ_HEAD(name, type)                      \
struct name {                               \
    struct type *tqh_first; /* first element */         \
    struct type **tqh_last; /* addr of last next element */     \
}

#define TAILQ_HEAD_INITIALIZER(head)                    \
    { NULL, &(head).tqh_first }

#define TAILQ_ENTRY(type)                       \
struct {                                \
    struct type *tqe_next;  /* next element */          \
    struct type **tqe_prev; /* address of previous next element */  \
}

/* 
 * tail queue access methods 
 */
#define TAILQ_FIRST(head)       ((head)->tqh_first)
#define TAILQ_END(head)         NULL
#define TAILQ_NEXT(elm, field)      ((elm)->field.tqe_next)
#define TAILQ_LAST(head, headname)                  \
    (*(((struct headname *)((head)->tqh_last))->tqh_last))
/* XXX */
#define TAILQ_PREV(elm, headname, field)                \
    (*(((struct headname *)((elm)->field.tqe_prev))->tqh_last))
#define TAILQ_EMPTY(head)                       \
    (TAILQ_FIRST(head) == TAILQ_END(head))

#define TAILQ_FOREACH(var, head, field)                 \
    for((var) = TAILQ_FIRST(head);                  \
        (var) != TAILQ_END(head);                   \
        (var) = TAILQ_NEXT(var, field))

#define TAILQ_FOREACH_REVERSE(var, head, field, headname)       \
    for((var) = TAILQ_LAST(head, headname);             \
        (var) != TAILQ_END(head);                   \
        (var) = TAILQ_PREV(var, headname, field))

/*
 * Tail queue functions.
 */
#define TAILQ_INIT(head) do {                       \
    (head)->tqh_first = NULL;                   \
    (head)->tqh_last = &(head)->tqh_first;              \
} while (0)

#define TAILQ_INSERT_HEAD(head, elm, field) do {            \
    if (((elm)->field.tqe_next = (head)->tqh_first) != NULL)    \
        (head)->tqh_first->field.tqe_prev =         \
            &(elm)->field.tqe_next;             \
    else                                \
        (head)->tqh_last = &(elm)->field.tqe_next;      \
    (head)->tqh_first = (elm);                  \
    (elm)->field.tqe_prev = &(head)->tqh_first;         \
} while (0)

#define TAILQ_INSERT_TAIL(head, elm, field) do {            \
    (elm)->field.tqe_next = NULL;                   \
    (elm)->field.tqe_prev = (head)->tqh_last;           \
    *(head)->tqh_last = (elm);                  \
    (head)->tqh_last = &(elm)->field.tqe_next;          \
} while (0)

#define TAILQ_INSERT_AFTER(head, listelm, elm, field) do {      \
    if (((elm)->field.tqe_next = (listelm)->field.tqe_next) != NULL)\
        (elm)->field.tqe_next->field.tqe_prev =         \
            &(elm)->field.tqe_next;             \
    else                                \
        (head)->tqh_last = &(elm)->field.tqe_next;      \
    (listelm)->field.tqe_next = (elm);              \
    (elm)->field.tqe_prev = &(listelm)->field.tqe_next;     \
} while (0)

#define TAILQ_INSERT_BEFORE(listelm, elm, field) do {           \
    (elm)->field.tqe_prev = (listelm)->field.tqe_prev;      \
    (elm)->field.tqe_next = (listelm);              \
    *(listelm)->field.tqe_prev = (elm);             \
    (listelm)->field.tqe_prev = &(elm)->field.tqe_next;     \
} while (0)

#define TAILQ_REMOVE(head, elm, field) do {             \
    if (((elm)->field.tqe_next) != NULL)                \
        (elm)->field.tqe_next->field.tqe_prev =         \
            (elm)->field.tqe_prev;              \
    else                                \
        (head)->tqh_last = (elm)->field.tqe_prev;       \
    *(elm)->field.tqe_prev = (elm)->field.tqe_next;         \
} while (0)

/* See similar LIST_ functions for comments.
 */
#define TAILQ_REMOVEZ(head, elm, field) do {       \
    TAILQ_REMOVE(head, elm, field);            \
    TAILQ_INIT_ELM(elm, field);                \
} while (0)

#define TAILQ_INIT_ELM(elm, field) do { \
        (elm)->field.tqe_prev = NULL;   \
        (elm)->field.tqe_next = NULL;   \
} while (0)

#define TAILQ_ELM_ZEROED(elm, field)    \
    (((elm)->field.tqe_prev == NULL) && \
     ((elm)->field.tqe_next == NULL))

#define TAILQ_REPLACE(head, elm, elm2, field) do {          \
    if (((elm2)->field.tqe_next = (elm)->field.tqe_next) != NULL)   \
        (elm2)->field.tqe_next->field.tqe_prev =        \
            &(elm2)->field.tqe_next;                \
    else                                \
        (head)->tqh_last = &(elm2)->field.tqe_next;     \
    (elm2)->field.tqe_prev = (elm)->field.tqe_prev;         \
    *(elm2)->field.tqe_prev = (elm2);               \
} while (0)

/* Move the tailq list from s_head to d_head. Once the move is done
 * init the s_head to NULL
 */
#define TAILQ_MOVE(d_head, s_head, field, headname) do {                       \
    TAILQ_INIT(d_head);                                                        \
    if (TAILQ_FIRST(s_head) != NULL) {                                         \
        (d_head)->tqh_first = TAILQ_FIRST(s_head);                             \
        (TAILQ_FIRST(s_head))->field.tqe_prev = &(d_head)->tqh_first;          \
        (d_head)->tqh_last = &(TAILQ_LAST(s_head, headname))->field.tqe_next;  \
        TAILQ_INIT(s_head);                                                    \
    }                                                                          \
} while (0)

/* Append the s_head list to d_head list. Once the append is done
 * init the s_head to NULL
 */
#define TAILQ_APPEND_TAIL(d_head, s_head, field) do {                 \
    if (TAILQ_FIRST(s_head) != NULL) {                                \
        (TAILQ_FIRST(s_head))->field.tqe_prev = (d_head)->tqh_last;   \
        (d_head)->tqh_last = (s_head)->tqh_last;                      \
        TAILQ_INIT(s_head);                                           \
    }                                                                 \
} while (0)

/*
 * Listq queue definitions. This is double-linked list with 2 pointers for
 * the head of the list, it allows insertion to the head and tail and removal
 * of the element using only the pointer to the element.
 */
#define LISTQ_HEAD(name, type)                                            \
struct name {                                                                \
    /* This are not really pointers to struct type *, but void **.  The type \
     * is saved here so we do not need to pass an extra type arguments to    \
     * certain routine and just do typeof().                                 \
     */                                                                      \
    struct type *lqh_first; /* address of first prev element */              \
    struct type *lqh_last;  /* addr of last next element */                  \
}

#define LISTQ_HEAD_INITIALIZER(head)                    \
    { &(head).lqh_last, &(head).lqh_first }

#define LISTQ_ENTRY(type)                                \
struct {                                                    \
    void *lqe_next; /* address of next previous element */  \
    void *lqe_prev; /* address of previous next element */  \
}

/* 
 * Listq queue access methods 
 */
#define LISTQ_EMPTY(head) ((head)->lqh_first == (void *)&(head)->lqh_last)


#ifdef SSC

#include <stddef.h>

/* Note: On SSC, we do not have typeof() operator, we have to circumvent
 *       that by passing the type to the methods
 */
#define LISTQ_FIRST(head, field, type)      \
    ((type *)((void *)(head)->lqh_first -   \
     offsetof(type, field.lqe_prev)))
     
#define LISTQ_END(head, field, type)        \
    ((type *)((void *)&(head)->lqh_last -   \
     offsetof(type, field.lqe_prev)))

#define LISTQ_NEXT(elt, field, type)			\
    ((type *)((void *)(elt)->field.lqe_next -	\
     offsetof(type, field.lqe_prev)))

#define LISTQ_FOREACH(var, head, field, type)	\
    for (var = LISTQ_FIRST(head, field, type);	\
         var != LISTQ_END(head, field, type);	\
         var = LISTQ_NEXT(var, field, type))	\

#else   // SSC

#define LISTQ_FIRST(head, field)                                \
    ((typeof((head)->lqh_first))((void *)(head)->lqh_first -    \
     offsetof(typeof(*(head)->lqh_first), field.lqe_prev)))
     
#define LISTQ_END(head, field)                                  \
    ((typeof((head)->lqh_first))((void *)&(head)->lqh_last -    \
     offsetof(typeof(*(head)->lqh_first), field.lqe_prev)))

#define LISTQ_NEXT(elt, field)						\
     ((typeof(elt)((void *)(elt)->field.lqe_next -	\
      offsetof(typeof(elt), field.lqe_prev)))

#define LISTQ_FOREACH(var, head, field)	\
    for (var = LISTQ_FIRST(head, field);\
         var != LISTQ_END(head, field);	\
         var = LISTQ_NEXT(var, field))	\

#endif // SSC


/*++

Routine Description:

    Initialize the recycle queue to empty.

Arguments:

    head - pointer to the head of the queue

Return Value:

    None.

--*/

#if __GNUC__ >= 3

#define LISTQ_INIT(head) do {                       \
    (head)->lqh_first = (typeof((head)->lqh_first))&(head)->lqh_last;  \
    (head)->lqh_last = (typeof((head)->lqh_last))&(head)->lqh_first;   \
} while (0)

#else

#define LISTQ_INIT(head) do {                       \
    (void **)(head)->lqh_first = &(head)->lqh_last;             \
    (void **)(head)->lqh_last = &(head)->lqh_first;             \
} while (0)

#endif

/*++

Routine Description:

    Insert element to the head of the queue (most recently used if used as a 
    LRU queue).  

Arguments:

    head - pointer to the head of the queue
    elm - element to insert
    field - the queue entry field

Return Value:

    None.

--*/

#if  __GNUC__ >= 3

#define LISTQ_INSERT_HEAD(head, elm, field) do {      \
    (elm)->field.lqe_next = (head)->lqh_first;          \
    (elm)->field.lqe_prev = &(head)->lqh_first;         \
    *((void **)(head)->lqh_first) =                          \
       &((elm)->field.lqe_next);  \
    (head)->lqh_first = (typeof((head)->lqh_first))&(elm)->field.lqe_prev;              \
} while (0)

#else

#define LISTQ_INSERT_HEAD(head, elm, field) do {               \
    (elm)->field.lqe_next = (head)->lqh_first;                 \
    (elm)->field.lqe_prev = &(head)->lqh_first;                \
    *((void **)(head)->lqh_first) = &((elm)->field.lqe_next);    \
    (void **)(head)->lqh_first = &(elm)->field.lqe_prev;         \
} while (0)

#endif

#define LISTQ_INSERT_MRU(head, elm, field) LISTQ_INSERT_HEAD(head, elm, field)

#if  __GNUC__ >= 3

#define LISTQ_INSERT_TAIL(head, elm, field) do {             \
    (elm)->field.lqe_next = &(head)->lqh_last;               \
    (elm)->field.lqe_prev = (head)->lqh_last;                \
    *((void **)(head)->lqh_last) = &((elm)->field.lqe_prev); \
    (head)->lqh_last = (typeof((head)->lqh_last))&(elm)->field.lqe_next;  \
} while (0)

#else

#define LISTQ_INSERT_TAIL(head, elm, field) do {             \
    (elm)->field.lqe_next = &(head)->lqh_last;               \
    (elm)->field.lqe_prev = (head)->lqh_last;                \
    *((void **)(head)->lqh_last) = &((elm)->field.lqe_prev); \
    (void **)(head)->lqh_last = &(elm)->field.lqe_next;      \
} while (0)

#endif

#define LISTQ_INSERT_LRU(head, elm, field) LISTQ_INSERT_TAIL(head, elm, field)

/*++

Routine Description:

    Remove the element from the queue.

Arguments:

    elm - element to remove
    field - the queue entry field

Return Value:

    None.

--*/

#define LISTQ_REMOVE(elm, field) do {                    \
    void **next_prev = (elm)->field.lqe_next;  \
    void **prev_next = (elm)->field.lqe_prev;  \
    *next_prev = prev_next;                                 \
    *prev_next = next_prev;                                 \
} while (0)


/*++

Routine Description:

    Get the last (least recently used if used as LRU list) element in the 
    queue.  

Arguments:

    head - pointer to the queue head
    field - the queue entry field

Return Value:

    Returns pointer to the least recently used element in the queue.

--*/

#ifdef SSC
#define LISTQ_LAST(head, field, type) ((void *)(head)->lqh_last -   \
                                   offsetof(type,                   \
                                            field.lqe_next))
#define LISTQ_LRU(head, field, type) LISTQ_LAST(head, field, type)
#else
#define LISTQ_LAST(head, field) ((void *)(head)->lqh_last -           \
                                   offsetof(typeof(*((head)->lqh_last)),  \
                                            field.lqe_next))
#define LISTQ_LRU(head, field) LISTQ_LAST(head, field)
#endif
                                             
                                            
    

/*
 * Circular queue definitions.
 */
#define CIRCLEQ_HEAD(name, type)                    \
struct name {                               \
    struct type *cqh_first;     /* first element */     \
    struct type *cqh_last;      /* last element */      \
}

#define CIRCLEQ_HEAD_INITIALIZER(head)                  \
    { CIRCLEQ_END(&head), CIRCLEQ_END(&head) }

#define CIRCLEQ_ENTRY(type)                     \
struct {                                \
    struct type *cqe_next;      /* next element */      \
    struct type *cqe_prev;      /* previous element */      \
}

/*
 * Circular queue access methods 
 */
#define CIRCLEQ_FIRST(head)     ((head)->cqh_first)
#define CIRCLEQ_LAST(head)      ((head)->cqh_last)
#define CIRCLEQ_END(head)       ((void *)(head))
#define CIRCLEQ_NEXT(elm, field)    ((elm)->field.cqe_next)
#define CIRCLEQ_PREV(elm, field)    ((elm)->field.cqe_prev)
#define CIRCLEQ_EMPTY(head)                     \
    (CIRCLEQ_FIRST(head) == CIRCLEQ_END(head))

#define CIRCLEQ_FOREACH(var, head, field)               \
    for((var) = CIRCLEQ_FIRST(head);                \
        (var) != CIRCLEQ_END(head);                 \
        (var) = CIRCLEQ_NEXT(var, field))

#define CIRCLEQ_FOREACH_REVERSE(var, head, field)           \
    for((var) = CIRCLEQ_LAST(head);                 \
        (var) != CIRCLEQ_END(head);                 \
        (var) = CIRCLEQ_PREV(var, field))

/*
 * Circular queue functions.
 */
#define CIRCLEQ_INIT(head) do {                     \
    (head)->cqh_first = CIRCLEQ_END(head);              \
    (head)->cqh_last = CIRCLEQ_END(head);               \
} while (0)

#define CIRCLEQ_INSERT_AFTER(head, listelm, elm, field) do {        \
    (elm)->field.cqe_next = (listelm)->field.cqe_next;      \
    (elm)->field.cqe_prev = (listelm);              \
    if ((listelm)->field.cqe_next == CIRCLEQ_END(head))     \
        (head)->cqh_last = (elm);               \
    else                                \
        (listelm)->field.cqe_next->field.cqe_prev = (elm);  \
    (listelm)->field.cqe_next = (elm);              \
} while (0)

#define CIRCLEQ_INSERT_BEFORE(head, listelm, elm, field) do {       \
    (elm)->field.cqe_next = (listelm);              \
    (elm)->field.cqe_prev = (listelm)->field.cqe_prev;      \
    if ((listelm)->field.cqe_prev == CIRCLEQ_END(head))     \
        (head)->cqh_first = (elm);              \
    else                                \
        (listelm)->field.cqe_prev->field.cqe_next = (elm);  \
    (listelm)->field.cqe_prev = (elm);              \
} while (0)

#define CIRCLEQ_INSERT_HEAD(head, elm, field) do {          \
    (elm)->field.cqe_next = (head)->cqh_first;          \
    (elm)->field.cqe_prev = CIRCLEQ_END(head);          \
    if ((head)->cqh_last == CIRCLEQ_END(head))          \
        (head)->cqh_last = (elm);               \
    else                                \
        (head)->cqh_first->field.cqe_prev = (elm);      \
    (head)->cqh_first = (elm);                  \
} while (0)

#define CIRCLEQ_INSERT_TAIL(head, elm, field) do {          \
    (elm)->field.cqe_next = CIRCLEQ_END(head);          \
    (elm)->field.cqe_prev = (head)->cqh_last;           \
    if ((head)->cqh_first == CIRCLEQ_END(head))         \
        (head)->cqh_first = (elm);              \
    else                                \
        (head)->cqh_last->field.cqe_next = (elm);       \
    (head)->cqh_last = (elm);                   \
} while (0)

#define CIRCLEQ_REMOVE(head, elm, field) do {               \
    if ((elm)->field.cqe_next == CIRCLEQ_END(head))         \
        (head)->cqh_last = (elm)->field.cqe_prev;       \
    else                                \
        (elm)->field.cqe_next->field.cqe_prev =         \
            (elm)->field.cqe_prev;              \
    if ((elm)->field.cqe_prev == CIRCLEQ_END(head))         \
        (head)->cqh_first = (elm)->field.cqe_next;      \
    else                                \
        (elm)->field.cqe_prev->field.cqe_next =         \
            (elm)->field.cqe_next;              \
} while (0)

#define CIRCLEQ_REPLACE(head, elm, elm2, field) do {            \
    if (((elm2)->field.cqe_next = (elm)->field.cqe_next) ==     \
        CIRCLEQ_END(head))                      \
        (head).cqh_last = (elm2);               \
    else                                \
        (elm2)->field.cqe_next->field.cqe_prev = (elm2);    \
    if (((elm2)->field.cqe_prev = (elm)->field.cqe_prev) ==     \
        CIRCLEQ_END(head))                      \
        (head).cqh_first = (elm2);              \
    else                                \
        (elm2)->field.cqe_prev->field.cqe_next = (elm2);    \
} while (0)

#endif  /* !_SYS_QUEUE_H_ */
