/* vi:ts=4
 *-----------------------------------------------------------------
 * Name        : pkt-queue-api.c
 * Description : Linux implementation of packet queue api
 * Copyright (C) 2009, OnStor Inc.
 *-----------------------------------------------------------------
 */
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>

#include <linux/gfp.h>
#include <linux/slab.h>

#include "nfx-defs.h"
#include "nfx-types.h"
#include "nfx-error.h"

#include "tpl-api.h"
#include "tpl.h"

#include "eee-desc.h"
#include "pkt-api.h"
#include "pkt-queue-struct.h"
#include "pkt-queue-api.h"

#ifdef LATER
/* Packet allocator free lists
 */
struct pkt_free_list pkt_acpu_free_list;
struct pkt_free_list pkt_ncpu_free_list;

/* Pointer to the packet array
 */
struct sk_buff *pkt_acpu_memory;

/* State machine function prototypes.
 */
void pq_pullup_finish(req_hdr_t *req);
NTSTATUS req_alloc_packet_continue(cifs_request_t *req);
NTSTATUS req_copy_dcaches_cont(any_request_ptr_t req);

/* The allocation statistics 
 */
int32 pkt_alloc_stats[PKT_ALLOC_TYPE_MAX];

/* The code for appending the packets to the queue requires pNext to be the
 * first field of the pkt_desc_t structure.
 */
NFX_ASSERT_STATIC(offsetof(pkt_desc_t, pNext) == 0);
#endif /* LATER */

#ifdef DEBUG
extern void *ultranew_memcpy(void *, const void *, int);
#endif /* DEBUG */

/*
 * Description:
 *  Given a location in the queue, calculate the offset from the  
 *  beginning of the queue.
 *    
 * Arguments:
 *  pq - packet queue
 *  loc - location
 *    
 * Return Value:
 *  Returns offset from the beginning of the queue.
 */
int
pq_location_offset(pkt_queue_t *pq, struct pq_location *loc)
{
    struct sk_buff *pkt;
    int offset = 0;

    for (pkt = pq->pq_head; pkt != loc->pql_current; pkt = pkt->next) {
        offset += pkt->len;
    }
    // VERIFY(pkt != NULL);
    return (offset + ((char *)loc->pql_data - (char *)loc->pql_current->data));
}

#ifdef LATER
/*
 * Description:
 *  Wait on the free list for the packet to become available.
 *
 * Arguments:
 *  req - request
 *  free_list - the packet free list
 *
 * Return Value:
 *  Returns STATUS_PENDING.
 */
NTSTATUS
pkt_acpu_wait_on_list(any_request_ptr_t req, struct pkt_free_list *free_list)
{
    SIMPLEQ_INSERT_TAIL(&free_list->pf_waiters, req.v, rh_wait_entry);
    return req_block_x(req.v, KPI_REQ_WAIT_PKT);
}
#endif /* LATER */

/*
 * Description:
 *  This routine appends a packet chain to the tail of the packet queue
 *  specified by the argument pq.
 *
 * Arguments:
 *  pq - packet queue
 *  head - start of the packet chain
 *  tail - tail of the packet chain
 *  size - the size of the packet chain
 *
 * Return Value:
 *  None.
 */
void 
pq_append_chain_to_queue(pkt_queue_t *pq, struct sk_buff *head, struct sk_buff *tail, uint32 size)
{
    // ASSERT(tail->next == NULL);

    pq->pq_tail->next = head;
    pq->pq_tail = tail;
    if (pq->pq_current == NULL) {
        pq->pq_current = head;
        pq->pq_offset = 0;
    }
    pq->pq_len += size;

    pq_validate(pq);
    return;
}

/*
 * Description:
 *  Add a packet to the packet queue.
 *
 * Arguments:
 *  pq - queue
 *  packet - packet to add
 *
 * Return Value:
 *  None.
 */
void
pq_add_packet(pkt_queue_t *pq, struct sk_buff *packet)
{
    // ASSERT((packet->flags & PKT_DESC_INUSE));
    // ASSERT(packet->next == NULL);

    pq_append_chain_to_queue(pq, packet, packet, packet->len);
    return;
}

/*
 * Description:
 *  Append packet queue specified by the argument add_queue to the tail of the 
 *  packet queue specified by the argument pq.  
 *
 * Arguments:
 *  pq - packet queue
 *  add_queue - packet queue to add
 *
 * Return Value:
 *  None.
 */
void
pq_append_queue(pkt_queue_t *pq, pkt_queue_t *add_queue)
{
    pq_append_chain_to_queue(pq, add_queue->pq_head, add_queue->pq_tail, add_queue->pq_len);
    return;
}

/*
 * Description:
 *  This routine initializes the packet queue to contain the packet
 *  specified by the argument packet. The pq_current field
 *  is initialized to point to the start of the queue, pq_offset
 *  and pq_start_offset are initialized to 0.
 *
 * Arguments:
 *  pq - queue to initialize
 *  packet - pointer to the first packet in the packet chain
 *  len - the length of the queue
 *
 * Return Value:
 *  None.
 */
void
pq_init_from_packet(pkt_queue_t *pq, struct sk_buff *packet, uint32 len)
{
    struct sk_buff *tail;

#if defined(DEBUG)
    uint32 pktTotalLen = 0;
#endif

    pq->pq_head = packet;
    pq->pq_current = packet;

    tail = packet;
#if defined(DEBUG)
    pktTotalLen += tail->len;
#endif        
    while (tail->next != NULL) {
        tail = tail->next;
#if defined(DEBUG)
        pktTotalLen += tail->len;
#endif        
    }
    pq->pq_tail = tail;

    pq->pq_start_offset = pq->pq_offset = 0;
    pq->pq_len = len;
    pq->pq_flags = 0;

    // ASSERT(len == pktTotalLen);    

    pq_validate(pq);
    return;
}

/*
 * Description:
 *  Make empty packet queue.
 *
 * Arguments:
 *  pq - packet queue
 *
 * Return Value:
 *  None.
 */
void
pq_make_empty(pkt_queue_t *pq)
{
#ifdef LATER
    pq->pq_head = NULL;
    pq->pq_tail = (struct sk_buff *)&pq->pq_head;
    pq->pq_len = 0;
    pq->pq_flags = 0;
    pq->pq_start_offset = 0;
    pq->pq_offset = 0;
    pq->pq_current = NULL;
#else
    memset(pq, 0, sizeof(pkt_queue_t));
    pq->pq_tail = (struct sk_buff *)&pq->pq_head;
#endif /* LATER */
    return;
}

/*
 * Description:
 *  Adjust the current pointers by the given offset.
 *
 * Arguments:
 *  pq - packet queue
 *  offset - number of bytes to adjust
 *
 * Return Value:
 *  None.
 */
void
pq_adjust(pkt_queue_t *pq, uint32 offset)
{
    struct sk_buff *current_skb;
    uint32 remain;

    /* The maximum offset which can be set is the next byte after the queue
     * end.
     */
#if 0
    // ASSERT(pq->pq_start_offset + offset <= pq->pq_len);
    /*@@@ - can't turn this on yet
     */
#else
    if (pq->pq_start_offset + offset > pq->pq_len) {
        offset = pq->pq_len - pq->pq_start_offset;
    }
#endif    

    remain = offset;
    while (remain > 0 && pq->pq_current != NULL) {
        current_skb = pq->pq_current;

        if (remain < current_skb->len - pq->pq_offset) {
            pq->pq_offset += remain;
            remain = 0;
        } else {
            remain -= current_skb->len - pq->pq_offset;
            pq->pq_current = current_skb->next;
            pq->pq_offset = 0;
        }
    }
    pq->pq_start_offset += offset;

    /* Skip packets containing no data
     */
    while (pq->pq_current != NULL && pq->pq_current->len == 0) {
        pq->pq_current = pq->pq_current->next;
    }

    pq_validate(pq);
    return;
}

/*
 * Description:
 *  Set the queue current pointers to the given offset from the
 *  beginning of the packet queue.
 *
 * Arguments:
 *  pq - packet queue
 *  offset_from_start - offset from the beginning of the packet queue
 *
 * Return Value:
 *  None.
 */
void
pq_seek_to(pkt_queue_t *pq, uint32 offset_from_start)
{
    uint32 adjust_offset;
    // ASSERT(offset_from_start <= pq->pq_len);

    if (offset_from_start < pq->pq_start_offset) {
        /* Moving backward from the current position
         */
        adjust_offset = pq->pq_start_offset - offset_from_start;

        if (pq->pq_current == NULL) {
            pq->pq_current = pq->pq_tail;
            pq->pq_offset = pq->pq_current->len;
        }
        if (pq->pq_offset >= adjust_offset) {
            /* If we are moving within the last packet, just change
             * the offsets and don't search from the beginning of the
             * queue.
             */
            pq->pq_offset -= adjust_offset;
            pq->pq_start_offset = offset_from_start;
        } else {
            pq->pq_current = pq->pq_head;
            pq->pq_offset = pq->pq_start_offset = 0;
            pq_adjust(pq, offset_from_start);
        }
    } else if (offset_from_start > pq->pq_start_offset) {
        /* Moving forward from the current position.
         */
        pq_adjust(pq, offset_from_start - pq->pq_start_offset);
    } else {
        /* Current position is not changing.
         */
    }

    pq_validate(pq);
    return;
}

/*
 * Description:
 *  Check if packet at pq_current pointer contains at least
 *  'size' bytes of data.  
 *
 * Arguments:
 *  pq - packet queue
 *  size - the requested size of contiguous data
 *
 * Return Value:
 *  Returns true if the data is contiguous, otherwise returns FALSE.
 */
int
pq_is_contig(pkt_queue_t *pq, uint32 size)
{
    return (pq->pq_current->len - pq->pq_offset >= size);
}

/*
 * Description:
 *  Move data from skb starting from the next packet into 
 *
 * Arguments:
 *  pq - packet queue
 *  to - skb for the destination packet
 *  data_ptr - pointer to the memory where to copy the data
 *  bytes - number of bytes to copy
 *
 * Return Value:
 *  None.
 */
void
pq_movedata(pkt_queue_t *pq, struct sk_buff *to, void *data_ptr, uint32 bytes)
{
    struct sk_buff *prev, *from;
    void *data_ptr_local;
    int copy_bytes;

    prev = to;
    from = to->next;
    data_ptr_local = data_ptr;

    while ((from != NULL) && (bytes > 0)) {

        copy_bytes = bytes < from->len ? bytes : from->len;

        memcpy(data_ptr, from->data, copy_bytes);

        if ((from->len - copy_bytes) != 0) {
            /* XXX skb_push(from, copy_bytes) */
	    from->data += copy_bytes;
            from->len -= copy_bytes;

            prev = from;
            from = from->next;
        } else {
            prev->next = from->next;
            if (from == pq->pq_tail) {
                pq->pq_tail = prev;
            }
            from->next = NULL;
            pkt_free(from);
            from = prev->next;
        }
        bytes -= copy_bytes;
	data_ptr_local += copy_bytes;
        // (void *)data_ptr += copy_bytes;
    }
    // ASSERT(bytes == 0);
    return;
}

/*
 * Description:
 *  Insert a packet into the queue.
 *
 * Arguments:
 *  pq - packet queue
 *  after - insert after this packet
 *  new_packet - packet to insert
 *
 * Return Value:
 *  None.
 */
void
pq_insert_after(pkt_queue_t *pq, struct sk_buff *after, struct sk_buff *new_packet)
{
    new_packet->next = after->next;
    after->next = new_packet;
    if (new_packet->next == NULL) {
        pq->pq_tail = new_packet;
    }
    return;
}

/*
 * Description:
 *  Split the packet in the queue at the given offset moving the rest of
 *  the data into the provided packet.
 *  The queue position is not updated, the caller should update it if necessary.
 *
 * Arguments:
 *  pq - packet queue
 *  split_packet - packet to split
 *  offset - offset in the split_packet
 *  new_packet - new packet
 *
 * Return Value:
 *  None.
 */
void
pq_split_right(pkt_queue_t *pq, struct sk_buff *split_packet, uint32 offset, struct sk_buff *new_packet)
{
    int move_data_size;

    pq_save_data(pq);

    pq_insert_after(pq, split_packet, new_packet);

    /* Move the rest of the current packet starting at the current
     * offset to new packet
     */
    move_data_size = split_packet->len - offset;

    // VERIFY(move_data_size >= 0);

    /* XXX */
    memcpy(new_packet->data, split_packet->data + offset, move_data_size);
    split_packet->len = offset;
    new_packet->len = move_data_size;

    pq_compare_saved_data(pq);
    return;
}

/*
 * Description:
 *  Split current packet at current offset, moving the rest of the data
 *   into the provided packet.
 *
 * Arguments:
 *  pq - packet queue
 *  new_packet - new packet
 *
 * Return Value:
 *  None.
 */
void
pq_split_current_packet_right(pkt_queue_t *pq, struct sk_buff *new_packet)
{
    // ASSERT(pq->pq_offset != 0);

    pq_split_right(pq, pq->pq_current, pq->pq_offset, new_packet);

    /* Move the current pointer to the beginning of the new packet
     */
    pq->pq_offset = 0;
    pq->pq_current = new_packet;

    pq_validate(pq);
    return;
}

/*
 * Description:
 *  Split current packet at current offset. The data before the
 *  current position is moved to the new packet.
 *
 * Arguments:
 *  pq - packet queue
 *  new_packet - new packet
 *
 * Return Value:
 *  None.
 */
void
pq_split_current_packet_left(pkt_queue_t *pq, struct sk_buff *new_packet)
{
    struct sk_buff *pkt;
    // ASSERT(pq->pq_offset != 0);

    pq_save_data(pq);

    pkt = pq->pq_head;

    if (pkt == pq->pq_current) {
        pq->pq_head = new_packet;
        new_packet->next = pq->pq_current;
    } else {
        while (pkt->next != pq->pq_current) {
            pkt = pkt->next;
        }
        pq_insert_after(pq, pkt, new_packet);
    }
    memcpy(new_packet->data, pq->pq_current->data, pq->pq_offset);

    // VERIFY(pq->pq_current->len >= pq->pq_offset);

    new_packet->len = pq->pq_offset;
    /* XXX skb_push(pq->pq_current, pq->pq_offset) */
    pq->pq_current->data += pq->pq_offset;
    pq->pq_current->len -= pq->pq_offset;
    pq->pq_offset = 0;

    pq_compare_saved_data(pq);
    pq_validate(pq);
    return;
}

/*
 * Description:
 *  Pullup the packet data at pq_current/pq_offset into newly allocated packet.  
 *
 * Arguments:
 *  pq - packet queue
 *  new_packet - newly allocated packet
 *
 * Return Value:  
 *  None.
 */
void
pq_pullup_into_new_packet(pkt_queue_t *pq, struct sk_buff *new_packet, uint32 pullup_size)
{
    int copy_bytes;

    // VERIFY(pullup_size >= (pq->pq_current->len - pq->pq_offset));
    // VERIFY(pullup_size <= (EEE_BUFSIZE_LARGE - (new_packet->data - (char *)new_packet)));

    new_packet->len = pullup_size;

    /* Insert the new packet after the current
     */
    pq_insert_after(pq, pq->pq_current, new_packet);

    /* Move the data from the pq_current packet into the new_packet
     */
    copy_bytes = pq->pq_current->len - pq->pq_offset;

    /* XXX */
    memcpy(new_packet->data, pq_current_ptr(pq), copy_bytes);
    pq->pq_current->len = pq->pq_offset;

    /* Move the rest of the data from the next packets
     */
    pq_movedata(pq,
		new_packet,
		new_packet->data + copy_bytes,
                new_packet->len - copy_bytes);

    /* Remove the packet which contained the data we moved if it is now empty.  
     */
    if (pq->pq_current->len == 0) {
        struct sk_buff **prev = &pq->pq_head;

        while (*prev != pq->pq_current) {
            prev = &((*prev)->next);
        }
        *prev = pq->pq_current->next;
    }

    /* Setup pq_current to point to the new allocated packet
     */
    pq->pq_current = new_packet;
    pq->pq_offset = 0;

    pq_validate(pq);
    return;
}

#ifdef LATER
/*
 * Description:
 *  Continuation function is called after memory allocation for pullup data is done.  
 *  The current state is PQ_PULLUP_FINISH.  The request context is the packet 
 *  queue to pull up.  The allocation routine has stored the pointer to 
 *  allocated packet by calling req_set_result().  The pullup size is stored 
 *  in pq_alloc_len field of the queue.  
 *
 * Arguments:
 *  req - current request
 *
 * Return Value:
 *  None.
 */
void
pq_pullup_finish(req_hdr_t *req)
{
    pkt_queue_t *pq = req_context(req);
    struct sk_buff *pkt = req_last_result(req);

    /* XXX */
    pkt->data = (char *)pkt + sizeof(struct sk_buff);
    pq_pullup_into_new_packet(pq, pkt, pq->pq_alloc_len);
    pq_compare_saved_data(pq);
    req_pop_state(req, STATUS_SUCCESS);

    /* XXX */
    req_run(req);
    return;
}

/*
 * Description:
 *  This routines ensures the packet data starting at the current
 *  pointer is occupying contiguous locations.
 *  If M_NOWAIT is specified in allocation flags, the packet allocation is
 *  required and there are no memory available, STATUS_INSUFF_SERVER_RESOURCES
 *  will be returned. Otherwise, the request will block for memory allocation.
 *
 *  This routine may have to perform memory allocation to have large
 *  enough contiguous region, so the caller should prepare to block
 *    before calling this routine.
 *
 * Arguments:
 *  req - current request
 *  pq - packet queue
 *  pullup_size - the required contiguous size
 *  free_list - free list to use for packet allocation if required
 *  flags - allocation flags 
 *
 * Return Value:
 *  STATUS_SUCCESS - pullup successful.
 *  STATUS_PENDING - need to wait for the memory allocation.
 */
NTSTATUS
pq_pullup(req_hdr_t *reqHdr,
          pkt_queue_t *pq,
          uint32 pullup_size,
          struct pkt_free_list *free_list,
          uint32 flags)
{
    struct sk_buff *current, *pkt;
    int bytes_in_this_buffer;
    uint32 can_use_this_buffer, bytes_left, gpf_flags;

    pq_validate(pq);

    if (pullup_size == 0) {
        /* There is no data to pull up.
         */
        return STATUS_SUCCESS;
    }

    // VERIFY(pullup_size <= EEE_BUFSIZE_LARGE);

    /* M_WAIT is not supported on NCPU.
     */
    // ASSERT((free_list == &pkt_acpu_free_list) || (flags & M_NOWAIT));

    current = pq->pq_current;
    bytes_in_this_buffer = current->len - pq->pq_offset;

    if (bytes_in_this_buffer >= pullup_size) {
        /* The data is already contiguous
         */
        return STATUS_SUCCESS;
    } else {
        pq_save_data(pq);

        // ASSERT(current->data - pkt_buf(current) < pkt_buflen(current));

        /* XXX */
        can_use_this_buffer = pkt_buflen(current) -
            ((current->data + pq->pq_offset) - pkt_buf(current));

        if (can_use_this_buffer >= pullup_size) {
            /* The data fits in the leftover of the current buffer
             * Move the rest of the data from next buffers.
             */
            bytes_left = pullup_size - bytes_in_this_buffer;

	    /* XXX */
            pq_movedata(pq,
                        current,
                        current->data + current->len,
                        bytes_left);
            current->len += bytes_left;
            pq_compare_saved_data(pq);
            pq_validate(pq);
            return STATUS_SUCCESS;
        } else {
            /* Need to allocate more memory to pull up.
             */
#ifdef LATER_LATER
            pkt = ((free_list == &pkt_ncpu_free_list) ? 
                               pkt_allocPkt() :
                               pkt_alloc_from_list(free_list, reqHdr->rh_alloc_type));
#else
	    gpf_flags = (flags & M_NOWAIT) 
	      ? (__GFP_HIGHMEM | __GFP_NORETRY | __GFP_DMA)
	      : (__GFP_HIGHMEM | __GFP_NOFAIL | __GFP_DMA);

	    pkt = alloc_skb(EEE_BUF_LARGE, gpf_flags);
#endif /* LATER_LATER */

            if (pkt == NULL) {
                if (flags & M_NOWAIT) {
                    return STATUS_INSUFF_SERVER_RESOURCES;
                } else {

                    /* Wait for the memory to become available.
		     * XXX
                     */
                    pq->pq_alloc_len = pullup_size;
                    req_push_state(reqHdr, PQ_PULLUP_FINISH, pq);
                    req_push_state(reqHdr, REQ_WAIT_ONE_PACKET, NULL);
                    return pkt_acpu_wait_on_list(reqHdr, free_list);
                }
            } else {
	        /* XXX */
                pkt->data = (char *)pkt + sizeof(struct sk_buff);

                pq_pullup_into_new_packet(pq, pkt, pullup_size);
                return STATUS_SUCCESS;
            }
        }
    }
    return;
}

/*
 * Description:
 *
 *  This routine attempts to align the data starting from the current pointer
 *  to the requested alignment.  If possible the data will be right shifted
 *  within the packet, but if that isn't possible a new packet will be allocated.
 *
 *  We will not try to align more than the maximum bytes specified by caller.
 *
 *  If the amount of data remaining in the current packet is not a multiple
 *  of the given alignment size, we will pullup some characters from the
 *  next packet in the chain so that there are no partial units.
 *
 *  This primary purpose of this call is to pullup misaligned Unicode
 *  strings.  In that case there should be at most a single byte movement
 *  but we do want to avoid having a code point span a packet boundary.
 *
 *  WARNING - This move is a destructive process.  If the caller were to
 *             pq_seek backwards they would not be able to easily locate the
 *             buffer.  For example, the CIFS Unicode string would not be
 *             found in the expected place after the fixed portion of the
 *             request structure.  As long as the caller knew that they had
 *             made this call then they could use that knowlege to pq_adjust
 *             them packet to the location we will be moving things.
 *
 *  This is a blocking call as a new packet might need to be allocated.
 *
 * Arguments:
 *  reqHdr - The request.
 *  pq - packet queue
 *  align_size - The alignment size in bytes.
 *  max_size - max num bytes to pull up.  We may pull up
 *             less, but it must be a multiple of the alignment size.
 *
 * Return Value:
 *  STATUS_SUCCESS - Success.
 *  STATUS_PENDING - The request was queued.  Return back to the dispatcher.
 */
NTSTATUS
pq_alignAtOffset(req_hdr_t *reqHdr, pkt_queue_t *pq, uint32 align_size, uint32 max_size)
{
    struct sk_buff *current, *pkt;
    void *source, *target;
    uint32 pullup_size, shift_bytes, bytes_to_shift;
    uint32 bytes_needed_from_next_buf, space_in_this_buffer;
    uint32 remaining_current_pkt;

    // ASSERT(align_size > 1);
    // ASSERT((max_size & (align_size - 1)) == 0);
    // ASSERT(max_size <= (pq->pq_len - pq->pq_start_offset));

    if (max_size == 0) {
        return STATUS_SUCCESS;
    }

    current = pq->pq_current;
    pq_save_data(pq);

    // ASSERT(current->data - pkt_buf(current) < pkt_buflen(current));

    /* Get the target position for the moved bytes.
     */
    /* XXX */
    source = current->data + pq->pq_offset;
    target = alignptr(source, align_size);

    /* Keep track of how many bytes we need to shift things to the right.
     */
    shift_bytes = (uintptr_t)target - (uintptr_t)source;

    /* Get the number of bytes we need to right shift as well as how many
     * needed from the next packet to satisfy the alignment requirements.
     */
    bytes_to_shift = current->len - pq->pq_offset; 
    bytes_needed_from_next_buf;

    if (max_size > (current->len - pq->pq_offset)) {
        bytes_needed_from_next_buf = GET_PAD(bytes_to_shift, align_size);
    } else {
        bytes_needed_from_next_buf = 0;
    }

    // ASSERT((bytes_needed_from_next_buf == 0) || (current->next != NULL));

    space_in_this_buffer = pkt_buflen(current) -
      ((uintptr_t)target - (uintptr_t)(pkt_buf(current)));

    if ((pq->pq_offset == 0) &&
        (space_in_this_buffer >= (bytes_to_shift + bytes_needed_from_next_buf))) {

        /* The data will fit in the current buffer.
         * Right shift the current data and copy what we need from the next
         * buffer.
         */
        //@@@; look into a datamover call.
        memmove(target, source, bytes_to_shift);

        if (bytes_needed_from_next_buf > 0) {
            pq_movedata(pq,
                        current,
                        target + bytes_to_shift,
                        bytes_needed_from_next_buf);
        }
        current->len += bytes_needed_from_next_buf;

	/* XXX skb_push(current, shift_bytes) */
        current->data += shift_bytes;
        pq_compare_saved_data(pq);
        pq_validate(pq);
        return STATUS_SUCCESS;
    } else {
        /* Need to allocate more memory to pull up.
         * cifs_conn_alloc_one_packet() stores the allocated
         * packet in req_last_result().
         *
         * Note that in this case we try to pull up as much data as possible
         * into this new packet.  The logic here is based on the primary use
         * case, which is a CIFS Unicode string.  If this starts unaligned,
         * the whole chain of packets holding this string will unaligned.
         * Therefore we will eventually be copying all of the bytes in any
         * event.  Additionally, the CIFS code assumes that a single path
         * component will not span more than two packets, and it can be up
         * to FS_NAME_MAX_CHARS * 2 bytes long.
         */
        if (EEE_BUFSIZE_LARGE >= max_size) {
            pullup_size = max_size;
        } else {
            pullup_size = EEE_BUFSIZE_LARGE & ~(align_size - 1);
        }

        /* Pullup at least the remainder of the current packet, otherwise
         * it will require allocation of another packet.
         */
        remaining_current_pkt = (pq->pq_current->len - pq->pq_offset);

        pullup_size = ((pullup_size > remaining_current_pkt) ?
                       pullup_size : remaining_current_pkt);

        pq->pq_alloc_len = pullup_size;

	/* XXX */
        req_push_state(reqHdr, PQ_PULLUP_FINISH, pq);

        switch (req_alloc_packet(reqHdr)) {

        case STATUS_SUCCESS:
            pkt = req_last_result(reqHdr);

	    /* XXX */
            pkt->data = (char *)pkt + sizeof(struct sk_buff);
            pq_pullup_into_new_packet(pq, pkt, pullup_size);
            req_pop_state(reqHdr, STATUS_SUCCESS);
            return STATUS_SUCCESS;

        case STATUS_PENDING:
            return STATUS_PENDING;

        default:
            /* Invalid return code */
            // ASSERT(0);
            return STATUS_PENDING;
        }
    }
}
#endif /* LATER */

/*-------------------------------------------------------------------
 * Name :    pq_copyPktToBuf()
 * Description: Copies data from a pkt_desc chain into an eDesc chain
 *  
 *  * Arguments:
 *  src           - source pkt_desc chain
 *  srcOffset     - start offset in the first pkt_desc
 *  dest          - destination eDesc chain
 *  destOffset    - offset within the first eDesc
 *  num_bytes     - number of bytes to copy
 *  srcEnd        - pointer to pkt_desc where copy finished (optional arg)
 *  srcEndOffset  - offset within the srcEnd pkt_desc (optional arg)
 *  destEnd       - pointer to eDesc where copy finished (optional arg)
 *  destEndOffset - offset within the destEnd eDesc (optional arg)
 *  
 *  Return Value:
 *   None
 *-------------------------------------------------------------------
 */
void
pq_copyPktToBuf(void *src, uint32 srcOffset,
		void *dest, uint32 destOffset, uint32 num_bytes,
		void **srcEnd, uint32 *srcEndOffset,
                void **destEnd, uint32 *destEndOffset)
{
    struct sk_buff *pkt = (struct sk_buff *)src;
    eee_desc_t *eDesc = (eee_desc_t *)dest;
    eee_descBuf_t *bd;
    uint32 bytes_left, bytes_to_copy, pkt_offset, bd_offset;

    pkt_offset = srcOffset;
    bd_offset = destOffset;
    // ASSERT(EEE_GET_PKT_LEN(eDesc) >= num_bytes);

    /* for all packet descriptors */
    do {

        /* copy 1 pkt at a time */
        bytes_left = (pkt->len - pkt_offset > num_bytes) ? num_bytes : 
        pkt->len - pkt_offset;
        bytes_to_copy = bytes_left;

        do {
            bd = &eDesc->bd[1];

            /* if all bytes will fit in current eDesc */
            if (bd->len - bd_offset >= bytes_left) {
	    /* XXX */
                memcpy((void *)&bd->buf[bd_offset], &pkt->data[pkt_offset], bytes_left);
                pkt_offset += bytes_left;
                bd_offset += bytes_left;
                bytes_left = 0;
                if (srcEnd) *srcEnd = (void *)pkt;
                if (srcEndOffset) *srcEndOffset = pkt_offset;
                if (destEnd) *destEnd = (void *)eDesc;
                if (destEndOffset) *destEndOffset = bd_offset;
            } else {
	    /* XXX */
                memcpy((void *)&bd->buf[bd_offset],
		       &pkt->data[pkt_offset], 
                       bd->len - bd_offset);

                pkt_offset += (bd->len - bd_offset);
                bytes_left -= (bd->len - bd_offset);
                eDesc = eDesc->hdr.next;
                // ASSERT(eDesc);
                bd_offset = 0;
            }
        } while (bytes_left > 0);

        num_bytes -= bytes_to_copy;
        pkt = pkt->next;
        pkt_offset = 0;
    } while (num_bytes > 0);

    return;
}

/*-------------------------------------------------------------------
 * Name :    pq_copyBufPkt()
 * Description: Copies flat buffer into pkt descriptor queue OR
 *              Copies a pkt descriptor queue into a flat buffer
 *              Assumes there is sufficient data/space in pkt_desc
 * 
 * Arguments:
 * copy_type    - type of copy, COPY_TYPE_FROM_PKT or COPY_TYPE_TO_PKT
 * struct sk_buff*  - packet descriptor
 * pkt_offset   - start offset in the pkt desc.
 * buf          - flat buffer
 * num_bytes    - number of bytes to copy
 * pktEnd       - ending packet desc. (optional arg)
 * pkt_endOffset- ending packet offset (optional arg)
 * 
 * Return Value:
 *  None
 *-------------------------------------------------------------------
 */
void
pq_copyBufPkt(int copy_type,
	      struct sk_buff *pkt, uint16 pkt_offset, 
              char *buf, uint32 num_bytes, 
              struct sk_buff **pktEnd, uint16 *pkt_endOffset)
{
#ifdef LATER
    uint32 buf_offset, bytes_in_pkt, header_bytes;
    uint32 skip_data_bytes, save_num_bytes;
    boolean consumed_buf;
    int i;

    buf_offset = 0;
    // ASSERT(copy_type == COPY_TYPE_FROM_PKT || copy_type == COPY_TYPE_TO_PKT);

    do {
        if (pkt->flags & PKT_DCACHE_REF) {
            header_bytes = pkt->len - pkt->data_len;
            if (header_bytes > pkt_offset) {
                bytes_in_pkt = header_bytes - pkt_offset;
            } else {
                bytes_in_pkt = 0;
            }
        } else {
            header_bytes = pkt->len;
            bytes_in_pkt = header_bytes - pkt_offset;
        }

        if (num_bytes <= bytes_in_pkt) {
            if (copy_type == COPY_TYPE_TO_PKT) {
	    /* XXX */
                memcpy(&pkt->data[pkt_offset], &buf[buf_offset], num_bytes);
            } else if (copy_type == COPY_TYPE_FROM_PKT) {
	    /* XXX */
                memcpy(&buf[buf_offset], &pkt->data[pkt_offset], num_bytes);
            }

            if (num_bytes < bytes_in_pkt) {
                if (pktEnd != NULL) {
                    *pktEnd = pkt;
                }
                if (pkt_endOffset != NULL) {
                    *pkt_endOffset = pkt_offset + num_bytes;
                }
            } else {
                if (pktEnd != NULL) {
                    *pktEnd = pkt->next;
                }
                if (pkt_endOffset != NULL) {
                    *pkt_endOffset = 0;
                }
            }
            num_bytes = 0;
        } else {
            if (bytes_in_pkt > 0) {
                /* Just copy the header bytes in this pass, anything that comes
                 * from the data area will come on the next iteration.
                 */
                if (copy_type == COPY_TYPE_TO_PKT) {
	    /* XXX */
                    memcpy(&pkt->data[pkt_offset],
			   &buf[buf_offset],
                           bytes_in_pkt);
                } else if (copy_type == COPY_TYPE_FROM_PKT) {
	    /* XXX */
                    memcpy(&buf[buf_offset],
			   &pkt->data[pkt_offset],
                           bytes_in_pkt);
                }

                buf_offset += bytes_in_pkt;
                num_bytes -= bytes_in_pkt;
                if (!(pkt->flags & PKT_DCACHE_REF)) {
                    pkt = pkt->next;
                    pkt_offset = 0;
                } else {
                    pkt_offset += bytes_in_pkt;
                }
            } else if (pkt->flags & PKT_DCACHE_REF) {
                /* Copy from the data area.  This must be copying from a
                 * packet or we would risk corrupting the dcache buffers.
                 */
                // VERIFY(copy_type == COPY_TYPE_FROM_PKT);
                skip_data_bytes = pkt_offset - header_bytes;
                consumed_buf = FALSE;
                save_num_bytes = num_bytes;

                for (i = 0; (i < pkt->num_bufs) && (num_bytes > 0); ++i) {
                    struct edesc_buf *bd = pkt_ref_bd(pkt, i);
                    if (skip_data_bytes > bd->len) {
                        skip_data_bytes -= bd->len;
                        consumed_buf = TRUE;
                    } else {
                        int copy_this_time;
                        if (num_bytes < (bd->len - skip_data_bytes)) {
                            copy_this_time = num_bytes;
                            consumed_buf = FALSE;
                        } else {
                            copy_this_time = bd->len - skip_data_bytes;
                            consumed_buf = TRUE;
                        }
                        memcpy(&buf[buf_offset], bd->buf + skip_data_bytes,
                               copy_this_time);
                        buf_offset += copy_this_time;
                        num_bytes -= copy_this_time;
                        skip_data_bytes = 0;
                    }
                }
                if (num_bytes == 0) {
                    if (!consumed_buf) {
                        if (pktEnd != NULL) {
                            *pktEnd = pkt;
                        }
                        if (pkt_endOffset != NULL) {
                            *pkt_endOffset = pkt_offset + save_num_bytes;
                        }
                    } else {
                        if (pktEnd != NULL) {
                            *pktEnd = pkt->next;
                        }
                        if (pkt_endOffset != NULL) {
                            *pkt_endOffset = 0;
                        }
                    }
                } else {
                    pkt = pkt->next;
                    pkt_offset = 0;
                }
            } else {
                    pkt = pkt->next;
                    pkt_offset = 0;
            }
        }
    } while (num_bytes > 0);        
#endif /* LATER */
    return;
}

/*
 * Description:
 *  This routine copies data from one packet chain into another.
 * Arguments:
 *  pkt_src - The source data packet.
 *  pkt_src_offset - The starting offset within the source data packet.
 *  num_bytes - The number of bytes to copy.
 *  pkt_dest - The destination data packet.
 *  pkt_dest_offset - The starting offset within the destination data packet.
 *
 * Return Value:
 *  None.
 */
void
pq_copyPkt(struct sk_buff *pkt_src,
           uint32 pkt_src_offset,
           uint32 num_bytes, 
           struct sk_buff *pkt_dest,
           uint32 pkt_dest_offset)
{
    uint32 bytes_in_pkt_src, bytes_in_pkt_dest, j, size;

    size = 0;
    for (j = 0; j < num_bytes; j += size) {
        bytes_in_pkt_src = pkt_src->len - pkt_src_offset;
        bytes_in_pkt_dest = pkt_dest->len - pkt_dest_offset;
        size = MIN(num_bytes - j, MIN(bytes_in_pkt_src, bytes_in_pkt_dest));
	    /* XXX */
        memcpy(&pkt_dest->data[pkt_dest_offset],
               &pkt_src->data[pkt_src_offset],
               size);
        if (size == bytes_in_pkt_src) {
            pkt_src = pkt_src->next;
            while ((pkt_src != NULL) && (pkt_src->len == 0)) {
                pkt_src = pkt_src->next;
            }
            pkt_src_offset = 0;
        } else {
            pkt_src_offset += size;
        }
        if (size == bytes_in_pkt_dest) {
            pkt_dest = pkt_dest->next;
            while ((pkt_dest != NULL) && (pkt_dest->len == 0)) {
                pkt_dest = pkt_dest->next;
            }
            pkt_dest_offset = 0;
        } else {
            pkt_dest_offset += size;
        }
    }
    return;
}

/*-------------------------------------------------------------------
 * Name :    pq_copyEdescPkt()
 *
 * Description: Copies eDesc chain into a pkt descriptor queue OR
 *              Copies pkt descriptor queue into a eDesc chain
 *              Assumes that there is sufficient space in eDesc chain
 *              and pkt desc chain
 * 
 * Arguments:
 * copy_type       - type of copy, COPY_TYPE_FROM_PKT or COPY_TYPE_TO_PKT
 * pkt             - packet descriptor
 * pkt_offset      - start offset in the pkt desc.
 * eDesc           - eDesc chain
 * eDesc_offset    - start offset in eDesc chain first eDesc
 * num_bytes       - number of bytes to copy
 * pktEnd          - ending packet desc. (optional arg)
 * pkt_endOffset   - ending packet offset (optional arg)
 * eDescEnd        - ending eDesc (optional arg)
 * eDesc_endOffset - ending eDesc offset (optional arg)
 * 
 * Return Value:
 *  None
 *-------------------------------------------------------------------
 */
void
pq_copyEdescPkt(int copy_type,
	struct sk_buff *pkt,
	uint16 pkt_offset,
        eee_desc_t *eDesc,
	uint16 eDesc_offset,
	uint32 num_bytes,
        struct sk_buff **pktEnd,
	uint16 *pkt_endOffset)
{
    struct sk_buff *nextPkt = pkt;
    uint16 nextPktOffset = pkt_offset; 
    uint32 bytes_in_edesc;
    int bufidx, firstidx;

    // ASSERT(copy_type == COPY_TYPE_FROM_PKT || copy_type == COPY_TYPE_TO_PKT);

    bufidx = firstidx = edesc_firstBufferIdx(eDesc);            

    /* copy 1 eDesc to/from multiple pkts */                
    do {
        bytes_in_edesc = eDesc->bd[bufidx].len - eDesc_offset;

        if (num_bytes <= bytes_in_edesc) {
            pq_copyBufPkt(copy_type, nextPkt, nextPktOffset, 
                          &eDesc->bd[bufidx].buf[eDesc_offset], num_bytes, 
                          pktEnd, pkt_endOffset);
            num_bytes = 0;
        } else {
            pq_copyBufPkt(copy_type, nextPkt, nextPktOffset, 
                          &eDesc->bd[bufidx].buf[eDesc_offset],
                          bytes_in_edesc, 
                          &nextPkt, &nextPktOffset);
            num_bytes -= bytes_in_edesc;
            if (++bufidx >= firstidx + EEE_GET_NUM_BUFS(eDesc)) {
                eDesc = eDesc->hdr.next;
                bufidx = firstidx = edesc_firstBufferIdx(eDesc);
            }
            eDesc_offset = 0;
        }
    } while (num_bytes > 0);
}

/*-------------------------------------------------------------------
 * Name :    pq_parseAsciiString()
 *
 * Description: Parses a packet looking for an NULL Terminated ASCII string
 *              Returns the string in the destination.
 *
 *  * Arguments:
 *   pkt           - source packet
 *   pkt_offset    - starting offset in the packet
 *   destn         - destination string
 *   destn_size    - maximum size of destination (including NULL terminator)
 *   pktEnd        - packet containing next data byte (optional arg)
 *   pkt_endOffset - start of next data byte (optonal arg)
 *
 *  * Return Value:
 *   NFX_OK              - if string was successfully parsed from packet
 *   PQ_SOURCE_UNDERFLOW - if the source got exhausted and NULL terminator was
 *                         not found
 *   PQ_DESTN_OVERFLOW   - if the destination was too small
 *-------------------------------------------------------------------
 */
int32
pq_parseAsciiString(struct sk_buff *pkt,
	uint16 pkt_offset,
        char *destn,
	uint16 *destn_size,
        struct sk_buff **pktEnd,
	uint16 *pkt_endOffset)
{
    uint32  max_size = *destn_size;
    uint32  strLen = 0;
    char    *data;
    uint16  data_offset;
    boolean found_null = FALSE;
    uchar8 next_char;

    while (pkt && max_size && !found_null) {
        data_offset = 0;
	    /* XXX */
        for (data = &pkt->data[pkt_offset];
            pkt_offset < pkt->len && !found_null && max_size;
            ++pkt_offset, --max_size) {

            next_char = data[data_offset++];
            destn[strLen++] = next_char;
            found_null = next_char == '\0';
        }

        /* if end of packet, goto next packet */
        if (pkt_offset == pkt->len) {
            pkt = pkt->next;
            pkt_offset = 0;
        }
    }

    /* set output params */
    *destn_size = strLen;
    if (pktEnd) *pktEnd = pkt;
    if (pkt_endOffset) *pkt_endOffset = pkt_offset;

    /* error in parsing ? */
    if (!found_null) {
        if (pkt)
            return PQ_DESTN_OVERFLOW;
        else
            return PQ_SOURCE_UNDERFLOW;
    } else { /* string parsed successfully */
        return NFX_OK;
    }
}

/*
 * Description:
 *  Validate the queue fields.
 *
 * Arguments:
 *  pq - packet queue
 *
 * Return Value:
 *  None.
 */
void
pq_validate(pkt_queue_t *pq)
{
    struct sk_buff *pkt;
    uint32 totalLen, calc_start_offset;

#ifdef LATER
    // ASSERT(pq->pq_start_offset <= pq->pq_len);

    if (pq->pq_start_offset == pq->pq_len) {
        // ASSERT(pq->pq_current == NULL);
    }
    pkt = pq->pq_head;
    if (pkt == NULL) {
        // ASSERT(pq->pq_tail == (struct sk_buff *)&pq->pq_head);
    }
#else
    pkt = pq->pq_head;
#endif /* LATER */

    totalLen = 0;
    while (pkt != NULL) {
        // ASSERT(pkt_validate(pkt));
        // ASSERT(pkt->len != 0);
        totalLen += pkt->len;
        if (pkt->next == NULL) {
            // ASSERT(pkt == pq->pq_tail);
        }
        pkt = pkt->next;
    }

    // ASSERT(totalLen == pq->pq_len);
    if (pq->pq_start_offset < pq->pq_len) {

        // ASSERT(pq->pq_current != NULL);
        // ASSERT(pq->pq_offset < pq->pq_current->len);

        /* If there is a current packet, make sure it is one of the
         * queue packets.
         */
        calc_start_offset = 0;
        pkt = pq->pq_head;

        while (pkt != NULL && pkt != pq->pq_current) {
            calc_start_offset += pkt->len;
            pkt = pkt->next;
        }
        // ASSERT(pkt != NULL);
        // ASSERT(calc_start_offset + pq->pq_offset == pq->pq_start_offset);
    }
}

#if defined(BASIC_STATS)
/* @@@ move somewhere more appropriate
 */
    #define UPDATE_STAT_INC(stat) ++(stat)
    #define UPDATE_MAXSTAT_INC(stat, maxstat) \
    { if (++(stat) > maxstat) maxstat = stat; }
    #define UPDATE_STAT_DEC(stat) --(stat)
    #define UPDATE_STAT_MINUS(stat, dec) ((stat) -= dec)
    #define UPDATE_STAT_PLUS(stat, plus) ((stat) += plus)
    #define UPDATE_MAXSTAT_PLUS(stat, plus, max) \
    { if (((stat) += (plus) > maxstat) maxstat = stat; }
    #define UPDATE_MAXSTAT(stat, max) \
    { if ((stat) > (maxstat)) maxstat = stat; }
#else
    #define UPDATE_STAT_INC(stat)
    #define UPDATE_MAXSTAT_INC(stat, maxstat)
    #define UPDATE_STAT_DEC(stat)
    #define UPDATE_STAT_MINUS(stat, dec)
    #define UPDATE_STAT_PLUS(stat, plus)
    #define UPDATE_MAXSTAT_PLUS(stat, plus, max)
    #define UPDATE_MAXSTAT(stat, maxstat) 
#endif

#ifdef LATER

/*
 * Description:
 *  Give pkt to a request waiting for allocation.
 * Arguments:
 * free_list - packet free list
 * reqHdr - request
 * pkt - packet descriptor
 *
 * Return Value:
 * None.
 */
void
pkt_give_to_req(struct pkt_free_list *free_list, req_hdr_t *reqHdr, struct sk_buff *pkt)
{

    UPDATE_STAT_INC(free_list->pf_calls);

#ifdef REQ_VALIDATE_PACKET_COUNTS
    if (! CM_IS_TXRX_NCPU) {
        reqHdr->rh_packets_allocated++;
    }
#endif

    int32 alloc_type = reqHdr->rh_alloc_type;

    switch (reqHdr->rh_state) {

    case CIFS_REQ_ALLOC_PACKETS_CONTINUE:
    case NFS_REQ_ALLOC_PACKETS_CONTINUE:
    case RPC_ALLOC_PACKETS_CONTINUE:
        {
            /* The allocation is done into a packet queue and desired
             * queue len is stored in pq_alloc_len field.
             */
            pkt_queue_t *pq = req_context(reqHdr);
            uint32 can_use = pkt_usable_len(reqHdr->rh_conn,
                                            pq_is_empty(pq));
            int remain = pq->pq_alloc_len - pq->pq_len;
            pkt_setHdrSize(pkt, remain < can_use ? remain : can_use);
            
            pq_add_packet(pq, pkt);

	    /* XXX */
            pkt->data = (char *)pkt + PKT_BUFF_INDENT;

            /* If all the memory was allocated, remove the request from
             * the wait queue and wake it up
             */
            if (pq->pq_alloc_len <= pq->pq_len) {
                SIMPLEQ_REMOVE_HEAD(&free_list->pf_waiters, reqHdr, rh_wait_entry);
                reqHdr->rh_error_code = STATUS_SUCCESS;
                req_wakeup(reqHdr, REQ_PRIORITY_WAKEUP);
            }
            break;
        }
    case REQ_WAIT_ONE_PACKET:
        /* Can not align the packet in this case since this depends on the 
         * position of the packet in the queue.  The caller must do it itself 
         * if it matters.  
         */
        reqHdr->rh_error_code = STATUS_SUCCESS;
        req_set_result(reqHdr, pkt);
        SIMPLEQ_REMOVE_HEAD(&free_list->pf_waiters, reqHdr, rh_wait_entry);
        req_wakeup(reqHdr, REQ_PRIORITY_WAKEUP);
        break;

    case REQ_WAIT_N_PACKETS:
        {
            pkt_queue_t *pq = req_context(reqHdr);
            pkt->len = 0;
            pq->pq_tail->next = pkt;
            pq->pq_tail = pkt;
            
	    /* XXX */
            pkt->data = (char *)pkt + PKT_BUFF_INDENT;

            if (--pq->pq_alloc_len == 0) {
                SIMPLEQ_REMOVE_HEAD(&free_list->pf_waiters, reqHdr, rh_wait_entry);
                reqHdr->rh_error_code = STATUS_SUCCESS;
                req_wakeup(reqHdr, REQ_PRIORITY_WAKEUP);
            }
            break;
        }

    default:
        panic("request is waiting for packet memory in invalid state\n");
        return;
    }
    if ((alloc_type <= 0) || (alloc_type >= PKT_ALLOC_TYPE_MAX)) {
        panic("pkt_give_to_req: invalid allocation type\n");
    }
    pkt_alloc_stats[alloc_type]++;
    pkt->alloc_type = alloc_type;
    return;
}

/*
 * Description:
 * Allocate EEE descriptor and buffer and initialize packet descriptor.
 *
 * Arguments:
 *  pkt - packet descriptor
 *
 * Return Value:
 *  None.
 */
void
pkt_init_new(struct sk_buff *pkt)
{
    pkt->flags = (PKT_DESC_INUSE | PKT_FAST_PATH | PKT_ALLOCATED_ON_ACPU | PKT_DATA);
    pkt->data = (char *)pkt + PKT_BUFF_INDENT;
    pkt->pTplNext = NULL;
    pkt->pTplConn = NULL;
    pkt->pktSize = 0;
    pkt->len = 0;
    pkt->alloc_type = PKT_ALLOC_TYPE_UNKNOWN;
    return;
}

/*
 * Description:
 *  Initialize the packet after allocation.
 * Arguments:
 *  pkt - packet
 * Return Value:
 *  None.
 */
void
pkt_init(struct sk_buff *pkt)
{
    pkt->flags = (PKT_FAST_PATH | PKT_ALLOCATED_ON_ACPU | PKT_DATA | PKT_DESC_INUSE);
    pkt->next = NULL;
}

/*
 * Description:
 *  Free the single packet to the ACPU free packet list.
 *
 * Arguments:
 *  pkt - packet descriptor
 *
 * Return Value:
 *  None.
 */
void
pkt_acpu_free(struct sk_buff *pkt)
{
    // ASSERT(pkt_validate(pkt));
    // ASSERT((pkt->flags & PKT_ALLOCATED_ON_ACPU));
    // ASSERT(! CM_IS_TXRX_NCPU);

#ifdef REQ_VALIDATE_PACKET_COUNTS
    if (creq.v != NULL) {
        creq.v->rh_packets_freed++;
    }
#endif
    int32 alloc_type = pkt->alloc_type;

    pkt->pktSize = 0;
    pkt->data = (char *)pkt + PKT_BUFF_INDENT;
    pkt->p_req = NULL;

    if ((alloc_type <= 0) || (alloc_type >= PKT_ALLOC_TYPE_MAX)) {
        panic("%s: invalid allocation type\n", __FUNCTION__);
    }
    pkt_alloc_stats[alloc_type]--;

    if (SIMPLEQ_EMPTY(&pkt_acpu_free_list.pf_waiters)) {
        pkt->flags = (PKT_FAST_PATH | PKT_ALLOCATED_ON_ACPU | PKT_DATA);
        pkt_free_to_list(&pkt_acpu_free_list, pkt);
    } else {
        pkt_init(pkt);
        pkt_give_to_req(&pkt_acpu_free_list,
                        SIMPLEQ_FIRST(&pkt_acpu_free_list.pf_waiters),
                        pkt);
    }
}
#endif /* LATER */

/*
 * Description:
 *  Initialize packet free list.
 *
 * Arguments:
 *  free_list - free list to initialize
 *  num_element - number of elements in the list
 *
 * Return Value:
 *  None.
 */
void
pkt_init_free_list(struct pkt_free_list *free_list, int num_elements unused__)
{
    SIMPLEQ_INIT(&free_list->pf_waiters);

#ifdef PKT_ALLOC_USE_TAILQ
    free_list->pf_head = NULL;
    free_list->pf_tail = &free_list->pf_head;
#else    
    free_list->pf_head = NULL;
#endif    

#if defined(BASIC_STATS)    
    free_list->pf_inuse = num_elements;
    free_list->pf_calls = 0;
    free_list->pf_maxinuse = 0;
    free_list->pf_no_memory = 0;
#endif    
}

#ifdef LATER
/*
 * Description:
 *  Initialize ACPU packet allocator.
 *
 * Arguments:
 *  None.
 *
 * Return Value:
 *  None.
 */
void
pkt_acpu_alloc_init(void)
{
    int i;

    pkt_init_free_list(&pkt_acpu_free_list, PKT_ACPU_LIMIT);

    pkt_alloc_stats[PKT_ALLOC_TYPE_UNKNOWN] = PKT_ACPU_LIMIT;

    for (i = 0; i < PKT_ACPU_LIMIT; ++i) {
        void *buf = eee_alloc_buf(&local_buf_pool[EEE_BUF_LARGE]);
        if (buf != NULL) {
            pkt_init_new(buf);
            pkt_acpu_free(buf);
        } else { 
            panic("Can not allocate ACPU packet buffers\n");
        }
    }
}

/*
 * Description:
 *  Wait until the memory in the packet free list becomes available.
 *  The request state should be either CIFS_REQ_ALLOC_PACKETS_CONTINUE if the
 *  request is allocating into a packet queue or REQ_WAIT_ONE_PACKET if the
 *  request wants to allocate just one packet.
 *
 * Arguments:
 *  req - request
 *
 * Return Value:
 *  Returns STATUS_PENDING.
 */
NTSTATUS
pkt_acpu_wait(any_request_ptr_t req)
{
    // ASSERT((req.v->rh_state == NFS_REQ_ALLOC_PACKETS_CONTINUE) ||
           (req.v->rh_state == CIFS_REQ_ALLOC_PACKETS_CONTINUE) ||
           (req.v->rh_state == RPC_ALLOC_PACKETS_CONTINUE) ||
           (req.v->rh_state == REQ_WAIT_ONE_PACKET) ||
           (req.v->rh_state == REQ_WAIT_N_PACKETS));
    return pkt_acpu_wait_on_list(req, &pkt_acpu_free_list);
}

/*
 * Description:
    Allocate a packet for a request. The pointer to allocated packet is stored
    in the request structure and can be accessed by calling req_last_result().
    This is a blocking call. The caller needs to setup the request
    state before calling this function.

 * Arguments:
    req - request

 * Return Value:
    STATUS_SUCCESS - the packet was allocated
    STATUS_PENDING - the request was blocked for memory allocation
 */
NTSTATUS
req_alloc_packet(any_request_ptr_t req)
{
    struct sk_buff *pkt = pkt_acpu_alloc(req.v->rh_alloc_type);
    if (pkt != NULL) {
        req_set_result(req.v, pkt);
        return STATUS_SUCCESS;
    } else {
        req_push_state(req, REQ_WAIT_ONE_PACKET, NULL);
        return pkt_acpu_wait(req);
    }
}

/*
 * Description:
    Continuation function for req_alloc_packet. It is called when the packet
    allocation for the request is completed.

 * Arguments:
    req - request

 * Return Value:
    Returns STATUS_PENDING.
 */
NTSTATUS
req_alloc_packet_continue(cifs_request_t *req)
{
    CALL_NEXT(req, STATUS_SUCCESS);
}

/*
 * Description:
 *  Free a chain of packets linked with next field.
 *
 * Arguments:
 *  pkt - start of the packet chain
 *
 * Return Value:
 *  None.
 */
void
pkt_acpu_free_chain(struct sk_buff *pkt)
{
    struct sk_buff *next;

    while (pkt != NULL) {
        next = pkt->next;
        pkt->next = NULL;
        pkt_free(pkt);
        pkt = next;
    }
}

/*
 * Description:
 *  Validate the packet queue after the allocation done by pq_alloc_packets().
 *  The allocated packets should match the request's connection MTU and the
 *  first packet should have the space reserved according to the protocol.
 *
 * Arguments:
 *  req - request
 *  pq - packet queue
 *
 * Return Value:
 *  None.
 */
void
pq_validate_after_alloc(any_request_ptr_t req, pkt_queue_t *pq)
{
    // ASSERT(pkt_usable_data_len(req.v->rh_conn, TRUE) >= pq->pq_head->len);
    struct sk_buff *pkt;
    for (pkt = pq->pq_head->next; pkt != NULL; pkt = pkt->next) {
        // ASSERT(pkt_usable_data_len(req.v->rh_conn, FALSE) >= pkt->len);
    }
}

/*
 * Description:
 *  Allocate packets for the request output queue.
 *
 * Arguments:
 *  req - request
 *  pq - packet queue to do allocations for
 *  need_bytes - desired queue length
 *  wait_state - state to wait for memory allocation
 *
 * Return Value:
 *  STATUS_SUCCESS - allocated successfully
 *  STATUS_PENDING - request was queued, return to dispatcher.
 */
NTSTATUS
pq_alloc_packets(any_request_ptr_t req,
                 pkt_queue_t *pq,
                 uint32 need_bytes,
                 req_state_t wait_state)
{
    // ASSERT(need_bytes >= pq->pq_len);

    req_hdr_t *reqHdr = req.v;
    int remain = need_bytes - pq->pq_len;
    boolean is_first;

    if (! pq_is_empty(pq)) {
        is_first = pq_is_first_packet(pq);
        int usable_len = pkt_usable_len(reqHdr->rh_conn, is_first);
        if ((usable_len - pq->pq_tail->len) >= remain) {

            /* There is sufficient space in the last allocated packet.
             */

            pq->pq_tail->len += remain;
            pq->pq_len = need_bytes;
            if (pq->pq_current == NULL) {
                pq->pq_current = pq->pq_tail;
                pq->pq_offset = pq->pq_tail->len - remain;
            }

            pq_validate(pq);
            return STATUS_SUCCESS;
        } else {

            /* We are going to allocate a new packet so the new response
             * structure is contiguous in memory.
             */

            is_first = FALSE;
        }
    } else {

        /* The queue is empty, allocate the first packet.
         */

        is_first = TRUE;
    }

    while (remain > 0) {
        uint32 usable_len = pkt_usable_len(reqHdr->rh_conn, is_first);
        uint32 alloc_length = ((remain < usable_len) ? remain : usable_len);
        struct sk_buff *packet = pkt_acpu_alloc(req.v->rh_alloc_type);

        if (packet) {
            pkt_setHdrSize(packet, alloc_length);
            packet->data = (char *)packet + PKT_BUFF_INDENT;
            pq_add_packet(pq, packet);

            remain -= alloc_length;
            is_first = FALSE;
        } else {
            if (!(pq->pq_flags & PQ_ALLOC_DONT_WAIT)) {

                /* Setup the request context to point to the packet queue
                 * that needs memory allocation. The pq_alloc_len field
                 * specifies how many bytes need to be in the queue when
                 * the memory allocation completes.
                 */

                pq->pq_alloc_len = need_bytes;
                req_push_state(reqHdr, wait_state, pq);
                return pkt_acpu_wait(reqHdr);
            } else {

                /* The caller does not want to wait for the memory to
                 * become available.
                 */

                return STATUS_INSUFF_SERVER_RESOURCES;
            }
        }
    }

    pq_validate(pq);

    pq_validate_after_alloc(req, pq);

    return STATUS_SUCCESS;
}
#endif /* LATER */

/*
 * Description:
 *  Discard unused data at end of queue, updating current pointers if necessary.
 * Arguments:
 *  pq - packet queue
 *  num_bytes - number of bytes to discard
 *
 * Return Value:
 *  None.
 */
void
pq_discard(pkt_queue_t *pq, uint32 num_bytes)
{
    pq_validate(pq);
    // ASSERT(pq->pq_len >= num_bytes);

    if (pq->pq_tail->len > num_bytes) {

        /* The bytes to discard are completely in the last queue packet.
         */

        pq->pq_tail->len -= num_bytes;
    } else if (pq->pq_len == num_bytes) {

        /* Discard all bytes in the queue.
         */

        pkt_acpu_free_chain(pq->pq_head);
        pq_make_empty(pq);
    } else {
        uint32 desired_len = pq->pq_len - num_bytes;

        /* Find the packet containing the last remaining byte
         */

        struct sk_buff *pkt;
        uint32 end_offset;

        if ((pq->pq_current == NULL) ||
            (desired_len <= (pq->pq_start_offset - pq->pq_offset))) {

            /* Start iteration from the beginning of the queue
             */

            pkt = pq->pq_head;
            end_offset = pkt->len;
        } else {

            /* Skip to the current packet
             */

            pkt = pq->pq_current;
            end_offset = pq->pq_start_offset - pq->pq_offset + pkt->len;
        }

        while (end_offset < desired_len) {
            pkt = pkt->next;
            end_offset += pkt->len;
        }

        // ASSERT(pkt->len > (end_offset - desired_len));

        pq->pq_tail = pkt;
        pkt_acpu_free_chain(pkt->next);
        pkt->next = NULL;
        pkt->len -= (end_offset - desired_len);
    }

    pq->pq_len -= num_bytes;

    if (pq->pq_start_offset >= pq->pq_len) {
        pq->pq_current = NULL;
        pq->pq_start_offset = pq->pq_len;
    }

    pq_validate(pq);
}

uint32
pq_random(void)
{
return 0;
}

#ifdef LATER
/*
 * Description:
 * Intentionally misalign and split a packet at a random offset.
 *
 * Arguments:
 *  pq - packet queue
 *
 * Return Value:
 *  None.
 */
#define EEE_BUF_LARGE_XXX 2048

void
pq_split_at_random_offset(pkt_queue_t *pq)
{
    struct sk_buff *pkt;
    uint32 saveOff, num_bytes, split_pt;

    // pkt = pkt_acpu_alloc(PKT_ALLOC_TYPE_UNKNOWN);
    pkt = alloc_skb(EEE_BUF_LARGE_XXX, (__GFP_HIGHMEM | __GFP_NOFAIL | __GFP_DMA));

    if (pkt) {

        saveOff = pq->pq_start_offset;
        num_bytes = pq->pq_len - saveOff;
        // ASSERT(num_bytes <= EEE_BUFSIZE_LARGE);
        split_pt = 1 + (pq_random() % (num_bytes - 2));

        pq_adjust(pq, split_pt);
        if ((pq_random() % 0x1) == 0) {
            // misalign this packet
            if (((void *)pkt->data & (void *)0x1) == 0) {
                pkt->data++;
            }
        }
        pkt->data = (char *)pkt + sizeof(struct sk_buff);

        pq_pullup_into_new_packet(pq, pkt, num_bytes - split_pt);
        pq_seek_to(pq, saveOff);
    }
    return;
}

/*
 * Description:
 *  Initialize the dcache reference structure at the end of the packet.
 *    
 * Arguments:
 *  pkt - packet descriptor
 *  dc - referenced dcache
 *  add_ref - if true, add reference to the dcache
 *
 * Return Value:
 *  None.
 */
void
pkt_init_ref(struct sk_buff *pkt, dcache_t *dc, boolean add_ref)
{
    pkt->flags |= PKT_DCACHE_REF;
    pkt->dcache = dc;
    // ASSERT(dc->dc_refCnt > 0);
    if (add_ref) {
    	dcache_setBusy(dc, pkt, ref_get_typeid(pkt));
    }
}

/*
 * Description:
 *  Add the reference to a dcache buffer to the packet.
 *    
 * Arguments:
 *  pkt - packet descriptor
 *  buf - pointer to the data in the buffer
 *  len - the length of the referenced data
 *  dc - the referenced dcache
 *  add_ref - if true, add reference to the dcache
 *    
 * Return Value:
 *  None.
 */
void
pkt_add_buffer(struct sk_buff *pkt, void *buf, uint32 len, dcache_t *dc, boolean add_ref)
{
    struct edesc_buf *bd;
    uint32 data_len, hdr_size;
    int num_bufs; 

    // ASSERT(pkt_validate(pkt));

    hdr_size = pkt->len;
    if (!(pkt->flags & PKT_DCACHE_REF)) {
        num_bufs = 0;
        data_len = 0;
        pkt_init_ref(pkt, dc, add_ref);
    } else {
        num_bufs = pkt->num_bufs;
        data_len = pkt->data_len;
    }

    bd = pkt_ref_bd(pkt, num_bufs);

    /* Check if we run out of unused space in the packet.
     */
    // ASSERT((void *)bd > (void *)pkt->data + (pkt->len - pkt->data_len));

    pkt->num_bufs = num_bufs + 1;
    pkt->data_len = data_len + len;
    bd->buf = buf;
    bd->len = len;
    pkt->len = hdr_size + len;

    // ASSERT(pkt_validate(pkt));
}

/*
 * Description:
 *  Validate the packet.
 *    
 * Arguments:
 *  pkt - packet descriptor
 *    
 * Return Value:
 *  Returns true if the packet is valid.
 */
boolean
pkt_validate(struct sk_buff *pkt)
{
    address_t ra unused__ = get_ra();
    if ((eee.buf_base[EEE_SHARED_POOL][EEE_BUF_LARGE] <= (void *)pkt) &&
        ((void *)pkt < eee.buf_end[EEE_SHARED_POOL][EEE_BUF_LARGE])) {
        // ASSERT((pkt->flags & PKT_DESC_INUSE) && 
               (pkt->flags & PKT_DATA));
        int pkt_header_len = pkt->data - (char *)pkt;
        int buf_usable_len unused__ = PKT_USABLE_SIZE - pkt_header_len;
        if ((pkt->flags & PKT_DCACHE_REF)) {
            // ASSERT(buf_usable_len >= ((pkt->len - pkt->data_len) +
                                      (sizeof(struct edesc_buf) * pkt->num_bufs)));
            // ASSERT((pkt->num_bufs == 0) || (pkt->data_len > 0));
            // ASSERT(pkt->len >= pkt->data_len);
            if (pkt->num_bufs != 0) {
                
                /* If there are additional referenced buffers, 
                 * the inline data must end on the cache boundary.
                 */

                // ASSERT(CACHE_ALIGNED(pkt->data + 
                                     pkt->len - 
                                     pkt->data_len));
            }
            int i;
            uint32 len_sum = 0;

            for (i = 0; i < pkt->num_bufs; ++i) {
                struct edesc_buf *bd = pkt_ref_bd(pkt, i);
                int checked_len;

                /* All buffers but the first must be cache aligned.  If the 
                 * first buffer is not aligned, we can use the "offset into 
                 * second buffer" feature of sibyte dma descriptors.  In this 
                 * case the length of the inline packet data must be cache 
                 * aligned, but we can not check this until the packet is 
                 * fully constructed.  This is going to be verified in the 
                 * network driver.  
                 */

                // ASSERT((i == 0) || CACHE_ALIGNED(bd->buf));

                /* All buffers but the last must end on the cache boundary.  
                 */

                // ASSERT((i == (pkt->num_bufs - 1)) ||
                       CACHE_ALIGNED(bd->buf + bd->len));

                /*
                 * Discount possible NFS padding for the last buffer.
                 */
                checked_len = ((i == (pkt->num_bufs - 1)) && (bd->len > 3) ?
                               (bd->len - 3) : bd->len);
                // ASSERT(eee_buf_valid(bd->buf, checked_len, ra));
                len_sum += bd->len;
            }
            // ASSERT(pkt->dcache->dc_refCnt != 0);
            // ASSERT(pkt->dcache->dc_wbBusyCnt != 0);
            // ASSERT(len_sum == pkt->data_len);
        } else if (pkt->flags & PKT_INPUT) {
            // ASSERT(pkt->len >= 0);
            // ASSERT(eee_buf_valid(pkt->data, pkt->len, ra));
        }
    }
    return TRUE;
}

/*
 * Description:
 *  Validate the packet and it's next pointers.
 *    
 * Arguments:
 *  pkt_head - packet descriptor
 *    
 * Return Value:
 *  Returns true if the packet is valid.
 */
boolean
pkt_validate_chain(struct sk_buff *pkt_head)
{
    struct sk_buff *pkt = pkt_head;

    while (pkt != NULL) {
        // ASSERT(pkt_validate(pkt));
        pkt = pkt->next;
    }
    return TRUE;
}

/*
 * Description:
 *  Verify that the data referenced is the same as the original and
 *  that the packets conform to the sibyte restrictions.
 *    
 * Arguments:
 *  req - current request
 *  ctx - copy context
 *  already_copied - the number of bytes that should be valid
 *
 * Return Value:
 *  Returns TRUE if validation succeeds or panics.
 */
boolean
req_copy_dcaches_verify(any_request_ptr_t req,
        struct req_copy_dcaches_ctx *ctx,
        uint32 already_copied)
{
    pkt_queue_t *pq;
    struct sk_buff *start_pkt, *it;
    eee_desc_t *edesc;
    struct edesc_buf *bd;

    uint32 pq_offset, start_offset, save_offset, len_verified;
    uint32 cod_offset, dcache_len;
    uint32 ref_data_len, num_bufs, bd_offset;
    int32 verified_this_packet, remain, pkt_offset;

    int dc_idx, i, data_offset, hdr_size;
    dcache_t *dc;
    nfx_cod_t cod;
    char *dc_data;
    boolean is_first;

    pq = ctx->nr_pq;

    // ASSERT(already_copied <= ctx->nr_read_count);

    if (already_copied == 0) {
        return TRUE;
    }

    pq_offset = 0;
    start_offset = pq->pq_start_offset;
    start_pkt = pq->pq_head;
    save_offset = 0;

    while (start_pkt->len + pq_offset <= start_offset) {
        pq_offset += start_pkt->len;
        start_pkt = start_pkt->next;
    }
    save_offset = start_offset - pq_offset;

    len_verified = 0;

    for (dc_idx = 0; ((dc_idx < ctx->nr_read_num_dcache_entries) &&
                      (len_verified < already_copied)); ++dc_idx) {

        dc = ctx->nr_read_dcache_entries[dc_idx];
        cod_offset = ((dc_idx == 0) ? 
                             (ctx->nr_read_offset - (dc->dc_block * EFS_BLOCK_SIZE)) : 0);
        dcache_len = ((dc_idx < (ctx->nr_read_num_dcache_entries - 1)) ?
                             ((dc->dc_nBlocks * EFS_BLOCK_SIZE) - cod_offset) :
                             MIN(dc->dc_nBlocks * EFS_BLOCK_SIZE,
                                 already_copied - len_verified));
        cod = dc->dc_cod;
        edesc = sbm_getDescFromCOD(cod);
        dc_data = eee_ramAlloc(dcache_len);

        if (dc_data != NULL) {
            int rc = eee_copyPacketData(dc_data,
                                        edesc,
                                        cod_offset,
                                        dcache_len,
                                        FALSE);
            // ASSERT(rc == dcache_len);

            buf = dc_data;
            remain = ((dcache_len > (already_copied - len_verified)) ?
                            (already_copied - len_verified) : dcache_len);
            pkt_offset = save_offset;
            is_first = (start_pkt == pq->pq_head);

            for (it = start_pkt; remain > 0; 
                  remain -= it->len - pkt_offset,
                  it = it->next, 
                  pkt_offset = 0,
                  is_first = FALSE) {

                // ASSERT(pkt_validate(it));

                ref_data_len = ((it->flags & PKT_DCACHE_REF) ? it->data_len : 0);
                num_bufs = ((it->flags & PKT_DCACHE_REF) ? it->num_bufs : 0);
                verified_this_packet = 0;
                i = 0;
                bd_offset = 0;

                if (pkt_offset >= it->len - ref_data_len) {
		    data_offset = pkt_offset - (it->len - ref_data_len);

                    for (; (i < num_bufs); ++i) {

                        bd = pkt_ref_bd(it, i);
                        if (data_offset < bd->len) {
                            bd_offset = data_offset;
                            break;
                        } else {
                            data_offset -= bd->len;
                        }
                    }
                } else {
                    hdr_size = (it->len - pkt_offset - ref_data_len);
                    if (hdr_size != 0) {
                        // ASSERT(memcmp(buf, 
                                      it->data + pkt_offset, 
                                      MIN(hdr_size, remain - verified_this_packet)) == 0);
                        verified_this_packet += hdr_size;
                        buf += hdr_size;
                    }
                }

                for (; (i < num_bufs) && (verified_this_packet < remain); ++i, bd_offset = 0) {
                    bd = pkt_ref_bd(it, i);
                    
                    // ASSERT(memcmp(buf, 
                                  bd->buf + bd_offset, 
                                  MIN(remain - verified_this_packet,
                                      bd->len - bd_offset)) == 0);

                    buf += bd->len - bd_offset;
                    verified_this_packet += bd->len - bd_offset;
                }
            }
            eee_ramDealloc(dc_data);
        }
        len_verified += dcache_len;

        if (len_verified < already_copied) {
            start_offset += dcache_len;

            while (start_pkt->len + pq_offset <= start_offset) {
                pq_offset += start_pkt->len;
                start_pkt = start_pkt->next;
            }
            save_offset = start_offset - pq_offset;
        }
    }
    return TRUE;
}

/*
 * Description:
 *  Continuation function for req_copy_dcaches_proc() called when the packet
 *  allocation is complete.
 *    
 * Arguments:
 *  req - request.
 *    
 * Return Value:
 *  Returns STATUS_PENDING.
 */
NTSTATUS
req_copy_dcaches_cont(any_request_ptr_t req)
{
    struct req_copy_dcaches_ctx *ctx = req_context(req.v);
    req_pop_state(req.v, STATUS_SUCCESS);
    ctx->nr_pq->pq_tail = ctx->nr_pq_tail;
    ctx->nr_read_dcache_index = ctx->nr_read_dcache_index_save;

    if (req_copy_dcaches_proc(req, ctx) == STATUS_SUCCESS) {
        req_run(req);
    }
    return STATUS_PENDING;
}

/*
 * Description:
 * If the data in the packet is not 8 bytes aligned, split the last buffer in 
 * the packet so it ends cache aligned and some data from the next packet can 
 * be moved into this packet. The data to the right from the split point is
 * copied into the packet and a reference is added.
 *   
 * Arguments:
 *  pkt - packet
 *  copy_this_time - number of bytes to copy from the next buffer
 *
 * Return Value:
 *  Returns pointer past the end of the copied data or NULL if no alignment required. 
 */
void *
req_align_last_buffer(struct sk_buff *pkt, uint32 *copy_this_time)
{
    struct edesc_buf *last_bd, *bd;
    uint32 unaligned;
    void *data_end;
    int i;

    if ((pkt->len % 8) != 0) {

        *copy_this_time = MIN(*copy_this_time, GET_PAD(pkt->len, 8));

        if ((pkt->flags & PKT_DCACHE_REF)) {
            last_bd = pkt_ref_bd(pkt, pkt->num_bufs - 1);
            unaligned = MIN((uintptr_t)(last_bd->buf + last_bd->len) % CACHELINE_SIZE,
                    last_bd->len);

            data_end = alignptr(pkt->data + (pkt->len - pkt->data_len), CACHELINE_SIZE);

            /* In case we have copied more than one buffer, find where the 
             * data copied so far ends. It ends after the last buffer
             * with a reference inside the packet.
             */
            int i;
            for (i = 0; i < pkt->num_bufs; ++i) {
                bd = pkt_ref_bd(pkt, i);

                if (((pkt->data <= (char *)bd->buf) && 
                     ((char *)bd->buf < pkt->data + EEE_BUFSIZE_LARGE))) {
                    data_end = alignptr(bd->buf + bd->len, CACHELINE_SIZE);
                }
            }

            if (unaligned != 0) {
                memcpy(data_end, last_bd->buf + last_bd->len - unaligned, unaligned);
                last_bd->len -= unaligned;
                pkt->data_len -= unaligned;
                pkt->len -= unaligned;
                if (last_bd->len == 0) {
                    pkt->num_bufs--;
                }
            }
            pkt_add_buffer(pkt, data_end, unaligned + *copy_this_time, NULL, FALSE);

            /* The caller will adjust the packet length again, compensate for this.
             */
            pkt->len -= *copy_this_time;
            return data_end + unaligned;
        } else {
            /* There are no buffers present, copy the data to satisfy 8 byte alignment
             * at the end of the current packet data.
             */
            return pkt->data + pkt->len;
        }
    } else {
        /* No alignement is necessary.
         */
        return NULL;
    }
}

/*
 * Description:
 *  Free the dcache reference that the packet may be has.
 *        
 * Arguments:
 *  pkt - packet
 *
 * Return Value:
 *  None.
 */
void
pkt_free_dcache_ref(struct sk_buff *pkt)
{
    if (pkt->flags & PKT_DCACHE_REF) {
        dcache_decBusy(pkt->dcache, pkt);
        pkt->flags &= ~PKT_DCACHE_REF;
    }
}

/*
 * Description:
 *
 * Compact the referenced dcache entries, ensuring that all dcache entries 
 * consist of buffers that are cache aligned and have cache aligned length.  
 *
 * The buffers of this configuration can be transmitted utilizing full MTU 
 * with minimal memory copies, maximum cache line per packet; The algorithm is:
 *
 * In case we start at some offset in the first buffer in the range, we use 
 * "offset to second buffer" feature of BMC12500, copy some number of bytes 
 * from the buffer into the packet so the data is cache aligned.  The next 
 * buffers can be added to the packet because all of them are cache aligned 
 * and end on cache aligned boundary.  
 *
 * For the next packet the process is repeated, in case we have to start with 
 * not necessarily cache aligned offset into the buffer because the packet 
 * ends in the middle of the buffer, we copy some data to make the packet 
 * header cache aligned and use "offset to second buffer" feature.  
 *
 * See req_copy_dcaches_proc() for more details.
 *
 * Arguments:
 *  req - request
 *
 * Return Value:
 *  STATUS_SUCCESS
 *  STATUS_PENDING
 */
NTSTATUS
req_copy_dcaches_compact_proc(req_hdr_t *req)
{
    eee_desc_t *edesc;
    struct edesc_buf *bd, *last_bd;
    struct req_copy_dcaches_ctx *ctx = req_context(req);
    dcache_t *dc = ctx->nr_read_dcache_entries[ctx->nr_read_dcache_index];

    req_copy_dcaches_compact_proc_dispatch_x(req);

    while (ctx->nr_read_dcache_index < ctx->nr_read_num_dcache_entries) {

        edesc = sbm_getDescFromCOD(dc->dc_cod);

        while (edesc != NULL) {
            bd = edesc_firstBuf(edesc);
            last_bd = bd + EEE_GET_NUM_BUFS(edesc);

            while (bd < last_bd) {

                if (!CACHE_ALIGNED(bd->buf) || !CACHE_ALIGNED(bd->len)) {
                    /* Found unaligned buffer, compact this dcache.  
                     */
                    BCALL(req, REQ_COPY_DCACHE_COMPACT_1, dcache_compact(req, dc));
                    goto next_dcache;
                }
                ++bd;
            }
            edesc = edesc->hdr.next;
        }
  next_dcache:
        ctx->nr_read_dcache_index++;
        dc = ctx->nr_read_dcache_entries[ctx->nr_read_dcache_index];
    }

    ctx->nr_read_dcache_index = 0;
    ctx->nr_read_dcache_index_save = 0;
    ctx->nr_mtu_unused = 0;
    req_pop_state(req, STATUS_SUCCESS);
    if (req_copy_dcaches_proc(req, ctx) != STATUS_PENDING) {
        req_run(req);
    }
    return STATUS_PENDING;
}

NTSTATUS
req_copy_dcaches_compact(any_request_ptr_t req, struct req_copy_dcaches_ctx *ctx)
{
    ctx->nr_read_dcache_index = 0;
    req_push_state(req, REQ_COPY_DCACHE_COMPACT_0, ctx);
    return req_copy_dcaches_compact_proc(req.v);
}

NTSTATUS
req_copy_dcaches_wait_pending(req_hdr_t *req)
{
    struct req_copy_dcaches_ctx *ctx = req_context(req);
    req_pop_state(req, STATUS_SUCCESS);
    if (req_copy_dcaches_proc(req, ctx) == STATUS_SUCCESS) {
        req_run(req);
    }
    return STATUS_PENDING;
}

/*
 * Description:
 *  Make a reference to dcache entries in packets of pkt queue.
 *    
 * Arguments:
 *  req - request.
 *  ctx - the request context
 *
 * Return Value:
 *  STATUS_SUCCESS - copied the data successfully
 *  STATUS_PENDING - the request was queued, return to the dispatcher.
 */
NTSTATUS    
req_copy_dcaches_proc(any_request_ptr_t req, struct req_copy_dcaches_ctx *ctx)
{
    struct sk_buff *new_pkt;
    acpu_conn_t *conn = req.v->rh_conn;
    pkt_queue_t *pq = ctx->nr_pq;
    struct sk_buff *pkt = pq->pq_tail;

    uint32 pkt_offset = pkt->len;
    uint32 next_usable_buf_len = pkt_usable_len(conn, FALSE);
    uint32 next_usable_len = pkt_usable_data_len(conn, FALSE);
    uint32 pkt_remain = (pkt_usable_data_len(conn, pkt == pq->pq_head) - pkt_offset);
    uint32 pkt_buf_remain = (pkt_usable_len(conn, pkt == pq->pq_head) - pkt_offset);
    uint32 already_copied = (pq->pq_len - pq->pq_start_offset);
    uint32 copy_remain = ctx->nr_read_count - already_copied;

    // ASSERT(req_copy_dcaches_verify(req, ctx, (pq->pq_len - pq->pq_start_offset)));

    dcache_t *dc = ctx->nr_read_dcache_entries[ctx->nr_read_dcache_index];
    uint32 cod_offset = ((ctx->nr_read_offset + already_copied) - 
                         (dc->dc_block * EFS_BLOCK_SIZE));
    void * __restrict__ copy_to = pkt->data + pkt_offset;
    boolean alloc_failed = FALSE;
    char tmp_packet[EEE_BUFSIZE_LARGE + CACHELINE_SIZE];
    int num_allocs = 0;

    while (copy_remain != 0) {

        if (dc->dc_fflags & DCACHE_F_WRITE_LOCK) {

            /* This dcache is being compacted, wait for this to complete.  
             */

            ctx->nr_pq = pq;
            ctx->nr_pq_tail = pkt;
            ctx->nr_read_dcache_index_save = ctx->nr_read_dcache_index;
            pq->pq_len = pq->pq_start_offset + (ctx->nr_read_count - copy_remain);
            pq->pq_tail = pkt;

            dcache_addToDcacheWaitingList(req.v, dc);
            req_push_state(req, REQ_COPY_DCACHE_WAIT_PENDING, ctx);
            return req_block_x(req, KPI_DCACHE_WAIT_COMPACT);
        }

        uint32 remain_this_dcache = MIN((dc->dc_nBlocks * EFS_BLOCK_SIZE) - cod_offset, 
                                        copy_remain);
        eee_desc_t *edesc = sbm_getDescFromCOD(dc->dc_cod);

        /* Find the descriptor and buffer containing the specified cod_offset
         */
        uint32 buf_offset = cod_offset;
        uint32 num_bufs;
        eee_descBuf_t *bd;

        while (1) {
            num_bufs = EEE_GET_NUM_BUFS(edesc);

            bd = edesc_firstBuf(edesc);

            while ((buf_offset != 0) && (num_bufs != 0)) {
                if (buf_offset >= bd->len) {
                    buf_offset -= bd->len;
                    bd++;
                    num_bufs--;
                } else {
                    break;
                }
            }
            if (num_bufs == 0) {
                edesc = edesc->hdr.next;
            } else if (buf_offset < bd->len) {
                break;
            } else {
                edesc = edesc->hdr.next;
            }
        }

        /* Arrange the data of the last packet header buffer so the misaligned 
         * portion of the first data buffer can be copied into the buffer 
         * containing the RPC headers making the end of the first packet 
         * buffer cache aligned.  
         */

        void * __restrict__ copy_from = bd->buf + buf_offset;
        uint32 buf_remain = bd->len - buf_offset;

        while (remain_this_dcache != 0) {
            boolean make_ref = TRUE;

            uint32 copy_this_time = MIN(remain_this_dcache, buf_remain);

            if (!(pkt->flags & PKT_DCACHE_REF)) {

                /* There are no referenced buffers yet. 
                     */

                if (!CACHE_ALIGNED(copy_from) || (pkt->len != 0)) {

                    /* The buffer is not cache aligned or there already is 
                     * some data in the buffer.  Copy some data from the 
                     * current dcache buffer so the data ends up on the cache 
                     * line boundary.  The packets are supposed to be 
                     * allocated so the network headers start at the cache 
                     * line boundary.  We are going to use "offset into the 
                     * second buffer" option when transmitting the packet.  
                     */

                    uint32 move_bytes = MIN(GET_PAD((uintptr_t)copy_to,
                                                    CACHELINE_SIZE),
                                            copy_this_time);
                    if (move_bytes != 0) {
#ifdef DEBUG
                        ultranew_memcpy(copy_to,
                                        copy_from,
                                        CACHELINE_SIZE);
#else
                        __builtin_memcpy(copy_to, 
                                         copy_from, 
                                         CACHELINE_SIZE);
#endif
                        pkt->len += move_bytes;
                        copy_remain -= move_bytes;
                        pkt_remain -= move_bytes;
                        buf_remain -= move_bytes;
                        copy_from += move_bytes;
                        copy_to += move_bytes;
                        copy_this_time -= move_bytes;
                        remain_this_dcache -= move_bytes;
                    }
                } else {

                    /* The buffer is cache aligned and there is no data in
                     * the buffer.  Adjust the payload start so that network
                     * headers will end on the cache line boundary.  When
                     * transmitting the packet we will use "offset into first
                     * buffer" option.
                     */

                    pkt->data = alignptr(pkt->data, CACHELINE_SIZE);
                }
            } else {
                struct edesc_buf *bd = pkt_ref_bd(pkt, pkt->num_bufs - 1);
                if (!(CACHE_ALIGNED(bd->buf + bd->len) &&
                      CACHE_ALIGNED(copy_from))) {

                    /* Can not add another buffer to this packet because
                     * either the previous or the next buffer do not conform
                     * to the alignment restrictions.
                     */

                    if ((conn->prot == TPL_PROT_UDP) &&
                        ((copy_to = req_align_last_buffer(pkt, &copy_this_time)) != NULL)) {
                        make_ref = FALSE;
                    } else {
                        copy_this_time = 0;
                        goto alloc_new;
                    }
                }
            }

            if (make_ref && (pkt_buf_remain < sizeof(struct edesc_buf))) {
                /* Not enough space to embed the next reference.
                 */
                goto alloc_new;
            }

            copy_this_time =
            MIN(pkt_remain, 
                make_ref ? 
                copy_this_time : MIN(pkt_buf_remain, copy_this_time));
            if (copy_this_time != 0) {
                if (make_ref) {
                    pkt_add_buffer(pkt,
                                   copy_from,
                                   copy_this_time,
                                   dc,
                                   !alloc_failed);
                    pkt_buf_remain -= sizeof(struct edesc_buf);
                } else {
                    memcpy(copy_to, copy_from, copy_this_time);
                    pkt->len += copy_this_time;
                    pkt_buf_remain -= copy_this_time;
                }
            }

            /* Advance iteration variables 
             */
            remain_this_dcache -= copy_this_time;
            copy_remain -= copy_this_time;

            if (remain_this_dcache != 0) {
                if (buf_remain == copy_this_time) {
                    void *buf = bd[1].buf;
                    uint32 len = bd[1].len;
                    eee_desc_t *next = edesc->hdr.next;
                    num_bufs--;
                    if (num_bufs == 0) {
                        edesc = next;
                        bd = edesc_firstBuf(edesc);
                        num_bufs = EEE_GET_NUM_BUFS(edesc);
                        buf = bd->buf;
                        len = bd->len;
                    } else {
                        bd++;
                        if (num_bufs == 1) {
                            PREFETCH(PREF_LOAD, next, 0);
                            PREFETCH(PREF_LOAD, next, CACHELINE_SIZE);
                        } else {
                            PREFETCH(PREF_LOAD, bd, CACHELINE_SIZE);
                        }
                    }
                    copy_from = buf;
                    buf_remain = len;
                } else {
                    copy_from += copy_this_time;
                    buf_remain -= copy_this_time;
                }
            } else {
                ++ctx->nr_read_dcache_index;
            }

            if (copy_remain != 0) {
                if (pkt_remain == copy_this_time) {
                    alloc_new:
                    if (!alloc_failed) {
                        if (pkt->next != 0) {
                            new_pkt = pkt->next;
                        } else {

                            /* Sum the number of bytes not used compared with
                             * full MTU utilization each time we allocate the
                             * packet. The last packet will not be counted.
                             */
                            // ASSERT(pkt_remain >= copy_this_time);
                            ctx->nr_mtu_unused += pkt_remain - copy_this_time;

                            new_pkt = pkt_acpu_alloc(req.v->rh_alloc_type);
                            if (new_pkt == NULL) {
                                /* Failed the allocation, continue
                                 * through this routine to calculate the
                                 * remaining packets we will need using
                                 * the temporary buffer on the stack for
                                 * packet.
                                 */
                                alloc_failed = TRUE;
                                new_pkt = ((struct sk_buff *)
                                           alignto64((uintptr_t)tmp_packet, CACHELINE_SIZE));
                                new_pkt->flags = (PKT_ALLOCATED_ON_ACPU |
                                                  PKT_DATA | PKT_DESC_INUSE);
                                new_pkt->data = (char *)new_pkt + PKT_BUFF_INDENT;
                                ++num_allocs;

                                /* Save the context for waiting.
                                 */
                                ctx->nr_pq = pq;
                                ctx->nr_pq_tail = pkt;
                                ctx->nr_read_dcache_index_save = ctx->nr_read_dcache_index;
                                pq->pq_len = pq->pq_start_offset + (ctx->nr_read_count - copy_remain);
                                pq->pq_tail = pkt;
                            } else {
                                // ASSERT(pkt->len != 0);

                                new_pkt->data = ((char *)new_pkt + PKT_BUFF_INDENT);
                                pkt->next = new_pkt;
                            }
                        }
                    } else {

                        /* Sum the number of bytes not used compared with
                         * full MTU utilization each time we allocate the
                         * packet. The last packet will not be counted.
                         */
                        // ASSERT(pkt_remain >= copy_this_time);
                        ctx->nr_mtu_unused += pkt_remain - copy_this_time;

                        ++num_allocs;
                        new_pkt = ((struct sk_buff *)
                                   alignto64((uintptr_t)tmp_packet, CACHELINE_SIZE));
                        new_pkt->flags = (PKT_ALLOCATED_ON_ACPU |
                                          PKT_DATA | PKT_DESC_INUSE);
                        new_pkt->data = (char *)new_pkt + PKT_BUFF_INDENT;
                    }
                    pkt = new_pkt;
                    pkt->len = 0;
                    copy_to = pkt->data;
                    pkt_remain = next_usable_len;
                    pkt_buf_remain = next_usable_buf_len;
                } else {
                    pkt_remain -= copy_this_time;
                    copy_to += copy_this_time;
                }
            }
        }
        dc = ctx->nr_read_dcache_entries[ctx->nr_read_dcache_index];
        cod_offset = 0;
    }

    if ((ctx->nr_mtu_unused * VS_MTU_UTILIZATION_DIVISOR) >  
        ((pq->pq_start_offset + 
          ctx->nr_read_count - 
          pkt->len + 
          ctx->nr_mtu_unused) * req.v->rh_vs->vs_max_unused_read_mtu)) {

        /* We are out of allowed fragmentation limits. Release the references
         * to the dcaches we have acquired, compact the dcaches, and retry.
         * 
         * After compaction the dcaches will consist of buffers which are
         * aligned on cache line boundary and have cache aligned length.
         * This buffer configuration can always be transmitted utilizing full
         * MTU size with some minimal memory copies (maximum cacheline per
         * packet).
         */

        pkt = ctx->nr_save_tail;
        pkt->len = ctx->nr_save_len;
        pkt_acpu_free_chain(pkt->next);
        pkt->next = NULL;
        pkt_free_dcache_ref(pkt);

        /* Restore the packet queue to original.
                                     */
        pq->pq_len = pq->pq_start_offset;
        pq->pq_tail = pkt;

        return req_copy_dcaches_compact(req, ctx);
    } else if (!alloc_failed) {
        /* Verify that the copy is performed correctly.
         */
        // ASSERT(req_copy_dcaches_verify(req, ctx, ctx->nr_read_count - copy_remain));

        /* Set the resulting queue position.
         */
        pq->pq_len = pq->pq_start_offset + ctx->nr_read_count;
        pq->pq_tail = pkt;
        pq->pq_current = NULL;
        pq->pq_start_offset = pq->pq_len;

        pq_validate(pq);
        return STATUS_SUCCESS;
    } else {
        pq->pq_alloc_len = num_allocs;

        req_push_state(req, REQ_COPY_DCACHE_ALLOC, ctx);
        req_push_state(req, REQ_WAIT_N_PACKETS, pq);
        return pkt_acpu_wait(req);
    }
}

/*
 * Description:
 *  Free a packet. 
 *
 * Arguments:
 *  pkt - packet to free
 *
 * Return Value:
 *  None.
 */
void
pkt_free(struct sk_buff *pkt)
{
    req_hdr_t *req = pkt->p_req;
    // VERIFY(pkt->flags & PKT_DESC_INUSE);

    if (!(pkt->flags & PKT_INPUT) && (req != NULL)) {
        // ASSERT(req->rh_flags & REQ_PKT_REFERENCE);

        if (req->rh_kpi.ks_idx != 1) {
            switch (req->rh_kpi.ks_idx) {
            case 3:
                kpi_leave_unchecked(&req->rh_kpi, KPI_LEVEL_3);
            case 2:
                kpi_done(&req->rh_kpi);
                break;
            default:
                // ASSERT(0);
            }
        }

        if (req->rh_flags & REQ_IS_CIFS) {
            req->rh_flags &= ~REQ_PKT_REFERENCE;
            cifs_req_dealloc((cifs_request_t *)req);
        } else {
            req->rh_flags = 0;
            nfs_req_dealloc((nfs_request_t *)req);
        }
        pkt->p_req = NULL;
    }
    if (pkt->flags & PKT_ALLOCATED_ON_ACPU) {
        pkt_free_dcache_ref(pkt);
        pkt_acpu_free(pkt);
    } else {
        pkt_freePkt(pkt);
    }
}

/*
 * Description:
 * Copy out the cod entry into the response packet queue at the
 *  current queue position.
 *
 * The queue position is adjusted to point after the last written byte.
 *
 * Arguments:
 *  pq - packet queue to fill
 *  cod_offset - offset in cod
 *  cod - cod
 *  length - data length
 *
 * Return Value:
 *  None.
 */
void
req_copy_cod_entry(pkt_queue_t *pq, uint32 cod_offset, nfx_cod_t cod, uint32 length)
{
    eee_desc_t *edesc;
    eee_descBuf_t *bd;
    uint32 buf_offset, num_bufs, remain, pkt_offset;
    struct sk_buff *pkt;
    uint32 pkt_remain, buf_remain, copy_this_time;
    void *copy_to, *copy_from;

#ifdef DEBUG
    address_t ra = get_ra();
#endif
    /* Check if the cod is valid
     */
    // ASSERT(sbm_verifyIsCODInSac(cod, 1));
    edesc = sbm_getDescFromCOD(cod);

    // ASSERT(EEE_GET_PKT_LEN(edesc) >= (length + cod_offset));

    /* Find the descriptor and buffer containing the specified cod_offset
     */
    buf_offset = cod_offset;

    while (1) {

        num_bufs = EEE_GET_NUM_BUFS(edesc);
        bd = edesc_firstBuf(edesc);

        while ((num_bufs != 0) && (buf_offset != 0)) {
            if (buf_offset >= bd->len) {
                buf_offset -= bd->len;
                bd++;
                num_bufs--;
            } else {
                break;
            }
        }
        if (num_bufs == 0) {
            edesc = edesc->hdr.next;
        } else if (buf_offset < bd->len) {
            break;
        } else {
            edesc = edesc->hdr.next;
        }
    }

    remain = length;
    pkt = pq->pq_current;
    pkt_offset = pq->pq_offset;

    while (remain != 0) {
        pkt_remain = pkt->len - pkt_offset;
        buf_remain = bd->len - buf_offset;
        copy_this_time = ((pkt_remain < buf_remain) ? pkt_remain : buf_remain);
        copy_to = pkt->data + pkt_offset;
        copy_from = bd->buf + buf_offset;

        if (remain <= copy_this_time) {
            copy_this_time = remain;
        }

        /* Advance iteration variables 
         */
        if (pkt_remain == copy_this_time) {
            pkt = pkt->next;
            pkt_offset = 0;
        } else {
            pkt_offset += copy_this_time;
        }

        if (buf_remain == copy_this_time) {
            num_bufs--;
            if (num_bufs == 0) {
                edesc = edesc->hdr.next;
                if (remain != copy_this_time) {
                    bd = edesc_firstBuf(edesc);
                    num_bufs = EEE_GET_NUM_BUFS(edesc);
                }
            } else {
                bd++;
            }
            buf_offset = 0;
        } else {
            buf_offset += copy_this_time;
        }

        remain -= copy_this_time;

        // ASSERT(eee_buf_valid(copy_to, copy_this_time, ra));
        // ASSERT(eee_buf_valid(copy_from, copy_this_time, ra));

        memcpy(copy_to, copy_from, copy_this_time);
    }

    /* Set the resulting queue position
     */

    pq->pq_current = pkt;
    pq->pq_offset = pkt_offset;
    pq->pq_start_offset += length;

    return;
}

/*
 * Description:
 *  Add pad at the end of the response queue. The caller should
 *  make sure there's enough space in the last packet of queue.
 *
 *  When the padding is less than 4 bytes, there is always enough
 *  space in the queue and there should be no need to check.
 *  The current queue position is advanced forward by pad.
 *
 * Arguments:
 *  pq - packet queue
 *  pad - pad to add.
 *
 * Return Value:
 *  None.
 */
void
pq_pad(pkt_queue_t *pq, uint32 pad)
{
    struct sk_buff *pkt;
    uint16 offset;

    pq_validate(pq);

    if (pad != 0) {
        pq->pq_tail->len += pad;

        if (pq->pq_tail->flags & PKT_DCACHE_REF) {

            /* If the packet is referencing a dcache buffer, add padding the 
             * to the last buffer.
             */

            pkt = pq->pq_tail;

            if (pkt->num_bufs != 0) {
                pkt_ref_bd(pkt, pkt->num_bufs - 1)->len += pad;
                pkt->data_len += pad;
            }
        }

        pq->pq_start_offset += pad;
        pq->pq_len += pad;

        if (pq->pq_current != NULL) {

            /* Advance forward by pad if not already at the end, in which case
             * we advance implicitly.
             */
	    offset = pq->pq_offset + pad;

            if (offset == pq->pq_current->len) {
                pq->pq_offset = 0;
                pq->pq_current = pq->pq_current->next;
            } else {
                // ASSERT(offset < pq->pq_current->len);
                pq->pq_offset = offset;
            }
        }

        pq_validate(pq);
    }
}

/*
 * Description:
 *
 *  Delete some data in the middle of the queue.  It is assumed that the range 
 *  to be deleted spans at most two packets.  If the range is ending before 
 *  the end of the packet, the remaining data in the packet is moved towards 
 *  the beginning of the packet.  If any data was moved, The addresses of the 
 *  affected range are returned in affectedRangeStart and affectedRangeEnd.  
 *    
 * Arguments:
 *  pq - packet queue 
 *  location - the location of the beginning of the range to be deleted
 *  prev_pkt - the previous packet (prev_pkt->next == pkt) OR
 *             NULL if pkt is the head of the queue
 *  size - the size of the range to be deleted in bytes
 *  affectedRangeStart - the start of the moved range is returned here
 *  affectedRangeEnd - the end of the moved range is returned here
 *  moveBy - offset by which the range was moved
 *    
 * Return Value:
 *  None.
 */
void
pq_delete_range(pkt_queue_t *pq,    
       struct pq_location *location,
       struct sk_buff *prev_pkt,
       uint32 size,
       void **affectedRangeStart,
       void **affectedRangeEnd,
       int *moveBy)
{
    pq_validate(pq);

    /* This function works only if the deleted range is before the end
     * of written data.
     */
    // ASSERT(pq_location_offset(pq, location) + size <= pq->pq_start_offset);

    struct sk_buff *pkt = location->pql_current;
    void *addr = location->pql_data;
    struct sk_buff *curr_prev_pkt = prev_pkt;

    pq->pq_len -= size;
    pq->pq_start_offset -= size;

    if (pq->pq_start_offset > pq->pq_len) {
        pq->pq_start_offset = pq->pq_len;
    }

    /* Adjust the packets containing the entry.
     */
    int remain = size;
    // ASSERT(((char *)addr - pkt->data) <= 0xFFFFFFFF);
    uint32 offset = (char *)addr - pkt->data;

    while (remain > 0) {
        if (offset + remain <= pkt->len) {
            /* The entry is completely within this packet
             */

            /* Setup the return addresses of the affected range
             */
            *affectedRangeStart = pkt->data + offset;
            *affectedRangeEnd = pkt->data + pkt->len;
            *moveBy = -remain;

            /* Remove the entry
             */
            memmove(pkt->data + offset,
                    pkt->data + offset + remain,
                    pkt->len - (offset + remain));
            pkt->len -= remain;
            remain = 0;
        } else {
            /* The entry is the last in this packet and
             * goes past the packet.
             */
            remain -= pkt->len - offset;
            pkt_setHdrSize(pkt, offset);
            offset = 0;
        }
        struct sk_buff *nextPkt = pkt->next;
        if (pkt->len == 0) {
            /* All data in this packet has been deleted. Free the
             * packet.
             */
            // ASSERT((curr_prev_pkt == NULL) || (curr_prev_pkt->next == pkt));

            if (curr_prev_pkt != NULL) {
                curr_prev_pkt->next = pkt->next;
                if (pkt->next == NULL) {
                    pq->pq_tail = curr_prev_pkt;
                }
            } else {
                pq->pq_head = pkt->next;
            }

            pkt->next = NULL;
            pkt_free(pkt);
        } else {
            curr_prev_pkt = pkt;
        }
        pkt = nextPkt;
    }

    /* If the current packet is one of the affected packets, we may need to 
     * adjust the pq_offset field.  
     */
    uint32 start_offset = 0;
    for (pkt = pq->pq_head; pkt != NULL; pkt = pkt->next) {
        if ((start_offset + pkt->len) > pq->pq_start_offset) {
            pq->pq_offset = pq->pq_start_offset - start_offset;
            break;
        } else {
            start_offset += pkt->len;
        }
    }

    pq_validate(pq);
    return;
}

/*
 * Description:
 *  Allocate a packet descriptor from free packet list.
 *
 * Arguments:
 *  free_list - the free list to allocate from
 *  alloc_type - the allocation type
 *
 * Return Value:
 *  Returns address of allocated pkt descriptor or NULL if not enough memory.
 */
struct sk_buff *
pkt_alloc_from_list(struct pkt_free_list *free_list, int32 alloc_type)
{
    struct sk_buff *pkt = free_list->pf_head;
    if (pkt != NULL) {
#ifdef REQ_VALIDATE_PACKET_COUNTS
        if (! CM_IS_TXRX_NCPU) {
            if (creq.v != NULL) {
                creq.v->rh_packets_allocated++;
            }
        }
#endif

#ifdef PKT_ALLOC_USE_TAILQ        
        free_list->pf_head = pkt->next;
        if (free_list->pf_head == NULL) {
            free_list->pf_tail = &free_list->pf_head;
        }
#else
        free_list->pf_head = pkt->next;
#endif        

        pkt_init(pkt);
        pkt->alloc_type = alloc_type;
        if ((alloc_type <= 0) || (alloc_type >= PKT_ALLOC_TYPE_MAX)) {
            panic("pkt_alloc_from_list: invalid allocation type\n");
        }
        pkt_alloc_stats[alloc_type]++;
        UPDATE_STAT_INC(free_list->pf_calls);
        UPDATE_MAXSTAT_INC(free_list->pf_inuse, free_list->pf_maxinuse);
    }
    else {
        UPDATE_STAT_INC(free_list->pf_no_memory);
    }
    return pkt;
}

/*
 * Description:
 *  Free dcache entries stored in req_copy_dcaches() context.    
 *    
 * Arguments:
 *  ctx - req_copy_dcaches context
 *
 * Return Value:
 *  None.
 */
void
req_free_dcache_entries(struct req_copy_dcaches_ctx *ctx)
{
    int i;

    for (i = 0; i < ctx->nr_read_num_dcache_entries; ++i) {
        dcache_rele(ref_delz(ctx, nr_read_dcache_entries[i]));
    }
    return;
}
#endif /* LATER */

/*
 * Description:
 *  Return location corresponding to specified offset 
 *  from beginning of queue.
 *    
 * Arguments:
 *  pq - packet queue
 *  loc - the location is stored here
 *  offset - offset from the beginning of the queue
 *
 * Return Value:
 *  Returns loc, the updated pg_location structure adddress.
 */
struct pq_location *
pq_get_location(pkt_queue_t *pq, struct pq_location *loc, int offset)
{
    int cur_offset = pq->pq_start_offset;
    pq_seek_to(pq, offset);

    *loc = pq_location(pq);
    pq_seek_to(pq, cur_offset);

    return loc;
}
