/* vi:ts=4
 *-----------------------------------------------------------------
 * Name         : pkt-queue-struct.h
 * Description  : Defines the structure for managing packet queues.
 * Copyright (C) 2009, OnStor, Inc.
 *-----------------------------------------------------------------
 */
#ifndef _PKT_QUEUE_STRUCT_H
#define _PKT_QUEUE_STRUCT_H

/* The pkt_queue structure is used for managing packet queues.
 */
typedef struct pkt_queue {
    struct sk_buff *pq_head; /* head of queue. Pkt will contain the SMB header */
    struct sk_buff *pq_tail; /* tail of queue. If empty, this field points to pq_head field */

    /* Current read or write pointer. This is used for processing of
     * andX requests. When processing the X part of andX request,
     * pq_current points to the packet containing the beginning of the
     * payload. The request processing code for the requests that can
     * be used as andX part should use this pointer to access the
     * request payload and not assume that the payload follows the SMB header.
     */
    struct sk_buff *pq_current;

    unsigned short pq_offset; /* Offset in pq_current pkt to beginning of payload data */

    /* If pq_flags set, allocations to packet queue should not block but
     * return a failure if there is not enough memory immediately.
     */
#define PQ_ALLOC_DONT_WAIT 0x2
    unsigned short pq_flags;    /* Flags */

    unsigned int pq_start_offset; /* Corresponding offset from the beginning of the queue */
    unsigned int pq_len;    /* Total number of bytes in the queue */

    /* Size of requested allocation if the request is blocked waiting for memory
     * If the request allocates contiguous data into packet queue, this
     * is the desired queue length.
     *
     * If the request is allocating specific number of packets, this is
     * the number of packets not allocated yet.
     */
    unsigned int pq_alloc_len;

#ifdef JONG_HACK
    void *pq_save_data;
#endif    
} pkt_queue_t;


/* The structure to save the position in the packet queue
 */
struct pq_location {
   struct sk_buff *pql_current; /* current pkt in queue, equal to pq_current */
   void *pql_data; /* current data pointer, equal to pq_current->pHdr + pq_offset */
};

/* The structure for the free list packet descriptors
 */
struct pkt_free_list {
    struct sk_buff *pf_head;    /* Head of the list of free packets */
    struct sk_buff **pf_tail;   /* Pointer to the last packet next pointer */

    /* queue of requests or TPL connections waiting for a packet allocation */
    union {
        SIMPLEQ_HEAD(, req_hdr) pf_waiters;
        SIMPLEQ_HEAD(, _tpl_connCb) pf_conn_waiters;
    };
    
    unsigned int pf_inuse;      /* Number of packets currently in use */
    unsigned int pf_maxinuse;   /* Maximum number of packets currently in use */
    unsigned long pf_calls;     /* Number of packet allocations */
    unsigned long pf_no_memory; /* Number of failed allocations */
};

#ifdef LATER
/*
 * The size of the standard ethernet header.
 */
#define PKT_ETH_HEADER_SIZE	14

/*
 * Extra bytes added to standard ethernet header to account
 * for the 802.1q header.
 */
#define PKT_VLAN_EXTRA_ETH_HEADER_SIZE	4

/* The size of luc header prepended to each packet.
 */
#define PKT_LUC_HEADER_SIZE 22

/* The size of the header of the first TCP packet in a rpc reply
 * Ethernet header + TCP/IP header + RPC header
 */
#define PKT_TCP_FIRST_HEADER_SIZE (PKT_ETH_HEADER_SIZE + 40 + 4)

/* The size of the subsequent TCP packet in a rpc reply
 * Ethernet header + TCP/IP header
 */
#define PKT_TCP_HEADER_SIZE (PKT_ETH_HEADER_SIZE + 40)

/* The size of the header of the first UDP packet in a rpc reply
 * Ethernet header + IP header + UDP header
 */
#define PKT_UDP_FIRST_HEADER_SIZE (PKT_ETH_HEADER_SIZE + 20 + 8)

/* The size of the header of the subsequent UDP packet in a rpc reply
 * Ethernet header + IP header
 */
#define PKT_UDP_HEADER_SIZE (PKT_ETH_HEADER_SIZE + 20)

/* Context structure for req_copy_dcaches().
 */
struct req_copy_dcaches_ctx {
    unsigned long nr_read_offset; /* The offset in the file read from */
    unsigned int nr_read_count;  /* The number of bytes read */
    short nr_read_num_dcache_entries; /* Number dcache entries covered by the range */
    short nr_read_dcache_index;  /* index of current dcache being copied */

    pkt_queue_t *nr_pq; /* packet queue to fill with the data */
    struct sk_buff *nr_pq_tail;    /* The saved tail of the queue */

    struct dcache_s *nr_read_dcache_entries[NFX_MAX_COD_MSG]; /* array of dcache entries */
    short nr_read_dcache_index_save;    /* The saved dcache index */
    int nr_mtu_unused; /* number bytes not used compared with full MTU utilization */

    struct sk_buff *nr_save_tail; /* Saved original last packet in queue, used to undo the conversion */
    int nr_save_hdrSize;    /* Saved original length of the last packet in the queue */
};

/* Packet allocation types. @@@ use defines for cifs counters.
 */
enum pkt_alloc_type {
    PKT_ALLOC_TYPE_UNKNOWN = 1,
    PKT_ALLOC_PMAP_NULL = 2,
    PKT_ALLOC_NFS_V2_NULL = PKT_ALLOC_PMAP_NULL + PMAP_PROC_CALLIT + 1,
    PKT_ALLOC_NFS_V3_NULL = PKT_ALLOC_NFS_V2_NULL + NFS_V2_PROC_STATFS + 1,
    PKT_ALLOC_NLM_NULL = PKT_ALLOC_NFS_V3_NULL + NFS_V3_PROC_COMMIT + 1,
    PKT_ALLOC_NSM_NULL = PKT_ALLOC_NLM_NULL + NLM_FREE_ALL + 1,
    PKT_ALLOC_MOUNT_NULL = PKT_ALLOC_NSM_NULL + NSM_NOTIFY + 1,
    PKT_ALLOC_RQD_NULL = PKT_ALLOC_MOUNT_NULL + MNT_PATHCONF + 1,
    PKT_ALLOC_VSCAN_NULL = PKT_ALLOC_RQD_NULL + RQD_GETACTIVEQUOTA + 1,
    PKT_ALLOC_RPC_GETPORT = PKT_ALLOC_VSCAN_NULL + 9,
    PKT_ALLOC_NETBIOS = PKT_ALLOC_RPC_GETPORT + 1,
    PKT_ALLOC_CIFS = PKT_ALLOC_NETBIOS + 1,
    PKT_ALLOC_CIFS_TRANS = PKT_ALLOC_CIFS + 74 + 1,
    /* See TRANS2_MAX_FUNCTION (0x11)
     */
    PKT_ALLOC_CIFS_TRANS_2 = PKT_ALLOC_CIFS_TRANS + 0x11 + 1,
    PKT_ALLOC_CIFS_NT_TRANS = PKT_ALLOC_CIFS_TRANS_2 + 17 + 1,
    PKT_ALLOC_TYPE_MAX = PKT_ALLOC_CIFS_NT_TRANS + 8 + 1,
};

#endif /* LATER */

#endif /* _PKT_QUEUE_STRUCT_H */
