/* vi:ts=4
 *-----------------------------------------------------------------
 * Copyright (C) 2003, ClariStor Inc.
 * Name         : pkt-queue-test.c
 * Description : Implementation of packet queue api.
 *-----------------------------------------------------------------
 */
#include "nfx-incl.h"

#include "../sm-cifs/cifs-conn-api.h"
#include "../sm-pkt/pkt-queue-api.h"
#include "../sm-debug/debug.h"
#include "sm-req-queue/req-tbl.h"
#include "../sm-utils/txrx-utils-api.h"

#include "../sm-sbm-rev2/sbm-api.h"
#include "../sm-sbm-rev2/sbm.h"
#include "../sm-sb1250-dm/sb1250-dm-api.h"

#include "../sm-cifs/cifs-req-api.h"
#include "../sm-nfs/nfs-req-api.h"
#include "../sm-dcache/dcache.h"

#include "../sm-req-queue/req-test.h"
#include "../sm-sbm-rev2/sbm-api.h"
#include "../sm-sbm-rev2/sbm.h"
#include "../sm-ui/uihdr.h"

/* State machine function prototypes.
 */
void pq_pullup_finish(req_hdr_t *req);
NTSTATUS req_alloc_packet_continue(cifs_request_t *req);
NTSTATUS req_copy_dcaches_cont(any_request_ptr_t req);

/*++

Routine Description:

    Allocate a packet queue with associated buffer(s) for unit tests.
    These tests often run on the ACPU so it is not safe to access all of
    the same fields as the standard pkt_allocPkt().
    
Arguments:

    pq - The packet queue to initialize.  On failure pq->pq_head will be NULL.
    rpc_length - The length of the rpc information
    data_length - The length of the COD data.
    numCods - The returned number of COD(s) we allocated for data_length.
    cods - The populated COD array, the caller should pass in an array of
           size NFX_MAX_COD_MSG

Return Values:

    TRUE - Success, FALSE failure.

--*/

boolean
pq_alloc_test(pkt_queue_t *pq,
              uint32 rpc_length,
              uint32 data_length,
              uint32 *numCods,
              nfx_cod_t *cods)
{
    NTSTATUS status = STATUS_SUCCESS;
    boolean failed = FALSE;
    int j = 0;

    VERIFY((data_length == 0) || ((numCods != NULL) && (cods != NULL)));
    pq_make_empty(pq);
    if (numCods != NULL) {
        *numCods = 0;
    }
    uint32 remaining_length = rpc_length;
    uint64 tval = eee_timestamp();
    while ((remaining_length > 0) && !failed) {

        pkt_desc_t *pkt = eee_alloc_buf(&local_buf_pool[EEE_BUF_LARGE]);

        if (pkt != NULL) {
            memset(pkt, 0, sizeof(*pkt));
            pkt->pHdr = (char *)pkt + PKT_BUFF_INDENT;
            pkt->hdrSize = MIN(remaining_length,
                               (EEE_BUFSIZE_LARGE - PKT_BUFF_INDENT));
            remaining_length -= pkt->hdrSize;
            pkt->flags = (PKT_INPUT | PKT_DESC_INUSE | PKT_DATA);
            pkt->p_time_stamp = tval;
            pq_add_packet(pq, pkt);
        } else {
            failed = TRUE;
        }
    }

    remaining_length = data_length;
    while ((remaining_length > 0) && !failed) {
        struct edesc_buffer_disposition buf_disposition;
        int buf_size_array[8];
        uint32 remaining = MIN(remaining_length, 8192);
        eee_desc_t *edesc;

        while (remaining > 0) {
            if (remaining < EEE_BUFSIZE_LARGE) {
                buf_size_array[j] = remaining;
                remaining = 0;
            } else {
                buf_size_array[j] = EEE_BUFSIZE_LARGE;
                remaining -= EEE_BUFSIZE_LARGE;
            }
            ++j;
        }
        buf_disposition.num_buffers = j;
        buf_disposition.buf_offsets = NULL;
        buf_disposition.buf_sizes = buf_size_array;
        status = test_make_edesc(&edesc, &buf_disposition, EEE_BUF_LARGE);
        if (status == STATUS_SUCCESS) {
            cods[*numCods] = sbm_getCODFromDesc(edesc);
            sbm_sendCODToSac(cods[*numCods]);
            (*numCods)++;
            remaining_length -= MIN(remaining_length, 8192);
        } else {
            failed = TRUE;
        }
    }
    if (failed) {
        /* Free the partially allocated info.
         */
        if (numCods != NULL) {
            for (j = 0; j < *numCods; ++j) {
                sbm_freeCOD(cods[j]);
            }
            *numCods = 0;
        }
        if (pq->pq_head != NULL) {
            pkt_freePkt(pq->pq_head);
            pq_make_empty(pq);
        }
        return FALSE;
    }
    return TRUE;
}

/* Context for pqtest_1().
 */
struct pqtest_1_ctx
{
    pkt_queue_t pq;
};


/*++

Routine Description:

    Get the location corresponding to the specified offset 
    from the beginning of the queue.    
    
Arguments:

    pq - packet queue
    loc - the location is stored here
    offset - offset from the beginning of the queue

Return Values:

    Returns loc.

--*/

struct pq_location *
pq_get_location(pkt_queue_t *pq, struct pq_location *loc, int offset)
{
    int cur_offset = pq->pq_start_offset;
    pq_seek_to(pq, offset);
    *loc = pq_location(pq);
    pq_seek_to(pq, cur_offset);
    return loc;
}


/*++

Routine Description:
    Tests for pq_delete_range().    
    
Arguments:
    None

Return Values:
    Returns STATUS_SUCCESS or panic().
--*/
NTSTATUS 
pqtest_1_proc()
{
    struct pqtest_1_ctx *ctx = req_context(creq.v);
    struct pq_location loc;
    void *range_start;
    void *range_end;
    int move_by;
    void *range_start_should_be, *range_end_should_be;

    dd();

    bc(pqtest_make_pq(&ctx->pq, 2, (int []){100, 200}));

    pq_get_location(&ctx->pq, &loc, 30);

    pq_seek_to(&ctx->pq, 150);

    range_start_should_be = loc.pql_data;
    range_end_should_be = loc.pql_current->pHdr + loc.pql_current->hdrSize;
    pq_delete_range(&ctx->pq, 
                    &loc,
                    NULL,
                    30,
                    &range_start,
                    &range_end,
                    &move_by);

    VERIFY(ctx->pq.pq_head->hdrSize == 70);
    VERIFY(ctx->pq.pq_head->pNext->hdrSize == 200);
    VERIFY(ctx->pq.pq_start_offset == 120);
    VERIFY(range_start == range_start_should_be);
    VERIFY(range_end == range_end_should_be);
    VERIFY(ctx->pq.pq_len == 270);

    pkt_acpu_free_chain(ctx->pq.pq_head);

    bc(pqtest_make_pq(&ctx->pq, 2, (int []){100, 200}));

    pq_get_location(&ctx->pq, &loc, 30);

    pq_seek_to(&ctx->pq, 100);

    range_start_should_be = loc.pql_data;
    range_end_should_be = loc.pql_current->pHdr + loc.pql_current->hdrSize;

    pq_delete_range(&ctx->pq,
                    &loc,
                    NULL,
                    70,
                    &range_start,
                    &range_end,
                    &move_by);

    VERIFY(ctx->pq.pq_head->hdrSize == 30);
    VERIFY(ctx->pq.pq_head->pNext->hdrSize == 200);
    VERIFY(ctx->pq.pq_start_offset == 30);
    VERIFY(range_start == range_start_should_be);
    VERIFY(range_end == range_end_should_be);
    VERIFY(ctx->pq.pq_len == 230);

    pkt_acpu_free_chain(ctx->pq.pq_head);

    bc(pqtest_make_pq(&ctx->pq, 2, (int []){100, 200}));

    pq_get_location(&ctx->pq, &loc, 30);

    pq_seek_to(&ctx->pq, 110);

    range_start_should_be = ctx->pq.pq_head->pNext->pHdr;
    range_end_should_be = ctx->pq.pq_head->pNext->pHdr + ctx->pq.pq_head->pNext->hdrSize;

    pq_delete_range(&ctx->pq,
                    &loc,
                    NULL,
                    80,
                    &range_start,
                    &range_end,
                    &move_by);

    VERIFY(ctx->pq.pq_head->hdrSize == 30);
    VERIFY(ctx->pq.pq_head->pNext->hdrSize == 190);
    VERIFY(ctx->pq.pq_start_offset == 30);
    VERIFY(range_start == range_start_should_be);
    VERIFY(range_end == range_end_should_be);
    VERIFY(ctx->pq.pq_len == 220);

    pkt_acpu_free_chain(ctx->pq.pq_head);

    bc(pqtest_make_pq(&ctx->pq, 2, (int []){100, 200}));

    pq_get_location(&ctx->pq, &loc, 30);

    pq_seek_to(&ctx->pq, 300);

    range_start_should_be = ctx->pq.pq_head->pNext->pHdr;
    range_end_should_be = ctx->pq.pq_head->pNext->pHdr + ctx->pq.pq_head->pNext->hdrSize;

    pq_delete_range(&ctx->pq,
                    &loc,
                    NULL,
                    270,
                    &range_start,
                    &range_end,
                    &move_by);

    VERIFY(ctx->pq.pq_head->hdrSize == 30);
    VERIFY(ctx->pq.pq_start_offset == 30);
    VERIFY(range_start == range_start_should_be);
    VERIFY(range_end == range_end_should_be);
    VERIFY(ctx->pq.pq_len == 30);

    pkt_acpu_free_chain(ctx->pq.pq_head);

    eee_ramDealloc(ctx);
    CALL_NEXT(creq.v, STATUS_SUCCESS);
}

NTSTATUS 
pqtest_1()
{
    struct pqtest_1_ctx *ctx;
    NTSTATUS status;
    if ((status = xalloc(&ctx, sizeof(*ctx))) == STATUS_SUCCESS) {
        req_push_state(creq.v, state0(0, pqtest_1_proc), ctx);
        status = pqtest_1_proc();
    }
    return status;
}

/* Request context for pqtest_2_proc() and pqtest_2_cifs_proc()
 */
struct pqtest_2_ctx {
    /* Our FCB.
     */
    fcb_t *fcb;

    /* Write context.
     */
    write_t w;

    /* Read context.
     */
    read_t r;

    /* Context for req_copy_dcaches(). The caller passes offset in the first
     * dcache block in c.nr_read_offset and length in c.nr_read_count.
     */
    struct req_copy_dcaches_ctx c;

    /* Packet queue to fill.
     */
    pkt_queue_t q;

    /* Context for ofh_close().
     */
    ofh_close_ctx_t close_ctx;

    /* Array of structures describing the dcaches configuration.
     */
    struct pqtest_2_dcache_descr {
        /* If TRUE, there is a hole in the file at this location.
         */
        boolean is_hole;

        /* How the buffers for this dcache should be allocated.
         */
        struct edesc_buffer_disposition buf_disposition;
    } dcache_descriptor[NFX_MAX_COD_MSG];

    /* Current block number being written/read.
     */
    int block_index;

    /* If true, verify that all packets except the last use the whole MTU 
     * size.  
     */
    boolean verify_mtu;

    /* If true, print the packet sizes.
     */
    boolean print_packets;

    /* Protocol to use.
     */
    int protocol;

    /* MTU to use.
     */
    int mtu;

    /* Save vs_max_unused_read_mtu if we are going to set it to 0 for testing.
     */
    uint32 save_vs_max_unused_read_mtu;

    /* Number of child requests completed.
     */
    int num_completed;

    /* Child requests
     */
    cifs_request_t *children[2];

    /* If TRUE, make the dcache busy before starting to copy.
     */
    boolean make_busy;

    /* End wait time.
     */
    eeeRTC_t end_wait;    
};


NTSTATUS
pqtest_2_cifs_proc()
{
    NTSTATUS status;
    struct pqtest_2_ctx *ctx = req_context(&testreq);

    dd();

    status = bc(topen(FILE_SUPERSEDE,
                      GENERIC_ALL,
                      FILE_NON_DIRECTORY_FILE,
                      FS_FTYPE_REG,
                      NULL,
                      "pqtest2.txt"));
    if (status == STATUS_SUCCESS) {
        ref_set(&testreq, rh_last_result, 
                fcb_hold(creq.cifs->cr_ofh->ofh_fns->fns_fcb));
    }
    testreq.rh_error_code = status;
    
    bc(ofh_close(creq.v, 
                 creq.cifs->cr_ofh, 
                 OFH_CLOSE_NORMAL, 
                 &ctx->close_ctx));

    req_wakeup(&testreq, REQ_PRIORITY_WAKEUP);

    return cifs_req_completed(creq.cifs);
}


NTSTATUS
pqtest_2_proc()
{
    struct pqtest_2_ctx *ctx = req_context(creq.v);
    cifs_request_t *cifs_req;
    eee_desc_t *edesc;
    pkt_desc_t *pkt;
    NTSTATUS status;

    dd();

    status = bc(find_writable_volume());
    if (status == STATUS_SUCCESS) {
        status = tcreate_cifs_req(&cifs_req, NULL, state0(0, pqtest_2_cifs_proc));
    }

    if (status == STATUS_SUCCESS) {
        status = bc(req_block(creq.v));
        if (status == STATUS_SUCCESS) {
            ref_set(ctx, fcb, ref_delz(creq.v, rh_last_result));
        }
    }

    if (status == STATUS_SUCCESS) {
        status = xalloc(&ctx->w.w_buf, EFS_BLOCK_SIZE);
    }
    
    /* Initialize the file data, write the blocks which are not holes.  
     */
    if (status == STATUS_SUCCESS) {
        int j;
        for (j = 0; j < EFS_BLOCK_SIZE; ++j) {
            ((char *)ctx->w.w_buf)[j] = j;
        }
        ctx->w.w_writeLen = EFS_BLOCK_SIZE;
        ctx->w.w_fcb = ctx->fcb;
        ctx->w.w_iFlags = W_I_FLAT_BUF_SUPPLIED | W_I_UNSTABLE;
        ctx->w.w_lockTypeHeld = 0;
        ctx->w.w_nCods = 0;
        ctx->w.w_ofh = NULL;

        for (ctx->w.w_fileOffset = ctx->c.nr_read_offset & ~(EFS_BLOCK_SIZE - 1),
             ctx->block_index = 0;
              (status == STATUS_SUCCESS) &&
              (ctx->w.w_fileOffset < 
               ((ctx->c.nr_read_offset + ctx->c.nr_read_count)));
              ctx->w.w_fileOffset += EFS_BLOCK_SIZE, 
                  ctx->block_index++) {
            if (!ctx->dcache_descriptor[ctx->block_index].is_hole) {
                status = bc(dcache_write(creq.v, &ctx->w));
            }
        }

        if (status == STATUS_SUCCESS) {
            /* Write a block past the last one to force the file extend in
             * case there were only holes at the end.
             */
            status = bc(dcache_write(creq.v, &ctx->w));
        }

        if (ctx->w.w_lockTypeHeld != 0) {
            fcb_lockRelease(creq, ctx->fcb);
        }
    }

    ctx->save_vs_max_unused_read_mtu = creq.v->rh_vs->vs_max_unused_read_mtu;
    creq.v->rh_vs->vs_max_unused_read_mtu = 0;

    /* Read back the data, initializing the nr_read_dcache_entries.  If a 
     * block is not a hole, replace the cods with cods of our configuration.  
     */
    if (status == STATUS_SUCCESS) {
        ctx->r.r_buf = NULL;
        ctx->r.r_bufSize = NULL;
        ctx->r.r_dcache = NULL;
        ctx->r.r_fcb = ctx->fcb;
        ctx->r.r_iFlags = R_I_DCACHE_REQUESTED;
        ctx->r.r_lastDcache = NULL;
        ctx->r.r_lockTypeHeld = 0;
        ctx->r.r_nBytes = EFS_BLOCK_SIZE;
        ctx->r.r_ofh = 0;

        for (ctx->r.r_fileOffset = ctx->c.nr_read_offset & ~(EFS_BLOCK_SIZE - 1),
             ctx->block_index = 0;
              (status == STATUS_SUCCESS) &&
              (ctx->r.r_fileOffset < 
               ((ctx->c.nr_read_offset + ctx->c.nr_read_count)));
              ctx->r.r_fileOffset += EFS_BLOCK_SIZE, 
                  ctx->block_index++) {
            status = bc(dcache_read(creq.v, &ctx->r));
            if (status == STATUS_SUCCESS) {
                if (!ctx->dcache_descriptor[ctx->block_index].is_hole) {
                    status = test_make_edesc(&edesc,
                                             &ctx->dcache_descriptor[ctx->block_index].buf_disposition,
                                             EEE_BUF_LARGE);
                    if (status == STATUS_SUCCESS) {
                        dcache_free_cod(ctx->r.r_dcache);
                        dcache_set_cod(ctx->r.r_dcache,
                                       sbm_getCODFromDesc(edesc));
                        sbm_sendCODToSac(ctx->r.r_dcache->dc_cod);
                    }
                }
                if (status == STATUS_SUCCESS) {
                    ref_del_holder(&ctx->r.r_dcaches[0], ctx->r.r_dcaches[0]);
                    ref_set(&ctx->c, nr_read_dcache_entries[ctx->c.nr_read_num_dcache_entries], 
                            ctx->r.r_dcaches[0]);
                    ctx->r.r_dcache = NULL;
                    ctx->c.nr_read_num_dcache_entries++;
                }
            }
        }
    }

    if (status == STATUS_SUCCESS) {
        creq.v->rh_conn = (acpu_conn_t *)nfs_createPseudoConn(creq.v->rh_vs);
        if (creq.v->rh_conn == NULL) {
            status = STATUS_INSUFF_SERVER_RESOURCES;
        }
    }

    /* Allocate the beginning of packet queue.
     */
    if (status == STATUS_SUCCESS) {
        creq.v->rh_conn->prot = ctx->protocol;
        tpl_connSetupMtu((tpl_connCb_t *)creq.v->rh_conn, ctx->mtu);
        bc(pq_alloc_packets(creq, &ctx->q, 40, RPC_ALLOC_PACKETS_CONTINUE));
        pq_adjust(&ctx->q, 40);
        status = STATUS_SUCCESS;
    }

    if (status == STATUS_SUCCESS) {
        status = bc(req_copy_dcaches(creq, &ctx->c, &ctx->q));
        VERIFY(status == STATUS_SUCCESS);
    }

    if ((status == STATUS_SUCCESS) && ctx->print_packets) {
        for (pkt = ctx->q.pq_head; pkt != NULL; pkt = pkt->pNext) {
            printf("hdr = %p hdrSize = %d\n", pkt->pHdr, pkt->hdrSize);
        }
    }

    if ((status == STATUS_SUCCESS) && ctx->verify_mtu) {
        for (pkt = ctx->q.pq_head; pkt != NULL; pkt = pkt->pNext) {
            if ((pkt != ctx->q.pq_tail)) {
                if (pkt->hdrSize != pkt_usable_len(creq.v->rh_conn,
                                                   pkt == ctx->q.pq_head)) {
                    status = STATUS_UNSUCCESSFUL;
                    break;
                }
            }
        }
    }

    if (ctx->fcb != NULL) {
        if (ctx->w.w_lockTypeHeld != 0) {
            fcb_lockRelease(creq, ctx->fcb);
        } else if (ctx->r.r_lockTypeHeld != 0) {
            fcb_lockRelease(creq, ctx->fcb);
        }
        fcb_rele(ref_delz(ctx, fcb));
    }

    req_free_dcache_entries(&ctx->c);

    if (!pq_is_empty(&ctx->q)) {
        pkt_acpu_free_chain(ctx->q.pq_head);
        pq_make_empty(&ctx->q);
    }

    if (ctx->w.w_buf != NULL) {
        eee_ramDealloc(ctx->w.w_buf);
    }

    creq.v->rh_vs->vs_max_unused_read_mtu = ctx->save_vs_max_unused_read_mtu;

    eee_ramDealloc(ctx);

    CALL_NEXT(creq.v, status);
}


int buf_size_array_1[] = {1460, 1460, 1460, 1460, 1460, 892};
int buf_size_array_2[] = {1792 - 96, 1792, 1792, 1792, 1120};
int buf_size_array_3[] = {1111, 1111, 1111, 1111, 1111, 1111, 1111, 415};
int buf_offsets_array_3[] = {1, 2, 3, 4, 5, 6, 7, 8};



/* Random data at 0 offset
 */
NTSTATUS
pqtest_2()
{
    struct pqtest_2_ctx *ctx;
    NTSTATUS status;
    status = xalloc(&ctx, sizeof(*ctx));
    if (status == STATUS_SUCCESS) {
        memset(ctx, 0, sizeof(*ctx));
        ctx->dcache_descriptor[0].buf_disposition.num_buffers = 
            sizeof(buf_size_array_1) / sizeof(int);
        ctx->dcache_descriptor[0].buf_disposition.buf_offsets = NULL;
        ctx->dcache_descriptor[0].buf_disposition.buf_sizes = buf_size_array_1;
        ctx->c.nr_read_count = EFS_BLOCK_SIZE;
        ctx->c.nr_read_offset = 0;
        ctx->protocol = TPL_PROT_UDP;
        ctx->mtu = 1480;
        ctx->verify_mtu = FALSE;
        pq_make_empty(&ctx->q);
        req_push_state(creq.v, state0(0, pqtest_2_proc), ctx);
        status = pqtest_2_proc();
    }

    return status;
}


/* Read from disk/compact dcache at 0 offset
 */
NTSTATUS
pqtest_3()
{
    struct pqtest_2_ctx *ctx;
    NTSTATUS status;
    status = xalloc(&ctx, sizeof(*ctx));
    if (status == STATUS_SUCCESS) {
        memset(ctx, 0, sizeof(*ctx));
        ctx->dcache_descriptor[0].buf_disposition.num_buffers = 
            sizeof(buf_size_array_1) / sizeof(int);
        ctx->dcache_descriptor[0].buf_disposition.buf_offsets = NULL;
        ctx->dcache_descriptor[0].buf_disposition.buf_sizes = buf_size_array_2;
        ctx->c.nr_read_count = EFS_BLOCK_SIZE;
        ctx->c.nr_read_offset = 0;
        ctx->protocol = TPL_PROT_UDP;
        ctx->mtu = 1480;
        ctx->verify_mtu = TRUE;
        pq_make_empty(&ctx->q);
        req_push_state(creq.v, state0(0, pqtest_2_proc), ctx);
        status = pqtest_2_proc();
    }

    return status;
}

/* Read from disk/compact dcache at non-zero offset 
 */
NTSTATUS
pqtest_4()
{
    struct pqtest_2_ctx *ctx;
    NTSTATUS status;
    status = xalloc(&ctx, sizeof(*ctx));
    if (status == STATUS_SUCCESS) {
        memset(ctx, 0, sizeof(*ctx));
        ctx->dcache_descriptor[0].buf_disposition.num_buffers = 
            sizeof(buf_size_array_1) / sizeof(int);
        ctx->dcache_descriptor[0].buf_disposition.buf_offsets = NULL;
        ctx->dcache_descriptor[0].buf_disposition.buf_sizes = buf_size_array_2;
        ctx->c.nr_read_count = EFS_BLOCK_SIZE - 733;
        ctx->c.nr_read_offset = 14;
        ctx->protocol = TPL_PROT_UDP;
        ctx->mtu = 1480;
        ctx->verify_mtu = TRUE;
        pq_make_empty(&ctx->q);
        req_push_state(creq.v, state0(0, pqtest_2_proc), ctx);
        status = pqtest_2_proc();
    }

    return status;
}


/* Data followed by hole
 */
NTSTATUS
pqtest_5()
{
    struct pqtest_2_ctx *ctx;
    NTSTATUS status;
    status = xalloc(&ctx, sizeof(*ctx));
    if (status == STATUS_SUCCESS) {
        memset(ctx, 0, sizeof(*ctx));
        ctx->dcache_descriptor[0].buf_disposition.num_buffers = 
            sizeof(buf_size_array_1) / sizeof(int);
        ctx->dcache_descriptor[0].buf_disposition.buf_offsets = NULL;
        ctx->dcache_descriptor[0].buf_disposition.buf_sizes = buf_size_array_2;
        ctx->dcache_descriptor[1].is_hole = TRUE;
        ctx->c.nr_read_count = EFS_BLOCK_SIZE + 3811;
        ctx->c.nr_read_offset = 44;
        ctx->protocol = TPL_PROT_UDP;
        ctx->mtu = 1480;
        ctx->verify_mtu = TRUE;
        ctx->print_packets = FALSE;
        pq_make_empty(&ctx->q);
        req_push_state(creq.v, state0(0, pqtest_2_proc), ctx);
        status = pqtest_2_proc();
    }

    return status;
}


/* Hole followed by data
 */
NTSTATUS
pqtest_6()
{
    struct pqtest_2_ctx *ctx;
    NTSTATUS status;
    status = xalloc(&ctx, sizeof(*ctx));
    if (status == STATUS_SUCCESS) {
        memset(ctx, 0, sizeof(*ctx));
        ctx->dcache_descriptor[0].is_hole = TRUE;
        ctx->dcache_descriptor[1].buf_disposition.num_buffers = 
            sizeof(buf_size_array_1) / sizeof(int);
        ctx->dcache_descriptor[1].buf_disposition.buf_offsets = NULL;
        ctx->dcache_descriptor[1].buf_disposition.buf_sizes = buf_size_array_2;
        ctx->c.nr_read_count = EFS_BLOCK_SIZE + 3811;
        ctx->c.nr_read_offset = 44;
        ctx->protocol = TPL_PROT_UDP;
        ctx->mtu = 1480;
        ctx->verify_mtu = TRUE;
        ctx->print_packets = FALSE;
        pq_make_empty(&ctx->q);
        req_push_state(creq.v, state0(0, pqtest_2_proc), ctx);
        status = pqtest_2_proc();
    }

    return status;
}


/* Check if compacting is invoked when needed.
 */
NTSTATUS
pqtest_7()
{
    struct pqtest_2_ctx *ctx;
    NTSTATUS status;
    status = xalloc(&ctx, sizeof(*ctx));
    if (status == STATUS_SUCCESS) {
        memset(ctx, 0, sizeof(*ctx));
        ctx->dcache_descriptor[0].buf_disposition.num_buffers = 
            sizeof(buf_size_array_3) / sizeof(int);
        ctx->dcache_descriptor[0].buf_disposition.buf_offsets = buf_offsets_array_3;
        ctx->dcache_descriptor[0].buf_disposition.buf_sizes = buf_size_array_3;
        ctx->c.nr_read_count = EFS_BLOCK_SIZE - 100;
        ctx->c.nr_read_offset = 44;
        ctx->protocol = TPL_PROT_UDP;
        ctx->mtu = 1480;
        ctx->verify_mtu = TRUE;
        ctx->print_packets = FALSE;
        pq_make_empty(&ctx->q);
        req_push_state(creq.v, state0(0, pqtest_2_proc), ctx);
        status = pqtest_2_proc();
    }

    return status;
}

NTSTATUS
pqtest_8_cifs_proc()
{
    NTSTATUS status;
    struct pqtest_2_ctx *ctx = req_context(&testreq);
    pkt_desc_t *pkt;

    dd();

    bc(pq_alloc_packets(creq,
                        &creq.cifs->cr_smb_response,
                        40,
                        RPC_ALLOC_PACKETS_CONTINUE));
    pq_adjust(&creq.cifs->cr_smb_response, 40);
    status = STATUS_SUCCESS;

    if (status == STATUS_SUCCESS) {
        status = bc(req_copy_dcaches(creq,
                                     &creq.cifs->read.cr_read_copy_ctx,
                                     &creq.cifs->cr_smb_response));
        VERIFY(status == STATUS_SUCCESS);
    }

    if ((status == STATUS_SUCCESS) && ctx->print_packets) {
        for (pkt = creq.cifs->cr_smb_response.pq_head; pkt != NULL; pkt = pkt->pNext) {
            printf("hdr = %p hdrSize = %d\n", pkt->pHdr, pkt->hdrSize);
        }
    }

    if ((status == STATUS_SUCCESS) && ctx->verify_mtu) {
        for (pkt = creq.cifs->cr_smb_response.pq_head; pkt != NULL; pkt = pkt->pNext) {
            if ((pkt != creq.cifs->cr_smb_response.pq_tail)) {
                if (pkt->hdrSize != pkt_usable_len(creq.v->rh_conn,
                                                   pkt == creq.cifs->cr_smb_response.pq_head)) {
                    status = STATUS_UNSUCCESSFUL;
                    break;
                }
            }
        }
    }

    req_free_dcache_entries(&creq.cifs->read.cr_read_copy_ctx);

    cifs_req_completed(creq.cifs);
    ctx->num_completed++;
    test_done(status);

    return STATUS_PENDING;
}

NTSTATUS
pqtest_8_proc()
{
    struct pqtest_2_ctx *ctx = req_context(creq.v);
    cifs_request_t *cifs_req;
    eee_desc_t *edesc;
    NTSTATUS status;
    int i;

    dd();

    status = bc(find_writable_volume());
    if (status == STATUS_SUCCESS) {
        status = tcreate_cifs_req(&cifs_req, NULL, state0(0, pqtest_2_cifs_proc));
    }

    if (status == STATUS_SUCCESS) {
        status = bc(req_block(creq.v));
        if (status == STATUS_SUCCESS) {
            ref_set(ctx, fcb, ref_delz(creq.v, rh_last_result));
        }
    }

    if (status == STATUS_SUCCESS) {
        status = xalloc(&ctx->w.w_buf, EFS_BLOCK_SIZE);
    }
    
    ctx->save_vs_max_unused_read_mtu = creq.v->rh_vs->vs_max_unused_read_mtu;
    creq.v->rh_vs->vs_max_unused_read_mtu = 0;

    /* Initialize the file data, write the blocks which are not holes.  
     */
    if (status == STATUS_SUCCESS) {
        int j;
        for (j = 0; j < EFS_BLOCK_SIZE; ++j) {
            ((char *)ctx->w.w_buf)[j] = j;
        }
        ctx->w.w_writeLen = EFS_BLOCK_SIZE;
        ctx->w.w_fcb = ctx->fcb;
        ctx->w.w_iFlags = W_I_FLAT_BUF_SUPPLIED | W_I_UNSTABLE;
        ctx->w.w_lockTypeHeld = 0;
        ctx->w.w_nCods = 0;
        ctx->w.w_ofh = NULL;

        for (ctx->w.w_fileOffset = ctx->c.nr_read_offset & ~(EFS_BLOCK_SIZE - 1),
             ctx->block_index = 0;
              (status == STATUS_SUCCESS) &&
              (ctx->w.w_fileOffset < 
               ((ctx->c.nr_read_offset + ctx->c.nr_read_count)));
              ctx->w.w_fileOffset += EFS_BLOCK_SIZE, 
                  ctx->block_index++) {
            if (!ctx->dcache_descriptor[ctx->block_index].is_hole) {
                status = bc(dcache_write(creq.v, &ctx->w));
            }
        }

        if (status == STATUS_SUCCESS) {
            /* Write a block past the last one to force the file extend in
             * case there were only holes at the end.
             */
            status = bc(dcache_write(creq.v, &ctx->w));
        }

        if (ctx->w.w_lockTypeHeld != 0) {
            fcb_lockRelease(creq, ctx->fcb);
        }
    }

    /* Read back the data, initializing the nr_read_dcache_entries.  If a 
     * block is not a hole, replace the cods with cods of our configuration.  
     */
    if (status == STATUS_SUCCESS) {
        ctx->r.r_buf = NULL;
        ctx->r.r_bufSize = NULL;
        ctx->r.r_dcache = NULL;
        ctx->r.r_fcb = ctx->fcb;
        ctx->r.r_iFlags = R_I_DCACHE_REQUESTED;
        ctx->r.r_lastDcache = NULL;
        ctx->r.r_lockTypeHeld = 0;
        ctx->r.r_nBytes = EFS_BLOCK_SIZE;
        ctx->r.r_ofh = 0;

        for (ctx->r.r_fileOffset = ctx->c.nr_read_offset & ~(EFS_BLOCK_SIZE - 1),
             ctx->block_index = 0;
              (status == STATUS_SUCCESS) &&
              (ctx->r.r_fileOffset < 
               ((ctx->c.nr_read_offset + ctx->c.nr_read_count)));
              ctx->r.r_fileOffset += EFS_BLOCK_SIZE, 
                  ctx->block_index++) {
            status = bc(dcache_read(creq.v, &ctx->r));
            if (status == STATUS_SUCCESS) {
                if (!ctx->dcache_descriptor[ctx->block_index].is_hole) {
                    status = test_make_edesc(&edesc,
                                             &ctx->dcache_descriptor[ctx->block_index].buf_disposition,
                                             EEE_BUF_LARGE);
                    if (status == STATUS_SUCCESS) {
                        dcache_free_cod(ctx->r.r_dcache);
                        dcache_set_cod(ctx->r.r_dcache,
                                       sbm_getCODFromDesc(edesc));
                        sbm_sendCODToSac(ctx->r.r_dcache->dc_cod);
                    }
                }
                if (status == STATUS_SUCCESS) {
                    ref_del_holder(&ctx->r.r_dcaches[0], ctx->r.r_dcaches[0]);
                    ref_set(&ctx->c, nr_read_dcache_entries[ctx->c.nr_read_num_dcache_entries], 
                            ctx->r.r_dcaches[0]);
                    ctx->r.r_dcache = NULL;
                    ctx->c.nr_read_num_dcache_entries++;
                }
            }
        }
    }

    if (status == STATUS_SUCCESS) {
        if (ctx->make_busy) {
            /* Make the dcache entry busy so the children will block trying
             * to do compact.
             */
            dcache_setBusy(ctx->c.nr_read_dcache_entries[0],
                           &ctx->c, ref_get_typeid(&ctx->c));
        }

        ctx->num_completed = 0;
        for (i = 0; (i < 2) && (status == STATUS_SUCCESS); ++i) {
            status = tcreate_cifs_req(&ctx->children[i],
                                      NULL,
                                      state0(0, pqtest_8_cifs_proc));
            if (status == STATUS_SUCCESS) {
                ctx->children[i]->rh_conn->prot = TPL_PROT_TCP;
                tpl_connSetupMtu((tpl_connCb_t *)ctx->children[i]->rh_conn,
                                 ctx->mtu);
                memcpy(&ctx->children[i]->read.cr_read_copy_ctx,
                       &ctx->c,
                       sizeof(ctx->c));
                int j;
                for (j = 0; j < ctx->c.nr_read_num_dcache_entries; ++j) {
                    ref_set(&ctx->children[i]->read.cr_read_copy_ctx,
                            nr_read_dcache_entries[j],
                            dcache_hold(ctx->c.nr_read_dcache_entries[j]));
                }
                req_add_to_active_queue(ctx->children[i], REQ_PRIORITY_WAKEUP);
            }
        }
    }

    if (status == STATUS_SUCCESS) {

        if (ctx->make_busy) {
            /* Wait until all children become blocked.
             */

            while (1) {
                int num_blocked = 0;
                for (i = 0; i < 2; ++i) {
                    if (ctx->children[i]->rh_flags & REQ_BLOCKED) {
                        ++num_blocked;
                    }
                }
                if (num_blocked < 2) {
                    bc(req_yield(creq.v, REQ_PRIORITY_NEW));
                    status = STATUS_SUCCESS;
                } else {
                    break;
                }
            }

            eeeRTC_t now;
            eee_rtc(&now);
            now.val += 500000;

            while (1) {
                bc(req_yield(creq.v, REQ_PRIORITY_NEW));
                status = STATUS_SUCCESS;
                eee_rtc(&now);
                if (eee_rtcCompare(&now, &ctx->end_wait) > 0) {
                    break;
                }
            }

            /* Release the dcache entry that we have made busy.
             */
            dcache_decBusy(ctx->c.nr_read_dcache_entries[0], &ctx->c);

        }
        /* The children now should be able to complete.
         */
        while ((ctx->num_completed < 2) && (status == STATUS_SUCCESS)) {
            status = bc(req_block(creq.v));
        }
    }

    if (ctx->fcb != NULL) {
        if (ctx->w.w_lockTypeHeld != 0) {
            fcb_lockRelease(creq, ctx->fcb);
        } else if (ctx->r.r_lockTypeHeld != 0) {
            fcb_lockRelease(creq, ctx->fcb);
        }
        fcb_rele(ref_delz(ctx, fcb));
    }

    req_free_dcache_entries(&ctx->c);

    if (!pq_is_empty(&ctx->q)) {
        pkt_acpu_free_chain(ctx->q.pq_head);
        pq_make_empty(&ctx->q);
    }

    if (ctx->w.w_buf != NULL) {
        eee_ramDealloc(ctx->w.w_buf);
    }

    creq.v->rh_vs->vs_max_unused_read_mtu = ctx->save_vs_max_unused_read_mtu;

    eee_ramDealloc(ctx);

    CALL_NEXT(creq.v, status);
}

/* Multiple requests trying to compact the same dcache which is busy
 */
NTSTATUS
pqtest_8()
{
    struct pqtest_2_ctx *ctx;
    NTSTATUS status;

    status = xalloc(&ctx, sizeof(*ctx));
    if (status == STATUS_SUCCESS) {
        memset(ctx, 0, sizeof(*ctx));
        ctx->dcache_descriptor[0].buf_disposition.num_buffers = 
            sizeof(buf_size_array_3) / sizeof(int);
        ctx->dcache_descriptor[0].buf_disposition.buf_offsets = buf_offsets_array_3;
        ctx->dcache_descriptor[0].buf_disposition.buf_sizes = buf_size_array_3;
        ctx->c.nr_read_count = EFS_BLOCK_SIZE - 100;
        ctx->c.nr_read_offset = 44;
        ctx->protocol = TPL_PROT_TCP;
        ctx->mtu = 1460;
        ctx->verify_mtu = TRUE;
        ctx->print_packets = FALSE;
        ctx->make_busy = TRUE;
        pq_make_empty(&ctx->q);
        req_push_state(creq.v, state0(0, pqtest_8_proc), ctx);
        status = pqtest_8_proc();
    }

    return status;
}


/* Multiple requests trying to read same dcache range, second dcache requires
 * compaction.
 */
NTSTATUS
pqtest_9()
{
    struct pqtest_2_ctx *ctx;
    NTSTATUS status;

    status = xalloc(&ctx, sizeof(*ctx));
    if (status == STATUS_SUCCESS) {
        memset(ctx, 0, sizeof(*ctx));
        ctx->dcache_descriptor[0].buf_disposition.num_buffers = 
            sizeof(buf_size_array_2) / sizeof(int);
        ctx->dcache_descriptor[0].buf_disposition.buf_offsets = NULL;
        ctx->dcache_descriptor[0].buf_disposition.buf_sizes = buf_size_array_2;
        ctx->dcache_descriptor[1].buf_disposition.num_buffers = 
            sizeof(buf_size_array_3) / sizeof(int);
        ctx->dcache_descriptor[1].buf_disposition.buf_offsets = buf_offsets_array_3;
        ctx->dcache_descriptor[1].buf_disposition.buf_sizes = buf_size_array_3;
        ctx->c.nr_read_count = EFS_BLOCK_SIZE + EFS_BLOCK_SIZE - 100;
        ctx->c.nr_read_offset = 44;
        ctx->protocol = TPL_PROT_TCP;
        ctx->mtu = 1460;
        ctx->verify_mtu = TRUE;
        ctx->print_packets = FALSE;
        ctx->make_busy = FALSE;
        pq_make_empty(&ctx->q);
        req_push_state(creq.v, state0(0, pqtest_8_proc), ctx);
        status = pqtest_8_proc();
    }

    return status;
}


/* Test the copy from one packet to another.
 */
NTSTATUS
pqtest_10()
{
#define PQTEST_10_SRC_SIZE 7
#define PQTEST_10_DEST_SIZE 7
#define PQTEST_10_MAX_BUF_SIZE 50
#define PQTEST_10_TESTS 3
#define PQTEST_10_MAX_COPY_SIZE 100
    char compareBuf1[PQTEST_10_MAX_COPY_SIZE];
    char compareBuf2[PQTEST_10_MAX_COPY_SIZE];
    struct pqtest_10_test {
        struct {
            char buf[PQTEST_10_MAX_BUF_SIZE];
            uint32 size;
            pkt_desc_t pkt;
        } src[PQTEST_10_SRC_SIZE];
        struct {
            char buf[PQTEST_10_MAX_BUF_SIZE];
            uint32 size;
            pkt_desc_t pkt;
        } dest[PQTEST_10_DEST_SIZE];
        uint32 srcPktOff;
        uint32 destPktOff;
        uint32 copySize;
    } in[PQTEST_10_TESTS];
    int j;

    in[0].src[0].size = 0;
    in[0].src[1].size = 10;
    in[0].src[2].size = 20;
    in[0].src[3].size = 0;
    in[0].src[4].size = 0;
    in[0].src[5].size = 10;
    in[0].src[6].size = 0;
    in[0].srcPktOff = 0;
    in[0].dest[0].size = 0;
    in[0].dest[1].size = 10;
    in[0].dest[2].size = 20;
    in[0].dest[3].size = 0;
    in[0].dest[4].size = 0;
    in[0].dest[5].size = 10;
    in[0].dest[6].size = 0;
    in[0].destPktOff = 0;
    in[0].copySize = 40;

    in[1].src[0].size = 0;
    in[1].src[1].size = 10;
    in[1].src[2].size = 20;
    in[1].src[3].size = 0;
    in[1].src[4].size = 0;
    in[1].src[5].size = 10;
    in[1].src[6].size = 0;
    in[1].srcPktOff = 0;
    in[1].dest[0].size = 50;
    in[1].dest[1].size = 1;
    in[1].dest[2].size = 2;
    in[1].dest[3].size = 2;
    in[1].dest[4].size = 30;
    in[1].dest[5].size = 10;
    in[1].dest[6].size = 10;
    in[1].destPktOff = 15;
    in[1].copySize = 40;

    in[2].src[0].size = 15;
    in[2].src[1].size = 1;
    in[2].src[2].size = 2;
    in[2].src[3].size = 2;
    in[2].src[4].size = 0;
    in[2].src[5].size = 10;
    in[2].src[6].size = 10;
    in[2].srcPktOff = 15;
    in[2].dest[0].size = 15;
    in[2].dest[1].size = 0;
    in[2].dest[2].size = 0;
    in[2].dest[3].size = 20;
    in[2].dest[4].size = 5;
    in[2].dest[5].size = 0;
    in[2].dest[6].size = 0;
    in[2].destPktOff = 15;
    in[2].copySize = 25;

    int k;
    for (k = 0; k < PQTEST_10_TESTS; ++k) {
        pkt_desc_t *itr;
        int offset = 0;

        for (j=0; j<PQTEST_10_SRC_SIZE; ++j) {
            itr = &in[k].src[j].pkt;
            bzero(itr, sizeof(*itr));
            itr->pHdr = in[k].src[j].buf;
            itr->hdrSize = in[k].src[j].size;
            itr->num_bufs = 1;
            ASSERT((offset + itr->hdrSize) <= PQTEST_10_MAX_COPY_SIZE);
            if (itr->hdrSize > 0) {
                memset(itr->pHdr, j + 1, itr->hdrSize);
                offset += itr->hdrSize;
            }
            if (j < (PQTEST_10_SRC_SIZE - 1)) {
                itr->pNext = &in[k].src[j + 1].pkt;
            }
        }
        ASSERT(offset >= in[k].copySize);
        pq_copydata(&in[k].src[0].pkt, in[k].srcPktOff, compareBuf1,
                    in[k].copySize, NULL, NULL);
        offset = 0;
        for (j=0; j<PQTEST_10_DEST_SIZE; ++j) {
            itr = &in[k].dest[j].pkt;
            bzero(itr, sizeof(*itr));
            itr->pHdr = in[k].dest[j].buf;
            itr->hdrSize = in[k].dest[j].size;
            itr->num_bufs = 1;
            if (itr->hdrSize > 0) {
                memset(itr->pHdr, 0xbaddcafe, PQTEST_10_MAX_BUF_SIZE);
                offset += itr->hdrSize;
            }
            if (j < (PQTEST_10_DEST_SIZE - 1)) {
                itr->pNext = &in[k].dest[j + 1].pkt;
            }
        }
        ASSERT(offset >= in[k].copySize);

        pq_copyPkt(&in[k].src[0].pkt,
                   in[k].srcPktOff,
                   in[k].copySize,
                   &in[k].dest[0].pkt,
                   in[k].destPktOff);

        pq_copydata(&in[k].dest[0].pkt, in[k].destPktOff, compareBuf2,
                    in[k].copySize, NULL, NULL);
        if (memcmp(compareBuf2, compareBuf1, in[k].copySize) != 0) {
            printf("pqtest_10: test %d failed\n", k);
            return STATUS_UNSUCCESSFUL;
        }
    }

    return STATUS_SUCCESS;
}


struct pqtest_11_ctx
{
    tpl_bindCb_t *b1;
    tpl_bindCb_t *b2;
    NTSTATUS status;
};

int32
pqtest_11_tplAddConnInd(tpl_bindCb_t *bind_cb,
                        void *conn_id,
                        void **app_handle,
                        uint32 raddr unused__,
                        uint16 rport unused__)
{
    return NFX_OK;
}

int32
pqtest_11_tplDelConnInd(void *app_handle,
                        void *conn_id unused__)
{
    return NFX_OK;
}

void
pqtest_11_tplRcvPktInd(void *appHdlp,
                       pkt_desc_t *pkt,
                       void *pHdr, 
                       uint64 tval)
{
}

int32
pqtest_11_tplBindReqAck(void *app_handle unused__,
                  int32 rc,
                  void *pbind)
{
    struct pqtest_11_ctx *ctx = testreq.rh_context;
    if (rc == 0) {
        testreq.rh_error_code = STATUS_SUCCESS;
        if (ctx->b1 == NULL) {
            ctx->b1 = pbind;
        } else {
            ctx->b2 = pbind;
        }
    } else {
        testreq.rh_error_code = STATUS_UNSUCCESSFUL;
    }
    req_wakeup(&testreq, REQ_PRIORITY_WAKEUP);
    return NFX_OK;
}


int32
pqtest_11_tplunbindAck(int32 rc unused__,
                 uint32 addr,
                 uint16 port,
                 uint16 proto,
                 void *info,
                 void *bind unused__)
{
    return NFX_OK;
}

NTSTATUS
pqtest_11_bind()
{
    tpl_bindCb_t bindCB;
    tpl_bindCb_t *bindCBp;
    NTSTATUS status;
    int rc;

    memset(&bindCB, 0, sizeof(tpl_bindCb_t));
    bindCBp = &bindCB;
    bindCBp->laddr = test_pick_ip();
    bindCBp->lport = 3000;
    bindCBp->tplProt = TPL_PROT_UDP;
    bindCBp->vsId = creq.v->rh_vs->vs_id;
    bindCBp->ul_addConnInd = pqtest_11_tplAddConnInd;
    bindCBp->ul_delConnInd = pqtest_11_tplDelConnInd;
    bindCBp->ul_rcvPktInd = pqtest_11_tplRcvPktInd;
    bindCBp->ul_bindReqAck = pqtest_11_tplBindReqAck;
    bindCBp->ul_unbindReqAck = pqtest_11_tplunbindAck;
    bindCBp->usrInfo = NULL;

    rc = tpl_bindReq(bindCBp, NULL, TPL_BIND_SERVER | TPL_BIND_USE_RPC);
    status = STATUS_UNSUCCESSFUL;
    if (rc == NFX_OK) {
        status = STATUS_SUCCESS;
    }

    return status;
}


NTSTATUS
pqtest_11_proc()
{
    NTSTATUS status;
    struct pqtest_11_ctx *ctx = req_context(creq.v);

    dd();

    status = pqtest_11_bind();

    if (status == STATUS_SUCCESS) {
        status = bc(req_block(creq.v));
    }

    if (status == STATUS_SUCCESS) {
        status = pqtest_11_bind();
    }

    if (status == STATUS_SUCCESS) {
        status = bc(req_block(creq.v));
        if (status == STATUS_SUCCESS) {
            printf("%s: succeeded in opening duplicate connection\n", 
                   __func__);
            status = STATUS_UNSUCCESSFUL;
        } else {
            status = STATUS_SUCCESS;
        }
    }

    if (ctx->b1 != NULL) {
        tpl_unbindReq(ctx->b1);
    }
    if (ctx->b2 != NULL) {
        tpl_unbindReq(ctx->b2);
    }

    CALL_NEXT(creq.v, status);
}


NTSTATUS
pqtest_11()
{
    struct pqtest_11_ctx *ctx;
    NTSTATUS status = xalloc(&ctx, sizeof(*ctx));
    if (status == STATUS_SUCCESS) {
        memset(ctx, 0, sizeof(*ctx));
        req_push_state(creq.v, state0(0, pqtest_11_proc), ctx);
        status = pqtest_11_proc();
    }
    return status;
}


NTSTATUS
pqtest_12()
{
    pkt_queue_t pq;
    int len;
    pkt_desc_t *pkt;
    NTSTATUS status;
    pq_make_empty(&pq);
    pq_alloc_test(&pq, 5000, 0, NULL, NULL);
    pkt_AdjPkt(pq.pq_head, -4000);
    for (pkt = pq.pq_head, len = 0; pkt != NULL; pkt = pkt->pNext) {
        len += pkt->hdrSize;
    }
    if (len == 1000) {
        status = STATUS_SUCCESS;
    } else {
        printf("%s: length mismatch\n", __func__);
        status = STATUS_UNSUCCESSFUL;
    }
    pkt_freePkt(pq.pq_head);
    return status;
}


testfunc_t pqtests[] = {
    {find_writable_volume, "pqtest init"},
    {pqtest_1, "pqtest_1"},
    {pqtest_2, "pqtest_2"},
    {pqtest_3, "pqtest_3"},
    {pqtest_4, "pqtest_4"},
    {pqtest_5, "pqtest_5"},
    {pqtest_6, "pqtest_6"},
    {pqtest_7, "pqtest_7"},
    {pqtest_8, "pqtest_8"},
    {pqtest_9, "pqtest_9"},
    {pqtest_10, "pqtest_10"},
    {pqtest_11, "pqtest_11"},
    {pqtest_12, "pqtest_12"},
    {test_success, "pqtest uninit"},
    {NULL, ""}
};

testfunc_t *pqtests_array[] = {
    pqtests,
    NULL
};


CLIFUNC(pqtest)
{
    return runtests(pqtests_array, argc, (char **)argv);
}

pkt_queue_t eatmem_pq;

CLIFUNC(pq_eatmem)
{
    pkt_desc_t *pkt;
    char *endp;
    int i;
    int packets_left = strtol(argv[1], &endp, 0);
    if (*endp != 0) {
        printf("%s is not a number\n", argv[1]);
        return -1;
    }
    if (packets_left < 0) {
        printf("%s must be positive number\n", argv[1]);
        return -1;
    }
    
    pq_make_empty(&eatmem_pq);
    while ((pkt = pkt_alloc_from_list(&pkt_acpu_free_list, 1)) != NULL) {
        eatmem_pq.pq_tail->pNext = pkt;
        eatmem_pq.pq_tail = pkt;
    }

    for (i = 0;
         (i < packets_left) && (eatmem_pq.pq_head != NULL); ++i) {
        pkt = eatmem_pq.pq_head;
        eatmem_pq.pq_head = pkt->pNext;
        pkt->pNext = NULL;
        if (eatmem_pq.pq_head == NULL) {
            pq_make_empty(&eatmem_pq);
        }
        pkt_free(pkt);
    }
    
    return 0;
}

