/*
 *--------------------------------------------------------------------------
 *
 * Name: tpl-fp.h
 *
 * Copyright (C) 2004, ONStor, Inc.
 *
 * Description: Transport Layer Connection information on FP
 *
 * Created by:	Nitin Bahadur
 *
 * Date Created: 07/20/2004
 *
 *--------------------------------------------------------------------------
 */

#ifndef __TPL_FP_H
#define __TPL_FP_H

#ifdef LATER
#include "nfx-incl.h"
#include "nfx-types.h"
#include "nfx-defs.h"
#include "nfx-error.h"
#include "../sm-eee/eee-api.h"
#include "../sm-timer/timer-api.h"
#include "../sm-cifs/win-compat.h"
#include "../sm-elog/elog-api.h"
#include "../sm-cifs/ntstatus.h"
#include "../sm-pkt/pkt.h"
#include "../sm-esm/esm-api.h"
#endif


/* Maximum data to fill in a eee_descBuf_t
 * Set to 8k to match filesystem block size.
 */
#define TPL_MAX_BUF_LEN         (EEE_BUFSIZE_ULTRA-512)

/* Leave some space in the buffer for use by TPL/pkt_desc headers
 */
#define TPL_BUF_HDR_LEN         PKT_BUFF_INDENT

/* Default mtu size ... used as a hint for buffer allocation
 */
#define TPL_DEFAULT_MTU			1456

/* TPL connection states
 */
enum tpl_fp_conn_state
{
    /* tpl_close() response has bee received and connection is now closed
     */
    TPL_FP_CONN_CLOSED,
            
    /* The connection is opened
     */
    TPL_FP_CONN_OPENED,

    /* The socket is listening for the requests
     */
    TPL_FP_CONN_LISTEN,

    /* tpl_connect() has been called and we are waiting for TXRX response
     */
    TPL_FP_CONN_CONNECTING,

    /* The connection has been successfully established
     */
    TPL_FP_CONN_CONNECTED,

    /* Peer has closed the connection
     */
    TPL_FP_CONN_PEER_CLOSED,

    /* tpl_close() has been called
     */
    TPL_FP_CONN_CLOSING,
};


/* Types of TPL_FP queues
 */
typedef enum {
    TPL_FP_INPUT_QUEUE,
    TPL_FP_OUTPUT_QUEUE,
} tpl_fp_queue_type;


/* Input/output queue for communication between the TXRX and FP.
 * The queue allows moving data in one direction, from FP to TxRx
 * for write queue and from TxRx to FP for read queue
 * The queues are typical ring queues. The queue is empty when
 * the read index equals the write index. When the queue is full,
 * the write index will immediately preceed the read index.
 * Note that since the write index is not allowed to catch the read index,
 * when the queue is full, the queue will actually contain max_buffers-1
 * filled buffers. 
 */
typedef struct tpl_fp_queue
{
    /* Current read index
     */
    int32 read_idx;

    /* Current write index
     */
    int32 write_idx;

    /* Current local read index
     */
    int32 local_read_idx;

    /* Current local write index
     */
    int32 local_write_idx;
    
    tpl_fp_queue_type type;

    /* number of bytes read from the queue
     */
    uint64 num_bytes_read;

    /* number of bytes written in the queue
     */
    uint64 num_bytes_written;

    /* number of possibly unacked bytes read in the queue via datamover
     */
    uint64 local_bytes_read;

    /* number of possibly unacked bytes written in the queue via datamover
     */
    uint64 local_bytes_written;

    /* Offset from the beginning of the current read buffer
     */
    uint32 cur_read_offset;

    /* Offset from the beginning of the current write buffer
     */
    uint32 cur_write_offset;

    /* Circular array used for communication of the data between FP and TxRx.
     * The array is preallocated when the connection structure is created.
     * The array entries are pointers to EEE buffers
     * The allocated array address is stored in physical address format
     */
    void **buffers;

    /* Maximum number of bytes that should be placed in the array
     */
    uint32 max_bytes;

    /* Maximum number of buffers that the bytes should occupy
     */
    uint32 max_buffers;

    /* Max valid data we will put in 1 ethernet packet. We choose to write
     * data at 8-byte boundary for easier pointer manipualtion.
     */
    uint32 mtu_size;

    /* Spinlock protecting the queue
     */
#ifdef LATER
    nfx_spinlock_t spin;
#else
    spinlock_t spin;
#endif /* LATER */
    
} tpl_fp_queue_t;


/* Types of callback functions on the FP
 */
enum tpl_fp_callback_type {
    TPL_FP_OPEN_CALLBACK = 1,
    TPL_FP_CLOSE_CALLBACK,
    TPL_FP_READ_CALLBACK,
    TPL_FP_WRITE_CALLBACK,
    TPL_FP_LISTEN_CALLBACK,
    TPL_FP_ACCEPT_CALLBACK,
    TPL_FP_CONNECT_CALLBACK
};


/* Structure describing the pending request on the connection
 */
typedef struct tpl_fp_callback {
    /* Callback type
     */
    enum tpl_fp_callback_type type;

    /* Callback argument
     */
    void *arg;

    /* Callback function
     */
    void *callback;

    /* Entry in the list of callbacks for the connection
     */
    LIST_ENTRY(tpl_fp_callback) next;
} tpl_fp_callback_t;


/* Structure for maintaining statistics about a TPL connection
 */
typedef struct tpl_fp_conn_stats {
    /* Number of accepts issued on this connection
     */
    uint64      num_accepts;	

    /* Number of of user read requests
     */
    uint64      num_user_reads;

    /* Number of user write requests
     */
    uint64      num_user_writes;

    /* Number of tpl read issued
     */
    uint64      num_tpl_reads;

    /* Number of tpl writes issued
     */
    uint64      num_tpl_writes;
    
} tpl_fp_conn_stats_t;


/* Structure describing the TPL connection
 */
typedef struct tpl_fp_conn {
    /* Connection state
     */
    enum tpl_fp_conn_state state;

    /* Virtual server id
     */
    vs_id_t vs_id;
 
    /* Spinlock protecting the connection structure
     */
#ifdef LATER
       nfx_spinlock_t spin;
#else
        spinlock_t spin;
#endif

	/* local IP address
	 */
	uint32 laddr;

	/* remote IP address
	 */
	uint32 raddr;
	
	/* local port
	 */
	ushort16 lport;

	/* remote port
	 */
	ushort16 rport;	

    /* Protocol, UDP or TCP
     */
    uint8 protocol;
    
    /* If set, then the connection is released on a close
     */
    boolean release_on_close;
    
	/* time when this tpl connection got connected
	 */
#ifdef LATER
	eeeRTC_t when_connected;
#endif /* LATER */
	
	/* duration in secs for which this tpl connection was in
	 * CONNECTED state
	 */
	uint64 connect_time;

    /* timer for synchronous reads
     */
    int32 read_timer;

    /* timer for synchronous writes
     */
    int32 write_timer;

    /* Pointer to TPL connection on TxRx
     * In case of a "opened" connection...this is a pointer to tpl_bindCb_t
     * In case of a "connected" connection...this is a pointer to tpl_connCb_t
     */
    void *tpl_txrx_cb;

    /* connection created while waiting for an accept response
     */
    struct tpl_fp_conn *accept_conn; 

    /* Input queue. Data is added to this queue by TPL on TxRx and
     * retrieved by calling tpl_read()
     */
    tpl_fp_queue_t in_queue;

    /* Output queue. Data is added to this queue on FP by calling
     * tpl_write() and retrieved by TPL on TxRX and moved to the socket
     * output buffer when space becomes available
     */
    tpl_fp_queue_t out_queue;

    /* Connection statistics
     */
    tpl_fp_conn_stats_t fp_stats;

    /* state machine handle for this connection
     */
    // esm_handle_t sm_handle;

    /* Indicates if we need to send a write request to TxRx on data-mover
     * operation completion when writing to the output queue.
     */
    boolean send_write_req;

	/* Entry in the global list of open connections. This list is used when
     * we want to close the connection by virtual server id
     */
    LIST_ENTRY(tpl_fp_conn) next;

    /* List of pending requests to the connection
     */
    // LIST_HEAD(, tpl_fp_callback) callbacks;
    SLIST_HEAD(, tpl_fp_callback) callbacks;

} tpl_fp_conn_t;


#if defined (NFP_FP)

typedef struct tpl_fp_stats {
    /* Number of open tpl connections
     */
    uint32      num_tpl_conns;

    /* padding
     */
    uint32      pad;

    /* Number of tpl opens issued
     */
    uint64      num_tpl_open;

    /* Number of tpl closes issued
     */
    uint64      num_tpl_close;

    /* Number of tpl listens issued
     */
    uint64      num_tpl_listen;

    /* Number of tpl accepts issued
     */
    uint64      num_tpl_accept;

    /* Number of tpl connects issued
     */
    uint64      num_tpl_connect;

    /* Number of tpl reads issued
     */
    uint64      num_tpl_reads;

    /* Number of tpl writes issued
     */
    uint64      num_tpl_writes;

    /* Number of data-mover ops for read
     */
    uint64      num_dm_reads;

    /* Number of data-mover ops for write
     */
    uint64      num_dm_writes;

    /* Number of data-mover read write operation callbacks
     */
    uint64      num_dm_read_callbacks;

    /* Number of data-mover write operation callbacks
     */
    uint64      num_dm_write_callbacks;
} tpl_fp_stats_t;


/* Structure for storing tpl state information about connections
 */
typedef struct tpl_context {
    /* tpl application id
     */
    int32    tpl_app_id;

    /* tpl source address (FP) for sending packets
     */
    uint32   tpl_fp_addr;

    /* tpl destination address (TxRx0) for sending packets
     */
    uint32   tpl_txrx_addr;

    /* lock for protecting this structure
     */
#ifdef LATER
    nfx_spinlock_t  spin;
#else
    spinlock_t  spin;
#endif
    
    /* data mover channel id
     */
    int dm_channel_id[SMP_MAX_CPUS];

    /* tpl fp statistics
     */
    tpl_fp_stats_t  stats;
    
    /* tpl fp error statistics
     */
    tpl_fp_stats_t  err_stats;
    
    /* list of tpl connections
     */
    LIST_HEAD(, tpl_fp_conn) conns;
} tpl_context_t;


/* Structure for storing context during a listen
 */
typedef struct {
    tpl_fp_conn_t   **conn;         /* address of tpl connection */
    void            *callback_arg;  /* argument of user callback function */
} tpl_listen_context_t;

/* Local Variables and Defines */

#define TPL_RELEASE_TIMEOUT     (2*60)  /* 2 minutes */

extern tpl_context_t tpl_context;

/* Local Functions */

/*-----------------------------------------------------------------
 * Name:         tpl_fp_init
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Initializes TPL on FP
 *-----------------------------------------------------------------
 */
void tpl_fp_init(void);

/*-----------------------------------------------------------------
 * Name:         tpl_receive_msg
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Processes messages received from TxRx
 *
 * Arguments:
 *  edesc - incoming message
 *-----------------------------------------------------------------
 */
void tpl_receive_msg(eee_desc_t *edesc);

/*-----------------------------------------------------------------
 * Name:         tpl_get_new_conn
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Allocates a new FP TPL Connection
 *
 * Arguments:
 *  vs_id - virtual server id
 *
 * Return Values:
 *  tpl_fp_conn_t* if a new connection could be allocated
 *  NULL otherwise
 *-----------------------------------------------------------------
 */
tpl_fp_conn_t * tpl_get_new_conn(vs_id_t vs_id);

/*-----------------------------------------------------------------
 * Name:         tpl_release_conn
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Releases a FP TPL Connection and frees all memory
 *
 * Arguments:
 *  conn - TPL connection to release
 *-----------------------------------------------------------------
 */
void tpl_release_conn(tpl_fp_conn_t *conn);


/*-----------------------------------------------------------------------
 * Name:         tpl_release_timeout_handler
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/21/04
 *
 * Description:  Handler for handling release timeout. Frees the tpl memory
 *
 * Arguments:
 *  timer_handle - timer handler of expired timer
 *  usr_p1       - user private 1 (tpl_fp_conn_t*)
 *  usr_p2       - user private 2 (not used)
 *  usr_p3       - user private 3 (not used)
 *  usr_p4       - user private 4 (not used)
 *
 * Return Value:
 *	NFX_OK
 *------------------------------------------------------------------------
 */
int32 tpl_release_timeout_handler(uint32 timer_handle, 
                                  void *usr_p1, 
                                  void *usr_p2 unused__, 
                                  void *usr_p3 unused__, 
                                  void *usr_p4 unused__);
                                 

/*-----------------------------------------------------------------------
 * Name:         tpl_save_callback
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Saves the callback function details for a TPL Connection
 *
 * Arguments:
 *  conn     - tpl connection
 *  type     - type of callback
 *  callback - callback function
 *  arg      - argument associated with the callback
 *
 * Return Values:
 *  NFX_OK if callback could be saved
 *  NFX_ERR otherwise
 *------------------------------------------------------------------------
 */
int tpl_save_callback(tpl_fp_conn_t *conn,
                      enum tpl_fp_callback_type type,
                      void *callback,
                      void *arg);

/*-----------------------------------------------------------------------
 * Name:         tpl_delete_callback
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/20/04
 *
 * Description:  Deletes the callback function for a TPL Connection
 *               and releases allocated memory
 *
 * Arguments:
 *  conn     - tpl connection
 *  type     - type of callback
 *  arg      - argument associated with the callback
 *------------------------------------------------------------------------
 */
void tpl_delete_callback(tpl_fp_conn_t *conn,
                         enum tpl_fp_callback_type type);


/*-----------------------------------------------------------------------
 * Name:         tpl_find_callback
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/22/04
 *
 * Description:  Finds the callback function for a TPL Connection and if
 *               specified removes it from the connection
 *
 * Arguments:
 *  conn     - tpl connection
 *  type     - type of callback
 *  delete   - if TRUE, then remove the callback from the connection
 *             callback list
 *------------------------------------------------------------------------
 */
tpl_fp_callback_t* tpl_find_callback(tpl_fp_conn_t *conn,
                                     enum tpl_fp_callback_type type,
									 boolean delete);


/*-----------------------------------------------------------------------
 * Name:         tpl_start_timer
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/21/04
 *
 * Description:  Starts a Timer
 *
 * Arguments:
 *  timer_handle - timer handler of timer created is returned here
 *  timeout      - timeout in seconds
 *  timeout_func - function for handling timer expiration
 *  arg          - argument to timeout handler
 *
 * Return Values:
 *  NFX_OK  - timer was successfully started
 *  Other errors codes returned by timer creation routines
 *------------------------------------------------------------------------
 */
int tpl_start_timer(int32 *timer_handle, 
                    uint32 timeout,
                    timer_callback_func timeout_func,
                    void *arg);


/*-----------------------------------------------------------------------
 * Name:         tpl_stop_timer
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/21/04
 *
 * Description:  Stops a Timer 
 *
 * Arguments:
 *  timer_handle - timer handler of timer to stop
 *------------------------------------------------------------------------
 */
void tpl_stop_timer(int32 *timer_handle);


/*-----------------------------------------------------------------------
 * Name:         tpl_io_timeout_handler
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/21/04
 *
 * Description:  Handler for handling I/O timeouts
 *
 * Arguments:
 *  timer_handle - timer handler of expired timer
 *  usr_p1       - user private 1 (tpl_fp_conn_t*)
 *  usr_p2       - user private 2 (not used)
 *  usr_p3       - user private 3 (not used)
 *  usr_p4       - user private 4 (not used)
 *
 * Return Value:
 *	NFX_OK
 *------------------------------------------------------------------------
 */
int32 tpl_io_timeout_handler(uint32 timer_handle, 
                             void *usr_p1, 
                             void *usr_p2 unused__, 
                             void *usr_p3 unused__, 
                             void *usr_p4 unused__);

/*-----------------------------------------------------------------------
 * Name:         tpl_listen_multiple_done
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/21/04
 *
 * Description:  Function handler to handle listen response from
 *               TxRx when trying to connect from a list of addresses
 *
 * Arguments:
 *  conn     - tpl connection
 *  arg      - argument passed to the tpl_listen() call
 *  status   - status from the listen response
 *------------------------------------------------------------------------
 */
void
tpl_listen_multiple_done(tpl_fp_conn_t **conn,
                         void *arg,
                         NTSTATUS status);


/*-----------------------------------------------------------------------
 * Name:         tpl_connect_multiple_done
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/21/04
 *
 * Description:  Function handler to handle connect response from
 *               TxRx when trying to connect from a list of addresses
 *               If the connect fails and there are more addresses to
 *               to connect to, then we will close the current socket
 *               used for the (failed) connect.
 *
 * Arguments:
 *  conn     - tpl connection
 *  arg      - argument passed to the tpl_connect_multiple() call
 *  status   - status from the connect response
 *------------------------------------------------------------------------
 */
void
tpl_connect_multiple_done(tpl_fp_conn_t *conn,
                          void *arg,
                          NTSTATUS status);


/*-----------------------------------------------------------------------
 * Name:         tpl_connect_multiple_close_done
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 08/12/05
 *
 * Description:  Function handler to handle close response from
 *               TxRx when we closed the socket associated with a previously
 *               failed connect. This will now open a new socket to do the
 *               next connect.
 *
 * Arguments:
 *  conn     - tpl connection
 *  arg      - argument passed to the tpl_connect_multiple() call
 *  status   - status from the close response
 *------------------------------------------------------------------------
 */
void
tpl_connect_multiple_close_done(tpl_fp_conn_t *conn,
                                void *arg,
                                NTSTATUS status);
                                

/*-----------------------------------------------------------------------
 * Name:         tpl_connect_multiple_reopen_done
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 08/12/05
 *
 * Description:  Function handler to handle open response from
 *               TxRx when trying to connect from a list of addresses.
 *               Once a new socket has been opened, we will try to connect to
 *               the next address.
 *
 * Arguments:
 *  conn     - tpl connection
 *  arg      - argument passed to the tpl_connect_multiple() call
 *  status   - status from the open response
 *------------------------------------------------------------------------
 */
void
tpl_connect_multiple_reopen_done(tpl_fp_conn_t *conn,
                                 void *arg,
                                 NTSTATUS status);


/*-----------------------------------------------------------------------
 * Name:         tpl_connect_timeout_handler
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/21/04
 *
 * Description:  Handler for handling connect timeout
 *
 * Arguments:
 *  timer_handle - timer handler of expired timer
 *  usr_p1       - user private 1 (tpl_fp_conn_t*)
 *  usr_p2       - user private 2 (not used)
 *  usr_p3       - user private 3 (not used)
 *  usr_p4       - user private 4 (not used)
 *
 * Return Value:
 *	NFX_OK
 *------------------------------------------------------------------------
 */
int32 tpl_connect_timeout_handler(uint32 timer_handle, 
                                  void *usr_p1, 
                                  void *usr_p2 unused__, 
                                  void *usr_p3 unused__, 
                                  void *usr_p4 unused__);


/*-----------------------------------------------------------------------
 * Name:         tpl_queue_alloc
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/22/04
 *
 * Description:  Allocates an I/O queue associated with a TPL connection
 *               allocating all associated memory and buffers		
 *
 * Arguments:
 *  queue         - queue to be allocated
 *  max_bytes     - maximum data bytes this queue can hold
 *  max_buffers - maximum buffers that can be used to hold the bytes
 *------------------------------------------------------------------------
 */
int tpl_queue_alloc(tpl_fp_queue_t *queue,
                    uint32 max_bytes, 
                    uint32 max_buffers);


/*-----------------------------------------------------------------------
 * Name:         tpl_queue_dealloc
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/22/04
 *
 * Description:  Deallocates an I/O queue associated with a TPL connection
 *               freeing all associated memory and buffers		
 *
 * Arguments:
 *  queue   - queue to be deallocated
 *------------------------------------------------------------------------
 */
void tpl_queue_dealloc(tpl_fp_queue_t *queue);


/*-----------------------------------------------------------------------
 * Name:         tpl_close_flush_done
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/19/04
 *
 * Description:  Function handler to handle write response from
 *               TxRx when flushing data during connection close time
 *
 * Arguments:
 *  conn     - tpl connection
 *  arg      - argument passed to the tpl_request_write() call
 *  status   - status from the write response
 *------------------------------------------------------------------------
 */
void tpl_close_flush_done(tpl_fp_conn_t *conn, void *arg, NTSTATUS status);


/*-----------------------------------------------------------------------
 * Name:         tpl_read_n_done
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/22/04
 *
 * Description:  Function handler to handle read response from
 *               TxRx when trying to do synchronous read of n bytes
 *
 * Arguments:
 *  conn     - tpl connection
 *  arg      - argument passed to the tpl_request_read() call
 *  status   - status from the read response
 *
 * Return Values:
 *  STATUS_SUCCESS - read all the requested bytes successfully
 *  STATUS_TIMEOUT - the read has timed out
 *  STATUS_ALREADY_DISCONNECTED - the connection has been closed.
 *  STATUS_UNSUCCESSFUL - read failed due to some other reason
 *------------------------------------------------------------------------
 */
NTSTATUS tpl_read_n_done(tpl_fp_conn_t *conn,
                         void *arg,
                         NTSTATUS status);


/*-----------------------------------------------------------------------
 * Name:         tpl_write_n_done
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/22/04
 *
 * Description:  Function handler to handle write response from
 *               TxRx when trying to do synchronous write of n bytes
 *
 * Arguments:
 *  conn     - tpl connection
 *  arg      - argument passed to the tpl_request_write() call
 *  status   - status from the read response
 *
 * Return Values:
 *  STATUS_SUCCESS - wrote all the requested bytes successfully
 *  STATUS_INSUFF_SERVER_RESOURCES - not enough memory
 *  STATUS_TIMEOUT - the read has timed out
 *  STATUS_ALREADY_DISCONNECTED - the connection has been closed.
 *  STATUS_UNSUCCESSFUL - write failed due to some other reason
 *------------------------------------------------------------------------
 */
NTSTATUS tpl_write_n_done(tpl_fp_conn_t *conn,
                          void *arg,
                          NTSTATUS status);


/*-----------------------------------------------------------------------
 * Name:         tpl_read_buffer_data
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/23/04
 *
 * Description:  Reads data from a queue into the provided buffer
 *               Data copy from TxRx buffers to FP buffers will
 *               make use of the data-mover only if the caller is a thread
 *               (and not a state machine).
 *
 * Arguments:
 *  buf         - buffer to read data into
 *  conn        - tpl connection
 *  queue       - TPL queue from which the data is to be read
 *  nbytes      - number of bytes to read
 *  nbytes_read - actual number of bytes read
 *
 * Return Values:
 *  NFX_OK  if we were able to access the queue and read data
 *  NFX_ERR if an error occured while reading data
 *------------------------------------------------------------------------
 */
int tpl_read_buffer_data(char *buf,
                         tpl_fp_conn_t *conn,
                         tpl_fp_queue_t *queue, 
                         uint32 nbytes, 
                         uint32 *nbytes_read);


/*-----------------------------------------------------------------------
 * Name:         tpl_read_edesc_data
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/23/04
 *
 * Description:  Reads from from a queue into the provided edesc at the offset
 *               within the edesc
 *               Data copy from TxRx buffers to FP buffers will
 *               make use of the data-mover only if the caller is a thread
 *               (and not a state machine).
 *
 * Arguments:
 *  start_edesc   - edesc to read data into
 *  data_offset   - offset in the edesc where the data is to be read into
 *  conn          - tpl connection
 *  queue         - TPL queue from which the data is to be read
 *  nbytes        - number of bytes to read
 *  nbytes_read   - actual number of bytes read
 *
 * Return Values:
 *  NFX_OK  if we were able to access the queue and read data
 *  NFX_ERR if memory allocation failed while reading data
 *------------------------------------------------------------------------
 */
int tpl_read_edesc_data(eee_desc_t **start_edesc, 
                        uint32 data_offset,
                        tpl_fp_conn_t *conn,
                        tpl_fp_queue_t *queue, 
                        uint32 nbytes, 
                        uint32 *nbytes_read);


/*-----------------------------------------------------------------------
 * Name:         tpl_write_buffer_data
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/23/04
 *
 * Description:  Writes data from the provided buffer into a queue
 *
 * Arguments:
 *  buf             - buffer to write data from
 *  conn            - tpl connection
 *  queue           - TPL queue into which the data is to be written
 *  nbytes          - number of bytes to write
 *  nbytes_written  - actual number of bytes written
 *
 * Return Values:
 *  NFX_OK  if we were able to access the queue and write data
 *  NFX_ERR if an error occured while writing data
 *------------------------------------------------------------------------
 */
int tpl_write_buffer_data(char *buf, 
                          tpl_fp_conn_t *conn,
                          tpl_fp_queue_t *queue, 
                          uint32 nbytes, 
                          uint32 *nbytes_written);


/*-----------------------------------------------------------------------
 * Name:         tpl_write_edesc_data
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 07/23/04
 *
 * Description:  Function handler to handle write response from
 *               TxRx when trying to do synchronous write of n bytes
 *
 * Arguments:
 *  edesc           - edesc to write data from
 *  data_offset     - offset in the edesc where the data is to be written
 *  conn            - tpl connection
 *  queue           - TPL queue into which the data is to be written
 *  nbytes          - number of bytes to write
 *  nbytes_written  - actual number of bytes written
 *
 * Return Values:
 *  NFX_OK  if we were able to access the queue and write data
 *  NFX_ERR if an error occured while writing data
 *------------------------------------------------------------------------
 */
int tpl_write_edesc_data(eee_desc_t *edesc, 
                         uint32 data_offset,
                         tpl_fp_conn_t *conn,
                         tpl_fp_queue_t *queue,
                         uint32 nbytes, 
                         uint32 *nbytes_written);


/*-----------------------------------------------------------------
 * Name:         tpl_send_write_request
 *
 * Created by :  Nitin Bahadur
 *
 * Date Created: 10/06/04
 *
 * Description:  Sends a write request to the TxRx
 *
 * Arguments:
 *  conn - tpl connection
 *-----------------------------------------------------------------
 */
void tpl_send_write_request(tpl_fp_conn_t *conn);


/*-----------------------------------------------------------------------------
 * Name:         tpl_allocateDesc
 *
 * Created by:   Nitin Bahadur      
 *
 * Date Created: 07/23/04
 *
 * Description:  Allocate an edesc chain and memory to store a record of 
 *               stream data. The record is divided into ULTRA buf size 
 *               chunks. Modeled on code in drs-api.c from drs_allocateDesc()
 *
 * Argument:
 *  size    - amount of data to be contained in the edesc chain
 *
 * Return Values:
 *   Allocated edesc.
 *   NULL on error.
 *-----------------------------------------------------------------------------
 */
eee_desc_t* tpl_allocate_edesc(uint32 size);


/*-----------------------------------------------------------------------------
 * Name:         tpl_print_conn
 *
 * Created by:   Nitin Bahadur      
 *
 * Date Created: 08/04/04
 *
 * Description:  Prints details about a tpl connection
 *
 * Argument:
 *  conn    - tpl connection to print
 *
 *-----------------------------------------------------------------------------
 */
void tpl_print_conn(tpl_fp_conn_t *conn);

#endif /* NFP_FP */

#endif /* __TPL_FP_H */
