/*
 *-----------------------------------------------------------------
 * Name: tpl-ipc.c
 * Description: Transport Packet Layer(TPL) event operations and
 *              IPC queue support between txrx NCPU(transport) and ACPU(application) cores.
 * Copyright (C) 2009, OnStor, Inc.
 *-----------------------------------------------------------------
 */
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>

#include <linux/gfp.h>
#include <linux/slab.h>

#include "nfx-defs.h"
#include "nfx-types.h"
#include "nfx-error.h"

#include "eee-desc.h"

#include "tpl-api.h"
#include "tpl.h"
#include "tpl-ipc.h"

#ifdef LATER
#include "../sm-cifs/win-compat.h"
#include "../sm-req-queue/req-queue-api.h"
#include "../sm-req-queue/req-queue-struct.h"
#endif

#define TRUE 1
#define FALSE 0
#define NFX_ERR -1 /* XXX */

#define TPL_IPC_PRINTK(a)   printk a

extern tpl_cfg_t *pTplCfg;
// extern uint64 eee_timestamp(void);

extern int tpl_bindReqCore(tpl_bindCb_t *bindReq, tpl_bindCb_t **bindRsp, int bindOpt);
extern int tpl_unbindReqCore(tpl_bindCb_t *bindReq);
extern void tpl_delConnReqCore(tpl_connCb_t *pConn);
extern void tpl_delConnCore(tpl_connCb_t *pConn);

extern void tpl_xmtPktCore(struct sock *sk, struct sk_buff *skb);

/*
 * Forward Referenced Procedures
 */
void tpl_ipcInit(void);

extern void pkt_acpu_free_chain(struct sk_buff *);

int tpl_ipc_evnt_requests_available(tpl_evntMsg_t *evntMsg);
int tpl_ipc_evnt_loopback(tpl_evntMsg_t *evntMsg);

#ifdef LATER
/*
 * Procedures called by NCPU Functions
 */
void tpl_ipcEvntProc(void *cb, unsigned int tref);
int tpl_ipcTxPkt(tpl_evntMsg_t *msg);
int tpl_ipcEvntBindReq(tpl_evntMsg_t *evntMsg);
int tpl_ipcEvntUnbindReq(tpl_evntMsg_t *evntMsg);
int tpl_ipcEvntUnbindReqAck2(tpl_evntMsg_t *evntMsg);
int tpl_ipcEvntCloseReq(tpl_evntMsg_t *evntMsg);
int tpl_ipcEvntAddConnAck(tpl_evntMsg_t *evntMsg);
int tpl_ipcEvntDelConnAck(tpl_evntMsg_t *evntMsg);

/*
 * Procedures called by ACPU Functions
 */
void pkt_acpu_free_from_ncpu(struct sk_buff *skb);
int tpl_ipcAcpuTxPktCompl(tpl_evntMsg_t *msg);
int tpl_ipcAcpuRxPkt(tpl_evntMsg_t *msg);
int tpl_ipcAcpuEvntBindReqAck(tpl_evntMsg_t *evntMsg);
int tpl_ipcAcpuEvntUnbindReqAck(tpl_evntMsg_t *evntMsg);
int tpl_ipcAcpuEvntAddConn(tpl_evntMsg_t *evntMsg);
int tpl_ipcAcpuEvntDelConn(tpl_evntMsg_t *evntMsg);
int tpl_ipcAcpuProc(tpl_ipcq_t *myQueuePtr);
int tpl_ipcAcpuEvntRxPktProc(void);

/*
 * External Procedures
 */
extern void *vs_getVsvrById(vs_id_t vsId);

#endif /* LATER */

/*
 * Forward Referenced Procedures
 */
void tpl_ipcq_read(tpl_ipcq_t *qCommon, tpl_evntMsg_t **evntMsgPtr, tpl_evntMsg_t **return_msg);

void tpl_ipcq_write(tpl_ipcq_t *qCommon, tpl_evntMsg_t *entry, unsigned int msg_type);

int    tpl_ipcTxPkt(tpl_evntMsg_t *evntMsg);
int    tpl_ipcAcpuTxPktCompl(tpl_evntMsg_t *evntMsg);
int    tpl_ipcAcpuRxPkt(tpl_evntMsg_t *evntMsg);
int    tpl_ipcAcpuEvntBindReqAck(tpl_evntMsg_t *evntMsg);
int    tpl_ipcAcpuEvntUnbindReqAck(tpl_evntMsg_t *evntMsg);
int    tpl_ipcAcpuEvntAddConn(tpl_evntMsg_t *evntMsg);
int    tpl_ipcAcpuEvntDelConn(tpl_evntMsg_t *evntMsg);

void tpl_ipcEvntProc(void *cb, unsigned int tref);
int    tpl_ipcEvntBindReq(tpl_evntMsg_t *evntMsg);
int    tpl_ipcEvntUnbindReq(tpl_evntMsg_t *evntMsg);
int    tpl_ipcEvntUnbindReqAck2(tpl_evntMsg_t *evntMsg);
int    tpl_ipcEvntCloseReq(tpl_evntMsg_t *evntMsg);
int    tpl_ipcEvntAddConnAck(tpl_evntMsg_t *evntMsg);
int    tpl_ipcEvntDelConnAck(tpl_evntMsg_t *evntMsg);

int tpl_ipc_evnt_requests_available(tpl_evntMsg_t *evntMsg);

/*
 * Global Data Structures
 */
typedef struct tpl_q_const {
    unsigned int  qEntSize;
    unsigned int  qLength;
} tpl_q_const_t;

tpl_q_const_t tpl_ipc_qConst[TPL_IPC_Q_CNT] = {
    {sizeof(tpl_qent_t), (MAX_NTOA_EVNTS+TPL_READ_WRITE_GAP)},     /* TPL_IPC_EVNTQ_NTOA */ 
    {sizeof(tpl_qent_t), (PKT_MAX_DESC_CNT*2+TPL_READ_WRITE_GAP)}, /* TPL_IPC_EVNTQ_ATON */
    {sizeof(tpl_qent_t), (PKT_MAX_DESC_CNT*2+TPL_READ_WRITE_GAP)}, /* TPL_IPC_EVNTQ_RXPKTS */
};

typedef int  (*tpl_evnt_fn_t)(tpl_evntMsg_t *msg);

tpl_evnt_fn_t    tpl_evnt_aproc[TPL_EVNT_CNT] = {
    NULL,                       /* TPL_EVNT_BIND_REQ */
    tpl_ipcAcpuEvntBindReqAck,  /* TPL_EVNT_BIND_REQ_ACK */
    NULL,                       /* TPL_EVNT_UNBIND_REQ */
    tpl_ipcAcpuEvntUnbindReqAck,/* TPL_EVNT_UNBIND_REQ_ACK */
    NULL,                       /* TPL_EVNT_CLOSE_REQ */
    tpl_ipcAcpuEvntAddConn,     /* TPL_EVNT_ADD_CONN */
    NULL,                       /* TPL_EVNT_ADD_CONN_ACK */
    tpl_ipcAcpuEvntDelConn,     /* TPL_EVNT_DEL_CONN */
    NULL,                       /* TPL_EVNT_DEL_CONN_ACK */ 
    NULL,                       /* TPL_EVNT_PKTQ_TX */
    tpl_ipcAcpuTxPktCompl,      /* TPL_EVNT_PKTQ_TX_COMP */
    tpl_ipcAcpuRxPkt,           /* TPL_EVNT_PKTQ_RX */
    NULL,                       /* TPL_EVNT_PKTQ_TX_SLOW */
    NULL,                       /* TPL_EVNT_UNBIND_REQ_ACK2 */
    NULL,                       /* TPL_EVNT_REQUESTS_AVAILABLE */
    NULL,                       /* TPL_EVNT_LOOPBACK */
};

tpl_evnt_fn_t    tpl_evnt_nproc[TPL_EVNT_CNT] = {
    tpl_ipcEvntBindReq,         /* TPL_EVNT_BIND_REQ */
    NULL,                       /* TPL_EVNT_BIND_REQ_ACK */
    tpl_ipcEvntUnbindReq,       /* TPL_EVNT_UNBIND_REQ */
    NULL,                       /* TPL_EVNT_UNBIND_REQ_ACK */
    tpl_ipcEvntCloseReq,        /* TPL_EVNT_CLOSE_REQ */
    NULL,                       /* TPL_EVNT_ADD_CONN */
    tpl_ipcEvntAddConnAck,      /* TPL_EVNT_ADD_CONN_ACK */
    NULL,                       /* TPL_EVNT_DEL_CONN */
    tpl_ipcEvntDelConnAck,      /* TPL_EVNT_DEL_CONN_ACK */
    tpl_ipcTxPkt,               /* TPL_EVNT_PKTQ_TX */
    NULL,                       /* TPL_EVNT_PKTQ_TX_COMP */
    NULL,                       /* TPL_EVNT_PKTQ_RX */
    tpl_ipcTxPkt,           /* TPL_EVNT_PKTQ_TX_SLOW */
    tpl_ipcEvntUnbindReqAck2,    /* TPL_EVNT_UNBIND_REQ_ACK2 */
    tpl_ipc_evnt_requests_available, /* TPL_EVNT_REQUESTS_AVAILABLE */
    tpl_ipc_evnt_loopback,           /* TPL_EVNT_LOOPBACK */
};

NFX_ASSERT_STATIC(sizeof(tpl_evnt_aproc) == (TPL_EVNT_CNT * sizeof(tpl_evnt_fn_t)));
NFX_ASSERT_STATIC(sizeof(tpl_evnt_nproc) == (TPL_EVNT_CNT * sizeof(tpl_evnt_fn_t)));

inline unsigned int
TPL_IPCQ_ENTRY(tpl_ipcq_t *_qCommon)
{
   return (((_qCommon)->tpl_ipcqRdPtr.rdPtr)->ptr == NULL);
}

inline unsigned int
TPL_IPCQ_FULL(tpl_ipcq_t *_qCommon)
{
  return ((_qCommon)->tpl_ipcqWrPtr.tailRdPtr == (_qCommon)->tpl_ipcqWrPtr.wrPtr);
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcq_write
 * Description: Write a message to the IPC Queue
 *-----------------------------------------------------------------
 */
void
tpl_ipcq_write(tpl_ipcq_t *qCommon, tpl_evntMsg_t *entry, unsigned int msg_type)
{
    tpl_qent_t *wr = qCommon->tpl_ipcqWrPtr.wrPtr;

    while (qCommon->tpl_ipcqWrPtr.tailRdPtr == wr) {
        /* spin until the queue is not full
         */
        ;
    }
    wr->msg = (void *)msg_type;
    wr->ptr = (void *)entry;

    if (wr == qCommon->tpl_ipcqEndPtr)
        qCommon->tpl_ipcqWrPtr.wrPtr = qCommon->tpl_ipcqStartPtr;
    else
        qCommon->tpl_ipcqWrPtr.wrPtr = wr + 1;

    qCommon->tpl_ipcqWrPtr.total_wr++;
    return;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcq_read
 * Description: Read a message from the IPC Queue
 *              Returns the two address'es contained in the read message
 *-----------------------------------------------------------------
 */
void
tpl_ipcq_read(tpl_ipcq_t *qCommon, tpl_evntMsg_t **evntMsgPtr, tpl_evntMsg_t **return_msg)
{
    *evntMsgPtr = (tpl_evntMsg_t *)((qCommon->tpl_ipcqRdPtr.rdPtr)->ptr);
    *return_msg = (tpl_evntMsg_t *)((qCommon->tpl_ipcqRdPtr.rdPtr)->msg);

    qCommon->tpl_ipcqRdPtr.rdPtr->ptr = NULL;

    /* update tail rd ptr */
    if (++(qCommon->tpl_ipcqRdPtr.lag) >= TPL_READ_WRITE_GAP) {
        qCommon->tpl_ipcqWrPtr.tailRdPtr = qCommon->tpl_ipcqRdPtr.rdPtr;
        qCommon->tpl_ipcqRdPtr.lag = 0;
    }

    if (qCommon->tpl_ipcqRdPtr.rdPtr == qCommon->tpl_ipcqEndPtr) { /* wrap-around */
        qCommon->tpl_ipcqRdPtr.rdPtr = qCommon->tpl_ipcqStartPtr;
    } else {
        qCommon->tpl_ipcqRdPtr.rdPtr++;
    }

    qCommon->tpl_ipcqRdPtr.total_rd++;
    return;
}                                                                       

/*-----------------------------------------------------------------
 * Name:        tpl_ipcEvntDelConnAck
 * Description: Free the packet sent by an application from NCPU, ACPU function
 *-----------------------------------------------------------------
 */
void
pkt_acpu_free_from_ncpu(struct sk_buff *skb)
{
    // ASSERT(skb->next == NULL);
    tpl_ipcq_t *pktTxCmplQueuePtr = &(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_PKTS_COMP]);

    pTplCfg->tplStats.cntIpcCompFwd++;
    tpl_ipcq_write(pktTxCmplQueuePtr, (tpl_evntMsg_t *)skb, TPL_EVNT_PKTQ_TXCOMP);
    return;
}

/*-----------------------------------------------------------------
 * Name: tpl_ipc_allocEvntMsg
 * Description: Allocate evntMsg from shared memory area
 *-----------------------------------------------------------------
 */
inline tpl_evntMsg_t *
tpl_ipc_allocEvntMsg(unsigned int msgSize)
{
    tpl_evntMsg_t *evntMsgPtr = (tpl_evntMsg_t *)kmalloc(msgSize, (__GFP_HIGHMEM | __GFP_NOFAIL));
    return evntMsgPtr;
}

/*-----------------------------------------------------------------
 * Name: tpl_ipc_allocEvntMsg
 * Description: Deallocate evntMsg back to shared memory area
 *-----------------------------------------------------------------
 */
inline void
tpl_ipc_deallocEvntMsg(tpl_evntMsg_t *evntMsgPtr)
{
  // kfree(evntMsgPtr, (__GFP_HIGHMEM | __GFP_NOFAIL));
    kfree(evntMsgPtr);
    return;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcSendEvntMsg
 * Description: Send event message between NCPU and ACPU
 *              depending on specified direction.
 *
 * Parameters:
 * msgType => type of message to send
 * data => Address of data to store in message
 * whereto => Where to sent event message
 *  0 => ACPU to NCPU, 1 => NCPU to ACPU
 * vsId = Virtual Server Identifier 
 *-----------------------------------------------------------------
 */
int
tpl_ipcSendEvntMsg(unsigned int msgType, void *data, int whereto, vs_id_t vsId)
{
    tpl_evntMsg_t   *evntMsgPtr;
    tpl_ipcq_t      *evntQueuePtr;
    int msgSize, rc, which_queue;

    TPL_IPC_PRINTK(("%s: msg %d, queue %s\n", __FUNCTION__, 
                    msgType, whereto ? "NtoA":"AtoN"));
    
    rc = 0;
    /* determine msg size */
    if ((msgType == TPL_EVNT_BIND_REQ) || (msgType == TPL_EVNT_UNBIND_REQ)) {
        msgSize = sizeof(tpl_evntMsg_t) + sizeof(tpl_bindCb_t);
    } else {
        msgSize = sizeof(tpl_evntMsg_t);
    }

    /* allocate evntMsg */
    evntMsgPtr = tpl_ipc_allocEvntMsg(msgSize);

    if (evntMsgPtr == NULL) {
        pTplCfg->tplStats.msgAllocFail++;
        return rc;
    }
    
    /* which event queue to put msg onto */
    if (whereto) {
        which_queue = TPL_IPC_EVNTQ_NTOA;
        pTplCfg->tplStats.NtoAMsgCnt++;
    } else {
	which_queue = TPL_IPC_EVNTQ_ATON;
        pTplCfg->tplStats.AtoNMsgCnt++;
    }

    evntMsgPtr->evnt_msg = msgType;
    evntMsgPtr->evnt_rc = 0;
    evntMsgPtr->evnt_data = data;
    evntMsgPtr->evnt_vsId = vsId;

    if (msgType == TPL_EVNT_BIND_REQ) {
        /* copy bind request into msg body */
        memcpy((char*)(evntMsgPtr + 1), (char*)data, sizeof(tpl_bindCb_t));
        
        /* reset data ptr to bind req */
        evntMsgPtr->evnt_data = (evntMsgPtr + 1);
    }

    evntQueuePtr = &(pTplCfg->tplIpcQ[which_queue]); 

    /* put msg on evnt queue */
    tpl_ipcq_write(evntQueuePtr, evntMsgPtr, msgType);
    
    /* successful msgs are freed upon acknowledgements */
    if (rc != 0) {
        tpl_ipc_deallocEvntMsg(evntMsgPtr);

	if (whereto)
	  pTplCfg->tplStats.NtoAMsgCnt--;
	else
	  pTplCfg->tplStats.AtoNMsgCnt--;
    }        
    return rc;
}

/*-----------------------------------------------------------------
 * Name:        tpl_loopback_packet
 * Description: Forward packet directly to the ACPU receive queue.
 * Arguments:
 *  pkt - head of the packet chain.
 * Return Value:
 *  None.
 *-----------------------------------------------------------------
 */
void
tpl_loopback_packet(struct sk_buff *pkt)
{
    tpl_ipcq_write(&(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_ATON]), (tpl_evntMsg_t *)pkt, TPL_EVNT_LOOPBACK);
    return;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcAcpuTxPktCompl
 * Description: process pkts in TPL_IPC_PKTQ_TXCOMPL queue, ACPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipcAcpuTxPktCompl(tpl_evntMsg_t *msg)
{
    pTplCfg->tplStats.cntIpcCompRcv++;

    pkt_acpu_free_chain((struct sk_buff *)msg);
    return 0;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcAcpuRxPkt
 * Description: process pkts in TPL_IPC_PKTQ_RX queue, ACPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipcAcpuRxPkt(tpl_evntMsg_t *msg)
{
  /* XXX */
#ifdef LATER
    struct sk_buff *skb;
    // tpl_connCb_t *pConn = skb->pTplConn;
    // char *pHdr = skb->pHdr;
    // uint64 tval = eee_timestamp();
    uint64 tval = 0;
    tpl_connCb_t *pConn = NULL;
    char *pHdr = NULL;
    int state;
    void *app_handle;
    tpl_rcvPktInd_func rcv_func;

    // PREFETCH(PREF_LOAD_STREAMED, msg, offsetof(struct sk_buff, pHdr));

    skb = (struct sk_buff*)msg;

    /* handle bcast pkt */
    if (skb->flags & PKT_BROADCAST) {

        if(((tpl_bindCb_t *)pConn)->ul_rcvPktInd == NULL) {
            
            /* Just drop the packet and return
             */
            txrx_request_complete();

            pTplCfg->tplStats.cntIpcRxDrop++;
            pkt_freePkt(skb);
            return 0;
        }
        /* If its broacast packet then there is no pConn
         * instead we have a pBind. See bsd_RcvSock().
         */
        ((tpl_bindCb_t *)pConn)->ul_rcvPktInd(0, skb, pHdr, 0);
        return 0;
    }
    
    state = pConn->acpuState;
    app_handle = pConn->pAppHdl;
    rcv_func = pConn->rcv_func;
    
    /* make sure connection has not be removed 
     */
    if (state != TPL_IPC_CONN_ADDED) {
        txrx_request_complete();

        pTplCfg->tplStats.cntIpcRxDrop++;
        pkt_freePkt(skb);
    } else {
      rcv_func(app_handle, skb, pHdr, tval);
    }
#endif /* LATER */

    return 0;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcAcpuEvntBindReqAck
 * Description: process bind request ack, ACPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipcAcpuEvntBindReqAck(tpl_evntMsg_t *evntMsg)
{
    tpl_bindCb_t *pBind, *pBindReq;
    int rc;     

    TPL_IPC_PRINTK(("%s: evnt_msg %d, evnt_rc %d\n", __FUNCTION__, evntMsg->evnt_msg, evntMsg->evnt_rc));

    /* XXX */
#ifdef LATER
    /* Check if the corresponding VS still exists, if not drop
     * the event
     */
    // ASSERT(evntMsg->evnt_vsId != VS_ID_INVALID);
    if(vs_getVsvrById(evntMsg->evnt_vsId) == NULL) {
      // ASSERT(FALSE);

        /* free msg that initiated from ACPU */
        tpl_ipc_deallocEvntMsg(evntMsg);
        return NFX_ERR;
    }
#endif /* LATER */

    pBind = evntMsg->evnt_data;
    pBindReq = (tpl_bindCb_t*)(evntMsg + 1);
    rc = evntMsg->evnt_rc;

    if (pBind->acpuState == TPL_IPC_CONN_INIT) {

        TPL_IPC_PRINTK(("%s: pBind->acpuState == TPL_IPC_CONN_INIT, evnt_msg %d\n",
		__FUNCTION__, evntMsg->evnt_msg));

        pBind->acpuState = ((rc == 0) ? TPL_IPC_CONN_ADDED : TPL_IPC_CONN_DELED);

	/* let application know, bind result */
	if (pBindReq->ul_bindReqAck) {
	  (*pBindReq->ul_bindReqAck)(pBindReq->pSock, rc, pBind);
	}
    
	/* free msg that initiated from ACPU */
	tpl_ipc_deallocEvntMsg(evntMsg);
    } else {
        TPL_IPC_PRINTK(("%s: pBind->acpuState != TPL_IPC_CONN_INIT, msg %d\n",
		__FUNCTION__, evntMsg->evnt_msg));
    }
    return 0;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcAcpuEvntUnbindReqAck
 * Description: process unbind request ack, ACPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipcAcpuEvntUnbindReqAck(tpl_evntMsg_t *evntMsg)
{
    tpl_ipcq_t *eventQueuePtr = &(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_ATON]);
    tpl_bindCb_t *pBind;
    int rc;

    TPL_IPC_PRINTK(("%s: evnt_msg %d, evnt_rc %d\n",
	__FUNCTION__, evntMsg->evnt_msg, evntMsg->evnt_rc));

#ifdef LATER
    /* Check if the corresponding VS still exists, if not drop
     * the event
     */
    // ASSERT(evntMsg->evnt_vsId != VS_ID_INVALID);

    if(vs_getVsvrById(evntMsg->evnt_vsId) == NULL) {
      // ASSERT(FALSE);

        /* free msg that initiated from ACPU */
        tpl_ipc_deallocEvntMsg(evntMsg);
        return NFX_ERR;
    }
#endif /* LATER */

    pBind = evntMsg->evnt_data;
    rc = evntMsg->evnt_rc;

    if (pBind->acpuState == TPL_IPC_CONN_CLOSE_SENT) {
        pBind->acpuState = TPL_IPC_CONN_DELED;
    } else { /* error */
        TPL_IPC_PRINTK(("%s: ERROR: pBind->acpuState == TPL_IPC_CONN_CLOSE_SENT, evnt_msg %d, evnt_rc %d\n",
		__FUNCTION__, evntMsg->evnt_msg, evntMsg->evnt_rc));
        return 0;
    }

    /* Call the unbind function to do the work */
    (*pBind->ul_unbindReqAck)(rc, pBind->laddr, pBind->lport, pBind->tplProt, pBind->usrInfo, pBind);

    /* Send the last ACK to NCPU
     * Note:1.  The unbind handler's return code is ignored here.
     *      2.  The event structure is deallocated on NCPU side
     *          as part of processing the last ACK
     */
    tpl_ipcq_write(eventQueuePtr, evntMsg, TPL_EVNT_UNBIND_REQ_ACK2);
    return 0;  
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcAcpuEvntAddConn
 * Description: process add connection indication messager, ACPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipcAcpuEvntAddConn(tpl_evntMsg_t *evntMsg)
{
    tpl_connCb_t *pConn;
    tpl_bindCb_t *pBind;
    unsigned int rc;
    tpl_ipcq_t *evntQAtoNPtr = &(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_ATON]);
    
    TPL_IPC_PRINTK(("%s: msg %d\n", __FUNCTION__, evntMsg->evnt_msg));

#ifdef LATER
    /* Check if the corresponding VS still exists, if not drop
     * the event
     */
    // ASSERT(evntMsg->evnt_vsId != VS_ID_INVALID);
    if(vs_getVsvrById(evntMsg->evnt_vsId) == NULL) {
      // ASSERT(FALSE);
        
        pConn = NULL;
        rc = NFX_ERR;
        /* Fall thru and send the response to NCPU
         */
        goto send_resp;
    }
#endif /* LATER */

    pConn = (tpl_connCb_t*)evntMsg->evnt_data;
    pBind = pConn->pBindCb;
    
    rc = (*pBind->ul_addConnInd)(pBind, pConn, &pConn->pAppHdl, pConn->raddr, pConn->rport);

    /* copy result bck to evntMsg */
    evntMsg->evnt_msg = TPL_EVNT_ADD_CONN_ACK;
    evntMsg->evnt_rc = rc;

    if (rc == 0) {
        pConn->acpuState = TPL_IPC_CONN_ADDED;
    }

#ifdef LATER
send_resp:
#endif /* LATER */
          
    tpl_ipcq_write(evntQAtoNPtr, evntMsg, TPL_EVNT_ADD_CONN_ACK);
    return 0;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcAcpuEvntDelConn
 * Description: process delete conn ind, ACPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipcAcpuEvntDelConn(tpl_evntMsg_t *evntMsg)
{
    tpl_connCb_t *pConn;
    tpl_bindCb_t *pBind;
    unsigned int rc = 0;

    TPL_IPC_PRINTK(("%s: msg %d\n", __FUNCTION__, evntMsg->evnt_msg));
    
#ifdef LATER
    /* Check if the corresponding VS still exists, if not drop
     * the event
     */
    // ASSERT(evntMsg->evnt_vsId != VS_ID_INVALID);

    if(vs_getVsvrById(evntMsg->evnt_vsId) == NULL) {
      // ASSERT(FALSE);

        pConn = NULL;
        rc = NFX_ERR;

        /* Send an error message to the other side */
        goto send_resp;
    }
#endif /* LATER */
        pConn = (tpl_connCb_t*)evntMsg->evnt_data;
        
        /* application may never received the connAddInd, perhaps due to
         * out of eventMsg buffer
         */
        if ((pConn->acpuState == TPL_IPC_CONN_ADDED)
            || (pConn->acpuState == TPL_IPC_CONN_CLOSE_SENT)) {

            pBind = pConn->pBindCb;
            rc = (*pBind->ul_delConnInd)(pConn->pAppHdl, pConn);

            if (rc == 0) {
                pConn->acpuState = TPL_IPC_CONN_DELED;
            }
        }

#ifdef LATER
send_resp:
#endif /* LATER */
          
    /* Note: We would be sending an ACK to the delete connection
     *       request later at the time of acpu_conn_uninit,
     *       Currently, just free the event message.
     *       This message could be allocated on ACPU (for the case
     *       of an active close) or on NCPU (Passive close).
     */
    tpl_ipc_deallocEvntMsg(evntMsg);
    return rc;
}
                                        
/*-----------------------------------------------------------------
 * Name:        tpl_ipcAcpuProc
 * Description: Generic Event Processing function on ACPU
 * Return Values:
 *  Returns a count of the number of events processed.
 *-----------------------------------------------------------------
 */
int
tpl_ipcAcpuProc(tpl_ipcq_t *myQueuePtr)
{
    tpl_evntMsg_t *myEvntMsgPtr, *return_msg;
    unsigned int ix;
    int count = 0;

    while (! TPL_IPCQ_ENTRY(myQueuePtr)) {
        
        myEvntMsgPtr = NULL;
        return_msg = NULL;
        tpl_ipcq_read(myQueuePtr, &myEvntMsgPtr, &return_msg);

        ix = ((unsigned int)return_msg & 0xffffffff); /* XXX */

        if (ix > TPL_EVNT_CNT) {
            TPL_IPC_PRINTK(("%s: BOGUS index %d, return_msg 0x%p, myEvntMsgPtr 0x%p\n",
			__FUNCTION__, ix, return_msg, myEvntMsgPtr));
            break;
        }
        if (tpl_evnt_aproc[ix]) {
            TPL_IPC_PRINTK(("%s: ix %d, return_msg 0x%p, myEvntMsgPtr 0x%p, func 0x%p\n",
		__FUNCTION__, ix, return_msg, myQueuePtr, tpl_evnt_aproc[ix]));

            (*tpl_evnt_aproc[ix])(myEvntMsgPtr);
        } else {
            TPL_IPC_PRINTK(("%s: ix %d, return_msg 0x%p, myEvntMsgPtr 0x%p, NULL func 0x%p\n",
		__FUNCTION__, ix, return_msg, myQueuePtr, tpl_evnt_aproc[ix]));
        }
        count++;
    }
    return count;
}

/* Limit the number of requests processing per polling loop.
 * Make this high enough to not trash the caches without reason, but
 * give up the CPU once in a while to avoid watchdog timeout.
 */
#define TPL_ACPU_REQUESTS_PER_POLL 100

/*-----------------------------------------------------------------
 * Name:        tpl_ipcAcpuEvntRxPktProc
 * Description: Special type of event processing function 
 *              on ACPU which stops processing events
 *              in the IPC queue when there are requests on the
 *              PRIORITY_WAKEUP queue.
 *-----------------------------------------------------------------
 */
int
tpl_ipcAcpuEvntRxPktProc(void)
{
    int count = 0;
#ifdef LATER
    tpl_ipcq_t *myQueuePtr = &(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_NTOA]);
    tpl_evntMsg_t *myEvntMsgPtr, *returned_msg;
    unsigned int ix;

    boolean wasnt_throttled = TRUE;

    while ((!TPL_IPCQ_ENTRY(myQueuePtr)) &&

           SIMPLEQ_EMPTY(&req_queues[REQ_PRIORITY_WAKEUP]) &&
           (count < TPL_ACPU_REQUESTS_PER_POLL) &&
           (!io_throttle_enabled ||
            (wasnt_throttled = (io_throttle_try_reserve_req() == STATUS_SUCCESS)))) {

        myEvntMsgPtr = NULL;
        returned_msg = NULL;
        tpl_ipcq_read(myQueuePtr, &myEvntMsgPtr, &returned_msg);

        ix = ((unsigned int)return_msg & 0xffffffff); /* XXX */

        if (ix > TPL_EVNT_CNT) {
            TPL_IPC_PRINTK(("%s: BOGUS index %d, return_msg 0x%p, myEvntMsgPtr 0x%p\n",
			__FUNCTION__, ix, return_msg, myEvntMsgPtr));
            break;
        }

        if (tpl_evnt_aproc[ix]) {
                
            TPL_IPC_PRINTK(("%s: ix %d, returned_msg 0x%p, func 0x%p\n", __FUNCTION__,
                        ix, returned_msg, tpl_evnt_aproc[ix]));

            (*tpl_evnt_aproc[x])(myEvntMsgPtr);
        } else {
            TPL_IPC_PRINTK(("%s: ix %d, func NULL 0x%p\n", __FUNCTION__,
                        ix, (unsigned int)tpl_evnt_aproc[ix]));
        }
        count++;
    }
    if (!wasnt_throttled) {
        /* Bump the count so that the caller knows that there was work to be
         * done here.  That prevents them from starting lower-priority tasks
         * when we are throttled here.  This is necessary because the poll
         * loop might check the timers after this and before attempting to
         * run lower priority work.  Running the timers can free up request
         * slots.
         */
        count++;
    }
#endif /* LATER */
    return count;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcEvntProc
 * Description: evnt processing function on NCPU
 *-----------------------------------------------------------------
 */
void
tpl_ipcEvntProc(void *cb, unsigned int tref)
{
    tpl_ipcq_t *myQueuePtr = &(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_ATON]);
    tpl_evntMsg_t *myEvntMsgPtr, *return_msg;
    unsigned int ix;
    
    while (!TPL_IPCQ_ENTRY(myQueuePtr)) {

        myEvntMsgPtr = NULL;
        return_msg = NULL;
        tpl_ipcq_read(myQueuePtr, &myEvntMsgPtr, &return_msg);

        ix = ((unsigned int)return_msg & 0xffffffff); /* XXX */

        if (ix > TPL_EVNT_CNT) {
            TPL_IPC_PRINTK(("%s: BOGUS index %d, return_msg 0x%p, myEvntMsgPtr 0x%p\n",
			    __FUNCTION__, ix, return_msg, myEvntMsgPtr));
            break;
        }
        if (tpl_evnt_nproc[ix]) {
            TPL_IPC_PRINTK(("%s: ix %d, return_msg 0x%p, msg 0x%p, tpl_evnt_nproc[%d] 0x%p\n", __FUNCTION__,
                        ix, return_msg, myEvntMsgPtr, ix, tpl_evnt_nproc[ix]));

            (*tpl_evnt_nproc[ix])(myEvntMsgPtr);
        } else {
            TPL_IPC_PRINTK(("%s: ix %d, return_msg 0x%p, msg 0x%p, tpl_evnt_nproc[%d] NULL 0x%p\n", __FUNCTION__,
                        ix, return_msg, myEvntMsgPtr, ix, tpl_evnt_nproc[ix]));
        }
    }
    return;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcInit
 * Description: Allocate and initialize shared memory queues
 *              between ACPU and NCPU's.
 *-----------------------------------------------------------------
 */
void
tpl_ipcInit()
{
    unsigned int i, qSize;
    
    /* initialization, so allocate ipc queues in shared space */

    for (i=0; i < TPL_IPC_Q_CNT; i++) {

        qSize = tpl_ipc_qConst[i].qEntSize * tpl_ipc_qConst[i].qLength;

        pTplCfg->tplIpcQ[i].tpl_ipcqStartPtr = (tpl_qent_t*)kmalloc(qSize, (__GFP_HIGHMEM | __GFP_NOFAIL));

        memset(pTplCfg->tplIpcQ[i].tpl_ipcqStartPtr, 0, qSize);
            
        pTplCfg->tplIpcQ[i].tpl_ipcqEndPtr = pTplCfg->tplIpcQ[i].tpl_ipcqStartPtr 
            + tpl_ipc_qConst[i].qLength - 1;
        
        /* init. read/write ptrs */
        pTplCfg->tplIpcQ[i].tpl_ipcqRdPtr.rdPtr = 
            pTplCfg->tplIpcQ[i].tpl_ipcqWrPtr.wrPtr =  
            pTplCfg->tplIpcQ[i].tpl_ipcqStartPtr;

        pTplCfg->tplIpcQ[i].tpl_ipcqWrPtr.tailRdPtr = pTplCfg->tplIpcQ[i].tpl_ipcqEndPtr;
        pTplCfg->tplIpcQ[i].tpl_ipcqDropCnt = pTplCfg->tplIpcQ[i].tpl_ipcqHWater = 0;
    }

#ifdef LATER
    /* register with eee polling */
    int rc;
    rc = eee_registerPollingFunc(EEE_HIGH_PRIORITY, tpl_ipcEvntProc, 0, NULL);
    if (rc < 0)
        panic("%s: tpl ipc poll regsiter failed %d\n", __FUNCTION__, rc);
#endif
    return;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcTxPkt
 * Description: process pkts in TPL_IPC_PKTQ_TX queue, NCPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipcTxPkt(tpl_evntMsg_t *msg)
{
    struct sk_buff *skb = (struct sk_buff *)msg;
    pTplCfg->tplStats.cntIpcTxRcv++;

    tpl_xmtPktCore(skb->sk, skb);
    return 0;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcEvntBindReq
 * Description: process bind request, NCPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipcEvntBindReq(tpl_evntMsg_t *evntMsg)
{
    tpl_bindCb_t *bindReq, *bindRsp;
    int bindOpt, rc;

    TPL_IPC_PRINTK(("%s: msg %d\n", __FUNCTION__, evntMsg->evnt_msg));

    bindReq = (tpl_bindCb_t*)(evntMsg + 1);
    bindRsp = NULL;
    bindOpt = bindReq->bindFlags;
    
    rc = tpl_bindReqCore(bindReq, &bindRsp, bindOpt);
    
    /* copy result bck to evntMsg */
    evntMsg->evnt_msg = TPL_EVNT_BIND_REQ_ACK;
    evntMsg->evnt_rc = rc;
    evntMsg->evnt_data = bindRsp;
    
    tpl_ipcq_write(&(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_NTOA]), evntMsg, TPL_EVNT_BIND_REQ_ACK);

    return 0;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcEvntUnbindReq
 * Description: process unbind request, NCPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipcEvntUnbindReq(tpl_evntMsg_t *evntMsg)
{
    tpl_bindCb_t *bindReq;
    int rc, replyNow;
    
    TPL_IPC_PRINTK(("%s: msg %d\n", __FUNCTION__, evntMsg->evnt_msg));

    bindReq = (tpl_bindCb_t*)(evntMsg->evnt_data);
    
    /* cannot return unbind result yet, because
     * unbind result should always happen after 
     * all associated connections are removed.
     * Save req msg, to reply later
     */
    replyNow = 0;

    if (!bindReq->acpuMsg) {
        
        bindReq->acpuMsg = evntMsg;
        rc = tpl_unbindReqCore(bindReq);
        if (rc != 0) {
            /* We should never come here
             */
            bindReq->acpuMsg = NULL;
            replyNow = 1;
        }
    } else {
        /* there already a pending request */
        /* We should never come here */
        rc = NFX_ERR;
        replyNow = 1;
    }
    
    if (replyNow) {

        /* unbind request failed, return now */
        /* copy result bck to evntMsg */
        evntMsg->evnt_msg = TPL_EVNT_UNBIND_REQ_ACK;
        evntMsg->evnt_rc = rc;
        tpl_ipcq_write(&(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_NTOA]), evntMsg, TPL_EVNT_UNBIND_REQ_ACK);
    }
    return 0;
}

/*-----------------------------------------------------------------
 * Name         : tpl_ipcEvntUnbindReqAck2
 * Description  : process unbind ack2 (ACK from ACPU for our ACK) 
 *                request, NCPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipcEvntUnbindReqAck2(tpl_evntMsg_t *evntMsg)
{
    tpl_bindCb_t    *bindReq;

    TPL_IPC_PRINTK(("%s: msg %d\n", __FUNCTION__, evntMsg->evnt_msg));
    bindReq = (tpl_bindCb_t*)(evntMsg->evnt_data);

 // ASSERT(bindReq->bindFlags & TPL_BIND_CLOSE);
 // ASSERT(((void *)pTplCfg->bindTblBase <= (void *)bindReq) && ((void *)bindReq < (void *)pTplCfg->bindTblEnd));
 // ASSERT(bindReq->pSock == NULL);
    
    /* Deallocate bindCb. 
     */
    tpl_freeBindCb(bindReq);

    pTplCfg->tplStats.cntUnbindReq++;
    pTplCfg->numBindCb--;

    /* Free the event message.
     * Note: This was allocated on the ACPU.
     */
    tpl_ipc_deallocEvntMsg(evntMsg);
    return 0;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcEvntCloseReq
 * Description: process close request, NCPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipcEvntCloseReq(tpl_evntMsg_t *evntMsg)
{
    tpl_connCb_t    *pConn;

    TPL_IPC_PRINTK(("%s: msg %d\n", __FUNCTION__, evntMsg->evnt_msg));

    /* since tpl_delConnReq() always return 0, acknowledge request
     * first. This is to avoid race condition, where TPL_EVNT_DEL_CONN 
     * will come before TPL_EVNT_CLOSE_REQ_ACK for udp connections
     */
    evntMsg->evnt_rc = 0;
    pConn = (tpl_connCb_t*)evntMsg->evnt_data;
    
    /* Check if there was a simultaneous close..
     */
    if((evntMsg->evnt_rc = (pConn->flags & TPL_CONN_CLOSED) ? NFX_ERR : 0) == 0) {
        /* Close the connection from our side.
         * Note: A DEL_CONN_REQ will be generated when we close
         *       the socket via bsd_closeSock(), via a new message.
         */
        pTplCfg->tplStats.cntDelConnReq++;
    }
    
    tpl_ipc_deallocEvntMsg(evntMsg);
    return 0;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcEvntAddConnAck
 * Description: process add conn ack, NCPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipcEvntAddConnAck(tpl_evntMsg_t *evntMsg)
{
#ifdef LATER
    struct sk_buff *tp, *ntp;
    tpl_connCb_t *pConn;
    int rc;

    TPL_IPC_PRINTK(("%s: msg %d\n", __FUNCTION__, evntMsg->evnt_msg));

    pConn = (tpl_connCb_t*)evntMsg->evnt_data;
    rc = evntMsg->evnt_rc;

    /* Switch to the appropriate vstack context
     */
    vstack_retCode_t vstackRC;

    vstackRC = vstack_switchContext(pConn->pBindCb->vsId);
    if (vstackRC != VSTACK_SUCCESS) {
        TPL_IPC_PRINTK(("%s: Unable to switch Context for 0x%p,  %d\n", 
          __FUNCTION__, pConn, evntMsg->evnt_msg));
        return NFX_ERR;
    }

    /* application don't like this connection
     * close it
     */
    if (rc != 0) {
         pConn->flags |= TPL_CONN_BAD_APP;
        tpl_delConnReqCore(pConn);
    } else {
        pConn->flags |= TPL_CONN_GOOD_APP;

        /* any queue pkts need to sent ?? */
        for (tp = pConn->acpuPktQ; tp;) {

            ntp = tp->next;
            tp->next = NULL;
            tpl_givePktToApp(pConn, tp);
            tp = ntp;
        }
        pConn->acpuPktQ = NULL;

    }

    /* free msg that initiated from NCPU */
    tpl_ipc_deallocEvntMsg(evntMsg);
#else
    TPL_IPC_PRINTK(("%s: msg %d\n", __FUNCTION__, evntMsg->evnt_msg));
#endif /* LATER */
    return 0;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcEvntDelConnAck
 * Description: process del conn ack, NCPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipcEvntDelConnAck(tpl_evntMsg_t *evntMsg)
{
    tpl_connCb_t    *pConn;
    int           rc;

    TPL_IPC_PRINTK(("%s: msg %d\n", __FUNCTION__, evntMsg->evnt_msg));

    pConn = (tpl_connCb_t*)evntMsg->evnt_data;
    rc    = evntMsg->evnt_rc;
    
    tpl_delConnCore(pConn);
    
    /* free msg that initiated from NCPU */
    tpl_ipc_deallocEvntMsg(evntMsg);
    
    return 0;  
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipc_evnt_requests_available
 * Description: Process notification from ACPU that number of pending
 *              requests has dropped below maximum. NCPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipc_evnt_requests_available(tpl_evntMsg_t *evntMsg)
{
#ifdef LATER
    tpl_connCb_t *conn;

    while ((txrx_requests_pending < TXRX_MAX_PENDING_REQUESTS) &&
           ((conn = SIMPLEQ_FIRST(&pkt_ncpu_free_list.pf_conn_waiters)) != NULL)) {

        SIMPLEQ_REMOVE_HEAD(&pkt_ncpu_free_list.pf_conn_waiters, conn, waitNext);
        conn->flags &= ~TPL_CONN_WAIT_PACKETS;
        tpl_process_receive_queue(conn);
    }
#endif /* LATER */
    return 0;
}

/*-----------------------------------------------------------------
 * Name:        tpl_ipcEvntDelConnAck
 * Description: Process loopback event from acpu, NCPU function
 *-----------------------------------------------------------------
 */
int
tpl_ipc_evnt_loopback(tpl_evntMsg_t *evntMsg)
{
    // struct sk_buff *pkt = (struct sk_buff *)evntMsg;
    // pkt->p_to_acpu_timestamp = eee_timestamp();

    tpl_ipcq_write(&(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_NTOA]), evntMsg, TPL_EVNT_PKTQ_RX);
    return 0;
}

char *tpl_ipcq_name_str[] = {
    "ALLOC SMALL",
    "ALLOC LARGE",
    "ALLOC ULTRA",
    "EVNTQ NTOA",
    "EVNTQ ATON",
    "PKTQ  RX"
};

/*-----------------------------------------------------------------
 * Name:        tpl_ipcShowQueue
 * Description: show IPC queue
 *-----------------------------------------------------------------
 */
void
tpl_ipcShowQueues(void)
{
    int i;

    printk("tpl_ipcShowQueues: event message allocated: ntoa %d, aton %d, no mem %d\n\n",
           pTplCfg->tplStats.NtoAMsgCnt,
	   pTplCfg->tplStats.AtoNMsgCnt, 
           pTplCfg->tplStats.msgAllocFail);
    
    printk("ipc pkt stats \n");
    printk("-----------------------------------------------------------------\n");
    printk(" Pkt_Type       Forwarded          Received          droped      \n");
    printk("-----------------------------------------------------------------\n");

    printk(" Rx pkts:       %-20d%-20d%-20d\n",
	   pTplCfg->tplStats.cntIpcRxFwd, pTplCfg->tplStats.cntIpcRxRcv, pTplCfg->tplStats.cntIpcRxDrop);

    printk(" Tx pkts:       %-20d%-20d%-20d\n",
	   pTplCfg->tplStats.cntIpcTxFwd, pTplCfg->tplStats.cntIpcTxRcv, pTplCfg->tplStats.cntIpcTxDrop);

    printk(" Free pkts:     %-20d%-20d%-20d\n",
	   pTplCfg->tplStats.cntIpcFreeFwd, pTplCfg->tplStats.cntIpcFreeRcv, pTplCfg->tplStats.cntIpcFreeDrop); 

    printk(" Compl pkts:    %-20d%-20d%-20d\n",
	   pTplCfg->tplStats.cntIpcCompFwd, pTplCfg->tplStats.cntIpcCompRcv, pTplCfg->tplStats.cntIpcCompDrop);
    
 printk("tpl ipc queues\n");
 printk("----------------------------------------------------------------\n");
 printk("Name        Start       End         RdPtr       WrPtr       Drop        Rcv        Fwd         HWater\n");
 printk("----------------------------------------------------------------\n");
    
    for (i=0; i<TPL_IPC_Q_CNT; i++) {
        
        printk("%-12s%-12u%-12u%-12u%-12u%-12u%-12u\n",
               tpl_ipcq_name_str[i], 
               (unsigned int)pTplCfg->tplIpcQ[i].tpl_ipcqStartPtr,
               (unsigned int)pTplCfg->tplIpcQ[i].tpl_ipcqEndPtr,
               (unsigned int)pTplCfg->tplIpcQ[i].tpl_ipcqDropCnt,
               (unsigned int)pTplCfg->tplIpcQ[i].tpl_ipcqRdPtr.total_rd,
               (unsigned int)pTplCfg->tplIpcQ[i].tpl_ipcqWrPtr.total_wr,
               (unsigned int)pTplCfg->tplIpcQ[i].tpl_ipcqHWater);
    }
    return;
}

#ifdef LATER
void
tpl_ipcPollView()
{
    printk("tpl_ipcPollView()\n");
    return;
}

void
tpl_ipcPollClear()
{
    printk("tpl_ipcPollClear()\n");
    return;
}

void
tpl_ipcPollCfg(unsigned int  max_val)
{
    printk("tpl_ipcPollCfg(%d)\n", max_val);
    return;
}

eee_poll_cfg_t tpl_ipcPollCfgStruct[] = {
    {"view",    tpl_ipcPollView},
    {"clear",   tpl_ipcPollClear},
    {"tpl",     tpl_ipcPollCfg},
    {NULL, 0}
};
#endif /* LATER */


