/*
 *-----------------------------------------------------------------
 * Name: tpl-utils.c
 * Copyright (C) 2009, OnStor, Inc.
 * Description: Transport Packet Layer utility procedures
 *-----------------------------------------------------------------
 */
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/gfp.h>
#include <linux/slab.h>

#include "nfx-defs.h"
#include "nfx-types.h"
#include "nfx-error.h"

#include "eee-desc.h"

#include "tpl-api.h"
#include "tpl.h"

#include "tpl-ipc.h"

#include "pkt-api.h"

#include "pkt-queue-api.h"

// include "../sm-cifs/win-compat.h"

extern void pkt_align_data(void **data, unsigned int len);
extern int socket_receive(struct sock *sk, struct sk_buff **retp);

int
eee_sendMessage(void *h, unsigned int len, unsigned short src_port, unsigned short dst_port)
{
  return 0;
}

void
eee_freePacket(eee_desc_t *d)
{
  return;
}

int
eee_getApplicationId(char *name)
{
  return 0;
}

inline uint32
nb_message_len(void *ptr)
{
    uint8 *data = ptr;
    return (data[3]) + (data[2] << 8) + ((data[1] & 1) << 16);
}

#ifdef LATER
tpl-utils.c:(.text+0x5f1b0): undefined reference to `pq_split_right'
tpl-utils.c:(.text+0x5f6d4): undefined reference to `pq_pullup'
tpl-utils.c:(.text+0x5f9c8): undefined reference to `pq_make_empty'
tpl-utils.c:(.text+0x5f9e8): undefined reference to `pq_adjust'

tpl-utils.c:(.text+0x5f594): undefined reference to `pkt_freePkt'
tpl-utils.c:(.text+0x5f6c0): undefined reference to `pkt_ncpu_free_list'
tpl-utils.c:(.text+0x5f6c0): undefined reference to `pkt_acpu_free_chain'
tpl-utils.c:(.text+0x5f810): undefined reference to `pkt_align_data'

tpl-utils.c:(.text+0x5fa00): undefined reference to `nb_message_len'
#endif /* LATER */

int
linux_connect(struct socket *sock, unsigned int a, unsigned int b)
{
  return 0;
}

int
linux_listen(struct socket *sock, unsigned int a, unsigned int b)
{
  return 0;
}

int
linux_socket_close(struct socket *sock)
{
  return 0;
}

int
linux_socket_abort (struct socket *sock)
{
  return 0;
}

int
linux_socket(int protocol, int bindType, bsd_sarg_t *sockArg)
{
  return 0;
}

/*-------------------------------
 * TPL error control block.
 *-------------------------------
 */
typedef struct _tpl_errCb_t {
	char	*errMsg;
	int32	errCode;
	int32	action;
} tpl_errCb_t;

#define TPL_DO_NOTHING	0
#define TPL_DO_PANIC	1

tpl_errCb_t errCb[ ]= {
	{"No Shared Memory Resource",	TPL_ERR_NO_SHMEM_RSRC,	TPL_DO_PANIC},
	{"No Bind Resource",		TPL_ERR_NO_BIND_RSRC,	TPL_DO_NOTHING},
	{"No CAM Resource",		TPL_ERR_NO_CAM_RSRC,	TPL_DO_NOTHING},
	{"No Pkt Resource",		TPL_ERR_NO_PKT_RSRC,	TPL_DO_PANIC},
	{"No Resource",			TPL_ERR_NO_RSRC,	TPL_DO_PANIC},
	{"Duplicate Connection",	TPL_ERR_DUP_CONN,	TPL_DO_PANIC},
	{"Bad Handle",			TPL_ERR_BAD_HDL,	TPL_DO_PANIC},
	{"Invalid  Protocol",		TPL_ERR_BAD_PROT,	TPL_DO_PANIC},
	{"No Indication Function",	TPL_ERR_BAD_FUNC,	TPL_DO_PANIC},
	{"Invalid Bind CB",		TPL_ERR_BAD_BIND,	TPL_DO_PANIC},
	{"Invalid Conn CB",		TPL_ERR_BAD_CONN,	TPL_DO_NOTHING},
	{"Invalid BSD Event",		TPL_ERR_BAD_EVT,	TPL_DO_PANIC},
	{"Invalid Pkt",			TPL_ERR_BAD_PKT,	TPL_DO_PANIC},
	{"Duplicate Bind",		TPL_ERR_DUP_BIND,	TPL_DO_NOTHING },
	{"No Binding found",		TPL_ERR_NOT_BIND,	TPL_DO_NOTHING },
	{"Upper layer add failed",	TPL_ERR_UL_ADD_FAIL,	TPL_DO_NOTHING},
	{"Upper layer delet failed",	TPL_ERR_UL_DEL_FAIL,	TPL_DO_PANIC},
	{"Free PKT ring full",		TPL_ERR_RING_FULL,	TPL_DO_PANIC},
	{"Failed to send",		TPL_ERR_XMT_FAIL,	TPL_DO_PANIC},
	{"Missing local address",	TPL_ERR_NO_LADDR,	TPL_DO_PANIC},
	{"Missing remote address",	TPL_ERR_NO_RADDR,	TPL_DO_PANIC},
	{"Missing local port",		TPL_ERR_NO_LPORT,	TPL_DO_PANIC},
	{"Missing remote port",		TPL_ERR_NO_RPORT,	TPL_DO_PANIC},
	{"Remote address in server",	TPL_ERR_RADDR_IN_SRVR,	TPL_DO_PANIC},
	{"Remote port  in server",	TPL_ERR_RPORT_IN_SRVR,	TPL_DO_PANIC},
	{"Bad binding mode",		TPL_ERR_BAD_MODE,	TPL_DO_PANIC},
	{"Bad link ID ",		TPL_ERR_BAD_LINK,	TPL_DO_PANIC},
	{"Vstack not configured",	TPL_ERR_VSTACK_ERR,	TPL_DO_NOTHING},
	{"Unknown Error",		TPL_ERR_MAX,		TPL_DO_PANIC},
};

extern tpl_cfg_t *pTplCfg;
extern struct pkt_free_list pkt_ncpu_free_list;

extern void tpl_ipcq_write(tpl_ipcq_t *qCommon, tpl_evntMsg_t *entry, unsigned int msg_type);

/* define TPL_KPI_SUPPORT */
#define TPL_RPC_PULLUP_SIZE 512
#define TPL_NB_PULLUP_SIZE 36

#define TRUE 1
#define FALSE 0

extern int bsdrl_appid;

/* The number of requests currently executing on TxRx.
 */
atomic_t txrx_requests_pending;

/* The number of requests dropped because the pending requests have
 * exceeded the limit
 */
uint32 txrx_requests_dropped = 0;

// @@@; debug variable
boolean pkt_align_flag = TRUE;

/*-----------------------------------------------------------------
 * Name: tpl_logErr
 * Description: Log TPL Errors. Panic if TPL_DEBUG is defined.
 *-----------------------------------------------------------------
 */
void
tpl_logErr( char *fileName, int32 lineNum, int32 errCode, void *info)
{
	tpl_errCb_t *pErrCb;
	int32	msgIdx, maxIdx;

	/*-------------------------------
	 * Get msg Id relative to base.
	 *-------------------------------
	 */
	msgIdx = ((TPL_ERROR_BASE) - errCode);
	maxIdx = ((TPL_ERROR_BASE) - TPL_ERR_MAX);

	/*-------------------------------
	 * Make sure err code is in range.
	 *-------------------------------
	 */
	if ((msgIdx > maxIdx) || (msgIdx < 0)) msgIdx = maxIdx;

	/*-------------------------------
	 * Get error CB.
	 *-------------------------------
	 */
	pErrCb = &errCb[msgIdx];

	if (pErrCb->action == TPL_DO_PANIC) {
	  printk("TPL PANIC errCode %d: %s, File: %s, Line: %d Info:%lx\n",
	            errCode, pErrCb->errMsg, fileName, lineNum, (address_t)info);
	} else {
	  printk("TPL errCode, %d: %s, File: %s, Line: %d Info:%lx\n",
	            errCode, pErrCb->errMsg, fileName, lineNum, (address_t)info);
	}
	return;
}
 		
/*
 * Routine Description:
 *  Count the total number of bytes in the packet chain.
 * Arguments:
 *  pkt - the head of the packet chain
 * Return Value:
 *  Returns the number of bytes in the packet chain.
 */
uint32
pkt_chain_len(struct sk_buff *pkt)
{
    struct sk_buff *p;
    uint32 size = 0;

    for (p = pkt; p != NULL; p = p->next) {
      size += p->len; /* p->hdrSize XXX */
    }
    return size;
}

/*
 * Routine Description:
    Split the queue at the given offset from the beginning of the queue,
    allocating the packets if necessary.
    The data to the right of the split point will be left in the queue, and
    the data to the left of the split point is returned.
 * Arguments:
    pq - packet queue
    offset - offset from the beginning of the queue
 * Return Value:
    Returns the pointer to the beginning of the packet chain or NULL if
    failed to allocated the memory for splitting the queue.
 */
struct sk_buff *
tpl_pq_split(pkt_queue_t *pq, uint32 offset)
{
    struct sk_buff *split_skb, *skb, *head;
    uint32 split_skb_offset;

    head = NULL;
    pq_validate(pq);
    
    if (pq->pq_len > offset) {
        
      if ((pq->pq_len - pq->pq_tail->len) < offset) { /* pq->pq_tail->hdrSize XXX */
            /* Splitting the queue in the middle of the last packet
             */
            split_skb = pq->pq_tail;
            split_skb_offset = pq->pq_len - pq->pq_tail->len; /* pq->pq_tail->hdrSize XXX */

        } else {
            /* Search for the packet containing the split point.
             */
            split_skb = pq->pq_head;
            split_skb_offset = 0;

            while ((split_skb->len + split_skb_offset) < offset) { /* split_skb->hdrSize XXX */
	        split_skb_offset += split_skb->len; /* split_skb->hdrSize XXX */
                split_skb = split_skb->next;
            }

            if ((split_skb->len + split_skb_offset) == offset) { /* split_skb->hdrSize XXX */
                /* No new packet need to be allocated.
                 */
                goto done;
            }
        }

        skb = alloc_skb(pq->pq_len, (__GFP_HIGHMEM | __GFP_NOFAIL | __GFP_DMA));
        if (skb == NULL) {
            return NULL;
        }

        // skb->flags |= PKT_INPUT; XXX
        // skb->pTplConn = split_skb->pTplConn;

#ifdef TPL_KPI_SUPPORT
        skb->tstamp = split_skb->tstamp;
#endif /* TPL_KPI_SUPPORT */

        pq_split_right(pq, split_skb, offset - split_skb_offset, skb);
done:
        pq->pq_len -= offset;

        head = pq->pq_head;
        pq->pq_current = pq->pq_head = split_skb->next;

        pq->pq_start_offset = 0;
        pq->pq_offset = 0;
        split_skb->next = NULL;
        
        pq_validate(pq);

        // ASSERT(skb_chain_len(head) == offset);

    }  else {
        /* The offset is past the end of the queue.
         */
        head = pq->pq_head;
        pq_make_empty(pq);
        
        // ASSERT(skb_chain_len(head) == offset);
    }
    return head;
}

/*
 * Routine Description:
    Wait for the packets to become available.
 * Arguments:
    pConn - tpl connection
 * Return Value:
    None.
 */
void
tpl_wait_packets(tpl_connCb_t *pConn)
{
    if (!(pConn->flags & TPL_CONN_WAIT_PACKETS)) {

        pConn->flags |= TPL_CONN_WAIT_PACKETS;
        SIMPLEQ_INSERT_TAIL(&pkt_ncpu_free_list.pf_conn_waiters, pConn, waitNext);
    }
    return;
}

/*
 * Routine Description:
 *  Send the packet to the application on ACPU to process.  If the application 
 *  has not yet acknowledged the connection establishment, queue the packet 
 *  to be send later.
 * Arguments:
 *  pConn - the TPL connection
 *  skb - the packet descriptor
 * Return Values:
 *  None.
 */
void
tpl_givePktToApp(tpl_connCb_t *pConn, struct sk_buff *skb)
{
    tpl_ipcq_t  *pktRxQueuePtr = &(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_NTOA]);
    struct sk_buff *tp;

    /* application not ready to receive ? */
    if (!(pConn->flags & TPL_CONN_GOOD_APP)) {

		if (!(pConn->acpuPktQ)) {
			pConn->acpuPktQ = skb;
		} else {
			for (tp = pConn->acpuPktQ; tp->next; tp = tp->next) ;
			tp->next = skb;
		}
    } else {
	  pTplCfg->tplStats.cntIpcRxFwd++;
	  //skb->pTplConn = pConn;

#ifdef TPL_KPI_SUPPORT
	  skb->tstamp = eee_timestamp();
#endif /* TPL_KPI_SUPPORT */

	  /* XXX is this correct, skb does NOT have eventMsg header in from */
	  tpl_ipcq_write(pktRxQueuePtr, (tpl_evntMsg_t *)skb, TPL_EVNT_PKTQ_RX);
    }
    return;
}

/*
 * Routine Description:
    This function processes the data in the connection input queue. The
    incoming data is split into packet chains according to the RPC boundary
    and the resulting packet chains are sent to the application on ACPU.

    This function is called by the TCP input when the connection receives new
    data or after the packets required to finish the queue split operation
    become available.

    If it is required to allocate new packets and there are currently no
    packets available, the connection will be put on the allocator wait
    list. The allocator will call this function again after the packets will
    become available.

    The RPC fragments are accumulated in the queue, the current queue position
    points to the beginning of the last fragment which is currently being
    reassembled. After the last fragment is received, the whole RPC is from
    start of the queue to (pq_start_offset + last fragment len).
 * Arguments:
 *  pConn - TPL connection.
 * Return Value:
 *  None.
 */
void
tpl_process_receive_queue(tpl_connCb_t *pConn)
{
    struct sk_buff *pkt, *prev;
    uint32 rpcLen, pq_len_adj, rpcMarker;
    int status;

    while (pConn->rpc_queue.pq_len != 0) {

        if (! (pConn->flags & TPL_CONN_RPC_REASM)) {

            if ((pConn->rpc_queue.pq_len - pConn->rpc_queue.pq_start_offset) < 4) {
                /* Wait for more data to arrive
                 */
                return;
            }
            status = pq_pullup(NULL, &pConn->rpc_queue, sizeof(uint32),
                      &pkt_ncpu_free_list, M_NOWAIT);

            if (status != STATUS_SUCCESS) {
                /* Not enough memory, wait for the packets to become available
                 */
                tpl_wait_packets(pConn);
                return;
            }

            if (pConn->flags & TPL_CONN_USE_RPC) {
                /* Get RPC len from marker.
                 */
                rpcMarker = (*(uint32 *)pq_current_ptr(&pConn->rpc_queue));
                rpcMarker = ntohl(rpcMarker);
                rpcLen = rpcMarker & 0x7FFFFFFF;

                if ((rpcMarker & 0xFFF00000) == RPC_LAST_FRAG) {
                    /* This is the last RPC fragment, set the connection flag
                     * indicating this.
                     */
                    pConn->flags |= TPL_CONN_LAST_RPC;
                } else if ((rpcMarker & 0xFFF00000) != RPC_DELIMIT_FRAG) {
                    /* Invalid fragment flag, the RPC boundary has been lost,
                     * abort the connection.
                     */
                    pTplCfg->tplStats.cntRpcRasmErr++;
#ifdef LATER
                    E_LOG (class_1, info_s, bsdrl_appid, 0, 0, 0,
                           ("%s: Aborting connection {0x%x,0x%x,0x%x,0x%x} because of RPC reasm errors\n",
                            __FUNCTION__,
                            pConn->laddr,
                            pConn->lport,
                            pConn->raddr,
                            pConn->rport));
#endif /* LATER */
                    tpl_abortConn(pConn);
                    return;
                } 

                if (!((0 < rpcLen) && (rpcLen <= TPL_MAX_RPC_LEN)) || 
					((rpcLen & 0x3) != 0)) {
                    /* The RPC is zero length, too large or is not aligned, 
                     * abort the connection.  
                     */
                    pTplCfg->tplStats.cntRpcLenErr++;
#ifdef LATER
                    E_LOG (class_1, info_s, bsdrl_appid, 0, 0, 0,
                       ("%s: Aborting connection {0x%x,0x%x,0x%x,0x%x} because of RPC len errors, rpc len = %u\n",
                            __FUNCTION__,
                            pConn->laddr,
                            pConn->lport,
                            pConn->raddr,
                            pConn->rport,
                            rpcLen));
#endif /* LATER */
                    tpl_abortConn(pConn);
                    return;
                }

                /* Exclude RPC Marker.
                 */
                pq_validate(&pConn->rpc_queue);

                if (pConn->rpc_queue.pq_len == 4) {
                    kfree_skb(pConn->rpc_queue.pq_head);
                    pq_make_empty(&pConn->rpc_queue);

                } else {
                    if (pConn->rpc_queue.pq_offset == 0) {
                        pConn->rpc_queue.pq_current->data += 4;
                    } else {
                        memmove((pConn->rpc_queue.pq_current->data + pConn->rpc_queue.pq_offset),
                         (pConn->rpc_queue.pq_current->data + pConn->rpc_queue.pq_offset + 4),
                         (pConn->rpc_queue.pq_current->len - pConn->rpc_queue.pq_offset - 4));

			/*
			 * XXX
			 * (pConn->rpc_queue.pq_current->data + pConn->rpc_queue.pq_offset + 4),
			 * (pConn->rpc_queue.pq_current->len - pConn->rpc_queue.pq_offset - 4));
			 */
                    }
                    pConn->rpc_queue.pq_len -= 4;
                    // pConn->rpc_queue.pq_current->len -= 4; XXX
                    pConn->rpc_queue.pq_current->len -= 4;

                    // if (pConn->rpc_queue.pq_current->len == 0) XXX
                    if (pConn->rpc_queue.pq_current->len == 0) {

                        pkt = pConn->rpc_queue.pq_current;

                        if (pConn->rpc_queue.pq_head == pkt) {
                            pConn->rpc_queue.pq_head = pkt->next;

                        } else {
                            for (prev = pConn->rpc_queue.pq_head;
                                  prev->next != pkt;
                                  prev = prev->next) {
                                ;
                            }
                            prev->next = pkt->next;
                        }
                        pConn->rpc_queue.pq_current = pkt->next;
                        pConn->rpc_queue.pq_offset = 0;
                        pkt->next = NULL;

                        /* Only the first packet in the chain can become
                         * misaligned in case the RPC marker is split across 2
                         * packets.  The alignment of the rest of the chain is
                         * checked in tpl_rcvPkt().
                         */
			// pConn->rpc_queue.pq_current->len); XXX

                        pkt_align_data(&pConn->rpc_queue.pq_current->data,
                                       pConn->rpc_queue.pq_current->len);
                        kfree_skb(pkt);
                    } else {

                        /* In this case the data will be aligned, because of
                         * the check in tpl_rcvPkt().
                         */
		      // if (pConn->rpc_queue.pq_offset == pConn->rpc_queue.pq_current->len)

                        if (pConn->rpc_queue.pq_offset == pConn->rpc_queue.pq_current->len) {
                            pConn->rpc_queue.pq_current = pConn->rpc_queue.pq_current->next;
                            pConn->rpc_queue.pq_offset = 0;
                        }
                    }
                }
                pq_validate(&pConn->rpc_queue);
            } else {
	      // ASSERT(pConn->flags & TPL_CONN_USE_NETBIOS);

                /* Assume netbios messages are unfragmented
                 */
                pConn->flags |= TPL_CONN_LAST_RPC;

                /* Add 4 to rpcLen for the calculations using the total packet
                 * length below (we include the Netbios header for CIFS
                 * requests processing)
                 */
                rpcLen = nb_message_len(pq_current_ptr(&pConn->rpc_queue)) + 4;
            }

            /* Save the expected length in the connection in case we will have
             * to block and set the connection state.
             */
            pConn->rpcRasmLen = rpcLen;
            pConn->flags |= TPL_CONN_RPC_REASM;
        
            atomic_inc(&txrx_requests_pending);
        } else {
            /* The rpc length is saved in the connection.
             */
            rpcLen = pConn->rpcRasmLen;
        }

        if (txrx_requests_pending.counter >= TXRX_MAX_PENDING_REQUESTS) {
            /* If over the limit of pending requests, leave the data on
             * the queue and wait for the requests to finish.
             */
            tpl_wait_packets(pConn);
            return;
        }

        pq_len_adj = (pConn->rpc_queue.pq_len - pConn->rpc_queue.pq_start_offset);

        if (rpcLen > pq_len_adj) {
            /* Wait for more data to arrive
             */
            return;
        }

        /* We have received all the data for this RPC fragment.
         */
        if (pConn->flags & TPL_CONN_LAST_RPC) {

            /* Received the last RPC fragment.
             * Calculate the total RPC length.
             */
            rpcLen += pConn->rpc_queue.pq_start_offset;

            /* Reset the queue position to the beginning of the queue.
             */   
            pConn->rpc_queue.pq_offset = 0;
            pConn->rpc_queue.pq_start_offset = 0;
            pConn->rpc_queue.pq_current = pConn->rpc_queue.pq_head;

            /* Do the initial pullup for the NFS or CIFS.
             */
            status = pq_pullup(NULL,
                               &pConn->rpc_queue,
                               MIN(((pConn->flags & TPL_CONN_USE_NETBIOS) ?
                                    TPL_NB_PULLUP_SIZE : TPL_RPC_PULLUP_SIZE),
                                   pConn->rpc_queue.pq_len),
                               &pkt_ncpu_free_list,
                               M_NOWAIT);
                                      
            if (pkt_align_flag) {
                /* Align the pulled up packet (if any)
                 */
                if ((pConn->flags & TPL_CONN_USE_RPC)
                    && (pConn->rpc_queue.pq_current->next != NULL)) {

                   pkt_align_data(&pConn->rpc_queue.pq_current->next->data,
                                   pConn->rpc_queue.pq_current->next->len);
                }
            }
            if (status != STATUS_SUCCESS) {
                /* Can not pullup, wait for the packets to become available
                 */
                return;
            }

            if (rpcLen < pConn->rpc_queue.pq_len) {

                /* The current queue contains more data in addition to the
                 * current RPC.
                 */
                pkt = tpl_pq_split(&pConn->rpc_queue, rpcLen);
                if (pkt == NULL) {

                    /* Not enough memory, wait for the packets to become
                     * available.
                     */
                    tpl_wait_packets(pConn);
                    return;
                }
            } else {
                pkt = pConn->rpc_queue.pq_head;
                pq_make_empty(&pConn->rpc_queue);
            }

            /* Clear reassembly state and deliver the current packet to the
             * application.
             */
            pConn->flags &= ~(TPL_CONN_RPC_REASM | TPL_CONN_LAST_RPC);
            pConn->rpcRasmLen = 0;
            pkt->len = rpcLen;
            tpl_givePktToApp(pConn, pkt);
        } else {
            /* Wait for the next RPC fragment.
             */
            pConn->flags &= ~TPL_CONN_RPC_REASM;
            pq_adjust(&pConn->rpc_queue, rpcLen);
        }
    }
    return;
}

/*-----------------------------------------------------------------
 * Name:        tpl_rcvPkt
 * Description: Called by linux kernel when receiving a packet
 *              designated for an RPC/XDR/NFS Connection.
 * Arguments:
 *  pConnId => Address of TPL Connection Control Block
 *  skb => Address of socket buffer descriptor
 *-----------------------------------------------------------------
 */
void
tpl_rcvPkt(void *pConnId, struct sk_buff *skb)
{
    tpl_bindCb_t    *pBind;
    tpl_connCb_t    *pConn = (tpl_connCb_t *)pConnId;
#ifdef LATER
    struct sk_buff  *retp, *pkt;
    int ret;
#endif /* LATER */

    /*-------------------------------
     * Make sure connCb is still vaild.
     *-------------------------------
     */
    if (!pConn || ((pConn->flags & TPL_CONN_INUSE) == 0) || (pConn->flags & TPL_CONN_CLOSED)) {
        pTplCfg->tplStats.cntBadConnErr++;
        goto drop;
    }

    /*-------------------------------
     * Make sure bindCb is still vaild.
     *-------------------------------
     */
    pBind = pConn->pBindCb;
    if ((!pBind) || (!pBind->ul_rcvPktInd) || (pBind->bindFlags & TPL_BIND_CLOSE)) {
        pTplCfg->tplStats.cntBadBindErr++;
        goto drop;
    }

    pTplCfg->tplStats.cntRcvPkt++;
    
    /*-------------------------------
     * Save pConn to be used later
     * by application.
     *-------------------------------
     */
    // skb->pTplConn = pConn;

    /*-------------------------------
     * Handle TCP packets.
     *-------------------------------
     */
#ifdef LATER
    switch (pConn->prot) {

    case  TPL_PROT_TCP:
            /*-------------------------------
             * Call tcp_inputC to update 
             * tcp ack, seq & state etc.
             *-------------------------------
             */
            ret = socket_receive(pConn->pSock, &retp);
	    if (ret == 0) skb = retp;

	    /*-------------------------------
	     * Handle RPC marker & reassembly.
	     *-------------------------------
	     */

    if ((pConn->flags & (TPL_CONN_USE_RPC|TPL_CONN_USE_NETBIOS | TPL_CONN_FP_CONN))) {

            while ((skb != NULL) && (skb->len == 0)) {
                pkt = skb->next;
                skb->next = NULL;
                kfree_skb(skb);
                skb = pkt;
            }

#ifdef TPL_KPI_SUPPORT
            uint64 tval = eee_timestamp();
            kpi_time_count *time_count = &pTplCfg->tplStats.tpl_times_counts[TPL_TIME_RX];
#endif /* TPL_KPI_SUPPORT */

            if (skb) {
                struct sk_buff *last_packet = skb;
                uint32 size = skb->len;

#ifdef TPL_KPI_SUPPORT
                if (skb->tstamp != 0) {
                    // ASSERT(tval > skb->tstamp);
                    time_count->kt_time += (tval - skb->tstamp);
                    ++time_count->kt_count;
                }
#endif /* TPL_KPI_SUPPORT */

                pkt = skb->next;

                if (skb->hdrSize > (EEE_BUFSIZE_LARGE - (skb->data - (char *)skb))) {

                    E_LOG (class_1, info_s, bsdrl_appid, 0, 0, 0,
                           ("%s: Aborting connection {0x%x,0x%x,0x%x,0x%x} because of bad packet length %d\n",
                            __FUNCTION__,
                            pConn->laddr,
                            pConn->lport,
                            pConn->raddr,
                            pConn->rport,
                            skb->hdrSize));

                    while (skb != NULL) {
                        pkt = skb->next;
                        skb->next = NULL;
                        kfree_skb(skb);
                        skb = pkt;
                    }
                    tpl_abortConn(pConn);
                    return;
                }

                if ((pConn->flags & TPL_CONN_USE_RPC)) {
                    pkt_align_data(&skb->data, skb->len);
                }

                while (pkt != NULL) {

                    if (pkt->len == 0) {
                        last_packet->next = pkt->next;
                        pkt->next = NULL;
                        kfree_skb(pkt);
                        pkt = last_packet->next;
                    } else {
                        if (pkt->hdrSize > (EEE_BUFSIZE_LARGE - (pkt->data - (char *)pkt))) {
                             E_LOG (class_1, info_s, bsdrl_appid, 0, 0, 0,
                             ("%s: Aborting connection {0x%x,0x%x,0x%x,0x%x} because of bad packet length %d\n",
                                     __FUNCTION__,
                                     pConn->laddr,
                                     pConn->lport,
                                     pConn->raddr,
                                     pConn->rport,
                                     pkt->hdrSize));

                             while (skb != NULL) {
                                 pkt = skb->next;
                                 skb->next = NULL;
                                 kfree_skb(skb);
                                 skb = pkt;
                             }
                             tpl_abortConn(pConn);
                             return;
                         }
                        last_packet = pkt;
                        size += pkt->len;

                        if ((pConn->flags & TPL_CONN_USE_RPC)) {
                            pkt_align_data(&pkt->data, pkt->len);
                        }
#ifdef TPL_KPI_SUPPORT
                        if (pkt->tstamp != 0) {
                            // ASSERT(tval > pkt->tstamp);
                            time_count->kt_time += (tval - pkt->tstamp);
                            ++time_count->kt_count;
                        }
#endif /* TPL_KPI_SUPPORT */
                        pkt = pkt->next;
                    }
                } /* end of while */

                pq_append_chain_to_queue(&pConn->rpc_queue, skb, last_packet, size);
				
		if (!(pConn->flags & TPL_CONN_FP_CONN))
		  tpl_process_receive_queue(pConn);
		else 
		  goto sendPkt;
            } /* end of skb */
          } /* end of if conn fp */
	  return;

    case  TPL_PROT_UDP:
        /* UDP case
	 * Check if the length according to udp matches the length 
         * according to IP and set pktSize to UDP length excluding headers.  
         */
        // skb->len = udp_pkt_size;
        
        /* conn. is active again 
         */
        pConn->last_aTic = 0;
        break;

    default:
        break;
     } /* end of switch for connection type */

#endif /* LATER */

    if (atomic_add_return(1, &txrx_requests_pending) >= TXRX_MAX_PENDING_REQUESTS) {
       /* If over the limit of pending requests, drop the packet.
        */
        atomic_dec(&txrx_requests_pending);
        txrx_requests_dropped++;
        goto drop;
    }
    
    /*-------------------------------
     * Upcall app. for rcv pkt.
     *-------------------------------
     */
#if defined(NFX_SMP)
    /*------------------------------------
     * FP Connections need to use callback
     *------------------------------------
     */
    if (pConn->flags & TPL_CONN_FP_CONN) {
		if ((pConn->flags & TPL_CANTRECVMORE) == 0)
        	(pBind->ul_rcvPktInd)(pConn->pAppHdl, skb, skb->data, 0);
    } else {
		tpl_givePktToApp(pConn, skb);
    }
    /* Check if there the packet was received with FIN, if so do the FIN
     * processing here after the data has been processed
     */
    if (pConn->pSock) {
        struct socket *sock = (struct socket *)pConn->pSock;
        sock->so_connevent(sock, 0, SO_CONN_DOWN_IND, &(sock->so_handle)); 
    }
#else
    (pBind->ul_rcvPktInd)(pConn->pAppHdl, skb, skb->data, 0);
#endif /* NFX_SMP */
    return;
    
 drop:
    kfree_skb(skb);
    return;
}

/*
 * Routine Description:
 *  Free the TPL connection's RPC reassembly queue and reinitialize it to empty.  
 * Arguments:
 *  pConn - TPL connection
 * Return Value:
 *  None.
 */
void 
tpl_flushRpcRasm(tpl_connCb_t *pConn)
{
    /* Free pkt in reassembly chain. 
     */
    if (!pq_is_empty(&pConn->rpc_queue)) {
        kfree_skb(pConn->rpc_queue.pq_head);
    }
    pq_make_empty(&pConn->rpc_queue);
                    
    if (pConn->flags & TPL_CONN_RPC_REASM) {
        /* Clear reassembly state.
         */
        pConn->flags &= ~(TPL_CONN_RPC_REASM | TPL_CONN_LAST_RPC);
        atomic_dec(&txrx_requests_pending);
    }
    return;
}

/*
 * Routine Description:
 *  Finish the request processing on the ACPU. Decrement the number of pending
 *  requests and notify the NCPU if the number of requests has dropped below
 *  the maximum and there are blocked connections on the NCPU side.
 *
 * Arguments:
 *  None
 *
 * Return Values:
 *  None
 */
void
txrx_request_complete(void)
{
    // ASSERT(txrx_requests_pending > 0);
    uint32 pending = atomic_sub_return(1, &txrx_requests_pending);

    if (! SIMPLEQ_EMPTY(&pkt_ncpu_free_list.pf_conn_waiters) &&
        (pending < TXRX_MAX_PENDING_REQUESTS)) {
        /* Allow the TPL to proceed with new requests.
         */
        tpl_ipcq_write(&(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_ATON]), 
		       (tpl_evntMsg_t *)TPL_EVNT_REQUESTS_AVAILABLE,
                       TPL_EVNT_REQUESTS_AVAILABLE);
    }
}

#ifdef TPL_KPI_SUPPORT
/*
 * Routine Description:
 *  Initialize the KPIs for the request received from TPL.
 *
 * Arguments:
 *  req - request
 *  pkt - received packet
 *  tval - timestamp when the request was received on ACPU
 *  to - the request KPI
 *
 * Return Value:
 *  None.
 */
void
tpl_kpi_start(any_request_ptr_t req, struct sk_buff *pkt, uint64 tval, enum kpi_symbols to)
{
    kpi_start_at(&req.v->rh_kpi, KPI_TOP, to, pkt->tstamp);
    kpi_enter_at(&req.v->rh_kpi, KPI_PKT_RPC_REASM, KPI_LEVEL_3, pkt->tstamp);
    kpi_leave_at(&req.v->rh_kpi, KPI_PKT_RPC_REASM, KPI_LEVEL_3, pkt->p_to_acpu_timestamp);
    kpi_enter_at(&req.v->rh_kpi, KPI_PKT_ACPU_QUEUE, KPI_LEVEL_3, pkt->p_to_acpu_timestamp);
    kpi_leave_at(&req.v->rh_kpi, KPI_PKT_ACPU_QUEUE, KPI_LEVEL_3, tval);
}
#endif /* TPL_KPI_SUPPORT */

/*-----------------------------------------------------------------
 * Name: tpl_xmtPkt
 * Description: Transmit packet, ACPU Function
 * Returns: NFX_OK or error
 *-----------------------------------------------------------------
 */
int32
tpl_xmtPkt(void *pConnId, struct sk_buff *skb)
{
    tpl_ipcq_t *pktTxQueuePtr = &(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_ATON]);
    tpl_connCb_t *pConn = (tpl_connCb_t *)pConnId;
    // req_hdr_t *req = skb->p_req;
    int rc = NFX_OK;
    
    /* Make sure connCb is still vaild.
     */
    // ASSERT(pConn->flags & TPL_CONN_INUSE);
    // ASSERT(skb->flags & PKT_DESC_INUSE);

    /* If connection is closed on ncpu, we cannot forward this pkt.  
     */
    if (!(pConn->flags & TPL_CONN_CLOSED)) {

                /* conn. is active again */
                pConn->last_aTic = 0;

		/* no need to pass on pConnId, it's in skb->pTplConn
		 */
		pTplCfg->tplStats.cntIpcTxFwd++;
		tpl_ipcq_write(pktTxQueuePtr, (tpl_evntMsg_t *)skb, TPL_EVNT_PKTQ_TX);
    } else {
                TPL_LOGERR(TPL_ERR_BAD_PROT, skb);
                rc = TPL_ERR_BAD_PROT;
		pkt_acpu_free_from_ncpu(skb);
    }
    return rc;
}

/*-----------------------------------------------------------------
 * Name: tpl_xmtPktCore
 * Description: Transmit packet, ACPU Function
 * Returns: NFX_OK or error
 *-----------------------------------------------------------------
 */
int32
tpl_xmtPktCore(struct sock *sk, struct sk_buff *skb)
{
    pTplCfg->tplStats.cntIpcTxFwd++;
    tpl_ipcq_write(&(pTplCfg->tplIpcQ[TPL_IPC_EVNTQ_ATON]), (tpl_evntMsg_t *)skb, TPL_EVNT_PKTQ_TX);
    return NFX_OK;
}

/*-----------------------------------------------------------------
 * Name: tpl_unbindPort
 * Description: Removes bind control blocks of given local port. 
 *-----------------------------------------------------------------
 */
void tpl_unbindPort(ushort16 lport)
{
  tpl_bindCb_t *pBind;
  int32 i;

    pBind = pTplCfg->bindTblBase;
    for (i=0; i < pTplCfg->maxBindCb; i++) {
        if (pBind->lport || pBind->rport) {
	  if ((lport == ntohs(pBind->lport)) || (lport == TPL_ANY_PORT)) {
			tpl_unbindReq(pBind);
	  }
	}
        pBind++;
    }
}

/*-----------------------------------------------------------------
 * Name:        tpl_showCfg
 * Description: Show TPL config block contents.
 *-----------------------------------------------------------------
 */
void 
tpl_showCfg(void)
{
    printk("TPL CFG: %lx\n", (address_t)pTplCfg);
    printk("----------------------------------------------------\n");
    printk("\tBind CB Base 0x%lx\n", (address_t)pTplCfg->bindTblBase);
    printk("\tConn CB Base 0x%lx\n", (address_t)pTplCfg->connTblBase);
    printk("\tBind CB End  0x%p\n", pTplCfg->bindTblEnd);
    printk("\tConn CB End  0x%p\n", pTplCfg->connTblEnd);
    printk("\tNum Bind CB %8d\n", pTplCfg->numBindCb);
    printk("\tNum Conn CB %8d\n", pTplCfg->numConnCb);
    printk("\tMax Bind CB %8d\n", pTplCfg->maxBindCb);
    printk("\tMax Conn CB %8d\n", pTplCfg->maxConnCb);
    return;
}

/*-----------------------------------------------------------------
 * Name:        tpl_showStats
 * Description: Show TPL config block contents.
 *-----------------------------------------------------------------
 */
void 
tpl_showStats(void)
{
    printk("Request Statistics:\n");
    printk("----------------\n");
    printk("  Number of requests pending %u, dropped %u\n",
           atomic_read(&txrx_requests_pending), txrx_requests_dropped);

    printk("TPL Statistics:\n" );
    printk("----------------------------------------------------\n");
    printk("  Bind Req    :    %12u\n", pTplCfg->tplStats.cntBindReq);
    printk("  Unbind Req  :    %12u\n", pTplCfg->tplStats.cntUnbindReq);
    printk("  Add Conn Ind:    %12u\n", pTplCfg->tplStats.cntAddConnInd);
    printk("  Del Conn Ind:    %12u\n", pTplCfg->tplStats.cntDelConnInd);
    printk("  Del Conn Req:    %12u\n", pTplCfg->tplStats.cntDelConnReq);
    printk("  Alloc Pkt   :    %12u\n", pTplCfg->tplStats.cntAllocPkt + pTplCfg->tplStats.cntAcpuAllocPkt);
    printk("  Alloc OBd   :    %12u\n", pTplCfg->tplStats.cntOutConnAlloc + pTplCfg->tplStats.cntAcpuOutConnAlloc);
    printk("  HW  Outbd   :    %12u\n", pTplCfg->tplStats.cntOutHWater);
    printk("  Free  Pkt   :    %12u\n", pTplCfg->tplStats.cntFreePkt + pTplCfg->tplStats.cntAcpuFreePkt);
    printk("  Rcv   Pkt   :    %12u\n", pTplCfg->tplStats.cntRcvPkt);
    printk("  Xmt   Pkt   :    %12u\n", pTplCfg->tplStats.cntXmtPkt);
    printk("  Xmt   Cmp   :    %12u\n", pTplCfg->tplStats.cntXmtCmp);
    printk("  Xmt Up Call :    %12u\n", pTplCfg->tplStats.cntXmtUpCall);
    printk("  Xmt Dwn Call:    %12u\n", pTplCfg->tplStats.cntXmtDownCall);
    printk("  MAC Pkt Drop:    %12u\n", pTplCfg->tplStats.cntXmtNoMacErr);
    printk("  MAC Queue   :    %12u\n", pTplCfg->tplStats.cntXmtMacQueue);
    printk("  MAC Uncalled:    %12u\n", pTplCfg->tplStats.cntXmtMacUncall);
    printk("  MAC Pkt Drop:    %12u\n", pTplCfg->tplStats.cntXmtNoMacErr);
    printk("  Tcp Queue   :    %12u\n", pTplCfg->tplStats.cntXmtTcpQueue);
    printk("  Conn. lnkChg:    %12u\n", pTplCfg->tplStats.cntConnLinkChge);
    printk("  pkt lnkChg  :    %12u\n", pTplCfg->tplStats.cntPktLinkChge);

    printk("\n\nTPL Error Statistics:\n" );
    printk("----------------------------------------------------\n");
    printk("  Bad Bind Err:    %12u\n", pTplCfg->tplStats.cntBadBindErr);
    printk("  Bad Conn Err:    %12u\n", pTplCfg->tplStats.cntBadConnErr);
    printk("  Bad Pkt  Err:    %12u\n", pTplCfg->tplStats.cntBadPktErr);
    printk("  No  CAM Rsrc:    %12u\n", pTplCfg->tplStats.cntNoCamRsrc);
    printk("  No  Pkt Rsrc:    %12u\n", pTplCfg->tplStats.cntNoPktRsrc);
    printk("  Ip csum Err :    %12u\n", pTplCfg->tplStats.cntIpCsumErr);
    printk("  Udp csum Err:    %12u\n", pTplCfg->tplStats.cntUdpCsumErr);
    printk("  Tcp csum Err:    %12u\n", pTplCfg->tplStats.cntTcpCsumErr);
    printk("  In Tcp TxQ  :    %12u\n", pTplCfg->tplStats.cntInTcpTxQ);
    printk("  In Tcp Xmt  :    %12u\n", pTplCfg->tplStats.cntInTcpXmt);
    printk("  RPC Len Err :    %12u\n", pTplCfg->tplStats.cntRpcLenErr);
    printk("  RPC Rasm Err:    %12u\n", pTplCfg->tplStats.cntRpcRasmErr);
    printk("  Dup.Pkt Err :    %12u\n", pTplCfg->tplStats.cntXmtDup);
    printk("  Bad Pkt Tpl :    %12u\n", pTplCfg->tplStats.cntXmtNoTplUsr);
    printk("  Dup.Tcp Err :    %12u\n", pTplCfg->tplStats.cntXmtTcpDup);
    printk("  Tcp pkt Err :    %12u\n", pTplCfg->tplStats.cntXmtTcpErr);
    printk("  Pkt Drop Err:    %12u\n", pTplCfg->tplStats.cntPktDropFail);

    return;
}

/*-----------------------------------------------------------------
 * Name:        tpl_showBindCb
 * Description: Show given bind control block contents.
 *-----------------------------------------------------------------
 */
static void 
tpl_showBindCb(tpl_bindCb_t *pBind)
{
    printk("\nBind CB 0x%lx, protocol:%d,mode: %s\n",
	   (address_t)pBind, pBind->tplProt,
	   ((pBind->bindFlags & TPL_FLAG_CLIENT) ? "Client" : "Server") );
    printk("----------------------------------------------\n");
    printk("laddr=0x%x lport=%d\n", ntohl(pBind->laddr), ntohs(pBind->lport));
    printk("raddr=0x%x rport=%d\n", ntohl(pBind->raddr), ntohs(pBind->rport));
    printk("usrInfo       = 0x%lx\n", (address_t)pBind->usrInfo);
    printk("pConnList     = 0x%lx\t", (address_t)pBind->pConnList);
    printk("num of Conn   = %d\n",   pBind->numConnCb);
    printk("pSock         = 0x%lx\n", (address_t)pBind->pSock);
    printk("ul_addConnInd = 0x%lx\n", (address_t)pBind->ul_addConnInd);
    printk("ul_delConnInd = 0x%lx\n", (address_t)pBind->ul_delConnInd);
    printk("ul_rcvPktInd  = 0x%lx\n", (address_t)pBind->ul_rcvPktInd);

    return;
}

/*-----------------------------------------------------------------
 * Name:        tpl_showBind
 * Description: Show all bind control blocks of given local port. 
 *-----------------------------------------------------------------
 */
void tpl_showBind(ushort16 lport)
{
  tpl_bindCb_t	*pBind;
  int32			i;

    pBind = pTplCfg->bindTblBase;
    for (i=0; i < pTplCfg->maxBindCb; i++) {
        if (pBind->lport || pBind->rport) {
		if ((lport == ntohs(pBind->lport)) || (lport == TPL_ANY_PORT))
		  tpl_showBindCb(pBind);
	}
        pBind++;
    }
}

/*-----------------------------------------------------------------
 * Name:        tpl_showConnCb
 * Description: Show given connection control block contents.
 *-----------------------------------------------------------------
 */
void 
tpl_showConnCb(tpl_connCb_t *pConn)
{
    printk("TPL CONN CB=0x%lx protocol=%d\n", (address_t)pConn, pConn->prot );
    printk("----------------------------------------------\n");
    printk("laddr=0x%x lport=%d\n", ntohl(pConn->laddr), ntohs(pConn->lport));
    printk("raddr=0x%x rport=%d\n", ntohl(pConn->raddr), ntohs(pConn->rport));

    if (pConn->prot == TPL_PROT_TCP) {
      printk("Connection state : TCP - 0x%p\n", pConn->pSock);
    } else {
      printk("Connection state : UDP - 0x%p\n", pConn->pSock);
    }
    printk("bindCb 0x%p\n", (void *)(pConn->pBindCb));
    printk("appHdl 0x%p\n", (void *)(pConn->pAppHdl));

    //printk("link %d\n", pConn->linkNum);

    printk("flags 0x%x\n", pConn->flags);
    printk("last_atic %d\n", pConn->last_aTic);

    // pkt_showDescRing(&pConn->txPktRing);
    return;
}

/*-----------------------------------------------------------------
 * Name: tpl_showConn
 * Description: Show all connection control blocks of given local port. 
 *-----------------------------------------------------------------
 */
void tpl_showConn(ushort16 lport)
{
    tpl_bindCb_t *pBind;
    tpl_connCb_t *pConn;
    uint32 connCnt = 0;
    int32 i;

    pBind = pTplCfg->bindTblBase;
    for (i=0; i < pTplCfg->maxBindCb; i++) {
        if (pBind->lport || pBind->rport) {
		if ((lport == ntohs(pBind->lport)) || (lport == TPL_ANY_PORT)) {
		  pConn = pBind->pConnList;
		  while(pConn) {
		    tpl_showConnCb(pConn);
                    connCnt++;
		    pConn = pConn->pNext;
		  }
		}
	}
        pBind++;
    }
    printk("Total number of connections seen in TPL layer: %u\n", connCnt);

    return;
}

void tpl_clearStats(void)
{
	memset(&pTplCfg->tplStats, 0, sizeof(tpl_stats_t));
	return;
}

/*-----------------------------------------------------------------
 * Name:        tpl_validateConnCb
 * Description: Validate a given connection control block.
 *-----------------------------------------------------------------
 */
int32 tpl_validateConnCb(void *pConnCb)
{
	tpl_any_conn_t *pConn = pConnCb;

	/*-------------------------------
	 * Check for valid range & flag.
	 *-------------------------------
	 */
	if ((pConn < pTplCfg->connTblBase ) || (pConn > pTplCfg->connTblEnd )	||
		((((tpl_connCb_t *)pConn)->flags & TPL_CONN_INUSE) == 0)) {
		pTplCfg->tplStats.cntBadConnErr++;
		return TPL_ERR_BAD_CONN;
	}
	return (NFX_OK);
}

/*-----------------------------------------------------------------
 * Name:        tpl_validateBindCb
 * Description: Validate a given Binding control block.
 *-----------------------------------------------------------------
 */
int32 tpl_validateBindCb(tpl_bindCb_t *pBind)
{
  /*-------------------------------
   * Check for valid range. 
   *-------------------------------
   */
  if ((pBind < pTplCfg->bindTblBase) || (pBind > pTplCfg->bindTblEnd)) {
    pTplCfg->tplStats.cntBadBindErr++;
    return TPL_ERR_BAD_BIND;
  }

    /* The bind endpoint should represent a UDP or a TCP endpoint
     */
    switch (pBind->tplProt) {
      case TPL_PROT_UDP:
      case TPL_PROT_TCP:
		break;
      default:
		pTplCfg->tplStats.cntBadBindErr++;
		return TPL_ERR_BAD_BIND;
    }
    /*-------------------------------
     * Check if upcalls missing.
     *-------------------------------
     */
    if ((!pBind->ul_delConnInd) || (!pBind->ul_addConnInd) || (!pBind->ul_rcvPktInd)) {
      pTplCfg->tplStats.cntBadBindErr++;
      return TPL_ERR_NOT_BIND;
    }
    return (NFX_OK);
}

